package App::Manoc::ControllerRole::ObjectList;
#ABSTRACT: Role for controllers accessing a list of results
use Moose::Role;

our $VERSION = '2.99.2'; ##TRIAL VERSION

use namespace::autoclean;

use MooseX::MethodAttributes::Role;


requires 'base';


has object_list_filter_columns => (
    is      => 'rw',
    isa     => 'ArrayRef[Str]',
    default => sub { [] }
);


has object_list_options => (
    is      => 'rw',
    isa     => 'HashRef',
    default => sub { {} }
);


sub object_list : Chained('base') : PathPart('') : CaptureArgs(0) {
    my ( $self, $c ) = @_;
    $c->stash( object_list => $self->get_object_list($c) );
}


sub get_object_list {
    my ( $self, $c ) = @_;

    my $rs      = $c->stash->{resultset};
    my $filter  = $self->get_object_list_filter($c);
    my $options = $c->stash->{object_list_options} || $self->object_list_options;
    return [ $rs->search( $filter, $options )->all ];
}


sub get_object_list_filter {
    my ( $self, $c ) = @_;

    my %filter;

    my $qp = $c->req->query_parameters;
    foreach my $col ( @{ $self->object_list_filter_columns } ) {
        my $param = $qp->{$col};
        defined($param) or next;
        ref($param) eq "ARRAY" and next;
        $filter{$col} = $param;
        $c->log->debug("filter object list $col = $param") if $c->debug;
    }

    return \%filter;
}

1;
# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::ControllerRole::ObjectList - Role for controllers accessing a list of results

=head1 VERSION

version 2.99.2

=head1 DESCRIPTION

This is a base role for Manoc controllers which manage a list of rows from
a resultset.

=head1 ATTRIBUTES

=head2 object_list_filter_columns

=head2 object_list_options

Options for the DBIc search in C<get_object_list>.
Can be overridden by $c->stash->{object_list_options}.

=head1 ACTIONS

=head2 object_list

Load the list of objects from the resultset into the stash. Chained to base.
This is the point for chaining all actions using the list of object.

Objects are fetched by C<get_object_list> and stored in $c->stash->{object_list}.

=head2 object_list

Populate object_list in stash using get_object_list method.

=head1 METHODS

=head2 get_object_list

Search in $c->stash->{resultset} using the filter returned by
C<get_object_list_filter> and the options in $c->stash->{object_list_options}
or object_list_options.

=head2 get_object_list_filter

=head1 SYNOPSYS

  package App::Manoc::Controller::Artist;

  use Moose;
  extends "Catalyst::Controller";
  with "App::Manoc::ControllerRole::Object";

  __PACKAGE__->config(
      # define PathPart
      action => {
          setup => {
              PathPart => 'artist',
          }
      },
      class      => 'ManocDB::Artist',
      );

  # manages /artist/
  sub list : Chained('object_list') : PathPart('') : Args(0) {
     my ( $self, $c ) = @_;

     # render with default template
     # objects are stored in $c->{object_list}
  }

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
