package Git::Wrapper::More;

use strict;
use warnings;

use parent 'Git::Wrapper';

use List::MoreUtils qw(uniq);
use Try::Tiny;
use Scalar::Util ();
use Carp;

our $VERSION = '0.04';

sub new {
    my ( $class, %args ) = @_;

    my $self = {
        gw_obj  => Git::Wrapper->new( $args{dir} ),
        verbose => $args{verbose},
    };

    return bless $self, $class;
}

sub canstatus {
    my $self = shift;
    my $ret;

    Try::Tiny::try {
        my $try = $self->{gw_obj}->status();
        $ret = 1;
    }
    Try::Tiny::catch {
        $ret = 0;
    };

    return $ret;
}

sub cwdbranch {
    my $self = shift;

    my $ret = ( $self->{gw_obj}->rev_parse(qw/ --abbrev-ref HEAD /) )[0];

    return $ret;
}

sub toplevel {
    my $self = shift;

    my $ret = ( $self->{gw_obj}->rev_parse(qw/ --show-toplevel /) )[0];

    return $ret;
}

sub lsfiles {
    my ( $self, $top_level_dir ) = @_;

    my @ret = $self->{gw_obj}->ls_files( $top_level_dir, { full_name => 1 } );

    return @ret;
}

sub mergebase {
    my ( $self, $branch_cur, $branch_parent, $remote ) = @_;

    $branch_parent //= 'master';
    $remote //= $self->getremote($branch_cur);

    if ( !defined $remote ) {
        $self->_print_warn('No git remote found for current branch, trying origin.');
        $remote = 'origin';
    }

    # try with remote
    my $cmd = sub {
        ( $self->{gw_obj}->merge_base( "$remote/$branch_cur", "$remote/$branch_parent" ) )[0];
    };
    my $ret = $self->_run_cmd($cmd);

    # try without remote
    if ( !defined $ret ) {
        $self->_print_warn('No git remote found for current branch, trying without any remote.');

        $cmd = sub {
            ( $self->{gw_obj}->merge_base( "$branch_cur", "$branch_parent" ) )[0];
        };

        $ret = $self->_run_cmd($cmd);
    }

    $self->_print_warn("No common commit (merge base) found!\n") unless defined $ret;

    return $ret;
}

sub getremote {
    my ( $self, $branch ) = @_;

    my $cmd = sub {
        ( $self->{gw_obj}->config( "--get", "branch.$branch.remote" ) )[0];
    };

    my $ret = $self->_run_cmd($cmd);

    return $ret;
}

sub logsincecommit {
    my ( $self, $commit ) = @_;

    my @ret = $self->{gw_obj}->log(qq/${commit}../);

    return @ret;
}

sub lastmergecommit {
    my ( $self, $commit ) = @_;

    my $ret = ( $self->{gw_obj}->rev_list( { merges => 1 }, ${commit} ) )[0];

    return $ret;
}

sub logfiles {
    my ( $self, $num ) = @_;

    my @raw_results =
        $self->{gw_obj}->RUN( 'log', "--max-count=$num", '--name-only', '--pretty=format:' );

    my @ret = grep { $_ ne '' } List::MoreUtils::uniq(@raw_results);

    return @ret;
}

sub showcommit {
    my ( $self, $commit ) = @_;

    my $cmd = sub {
        $self->{gw_obj}->show( { summary => 1 }, $commit );
    };

    my @ret = $self->_run_cmd( $cmd, 'array' );

    return @ret;
}

sub logsincedate {
    my ( $self, $days ) = @_;

    my @raw_results = $self->{gw_obj}->RUN( 'log', "--since=$days", '--name-only', '--pretty=format:' );

    my @ret = grep { $_ ne '' } List::MoreUtils::uniq(@raw_results);

    return @ret;
}

sub diffbranches {
    my ( $self, $branch1, $branch2 ) = @_;

    my $range = qq{$branch1..$branch2};
    my @raw_results = $self->{gw_obj}->diff( qw/--name-status --diff-filter=M/, $range );

    # filter out just the files
    my @ret = map {m/M\s+(.+)$/} @raw_results;

    return @ret;
}

sub diffbranchesmergecommit {
    my ( $self, $branch1, $branch2 ) = @_;

    my $merge_commit = $self->mergebase( $branch1, $branch2 );
    die "[error] No common commit (merge base) found!\n" unless defined $merge_commit;

    my @coms        = $self->logsincecommit($merge_commit);
    my $num_commits = ( scalar @coms ) + 1;
    my @ret         = $self->logfiles($num_commits);

    return @ret;
}

sub unstagedfiles {
    my $self = shift;

    my @raw_results = $self->{gw_obj}->diff(qw/--name-status --diff-filter=M/);

    # filter out just the files
    my @ret = map {m/M\s+(.+)$/} @raw_results;

    return @ret;
}

sub _print_warn {
    my $self = shift;
    my $msg  = shift;

    my $ret = '[warn] ' . $msg;
    $self->{verbose} ? warn $ret : print "\n" . $ret;

    return;
}

sub _run_cmd {
    my ( $self, $cmd, $ref_type ) = @_;

    $ref_type //= 'scalar';
    die 'invalid value for $ref_type' unless $ref_type =~ /^(scalar|array)$/;

    my $ret;
    my @ret;

    Try::Tiny::try {
        ( $ref_type eq 'array' ) ? @ret = &{$cmd}() : $ret = &{$cmd}();
    }
    Try::Tiny::catch {

        # $_ is the error, same as eval's $@

        # if GW error is not an object, something went very wrong
        # quotes are overloaded
        Carp::cluck "[error][string] $_\n" and return unless Scalar::Util::blessed $_;

        # otherwise, normal errors can be printed if user asks with verbose flag
        # print STERR from erroneous git command
        warn $_->error . "\n" if $_->error && $self->{verbose};

        # print STOUT from git command
        warn $_->output . "\n" if $_->output && $self->{verbose};
    };

    return ( $ref_type eq 'array' ) ? @ret : $ret;
}

1;

__END__

=pod

=head1 NAME

Git::Wrapper::More - A handy collection of Git::Wrapper usages

=head1 SYNOPSIS

    my $git = Git::Wrapper::More->new(
        dir     => $dir,
        verbose => 1,
    );

    my $branch_from_working_dir = $git->cwdbranch;

=head1 DESCRIPTION

Git::Wrapper powerfully enables git repository manipulation, but has multiple command invocations. They sometimes require only one style or another.

This module can serve as a cheatsheet for some of the more common ones, or used to export a list of simple commands.

=head1 SUBROUTINES/METHODS

=head2 canstatus

    command line example:
        git status

=head3 ARGUMENTS

    none

=head3 RETURNS

    1 if git object can perform the most basic query, 0 if not


=head2 cwdbranch

    command line example:
        git rev-parse --abbrev-ref HEAD

=head3 ARGUMENTS

    none

=head3 RETURNS

    scalar: branch of current working directory


=head2 toplevel

    command line example:
        git rev-parse --show-toplevel

=head3 ARGUMENTS

    none

=head3 RETURNS

    scalar: top level directory of the git repository


=head2 lsfiles

    command line example:
        cd /top/level/directory; git ls-files

=head3 ARGUMENTS

    scalar: top level directory

=head3 RETURNS

    array: all repo files from top-level directory

=head2 mergebase

    finds the commit from which the current branch forked from parent branch

    first tries with the remote provided as input
    then tries with a default remote named 'origin', as that's the most common
     remote in current environment
    finally tries without any remote at all

    command line example:
        git merge-base my-current-branch master

=head3 ARGUMENTS

    scalar: current branch
    scalar: parent branch
    scalar (optional): git remote these branches are on

=head3 RETURNS

    scalar: commit

=head2 getremote

    command line example:
        git config --get branch.$branch.remote

    note: if there's no remote tracking, Git::Wrapper dies

=head3 ARGUMENTS

    scalar: branch

=head3 RETURNS

    the remote in use

=head2 logsincecommit

    command line example:
        git log 39631b5e862ec5fda9bd73f35d775136839a6f85..

=head3 ARGUMENTS

    scalar: commit

=head3 RETURNS

    array: commits since a particular commit

=head2 lastmergecommit

    command line example:
        git rev-list --merges c4a366176b77e806b2b1a3d19f93a74acd05c599 | head -1

=head3 ARGUMENTS

    scalar: commit

=head3 RETURNS

    scalar: most recent merge commit

=head2 logfiles

    command line example:
        git log -n 3 --name-only --pretty=format: | sort -u

=head3 ARGUMENTS

    scalar: number

=head3 RETURNS

    array: files modified in the last n logs

=head2 showcommit

    command line example:
        git show --summary 39631b5e862ec5fda9bd73f35d775136839a6f85

=head3 ARGUMENTS

    scalar: commit hash

=head3 RETURNS

    array: commit summary and details

=head2 logsincedate

    command line example:
        git log --since="$days days ago" --name-only --pretty=format:

=head3 ARGUMENTS

    scalar: number of days 

=head3 RETURNS

    array: files modified in the last n days

=head2 diffbranches

    command line example:
        git diff --name-status --diff-filter=M branch1..branch2 | awk '/^M/ {print $2}'

=head3 ARGUMENTS

    scalars: $branch1, $branch2 

=head3 RETURNS

    array: files different between 2 branches

=head2 diffbranchesmergecommit

    (experimental) diff of branches

=head3 ARGUMENTS

    scalars: $branch1, $branch2 

=head3 RETURNS

    array: files different between 2 branches, using merge commit

=head2 unstagedfiles

    command line example:
        git diff --name-status --diff-filter=M | awk '/^M/ {print $2}'

=head3 ARGUMENTS

    none

=head3 RETURNS

    array: changed files in current branch

=head2 _print_warn

    prints warn message with optional verbosity

=head3 ARGUMENTS

    scalar: warnning message

=head3 RETURNS

    none

=head2 _run_cmd

    run the Git::Wrapper command
    catches and prints any errors, without letting it die

=head3 ARGUMENTS

    reference: subroutine of a Git::Wrapper command

=head3 RETURNS

    results of the Git::Wrapper command
    scalar or array: depends on input variable

=head1 AUTHOR

Marco Ferrufino, C<< <marcojm02+cpan at gmail.com> >>

=head1 BUGS

Please report any bugs or feature requests to C<bug-git-wrapper-more at rt.cpan.org>, or through
the web interface at L<https://rt.cpan.org/NoAuth/ReportBug.html?Queue=Git-Wrapper-More>.  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

=head1 LICENSE AND COPYRIGHT

This software is Copyright (c) 2019 by cPanel L.L.C

copyright@cpanel.net

This is free software, licensed under:

  The Artistic License 2.0 (GPL Compatible)

=cut
