// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package core

import (
	"context"
	"errors"
	"regexp"

	"github.com/drone/drone-yaml/yaml"
)

var (
	errSecretNameInvalid = errors.New("Invalid Secret Name")
	errSecretDataInvalid = errors.New("Invalid Secret Value")
)

type (
	// Secret represents a secret variable, such as a password or token,
	// that is provided to the build at runtime.
	Secret struct {
		ID              int64  `json:"id,omitempty"`
		RepoID          int64  `json:"repo_id,omitempty"`
		Namespace       string `json:"namespace,omitempty"`
		Name            string `json:"name,omitempty"`
		Type            string `json:"type,omitempty"`
		Data            string `json:"data,omitempty"`
		PullRequest     bool   `json:"pull_request,omitempty"`
		PullRequestPush bool   `json:"pull_request_push,omitempty"`
	}

	// SecretArgs provides arguments for requesting secrets
	// from the remote service.
	SecretArgs struct {
		Name  string         `json:"name"`
		Repo  *Repository    `json:"repo,omitempty"`
		Build *Build         `json:"build,omitempty"`
		Conf  *yaml.Manifest `json:"-"`
	}

	// SecretStore manages repository secrets.
	SecretStore interface {
		// List returns a secret list from the datastore.
		List(context.Context, int64) ([]*Secret, error)

		// Find returns a secret from the datastore.
		Find(context.Context, int64) (*Secret, error)

		// FindName returns a secret from the datastore.
		FindName(context.Context, int64, string) (*Secret, error)

		// Create persists a new secret to the datastore.
		Create(context.Context, *Secret) error

		// Update persists an updated secret to the datastore.
		Update(context.Context, *Secret) error

		// Delete deletes a secret from the datastore.
		Delete(context.Context, *Secret) error
	}

	// GlobalSecretStore manages global secrets accessible to
	// all repositories in the system.
	GlobalSecretStore interface {
		// List returns a secret list from the datastore.
		List(ctx context.Context, namespace string) ([]*Secret, error)

		// ListAll returns a secret list from the datastore
		// for all namespaces.
		ListAll(ctx context.Context) ([]*Secret, error)

		// Find returns a secret from the datastore.
		Find(ctx context.Context, id int64) (*Secret, error)

		// FindName returns a secret from the datastore.
		FindName(ctx context.Context, namespace, name string) (*Secret, error)

		// Create persists a new secret to the datastore.
		Create(ctx context.Context, secret *Secret) error

		// Update persists an updated secret to the datastore.
		Update(ctx context.Context, secret *Secret) error

		// Delete deletes a secret from the datastore.
		Delete(ctx context.Context, secret *Secret) error
	}

	// SecretService provides secrets from an external service.
	SecretService interface {
		// Find returns a named secret from the global remote service.
		Find(context.Context, *SecretArgs) (*Secret, error)
	}
)

// Validate validates the required fields and formats.
func (s *Secret) Validate() error {
	switch {
	case len(s.Name) == 0:
		return errSecretNameInvalid
	case len(s.Data) == 0:
		return errSecretDataInvalid
	case slugRE.MatchString(s.Name):
		return errSecretNameInvalid
	default:
		return nil
	}
}

// Copy makes a copy of the secret without the value.
func (s *Secret) Copy() *Secret {
	return &Secret{
		ID:              s.ID,
		RepoID:          s.RepoID,
		Namespace:       s.Namespace,
		Name:            s.Name,
		Type:            s.Type,
		PullRequest:     s.PullRequest,
		PullRequestPush: s.PullRequestPush,
	}
}

// slug regular expression
var slugRE = regexp.MustCompile("[^a-zA-Z0-9-_.]+")
