{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The object that handles DNS resolution. Use 'GI.Gio.Objects.Resolver.resolverGetDefault'
-- to get the default resolver.
-- 
-- @GResolver@ provides cancellable synchronous and asynchronous DNS
-- resolution, for hostnames ('GI.Gio.Objects.Resolver.resolverLookupByAddress',
-- 'GI.Gio.Objects.Resolver.resolverLookupByName' and their async variants) and SRV
-- (service) records ('GI.Gio.Objects.Resolver.resolverLookupService').
-- 
-- t'GI.Gio.Objects.NetworkAddress.NetworkAddress' and t'GI.Gio.Objects.NetworkService.NetworkService' provide wrappers
-- around @GResolver@ functionality that also implement
-- t'GI.Gio.Interfaces.SocketConnectable.SocketConnectable', making it easy to connect to a remote
-- host\/service.
-- 
-- The default resolver (see 'GI.Gio.Objects.Resolver.resolverGetDefault') has a timeout of
-- 30s set on it since GLib 2.78. Earlier versions of GLib did not support
-- resolver timeouts.
-- 
-- This is an abstract type; subclasses of it implement different resolvers for
-- different platforms and situations.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.Resolver
    ( 

-- * Exported types
    Resolver(..)                            ,
    IsResolver                              ,
    toResolver                              ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [lookupByAddress]("GI.Gio.Objects.Resolver#g:method:lookupByAddress"), [lookupByAddressAsync]("GI.Gio.Objects.Resolver#g:method:lookupByAddressAsync"), [lookupByAddressFinish]("GI.Gio.Objects.Resolver#g:method:lookupByAddressFinish"), [lookupByName]("GI.Gio.Objects.Resolver#g:method:lookupByName"), [lookupByNameAsync]("GI.Gio.Objects.Resolver#g:method:lookupByNameAsync"), [lookupByNameFinish]("GI.Gio.Objects.Resolver#g:method:lookupByNameFinish"), [lookupByNameWithFlags]("GI.Gio.Objects.Resolver#g:method:lookupByNameWithFlags"), [lookupByNameWithFlagsAsync]("GI.Gio.Objects.Resolver#g:method:lookupByNameWithFlagsAsync"), [lookupByNameWithFlagsFinish]("GI.Gio.Objects.Resolver#g:method:lookupByNameWithFlagsFinish"), [lookupRecords]("GI.Gio.Objects.Resolver#g:method:lookupRecords"), [lookupRecordsAsync]("GI.Gio.Objects.Resolver#g:method:lookupRecordsAsync"), [lookupRecordsFinish]("GI.Gio.Objects.Resolver#g:method:lookupRecordsFinish"), [lookupService]("GI.Gio.Objects.Resolver#g:method:lookupService"), [lookupServiceAsync]("GI.Gio.Objects.Resolver#g:method:lookupServiceAsync"), [lookupServiceFinish]("GI.Gio.Objects.Resolver#g:method:lookupServiceFinish"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getTimeout]("GI.Gio.Objects.Resolver#g:method:getTimeout").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDefault]("GI.Gio.Objects.Resolver#g:method:setDefault"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setTimeout]("GI.Gio.Objects.Resolver#g:method:setTimeout").

#if defined(ENABLE_OVERLOADING)
    ResolveResolverMethod                   ,
#endif

-- ** getDefault #method:getDefault#

    resolverGetDefault                      ,


-- ** getTimeout #method:getTimeout#

#if defined(ENABLE_OVERLOADING)
    ResolverGetTimeoutMethodInfo            ,
#endif
    resolverGetTimeout                      ,


-- ** lookupByAddress #method:lookupByAddress#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupByAddressMethodInfo       ,
#endif
    resolverLookupByAddress                 ,


-- ** lookupByAddressAsync #method:lookupByAddressAsync#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupByAddressAsyncMethodInfo  ,
#endif
    resolverLookupByAddressAsync            ,


-- ** lookupByAddressFinish #method:lookupByAddressFinish#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupByAddressFinishMethodInfo ,
#endif
    resolverLookupByAddressFinish           ,


-- ** lookupByName #method:lookupByName#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupByNameMethodInfo          ,
#endif
    resolverLookupByName                    ,


-- ** lookupByNameAsync #method:lookupByNameAsync#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupByNameAsyncMethodInfo     ,
#endif
    resolverLookupByNameAsync               ,


-- ** lookupByNameFinish #method:lookupByNameFinish#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupByNameFinishMethodInfo    ,
#endif
    resolverLookupByNameFinish              ,


-- ** lookupByNameWithFlags #method:lookupByNameWithFlags#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupByNameWithFlagsMethodInfo ,
#endif
    resolverLookupByNameWithFlags           ,


-- ** lookupByNameWithFlagsAsync #method:lookupByNameWithFlagsAsync#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupByNameWithFlagsAsyncMethodInfo,
#endif
    resolverLookupByNameWithFlagsAsync      ,


-- ** lookupByNameWithFlagsFinish #method:lookupByNameWithFlagsFinish#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupByNameWithFlagsFinishMethodInfo,
#endif
    resolverLookupByNameWithFlagsFinish     ,


-- ** lookupRecords #method:lookupRecords#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupRecordsMethodInfo         ,
#endif
    resolverLookupRecords                   ,


-- ** lookupRecordsAsync #method:lookupRecordsAsync#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupRecordsAsyncMethodInfo    ,
#endif
    resolverLookupRecordsAsync              ,


-- ** lookupRecordsFinish #method:lookupRecordsFinish#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupRecordsFinishMethodInfo   ,
#endif
    resolverLookupRecordsFinish             ,


-- ** lookupService #method:lookupService#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupServiceMethodInfo         ,
#endif
    resolverLookupService                   ,


-- ** lookupServiceAsync #method:lookupServiceAsync#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupServiceAsyncMethodInfo    ,
#endif
    resolverLookupServiceAsync              ,


-- ** lookupServiceFinish #method:lookupServiceFinish#

#if defined(ENABLE_OVERLOADING)
    ResolverLookupServiceFinishMethodInfo   ,
#endif
    resolverLookupServiceFinish             ,


-- ** setDefault #method:setDefault#

#if defined(ENABLE_OVERLOADING)
    ResolverSetDefaultMethodInfo            ,
#endif
    resolverSetDefault                      ,


-- ** setTimeout #method:setTimeout#

#if defined(ENABLE_OVERLOADING)
    ResolverSetTimeoutMethodInfo            ,
#endif
    resolverSetTimeout                      ,




 -- * Properties


-- ** timeout #attr:timeout#
-- | The timeout applied to all resolver lookups, in milliseconds.
-- 
-- This may be changed through the lifetime of the t'GI.Gio.Objects.Resolver.Resolver'. The new value
-- will apply to any lookups started after the change, but not to any
-- already-ongoing lookups.
-- 
-- If this is @0@, no timeout is applied to lookups.
-- 
-- No timeout was applied to lookups before this property was added in
-- GLib 2.78.
-- 
-- /Since: 2.78/

#if defined(ENABLE_OVERLOADING)
    ResolverTimeoutPropertyInfo             ,
#endif
    constructResolverTimeout                ,
    getResolverTimeout                      ,
#if defined(ENABLE_OVERLOADING)
    resolverTimeout                         ,
#endif
    setResolverTimeout                      ,




 -- * Signals


-- ** reload #signal:reload#

    ResolverReloadCallback                  ,
#if defined(ENABLE_OVERLOADING)
    ResolverReloadSignalInfo                ,
#endif
    afterResolverReload                     ,
    onResolverReload                        ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.InetAddress as Gio.InetAddress
import {-# SOURCE #-} qualified GI.Gio.Structs.SrvTarget as Gio.SrvTarget

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Enums as Gio.Enums
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.InetAddress as Gio.InetAddress
import {-# SOURCE #-} qualified GI.Gio.Structs.SrvTarget as Gio.SrvTarget

#endif

-- | Memory-managed wrapper type.
newtype Resolver = Resolver (SP.ManagedPtr Resolver)
    deriving (Eq)

instance SP.ManagedPtrNewtype Resolver where
    toManagedPtr (Resolver p) = p

foreign import ccall "g_resolver_get_type"
    c_g_resolver_get_type :: IO B.Types.GType

instance B.Types.TypedObject Resolver where
    glibType = c_g_resolver_get_type

instance B.Types.GObject Resolver

-- | Type class for types which can be safely cast to t'Resolver', for instance with `toResolver`.
class (SP.GObject o, O.IsDescendantOf Resolver o) => IsResolver o
instance (SP.GObject o, O.IsDescendantOf Resolver o) => IsResolver o

instance O.HasParentTypes Resolver
type instance O.ParentTypes Resolver = '[GObject.Object.Object]

-- | Cast to t'Resolver', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toResolver :: (MIO.MonadIO m, IsResolver o) => o -> m Resolver
toResolver = MIO.liftIO . B.ManagedPtr.unsafeCastTo Resolver

-- | Convert t'Resolver' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Resolver) where
    gvalueGType_ = c_g_resolver_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Resolver)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Resolver)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Resolver ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveResolverMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveResolverMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveResolverMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveResolverMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveResolverMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveResolverMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveResolverMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveResolverMethod "lookupByAddress" o = ResolverLookupByAddressMethodInfo
    ResolveResolverMethod "lookupByAddressAsync" o = ResolverLookupByAddressAsyncMethodInfo
    ResolveResolverMethod "lookupByAddressFinish" o = ResolverLookupByAddressFinishMethodInfo
    ResolveResolverMethod "lookupByName" o = ResolverLookupByNameMethodInfo
    ResolveResolverMethod "lookupByNameAsync" o = ResolverLookupByNameAsyncMethodInfo
    ResolveResolverMethod "lookupByNameFinish" o = ResolverLookupByNameFinishMethodInfo
    ResolveResolverMethod "lookupByNameWithFlags" o = ResolverLookupByNameWithFlagsMethodInfo
    ResolveResolverMethod "lookupByNameWithFlagsAsync" o = ResolverLookupByNameWithFlagsAsyncMethodInfo
    ResolveResolverMethod "lookupByNameWithFlagsFinish" o = ResolverLookupByNameWithFlagsFinishMethodInfo
    ResolveResolverMethod "lookupRecords" o = ResolverLookupRecordsMethodInfo
    ResolveResolverMethod "lookupRecordsAsync" o = ResolverLookupRecordsAsyncMethodInfo
    ResolveResolverMethod "lookupRecordsFinish" o = ResolverLookupRecordsFinishMethodInfo
    ResolveResolverMethod "lookupService" o = ResolverLookupServiceMethodInfo
    ResolveResolverMethod "lookupServiceAsync" o = ResolverLookupServiceAsyncMethodInfo
    ResolveResolverMethod "lookupServiceFinish" o = ResolverLookupServiceFinishMethodInfo
    ResolveResolverMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveResolverMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveResolverMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveResolverMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveResolverMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveResolverMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveResolverMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveResolverMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveResolverMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveResolverMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveResolverMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveResolverMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveResolverMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveResolverMethod "getTimeout" o = ResolverGetTimeoutMethodInfo
    ResolveResolverMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveResolverMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveResolverMethod "setDefault" o = ResolverSetDefaultMethodInfo
    ResolveResolverMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveResolverMethod "setTimeout" o = ResolverSetTimeoutMethodInfo
    ResolveResolverMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveResolverMethod t Resolver, O.OverloadedMethod info Resolver p) => OL.IsLabel t (Resolver -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveResolverMethod t Resolver, O.OverloadedMethod info Resolver p, R.HasField t Resolver p) => R.HasField t Resolver p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveResolverMethod t Resolver, O.OverloadedMethodInfo info Resolver) => OL.IsLabel t (O.MethodProxy info Resolver) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal Resolver::reload
-- | Emitted when the resolver notices that the system resolver
-- configuration has changed.
type ResolverReloadCallback =
    IO ()

type C_ResolverReloadCallback =
    Ptr Resolver ->                         -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ResolverReloadCallback`.
foreign import ccall "wrapper"
    mk_ResolverReloadCallback :: C_ResolverReloadCallback -> IO (FunPtr C_ResolverReloadCallback)

wrap_ResolverReloadCallback :: 
    GObject a => (a -> ResolverReloadCallback) ->
    C_ResolverReloadCallback
wrap_ResolverReloadCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [reload](#signal:reload) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' resolver #reload callback
-- @
-- 
-- 
onResolverReload :: (IsResolver a, MonadIO m) => a -> ((?self :: a) => ResolverReloadCallback) -> m SignalHandlerId
onResolverReload obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ResolverReloadCallback wrapped
    wrapped'' <- mk_ResolverReloadCallback wrapped'
    connectSignalFunPtr obj "reload" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [reload](#signal:reload) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' resolver #reload callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterResolverReload :: (IsResolver a, MonadIO m) => a -> ((?self :: a) => ResolverReloadCallback) -> m SignalHandlerId
afterResolverReload obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ResolverReloadCallback wrapped
    wrapped'' <- mk_ResolverReloadCallback wrapped'
    connectSignalFunPtr obj "reload" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data ResolverReloadSignalInfo
instance SignalInfo ResolverReloadSignalInfo where
    type HaskellCallbackType ResolverReloadSignalInfo = ResolverReloadCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ResolverReloadCallback cb
        cb'' <- mk_ResolverReloadCallback cb'
        connectSignalFunPtr obj "reload" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver::reload"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#g:signal:reload"})

#endif

-- VVV Prop "timeout"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@timeout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' resolver #timeout
-- @
getResolverTimeout :: (MonadIO m, IsResolver o) => o -> m Word32
getResolverTimeout obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "timeout"

-- | Set the value of the “@timeout@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' resolver [ #timeout 'Data.GI.Base.Attributes.:=' value ]
-- @
setResolverTimeout :: (MonadIO m, IsResolver o) => o -> Word32 -> m ()
setResolverTimeout obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyUInt32 obj "timeout" val

-- | Construct a t'GValueConstruct' with valid value for the “@timeout@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructResolverTimeout :: (IsResolver o, MIO.MonadIO m) => Word32 -> m (GValueConstruct o)
constructResolverTimeout val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyUInt32 "timeout" val

#if defined(ENABLE_OVERLOADING)
data ResolverTimeoutPropertyInfo
instance AttrInfo ResolverTimeoutPropertyInfo where
    type AttrAllowedOps ResolverTimeoutPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint ResolverTimeoutPropertyInfo = IsResolver
    type AttrSetTypeConstraint ResolverTimeoutPropertyInfo = (~) Word32
    type AttrTransferTypeConstraint ResolverTimeoutPropertyInfo = (~) Word32
    type AttrTransferType ResolverTimeoutPropertyInfo = Word32
    type AttrGetType ResolverTimeoutPropertyInfo = Word32
    type AttrLabel ResolverTimeoutPropertyInfo = "timeout"
    type AttrOrigin ResolverTimeoutPropertyInfo = Resolver
    attrGet = getResolverTimeout
    attrSet = setResolverTimeout
    attrPut = setResolverTimeout
    attrTransfer _ v = do
        return v
    attrConstruct = constructResolverTimeout
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.timeout"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#g:attr:timeout"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Resolver
type instance O.AttributeList Resolver = ResolverAttributeList
type ResolverAttributeList = ('[ '("timeout", ResolverTimeoutPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
resolverTimeout :: AttrLabelProxy "timeout"
resolverTimeout = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Resolver = ResolverSignalList
type ResolverSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo), '("reload", ResolverReloadSignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Resolver::get_timeout
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_resolver_get_timeout" g_resolver_get_timeout :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    IO Word32

-- | Get the timeout applied to all resolver lookups. See [Resolver:timeout]("GI.Gio.Objects.Resolver#g:attr:timeout").
-- 
-- /Since: 2.78/
resolverGetTimeout ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> m Word32
    -- ^ __Returns:__ the resolver timeout, in milliseconds, or @0@ for no timeout
resolverGetTimeout resolver = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    result <- g_resolver_get_timeout resolver'
    touchManagedPtr resolver
    return result

#if defined(ENABLE_OVERLOADING)
data ResolverGetTimeoutMethodInfo
instance (signature ~ (m Word32), MonadIO m, IsResolver a) => O.OverloadedMethod ResolverGetTimeoutMethodInfo a signature where
    overloadedMethod = resolverGetTimeout

instance O.OverloadedMethodInfo ResolverGetTimeoutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverGetTimeout",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverGetTimeout"
        })


#endif

-- method Resolver::lookup_by_address
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "address"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InetAddress" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the address to reverse-resolve"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_resolver_lookup_by_address" g_resolver_lookup_by_address :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    Ptr Gio.InetAddress.InetAddress ->      -- address : TInterface (Name {namespace = "Gio", name = "InetAddress"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Synchronously reverse-resolves /@address@/ to determine its
-- associated hostname.
-- 
-- If the DNS resolution fails, /@error@/ (if non-'P.Nothing') will be set to
-- a value from t'GI.Gio.Enums.ResolverError'.
-- 
-- If /@cancellable@/ is non-'P.Nothing', it can be used to cancel the
-- operation, in which case /@error@/ (if non-'P.Nothing') will be set to
-- 'GI.Gio.Enums.IOErrorEnumCancelled'.
-- 
-- /Since: 2.22/
resolverLookupByAddress ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.InetAddress.IsInetAddress b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> b
    -- ^ /@address@/: the address to reverse-resolve
    -> Maybe (c)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> m T.Text
    -- ^ __Returns:__ a hostname (either ASCII-only, or in ASCII-encoded
    --     form), or 'P.Nothing' on error. /(Can throw 'Data.GI.Base.GError.GError')/
resolverLookupByAddress resolver address cancellable = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    address' <- unsafeManagedPtrCastPtr address
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_resolver_lookup_by_address resolver' address' maybeCancellable
        checkUnexpectedReturnNULL "resolverLookupByAddress" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr resolver
        touchManagedPtr address
        whenJust cancellable touchManagedPtr
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data ResolverLookupByAddressMethodInfo
instance (signature ~ (b -> Maybe (c) -> m T.Text), MonadIO m, IsResolver a, Gio.InetAddress.IsInetAddress b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod ResolverLookupByAddressMethodInfo a signature where
    overloadedMethod = resolverLookupByAddress

instance O.OverloadedMethodInfo ResolverLookupByAddressMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupByAddress",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupByAddress"
        })


#endif

-- method Resolver::lookup_by_address_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "address"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "InetAddress" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the address to reverse-resolve"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "callback to call after resolution completes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data for @callback" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_resolver_lookup_by_address_async" g_resolver_lookup_by_address_async :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    Ptr Gio.InetAddress.InetAddress ->      -- address : TInterface (Name {namespace = "Gio", name = "InetAddress"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Begins asynchronously reverse-resolving /@address@/ to determine its
-- associated hostname, and eventually calls /@callback@/, which must
-- call 'GI.Gio.Objects.Resolver.resolverLookupByAddressFinish' to get the final result.
-- 
-- /Since: 2.22/
resolverLookupByAddressAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.InetAddress.IsInetAddress b, Gio.Cancellable.IsCancellable c) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> b
    -- ^ /@address@/: the address to reverse-resolve
    -> Maybe (c)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: callback to call after resolution completes
    -> m ()
resolverLookupByAddressAsync resolver address cancellable callback = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    address' <- unsafeManagedPtrCastPtr address
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_resolver_lookup_by_address_async resolver' address' maybeCancellable maybeCallback userData
    touchManagedPtr resolver
    touchManagedPtr address
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data ResolverLookupByAddressAsyncMethodInfo
instance (signature ~ (b -> Maybe (c) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsResolver a, Gio.InetAddress.IsInetAddress b, Gio.Cancellable.IsCancellable c) => O.OverloadedMethod ResolverLookupByAddressAsyncMethodInfo a signature where
    overloadedMethod = resolverLookupByAddressAsync

instance O.OverloadedMethodInfo ResolverLookupByAddressAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupByAddressAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupByAddressAsync"
        })


#endif

-- method Resolver::lookup_by_address_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the result passed to your #GAsyncReadyCallback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_resolver_lookup_by_address_finish" g_resolver_lookup_by_address_finish :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Retrieves the result of a previous call to
-- 'GI.Gio.Objects.Resolver.resolverLookupByAddressAsync'.
-- 
-- If the DNS resolution failed, /@error@/ (if non-'P.Nothing') will be set to
-- a value from t'GI.Gio.Enums.ResolverError'. If the operation was cancelled,
-- /@error@/ will be set to 'GI.Gio.Enums.IOErrorEnumCancelled'.
-- 
-- /Since: 2.22/
resolverLookupByAddressFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> b
    -- ^ /@result@/: the result passed to your t'GI.Gio.Callbacks.AsyncReadyCallback'
    -> m T.Text
    -- ^ __Returns:__ a hostname (either ASCII-only, or in ASCII-encoded
    -- form), or 'P.Nothing' on error. /(Can throw 'Data.GI.Base.GError.GError')/
resolverLookupByAddressFinish resolver result_ = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_resolver_lookup_by_address_finish resolver' result_'
        checkUnexpectedReturnNULL "resolverLookupByAddressFinish" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr resolver
        touchManagedPtr result_
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data ResolverLookupByAddressFinishMethodInfo
instance (signature ~ (b -> m T.Text), MonadIO m, IsResolver a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod ResolverLookupByAddressFinishMethodInfo a signature where
    overloadedMethod = resolverLookupByAddressFinish

instance O.OverloadedMethodInfo ResolverLookupByAddressFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupByAddressFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupByAddressFinish"
        })


#endif

-- method Resolver::lookup_by_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hostname"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the hostname to look up"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList
--                  (TInterface Name { namespace = "Gio" , name = "InetAddress" }))
-- throws : True
-- Skip return : False

foreign import ccall "g_resolver_lookup_by_name" g_resolver_lookup_by_name :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    CString ->                              -- hostname : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr (GList (Ptr Gio.InetAddress.InetAddress)))

-- | Synchronously resolves /@hostname@/ to determine its associated IP
-- address(es). /@hostname@/ may be an ASCII-only or UTF-8 hostname, or
-- the textual form of an IP address (in which case this just becomes
-- a wrapper around 'GI.Gio.Objects.InetAddress.inetAddressNewFromString').
-- 
-- On success, 'GI.Gio.Objects.Resolver.resolverLookupByName' will return a non-empty t'GI.GLib.Structs.List.List' of
-- t'GI.Gio.Objects.InetAddress.InetAddress', sorted in order of preference and guaranteed to not
-- contain duplicates. That is, if using the result to connect to
-- /@hostname@/, you should attempt to connect to the first address
-- first, then the second if the first fails, etc. If you are using
-- the result to listen on a socket, it is appropriate to add each
-- result using e.g. 'GI.Gio.Objects.SocketListener.socketListenerAddAddress'.
-- 
-- If the DNS resolution fails, /@error@/ (if non-'P.Nothing') will be set to a
-- value from t'GI.Gio.Enums.ResolverError' and 'P.Nothing' will be returned.
-- 
-- If /@cancellable@/ is non-'P.Nothing', it can be used to cancel the
-- operation, in which case /@error@/ (if non-'P.Nothing') will be set to
-- 'GI.Gio.Enums.IOErrorEnumCancelled'.
-- 
-- If you are planning to connect to a socket on the resolved IP
-- address, it may be easier to create a t'GI.Gio.Objects.NetworkAddress.NetworkAddress' and use its
-- t'GI.Gio.Interfaces.SocketConnectable.SocketConnectable' interface.
-- 
-- /Since: 2.22/
resolverLookupByName ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> T.Text
    -- ^ /@hostname@/: the hostname to look up
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> m [Gio.InetAddress.InetAddress]
    -- ^ __Returns:__ a non-empty t'GI.GLib.Structs.List.List'
    -- of t'GI.Gio.Objects.InetAddress.InetAddress', or 'P.Nothing' on error. You
    -- must unref each of the addresses and free the list when you are
    -- done with it. (You can use @/g_resolver_free_addresses()/@ to do this.) /(Can throw 'Data.GI.Base.GError.GError')/
resolverLookupByName resolver hostname cancellable = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    hostname' <- textToCString hostname
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_resolver_lookup_by_name resolver' hostname' maybeCancellable
        result' <- unpackGList result
        result'' <- mapM (wrapObject Gio.InetAddress.InetAddress) result'
        g_list_free result
        touchManagedPtr resolver
        whenJust cancellable touchManagedPtr
        freeMem hostname'
        return result''
     ) (do
        freeMem hostname'
     )

#if defined(ENABLE_OVERLOADING)
data ResolverLookupByNameMethodInfo
instance (signature ~ (T.Text -> Maybe (b) -> m [Gio.InetAddress.InetAddress]), MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod ResolverLookupByNameMethodInfo a signature where
    overloadedMethod = resolverLookupByName

instance O.OverloadedMethodInfo ResolverLookupByNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupByName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupByName"
        })


#endif

-- method Resolver::lookup_by_name_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hostname"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the hostname to look up the address of"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "callback to call after resolution completes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data for @callback" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_resolver_lookup_by_name_async" g_resolver_lookup_by_name_async :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    CString ->                              -- hostname : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Begins asynchronously resolving /@hostname@/ to determine its
-- associated IP address(es), and eventually calls /@callback@/, which
-- must call 'GI.Gio.Objects.Resolver.resolverLookupByNameFinish' to get the result.
-- See 'GI.Gio.Objects.Resolver.resolverLookupByName' for more details.
-- 
-- /Since: 2.22/
resolverLookupByNameAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> T.Text
    -- ^ /@hostname@/: the hostname to look up the address of
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: callback to call after resolution completes
    -> m ()
resolverLookupByNameAsync resolver hostname cancellable callback = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    hostname' <- textToCString hostname
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_resolver_lookup_by_name_async resolver' hostname' maybeCancellable maybeCallback userData
    touchManagedPtr resolver
    whenJust cancellable touchManagedPtr
    freeMem hostname'
    return ()

#if defined(ENABLE_OVERLOADING)
data ResolverLookupByNameAsyncMethodInfo
instance (signature ~ (T.Text -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod ResolverLookupByNameAsyncMethodInfo a signature where
    overloadedMethod = resolverLookupByNameAsync

instance O.OverloadedMethodInfo ResolverLookupByNameAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupByNameAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupByNameAsync"
        })


#endif

-- method Resolver::lookup_by_name_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the result passed to your #GAsyncReadyCallback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList
--                  (TInterface Name { namespace = "Gio" , name = "InetAddress" }))
-- throws : True
-- Skip return : False

foreign import ccall "g_resolver_lookup_by_name_finish" g_resolver_lookup_by_name_finish :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr (GList (Ptr Gio.InetAddress.InetAddress)))

-- | Retrieves the result of a call to
-- 'GI.Gio.Objects.Resolver.resolverLookupByNameAsync'.
-- 
-- If the DNS resolution failed, /@error@/ (if non-'P.Nothing') will be set to
-- a value from t'GI.Gio.Enums.ResolverError'. If the operation was cancelled,
-- /@error@/ will be set to 'GI.Gio.Enums.IOErrorEnumCancelled'.
-- 
-- /Since: 2.22/
resolverLookupByNameFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> b
    -- ^ /@result@/: the result passed to your t'GI.Gio.Callbacks.AsyncReadyCallback'
    -> m [Gio.InetAddress.InetAddress]
    -- ^ __Returns:__ a t'GI.GLib.Structs.List.List'
    -- of t'GI.Gio.Objects.InetAddress.InetAddress', or 'P.Nothing' on error. See 'GI.Gio.Objects.Resolver.resolverLookupByName'
    -- for more details. /(Can throw 'Data.GI.Base.GError.GError')/
resolverLookupByNameFinish resolver result_ = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_resolver_lookup_by_name_finish resolver' result_'
        result' <- unpackGList result
        result'' <- mapM (wrapObject Gio.InetAddress.InetAddress) result'
        g_list_free result
        touchManagedPtr resolver
        touchManagedPtr result_
        return result''
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data ResolverLookupByNameFinishMethodInfo
instance (signature ~ (b -> m [Gio.InetAddress.InetAddress]), MonadIO m, IsResolver a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod ResolverLookupByNameFinishMethodInfo a signature where
    overloadedMethod = resolverLookupByNameFinish

instance O.OverloadedMethodInfo ResolverLookupByNameFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupByNameFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupByNameFinish"
        })


#endif

-- method Resolver::lookup_by_name_with_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hostname"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the hostname to look up"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ResolverNameLookupFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "extra #GResolverNameLookupFlags for the lookup"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList
--                  (TInterface Name { namespace = "Gio" , name = "InetAddress" }))
-- throws : True
-- Skip return : False

foreign import ccall "g_resolver_lookup_by_name_with_flags" g_resolver_lookup_by_name_with_flags :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    CString ->                              -- hostname : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "ResolverNameLookupFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr (GList (Ptr Gio.InetAddress.InetAddress)))

-- | This differs from 'GI.Gio.Objects.Resolver.resolverLookupByName' in that you can modify
-- the lookup behavior with /@flags@/. For example this can be used to limit
-- results with 'GI.Gio.Flags.ResolverNameLookupFlagsIpv4Only'.
-- 
-- /Since: 2.60/
resolverLookupByNameWithFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> T.Text
    -- ^ /@hostname@/: the hostname to look up
    -> [Gio.Flags.ResolverNameLookupFlags]
    -- ^ /@flags@/: extra t'GI.Gio.Flags.ResolverNameLookupFlags' for the lookup
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> m [Gio.InetAddress.InetAddress]
    -- ^ __Returns:__ a non-empty t'GI.GLib.Structs.List.List'
    -- of t'GI.Gio.Objects.InetAddress.InetAddress', or 'P.Nothing' on error. You
    -- must unref each of the addresses and free the list when you are
    -- done with it. (You can use @/g_resolver_free_addresses()/@ to do this.) /(Can throw 'Data.GI.Base.GError.GError')/
resolverLookupByNameWithFlags resolver hostname flags cancellable = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    hostname' <- textToCString hostname
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_resolver_lookup_by_name_with_flags resolver' hostname' flags' maybeCancellable
        result' <- unpackGList result
        result'' <- mapM (wrapObject Gio.InetAddress.InetAddress) result'
        g_list_free result
        touchManagedPtr resolver
        whenJust cancellable touchManagedPtr
        freeMem hostname'
        return result''
     ) (do
        freeMem hostname'
     )

#if defined(ENABLE_OVERLOADING)
data ResolverLookupByNameWithFlagsMethodInfo
instance (signature ~ (T.Text -> [Gio.Flags.ResolverNameLookupFlags] -> Maybe (b) -> m [Gio.InetAddress.InetAddress]), MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod ResolverLookupByNameWithFlagsMethodInfo a signature where
    overloadedMethod = resolverLookupByNameWithFlags

instance O.OverloadedMethodInfo ResolverLookupByNameWithFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupByNameWithFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupByNameWithFlags"
        })


#endif

-- method Resolver::lookup_by_name_with_flags_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "hostname"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the hostname to look up the address of"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface
--                 Name { namespace = "Gio" , name = "ResolverNameLookupFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "extra #GResolverNameLookupFlags for the lookup"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "callback to call after resolution completes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data for @callback" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_resolver_lookup_by_name_with_flags_async" g_resolver_lookup_by_name_with_flags_async :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    CString ->                              -- hostname : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "ResolverNameLookupFlags"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Begins asynchronously resolving /@hostname@/ to determine its
-- associated IP address(es), and eventually calls /@callback@/, which
-- must call 'GI.Gio.Objects.Resolver.resolverLookupByNameWithFlagsFinish' to get the result.
-- See 'GI.Gio.Objects.Resolver.resolverLookupByName' for more details.
-- 
-- /Since: 2.60/
resolverLookupByNameWithFlagsAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> T.Text
    -- ^ /@hostname@/: the hostname to look up the address of
    -> [Gio.Flags.ResolverNameLookupFlags]
    -- ^ /@flags@/: extra t'GI.Gio.Flags.ResolverNameLookupFlags' for the lookup
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: callback to call after resolution completes
    -> m ()
resolverLookupByNameWithFlagsAsync resolver hostname flags cancellable callback = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    hostname' <- textToCString hostname
    let flags' = gflagsToWord flags
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_resolver_lookup_by_name_with_flags_async resolver' hostname' flags' maybeCancellable maybeCallback userData
    touchManagedPtr resolver
    whenJust cancellable touchManagedPtr
    freeMem hostname'
    return ()

#if defined(ENABLE_OVERLOADING)
data ResolverLookupByNameWithFlagsAsyncMethodInfo
instance (signature ~ (T.Text -> [Gio.Flags.ResolverNameLookupFlags] -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod ResolverLookupByNameWithFlagsAsyncMethodInfo a signature where
    overloadedMethod = resolverLookupByNameWithFlagsAsync

instance O.OverloadedMethodInfo ResolverLookupByNameWithFlagsAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupByNameWithFlagsAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupByNameWithFlagsAsync"
        })


#endif

-- method Resolver::lookup_by_name_with_flags_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the result passed to your #GAsyncReadyCallback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList
--                  (TInterface Name { namespace = "Gio" , name = "InetAddress" }))
-- throws : True
-- Skip return : False

foreign import ccall "g_resolver_lookup_by_name_with_flags_finish" g_resolver_lookup_by_name_with_flags_finish :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr (GList (Ptr Gio.InetAddress.InetAddress)))

-- | Retrieves the result of a call to
-- 'GI.Gio.Objects.Resolver.resolverLookupByNameWithFlagsAsync'.
-- 
-- If the DNS resolution failed, /@error@/ (if non-'P.Nothing') will be set to
-- a value from t'GI.Gio.Enums.ResolverError'. If the operation was cancelled,
-- /@error@/ will be set to 'GI.Gio.Enums.IOErrorEnumCancelled'.
-- 
-- /Since: 2.60/
resolverLookupByNameWithFlagsFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> b
    -- ^ /@result@/: the result passed to your t'GI.Gio.Callbacks.AsyncReadyCallback'
    -> m [Gio.InetAddress.InetAddress]
    -- ^ __Returns:__ a t'GI.GLib.Structs.List.List'
    -- of t'GI.Gio.Objects.InetAddress.InetAddress', or 'P.Nothing' on error. See 'GI.Gio.Objects.Resolver.resolverLookupByName'
    -- for more details. /(Can throw 'Data.GI.Base.GError.GError')/
resolverLookupByNameWithFlagsFinish resolver result_ = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_resolver_lookup_by_name_with_flags_finish resolver' result_'
        result' <- unpackGList result
        result'' <- mapM (wrapObject Gio.InetAddress.InetAddress) result'
        g_list_free result
        touchManagedPtr resolver
        touchManagedPtr result_
        return result''
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data ResolverLookupByNameWithFlagsFinishMethodInfo
instance (signature ~ (b -> m [Gio.InetAddress.InetAddress]), MonadIO m, IsResolver a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod ResolverLookupByNameWithFlagsFinishMethodInfo a signature where
    overloadedMethod = resolverLookupByNameWithFlagsFinish

instance O.OverloadedMethodInfo ResolverLookupByNameWithFlagsFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupByNameWithFlagsFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupByNameWithFlagsFinish"
        })


#endif

-- method Resolver::lookup_records
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "rrname"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the DNS name to look up the record for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "record_type"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ResolverRecordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the type of DNS record to look up"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TGList TVariant)
-- throws : True
-- Skip return : False

foreign import ccall "g_resolver_lookup_records" g_resolver_lookup_records :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    CString ->                              -- rrname : TBasicType TUTF8
    CUInt ->                                -- record_type : TInterface (Name {namespace = "Gio", name = "ResolverRecordType"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr (GList (Ptr GVariant)))

-- | Synchronously performs a DNS record lookup for the given /@rrname@/ and returns
-- a list of records as t'GVariant' tuples. See t'GI.Gio.Enums.ResolverRecordType' for
-- information on what the records contain for each /@recordType@/.
-- 
-- If the DNS resolution fails, /@error@/ (if non-'P.Nothing') will be set to
-- a value from t'GI.Gio.Enums.ResolverError' and 'P.Nothing' will be returned.
-- 
-- If /@cancellable@/ is non-'P.Nothing', it can be used to cancel the
-- operation, in which case /@error@/ (if non-'P.Nothing') will be set to
-- 'GI.Gio.Enums.IOErrorEnumCancelled'.
-- 
-- /Since: 2.34/
resolverLookupRecords ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> T.Text
    -- ^ /@rrname@/: the DNS name to look up the record for
    -> Gio.Enums.ResolverRecordType
    -- ^ /@recordType@/: the type of DNS record to look up
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> m [GVariant]
    -- ^ __Returns:__ a non-empty t'GI.GLib.Structs.List.List' of
    -- t'GVariant', or 'P.Nothing' on error. You must free each of the records and the list
    -- when you are done with it. (You can use @/g_list_free_full()/@ with
    -- 'GI.GLib.Structs.Variant.variantUnref' to do this.) /(Can throw 'Data.GI.Base.GError.GError')/
resolverLookupRecords resolver rrname recordType cancellable = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    rrname' <- textToCString rrname
    let recordType' = (fromIntegral . fromEnum) recordType
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_resolver_lookup_records resolver' rrname' recordType' maybeCancellable
        result' <- unpackGList result
        result'' <- mapM B.GVariant.wrapGVariantPtr result'
        g_list_free result
        touchManagedPtr resolver
        whenJust cancellable touchManagedPtr
        freeMem rrname'
        return result''
     ) (do
        freeMem rrname'
     )

#if defined(ENABLE_OVERLOADING)
data ResolverLookupRecordsMethodInfo
instance (signature ~ (T.Text -> Gio.Enums.ResolverRecordType -> Maybe (b) -> m [GVariant]), MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod ResolverLookupRecordsMethodInfo a signature where
    overloadedMethod = resolverLookupRecords

instance O.OverloadedMethodInfo ResolverLookupRecordsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupRecords",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupRecords"
        })


#endif

-- method Resolver::lookup_records_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "rrname"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the DNS name to look up the record for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "record_type"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ResolverRecordType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the type of DNS record to look up"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "callback to call after resolution completes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 5
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data for @callback" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_resolver_lookup_records_async" g_resolver_lookup_records_async :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    CString ->                              -- rrname : TBasicType TUTF8
    CUInt ->                                -- record_type : TInterface (Name {namespace = "Gio", name = "ResolverRecordType"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Begins asynchronously performing a DNS lookup for the given
-- /@rrname@/, and eventually calls /@callback@/, which must call
-- 'GI.Gio.Objects.Resolver.resolverLookupRecordsFinish' to get the final result. See
-- 'GI.Gio.Objects.Resolver.resolverLookupRecords' for more details.
-- 
-- /Since: 2.34/
resolverLookupRecordsAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> T.Text
    -- ^ /@rrname@/: the DNS name to look up the record for
    -> Gio.Enums.ResolverRecordType
    -- ^ /@recordType@/: the type of DNS record to look up
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: callback to call after resolution completes
    -> m ()
resolverLookupRecordsAsync resolver rrname recordType cancellable callback = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    rrname' <- textToCString rrname
    let recordType' = (fromIntegral . fromEnum) recordType
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_resolver_lookup_records_async resolver' rrname' recordType' maybeCancellable maybeCallback userData
    touchManagedPtr resolver
    whenJust cancellable touchManagedPtr
    freeMem rrname'
    return ()

#if defined(ENABLE_OVERLOADING)
data ResolverLookupRecordsAsyncMethodInfo
instance (signature ~ (T.Text -> Gio.Enums.ResolverRecordType -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod ResolverLookupRecordsAsyncMethodInfo a signature where
    overloadedMethod = resolverLookupRecordsAsync

instance O.OverloadedMethodInfo ResolverLookupRecordsAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupRecordsAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupRecordsAsync"
        })


#endif

-- method Resolver::lookup_records_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the result passed to your #GAsyncReadyCallback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TGList TVariant)
-- throws : True
-- Skip return : False

foreign import ccall "g_resolver_lookup_records_finish" g_resolver_lookup_records_finish :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr (GList (Ptr GVariant)))

-- | Retrieves the result of a previous call to
-- 'GI.Gio.Objects.Resolver.resolverLookupRecordsAsync'. Returns a non-empty list of records as
-- t'GVariant' tuples. See t'GI.Gio.Enums.ResolverRecordType' for information on what the
-- records contain.
-- 
-- If the DNS resolution failed, /@error@/ (if non-'P.Nothing') will be set to
-- a value from t'GI.Gio.Enums.ResolverError'. If the operation was cancelled,
-- /@error@/ will be set to 'GI.Gio.Enums.IOErrorEnumCancelled'.
-- 
-- /Since: 2.34/
resolverLookupRecordsFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> b
    -- ^ /@result@/: the result passed to your t'GI.Gio.Callbacks.AsyncReadyCallback'
    -> m [GVariant]
    -- ^ __Returns:__ a non-empty t'GI.GLib.Structs.List.List' of
    -- t'GVariant', or 'P.Nothing' on error. You must free each of the records and the list
    -- when you are done with it. (You can use @/g_list_free_full()/@ with
    -- 'GI.GLib.Structs.Variant.variantUnref' to do this.) /(Can throw 'Data.GI.Base.GError.GError')/
resolverLookupRecordsFinish resolver result_ = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_resolver_lookup_records_finish resolver' result_'
        result' <- unpackGList result
        result'' <- mapM B.GVariant.wrapGVariantPtr result'
        g_list_free result
        touchManagedPtr resolver
        touchManagedPtr result_
        return result''
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data ResolverLookupRecordsFinishMethodInfo
instance (signature ~ (b -> m [GVariant]), MonadIO m, IsResolver a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod ResolverLookupRecordsFinishMethodInfo a signature where
    overloadedMethod = resolverLookupRecordsFinish

instance O.OverloadedMethodInfo ResolverLookupRecordsFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupRecordsFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupRecordsFinish"
        })


#endif

-- method Resolver::lookup_service
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "service"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the service type to look up (eg, \"ldap\")"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "protocol"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the networking protocol to use for @service (eg, \"tcp\")"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "domain"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the DNS domain to look up the service in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList
--                  (TInterface Name { namespace = "Gio" , name = "SrvTarget" }))
-- throws : True
-- Skip return : False

foreign import ccall "g_resolver_lookup_service" g_resolver_lookup_service :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    CString ->                              -- service : TBasicType TUTF8
    CString ->                              -- protocol : TBasicType TUTF8
    CString ->                              -- domain : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr (GList (Ptr Gio.SrvTarget.SrvTarget)))

-- | Synchronously performs a DNS SRV lookup for the given /@service@/ and
-- /@protocol@/ in the given /@domain@/ and returns an array of t'GI.Gio.Structs.SrvTarget.SrvTarget'.
-- /@domain@/ may be an ASCII-only or UTF-8 hostname. Note also that the
-- /@service@/ and /@protocol@/ arguments do not include the leading underscore
-- that appears in the actual DNS entry.
-- 
-- On success, 'GI.Gio.Objects.Resolver.resolverLookupService' will return a non-empty t'GI.GLib.Structs.List.List' of
-- t'GI.Gio.Structs.SrvTarget.SrvTarget', sorted in order of preference. (That is, you should
-- attempt to connect to the first target first, then the second if
-- the first fails, etc.)
-- 
-- If the DNS resolution fails, /@error@/ (if non-'P.Nothing') will be set to
-- a value from t'GI.Gio.Enums.ResolverError' and 'P.Nothing' will be returned.
-- 
-- If /@cancellable@/ is non-'P.Nothing', it can be used to cancel the
-- operation, in which case /@error@/ (if non-'P.Nothing') will be set to
-- 'GI.Gio.Enums.IOErrorEnumCancelled'.
-- 
-- If you are planning to connect to the service, it is usually easier
-- to create a t'GI.Gio.Objects.NetworkService.NetworkService' and use its t'GI.Gio.Interfaces.SocketConnectable.SocketConnectable'
-- interface.
-- 
-- /Since: 2.22/
resolverLookupService ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> T.Text
    -- ^ /@service@/: the service type to look up (eg, \"ldap\")
    -> T.Text
    -- ^ /@protocol@/: the networking protocol to use for /@service@/ (eg, \"tcp\")
    -> T.Text
    -- ^ /@domain@/: the DNS domain to look up the service in
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> m [Gio.SrvTarget.SrvTarget]
    -- ^ __Returns:__ a non-empty t'GI.GLib.Structs.List.List' of
    -- t'GI.Gio.Structs.SrvTarget.SrvTarget', or 'P.Nothing' on error. You must free each of the targets and the
    -- list when you are done with it. (You can use @/g_resolver_free_targets()/@ to do
    -- this.) /(Can throw 'Data.GI.Base.GError.GError')/
resolverLookupService resolver service protocol domain cancellable = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    service' <- textToCString service
    protocol' <- textToCString protocol
    domain' <- textToCString domain
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        result <- propagateGError $ g_resolver_lookup_service resolver' service' protocol' domain' maybeCancellable
        result' <- unpackGList result
        result'' <- mapM (wrapBoxed Gio.SrvTarget.SrvTarget) result'
        g_list_free result
        touchManagedPtr resolver
        whenJust cancellable touchManagedPtr
        freeMem service'
        freeMem protocol'
        freeMem domain'
        return result''
     ) (do
        freeMem service'
        freeMem protocol'
        freeMem domain'
     )

#if defined(ENABLE_OVERLOADING)
data ResolverLookupServiceMethodInfo
instance (signature ~ (T.Text -> T.Text -> T.Text -> Maybe (b) -> m [Gio.SrvTarget.SrvTarget]), MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod ResolverLookupServiceMethodInfo a signature where
    overloadedMethod = resolverLookupService

instance O.OverloadedMethodInfo ResolverLookupServiceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupService",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupService"
        })


#endif

-- method Resolver::lookup_service_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "service"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the service type to look up (eg, \"ldap\")"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "protocol"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the networking protocol to use for @service (eg, \"tcp\")"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "domain"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the DNS domain to look up the service in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "callback to call after resolution completes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 6
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data for @callback" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_resolver_lookup_service_async" g_resolver_lookup_service_async :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    CString ->                              -- service : TBasicType TUTF8
    CString ->                              -- protocol : TBasicType TUTF8
    CString ->                              -- domain : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Begins asynchronously performing a DNS SRV lookup for the given
-- /@service@/ and /@protocol@/ in the given /@domain@/, and eventually calls
-- /@callback@/, which must call 'GI.Gio.Objects.Resolver.resolverLookupServiceFinish' to
-- get the final result. See 'GI.Gio.Objects.Resolver.resolverLookupService' for more
-- details.
-- 
-- /Since: 2.22/
resolverLookupServiceAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> T.Text
    -- ^ /@service@/: the service type to look up (eg, \"ldap\")
    -> T.Text
    -- ^ /@protocol@/: the networking protocol to use for /@service@/ (eg, \"tcp\")
    -> T.Text
    -- ^ /@domain@/: the DNS domain to look up the service in
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: callback to call after resolution completes
    -> m ()
resolverLookupServiceAsync resolver service protocol domain cancellable callback = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    service' <- textToCString service
    protocol' <- textToCString protocol
    domain' <- textToCString domain
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_resolver_lookup_service_async resolver' service' protocol' domain' maybeCancellable maybeCallback userData
    touchManagedPtr resolver
    whenJust cancellable touchManagedPtr
    freeMem service'
    freeMem protocol'
    freeMem domain'
    return ()

#if defined(ENABLE_OVERLOADING)
data ResolverLookupServiceAsyncMethodInfo
instance (signature ~ (T.Text -> T.Text -> T.Text -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsResolver a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod ResolverLookupServiceAsyncMethodInfo a signature where
    overloadedMethod = resolverLookupServiceAsync

instance O.OverloadedMethodInfo ResolverLookupServiceAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupServiceAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupServiceAsync"
        })


#endif

-- method Resolver::lookup_service_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the result passed to your #GAsyncReadyCallback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGList
--                  (TInterface Name { namespace = "Gio" , name = "SrvTarget" }))
-- throws : True
-- Skip return : False

foreign import ccall "g_resolver_lookup_service_finish" g_resolver_lookup_service_finish :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr (GList (Ptr Gio.SrvTarget.SrvTarget)))

-- | Retrieves the result of a previous call to
-- 'GI.Gio.Objects.Resolver.resolverLookupServiceAsync'.
-- 
-- If the DNS resolution failed, /@error@/ (if non-'P.Nothing') will be set to
-- a value from t'GI.Gio.Enums.ResolverError'. If the operation was cancelled,
-- /@error@/ will be set to 'GI.Gio.Enums.IOErrorEnumCancelled'.
-- 
-- /Since: 2.22/
resolverLookupServiceFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> b
    -- ^ /@result@/: the result passed to your t'GI.Gio.Callbacks.AsyncReadyCallback'
    -> m [Gio.SrvTarget.SrvTarget]
    -- ^ __Returns:__ a non-empty t'GI.GLib.Structs.List.List' of
    -- t'GI.Gio.Structs.SrvTarget.SrvTarget', or 'P.Nothing' on error. See 'GI.Gio.Objects.Resolver.resolverLookupService' for more
    -- details. /(Can throw 'Data.GI.Base.GError.GError')/
resolverLookupServiceFinish resolver result_ = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        result <- propagateGError $ g_resolver_lookup_service_finish resolver' result_'
        result' <- unpackGList result
        result'' <- mapM (wrapBoxed Gio.SrvTarget.SrvTarget) result'
        g_list_free result
        touchManagedPtr resolver
        touchManagedPtr result_
        return result''
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data ResolverLookupServiceFinishMethodInfo
instance (signature ~ (b -> m [Gio.SrvTarget.SrvTarget]), MonadIO m, IsResolver a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod ResolverLookupServiceFinishMethodInfo a signature where
    overloadedMethod = resolverLookupServiceFinish

instance O.OverloadedMethodInfo ResolverLookupServiceFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverLookupServiceFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverLookupServiceFinish"
        })


#endif

-- method Resolver::set_default
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new default #GResolver"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_resolver_set_default" g_resolver_set_default :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    IO ()

-- | Sets /@resolver@/ to be the application\'s default resolver (reffing
-- /@resolver@/, and unreffing the previous default resolver, if any).
-- Future calls to 'GI.Gio.Objects.Resolver.resolverGetDefault' will return this resolver.
-- 
-- This can be used if an application wants to perform any sort of DNS
-- caching or \"pinning\"; it can implement its own t'GI.Gio.Objects.Resolver.Resolver' that
-- calls the original default resolver for DNS operations, and
-- implements its own cache policies on top of that, and then set
-- itself as the default resolver for all later code to use.
-- 
-- /Since: 2.22/
resolverSetDefault ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a) =>
    a
    -- ^ /@resolver@/: the new default t'GI.Gio.Objects.Resolver.Resolver'
    -> m ()
resolverSetDefault resolver = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    g_resolver_set_default resolver'
    touchManagedPtr resolver
    return ()

#if defined(ENABLE_OVERLOADING)
data ResolverSetDefaultMethodInfo
instance (signature ~ (m ()), MonadIO m, IsResolver a) => O.OverloadedMethod ResolverSetDefaultMethodInfo a signature where
    overloadedMethod = resolverSetDefault

instance O.OverloadedMethodInfo ResolverSetDefaultMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverSetDefault",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverSetDefault"
        })


#endif

-- method Resolver::set_timeout
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "resolver"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Resolver" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GResolver" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "timeout_ms"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "timeout in milliseconds, or `0` for no timeouts"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_resolver_set_timeout" g_resolver_set_timeout :: 
    Ptr Resolver ->                         -- resolver : TInterface (Name {namespace = "Gio", name = "Resolver"})
    Word32 ->                               -- timeout_ms : TBasicType TUInt
    IO ()

-- | Set the timeout applied to all resolver lookups. See [Resolver:timeout]("GI.Gio.Objects.Resolver#g:attr:timeout").
-- 
-- /Since: 2.78/
resolverSetTimeout ::
    (B.CallStack.HasCallStack, MonadIO m, IsResolver a) =>
    a
    -- ^ /@resolver@/: a t'GI.Gio.Objects.Resolver.Resolver'
    -> Word32
    -- ^ /@timeoutMs@/: timeout in milliseconds, or @0@ for no timeouts
    -> m ()
resolverSetTimeout resolver timeoutMs = liftIO $ do
    resolver' <- unsafeManagedPtrCastPtr resolver
    g_resolver_set_timeout resolver' timeoutMs
    touchManagedPtr resolver
    return ()

#if defined(ENABLE_OVERLOADING)
data ResolverSetTimeoutMethodInfo
instance (signature ~ (Word32 -> m ()), MonadIO m, IsResolver a) => O.OverloadedMethod ResolverSetTimeoutMethodInfo a signature where
    overloadedMethod = resolverSetTimeout

instance O.OverloadedMethodInfo ResolverSetTimeoutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Resolver.resolverSetTimeout",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Resolver.html#v:resolverSetTimeout"
        })


#endif

-- method Resolver::get_default
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Resolver" })
-- throws : False
-- Skip return : False

foreign import ccall "g_resolver_get_default" g_resolver_get_default :: 
    IO (Ptr Resolver)

-- | Gets the default t'GI.Gio.Objects.Resolver.Resolver'. You should unref it when you are done
-- with it. t'GI.Gio.Objects.Resolver.Resolver' may use its reference count as a hint about how
-- many threads it should allocate for concurrent DNS resolutions.
-- 
-- /Since: 2.22/
resolverGetDefault ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Resolver
    -- ^ __Returns:__ the default t'GI.Gio.Objects.Resolver.Resolver'.
resolverGetDefault  = liftIO $ do
    result <- g_resolver_get_default
    checkUnexpectedReturnNULL "resolverGetDefault" result
    result' <- (wrapObject Resolver) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif


