// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
/*
 * Implementation of the https://nvlpubs.nist.gov/nistpubs/SpecialPublications/NIST.SP.800-108r1.pdf
 * Key Derivation in Counter Mode Using Pseudorandom Functions. This
 * implementation mirrors the Dafny one: https://github.com/aws/aws-cryptographic-material-providers-library/blob/main/AwsCryptographyPrimitives/src/KDF/KdfCtr.dfy
 */
import { createHash, createHmac } from 'crypto';
import { needs } from '@aws-crypto/material-management';
import { uInt32BE } from '@aws-crypto/serialize';
const SEPARATION_INDICATOR = Buffer.from([0x00]);
const COUNTER_START_VALUE = 1;
export const INT32_MAX_LIMIT = 2147483647;
const SUPPORTED_IKM_LENGTHS = [32, 48, 66];
const SUPPORTED_NONCE_LENGTHS = [16, 32];
const SUPPORTED_DERIVED_KEY_LENGTHS = [32, 64];
const SUPPORTED_DIGEST_ALGORITHMS = ['sha256', 'sha384'];
export function kdfCounterMode({ digestAlgorithm, ikm, nonce, purpose, expectedLength, }) {
    /* Precondition: the ikm must be 32, 48, 66 bytes long */
    needs(SUPPORTED_IKM_LENGTHS.includes(ikm.length), `Unsupported IKM length ${ikm.length}`);
    /* Precondition: the nonce is required */
    needs(nonce, 'The nonce must be provided');
    /* Precondition: the nonce must be 16, 32 bytes long */
    needs(SUPPORTED_NONCE_LENGTHS.includes(nonce.length), `Unsupported nonce length ${nonce.length}`);
    /* Precondition: the expected length must be 32, 64 bytes */
    /* Precondition: the expected length * 8 must be under the max 32-bit signed integer */
    needs(SUPPORTED_DERIVED_KEY_LENGTHS.includes(expectedLength) &&
        expectedLength * 8 < INT32_MAX_LIMIT &&
        expectedLength * 8 > 0, `Unsupported requested length ${expectedLength}`);
    const label = purpose || Buffer.alloc(0);
    const info = nonce;
    const internalLength = 8 + SEPARATION_INDICATOR.length;
    /* Precondition: the input length must be under the max 32-bit signed integer */
    needs(internalLength + label.length + info.length < INT32_MAX_LIMIT, `Input Length ${internalLength + label.length + info.length} must be under ${INT32_MAX_LIMIT} bytes`);
    const lengthBits = Buffer.from(uInt32BE(expectedLength * 8));
    const explicitInfo = Buffer.concat([
        label,
        SEPARATION_INDICATOR,
        info,
        lengthBits,
    ]);
    return rawDerive(ikm, explicitInfo, expectedLength, digestAlgorithm);
}
export function rawDerive(ikm, explicitInfo, length, 
// omit offset as a parameter because it is unused, causing compile errors due
// to configured project settings
digestAlgorithm) {
    const h = createHash(digestAlgorithm).digest().length;
    /* Precondition: expected length must be positive */
    needs(length > 0, `Requested length ${length} must be positive`);
    /* Precondition: length of explicit info + 4 bytes should be under the max 32-bit signed integer */
    needs(4 + explicitInfo.length < INT32_MAX_LIMIT, `Explicit info length ${explicitInfo.length} must be under ${INT32_MAX_LIMIT - 4} bytes`);
    /* Precondition: the digest algorithm should be sha256 */
    needs(SUPPORTED_DIGEST_ALGORITHMS.includes(digestAlgorithm), `Unsupported digest algorithm ${digestAlgorithm}`);
    /* Precondition: the expected length + digest hash length should be under the max 32-bit signed integer - 1 */
    needs(length + h < INT32_MAX_LIMIT - 1, `The combined requested and digest hash length ${length + h} must be under ${INT32_MAX_LIMIT - 1} bytes`);
    // number of iterations calculated in accordance with SP800-108
    const iterations = Math.floor((length + h - 1) / h);
    let buffer = Buffer.alloc(0);
    let i = Buffer.from(uInt32BE(COUNTER_START_VALUE));
    for (let iteration = 1; iteration <= iterations; iteration++) {
        const digest = createHmac(digestAlgorithm, ikm)
            .update(i)
            .update(explicitInfo)
            .digest();
        buffer = Buffer.concat([buffer, digest]);
        i = increment(i);
    }
    needs(buffer.length >= length, 'Failed to derive key of requested length');
    return buffer.subarray(0, length);
}
export function increment(x) {
    /* Precondition: buffer length must be 4 bytes */
    needs(x.length === 4, `Buffer length ${x.length} must be 4 bytes`);
    let output;
    if (x[3] < 255) {
        output = Buffer.from([x[0], x[1], x[2], x[3] + 1]);
    }
    else if (x[2] < 255) {
        output = Buffer.from([x[0], x[1], x[2] + 1, 0]);
    }
    else if (x[1] < 255) {
        output = Buffer.from([x[0], x[1] + 1, 0, 0]);
    }
    else if (x[0] < 255) {
        output = Buffer.from([x[0] + 1, 0, 0, 0]);
    }
    else {
        throw new Error('Unable to derive key material; may have exceeded limit.');
    }
    /* Postcondition: incremented buffer length must be 4 bytes */
    needs(output.length === 4, `Incremented buffer length ${output.length} must be 4 bytes`);
    return output;
}
//# sourceMappingURL=data:application/json;base64,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