/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.common.security.scram.internals;

import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import javax.security.auth.callback.Callback;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.callback.NameCallback;
import javax.security.sasl.SaslException;
import javax.security.sasl.SaslServer;
import javax.security.sasl.SaslServerFactory;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.errors.IllegalSaslStateException;
import org.apache.kafka.common.errors.SaslAuthenticationException;
import org.apache.kafka.common.security.scram.ScramCredential;
import org.apache.kafka.common.security.scram.ScramCredentialCallback;
import org.apache.kafka.common.security.scram.internals.ScramExtensions;
import org.apache.kafka.common.security.scram.internals.ScramFormatter;
import org.apache.kafka.common.security.scram.internals.ScramMechanism;
import org.apache.kafka.common.security.scram.internals.ScramMessages;
import org.apache.kafka.common.security.token.delegation.internals.DelegationTokenCredentialCallback;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ScramSaslServer
implements SaslServer {
    private static final Logger log = LoggerFactory.getLogger(ScramSaslServer.class);
    private static final Set<String> SUPPORTED_EXTENSIONS = Utils.mkSet("tokenauth");
    private final ScramMechanism mechanism;
    private final ScramFormatter formatter;
    private final CallbackHandler callbackHandler;
    private State state;
    private ScramMessages.ClientFirstMessage clientFirstMessage;
    private ScramMessages.ServerFirstMessage serverFirstMessage;
    private ScramExtensions scramExtensions;
    private ScramCredential scramCredential;
    private String authorizationId;
    private Long tokenExpiryTimestamp;

    public ScramSaslServer(ScramMechanism mechanism, Map<String, ?> props, CallbackHandler callbackHandler) throws NoSuchAlgorithmException {
        this.mechanism = mechanism;
        this.formatter = new ScramFormatter(mechanism);
        this.callbackHandler = callbackHandler;
        this.setState(State.RECEIVE_CLIENT_FIRST_MESSAGE);
    }

    @Override
    public byte[] evaluateResponse(byte[] response) throws SaslException, SaslAuthenticationException {
        try {
            switch (this.state) {
                case RECEIVE_CLIENT_FIRST_MESSAGE: {
                    this.clientFirstMessage = new ScramMessages.ClientFirstMessage(response);
                    this.scramExtensions = this.clientFirstMessage.extensions();
                    if (!SUPPORTED_EXTENSIONS.containsAll(this.scramExtensions.map().keySet())) {
                        log.debug("Unsupported extensions will be ignored, supported {}, provided {}", SUPPORTED_EXTENSIONS, this.scramExtensions.map().keySet());
                    }
                    String serverNonce = this.formatter.secureRandomString();
                    try {
                        ScramCredentialCallback credentialCallback;
                        String saslName = this.clientFirstMessage.saslName();
                        String username = ScramFormatter.username(saslName);
                        NameCallback nameCallback = new NameCallback("username", username);
                        if (this.scramExtensions.tokenAuthenticated()) {
                            DelegationTokenCredentialCallback tokenCallback = new DelegationTokenCredentialCallback();
                            credentialCallback = tokenCallback;
                            this.callbackHandler.handle(new Callback[]{nameCallback, tokenCallback});
                            if (tokenCallback.tokenOwner() == null) {
                                throw new SaslException("Token Authentication failed: Invalid tokenId : " + username);
                            }
                            this.authorizationId = tokenCallback.tokenOwner();
                            this.tokenExpiryTimestamp = tokenCallback.tokenExpiryTimestamp();
                        } else {
                            credentialCallback = new ScramCredentialCallback();
                            this.callbackHandler.handle(new Callback[]{nameCallback, credentialCallback});
                            this.authorizationId = username;
                            this.tokenExpiryTimestamp = null;
                        }
                        this.scramCredential = credentialCallback.scramCredential();
                        if (this.scramCredential == null) {
                            throw new SaslException("Authentication failed: Invalid user credentials");
                        }
                        String authorizationIdFromClient = this.clientFirstMessage.authorizationId();
                        if (!authorizationIdFromClient.isEmpty() && !authorizationIdFromClient.equals(username)) {
                            throw new SaslAuthenticationException("Authentication failed: Client requested an authorization id that is different from username");
                        }
                        if (this.scramCredential.iterations() < this.mechanism.minIterations()) {
                            throw new SaslException("Iterations " + this.scramCredential.iterations() + " is less than the minimum " + this.mechanism.minIterations() + " for " + (Object)((Object)this.mechanism));
                        }
                        this.serverFirstMessage = new ScramMessages.ServerFirstMessage(this.clientFirstMessage.nonce(), serverNonce, this.scramCredential.salt(), this.scramCredential.iterations());
                        this.setState(State.RECEIVE_CLIENT_FINAL_MESSAGE);
                        return this.serverFirstMessage.toBytes();
                    }
                    catch (SaslException | AuthenticationException e) {
                        throw e;
                    }
                    catch (Throwable e) {
                        throw new SaslException("Authentication failed: Credentials could not be obtained", e);
                    }
                }
                case RECEIVE_CLIENT_FINAL_MESSAGE: {
                    try {
                        ScramMessages.ClientFinalMessage clientFinalMessage = new ScramMessages.ClientFinalMessage(response);
                        if (!clientFinalMessage.nonce().endsWith(this.serverFirstMessage.nonce())) {
                            throw new SaslException("Invalid client nonce in the final client message.");
                        }
                        this.verifyClientProof(clientFinalMessage);
                        byte[] serverKey = this.scramCredential.serverKey();
                        byte[] serverSignature = this.formatter.serverSignature(serverKey, this.clientFirstMessage, this.serverFirstMessage, clientFinalMessage);
                        ScramMessages.ServerFinalMessage serverFinalMessage = new ScramMessages.ServerFinalMessage(null, serverSignature);
                        this.clearCredentials();
                        this.setState(State.COMPLETE);
                        return serverFinalMessage.toBytes();
                    }
                    catch (InvalidKeyException e) {
                        throw new SaslException("Authentication failed: Invalid client final message", e);
                    }
                }
            }
            throw new IllegalSaslStateException("Unexpected challenge in Sasl server state " + (Object)((Object)this.state));
        }
        catch (SaslException | AuthenticationException e) {
            this.clearCredentials();
            this.setState(State.FAILED);
            throw e;
        }
    }

    @Override
    public String getAuthorizationID() {
        if (!this.isComplete()) {
            throw new IllegalStateException("Authentication exchange has not completed");
        }
        return this.authorizationId;
    }

    @Override
    public String getMechanismName() {
        return this.mechanism.mechanismName();
    }

    @Override
    public Object getNegotiatedProperty(String propName) {
        if (!this.isComplete()) {
            throw new IllegalStateException("Authentication exchange has not completed");
        }
        if ("CREDENTIAL.LIFETIME.MS".equals(propName)) {
            return this.tokenExpiryTimestamp;
        }
        if (SUPPORTED_EXTENSIONS.contains(propName)) {
            return this.scramExtensions.map().get(propName);
        }
        return null;
    }

    @Override
    public boolean isComplete() {
        return this.state == State.COMPLETE;
    }

    @Override
    public byte[] unwrap(byte[] incoming, int offset, int len) {
        if (!this.isComplete()) {
            throw new IllegalStateException("Authentication exchange has not completed");
        }
        return Arrays.copyOfRange(incoming, offset, offset + len);
    }

    @Override
    public byte[] wrap(byte[] outgoing, int offset, int len) {
        if (!this.isComplete()) {
            throw new IllegalStateException("Authentication exchange has not completed");
        }
        return Arrays.copyOfRange(outgoing, offset, offset + len);
    }

    @Override
    public void dispose() {
    }

    private void setState(State state) {
        log.debug("Setting SASL/{} server state to {}", (Object)this.mechanism, (Object)state);
        this.state = state;
    }

    void verifyClientProof(ScramMessages.ClientFinalMessage clientFinalMessage) throws SaslException {
        try {
            byte[] expectedStoredKey = this.scramCredential.storedKey();
            byte[] clientSignature = this.formatter.clientSignature(expectedStoredKey, this.clientFirstMessage, this.serverFirstMessage, clientFinalMessage);
            byte[] computedStoredKey = this.formatter.storedKey(clientSignature, clientFinalMessage.proof());
            if (!MessageDigest.isEqual(computedStoredKey, expectedStoredKey)) {
                throw new SaslException("Invalid client credentials");
            }
        }
        catch (InvalidKeyException e) {
            throw new SaslException("Sasl client verification failed", e);
        }
    }

    private void clearCredentials() {
        this.scramCredential = null;
        this.clientFirstMessage = null;
        this.serverFirstMessage = null;
    }

    public static class ScramSaslServerFactory
    implements SaslServerFactory {
        @Override
        public SaslServer createSaslServer(String mechanism, String protocol, String serverName, Map<String, ?> props, CallbackHandler cbh) throws SaslException {
            if (!ScramMechanism.isScram(mechanism)) {
                throw new SaslException(String.format("Requested mechanism '%s' is not supported. Supported mechanisms are '%s'.", mechanism, ScramMechanism.mechanismNames()));
            }
            try {
                return new ScramSaslServer(ScramMechanism.forMechanismName(mechanism), props, cbh);
            }
            catch (NoSuchAlgorithmException e) {
                throw new SaslException("Hash algorithm not supported for mechanism " + mechanism, e);
            }
        }

        @Override
        public String[] getMechanismNames(Map<String, ?> props) {
            Collection<String> mechanisms = ScramMechanism.mechanismNames();
            return mechanisms.toArray(new String[0]);
        }
    }

    static enum State {
        RECEIVE_CLIENT_FIRST_MESSAGE,
        RECEIVE_CLIENT_FINAL_MESSAGE,
        COMPLETE,
        FAILED;

    }
}

