"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSet = exports.FileMap = void 0;
const typescript_1 = __importDefault(require("typescript"));
const utils_1 = require("../../utils");
/**
 * wrapper around Map<string, T> for case insensitive file systems
 */
class FileMap {
    constructor(useCaseSensitiveFileNames = typescript_1.default.sys.useCaseSensitiveFileNames) {
        this.map = new Map();
        this.getCanonicalFileName = (0, utils_1.createGetCanonicalFileName)(useCaseSensitiveFileNames);
    }
    get(filePath) {
        return this.map.get(this.getCanonicalFileName(filePath));
    }
    set(filePath, value) {
        const canonicalFileName = this.getCanonicalFileName(filePath);
        return this.map.set(canonicalFileName, value);
    }
    has(filePath) {
        return this.map.has(this.getCanonicalFileName(filePath));
    }
    delete(filePath) {
        return this.map.delete(this.getCanonicalFileName(filePath));
    }
    /**
     * Returns an iterable of key, value pairs for every entry in the map.
     * In case insensitive file system the key in the key-value pairs is in lowercase
     */
    entries() {
        return this.map.entries();
    }
    /**
     *
     * @param callbackfn In case insensitive file system the key parameter for the callback is in lowercase
     */
    forEach(callbackfn) {
        return this.map.forEach(callbackfn);
    }
    values() {
        return this.map.values();
    }
    clear() {
        this.map.clear();
    }
    /**
     * Returns an iterable of values in the map.
     * In case insensitive file system the key is in lowercase
     */
    keys() {
        return this.map.keys();
    }
    get size() {
        return this.map.size;
    }
    [Symbol.iterator]() {
        return this.map[Symbol.iterator]();
    }
}
exports.FileMap = FileMap;
class FileSet {
    constructor(useCaseSensitiveFileNames = typescript_1.default.sys.useCaseSensitiveFileNames) {
        this.set = new Set();
        this.getCanonicalFileName = (0, utils_1.createGetCanonicalFileName)(useCaseSensitiveFileNames);
    }
    add(filePath) {
        this.set.add(this.getCanonicalFileName(filePath));
    }
    has(filePath) {
        return this.set.has(this.getCanonicalFileName(filePath));
    }
    delete(filePath) {
        return this.set.delete(this.getCanonicalFileName(filePath));
    }
    clear() {
        this.set.clear();
    }
    get size() {
        return this.set.size;
    }
    [Symbol.iterator]() {
        return this.set[Symbol.iterator]();
    }
}
exports.FileSet = FileSet;
//# sourceMappingURL=fileCollection.js.map