
#include <stdlib.h>
#include <string.h>

#include <memory>

#include "trace_writer_local.hpp"

// To validate our prototypes
#define GL_GLEXT_PROTOTYPES
#define GLX_GLXEXT_PROTOTYPES

#include "glproc.hpp"
#include "glsize.hpp"

static __GLXextFuncPtr _wrapProcAddress(const GLubyte * procName, __GLXextFuncPtr procPtr);

#ifdef _WIN32
#  include <malloc.h> // alloca
#  ifndef alloca
#    define alloca _alloca
#  endif
#else
#  include <alloca.h> // alloca
#endif


/*
 * g_WrappedObjects is already protected by trace::LocalWriter::mutex
 * This lock is hold during the beginEnter/endEnter and beginLeave/endLeave sections
 */
static std::map<void *, void *> g_WrappedObjects;
#include <algorithm>
#include "cxx_compat.hpp"

#include "gltrace.hpp"
#include "gltrace_arrays.hpp"
#include "glmemshadow.hpp"
#include "gltrace_unpack_compressed.hpp"

static inline bool _need_user_arrays(gltrace::Context *_ctx)
{
    if (!_ctx->user_arrays) {
        return false;
    }

    glfeatures::Profile profile = _ctx->profile;
    bool es1 = profile.es() && profile.major == 1;

    // glSecondaryColorPointer
  if (profile.desktop()) {
    if (_glIsEnabled(GL_SECONDARY_COLOR_ARRAY) &&
        _glGetInteger(GL_SECONDARY_COLOR_ARRAY_BUFFER_BINDING) == 0) {
        return true;
    }
  }

    // glFogCoordPointer
  if (profile.desktop()) {
    if (_glIsEnabled(GL_FOG_COORD_ARRAY) &&
        _glGetInteger(GL_FOG_COORD_ARRAY_BUFFER_BINDING) == 0) {
        return true;
    }
  }

    // glEdgeFlagPointer
  if (profile.desktop()) {
    if (_glIsEnabled(GL_EDGE_FLAG_ARRAY) &&
        _glGetInteger(GL_EDGE_FLAG_ARRAY_BUFFER_BINDING) == 0) {
        return true;
    }
  }

    // glTexCoordPointer
  if ((profile.desktop() || es1)) {
    GLint max_units = 0;
    if (_ctx->profile.desktop())
        _glGetIntegerv(GL_MAX_TEXTURE_COORDS, &max_units);
    else
        _glGetIntegerv(GL_MAX_TEXTURE_UNITS, &max_units);
    GLint client_active_texture = GL_TEXTURE0;
    if (max_units > 0) {
        _glGetIntegerv(GL_CLIENT_ACTIVE_TEXTURE, &client_active_texture);
    }
    GLint unit = 0;
    do {
        GLint texture = GL_TEXTURE0 + unit;
        if (max_units > 0) {
            _glClientActiveTexture(texture);
        }
    if (_glIsEnabled(GL_TEXTURE_COORD_ARRAY) &&
        _glGetInteger(GL_TEXTURE_COORD_ARRAY_BUFFER_BINDING) == 0) {
    if (max_units > 0) {
        _glClientActiveTexture(client_active_texture);
    }
        return true;
    }
    } while (++unit < max_units);
    if (max_units > 0) {
        _glClientActiveTexture(client_active_texture);
    }
  }

    // glIndexPointer
  if (profile.desktop()) {
    if (_glIsEnabled(GL_INDEX_ARRAY) &&
        _glGetInteger(GL_INDEX_ARRAY_BUFFER_BINDING) == 0) {
        return true;
    }
  }

    // glColorPointer
  if ((profile.desktop() || es1)) {
    if (_glIsEnabled(GL_COLOR_ARRAY) &&
        _glGetInteger(GL_COLOR_ARRAY_BUFFER_BINDING) == 0) {
        return true;
    }
  }

    // glNormalPointer
  if ((profile.desktop() || es1)) {
    if (_glIsEnabled(GL_NORMAL_ARRAY) &&
        _glGetInteger(GL_NORMAL_ARRAY_BUFFER_BINDING) == 0) {
        return true;
    }
  }

    // glVertexPointer
  if ((profile.desktop() || es1)) {
    if (_glIsEnabled(GL_VERTEX_ARRAY) &&
        _glGetInteger(GL_VERTEX_ARRAY_BUFFER_BINDING) == 0) {
        return true;
    }
  }

    // ES1 does not support generic vertex attributes
    if (es1)
        return false;

    // glVertexAttribPointer
    GLint _max_vertex_attribs = _glGetInteger(GL_MAX_VERTEX_ATTRIBS);
    for (GLint index = 0; index < _max_vertex_attribs; ++index) {
        if (_glGetVertexAttribi(index, GL_VERTEX_ATTRIB_ARRAY_ENABLED) &&
            _glGetVertexAttribi(index, GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING) == 0) {
            return true;
        }
    }

    return false;
}

static void _trace_user_arrays(gltrace::Context *_ctx, GLuint count, bool instanced=false, GLuint instancecount=1);

void APIENTRY _fake_glScissor(GLint x, GLint y, GLsizei width, GLsizei height);
void APIENTRY _fake_glTexImage2D(GLenum target, GLint level, GLint internalformat, GLsizei width, GLsizei height, GLint border, GLenum format, GLenum type, const GLvoid * pixels);
void APIENTRY _fake_glViewport(GLint x, GLint y, GLsizei width, GLsizei height);
void APIENTRY _fake_glNewList(GLuint list, GLenum mode);
void APIENTRY _fake_glEndList(void);
void APIENTRY _fake_glBitmap(GLsizei width, GLsizei height, GLfloat xorig, GLfloat yorig, GLfloat xmove, GLfloat ymove, const GLubyte * bitmap);
void APIENTRY _fake_glDisableClientState(GLenum array);
void APIENTRY _fake_glEnableClientState(GLenum array);
void APIENTRY _fake_glClientActiveTexture(GLenum texture);
void APIENTRY _fake_glBindBuffer(GLenum target, GLuint buffer);
void APIENTRY _fake_glBindAttribLocation(GLuint program, GLuint index, const GLchar * name);
void APIENTRY _fake_glBindAttribLocationARB(GLhandleARB programObj, GLuint index, const GLcharARB * name);
void APIENTRY _fake_glStringMarkerGREMEDY(GLsizei len, const GLvoid * string);

inline void
_fakeStringMarker(const std::string &s) {
    _fake_glStringMarkerGREMEDY(s.length(), s.data());
}

// whether glMapBufferRange(GL_MAP_WRITE_BIT) has ever been called
static bool _checkBufferMapRange = false;

// whether glBufferParameteriAPPLE(GL_BUFFER_FLUSHING_UNMAP_APPLE, GL_FALSE) has ever been called
static bool _checkBufferFlushingUnmapAPPLE = false;

static bool
is_symbolic_pname(GLenum pname) {
    switch (pname) {
    case GL_LIST_MODE:
    case GL_POLYGON_MODE:
    case GL_CULL_FACE_MODE:
    case GL_FRONT_FACE:
    case GL_SHADE_MODEL:
    case GL_COLOR_MATERIAL_FACE:
    case GL_COLOR_MATERIAL_PARAMETER:
    case GL_FOG_MODE:
    case GL_DEPTH_FUNC:
    case GL_STENCIL_FUNC:
    case GL_STENCIL_FAIL:
    case GL_STENCIL_PASS_DEPTH_FAIL:
    case GL_STENCIL_PASS_DEPTH_PASS:
    case GL_MATRIX_MODE:
    case GL_ALPHA_TEST_FUNC:
    case GL_BLEND_DST:
    case GL_BLEND_SRC:
    case GL_LOGIC_OP_MODE:
    case GL_DRAW_BUFFER:
    case GL_READ_BUFFER:
    case GL_RENDER_MODE:
    case GL_PERSPECTIVE_CORRECTION_HINT:
    case GL_POINT_SMOOTH_HINT:
    case GL_LINE_SMOOTH_HINT:
    case GL_POLYGON_SMOOTH_HINT:
    case GL_FOG_HINT:
    case GL_FEEDBACK_BUFFER_TYPE:
    case GL_TEXTURE_INTERNAL_FORMAT:
    case GL_TEXTURE_TARGET:
    case GL_TEXTURE_ENV_MODE:
    case GL_TEXTURE_GEN_MODE:
    case GL_TEXTURE_MAG_FILTER:
    case GL_TEXTURE_MIN_FILTER:
    case GL_TEXTURE_WRAP_S:
    case GL_TEXTURE_WRAP_T:
    case GL_BLEND_EQUATION:
    case GL_CONVOLUTION_BORDER_MODE:
    case GL_CONVOLUTION_FORMAT:
    case GL_HISTOGRAM_FORMAT:
    case GL_MINMAX_FORMAT:
    case GL_TEXTURE_WRAP_R:
    case GL_VERTEX_ARRAY_TYPE:
    case GL_NORMAL_ARRAY_TYPE:
    case GL_COLOR_ARRAY_TYPE:
    case GL_INDEX_ARRAY_TYPE:
    case GL_TEXTURE_COORD_ARRAY_TYPE:
    case GL_BLEND_DST_RGB:
    case GL_BLEND_SRC_RGB:
    case GL_BLEND_DST_ALPHA:
    case GL_BLEND_SRC_ALPHA:
    case GL_COLOR_TABLE_FORMAT:
    case GL_CLIP_VOLUME_CLIPPING_HINT_EXT:
    case GL_GENERATE_MIPMAP_HINT:
    case GL_TEXTURE_COMPARE_OPERATOR_SGIX:
    case GL_LIGHT_MODEL_COLOR_CONTROL:
    case GL_FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING:
    case GL_FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE:
    case GL_RESET_NOTIFICATION_STRATEGY:
    case GL_LAYER_PROVOKING_VERTEX:
    case GL_VIEWPORT_INDEX_PROVOKING_VERTEX:
    case GL_INTERNALFORMAT_PREFERRED:
    case GL_INTERNALFORMAT_RED_TYPE:
    case GL_INTERNALFORMAT_GREEN_TYPE:
    case GL_INTERNALFORMAT_BLUE_TYPE:
    case GL_INTERNALFORMAT_ALPHA_TYPE:
    case GL_INTERNALFORMAT_DEPTH_TYPE:
    case GL_INTERNALFORMAT_STENCIL_TYPE:
    case GL_FRAMEBUFFER_RENDERABLE:
    case GL_FRAMEBUFFER_RENDERABLE_LAYERED:
    case GL_FRAMEBUFFER_BLEND:
    case GL_READ_PIXELS:
    case GL_READ_PIXELS_FORMAT:
    case GL_READ_PIXELS_TYPE:
    case GL_TEXTURE_IMAGE_FORMAT:
    case GL_TEXTURE_IMAGE_TYPE:
    case GL_GET_TEXTURE_IMAGE_FORMAT:
    case GL_GET_TEXTURE_IMAGE_TYPE:
    case GL_MANUAL_GENERATE_MIPMAP:
    case GL_AUTO_GENERATE_MIPMAP:
    case GL_COLOR_ENCODING:
    case GL_SRGB_READ:
    case GL_SRGB_WRITE:
    case GL_SRGB_DECODE_ARB:
    case GL_FILTER:
    case GL_VERTEX_TEXTURE:
    case GL_TESS_CONTROL_TEXTURE:
    case GL_TESS_EVALUATION_TEXTURE:
    case GL_GEOMETRY_TEXTURE:
    case GL_FRAGMENT_TEXTURE:
    case GL_COMPUTE_TEXTURE:
    case GL_TEXTURE_SHADOW:
    case GL_TEXTURE_GATHER:
    case GL_TEXTURE_GATHER_SHADOW:
    case GL_SHADER_IMAGE_LOAD:
    case GL_SHADER_IMAGE_STORE:
    case GL_SHADER_IMAGE_ATOMIC:
    case GL_IMAGE_COMPATIBILITY_CLASS:
    case GL_IMAGE_PIXEL_FORMAT:
    case GL_IMAGE_PIXEL_TYPE:
    case GL_SIMULTANEOUS_TEXTURE_AND_DEPTH_TEST:
    case GL_SIMULTANEOUS_TEXTURE_AND_STENCIL_TEST:
    case GL_SIMULTANEOUS_TEXTURE_AND_DEPTH_WRITE:
    case GL_SIMULTANEOUS_TEXTURE_AND_STENCIL_WRITE:
    case GL_CLEAR_BUFFER:
    case GL_TEXTURE_VIEW:
    case GL_VIEW_COMPATIBILITY_CLASS:
    case GL_CONTEXT_RELEASE_BEHAVIOR:
    case GL_CONTEXT_RELEASE_BEHAVIOR_FLUSH:
    case GL_TEXTURE_MEMORY_LAYOUT_INTEL:
    case GL_TANGENT_ARRAY_TYPE_EXT:
    case GL_BINORMAL_ARRAY_TYPE_EXT:
    case GL_FOG_COORD_SRC:
    case GL_FOG_COORD_ARRAY_TYPE:
    case GL_SECONDARY_COLOR_ARRAY_TYPE:
    case GL_ACTIVE_TEXTURE:
    case GL_CLIENT_ACTIVE_TEXTURE:
    case GL_TEXTURE_COMPRESSION_HINT:
    case GL_VERTEX_WEIGHT_ARRAY_TYPE_EXT:
    case GL_MULTISAMPLE_FILTER_HINT_NV:
    case GL_FOG_DISTANCE_MODE_NV:
    case GL_COMBINE_RGB:
    case GL_COMBINE_ALPHA:
    case GL_SRC0_RGB:
    case GL_SRC1_RGB:
    case GL_SRC2_RGB:
    case GL_SOURCE3_RGB_NV:
    case GL_SRC0_ALPHA:
    case GL_SRC1_ALPHA:
    case GL_SRC2_ALPHA:
    case GL_SOURCE3_ALPHA_NV:
    case GL_OPERAND0_RGB:
    case GL_OPERAND1_RGB:
    case GL_OPERAND2_RGB:
    case GL_OPERAND3_RGB_NV:
    case GL_OPERAND0_ALPHA:
    case GL_OPERAND1_ALPHA:
    case GL_OPERAND2_ALPHA:
    case GL_OPERAND3_ALPHA_NV:
    case GL_TEXTURE_STORAGE_HINT_APPLE:
    case GL_REPLACEMENT_CODE_ARRAY_TYPE_SUN:
    case GL_VERTEX_ATTRIB_ARRAY_TYPE:
    case GL_COMPRESSED_TEXTURE_FORMATS:
    case GL_WEIGHT_ARRAY_TYPE_ARB:
    case GL_RGBA_UNSIGNED_DOT_PRODUCT_MAPPING_NV:
    case GL_SHADER_OPERATION_NV:
    case GL_CULL_MODES_NV:
    case GL_PREVIOUS_TEXTURE_INPUT_NV:
    case GL_BUFFER_USAGE:
    case GL_ELEMENT_ARRAY_TYPE_ATI:
    case GL_BUMP_TARGET_ATI:
    case GL_VARIANT_ARRAY_TYPE_EXT:
    case GL_PROGRAM_BINARY_FORMATS:
    case GL_STENCIL_BACK_FUNC:
    case GL_STENCIL_BACK_FAIL:
    case GL_STENCIL_BACK_PASS_DEPTH_FAIL:
    case GL_STENCIL_BACK_PASS_DEPTH_PASS:
    case GL_DRAW_BUFFER0:
    case GL_DRAW_BUFFER1:
    case GL_DRAW_BUFFER2:
    case GL_DRAW_BUFFER3:
    case GL_DRAW_BUFFER4:
    case GL_DRAW_BUFFER5:
    case GL_DRAW_BUFFER6:
    case GL_DRAW_BUFFER7:
    case GL_DRAW_BUFFER8:
    case GL_DRAW_BUFFER9:
    case GL_DRAW_BUFFER10:
    case GL_DRAW_BUFFER11:
    case GL_DRAW_BUFFER12:
    case GL_DRAW_BUFFER13:
    case GL_DRAW_BUFFER14:
    case GL_DRAW_BUFFER15:
    case GL_BLEND_EQUATION_ALPHA:
    case GL_MATRIX_INDEX_ARRAY_TYPE_ARB:
    case GL_DEPTH_TEXTURE_MODE:
    case GL_TEXTURE_COMPARE_MODE:
    case GL_TEXTURE_COMPARE_FUNC:
    case GL_POINT_SPRITE_R_MODE_NV:
    case GL_PROGRAM_FORMAT_ARB:
    case GL_TEXTURE_UNSIGNED_REMAP_MODE_NV:
    case GL_BUFFER_ACCESS:
    case GL_ACTIVE_STENCIL_FACE_EXT:
    case GL_GEOMETRY_INPUT_TYPE:
    case GL_GEOMETRY_OUTPUT_TYPE:
    case GL_CLAMP_VERTEX_COLOR:
    case GL_CLAMP_FRAGMENT_COLOR:
    case GL_CLAMP_READ_COLOR:
    case GL_ELEMENT_ARRAY_TYPE_APPLE:
    case GL_UNIFORM_TYPE:
    case GL_TEXTURE_SRGB_DECODE_EXT:
    case GL_OBJECT_TYPE_ARB:
    case GL_SHADER_TYPE:
    case GL_IMPLEMENTATION_COLOR_READ_TYPE:
    case GL_IMPLEMENTATION_COLOR_READ_FORMAT:
    case GL_TEXTURE_RED_TYPE:
    case GL_TEXTURE_GREEN_TYPE:
    case GL_TEXTURE_BLUE_TYPE:
    case GL_TEXTURE_ALPHA_TYPE:
    case GL_TEXTURE_LUMINANCE_TYPE:
    case GL_TEXTURE_INTENSITY_TYPE:
    case GL_TEXTURE_DEPTH_TYPE:
    case GL_TEXTURE_BUFFER_FORMAT_ARB:
    case GL_TRANSFORM_FEEDBACK_BUFFER_MODE:
    case GL_POINT_SPRITE_COORD_ORIGIN:
    case GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE:
    case GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE:
    case GL_RENDERBUFFER_INTERNAL_FORMAT:
    case GL_GEOMETRY_INPUT_TYPE_ARB:
    case GL_GEOMETRY_OUTPUT_TYPE_ARB:
    case GL_NAMED_STRING_TYPE_ARB:
    case GL_SHADER_BINARY_FORMATS:
    case GL_TEXTURE_SWIZZLE_R:
    case GL_TEXTURE_SWIZZLE_G:
    case GL_TEXTURE_SWIZZLE_B:
    case GL_TEXTURE_SWIZZLE_A:
    case GL_TEXTURE_SWIZZLE_RGBA:
    case GL_PROVOKING_VERTEX:
    case GL_TESS_GEN_MODE:
    case GL_TESS_GEN_SPACING:
    case GL_TESS_GEN_VERTEX_ORDER:
    case GL_TESS_GEN_POINT_MODE:
    case GL_WINDOW_RECTANGLE_MODE_EXT:
    case GL_IMAGE_BINDING_ACCESS:
    case GL_TEXTURE_ASTC_DECODE_PRECISION_EXT:
    case GL_IMAGE_BINDING_FORMAT:
    case GL_DEPTH_STENCIL_TEXTURE_MODE:
    case GL_TYPE:
    case GL_CLIP_ORIGIN:
    case GL_CLIP_DEPTH_MODE:
    case GL_TEXTURE_REDUCTION_MODE_ARB:
        return true;
    default:
        return false;
    }
}

template<class T>
static inline bool
is_symbolic_param(T param) {
    return static_cast<T>(static_cast<GLenum>(param)) == param;
}

static size_t
_gl_param_size(GLenum pname) {
    switch (pname) {
    case GL_CURRENT_COLOR: return 4;
    case GL_CURRENT_INDEX: return 1;
    case GL_CURRENT_NORMAL: return 3;
    case GL_CURRENT_TEXTURE_COORDS: return 4;
    case GL_CURRENT_RASTER_COLOR: return 4;
    case GL_CURRENT_RASTER_INDEX: return 1;
    case GL_CURRENT_RASTER_TEXTURE_COORDS: return 4;
    case GL_CURRENT_RASTER_POSITION: return 4;
    case GL_CURRENT_RASTER_POSITION_VALID: return 1;
    case GL_CURRENT_RASTER_DISTANCE: return 1;
    case GL_POINT_SMOOTH: return 1;
    case GL_POINT_SIZE: return 1;
    case GL_POINT_SIZE_RANGE: return 2;
    case GL_POINT_SIZE_GRANULARITY: return 1;
    case GL_LINE_SMOOTH: return 1;
    case GL_LINE_WIDTH: return 1;
    case GL_LINE_WIDTH_RANGE: return 2;
    case GL_LINE_WIDTH_GRANULARITY: return 1;
    case GL_LINE_STIPPLE: return 1;
    case GL_LINE_STIPPLE_PATTERN: return 1;
    case GL_LINE_STIPPLE_REPEAT: return 1;
    case GL_LIST_MODE: return 1;
    case GL_MAX_LIST_NESTING: return 1;
    case GL_LIST_BASE: return 1;
    case GL_LIST_INDEX: return 1;
    case GL_POLYGON_MODE: return 2;
    case GL_POLYGON_SMOOTH: return 1;
    case GL_POLYGON_STIPPLE: return 1;
    case GL_EDGE_FLAG: return 1;
    case GL_CULL_FACE: return 1;
    case GL_CULL_FACE_MODE: return 1;
    case GL_FRONT_FACE: return 1;
    case GL_LIGHTING: return 1;
    case GL_LIGHT_MODEL_LOCAL_VIEWER: return 1;
    case GL_LIGHT_MODEL_TWO_SIDE: return 1;
    case GL_LIGHT_MODEL_AMBIENT: return 4;
    case GL_SHADE_MODEL: return 1;
    case GL_COLOR_MATERIAL_FACE: return 1;
    case GL_COLOR_MATERIAL_PARAMETER: return 1;
    case GL_COLOR_MATERIAL: return 1;
    case GL_FOG: return 1;
    case GL_FOG_INDEX: return 1;
    case GL_FOG_DENSITY: return 1;
    case GL_FOG_START: return 1;
    case GL_FOG_END: return 1;
    case GL_FOG_MODE: return 1;
    case GL_FOG_COLOR: return 4;
    case GL_DEPTH_RANGE: return 2;
    case GL_DEPTH_TEST: return 1;
    case GL_DEPTH_WRITEMASK: return 1;
    case GL_DEPTH_CLEAR_VALUE: return 1;
    case GL_DEPTH_FUNC: return 1;
    case GL_ACCUM_CLEAR_VALUE: return 4;
    case GL_STENCIL_TEST: return 1;
    case GL_STENCIL_CLEAR_VALUE: return 1;
    case GL_STENCIL_FUNC: return 1;
    case GL_STENCIL_VALUE_MASK: return 1;
    case GL_STENCIL_FAIL: return 1;
    case GL_STENCIL_PASS_DEPTH_FAIL: return 1;
    case GL_STENCIL_PASS_DEPTH_PASS: return 1;
    case GL_STENCIL_REF: return 1;
    case GL_STENCIL_WRITEMASK: return 1;
    case GL_MATRIX_MODE: return 1;
    case GL_NORMALIZE: return 1;
    case GL_VIEWPORT: return 4;
    case GL_MODELVIEW_STACK_DEPTH: return 1;
    case GL_PROJECTION_STACK_DEPTH: return 1;
    case GL_TEXTURE_STACK_DEPTH: return 1;
    case GL_MODELVIEW_MATRIX: return 16;
    case GL_PROJECTION_MATRIX: return 16;
    case GL_TEXTURE_MATRIX: return 16;
    case GL_ATTRIB_STACK_DEPTH: return 1;
    case GL_CLIENT_ATTRIB_STACK_DEPTH: return 1;
    case GL_ALPHA_TEST: return 1;
    case GL_ALPHA_TEST_FUNC: return 1;
    case GL_ALPHA_TEST_REF: return 1;
    case GL_DITHER: return 1;
    case GL_BLEND_DST: return 1;
    case GL_BLEND_SRC: return 1;
    case GL_BLEND: return 1;
    case GL_LOGIC_OP_MODE: return 1;
    case GL_INDEX_LOGIC_OP: return 1;
    case GL_COLOR_LOGIC_OP: return 1;
    case GL_AUX_BUFFERS: return 1;
    case GL_DRAW_BUFFER: return 1;
    case GL_READ_BUFFER: return 1;
    case GL_SCISSOR_BOX: return 4;
    case GL_SCISSOR_TEST: return 1;
    case GL_INDEX_CLEAR_VALUE: return 1;
    case GL_INDEX_WRITEMASK: return 1;
    case GL_COLOR_CLEAR_VALUE: return 4;
    case GL_COLOR_WRITEMASK: return 4;
    case GL_INDEX_MODE: return 1;
    case GL_RGBA_MODE: return 1;
    case GL_DOUBLEBUFFER: return 1;
    case GL_STEREO: return 1;
    case GL_RENDER_MODE: return 1;
    case GL_PERSPECTIVE_CORRECTION_HINT: return 1;
    case GL_POINT_SMOOTH_HINT: return 1;
    case GL_LINE_SMOOTH_HINT: return 1;
    case GL_POLYGON_SMOOTH_HINT: return 1;
    case GL_FOG_HINT: return 1;
    case GL_TEXTURE_GEN_S: return 1;
    case GL_TEXTURE_GEN_T: return 1;
    case GL_TEXTURE_GEN_R: return 1;
    case GL_TEXTURE_GEN_Q: return 1;
    case GL_PIXEL_MAP_I_TO_I_SIZE: return 1;
    case GL_PIXEL_MAP_S_TO_S_SIZE: return 1;
    case GL_PIXEL_MAP_I_TO_R_SIZE: return 1;
    case GL_PIXEL_MAP_I_TO_G_SIZE: return 1;
    case GL_PIXEL_MAP_I_TO_B_SIZE: return 1;
    case GL_PIXEL_MAP_I_TO_A_SIZE: return 1;
    case GL_PIXEL_MAP_R_TO_R_SIZE: return 1;
    case GL_PIXEL_MAP_G_TO_G_SIZE: return 1;
    case GL_PIXEL_MAP_B_TO_B_SIZE: return 1;
    case GL_PIXEL_MAP_A_TO_A_SIZE: return 1;
    case GL_UNPACK_SWAP_BYTES: return 1;
    case GL_UNPACK_LSB_FIRST: return 1;
    case GL_UNPACK_ROW_LENGTH: return 1;
    case GL_UNPACK_SKIP_ROWS: return 1;
    case GL_UNPACK_SKIP_PIXELS: return 1;
    case GL_UNPACK_ALIGNMENT: return 1;
    case GL_PACK_SWAP_BYTES: return 1;
    case GL_PACK_LSB_FIRST: return 1;
    case GL_PACK_ROW_LENGTH: return 1;
    case GL_PACK_SKIP_ROWS: return 1;
    case GL_PACK_SKIP_PIXELS: return 1;
    case GL_PACK_ALIGNMENT: return 1;
    case GL_MAP_COLOR: return 1;
    case GL_MAP_STENCIL: return 1;
    case GL_INDEX_SHIFT: return 1;
    case GL_INDEX_OFFSET: return 1;
    case GL_RED_SCALE: return 1;
    case GL_RED_BIAS: return 1;
    case GL_ZOOM_X: return 1;
    case GL_ZOOM_Y: return 1;
    case GL_GREEN_SCALE: return 1;
    case GL_GREEN_BIAS: return 1;
    case GL_BLUE_SCALE: return 1;
    case GL_BLUE_BIAS: return 1;
    case GL_ALPHA_SCALE: return 1;
    case GL_ALPHA_BIAS: return 1;
    case GL_DEPTH_SCALE: return 1;
    case GL_DEPTH_BIAS: return 1;
    case GL_MAX_EVAL_ORDER: return 1;
    case GL_MAX_LIGHTS: return 1;
    case GL_MAX_CLIP_DISTANCES: return 1;
    case GL_MAX_TEXTURE_SIZE: return 1;
    case GL_MAX_PIXEL_MAP_TABLE: return 1;
    case GL_MAX_ATTRIB_STACK_DEPTH: return 1;
    case GL_MAX_MODELVIEW_STACK_DEPTH: return 1;
    case GL_MAX_NAME_STACK_DEPTH: return 1;
    case GL_MAX_PROJECTION_STACK_DEPTH: return 1;
    case GL_MAX_TEXTURE_STACK_DEPTH: return 1;
    case GL_MAX_VIEWPORT_DIMS: return 2;
    case GL_MAX_CLIENT_ATTRIB_STACK_DEPTH: return 1;
    case GL_SUBPIXEL_BITS: return 1;
    case GL_INDEX_BITS: return 1;
    case GL_RED_BITS: return 1;
    case GL_GREEN_BITS: return 1;
    case GL_BLUE_BITS: return 1;
    case GL_ALPHA_BITS: return 1;
    case GL_DEPTH_BITS: return 1;
    case GL_STENCIL_BITS: return 1;
    case GL_ACCUM_RED_BITS: return 1;
    case GL_ACCUM_GREEN_BITS: return 1;
    case GL_ACCUM_BLUE_BITS: return 1;
    case GL_ACCUM_ALPHA_BITS: return 1;
    case GL_NAME_STACK_DEPTH: return 1;
    case GL_AUTO_NORMAL: return 1;
    case GL_MAP1_COLOR_4: return 1;
    case GL_MAP1_INDEX: return 1;
    case GL_MAP1_NORMAL: return 1;
    case GL_MAP1_TEXTURE_COORD_1: return 1;
    case GL_MAP1_TEXTURE_COORD_2: return 1;
    case GL_MAP1_TEXTURE_COORD_3: return 1;
    case GL_MAP1_TEXTURE_COORD_4: return 1;
    case GL_MAP1_VERTEX_3: return 1;
    case GL_MAP1_VERTEX_4: return 1;
    case GL_MAP2_COLOR_4: return 1;
    case GL_MAP2_INDEX: return 1;
    case GL_MAP2_NORMAL: return 1;
    case GL_MAP2_TEXTURE_COORD_1: return 1;
    case GL_MAP2_TEXTURE_COORD_2: return 1;
    case GL_MAP2_TEXTURE_COORD_3: return 1;
    case GL_MAP2_TEXTURE_COORD_4: return 1;
    case GL_MAP2_VERTEX_3: return 1;
    case GL_MAP2_VERTEX_4: return 1;
    case GL_MAP1_GRID_DOMAIN: return 2;
    case GL_MAP1_GRID_SEGMENTS: return 1;
    case GL_MAP2_GRID_DOMAIN: return 4;
    case GL_MAP2_GRID_SEGMENTS: return 2;
    case GL_TEXTURE_1D: return 1;
    case GL_TEXTURE_2D: return 1;
    case GL_FEEDBACK_BUFFER_POINTER: return 1;
    case GL_FEEDBACK_BUFFER_SIZE: return 1;
    case GL_FEEDBACK_BUFFER_TYPE: return 1;
    case GL_SELECTION_BUFFER_POINTER: return 1;
    case GL_SELECTION_BUFFER_SIZE: return 1;
    case GL_TEXTURE_WIDTH: return 1;
    case GL_TEXTURE_HEIGHT: return 1;
    case GL_TEXTURE_INTERNAL_FORMAT: return 1;
    case GL_TEXTURE_BORDER_COLOR: return 4;
    case GL_TEXTURE_BORDER: return 1;
    case GL_TEXTURE_TARGET: return 1;
    case GL_AMBIENT: return 4;
    case GL_DIFFUSE: return 4;
    case GL_SPECULAR: return 4;
    case GL_POSITION: return 4;
    case GL_SPOT_DIRECTION: return 3;
    case GL_SPOT_EXPONENT: return 1;
    case GL_SPOT_CUTOFF: return 1;
    case GL_CONSTANT_ATTENUATION: return 1;
    case GL_LINEAR_ATTENUATION: return 1;
    case GL_QUADRATIC_ATTENUATION: return 1;
    case GL_EMISSION: return 4;
    case GL_SHININESS: return 1;
    case GL_AMBIENT_AND_DIFFUSE: return 4;
    case GL_COLOR_INDEXES: return 3;
    case GL_VENDOR: return 1;
    case GL_RENDERER: return 1;
    case GL_VERSION: return 1;
    case GL_EXTENSIONS: return 1;
    case GL_TEXTURE_ENV_MODE: return 1;
    case GL_TEXTURE_ENV_COLOR: return 4;
    case GL_TEXTURE_GEN_MODE: return 1;
    case GL_OBJECT_PLANE: return 4;
    case GL_EYE_PLANE: return 4;
    case GL_TEXTURE_MAG_FILTER: return 1;
    case GL_TEXTURE_MIN_FILTER: return 1;
    case GL_TEXTURE_WRAP_S: return 1;
    case GL_TEXTURE_WRAP_T: return 1;
    case GL_POLYGON_OFFSET_UNITS: return 1;
    case GL_POLYGON_OFFSET_POINT: return 1;
    case GL_POLYGON_OFFSET_LINE: return 1;
    case GL_CLIP_DISTANCE0: return 1;
    case GL_CLIP_DISTANCE1: return 1;
    case GL_CLIP_DISTANCE2: return 1;
    case GL_CLIP_DISTANCE3: return 1;
    case GL_CLIP_DISTANCE4: return 1;
    case GL_CLIP_DISTANCE5: return 1;
    case GL_CLIP_DISTANCE6: return 1;
    case GL_CLIP_DISTANCE7: return 1;
    case GL_LIGHT0: return 1;
    case GL_LIGHT1: return 1;
    case GL_LIGHT2: return 1;
    case GL_LIGHT3: return 1;
    case GL_LIGHT4: return 1;
    case GL_LIGHT5: return 1;
    case GL_LIGHT6: return 1;
    case GL_LIGHT7: return 1;
    case GL_BLEND_COLOR: return 4;
    case GL_BLEND_EQUATION: return 1;
    case GL_PACK_CMYK_HINT_EXT: return 1;
    case GL_UNPACK_CMYK_HINT_EXT: return 1;
    case GL_CONVOLUTION_1D: return 1;
    case GL_CONVOLUTION_2D: return 1;
    case GL_SEPARABLE_2D: return 1;
    case GL_CONVOLUTION_BORDER_MODE: return 1;
    case GL_CONVOLUTION_FILTER_SCALE: return 4;
    case GL_CONVOLUTION_FILTER_BIAS: return 4;
    case GL_CONVOLUTION_FORMAT: return 1;
    case GL_CONVOLUTION_WIDTH: return 1;
    case GL_CONVOLUTION_HEIGHT: return 1;
    case GL_MAX_CONVOLUTION_WIDTH: return 1;
    case GL_MAX_CONVOLUTION_HEIGHT: return 1;
    case GL_POST_CONVOLUTION_RED_SCALE: return 1;
    case GL_POST_CONVOLUTION_GREEN_SCALE: return 1;
    case GL_POST_CONVOLUTION_BLUE_SCALE: return 1;
    case GL_POST_CONVOLUTION_ALPHA_SCALE: return 1;
    case GL_POST_CONVOLUTION_RED_BIAS: return 1;
    case GL_POST_CONVOLUTION_GREEN_BIAS: return 1;
    case GL_POST_CONVOLUTION_BLUE_BIAS: return 1;
    case GL_POST_CONVOLUTION_ALPHA_BIAS: return 1;
    case GL_HISTOGRAM: return 1;
    case GL_HISTOGRAM_WIDTH: return 1;
    case GL_HISTOGRAM_FORMAT: return 1;
    case GL_HISTOGRAM_RED_SIZE: return 1;
    case GL_HISTOGRAM_GREEN_SIZE: return 1;
    case GL_HISTOGRAM_BLUE_SIZE: return 1;
    case GL_HISTOGRAM_ALPHA_SIZE: return 1;
    case GL_HISTOGRAM_LUMINANCE_SIZE: return 1;
    case GL_HISTOGRAM_SINK: return 1;
    case GL_MINMAX: return 1;
    case GL_MINMAX_FORMAT: return 1;
    case GL_MINMAX_SINK: return 1;
    case GL_POLYGON_OFFSET_FILL: return 1;
    case GL_POLYGON_OFFSET_FACTOR: return 1;
    case GL_POLYGON_OFFSET_BIAS_EXT: return 1;
    case GL_RESCALE_NORMAL: return 1;
    case GL_TEXTURE_RED_SIZE: return 1;
    case GL_TEXTURE_GREEN_SIZE: return 1;
    case GL_TEXTURE_BLUE_SIZE: return 1;
    case GL_TEXTURE_ALPHA_SIZE: return 1;
    case GL_TEXTURE_LUMINANCE_SIZE: return 1;
    case GL_TEXTURE_INTENSITY_SIZE: return 1;
    case GL_TEXTURE_PRIORITY: return 1;
    case GL_TEXTURE_RESIDENT: return 1;
    case GL_TEXTURE_BINDING_1D: return 1;
    case GL_TEXTURE_BINDING_2D: return 1;
    case GL_TEXTURE_BINDING_3D: return 1;
    case GL_PACK_SKIP_IMAGES: return 1;
    case GL_PACK_IMAGE_HEIGHT: return 1;
    case GL_UNPACK_SKIP_IMAGES: return 1;
    case GL_UNPACK_IMAGE_HEIGHT: return 1;
    case GL_TEXTURE_3D: return 1;
    case GL_TEXTURE_DEPTH: return 1;
    case GL_TEXTURE_WRAP_R: return 1;
    case GL_MAX_3D_TEXTURE_SIZE: return 1;
    case GL_VERTEX_ARRAY: return 1;
    case GL_NORMAL_ARRAY: return 1;
    case GL_COLOR_ARRAY: return 1;
    case GL_INDEX_ARRAY: return 1;
    case GL_TEXTURE_COORD_ARRAY: return 1;
    case GL_EDGE_FLAG_ARRAY: return 1;
    case GL_VERTEX_ARRAY_SIZE: return 1;
    case GL_VERTEX_ARRAY_TYPE: return 1;
    case GL_VERTEX_ARRAY_STRIDE: return 1;
    case GL_VERTEX_ARRAY_COUNT_EXT: return 1;
    case GL_NORMAL_ARRAY_TYPE: return 1;
    case GL_NORMAL_ARRAY_STRIDE: return 1;
    case GL_NORMAL_ARRAY_COUNT_EXT: return 1;
    case GL_COLOR_ARRAY_SIZE: return 1;
    case GL_COLOR_ARRAY_TYPE: return 1;
    case GL_COLOR_ARRAY_STRIDE: return 1;
    case GL_COLOR_ARRAY_COUNT_EXT: return 1;
    case GL_INDEX_ARRAY_TYPE: return 1;
    case GL_INDEX_ARRAY_STRIDE: return 1;
    case GL_INDEX_ARRAY_COUNT_EXT: return 1;
    case GL_TEXTURE_COORD_ARRAY_SIZE: return 1;
    case GL_TEXTURE_COORD_ARRAY_TYPE: return 1;
    case GL_TEXTURE_COORD_ARRAY_STRIDE: return 1;
    case GL_TEXTURE_COORD_ARRAY_COUNT_EXT: return 1;
    case GL_EDGE_FLAG_ARRAY_STRIDE: return 1;
    case GL_EDGE_FLAG_ARRAY_COUNT_EXT: return 1;
    case GL_VERTEX_ARRAY_POINTER: return 1;
    case GL_NORMAL_ARRAY_POINTER: return 1;
    case GL_COLOR_ARRAY_POINTER: return 1;
    case GL_INDEX_ARRAY_POINTER: return 1;
    case GL_TEXTURE_COORD_ARRAY_POINTER: return 1;
    case GL_EDGE_FLAG_ARRAY_POINTER: return 1;
    case GL_INTERLACE_SGIX: return 1;
    case GL_DETAIL_TEXTURE_2D_BINDING_SGIS: return 1;
    case GL_MULTISAMPLE: return 1;
    case GL_SAMPLE_ALPHA_TO_COVERAGE: return 1;
    case GL_SAMPLE_ALPHA_TO_ONE: return 1;
    case GL_SAMPLE_COVERAGE: return 1;
    case GL_SAMPLE_BUFFERS: return 1;
    case GL_SAMPLES: return 1;
    case GL_SAMPLE_COVERAGE_VALUE: return 1;
    case GL_SAMPLE_COVERAGE_INVERT: return 1;
    case GL_SAMPLE_PATTERN_EXT: return 1;
    case GL_COLOR_MATRIX: return 16;
    case GL_COLOR_MATRIX_STACK_DEPTH: return 1;
    case GL_MAX_COLOR_MATRIX_STACK_DEPTH: return 1;
    case GL_POST_COLOR_MATRIX_RED_SCALE: return 1;
    case GL_POST_COLOR_MATRIX_GREEN_SCALE: return 1;
    case GL_POST_COLOR_MATRIX_BLUE_SCALE: return 1;
    case GL_POST_COLOR_MATRIX_ALPHA_SCALE: return 1;
    case GL_POST_COLOR_MATRIX_RED_BIAS: return 1;
    case GL_POST_COLOR_MATRIX_GREEN_BIAS: return 1;
    case GL_POST_COLOR_MATRIX_BLUE_BIAS: return 1;
    case GL_POST_COLOR_MATRIX_ALPHA_BIAS: return 1;
    case GL_TEXTURE_COLOR_TABLE_SGI: return 1;
    case GL_TEXTURE_COMPARE_FAIL_VALUE_ARB: return 1;
    case GL_BLEND_DST_RGB: return 1;
    case GL_BLEND_SRC_RGB: return 1;
    case GL_BLEND_DST_ALPHA: return 1;
    case GL_BLEND_SRC_ALPHA: return 1;
    case GL_COLOR_TABLE: return 1;
    case GL_POST_CONVOLUTION_COLOR_TABLE: return 1;
    case GL_POST_COLOR_MATRIX_COLOR_TABLE: return 1;
    case GL_COLOR_TABLE_SCALE: return 4;
    case GL_COLOR_TABLE_BIAS: return 4;
    case GL_COLOR_TABLE_FORMAT: return 1;
    case GL_COLOR_TABLE_WIDTH: return 1;
    case GL_COLOR_TABLE_RED_SIZE: return 1;
    case GL_COLOR_TABLE_GREEN_SIZE: return 1;
    case GL_COLOR_TABLE_BLUE_SIZE: return 1;
    case GL_COLOR_TABLE_ALPHA_SIZE: return 1;
    case GL_COLOR_TABLE_LUMINANCE_SIZE: return 1;
    case GL_COLOR_TABLE_INTENSITY_SIZE: return 1;
    case GL_MAX_ELEMENTS_VERTICES: return 1;
    case GL_MAX_ELEMENTS_INDICES: return 1;
    case GL_TEXTURE_INDEX_SIZE_EXT: return 1;
    case GL_CLIP_VOLUME_CLIPPING_HINT_EXT: return 1;
    case GL_POINT_SIZE_MIN: return 1;
    case GL_POINT_SIZE_MAX: return 1;
    case GL_POINT_FADE_THRESHOLD_SIZE: return 1;
    case GL_POINT_DISTANCE_ATTENUATION: return 3;
    case GL_FOG_FUNC_POINTS_SGIS: return 1;
    case GL_MAX_FOG_FUNC_POINTS_SGIS: return 1;
    case GL_PACK_SKIP_VOLUMES_SGIS: return 1;
    case GL_PACK_IMAGE_DEPTH_SGIS: return 1;
    case GL_UNPACK_SKIP_VOLUMES_SGIS: return 1;
    case GL_UNPACK_IMAGE_DEPTH_SGIS: return 1;
    case GL_TEXTURE_4D_SGIS: return 1;
    case GL_MAX_4D_TEXTURE_SIZE_SGIS: return 1;
    case GL_PIXEL_TEX_GEN_SGIX: return 1;
    case GL_TEXTURE_MIN_LOD: return 1;
    case GL_TEXTURE_MAX_LOD: return 1;
    case GL_TEXTURE_BASE_LEVEL: return 1;
    case GL_TEXTURE_MAX_LEVEL: return 1;
    case GL_PIXEL_TILE_BEST_ALIGNMENT_SGIX: return 1;
    case GL_PIXEL_TILE_CACHE_INCREMENT_SGIX: return 1;
    case GL_PIXEL_TILE_WIDTH_SGIX: return 1;
    case GL_PIXEL_TILE_HEIGHT_SGIX: return 1;
    case GL_PIXEL_TILE_GRID_WIDTH_SGIX: return 1;
    case GL_PIXEL_TILE_GRID_HEIGHT_SGIX: return 1;
    case GL_PIXEL_TILE_GRID_DEPTH_SGIX: return 1;
    case GL_PIXEL_TILE_CACHE_SIZE_SGIX: return 1;
    case GL_SPRITE_SGIX: return 1;
    case GL_SPRITE_MODE_SGIX: return 1;
    case GL_SPRITE_AXIS_SGIX: return 3;
    case GL_SPRITE_TRANSLATION_SGIX: return 3;
    case GL_TEXTURE_4D_BINDING_SGIS: return 1;
    case GL_CONVOLUTION_BORDER_COLOR: return 4;
    case GL_OCCLUSION_TEST_HP: return 1;
    case GL_OCCLUSION_TEST_RESULT_HP: return 1;
    case GL_TEXTURE_CLIPMAP_CENTER_SGIX: return 2;
    case GL_TEXTURE_CLIPMAP_FRAME_SGIX: return 1;
    case GL_TEXTURE_CLIPMAP_OFFSET_SGIX: return 2;
    case GL_TEXTURE_CLIPMAP_VIRTUAL_DEPTH_SGIX: return 3;
    case GL_TEXTURE_CLIPMAP_DEPTH_SGIX: return 1;
    case GL_MAX_CLIPMAP_DEPTH_SGIX: return 1;
    case GL_MAX_CLIPMAP_VIRTUAL_DEPTH_SGIX: return 1;
    case GL_POST_TEXTURE_FILTER_BIAS_SGIX: return 1;
    case GL_POST_TEXTURE_FILTER_SCALE_SGIX: return 1;
    case GL_POST_TEXTURE_FILTER_BIAS_RANGE_SGIX: return 2;
    case GL_POST_TEXTURE_FILTER_SCALE_RANGE_SGIX: return 2;
    case GL_REFERENCE_PLANE_SGIX: return 1;
    case GL_REFERENCE_PLANE_EQUATION_SGIX: return 4;
    case GL_IR_INSTRUMENT1_SGIX: return 1;
    case GL_INSTRUMENT_MEASUREMENTS_SGIX: return 1;
    case GL_CALLIGRAPHIC_FRAGMENT_SGIX: return 1;
    case GL_FRAMEZOOM_SGIX: return 1;
    case GL_FRAMEZOOM_FACTOR_SGIX: return 1;
    case GL_MAX_FRAMEZOOM_FACTOR_SGIX: return 1;
    case GL_TEXTURE_LOD_BIAS_S_SGIX: return 1;
    case GL_TEXTURE_LOD_BIAS_T_SGIX: return 1;
    case GL_TEXTURE_LOD_BIAS_R_SGIX: return 1;
    case GL_GENERATE_MIPMAP: return 1;
    case GL_GENERATE_MIPMAP_HINT: return 1;
    case GL_DEFORMATIONS_MASK_SGIX: return 1;
    case GL_FOG_OFFSET_SGIX: return 1;
    case GL_FOG_OFFSET_VALUE_SGIX: return 4;
    case GL_TEXTURE_COMPARE_SGIX: return 1;
    case GL_TEXTURE_COMPARE_OPERATOR_SGIX: return 1;
    case GL_ARRAY_ELEMENT_LOCK_FIRST_EXT: return 1;
    case GL_ARRAY_ELEMENT_LOCK_COUNT_EXT: return 1;
    case GL_CULL_VERTEX_EXT: return 1;
    case GL_CULL_VERTEX_EYE_POSITION_EXT: return 4;
    case GL_CULL_VERTEX_OBJECT_POSITION_EXT: return 4;
    case GL_TEXTURE_COLOR_WRITEMASK_SGIS: return 4;
    case GL_LIGHT_MODEL_COLOR_CONTROL: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_RED_SIZE: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_GREEN_SIZE: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_BLUE_SIZE: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE: return 1;
    case GL_MAJOR_VERSION: return 1;
    case GL_MINOR_VERSION: return 1;
    case GL_NUM_EXTENSIONS: return 1;
    case GL_CONTEXT_FLAGS: return 1;
    case GL_BUFFER_IMMUTABLE_STORAGE: return 1;
    case GL_BUFFER_STORAGE_FLAGS: return 1;
    case GL_PRIMITIVE_RESTART_FOR_PATCHES_SUPPORTED: return 1;
    case GL_DEBUG_OUTPUT_SYNCHRONOUS: return 1;
    case GL_DEBUG_NEXT_LOGGED_MESSAGE_LENGTH: return 1;
    case GL_DEBUG_CALLBACK_FUNCTION: return 1;
    case GL_DEBUG_CALLBACK_USER_PARAM: return 1;
    case GL_RESET_NOTIFICATION_STRATEGY: return 1;
    case GL_PROGRAM_BINARY_RETRIEVABLE_HINT: return 1;
    case GL_PROGRAM_SEPARABLE: return 1;
    case GL_ACTIVE_PROGRAM: return 1;
    case GL_PROGRAM_PIPELINE_BINDING: return 1;
    case GL_MAX_VIEWPORTS: return 1;
    case GL_VIEWPORT_SUBPIXEL_BITS: return 1;
    case GL_VIEWPORT_BOUNDS_RANGE: return 2;
    case GL_LAYER_PROVOKING_VERTEX: return 1;
    case GL_VIEWPORT_INDEX_PROVOKING_VERTEX: return 1;
    case GL_MAX_COMPUTE_SHARED_MEMORY_SIZE: return 1;
    case GL_MAX_COMPUTE_UNIFORM_COMPONENTS: return 1;
    case GL_MAX_COMPUTE_ATOMIC_COUNTER_BUFFERS: return 1;
    case GL_MAX_COMPUTE_ATOMIC_COUNTERS: return 1;
    case GL_MAX_COMBINED_COMPUTE_UNIFORM_COMPONENTS: return 1;
    case GL_COMPUTE_WORK_GROUP_SIZE: return 3;
    case GL_MAX_DEBUG_GROUP_STACK_DEPTH: return 1;
    case GL_DEBUG_GROUP_STACK_DEPTH: return 1;
    case GL_MAX_UNIFORM_LOCATIONS: return 1;
    case GL_INTERNALFORMAT_SUPPORTED: return 1;
    case GL_INTERNALFORMAT_PREFERRED: return 1;
    case GL_INTERNALFORMAT_RED_SIZE: return 1;
    case GL_INTERNALFORMAT_GREEN_SIZE: return 1;
    case GL_INTERNALFORMAT_BLUE_SIZE: return 1;
    case GL_INTERNALFORMAT_ALPHA_SIZE: return 1;
    case GL_INTERNALFORMAT_DEPTH_SIZE: return 1;
    case GL_INTERNALFORMAT_STENCIL_SIZE: return 1;
    case GL_INTERNALFORMAT_SHARED_SIZE: return 1;
    case GL_INTERNALFORMAT_RED_TYPE: return 1;
    case GL_INTERNALFORMAT_GREEN_TYPE: return 1;
    case GL_INTERNALFORMAT_BLUE_TYPE: return 1;
    case GL_INTERNALFORMAT_ALPHA_TYPE: return 1;
    case GL_INTERNALFORMAT_DEPTH_TYPE: return 1;
    case GL_INTERNALFORMAT_STENCIL_TYPE: return 1;
    case GL_MAX_WIDTH: return 1;
    case GL_MAX_HEIGHT: return 1;
    case GL_MAX_DEPTH: return 1;
    case GL_MAX_LAYERS: return 1;
    case GL_MAX_COMBINED_DIMENSIONS: return 1;
    case GL_COLOR_COMPONENTS: return 1;
    case GL_DEPTH_COMPONENTS: return 1;
    case GL_STENCIL_COMPONENTS: return 1;
    case GL_COLOR_RENDERABLE: return 1;
    case GL_DEPTH_RENDERABLE: return 1;
    case GL_STENCIL_RENDERABLE: return 1;
    case GL_FRAMEBUFFER_RENDERABLE: return 1;
    case GL_FRAMEBUFFER_RENDERABLE_LAYERED: return 1;
    case GL_FRAMEBUFFER_BLEND: return 1;
    case GL_READ_PIXELS: return 1;
    case GL_READ_PIXELS_FORMAT: return 1;
    case GL_READ_PIXELS_TYPE: return 1;
    case GL_TEXTURE_IMAGE_FORMAT: return 1;
    case GL_TEXTURE_IMAGE_TYPE: return 1;
    case GL_GET_TEXTURE_IMAGE_FORMAT: return 1;
    case GL_GET_TEXTURE_IMAGE_TYPE: return 1;
    case GL_MIPMAP: return 1;
    case GL_MANUAL_GENERATE_MIPMAP: return 1;
    case GL_AUTO_GENERATE_MIPMAP: return 1;
    case GL_COLOR_ENCODING: return 1;
    case GL_SRGB_READ: return 1;
    case GL_SRGB_WRITE: return 1;
    case GL_SRGB_DECODE_ARB: return 1;
    case GL_FILTER: return 1;
    case GL_VERTEX_TEXTURE: return 1;
    case GL_TESS_CONTROL_TEXTURE: return 1;
    case GL_TESS_EVALUATION_TEXTURE: return 1;
    case GL_GEOMETRY_TEXTURE: return 1;
    case GL_FRAGMENT_TEXTURE: return 1;
    case GL_COMPUTE_TEXTURE: return 1;
    case GL_TEXTURE_SHADOW: return 1;
    case GL_TEXTURE_GATHER: return 1;
    case GL_TEXTURE_GATHER_SHADOW: return 1;
    case GL_SHADER_IMAGE_LOAD: return 1;
    case GL_SHADER_IMAGE_STORE: return 1;
    case GL_SHADER_IMAGE_ATOMIC: return 1;
    case GL_IMAGE_TEXEL_SIZE: return 1;
    case GL_IMAGE_COMPATIBILITY_CLASS: return 1;
    case GL_IMAGE_PIXEL_FORMAT: return 1;
    case GL_IMAGE_PIXEL_TYPE: return 1;
    case GL_SIMULTANEOUS_TEXTURE_AND_DEPTH_TEST: return 1;
    case GL_SIMULTANEOUS_TEXTURE_AND_STENCIL_TEST: return 1;
    case GL_SIMULTANEOUS_TEXTURE_AND_DEPTH_WRITE: return 1;
    case GL_SIMULTANEOUS_TEXTURE_AND_STENCIL_WRITE: return 1;
    case GL_TEXTURE_COMPRESSED_BLOCK_WIDTH: return 1;
    case GL_TEXTURE_COMPRESSED_BLOCK_HEIGHT: return 1;
    case GL_TEXTURE_COMPRESSED_BLOCK_SIZE: return 1;
    case GL_CLEAR_BUFFER: return 1;
    case GL_TEXTURE_VIEW: return 1;
    case GL_VIEW_COMPATIBILITY_CLASS: return 1;
    case GL_VERTEX_ATTRIB_BINDING: return 1;
    case GL_VERTEX_ATTRIB_RELATIVE_OFFSET: return 1;
    case GL_VERTEX_BINDING_DIVISOR: return 1;
    case GL_VERTEX_BINDING_OFFSET: return 1;
    case GL_VERTEX_BINDING_STRIDE: return 1;
    case GL_MAX_VERTEX_ATTRIB_RELATIVE_OFFSET: return 1;
    case GL_MAX_VERTEX_ATTRIB_BINDINGS: return 1;
    case GL_TEXTURE_VIEW_MIN_LEVEL: return 1;
    case GL_TEXTURE_VIEW_NUM_LEVELS: return 1;
    case GL_TEXTURE_VIEW_MIN_LAYER: return 1;
    case GL_TEXTURE_VIEW_NUM_LAYERS: return 1;
    case GL_TEXTURE_IMMUTABLE_LEVELS: return 1;
    case GL_MAX_VERTEX_ATTRIB_STRIDE: return 1;
    case GL_MAX_LABEL_LENGTH: return 1;
    case GL_SPARSE_BUFFER_PAGE_SIZE_ARB: return 1;
    case GL_MAX_CULL_DISTANCES: return 1;
    case GL_MAX_COMBINED_CLIP_AND_CULL_DISTANCES: return 1;
    case GL_CONTEXT_RELEASE_BEHAVIOR: return 1;
    case GL_CONTEXT_RELEASE_BEHAVIOR_FLUSH: return 1;
    case GL_CONVOLUTION_HINT_SGIX: return 1;
    case GL_PIXEL_TEX_GEN_MODE_SGIX: return 1;
    case GL_TEXTURE_APPLICATION_MODE_EXT: return 1;
    case GL_TEXTURE_LIGHT_EXT: return 1;
    case GL_TEXTURE_MATERIAL_FACE_EXT: return 1;
    case GL_TEXTURE_MATERIAL_PARAMETER_EXT: return 1;
    case GL_PIXEL_TEXTURE_SGIS: return 1;
    case GL_PIXEL_FRAGMENT_RGB_SOURCE_SGIS: return 1;
    case GL_PIXEL_FRAGMENT_ALPHA_SOURCE_SGIS: return 1;
    case GL_PIXEL_GROUP_COLOR_SGIS: return 1;
    case GL_TEXTURE_MAX_CLAMP_S_SGIX: return 1;
    case GL_TEXTURE_MAX_CLAMP_T_SGIX: return 1;
    case GL_TEXTURE_MAX_CLAMP_R_SGIX: return 1;
    case GL_TEXTURE_MEMORY_LAYOUT_INTEL: return 1;
    case GL_FRAGMENT_LIGHTING_SGIX: return 1;
    case GL_FRAGMENT_COLOR_MATERIAL_SGIX: return 1;
    case GL_FRAGMENT_COLOR_MATERIAL_FACE_SGIX: return 1;
    case GL_FRAGMENT_COLOR_MATERIAL_PARAMETER_SGIX: return 1;
    case GL_MAX_FRAGMENT_LIGHTS_SGIX: return 1;
    case GL_MAX_ACTIVE_LIGHTS_SGIX: return 1;
    case GL_CURRENT_RASTER_NORMAL_SGIX: return 1;
    case GL_LIGHT_ENV_MODE_SGIX: return 1;
    case GL_FRAGMENT_LIGHT_MODEL_LOCAL_VIEWER_SGIX: return 1;
    case GL_FRAGMENT_LIGHT_MODEL_TWO_SIDE_SGIX: return 1;
    case GL_FRAGMENT_LIGHT_MODEL_AMBIENT_SGIX: return 4;
    case GL_FRAGMENT_LIGHT_MODEL_NORMAL_INTERPOLATION_SGIX: return 1;
    case GL_FRAGMENT_LIGHT0_SGIX: return 1;
    case GL_TANGENT_ARRAY_TYPE_EXT: return 1;
    case GL_BINORMAL_ARRAY_TYPE_EXT: return 1;
    case GL_TANGENT_ARRAY_POINTER_EXT: return 1;
    case GL_BINORMAL_ARRAY_POINTER_EXT: return 1;
    case GL_FOG_COORD_SRC: return 1;
    case GL_CURRENT_FOG_COORD: return 1;
    case GL_FOG_COORD_ARRAY_TYPE: return 1;
    case GL_FOG_COORD_ARRAY_STRIDE: return 1;
    case GL_FOG_COORD_ARRAY: return 1;
    case GL_COLOR_SUM: return 1;
    case GL_CURRENT_SECONDARY_COLOR: return 4;
    case GL_SECONDARY_COLOR_ARRAY_SIZE: return 1;
    case GL_SECONDARY_COLOR_ARRAY_TYPE: return 1;
    case GL_SECONDARY_COLOR_ARRAY_STRIDE: return 1;
    case GL_SECONDARY_COLOR_ARRAY: return 1;
    case GL_ALIASED_POINT_SIZE_RANGE: return 2;
    case GL_ALIASED_LINE_WIDTH_RANGE: return 2;
    case GL_ACTIVE_TEXTURE: return 1;
    case GL_CLIENT_ACTIVE_TEXTURE: return 1;
    case GL_MAX_TEXTURE_UNITS: return 1;
    case GL_TRANSPOSE_MODELVIEW_MATRIX: return 16;
    case GL_TRANSPOSE_PROJECTION_MATRIX: return 16;
    case GL_TRANSPOSE_TEXTURE_MATRIX: return 16;
    case GL_TRANSPOSE_COLOR_MATRIX: return 16;
    case GL_MAX_RENDERBUFFER_SIZE: return 1;
    case GL_TEXTURE_COMPRESSION_HINT: return 1;
    case GL_UNIFORM_BLOCK_REFERENCED_BY_TESS_CONTROL_SHADER: return 1;
    case GL_UNIFORM_BLOCK_REFERENCED_BY_TESS_EVALUATION_SHADER: return 1;
    case GL_TEXTURE_RECTANGLE: return 1;
    case GL_TEXTURE_BINDING_RECTANGLE: return 1;
    case GL_MAX_RECTANGLE_TEXTURE_SIZE: return 1;
    case GL_MAX_TEXTURE_LOD_BIAS: return 1;
    case GL_TEXTURE_MAX_ANISOTROPY_EXT: return 1;
    case GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT: return 1;
    case GL_TEXTURE_LOD_BIAS: return 1;
    case GL_MAX_SHININESS_NV: return 1;
    case GL_MAX_SPOT_EXPONENT_NV: return 1;
    case GL_VERTEX_WEIGHT_ARRAY_TYPE_EXT: return 1;
    case GL_TEXTURE_CUBE_MAP: return 1;
    case GL_TEXTURE_BINDING_CUBE_MAP: return 1;
    case GL_MAX_CUBE_MAP_TEXTURE_SIZE: return 1;
    case GL_VERTEX_ARRAY_RANGE_NV: return 1;
    case GL_VERTEX_ARRAY_RANGE_LENGTH_NV: return 1;
    case GL_VERTEX_ARRAY_RANGE_VALID_NV: return 1;
    case GL_MAX_VERTEX_ARRAY_RANGE_ELEMENT_NV: return 1;
    case GL_VERTEX_ARRAY_RANGE_POINTER_NV: return 1;
    case GL_CONSTANT_COLOR0_NV: return 4;
    case GL_CONSTANT_COLOR1_NV: return 4;
    case GL_MULTISAMPLE_FILTER_HINT_NV: return 1;
    case GL_MAX_GENERAL_COMBINERS_NV: return 1;
    case GL_NUM_GENERAL_COMBINERS_NV: return 1;
    case GL_COLOR_SUM_CLAMP_NV: return 1;
    case GL_PRIMITIVE_RESTART_NV: return 1;
    case GL_PRIMITIVE_RESTART_INDEX_NV: return 1;
    case GL_FOG_DISTANCE_MODE_NV: return 1;
    case GL_COMBINE_RGB: return 1;
    case GL_COMBINE_ALPHA: return 1;
    case GL_RGB_SCALE: return 1;
    case GL_SRC0_RGB: return 1;
    case GL_SRC1_RGB: return 1;
    case GL_SRC2_RGB: return 1;
    case GL_SOURCE3_RGB_NV: return 1;
    case GL_SRC0_ALPHA: return 1;
    case GL_SRC1_ALPHA: return 1;
    case GL_SRC2_ALPHA: return 1;
    case GL_SOURCE3_ALPHA_NV: return 1;
    case GL_OPERAND0_RGB: return 1;
    case GL_OPERAND1_RGB: return 1;
    case GL_OPERAND2_RGB: return 1;
    case GL_OPERAND3_RGB_NV: return 1;
    case GL_OPERAND0_ALPHA: return 1;
    case GL_OPERAND1_ALPHA: return 1;
    case GL_OPERAND2_ALPHA: return 1;
    case GL_OPERAND3_ALPHA_NV: return 1;
    case GL_VERTEX_ARRAY_BINDING: return 1;
    case GL_TEXTURE_RANGE_LENGTH_APPLE: return 1;
    case GL_TEXTURE_RANGE_POINTER_APPLE: return 1;
    case GL_TEXTURE_STORAGE_HINT_APPLE: return 1;
    case GL_REPLACEMENT_CODE_ARRAY_TYPE_SUN: return 1;
    case GL_VERTEX_PROGRAM_ARB: return 1;
    case GL_VERTEX_ATTRIB_ARRAY_ENABLED: return 1;
    case GL_VERTEX_ATTRIB_ARRAY_SIZE: return 1;
    case GL_VERTEX_ATTRIB_ARRAY_STRIDE: return 1;
    case GL_VERTEX_ATTRIB_ARRAY_TYPE: return 1;
    case GL_CURRENT_VERTEX_ATTRIB: return 4;
    case GL_PROGRAM_LENGTH_ARB: return 1;
    case GL_PROGRAM_STRING_ARB: return 1;
    case GL_MAX_PROGRAM_MATRIX_STACK_DEPTH_ARB: return 1;
    case GL_MAX_PROGRAM_MATRICES_ARB: return 1;
    case GL_CURRENT_MATRIX_STACK_DEPTH_ARB: return 1;
    case GL_CURRENT_MATRIX_ARB: return 16;
    case GL_PROGRAM_POINT_SIZE: return 1;
    case GL_VERTEX_PROGRAM_TWO_SIDE: return 1;
    case GL_VERTEX_ATTRIB_ARRAY_POINTER: return 1;
    case GL_PROGRAM_TARGET_NV: return 1;
    case GL_PROGRAM_RESIDENT_NV: return 1;
    case GL_VERTEX_PROGRAM_BINDING_NV: return 1;
    case GL_PROGRAM_ERROR_POSITION_ARB: return 1;
    case GL_DEPTH_CLAMP: return 1;
    case GL_VERTEX_ATTRIB_ARRAY0_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY1_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY2_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY3_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY4_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY5_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY6_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY7_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY8_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY9_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY10_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY11_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY12_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY13_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY14_NV: return 1;
    case GL_VERTEX_ATTRIB_ARRAY15_NV: return 1;
    case GL_PROGRAM_BINDING_ARB: return 1;
    case GL_TEXTURE_COMPRESSED_IMAGE_SIZE: return 1;
    case GL_TEXTURE_COMPRESSED: return 1;
    case GL_NUM_COMPRESSED_TEXTURE_FORMATS: return 1;
    case GL_COMPRESSED_TEXTURE_FORMATS: return _glGetInteger(GL_NUM_COMPRESSED_TEXTURE_FORMATS);
    case GL_MAX_VERTEX_UNITS_ARB: return 1;
    case GL_ACTIVE_VERTEX_UNITS_ARB: return 1;
    case GL_WEIGHT_SUM_UNITY_ARB: return 1;
    case GL_VERTEX_BLEND_ARB: return 1;
    case GL_CURRENT_WEIGHT_ARB: return 1;
    case GL_WEIGHT_ARRAY_TYPE_ARB: return 1;
    case GL_WEIGHT_ARRAY_STRIDE_ARB: return 1;
    case GL_WEIGHT_ARRAY_SIZE_ARB: return 1;
    case GL_WEIGHT_ARRAY_POINTER_ARB: return 1;
    case GL_WEIGHT_ARRAY_ARB: return 1;
    case GL_RGBA_UNSIGNED_DOT_PRODUCT_MAPPING_NV: return 1;
    case GL_SHADER_OPERATION_NV: return 1;
    case GL_CULL_MODES_NV: return 4;
    case GL_OFFSET_TEXTURE_MATRIX_NV: return 4;
    case GL_OFFSET_TEXTURE_SCALE_NV: return 1;
    case GL_OFFSET_TEXTURE_BIAS_NV: return 1;
    case GL_PREVIOUS_TEXTURE_INPUT_NV: return 1;
    case GL_CONST_EYE_NV: return 3;
    case GL_MODELVIEW2_ARB: return 16;
    case GL_MODELVIEW3_ARB: return 16;
    case GL_MODELVIEW4_ARB: return 16;
    case GL_MODELVIEW5_ARB: return 16;
    case GL_MODELVIEW6_ARB: return 16;
    case GL_MODELVIEW7_ARB: return 16;
    case GL_MODELVIEW8_ARB: return 16;
    case GL_MODELVIEW9_ARB: return 16;
    case GL_MODELVIEW10_ARB: return 16;
    case GL_MODELVIEW11_ARB: return 16;
    case GL_MODELVIEW12_ARB: return 16;
    case GL_MODELVIEW13_ARB: return 16;
    case GL_MODELVIEW14_ARB: return 16;
    case GL_MODELVIEW15_ARB: return 16;
    case GL_MODELVIEW16_ARB: return 16;
    case GL_MODELVIEW17_ARB: return 16;
    case GL_MODELVIEW18_ARB: return 16;
    case GL_MODELVIEW19_ARB: return 16;
    case GL_MODELVIEW20_ARB: return 16;
    case GL_MODELVIEW21_ARB: return 16;
    case GL_MODELVIEW22_ARB: return 16;
    case GL_MODELVIEW23_ARB: return 16;
    case GL_MODELVIEW24_ARB: return 16;
    case GL_MODELVIEW25_ARB: return 16;
    case GL_MODELVIEW26_ARB: return 16;
    case GL_MODELVIEW27_ARB: return 16;
    case GL_MODELVIEW28_ARB: return 16;
    case GL_MODELVIEW29_ARB: return 16;
    case GL_MODELVIEW30_ARB: return 16;
    case GL_MODELVIEW31_ARB: return 16;
    case GL_PROGRAM_BINARY_LENGTH: return 1;
    case GL_VERTEX_ATTRIB_ARRAY_LONG: return 1;
    case GL_PACK_INVERT_MESA: return 1;
    case GL_BUFFER_SIZE: return 1;
    case GL_BUFFER_USAGE: return 1;
    case GL_ELEMENT_ARRAY_TYPE_ATI: return 1;
    case GL_BUMP_TARGET_ATI: return 1;
    case GL_VERTEX_SHADER_BINDING_EXT: return 1;
    case GL_VARIANT_ARRAY_TYPE_EXT: return 1;
    case GL_VBO_FREE_MEMORY_ATI: return 4;
    case GL_TEXTURE_FREE_MEMORY_ATI: return 4;
    case GL_RENDERBUFFER_FREE_MEMORY_ATI: return 4;
    case GL_NUM_PROGRAM_BINARY_FORMATS: return 1;
    case GL_PROGRAM_BINARY_FORMATS: return 0;
    case GL_STENCIL_BACK_FUNC: return 1;
    case GL_STENCIL_BACK_FAIL: return 1;
    case GL_STENCIL_BACK_PASS_DEPTH_FAIL: return 1;
    case GL_STENCIL_BACK_PASS_DEPTH_PASS: return 1;
    case GL_FRAGMENT_PROGRAM_ARB: return 1;
    case GL_PROGRAM_ALU_INSTRUCTIONS_ARB: return 1;
    case GL_PROGRAM_TEX_INSTRUCTIONS_ARB: return 1;
    case GL_PROGRAM_TEX_INDIRECTIONS_ARB: return 1;
    case GL_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB: return 1;
    case GL_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB: return 1;
    case GL_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB: return 1;
    case GL_MAX_PROGRAM_ALU_INSTRUCTIONS_ARB: return 1;
    case GL_MAX_PROGRAM_TEX_INSTRUCTIONS_ARB: return 1;
    case GL_MAX_PROGRAM_TEX_INDIRECTIONS_ARB: return 1;
    case GL_MAX_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB: return 1;
    case GL_MAX_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB: return 1;
    case GL_MAX_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB: return 1;
    case GL_RGBA_FLOAT_MODE_ARB: return 1;
    case GL_MAX_DRAW_BUFFERS: return 1;
    case GL_DRAW_BUFFER0: return 1;
    case GL_DRAW_BUFFER1: return 1;
    case GL_DRAW_BUFFER2: return 1;
    case GL_DRAW_BUFFER3: return 1;
    case GL_DRAW_BUFFER4: return 1;
    case GL_DRAW_BUFFER5: return 1;
    case GL_DRAW_BUFFER6: return 1;
    case GL_DRAW_BUFFER7: return 1;
    case GL_DRAW_BUFFER8: return 1;
    case GL_DRAW_BUFFER9: return 1;
    case GL_DRAW_BUFFER10: return 1;
    case GL_DRAW_BUFFER11: return 1;
    case GL_DRAW_BUFFER12: return 1;
    case GL_DRAW_BUFFER13: return 1;
    case GL_DRAW_BUFFER14: return 1;
    case GL_DRAW_BUFFER15: return 1;
    case GL_BLEND_EQUATION_ALPHA: return 1;
    case GL_MATRIX_PALETTE_ARB: return 1;
    case GL_MAX_MATRIX_PALETTE_STACK_DEPTH_ARB: return 1;
    case GL_MAX_PALETTE_MATRICES_ARB: return 1;
    case GL_CURRENT_PALETTE_MATRIX_ARB: return 1;
    case GL_MATRIX_INDEX_ARRAY_ARB: return 1;
    case GL_CURRENT_MATRIX_INDEX_ARB: return 1;
    case GL_MATRIX_INDEX_ARRAY_SIZE_ARB: return 1;
    case GL_MATRIX_INDEX_ARRAY_TYPE_ARB: return 1;
    case GL_MATRIX_INDEX_ARRAY_STRIDE_ARB: return 1;
    case GL_MATRIX_INDEX_ARRAY_POINTER_ARB: return 1;
    case GL_TEXTURE_DEPTH_SIZE: return 1;
    case GL_DEPTH_TEXTURE_MODE: return 1;
    case GL_TEXTURE_COMPARE_MODE: return 1;
    case GL_TEXTURE_COMPARE_FUNC: return 1;
    case GL_TEXTURE_CUBE_MAP_SEAMLESS: return 1;
    case GL_POINT_SPRITE: return 1;
    case GL_COORD_REPLACE: return 1;
    case GL_POINT_SPRITE_R_MODE_NV: return 1;
    case GL_QUERY_COUNTER_BITS: return 1;
    case GL_CURRENT_QUERY: return 1;
    case GL_QUERY_RESULT: return 1;
    case GL_QUERY_RESULT_AVAILABLE: return 1;
    case GL_MAX_FRAGMENT_PROGRAM_LOCAL_PARAMETERS_NV: return 1;
    case GL_MAX_VERTEX_ATTRIBS: return 1;
    case GL_VERTEX_ATTRIB_ARRAY_NORMALIZED: return 1;
    case GL_MAX_TESS_CONTROL_INPUT_COMPONENTS: return 1;
    case GL_MAX_TESS_EVALUATION_INPUT_COMPONENTS: return 1;
    case GL_MAX_TEXTURE_COORDS: return 1;
    case GL_MAX_TEXTURE_IMAGE_UNITS: return 1;
    case GL_FRAGMENT_PROGRAM_BINDING_NV: return 1;
    case GL_PROGRAM_ERROR_STRING_ARB: return 1;
    case GL_PROGRAM_FORMAT_ARB: return 1;
    case GL_TEXTURE_UNSIGNED_REMAP_MODE_NV: return 1;
    case GL_DEPTH_BOUNDS_TEST_EXT: return 1;
    case GL_DEPTH_BOUNDS_EXT: return 2;
    case GL_ARRAY_BUFFER_BINDING: return 1;
    case GL_ELEMENT_ARRAY_BUFFER_BINDING: return 1;
    case GL_VERTEX_ARRAY_BUFFER_BINDING: return 1;
    case GL_NORMAL_ARRAY_BUFFER_BINDING: return 1;
    case GL_COLOR_ARRAY_BUFFER_BINDING: return 1;
    case GL_INDEX_ARRAY_BUFFER_BINDING: return 1;
    case GL_TEXTURE_COORD_ARRAY_BUFFER_BINDING: return 1;
    case GL_EDGE_FLAG_ARRAY_BUFFER_BINDING: return 1;
    case GL_SECONDARY_COLOR_ARRAY_BUFFER_BINDING: return 1;
    case GL_FOG_COORD_ARRAY_BUFFER_BINDING: return 1;
    case GL_WEIGHT_ARRAY_BUFFER_BINDING: return 1;
    case GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING: return 1;
    case GL_PROGRAM_INSTRUCTIONS_ARB: return 1;
    case GL_MAX_PROGRAM_INSTRUCTIONS_ARB: return 1;
    case GL_PROGRAM_NATIVE_INSTRUCTIONS_ARB: return 1;
    case GL_MAX_PROGRAM_NATIVE_INSTRUCTIONS_ARB: return 1;
    case GL_PROGRAM_TEMPORARIES_ARB: return 1;
    case GL_MAX_PROGRAM_TEMPORARIES_ARB: return 1;
    case GL_PROGRAM_NATIVE_TEMPORARIES_ARB: return 1;
    case GL_MAX_PROGRAM_NATIVE_TEMPORARIES_ARB: return 1;
    case GL_PROGRAM_PARAMETERS_ARB: return 1;
    case GL_MAX_PROGRAM_PARAMETERS_ARB: return 1;
    case GL_PROGRAM_NATIVE_PARAMETERS_ARB: return 1;
    case GL_MAX_PROGRAM_NATIVE_PARAMETERS_ARB: return 1;
    case GL_PROGRAM_ATTRIBS_ARB: return 1;
    case GL_MAX_PROGRAM_ATTRIBS_ARB: return 1;
    case GL_PROGRAM_NATIVE_ATTRIBS_ARB: return 1;
    case GL_MAX_PROGRAM_NATIVE_ATTRIBS_ARB: return 1;
    case GL_PROGRAM_ADDRESS_REGISTERS_ARB: return 1;
    case GL_MAX_PROGRAM_ADDRESS_REGISTERS_ARB: return 1;
    case GL_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB: return 1;
    case GL_MAX_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB: return 1;
    case GL_MAX_PROGRAM_LOCAL_PARAMETERS_ARB: return 1;
    case GL_MAX_PROGRAM_ENV_PARAMETERS_ARB: return 1;
    case GL_PROGRAM_UNDER_NATIVE_LIMITS_ARB: return 1;
    case GL_TRANSPOSE_CURRENT_MATRIX_ARB: return 16;
    case GL_BUFFER_ACCESS: return 1;
    case GL_BUFFER_MAPPED: return 1;
    case GL_BUFFER_MAP_POINTER: return 1;
    case GL_PIXEL_PACK_BUFFER_BINDING: return 1;
    case GL_PIXEL_UNPACK_BUFFER_BINDING: return 1;
    case GL_TEXTURE_STENCIL_SIZE: return 1;
    case GL_MAX_PROGRAM_EXEC_INSTRUCTIONS_NV: return 1;
    case GL_MAX_PROGRAM_CALL_DEPTH_NV: return 1;
    case GL_MAX_PROGRAM_IF_DEPTH_NV: return 1;
    case GL_MAX_PROGRAM_LOOP_DEPTH_NV: return 1;
    case GL_MAX_PROGRAM_LOOP_COUNT_NV: return 1;
    case GL_MAX_DUAL_SOURCE_DRAW_BUFFERS: return 1;
    case GL_VERTEX_ATTRIB_ARRAY_INTEGER: return 1;
    case GL_VERTEX_ATTRIB_ARRAY_DIVISOR: return 1;
    case GL_MAX_ARRAY_TEXTURE_LAYERS: return 1;
    case GL_MIN_PROGRAM_TEXEL_OFFSET: return 1;
    case GL_MAX_PROGRAM_TEXEL_OFFSET: return 1;
    case GL_MAX_PROGRAM_ATTRIB_COMPONENTS_NV: return 1;
    case GL_MAX_PROGRAM_RESULT_COMPONENTS_NV: return 1;
    case GL_STENCIL_TEST_TWO_SIDE_EXT: return 1;
    case GL_ACTIVE_STENCIL_FACE_EXT: return 1;
    case GL_GEOMETRY_VERTICES_OUT: return 1;
    case GL_GEOMETRY_INPUT_TYPE: return 1;
    case GL_GEOMETRY_OUTPUT_TYPE: return 1;
    case GL_SAMPLER_BINDING: return 1;
    case GL_CLAMP_VERTEX_COLOR: return 1;
    case GL_CLAMP_FRAGMENT_COLOR: return 1;
    case GL_CLAMP_READ_COLOR: return 1;
    case GL_ELEMENT_ARRAY_TYPE_APPLE: return 1;
    case GL_PURGEABLE_APPLE: return 1;
    case GL_UNIFORM_BUFFER_BINDING: return 1;
    case GL_UNIFORM_BUFFER_START: return 1;
    case GL_UNIFORM_BUFFER_SIZE: return 1;
    case GL_MAX_VERTEX_UNIFORM_BLOCKS: return 1;
    case GL_MAX_GEOMETRY_UNIFORM_BLOCKS: return 1;
    case GL_MAX_FRAGMENT_UNIFORM_BLOCKS: return 1;
    case GL_MAX_COMBINED_UNIFORM_BLOCKS: return 1;
    case GL_MAX_UNIFORM_BUFFER_BINDINGS: return 1;
    case GL_MAX_UNIFORM_BLOCK_SIZE: return 1;
    case GL_MAX_COMBINED_VERTEX_UNIFORM_COMPONENTS: return 1;
    case GL_MAX_COMBINED_GEOMETRY_UNIFORM_COMPONENTS: return 1;
    case GL_MAX_COMBINED_FRAGMENT_UNIFORM_COMPONENTS: return 1;
    case GL_UNIFORM_BUFFER_OFFSET_ALIGNMENT: return 1;
    case GL_ACTIVE_UNIFORM_BLOCK_MAX_NAME_LENGTH: return 1;
    case GL_ACTIVE_UNIFORM_BLOCKS: return 1;
    case GL_UNIFORM_TYPE: return 1;
    case GL_UNIFORM_SIZE: return 1;
    case GL_UNIFORM_NAME_LENGTH: return 1;
    case GL_UNIFORM_BLOCK_INDEX: return 1;
    case GL_UNIFORM_OFFSET: return 1;
    case GL_UNIFORM_ARRAY_STRIDE: return 1;
    case GL_UNIFORM_MATRIX_STRIDE: return 1;
    case GL_UNIFORM_IS_ROW_MAJOR: return 1;
    case GL_UNIFORM_BLOCK_BINDING: return 1;
    case GL_UNIFORM_BLOCK_DATA_SIZE: return 1;
    case GL_UNIFORM_BLOCK_NAME_LENGTH: return 1;
    case GL_UNIFORM_BLOCK_ACTIVE_UNIFORMS: return 1;
    case GL_UNIFORM_BLOCK_ACTIVE_UNIFORM_INDICES: return 1;
    case GL_UNIFORM_BLOCK_REFERENCED_BY_VERTEX_SHADER: return 1;
    case GL_UNIFORM_BLOCK_REFERENCED_BY_GEOMETRY_SHADER: return 1;
    case GL_UNIFORM_BLOCK_REFERENCED_BY_FRAGMENT_SHADER: return 1;
    case GL_TEXTURE_SRGB_DECODE_EXT: return 1;
    case GL_FRAGMENT_SHADER: return 1;
    case GL_VERTEX_SHADER: return 1;
    case GL_PROGRAM_OBJECT_ARB: return 1;
    case GL_MAX_FRAGMENT_UNIFORM_COMPONENTS: return 1;
    case GL_MAX_VERTEX_UNIFORM_COMPONENTS: return 1;
    case GL_MAX_VARYING_COMPONENTS: return 1;
    case GL_MAX_VERTEX_TEXTURE_IMAGE_UNITS: return 1;
    case GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS: return 1;
    case GL_OBJECT_TYPE_ARB: return 1;
    case GL_SHADER_TYPE: return 1;
    case GL_DELETE_STATUS: return 1;
    case GL_COMPILE_STATUS: return 1;
    case GL_LINK_STATUS: return 1;
    case GL_VALIDATE_STATUS: return 1;
    case GL_INFO_LOG_LENGTH: return 1;
    case GL_ATTACHED_SHADERS: return 1;
    case GL_ACTIVE_UNIFORMS: return 1;
    case GL_ACTIVE_UNIFORM_MAX_LENGTH: return 1;
    case GL_SHADER_SOURCE_LENGTH: return 1;
    case GL_ACTIVE_ATTRIBUTES: return 1;
    case GL_ACTIVE_ATTRIBUTE_MAX_LENGTH: return 1;
    case GL_SHADING_LANGUAGE_VERSION: return 1;
    case GL_CURRENT_PROGRAM: return 1;
    case GL_IMPLEMENTATION_COLOR_READ_TYPE: return 1;
    case GL_IMPLEMENTATION_COLOR_READ_FORMAT: return 1;
    case GL_TEXTURE_CROP_RECT_OES: return 4;
    case GL_MATRIX_INDEX_ARRAY_BUFFER_BINDING_OES: return 1;
    case GL_POINT_SIZE_ARRAY_BUFFER_BINDING_OES: return 1;
    case GL_TEXTURE_RED_TYPE: return 1;
    case GL_TEXTURE_GREEN_TYPE: return 1;
    case GL_TEXTURE_BLUE_TYPE: return 1;
    case GL_TEXTURE_ALPHA_TYPE: return 1;
    case GL_TEXTURE_LUMINANCE_TYPE: return 1;
    case GL_TEXTURE_INTENSITY_TYPE: return 1;
    case GL_TEXTURE_DEPTH_TYPE: return 1;
    case GL_TEXTURE_1D_ARRAY: return 1;
    case GL_TEXTURE_2D_ARRAY: return 1;
    case GL_TEXTURE_BINDING_1D_ARRAY: return 1;
    case GL_TEXTURE_BINDING_2D_ARRAY: return 1;
    case GL_MAX_PROGRAM_OUTPUT_VERTICES_NV: return 1;
    case GL_MAX_PROGRAM_TOTAL_OUTPUT_COMPONENTS_NV: return 1;
    case GL_MAX_GEOMETRY_TEXTURE_IMAGE_UNITS: return 1;
    case GL_TEXTURE_BUFFER: return 1;
    case GL_MAX_TEXTURE_BUFFER_SIZE: return 1;
    case GL_TEXTURE_BINDING_BUFFER: return 1;
    case GL_TEXTURE_BUFFER_DATA_STORE_BINDING: return 1;
    case GL_TEXTURE_BUFFER_FORMAT_ARB: return 1;
    case GL_ANY_SAMPLES_PASSED: return 1;
    case GL_SAMPLE_SHADING: return 1;
    case GL_MIN_SAMPLE_SHADING_VALUE: return 1;
    case GL_TRANSFORM_FEEDBACK_VARYING_MAX_LENGTH: return 1;
    case GL_TRANSFORM_FEEDBACK_BUFFER_MODE: return 1;
    case GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_COMPONENTS: return 1;
    case GL_TRANSFORM_FEEDBACK_VARYINGS: return 1;
    case GL_TRANSFORM_FEEDBACK_BUFFER_START: return 1;
    case GL_TRANSFORM_FEEDBACK_BUFFER_SIZE: return 1;
    case GL_RASTERIZER_DISCARD: return 1;
    case GL_MAX_TRANSFORM_FEEDBACK_INTERLEAVED_COMPONENTS: return 1;
    case GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_ATTRIBS: return 1;
    case GL_TRANSFORM_FEEDBACK_BUFFER_BINDING: return 1;
    case GL_POINT_SPRITE_COORD_ORIGIN: return 1;
    case GL_DRAW_FRAMEBUFFER_BINDING: return 1;
    case GL_RENDERBUFFER_BINDING: return 1;
    case GL_READ_FRAMEBUFFER: return 1;
    case GL_DRAW_FRAMEBUFFER: return 1;
    case GL_READ_FRAMEBUFFER_BINDING: return 1;
    case GL_RENDERBUFFER_SAMPLES: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER: return 1;
    case GL_MAX_COLOR_ATTACHMENTS: return 1;
    case GL_RENDERBUFFER_WIDTH: return 1;
    case GL_RENDERBUFFER_HEIGHT: return 1;
    case GL_RENDERBUFFER_INTERNAL_FORMAT: return 1;
    case GL_RENDERBUFFER_RED_SIZE: return 1;
    case GL_RENDERBUFFER_GREEN_SIZE: return 1;
    case GL_RENDERBUFFER_BLUE_SIZE: return 1;
    case GL_RENDERBUFFER_ALPHA_SIZE: return 1;
    case GL_RENDERBUFFER_DEPTH_SIZE: return 1;
    case GL_RENDERBUFFER_STENCIL_SIZE: return 1;
    case GL_MAX_SAMPLES: return 1;
    case GL_PRIMITIVE_RESTART_FIXED_INDEX: return 1;
    case GL_ANY_SAMPLES_PASSED_CONSERVATIVE: return 1;
    case GL_MAX_ELEMENT_INDEX: return 1;
    case GL_RGBA_INTEGER_MODE_EXT: return 1;
    case GL_MAX_PROGRAM_PARAMETER_BUFFER_BINDINGS_NV: return 1;
    case GL_MAX_PROGRAM_PARAMETER_BUFFER_SIZE_NV: return 1;
    case GL_MAX_PROGRAM_GENERIC_ATTRIBS_NV: return 1;
    case GL_MAX_PROGRAM_GENERIC_RESULTS_NV: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_LAYERED: return 1;
    case GL_FRAMEBUFFER_SRGB: return 1;
    case GL_FRAMEBUFFER_SRGB_CAPABLE_EXT: return 1;
    case GL_GEOMETRY_SHADER: return 1;
    case GL_GEOMETRY_VERTICES_OUT_ARB: return 1;
    case GL_GEOMETRY_INPUT_TYPE_ARB: return 1;
    case GL_GEOMETRY_OUTPUT_TYPE_ARB: return 1;
    case GL_MAX_GEOMETRY_VARYING_COMPONENTS_ARB: return 1;
    case GL_MAX_VERTEX_VARYING_COMPONENTS_ARB: return 1;
    case GL_MAX_GEOMETRY_UNIFORM_COMPONENTS: return 1;
    case GL_MAX_GEOMETRY_OUTPUT_VERTICES: return 1;
    case GL_MAX_GEOMETRY_TOTAL_OUTPUT_COMPONENTS: return 1;
    case GL_MAX_VERTEX_BINDABLE_UNIFORMS_EXT: return 1;
    case GL_MAX_FRAGMENT_BINDABLE_UNIFORMS_EXT: return 1;
    case GL_MAX_GEOMETRY_BINDABLE_UNIFORMS_EXT: return 1;
    case GL_ACTIVE_SUBROUTINES: return 1;
    case GL_ACTIVE_SUBROUTINE_UNIFORMS: return 1;
    case GL_MAX_SUBROUTINES: return 1;
    case GL_MAX_SUBROUTINE_UNIFORM_LOCATIONS: return 1;
    case GL_NAMED_STRING_LENGTH_ARB: return 1;
    case GL_NAMED_STRING_TYPE_ARB: return 1;
    case GL_MAX_BINDABLE_UNIFORM_SIZE_EXT: return 1;
    case GL_UNIFORM_BUFFER_BINDING_EXT: return 1;
    case GL_SHADER_BINARY_FORMATS: return _glGetInteger(GL_NUM_SHADER_BINARY_FORMATS);
    case GL_NUM_SHADER_BINARY_FORMATS: return 1;
    case GL_SHADER_COMPILER: return 1;
    case GL_MAX_VERTEX_UNIFORM_VECTORS: return 1;
    case GL_MAX_VARYING_VECTORS: return 1;
    case GL_MAX_FRAGMENT_UNIFORM_VECTORS: return 1;
    case GL_MAX_MULTISAMPLE_COVERAGE_MODES_NV: return 1;
    case GL_MAX_COMBINED_TESS_CONTROL_UNIFORM_COMPONENTS: return 1;
    case GL_MAX_COMBINED_TESS_EVALUATION_UNIFORM_COMPONENTS: return 1;
    case GL_TRANSFORM_FEEDBACK_BUFFER_PAUSED: return 1;
    case GL_TRANSFORM_FEEDBACK_BUFFER_ACTIVE: return 1;
    case GL_TRANSFORM_FEEDBACK_BINDING: return 1;
    case GL_TIMESTAMP: return 1;
    case GL_TEXTURE_SWIZZLE_R: return 1;
    case GL_TEXTURE_SWIZZLE_G: return 1;
    case GL_TEXTURE_SWIZZLE_B: return 1;
    case GL_TEXTURE_SWIZZLE_A: return 1;
    case GL_TEXTURE_SWIZZLE_RGBA: return 4;
    case GL_ACTIVE_SUBROUTINE_UNIFORM_LOCATIONS: return 1;
    case GL_ACTIVE_SUBROUTINE_MAX_LENGTH: return 1;
    case GL_ACTIVE_SUBROUTINE_UNIFORM_MAX_LENGTH: return 1;
    case GL_NUM_COMPATIBLE_SUBROUTINES: return 1;
    case GL_COMPATIBLE_SUBROUTINES: return 1;
    case GL_QUADS_FOLLOW_PROVOKING_VERTEX_CONVENTION: return 1;
    case GL_PROVOKING_VERTEX: return 1;
    case GL_SAMPLE_POSITION: return 2;
    case GL_SAMPLE_MASK: return 1;
    case GL_SAMPLE_MASK_VALUE: return 1;
    case GL_TEXTURE_RENDERBUFFER_DATA_STORE_BINDING_NV: return 1;
    case GL_MAX_SAMPLE_MASK_WORDS: return 1;
    case GL_MAX_GEOMETRY_SHADER_INVOCATIONS: return 1;
    case GL_MIN_FRAGMENT_INTERPOLATION_OFFSET: return 1;
    case GL_MAX_FRAGMENT_INTERPOLATION_OFFSET: return 1;
    case GL_FRAGMENT_INTERPOLATION_OFFSET_BITS: return 1;
    case GL_MIN_PROGRAM_TEXTURE_GATHER_OFFSET: return 1;
    case GL_MAX_PROGRAM_TEXTURE_GATHER_OFFSET: return 1;
    case GL_MAX_TRANSFORM_FEEDBACK_BUFFERS: return 1;
    case GL_MAX_VERTEX_STREAMS: return 1;
    case GL_PATCH_VERTICES: return 1;
    case GL_PATCH_DEFAULT_INNER_LEVEL: return 2;
    case GL_PATCH_DEFAULT_OUTER_LEVEL: return 4;
    case GL_TESS_CONTROL_OUTPUT_VERTICES: return 1;
    case GL_TESS_GEN_MODE: return 1;
    case GL_TESS_GEN_SPACING: return 1;
    case GL_TESS_GEN_VERTEX_ORDER: return 1;
    case GL_TESS_GEN_POINT_MODE: return 1;
    case GL_MAX_PATCH_VERTICES: return 1;
    case GL_MAX_TESS_GEN_LEVEL: return 1;
    case GL_MAX_TESS_CONTROL_UNIFORM_COMPONENTS: return 1;
    case GL_MAX_TESS_EVALUATION_UNIFORM_COMPONENTS: return 1;
    case GL_MAX_TESS_CONTROL_TEXTURE_IMAGE_UNITS: return 1;
    case GL_MAX_TESS_EVALUATION_TEXTURE_IMAGE_UNITS: return 1;
    case GL_MAX_TESS_CONTROL_OUTPUT_COMPONENTS: return 1;
    case GL_MAX_TESS_PATCH_COMPONENTS: return 1;
    case GL_MAX_TESS_CONTROL_TOTAL_OUTPUT_COMPONENTS: return 1;
    case GL_MAX_TESS_EVALUATION_OUTPUT_COMPONENTS: return 1;
    case GL_TESS_EVALUATION_SHADER: return 1;
    case GL_TESS_CONTROL_SHADER: return 1;
    case GL_MAX_TESS_CONTROL_UNIFORM_BLOCKS: return 1;
    case GL_MAX_TESS_EVALUATION_UNIFORM_BLOCKS: return 1;
    case GL_WINDOW_RECTANGLE_EXT: return 4;
    case GL_WINDOW_RECTANGLE_MODE_EXT: return 1;
    case GL_MAX_WINDOW_RECTANGLES_EXT: return 1;
    case GL_NUM_WINDOW_RECTANGLES_EXT: return 1;
    case GL_BUFFER_GPU_ADDRESS_NV: return 1;
    case GL_MAX_SHADER_BUFFER_ADDRESS_NV: return 1;
    case GL_COPY_READ_BUFFER: return 1;
    case GL_COPY_WRITE_BUFFER: return 1;
    case GL_MAX_IMAGE_UNITS: return 1;
    case GL_MAX_COMBINED_IMAGE_UNITS_AND_FRAGMENT_OUTPUTS: return 1;
    case GL_IMAGE_BINDING_NAME: return 1;
    case GL_IMAGE_BINDING_LEVEL: return 1;
    case GL_IMAGE_BINDING_LAYERED: return 1;
    case GL_IMAGE_BINDING_LAYER: return 1;
    case GL_IMAGE_BINDING_ACCESS: return 1;
    case GL_DRAW_INDIRECT_BUFFER_BINDING: return 1;
    case GL_TEXTURE_ASTC_DECODE_PRECISION_EXT: return 1;
    case GL_PRIMITIVE_RESTART: return 1;
    case GL_PRIMITIVE_RESTART_INDEX: return 1;
    case GL_MAX_PROGRAM_TEXTURE_GATHER_COMPONENTS_ARB: return 1;
    case GL_TEXTURE_CUBE_MAP_ARRAY: return 1;
    case GL_TEXTURE_BINDING_CUBE_MAP_ARRAY: return 1;
    case GL_DEPTH_CLAMP_NEAR_AMD: return 1;
    case GL_DEPTH_CLAMP_FAR_AMD: return 1;
    case GL_VIDEO_BUFFER_BINDING_NV: return 1;
    case GL_GPU_MEMORY_INFO_DEDICATED_VIDMEM_NVX: return 1;
    case GL_GPU_MEMORY_INFO_TOTAL_AVAILABLE_MEMORY_NVX: return 1;
    case GL_GPU_MEMORY_INFO_CURRENT_AVAILABLE_VIDMEM_NVX: return 1;
    case GL_GPU_MEMORY_INFO_EVICTION_COUNT_NVX: return 1;
    case GL_GPU_MEMORY_INFO_EVICTED_MEMORY_NVX: return 1;
    case GL_MAX_IMAGE_SAMPLES: return 1;
    case GL_IMAGE_BINDING_FORMAT: return 1;
    case GL_MIN_MAP_BUFFER_ALIGNMENT: return 1;
    case GL_MAX_VERTEX_IMAGE_UNIFORMS: return 1;
    case GL_MAX_TESS_CONTROL_IMAGE_UNIFORMS: return 1;
    case GL_MAX_TESS_EVALUATION_IMAGE_UNIFORMS: return 1;
    case GL_MAX_GEOMETRY_IMAGE_UNIFORMS: return 1;
    case GL_MAX_FRAGMENT_IMAGE_UNIFORMS: return 1;
    case GL_MAX_COMBINED_IMAGE_UNIFORMS: return 1;
    case GL_SHADER_STORAGE_BUFFER: return 1;
    case GL_SHADER_STORAGE_BUFFER_BINDING: return 1;
    case GL_SHADER_STORAGE_BUFFER_START: return 1;
    case GL_SHADER_STORAGE_BUFFER_SIZE: return 1;
    case GL_MAX_VERTEX_SHADER_STORAGE_BLOCKS: return 1;
    case GL_MAX_GEOMETRY_SHADER_STORAGE_BLOCKS: return 1;
    case GL_MAX_TESS_CONTROL_SHADER_STORAGE_BLOCKS: return 1;
    case GL_MAX_TESS_EVALUATION_SHADER_STORAGE_BLOCKS: return 1;
    case GL_MAX_FRAGMENT_SHADER_STORAGE_BLOCKS: return 1;
    case GL_MAX_COMPUTE_SHADER_STORAGE_BLOCKS: return 1;
    case GL_MAX_COMBINED_SHADER_STORAGE_BLOCKS: return 1;
    case GL_MAX_SHADER_STORAGE_BUFFER_BINDINGS: return 1;
    case GL_MAX_SHADER_STORAGE_BLOCK_SIZE: return 1;
    case GL_SHADER_STORAGE_BUFFER_OFFSET_ALIGNMENT: return 1;
    case GL_DEPTH_STENCIL_TEXTURE_MODE: return 1;
    case GL_MAX_COMPUTE_WORK_GROUP_INVOCATIONS: return 1;
    case GL_DISPATCH_INDIRECT_BUFFER_BINDING: return 1;
    case GL_TEXTURE_BINDING_2D_MULTISAMPLE: return 1;
    case GL_TEXTURE_BINDING_2D_MULTISAMPLE_ARRAY: return 1;
    case GL_TEXTURE_SAMPLES: return 1;
    case GL_TEXTURE_FIXED_SAMPLE_LOCATIONS: return 1;
    case GL_MAX_COLOR_TEXTURE_SAMPLES: return 1;
    case GL_MAX_DEPTH_TEXTURE_SAMPLES: return 1;
    case GL_MAX_INTEGER_SAMPLES: return 1;
    case GL_MAX_SERVER_WAIT_TIMEOUT: return 1;
    case GL_BUFFER_ACCESS_FLAGS: return 1;
    case GL_BUFFER_MAP_LENGTH: return 1;
    case GL_BUFFER_MAP_OFFSET: return 1;
    case GL_MAX_VERTEX_OUTPUT_COMPONENTS: return 1;
    case GL_MAX_GEOMETRY_INPUT_COMPONENTS: return 1;
    case GL_MAX_GEOMETRY_OUTPUT_COMPONENTS: return 1;
    case GL_MAX_FRAGMENT_INPUT_COMPONENTS: return 1;
    case GL_CONTEXT_PROFILE_MASK: return 1;
    case GL_UNPACK_COMPRESSED_BLOCK_WIDTH: return 1;
    case GL_UNPACK_COMPRESSED_BLOCK_HEIGHT: return 1;
    case GL_UNPACK_COMPRESSED_BLOCK_DEPTH: return 1;
    case GL_UNPACK_COMPRESSED_BLOCK_SIZE: return 1;
    case GL_PACK_COMPRESSED_BLOCK_WIDTH: return 1;
    case GL_PACK_COMPRESSED_BLOCK_HEIGHT: return 1;
    case GL_PACK_COMPRESSED_BLOCK_DEPTH: return 1;
    case GL_PACK_COMPRESSED_BLOCK_SIZE: return 1;
    case GL_TEXTURE_IMMUTABLE_FORMAT: return 1;
    case GL_RENDERBUFFER_SAMPLES_IMG: return 1;
    case GL_MAX_SAMPLES_IMG: return 1;
    case GL_TEXTURE_SAMPLES_IMG: return 1;
    case GL_MAX_DEBUG_MESSAGE_LENGTH: return 1;
    case GL_MAX_DEBUG_LOGGED_MESSAGES: return 1;
    case GL_DEBUG_LOGGED_MESSAGES: return 1;
    case GL_QUERY_BUFFER_BINDING: return 1;
    case GL_QUERY_RESULT_NO_WAIT: return 1;
    case GL_MAX_SPARSE_TEXTURE_SIZE_ARB: return 1;
    case GL_MAX_SPARSE_3D_TEXTURE_SIZE_ARB: return 1;
    case GL_MAX_SPARSE_ARRAY_TEXTURE_LAYERS: return 1;
    case GL_TEXTURE_BUFFER_OFFSET: return 1;
    case GL_TEXTURE_BUFFER_SIZE: return 1;
    case GL_TEXTURE_BUFFER_OFFSET_ALIGNMENT: return 1;
    case GL_SPARSE_TEXTURE_FULL_ARRAY_CUBE_MIPMAPS_ARB: return 1;
    case GL_MAX_SHADER_COMPILER_THREADS_ARB: return 1;
    case GL_COMPLETION_STATUS_ARB: return 1;
    case GL_COMPUTE_SHADER: return 1;
    case GL_MAX_COMPUTE_UNIFORM_BLOCKS: return 1;
    case GL_MAX_COMPUTE_TEXTURE_IMAGE_UNITS: return 1;
    case GL_MAX_COMPUTE_IMAGE_UNIFORMS: return 1;
    case GL_MAX_COMPUTE_WORK_GROUP_COUNT: return 3;
    case GL_MAX_COMPUTE_WORK_GROUP_SIZE: return 3;
    case GL_PRIMITIVE_BOUNDING_BOX_EXT: return 8;
    case GL_ATOMIC_COUNTER_BUFFER_BINDING: return 1;
    case GL_ATOMIC_COUNTER_BUFFER_START: return 1;
    case GL_ATOMIC_COUNTER_BUFFER_SIZE: return 1;
    case GL_MAX_VERTEX_ATOMIC_COUNTER_BUFFERS: return 1;
    case GL_MAX_TESS_CONTROL_ATOMIC_COUNTER_BUFFERS: return 1;
    case GL_MAX_TESS_EVALUATION_ATOMIC_COUNTER_BUFFERS: return 1;
    case GL_MAX_GEOMETRY_ATOMIC_COUNTER_BUFFERS: return 1;
    case GL_MAX_FRAGMENT_ATOMIC_COUNTER_BUFFERS: return 1;
    case GL_MAX_COMBINED_ATOMIC_COUNTER_BUFFERS: return 1;
    case GL_MAX_VERTEX_ATOMIC_COUNTERS: return 1;
    case GL_MAX_TESS_CONTROL_ATOMIC_COUNTERS: return 1;
    case GL_MAX_TESS_EVALUATION_ATOMIC_COUNTERS: return 1;
    case GL_MAX_GEOMETRY_ATOMIC_COUNTERS: return 1;
    case GL_MAX_FRAGMENT_ATOMIC_COUNTERS: return 1;
    case GL_MAX_COMBINED_ATOMIC_COUNTERS: return 1;
    case GL_MAX_ATOMIC_COUNTER_BUFFER_SIZE: return 1;
    case GL_ACTIVE_ATOMIC_COUNTER_BUFFERS: return 1;
    case GL_MAX_ATOMIC_COUNTER_BUFFER_BINDINGS: return 1;
    case GL_DEBUG_OUTPUT: return 1;
    case GL_IS_PER_PATCH: return 1;
    case GL_ACTIVE_RESOURCES: return 1;
    case GL_MAX_NAME_LENGTH: return 1;
    case GL_MAX_NUM_ACTIVE_VARIABLES: return 1;
    case GL_MAX_NUM_COMPATIBLE_SUBROUTINES: return 1;
    case GL_NAME_LENGTH: return 1;
    case GL_TYPE: return 1;
    case GL_ARRAY_SIZE: return 1;
    case GL_OFFSET: return 1;
    case GL_BLOCK_INDEX: return 1;
    case GL_ARRAY_STRIDE: return 1;
    case GL_MATRIX_STRIDE: return 1;
    case GL_IS_ROW_MAJOR: return 1;
    case GL_ATOMIC_COUNTER_BUFFER_INDEX: return 1;
    case GL_BUFFER_BINDING: return 1;
    case GL_BUFFER_DATA_SIZE: return 1;
    case GL_NUM_ACTIVE_VARIABLES: return 1;
    case GL_ACTIVE_VARIABLES: return 1;
    case GL_REFERENCED_BY_VERTEX_SHADER: return 1;
    case GL_REFERENCED_BY_TESS_CONTROL_SHADER: return 1;
    case GL_REFERENCED_BY_TESS_EVALUATION_SHADER: return 1;
    case GL_REFERENCED_BY_GEOMETRY_SHADER: return 1;
    case GL_REFERENCED_BY_FRAGMENT_SHADER: return 1;
    case GL_REFERENCED_BY_COMPUTE_SHADER: return 1;
    case GL_TOP_LEVEL_ARRAY_SIZE: return 1;
    case GL_TOP_LEVEL_ARRAY_STRIDE: return 1;
    case GL_LOCATION: return 1;
    case GL_LOCATION_INDEX: return 1;
    case GL_FRAMEBUFFER_DEFAULT_WIDTH: return 1;
    case GL_FRAMEBUFFER_DEFAULT_HEIGHT: return 1;
    case GL_FRAMEBUFFER_DEFAULT_LAYERS: return 1;
    case GL_FRAMEBUFFER_DEFAULT_SAMPLES: return 1;
    case GL_FRAMEBUFFER_DEFAULT_FIXED_SAMPLE_LOCATIONS: return 1;
    case GL_MAX_FRAMEBUFFER_WIDTH: return 1;
    case GL_MAX_FRAMEBUFFER_HEIGHT: return 1;
    case GL_MAX_FRAMEBUFFER_LAYERS: return 1;
    case GL_MAX_FRAMEBUFFER_SAMPLES: return 1;
    case GL_SAMPLE_LOCATION_SUBPIXEL_BITS_ARB: return 1;
    case GL_SAMPLE_LOCATION_PIXEL_GRID_WIDTH_ARB: return 1;
    case GL_SAMPLE_LOCATION_PIXEL_GRID_HEIGHT_ARB: return 1;
    case GL_PROGRAMMABLE_SAMPLE_LOCATION_ARB: return 2;
    case GL_FRAMEBUFFER_PROGRAMMABLE_SAMPLE_LOCATIONS_ARB: return 1;
    case GL_FRAMEBUFFER_SAMPLE_LOCATION_PIXEL_GRID_ARB: return 1;
    case GL_MAX_COMPUTE_VARIABLE_GROUP_INVOCATIONS_ARB: return 1;
    case GL_MAX_COMPUTE_VARIABLE_GROUP_SIZE_ARB: return 1;
    case GL_CLIP_ORIGIN: return 1;
    case GL_CLIP_DEPTH_MODE: return 1;
    case GL_TEXTURE_REDUCTION_MODE_ARB: return 1;
    case GL_MULTISAMPLE_LINE_WIDTH_RANGE_ARB: return 2;
    case GL_MULTISAMPLE_LINE_WIDTH_GRANULARITY_ARB: return 1;
    case GL_SPIR_V_BINARY: return 1;
    case GL_SPIR_V_EXTENSIONS: return _glGetInteger(GL_NUM_SPIR_V_EXTENSIONS);
    case GL_NUM_SPIR_V_EXTENSIONS: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_NUM_VIEWS_OVR: return 1;
    case GL_MAX_VIEWS_OVR: return 1;
    case GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_BASE_VIEW_INDEX_OVR: return 1;
    case GL_RASTER_POSITION_UNCLIPPED_IBM: return 1;
    default:
        os::log("apitrace: warning: %s: unknown GLenum 0x%04X\n", __FUNCTION__, pname);
        return 1;
    }
}

static GLenum
getBufferBinding(GLenum target) {
    switch (target) {
    case GL_ARRAY_BUFFER:
        return GL_ARRAY_BUFFER_BINDING;
    case GL_ATOMIC_COUNTER_BUFFER:
        return GL_ATOMIC_COUNTER_BUFFER_BINDING;
    case GL_COPY_READ_BUFFER:
        return GL_COPY_READ_BUFFER_BINDING;
    case GL_COPY_WRITE_BUFFER:
        return GL_COPY_WRITE_BUFFER_BINDING;
    case GL_DRAW_INDIRECT_BUFFER:
        return GL_DRAW_INDIRECT_BUFFER_BINDING;
    case GL_DISPATCH_INDIRECT_BUFFER:
        return GL_DISPATCH_INDIRECT_BUFFER_BINDING;
    case GL_ELEMENT_ARRAY_BUFFER:
        return GL_ELEMENT_ARRAY_BUFFER_BINDING;
    case GL_PIXEL_PACK_BUFFER:
        return GL_PIXEL_PACK_BUFFER_BINDING;
    case GL_PIXEL_UNPACK_BUFFER:
        return GL_PIXEL_UNPACK_BUFFER_BINDING;
    case GL_QUERY_BUFFER:
        return GL_QUERY_BUFFER_BINDING;
    case GL_SHADER_STORAGE_BUFFER:
        return GL_SHADER_STORAGE_BUFFER_BINDING;
    case GL_TEXTURE_BUFFER:
        return GL_TEXTURE_BUFFER_BINDING;
    case GL_TRANSFORM_FEEDBACK_BUFFER:
        return GL_TRANSFORM_FEEDBACK_BUFFER_BINDING;
    case GL_UNIFORM_BUFFER:
        return GL_UNIFORM_BUFFER_BINDING;
    default:
        assert(false);
        return 0;
    }
}

static GLint
getBufferName(GLenum target) {
    GLint bufferName = 0;
    _glGetIntegerv(getBufferBinding(target), &bufferName);
    assert(bufferName != 0);
    return bufferName;
}

static inline bool
can_unpack_subimage(void) {
    gltrace::Context *_ctx = gltrace::getContext();
    return _ctx->features.unpack_subimage;
}

extern "C" PUBLIC
void APIENTRY
glNotifyMappedBufferRangeVMWX(const void * start, GLsizeiptr length) {
    trace::fakeMemcpy(start, length);
}


static const trace::EnumValue _enumint7_values[] = {
    {"GLX_NONE", GLX_NONE},
    {"GLX_SLOW_CONFIG", GLX_SLOW_CONFIG},
    {"GLX_NON_CONFORMANT_CONFIG", GLX_NON_CONFORMANT_CONFIG},
};

static const trace::EnumSig _enumint7_sig = {
    11, 3, _enumint7_values
};

static const trace::EnumValue _enumint8_values[] = {
    {"GLX_TRUE_COLOR", GLX_TRUE_COLOR},
    {"GLX_DIRECT_COLOR", GLX_DIRECT_COLOR},
    {"GLX_PSEUDO_COLOR", GLX_PSEUDO_COLOR},
    {"GLX_STATIC_COLOR", GLX_STATIC_COLOR},
};

static const trace::EnumSig _enumint8_sig = {
    12, 4, _enumint8_values
};

static const trace::EnumValue _enumint9_values[] = {
    {"GLX_NONE", GLX_NONE},
    {"GLX_TRANSPARENT_RGB", GLX_TRANSPARENT_RGB},
    {"GLX_TRANSPARENT_INDEX", GLX_TRANSPARENT_INDEX},
};

static const trace::EnumSig _enumint9_sig = {
    13, 3, _enumint9_values
};

static const char * _structXVisualInfo_members[10] = {
    "visual",
    "visualid",
    "screen",
    "depth",
    "c_class",
    "red_mask",
    "green_mask",
    "blue_mask",
    "colormap_size",
    "bits_per_rgb",
};
static const trace::StructSig _structXVisualInfo_sig = {
    0, "XVisualInfo", 10, _structXVisualInfo_members
};

static const trace::EnumValue _enumint1_values[] = {
    {"False", False},
    {"True", True},
};

static const trace::EnumSig _enumint1_sig = {
    7, 2, _enumint1_values
};

static const trace::EnumValue _enumint2_values[] = {
    {"GLX_BUFFER_SIZE", GLX_BUFFER_SIZE},
    {"GLX_LEVEL", GLX_LEVEL},
    {"GLX_RGBA", GLX_RGBA},
    {"GLX_DOUBLEBUFFER", GLX_DOUBLEBUFFER},
    {"GLX_STEREO", GLX_STEREO},
    {"GLX_AUX_BUFFERS", GLX_AUX_BUFFERS},
    {"GLX_RED_SIZE", GLX_RED_SIZE},
    {"GLX_GREEN_SIZE", GLX_GREEN_SIZE},
    {"GLX_BLUE_SIZE", GLX_BLUE_SIZE},
    {"GLX_ALPHA_SIZE", GLX_ALPHA_SIZE},
    {"GLX_DEPTH_SIZE", GLX_DEPTH_SIZE},
    {"GLX_STENCIL_SIZE", GLX_STENCIL_SIZE},
    {"GLX_ACCUM_RED_SIZE", GLX_ACCUM_RED_SIZE},
    {"GLX_ACCUM_GREEN_SIZE", GLX_ACCUM_GREEN_SIZE},
    {"GLX_ACCUM_BLUE_SIZE", GLX_ACCUM_BLUE_SIZE},
    {"GLX_ACCUM_ALPHA_SIZE", GLX_ACCUM_ALPHA_SIZE},
    {"GLX_CONFIG_CAVEAT", GLX_CONFIG_CAVEAT},
    {"GLX_X_VISUAL_TYPE", GLX_X_VISUAL_TYPE},
    {"GLX_TRANSPARENT_TYPE", GLX_TRANSPARENT_TYPE},
    {"GLX_TRANSPARENT_INDEX_VALUE", GLX_TRANSPARENT_INDEX_VALUE},
    {"GLX_TRANSPARENT_RED_VALUE", GLX_TRANSPARENT_RED_VALUE},
    {"GLX_TRANSPARENT_GREEN_VALUE", GLX_TRANSPARENT_GREEN_VALUE},
    {"GLX_TRANSPARENT_BLUE_VALUE", GLX_TRANSPARENT_BLUE_VALUE},
    {"GLX_TRANSPARENT_ALPHA_VALUE", GLX_TRANSPARENT_ALPHA_VALUE},
    {"GLX_HYPERPIPE_PIPE_NAME_LENGTH_SGIX", GLX_HYPERPIPE_PIPE_NAME_LENGTH_SGIX},
    {"GLX_BAD_HYPERPIPE_CONFIG_SGIX", GLX_BAD_HYPERPIPE_CONFIG_SGIX},
    {"GLX_BAD_HYPERPIPE_SGIX", GLX_BAD_HYPERPIPE_SGIX},
    {"GLX_GPU_VENDOR_AMD", GLX_GPU_VENDOR_AMD},
    {"GLX_GPU_RENDERER_STRING_AMD", GLX_GPU_RENDERER_STRING_AMD},
    {"GLX_GPU_OPENGL_VERSION_STRING_AMD", GLX_GPU_OPENGL_VERSION_STRING_AMD},
    {"GLX_CONTEXT_MAJOR_VERSION_ARB", GLX_CONTEXT_MAJOR_VERSION_ARB},
    {"GLX_CONTEXT_MINOR_VERSION_ARB", GLX_CONTEXT_MINOR_VERSION_ARB},
    {"GLX_CONTEXT_FLAGS_ARB", GLX_CONTEXT_FLAGS_ARB},
    {"GLX_CONTEXT_ALLOW_BUFFER_BYTE_ORDER_MISMATCH_ARB", GLX_CONTEXT_ALLOW_BUFFER_BYTE_ORDER_MISMATCH_ARB},
    {"GLX_CONTEXT_RELEASE_BEHAVIOR_ARB", GLX_CONTEXT_RELEASE_BEHAVIOR_ARB},
    {"GLX_CONTEXT_RELEASE_BEHAVIOR_FLUSH_ARB", GLX_CONTEXT_RELEASE_BEHAVIOR_FLUSH_ARB},
    {"GLX_FLOAT_COMPONENTS_NV", GLX_FLOAT_COMPONENTS_NV},
    {"GLX_RGBA_UNSIGNED_FLOAT_TYPE_EXT", GLX_RGBA_UNSIGNED_FLOAT_TYPE_EXT},
    {"GLX_FRAMEBUFFER_SRGB_CAPABLE_ARB", GLX_FRAMEBUFFER_SRGB_CAPABLE_ARB},
    {"GLX_COLOR_SAMPLES_NV", GLX_COLOR_SAMPLES_NV},
    {"GLX_RGBA_FLOAT_TYPE_ARB", GLX_RGBA_FLOAT_TYPE_ARB},
    {"GLX_VIDEO_OUT_COLOR_NV", GLX_VIDEO_OUT_COLOR_NV},
    {"GLX_VIDEO_OUT_ALPHA_NV", GLX_VIDEO_OUT_ALPHA_NV},
    {"GLX_VIDEO_OUT_DEPTH_NV", GLX_VIDEO_OUT_DEPTH_NV},
    {"GLX_VIDEO_OUT_COLOR_AND_ALPHA_NV", GLX_VIDEO_OUT_COLOR_AND_ALPHA_NV},
    {"GLX_VIDEO_OUT_COLOR_AND_DEPTH_NV", GLX_VIDEO_OUT_COLOR_AND_DEPTH_NV},
    {"GLX_VIDEO_OUT_FRAME_NV", GLX_VIDEO_OUT_FRAME_NV},
    {"GLX_VIDEO_OUT_FIELD_1_NV", GLX_VIDEO_OUT_FIELD_1_NV},
    {"GLX_VIDEO_OUT_FIELD_2_NV", GLX_VIDEO_OUT_FIELD_2_NV},
    {"GLX_VIDEO_OUT_STACKED_FIELDS_1_2_NV", GLX_VIDEO_OUT_STACKED_FIELDS_1_2_NV},
    {"GLX_VIDEO_OUT_STACKED_FIELDS_2_1_NV", GLX_VIDEO_OUT_STACKED_FIELDS_2_1_NV},
    {"GLX_DEVICE_ID_NV", GLX_DEVICE_ID_NV},
    {"GLX_UNIQUE_ID_NV", GLX_UNIQUE_ID_NV},
    {"GLX_NUM_VIDEO_CAPTURE_SLOTS_NV", GLX_NUM_VIDEO_CAPTURE_SLOTS_NV},
    {"GLX_BIND_TO_TEXTURE_RGB_EXT", GLX_BIND_TO_TEXTURE_RGB_EXT},
    {"GLX_BIND_TO_TEXTURE_RGBA_EXT", GLX_BIND_TO_TEXTURE_RGBA_EXT},
    {"GLX_BIND_TO_MIPMAP_TEXTURE_EXT", GLX_BIND_TO_MIPMAP_TEXTURE_EXT},
    {"GLX_BIND_TO_TEXTURE_TARGETS_EXT", GLX_BIND_TO_TEXTURE_TARGETS_EXT},
    {"GLX_Y_INVERTED_EXT", GLX_Y_INVERTED_EXT},
    {"GLX_TEXTURE_FORMAT_EXT", GLX_TEXTURE_FORMAT_EXT},
    {"GLX_TEXTURE_TARGET_EXT", GLX_TEXTURE_TARGET_EXT},
    {"GLX_MIPMAP_TEXTURE_EXT", GLX_MIPMAP_TEXTURE_EXT},
    {"GLX_TEXTURE_FORMAT_NONE_EXT", GLX_TEXTURE_FORMAT_NONE_EXT},
    {"GLX_TEXTURE_FORMAT_RGB_EXT", GLX_TEXTURE_FORMAT_RGB_EXT},
    {"GLX_TEXTURE_FORMAT_RGBA_EXT", GLX_TEXTURE_FORMAT_RGBA_EXT},
    {"GLX_TEXTURE_1D_EXT", GLX_TEXTURE_1D_EXT},
    {"GLX_TEXTURE_2D_EXT", GLX_TEXTURE_2D_EXT},
    {"GLX_TEXTURE_RECTANGLE_EXT", GLX_TEXTURE_RECTANGLE_EXT},
    {"GLX_FRONT_LEFT_EXT", GLX_FRONT_LEFT_EXT},
    {"GLX_FRONT_RIGHT_EXT", GLX_FRONT_RIGHT_EXT},
    {"GLX_BACK_LEFT_EXT", GLX_BACK_LEFT_EXT},
    {"GLX_BACK_RIGHT_EXT", GLX_BACK_RIGHT_EXT},
    {"GLX_AUX0_EXT", GLX_AUX0_EXT},
    {"GLX_AUX1_EXT", GLX_AUX1_EXT},
    {"GLX_AUX2_EXT", GLX_AUX2_EXT},
    {"GLX_AUX3_EXT", GLX_AUX3_EXT},
    {"GLX_AUX4_EXT", GLX_AUX4_EXT},
    {"GLX_AUX5_EXT", GLX_AUX5_EXT},
    {"GLX_AUX6_EXT", GLX_AUX6_EXT},
    {"GLX_AUX7_EXT", GLX_AUX7_EXT},
    {"GLX_AUX8_EXT", GLX_AUX8_EXT},
    {"GLX_AUX9_EXT", GLX_AUX9_EXT},
    {"GLX_NUM_VIDEO_SLOTS_NV", GLX_NUM_VIDEO_SLOTS_NV},
    {"GLX_SWAP_INTERVAL_EXT", GLX_SWAP_INTERVAL_EXT},
    {"GLX_MAX_SWAP_INTERVAL_EXT", GLX_MAX_SWAP_INTERVAL_EXT},
    {"GLX_LATE_SWAPS_TEAR_EXT", GLX_LATE_SWAPS_TEAR_EXT},
    {"GLX_BACK_BUFFER_AGE_EXT", GLX_BACK_BUFFER_AGE_EXT},
    {"GLX_STEREO_TREE_EXT", GLX_STEREO_TREE_EXT},
    {"GLX_GPU_FASTEST_TARGET_GPUS_AMD", GLX_GPU_FASTEST_TARGET_GPUS_AMD},
    {"GLX_GPU_RAM_AMD", GLX_GPU_RAM_AMD},
    {"GLX_GPU_CLOCK_AMD", GLX_GPU_CLOCK_AMD},
    {"GLX_GPU_NUM_PIPES_AMD", GLX_GPU_NUM_PIPES_AMD},
    {"GLX_GPU_NUM_SIMD_AMD", GLX_GPU_NUM_SIMD_AMD},
    {"GLX_GPU_NUM_RB_AMD", GLX_GPU_NUM_RB_AMD},
    {"GLX_GPU_NUM_SPI_AMD", GLX_GPU_NUM_SPI_AMD},
    {"GLX_CONTEXT_OPENGL_NO_ERROR_ARB", GLX_CONTEXT_OPENGL_NO_ERROR_ARB},
    {"GLX_NONE", GLX_NONE},
    {"GLX_SLOW_CONFIG", GLX_SLOW_CONFIG},
    {"GLX_TRUE_COLOR", GLX_TRUE_COLOR},
    {"GLX_DIRECT_COLOR", GLX_DIRECT_COLOR},
    {"GLX_PSEUDO_COLOR", GLX_PSEUDO_COLOR},
    {"GLX_STATIC_COLOR", GLX_STATIC_COLOR},
    {"GLX_GRAY_SCALE", GLX_GRAY_SCALE},
    {"GLX_STATIC_GRAY", GLX_STATIC_GRAY},
    {"GLX_TRANSPARENT_RGB", GLX_TRANSPARENT_RGB},
    {"GLX_TRANSPARENT_INDEX", GLX_TRANSPARENT_INDEX},
    {"GLX_SHARE_CONTEXT_EXT", GLX_SHARE_CONTEXT_EXT},
    {"GLX_VISUAL_ID", GLX_VISUAL_ID},
    {"GLX_SCREEN", GLX_SCREEN},
    {"GLX_NON_CONFORMANT_CONFIG", GLX_NON_CONFORMANT_CONFIG},
    {"GLX_DRAWABLE_TYPE", GLX_DRAWABLE_TYPE},
    {"GLX_RENDER_TYPE", GLX_RENDER_TYPE},
    {"GLX_X_RENDERABLE", GLX_X_RENDERABLE},
    {"GLX_FBCONFIG_ID", GLX_FBCONFIG_ID},
    {"GLX_RGBA_TYPE", GLX_RGBA_TYPE},
    {"GLX_COLOR_INDEX_TYPE", GLX_COLOR_INDEX_TYPE},
    {"GLX_MAX_PBUFFER_WIDTH", GLX_MAX_PBUFFER_WIDTH},
    {"GLX_MAX_PBUFFER_HEIGHT", GLX_MAX_PBUFFER_HEIGHT},
    {"GLX_MAX_PBUFFER_PIXELS", GLX_MAX_PBUFFER_PIXELS},
    {"GLX_OPTIMAL_PBUFFER_WIDTH_SGIX", GLX_OPTIMAL_PBUFFER_WIDTH_SGIX},
    {"GLX_OPTIMAL_PBUFFER_HEIGHT_SGIX", GLX_OPTIMAL_PBUFFER_HEIGHT_SGIX},
    {"GLX_PRESERVED_CONTENTS", GLX_PRESERVED_CONTENTS},
    {"GLX_LARGEST_PBUFFER", GLX_LARGEST_PBUFFER},
    {"GLX_WIDTH", GLX_WIDTH},
    {"GLX_HEIGHT", GLX_HEIGHT},
    {"GLX_EVENT_MASK", GLX_EVENT_MASK},
    {"GLX_DAMAGED", GLX_DAMAGED},
    {"GLX_SAVED", GLX_SAVED},
    {"GLX_WINDOW", GLX_WINDOW},
    {"GLX_PBUFFER", GLX_PBUFFER},
    {"GLX_BLENDED_RGBA_SGIS", GLX_BLENDED_RGBA_SGIS},
    {"GLX_MULTISAMPLE_SUB_RECT_WIDTH_SGIS", GLX_MULTISAMPLE_SUB_RECT_WIDTH_SGIS},
    {"GLX_MULTISAMPLE_SUB_RECT_HEIGHT_SGIS", GLX_MULTISAMPLE_SUB_RECT_HEIGHT_SGIS},
    {"GLX_VISUAL_SELECT_GROUP_SGIX", GLX_VISUAL_SELECT_GROUP_SGIX},
    {"GLX_HYPERPIPE_ID_SGIX", GLX_HYPERPIPE_ID_SGIX},
    {"GLX_PBUFFER_HEIGHT", GLX_PBUFFER_HEIGHT},
    {"GLX_PBUFFER_WIDTH", GLX_PBUFFER_WIDTH},
    {"GLX_SAMPLE_BUFFERS_3DFX", GLX_SAMPLE_BUFFERS_3DFX},
    {"GLX_SAMPLES_3DFX", GLX_SAMPLES_3DFX},
    {"GLX_SWAP_METHOD_OML", GLX_SWAP_METHOD_OML},
    {"GLX_SWAP_EXCHANGE_OML", GLX_SWAP_EXCHANGE_OML},
    {"GLX_SWAP_COPY_OML", GLX_SWAP_COPY_OML},
    {"GLX_SWAP_UNDEFINED_OML", GLX_SWAP_UNDEFINED_OML},
    {"GLX_EXCHANGE_COMPLETE_INTEL", GLX_EXCHANGE_COMPLETE_INTEL},
    {"GLX_COPY_COMPLETE_INTEL", GLX_COPY_COMPLETE_INTEL},
    {"GLX_FLIP_COMPLETE_INTEL", GLX_FLIP_COMPLETE_INTEL},
    {"GLX_RENDERER_VENDOR_ID_MESA", GLX_RENDERER_VENDOR_ID_MESA},
    {"GLX_RENDERER_DEVICE_ID_MESA", GLX_RENDERER_DEVICE_ID_MESA},
    {"GLX_RENDERER_VERSION_MESA", GLX_RENDERER_VERSION_MESA},
    {"GLX_RENDERER_ACCELERATED_MESA", GLX_RENDERER_ACCELERATED_MESA},
    {"GLX_RENDERER_VIDEO_MEMORY_MESA", GLX_RENDERER_VIDEO_MEMORY_MESA},
    {"GLX_RENDERER_UNIFIED_MEMORY_ARCHITECTURE_MESA", GLX_RENDERER_UNIFIED_MEMORY_ARCHITECTURE_MESA},
    {"GLX_RENDERER_PREFERRED_PROFILE_MESA", GLX_RENDERER_PREFERRED_PROFILE_MESA},
    {"GLX_RENDERER_OPENGL_CORE_PROFILE_VERSION_MESA", GLX_RENDERER_OPENGL_CORE_PROFILE_VERSION_MESA},
    {"GLX_RENDERER_OPENGL_COMPATIBILITY_PROFILE_VERSION_MESA", GLX_RENDERER_OPENGL_COMPATIBILITY_PROFILE_VERSION_MESA},
    {"GLX_RENDERER_OPENGL_ES_PROFILE_VERSION_MESA", GLX_RENDERER_OPENGL_ES_PROFILE_VERSION_MESA},
    {"GLX_RENDERER_OPENGL_ES2_PROFILE_VERSION_MESA", GLX_RENDERER_OPENGL_ES2_PROFILE_VERSION_MESA},
    {"GLX_LOSE_CONTEXT_ON_RESET_ARB", GLX_LOSE_CONTEXT_ON_RESET_ARB},
    {"GLX_CONTEXT_RESET_NOTIFICATION_STRATEGY_ARB", GLX_CONTEXT_RESET_NOTIFICATION_STRATEGY_ARB},
    {"GLX_NO_RESET_NOTIFICATION_ARB", GLX_NO_RESET_NOTIFICATION_ARB},
    {"GLX_CONTEXT_PROFILE_MASK_ARB", GLX_CONTEXT_PROFILE_MASK_ARB},
    {"GLX_SAMPLE_BUFFERS", GLX_SAMPLE_BUFFERS},
    {"GLX_SAMPLES", GLX_SAMPLES},
    {"GLX_DONT_CARE", GLX_DONT_CARE},
};

static const trace::EnumSig _enumint2_sig = {
    8, 164, _enumint2_values
};

static const trace::EnumValue _enumint3_values[] = {
    {"GLX_BAD_SCREEN", GLX_BAD_SCREEN},
    {"GLX_BAD_ATTRIBUTE", GLX_BAD_ATTRIBUTE},
    {"GLX_NO_EXTENSION", GLX_NO_EXTENSION},
    {"GLX_BAD_VISUAL", GLX_BAD_VISUAL},
    {"GLX_BAD_CONTEXT", GLX_BAD_CONTEXT},
    {"GLX_BAD_VALUE", GLX_BAD_VALUE},
    {"GLX_BAD_ENUM", GLX_BAD_ENUM},
};

static const trace::EnumSig _enumint3_sig = {
    9, 7, _enumint3_values
};

static const trace::EnumValue _enumint4_values[] = {
    {"GLX_VENDOR", GLX_VENDOR},
    {"GLX_VERSION", GLX_VERSION},
    {"GLX_EXTENSIONS", GLX_EXTENSIONS},
};

static const trace::EnumSig _enumint4_sig = {
    10, 3, _enumint4_values
};

static const trace::BitmaskFlag _bitmaskint11_flags[] = {
    {"GLX_RGBA_BIT", (unsigned long long)GLX_RGBA_BIT},
    {"GLX_COLOR_INDEX_BIT", (unsigned long long)GLX_COLOR_INDEX_BIT},
};

static const trace::BitmaskSig _bitmaskint11_sig = {
    12, 2, _bitmaskint11_flags
};

static const trace::BitmaskFlag _bitmaskint12_flags[] = {
    {"GLX_WINDOW_BIT", (unsigned long long)GLX_WINDOW_BIT},
    {"GLX_PIXMAP_BIT", (unsigned long long)GLX_PIXMAP_BIT},
    {"GLX_PBUFFER_BIT", (unsigned long long)GLX_PBUFFER_BIT},
};

static const trace::BitmaskSig _bitmaskint12_sig = {
    13, 3, _bitmaskint12_flags
};

static const trace::EnumValue _enumint15_values[] = {
    {"GLX_RGBA_TYPE", GLX_RGBA_TYPE},
    {"GLX_COLOR_INDEX_TYPE", GLX_COLOR_INDEX_TYPE},
};

static const trace::EnumSig _enumint15_sig = {
    14, 2, _enumint15_values
};

static const trace::BitmaskFlag _bitmaskint16_flags[] = {
    {"GLX_CONTEXT_DEBUG_BIT_ARB", (unsigned long long)GLX_CONTEXT_DEBUG_BIT_ARB},
    {"GLX_CONTEXT_FORWARD_COMPATIBLE_BIT_ARB", (unsigned long long)GLX_CONTEXT_FORWARD_COMPATIBLE_BIT_ARB},
    {"GLX_CONTEXT_ROBUST_ACCESS_BIT_ARB", (unsigned long long)GLX_CONTEXT_ROBUST_ACCESS_BIT_ARB},
};

static const trace::BitmaskSig _bitmaskint16_sig = {
    15, 3, _bitmaskint16_flags
};

static const trace::BitmaskFlag _bitmaskint14_flags[] = {
    {"GLX_CONTEXT_CORE_PROFILE_BIT_ARB", (unsigned long long)GLX_CONTEXT_CORE_PROFILE_BIT_ARB},
    {"GLX_CONTEXT_COMPATIBILITY_PROFILE_BIT_ARB", (unsigned long long)GLX_CONTEXT_COMPATIBILITY_PROFILE_BIT_ARB},
    {"GLX_CONTEXT_ES_PROFILE_BIT_EXT", (unsigned long long)GLX_CONTEXT_ES_PROFILE_BIT_EXT},
};

static const trace::BitmaskSig _bitmaskint14_sig = {
    14, 3, _bitmaskint14_flags
};

static const trace::EnumValue _enumint17_values[] = {
    {"GLX_CONTEXT_RELEASE_BEHAVIOR_NONE_ARB", GLX_CONTEXT_RELEASE_BEHAVIOR_NONE_ARB},
    {"GLX_CONTEXT_RELEASE_BEHAVIOR_FLUSH_ARB", GLX_CONTEXT_RELEASE_BEHAVIOR_FLUSH_ARB},
};

static const trace::EnumSig _enumint17_sig = {
    15, 2, _enumint17_values
};

static void _write__unsignedint6(int selector, unsigned int * const & value) {
    switch (selector) {
    default:
    if ((unsigned int *)(value)) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeUInt(*(unsigned int *)(value));
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_VENDOR_ID_MESA:
    if ((unsigned int *)(value)) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeUInt(*(unsigned int *)(value));
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_DEVICE_ID_MESA:
    if ((unsigned int *)(value)) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeUInt(*(unsigned int *)(value));
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_VERSION_MESA:
    if ((unsigned int *)(value)) {
        size_t _cunsignedint = 3 > 0 ? 3 : 0;
        trace::localWriter.beginArray(_cunsignedint);
        for (size_t _iunsignedint = 0; _iunsignedint < _cunsignedint; ++_iunsignedint) {
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt(((unsigned int *)(value))[_iunsignedint]);
            trace::localWriter.endElement();
        }
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_ACCELERATED_MESA:
    if ((int *)(value)) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint1_sig, *(int *)(value));
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_VIDEO_MEMORY_MESA:
    if ((unsigned int *)(value)) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeUInt(*(unsigned int *)(value));
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_UNIFIED_MEMORY_ARCHITECTURE_MESA:
    if ((int *)(value)) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint1_sig, *(int *)(value));
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_PREFERRED_PROFILE_MESA:
    if ((int *)(value)) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeBitmask(&_bitmaskint14_sig, *(int *)(value));
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_OPENGL_CORE_PROFILE_VERSION_MESA:
    if ((unsigned int *)(value)) {
        size_t _cunsignedint = 2 > 0 ? 2 : 0;
        trace::localWriter.beginArray(_cunsignedint);
        for (size_t _iunsignedint = 0; _iunsignedint < _cunsignedint; ++_iunsignedint) {
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt(((unsigned int *)(value))[_iunsignedint]);
            trace::localWriter.endElement();
        }
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_OPENGL_COMPATIBILITY_PROFILE_VERSION_MESA:
    if ((unsigned int *)(value)) {
        size_t _cunsignedint = 2 > 0 ? 2 : 0;
        trace::localWriter.beginArray(_cunsignedint);
        for (size_t _iunsignedint = 0; _iunsignedint < _cunsignedint; ++_iunsignedint) {
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt(((unsigned int *)(value))[_iunsignedint]);
            trace::localWriter.endElement();
        }
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_OPENGL_ES_PROFILE_VERSION_MESA:
    if ((unsigned int *)(value)) {
        size_t _cunsignedint = 2 > 0 ? 2 : 0;
        trace::localWriter.beginArray(_cunsignedint);
        for (size_t _iunsignedint = 0; _iunsignedint < _cunsignedint; ++_iunsignedint) {
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt(((unsigned int *)(value))[_iunsignedint]);
            trace::localWriter.endElement();
        }
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GLX_RENDERER_OPENGL_ES2_PROFILE_VERSION_MESA:
    if ((unsigned int *)(value)) {
        size_t _cunsignedint = 2 > 0 ? 2 : 0;
        trace::localWriter.beginArray(_cunsignedint);
        for (size_t _iunsignedint = 0; _iunsignedint < _cunsignedint; ++_iunsignedint) {
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt(((unsigned int *)(value))[_iunsignedint]);
            trace::localWriter.endElement();
        }
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    }
}

static const trace::EnumValue _enumGLboolean_values[] = {
    {"GL_TRUE", GL_TRUE},
    {"GL_FALSE", GL_FALSE},
};

static const trace::EnumSig _enumGLboolean_sig = {
    0, 2, _enumGLboolean_values
};

static const trace::EnumValue _enumGLenum_values[] = {
    {"GL_ZERO", GL_ZERO},
    {"GL_ONE", GL_ONE},
    {"GL_RESTART_PATH_NV", GL_RESTART_PATH_NV},
    {"GL_DUP_FIRST_CUBIC_CURVE_TO_NV", GL_DUP_FIRST_CUBIC_CURVE_TO_NV},
    {"GL_DUP_LAST_CUBIC_CURVE_TO_NV", GL_DUP_LAST_CUBIC_CURVE_TO_NV},
    {"GL_RECT_NV", GL_RECT_NV},
    {"GL_RELATIVE_RECT_NV", GL_RELATIVE_RECT_NV},
    {"GL_CIRCULAR_CCW_ARC_TO_NV", GL_CIRCULAR_CCW_ARC_TO_NV},
    {"GL_CIRCULAR_CW_ARC_TO_NV", GL_CIRCULAR_CW_ARC_TO_NV},
    {"GL_CIRCULAR_TANGENT_ARC_TO_NV", GL_CIRCULAR_TANGENT_ARC_TO_NV},
    {"GL_ARC_TO_NV", GL_ARC_TO_NV},
    {"GL_RELATIVE_ARC_TO_NV", GL_RELATIVE_ARC_TO_NV},
    {"GL_ACCUM", GL_ACCUM},
    {"GL_LOAD", GL_LOAD},
    {"GL_RETURN", GL_RETURN},
    {"GL_MULT", GL_MULT},
    {"GL_ADD", GL_ADD},
    {"GL_NEVER", GL_NEVER},
    {"GL_LESS", GL_LESS},
    {"GL_EQUAL", GL_EQUAL},
    {"GL_LEQUAL", GL_LEQUAL},
    {"GL_GREATER", GL_GREATER},
    {"GL_NOTEQUAL", GL_NOTEQUAL},
    {"GL_GEQUAL", GL_GEQUAL},
    {"GL_ALWAYS", GL_ALWAYS},
    {"GL_SRC_COLOR", GL_SRC_COLOR},
    {"GL_ONE_MINUS_SRC_COLOR", GL_ONE_MINUS_SRC_COLOR},
    {"GL_SRC_ALPHA", GL_SRC_ALPHA},
    {"GL_ONE_MINUS_SRC_ALPHA", GL_ONE_MINUS_SRC_ALPHA},
    {"GL_DST_ALPHA", GL_DST_ALPHA},
    {"GL_ONE_MINUS_DST_ALPHA", GL_ONE_MINUS_DST_ALPHA},
    {"GL_DST_COLOR", GL_DST_COLOR},
    {"GL_ONE_MINUS_DST_COLOR", GL_ONE_MINUS_DST_COLOR},
    {"GL_SRC_ALPHA_SATURATE", GL_SRC_ALPHA_SATURATE},
    {"GL_FRONT_LEFT", GL_FRONT_LEFT},
    {"GL_FRONT_RIGHT", GL_FRONT_RIGHT},
    {"GL_BACK_LEFT", GL_BACK_LEFT},
    {"GL_BACK_RIGHT", GL_BACK_RIGHT},
    {"GL_FRONT", GL_FRONT},
    {"GL_BACK", GL_BACK},
    {"GL_LEFT", GL_LEFT},
    {"GL_RIGHT", GL_RIGHT},
    {"GL_FRONT_AND_BACK", GL_FRONT_AND_BACK},
    {"GL_AUX0", GL_AUX0},
    {"GL_AUX1", GL_AUX1},
    {"GL_AUX2", GL_AUX2},
    {"GL_AUX3", GL_AUX3},
    {"GL_INVALID_ENUM", GL_INVALID_ENUM},
    {"GL_INVALID_VALUE", GL_INVALID_VALUE},
    {"GL_INVALID_OPERATION", GL_INVALID_OPERATION},
    {"GL_STACK_OVERFLOW", GL_STACK_OVERFLOW},
    {"GL_STACK_UNDERFLOW", GL_STACK_UNDERFLOW},
    {"GL_OUT_OF_MEMORY", GL_OUT_OF_MEMORY},
    {"GL_INVALID_FRAMEBUFFER_OPERATION", GL_INVALID_FRAMEBUFFER_OPERATION},
    {"GL_CONTEXT_LOST", GL_CONTEXT_LOST},
    {"GL_2D", GL_2D},
    {"GL_3D", GL_3D},
    {"GL_3D_COLOR", GL_3D_COLOR},
    {"GL_3D_COLOR_TEXTURE", GL_3D_COLOR_TEXTURE},
    {"GL_4D_COLOR_TEXTURE", GL_4D_COLOR_TEXTURE},
    {"GL_PASS_THROUGH_TOKEN", GL_PASS_THROUGH_TOKEN},
    {"GL_POINT_TOKEN", GL_POINT_TOKEN},
    {"GL_LINE_TOKEN", GL_LINE_TOKEN},
    {"GL_POLYGON_TOKEN", GL_POLYGON_TOKEN},
    {"GL_BITMAP_TOKEN", GL_BITMAP_TOKEN},
    {"GL_DRAW_PIXEL_TOKEN", GL_DRAW_PIXEL_TOKEN},
    {"GL_COPY_PIXEL_TOKEN", GL_COPY_PIXEL_TOKEN},
    {"GL_LINE_RESET_TOKEN", GL_LINE_RESET_TOKEN},
    {"GL_EXP", GL_EXP},
    {"GL_EXP2", GL_EXP2},
    {"GL_CW", GL_CW},
    {"GL_CCW", GL_CCW},
    {"GL_COEFF", GL_COEFF},
    {"GL_ORDER", GL_ORDER},
    {"GL_DOMAIN", GL_DOMAIN},
    {"GL_CURRENT_COLOR", GL_CURRENT_COLOR},
    {"GL_CURRENT_INDEX", GL_CURRENT_INDEX},
    {"GL_CURRENT_NORMAL", GL_CURRENT_NORMAL},
    {"GL_CURRENT_TEXTURE_COORDS", GL_CURRENT_TEXTURE_COORDS},
    {"GL_CURRENT_RASTER_COLOR", GL_CURRENT_RASTER_COLOR},
    {"GL_CURRENT_RASTER_INDEX", GL_CURRENT_RASTER_INDEX},
    {"GL_CURRENT_RASTER_TEXTURE_COORDS", GL_CURRENT_RASTER_TEXTURE_COORDS},
    {"GL_CURRENT_RASTER_POSITION", GL_CURRENT_RASTER_POSITION},
    {"GL_CURRENT_RASTER_POSITION_VALID", GL_CURRENT_RASTER_POSITION_VALID},
    {"GL_CURRENT_RASTER_DISTANCE", GL_CURRENT_RASTER_DISTANCE},
    {"GL_POINT_SMOOTH", GL_POINT_SMOOTH},
    {"GL_POINT_SIZE", GL_POINT_SIZE},
    {"GL_POINT_SIZE_RANGE", GL_POINT_SIZE_RANGE},
    {"GL_POINT_SIZE_GRANULARITY", GL_POINT_SIZE_GRANULARITY},
    {"GL_LINE_SMOOTH", GL_LINE_SMOOTH},
    {"GL_LINE_WIDTH", GL_LINE_WIDTH},
    {"GL_LINE_WIDTH_RANGE", GL_LINE_WIDTH_RANGE},
    {"GL_LINE_WIDTH_GRANULARITY", GL_LINE_WIDTH_GRANULARITY},
    {"GL_LINE_STIPPLE", GL_LINE_STIPPLE},
    {"GL_LINE_STIPPLE_PATTERN", GL_LINE_STIPPLE_PATTERN},
    {"GL_LINE_STIPPLE_REPEAT", GL_LINE_STIPPLE_REPEAT},
    {"GL_LIST_MODE", GL_LIST_MODE},
    {"GL_MAX_LIST_NESTING", GL_MAX_LIST_NESTING},
    {"GL_LIST_BASE", GL_LIST_BASE},
    {"GL_LIST_INDEX", GL_LIST_INDEX},
    {"GL_POLYGON_MODE", GL_POLYGON_MODE},
    {"GL_POLYGON_SMOOTH", GL_POLYGON_SMOOTH},
    {"GL_POLYGON_STIPPLE", GL_POLYGON_STIPPLE},
    {"GL_EDGE_FLAG", GL_EDGE_FLAG},
    {"GL_CULL_FACE", GL_CULL_FACE},
    {"GL_CULL_FACE_MODE", GL_CULL_FACE_MODE},
    {"GL_FRONT_FACE", GL_FRONT_FACE},
    {"GL_LIGHTING", GL_LIGHTING},
    {"GL_LIGHT_MODEL_LOCAL_VIEWER", GL_LIGHT_MODEL_LOCAL_VIEWER},
    {"GL_LIGHT_MODEL_TWO_SIDE", GL_LIGHT_MODEL_TWO_SIDE},
    {"GL_LIGHT_MODEL_AMBIENT", GL_LIGHT_MODEL_AMBIENT},
    {"GL_SHADE_MODEL", GL_SHADE_MODEL},
    {"GL_COLOR_MATERIAL_FACE", GL_COLOR_MATERIAL_FACE},
    {"GL_COLOR_MATERIAL_PARAMETER", GL_COLOR_MATERIAL_PARAMETER},
    {"GL_COLOR_MATERIAL", GL_COLOR_MATERIAL},
    {"GL_FOG", GL_FOG},
    {"GL_FOG_INDEX", GL_FOG_INDEX},
    {"GL_FOG_DENSITY", GL_FOG_DENSITY},
    {"GL_FOG_START", GL_FOG_START},
    {"GL_FOG_END", GL_FOG_END},
    {"GL_FOG_MODE", GL_FOG_MODE},
    {"GL_FOG_COLOR", GL_FOG_COLOR},
    {"GL_DEPTH_RANGE", GL_DEPTH_RANGE},
    {"GL_DEPTH_TEST", GL_DEPTH_TEST},
    {"GL_DEPTH_WRITEMASK", GL_DEPTH_WRITEMASK},
    {"GL_DEPTH_CLEAR_VALUE", GL_DEPTH_CLEAR_VALUE},
    {"GL_DEPTH_FUNC", GL_DEPTH_FUNC},
    {"GL_ACCUM_CLEAR_VALUE", GL_ACCUM_CLEAR_VALUE},
    {"GL_STENCIL_TEST", GL_STENCIL_TEST},
    {"GL_STENCIL_CLEAR_VALUE", GL_STENCIL_CLEAR_VALUE},
    {"GL_STENCIL_FUNC", GL_STENCIL_FUNC},
    {"GL_STENCIL_VALUE_MASK", GL_STENCIL_VALUE_MASK},
    {"GL_STENCIL_FAIL", GL_STENCIL_FAIL},
    {"GL_STENCIL_PASS_DEPTH_FAIL", GL_STENCIL_PASS_DEPTH_FAIL},
    {"GL_STENCIL_PASS_DEPTH_PASS", GL_STENCIL_PASS_DEPTH_PASS},
    {"GL_STENCIL_REF", GL_STENCIL_REF},
    {"GL_STENCIL_WRITEMASK", GL_STENCIL_WRITEMASK},
    {"GL_MATRIX_MODE", GL_MATRIX_MODE},
    {"GL_NORMALIZE", GL_NORMALIZE},
    {"GL_VIEWPORT", GL_VIEWPORT},
    {"GL_MODELVIEW_STACK_DEPTH", GL_MODELVIEW_STACK_DEPTH},
    {"GL_PROJECTION_STACK_DEPTH", GL_PROJECTION_STACK_DEPTH},
    {"GL_TEXTURE_STACK_DEPTH", GL_TEXTURE_STACK_DEPTH},
    {"GL_MODELVIEW_MATRIX", GL_MODELVIEW_MATRIX},
    {"GL_PROJECTION_MATRIX", GL_PROJECTION_MATRIX},
    {"GL_TEXTURE_MATRIX", GL_TEXTURE_MATRIX},
    {"GL_ATTRIB_STACK_DEPTH", GL_ATTRIB_STACK_DEPTH},
    {"GL_CLIENT_ATTRIB_STACK_DEPTH", GL_CLIENT_ATTRIB_STACK_DEPTH},
    {"GL_ALPHA_TEST", GL_ALPHA_TEST},
    {"GL_ALPHA_TEST_FUNC", GL_ALPHA_TEST_FUNC},
    {"GL_ALPHA_TEST_REF", GL_ALPHA_TEST_REF},
    {"GL_DITHER", GL_DITHER},
    {"GL_BLEND_DST", GL_BLEND_DST},
    {"GL_BLEND_SRC", GL_BLEND_SRC},
    {"GL_BLEND", GL_BLEND},
    {"GL_LOGIC_OP_MODE", GL_LOGIC_OP_MODE},
    {"GL_INDEX_LOGIC_OP", GL_INDEX_LOGIC_OP},
    {"GL_COLOR_LOGIC_OP", GL_COLOR_LOGIC_OP},
    {"GL_AUX_BUFFERS", GL_AUX_BUFFERS},
    {"GL_DRAW_BUFFER", GL_DRAW_BUFFER},
    {"GL_READ_BUFFER", GL_READ_BUFFER},
    {"GL_SCISSOR_BOX", GL_SCISSOR_BOX},
    {"GL_SCISSOR_TEST", GL_SCISSOR_TEST},
    {"GL_INDEX_CLEAR_VALUE", GL_INDEX_CLEAR_VALUE},
    {"GL_INDEX_WRITEMASK", GL_INDEX_WRITEMASK},
    {"GL_COLOR_CLEAR_VALUE", GL_COLOR_CLEAR_VALUE},
    {"GL_COLOR_WRITEMASK", GL_COLOR_WRITEMASK},
    {"GL_INDEX_MODE", GL_INDEX_MODE},
    {"GL_RGBA_MODE", GL_RGBA_MODE},
    {"GL_DOUBLEBUFFER", GL_DOUBLEBUFFER},
    {"GL_STEREO", GL_STEREO},
    {"GL_RENDER_MODE", GL_RENDER_MODE},
    {"GL_PERSPECTIVE_CORRECTION_HINT", GL_PERSPECTIVE_CORRECTION_HINT},
    {"GL_POINT_SMOOTH_HINT", GL_POINT_SMOOTH_HINT},
    {"GL_LINE_SMOOTH_HINT", GL_LINE_SMOOTH_HINT},
    {"GL_POLYGON_SMOOTH_HINT", GL_POLYGON_SMOOTH_HINT},
    {"GL_FOG_HINT", GL_FOG_HINT},
    {"GL_TEXTURE_GEN_S", GL_TEXTURE_GEN_S},
    {"GL_TEXTURE_GEN_T", GL_TEXTURE_GEN_T},
    {"GL_TEXTURE_GEN_R", GL_TEXTURE_GEN_R},
    {"GL_TEXTURE_GEN_Q", GL_TEXTURE_GEN_Q},
    {"GL_PIXEL_MAP_I_TO_I", GL_PIXEL_MAP_I_TO_I},
    {"GL_PIXEL_MAP_S_TO_S", GL_PIXEL_MAP_S_TO_S},
    {"GL_PIXEL_MAP_I_TO_R", GL_PIXEL_MAP_I_TO_R},
    {"GL_PIXEL_MAP_I_TO_G", GL_PIXEL_MAP_I_TO_G},
    {"GL_PIXEL_MAP_I_TO_B", GL_PIXEL_MAP_I_TO_B},
    {"GL_PIXEL_MAP_I_TO_A", GL_PIXEL_MAP_I_TO_A},
    {"GL_PIXEL_MAP_R_TO_R", GL_PIXEL_MAP_R_TO_R},
    {"GL_PIXEL_MAP_G_TO_G", GL_PIXEL_MAP_G_TO_G},
    {"GL_PIXEL_MAP_B_TO_B", GL_PIXEL_MAP_B_TO_B},
    {"GL_PIXEL_MAP_A_TO_A", GL_PIXEL_MAP_A_TO_A},
    {"GL_PIXEL_MAP_I_TO_I_SIZE", GL_PIXEL_MAP_I_TO_I_SIZE},
    {"GL_PIXEL_MAP_S_TO_S_SIZE", GL_PIXEL_MAP_S_TO_S_SIZE},
    {"GL_PIXEL_MAP_I_TO_R_SIZE", GL_PIXEL_MAP_I_TO_R_SIZE},
    {"GL_PIXEL_MAP_I_TO_G_SIZE", GL_PIXEL_MAP_I_TO_G_SIZE},
    {"GL_PIXEL_MAP_I_TO_B_SIZE", GL_PIXEL_MAP_I_TO_B_SIZE},
    {"GL_PIXEL_MAP_I_TO_A_SIZE", GL_PIXEL_MAP_I_TO_A_SIZE},
    {"GL_PIXEL_MAP_R_TO_R_SIZE", GL_PIXEL_MAP_R_TO_R_SIZE},
    {"GL_PIXEL_MAP_G_TO_G_SIZE", GL_PIXEL_MAP_G_TO_G_SIZE},
    {"GL_PIXEL_MAP_B_TO_B_SIZE", GL_PIXEL_MAP_B_TO_B_SIZE},
    {"GL_PIXEL_MAP_A_TO_A_SIZE", GL_PIXEL_MAP_A_TO_A_SIZE},
    {"GL_UNPACK_SWAP_BYTES", GL_UNPACK_SWAP_BYTES},
    {"GL_UNPACK_LSB_FIRST", GL_UNPACK_LSB_FIRST},
    {"GL_UNPACK_ROW_LENGTH", GL_UNPACK_ROW_LENGTH},
    {"GL_UNPACK_SKIP_ROWS", GL_UNPACK_SKIP_ROWS},
    {"GL_UNPACK_SKIP_PIXELS", GL_UNPACK_SKIP_PIXELS},
    {"GL_UNPACK_ALIGNMENT", GL_UNPACK_ALIGNMENT},
    {"GL_PACK_SWAP_BYTES", GL_PACK_SWAP_BYTES},
    {"GL_PACK_LSB_FIRST", GL_PACK_LSB_FIRST},
    {"GL_PACK_ROW_LENGTH", GL_PACK_ROW_LENGTH},
    {"GL_PACK_SKIP_ROWS", GL_PACK_SKIP_ROWS},
    {"GL_PACK_SKIP_PIXELS", GL_PACK_SKIP_PIXELS},
    {"GL_PACK_ALIGNMENT", GL_PACK_ALIGNMENT},
    {"GL_MAP_COLOR", GL_MAP_COLOR},
    {"GL_MAP_STENCIL", GL_MAP_STENCIL},
    {"GL_INDEX_SHIFT", GL_INDEX_SHIFT},
    {"GL_INDEX_OFFSET", GL_INDEX_OFFSET},
    {"GL_RED_SCALE", GL_RED_SCALE},
    {"GL_RED_BIAS", GL_RED_BIAS},
    {"GL_ZOOM_X", GL_ZOOM_X},
    {"GL_ZOOM_Y", GL_ZOOM_Y},
    {"GL_GREEN_SCALE", GL_GREEN_SCALE},
    {"GL_GREEN_BIAS", GL_GREEN_BIAS},
    {"GL_BLUE_SCALE", GL_BLUE_SCALE},
    {"GL_BLUE_BIAS", GL_BLUE_BIAS},
    {"GL_ALPHA_SCALE", GL_ALPHA_SCALE},
    {"GL_ALPHA_BIAS", GL_ALPHA_BIAS},
    {"GL_DEPTH_SCALE", GL_DEPTH_SCALE},
    {"GL_DEPTH_BIAS", GL_DEPTH_BIAS},
    {"GL_MAX_EVAL_ORDER", GL_MAX_EVAL_ORDER},
    {"GL_MAX_LIGHTS", GL_MAX_LIGHTS},
    {"GL_MAX_CLIP_DISTANCES", GL_MAX_CLIP_DISTANCES},
    {"GL_MAX_TEXTURE_SIZE", GL_MAX_TEXTURE_SIZE},
    {"GL_MAX_PIXEL_MAP_TABLE", GL_MAX_PIXEL_MAP_TABLE},
    {"GL_MAX_ATTRIB_STACK_DEPTH", GL_MAX_ATTRIB_STACK_DEPTH},
    {"GL_MAX_MODELVIEW_STACK_DEPTH", GL_MAX_MODELVIEW_STACK_DEPTH},
    {"GL_MAX_NAME_STACK_DEPTH", GL_MAX_NAME_STACK_DEPTH},
    {"GL_MAX_PROJECTION_STACK_DEPTH", GL_MAX_PROJECTION_STACK_DEPTH},
    {"GL_MAX_TEXTURE_STACK_DEPTH", GL_MAX_TEXTURE_STACK_DEPTH},
    {"GL_MAX_VIEWPORT_DIMS", GL_MAX_VIEWPORT_DIMS},
    {"GL_MAX_CLIENT_ATTRIB_STACK_DEPTH", GL_MAX_CLIENT_ATTRIB_STACK_DEPTH},
    {"GL_SUBPIXEL_BITS", GL_SUBPIXEL_BITS},
    {"GL_INDEX_BITS", GL_INDEX_BITS},
    {"GL_RED_BITS", GL_RED_BITS},
    {"GL_GREEN_BITS", GL_GREEN_BITS},
    {"GL_BLUE_BITS", GL_BLUE_BITS},
    {"GL_ALPHA_BITS", GL_ALPHA_BITS},
    {"GL_DEPTH_BITS", GL_DEPTH_BITS},
    {"GL_STENCIL_BITS", GL_STENCIL_BITS},
    {"GL_ACCUM_RED_BITS", GL_ACCUM_RED_BITS},
    {"GL_ACCUM_GREEN_BITS", GL_ACCUM_GREEN_BITS},
    {"GL_ACCUM_BLUE_BITS", GL_ACCUM_BLUE_BITS},
    {"GL_ACCUM_ALPHA_BITS", GL_ACCUM_ALPHA_BITS},
    {"GL_NAME_STACK_DEPTH", GL_NAME_STACK_DEPTH},
    {"GL_AUTO_NORMAL", GL_AUTO_NORMAL},
    {"GL_MAP1_COLOR_4", GL_MAP1_COLOR_4},
    {"GL_MAP1_INDEX", GL_MAP1_INDEX},
    {"GL_MAP1_NORMAL", GL_MAP1_NORMAL},
    {"GL_MAP1_TEXTURE_COORD_1", GL_MAP1_TEXTURE_COORD_1},
    {"GL_MAP1_TEXTURE_COORD_2", GL_MAP1_TEXTURE_COORD_2},
    {"GL_MAP1_TEXTURE_COORD_3", GL_MAP1_TEXTURE_COORD_3},
    {"GL_MAP1_TEXTURE_COORD_4", GL_MAP1_TEXTURE_COORD_4},
    {"GL_MAP1_VERTEX_3", GL_MAP1_VERTEX_3},
    {"GL_MAP1_VERTEX_4", GL_MAP1_VERTEX_4},
    {"GL_MAP2_COLOR_4", GL_MAP2_COLOR_4},
    {"GL_MAP2_INDEX", GL_MAP2_INDEX},
    {"GL_MAP2_NORMAL", GL_MAP2_NORMAL},
    {"GL_MAP2_TEXTURE_COORD_1", GL_MAP2_TEXTURE_COORD_1},
    {"GL_MAP2_TEXTURE_COORD_2", GL_MAP2_TEXTURE_COORD_2},
    {"GL_MAP2_TEXTURE_COORD_3", GL_MAP2_TEXTURE_COORD_3},
    {"GL_MAP2_TEXTURE_COORD_4", GL_MAP2_TEXTURE_COORD_4},
    {"GL_MAP2_VERTEX_3", GL_MAP2_VERTEX_3},
    {"GL_MAP2_VERTEX_4", GL_MAP2_VERTEX_4},
    {"GL_MAP1_GRID_DOMAIN", GL_MAP1_GRID_DOMAIN},
    {"GL_MAP1_GRID_SEGMENTS", GL_MAP1_GRID_SEGMENTS},
    {"GL_MAP2_GRID_DOMAIN", GL_MAP2_GRID_DOMAIN},
    {"GL_MAP2_GRID_SEGMENTS", GL_MAP2_GRID_SEGMENTS},
    {"GL_TEXTURE_1D", GL_TEXTURE_1D},
    {"GL_TEXTURE_2D", GL_TEXTURE_2D},
    {"GL_FEEDBACK_BUFFER_POINTER", GL_FEEDBACK_BUFFER_POINTER},
    {"GL_FEEDBACK_BUFFER_SIZE", GL_FEEDBACK_BUFFER_SIZE},
    {"GL_FEEDBACK_BUFFER_TYPE", GL_FEEDBACK_BUFFER_TYPE},
    {"GL_SELECTION_BUFFER_POINTER", GL_SELECTION_BUFFER_POINTER},
    {"GL_SELECTION_BUFFER_SIZE", GL_SELECTION_BUFFER_SIZE},
    {"GL_TEXTURE_WIDTH", GL_TEXTURE_WIDTH},
    {"GL_TEXTURE_HEIGHT", GL_TEXTURE_HEIGHT},
    {"GL_TEXTURE_INTERNAL_FORMAT", GL_TEXTURE_INTERNAL_FORMAT},
    {"GL_TEXTURE_BORDER_COLOR", GL_TEXTURE_BORDER_COLOR},
    {"GL_TEXTURE_BORDER", GL_TEXTURE_BORDER},
    {"GL_TEXTURE_TARGET", GL_TEXTURE_TARGET},
    {"GL_DONT_CARE", GL_DONT_CARE},
    {"GL_FASTEST", GL_FASTEST},
    {"GL_NICEST", GL_NICEST},
    {"GL_AMBIENT", GL_AMBIENT},
    {"GL_DIFFUSE", GL_DIFFUSE},
    {"GL_SPECULAR", GL_SPECULAR},
    {"GL_POSITION", GL_POSITION},
    {"GL_SPOT_DIRECTION", GL_SPOT_DIRECTION},
    {"GL_SPOT_EXPONENT", GL_SPOT_EXPONENT},
    {"GL_SPOT_CUTOFF", GL_SPOT_CUTOFF},
    {"GL_CONSTANT_ATTENUATION", GL_CONSTANT_ATTENUATION},
    {"GL_LINEAR_ATTENUATION", GL_LINEAR_ATTENUATION},
    {"GL_QUADRATIC_ATTENUATION", GL_QUADRATIC_ATTENUATION},
    {"GL_COMPILE", GL_COMPILE},
    {"GL_COMPILE_AND_EXECUTE", GL_COMPILE_AND_EXECUTE},
    {"GL_BYTE", GL_BYTE},
    {"GL_UNSIGNED_BYTE", GL_UNSIGNED_BYTE},
    {"GL_SHORT", GL_SHORT},
    {"GL_UNSIGNED_SHORT", GL_UNSIGNED_SHORT},
    {"GL_INT", GL_INT},
    {"GL_UNSIGNED_INT", GL_UNSIGNED_INT},
    {"GL_FLOAT", GL_FLOAT},
    {"GL_2_BYTES", GL_2_BYTES},
    {"GL_3_BYTES", GL_3_BYTES},
    {"GL_4_BYTES", GL_4_BYTES},
    {"GL_DOUBLE", GL_DOUBLE},
    {"GL_HALF_FLOAT", GL_HALF_FLOAT},
    {"GL_FIXED", GL_FIXED},
    {"GL_INT64_ARB", GL_INT64_ARB},
    {"GL_UNSIGNED_INT64_ARB", GL_UNSIGNED_INT64_ARB},
    {"GL_CLEAR", GL_CLEAR},
    {"GL_AND", GL_AND},
    {"GL_AND_REVERSE", GL_AND_REVERSE},
    {"GL_COPY", GL_COPY},
    {"GL_AND_INVERTED", GL_AND_INVERTED},
    {"GL_NOOP", GL_NOOP},
    {"GL_XOR", GL_XOR},
    {"GL_OR", GL_OR},
    {"GL_NOR", GL_NOR},
    {"GL_EQUIV", GL_EQUIV},
    {"GL_INVERT", GL_INVERT},
    {"GL_OR_REVERSE", GL_OR_REVERSE},
    {"GL_COPY_INVERTED", GL_COPY_INVERTED},
    {"GL_OR_INVERTED", GL_OR_INVERTED},
    {"GL_NAND", GL_NAND},
    {"GL_SET", GL_SET},
    {"GL_EMISSION", GL_EMISSION},
    {"GL_SHININESS", GL_SHININESS},
    {"GL_AMBIENT_AND_DIFFUSE", GL_AMBIENT_AND_DIFFUSE},
    {"GL_COLOR_INDEXES", GL_COLOR_INDEXES},
    {"GL_MODELVIEW", GL_MODELVIEW},
    {"GL_PROJECTION", GL_PROJECTION},
    {"GL_TEXTURE", GL_TEXTURE},
    {"GL_COLOR", GL_COLOR},
    {"GL_DEPTH", GL_DEPTH},
    {"GL_STENCIL", GL_STENCIL},
    {"GL_COLOR_INDEX", GL_COLOR_INDEX},
    {"GL_STENCIL_INDEX", GL_STENCIL_INDEX},
    {"GL_DEPTH_COMPONENT", GL_DEPTH_COMPONENT},
    {"GL_RED", GL_RED},
    {"GL_GREEN", GL_GREEN},
    {"GL_BLUE", GL_BLUE},
    {"GL_ALPHA", GL_ALPHA},
    {"GL_RGB", GL_RGB},
    {"GL_RGBA", GL_RGBA},
    {"GL_LUMINANCE", GL_LUMINANCE},
    {"GL_LUMINANCE_ALPHA", GL_LUMINANCE_ALPHA},
    {"GL_BITMAP", GL_BITMAP},
    {"GL_POINT", GL_POINT},
    {"GL_LINE", GL_LINE},
    {"GL_FILL", GL_FILL},
    {"GL_RENDER", GL_RENDER},
    {"GL_FEEDBACK", GL_FEEDBACK},
    {"GL_SELECT", GL_SELECT},
    {"GL_FLAT", GL_FLAT},
    {"GL_SMOOTH", GL_SMOOTH},
    {"GL_KEEP", GL_KEEP},
    {"GL_REPLACE", GL_REPLACE},
    {"GL_INCR", GL_INCR},
    {"GL_DECR", GL_DECR},
    {"GL_VENDOR", GL_VENDOR},
    {"GL_RENDERER", GL_RENDERER},
    {"GL_VERSION", GL_VERSION},
    {"GL_EXTENSIONS", GL_EXTENSIONS},
    {"GL_S", GL_S},
    {"GL_T", GL_T},
    {"GL_R", GL_R},
    {"GL_Q", GL_Q},
    {"GL_MODULATE", GL_MODULATE},
    {"GL_DECAL", GL_DECAL},
    {"GL_TEXTURE_ENV_MODE", GL_TEXTURE_ENV_MODE},
    {"GL_TEXTURE_ENV_COLOR", GL_TEXTURE_ENV_COLOR},
    {"GL_TEXTURE_ENV", GL_TEXTURE_ENV},
    {"GL_EYE_LINEAR", GL_EYE_LINEAR},
    {"GL_OBJECT_LINEAR", GL_OBJECT_LINEAR},
    {"GL_SPHERE_MAP", GL_SPHERE_MAP},
    {"GL_TEXTURE_GEN_MODE", GL_TEXTURE_GEN_MODE},
    {"GL_OBJECT_PLANE", GL_OBJECT_PLANE},
    {"GL_EYE_PLANE", GL_EYE_PLANE},
    {"GL_NEAREST", GL_NEAREST},
    {"GL_LINEAR", GL_LINEAR},
    {"GL_NEAREST_MIPMAP_NEAREST", GL_NEAREST_MIPMAP_NEAREST},
    {"GL_LINEAR_MIPMAP_NEAREST", GL_LINEAR_MIPMAP_NEAREST},
    {"GL_NEAREST_MIPMAP_LINEAR", GL_NEAREST_MIPMAP_LINEAR},
    {"GL_LINEAR_MIPMAP_LINEAR", GL_LINEAR_MIPMAP_LINEAR},
    {"GL_TEXTURE_MAG_FILTER", GL_TEXTURE_MAG_FILTER},
    {"GL_TEXTURE_MIN_FILTER", GL_TEXTURE_MIN_FILTER},
    {"GL_TEXTURE_WRAP_S", GL_TEXTURE_WRAP_S},
    {"GL_TEXTURE_WRAP_T", GL_TEXTURE_WRAP_T},
    {"GL_CLAMP", GL_CLAMP},
    {"GL_REPEAT", GL_REPEAT},
    {"GL_POLYGON_OFFSET_UNITS", GL_POLYGON_OFFSET_UNITS},
    {"GL_POLYGON_OFFSET_POINT", GL_POLYGON_OFFSET_POINT},
    {"GL_POLYGON_OFFSET_LINE", GL_POLYGON_OFFSET_LINE},
    {"GL_R3_G3_B2", GL_R3_G3_B2},
    {"GL_V2F", GL_V2F},
    {"GL_V3F", GL_V3F},
    {"GL_C4UB_V2F", GL_C4UB_V2F},
    {"GL_C4UB_V3F", GL_C4UB_V3F},
    {"GL_C3F_V3F", GL_C3F_V3F},
    {"GL_N3F_V3F", GL_N3F_V3F},
    {"GL_C4F_N3F_V3F", GL_C4F_N3F_V3F},
    {"GL_T2F_V3F", GL_T2F_V3F},
    {"GL_T4F_V4F", GL_T4F_V4F},
    {"GL_T2F_C4UB_V3F", GL_T2F_C4UB_V3F},
    {"GL_T2F_C3F_V3F", GL_T2F_C3F_V3F},
    {"GL_T2F_N3F_V3F", GL_T2F_N3F_V3F},
    {"GL_T2F_C4F_N3F_V3F", GL_T2F_C4F_N3F_V3F},
    {"GL_T4F_C4F_N3F_V4F", GL_T4F_C4F_N3F_V4F},
    {"GL_CLIP_DISTANCE0", GL_CLIP_DISTANCE0},
    {"GL_CLIP_DISTANCE1", GL_CLIP_DISTANCE1},
    {"GL_CLIP_DISTANCE2", GL_CLIP_DISTANCE2},
    {"GL_CLIP_DISTANCE3", GL_CLIP_DISTANCE3},
    {"GL_CLIP_DISTANCE4", GL_CLIP_DISTANCE4},
    {"GL_CLIP_DISTANCE5", GL_CLIP_DISTANCE5},
    {"GL_CLIP_DISTANCE6", GL_CLIP_DISTANCE6},
    {"GL_CLIP_DISTANCE7", GL_CLIP_DISTANCE7},
    {"GL_LIGHT0", GL_LIGHT0},
    {"GL_LIGHT1", GL_LIGHT1},
    {"GL_LIGHT2", GL_LIGHT2},
    {"GL_LIGHT3", GL_LIGHT3},
    {"GL_LIGHT4", GL_LIGHT4},
    {"GL_LIGHT5", GL_LIGHT5},
    {"GL_LIGHT6", GL_LIGHT6},
    {"GL_LIGHT7", GL_LIGHT7},
    {"GL_ABGR_EXT", GL_ABGR_EXT},
    {"GL_CONSTANT_COLOR", GL_CONSTANT_COLOR},
    {"GL_ONE_MINUS_CONSTANT_COLOR", GL_ONE_MINUS_CONSTANT_COLOR},
    {"GL_CONSTANT_ALPHA", GL_CONSTANT_ALPHA},
    {"GL_ONE_MINUS_CONSTANT_ALPHA", GL_ONE_MINUS_CONSTANT_ALPHA},
    {"GL_BLEND_COLOR", GL_BLEND_COLOR},
    {"GL_FUNC_ADD", GL_FUNC_ADD},
    {"GL_MIN", GL_MIN},
    {"GL_MAX", GL_MAX},
    {"GL_BLEND_EQUATION", GL_BLEND_EQUATION},
    {"GL_FUNC_SUBTRACT", GL_FUNC_SUBTRACT},
    {"GL_FUNC_REVERSE_SUBTRACT", GL_FUNC_REVERSE_SUBTRACT},
    {"GL_CMYK_EXT", GL_CMYK_EXT},
    {"GL_CMYKA_EXT", GL_CMYKA_EXT},
    {"GL_PACK_CMYK_HINT_EXT", GL_PACK_CMYK_HINT_EXT},
    {"GL_UNPACK_CMYK_HINT_EXT", GL_UNPACK_CMYK_HINT_EXT},
    {"GL_CONVOLUTION_1D", GL_CONVOLUTION_1D},
    {"GL_CONVOLUTION_2D", GL_CONVOLUTION_2D},
    {"GL_SEPARABLE_2D", GL_SEPARABLE_2D},
    {"GL_CONVOLUTION_BORDER_MODE", GL_CONVOLUTION_BORDER_MODE},
    {"GL_CONVOLUTION_FILTER_SCALE", GL_CONVOLUTION_FILTER_SCALE},
    {"GL_CONVOLUTION_FILTER_BIAS", GL_CONVOLUTION_FILTER_BIAS},
    {"GL_REDUCE", GL_REDUCE},
    {"GL_CONVOLUTION_FORMAT", GL_CONVOLUTION_FORMAT},
    {"GL_CONVOLUTION_WIDTH", GL_CONVOLUTION_WIDTH},
    {"GL_CONVOLUTION_HEIGHT", GL_CONVOLUTION_HEIGHT},
    {"GL_MAX_CONVOLUTION_WIDTH", GL_MAX_CONVOLUTION_WIDTH},
    {"GL_MAX_CONVOLUTION_HEIGHT", GL_MAX_CONVOLUTION_HEIGHT},
    {"GL_POST_CONVOLUTION_RED_SCALE", GL_POST_CONVOLUTION_RED_SCALE},
    {"GL_POST_CONVOLUTION_GREEN_SCALE", GL_POST_CONVOLUTION_GREEN_SCALE},
    {"GL_POST_CONVOLUTION_BLUE_SCALE", GL_POST_CONVOLUTION_BLUE_SCALE},
    {"GL_POST_CONVOLUTION_ALPHA_SCALE", GL_POST_CONVOLUTION_ALPHA_SCALE},
    {"GL_POST_CONVOLUTION_RED_BIAS", GL_POST_CONVOLUTION_RED_BIAS},
    {"GL_POST_CONVOLUTION_GREEN_BIAS", GL_POST_CONVOLUTION_GREEN_BIAS},
    {"GL_POST_CONVOLUTION_BLUE_BIAS", GL_POST_CONVOLUTION_BLUE_BIAS},
    {"GL_POST_CONVOLUTION_ALPHA_BIAS", GL_POST_CONVOLUTION_ALPHA_BIAS},
    {"GL_HISTOGRAM", GL_HISTOGRAM},
    {"GL_PROXY_HISTOGRAM", GL_PROXY_HISTOGRAM},
    {"GL_HISTOGRAM_WIDTH", GL_HISTOGRAM_WIDTH},
    {"GL_HISTOGRAM_FORMAT", GL_HISTOGRAM_FORMAT},
    {"GL_HISTOGRAM_RED_SIZE", GL_HISTOGRAM_RED_SIZE},
    {"GL_HISTOGRAM_GREEN_SIZE", GL_HISTOGRAM_GREEN_SIZE},
    {"GL_HISTOGRAM_BLUE_SIZE", GL_HISTOGRAM_BLUE_SIZE},
    {"GL_HISTOGRAM_ALPHA_SIZE", GL_HISTOGRAM_ALPHA_SIZE},
    {"GL_HISTOGRAM_LUMINANCE_SIZE", GL_HISTOGRAM_LUMINANCE_SIZE},
    {"GL_HISTOGRAM_SINK", GL_HISTOGRAM_SINK},
    {"GL_MINMAX", GL_MINMAX},
    {"GL_MINMAX_FORMAT", GL_MINMAX_FORMAT},
    {"GL_MINMAX_SINK", GL_MINMAX_SINK},
    {"GL_TABLE_TOO_LARGE", GL_TABLE_TOO_LARGE},
    {"GL_UNSIGNED_BYTE_3_3_2", GL_UNSIGNED_BYTE_3_3_2},
    {"GL_UNSIGNED_SHORT_4_4_4_4", GL_UNSIGNED_SHORT_4_4_4_4},
    {"GL_UNSIGNED_SHORT_5_5_5_1", GL_UNSIGNED_SHORT_5_5_5_1},
    {"GL_UNSIGNED_INT_8_8_8_8", GL_UNSIGNED_INT_8_8_8_8},
    {"GL_UNSIGNED_INT_10_10_10_2", GL_UNSIGNED_INT_10_10_10_2},
    {"GL_POLYGON_OFFSET_FILL", GL_POLYGON_OFFSET_FILL},
    {"GL_POLYGON_OFFSET_FACTOR", GL_POLYGON_OFFSET_FACTOR},
    {"GL_POLYGON_OFFSET_BIAS_EXT", GL_POLYGON_OFFSET_BIAS_EXT},
    {"GL_RESCALE_NORMAL", GL_RESCALE_NORMAL},
    {"GL_ALPHA4", GL_ALPHA4},
    {"GL_ALPHA8", GL_ALPHA8},
    {"GL_ALPHA12", GL_ALPHA12},
    {"GL_ALPHA16", GL_ALPHA16},
    {"GL_LUMINANCE4", GL_LUMINANCE4},
    {"GL_LUMINANCE8", GL_LUMINANCE8},
    {"GL_LUMINANCE12", GL_LUMINANCE12},
    {"GL_LUMINANCE16", GL_LUMINANCE16},
    {"GL_LUMINANCE4_ALPHA4", GL_LUMINANCE4_ALPHA4},
    {"GL_LUMINANCE6_ALPHA2", GL_LUMINANCE6_ALPHA2},
    {"GL_LUMINANCE8_ALPHA8", GL_LUMINANCE8_ALPHA8},
    {"GL_LUMINANCE12_ALPHA4", GL_LUMINANCE12_ALPHA4},
    {"GL_LUMINANCE12_ALPHA12", GL_LUMINANCE12_ALPHA12},
    {"GL_LUMINANCE16_ALPHA16", GL_LUMINANCE16_ALPHA16},
    {"GL_INTENSITY", GL_INTENSITY},
    {"GL_INTENSITY4", GL_INTENSITY4},
    {"GL_INTENSITY8", GL_INTENSITY8},
    {"GL_INTENSITY12", GL_INTENSITY12},
    {"GL_INTENSITY16", GL_INTENSITY16},
    {"GL_RGB2_EXT", GL_RGB2_EXT},
    {"GL_RGB4", GL_RGB4},
    {"GL_RGB5", GL_RGB5},
    {"GL_RGB8", GL_RGB8},
    {"GL_RGB10", GL_RGB10},
    {"GL_RGB12", GL_RGB12},
    {"GL_RGB16", GL_RGB16},
    {"GL_RGBA2", GL_RGBA2},
    {"GL_RGBA4", GL_RGBA4},
    {"GL_RGB5_A1", GL_RGB5_A1},
    {"GL_RGBA8", GL_RGBA8},
    {"GL_RGB10_A2", GL_RGB10_A2},
    {"GL_RGBA12", GL_RGBA12},
    {"GL_RGBA16", GL_RGBA16},
    {"GL_TEXTURE_RED_SIZE", GL_TEXTURE_RED_SIZE},
    {"GL_TEXTURE_GREEN_SIZE", GL_TEXTURE_GREEN_SIZE},
    {"GL_TEXTURE_BLUE_SIZE", GL_TEXTURE_BLUE_SIZE},
    {"GL_TEXTURE_ALPHA_SIZE", GL_TEXTURE_ALPHA_SIZE},
    {"GL_TEXTURE_LUMINANCE_SIZE", GL_TEXTURE_LUMINANCE_SIZE},
    {"GL_TEXTURE_INTENSITY_SIZE", GL_TEXTURE_INTENSITY_SIZE},
    {"GL_REPLACE_EXT", GL_REPLACE_EXT},
    {"GL_PROXY_TEXTURE_1D", GL_PROXY_TEXTURE_1D},
    {"GL_PROXY_TEXTURE_2D", GL_PROXY_TEXTURE_2D},
    {"GL_TEXTURE_TOO_LARGE_EXT", GL_TEXTURE_TOO_LARGE_EXT},
    {"GL_TEXTURE_PRIORITY", GL_TEXTURE_PRIORITY},
    {"GL_TEXTURE_RESIDENT", GL_TEXTURE_RESIDENT},
    {"GL_TEXTURE_BINDING_1D", GL_TEXTURE_BINDING_1D},
    {"GL_TEXTURE_BINDING_2D", GL_TEXTURE_BINDING_2D},
    {"GL_TEXTURE_BINDING_3D", GL_TEXTURE_BINDING_3D},
    {"GL_PACK_SKIP_IMAGES", GL_PACK_SKIP_IMAGES},
    {"GL_PACK_IMAGE_HEIGHT", GL_PACK_IMAGE_HEIGHT},
    {"GL_UNPACK_SKIP_IMAGES", GL_UNPACK_SKIP_IMAGES},
    {"GL_UNPACK_IMAGE_HEIGHT", GL_UNPACK_IMAGE_HEIGHT},
    {"GL_TEXTURE_3D", GL_TEXTURE_3D},
    {"GL_PROXY_TEXTURE_3D", GL_PROXY_TEXTURE_3D},
    {"GL_TEXTURE_DEPTH", GL_TEXTURE_DEPTH},
    {"GL_TEXTURE_WRAP_R", GL_TEXTURE_WRAP_R},
    {"GL_MAX_3D_TEXTURE_SIZE", GL_MAX_3D_TEXTURE_SIZE},
    {"GL_VERTEX_ARRAY", GL_VERTEX_ARRAY},
    {"GL_NORMAL_ARRAY", GL_NORMAL_ARRAY},
    {"GL_COLOR_ARRAY", GL_COLOR_ARRAY},
    {"GL_INDEX_ARRAY", GL_INDEX_ARRAY},
    {"GL_TEXTURE_COORD_ARRAY", GL_TEXTURE_COORD_ARRAY},
    {"GL_EDGE_FLAG_ARRAY", GL_EDGE_FLAG_ARRAY},
    {"GL_VERTEX_ARRAY_SIZE", GL_VERTEX_ARRAY_SIZE},
    {"GL_VERTEX_ARRAY_TYPE", GL_VERTEX_ARRAY_TYPE},
    {"GL_VERTEX_ARRAY_STRIDE", GL_VERTEX_ARRAY_STRIDE},
    {"GL_VERTEX_ARRAY_COUNT_EXT", GL_VERTEX_ARRAY_COUNT_EXT},
    {"GL_NORMAL_ARRAY_TYPE", GL_NORMAL_ARRAY_TYPE},
    {"GL_NORMAL_ARRAY_STRIDE", GL_NORMAL_ARRAY_STRIDE},
    {"GL_NORMAL_ARRAY_COUNT_EXT", GL_NORMAL_ARRAY_COUNT_EXT},
    {"GL_COLOR_ARRAY_SIZE", GL_COLOR_ARRAY_SIZE},
    {"GL_COLOR_ARRAY_TYPE", GL_COLOR_ARRAY_TYPE},
    {"GL_COLOR_ARRAY_STRIDE", GL_COLOR_ARRAY_STRIDE},
    {"GL_COLOR_ARRAY_COUNT_EXT", GL_COLOR_ARRAY_COUNT_EXT},
    {"GL_INDEX_ARRAY_TYPE", GL_INDEX_ARRAY_TYPE},
    {"GL_INDEX_ARRAY_STRIDE", GL_INDEX_ARRAY_STRIDE},
    {"GL_INDEX_ARRAY_COUNT_EXT", GL_INDEX_ARRAY_COUNT_EXT},
    {"GL_TEXTURE_COORD_ARRAY_SIZE", GL_TEXTURE_COORD_ARRAY_SIZE},
    {"GL_TEXTURE_COORD_ARRAY_TYPE", GL_TEXTURE_COORD_ARRAY_TYPE},
    {"GL_TEXTURE_COORD_ARRAY_STRIDE", GL_TEXTURE_COORD_ARRAY_STRIDE},
    {"GL_TEXTURE_COORD_ARRAY_COUNT_EXT", GL_TEXTURE_COORD_ARRAY_COUNT_EXT},
    {"GL_EDGE_FLAG_ARRAY_STRIDE", GL_EDGE_FLAG_ARRAY_STRIDE},
    {"GL_EDGE_FLAG_ARRAY_COUNT_EXT", GL_EDGE_FLAG_ARRAY_COUNT_EXT},
    {"GL_VERTEX_ARRAY_POINTER", GL_VERTEX_ARRAY_POINTER},
    {"GL_NORMAL_ARRAY_POINTER", GL_NORMAL_ARRAY_POINTER},
    {"GL_COLOR_ARRAY_POINTER", GL_COLOR_ARRAY_POINTER},
    {"GL_INDEX_ARRAY_POINTER", GL_INDEX_ARRAY_POINTER},
    {"GL_TEXTURE_COORD_ARRAY_POINTER", GL_TEXTURE_COORD_ARRAY_POINTER},
    {"GL_EDGE_FLAG_ARRAY_POINTER", GL_EDGE_FLAG_ARRAY_POINTER},
    {"GL_INTERLACE_SGIX", GL_INTERLACE_SGIX},
    {"GL_DETAIL_TEXTURE_2D_SGIS", GL_DETAIL_TEXTURE_2D_SGIS},
    {"GL_DETAIL_TEXTURE_2D_BINDING_SGIS", GL_DETAIL_TEXTURE_2D_BINDING_SGIS},
    {"GL_LINEAR_DETAIL_SGIS", GL_LINEAR_DETAIL_SGIS},
    {"GL_LINEAR_DETAIL_ALPHA_SGIS", GL_LINEAR_DETAIL_ALPHA_SGIS},
    {"GL_LINEAR_DETAIL_COLOR_SGIS", GL_LINEAR_DETAIL_COLOR_SGIS},
    {"GL_DETAIL_TEXTURE_LEVEL_SGIS", GL_DETAIL_TEXTURE_LEVEL_SGIS},
    {"GL_DETAIL_TEXTURE_MODE_SGIS", GL_DETAIL_TEXTURE_MODE_SGIS},
    {"GL_DETAIL_TEXTURE_FUNC_POINTS_SGIS", GL_DETAIL_TEXTURE_FUNC_POINTS_SGIS},
    {"GL_MULTISAMPLE", GL_MULTISAMPLE},
    {"GL_SAMPLE_ALPHA_TO_COVERAGE", GL_SAMPLE_ALPHA_TO_COVERAGE},
    {"GL_SAMPLE_ALPHA_TO_ONE", GL_SAMPLE_ALPHA_TO_ONE},
    {"GL_SAMPLE_COVERAGE", GL_SAMPLE_COVERAGE},
    {"GL_1PASS_EXT", GL_1PASS_EXT},
    {"GL_2PASS_0_EXT", GL_2PASS_0_EXT},
    {"GL_2PASS_1_EXT", GL_2PASS_1_EXT},
    {"GL_4PASS_0_EXT", GL_4PASS_0_EXT},
    {"GL_4PASS_1_EXT", GL_4PASS_1_EXT},
    {"GL_4PASS_2_EXT", GL_4PASS_2_EXT},
    {"GL_4PASS_3_EXT", GL_4PASS_3_EXT},
    {"GL_SAMPLE_BUFFERS", GL_SAMPLE_BUFFERS},
    {"GL_SAMPLES", GL_SAMPLES},
    {"GL_SAMPLE_COVERAGE_VALUE", GL_SAMPLE_COVERAGE_VALUE},
    {"GL_SAMPLE_COVERAGE_INVERT", GL_SAMPLE_COVERAGE_INVERT},
    {"GL_SAMPLE_PATTERN_EXT", GL_SAMPLE_PATTERN_EXT},
    {"GL_LINEAR_SHARPEN_SGIS", GL_LINEAR_SHARPEN_SGIS},
    {"GL_LINEAR_SHARPEN_ALPHA_SGIS", GL_LINEAR_SHARPEN_ALPHA_SGIS},
    {"GL_LINEAR_SHARPEN_COLOR_SGIS", GL_LINEAR_SHARPEN_COLOR_SGIS},
    {"GL_SHARPEN_TEXTURE_FUNC_POINTS_SGIS", GL_SHARPEN_TEXTURE_FUNC_POINTS_SGIS},
    {"GL_COLOR_MATRIX", GL_COLOR_MATRIX},
    {"GL_COLOR_MATRIX_STACK_DEPTH", GL_COLOR_MATRIX_STACK_DEPTH},
    {"GL_MAX_COLOR_MATRIX_STACK_DEPTH", GL_MAX_COLOR_MATRIX_STACK_DEPTH},
    {"GL_POST_COLOR_MATRIX_RED_SCALE", GL_POST_COLOR_MATRIX_RED_SCALE},
    {"GL_POST_COLOR_MATRIX_GREEN_SCALE", GL_POST_COLOR_MATRIX_GREEN_SCALE},
    {"GL_POST_COLOR_MATRIX_BLUE_SCALE", GL_POST_COLOR_MATRIX_BLUE_SCALE},
    {"GL_POST_COLOR_MATRIX_ALPHA_SCALE", GL_POST_COLOR_MATRIX_ALPHA_SCALE},
    {"GL_POST_COLOR_MATRIX_RED_BIAS", GL_POST_COLOR_MATRIX_RED_BIAS},
    {"GL_POST_COLOR_MATRIX_GREEN_BIAS", GL_POST_COLOR_MATRIX_GREEN_BIAS},
    {"GL_POST_COLOR_MATRIX_BLUE_BIAS", GL_POST_COLOR_MATRIX_BLUE_BIAS},
    {"GL_POST_COLOR_MATRIX_ALPHA_BIAS", GL_POST_COLOR_MATRIX_ALPHA_BIAS},
    {"GL_TEXTURE_COLOR_TABLE_SGI", GL_TEXTURE_COLOR_TABLE_SGI},
    {"GL_PROXY_TEXTURE_COLOR_TABLE_SGI", GL_PROXY_TEXTURE_COLOR_TABLE_SGI},
    {"GL_TEXTURE_ENV_BIAS_SGIX", GL_TEXTURE_ENV_BIAS_SGIX},
    {"GL_TEXTURE_COMPARE_FAIL_VALUE_ARB", GL_TEXTURE_COMPARE_FAIL_VALUE_ARB},
    {"GL_BLEND_DST_RGB", GL_BLEND_DST_RGB},
    {"GL_BLEND_SRC_RGB", GL_BLEND_SRC_RGB},
    {"GL_BLEND_DST_ALPHA", GL_BLEND_DST_ALPHA},
    {"GL_BLEND_SRC_ALPHA", GL_BLEND_SRC_ALPHA},
    {"GL_422_EXT", GL_422_EXT},
    {"GL_422_REV_EXT", GL_422_REV_EXT},
    {"GL_422_AVERAGE_EXT", GL_422_AVERAGE_EXT},
    {"GL_422_REV_AVERAGE_EXT", GL_422_REV_AVERAGE_EXT},
    {"GL_COLOR_TABLE", GL_COLOR_TABLE},
    {"GL_POST_CONVOLUTION_COLOR_TABLE", GL_POST_CONVOLUTION_COLOR_TABLE},
    {"GL_POST_COLOR_MATRIX_COLOR_TABLE", GL_POST_COLOR_MATRIX_COLOR_TABLE},
    {"GL_PROXY_COLOR_TABLE", GL_PROXY_COLOR_TABLE},
    {"GL_PROXY_POST_CONVOLUTION_COLOR_TABLE", GL_PROXY_POST_CONVOLUTION_COLOR_TABLE},
    {"GL_PROXY_POST_COLOR_MATRIX_COLOR_TABLE", GL_PROXY_POST_COLOR_MATRIX_COLOR_TABLE},
    {"GL_COLOR_TABLE_SCALE", GL_COLOR_TABLE_SCALE},
    {"GL_COLOR_TABLE_BIAS", GL_COLOR_TABLE_BIAS},
    {"GL_COLOR_TABLE_FORMAT", GL_COLOR_TABLE_FORMAT},
    {"GL_COLOR_TABLE_WIDTH", GL_COLOR_TABLE_WIDTH},
    {"GL_COLOR_TABLE_RED_SIZE", GL_COLOR_TABLE_RED_SIZE},
    {"GL_COLOR_TABLE_GREEN_SIZE", GL_COLOR_TABLE_GREEN_SIZE},
    {"GL_COLOR_TABLE_BLUE_SIZE", GL_COLOR_TABLE_BLUE_SIZE},
    {"GL_COLOR_TABLE_ALPHA_SIZE", GL_COLOR_TABLE_ALPHA_SIZE},
    {"GL_COLOR_TABLE_LUMINANCE_SIZE", GL_COLOR_TABLE_LUMINANCE_SIZE},
    {"GL_COLOR_TABLE_INTENSITY_SIZE", GL_COLOR_TABLE_INTENSITY_SIZE},
    {"GL_BGR", GL_BGR},
    {"GL_BGRA", GL_BGRA},
    {"GL_COLOR_INDEX1_EXT", GL_COLOR_INDEX1_EXT},
    {"GL_COLOR_INDEX2_EXT", GL_COLOR_INDEX2_EXT},
    {"GL_COLOR_INDEX4_EXT", GL_COLOR_INDEX4_EXT},
    {"GL_COLOR_INDEX8_EXT", GL_COLOR_INDEX8_EXT},
    {"GL_COLOR_INDEX12_EXT", GL_COLOR_INDEX12_EXT},
    {"GL_COLOR_INDEX16_EXT", GL_COLOR_INDEX16_EXT},
    {"GL_MAX_ELEMENTS_VERTICES", GL_MAX_ELEMENTS_VERTICES},
    {"GL_MAX_ELEMENTS_INDICES", GL_MAX_ELEMENTS_INDICES},
    {"GL_PHONG_WIN", GL_PHONG_WIN},
    {"GL_PHONG_HINT_WIN", GL_PHONG_HINT_WIN},
    {"GL_FOG_SPECULAR_TEXTURE_WIN", GL_FOG_SPECULAR_TEXTURE_WIN},
    {"GL_TEXTURE_INDEX_SIZE_EXT", GL_TEXTURE_INDEX_SIZE_EXT},
    {"GL_PARAMETER_BUFFER_ARB", GL_PARAMETER_BUFFER_ARB},
    {"GL_PARAMETER_BUFFER_BINDING_ARB", GL_PARAMETER_BUFFER_BINDING_ARB},
    {"GL_CLIP_VOLUME_CLIPPING_HINT_EXT", GL_CLIP_VOLUME_CLIPPING_HINT_EXT},
    {"GL_DUAL_ALPHA4_SGIS", GL_DUAL_ALPHA4_SGIS},
    {"GL_DUAL_ALPHA8_SGIS", GL_DUAL_ALPHA8_SGIS},
    {"GL_DUAL_ALPHA12_SGIS", GL_DUAL_ALPHA12_SGIS},
    {"GL_DUAL_ALPHA16_SGIS", GL_DUAL_ALPHA16_SGIS},
    {"GL_DUAL_LUMINANCE4_SGIS", GL_DUAL_LUMINANCE4_SGIS},
    {"GL_DUAL_LUMINANCE8_SGIS", GL_DUAL_LUMINANCE8_SGIS},
    {"GL_DUAL_LUMINANCE12_SGIS", GL_DUAL_LUMINANCE12_SGIS},
    {"GL_DUAL_LUMINANCE16_SGIS", GL_DUAL_LUMINANCE16_SGIS},
    {"GL_DUAL_INTENSITY4_SGIS", GL_DUAL_INTENSITY4_SGIS},
    {"GL_DUAL_INTENSITY8_SGIS", GL_DUAL_INTENSITY8_SGIS},
    {"GL_DUAL_INTENSITY12_SGIS", GL_DUAL_INTENSITY12_SGIS},
    {"GL_DUAL_INTENSITY16_SGIS", GL_DUAL_INTENSITY16_SGIS},
    {"GL_DUAL_LUMINANCE_ALPHA4_SGIS", GL_DUAL_LUMINANCE_ALPHA4_SGIS},
    {"GL_DUAL_LUMINANCE_ALPHA8_SGIS", GL_DUAL_LUMINANCE_ALPHA8_SGIS},
    {"GL_QUAD_ALPHA4_SGIS", GL_QUAD_ALPHA4_SGIS},
    {"GL_QUAD_ALPHA8_SGIS", GL_QUAD_ALPHA8_SGIS},
    {"GL_QUAD_LUMINANCE4_SGIS", GL_QUAD_LUMINANCE4_SGIS},
    {"GL_QUAD_LUMINANCE8_SGIS", GL_QUAD_LUMINANCE8_SGIS},
    {"GL_QUAD_INTENSITY4_SGIS", GL_QUAD_INTENSITY4_SGIS},
    {"GL_QUAD_INTENSITY8_SGIS", GL_QUAD_INTENSITY8_SGIS},
    {"GL_DUAL_TEXTURE_SELECT_SGIS", GL_DUAL_TEXTURE_SELECT_SGIS},
    {"GL_QUAD_TEXTURE_SELECT_SGIS", GL_QUAD_TEXTURE_SELECT_SGIS},
    {"GL_POINT_SIZE_MIN", GL_POINT_SIZE_MIN},
    {"GL_POINT_SIZE_MAX", GL_POINT_SIZE_MAX},
    {"GL_POINT_FADE_THRESHOLD_SIZE", GL_POINT_FADE_THRESHOLD_SIZE},
    {"GL_POINT_DISTANCE_ATTENUATION", GL_POINT_DISTANCE_ATTENUATION},
    {"GL_FOG_FUNC_SGIS", GL_FOG_FUNC_SGIS},
    {"GL_FOG_FUNC_POINTS_SGIS", GL_FOG_FUNC_POINTS_SGIS},
    {"GL_MAX_FOG_FUNC_POINTS_SGIS", GL_MAX_FOG_FUNC_POINTS_SGIS},
    {"GL_CLAMP_TO_BORDER", GL_CLAMP_TO_BORDER},
    {"GL_TEXTURE_MULTI_BUFFER_HINT_SGIX", GL_TEXTURE_MULTI_BUFFER_HINT_SGIX},
    {"GL_CLAMP_TO_EDGE", GL_CLAMP_TO_EDGE},
    {"GL_PACK_SKIP_VOLUMES_SGIS", GL_PACK_SKIP_VOLUMES_SGIS},
    {"GL_PACK_IMAGE_DEPTH_SGIS", GL_PACK_IMAGE_DEPTH_SGIS},
    {"GL_UNPACK_SKIP_VOLUMES_SGIS", GL_UNPACK_SKIP_VOLUMES_SGIS},
    {"GL_UNPACK_IMAGE_DEPTH_SGIS", GL_UNPACK_IMAGE_DEPTH_SGIS},
    {"GL_TEXTURE_4D_SGIS", GL_TEXTURE_4D_SGIS},
    {"GL_PROXY_TEXTURE_4D_SGIS", GL_PROXY_TEXTURE_4D_SGIS},
    {"GL_TEXTURE_4DSIZE_SGIS", GL_TEXTURE_4DSIZE_SGIS},
    {"GL_TEXTURE_WRAP_Q_SGIS", GL_TEXTURE_WRAP_Q_SGIS},
    {"GL_MAX_4D_TEXTURE_SIZE_SGIS", GL_MAX_4D_TEXTURE_SIZE_SGIS},
    {"GL_PIXEL_TEX_GEN_SGIX", GL_PIXEL_TEX_GEN_SGIX},
    {"GL_TEXTURE_MIN_LOD", GL_TEXTURE_MIN_LOD},
    {"GL_TEXTURE_MAX_LOD", GL_TEXTURE_MAX_LOD},
    {"GL_TEXTURE_BASE_LEVEL", GL_TEXTURE_BASE_LEVEL},
    {"GL_TEXTURE_MAX_LEVEL", GL_TEXTURE_MAX_LEVEL},
    {"GL_PIXEL_TILE_BEST_ALIGNMENT_SGIX", GL_PIXEL_TILE_BEST_ALIGNMENT_SGIX},
    {"GL_PIXEL_TILE_CACHE_INCREMENT_SGIX", GL_PIXEL_TILE_CACHE_INCREMENT_SGIX},
    {"GL_PIXEL_TILE_WIDTH_SGIX", GL_PIXEL_TILE_WIDTH_SGIX},
    {"GL_PIXEL_TILE_HEIGHT_SGIX", GL_PIXEL_TILE_HEIGHT_SGIX},
    {"GL_PIXEL_TILE_GRID_WIDTH_SGIX", GL_PIXEL_TILE_GRID_WIDTH_SGIX},
    {"GL_PIXEL_TILE_GRID_HEIGHT_SGIX", GL_PIXEL_TILE_GRID_HEIGHT_SGIX},
    {"GL_PIXEL_TILE_GRID_DEPTH_SGIX", GL_PIXEL_TILE_GRID_DEPTH_SGIX},
    {"GL_PIXEL_TILE_CACHE_SIZE_SGIX", GL_PIXEL_TILE_CACHE_SIZE_SGIX},
    {"GL_FILTER4_SGIS", GL_FILTER4_SGIS},
    {"GL_TEXTURE_FILTER4_SIZE_SGIS", GL_TEXTURE_FILTER4_SIZE_SGIS},
    {"GL_SPRITE_SGIX", GL_SPRITE_SGIX},
    {"GL_SPRITE_MODE_SGIX", GL_SPRITE_MODE_SGIX},
    {"GL_SPRITE_AXIS_SGIX", GL_SPRITE_AXIS_SGIX},
    {"GL_SPRITE_TRANSLATION_SGIX", GL_SPRITE_TRANSLATION_SGIX},
    {"GL_SPRITE_AXIAL_SGIX", GL_SPRITE_AXIAL_SGIX},
    {"GL_SPRITE_OBJECT_ALIGNED_SGIX", GL_SPRITE_OBJECT_ALIGNED_SGIX},
    {"GL_SPRITE_EYE_ALIGNED_SGIX", GL_SPRITE_EYE_ALIGNED_SGIX},
    {"GL_TEXTURE_4D_BINDING_SGIS", GL_TEXTURE_4D_BINDING_SGIS},
    {"GL_IGNORE_BORDER_HP", GL_IGNORE_BORDER_HP},
    {"GL_CONSTANT_BORDER", GL_CONSTANT_BORDER},
    {"GL_REPLICATE_BORDER", GL_REPLICATE_BORDER},
    {"GL_CONVOLUTION_BORDER_COLOR", GL_CONVOLUTION_BORDER_COLOR},
    {"GL_IMAGE_SCALE_X_HP", GL_IMAGE_SCALE_X_HP},
    {"GL_IMAGE_SCALE_Y_HP", GL_IMAGE_SCALE_Y_HP},
    {"GL_IMAGE_TRANSLATE_X_HP", GL_IMAGE_TRANSLATE_X_HP},
    {"GL_IMAGE_TRANSLATE_Y_HP", GL_IMAGE_TRANSLATE_Y_HP},
    {"GL_IMAGE_ROTATE_ANGLE_HP", GL_IMAGE_ROTATE_ANGLE_HP},
    {"GL_IMAGE_ROTATE_ORIGIN_X_HP", GL_IMAGE_ROTATE_ORIGIN_X_HP},
    {"GL_IMAGE_ROTATE_ORIGIN_Y_HP", GL_IMAGE_ROTATE_ORIGIN_Y_HP},
    {"GL_IMAGE_MAG_FILTER_HP", GL_IMAGE_MAG_FILTER_HP},
    {"GL_IMAGE_MIN_FILTER_HP", GL_IMAGE_MIN_FILTER_HP},
    {"GL_IMAGE_CUBIC_WEIGHT_HP", GL_IMAGE_CUBIC_WEIGHT_HP},
    {"GL_CUBIC_HP", GL_CUBIC_HP},
    {"GL_AVERAGE_HP", GL_AVERAGE_HP},
    {"GL_IMAGE_TRANSFORM_2D_HP", GL_IMAGE_TRANSFORM_2D_HP},
    {"GL_POST_IMAGE_TRANSFORM_COLOR_TABLE_HP", GL_POST_IMAGE_TRANSFORM_COLOR_TABLE_HP},
    {"GL_PROXY_POST_IMAGE_TRANSFORM_COLOR_TABLE_HP", GL_PROXY_POST_IMAGE_TRANSFORM_COLOR_TABLE_HP},
    {"GL_OCCLUSION_TEST_HP", GL_OCCLUSION_TEST_HP},
    {"GL_OCCLUSION_TEST_RESULT_HP", GL_OCCLUSION_TEST_RESULT_HP},
    {"GL_TEXTURE_LIGHTING_MODE_HP", GL_TEXTURE_LIGHTING_MODE_HP},
    {"GL_TEXTURE_POST_SPECULAR_HP", GL_TEXTURE_POST_SPECULAR_HP},
    {"GL_TEXTURE_PRE_SPECULAR_HP", GL_TEXTURE_PRE_SPECULAR_HP},
    {"GL_LINEAR_CLIPMAP_LINEAR_SGIX", GL_LINEAR_CLIPMAP_LINEAR_SGIX},
    {"GL_TEXTURE_CLIPMAP_CENTER_SGIX", GL_TEXTURE_CLIPMAP_CENTER_SGIX},
    {"GL_TEXTURE_CLIPMAP_FRAME_SGIX", GL_TEXTURE_CLIPMAP_FRAME_SGIX},
    {"GL_TEXTURE_CLIPMAP_OFFSET_SGIX", GL_TEXTURE_CLIPMAP_OFFSET_SGIX},
    {"GL_TEXTURE_CLIPMAP_VIRTUAL_DEPTH_SGIX", GL_TEXTURE_CLIPMAP_VIRTUAL_DEPTH_SGIX},
    {"GL_TEXTURE_CLIPMAP_LOD_OFFSET_SGIX", GL_TEXTURE_CLIPMAP_LOD_OFFSET_SGIX},
    {"GL_TEXTURE_CLIPMAP_DEPTH_SGIX", GL_TEXTURE_CLIPMAP_DEPTH_SGIX},
    {"GL_MAX_CLIPMAP_DEPTH_SGIX", GL_MAX_CLIPMAP_DEPTH_SGIX},
    {"GL_MAX_CLIPMAP_VIRTUAL_DEPTH_SGIX", GL_MAX_CLIPMAP_VIRTUAL_DEPTH_SGIX},
    {"GL_POST_TEXTURE_FILTER_BIAS_SGIX", GL_POST_TEXTURE_FILTER_BIAS_SGIX},
    {"GL_POST_TEXTURE_FILTER_SCALE_SGIX", GL_POST_TEXTURE_FILTER_SCALE_SGIX},
    {"GL_POST_TEXTURE_FILTER_BIAS_RANGE_SGIX", GL_POST_TEXTURE_FILTER_BIAS_RANGE_SGIX},
    {"GL_POST_TEXTURE_FILTER_SCALE_RANGE_SGIX", GL_POST_TEXTURE_FILTER_SCALE_RANGE_SGIX},
    {"GL_REFERENCE_PLANE_SGIX", GL_REFERENCE_PLANE_SGIX},
    {"GL_REFERENCE_PLANE_EQUATION_SGIX", GL_REFERENCE_PLANE_EQUATION_SGIX},
    {"GL_IR_INSTRUMENT1_SGIX", GL_IR_INSTRUMENT1_SGIX},
    {"GL_INSTRUMENT_BUFFER_POINTER_SGIX", GL_INSTRUMENT_BUFFER_POINTER_SGIX},
    {"GL_INSTRUMENT_MEASUREMENTS_SGIX", GL_INSTRUMENT_MEASUREMENTS_SGIX},
    {"GL_LIST_PRIORITY_SGIX", GL_LIST_PRIORITY_SGIX},
    {"GL_CALLIGRAPHIC_FRAGMENT_SGIX", GL_CALLIGRAPHIC_FRAGMENT_SGIX},
    {"GL_PIXEL_TEX_GEN_Q_CEILING_SGIX", GL_PIXEL_TEX_GEN_Q_CEILING_SGIX},
    {"GL_PIXEL_TEX_GEN_Q_ROUND_SGIX", GL_PIXEL_TEX_GEN_Q_ROUND_SGIX},
    {"GL_PIXEL_TEX_GEN_Q_FLOOR_SGIX", GL_PIXEL_TEX_GEN_Q_FLOOR_SGIX},
    {"GL_PIXEL_TEX_GEN_ALPHA_REPLACE_SGIX", GL_PIXEL_TEX_GEN_ALPHA_REPLACE_SGIX},
    {"GL_PIXEL_TEX_GEN_ALPHA_NO_REPLACE_SGIX", GL_PIXEL_TEX_GEN_ALPHA_NO_REPLACE_SGIX},
    {"GL_PIXEL_TEX_GEN_ALPHA_LS_SGIX", GL_PIXEL_TEX_GEN_ALPHA_LS_SGIX},
    {"GL_PIXEL_TEX_GEN_ALPHA_MS_SGIX", GL_PIXEL_TEX_GEN_ALPHA_MS_SGIX},
    {"GL_FRAMEZOOM_SGIX", GL_FRAMEZOOM_SGIX},
    {"GL_FRAMEZOOM_FACTOR_SGIX", GL_FRAMEZOOM_FACTOR_SGIX},
    {"GL_MAX_FRAMEZOOM_FACTOR_SGIX", GL_MAX_FRAMEZOOM_FACTOR_SGIX},
    {"GL_TEXTURE_LOD_BIAS_S_SGIX", GL_TEXTURE_LOD_BIAS_S_SGIX},
    {"GL_TEXTURE_LOD_BIAS_T_SGIX", GL_TEXTURE_LOD_BIAS_T_SGIX},
    {"GL_TEXTURE_LOD_BIAS_R_SGIX", GL_TEXTURE_LOD_BIAS_R_SGIX},
    {"GL_GENERATE_MIPMAP", GL_GENERATE_MIPMAP},
    {"GL_GENERATE_MIPMAP_HINT", GL_GENERATE_MIPMAP_HINT},
    {"GL_GEOMETRY_DEFORMATION_SGIX", GL_GEOMETRY_DEFORMATION_SGIX},
    {"GL_TEXTURE_DEFORMATION_SGIX", GL_TEXTURE_DEFORMATION_SGIX},
    {"GL_DEFORMATIONS_MASK_SGIX", GL_DEFORMATIONS_MASK_SGIX},
    {"GL_MAX_DEFORMATION_ORDER_SGIX", GL_MAX_DEFORMATION_ORDER_SGIX},
    {"GL_FOG_OFFSET_SGIX", GL_FOG_OFFSET_SGIX},
    {"GL_FOG_OFFSET_VALUE_SGIX", GL_FOG_OFFSET_VALUE_SGIX},
    {"GL_TEXTURE_COMPARE_SGIX", GL_TEXTURE_COMPARE_SGIX},
    {"GL_TEXTURE_COMPARE_OPERATOR_SGIX", GL_TEXTURE_COMPARE_OPERATOR_SGIX},
    {"GL_TEXTURE_LEQUAL_R_SGIX", GL_TEXTURE_LEQUAL_R_SGIX},
    {"GL_TEXTURE_GEQUAL_R_SGIX", GL_TEXTURE_GEQUAL_R_SGIX},
    {"GL_DEPTH_COMPONENT16", GL_DEPTH_COMPONENT16},
    {"GL_DEPTH_COMPONENT24", GL_DEPTH_COMPONENT24},
    {"GL_DEPTH_COMPONENT32", GL_DEPTH_COMPONENT32},
    {"GL_ARRAY_ELEMENT_LOCK_FIRST_EXT", GL_ARRAY_ELEMENT_LOCK_FIRST_EXT},
    {"GL_ARRAY_ELEMENT_LOCK_COUNT_EXT", GL_ARRAY_ELEMENT_LOCK_COUNT_EXT},
    {"GL_CULL_VERTEX_EXT", GL_CULL_VERTEX_EXT},
    {"GL_CULL_VERTEX_EYE_POSITION_EXT", GL_CULL_VERTEX_EYE_POSITION_EXT},
    {"GL_CULL_VERTEX_OBJECT_POSITION_EXT", GL_CULL_VERTEX_OBJECT_POSITION_EXT},
    {"GL_IUI_V2F_EXT", GL_IUI_V2F_EXT},
    {"GL_IUI_V3F_EXT", GL_IUI_V3F_EXT},
    {"GL_IUI_N3F_V2F_EXT", GL_IUI_N3F_V2F_EXT},
    {"GL_IUI_N3F_V3F_EXT", GL_IUI_N3F_V3F_EXT},
    {"GL_T2F_IUI_V2F_EXT", GL_T2F_IUI_V2F_EXT},
    {"GL_T2F_IUI_V3F_EXT", GL_T2F_IUI_V3F_EXT},
    {"GL_T2F_IUI_N3F_V2F_EXT", GL_T2F_IUI_N3F_V2F_EXT},
    {"GL_T2F_IUI_N3F_V3F_EXT", GL_T2F_IUI_N3F_V3F_EXT},
    {"GL_INDEX_TEST_EXT", GL_INDEX_TEST_EXT},
    {"GL_INDEX_TEST_FUNC_EXT", GL_INDEX_TEST_FUNC_EXT},
    {"GL_INDEX_TEST_REF_EXT", GL_INDEX_TEST_REF_EXT},
    {"GL_INDEX_MATERIAL_EXT", GL_INDEX_MATERIAL_EXT},
    {"GL_INDEX_MATERIAL_PARAMETER_EXT", GL_INDEX_MATERIAL_PARAMETER_EXT},
    {"GL_INDEX_MATERIAL_FACE_EXT", GL_INDEX_MATERIAL_FACE_EXT},
    {"GL_YCRCB_422_SGIX", GL_YCRCB_422_SGIX},
    {"GL_YCRCB_444_SGIX", GL_YCRCB_444_SGIX},
    {"GL_WRAP_BORDER_SUN", GL_WRAP_BORDER_SUN},
    {"GL_UNPACK_CONSTANT_DATA_SUNX", GL_UNPACK_CONSTANT_DATA_SUNX},
    {"GL_TEXTURE_CONSTANT_DATA_SUNX", GL_TEXTURE_CONSTANT_DATA_SUNX},
    {"GL_TRIANGLE_LIST_SUN", GL_TRIANGLE_LIST_SUN},
    {"GL_REPLACEMENT_CODE_SUN", GL_REPLACEMENT_CODE_SUN},
    {"GL_GLOBAL_ALPHA_SUN", GL_GLOBAL_ALPHA_SUN},
    {"GL_GLOBAL_ALPHA_FACTOR_SUN", GL_GLOBAL_ALPHA_FACTOR_SUN},
    {"GL_TEXTURE_COLOR_WRITEMASK_SGIS", GL_TEXTURE_COLOR_WRITEMASK_SGIS},
    {"GL_EYE_DISTANCE_TO_POINT_SGIS", GL_EYE_DISTANCE_TO_POINT_SGIS},
    {"GL_OBJECT_DISTANCE_TO_POINT_SGIS", GL_OBJECT_DISTANCE_TO_POINT_SGIS},
    {"GL_EYE_DISTANCE_TO_LINE_SGIS", GL_EYE_DISTANCE_TO_LINE_SGIS},
    {"GL_OBJECT_DISTANCE_TO_LINE_SGIS", GL_OBJECT_DISTANCE_TO_LINE_SGIS},
    {"GL_EYE_POINT_SGIS", GL_EYE_POINT_SGIS},
    {"GL_OBJECT_POINT_SGIS", GL_OBJECT_POINT_SGIS},
    {"GL_EYE_LINE_SGIS", GL_EYE_LINE_SGIS},
    {"GL_OBJECT_LINE_SGIS", GL_OBJECT_LINE_SGIS},
    {"GL_LIGHT_MODEL_COLOR_CONTROL", GL_LIGHT_MODEL_COLOR_CONTROL},
    {"GL_SINGLE_COLOR", GL_SINGLE_COLOR},
    {"GL_SEPARATE_SPECULAR_COLOR", GL_SEPARATE_SPECULAR_COLOR},
    {"GL_SHARED_TEXTURE_PALETTE_EXT", GL_SHARED_TEXTURE_PALETTE_EXT},
    {"GL_FOG_SCALE_SGIX", GL_FOG_SCALE_SGIX},
    {"GL_FOG_SCALE_VALUE_SGIX", GL_FOG_SCALE_VALUE_SGIX},
    {"GL_TEXT_FRAGMENT_SHADER_ATI", GL_TEXT_FRAGMENT_SHADER_ATI},
    {"GL_FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING", GL_FRAMEBUFFER_ATTACHMENT_COLOR_ENCODING},
    {"GL_FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE", GL_FRAMEBUFFER_ATTACHMENT_COMPONENT_TYPE},
    {"GL_FRAMEBUFFER_ATTACHMENT_RED_SIZE", GL_FRAMEBUFFER_ATTACHMENT_RED_SIZE},
    {"GL_FRAMEBUFFER_ATTACHMENT_GREEN_SIZE", GL_FRAMEBUFFER_ATTACHMENT_GREEN_SIZE},
    {"GL_FRAMEBUFFER_ATTACHMENT_BLUE_SIZE", GL_FRAMEBUFFER_ATTACHMENT_BLUE_SIZE},
    {"GL_FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE", GL_FRAMEBUFFER_ATTACHMENT_ALPHA_SIZE},
    {"GL_FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE", GL_FRAMEBUFFER_ATTACHMENT_DEPTH_SIZE},
    {"GL_FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE", GL_FRAMEBUFFER_ATTACHMENT_STENCIL_SIZE},
    {"GL_FRAMEBUFFER_DEFAULT", GL_FRAMEBUFFER_DEFAULT},
    {"GL_FRAMEBUFFER_UNDEFINED", GL_FRAMEBUFFER_UNDEFINED},
    {"GL_DEPTH_STENCIL_ATTACHMENT", GL_DEPTH_STENCIL_ATTACHMENT},
    {"GL_MAJOR_VERSION", GL_MAJOR_VERSION},
    {"GL_MINOR_VERSION", GL_MINOR_VERSION},
    {"GL_NUM_EXTENSIONS", GL_NUM_EXTENSIONS},
    {"GL_CONTEXT_FLAGS", GL_CONTEXT_FLAGS},
    {"GL_BUFFER_IMMUTABLE_STORAGE", GL_BUFFER_IMMUTABLE_STORAGE},
    {"GL_BUFFER_STORAGE_FLAGS", GL_BUFFER_STORAGE_FLAGS},
    {"GL_PRIMITIVE_RESTART_FOR_PATCHES_SUPPORTED", GL_PRIMITIVE_RESTART_FOR_PATCHES_SUPPORTED},
    {"GL_INDEX", GL_INDEX},
    {"GL_COMPRESSED_RED", GL_COMPRESSED_RED},
    {"GL_COMPRESSED_RG", GL_COMPRESSED_RG},
    {"GL_RG", GL_RG},
    {"GL_RG_INTEGER", GL_RG_INTEGER},
    {"GL_R8", GL_R8},
    {"GL_R16", GL_R16},
    {"GL_RG8", GL_RG8},
    {"GL_RG16", GL_RG16},
    {"GL_R16F", GL_R16F},
    {"GL_R32F", GL_R32F},
    {"GL_RG16F", GL_RG16F},
    {"GL_RG32F", GL_RG32F},
    {"GL_R8I", GL_R8I},
    {"GL_R8UI", GL_R8UI},
    {"GL_R16I", GL_R16I},
    {"GL_R16UI", GL_R16UI},
    {"GL_R32I", GL_R32I},
    {"GL_R32UI", GL_R32UI},
    {"GL_RG8I", GL_RG8I},
    {"GL_RG8UI", GL_RG8UI},
    {"GL_RG16I", GL_RG16I},
    {"GL_RG16UI", GL_RG16UI},
    {"GL_RG32I", GL_RG32I},
    {"GL_RG32UI", GL_RG32UI},
    {"GL_SYNC_CL_EVENT_ARB", GL_SYNC_CL_EVENT_ARB},
    {"GL_SYNC_CL_EVENT_COMPLETE_ARB", GL_SYNC_CL_EVENT_COMPLETE_ARB},
    {"GL_DEBUG_OUTPUT_SYNCHRONOUS", GL_DEBUG_OUTPUT_SYNCHRONOUS},
    {"GL_DEBUG_NEXT_LOGGED_MESSAGE_LENGTH", GL_DEBUG_NEXT_LOGGED_MESSAGE_LENGTH},
    {"GL_DEBUG_CALLBACK_FUNCTION", GL_DEBUG_CALLBACK_FUNCTION},
    {"GL_DEBUG_CALLBACK_USER_PARAM", GL_DEBUG_CALLBACK_USER_PARAM},
    {"GL_DEBUG_SOURCE_API", GL_DEBUG_SOURCE_API},
    {"GL_DEBUG_SOURCE_WINDOW_SYSTEM", GL_DEBUG_SOURCE_WINDOW_SYSTEM},
    {"GL_DEBUG_SOURCE_SHADER_COMPILER", GL_DEBUG_SOURCE_SHADER_COMPILER},
    {"GL_DEBUG_SOURCE_THIRD_PARTY", GL_DEBUG_SOURCE_THIRD_PARTY},
    {"GL_DEBUG_SOURCE_APPLICATION", GL_DEBUG_SOURCE_APPLICATION},
    {"GL_DEBUG_SOURCE_OTHER", GL_DEBUG_SOURCE_OTHER},
    {"GL_DEBUG_TYPE_ERROR", GL_DEBUG_TYPE_ERROR},
    {"GL_DEBUG_TYPE_DEPRECATED_BEHAVIOR", GL_DEBUG_TYPE_DEPRECATED_BEHAVIOR},
    {"GL_DEBUG_TYPE_UNDEFINED_BEHAVIOR", GL_DEBUG_TYPE_UNDEFINED_BEHAVIOR},
    {"GL_DEBUG_TYPE_PORTABILITY", GL_DEBUG_TYPE_PORTABILITY},
    {"GL_DEBUG_TYPE_PERFORMANCE", GL_DEBUG_TYPE_PERFORMANCE},
    {"GL_DEBUG_TYPE_OTHER", GL_DEBUG_TYPE_OTHER},
    {"GL_LOSE_CONTEXT_ON_RESET", GL_LOSE_CONTEXT_ON_RESET},
    {"GL_GUILTY_CONTEXT_RESET", GL_GUILTY_CONTEXT_RESET},
    {"GL_INNOCENT_CONTEXT_RESET", GL_INNOCENT_CONTEXT_RESET},
    {"GL_UNKNOWN_CONTEXT_RESET", GL_UNKNOWN_CONTEXT_RESET},
    {"GL_RESET_NOTIFICATION_STRATEGY", GL_RESET_NOTIFICATION_STRATEGY},
    {"GL_PROGRAM_BINARY_RETRIEVABLE_HINT", GL_PROGRAM_BINARY_RETRIEVABLE_HINT},
    {"GL_PROGRAM_SEPARABLE", GL_PROGRAM_SEPARABLE},
    {"GL_ACTIVE_PROGRAM", GL_ACTIVE_PROGRAM},
    {"GL_PROGRAM_PIPELINE_BINDING", GL_PROGRAM_PIPELINE_BINDING},
    {"GL_MAX_VIEWPORTS", GL_MAX_VIEWPORTS},
    {"GL_VIEWPORT_SUBPIXEL_BITS", GL_VIEWPORT_SUBPIXEL_BITS},
    {"GL_VIEWPORT_BOUNDS_RANGE", GL_VIEWPORT_BOUNDS_RANGE},
    {"GL_LAYER_PROVOKING_VERTEX", GL_LAYER_PROVOKING_VERTEX},
    {"GL_VIEWPORT_INDEX_PROVOKING_VERTEX", GL_VIEWPORT_INDEX_PROVOKING_VERTEX},
    {"GL_UNDEFINED_VERTEX", GL_UNDEFINED_VERTEX},
    {"GL_NO_RESET_NOTIFICATION", GL_NO_RESET_NOTIFICATION},
    {"GL_MAX_COMPUTE_SHARED_MEMORY_SIZE", GL_MAX_COMPUTE_SHARED_MEMORY_SIZE},
    {"GL_MAX_COMPUTE_UNIFORM_COMPONENTS", GL_MAX_COMPUTE_UNIFORM_COMPONENTS},
    {"GL_MAX_COMPUTE_ATOMIC_COUNTER_BUFFERS", GL_MAX_COMPUTE_ATOMIC_COUNTER_BUFFERS},
    {"GL_MAX_COMPUTE_ATOMIC_COUNTERS", GL_MAX_COMPUTE_ATOMIC_COUNTERS},
    {"GL_MAX_COMBINED_COMPUTE_UNIFORM_COMPONENTS", GL_MAX_COMBINED_COMPUTE_UNIFORM_COMPONENTS},
    {"GL_COMPUTE_WORK_GROUP_SIZE", GL_COMPUTE_WORK_GROUP_SIZE},
    {"GL_DEBUG_TYPE_MARKER", GL_DEBUG_TYPE_MARKER},
    {"GL_DEBUG_TYPE_PUSH_GROUP", GL_DEBUG_TYPE_PUSH_GROUP},
    {"GL_DEBUG_TYPE_POP_GROUP", GL_DEBUG_TYPE_POP_GROUP},
    {"GL_DEBUG_SEVERITY_NOTIFICATION", GL_DEBUG_SEVERITY_NOTIFICATION},
    {"GL_MAX_DEBUG_GROUP_STACK_DEPTH", GL_MAX_DEBUG_GROUP_STACK_DEPTH},
    {"GL_DEBUG_GROUP_STACK_DEPTH", GL_DEBUG_GROUP_STACK_DEPTH},
    {"GL_MAX_UNIFORM_LOCATIONS", GL_MAX_UNIFORM_LOCATIONS},
    {"GL_INTERNALFORMAT_SUPPORTED", GL_INTERNALFORMAT_SUPPORTED},
    {"GL_INTERNALFORMAT_PREFERRED", GL_INTERNALFORMAT_PREFERRED},
    {"GL_INTERNALFORMAT_RED_SIZE", GL_INTERNALFORMAT_RED_SIZE},
    {"GL_INTERNALFORMAT_GREEN_SIZE", GL_INTERNALFORMAT_GREEN_SIZE},
    {"GL_INTERNALFORMAT_BLUE_SIZE", GL_INTERNALFORMAT_BLUE_SIZE},
    {"GL_INTERNALFORMAT_ALPHA_SIZE", GL_INTERNALFORMAT_ALPHA_SIZE},
    {"GL_INTERNALFORMAT_DEPTH_SIZE", GL_INTERNALFORMAT_DEPTH_SIZE},
    {"GL_INTERNALFORMAT_STENCIL_SIZE", GL_INTERNALFORMAT_STENCIL_SIZE},
    {"GL_INTERNALFORMAT_SHARED_SIZE", GL_INTERNALFORMAT_SHARED_SIZE},
    {"GL_INTERNALFORMAT_RED_TYPE", GL_INTERNALFORMAT_RED_TYPE},
    {"GL_INTERNALFORMAT_GREEN_TYPE", GL_INTERNALFORMAT_GREEN_TYPE},
    {"GL_INTERNALFORMAT_BLUE_TYPE", GL_INTERNALFORMAT_BLUE_TYPE},
    {"GL_INTERNALFORMAT_ALPHA_TYPE", GL_INTERNALFORMAT_ALPHA_TYPE},
    {"GL_INTERNALFORMAT_DEPTH_TYPE", GL_INTERNALFORMAT_DEPTH_TYPE},
    {"GL_INTERNALFORMAT_STENCIL_TYPE", GL_INTERNALFORMAT_STENCIL_TYPE},
    {"GL_MAX_WIDTH", GL_MAX_WIDTH},
    {"GL_MAX_HEIGHT", GL_MAX_HEIGHT},
    {"GL_MAX_DEPTH", GL_MAX_DEPTH},
    {"GL_MAX_LAYERS", GL_MAX_LAYERS},
    {"GL_MAX_COMBINED_DIMENSIONS", GL_MAX_COMBINED_DIMENSIONS},
    {"GL_COLOR_COMPONENTS", GL_COLOR_COMPONENTS},
    {"GL_DEPTH_COMPONENTS", GL_DEPTH_COMPONENTS},
    {"GL_STENCIL_COMPONENTS", GL_STENCIL_COMPONENTS},
    {"GL_COLOR_RENDERABLE", GL_COLOR_RENDERABLE},
    {"GL_DEPTH_RENDERABLE", GL_DEPTH_RENDERABLE},
    {"GL_STENCIL_RENDERABLE", GL_STENCIL_RENDERABLE},
    {"GL_FRAMEBUFFER_RENDERABLE", GL_FRAMEBUFFER_RENDERABLE},
    {"GL_FRAMEBUFFER_RENDERABLE_LAYERED", GL_FRAMEBUFFER_RENDERABLE_LAYERED},
    {"GL_FRAMEBUFFER_BLEND", GL_FRAMEBUFFER_BLEND},
    {"GL_READ_PIXELS", GL_READ_PIXELS},
    {"GL_READ_PIXELS_FORMAT", GL_READ_PIXELS_FORMAT},
    {"GL_READ_PIXELS_TYPE", GL_READ_PIXELS_TYPE},
    {"GL_TEXTURE_IMAGE_FORMAT", GL_TEXTURE_IMAGE_FORMAT},
    {"GL_TEXTURE_IMAGE_TYPE", GL_TEXTURE_IMAGE_TYPE},
    {"GL_GET_TEXTURE_IMAGE_FORMAT", GL_GET_TEXTURE_IMAGE_FORMAT},
    {"GL_GET_TEXTURE_IMAGE_TYPE", GL_GET_TEXTURE_IMAGE_TYPE},
    {"GL_MIPMAP", GL_MIPMAP},
    {"GL_MANUAL_GENERATE_MIPMAP", GL_MANUAL_GENERATE_MIPMAP},
    {"GL_AUTO_GENERATE_MIPMAP", GL_AUTO_GENERATE_MIPMAP},
    {"GL_COLOR_ENCODING", GL_COLOR_ENCODING},
    {"GL_SRGB_READ", GL_SRGB_READ},
    {"GL_SRGB_WRITE", GL_SRGB_WRITE},
    {"GL_SRGB_DECODE_ARB", GL_SRGB_DECODE_ARB},
    {"GL_FILTER", GL_FILTER},
    {"GL_VERTEX_TEXTURE", GL_VERTEX_TEXTURE},
    {"GL_TESS_CONTROL_TEXTURE", GL_TESS_CONTROL_TEXTURE},
    {"GL_TESS_EVALUATION_TEXTURE", GL_TESS_EVALUATION_TEXTURE},
    {"GL_GEOMETRY_TEXTURE", GL_GEOMETRY_TEXTURE},
    {"GL_FRAGMENT_TEXTURE", GL_FRAGMENT_TEXTURE},
    {"GL_COMPUTE_TEXTURE", GL_COMPUTE_TEXTURE},
    {"GL_TEXTURE_SHADOW", GL_TEXTURE_SHADOW},
    {"GL_TEXTURE_GATHER", GL_TEXTURE_GATHER},
    {"GL_TEXTURE_GATHER_SHADOW", GL_TEXTURE_GATHER_SHADOW},
    {"GL_SHADER_IMAGE_LOAD", GL_SHADER_IMAGE_LOAD},
    {"GL_SHADER_IMAGE_STORE", GL_SHADER_IMAGE_STORE},
    {"GL_SHADER_IMAGE_ATOMIC", GL_SHADER_IMAGE_ATOMIC},
    {"GL_IMAGE_TEXEL_SIZE", GL_IMAGE_TEXEL_SIZE},
    {"GL_IMAGE_COMPATIBILITY_CLASS", GL_IMAGE_COMPATIBILITY_CLASS},
    {"GL_IMAGE_PIXEL_FORMAT", GL_IMAGE_PIXEL_FORMAT},
    {"GL_IMAGE_PIXEL_TYPE", GL_IMAGE_PIXEL_TYPE},
    {"GL_SIMULTANEOUS_TEXTURE_AND_DEPTH_TEST", GL_SIMULTANEOUS_TEXTURE_AND_DEPTH_TEST},
    {"GL_SIMULTANEOUS_TEXTURE_AND_STENCIL_TEST", GL_SIMULTANEOUS_TEXTURE_AND_STENCIL_TEST},
    {"GL_SIMULTANEOUS_TEXTURE_AND_DEPTH_WRITE", GL_SIMULTANEOUS_TEXTURE_AND_DEPTH_WRITE},
    {"GL_SIMULTANEOUS_TEXTURE_AND_STENCIL_WRITE", GL_SIMULTANEOUS_TEXTURE_AND_STENCIL_WRITE},
    {"GL_TEXTURE_COMPRESSED_BLOCK_WIDTH", GL_TEXTURE_COMPRESSED_BLOCK_WIDTH},
    {"GL_TEXTURE_COMPRESSED_BLOCK_HEIGHT", GL_TEXTURE_COMPRESSED_BLOCK_HEIGHT},
    {"GL_TEXTURE_COMPRESSED_BLOCK_SIZE", GL_TEXTURE_COMPRESSED_BLOCK_SIZE},
    {"GL_CLEAR_BUFFER", GL_CLEAR_BUFFER},
    {"GL_TEXTURE_VIEW", GL_TEXTURE_VIEW},
    {"GL_VIEW_COMPATIBILITY_CLASS", GL_VIEW_COMPATIBILITY_CLASS},
    {"GL_FULL_SUPPORT", GL_FULL_SUPPORT},
    {"GL_CAVEAT_SUPPORT", GL_CAVEAT_SUPPORT},
    {"GL_IMAGE_CLASS_4_X_32", GL_IMAGE_CLASS_4_X_32},
    {"GL_IMAGE_CLASS_2_X_32", GL_IMAGE_CLASS_2_X_32},
    {"GL_IMAGE_CLASS_1_X_32", GL_IMAGE_CLASS_1_X_32},
    {"GL_IMAGE_CLASS_4_X_16", GL_IMAGE_CLASS_4_X_16},
    {"GL_IMAGE_CLASS_2_X_16", GL_IMAGE_CLASS_2_X_16},
    {"GL_IMAGE_CLASS_1_X_16", GL_IMAGE_CLASS_1_X_16},
    {"GL_IMAGE_CLASS_4_X_8", GL_IMAGE_CLASS_4_X_8},
    {"GL_IMAGE_CLASS_2_X_8", GL_IMAGE_CLASS_2_X_8},
    {"GL_IMAGE_CLASS_1_X_8", GL_IMAGE_CLASS_1_X_8},
    {"GL_IMAGE_CLASS_11_11_10", GL_IMAGE_CLASS_11_11_10},
    {"GL_IMAGE_CLASS_10_10_10_2", GL_IMAGE_CLASS_10_10_10_2},
    {"GL_VIEW_CLASS_128_BITS", GL_VIEW_CLASS_128_BITS},
    {"GL_VIEW_CLASS_96_BITS", GL_VIEW_CLASS_96_BITS},
    {"GL_VIEW_CLASS_64_BITS", GL_VIEW_CLASS_64_BITS},
    {"GL_VIEW_CLASS_48_BITS", GL_VIEW_CLASS_48_BITS},
    {"GL_VIEW_CLASS_32_BITS", GL_VIEW_CLASS_32_BITS},
    {"GL_VIEW_CLASS_24_BITS", GL_VIEW_CLASS_24_BITS},
    {"GL_VIEW_CLASS_16_BITS", GL_VIEW_CLASS_16_BITS},
    {"GL_VIEW_CLASS_8_BITS", GL_VIEW_CLASS_8_BITS},
    {"GL_VIEW_CLASS_S3TC_DXT1_RGB", GL_VIEW_CLASS_S3TC_DXT1_RGB},
    {"GL_VIEW_CLASS_S3TC_DXT1_RGBA", GL_VIEW_CLASS_S3TC_DXT1_RGBA},
    {"GL_VIEW_CLASS_S3TC_DXT3_RGBA", GL_VIEW_CLASS_S3TC_DXT3_RGBA},
    {"GL_VIEW_CLASS_S3TC_DXT5_RGBA", GL_VIEW_CLASS_S3TC_DXT5_RGBA},
    {"GL_VIEW_CLASS_RGTC1_RED", GL_VIEW_CLASS_RGTC1_RED},
    {"GL_VIEW_CLASS_RGTC2_RG", GL_VIEW_CLASS_RGTC2_RG},
    {"GL_VIEW_CLASS_BPTC_UNORM", GL_VIEW_CLASS_BPTC_UNORM},
    {"GL_VIEW_CLASS_BPTC_FLOAT", GL_VIEW_CLASS_BPTC_FLOAT},
    {"GL_VERTEX_ATTRIB_BINDING", GL_VERTEX_ATTRIB_BINDING},
    {"GL_VERTEX_ATTRIB_RELATIVE_OFFSET", GL_VERTEX_ATTRIB_RELATIVE_OFFSET},
    {"GL_VERTEX_BINDING_DIVISOR", GL_VERTEX_BINDING_DIVISOR},
    {"GL_VERTEX_BINDING_OFFSET", GL_VERTEX_BINDING_OFFSET},
    {"GL_VERTEX_BINDING_STRIDE", GL_VERTEX_BINDING_STRIDE},
    {"GL_MAX_VERTEX_ATTRIB_RELATIVE_OFFSET", GL_MAX_VERTEX_ATTRIB_RELATIVE_OFFSET},
    {"GL_MAX_VERTEX_ATTRIB_BINDINGS", GL_MAX_VERTEX_ATTRIB_BINDINGS},
    {"GL_TEXTURE_VIEW_MIN_LEVEL", GL_TEXTURE_VIEW_MIN_LEVEL},
    {"GL_TEXTURE_VIEW_NUM_LEVELS", GL_TEXTURE_VIEW_NUM_LEVELS},
    {"GL_TEXTURE_VIEW_MIN_LAYER", GL_TEXTURE_VIEW_MIN_LAYER},
    {"GL_TEXTURE_VIEW_NUM_LAYERS", GL_TEXTURE_VIEW_NUM_LAYERS},
    {"GL_TEXTURE_IMMUTABLE_LEVELS", GL_TEXTURE_IMMUTABLE_LEVELS},
    {"GL_BUFFER", GL_BUFFER},
    {"GL_SHADER", GL_SHADER},
    {"GL_PROGRAM", GL_PROGRAM},
    {"GL_QUERY", GL_QUERY},
    {"GL_PROGRAM_PIPELINE", GL_PROGRAM_PIPELINE},
    {"GL_MAX_VERTEX_ATTRIB_STRIDE", GL_MAX_VERTEX_ATTRIB_STRIDE},
    {"GL_SAMPLER", GL_SAMPLER},
    {"GL_DISPLAY_LIST", GL_DISPLAY_LIST},
    {"GL_MAX_LABEL_LENGTH", GL_MAX_LABEL_LENGTH},
    {"GL_NUM_SHADING_LANGUAGE_VERSIONS", GL_NUM_SHADING_LANGUAGE_VERSIONS},
    {"GL_QUERY_TARGET", GL_QUERY_TARGET},
    {"GL_TRANSFORM_FEEDBACK_OVERFLOW_ARB", GL_TRANSFORM_FEEDBACK_OVERFLOW_ARB},
    {"GL_TRANSFORM_FEEDBACK_STREAM_OVERFLOW_ARB", GL_TRANSFORM_FEEDBACK_STREAM_OVERFLOW_ARB},
    {"GL_VERTICES_SUBMITTED_ARB", GL_VERTICES_SUBMITTED_ARB},
    {"GL_PRIMITIVES_SUBMITTED_ARB", GL_PRIMITIVES_SUBMITTED_ARB},
    {"GL_VERTEX_SHADER_INVOCATIONS_ARB", GL_VERTEX_SHADER_INVOCATIONS_ARB},
    {"GL_TESS_CONTROL_SHADER_PATCHES_ARB", GL_TESS_CONTROL_SHADER_PATCHES_ARB},
    {"GL_TESS_EVALUATION_SHADER_INVOCATIONS_ARB", GL_TESS_EVALUATION_SHADER_INVOCATIONS_ARB},
    {"GL_GEOMETRY_SHADER_PRIMITIVES_EMITTED_ARB", GL_GEOMETRY_SHADER_PRIMITIVES_EMITTED_ARB},
    {"GL_FRAGMENT_SHADER_INVOCATIONS_ARB", GL_FRAGMENT_SHADER_INVOCATIONS_ARB},
    {"GL_COMPUTE_SHADER_INVOCATIONS_ARB", GL_COMPUTE_SHADER_INVOCATIONS_ARB},
    {"GL_CLIPPING_INPUT_PRIMITIVES_ARB", GL_CLIPPING_INPUT_PRIMITIVES_ARB},
    {"GL_CLIPPING_OUTPUT_PRIMITIVES_ARB", GL_CLIPPING_OUTPUT_PRIMITIVES_ARB},
    {"GL_SPARSE_BUFFER_PAGE_SIZE_ARB", GL_SPARSE_BUFFER_PAGE_SIZE_ARB},
    {"GL_MAX_CULL_DISTANCES", GL_MAX_CULL_DISTANCES},
    {"GL_MAX_COMBINED_CLIP_AND_CULL_DISTANCES", GL_MAX_COMBINED_CLIP_AND_CULL_DISTANCES},
    {"GL_CONTEXT_RELEASE_BEHAVIOR", GL_CONTEXT_RELEASE_BEHAVIOR},
    {"GL_CONTEXT_RELEASE_BEHAVIOR_FLUSH", GL_CONTEXT_RELEASE_BEHAVIOR_FLUSH},
    {"GL_DEPTH_PASS_INSTRUMENT_SGIX", GL_DEPTH_PASS_INSTRUMENT_SGIX},
    {"GL_DEPTH_PASS_INSTRUMENT_COUNTERS_SGIX", GL_DEPTH_PASS_INSTRUMENT_COUNTERS_SGIX},
    {"GL_DEPTH_PASS_INSTRUMENT_MAX_SGIX", GL_DEPTH_PASS_INSTRUMENT_MAX_SGIX},
    {"GL_CONVOLUTION_HINT_SGIX", GL_CONVOLUTION_HINT_SGIX},
    {"GL_YCRCB_SGIX", GL_YCRCB_SGIX},
    {"GL_YCRCBA_SGIX", GL_YCRCBA_SGIX},
    {"GL_ALPHA_MIN_SGIX", GL_ALPHA_MIN_SGIX},
    {"GL_ALPHA_MAX_SGIX", GL_ALPHA_MAX_SGIX},
    {"GL_SCALEBIAS_HINT_SGIX", GL_SCALEBIAS_HINT_SGIX},
    {"GL_ASYNC_MARKER_SGIX", GL_ASYNC_MARKER_SGIX},
    {"GL_PIXEL_TEX_GEN_MODE_SGIX", GL_PIXEL_TEX_GEN_MODE_SGIX},
    {"GL_ASYNC_HISTOGRAM_SGIX", GL_ASYNC_HISTOGRAM_SGIX},
    {"GL_MAX_ASYNC_HISTOGRAM_SGIX", GL_MAX_ASYNC_HISTOGRAM_SGIX},
    {"GL_PIXEL_TRANSFORM_2D_EXT", GL_PIXEL_TRANSFORM_2D_EXT},
    {"GL_PIXEL_MAG_FILTER_EXT", GL_PIXEL_MAG_FILTER_EXT},
    {"GL_PIXEL_MIN_FILTER_EXT", GL_PIXEL_MIN_FILTER_EXT},
    {"GL_PIXEL_CUBIC_WEIGHT_EXT", GL_PIXEL_CUBIC_WEIGHT_EXT},
    {"GL_CUBIC_EXT", GL_CUBIC_EXT},
    {"GL_AVERAGE_EXT", GL_AVERAGE_EXT},
    {"GL_PIXEL_TRANSFORM_2D_STACK_DEPTH_EXT", GL_PIXEL_TRANSFORM_2D_STACK_DEPTH_EXT},
    {"GL_MAX_PIXEL_TRANSFORM_2D_STACK_DEPTH_EXT", GL_MAX_PIXEL_TRANSFORM_2D_STACK_DEPTH_EXT},
    {"GL_PIXEL_TRANSFORM_2D_MATRIX_EXT", GL_PIXEL_TRANSFORM_2D_MATRIX_EXT},
    {"GL_FRAGMENT_MATERIAL_EXT", GL_FRAGMENT_MATERIAL_EXT},
    {"GL_FRAGMENT_NORMAL_EXT", GL_FRAGMENT_NORMAL_EXT},
    {"GL_FRAGMENT_COLOR_EXT", GL_FRAGMENT_COLOR_EXT},
    {"GL_ATTENUATION_EXT", GL_ATTENUATION_EXT},
    {"GL_SHADOW_ATTENUATION_EXT", GL_SHADOW_ATTENUATION_EXT},
    {"GL_TEXTURE_APPLICATION_MODE_EXT", GL_TEXTURE_APPLICATION_MODE_EXT},
    {"GL_TEXTURE_LIGHT_EXT", GL_TEXTURE_LIGHT_EXT},
    {"GL_TEXTURE_MATERIAL_FACE_EXT", GL_TEXTURE_MATERIAL_FACE_EXT},
    {"GL_TEXTURE_MATERIAL_PARAMETER_EXT", GL_TEXTURE_MATERIAL_PARAMETER_EXT},
    {"GL_PIXEL_TEXTURE_SGIS", GL_PIXEL_TEXTURE_SGIS},
    {"GL_PIXEL_FRAGMENT_RGB_SOURCE_SGIS", GL_PIXEL_FRAGMENT_RGB_SOURCE_SGIS},
    {"GL_PIXEL_FRAGMENT_ALPHA_SOURCE_SGIS", GL_PIXEL_FRAGMENT_ALPHA_SOURCE_SGIS},
    {"GL_PIXEL_GROUP_COLOR_SGIS", GL_PIXEL_GROUP_COLOR_SGIS},
    {"GL_ASYNC_TEX_IMAGE_SGIX", GL_ASYNC_TEX_IMAGE_SGIX},
    {"GL_ASYNC_DRAW_PIXELS_SGIX", GL_ASYNC_DRAW_PIXELS_SGIX},
    {"GL_ASYNC_READ_PIXELS_SGIX", GL_ASYNC_READ_PIXELS_SGIX},
    {"GL_MAX_ASYNC_TEX_IMAGE_SGIX", GL_MAX_ASYNC_TEX_IMAGE_SGIX},
    {"GL_MAX_ASYNC_DRAW_PIXELS_SGIX", GL_MAX_ASYNC_DRAW_PIXELS_SGIX},
    {"GL_MAX_ASYNC_READ_PIXELS_SGIX", GL_MAX_ASYNC_READ_PIXELS_SGIX},
    {"GL_UNSIGNED_BYTE_2_3_3_REV", GL_UNSIGNED_BYTE_2_3_3_REV},
    {"GL_UNSIGNED_SHORT_5_6_5", GL_UNSIGNED_SHORT_5_6_5},
    {"GL_UNSIGNED_SHORT_5_6_5_REV", GL_UNSIGNED_SHORT_5_6_5_REV},
    {"GL_UNSIGNED_SHORT_4_4_4_4_REV", GL_UNSIGNED_SHORT_4_4_4_4_REV},
    {"GL_UNSIGNED_SHORT_1_5_5_5_REV", GL_UNSIGNED_SHORT_1_5_5_5_REV},
    {"GL_UNSIGNED_INT_8_8_8_8_REV", GL_UNSIGNED_INT_8_8_8_8_REV},
    {"GL_UNSIGNED_INT_2_10_10_10_REV", GL_UNSIGNED_INT_2_10_10_10_REV},
    {"GL_TEXTURE_MAX_CLAMP_S_SGIX", GL_TEXTURE_MAX_CLAMP_S_SGIX},
    {"GL_TEXTURE_MAX_CLAMP_T_SGIX", GL_TEXTURE_MAX_CLAMP_T_SGIX},
    {"GL_TEXTURE_MAX_CLAMP_R_SGIX", GL_TEXTURE_MAX_CLAMP_R_SGIX},
    {"GL_MIRRORED_REPEAT", GL_MIRRORED_REPEAT},
    {"GL_RGB_S3TC", GL_RGB_S3TC},
    {"GL_RGB4_S3TC", GL_RGB4_S3TC},
    {"GL_RGBA_S3TC", GL_RGBA_S3TC},
    {"GL_RGBA4_S3TC", GL_RGBA4_S3TC},
    {"GL_RGBA_DXT5_S3TC", GL_RGBA_DXT5_S3TC},
    {"GL_RGBA4_DXT5_S3TC", GL_RGBA4_DXT5_S3TC},
    {"GL_VERTEX_PRECLIP_SGIX", GL_VERTEX_PRECLIP_SGIX},
    {"GL_VERTEX_PRECLIP_HINT_SGIX", GL_VERTEX_PRECLIP_HINT_SGIX},
    {"GL_COMPRESSED_RGB_S3TC_DXT1_EXT", GL_COMPRESSED_RGB_S3TC_DXT1_EXT},
    {"GL_COMPRESSED_RGBA_S3TC_DXT1_EXT", GL_COMPRESSED_RGBA_S3TC_DXT1_EXT},
    {"GL_COMPRESSED_RGBA_S3TC_DXT3_EXT", GL_COMPRESSED_RGBA_S3TC_DXT3_EXT},
    {"GL_COMPRESSED_RGBA_S3TC_DXT5_EXT", GL_COMPRESSED_RGBA_S3TC_DXT5_EXT},
    {"GL_PARALLEL_ARRAYS_INTEL", GL_PARALLEL_ARRAYS_INTEL},
    {"GL_VERTEX_ARRAY_PARALLEL_POINTERS_INTEL", GL_VERTEX_ARRAY_PARALLEL_POINTERS_INTEL},
    {"GL_NORMAL_ARRAY_PARALLEL_POINTERS_INTEL", GL_NORMAL_ARRAY_PARALLEL_POINTERS_INTEL},
    {"GL_COLOR_ARRAY_PARALLEL_POINTERS_INTEL", GL_COLOR_ARRAY_PARALLEL_POINTERS_INTEL},
    {"GL_TEXTURE_COORD_ARRAY_PARALLEL_POINTERS_INTEL", GL_TEXTURE_COORD_ARRAY_PARALLEL_POINTERS_INTEL},
    {"GL_PERFQUERY_DONOT_FLUSH_INTEL", GL_PERFQUERY_DONOT_FLUSH_INTEL},
    {"GL_PERFQUERY_FLUSH_INTEL", GL_PERFQUERY_FLUSH_INTEL},
    {"GL_PERFQUERY_WAIT_INTEL", GL_PERFQUERY_WAIT_INTEL},
    {"GL_TEXTURE_MEMORY_LAYOUT_INTEL", GL_TEXTURE_MEMORY_LAYOUT_INTEL},
    {"GL_FRAGMENT_LIGHTING_SGIX", GL_FRAGMENT_LIGHTING_SGIX},
    {"GL_FRAGMENT_COLOR_MATERIAL_SGIX", GL_FRAGMENT_COLOR_MATERIAL_SGIX},
    {"GL_FRAGMENT_COLOR_MATERIAL_FACE_SGIX", GL_FRAGMENT_COLOR_MATERIAL_FACE_SGIX},
    {"GL_FRAGMENT_COLOR_MATERIAL_PARAMETER_SGIX", GL_FRAGMENT_COLOR_MATERIAL_PARAMETER_SGIX},
    {"GL_MAX_FRAGMENT_LIGHTS_SGIX", GL_MAX_FRAGMENT_LIGHTS_SGIX},
    {"GL_MAX_ACTIVE_LIGHTS_SGIX", GL_MAX_ACTIVE_LIGHTS_SGIX},
    {"GL_CURRENT_RASTER_NORMAL_SGIX", GL_CURRENT_RASTER_NORMAL_SGIX},
    {"GL_LIGHT_ENV_MODE_SGIX", GL_LIGHT_ENV_MODE_SGIX},
    {"GL_FRAGMENT_LIGHT_MODEL_LOCAL_VIEWER_SGIX", GL_FRAGMENT_LIGHT_MODEL_LOCAL_VIEWER_SGIX},
    {"GL_FRAGMENT_LIGHT_MODEL_TWO_SIDE_SGIX", GL_FRAGMENT_LIGHT_MODEL_TWO_SIDE_SGIX},
    {"GL_FRAGMENT_LIGHT_MODEL_AMBIENT_SGIX", GL_FRAGMENT_LIGHT_MODEL_AMBIENT_SGIX},
    {"GL_FRAGMENT_LIGHT_MODEL_NORMAL_INTERPOLATION_SGIX", GL_FRAGMENT_LIGHT_MODEL_NORMAL_INTERPOLATION_SGIX},
    {"GL_FRAGMENT_LIGHT0_SGIX", GL_FRAGMENT_LIGHT0_SGIX},
    {"GL_FRAGMENT_LIGHT1_SGIX", GL_FRAGMENT_LIGHT1_SGIX},
    {"GL_FRAGMENT_LIGHT2_SGIX", GL_FRAGMENT_LIGHT2_SGIX},
    {"GL_FRAGMENT_LIGHT3_SGIX", GL_FRAGMENT_LIGHT3_SGIX},
    {"GL_FRAGMENT_LIGHT4_SGIX", GL_FRAGMENT_LIGHT4_SGIX},
    {"GL_FRAGMENT_LIGHT5_SGIX", GL_FRAGMENT_LIGHT5_SGIX},
    {"GL_FRAGMENT_LIGHT6_SGIX", GL_FRAGMENT_LIGHT6_SGIX},
    {"GL_FRAGMENT_LIGHT7_SGIX", GL_FRAGMENT_LIGHT7_SGIX},
    {"GL_PACK_RESAMPLE_SGIX", GL_PACK_RESAMPLE_SGIX},
    {"GL_UNPACK_RESAMPLE_SGIX", GL_UNPACK_RESAMPLE_SGIX},
    {"GL_RESAMPLE_DECIMATE_SGIX", GL_RESAMPLE_DECIMATE_SGIX},
    {"GL_RESAMPLE_REPLICATE_SGIX", GL_RESAMPLE_REPLICATE_SGIX},
    {"GL_RESAMPLE_ZERO_FILL_SGIX", GL_RESAMPLE_ZERO_FILL_SGIX},
    {"GL_TANGENT_ARRAY_EXT", GL_TANGENT_ARRAY_EXT},
    {"GL_BINORMAL_ARRAY_EXT", GL_BINORMAL_ARRAY_EXT},
    {"GL_CURRENT_TANGENT_EXT", GL_CURRENT_TANGENT_EXT},
    {"GL_CURRENT_BINORMAL_EXT", GL_CURRENT_BINORMAL_EXT},
    {"GL_TANGENT_ARRAY_TYPE_EXT", GL_TANGENT_ARRAY_TYPE_EXT},
    {"GL_TANGENT_ARRAY_STRIDE_EXT", GL_TANGENT_ARRAY_STRIDE_EXT},
    {"GL_BINORMAL_ARRAY_TYPE_EXT", GL_BINORMAL_ARRAY_TYPE_EXT},
    {"GL_BINORMAL_ARRAY_STRIDE_EXT", GL_BINORMAL_ARRAY_STRIDE_EXT},
    {"GL_TANGENT_ARRAY_POINTER_EXT", GL_TANGENT_ARRAY_POINTER_EXT},
    {"GL_BINORMAL_ARRAY_POINTER_EXT", GL_BINORMAL_ARRAY_POINTER_EXT},
    {"GL_MAP1_TANGENT_EXT", GL_MAP1_TANGENT_EXT},
    {"GL_MAP2_TANGENT_EXT", GL_MAP2_TANGENT_EXT},
    {"GL_MAP1_BINORMAL_EXT", GL_MAP1_BINORMAL_EXT},
    {"GL_MAP2_BINORMAL_EXT", GL_MAP2_BINORMAL_EXT},
    {"GL_NEAREST_CLIPMAP_NEAREST_SGIX", GL_NEAREST_CLIPMAP_NEAREST_SGIX},
    {"GL_NEAREST_CLIPMAP_LINEAR_SGIX", GL_NEAREST_CLIPMAP_LINEAR_SGIX},
    {"GL_LINEAR_CLIPMAP_NEAREST_SGIX", GL_LINEAR_CLIPMAP_NEAREST_SGIX},
    {"GL_FOG_COORD_SRC", GL_FOG_COORD_SRC},
    {"GL_FOG_COORD", GL_FOG_COORD},
    {"GL_FRAGMENT_DEPTH", GL_FRAGMENT_DEPTH},
    {"GL_CURRENT_FOG_COORD", GL_CURRENT_FOG_COORD},
    {"GL_FOG_COORD_ARRAY_TYPE", GL_FOG_COORD_ARRAY_TYPE},
    {"GL_FOG_COORD_ARRAY_STRIDE", GL_FOG_COORD_ARRAY_STRIDE},
    {"GL_FOG_COORD_ARRAY_POINTER", GL_FOG_COORD_ARRAY_POINTER},
    {"GL_FOG_COORD_ARRAY", GL_FOG_COORD_ARRAY},
    {"GL_COLOR_SUM", GL_COLOR_SUM},
    {"GL_CURRENT_SECONDARY_COLOR", GL_CURRENT_SECONDARY_COLOR},
    {"GL_SECONDARY_COLOR_ARRAY_SIZE", GL_SECONDARY_COLOR_ARRAY_SIZE},
    {"GL_SECONDARY_COLOR_ARRAY_TYPE", GL_SECONDARY_COLOR_ARRAY_TYPE},
    {"GL_SECONDARY_COLOR_ARRAY_STRIDE", GL_SECONDARY_COLOR_ARRAY_STRIDE},
    {"GL_SECONDARY_COLOR_ARRAY_POINTER", GL_SECONDARY_COLOR_ARRAY_POINTER},
    {"GL_SECONDARY_COLOR_ARRAY", GL_SECONDARY_COLOR_ARRAY},
    {"GL_CURRENT_RASTER_SECONDARY_COLOR", GL_CURRENT_RASTER_SECONDARY_COLOR},
    {"GL_ALIASED_POINT_SIZE_RANGE", GL_ALIASED_POINT_SIZE_RANGE},
    {"GL_ALIASED_LINE_WIDTH_RANGE", GL_ALIASED_LINE_WIDTH_RANGE},
    {"GL_SCREEN_COORDINATES_REND", GL_SCREEN_COORDINATES_REND},
    {"GL_INVERTED_SCREEN_W_REND", GL_INVERTED_SCREEN_W_REND},
    {"GL_TEXTURE0", GL_TEXTURE0},
    {"GL_TEXTURE1", GL_TEXTURE1},
    {"GL_TEXTURE2", GL_TEXTURE2},
    {"GL_TEXTURE3", GL_TEXTURE3},
    {"GL_TEXTURE4", GL_TEXTURE4},
    {"GL_TEXTURE5", GL_TEXTURE5},
    {"GL_TEXTURE6", GL_TEXTURE6},
    {"GL_TEXTURE7", GL_TEXTURE7},
    {"GL_TEXTURE8", GL_TEXTURE8},
    {"GL_TEXTURE9", GL_TEXTURE9},
    {"GL_TEXTURE10", GL_TEXTURE10},
    {"GL_TEXTURE11", GL_TEXTURE11},
    {"GL_TEXTURE12", GL_TEXTURE12},
    {"GL_TEXTURE13", GL_TEXTURE13},
    {"GL_TEXTURE14", GL_TEXTURE14},
    {"GL_TEXTURE15", GL_TEXTURE15},
    {"GL_TEXTURE16", GL_TEXTURE16},
    {"GL_TEXTURE17", GL_TEXTURE17},
    {"GL_TEXTURE18", GL_TEXTURE18},
    {"GL_TEXTURE19", GL_TEXTURE19},
    {"GL_TEXTURE20", GL_TEXTURE20},
    {"GL_TEXTURE21", GL_TEXTURE21},
    {"GL_TEXTURE22", GL_TEXTURE22},
    {"GL_TEXTURE23", GL_TEXTURE23},
    {"GL_TEXTURE24", GL_TEXTURE24},
    {"GL_TEXTURE25", GL_TEXTURE25},
    {"GL_TEXTURE26", GL_TEXTURE26},
    {"GL_TEXTURE27", GL_TEXTURE27},
    {"GL_TEXTURE28", GL_TEXTURE28},
    {"GL_TEXTURE29", GL_TEXTURE29},
    {"GL_TEXTURE30", GL_TEXTURE30},
    {"GL_TEXTURE31", GL_TEXTURE31},
    {"GL_ACTIVE_TEXTURE", GL_ACTIVE_TEXTURE},
    {"GL_CLIENT_ACTIVE_TEXTURE", GL_CLIENT_ACTIVE_TEXTURE},
    {"GL_MAX_TEXTURE_UNITS", GL_MAX_TEXTURE_UNITS},
    {"GL_TRANSPOSE_MODELVIEW_MATRIX", GL_TRANSPOSE_MODELVIEW_MATRIX},
    {"GL_TRANSPOSE_PROJECTION_MATRIX", GL_TRANSPOSE_PROJECTION_MATRIX},
    {"GL_TRANSPOSE_TEXTURE_MATRIX", GL_TRANSPOSE_TEXTURE_MATRIX},
    {"GL_TRANSPOSE_COLOR_MATRIX", GL_TRANSPOSE_COLOR_MATRIX},
    {"GL_SUBTRACT", GL_SUBTRACT},
    {"GL_MAX_RENDERBUFFER_SIZE", GL_MAX_RENDERBUFFER_SIZE},
    {"GL_COMPRESSED_ALPHA", GL_COMPRESSED_ALPHA},
    {"GL_COMPRESSED_LUMINANCE", GL_COMPRESSED_LUMINANCE},
    {"GL_COMPRESSED_LUMINANCE_ALPHA", GL_COMPRESSED_LUMINANCE_ALPHA},
    {"GL_COMPRESSED_INTENSITY", GL_COMPRESSED_INTENSITY},
    {"GL_COMPRESSED_RGB", GL_COMPRESSED_RGB},
    {"GL_COMPRESSED_RGBA", GL_COMPRESSED_RGBA},
    {"GL_TEXTURE_COMPRESSION_HINT", GL_TEXTURE_COMPRESSION_HINT},
    {"GL_UNIFORM_BLOCK_REFERENCED_BY_TESS_CONTROL_SHADER", GL_UNIFORM_BLOCK_REFERENCED_BY_TESS_CONTROL_SHADER},
    {"GL_UNIFORM_BLOCK_REFERENCED_BY_TESS_EVALUATION_SHADER", GL_UNIFORM_BLOCK_REFERENCED_BY_TESS_EVALUATION_SHADER},
    {"GL_ALL_COMPLETED_NV", GL_ALL_COMPLETED_NV},
    {"GL_FENCE_STATUS_NV", GL_FENCE_STATUS_NV},
    {"GL_FENCE_CONDITION_NV", GL_FENCE_CONDITION_NV},
    {"GL_TEXTURE_RECTANGLE", GL_TEXTURE_RECTANGLE},
    {"GL_TEXTURE_BINDING_RECTANGLE", GL_TEXTURE_BINDING_RECTANGLE},
    {"GL_PROXY_TEXTURE_RECTANGLE", GL_PROXY_TEXTURE_RECTANGLE},
    {"GL_MAX_RECTANGLE_TEXTURE_SIZE", GL_MAX_RECTANGLE_TEXTURE_SIZE},
    {"GL_DEPTH_STENCIL", GL_DEPTH_STENCIL},
    {"GL_UNSIGNED_INT_24_8", GL_UNSIGNED_INT_24_8},
    {"GL_MAX_TEXTURE_LOD_BIAS", GL_MAX_TEXTURE_LOD_BIAS},
    {"GL_TEXTURE_MAX_ANISOTROPY_EXT", GL_TEXTURE_MAX_ANISOTROPY_EXT},
    {"GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT", GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT},
    {"GL_TEXTURE_FILTER_CONTROL", GL_TEXTURE_FILTER_CONTROL},
    {"GL_TEXTURE_LOD_BIAS", GL_TEXTURE_LOD_BIAS},
    {"GL_MODELVIEW1_STACK_DEPTH_EXT", GL_MODELVIEW1_STACK_DEPTH_EXT},
    {"GL_COMBINE4_NV", GL_COMBINE4_NV},
    {"GL_MAX_SHININESS_NV", GL_MAX_SHININESS_NV},
    {"GL_MAX_SPOT_EXPONENT_NV", GL_MAX_SPOT_EXPONENT_NV},
    {"GL_MODELVIEW1_MATRIX_EXT", GL_MODELVIEW1_MATRIX_EXT},
    {"GL_INCR_WRAP", GL_INCR_WRAP},
    {"GL_DECR_WRAP", GL_DECR_WRAP},
    {"GL_VERTEX_WEIGHTING_EXT", GL_VERTEX_WEIGHTING_EXT},
    {"GL_MODELVIEW1_ARB", GL_MODELVIEW1_ARB},
    {"GL_CURRENT_VERTEX_WEIGHT_EXT", GL_CURRENT_VERTEX_WEIGHT_EXT},
    {"GL_VERTEX_WEIGHT_ARRAY_EXT", GL_VERTEX_WEIGHT_ARRAY_EXT},
    {"GL_VERTEX_WEIGHT_ARRAY_SIZE_EXT", GL_VERTEX_WEIGHT_ARRAY_SIZE_EXT},
    {"GL_VERTEX_WEIGHT_ARRAY_TYPE_EXT", GL_VERTEX_WEIGHT_ARRAY_TYPE_EXT},
    {"GL_VERTEX_WEIGHT_ARRAY_STRIDE_EXT", GL_VERTEX_WEIGHT_ARRAY_STRIDE_EXT},
    {"GL_VERTEX_WEIGHT_ARRAY_POINTER_EXT", GL_VERTEX_WEIGHT_ARRAY_POINTER_EXT},
    {"GL_NORMAL_MAP", GL_NORMAL_MAP},
    {"GL_REFLECTION_MAP", GL_REFLECTION_MAP},
    {"GL_TEXTURE_CUBE_MAP", GL_TEXTURE_CUBE_MAP},
    {"GL_TEXTURE_BINDING_CUBE_MAP", GL_TEXTURE_BINDING_CUBE_MAP},
    {"GL_TEXTURE_CUBE_MAP_POSITIVE_X", GL_TEXTURE_CUBE_MAP_POSITIVE_X},
    {"GL_TEXTURE_CUBE_MAP_NEGATIVE_X", GL_TEXTURE_CUBE_MAP_NEGATIVE_X},
    {"GL_TEXTURE_CUBE_MAP_POSITIVE_Y", GL_TEXTURE_CUBE_MAP_POSITIVE_Y},
    {"GL_TEXTURE_CUBE_MAP_NEGATIVE_Y", GL_TEXTURE_CUBE_MAP_NEGATIVE_Y},
    {"GL_TEXTURE_CUBE_MAP_POSITIVE_Z", GL_TEXTURE_CUBE_MAP_POSITIVE_Z},
    {"GL_TEXTURE_CUBE_MAP_NEGATIVE_Z", GL_TEXTURE_CUBE_MAP_NEGATIVE_Z},
    {"GL_PROXY_TEXTURE_CUBE_MAP", GL_PROXY_TEXTURE_CUBE_MAP},
    {"GL_MAX_CUBE_MAP_TEXTURE_SIZE", GL_MAX_CUBE_MAP_TEXTURE_SIZE},
    {"GL_VERTEX_ARRAY_RANGE_NV", GL_VERTEX_ARRAY_RANGE_NV},
    {"GL_VERTEX_ARRAY_RANGE_LENGTH_NV", GL_VERTEX_ARRAY_RANGE_LENGTH_NV},
    {"GL_VERTEX_ARRAY_RANGE_VALID_NV", GL_VERTEX_ARRAY_RANGE_VALID_NV},
    {"GL_MAX_VERTEX_ARRAY_RANGE_ELEMENT_NV", GL_MAX_VERTEX_ARRAY_RANGE_ELEMENT_NV},
    {"GL_VERTEX_ARRAY_RANGE_POINTER_NV", GL_VERTEX_ARRAY_RANGE_POINTER_NV},
    {"GL_REGISTER_COMBINERS_NV", GL_REGISTER_COMBINERS_NV},
    {"GL_VARIABLE_A_NV", GL_VARIABLE_A_NV},
    {"GL_VARIABLE_B_NV", GL_VARIABLE_B_NV},
    {"GL_VARIABLE_C_NV", GL_VARIABLE_C_NV},
    {"GL_VARIABLE_D_NV", GL_VARIABLE_D_NV},
    {"GL_VARIABLE_E_NV", GL_VARIABLE_E_NV},
    {"GL_VARIABLE_F_NV", GL_VARIABLE_F_NV},
    {"GL_VARIABLE_G_NV", GL_VARIABLE_G_NV},
    {"GL_CONSTANT_COLOR0_NV", GL_CONSTANT_COLOR0_NV},
    {"GL_CONSTANT_COLOR1_NV", GL_CONSTANT_COLOR1_NV},
    {"GL_PRIMARY_COLOR_NV", GL_PRIMARY_COLOR_NV},
    {"GL_SECONDARY_COLOR_NV", GL_SECONDARY_COLOR_NV},
    {"GL_SPARE0_NV", GL_SPARE0_NV},
    {"GL_SPARE1_NV", GL_SPARE1_NV},
    {"GL_DISCARD_NV", GL_DISCARD_NV},
    {"GL_E_TIMES_F_NV", GL_E_TIMES_F_NV},
    {"GL_SPARE0_PLUS_SECONDARY_COLOR_NV", GL_SPARE0_PLUS_SECONDARY_COLOR_NV},
    {"GL_VERTEX_ARRAY_RANGE_WITHOUT_FLUSH_NV", GL_VERTEX_ARRAY_RANGE_WITHOUT_FLUSH_NV},
    {"GL_MULTISAMPLE_FILTER_HINT_NV", GL_MULTISAMPLE_FILTER_HINT_NV},
    {"GL_PER_STAGE_CONSTANTS_NV", GL_PER_STAGE_CONSTANTS_NV},
    {"GL_UNSIGNED_IDENTITY_NV", GL_UNSIGNED_IDENTITY_NV},
    {"GL_UNSIGNED_INVERT_NV", GL_UNSIGNED_INVERT_NV},
    {"GL_EXPAND_NORMAL_NV", GL_EXPAND_NORMAL_NV},
    {"GL_EXPAND_NEGATE_NV", GL_EXPAND_NEGATE_NV},
    {"GL_HALF_BIAS_NORMAL_NV", GL_HALF_BIAS_NORMAL_NV},
    {"GL_HALF_BIAS_NEGATE_NV", GL_HALF_BIAS_NEGATE_NV},
    {"GL_SIGNED_IDENTITY_NV", GL_SIGNED_IDENTITY_NV},
    {"GL_SIGNED_NEGATE_NV", GL_SIGNED_NEGATE_NV},
    {"GL_SCALE_BY_TWO_NV", GL_SCALE_BY_TWO_NV},
    {"GL_SCALE_BY_FOUR_NV", GL_SCALE_BY_FOUR_NV},
    {"GL_SCALE_BY_ONE_HALF_NV", GL_SCALE_BY_ONE_HALF_NV},
    {"GL_BIAS_BY_NEGATIVE_ONE_HALF_NV", GL_BIAS_BY_NEGATIVE_ONE_HALF_NV},
    {"GL_COMBINER_INPUT_NV", GL_COMBINER_INPUT_NV},
    {"GL_COMBINER_MAPPING_NV", GL_COMBINER_MAPPING_NV},
    {"GL_COMBINER_COMPONENT_USAGE_NV", GL_COMBINER_COMPONENT_USAGE_NV},
    {"GL_COMBINER_AB_DOT_PRODUCT_NV", GL_COMBINER_AB_DOT_PRODUCT_NV},
    {"GL_COMBINER_CD_DOT_PRODUCT_NV", GL_COMBINER_CD_DOT_PRODUCT_NV},
    {"GL_COMBINER_MUX_SUM_NV", GL_COMBINER_MUX_SUM_NV},
    {"GL_COMBINER_SCALE_NV", GL_COMBINER_SCALE_NV},
    {"GL_COMBINER_BIAS_NV", GL_COMBINER_BIAS_NV},
    {"GL_COMBINER_AB_OUTPUT_NV", GL_COMBINER_AB_OUTPUT_NV},
    {"GL_COMBINER_CD_OUTPUT_NV", GL_COMBINER_CD_OUTPUT_NV},
    {"GL_COMBINER_SUM_OUTPUT_NV", GL_COMBINER_SUM_OUTPUT_NV},
    {"GL_MAX_GENERAL_COMBINERS_NV", GL_MAX_GENERAL_COMBINERS_NV},
    {"GL_NUM_GENERAL_COMBINERS_NV", GL_NUM_GENERAL_COMBINERS_NV},
    {"GL_COLOR_SUM_CLAMP_NV", GL_COLOR_SUM_CLAMP_NV},
    {"GL_COMBINER0_NV", GL_COMBINER0_NV},
    {"GL_COMBINER1_NV", GL_COMBINER1_NV},
    {"GL_COMBINER2_NV", GL_COMBINER2_NV},
    {"GL_COMBINER3_NV", GL_COMBINER3_NV},
    {"GL_COMBINER4_NV", GL_COMBINER4_NV},
    {"GL_COMBINER5_NV", GL_COMBINER5_NV},
    {"GL_COMBINER6_NV", GL_COMBINER6_NV},
    {"GL_COMBINER7_NV", GL_COMBINER7_NV},
    {"GL_PRIMITIVE_RESTART_NV", GL_PRIMITIVE_RESTART_NV},
    {"GL_PRIMITIVE_RESTART_INDEX_NV", GL_PRIMITIVE_RESTART_INDEX_NV},
    {"GL_FOG_DISTANCE_MODE_NV", GL_FOG_DISTANCE_MODE_NV},
    {"GL_EYE_RADIAL_NV", GL_EYE_RADIAL_NV},
    {"GL_EYE_PLANE_ABSOLUTE_NV", GL_EYE_PLANE_ABSOLUTE_NV},
    {"GL_EMBOSS_LIGHT_NV", GL_EMBOSS_LIGHT_NV},
    {"GL_EMBOSS_CONSTANT_NV", GL_EMBOSS_CONSTANT_NV},
    {"GL_EMBOSS_MAP_NV", GL_EMBOSS_MAP_NV},
    {"GL_RED_MIN_CLAMP_INGR", GL_RED_MIN_CLAMP_INGR},
    {"GL_GREEN_MIN_CLAMP_INGR", GL_GREEN_MIN_CLAMP_INGR},
    {"GL_BLUE_MIN_CLAMP_INGR", GL_BLUE_MIN_CLAMP_INGR},
    {"GL_ALPHA_MIN_CLAMP_INGR", GL_ALPHA_MIN_CLAMP_INGR},
    {"GL_RED_MAX_CLAMP_INGR", GL_RED_MAX_CLAMP_INGR},
    {"GL_GREEN_MAX_CLAMP_INGR", GL_GREEN_MAX_CLAMP_INGR},
    {"GL_BLUE_MAX_CLAMP_INGR", GL_BLUE_MAX_CLAMP_INGR},
    {"GL_ALPHA_MAX_CLAMP_INGR", GL_ALPHA_MAX_CLAMP_INGR},
    {"GL_INTERLACE_READ_INGR", GL_INTERLACE_READ_INGR},
    {"GL_COMBINE", GL_COMBINE},
    {"GL_COMBINE_RGB", GL_COMBINE_RGB},
    {"GL_COMBINE_ALPHA", GL_COMBINE_ALPHA},
    {"GL_RGB_SCALE", GL_RGB_SCALE},
    {"GL_ADD_SIGNED", GL_ADD_SIGNED},
    {"GL_INTERPOLATE", GL_INTERPOLATE},
    {"GL_CONSTANT", GL_CONSTANT},
    {"GL_PRIMARY_COLOR", GL_PRIMARY_COLOR},
    {"GL_PREVIOUS", GL_PREVIOUS},
    {"GL_SRC0_RGB", GL_SRC0_RGB},
    {"GL_SRC1_RGB", GL_SRC1_RGB},
    {"GL_SRC2_RGB", GL_SRC2_RGB},
    {"GL_SOURCE3_RGB_NV", GL_SOURCE3_RGB_NV},
    {"GL_SRC0_ALPHA", GL_SRC0_ALPHA},
    {"GL_SRC1_ALPHA", GL_SRC1_ALPHA},
    {"GL_SRC2_ALPHA", GL_SRC2_ALPHA},
    {"GL_SOURCE3_ALPHA_NV", GL_SOURCE3_ALPHA_NV},
    {"GL_OPERAND0_RGB", GL_OPERAND0_RGB},
    {"GL_OPERAND1_RGB", GL_OPERAND1_RGB},
    {"GL_OPERAND2_RGB", GL_OPERAND2_RGB},
    {"GL_OPERAND3_RGB_NV", GL_OPERAND3_RGB_NV},
    {"GL_OPERAND0_ALPHA", GL_OPERAND0_ALPHA},
    {"GL_OPERAND1_ALPHA", GL_OPERAND1_ALPHA},
    {"GL_OPERAND2_ALPHA", GL_OPERAND2_ALPHA},
    {"GL_OPERAND3_ALPHA_NV", GL_OPERAND3_ALPHA_NV},
    {"GL_PACK_SUBSAMPLE_RATE_SGIX", GL_PACK_SUBSAMPLE_RATE_SGIX},
    {"GL_UNPACK_SUBSAMPLE_RATE_SGIX", GL_UNPACK_SUBSAMPLE_RATE_SGIX},
    {"GL_PIXEL_SUBSAMPLE_4444_SGIX", GL_PIXEL_SUBSAMPLE_4444_SGIX},
    {"GL_PIXEL_SUBSAMPLE_2424_SGIX", GL_PIXEL_SUBSAMPLE_2424_SGIX},
    {"GL_PIXEL_SUBSAMPLE_4242_SGIX", GL_PIXEL_SUBSAMPLE_4242_SGIX},
    {"GL_PERTURB_EXT", GL_PERTURB_EXT},
    {"GL_TEXTURE_NORMAL_EXT", GL_TEXTURE_NORMAL_EXT},
    {"GL_LIGHT_MODEL_SPECULAR_VECTOR_APPLE", GL_LIGHT_MODEL_SPECULAR_VECTOR_APPLE},
    {"GL_TRANSFORM_HINT_APPLE", GL_TRANSFORM_HINT_APPLE},
    {"GL_UNPACK_CLIENT_STORAGE_APPLE", GL_UNPACK_CLIENT_STORAGE_APPLE},
    {"GL_BUFFER_OBJECT_APPLE", GL_BUFFER_OBJECT_APPLE},
    {"GL_STORAGE_CLIENT_APPLE", GL_STORAGE_CLIENT_APPLE},
    {"GL_VERTEX_ARRAY_BINDING", GL_VERTEX_ARRAY_BINDING},
    {"GL_TEXTURE_RANGE_LENGTH_APPLE", GL_TEXTURE_RANGE_LENGTH_APPLE},
    {"GL_TEXTURE_RANGE_POINTER_APPLE", GL_TEXTURE_RANGE_POINTER_APPLE},
    {"GL_YCBCR_422_APPLE", GL_YCBCR_422_APPLE},
    {"GL_UNSIGNED_SHORT_8_8_APPLE", GL_UNSIGNED_SHORT_8_8_APPLE},
    {"GL_UNSIGNED_SHORT_8_8_REV_APPLE", GL_UNSIGNED_SHORT_8_8_REV_APPLE},
    {"GL_TEXTURE_STORAGE_HINT_APPLE", GL_TEXTURE_STORAGE_HINT_APPLE},
    {"GL_STORAGE_PRIVATE_APPLE", GL_STORAGE_PRIVATE_APPLE},
    {"GL_STORAGE_CACHED_APPLE", GL_STORAGE_CACHED_APPLE},
    {"GL_STORAGE_SHARED_APPLE", GL_STORAGE_SHARED_APPLE},
    {"GL_REPLACEMENT_CODE_ARRAY_SUN", GL_REPLACEMENT_CODE_ARRAY_SUN},
    {"GL_REPLACEMENT_CODE_ARRAY_TYPE_SUN", GL_REPLACEMENT_CODE_ARRAY_TYPE_SUN},
    {"GL_REPLACEMENT_CODE_ARRAY_STRIDE_SUN", GL_REPLACEMENT_CODE_ARRAY_STRIDE_SUN},
    {"GL_REPLACEMENT_CODE_ARRAY_POINTER_SUN", GL_REPLACEMENT_CODE_ARRAY_POINTER_SUN},
    {"GL_R1UI_V3F_SUN", GL_R1UI_V3F_SUN},
    {"GL_R1UI_C4UB_V3F_SUN", GL_R1UI_C4UB_V3F_SUN},
    {"GL_R1UI_C3F_V3F_SUN", GL_R1UI_C3F_V3F_SUN},
    {"GL_R1UI_N3F_V3F_SUN", GL_R1UI_N3F_V3F_SUN},
    {"GL_R1UI_C4F_N3F_V3F_SUN", GL_R1UI_C4F_N3F_V3F_SUN},
    {"GL_R1UI_T2F_V3F_SUN", GL_R1UI_T2F_V3F_SUN},
    {"GL_R1UI_T2F_N3F_V3F_SUN", GL_R1UI_T2F_N3F_V3F_SUN},
    {"GL_R1UI_T2F_C4F_N3F_V3F_SUN", GL_R1UI_T2F_C4F_N3F_V3F_SUN},
    {"GL_SLICE_ACCUM_SUN", GL_SLICE_ACCUM_SUN},
    {"GL_QUAD_MESH_SUN", GL_QUAD_MESH_SUN},
    {"GL_TRIANGLE_MESH_SUN", GL_TRIANGLE_MESH_SUN},
    {"GL_VERTEX_PROGRAM_ARB", GL_VERTEX_PROGRAM_ARB},
    {"GL_VERTEX_STATE_PROGRAM_NV", GL_VERTEX_STATE_PROGRAM_NV},
    {"GL_VERTEX_ATTRIB_ARRAY_ENABLED", GL_VERTEX_ATTRIB_ARRAY_ENABLED},
    {"GL_VERTEX_ATTRIB_ARRAY_SIZE", GL_VERTEX_ATTRIB_ARRAY_SIZE},
    {"GL_VERTEX_ATTRIB_ARRAY_STRIDE", GL_VERTEX_ATTRIB_ARRAY_STRIDE},
    {"GL_VERTEX_ATTRIB_ARRAY_TYPE", GL_VERTEX_ATTRIB_ARRAY_TYPE},
    {"GL_CURRENT_VERTEX_ATTRIB", GL_CURRENT_VERTEX_ATTRIB},
    {"GL_PROGRAM_LENGTH_ARB", GL_PROGRAM_LENGTH_ARB},
    {"GL_PROGRAM_STRING_ARB", GL_PROGRAM_STRING_ARB},
    {"GL_MODELVIEW_PROJECTION_NV", GL_MODELVIEW_PROJECTION_NV},
    {"GL_IDENTITY_NV", GL_IDENTITY_NV},
    {"GL_INVERSE_NV", GL_INVERSE_NV},
    {"GL_TRANSPOSE_NV", GL_TRANSPOSE_NV},
    {"GL_INVERSE_TRANSPOSE_NV", GL_INVERSE_TRANSPOSE_NV},
    {"GL_MAX_PROGRAM_MATRIX_STACK_DEPTH_ARB", GL_MAX_PROGRAM_MATRIX_STACK_DEPTH_ARB},
    {"GL_MAX_PROGRAM_MATRICES_ARB", GL_MAX_PROGRAM_MATRICES_ARB},
    {"GL_MATRIX0_NV", GL_MATRIX0_NV},
    {"GL_MATRIX1_NV", GL_MATRIX1_NV},
    {"GL_MATRIX2_NV", GL_MATRIX2_NV},
    {"GL_MATRIX3_NV", GL_MATRIX3_NV},
    {"GL_MATRIX4_NV", GL_MATRIX4_NV},
    {"GL_MATRIX5_NV", GL_MATRIX5_NV},
    {"GL_MATRIX6_NV", GL_MATRIX6_NV},
    {"GL_MATRIX7_NV", GL_MATRIX7_NV},
    {"GL_CURRENT_MATRIX_STACK_DEPTH_ARB", GL_CURRENT_MATRIX_STACK_DEPTH_ARB},
    {"GL_CURRENT_MATRIX_ARB", GL_CURRENT_MATRIX_ARB},
    {"GL_PROGRAM_POINT_SIZE", GL_PROGRAM_POINT_SIZE},
    {"GL_VERTEX_PROGRAM_TWO_SIDE", GL_VERTEX_PROGRAM_TWO_SIDE},
    {"GL_PROGRAM_PARAMETER_NV", GL_PROGRAM_PARAMETER_NV},
    {"GL_VERTEX_ATTRIB_ARRAY_POINTER", GL_VERTEX_ATTRIB_ARRAY_POINTER},
    {"GL_PROGRAM_TARGET_NV", GL_PROGRAM_TARGET_NV},
    {"GL_PROGRAM_RESIDENT_NV", GL_PROGRAM_RESIDENT_NV},
    {"GL_TRACK_MATRIX_NV", GL_TRACK_MATRIX_NV},
    {"GL_TRACK_MATRIX_TRANSFORM_NV", GL_TRACK_MATRIX_TRANSFORM_NV},
    {"GL_VERTEX_PROGRAM_BINDING_NV", GL_VERTEX_PROGRAM_BINDING_NV},
    {"GL_PROGRAM_ERROR_POSITION_ARB", GL_PROGRAM_ERROR_POSITION_ARB},
    {"GL_OFFSET_TEXTURE_RECTANGLE_NV", GL_OFFSET_TEXTURE_RECTANGLE_NV},
    {"GL_OFFSET_TEXTURE_RECTANGLE_SCALE_NV", GL_OFFSET_TEXTURE_RECTANGLE_SCALE_NV},
    {"GL_DOT_PRODUCT_TEXTURE_RECTANGLE_NV", GL_DOT_PRODUCT_TEXTURE_RECTANGLE_NV},
    {"GL_DEPTH_CLAMP", GL_DEPTH_CLAMP},
    {"GL_VERTEX_ATTRIB_ARRAY0_NV", GL_VERTEX_ATTRIB_ARRAY0_NV},
    {"GL_VERTEX_ATTRIB_ARRAY1_NV", GL_VERTEX_ATTRIB_ARRAY1_NV},
    {"GL_VERTEX_ATTRIB_ARRAY2_NV", GL_VERTEX_ATTRIB_ARRAY2_NV},
    {"GL_VERTEX_ATTRIB_ARRAY3_NV", GL_VERTEX_ATTRIB_ARRAY3_NV},
    {"GL_VERTEX_ATTRIB_ARRAY4_NV", GL_VERTEX_ATTRIB_ARRAY4_NV},
    {"GL_VERTEX_ATTRIB_ARRAY5_NV", GL_VERTEX_ATTRIB_ARRAY5_NV},
    {"GL_VERTEX_ATTRIB_ARRAY6_NV", GL_VERTEX_ATTRIB_ARRAY6_NV},
    {"GL_VERTEX_ATTRIB_ARRAY7_NV", GL_VERTEX_ATTRIB_ARRAY7_NV},
    {"GL_VERTEX_ATTRIB_ARRAY8_NV", GL_VERTEX_ATTRIB_ARRAY8_NV},
    {"GL_VERTEX_ATTRIB_ARRAY9_NV", GL_VERTEX_ATTRIB_ARRAY9_NV},
    {"GL_VERTEX_ATTRIB_ARRAY10_NV", GL_VERTEX_ATTRIB_ARRAY10_NV},
    {"GL_VERTEX_ATTRIB_ARRAY11_NV", GL_VERTEX_ATTRIB_ARRAY11_NV},
    {"GL_VERTEX_ATTRIB_ARRAY12_NV", GL_VERTEX_ATTRIB_ARRAY12_NV},
    {"GL_VERTEX_ATTRIB_ARRAY13_NV", GL_VERTEX_ATTRIB_ARRAY13_NV},
    {"GL_VERTEX_ATTRIB_ARRAY14_NV", GL_VERTEX_ATTRIB_ARRAY14_NV},
    {"GL_VERTEX_ATTRIB_ARRAY15_NV", GL_VERTEX_ATTRIB_ARRAY15_NV},
    {"GL_MAP1_VERTEX_ATTRIB0_4_NV", GL_MAP1_VERTEX_ATTRIB0_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB1_4_NV", GL_MAP1_VERTEX_ATTRIB1_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB2_4_NV", GL_MAP1_VERTEX_ATTRIB2_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB3_4_NV", GL_MAP1_VERTEX_ATTRIB3_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB4_4_NV", GL_MAP1_VERTEX_ATTRIB4_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB5_4_NV", GL_MAP1_VERTEX_ATTRIB5_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB6_4_NV", GL_MAP1_VERTEX_ATTRIB6_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB7_4_NV", GL_MAP1_VERTEX_ATTRIB7_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB8_4_NV", GL_MAP1_VERTEX_ATTRIB8_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB9_4_NV", GL_MAP1_VERTEX_ATTRIB9_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB10_4_NV", GL_MAP1_VERTEX_ATTRIB10_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB11_4_NV", GL_MAP1_VERTEX_ATTRIB11_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB12_4_NV", GL_MAP1_VERTEX_ATTRIB12_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB13_4_NV", GL_MAP1_VERTEX_ATTRIB13_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB14_4_NV", GL_MAP1_VERTEX_ATTRIB14_4_NV},
    {"GL_MAP1_VERTEX_ATTRIB15_4_NV", GL_MAP1_VERTEX_ATTRIB15_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB0_4_NV", GL_MAP2_VERTEX_ATTRIB0_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB1_4_NV", GL_MAP2_VERTEX_ATTRIB1_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB2_4_NV", GL_MAP2_VERTEX_ATTRIB2_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB3_4_NV", GL_MAP2_VERTEX_ATTRIB3_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB4_4_NV", GL_MAP2_VERTEX_ATTRIB4_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB5_4_NV", GL_MAP2_VERTEX_ATTRIB5_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB6_4_NV", GL_MAP2_VERTEX_ATTRIB6_4_NV},
    {"GL_PROGRAM_BINDING_ARB", GL_PROGRAM_BINDING_ARB},
    {"GL_MAP2_VERTEX_ATTRIB8_4_NV", GL_MAP2_VERTEX_ATTRIB8_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB9_4_NV", GL_MAP2_VERTEX_ATTRIB9_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB10_4_NV", GL_MAP2_VERTEX_ATTRIB10_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB11_4_NV", GL_MAP2_VERTEX_ATTRIB11_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB12_4_NV", GL_MAP2_VERTEX_ATTRIB12_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB13_4_NV", GL_MAP2_VERTEX_ATTRIB13_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB14_4_NV", GL_MAP2_VERTEX_ATTRIB14_4_NV},
    {"GL_MAP2_VERTEX_ATTRIB15_4_NV", GL_MAP2_VERTEX_ATTRIB15_4_NV},
    {"GL_TEXTURE_COMPRESSED_IMAGE_SIZE", GL_TEXTURE_COMPRESSED_IMAGE_SIZE},
    {"GL_TEXTURE_COMPRESSED", GL_TEXTURE_COMPRESSED},
    {"GL_NUM_COMPRESSED_TEXTURE_FORMATS", GL_NUM_COMPRESSED_TEXTURE_FORMATS},
    {"GL_COMPRESSED_TEXTURE_FORMATS", GL_COMPRESSED_TEXTURE_FORMATS},
    {"GL_MAX_VERTEX_UNITS_ARB", GL_MAX_VERTEX_UNITS_ARB},
    {"GL_ACTIVE_VERTEX_UNITS_ARB", GL_ACTIVE_VERTEX_UNITS_ARB},
    {"GL_WEIGHT_SUM_UNITY_ARB", GL_WEIGHT_SUM_UNITY_ARB},
    {"GL_VERTEX_BLEND_ARB", GL_VERTEX_BLEND_ARB},
    {"GL_CURRENT_WEIGHT_ARB", GL_CURRENT_WEIGHT_ARB},
    {"GL_WEIGHT_ARRAY_TYPE_ARB", GL_WEIGHT_ARRAY_TYPE_ARB},
    {"GL_WEIGHT_ARRAY_STRIDE_ARB", GL_WEIGHT_ARRAY_STRIDE_ARB},
    {"GL_WEIGHT_ARRAY_SIZE_ARB", GL_WEIGHT_ARRAY_SIZE_ARB},
    {"GL_WEIGHT_ARRAY_POINTER_ARB", GL_WEIGHT_ARRAY_POINTER_ARB},
    {"GL_WEIGHT_ARRAY_ARB", GL_WEIGHT_ARRAY_ARB},
    {"GL_DOT3_RGB", GL_DOT3_RGB},
    {"GL_DOT3_RGBA", GL_DOT3_RGBA},
    {"GL_COMPRESSED_RGB_FXT1_3DFX", GL_COMPRESSED_RGB_FXT1_3DFX},
    {"GL_COMPRESSED_RGBA_FXT1_3DFX", GL_COMPRESSED_RGBA_FXT1_3DFX},
    {"GL_MULTISAMPLE_3DFX", GL_MULTISAMPLE_3DFX},
    {"GL_SAMPLE_BUFFERS_3DFX", GL_SAMPLE_BUFFERS_3DFX},
    {"GL_SAMPLES_3DFX", GL_SAMPLES_3DFX},
    {"GL_EVAL_2D_NV", GL_EVAL_2D_NV},
    {"GL_EVAL_TRIANGULAR_2D_NV", GL_EVAL_TRIANGULAR_2D_NV},
    {"GL_MAP_TESSELLATION_NV", GL_MAP_TESSELLATION_NV},
    {"GL_MAP_ATTRIB_U_ORDER_NV", GL_MAP_ATTRIB_U_ORDER_NV},
    {"GL_MAP_ATTRIB_V_ORDER_NV", GL_MAP_ATTRIB_V_ORDER_NV},
    {"GL_EVAL_FRACTIONAL_TESSELLATION_NV", GL_EVAL_FRACTIONAL_TESSELLATION_NV},
    {"GL_EVAL_VERTEX_ATTRIB0_NV", GL_EVAL_VERTEX_ATTRIB0_NV},
    {"GL_EVAL_VERTEX_ATTRIB1_NV", GL_EVAL_VERTEX_ATTRIB1_NV},
    {"GL_EVAL_VERTEX_ATTRIB2_NV", GL_EVAL_VERTEX_ATTRIB2_NV},
    {"GL_EVAL_VERTEX_ATTRIB3_NV", GL_EVAL_VERTEX_ATTRIB3_NV},
    {"GL_EVAL_VERTEX_ATTRIB4_NV", GL_EVAL_VERTEX_ATTRIB4_NV},
    {"GL_EVAL_VERTEX_ATTRIB5_NV", GL_EVAL_VERTEX_ATTRIB5_NV},
    {"GL_EVAL_VERTEX_ATTRIB6_NV", GL_EVAL_VERTEX_ATTRIB6_NV},
    {"GL_EVAL_VERTEX_ATTRIB7_NV", GL_EVAL_VERTEX_ATTRIB7_NV},
    {"GL_EVAL_VERTEX_ATTRIB8_NV", GL_EVAL_VERTEX_ATTRIB8_NV},
    {"GL_EVAL_VERTEX_ATTRIB9_NV", GL_EVAL_VERTEX_ATTRIB9_NV},
    {"GL_EVAL_VERTEX_ATTRIB10_NV", GL_EVAL_VERTEX_ATTRIB10_NV},
    {"GL_EVAL_VERTEX_ATTRIB11_NV", GL_EVAL_VERTEX_ATTRIB11_NV},
    {"GL_EVAL_VERTEX_ATTRIB12_NV", GL_EVAL_VERTEX_ATTRIB12_NV},
    {"GL_EVAL_VERTEX_ATTRIB13_NV", GL_EVAL_VERTEX_ATTRIB13_NV},
    {"GL_EVAL_VERTEX_ATTRIB14_NV", GL_EVAL_VERTEX_ATTRIB14_NV},
    {"GL_EVAL_VERTEX_ATTRIB15_NV", GL_EVAL_VERTEX_ATTRIB15_NV},
    {"GL_MAX_MAP_TESSELLATION_NV", GL_MAX_MAP_TESSELLATION_NV},
    {"GL_MAX_RATIONAL_EVAL_ORDER_NV", GL_MAX_RATIONAL_EVAL_ORDER_NV},
    {"GL_MAX_PROGRAM_PATCH_ATTRIBS_NV", GL_MAX_PROGRAM_PATCH_ATTRIBS_NV},
    {"GL_RGBA_UNSIGNED_DOT_PRODUCT_MAPPING_NV", GL_RGBA_UNSIGNED_DOT_PRODUCT_MAPPING_NV},
    {"GL_UNSIGNED_INT_S8_S8_8_8_NV", GL_UNSIGNED_INT_S8_S8_8_8_NV},
    {"GL_UNSIGNED_INT_8_8_S8_S8_REV_NV", GL_UNSIGNED_INT_8_8_S8_S8_REV_NV},
    {"GL_DSDT_MAG_INTENSITY_NV", GL_DSDT_MAG_INTENSITY_NV},
    {"GL_SHADER_CONSISTENT_NV", GL_SHADER_CONSISTENT_NV},
    {"GL_TEXTURE_SHADER_NV", GL_TEXTURE_SHADER_NV},
    {"GL_SHADER_OPERATION_NV", GL_SHADER_OPERATION_NV},
    {"GL_CULL_MODES_NV", GL_CULL_MODES_NV},
    {"GL_OFFSET_TEXTURE_MATRIX_NV", GL_OFFSET_TEXTURE_MATRIX_NV},
    {"GL_OFFSET_TEXTURE_SCALE_NV", GL_OFFSET_TEXTURE_SCALE_NV},
    {"GL_OFFSET_TEXTURE_BIAS_NV", GL_OFFSET_TEXTURE_BIAS_NV},
    {"GL_PREVIOUS_TEXTURE_INPUT_NV", GL_PREVIOUS_TEXTURE_INPUT_NV},
    {"GL_CONST_EYE_NV", GL_CONST_EYE_NV},
    {"GL_PASS_THROUGH_NV", GL_PASS_THROUGH_NV},
    {"GL_CULL_FRAGMENT_NV", GL_CULL_FRAGMENT_NV},
    {"GL_OFFSET_TEXTURE_2D_NV", GL_OFFSET_TEXTURE_2D_NV},
    {"GL_DEPENDENT_AR_TEXTURE_2D_NV", GL_DEPENDENT_AR_TEXTURE_2D_NV},
    {"GL_DEPENDENT_GB_TEXTURE_2D_NV", GL_DEPENDENT_GB_TEXTURE_2D_NV},
    {"GL_SURFACE_STATE_NV", GL_SURFACE_STATE_NV},
    {"GL_DOT_PRODUCT_NV", GL_DOT_PRODUCT_NV},
    {"GL_DOT_PRODUCT_DEPTH_REPLACE_NV", GL_DOT_PRODUCT_DEPTH_REPLACE_NV},
    {"GL_DOT_PRODUCT_TEXTURE_2D_NV", GL_DOT_PRODUCT_TEXTURE_2D_NV},
    {"GL_DOT_PRODUCT_TEXTURE_3D_NV", GL_DOT_PRODUCT_TEXTURE_3D_NV},
    {"GL_DOT_PRODUCT_TEXTURE_CUBE_MAP_NV", GL_DOT_PRODUCT_TEXTURE_CUBE_MAP_NV},
    {"GL_DOT_PRODUCT_DIFFUSE_CUBE_MAP_NV", GL_DOT_PRODUCT_DIFFUSE_CUBE_MAP_NV},
    {"GL_DOT_PRODUCT_REFLECT_CUBE_MAP_NV", GL_DOT_PRODUCT_REFLECT_CUBE_MAP_NV},
    {"GL_DOT_PRODUCT_CONST_EYE_REFLECT_CUBE_MAP_NV", GL_DOT_PRODUCT_CONST_EYE_REFLECT_CUBE_MAP_NV},
    {"GL_HILO_NV", GL_HILO_NV},
    {"GL_DSDT_NV", GL_DSDT_NV},
    {"GL_DSDT_MAG_NV", GL_DSDT_MAG_NV},
    {"GL_DSDT_MAG_VIB_NV", GL_DSDT_MAG_VIB_NV},
    {"GL_HILO16_NV", GL_HILO16_NV},
    {"GL_SIGNED_HILO_NV", GL_SIGNED_HILO_NV},
    {"GL_SIGNED_HILO16_NV", GL_SIGNED_HILO16_NV},
    {"GL_SIGNED_RGBA_NV", GL_SIGNED_RGBA_NV},
    {"GL_SIGNED_RGBA8_NV", GL_SIGNED_RGBA8_NV},
    {"GL_SURFACE_REGISTERED_NV", GL_SURFACE_REGISTERED_NV},
    {"GL_SIGNED_RGB_NV", GL_SIGNED_RGB_NV},
    {"GL_SIGNED_RGB8_NV", GL_SIGNED_RGB8_NV},
    {"GL_SURFACE_MAPPED_NV", GL_SURFACE_MAPPED_NV},
    {"GL_SIGNED_LUMINANCE_NV", GL_SIGNED_LUMINANCE_NV},
    {"GL_SIGNED_LUMINANCE8_NV", GL_SIGNED_LUMINANCE8_NV},
    {"GL_SIGNED_LUMINANCE_ALPHA_NV", GL_SIGNED_LUMINANCE_ALPHA_NV},
    {"GL_SIGNED_LUMINANCE8_ALPHA8_NV", GL_SIGNED_LUMINANCE8_ALPHA8_NV},
    {"GL_SIGNED_ALPHA_NV", GL_SIGNED_ALPHA_NV},
    {"GL_SIGNED_ALPHA8_NV", GL_SIGNED_ALPHA8_NV},
    {"GL_SIGNED_INTENSITY_NV", GL_SIGNED_INTENSITY_NV},
    {"GL_SIGNED_INTENSITY8_NV", GL_SIGNED_INTENSITY8_NV},
    {"GL_DSDT8_NV", GL_DSDT8_NV},
    {"GL_DSDT8_MAG8_NV", GL_DSDT8_MAG8_NV},
    {"GL_DSDT8_MAG8_INTENSITY8_NV", GL_DSDT8_MAG8_INTENSITY8_NV},
    {"GL_SIGNED_RGB_UNSIGNED_ALPHA_NV", GL_SIGNED_RGB_UNSIGNED_ALPHA_NV},
    {"GL_SIGNED_RGB8_UNSIGNED_ALPHA8_NV", GL_SIGNED_RGB8_UNSIGNED_ALPHA8_NV},
    {"GL_HI_SCALE_NV", GL_HI_SCALE_NV},
    {"GL_LO_SCALE_NV", GL_LO_SCALE_NV},
    {"GL_DS_SCALE_NV", GL_DS_SCALE_NV},
    {"GL_DT_SCALE_NV", GL_DT_SCALE_NV},
    {"GL_MAGNITUDE_SCALE_NV", GL_MAGNITUDE_SCALE_NV},
    {"GL_VIBRANCE_SCALE_NV", GL_VIBRANCE_SCALE_NV},
    {"GL_HI_BIAS_NV", GL_HI_BIAS_NV},
    {"GL_LO_BIAS_NV", GL_LO_BIAS_NV},
    {"GL_DS_BIAS_NV", GL_DS_BIAS_NV},
    {"GL_DT_BIAS_NV", GL_DT_BIAS_NV},
    {"GL_MAGNITUDE_BIAS_NV", GL_MAGNITUDE_BIAS_NV},
    {"GL_VIBRANCE_BIAS_NV", GL_VIBRANCE_BIAS_NV},
    {"GL_TEXTURE_BORDER_VALUES_NV", GL_TEXTURE_BORDER_VALUES_NV},
    {"GL_TEXTURE_HI_SIZE_NV", GL_TEXTURE_HI_SIZE_NV},
    {"GL_TEXTURE_LO_SIZE_NV", GL_TEXTURE_LO_SIZE_NV},
    {"GL_TEXTURE_DS_SIZE_NV", GL_TEXTURE_DS_SIZE_NV},
    {"GL_TEXTURE_DT_SIZE_NV", GL_TEXTURE_DT_SIZE_NV},
    {"GL_TEXTURE_MAG_SIZE_NV", GL_TEXTURE_MAG_SIZE_NV},
    {"GL_MODELVIEW2_ARB", GL_MODELVIEW2_ARB},
    {"GL_MODELVIEW3_ARB", GL_MODELVIEW3_ARB},
    {"GL_MODELVIEW4_ARB", GL_MODELVIEW4_ARB},
    {"GL_MODELVIEW5_ARB", GL_MODELVIEW5_ARB},
    {"GL_MODELVIEW6_ARB", GL_MODELVIEW6_ARB},
    {"GL_MODELVIEW7_ARB", GL_MODELVIEW7_ARB},
    {"GL_MODELVIEW8_ARB", GL_MODELVIEW8_ARB},
    {"GL_MODELVIEW9_ARB", GL_MODELVIEW9_ARB},
    {"GL_MODELVIEW10_ARB", GL_MODELVIEW10_ARB},
    {"GL_MODELVIEW11_ARB", GL_MODELVIEW11_ARB},
    {"GL_MODELVIEW12_ARB", GL_MODELVIEW12_ARB},
    {"GL_MODELVIEW13_ARB", GL_MODELVIEW13_ARB},
    {"GL_MODELVIEW14_ARB", GL_MODELVIEW14_ARB},
    {"GL_MODELVIEW15_ARB", GL_MODELVIEW15_ARB},
    {"GL_MODELVIEW16_ARB", GL_MODELVIEW16_ARB},
    {"GL_MODELVIEW17_ARB", GL_MODELVIEW17_ARB},
    {"GL_MODELVIEW18_ARB", GL_MODELVIEW18_ARB},
    {"GL_MODELVIEW19_ARB", GL_MODELVIEW19_ARB},
    {"GL_MODELVIEW20_ARB", GL_MODELVIEW20_ARB},
    {"GL_MODELVIEW21_ARB", GL_MODELVIEW21_ARB},
    {"GL_MODELVIEW22_ARB", GL_MODELVIEW22_ARB},
    {"GL_MODELVIEW23_ARB", GL_MODELVIEW23_ARB},
    {"GL_MODELVIEW24_ARB", GL_MODELVIEW24_ARB},
    {"GL_MODELVIEW25_ARB", GL_MODELVIEW25_ARB},
    {"GL_MODELVIEW26_ARB", GL_MODELVIEW26_ARB},
    {"GL_MODELVIEW27_ARB", GL_MODELVIEW27_ARB},
    {"GL_MODELVIEW28_ARB", GL_MODELVIEW28_ARB},
    {"GL_MODELVIEW29_ARB", GL_MODELVIEW29_ARB},
    {"GL_MODELVIEW30_ARB", GL_MODELVIEW30_ARB},
    {"GL_MODELVIEW31_ARB", GL_MODELVIEW31_ARB},
    {"GL_DOT3_RGB_EXT", GL_DOT3_RGB_EXT},
    {"GL_PROGRAM_BINARY_LENGTH", GL_PROGRAM_BINARY_LENGTH},
    {"GL_MIRROR_CLAMP_EXT", GL_MIRROR_CLAMP_EXT},
    {"GL_MIRROR_CLAMP_TO_EDGE", GL_MIRROR_CLAMP_TO_EDGE},
    {"GL_MODULATE_ADD_ATI", GL_MODULATE_ADD_ATI},
    {"GL_MODULATE_SIGNED_ADD_ATI", GL_MODULATE_SIGNED_ADD_ATI},
    {"GL_MODULATE_SUBTRACT_ATI", GL_MODULATE_SUBTRACT_ATI},
    {"GL_SET_AMD", GL_SET_AMD},
    {"GL_REPLACE_VALUE_AMD", GL_REPLACE_VALUE_AMD},
    {"GL_STENCIL_OP_VALUE_AMD", GL_STENCIL_OP_VALUE_AMD},
    {"GL_STENCIL_BACK_OP_VALUE_AMD", GL_STENCIL_BACK_OP_VALUE_AMD},
    {"GL_VERTEX_ATTRIB_ARRAY_LONG", GL_VERTEX_ATTRIB_ARRAY_LONG},
    {"GL_OCCLUSION_QUERY_EVENT_MASK_AMD", GL_OCCLUSION_QUERY_EVENT_MASK_AMD},
    {"GL_YCBCR_MESA", GL_YCBCR_MESA},
    {"GL_PACK_INVERT_MESA", GL_PACK_INVERT_MESA},
    {"GL_TEXTURE_1D_STACK_MESAX", GL_TEXTURE_1D_STACK_MESAX},
    {"GL_TEXTURE_2D_STACK_MESAX", GL_TEXTURE_2D_STACK_MESAX},
    {"GL_PROXY_TEXTURE_1D_STACK_MESAX", GL_PROXY_TEXTURE_1D_STACK_MESAX},
    {"GL_PROXY_TEXTURE_2D_STACK_MESAX", GL_PROXY_TEXTURE_2D_STACK_MESAX},
    {"GL_TEXTURE_1D_STACK_BINDING_MESAX", GL_TEXTURE_1D_STACK_BINDING_MESAX},
    {"GL_TEXTURE_2D_STACK_BINDING_MESAX", GL_TEXTURE_2D_STACK_BINDING_MESAX},
    {"GL_STATIC_ATI", GL_STATIC_ATI},
    {"GL_DYNAMIC_ATI", GL_DYNAMIC_ATI},
    {"GL_PRESERVE_ATI", GL_PRESERVE_ATI},
    {"GL_DISCARD_ATI", GL_DISCARD_ATI},
    {"GL_BUFFER_SIZE", GL_BUFFER_SIZE},
    {"GL_BUFFER_USAGE", GL_BUFFER_USAGE},
    {"GL_ARRAY_OBJECT_BUFFER_ATI", GL_ARRAY_OBJECT_BUFFER_ATI},
    {"GL_ARRAY_OBJECT_OFFSET_ATI", GL_ARRAY_OBJECT_OFFSET_ATI},
    {"GL_ELEMENT_ARRAY_ATI", GL_ELEMENT_ARRAY_ATI},
    {"GL_ELEMENT_ARRAY_TYPE_ATI", GL_ELEMENT_ARRAY_TYPE_ATI},
    {"GL_ELEMENT_ARRAY_POINTER_ATI", GL_ELEMENT_ARRAY_POINTER_ATI},
    {"GL_MAX_VERTEX_STREAMS_ATI", GL_MAX_VERTEX_STREAMS_ATI},
    {"GL_VERTEX_STREAM0_ATI", GL_VERTEX_STREAM0_ATI},
    {"GL_VERTEX_STREAM1_ATI", GL_VERTEX_STREAM1_ATI},
    {"GL_VERTEX_STREAM2_ATI", GL_VERTEX_STREAM2_ATI},
    {"GL_VERTEX_STREAM3_ATI", GL_VERTEX_STREAM3_ATI},
    {"GL_VERTEX_STREAM4_ATI", GL_VERTEX_STREAM4_ATI},
    {"GL_VERTEX_STREAM5_ATI", GL_VERTEX_STREAM5_ATI},
    {"GL_VERTEX_STREAM6_ATI", GL_VERTEX_STREAM6_ATI},
    {"GL_VERTEX_STREAM7_ATI", GL_VERTEX_STREAM7_ATI},
    {"GL_VERTEX_SOURCE_ATI", GL_VERTEX_SOURCE_ATI},
    {"GL_BUMP_ROT_MATRIX_ATI", GL_BUMP_ROT_MATRIX_ATI},
    {"GL_BUMP_ROT_MATRIX_SIZE_ATI", GL_BUMP_ROT_MATRIX_SIZE_ATI},
    {"GL_BUMP_NUM_TEX_UNITS_ATI", GL_BUMP_NUM_TEX_UNITS_ATI},
    {"GL_BUMP_TEX_UNITS_ATI", GL_BUMP_TEX_UNITS_ATI},
    {"GL_DUDV_ATI", GL_DUDV_ATI},
    {"GL_DU8DV8_ATI", GL_DU8DV8_ATI},
    {"GL_BUMP_ENVMAP_ATI", GL_BUMP_ENVMAP_ATI},
    {"GL_BUMP_TARGET_ATI", GL_BUMP_TARGET_ATI},
    {"GL_VERTEX_SHADER_EXT", GL_VERTEX_SHADER_EXT},
    {"GL_VERTEX_SHADER_BINDING_EXT", GL_VERTEX_SHADER_BINDING_EXT},
    {"GL_OP_INDEX_EXT", GL_OP_INDEX_EXT},
    {"GL_OP_NEGATE_EXT", GL_OP_NEGATE_EXT},
    {"GL_OP_DOT3_EXT", GL_OP_DOT3_EXT},
    {"GL_OP_DOT4_EXT", GL_OP_DOT4_EXT},
    {"GL_OP_MUL_EXT", GL_OP_MUL_EXT},
    {"GL_OP_ADD_EXT", GL_OP_ADD_EXT},
    {"GL_OP_MADD_EXT", GL_OP_MADD_EXT},
    {"GL_OP_FRAC_EXT", GL_OP_FRAC_EXT},
    {"GL_OP_MAX_EXT", GL_OP_MAX_EXT},
    {"GL_OP_MIN_EXT", GL_OP_MIN_EXT},
    {"GL_OP_SET_GE_EXT", GL_OP_SET_GE_EXT},
    {"GL_OP_SET_LT_EXT", GL_OP_SET_LT_EXT},
    {"GL_OP_CLAMP_EXT", GL_OP_CLAMP_EXT},
    {"GL_OP_FLOOR_EXT", GL_OP_FLOOR_EXT},
    {"GL_OP_ROUND_EXT", GL_OP_ROUND_EXT},
    {"GL_OP_EXP_BASE_2_EXT", GL_OP_EXP_BASE_2_EXT},
    {"GL_OP_LOG_BASE_2_EXT", GL_OP_LOG_BASE_2_EXT},
    {"GL_OP_POWER_EXT", GL_OP_POWER_EXT},
    {"GL_OP_RECIP_EXT", GL_OP_RECIP_EXT},
    {"GL_OP_RECIP_SQRT_EXT", GL_OP_RECIP_SQRT_EXT},
    {"GL_OP_SUB_EXT", GL_OP_SUB_EXT},
    {"GL_OP_CROSS_PRODUCT_EXT", GL_OP_CROSS_PRODUCT_EXT},
    {"GL_OP_MULTIPLY_MATRIX_EXT", GL_OP_MULTIPLY_MATRIX_EXT},
    {"GL_OP_MOV_EXT", GL_OP_MOV_EXT},
    {"GL_OUTPUT_VERTEX_EXT", GL_OUTPUT_VERTEX_EXT},
    {"GL_OUTPUT_COLOR0_EXT", GL_OUTPUT_COLOR0_EXT},
    {"GL_OUTPUT_COLOR1_EXT", GL_OUTPUT_COLOR1_EXT},
    {"GL_OUTPUT_TEXTURE_COORD0_EXT", GL_OUTPUT_TEXTURE_COORD0_EXT},
    {"GL_OUTPUT_TEXTURE_COORD1_EXT", GL_OUTPUT_TEXTURE_COORD1_EXT},
    {"GL_OUTPUT_TEXTURE_COORD2_EXT", GL_OUTPUT_TEXTURE_COORD2_EXT},
    {"GL_OUTPUT_TEXTURE_COORD3_EXT", GL_OUTPUT_TEXTURE_COORD3_EXT},
    {"GL_OUTPUT_TEXTURE_COORD4_EXT", GL_OUTPUT_TEXTURE_COORD4_EXT},
    {"GL_OUTPUT_TEXTURE_COORD5_EXT", GL_OUTPUT_TEXTURE_COORD5_EXT},
    {"GL_OUTPUT_TEXTURE_COORD6_EXT", GL_OUTPUT_TEXTURE_COORD6_EXT},
    {"GL_OUTPUT_TEXTURE_COORD7_EXT", GL_OUTPUT_TEXTURE_COORD7_EXT},
    {"GL_OUTPUT_TEXTURE_COORD8_EXT", GL_OUTPUT_TEXTURE_COORD8_EXT},
    {"GL_OUTPUT_TEXTURE_COORD9_EXT", GL_OUTPUT_TEXTURE_COORD9_EXT},
    {"GL_OUTPUT_TEXTURE_COORD10_EXT", GL_OUTPUT_TEXTURE_COORD10_EXT},
    {"GL_OUTPUT_TEXTURE_COORD11_EXT", GL_OUTPUT_TEXTURE_COORD11_EXT},
    {"GL_OUTPUT_TEXTURE_COORD12_EXT", GL_OUTPUT_TEXTURE_COORD12_EXT},
    {"GL_OUTPUT_TEXTURE_COORD13_EXT", GL_OUTPUT_TEXTURE_COORD13_EXT},
    {"GL_OUTPUT_TEXTURE_COORD14_EXT", GL_OUTPUT_TEXTURE_COORD14_EXT},
    {"GL_OUTPUT_TEXTURE_COORD15_EXT", GL_OUTPUT_TEXTURE_COORD15_EXT},
    {"GL_OUTPUT_TEXTURE_COORD16_EXT", GL_OUTPUT_TEXTURE_COORD16_EXT},
    {"GL_OUTPUT_TEXTURE_COORD17_EXT", GL_OUTPUT_TEXTURE_COORD17_EXT},
    {"GL_OUTPUT_TEXTURE_COORD18_EXT", GL_OUTPUT_TEXTURE_COORD18_EXT},
    {"GL_OUTPUT_TEXTURE_COORD19_EXT", GL_OUTPUT_TEXTURE_COORD19_EXT},
    {"GL_OUTPUT_TEXTURE_COORD20_EXT", GL_OUTPUT_TEXTURE_COORD20_EXT},
    {"GL_OUTPUT_TEXTURE_COORD21_EXT", GL_OUTPUT_TEXTURE_COORD21_EXT},
    {"GL_OUTPUT_TEXTURE_COORD22_EXT", GL_OUTPUT_TEXTURE_COORD22_EXT},
    {"GL_OUTPUT_TEXTURE_COORD23_EXT", GL_OUTPUT_TEXTURE_COORD23_EXT},
    {"GL_OUTPUT_TEXTURE_COORD24_EXT", GL_OUTPUT_TEXTURE_COORD24_EXT},
    {"GL_OUTPUT_TEXTURE_COORD25_EXT", GL_OUTPUT_TEXTURE_COORD25_EXT},
    {"GL_OUTPUT_TEXTURE_COORD26_EXT", GL_OUTPUT_TEXTURE_COORD26_EXT},
    {"GL_OUTPUT_TEXTURE_COORD27_EXT", GL_OUTPUT_TEXTURE_COORD27_EXT},
    {"GL_OUTPUT_TEXTURE_COORD28_EXT", GL_OUTPUT_TEXTURE_COORD28_EXT},
    {"GL_OUTPUT_TEXTURE_COORD29_EXT", GL_OUTPUT_TEXTURE_COORD29_EXT},
    {"GL_OUTPUT_TEXTURE_COORD30_EXT", GL_OUTPUT_TEXTURE_COORD30_EXT},
    {"GL_OUTPUT_TEXTURE_COORD31_EXT", GL_OUTPUT_TEXTURE_COORD31_EXT},
    {"GL_OUTPUT_FOG_EXT", GL_OUTPUT_FOG_EXT},
    {"GL_SCALAR_EXT", GL_SCALAR_EXT},
    {"GL_VECTOR_EXT", GL_VECTOR_EXT},
    {"GL_MATRIX_EXT", GL_MATRIX_EXT},
    {"GL_VARIANT_EXT", GL_VARIANT_EXT},
    {"GL_INVARIANT_EXT", GL_INVARIANT_EXT},
    {"GL_LOCAL_CONSTANT_EXT", GL_LOCAL_CONSTANT_EXT},
    {"GL_LOCAL_EXT", GL_LOCAL_EXT},
    {"GL_MAX_VERTEX_SHADER_INSTRUCTIONS_EXT", GL_MAX_VERTEX_SHADER_INSTRUCTIONS_EXT},
    {"GL_MAX_VERTEX_SHADER_VARIANTS_EXT", GL_MAX_VERTEX_SHADER_VARIANTS_EXT},
    {"GL_MAX_VERTEX_SHADER_INVARIANTS_EXT", GL_MAX_VERTEX_SHADER_INVARIANTS_EXT},
    {"GL_MAX_VERTEX_SHADER_LOCAL_CONSTANTS_EXT", GL_MAX_VERTEX_SHADER_LOCAL_CONSTANTS_EXT},
    {"GL_MAX_VERTEX_SHADER_LOCALS_EXT", GL_MAX_VERTEX_SHADER_LOCALS_EXT},
    {"GL_MAX_OPTIMIZED_VERTEX_SHADER_INSTRUCTIONS_EXT", GL_MAX_OPTIMIZED_VERTEX_SHADER_INSTRUCTIONS_EXT},
    {"GL_MAX_OPTIMIZED_VERTEX_SHADER_VARIANTS_EXT", GL_MAX_OPTIMIZED_VERTEX_SHADER_VARIANTS_EXT},
    {"GL_MAX_OPTIMIZED_VERTEX_SHADER_LOCAL_CONSTANTS_EXT", GL_MAX_OPTIMIZED_VERTEX_SHADER_LOCAL_CONSTANTS_EXT},
    {"GL_MAX_OPTIMIZED_VERTEX_SHADER_INVARIANTS_EXT", GL_MAX_OPTIMIZED_VERTEX_SHADER_INVARIANTS_EXT},
    {"GL_MAX_OPTIMIZED_VERTEX_SHADER_LOCALS_EXT", GL_MAX_OPTIMIZED_VERTEX_SHADER_LOCALS_EXT},
    {"GL_VERTEX_SHADER_INSTRUCTIONS_EXT", GL_VERTEX_SHADER_INSTRUCTIONS_EXT},
    {"GL_VERTEX_SHADER_VARIANTS_EXT", GL_VERTEX_SHADER_VARIANTS_EXT},
    {"GL_VERTEX_SHADER_INVARIANTS_EXT", GL_VERTEX_SHADER_INVARIANTS_EXT},
    {"GL_VERTEX_SHADER_LOCAL_CONSTANTS_EXT", GL_VERTEX_SHADER_LOCAL_CONSTANTS_EXT},
    {"GL_VERTEX_SHADER_LOCALS_EXT", GL_VERTEX_SHADER_LOCALS_EXT},
    {"GL_VERTEX_SHADER_OPTIMIZED_EXT", GL_VERTEX_SHADER_OPTIMIZED_EXT},
    {"GL_X_EXT", GL_X_EXT},
    {"GL_Y_EXT", GL_Y_EXT},
    {"GL_Z_EXT", GL_Z_EXT},
    {"GL_W_EXT", GL_W_EXT},
    {"GL_NEGATIVE_X_EXT", GL_NEGATIVE_X_EXT},
    {"GL_NEGATIVE_Y_EXT", GL_NEGATIVE_Y_EXT},
    {"GL_NEGATIVE_Z_EXT", GL_NEGATIVE_Z_EXT},
    {"GL_NEGATIVE_W_EXT", GL_NEGATIVE_W_EXT},
    {"GL_ZERO_EXT", GL_ZERO_EXT},
    {"GL_ONE_EXT", GL_ONE_EXT},
    {"GL_NEGATIVE_ONE_EXT", GL_NEGATIVE_ONE_EXT},
    {"GL_NORMALIZED_RANGE_EXT", GL_NORMALIZED_RANGE_EXT},
    {"GL_FULL_RANGE_EXT", GL_FULL_RANGE_EXT},
    {"GL_CURRENT_VERTEX_EXT", GL_CURRENT_VERTEX_EXT},
    {"GL_MVP_MATRIX_EXT", GL_MVP_MATRIX_EXT},
    {"GL_VARIANT_VALUE_EXT", GL_VARIANT_VALUE_EXT},
    {"GL_VARIANT_DATATYPE_EXT", GL_VARIANT_DATATYPE_EXT},
    {"GL_VARIANT_ARRAY_STRIDE_EXT", GL_VARIANT_ARRAY_STRIDE_EXT},
    {"GL_VARIANT_ARRAY_TYPE_EXT", GL_VARIANT_ARRAY_TYPE_EXT},
    {"GL_VARIANT_ARRAY_EXT", GL_VARIANT_ARRAY_EXT},
    {"GL_VARIANT_ARRAY_POINTER_EXT", GL_VARIANT_ARRAY_POINTER_EXT},
    {"GL_INVARIANT_VALUE_EXT", GL_INVARIANT_VALUE_EXT},
    {"GL_INVARIANT_DATATYPE_EXT", GL_INVARIANT_DATATYPE_EXT},
    {"GL_LOCAL_CONSTANT_VALUE_EXT", GL_LOCAL_CONSTANT_VALUE_EXT},
    {"GL_LOCAL_CONSTANT_DATATYPE_EXT", GL_LOCAL_CONSTANT_DATATYPE_EXT},
    {"GL_ATC_RGBA_INTERPOLATED_ALPHA_AMD", GL_ATC_RGBA_INTERPOLATED_ALPHA_AMD},
    {"GL_PN_TRIANGLES_ATI", GL_PN_TRIANGLES_ATI},
    {"GL_MAX_PN_TRIANGLES_TESSELATION_LEVEL_ATI", GL_MAX_PN_TRIANGLES_TESSELATION_LEVEL_ATI},
    {"GL_PN_TRIANGLES_POINT_MODE_ATI", GL_PN_TRIANGLES_POINT_MODE_ATI},
    {"GL_PN_TRIANGLES_NORMAL_MODE_ATI", GL_PN_TRIANGLES_NORMAL_MODE_ATI},
    {"GL_PN_TRIANGLES_TESSELATION_LEVEL_ATI", GL_PN_TRIANGLES_TESSELATION_LEVEL_ATI},
    {"GL_PN_TRIANGLES_POINT_MODE_LINEAR_ATI", GL_PN_TRIANGLES_POINT_MODE_LINEAR_ATI},
    {"GL_PN_TRIANGLES_POINT_MODE_CUBIC_ATI", GL_PN_TRIANGLES_POINT_MODE_CUBIC_ATI},
    {"GL_PN_TRIANGLES_NORMAL_MODE_LINEAR_ATI", GL_PN_TRIANGLES_NORMAL_MODE_LINEAR_ATI},
    {"GL_PN_TRIANGLES_NORMAL_MODE_QUADRATIC_ATI", GL_PN_TRIANGLES_NORMAL_MODE_QUADRATIC_ATI},
    {"GL_3DC_X_AMD", GL_3DC_X_AMD},
    {"GL_3DC_XY_AMD", GL_3DC_XY_AMD},
    {"GL_VBO_FREE_MEMORY_ATI", GL_VBO_FREE_MEMORY_ATI},
    {"GL_TEXTURE_FREE_MEMORY_ATI", GL_TEXTURE_FREE_MEMORY_ATI},
    {"GL_RENDERBUFFER_FREE_MEMORY_ATI", GL_RENDERBUFFER_FREE_MEMORY_ATI},
    {"GL_NUM_PROGRAM_BINARY_FORMATS", GL_NUM_PROGRAM_BINARY_FORMATS},
    {"GL_PROGRAM_BINARY_FORMATS", GL_PROGRAM_BINARY_FORMATS},
    {"GL_STENCIL_BACK_FUNC", GL_STENCIL_BACK_FUNC},
    {"GL_STENCIL_BACK_FAIL", GL_STENCIL_BACK_FAIL},
    {"GL_STENCIL_BACK_PASS_DEPTH_FAIL", GL_STENCIL_BACK_PASS_DEPTH_FAIL},
    {"GL_STENCIL_BACK_PASS_DEPTH_PASS", GL_STENCIL_BACK_PASS_DEPTH_PASS},
    {"GL_FRAGMENT_PROGRAM_ARB", GL_FRAGMENT_PROGRAM_ARB},
    {"GL_PROGRAM_ALU_INSTRUCTIONS_ARB", GL_PROGRAM_ALU_INSTRUCTIONS_ARB},
    {"GL_PROGRAM_TEX_INSTRUCTIONS_ARB", GL_PROGRAM_TEX_INSTRUCTIONS_ARB},
    {"GL_PROGRAM_TEX_INDIRECTIONS_ARB", GL_PROGRAM_TEX_INDIRECTIONS_ARB},
    {"GL_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB", GL_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB},
    {"GL_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB", GL_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB},
    {"GL_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB", GL_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB},
    {"GL_MAX_PROGRAM_ALU_INSTRUCTIONS_ARB", GL_MAX_PROGRAM_ALU_INSTRUCTIONS_ARB},
    {"GL_MAX_PROGRAM_TEX_INSTRUCTIONS_ARB", GL_MAX_PROGRAM_TEX_INSTRUCTIONS_ARB},
    {"GL_MAX_PROGRAM_TEX_INDIRECTIONS_ARB", GL_MAX_PROGRAM_TEX_INDIRECTIONS_ARB},
    {"GL_MAX_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB", GL_MAX_PROGRAM_NATIVE_ALU_INSTRUCTIONS_ARB},
    {"GL_MAX_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB", GL_MAX_PROGRAM_NATIVE_TEX_INSTRUCTIONS_ARB},
    {"GL_MAX_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB", GL_MAX_PROGRAM_NATIVE_TEX_INDIRECTIONS_ARB},
    {"GL_RGBA32F", GL_RGBA32F},
    {"GL_RGB32F", GL_RGB32F},
    {"GL_ALPHA32F_ARB", GL_ALPHA32F_ARB},
    {"GL_INTENSITY32F_ARB", GL_INTENSITY32F_ARB},
    {"GL_LUMINANCE32F_ARB", GL_LUMINANCE32F_ARB},
    {"GL_LUMINANCE_ALPHA32F_ARB", GL_LUMINANCE_ALPHA32F_ARB},
    {"GL_RGBA16F", GL_RGBA16F},
    {"GL_RGB16F", GL_RGB16F},
    {"GL_ALPHA16F_ARB", GL_ALPHA16F_ARB},
    {"GL_INTENSITY16F_ARB", GL_INTENSITY16F_ARB},
    {"GL_LUMINANCE16F_ARB", GL_LUMINANCE16F_ARB},
    {"GL_LUMINANCE_ALPHA16F_ARB", GL_LUMINANCE_ALPHA16F_ARB},
    {"GL_RGBA_FLOAT_MODE_ARB", GL_RGBA_FLOAT_MODE_ARB},
    {"GL_MAX_DRAW_BUFFERS", GL_MAX_DRAW_BUFFERS},
    {"GL_DRAW_BUFFER0", GL_DRAW_BUFFER0},
    {"GL_DRAW_BUFFER1", GL_DRAW_BUFFER1},
    {"GL_DRAW_BUFFER2", GL_DRAW_BUFFER2},
    {"GL_DRAW_BUFFER3", GL_DRAW_BUFFER3},
    {"GL_DRAW_BUFFER4", GL_DRAW_BUFFER4},
    {"GL_DRAW_BUFFER5", GL_DRAW_BUFFER5},
    {"GL_DRAW_BUFFER6", GL_DRAW_BUFFER6},
    {"GL_DRAW_BUFFER7", GL_DRAW_BUFFER7},
    {"GL_DRAW_BUFFER8", GL_DRAW_BUFFER8},
    {"GL_DRAW_BUFFER9", GL_DRAW_BUFFER9},
    {"GL_DRAW_BUFFER10", GL_DRAW_BUFFER10},
    {"GL_DRAW_BUFFER11", GL_DRAW_BUFFER11},
    {"GL_DRAW_BUFFER12", GL_DRAW_BUFFER12},
    {"GL_DRAW_BUFFER13", GL_DRAW_BUFFER13},
    {"GL_DRAW_BUFFER14", GL_DRAW_BUFFER14},
    {"GL_DRAW_BUFFER15", GL_DRAW_BUFFER15},
    {"GL_COLOR_CLEAR_UNCLAMPED_VALUE_ATI", GL_COLOR_CLEAR_UNCLAMPED_VALUE_ATI},
    {"GL_BLEND_EQUATION_ALPHA", GL_BLEND_EQUATION_ALPHA},
    {"GL_SUBSAMPLE_DISTANCE_AMD", GL_SUBSAMPLE_DISTANCE_AMD},
    {"GL_MATRIX_PALETTE_ARB", GL_MATRIX_PALETTE_ARB},
    {"GL_MAX_MATRIX_PALETTE_STACK_DEPTH_ARB", GL_MAX_MATRIX_PALETTE_STACK_DEPTH_ARB},
    {"GL_MAX_PALETTE_MATRICES_ARB", GL_MAX_PALETTE_MATRICES_ARB},
    {"GL_CURRENT_PALETTE_MATRIX_ARB", GL_CURRENT_PALETTE_MATRIX_ARB},
    {"GL_MATRIX_INDEX_ARRAY_ARB", GL_MATRIX_INDEX_ARRAY_ARB},
    {"GL_CURRENT_MATRIX_INDEX_ARB", GL_CURRENT_MATRIX_INDEX_ARB},
    {"GL_MATRIX_INDEX_ARRAY_SIZE_ARB", GL_MATRIX_INDEX_ARRAY_SIZE_ARB},
    {"GL_MATRIX_INDEX_ARRAY_TYPE_ARB", GL_MATRIX_INDEX_ARRAY_TYPE_ARB},
    {"GL_MATRIX_INDEX_ARRAY_STRIDE_ARB", GL_MATRIX_INDEX_ARRAY_STRIDE_ARB},
    {"GL_MATRIX_INDEX_ARRAY_POINTER_ARB", GL_MATRIX_INDEX_ARRAY_POINTER_ARB},
    {"GL_TEXTURE_DEPTH_SIZE", GL_TEXTURE_DEPTH_SIZE},
    {"GL_DEPTH_TEXTURE_MODE", GL_DEPTH_TEXTURE_MODE},
    {"GL_TEXTURE_COMPARE_MODE", GL_TEXTURE_COMPARE_MODE},
    {"GL_TEXTURE_COMPARE_FUNC", GL_TEXTURE_COMPARE_FUNC},
    {"GL_COMPARE_REF_TO_TEXTURE", GL_COMPARE_REF_TO_TEXTURE},
    {"GL_TEXTURE_CUBE_MAP_SEAMLESS", GL_TEXTURE_CUBE_MAP_SEAMLESS},
    {"GL_OFFSET_PROJECTIVE_TEXTURE_2D_NV", GL_OFFSET_PROJECTIVE_TEXTURE_2D_NV},
    {"GL_OFFSET_PROJECTIVE_TEXTURE_2D_SCALE_NV", GL_OFFSET_PROJECTIVE_TEXTURE_2D_SCALE_NV},
    {"GL_OFFSET_PROJECTIVE_TEXTURE_RECTANGLE_NV", GL_OFFSET_PROJECTIVE_TEXTURE_RECTANGLE_NV},
    {"GL_OFFSET_PROJECTIVE_TEXTURE_RECTANGLE_SCALE_NV", GL_OFFSET_PROJECTIVE_TEXTURE_RECTANGLE_SCALE_NV},
    {"GL_OFFSET_HILO_TEXTURE_2D_NV", GL_OFFSET_HILO_TEXTURE_2D_NV},
    {"GL_OFFSET_HILO_TEXTURE_RECTANGLE_NV", GL_OFFSET_HILO_TEXTURE_RECTANGLE_NV},
    {"GL_OFFSET_HILO_PROJECTIVE_TEXTURE_2D_NV", GL_OFFSET_HILO_PROJECTIVE_TEXTURE_2D_NV},
    {"GL_OFFSET_HILO_PROJECTIVE_TEXTURE_RECTANGLE_NV", GL_OFFSET_HILO_PROJECTIVE_TEXTURE_RECTANGLE_NV},
    {"GL_DEPENDENT_HILO_TEXTURE_2D_NV", GL_DEPENDENT_HILO_TEXTURE_2D_NV},
    {"GL_DEPENDENT_RGB_TEXTURE_3D_NV", GL_DEPENDENT_RGB_TEXTURE_3D_NV},
    {"GL_DEPENDENT_RGB_TEXTURE_CUBE_MAP_NV", GL_DEPENDENT_RGB_TEXTURE_CUBE_MAP_NV},
    {"GL_DOT_PRODUCT_PASS_THROUGH_NV", GL_DOT_PRODUCT_PASS_THROUGH_NV},
    {"GL_DOT_PRODUCT_TEXTURE_1D_NV", GL_DOT_PRODUCT_TEXTURE_1D_NV},
    {"GL_DOT_PRODUCT_AFFINE_DEPTH_REPLACE_NV", GL_DOT_PRODUCT_AFFINE_DEPTH_REPLACE_NV},
    {"GL_HILO8_NV", GL_HILO8_NV},
    {"GL_SIGNED_HILO8_NV", GL_SIGNED_HILO8_NV},
    {"GL_FORCE_BLUE_TO_ONE_NV", GL_FORCE_BLUE_TO_ONE_NV},
    {"GL_POINT_SPRITE", GL_POINT_SPRITE},
    {"GL_COORD_REPLACE", GL_COORD_REPLACE},
    {"GL_POINT_SPRITE_R_MODE_NV", GL_POINT_SPRITE_R_MODE_NV},
    {"GL_QUERY_COUNTER_BITS", GL_QUERY_COUNTER_BITS},
    {"GL_CURRENT_QUERY", GL_CURRENT_QUERY},
    {"GL_QUERY_RESULT", GL_QUERY_RESULT},
    {"GL_QUERY_RESULT_AVAILABLE", GL_QUERY_RESULT_AVAILABLE},
    {"GL_MAX_FRAGMENT_PROGRAM_LOCAL_PARAMETERS_NV", GL_MAX_FRAGMENT_PROGRAM_LOCAL_PARAMETERS_NV},
    {"GL_MAX_VERTEX_ATTRIBS", GL_MAX_VERTEX_ATTRIBS},
    {"GL_VERTEX_ATTRIB_ARRAY_NORMALIZED", GL_VERTEX_ATTRIB_ARRAY_NORMALIZED},
    {"GL_MAX_TESS_CONTROL_INPUT_COMPONENTS", GL_MAX_TESS_CONTROL_INPUT_COMPONENTS},
    {"GL_MAX_TESS_EVALUATION_INPUT_COMPONENTS", GL_MAX_TESS_EVALUATION_INPUT_COMPONENTS},
    {"GL_DEPTH_STENCIL_TO_RGBA_NV", GL_DEPTH_STENCIL_TO_RGBA_NV},
    {"GL_DEPTH_STENCIL_TO_BGRA_NV", GL_DEPTH_STENCIL_TO_BGRA_NV},
    {"GL_FRAGMENT_PROGRAM_NV", GL_FRAGMENT_PROGRAM_NV},
    {"GL_MAX_TEXTURE_COORDS", GL_MAX_TEXTURE_COORDS},
    {"GL_MAX_TEXTURE_IMAGE_UNITS", GL_MAX_TEXTURE_IMAGE_UNITS},
    {"GL_FRAGMENT_PROGRAM_BINDING_NV", GL_FRAGMENT_PROGRAM_BINDING_NV},
    {"GL_PROGRAM_ERROR_STRING_ARB", GL_PROGRAM_ERROR_STRING_ARB},
    {"GL_PROGRAM_FORMAT_ASCII_ARB", GL_PROGRAM_FORMAT_ASCII_ARB},
    {"GL_PROGRAM_FORMAT_ARB", GL_PROGRAM_FORMAT_ARB},
    {"GL_WRITE_PIXEL_DATA_RANGE_NV", GL_WRITE_PIXEL_DATA_RANGE_NV},
    {"GL_READ_PIXEL_DATA_RANGE_NV", GL_READ_PIXEL_DATA_RANGE_NV},
    {"GL_WRITE_PIXEL_DATA_RANGE_LENGTH_NV", GL_WRITE_PIXEL_DATA_RANGE_LENGTH_NV},
    {"GL_READ_PIXEL_DATA_RANGE_LENGTH_NV", GL_READ_PIXEL_DATA_RANGE_LENGTH_NV},
    {"GL_WRITE_PIXEL_DATA_RANGE_POINTER_NV", GL_WRITE_PIXEL_DATA_RANGE_POINTER_NV},
    {"GL_READ_PIXEL_DATA_RANGE_POINTER_NV", GL_READ_PIXEL_DATA_RANGE_POINTER_NV},
    {"GL_GEOMETRY_SHADER_INVOCATIONS", GL_GEOMETRY_SHADER_INVOCATIONS},
    {"GL_FLOAT_R_NV", GL_FLOAT_R_NV},
    {"GL_FLOAT_RG_NV", GL_FLOAT_RG_NV},
    {"GL_FLOAT_RGB_NV", GL_FLOAT_RGB_NV},
    {"GL_FLOAT_RGBA_NV", GL_FLOAT_RGBA_NV},
    {"GL_FLOAT_R16_NV", GL_FLOAT_R16_NV},
    {"GL_FLOAT_R32_NV", GL_FLOAT_R32_NV},
    {"GL_FLOAT_RG16_NV", GL_FLOAT_RG16_NV},
    {"GL_FLOAT_RG32_NV", GL_FLOAT_RG32_NV},
    {"GL_FLOAT_RGB16_NV", GL_FLOAT_RGB16_NV},
    {"GL_FLOAT_RGB32_NV", GL_FLOAT_RGB32_NV},
    {"GL_FLOAT_RGBA16_NV", GL_FLOAT_RGBA16_NV},
    {"GL_FLOAT_RGBA32_NV", GL_FLOAT_RGBA32_NV},
    {"GL_TEXTURE_FLOAT_COMPONENTS_NV", GL_TEXTURE_FLOAT_COMPONENTS_NV},
    {"GL_FLOAT_CLEAR_COLOR_VALUE_NV", GL_FLOAT_CLEAR_COLOR_VALUE_NV},
    {"GL_FLOAT_RGBA_MODE_NV", GL_FLOAT_RGBA_MODE_NV},
    {"GL_TEXTURE_UNSIGNED_REMAP_MODE_NV", GL_TEXTURE_UNSIGNED_REMAP_MODE_NV},
    {"GL_DEPTH_BOUNDS_TEST_EXT", GL_DEPTH_BOUNDS_TEST_EXT},
    {"GL_DEPTH_BOUNDS_EXT", GL_DEPTH_BOUNDS_EXT},
    {"GL_ARRAY_BUFFER", GL_ARRAY_BUFFER},
    {"GL_ELEMENT_ARRAY_BUFFER", GL_ELEMENT_ARRAY_BUFFER},
    {"GL_ARRAY_BUFFER_BINDING", GL_ARRAY_BUFFER_BINDING},
    {"GL_ELEMENT_ARRAY_BUFFER_BINDING", GL_ELEMENT_ARRAY_BUFFER_BINDING},
    {"GL_VERTEX_ARRAY_BUFFER_BINDING", GL_VERTEX_ARRAY_BUFFER_BINDING},
    {"GL_NORMAL_ARRAY_BUFFER_BINDING", GL_NORMAL_ARRAY_BUFFER_BINDING},
    {"GL_COLOR_ARRAY_BUFFER_BINDING", GL_COLOR_ARRAY_BUFFER_BINDING},
    {"GL_INDEX_ARRAY_BUFFER_BINDING", GL_INDEX_ARRAY_BUFFER_BINDING},
    {"GL_TEXTURE_COORD_ARRAY_BUFFER_BINDING", GL_TEXTURE_COORD_ARRAY_BUFFER_BINDING},
    {"GL_EDGE_FLAG_ARRAY_BUFFER_BINDING", GL_EDGE_FLAG_ARRAY_BUFFER_BINDING},
    {"GL_SECONDARY_COLOR_ARRAY_BUFFER_BINDING", GL_SECONDARY_COLOR_ARRAY_BUFFER_BINDING},
    {"GL_FOG_COORD_ARRAY_BUFFER_BINDING", GL_FOG_COORD_ARRAY_BUFFER_BINDING},
    {"GL_WEIGHT_ARRAY_BUFFER_BINDING", GL_WEIGHT_ARRAY_BUFFER_BINDING},
    {"GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING", GL_VERTEX_ATTRIB_ARRAY_BUFFER_BINDING},
    {"GL_PROGRAM_INSTRUCTIONS_ARB", GL_PROGRAM_INSTRUCTIONS_ARB},
    {"GL_MAX_PROGRAM_INSTRUCTIONS_ARB", GL_MAX_PROGRAM_INSTRUCTIONS_ARB},
    {"GL_PROGRAM_NATIVE_INSTRUCTIONS_ARB", GL_PROGRAM_NATIVE_INSTRUCTIONS_ARB},
    {"GL_MAX_PROGRAM_NATIVE_INSTRUCTIONS_ARB", GL_MAX_PROGRAM_NATIVE_INSTRUCTIONS_ARB},
    {"GL_PROGRAM_TEMPORARIES_ARB", GL_PROGRAM_TEMPORARIES_ARB},
    {"GL_MAX_PROGRAM_TEMPORARIES_ARB", GL_MAX_PROGRAM_TEMPORARIES_ARB},
    {"GL_PROGRAM_NATIVE_TEMPORARIES_ARB", GL_PROGRAM_NATIVE_TEMPORARIES_ARB},
    {"GL_MAX_PROGRAM_NATIVE_TEMPORARIES_ARB", GL_MAX_PROGRAM_NATIVE_TEMPORARIES_ARB},
    {"GL_PROGRAM_PARAMETERS_ARB", GL_PROGRAM_PARAMETERS_ARB},
    {"GL_MAX_PROGRAM_PARAMETERS_ARB", GL_MAX_PROGRAM_PARAMETERS_ARB},
    {"GL_PROGRAM_NATIVE_PARAMETERS_ARB", GL_PROGRAM_NATIVE_PARAMETERS_ARB},
    {"GL_MAX_PROGRAM_NATIVE_PARAMETERS_ARB", GL_MAX_PROGRAM_NATIVE_PARAMETERS_ARB},
    {"GL_PROGRAM_ATTRIBS_ARB", GL_PROGRAM_ATTRIBS_ARB},
    {"GL_MAX_PROGRAM_ATTRIBS_ARB", GL_MAX_PROGRAM_ATTRIBS_ARB},
    {"GL_PROGRAM_NATIVE_ATTRIBS_ARB", GL_PROGRAM_NATIVE_ATTRIBS_ARB},
    {"GL_MAX_PROGRAM_NATIVE_ATTRIBS_ARB", GL_MAX_PROGRAM_NATIVE_ATTRIBS_ARB},
    {"GL_PROGRAM_ADDRESS_REGISTERS_ARB", GL_PROGRAM_ADDRESS_REGISTERS_ARB},
    {"GL_MAX_PROGRAM_ADDRESS_REGISTERS_ARB", GL_MAX_PROGRAM_ADDRESS_REGISTERS_ARB},
    {"GL_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB", GL_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB},
    {"GL_MAX_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB", GL_MAX_PROGRAM_NATIVE_ADDRESS_REGISTERS_ARB},
    {"GL_MAX_PROGRAM_LOCAL_PARAMETERS_ARB", GL_MAX_PROGRAM_LOCAL_PARAMETERS_ARB},
    {"GL_MAX_PROGRAM_ENV_PARAMETERS_ARB", GL_MAX_PROGRAM_ENV_PARAMETERS_ARB},
    {"GL_PROGRAM_UNDER_NATIVE_LIMITS_ARB", GL_PROGRAM_UNDER_NATIVE_LIMITS_ARB},
    {"GL_TRANSPOSE_CURRENT_MATRIX_ARB", GL_TRANSPOSE_CURRENT_MATRIX_ARB},
    {"GL_READ_ONLY", GL_READ_ONLY},
    {"GL_WRITE_ONLY", GL_WRITE_ONLY},
    {"GL_READ_WRITE", GL_READ_WRITE},
    {"GL_BUFFER_ACCESS", GL_BUFFER_ACCESS},
    {"GL_BUFFER_MAPPED", GL_BUFFER_MAPPED},
    {"GL_BUFFER_MAP_POINTER", GL_BUFFER_MAP_POINTER},
    {"GL_WRITE_DISCARD_NV", GL_WRITE_DISCARD_NV},
    {"GL_TIME_ELAPSED", GL_TIME_ELAPSED},
    {"GL_MATRIX0_ARB", GL_MATRIX0_ARB},
    {"GL_MATRIX1_ARB", GL_MATRIX1_ARB},
    {"GL_MATRIX2_ARB", GL_MATRIX2_ARB},
    {"GL_MATRIX3_ARB", GL_MATRIX3_ARB},
    {"GL_MATRIX4_ARB", GL_MATRIX4_ARB},
    {"GL_MATRIX5_ARB", GL_MATRIX5_ARB},
    {"GL_MATRIX6_ARB", GL_MATRIX6_ARB},
    {"GL_MATRIX7_ARB", GL_MATRIX7_ARB},
    {"GL_MATRIX8_ARB", GL_MATRIX8_ARB},
    {"GL_MATRIX9_ARB", GL_MATRIX9_ARB},
    {"GL_MATRIX10_ARB", GL_MATRIX10_ARB},
    {"GL_MATRIX11_ARB", GL_MATRIX11_ARB},
    {"GL_MATRIX12_ARB", GL_MATRIX12_ARB},
    {"GL_MATRIX13_ARB", GL_MATRIX13_ARB},
    {"GL_MATRIX14_ARB", GL_MATRIX14_ARB},
    {"GL_MATRIX15_ARB", GL_MATRIX15_ARB},
    {"GL_MATRIX16_ARB", GL_MATRIX16_ARB},
    {"GL_MATRIX17_ARB", GL_MATRIX17_ARB},
    {"GL_MATRIX18_ARB", GL_MATRIX18_ARB},
    {"GL_MATRIX19_ARB", GL_MATRIX19_ARB},
    {"GL_MATRIX20_ARB", GL_MATRIX20_ARB},
    {"GL_MATRIX21_ARB", GL_MATRIX21_ARB},
    {"GL_MATRIX22_ARB", GL_MATRIX22_ARB},
    {"GL_MATRIX23_ARB", GL_MATRIX23_ARB},
    {"GL_MATRIX24_ARB", GL_MATRIX24_ARB},
    {"GL_MATRIX25_ARB", GL_MATRIX25_ARB},
    {"GL_MATRIX26_ARB", GL_MATRIX26_ARB},
    {"GL_MATRIX27_ARB", GL_MATRIX27_ARB},
    {"GL_MATRIX28_ARB", GL_MATRIX28_ARB},
    {"GL_MATRIX29_ARB", GL_MATRIX29_ARB},
    {"GL_MATRIX30_ARB", GL_MATRIX30_ARB},
    {"GL_MATRIX31_ARB", GL_MATRIX31_ARB},
    {"GL_STREAM_DRAW", GL_STREAM_DRAW},
    {"GL_STREAM_READ", GL_STREAM_READ},
    {"GL_STREAM_COPY", GL_STREAM_COPY},
    {"GL_STATIC_DRAW", GL_STATIC_DRAW},
    {"GL_STATIC_READ", GL_STATIC_READ},
    {"GL_STATIC_COPY", GL_STATIC_COPY},
    {"GL_DYNAMIC_DRAW", GL_DYNAMIC_DRAW},
    {"GL_DYNAMIC_READ", GL_DYNAMIC_READ},
    {"GL_DYNAMIC_COPY", GL_DYNAMIC_COPY},
    {"GL_PIXEL_PACK_BUFFER", GL_PIXEL_PACK_BUFFER},
    {"GL_PIXEL_UNPACK_BUFFER", GL_PIXEL_UNPACK_BUFFER},
    {"GL_PIXEL_PACK_BUFFER_BINDING", GL_PIXEL_PACK_BUFFER_BINDING},
    {"GL_ETC1_SRGB8_NV", GL_ETC1_SRGB8_NV},
    {"GL_PIXEL_UNPACK_BUFFER_BINDING", GL_PIXEL_UNPACK_BUFFER_BINDING},
    {"GL_DEPTH24_STENCIL8", GL_DEPTH24_STENCIL8},
    {"GL_TEXTURE_STENCIL_SIZE", GL_TEXTURE_STENCIL_SIZE},
    {"GL_STENCIL_TAG_BITS_EXT", GL_STENCIL_TAG_BITS_EXT},
    {"GL_STENCIL_CLEAR_TAG_VALUE_EXT", GL_STENCIL_CLEAR_TAG_VALUE_EXT},
    {"GL_MAX_PROGRAM_EXEC_INSTRUCTIONS_NV", GL_MAX_PROGRAM_EXEC_INSTRUCTIONS_NV},
    {"GL_MAX_PROGRAM_CALL_DEPTH_NV", GL_MAX_PROGRAM_CALL_DEPTH_NV},
    {"GL_MAX_PROGRAM_IF_DEPTH_NV", GL_MAX_PROGRAM_IF_DEPTH_NV},
    {"GL_MAX_PROGRAM_LOOP_DEPTH_NV", GL_MAX_PROGRAM_LOOP_DEPTH_NV},
    {"GL_MAX_PROGRAM_LOOP_COUNT_NV", GL_MAX_PROGRAM_LOOP_COUNT_NV},
    {"GL_SRC1_COLOR", GL_SRC1_COLOR},
    {"GL_ONE_MINUS_SRC1_COLOR", GL_ONE_MINUS_SRC1_COLOR},
    {"GL_ONE_MINUS_SRC1_ALPHA", GL_ONE_MINUS_SRC1_ALPHA},
    {"GL_MAX_DUAL_SOURCE_DRAW_BUFFERS", GL_MAX_DUAL_SOURCE_DRAW_BUFFERS},
    {"GL_VERTEX_ATTRIB_ARRAY_INTEGER", GL_VERTEX_ATTRIB_ARRAY_INTEGER},
    {"GL_VERTEX_ATTRIB_ARRAY_DIVISOR", GL_VERTEX_ATTRIB_ARRAY_DIVISOR},
    {"GL_MAX_ARRAY_TEXTURE_LAYERS", GL_MAX_ARRAY_TEXTURE_LAYERS},
    {"GL_MIN_PROGRAM_TEXEL_OFFSET", GL_MIN_PROGRAM_TEXEL_OFFSET},
    {"GL_MAX_PROGRAM_TEXEL_OFFSET", GL_MAX_PROGRAM_TEXEL_OFFSET},
    {"GL_PROGRAM_ATTRIB_COMPONENTS_NV", GL_PROGRAM_ATTRIB_COMPONENTS_NV},
    {"GL_PROGRAM_RESULT_COMPONENTS_NV", GL_PROGRAM_RESULT_COMPONENTS_NV},
    {"GL_MAX_PROGRAM_ATTRIB_COMPONENTS_NV", GL_MAX_PROGRAM_ATTRIB_COMPONENTS_NV},
    {"GL_MAX_PROGRAM_RESULT_COMPONENTS_NV", GL_MAX_PROGRAM_RESULT_COMPONENTS_NV},
    {"GL_STENCIL_TEST_TWO_SIDE_EXT", GL_STENCIL_TEST_TWO_SIDE_EXT},
    {"GL_ACTIVE_STENCIL_FACE_EXT", GL_ACTIVE_STENCIL_FACE_EXT},
    {"GL_MIRROR_CLAMP_TO_BORDER_EXT", GL_MIRROR_CLAMP_TO_BORDER_EXT},
    {"GL_SAMPLES_PASSED", GL_SAMPLES_PASSED},
    {"GL_GEOMETRY_VERTICES_OUT", GL_GEOMETRY_VERTICES_OUT},
    {"GL_GEOMETRY_INPUT_TYPE", GL_GEOMETRY_INPUT_TYPE},
    {"GL_GEOMETRY_OUTPUT_TYPE", GL_GEOMETRY_OUTPUT_TYPE},
    {"GL_SAMPLER_BINDING", GL_SAMPLER_BINDING},
    {"GL_CLAMP_VERTEX_COLOR", GL_CLAMP_VERTEX_COLOR},
    {"GL_CLAMP_FRAGMENT_COLOR", GL_CLAMP_FRAGMENT_COLOR},
    {"GL_CLAMP_READ_COLOR", GL_CLAMP_READ_COLOR},
    {"GL_FIXED_ONLY", GL_FIXED_ONLY},
    {"GL_TESS_CONTROL_PROGRAM_NV", GL_TESS_CONTROL_PROGRAM_NV},
    {"GL_TESS_EVALUATION_PROGRAM_NV", GL_TESS_EVALUATION_PROGRAM_NV},
    {"GL_FRAGMENT_SHADER_ATI", GL_FRAGMENT_SHADER_ATI},
    {"GL_REG_0_ATI", GL_REG_0_ATI},
    {"GL_REG_1_ATI", GL_REG_1_ATI},
    {"GL_REG_2_ATI", GL_REG_2_ATI},
    {"GL_REG_3_ATI", GL_REG_3_ATI},
    {"GL_REG_4_ATI", GL_REG_4_ATI},
    {"GL_REG_5_ATI", GL_REG_5_ATI},
    {"GL_REG_6_ATI", GL_REG_6_ATI},
    {"GL_REG_7_ATI", GL_REG_7_ATI},
    {"GL_REG_8_ATI", GL_REG_8_ATI},
    {"GL_REG_9_ATI", GL_REG_9_ATI},
    {"GL_REG_10_ATI", GL_REG_10_ATI},
    {"GL_REG_11_ATI", GL_REG_11_ATI},
    {"GL_REG_12_ATI", GL_REG_12_ATI},
    {"GL_REG_13_ATI", GL_REG_13_ATI},
    {"GL_REG_14_ATI", GL_REG_14_ATI},
    {"GL_REG_15_ATI", GL_REG_15_ATI},
    {"GL_REG_16_ATI", GL_REG_16_ATI},
    {"GL_REG_17_ATI", GL_REG_17_ATI},
    {"GL_REG_18_ATI", GL_REG_18_ATI},
    {"GL_REG_19_ATI", GL_REG_19_ATI},
    {"GL_REG_20_ATI", GL_REG_20_ATI},
    {"GL_REG_21_ATI", GL_REG_21_ATI},
    {"GL_REG_22_ATI", GL_REG_22_ATI},
    {"GL_REG_23_ATI", GL_REG_23_ATI},
    {"GL_REG_24_ATI", GL_REG_24_ATI},
    {"GL_REG_25_ATI", GL_REG_25_ATI},
    {"GL_REG_26_ATI", GL_REG_26_ATI},
    {"GL_REG_27_ATI", GL_REG_27_ATI},
    {"GL_REG_28_ATI", GL_REG_28_ATI},
    {"GL_REG_29_ATI", GL_REG_29_ATI},
    {"GL_REG_30_ATI", GL_REG_30_ATI},
    {"GL_REG_31_ATI", GL_REG_31_ATI},
    {"GL_CON_0_ATI", GL_CON_0_ATI},
    {"GL_CON_1_ATI", GL_CON_1_ATI},
    {"GL_CON_2_ATI", GL_CON_2_ATI},
    {"GL_CON_3_ATI", GL_CON_3_ATI},
    {"GL_CON_4_ATI", GL_CON_4_ATI},
    {"GL_CON_5_ATI", GL_CON_5_ATI},
    {"GL_CON_6_ATI", GL_CON_6_ATI},
    {"GL_CON_7_ATI", GL_CON_7_ATI},
    {"GL_CON_8_ATI", GL_CON_8_ATI},
    {"GL_CON_9_ATI", GL_CON_9_ATI},
    {"GL_CON_10_ATI", GL_CON_10_ATI},
    {"GL_CON_11_ATI", GL_CON_11_ATI},
    {"GL_CON_12_ATI", GL_CON_12_ATI},
    {"GL_CON_13_ATI", GL_CON_13_ATI},
    {"GL_CON_14_ATI", GL_CON_14_ATI},
    {"GL_CON_15_ATI", GL_CON_15_ATI},
    {"GL_CON_16_ATI", GL_CON_16_ATI},
    {"GL_CON_17_ATI", GL_CON_17_ATI},
    {"GL_CON_18_ATI", GL_CON_18_ATI},
    {"GL_CON_19_ATI", GL_CON_19_ATI},
    {"GL_CON_20_ATI", GL_CON_20_ATI},
    {"GL_CON_21_ATI", GL_CON_21_ATI},
    {"GL_CON_22_ATI", GL_CON_22_ATI},
    {"GL_CON_23_ATI", GL_CON_23_ATI},
    {"GL_CON_24_ATI", GL_CON_24_ATI},
    {"GL_CON_25_ATI", GL_CON_25_ATI},
    {"GL_CON_26_ATI", GL_CON_26_ATI},
    {"GL_CON_27_ATI", GL_CON_27_ATI},
    {"GL_CON_28_ATI", GL_CON_28_ATI},
    {"GL_CON_29_ATI", GL_CON_29_ATI},
    {"GL_CON_30_ATI", GL_CON_30_ATI},
    {"GL_CON_31_ATI", GL_CON_31_ATI},
    {"GL_MOV_ATI", GL_MOV_ATI},
    {"GL_ADD_ATI", GL_ADD_ATI},
    {"GL_MUL_ATI", GL_MUL_ATI},
    {"GL_SUB_ATI", GL_SUB_ATI},
    {"GL_DOT3_ATI", GL_DOT3_ATI},
    {"GL_DOT4_ATI", GL_DOT4_ATI},
    {"GL_MAD_ATI", GL_MAD_ATI},
    {"GL_LERP_ATI", GL_LERP_ATI},
    {"GL_CND_ATI", GL_CND_ATI},
    {"GL_CND0_ATI", GL_CND0_ATI},
    {"GL_DOT2_ADD_ATI", GL_DOT2_ADD_ATI},
    {"GL_SECONDARY_INTERPOLATOR_ATI", GL_SECONDARY_INTERPOLATOR_ATI},
    {"GL_NUM_FRAGMENT_REGISTERS_ATI", GL_NUM_FRAGMENT_REGISTERS_ATI},
    {"GL_NUM_FRAGMENT_CONSTANTS_ATI", GL_NUM_FRAGMENT_CONSTANTS_ATI},
    {"GL_NUM_PASSES_ATI", GL_NUM_PASSES_ATI},
    {"GL_NUM_INSTRUCTIONS_PER_PASS_ATI", GL_NUM_INSTRUCTIONS_PER_PASS_ATI},
    {"GL_NUM_INSTRUCTIONS_TOTAL_ATI", GL_NUM_INSTRUCTIONS_TOTAL_ATI},
    {"GL_NUM_INPUT_INTERPOLATOR_COMPONENTS_ATI", GL_NUM_INPUT_INTERPOLATOR_COMPONENTS_ATI},
    {"GL_NUM_LOOPBACK_COMPONENTS_ATI", GL_NUM_LOOPBACK_COMPONENTS_ATI},
    {"GL_COLOR_ALPHA_PAIRING_ATI", GL_COLOR_ALPHA_PAIRING_ATI},
    {"GL_SWIZZLE_STR_ATI", GL_SWIZZLE_STR_ATI},
    {"GL_SWIZZLE_STQ_ATI", GL_SWIZZLE_STQ_ATI},
    {"GL_SWIZZLE_STR_DR_ATI", GL_SWIZZLE_STR_DR_ATI},
    {"GL_SWIZZLE_STQ_DQ_ATI", GL_SWIZZLE_STQ_DQ_ATI},
    {"GL_SWIZZLE_STRQ_ATI", GL_SWIZZLE_STRQ_ATI},
    {"GL_SWIZZLE_STRQ_DQ_ATI", GL_SWIZZLE_STRQ_DQ_ATI},
    {"GL_INTERLACE_OML", GL_INTERLACE_OML},
    {"GL_INTERLACE_READ_OML", GL_INTERLACE_READ_OML},
    {"GL_FORMAT_SUBSAMPLE_24_24_OML", GL_FORMAT_SUBSAMPLE_24_24_OML},
    {"GL_FORMAT_SUBSAMPLE_244_244_OML", GL_FORMAT_SUBSAMPLE_244_244_OML},
    {"GL_PACK_RESAMPLE_OML", GL_PACK_RESAMPLE_OML},
    {"GL_UNPACK_RESAMPLE_OML", GL_UNPACK_RESAMPLE_OML},
    {"GL_RESAMPLE_REPLICATE_OML", GL_RESAMPLE_REPLICATE_OML},
    {"GL_RESAMPLE_ZERO_FILL_OML", GL_RESAMPLE_ZERO_FILL_OML},
    {"GL_RESAMPLE_AVERAGE_OML", GL_RESAMPLE_AVERAGE_OML},
    {"GL_RESAMPLE_DECIMATE_OML", GL_RESAMPLE_DECIMATE_OML},
    {"GL_POINT_SIZE_ARRAY_TYPE_OES", GL_POINT_SIZE_ARRAY_TYPE_OES},
    {"GL_POINT_SIZE_ARRAY_STRIDE_OES", GL_POINT_SIZE_ARRAY_STRIDE_OES},
    {"GL_POINT_SIZE_ARRAY_POINTER_OES", GL_POINT_SIZE_ARRAY_POINTER_OES},
    {"GL_MODELVIEW_MATRIX_FLOAT_AS_INT_BITS_OES", GL_MODELVIEW_MATRIX_FLOAT_AS_INT_BITS_OES},
    {"GL_PROJECTION_MATRIX_FLOAT_AS_INT_BITS_OES", GL_PROJECTION_MATRIX_FLOAT_AS_INT_BITS_OES},
    {"GL_TEXTURE_MATRIX_FLOAT_AS_INT_BITS_OES", GL_TEXTURE_MATRIX_FLOAT_AS_INT_BITS_OES},
    {"GL_VERTEX_ATTRIB_MAP1_APPLE", GL_VERTEX_ATTRIB_MAP1_APPLE},
    {"GL_VERTEX_ATTRIB_MAP2_APPLE", GL_VERTEX_ATTRIB_MAP2_APPLE},
    {"GL_VERTEX_ATTRIB_MAP1_SIZE_APPLE", GL_VERTEX_ATTRIB_MAP1_SIZE_APPLE},
    {"GL_VERTEX_ATTRIB_MAP1_COEFF_APPLE", GL_VERTEX_ATTRIB_MAP1_COEFF_APPLE},
    {"GL_VERTEX_ATTRIB_MAP1_ORDER_APPLE", GL_VERTEX_ATTRIB_MAP1_ORDER_APPLE},
    {"GL_VERTEX_ATTRIB_MAP1_DOMAIN_APPLE", GL_VERTEX_ATTRIB_MAP1_DOMAIN_APPLE},
    {"GL_VERTEX_ATTRIB_MAP2_SIZE_APPLE", GL_VERTEX_ATTRIB_MAP2_SIZE_APPLE},
    {"GL_VERTEX_ATTRIB_MAP2_COEFF_APPLE", GL_VERTEX_ATTRIB_MAP2_COEFF_APPLE},
    {"GL_VERTEX_ATTRIB_MAP2_ORDER_APPLE", GL_VERTEX_ATTRIB_MAP2_ORDER_APPLE},
    {"GL_VERTEX_ATTRIB_MAP2_DOMAIN_APPLE", GL_VERTEX_ATTRIB_MAP2_DOMAIN_APPLE},
    {"GL_DRAW_PIXELS_APPLE", GL_DRAW_PIXELS_APPLE},
    {"GL_FENCE_APPLE", GL_FENCE_APPLE},
    {"GL_ELEMENT_ARRAY_APPLE", GL_ELEMENT_ARRAY_APPLE},
    {"GL_ELEMENT_ARRAY_TYPE_APPLE", GL_ELEMENT_ARRAY_TYPE_APPLE},
    {"GL_ELEMENT_ARRAY_POINTER_APPLE", GL_ELEMENT_ARRAY_POINTER_APPLE},
    {"GL_COLOR_FLOAT_APPLE", GL_COLOR_FLOAT_APPLE},
    {"GL_UNIFORM_BUFFER", GL_UNIFORM_BUFFER},
    {"GL_BUFFER_SERIALIZED_MODIFY_APPLE", GL_BUFFER_SERIALIZED_MODIFY_APPLE},
    {"GL_BUFFER_FLUSHING_UNMAP_APPLE", GL_BUFFER_FLUSHING_UNMAP_APPLE},
    {"GL_AUX_DEPTH_STENCIL_APPLE", GL_AUX_DEPTH_STENCIL_APPLE},
    {"GL_PACK_ROW_BYTES_APPLE", GL_PACK_ROW_BYTES_APPLE},
    {"GL_UNPACK_ROW_BYTES_APPLE", GL_UNPACK_ROW_BYTES_APPLE},
    {"GL_RELEASED_APPLE", GL_RELEASED_APPLE},
    {"GL_VOLATILE_APPLE", GL_VOLATILE_APPLE},
    {"GL_RETAINED_APPLE", GL_RETAINED_APPLE},
    {"GL_UNDEFINED_APPLE", GL_UNDEFINED_APPLE},
    {"GL_PURGEABLE_APPLE", GL_PURGEABLE_APPLE},
    {"GL_RGB_422_APPLE", GL_RGB_422_APPLE},
    {"GL_UNIFORM_BUFFER_BINDING", GL_UNIFORM_BUFFER_BINDING},
    {"GL_UNIFORM_BUFFER_START", GL_UNIFORM_BUFFER_START},
    {"GL_UNIFORM_BUFFER_SIZE", GL_UNIFORM_BUFFER_SIZE},
    {"GL_MAX_VERTEX_UNIFORM_BLOCKS", GL_MAX_VERTEX_UNIFORM_BLOCKS},
    {"GL_MAX_GEOMETRY_UNIFORM_BLOCKS", GL_MAX_GEOMETRY_UNIFORM_BLOCKS},
    {"GL_MAX_FRAGMENT_UNIFORM_BLOCKS", GL_MAX_FRAGMENT_UNIFORM_BLOCKS},
    {"GL_MAX_COMBINED_UNIFORM_BLOCKS", GL_MAX_COMBINED_UNIFORM_BLOCKS},
    {"GL_MAX_UNIFORM_BUFFER_BINDINGS", GL_MAX_UNIFORM_BUFFER_BINDINGS},
    {"GL_MAX_UNIFORM_BLOCK_SIZE", GL_MAX_UNIFORM_BLOCK_SIZE},
    {"GL_MAX_COMBINED_VERTEX_UNIFORM_COMPONENTS", GL_MAX_COMBINED_VERTEX_UNIFORM_COMPONENTS},
    {"GL_MAX_COMBINED_GEOMETRY_UNIFORM_COMPONENTS", GL_MAX_COMBINED_GEOMETRY_UNIFORM_COMPONENTS},
    {"GL_MAX_COMBINED_FRAGMENT_UNIFORM_COMPONENTS", GL_MAX_COMBINED_FRAGMENT_UNIFORM_COMPONENTS},
    {"GL_UNIFORM_BUFFER_OFFSET_ALIGNMENT", GL_UNIFORM_BUFFER_OFFSET_ALIGNMENT},
    {"GL_ACTIVE_UNIFORM_BLOCK_MAX_NAME_LENGTH", GL_ACTIVE_UNIFORM_BLOCK_MAX_NAME_LENGTH},
    {"GL_ACTIVE_UNIFORM_BLOCKS", GL_ACTIVE_UNIFORM_BLOCKS},
    {"GL_UNIFORM_TYPE", GL_UNIFORM_TYPE},
    {"GL_UNIFORM_SIZE", GL_UNIFORM_SIZE},
    {"GL_UNIFORM_NAME_LENGTH", GL_UNIFORM_NAME_LENGTH},
    {"GL_UNIFORM_BLOCK_INDEX", GL_UNIFORM_BLOCK_INDEX},
    {"GL_UNIFORM_OFFSET", GL_UNIFORM_OFFSET},
    {"GL_UNIFORM_ARRAY_STRIDE", GL_UNIFORM_ARRAY_STRIDE},
    {"GL_UNIFORM_MATRIX_STRIDE", GL_UNIFORM_MATRIX_STRIDE},
    {"GL_UNIFORM_IS_ROW_MAJOR", GL_UNIFORM_IS_ROW_MAJOR},
    {"GL_UNIFORM_BLOCK_BINDING", GL_UNIFORM_BLOCK_BINDING},
    {"GL_UNIFORM_BLOCK_DATA_SIZE", GL_UNIFORM_BLOCK_DATA_SIZE},
    {"GL_UNIFORM_BLOCK_NAME_LENGTH", GL_UNIFORM_BLOCK_NAME_LENGTH},
    {"GL_UNIFORM_BLOCK_ACTIVE_UNIFORMS", GL_UNIFORM_BLOCK_ACTIVE_UNIFORMS},
    {"GL_UNIFORM_BLOCK_ACTIVE_UNIFORM_INDICES", GL_UNIFORM_BLOCK_ACTIVE_UNIFORM_INDICES},
    {"GL_UNIFORM_BLOCK_REFERENCED_BY_VERTEX_SHADER", GL_UNIFORM_BLOCK_REFERENCED_BY_VERTEX_SHADER},
    {"GL_UNIFORM_BLOCK_REFERENCED_BY_GEOMETRY_SHADER", GL_UNIFORM_BLOCK_REFERENCED_BY_GEOMETRY_SHADER},
    {"GL_UNIFORM_BLOCK_REFERENCED_BY_FRAGMENT_SHADER", GL_UNIFORM_BLOCK_REFERENCED_BY_FRAGMENT_SHADER},
    {"GL_TEXTURE_SRGB_DECODE_EXT", GL_TEXTURE_SRGB_DECODE_EXT},
    {"GL_DECODE_EXT", GL_DECODE_EXT},
    {"GL_SKIP_DECODE_EXT", GL_SKIP_DECODE_EXT},
    {"GL_PROGRAM_PIPELINE_OBJECT_EXT", GL_PROGRAM_PIPELINE_OBJECT_EXT},
    {"GL_RGB_RAW_422_APPLE", GL_RGB_RAW_422_APPLE},
    {"GL_FRAGMENT_SHADER_DISCARDS_SAMPLES_EXT", GL_FRAGMENT_SHADER_DISCARDS_SAMPLES_EXT},
    {"GL_SYNC_OBJECT_APPLE", GL_SYNC_OBJECT_APPLE},
    {"GL_COMPRESSED_SRGB_PVRTC_2BPPV1_EXT", GL_COMPRESSED_SRGB_PVRTC_2BPPV1_EXT},
    {"GL_COMPRESSED_SRGB_PVRTC_4BPPV1_EXT", GL_COMPRESSED_SRGB_PVRTC_4BPPV1_EXT},
    {"GL_COMPRESSED_SRGB_ALPHA_PVRTC_2BPPV1_EXT", GL_COMPRESSED_SRGB_ALPHA_PVRTC_2BPPV1_EXT},
    {"GL_COMPRESSED_SRGB_ALPHA_PVRTC_4BPPV1_EXT", GL_COMPRESSED_SRGB_ALPHA_PVRTC_4BPPV1_EXT},
    {"GL_FRAGMENT_SHADER", GL_FRAGMENT_SHADER},
    {"GL_VERTEX_SHADER", GL_VERTEX_SHADER},
    {"GL_PROGRAM_OBJECT_ARB", GL_PROGRAM_OBJECT_ARB},
    {"GL_SHADER_OBJECT_ARB", GL_SHADER_OBJECT_ARB},
    {"GL_MAX_FRAGMENT_UNIFORM_COMPONENTS", GL_MAX_FRAGMENT_UNIFORM_COMPONENTS},
    {"GL_MAX_VERTEX_UNIFORM_COMPONENTS", GL_MAX_VERTEX_UNIFORM_COMPONENTS},
    {"GL_MAX_VARYING_COMPONENTS", GL_MAX_VARYING_COMPONENTS},
    {"GL_MAX_VERTEX_TEXTURE_IMAGE_UNITS", GL_MAX_VERTEX_TEXTURE_IMAGE_UNITS},
    {"GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS", GL_MAX_COMBINED_TEXTURE_IMAGE_UNITS},
    {"GL_OBJECT_TYPE_ARB", GL_OBJECT_TYPE_ARB},
    {"GL_SHADER_TYPE", GL_SHADER_TYPE},
    {"GL_FLOAT_VEC2", GL_FLOAT_VEC2},
    {"GL_FLOAT_VEC3", GL_FLOAT_VEC3},
    {"GL_FLOAT_VEC4", GL_FLOAT_VEC4},
    {"GL_INT_VEC2", GL_INT_VEC2},
    {"GL_INT_VEC3", GL_INT_VEC3},
    {"GL_INT_VEC4", GL_INT_VEC4},
    {"GL_BOOL", GL_BOOL},
    {"GL_BOOL_VEC2", GL_BOOL_VEC2},
    {"GL_BOOL_VEC3", GL_BOOL_VEC3},
    {"GL_BOOL_VEC4", GL_BOOL_VEC4},
    {"GL_FLOAT_MAT2", GL_FLOAT_MAT2},
    {"GL_FLOAT_MAT3", GL_FLOAT_MAT3},
    {"GL_FLOAT_MAT4", GL_FLOAT_MAT4},
    {"GL_SAMPLER_1D", GL_SAMPLER_1D},
    {"GL_SAMPLER_2D", GL_SAMPLER_2D},
    {"GL_SAMPLER_3D", GL_SAMPLER_3D},
    {"GL_SAMPLER_CUBE", GL_SAMPLER_CUBE},
    {"GL_SAMPLER_1D_SHADOW", GL_SAMPLER_1D_SHADOW},
    {"GL_SAMPLER_2D_SHADOW", GL_SAMPLER_2D_SHADOW},
    {"GL_SAMPLER_2D_RECT", GL_SAMPLER_2D_RECT},
    {"GL_SAMPLER_2D_RECT_SHADOW", GL_SAMPLER_2D_RECT_SHADOW},
    {"GL_FLOAT_MAT2x3", GL_FLOAT_MAT2x3},
    {"GL_FLOAT_MAT2x4", GL_FLOAT_MAT2x4},
    {"GL_FLOAT_MAT3x2", GL_FLOAT_MAT3x2},
    {"GL_FLOAT_MAT3x4", GL_FLOAT_MAT3x4},
    {"GL_FLOAT_MAT4x2", GL_FLOAT_MAT4x2},
    {"GL_FLOAT_MAT4x3", GL_FLOAT_MAT4x3},
    {"GL_DELETE_STATUS", GL_DELETE_STATUS},
    {"GL_COMPILE_STATUS", GL_COMPILE_STATUS},
    {"GL_LINK_STATUS", GL_LINK_STATUS},
    {"GL_VALIDATE_STATUS", GL_VALIDATE_STATUS},
    {"GL_INFO_LOG_LENGTH", GL_INFO_LOG_LENGTH},
    {"GL_ATTACHED_SHADERS", GL_ATTACHED_SHADERS},
    {"GL_ACTIVE_UNIFORMS", GL_ACTIVE_UNIFORMS},
    {"GL_ACTIVE_UNIFORM_MAX_LENGTH", GL_ACTIVE_UNIFORM_MAX_LENGTH},
    {"GL_SHADER_SOURCE_LENGTH", GL_SHADER_SOURCE_LENGTH},
    {"GL_ACTIVE_ATTRIBUTES", GL_ACTIVE_ATTRIBUTES},
    {"GL_ACTIVE_ATTRIBUTE_MAX_LENGTH", GL_ACTIVE_ATTRIBUTE_MAX_LENGTH},
    {"GL_FRAGMENT_SHADER_DERIVATIVE_HINT", GL_FRAGMENT_SHADER_DERIVATIVE_HINT},
    {"GL_SHADING_LANGUAGE_VERSION", GL_SHADING_LANGUAGE_VERSION},
    {"GL_CURRENT_PROGRAM", GL_CURRENT_PROGRAM},
    {"GL_PALETTE4_RGB8_OES", GL_PALETTE4_RGB8_OES},
    {"GL_PALETTE4_RGBA8_OES", GL_PALETTE4_RGBA8_OES},
    {"GL_PALETTE4_R5_G6_B5_OES", GL_PALETTE4_R5_G6_B5_OES},
    {"GL_PALETTE4_RGBA4_OES", GL_PALETTE4_RGBA4_OES},
    {"GL_PALETTE4_RGB5_A1_OES", GL_PALETTE4_RGB5_A1_OES},
    {"GL_PALETTE8_RGB8_OES", GL_PALETTE8_RGB8_OES},
    {"GL_PALETTE8_RGBA8_OES", GL_PALETTE8_RGBA8_OES},
    {"GL_PALETTE8_R5_G6_B5_OES", GL_PALETTE8_R5_G6_B5_OES},
    {"GL_PALETTE8_RGBA4_OES", GL_PALETTE8_RGBA4_OES},
    {"GL_PALETTE8_RGB5_A1_OES", GL_PALETTE8_RGB5_A1_OES},
    {"GL_IMPLEMENTATION_COLOR_READ_TYPE", GL_IMPLEMENTATION_COLOR_READ_TYPE},
    {"GL_IMPLEMENTATION_COLOR_READ_FORMAT", GL_IMPLEMENTATION_COLOR_READ_FORMAT},
    {"GL_POINT_SIZE_ARRAY_OES", GL_POINT_SIZE_ARRAY_OES},
    {"GL_TEXTURE_CROP_RECT_OES", GL_TEXTURE_CROP_RECT_OES},
    {"GL_MATRIX_INDEX_ARRAY_BUFFER_BINDING_OES", GL_MATRIX_INDEX_ARRAY_BUFFER_BINDING_OES},
    {"GL_POINT_SIZE_ARRAY_BUFFER_BINDING_OES", GL_POINT_SIZE_ARRAY_BUFFER_BINDING_OES},
    {"GL_COUNTER_TYPE_AMD", GL_COUNTER_TYPE_AMD},
    {"GL_COUNTER_RANGE_AMD", GL_COUNTER_RANGE_AMD},
    {"GL_UNSIGNED_INT64_AMD", GL_UNSIGNED_INT64_AMD},
    {"GL_PERCENTAGE_AMD", GL_PERCENTAGE_AMD},
    {"GL_PERFMON_RESULT_AVAILABLE_AMD", GL_PERFMON_RESULT_AVAILABLE_AMD},
    {"GL_PERFMON_RESULT_SIZE_AMD", GL_PERFMON_RESULT_SIZE_AMD},
    {"GL_PERFMON_RESULT_AMD", GL_PERFMON_RESULT_AMD},
    {"GL_SAMPLER_EXTERNAL_2D_Y2Y_EXT", GL_SAMPLER_EXTERNAL_2D_Y2Y_EXT},
    {"GL_TEXTURE_RED_TYPE", GL_TEXTURE_RED_TYPE},
    {"GL_TEXTURE_GREEN_TYPE", GL_TEXTURE_GREEN_TYPE},
    {"GL_TEXTURE_BLUE_TYPE", GL_TEXTURE_BLUE_TYPE},
    {"GL_TEXTURE_ALPHA_TYPE", GL_TEXTURE_ALPHA_TYPE},
    {"GL_TEXTURE_LUMINANCE_TYPE", GL_TEXTURE_LUMINANCE_TYPE},
    {"GL_TEXTURE_INTENSITY_TYPE", GL_TEXTURE_INTENSITY_TYPE},
    {"GL_TEXTURE_DEPTH_TYPE", GL_TEXTURE_DEPTH_TYPE},
    {"GL_UNSIGNED_NORMALIZED", GL_UNSIGNED_NORMALIZED},
    {"GL_TEXTURE_1D_ARRAY", GL_TEXTURE_1D_ARRAY},
    {"GL_PROXY_TEXTURE_1D_ARRAY", GL_PROXY_TEXTURE_1D_ARRAY},
    {"GL_TEXTURE_2D_ARRAY", GL_TEXTURE_2D_ARRAY},
    {"GL_PROXY_TEXTURE_2D_ARRAY", GL_PROXY_TEXTURE_2D_ARRAY},
    {"GL_TEXTURE_BINDING_1D_ARRAY", GL_TEXTURE_BINDING_1D_ARRAY},
    {"GL_TEXTURE_BINDING_2D_ARRAY", GL_TEXTURE_BINDING_2D_ARRAY},
    {"GL_GEOMETRY_PROGRAM_NV", GL_GEOMETRY_PROGRAM_NV},
    {"GL_MAX_PROGRAM_OUTPUT_VERTICES_NV", GL_MAX_PROGRAM_OUTPUT_VERTICES_NV},
    {"GL_MAX_PROGRAM_TOTAL_OUTPUT_COMPONENTS_NV", GL_MAX_PROGRAM_TOTAL_OUTPUT_COMPONENTS_NV},
    {"GL_MAX_GEOMETRY_TEXTURE_IMAGE_UNITS", GL_MAX_GEOMETRY_TEXTURE_IMAGE_UNITS},
    {"GL_TEXTURE_BUFFER", GL_TEXTURE_BUFFER},
    {"GL_MAX_TEXTURE_BUFFER_SIZE", GL_MAX_TEXTURE_BUFFER_SIZE},
    {"GL_TEXTURE_BINDING_BUFFER", GL_TEXTURE_BINDING_BUFFER},
    {"GL_TEXTURE_BUFFER_DATA_STORE_BINDING", GL_TEXTURE_BUFFER_DATA_STORE_BINDING},
    {"GL_TEXTURE_BUFFER_FORMAT_ARB", GL_TEXTURE_BUFFER_FORMAT_ARB},
    {"GL_ANY_SAMPLES_PASSED", GL_ANY_SAMPLES_PASSED},
    {"GL_SAMPLE_SHADING", GL_SAMPLE_SHADING},
    {"GL_MIN_SAMPLE_SHADING_VALUE", GL_MIN_SAMPLE_SHADING_VALUE},
    {"GL_R11F_G11F_B10F", GL_R11F_G11F_B10F},
    {"GL_UNSIGNED_INT_10F_11F_11F_REV", GL_UNSIGNED_INT_10F_11F_11F_REV},
    {"GL_RGBA_SIGNED_COMPONENTS_EXT", GL_RGBA_SIGNED_COMPONENTS_EXT},
    {"GL_RGB9_E5", GL_RGB9_E5},
    {"GL_UNSIGNED_INT_5_9_9_9_REV", GL_UNSIGNED_INT_5_9_9_9_REV},
    {"GL_TEXTURE_SHARED_SIZE", GL_TEXTURE_SHARED_SIZE},
    {"GL_SRGB", GL_SRGB},
    {"GL_SRGB8", GL_SRGB8},
    {"GL_SRGB_ALPHA", GL_SRGB_ALPHA},
    {"GL_SRGB8_ALPHA8", GL_SRGB8_ALPHA8},
    {"GL_SLUMINANCE_ALPHA", GL_SLUMINANCE_ALPHA},
    {"GL_SLUMINANCE8_ALPHA8", GL_SLUMINANCE8_ALPHA8},
    {"GL_SLUMINANCE", GL_SLUMINANCE},
    {"GL_SLUMINANCE8", GL_SLUMINANCE8},
    {"GL_COMPRESSED_SRGB", GL_COMPRESSED_SRGB},
    {"GL_COMPRESSED_SRGB_ALPHA", GL_COMPRESSED_SRGB_ALPHA},
    {"GL_COMPRESSED_SLUMINANCE", GL_COMPRESSED_SLUMINANCE},
    {"GL_COMPRESSED_SLUMINANCE_ALPHA", GL_COMPRESSED_SLUMINANCE_ALPHA},
    {"GL_COMPRESSED_SRGB_S3TC_DXT1_EXT", GL_COMPRESSED_SRGB_S3TC_DXT1_EXT},
    {"GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT", GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT1_EXT},
    {"GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT", GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT3_EXT},
    {"GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT", GL_COMPRESSED_SRGB_ALPHA_S3TC_DXT5_EXT},
    {"GL_COMPRESSED_LUMINANCE_LATC1_EXT", GL_COMPRESSED_LUMINANCE_LATC1_EXT},
    {"GL_COMPRESSED_SIGNED_LUMINANCE_LATC1_EXT", GL_COMPRESSED_SIGNED_LUMINANCE_LATC1_EXT},
    {"GL_COMPRESSED_LUMINANCE_ALPHA_LATC2_EXT", GL_COMPRESSED_LUMINANCE_ALPHA_LATC2_EXT},
    {"GL_COMPRESSED_SIGNED_LUMINANCE_ALPHA_LATC2_EXT", GL_COMPRESSED_SIGNED_LUMINANCE_ALPHA_LATC2_EXT},
    {"GL_TESS_CONTROL_PROGRAM_PARAMETER_BUFFER_NV", GL_TESS_CONTROL_PROGRAM_PARAMETER_BUFFER_NV},
    {"GL_TESS_EVALUATION_PROGRAM_PARAMETER_BUFFER_NV", GL_TESS_EVALUATION_PROGRAM_PARAMETER_BUFFER_NV},
    {"GL_TRANSFORM_FEEDBACK_VARYING_MAX_LENGTH", GL_TRANSFORM_FEEDBACK_VARYING_MAX_LENGTH},
    {"GL_BACK_PRIMARY_COLOR_NV", GL_BACK_PRIMARY_COLOR_NV},
    {"GL_BACK_SECONDARY_COLOR_NV", GL_BACK_SECONDARY_COLOR_NV},
    {"GL_TEXTURE_COORD_NV", GL_TEXTURE_COORD_NV},
    {"GL_CLIP_DISTANCE_NV", GL_CLIP_DISTANCE_NV},
    {"GL_VERTEX_ID_NV", GL_VERTEX_ID_NV},
    {"GL_PRIMITIVE_ID_NV", GL_PRIMITIVE_ID_NV},
    {"GL_GENERIC_ATTRIB_NV", GL_GENERIC_ATTRIB_NV},
    {"GL_TRANSFORM_FEEDBACK_ATTRIBS_NV", GL_TRANSFORM_FEEDBACK_ATTRIBS_NV},
    {"GL_TRANSFORM_FEEDBACK_BUFFER_MODE", GL_TRANSFORM_FEEDBACK_BUFFER_MODE},
    {"GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_COMPONENTS", GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_COMPONENTS},
    {"GL_ACTIVE_VARYINGS_NV", GL_ACTIVE_VARYINGS_NV},
    {"GL_ACTIVE_VARYING_MAX_LENGTH_NV", GL_ACTIVE_VARYING_MAX_LENGTH_NV},
    {"GL_TRANSFORM_FEEDBACK_VARYINGS", GL_TRANSFORM_FEEDBACK_VARYINGS},
    {"GL_TRANSFORM_FEEDBACK_BUFFER_START", GL_TRANSFORM_FEEDBACK_BUFFER_START},
    {"GL_TRANSFORM_FEEDBACK_BUFFER_SIZE", GL_TRANSFORM_FEEDBACK_BUFFER_SIZE},
    {"GL_TRANSFORM_FEEDBACK_RECORD_NV", GL_TRANSFORM_FEEDBACK_RECORD_NV},
    {"GL_PRIMITIVES_GENERATED", GL_PRIMITIVES_GENERATED},
    {"GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN", GL_TRANSFORM_FEEDBACK_PRIMITIVES_WRITTEN},
    {"GL_RASTERIZER_DISCARD", GL_RASTERIZER_DISCARD},
    {"GL_MAX_TRANSFORM_FEEDBACK_INTERLEAVED_COMPONENTS", GL_MAX_TRANSFORM_FEEDBACK_INTERLEAVED_COMPONENTS},
    {"GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_ATTRIBS", GL_MAX_TRANSFORM_FEEDBACK_SEPARATE_ATTRIBS},
    {"GL_INTERLEAVED_ATTRIBS", GL_INTERLEAVED_ATTRIBS},
    {"GL_SEPARATE_ATTRIBS", GL_SEPARATE_ATTRIBS},
    {"GL_TRANSFORM_FEEDBACK_BUFFER", GL_TRANSFORM_FEEDBACK_BUFFER},
    {"GL_TRANSFORM_FEEDBACK_BUFFER_BINDING", GL_TRANSFORM_FEEDBACK_BUFFER_BINDING},
    {"GL_ATC_RGB_AMD", GL_ATC_RGB_AMD},
    {"GL_ATC_RGBA_EXPLICIT_ALPHA_AMD", GL_ATC_RGBA_EXPLICIT_ALPHA_AMD},
    {"GL_POINT_SPRITE_COORD_ORIGIN", GL_POINT_SPRITE_COORD_ORIGIN},
    {"GL_LOWER_LEFT", GL_LOWER_LEFT},
    {"GL_UPPER_LEFT", GL_UPPER_LEFT},
    {"GL_STENCIL_BACK_REF", GL_STENCIL_BACK_REF},
    {"GL_STENCIL_BACK_VALUE_MASK", GL_STENCIL_BACK_VALUE_MASK},
    {"GL_STENCIL_BACK_WRITEMASK", GL_STENCIL_BACK_WRITEMASK},
    {"GL_DRAW_FRAMEBUFFER_BINDING", GL_DRAW_FRAMEBUFFER_BINDING},
    {"GL_RENDERBUFFER_BINDING", GL_RENDERBUFFER_BINDING},
    {"GL_READ_FRAMEBUFFER", GL_READ_FRAMEBUFFER},
    {"GL_DRAW_FRAMEBUFFER", GL_DRAW_FRAMEBUFFER},
    {"GL_READ_FRAMEBUFFER_BINDING", GL_READ_FRAMEBUFFER_BINDING},
    {"GL_RENDERBUFFER_SAMPLES", GL_RENDERBUFFER_SAMPLES},
    {"GL_DEPTH_COMPONENT32F", GL_DEPTH_COMPONENT32F},
    {"GL_DEPTH32F_STENCIL8", GL_DEPTH32F_STENCIL8},
    {"GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE", GL_FRAMEBUFFER_ATTACHMENT_OBJECT_TYPE},
    {"GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME", GL_FRAMEBUFFER_ATTACHMENT_OBJECT_NAME},
    {"GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL", GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LEVEL},
    {"GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE", GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_CUBE_MAP_FACE},
    {"GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER", GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_LAYER},
    {"GL_FRAMEBUFFER_COMPLETE", GL_FRAMEBUFFER_COMPLETE},
    {"GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT", GL_FRAMEBUFFER_INCOMPLETE_ATTACHMENT},
    {"GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT", GL_FRAMEBUFFER_INCOMPLETE_MISSING_ATTACHMENT},
    {"GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT", GL_FRAMEBUFFER_INCOMPLETE_DIMENSIONS_EXT},
    {"GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT", GL_FRAMEBUFFER_INCOMPLETE_FORMATS_EXT},
    {"GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER", GL_FRAMEBUFFER_INCOMPLETE_DRAW_BUFFER},
    {"GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER", GL_FRAMEBUFFER_INCOMPLETE_READ_BUFFER},
    {"GL_FRAMEBUFFER_UNSUPPORTED", GL_FRAMEBUFFER_UNSUPPORTED},
    {"GL_MAX_COLOR_ATTACHMENTS", GL_MAX_COLOR_ATTACHMENTS},
    {"GL_COLOR_ATTACHMENT0", GL_COLOR_ATTACHMENT0},
    {"GL_COLOR_ATTACHMENT1", GL_COLOR_ATTACHMENT1},
    {"GL_COLOR_ATTACHMENT2", GL_COLOR_ATTACHMENT2},
    {"GL_COLOR_ATTACHMENT3", GL_COLOR_ATTACHMENT3},
    {"GL_COLOR_ATTACHMENT4", GL_COLOR_ATTACHMENT4},
    {"GL_COLOR_ATTACHMENT5", GL_COLOR_ATTACHMENT5},
    {"GL_COLOR_ATTACHMENT6", GL_COLOR_ATTACHMENT6},
    {"GL_COLOR_ATTACHMENT7", GL_COLOR_ATTACHMENT7},
    {"GL_COLOR_ATTACHMENT8", GL_COLOR_ATTACHMENT8},
    {"GL_COLOR_ATTACHMENT9", GL_COLOR_ATTACHMENT9},
    {"GL_COLOR_ATTACHMENT10", GL_COLOR_ATTACHMENT10},
    {"GL_COLOR_ATTACHMENT11", GL_COLOR_ATTACHMENT11},
    {"GL_COLOR_ATTACHMENT12", GL_COLOR_ATTACHMENT12},
    {"GL_COLOR_ATTACHMENT13", GL_COLOR_ATTACHMENT13},
    {"GL_COLOR_ATTACHMENT14", GL_COLOR_ATTACHMENT14},
    {"GL_COLOR_ATTACHMENT15", GL_COLOR_ATTACHMENT15},
    {"GL_COLOR_ATTACHMENT16", GL_COLOR_ATTACHMENT16},
    {"GL_COLOR_ATTACHMENT17", GL_COLOR_ATTACHMENT17},
    {"GL_COLOR_ATTACHMENT18", GL_COLOR_ATTACHMENT18},
    {"GL_COLOR_ATTACHMENT19", GL_COLOR_ATTACHMENT19},
    {"GL_COLOR_ATTACHMENT20", GL_COLOR_ATTACHMENT20},
    {"GL_COLOR_ATTACHMENT21", GL_COLOR_ATTACHMENT21},
    {"GL_COLOR_ATTACHMENT22", GL_COLOR_ATTACHMENT22},
    {"GL_COLOR_ATTACHMENT23", GL_COLOR_ATTACHMENT23},
    {"GL_COLOR_ATTACHMENT24", GL_COLOR_ATTACHMENT24},
    {"GL_COLOR_ATTACHMENT25", GL_COLOR_ATTACHMENT25},
    {"GL_COLOR_ATTACHMENT26", GL_COLOR_ATTACHMENT26},
    {"GL_COLOR_ATTACHMENT27", GL_COLOR_ATTACHMENT27},
    {"GL_COLOR_ATTACHMENT28", GL_COLOR_ATTACHMENT28},
    {"GL_COLOR_ATTACHMENT29", GL_COLOR_ATTACHMENT29},
    {"GL_COLOR_ATTACHMENT30", GL_COLOR_ATTACHMENT30},
    {"GL_COLOR_ATTACHMENT31", GL_COLOR_ATTACHMENT31},
    {"GL_DEPTH_ATTACHMENT", GL_DEPTH_ATTACHMENT},
    {"GL_STENCIL_ATTACHMENT", GL_STENCIL_ATTACHMENT},
    {"GL_FRAMEBUFFER", GL_FRAMEBUFFER},
    {"GL_RENDERBUFFER", GL_RENDERBUFFER},
    {"GL_RENDERBUFFER_WIDTH", GL_RENDERBUFFER_WIDTH},
    {"GL_RENDERBUFFER_HEIGHT", GL_RENDERBUFFER_HEIGHT},
    {"GL_RENDERBUFFER_INTERNAL_FORMAT", GL_RENDERBUFFER_INTERNAL_FORMAT},
    {"GL_STENCIL_INDEX1", GL_STENCIL_INDEX1},
    {"GL_STENCIL_INDEX4", GL_STENCIL_INDEX4},
    {"GL_STENCIL_INDEX8", GL_STENCIL_INDEX8},
    {"GL_STENCIL_INDEX16", GL_STENCIL_INDEX16},
    {"GL_RENDERBUFFER_RED_SIZE", GL_RENDERBUFFER_RED_SIZE},
    {"GL_RENDERBUFFER_GREEN_SIZE", GL_RENDERBUFFER_GREEN_SIZE},
    {"GL_RENDERBUFFER_BLUE_SIZE", GL_RENDERBUFFER_BLUE_SIZE},
    {"GL_RENDERBUFFER_ALPHA_SIZE", GL_RENDERBUFFER_ALPHA_SIZE},
    {"GL_RENDERBUFFER_DEPTH_SIZE", GL_RENDERBUFFER_DEPTH_SIZE},
    {"GL_RENDERBUFFER_STENCIL_SIZE", GL_RENDERBUFFER_STENCIL_SIZE},
    {"GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE", GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE},
    {"GL_MAX_SAMPLES", GL_MAX_SAMPLES},
    {"GL_TEXTURE_GEN_STR_OES", GL_TEXTURE_GEN_STR_OES},
    {"GL_HALF_FLOAT_OES", GL_HALF_FLOAT_OES},
    {"GL_RGB565", GL_RGB565},
    {"GL_ETC1_RGB8_OES", GL_ETC1_RGB8_OES},
    {"GL_TEXTURE_EXTERNAL_OES", GL_TEXTURE_EXTERNAL_OES},
    {"GL_SAMPLER_EXTERNAL_OES", GL_SAMPLER_EXTERNAL_OES},
    {"GL_TEXTURE_BINDING_EXTERNAL_OES", GL_TEXTURE_BINDING_EXTERNAL_OES},
    {"GL_REQUIRED_TEXTURE_IMAGE_UNITS_OES", GL_REQUIRED_TEXTURE_IMAGE_UNITS_OES},
    {"GL_PRIMITIVE_RESTART_FIXED_INDEX", GL_PRIMITIVE_RESTART_FIXED_INDEX},
    {"GL_ANY_SAMPLES_PASSED_CONSERVATIVE", GL_ANY_SAMPLES_PASSED_CONSERVATIVE},
    {"GL_MAX_ELEMENT_INDEX", GL_MAX_ELEMENT_INDEX},
    {"GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_SAMPLES_EXT", GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_SAMPLES_EXT},
    {"GL_RGBA32UI", GL_RGBA32UI},
    {"GL_RGB32UI", GL_RGB32UI},
    {"GL_ALPHA32UI_EXT", GL_ALPHA32UI_EXT},
    {"GL_INTENSITY32UI_EXT", GL_INTENSITY32UI_EXT},
    {"GL_LUMINANCE32UI_EXT", GL_LUMINANCE32UI_EXT},
    {"GL_LUMINANCE_ALPHA32UI_EXT", GL_LUMINANCE_ALPHA32UI_EXT},
    {"GL_RGBA16UI", GL_RGBA16UI},
    {"GL_RGB16UI", GL_RGB16UI},
    {"GL_ALPHA16UI_EXT", GL_ALPHA16UI_EXT},
    {"GL_INTENSITY16UI_EXT", GL_INTENSITY16UI_EXT},
    {"GL_LUMINANCE16UI_EXT", GL_LUMINANCE16UI_EXT},
    {"GL_LUMINANCE_ALPHA16UI_EXT", GL_LUMINANCE_ALPHA16UI_EXT},
    {"GL_RGBA8UI", GL_RGBA8UI},
    {"GL_RGB8UI", GL_RGB8UI},
    {"GL_ALPHA8UI_EXT", GL_ALPHA8UI_EXT},
    {"GL_INTENSITY8UI_EXT", GL_INTENSITY8UI_EXT},
    {"GL_LUMINANCE8UI_EXT", GL_LUMINANCE8UI_EXT},
    {"GL_LUMINANCE_ALPHA8UI_EXT", GL_LUMINANCE_ALPHA8UI_EXT},
    {"GL_RGBA32I", GL_RGBA32I},
    {"GL_RGB32I", GL_RGB32I},
    {"GL_ALPHA32I_EXT", GL_ALPHA32I_EXT},
    {"GL_INTENSITY32I_EXT", GL_INTENSITY32I_EXT},
    {"GL_LUMINANCE32I_EXT", GL_LUMINANCE32I_EXT},
    {"GL_LUMINANCE_ALPHA32I_EXT", GL_LUMINANCE_ALPHA32I_EXT},
    {"GL_RGBA16I", GL_RGBA16I},
    {"GL_RGB16I", GL_RGB16I},
    {"GL_ALPHA16I_EXT", GL_ALPHA16I_EXT},
    {"GL_INTENSITY16I_EXT", GL_INTENSITY16I_EXT},
    {"GL_LUMINANCE16I_EXT", GL_LUMINANCE16I_EXT},
    {"GL_LUMINANCE_ALPHA16I_EXT", GL_LUMINANCE_ALPHA16I_EXT},
    {"GL_RGBA8I", GL_RGBA8I},
    {"GL_RGB8I", GL_RGB8I},
    {"GL_ALPHA8I_EXT", GL_ALPHA8I_EXT},
    {"GL_INTENSITY8I_EXT", GL_INTENSITY8I_EXT},
    {"GL_LUMINANCE8I_EXT", GL_LUMINANCE8I_EXT},
    {"GL_LUMINANCE_ALPHA8I_EXT", GL_LUMINANCE_ALPHA8I_EXT},
    {"GL_RED_INTEGER", GL_RED_INTEGER},
    {"GL_GREEN_INTEGER", GL_GREEN_INTEGER},
    {"GL_BLUE_INTEGER", GL_BLUE_INTEGER},
    {"GL_ALPHA_INTEGER", GL_ALPHA_INTEGER},
    {"GL_RGB_INTEGER", GL_RGB_INTEGER},
    {"GL_RGBA_INTEGER", GL_RGBA_INTEGER},
    {"GL_BGR_INTEGER", GL_BGR_INTEGER},
    {"GL_BGRA_INTEGER", GL_BGRA_INTEGER},
    {"GL_LUMINANCE_INTEGER_EXT", GL_LUMINANCE_INTEGER_EXT},
    {"GL_LUMINANCE_ALPHA_INTEGER_EXT", GL_LUMINANCE_ALPHA_INTEGER_EXT},
    {"GL_RGBA_INTEGER_MODE_EXT", GL_RGBA_INTEGER_MODE_EXT},
    {"GL_INT_2_10_10_10_REV", GL_INT_2_10_10_10_REV},
    {"GL_MAX_PROGRAM_PARAMETER_BUFFER_BINDINGS_NV", GL_MAX_PROGRAM_PARAMETER_BUFFER_BINDINGS_NV},
    {"GL_MAX_PROGRAM_PARAMETER_BUFFER_SIZE_NV", GL_MAX_PROGRAM_PARAMETER_BUFFER_SIZE_NV},
    {"GL_VERTEX_PROGRAM_PARAMETER_BUFFER_NV", GL_VERTEX_PROGRAM_PARAMETER_BUFFER_NV},
    {"GL_GEOMETRY_PROGRAM_PARAMETER_BUFFER_NV", GL_GEOMETRY_PROGRAM_PARAMETER_BUFFER_NV},
    {"GL_FRAGMENT_PROGRAM_PARAMETER_BUFFER_NV", GL_FRAGMENT_PROGRAM_PARAMETER_BUFFER_NV},
    {"GL_MAX_PROGRAM_GENERIC_ATTRIBS_NV", GL_MAX_PROGRAM_GENERIC_ATTRIBS_NV},
    {"GL_MAX_PROGRAM_GENERIC_RESULTS_NV", GL_MAX_PROGRAM_GENERIC_RESULTS_NV},
    {"GL_FRAMEBUFFER_ATTACHMENT_LAYERED", GL_FRAMEBUFFER_ATTACHMENT_LAYERED},
    {"GL_FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS", GL_FRAMEBUFFER_INCOMPLETE_LAYER_TARGETS},
    {"GL_FRAMEBUFFER_INCOMPLETE_LAYER_COUNT_ARB", GL_FRAMEBUFFER_INCOMPLETE_LAYER_COUNT_ARB},
    {"GL_LAYER_NV", GL_LAYER_NV},
    {"GL_DEPTH_COMPONENT32F_NV", GL_DEPTH_COMPONENT32F_NV},
    {"GL_DEPTH32F_STENCIL8_NV", GL_DEPTH32F_STENCIL8_NV},
    {"GL_FLOAT_32_UNSIGNED_INT_24_8_REV", GL_FLOAT_32_UNSIGNED_INT_24_8_REV},
    {"GL_SHADER_INCLUDE_ARB", GL_SHADER_INCLUDE_ARB},
    {"GL_DEPTH_BUFFER_FLOAT_MODE_NV", GL_DEPTH_BUFFER_FLOAT_MODE_NV},
    {"GL_FRAMEBUFFER_SRGB", GL_FRAMEBUFFER_SRGB},
    {"GL_FRAMEBUFFER_SRGB_CAPABLE_EXT", GL_FRAMEBUFFER_SRGB_CAPABLE_EXT},
    {"GL_COMPRESSED_RED_RGTC1", GL_COMPRESSED_RED_RGTC1},
    {"GL_COMPRESSED_SIGNED_RED_RGTC1", GL_COMPRESSED_SIGNED_RED_RGTC1},
    {"GL_COMPRESSED_RG_RGTC2", GL_COMPRESSED_RG_RGTC2},
    {"GL_COMPRESSED_SIGNED_RG_RGTC2", GL_COMPRESSED_SIGNED_RG_RGTC2},
    {"GL_SAMPLER_1D_ARRAY", GL_SAMPLER_1D_ARRAY},
    {"GL_SAMPLER_2D_ARRAY", GL_SAMPLER_2D_ARRAY},
    {"GL_SAMPLER_BUFFER", GL_SAMPLER_BUFFER},
    {"GL_SAMPLER_1D_ARRAY_SHADOW", GL_SAMPLER_1D_ARRAY_SHADOW},
    {"GL_SAMPLER_2D_ARRAY_SHADOW", GL_SAMPLER_2D_ARRAY_SHADOW},
    {"GL_SAMPLER_CUBE_SHADOW", GL_SAMPLER_CUBE_SHADOW},
    {"GL_UNSIGNED_INT_VEC2", GL_UNSIGNED_INT_VEC2},
    {"GL_UNSIGNED_INT_VEC3", GL_UNSIGNED_INT_VEC3},
    {"GL_UNSIGNED_INT_VEC4", GL_UNSIGNED_INT_VEC4},
    {"GL_INT_SAMPLER_1D", GL_INT_SAMPLER_1D},
    {"GL_INT_SAMPLER_2D", GL_INT_SAMPLER_2D},
    {"GL_INT_SAMPLER_3D", GL_INT_SAMPLER_3D},
    {"GL_INT_SAMPLER_CUBE", GL_INT_SAMPLER_CUBE},
    {"GL_INT_SAMPLER_2D_RECT", GL_INT_SAMPLER_2D_RECT},
    {"GL_INT_SAMPLER_1D_ARRAY", GL_INT_SAMPLER_1D_ARRAY},
    {"GL_INT_SAMPLER_2D_ARRAY", GL_INT_SAMPLER_2D_ARRAY},
    {"GL_INT_SAMPLER_BUFFER", GL_INT_SAMPLER_BUFFER},
    {"GL_UNSIGNED_INT_SAMPLER_1D", GL_UNSIGNED_INT_SAMPLER_1D},
    {"GL_UNSIGNED_INT_SAMPLER_2D", GL_UNSIGNED_INT_SAMPLER_2D},
    {"GL_UNSIGNED_INT_SAMPLER_3D", GL_UNSIGNED_INT_SAMPLER_3D},
    {"GL_UNSIGNED_INT_SAMPLER_CUBE", GL_UNSIGNED_INT_SAMPLER_CUBE},
    {"GL_UNSIGNED_INT_SAMPLER_2D_RECT", GL_UNSIGNED_INT_SAMPLER_2D_RECT},
    {"GL_UNSIGNED_INT_SAMPLER_1D_ARRAY", GL_UNSIGNED_INT_SAMPLER_1D_ARRAY},
    {"GL_UNSIGNED_INT_SAMPLER_2D_ARRAY", GL_UNSIGNED_INT_SAMPLER_2D_ARRAY},
    {"GL_UNSIGNED_INT_SAMPLER_BUFFER", GL_UNSIGNED_INT_SAMPLER_BUFFER},
    {"GL_GEOMETRY_SHADER", GL_GEOMETRY_SHADER},
    {"GL_GEOMETRY_VERTICES_OUT_ARB", GL_GEOMETRY_VERTICES_OUT_ARB},
    {"GL_GEOMETRY_INPUT_TYPE_ARB", GL_GEOMETRY_INPUT_TYPE_ARB},
    {"GL_GEOMETRY_OUTPUT_TYPE_ARB", GL_GEOMETRY_OUTPUT_TYPE_ARB},
    {"GL_MAX_GEOMETRY_VARYING_COMPONENTS_ARB", GL_MAX_GEOMETRY_VARYING_COMPONENTS_ARB},
    {"GL_MAX_VERTEX_VARYING_COMPONENTS_ARB", GL_MAX_VERTEX_VARYING_COMPONENTS_ARB},
    {"GL_MAX_GEOMETRY_UNIFORM_COMPONENTS", GL_MAX_GEOMETRY_UNIFORM_COMPONENTS},
    {"GL_MAX_GEOMETRY_OUTPUT_VERTICES", GL_MAX_GEOMETRY_OUTPUT_VERTICES},
    {"GL_MAX_GEOMETRY_TOTAL_OUTPUT_COMPONENTS", GL_MAX_GEOMETRY_TOTAL_OUTPUT_COMPONENTS},
    {"GL_MAX_VERTEX_BINDABLE_UNIFORMS_EXT", GL_MAX_VERTEX_BINDABLE_UNIFORMS_EXT},
    {"GL_MAX_FRAGMENT_BINDABLE_UNIFORMS_EXT", GL_MAX_FRAGMENT_BINDABLE_UNIFORMS_EXT},
    {"GL_MAX_GEOMETRY_BINDABLE_UNIFORMS_EXT", GL_MAX_GEOMETRY_BINDABLE_UNIFORMS_EXT},
    {"GL_ACTIVE_SUBROUTINES", GL_ACTIVE_SUBROUTINES},
    {"GL_ACTIVE_SUBROUTINE_UNIFORMS", GL_ACTIVE_SUBROUTINE_UNIFORMS},
    {"GL_MAX_SUBROUTINES", GL_MAX_SUBROUTINES},
    {"GL_MAX_SUBROUTINE_UNIFORM_LOCATIONS", GL_MAX_SUBROUTINE_UNIFORM_LOCATIONS},
    {"GL_NAMED_STRING_LENGTH_ARB", GL_NAMED_STRING_LENGTH_ARB},
    {"GL_NAMED_STRING_TYPE_ARB", GL_NAMED_STRING_TYPE_ARB},
    {"GL_MAX_BINDABLE_UNIFORM_SIZE_EXT", GL_MAX_BINDABLE_UNIFORM_SIZE_EXT},
    {"GL_UNIFORM_BUFFER_EXT", GL_UNIFORM_BUFFER_EXT},
    {"GL_UNIFORM_BUFFER_BINDING_EXT", GL_UNIFORM_BUFFER_BINDING_EXT},
    {"GL_LOW_FLOAT", GL_LOW_FLOAT},
    {"GL_MEDIUM_FLOAT", GL_MEDIUM_FLOAT},
    {"GL_HIGH_FLOAT", GL_HIGH_FLOAT},
    {"GL_LOW_INT", GL_LOW_INT},
    {"GL_MEDIUM_INT", GL_MEDIUM_INT},
    {"GL_HIGH_INT", GL_HIGH_INT},
    {"GL_UNSIGNED_INT_10_10_10_2_OES", GL_UNSIGNED_INT_10_10_10_2_OES},
    {"GL_INT_10_10_10_2_OES", GL_INT_10_10_10_2_OES},
    {"GL_SHADER_BINARY_FORMATS", GL_SHADER_BINARY_FORMATS},
    {"GL_NUM_SHADER_BINARY_FORMATS", GL_NUM_SHADER_BINARY_FORMATS},
    {"GL_SHADER_COMPILER", GL_SHADER_COMPILER},
    {"GL_MAX_VERTEX_UNIFORM_VECTORS", GL_MAX_VERTEX_UNIFORM_VECTORS},
    {"GL_MAX_VARYING_VECTORS", GL_MAX_VARYING_VECTORS},
    {"GL_MAX_FRAGMENT_UNIFORM_VECTORS", GL_MAX_FRAGMENT_UNIFORM_VECTORS},
    {"GL_RENDERBUFFER_COLOR_SAMPLES_NV", GL_RENDERBUFFER_COLOR_SAMPLES_NV},
    {"GL_MAX_MULTISAMPLE_COVERAGE_MODES_NV", GL_MAX_MULTISAMPLE_COVERAGE_MODES_NV},
    {"GL_MULTISAMPLE_COVERAGE_MODES_NV", GL_MULTISAMPLE_COVERAGE_MODES_NV},
    {"GL_QUERY_WAIT", GL_QUERY_WAIT},
    {"GL_QUERY_NO_WAIT", GL_QUERY_NO_WAIT},
    {"GL_QUERY_BY_REGION_WAIT", GL_QUERY_BY_REGION_WAIT},
    {"GL_QUERY_BY_REGION_NO_WAIT", GL_QUERY_BY_REGION_NO_WAIT},
    {"GL_QUERY_WAIT_INVERTED", GL_QUERY_WAIT_INVERTED},
    {"GL_QUERY_NO_WAIT_INVERTED", GL_QUERY_NO_WAIT_INVERTED},
    {"GL_QUERY_BY_REGION_WAIT_INVERTED", GL_QUERY_BY_REGION_WAIT_INVERTED},
    {"GL_QUERY_BY_REGION_NO_WAIT_INVERTED", GL_QUERY_BY_REGION_NO_WAIT_INVERTED},
    {"GL_POLYGON_OFFSET_CLAMP_EXT", GL_POLYGON_OFFSET_CLAMP_EXT},
    {"GL_MAX_COMBINED_TESS_CONTROL_UNIFORM_COMPONENTS", GL_MAX_COMBINED_TESS_CONTROL_UNIFORM_COMPONENTS},
    {"GL_MAX_COMBINED_TESS_EVALUATION_UNIFORM_COMPONENTS", GL_MAX_COMBINED_TESS_EVALUATION_UNIFORM_COMPONENTS},
    {"GL_COLOR_SAMPLES_NV", GL_COLOR_SAMPLES_NV},
    {"GL_TRANSFORM_FEEDBACK", GL_TRANSFORM_FEEDBACK},
    {"GL_TRANSFORM_FEEDBACK_BUFFER_PAUSED", GL_TRANSFORM_FEEDBACK_BUFFER_PAUSED},
    {"GL_TRANSFORM_FEEDBACK_BUFFER_ACTIVE", GL_TRANSFORM_FEEDBACK_BUFFER_ACTIVE},
    {"GL_TRANSFORM_FEEDBACK_BINDING", GL_TRANSFORM_FEEDBACK_BINDING},
    {"GL_FRAME_NV", GL_FRAME_NV},
    {"GL_FIELDS_NV", GL_FIELDS_NV},
    {"GL_TIMESTAMP", GL_TIMESTAMP},
    {"GL_NUM_FILL_STREAMS_NV", GL_NUM_FILL_STREAMS_NV},
    {"GL_PRESENT_TIME_NV", GL_PRESENT_TIME_NV},
    {"GL_PRESENT_DURATION_NV", GL_PRESENT_DURATION_NV},
    {"GL_DEPTH_COMPONENT16_NONLINEAR_NV", GL_DEPTH_COMPONENT16_NONLINEAR_NV},
    {"GL_PROGRAM_MATRIX_EXT", GL_PROGRAM_MATRIX_EXT},
    {"GL_TRANSPOSE_PROGRAM_MATRIX_EXT", GL_TRANSPOSE_PROGRAM_MATRIX_EXT},
    {"GL_PROGRAM_MATRIX_STACK_DEPTH_EXT", GL_PROGRAM_MATRIX_STACK_DEPTH_EXT},
    {"GL_TEXTURE_SWIZZLE_R", GL_TEXTURE_SWIZZLE_R},
    {"GL_TEXTURE_SWIZZLE_G", GL_TEXTURE_SWIZZLE_G},
    {"GL_TEXTURE_SWIZZLE_B", GL_TEXTURE_SWIZZLE_B},
    {"GL_TEXTURE_SWIZZLE_A", GL_TEXTURE_SWIZZLE_A},
    {"GL_TEXTURE_SWIZZLE_RGBA", GL_TEXTURE_SWIZZLE_RGBA},
    {"GL_ACTIVE_SUBROUTINE_UNIFORM_LOCATIONS", GL_ACTIVE_SUBROUTINE_UNIFORM_LOCATIONS},
    {"GL_ACTIVE_SUBROUTINE_MAX_LENGTH", GL_ACTIVE_SUBROUTINE_MAX_LENGTH},
    {"GL_ACTIVE_SUBROUTINE_UNIFORM_MAX_LENGTH", GL_ACTIVE_SUBROUTINE_UNIFORM_MAX_LENGTH},
    {"GL_NUM_COMPATIBLE_SUBROUTINES", GL_NUM_COMPATIBLE_SUBROUTINES},
    {"GL_COMPATIBLE_SUBROUTINES", GL_COMPATIBLE_SUBROUTINES},
    {"GL_QUADS_FOLLOW_PROVOKING_VERTEX_CONVENTION", GL_QUADS_FOLLOW_PROVOKING_VERTEX_CONVENTION},
    {"GL_FIRST_VERTEX_CONVENTION", GL_FIRST_VERTEX_CONVENTION},
    {"GL_LAST_VERTEX_CONVENTION", GL_LAST_VERTEX_CONVENTION},
    {"GL_PROVOKING_VERTEX", GL_PROVOKING_VERTEX},
    {"GL_SAMPLE_POSITION", GL_SAMPLE_POSITION},
    {"GL_SAMPLE_MASK", GL_SAMPLE_MASK},
    {"GL_SAMPLE_MASK_VALUE", GL_SAMPLE_MASK_VALUE},
    {"GL_TEXTURE_BINDING_RENDERBUFFER_NV", GL_TEXTURE_BINDING_RENDERBUFFER_NV},
    {"GL_TEXTURE_RENDERBUFFER_DATA_STORE_BINDING_NV", GL_TEXTURE_RENDERBUFFER_DATA_STORE_BINDING_NV},
    {"GL_TEXTURE_RENDERBUFFER_NV", GL_TEXTURE_RENDERBUFFER_NV},
    {"GL_SAMPLER_RENDERBUFFER_NV", GL_SAMPLER_RENDERBUFFER_NV},
    {"GL_INT_SAMPLER_RENDERBUFFER_NV", GL_INT_SAMPLER_RENDERBUFFER_NV},
    {"GL_UNSIGNED_INT_SAMPLER_RENDERBUFFER_NV", GL_UNSIGNED_INT_SAMPLER_RENDERBUFFER_NV},
    {"GL_MAX_SAMPLE_MASK_WORDS", GL_MAX_SAMPLE_MASK_WORDS},
    {"GL_MAX_GEOMETRY_SHADER_INVOCATIONS", GL_MAX_GEOMETRY_SHADER_INVOCATIONS},
    {"GL_MIN_FRAGMENT_INTERPOLATION_OFFSET", GL_MIN_FRAGMENT_INTERPOLATION_OFFSET},
    {"GL_MAX_FRAGMENT_INTERPOLATION_OFFSET", GL_MAX_FRAGMENT_INTERPOLATION_OFFSET},
    {"GL_FRAGMENT_INTERPOLATION_OFFSET_BITS", GL_FRAGMENT_INTERPOLATION_OFFSET_BITS},
    {"GL_MIN_PROGRAM_TEXTURE_GATHER_OFFSET", GL_MIN_PROGRAM_TEXTURE_GATHER_OFFSET},
    {"GL_MAX_PROGRAM_TEXTURE_GATHER_OFFSET", GL_MAX_PROGRAM_TEXTURE_GATHER_OFFSET},
    {"GL_MAX_TRANSFORM_FEEDBACK_BUFFERS", GL_MAX_TRANSFORM_FEEDBACK_BUFFERS},
    {"GL_MAX_VERTEX_STREAMS", GL_MAX_VERTEX_STREAMS},
    {"GL_PATCH_VERTICES", GL_PATCH_VERTICES},
    {"GL_PATCH_DEFAULT_INNER_LEVEL", GL_PATCH_DEFAULT_INNER_LEVEL},
    {"GL_PATCH_DEFAULT_OUTER_LEVEL", GL_PATCH_DEFAULT_OUTER_LEVEL},
    {"GL_TESS_CONTROL_OUTPUT_VERTICES", GL_TESS_CONTROL_OUTPUT_VERTICES},
    {"GL_TESS_GEN_MODE", GL_TESS_GEN_MODE},
    {"GL_TESS_GEN_SPACING", GL_TESS_GEN_SPACING},
    {"GL_TESS_GEN_VERTEX_ORDER", GL_TESS_GEN_VERTEX_ORDER},
    {"GL_TESS_GEN_POINT_MODE", GL_TESS_GEN_POINT_MODE},
    {"GL_ISOLINES", GL_ISOLINES},
    {"GL_FRACTIONAL_ODD", GL_FRACTIONAL_ODD},
    {"GL_FRACTIONAL_EVEN", GL_FRACTIONAL_EVEN},
    {"GL_MAX_PATCH_VERTICES", GL_MAX_PATCH_VERTICES},
    {"GL_MAX_TESS_GEN_LEVEL", GL_MAX_TESS_GEN_LEVEL},
    {"GL_MAX_TESS_CONTROL_UNIFORM_COMPONENTS", GL_MAX_TESS_CONTROL_UNIFORM_COMPONENTS},
    {"GL_MAX_TESS_EVALUATION_UNIFORM_COMPONENTS", GL_MAX_TESS_EVALUATION_UNIFORM_COMPONENTS},
    {"GL_MAX_TESS_CONTROL_TEXTURE_IMAGE_UNITS", GL_MAX_TESS_CONTROL_TEXTURE_IMAGE_UNITS},
    {"GL_MAX_TESS_EVALUATION_TEXTURE_IMAGE_UNITS", GL_MAX_TESS_EVALUATION_TEXTURE_IMAGE_UNITS},
    {"GL_MAX_TESS_CONTROL_OUTPUT_COMPONENTS", GL_MAX_TESS_CONTROL_OUTPUT_COMPONENTS},
    {"GL_MAX_TESS_PATCH_COMPONENTS", GL_MAX_TESS_PATCH_COMPONENTS},
    {"GL_MAX_TESS_CONTROL_TOTAL_OUTPUT_COMPONENTS", GL_MAX_TESS_CONTROL_TOTAL_OUTPUT_COMPONENTS},
    {"GL_MAX_TESS_EVALUATION_OUTPUT_COMPONENTS", GL_MAX_TESS_EVALUATION_OUTPUT_COMPONENTS},
    {"GL_TESS_EVALUATION_SHADER", GL_TESS_EVALUATION_SHADER},
    {"GL_TESS_CONTROL_SHADER", GL_TESS_CONTROL_SHADER},
    {"GL_MAX_TESS_CONTROL_UNIFORM_BLOCKS", GL_MAX_TESS_CONTROL_UNIFORM_BLOCKS},
    {"GL_MAX_TESS_EVALUATION_UNIFORM_BLOCKS", GL_MAX_TESS_EVALUATION_UNIFORM_BLOCKS},
    {"GL_COMPRESSED_RGBA_BPTC_UNORM", GL_COMPRESSED_RGBA_BPTC_UNORM},
    {"GL_COMPRESSED_SRGB_ALPHA_BPTC_UNORM", GL_COMPRESSED_SRGB_ALPHA_BPTC_UNORM},
    {"GL_COMPRESSED_RGB_BPTC_SIGNED_FLOAT", GL_COMPRESSED_RGB_BPTC_SIGNED_FLOAT},
    {"GL_COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT", GL_COMPRESSED_RGB_BPTC_UNSIGNED_FLOAT},
    {"GL_INCLUSIVE_EXT", GL_INCLUSIVE_EXT},
    {"GL_EXCLUSIVE_EXT", GL_EXCLUSIVE_EXT},
    {"GL_WINDOW_RECTANGLE_EXT", GL_WINDOW_RECTANGLE_EXT},
    {"GL_WINDOW_RECTANGLE_MODE_EXT", GL_WINDOW_RECTANGLE_MODE_EXT},
    {"GL_MAX_WINDOW_RECTANGLES_EXT", GL_MAX_WINDOW_RECTANGLES_EXT},
    {"GL_NUM_WINDOW_RECTANGLES_EXT", GL_NUM_WINDOW_RECTANGLES_EXT},
    {"GL_BUFFER_GPU_ADDRESS_NV", GL_BUFFER_GPU_ADDRESS_NV},
    {"GL_VERTEX_ATTRIB_ARRAY_UNIFIED_NV", GL_VERTEX_ATTRIB_ARRAY_UNIFIED_NV},
    {"GL_ELEMENT_ARRAY_UNIFIED_NV", GL_ELEMENT_ARRAY_UNIFIED_NV},
    {"GL_VERTEX_ATTRIB_ARRAY_ADDRESS_NV", GL_VERTEX_ATTRIB_ARRAY_ADDRESS_NV},
    {"GL_VERTEX_ARRAY_ADDRESS_NV", GL_VERTEX_ARRAY_ADDRESS_NV},
    {"GL_NORMAL_ARRAY_ADDRESS_NV", GL_NORMAL_ARRAY_ADDRESS_NV},
    {"GL_COLOR_ARRAY_ADDRESS_NV", GL_COLOR_ARRAY_ADDRESS_NV},
    {"GL_INDEX_ARRAY_ADDRESS_NV", GL_INDEX_ARRAY_ADDRESS_NV},
    {"GL_TEXTURE_COORD_ARRAY_ADDRESS_NV", GL_TEXTURE_COORD_ARRAY_ADDRESS_NV},
    {"GL_EDGE_FLAG_ARRAY_ADDRESS_NV", GL_EDGE_FLAG_ARRAY_ADDRESS_NV},
    {"GL_SECONDARY_COLOR_ARRAY_ADDRESS_NV", GL_SECONDARY_COLOR_ARRAY_ADDRESS_NV},
    {"GL_FOG_COORD_ARRAY_ADDRESS_NV", GL_FOG_COORD_ARRAY_ADDRESS_NV},
    {"GL_ELEMENT_ARRAY_ADDRESS_NV", GL_ELEMENT_ARRAY_ADDRESS_NV},
    {"GL_VERTEX_ATTRIB_ARRAY_LENGTH_NV", GL_VERTEX_ATTRIB_ARRAY_LENGTH_NV},
    {"GL_VERTEX_ARRAY_LENGTH_NV", GL_VERTEX_ARRAY_LENGTH_NV},
    {"GL_NORMAL_ARRAY_LENGTH_NV", GL_NORMAL_ARRAY_LENGTH_NV},
    {"GL_COLOR_ARRAY_LENGTH_NV", GL_COLOR_ARRAY_LENGTH_NV},
    {"GL_INDEX_ARRAY_LENGTH_NV", GL_INDEX_ARRAY_LENGTH_NV},
    {"GL_TEXTURE_COORD_ARRAY_LENGTH_NV", GL_TEXTURE_COORD_ARRAY_LENGTH_NV},
    {"GL_EDGE_FLAG_ARRAY_LENGTH_NV", GL_EDGE_FLAG_ARRAY_LENGTH_NV},
    {"GL_SECONDARY_COLOR_ARRAY_LENGTH_NV", GL_SECONDARY_COLOR_ARRAY_LENGTH_NV},
    {"GL_FOG_COORD_ARRAY_LENGTH_NV", GL_FOG_COORD_ARRAY_LENGTH_NV},
    {"GL_ELEMENT_ARRAY_LENGTH_NV", GL_ELEMENT_ARRAY_LENGTH_NV},
    {"GL_GPU_ADDRESS_NV", GL_GPU_ADDRESS_NV},
    {"GL_MAX_SHADER_BUFFER_ADDRESS_NV", GL_MAX_SHADER_BUFFER_ADDRESS_NV},
    {"GL_COPY_READ_BUFFER", GL_COPY_READ_BUFFER},
    {"GL_COPY_WRITE_BUFFER", GL_COPY_WRITE_BUFFER},
    {"GL_MAX_IMAGE_UNITS", GL_MAX_IMAGE_UNITS},
    {"GL_MAX_COMBINED_IMAGE_UNITS_AND_FRAGMENT_OUTPUTS", GL_MAX_COMBINED_IMAGE_UNITS_AND_FRAGMENT_OUTPUTS},
    {"GL_IMAGE_BINDING_NAME", GL_IMAGE_BINDING_NAME},
    {"GL_IMAGE_BINDING_LEVEL", GL_IMAGE_BINDING_LEVEL},
    {"GL_IMAGE_BINDING_LAYERED", GL_IMAGE_BINDING_LAYERED},
    {"GL_IMAGE_BINDING_LAYER", GL_IMAGE_BINDING_LAYER},
    {"GL_IMAGE_BINDING_ACCESS", GL_IMAGE_BINDING_ACCESS},
    {"GL_DRAW_INDIRECT_BUFFER", GL_DRAW_INDIRECT_BUFFER},
    {"GL_DRAW_INDIRECT_UNIFIED_NV", GL_DRAW_INDIRECT_UNIFIED_NV},
    {"GL_DRAW_INDIRECT_ADDRESS_NV", GL_DRAW_INDIRECT_ADDRESS_NV},
    {"GL_DRAW_INDIRECT_LENGTH_NV", GL_DRAW_INDIRECT_LENGTH_NV},
    {"GL_DRAW_INDIRECT_BUFFER_BINDING", GL_DRAW_INDIRECT_BUFFER_BINDING},
    {"GL_MAX_PROGRAM_SUBROUTINE_PARAMETERS_NV", GL_MAX_PROGRAM_SUBROUTINE_PARAMETERS_NV},
    {"GL_MAX_PROGRAM_SUBROUTINE_NUM_NV", GL_MAX_PROGRAM_SUBROUTINE_NUM_NV},
    {"GL_DOUBLE_MAT2", GL_DOUBLE_MAT2},
    {"GL_DOUBLE_MAT3", GL_DOUBLE_MAT3},
    {"GL_DOUBLE_MAT4", GL_DOUBLE_MAT4},
    {"GL_DOUBLE_MAT2x3", GL_DOUBLE_MAT2x3},
    {"GL_DOUBLE_MAT2x4", GL_DOUBLE_MAT2x4},
    {"GL_DOUBLE_MAT3x2", GL_DOUBLE_MAT3x2},
    {"GL_DOUBLE_MAT3x4", GL_DOUBLE_MAT3x4},
    {"GL_DOUBLE_MAT4x2", GL_DOUBLE_MAT4x2},
    {"GL_DOUBLE_MAT4x3", GL_DOUBLE_MAT4x3},
    {"GL_VERTEX_BINDING_BUFFER", GL_VERTEX_BINDING_BUFFER},
    {"GL_MALI_SHADER_BINARY_ARM", GL_MALI_SHADER_BINARY_ARM},
    {"GL_MALI_PROGRAM_BINARY_ARM", GL_MALI_PROGRAM_BINARY_ARM},
    {"GL_MAX_SHADER_PIXEL_LOCAL_STORAGE_FAST_SIZE_EXT", GL_MAX_SHADER_PIXEL_LOCAL_STORAGE_FAST_SIZE_EXT},
    {"GL_SHADER_PIXEL_LOCAL_STORAGE_EXT", GL_SHADER_PIXEL_LOCAL_STORAGE_EXT},
    {"GL_FETCH_PER_SAMPLE_ARM", GL_FETCH_PER_SAMPLE_ARM},
    {"GL_FRAGMENT_SHADER_FRAMEBUFFER_FETCH_MRT_ARM", GL_FRAGMENT_SHADER_FRAMEBUFFER_FETCH_MRT_ARM},
    {"GL_MAX_SHADER_PIXEL_LOCAL_STORAGE_SIZE_EXT", GL_MAX_SHADER_PIXEL_LOCAL_STORAGE_SIZE_EXT},
    {"GL_TEXTURE_ASTC_DECODE_PRECISION_EXT", GL_TEXTURE_ASTC_DECODE_PRECISION_EXT},
    {"GL_NUM_SURFACE_COMPRESSION_FIXED_RATES_EXT", GL_NUM_SURFACE_COMPRESSION_FIXED_RATES_EXT},
    {"GL_RED_SNORM", GL_RED_SNORM},
    {"GL_RG_SNORM", GL_RG_SNORM},
    {"GL_RGB_SNORM", GL_RGB_SNORM},
    {"GL_RGBA_SNORM", GL_RGBA_SNORM},
    {"GL_R8_SNORM", GL_R8_SNORM},
    {"GL_RG8_SNORM", GL_RG8_SNORM},
    {"GL_RGB8_SNORM", GL_RGB8_SNORM},
    {"GL_RGBA8_SNORM", GL_RGBA8_SNORM},
    {"GL_R16_SNORM", GL_R16_SNORM},
    {"GL_RG16_SNORM", GL_RG16_SNORM},
    {"GL_RGB16_SNORM", GL_RGB16_SNORM},
    {"GL_RGBA16_SNORM", GL_RGBA16_SNORM},
    {"GL_SIGNED_NORMALIZED", GL_SIGNED_NORMALIZED},
    {"GL_PRIMITIVE_RESTART", GL_PRIMITIVE_RESTART},
    {"GL_PRIMITIVE_RESTART_INDEX", GL_PRIMITIVE_RESTART_INDEX},
    {"GL_MAX_PROGRAM_TEXTURE_GATHER_COMPONENTS_ARB", GL_MAX_PROGRAM_TEXTURE_GATHER_COMPONENTS_ARB},
    {"GL_PERFMON_GLOBAL_MODE_QCOM", GL_PERFMON_GLOBAL_MODE_QCOM},
    {"GL_BINNING_CONTROL_HINT_QCOM", GL_BINNING_CONTROL_HINT_QCOM},
    {"GL_CPU_OPTIMIZED_QCOM", GL_CPU_OPTIMIZED_QCOM},
    {"GL_GPU_OPTIMIZED_QCOM", GL_GPU_OPTIMIZED_QCOM},
    {"GL_RENDER_DIRECT_TO_FRAMEBUFFER_QCOM", GL_RENDER_DIRECT_TO_FRAMEBUFFER_QCOM},
    {"GL_GPU_DISJOINT_EXT", GL_GPU_DISJOINT_EXT},
    {"GL_SR8_EXT", GL_SR8_EXT},
    {"GL_SRG8_EXT", GL_SRG8_EXT},
    {"GL_SHADER_BINARY_VIV", GL_SHADER_BINARY_VIV},
    {"GL_INT8_NV", GL_INT8_NV},
    {"GL_INT8_VEC2_NV", GL_INT8_VEC2_NV},
    {"GL_INT8_VEC3_NV", GL_INT8_VEC3_NV},
    {"GL_INT8_VEC4_NV", GL_INT8_VEC4_NV},
    {"GL_INT16_NV", GL_INT16_NV},
    {"GL_INT16_VEC2_NV", GL_INT16_VEC2_NV},
    {"GL_INT16_VEC3_NV", GL_INT16_VEC3_NV},
    {"GL_INT16_VEC4_NV", GL_INT16_VEC4_NV},
    {"GL_INT64_VEC2_ARB", GL_INT64_VEC2_ARB},
    {"GL_INT64_VEC3_ARB", GL_INT64_VEC3_ARB},
    {"GL_INT64_VEC4_ARB", GL_INT64_VEC4_ARB},
    {"GL_UNSIGNED_INT8_NV", GL_UNSIGNED_INT8_NV},
    {"GL_UNSIGNED_INT8_VEC2_NV", GL_UNSIGNED_INT8_VEC2_NV},
    {"GL_UNSIGNED_INT8_VEC3_NV", GL_UNSIGNED_INT8_VEC3_NV},
    {"GL_UNSIGNED_INT8_VEC4_NV", GL_UNSIGNED_INT8_VEC4_NV},
    {"GL_UNSIGNED_INT16_NV", GL_UNSIGNED_INT16_NV},
    {"GL_UNSIGNED_INT16_VEC2_NV", GL_UNSIGNED_INT16_VEC2_NV},
    {"GL_UNSIGNED_INT16_VEC3_NV", GL_UNSIGNED_INT16_VEC3_NV},
    {"GL_UNSIGNED_INT16_VEC4_NV", GL_UNSIGNED_INT16_VEC4_NV},
    {"GL_UNSIGNED_INT64_VEC2_ARB", GL_UNSIGNED_INT64_VEC2_ARB},
    {"GL_UNSIGNED_INT64_VEC3_ARB", GL_UNSIGNED_INT64_VEC3_ARB},
    {"GL_UNSIGNED_INT64_VEC4_ARB", GL_UNSIGNED_INT64_VEC4_ARB},
    {"GL_FLOAT16_NV", GL_FLOAT16_NV},
    {"GL_FLOAT16_VEC2_NV", GL_FLOAT16_VEC2_NV},
    {"GL_FLOAT16_VEC3_NV", GL_FLOAT16_VEC3_NV},
    {"GL_FLOAT16_VEC4_NV", GL_FLOAT16_VEC4_NV},
    {"GL_DOUBLE_VEC2", GL_DOUBLE_VEC2},
    {"GL_DOUBLE_VEC3", GL_DOUBLE_VEC3},
    {"GL_DOUBLE_VEC4", GL_DOUBLE_VEC4},
    {"GL_SAMPLER_BUFFER_AMD", GL_SAMPLER_BUFFER_AMD},
    {"GL_INT_SAMPLER_BUFFER_AMD", GL_INT_SAMPLER_BUFFER_AMD},
    {"GL_UNSIGNED_INT_SAMPLER_BUFFER_AMD", GL_UNSIGNED_INT_SAMPLER_BUFFER_AMD},
    {"GL_TESSELLATION_MODE_AMD", GL_TESSELLATION_MODE_AMD},
    {"GL_TESSELLATION_FACTOR_AMD", GL_TESSELLATION_FACTOR_AMD},
    {"GL_DISCRETE_AMD", GL_DISCRETE_AMD},
    {"GL_CONTINUOUS_AMD", GL_CONTINUOUS_AMD},
    {"GL_TEXTURE_CUBE_MAP_ARRAY", GL_TEXTURE_CUBE_MAP_ARRAY},
    {"GL_TEXTURE_BINDING_CUBE_MAP_ARRAY", GL_TEXTURE_BINDING_CUBE_MAP_ARRAY},
    {"GL_PROXY_TEXTURE_CUBE_MAP_ARRAY", GL_PROXY_TEXTURE_CUBE_MAP_ARRAY},
    {"GL_SAMPLER_CUBE_MAP_ARRAY", GL_SAMPLER_CUBE_MAP_ARRAY},
    {"GL_SAMPLER_CUBE_MAP_ARRAY_SHADOW", GL_SAMPLER_CUBE_MAP_ARRAY_SHADOW},
    {"GL_INT_SAMPLER_CUBE_MAP_ARRAY", GL_INT_SAMPLER_CUBE_MAP_ARRAY},
    {"GL_UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY", GL_UNSIGNED_INT_SAMPLER_CUBE_MAP_ARRAY},
    {"GL_ALPHA_SNORM", GL_ALPHA_SNORM},
    {"GL_LUMINANCE_SNORM", GL_LUMINANCE_SNORM},
    {"GL_LUMINANCE_ALPHA_SNORM", GL_LUMINANCE_ALPHA_SNORM},
    {"GL_INTENSITY_SNORM", GL_INTENSITY_SNORM},
    {"GL_ALPHA8_SNORM", GL_ALPHA8_SNORM},
    {"GL_LUMINANCE8_SNORM", GL_LUMINANCE8_SNORM},
    {"GL_LUMINANCE8_ALPHA8_SNORM", GL_LUMINANCE8_ALPHA8_SNORM},
    {"GL_INTENSITY8_SNORM", GL_INTENSITY8_SNORM},
    {"GL_ALPHA16_SNORM", GL_ALPHA16_SNORM},
    {"GL_LUMINANCE16_SNORM", GL_LUMINANCE16_SNORM},
    {"GL_LUMINANCE16_ALPHA16_SNORM", GL_LUMINANCE16_ALPHA16_SNORM},
    {"GL_INTENSITY16_SNORM", GL_INTENSITY16_SNORM},
    {"GL_FACTOR_MIN_AMD", GL_FACTOR_MIN_AMD},
    {"GL_FACTOR_MAX_AMD", GL_FACTOR_MAX_AMD},
    {"GL_DEPTH_CLAMP_NEAR_AMD", GL_DEPTH_CLAMP_NEAR_AMD},
    {"GL_DEPTH_CLAMP_FAR_AMD", GL_DEPTH_CLAMP_FAR_AMD},
    {"GL_VIDEO_BUFFER_NV", GL_VIDEO_BUFFER_NV},
    {"GL_VIDEO_BUFFER_BINDING_NV", GL_VIDEO_BUFFER_BINDING_NV},
    {"GL_FIELD_UPPER_NV", GL_FIELD_UPPER_NV},
    {"GL_FIELD_LOWER_NV", GL_FIELD_LOWER_NV},
    {"GL_NUM_VIDEO_CAPTURE_STREAMS_NV", GL_NUM_VIDEO_CAPTURE_STREAMS_NV},
    {"GL_NEXT_VIDEO_CAPTURE_BUFFER_STATUS_NV", GL_NEXT_VIDEO_CAPTURE_BUFFER_STATUS_NV},
    {"GL_VIDEO_CAPTURE_TO_422_SUPPORTED_NV", GL_VIDEO_CAPTURE_TO_422_SUPPORTED_NV},
    {"GL_LAST_VIDEO_CAPTURE_STATUS_NV", GL_LAST_VIDEO_CAPTURE_STATUS_NV},
    {"GL_VIDEO_BUFFER_PITCH_NV", GL_VIDEO_BUFFER_PITCH_NV},
    {"GL_VIDEO_COLOR_CONVERSION_MATRIX_NV", GL_VIDEO_COLOR_CONVERSION_MATRIX_NV},
    {"GL_VIDEO_COLOR_CONVERSION_MAX_NV", GL_VIDEO_COLOR_CONVERSION_MAX_NV},
    {"GL_VIDEO_COLOR_CONVERSION_MIN_NV", GL_VIDEO_COLOR_CONVERSION_MIN_NV},
    {"GL_VIDEO_COLOR_CONVERSION_OFFSET_NV", GL_VIDEO_COLOR_CONVERSION_OFFSET_NV},
    {"GL_VIDEO_BUFFER_INTERNAL_FORMAT_NV", GL_VIDEO_BUFFER_INTERNAL_FORMAT_NV},
    {"GL_PARTIAL_SUCCESS_NV", GL_PARTIAL_SUCCESS_NV},
    {"GL_SUCCESS_NV", GL_SUCCESS_NV},
    {"GL_FAILURE_NV", GL_FAILURE_NV},
    {"GL_YCBYCR8_422_NV", GL_YCBYCR8_422_NV},
    {"GL_YCBAYCR8A_4224_NV", GL_YCBAYCR8A_4224_NV},
    {"GL_Z6Y10Z6CB10Z6Y10Z6CR10_422_NV", GL_Z6Y10Z6CB10Z6Y10Z6CR10_422_NV},
    {"GL_Z6Y10Z6CB10Z6A10Z6Y10Z6CR10Z6A10_4224_NV", GL_Z6Y10Z6CB10Z6A10Z6Y10Z6CR10Z6A10_4224_NV},
    {"GL_Z4Y12Z4CB12Z4Y12Z4CR12_422_NV", GL_Z4Y12Z4CB12Z4Y12Z4CR12_422_NV},
    {"GL_Z4Y12Z4CB12Z4A12Z4Y12Z4CR12Z4A12_4224_NV", GL_Z4Y12Z4CB12Z4A12Z4Y12Z4CR12Z4A12_4224_NV},
    {"GL_Z4Y12Z4CB12Z4CR12_444_NV", GL_Z4Y12Z4CB12Z4CR12_444_NV},
    {"GL_VIDEO_CAPTURE_FRAME_WIDTH_NV", GL_VIDEO_CAPTURE_FRAME_WIDTH_NV},
    {"GL_VIDEO_CAPTURE_FRAME_HEIGHT_NV", GL_VIDEO_CAPTURE_FRAME_HEIGHT_NV},
    {"GL_VIDEO_CAPTURE_FIELD_UPPER_HEIGHT_NV", GL_VIDEO_CAPTURE_FIELD_UPPER_HEIGHT_NV},
    {"GL_VIDEO_CAPTURE_FIELD_LOWER_HEIGHT_NV", GL_VIDEO_CAPTURE_FIELD_LOWER_HEIGHT_NV},
    {"GL_VIDEO_CAPTURE_SURFACE_ORIGIN_NV", GL_VIDEO_CAPTURE_SURFACE_ORIGIN_NV},
    {"GL_TEXTURE_COVERAGE_SAMPLES_NV", GL_TEXTURE_COVERAGE_SAMPLES_NV},
    {"GL_TEXTURE_COLOR_SAMPLES_NV", GL_TEXTURE_COLOR_SAMPLES_NV},
    {"GL_GPU_MEMORY_INFO_DEDICATED_VIDMEM_NVX", GL_GPU_MEMORY_INFO_DEDICATED_VIDMEM_NVX},
    {"GL_GPU_MEMORY_INFO_TOTAL_AVAILABLE_MEMORY_NVX", GL_GPU_MEMORY_INFO_TOTAL_AVAILABLE_MEMORY_NVX},
    {"GL_GPU_MEMORY_INFO_CURRENT_AVAILABLE_VIDMEM_NVX", GL_GPU_MEMORY_INFO_CURRENT_AVAILABLE_VIDMEM_NVX},
    {"GL_GPU_MEMORY_INFO_EVICTION_COUNT_NVX", GL_GPU_MEMORY_INFO_EVICTION_COUNT_NVX},
    {"GL_GPU_MEMORY_INFO_EVICTED_MEMORY_NVX", GL_GPU_MEMORY_INFO_EVICTED_MEMORY_NVX},
    {"GL_IMAGE_1D", GL_IMAGE_1D},
    {"GL_IMAGE_2D", GL_IMAGE_2D},
    {"GL_IMAGE_3D", GL_IMAGE_3D},
    {"GL_IMAGE_2D_RECT", GL_IMAGE_2D_RECT},
    {"GL_IMAGE_CUBE", GL_IMAGE_CUBE},
    {"GL_IMAGE_BUFFER", GL_IMAGE_BUFFER},
    {"GL_IMAGE_1D_ARRAY", GL_IMAGE_1D_ARRAY},
    {"GL_IMAGE_2D_ARRAY", GL_IMAGE_2D_ARRAY},
    {"GL_IMAGE_CUBE_MAP_ARRAY", GL_IMAGE_CUBE_MAP_ARRAY},
    {"GL_IMAGE_2D_MULTISAMPLE", GL_IMAGE_2D_MULTISAMPLE},
    {"GL_IMAGE_2D_MULTISAMPLE_ARRAY", GL_IMAGE_2D_MULTISAMPLE_ARRAY},
    {"GL_INT_IMAGE_1D", GL_INT_IMAGE_1D},
    {"GL_INT_IMAGE_2D", GL_INT_IMAGE_2D},
    {"GL_INT_IMAGE_3D", GL_INT_IMAGE_3D},
    {"GL_INT_IMAGE_2D_RECT", GL_INT_IMAGE_2D_RECT},
    {"GL_INT_IMAGE_CUBE", GL_INT_IMAGE_CUBE},
    {"GL_INT_IMAGE_BUFFER", GL_INT_IMAGE_BUFFER},
    {"GL_INT_IMAGE_1D_ARRAY", GL_INT_IMAGE_1D_ARRAY},
    {"GL_INT_IMAGE_2D_ARRAY", GL_INT_IMAGE_2D_ARRAY},
    {"GL_INT_IMAGE_CUBE_MAP_ARRAY", GL_INT_IMAGE_CUBE_MAP_ARRAY},
    {"GL_INT_IMAGE_2D_MULTISAMPLE", GL_INT_IMAGE_2D_MULTISAMPLE},
    {"GL_INT_IMAGE_2D_MULTISAMPLE_ARRAY", GL_INT_IMAGE_2D_MULTISAMPLE_ARRAY},
    {"GL_UNSIGNED_INT_IMAGE_1D", GL_UNSIGNED_INT_IMAGE_1D},
    {"GL_UNSIGNED_INT_IMAGE_2D", GL_UNSIGNED_INT_IMAGE_2D},
    {"GL_UNSIGNED_INT_IMAGE_3D", GL_UNSIGNED_INT_IMAGE_3D},
    {"GL_UNSIGNED_INT_IMAGE_2D_RECT", GL_UNSIGNED_INT_IMAGE_2D_RECT},
    {"GL_UNSIGNED_INT_IMAGE_CUBE", GL_UNSIGNED_INT_IMAGE_CUBE},
    {"GL_UNSIGNED_INT_IMAGE_BUFFER", GL_UNSIGNED_INT_IMAGE_BUFFER},
    {"GL_UNSIGNED_INT_IMAGE_1D_ARRAY", GL_UNSIGNED_INT_IMAGE_1D_ARRAY},
    {"GL_UNSIGNED_INT_IMAGE_2D_ARRAY", GL_UNSIGNED_INT_IMAGE_2D_ARRAY},
    {"GL_UNSIGNED_INT_IMAGE_CUBE_MAP_ARRAY", GL_UNSIGNED_INT_IMAGE_CUBE_MAP_ARRAY},
    {"GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE", GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE},
    {"GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE_ARRAY", GL_UNSIGNED_INT_IMAGE_2D_MULTISAMPLE_ARRAY},
    {"GL_MAX_IMAGE_SAMPLES", GL_MAX_IMAGE_SAMPLES},
    {"GL_IMAGE_BINDING_FORMAT", GL_IMAGE_BINDING_FORMAT},
    {"GL_RGB10_A2UI", GL_RGB10_A2UI},
    {"GL_PATH_FORMAT_SVG_NV", GL_PATH_FORMAT_SVG_NV},
    {"GL_PATH_FORMAT_PS_NV", GL_PATH_FORMAT_PS_NV},
    {"GL_STANDARD_FONT_NAME_NV", GL_STANDARD_FONT_NAME_NV},
    {"GL_SYSTEM_FONT_NAME_NV", GL_SYSTEM_FONT_NAME_NV},
    {"GL_FILE_NAME_NV", GL_FILE_NAME_NV},
    {"GL_PATH_STROKE_WIDTH_NV", GL_PATH_STROKE_WIDTH_NV},
    {"GL_PATH_END_CAPS_NV", GL_PATH_END_CAPS_NV},
    {"GL_PATH_INITIAL_END_CAP_NV", GL_PATH_INITIAL_END_CAP_NV},
    {"GL_PATH_TERMINAL_END_CAP_NV", GL_PATH_TERMINAL_END_CAP_NV},
    {"GL_PATH_JOIN_STYLE_NV", GL_PATH_JOIN_STYLE_NV},
    {"GL_PATH_MITER_LIMIT_NV", GL_PATH_MITER_LIMIT_NV},
    {"GL_PATH_DASH_CAPS_NV", GL_PATH_DASH_CAPS_NV},
    {"GL_PATH_INITIAL_DASH_CAP_NV", GL_PATH_INITIAL_DASH_CAP_NV},
    {"GL_PATH_TERMINAL_DASH_CAP_NV", GL_PATH_TERMINAL_DASH_CAP_NV},
    {"GL_PATH_DASH_OFFSET_NV", GL_PATH_DASH_OFFSET_NV},
    {"GL_PATH_CLIENT_LENGTH_NV", GL_PATH_CLIENT_LENGTH_NV},
    {"GL_PATH_FILL_MODE_NV", GL_PATH_FILL_MODE_NV},
    {"GL_PATH_FILL_MASK_NV", GL_PATH_FILL_MASK_NV},
    {"GL_PATH_FILL_COVER_MODE_NV", GL_PATH_FILL_COVER_MODE_NV},
    {"GL_PATH_STROKE_COVER_MODE_NV", GL_PATH_STROKE_COVER_MODE_NV},
    {"GL_PATH_STROKE_MASK_NV", GL_PATH_STROKE_MASK_NV},
    {"GL_PATH_SAMPLE_QUALITY_NV", GL_PATH_SAMPLE_QUALITY_NV},
    {"GL_PATH_STROKE_OVERSAMPLE_COUNT_NV", GL_PATH_STROKE_OVERSAMPLE_COUNT_NV},
    {"GL_COUNT_UP_NV", GL_COUNT_UP_NV},
    {"GL_COUNT_DOWN_NV", GL_COUNT_DOWN_NV},
    {"GL_PATH_OBJECT_BOUNDING_BOX_NV", GL_PATH_OBJECT_BOUNDING_BOX_NV},
    {"GL_CONVEX_HULL_NV", GL_CONVEX_HULL_NV},
    {"GL_MULTI_HULLS_NV", GL_MULTI_HULLS_NV},
    {"GL_BOUNDING_BOX_NV", GL_BOUNDING_BOX_NV},
    {"GL_TRANSLATE_X_NV", GL_TRANSLATE_X_NV},
    {"GL_TRANSLATE_Y_NV", GL_TRANSLATE_Y_NV},
    {"GL_TRANSLATE_2D_NV", GL_TRANSLATE_2D_NV},
    {"GL_TRANSLATE_3D_NV", GL_TRANSLATE_3D_NV},
    {"GL_AFFINE_2D_NV", GL_AFFINE_2D_NV},
    {"GL_PROJECTIVE_2D_NV", GL_PROJECTIVE_2D_NV},
    {"GL_AFFINE_3D_NV", GL_AFFINE_3D_NV},
    {"GL_PROJECTIVE_3D_NV", GL_PROJECTIVE_3D_NV},
    {"GL_TRANSPOSE_AFFINE_2D_NV", GL_TRANSPOSE_AFFINE_2D_NV},
    {"GL_TRANSPOSE_PROJECTIVE_2D_NV", GL_TRANSPOSE_PROJECTIVE_2D_NV},
    {"GL_TRANSPOSE_AFFINE_3D_NV", GL_TRANSPOSE_AFFINE_3D_NV},
    {"GL_TRANSPOSE_PROJECTIVE_3D_NV", GL_TRANSPOSE_PROJECTIVE_3D_NV},
    {"GL_UTF8_NV", GL_UTF8_NV},
    {"GL_UTF16_NV", GL_UTF16_NV},
    {"GL_BOUNDING_BOX_OF_BOUNDING_BOXES_NV", GL_BOUNDING_BOX_OF_BOUNDING_BOXES_NV},
    {"GL_PATH_COMMAND_COUNT_NV", GL_PATH_COMMAND_COUNT_NV},
    {"GL_PATH_COORD_COUNT_NV", GL_PATH_COORD_COUNT_NV},
    {"GL_PATH_DASH_ARRAY_COUNT_NV", GL_PATH_DASH_ARRAY_COUNT_NV},
    {"GL_PATH_COMPUTED_LENGTH_NV", GL_PATH_COMPUTED_LENGTH_NV},
    {"GL_PATH_FILL_BOUNDING_BOX_NV", GL_PATH_FILL_BOUNDING_BOX_NV},
    {"GL_PATH_STROKE_BOUNDING_BOX_NV", GL_PATH_STROKE_BOUNDING_BOX_NV},
    {"GL_SQUARE_NV", GL_SQUARE_NV},
    {"GL_ROUND_NV", GL_ROUND_NV},
    {"GL_TRIANGULAR_NV", GL_TRIANGULAR_NV},
    {"GL_BEVEL_NV", GL_BEVEL_NV},
    {"GL_MITER_REVERT_NV", GL_MITER_REVERT_NV},
    {"GL_MITER_TRUNCATE_NV", GL_MITER_TRUNCATE_NV},
    {"GL_SKIP_MISSING_GLYPH_NV", GL_SKIP_MISSING_GLYPH_NV},
    {"GL_USE_MISSING_GLYPH_NV", GL_USE_MISSING_GLYPH_NV},
    {"GL_PATH_ERROR_POSITION_NV", GL_PATH_ERROR_POSITION_NV},
    {"GL_PATH_FOG_GEN_MODE_NV", GL_PATH_FOG_GEN_MODE_NV},
    {"GL_ACCUM_ADJACENT_PAIRS_NV", GL_ACCUM_ADJACENT_PAIRS_NV},
    {"GL_ADJACENT_PAIRS_NV", GL_ADJACENT_PAIRS_NV},
    {"GL_FIRST_TO_REST_NV", GL_FIRST_TO_REST_NV},
    {"GL_PATH_GEN_MODE_NV", GL_PATH_GEN_MODE_NV},
    {"GL_PATH_GEN_COEFF_NV", GL_PATH_GEN_COEFF_NV},
    {"GL_PATH_GEN_COLOR_FORMAT_NV", GL_PATH_GEN_COLOR_FORMAT_NV},
    {"GL_PATH_GEN_COMPONENTS_NV", GL_PATH_GEN_COMPONENTS_NV},
    {"GL_PATH_DASH_OFFSET_RESET_NV", GL_PATH_DASH_OFFSET_RESET_NV},
    {"GL_MOVE_TO_RESETS_NV", GL_MOVE_TO_RESETS_NV},
    {"GL_MOVE_TO_CONTINUES_NV", GL_MOVE_TO_CONTINUES_NV},
    {"GL_PATH_STENCIL_FUNC_NV", GL_PATH_STENCIL_FUNC_NV},
    {"GL_PATH_STENCIL_REF_NV", GL_PATH_STENCIL_REF_NV},
    {"GL_PATH_STENCIL_VALUE_MASK_NV", GL_PATH_STENCIL_VALUE_MASK_NV},
    {"GL_SCALED_RESOLVE_FASTEST_EXT", GL_SCALED_RESOLVE_FASTEST_EXT},
    {"GL_SCALED_RESOLVE_NICEST_EXT", GL_SCALED_RESOLVE_NICEST_EXT},
    {"GL_MIN_MAP_BUFFER_ALIGNMENT", GL_MIN_MAP_BUFFER_ALIGNMENT},
    {"GL_PATH_STENCIL_DEPTH_OFFSET_FACTOR_NV", GL_PATH_STENCIL_DEPTH_OFFSET_FACTOR_NV},
    {"GL_PATH_STENCIL_DEPTH_OFFSET_UNITS_NV", GL_PATH_STENCIL_DEPTH_OFFSET_UNITS_NV},
    {"GL_PATH_COVER_DEPTH_FUNC_NV", GL_PATH_COVER_DEPTH_FUNC_NV},
    {"GL_IMAGE_FORMAT_COMPATIBILITY_TYPE", GL_IMAGE_FORMAT_COMPATIBILITY_TYPE},
    {"GL_IMAGE_FORMAT_COMPATIBILITY_BY_SIZE", GL_IMAGE_FORMAT_COMPATIBILITY_BY_SIZE},
    {"GL_IMAGE_FORMAT_COMPATIBILITY_BY_CLASS", GL_IMAGE_FORMAT_COMPATIBILITY_BY_CLASS},
    {"GL_MAX_VERTEX_IMAGE_UNIFORMS", GL_MAX_VERTEX_IMAGE_UNIFORMS},
    {"GL_MAX_TESS_CONTROL_IMAGE_UNIFORMS", GL_MAX_TESS_CONTROL_IMAGE_UNIFORMS},
    {"GL_MAX_TESS_EVALUATION_IMAGE_UNIFORMS", GL_MAX_TESS_EVALUATION_IMAGE_UNIFORMS},
    {"GL_MAX_GEOMETRY_IMAGE_UNIFORMS", GL_MAX_GEOMETRY_IMAGE_UNIFORMS},
    {"GL_MAX_FRAGMENT_IMAGE_UNIFORMS", GL_MAX_FRAGMENT_IMAGE_UNIFORMS},
    {"GL_MAX_COMBINED_IMAGE_UNIFORMS", GL_MAX_COMBINED_IMAGE_UNIFORMS},
    {"GL_MAX_DEEP_3D_TEXTURE_WIDTH_HEIGHT_NV", GL_MAX_DEEP_3D_TEXTURE_WIDTH_HEIGHT_NV},
    {"GL_MAX_DEEP_3D_TEXTURE_DEPTH_NV", GL_MAX_DEEP_3D_TEXTURE_DEPTH_NV},
    {"GL_SHADER_STORAGE_BUFFER", GL_SHADER_STORAGE_BUFFER},
    {"GL_SHADER_STORAGE_BUFFER_BINDING", GL_SHADER_STORAGE_BUFFER_BINDING},
    {"GL_SHADER_STORAGE_BUFFER_START", GL_SHADER_STORAGE_BUFFER_START},
    {"GL_SHADER_STORAGE_BUFFER_SIZE", GL_SHADER_STORAGE_BUFFER_SIZE},
    {"GL_MAX_VERTEX_SHADER_STORAGE_BLOCKS", GL_MAX_VERTEX_SHADER_STORAGE_BLOCKS},
    {"GL_MAX_GEOMETRY_SHADER_STORAGE_BLOCKS", GL_MAX_GEOMETRY_SHADER_STORAGE_BLOCKS},
    {"GL_MAX_TESS_CONTROL_SHADER_STORAGE_BLOCKS", GL_MAX_TESS_CONTROL_SHADER_STORAGE_BLOCKS},
    {"GL_MAX_TESS_EVALUATION_SHADER_STORAGE_BLOCKS", GL_MAX_TESS_EVALUATION_SHADER_STORAGE_BLOCKS},
    {"GL_MAX_FRAGMENT_SHADER_STORAGE_BLOCKS", GL_MAX_FRAGMENT_SHADER_STORAGE_BLOCKS},
    {"GL_MAX_COMPUTE_SHADER_STORAGE_BLOCKS", GL_MAX_COMPUTE_SHADER_STORAGE_BLOCKS},
    {"GL_MAX_COMBINED_SHADER_STORAGE_BLOCKS", GL_MAX_COMBINED_SHADER_STORAGE_BLOCKS},
    {"GL_MAX_SHADER_STORAGE_BUFFER_BINDINGS", GL_MAX_SHADER_STORAGE_BUFFER_BINDINGS},
    {"GL_MAX_SHADER_STORAGE_BLOCK_SIZE", GL_MAX_SHADER_STORAGE_BLOCK_SIZE},
    {"GL_SHADER_STORAGE_BUFFER_OFFSET_ALIGNMENT", GL_SHADER_STORAGE_BUFFER_OFFSET_ALIGNMENT},
    {"GL_SYNC_X11_FENCE_EXT", GL_SYNC_X11_FENCE_EXT},
    {"GL_DEPTH_STENCIL_TEXTURE_MODE", GL_DEPTH_STENCIL_TEXTURE_MODE},
    {"GL_MAX_COMPUTE_WORK_GROUP_INVOCATIONS", GL_MAX_COMPUTE_WORK_GROUP_INVOCATIONS},
    {"GL_UNIFORM_BLOCK_REFERENCED_BY_COMPUTE_SHADER", GL_UNIFORM_BLOCK_REFERENCED_BY_COMPUTE_SHADER},
    {"GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_COMPUTE_SHADER", GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_COMPUTE_SHADER},
    {"GL_DISPATCH_INDIRECT_BUFFER", GL_DISPATCH_INDIRECT_BUFFER},
    {"GL_DISPATCH_INDIRECT_BUFFER_BINDING", GL_DISPATCH_INDIRECT_BUFFER_BINDING},
    {"GL_COLOR_ATTACHMENT_EXT", GL_COLOR_ATTACHMENT_EXT},
    {"GL_MULTIVIEW_EXT", GL_MULTIVIEW_EXT},
    {"GL_MAX_MULTIVIEW_BUFFERS_EXT", GL_MAX_MULTIVIEW_BUFFERS_EXT},
    {"GL_CONTEXT_ROBUST_ACCESS", GL_CONTEXT_ROBUST_ACCESS},
    {"GL_COMPUTE_PROGRAM_NV", GL_COMPUTE_PROGRAM_NV},
    {"GL_COMPUTE_PROGRAM_PARAMETER_BUFFER_NV", GL_COMPUTE_PROGRAM_PARAMETER_BUFFER_NV},
    {"GL_TEXTURE_2D_MULTISAMPLE", GL_TEXTURE_2D_MULTISAMPLE},
    {"GL_PROXY_TEXTURE_2D_MULTISAMPLE", GL_PROXY_TEXTURE_2D_MULTISAMPLE},
    {"GL_TEXTURE_2D_MULTISAMPLE_ARRAY", GL_TEXTURE_2D_MULTISAMPLE_ARRAY},
    {"GL_PROXY_TEXTURE_2D_MULTISAMPLE_ARRAY", GL_PROXY_TEXTURE_2D_MULTISAMPLE_ARRAY},
    {"GL_TEXTURE_BINDING_2D_MULTISAMPLE", GL_TEXTURE_BINDING_2D_MULTISAMPLE},
    {"GL_TEXTURE_BINDING_2D_MULTISAMPLE_ARRAY", GL_TEXTURE_BINDING_2D_MULTISAMPLE_ARRAY},
    {"GL_TEXTURE_SAMPLES", GL_TEXTURE_SAMPLES},
    {"GL_TEXTURE_FIXED_SAMPLE_LOCATIONS", GL_TEXTURE_FIXED_SAMPLE_LOCATIONS},
    {"GL_SAMPLER_2D_MULTISAMPLE", GL_SAMPLER_2D_MULTISAMPLE},
    {"GL_INT_SAMPLER_2D_MULTISAMPLE", GL_INT_SAMPLER_2D_MULTISAMPLE},
    {"GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE", GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE},
    {"GL_SAMPLER_2D_MULTISAMPLE_ARRAY", GL_SAMPLER_2D_MULTISAMPLE_ARRAY},
    {"GL_INT_SAMPLER_2D_MULTISAMPLE_ARRAY", GL_INT_SAMPLER_2D_MULTISAMPLE_ARRAY},
    {"GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE_ARRAY", GL_UNSIGNED_INT_SAMPLER_2D_MULTISAMPLE_ARRAY},
    {"GL_MAX_COLOR_TEXTURE_SAMPLES", GL_MAX_COLOR_TEXTURE_SAMPLES},
    {"GL_MAX_DEPTH_TEXTURE_SAMPLES", GL_MAX_DEPTH_TEXTURE_SAMPLES},
    {"GL_MAX_INTEGER_SAMPLES", GL_MAX_INTEGER_SAMPLES},
    {"GL_MAX_SERVER_WAIT_TIMEOUT", GL_MAX_SERVER_WAIT_TIMEOUT},
    {"GL_OBJECT_TYPE", GL_OBJECT_TYPE},
    {"GL_SYNC_CONDITION", GL_SYNC_CONDITION},
    {"GL_SYNC_STATUS", GL_SYNC_STATUS},
    {"GL_SYNC_FLAGS", GL_SYNC_FLAGS},
    {"GL_SYNC_FENCE", GL_SYNC_FENCE},
    {"GL_SYNC_GPU_COMMANDS_COMPLETE", GL_SYNC_GPU_COMMANDS_COMPLETE},
    {"GL_UNSIGNALED", GL_UNSIGNALED},
    {"GL_SIGNALED", GL_SIGNALED},
    {"GL_ALREADY_SIGNALED", GL_ALREADY_SIGNALED},
    {"GL_TIMEOUT_EXPIRED", GL_TIMEOUT_EXPIRED},
    {"GL_CONDITION_SATISFIED", GL_CONDITION_SATISFIED},
    {"GL_WAIT_FAILED", GL_WAIT_FAILED},
    {"GL_BUFFER_ACCESS_FLAGS", GL_BUFFER_ACCESS_FLAGS},
    {"GL_BUFFER_MAP_LENGTH", GL_BUFFER_MAP_LENGTH},
    {"GL_BUFFER_MAP_OFFSET", GL_BUFFER_MAP_OFFSET},
    {"GL_MAX_VERTEX_OUTPUT_COMPONENTS", GL_MAX_VERTEX_OUTPUT_COMPONENTS},
    {"GL_MAX_GEOMETRY_INPUT_COMPONENTS", GL_MAX_GEOMETRY_INPUT_COMPONENTS},
    {"GL_MAX_GEOMETRY_OUTPUT_COMPONENTS", GL_MAX_GEOMETRY_OUTPUT_COMPONENTS},
    {"GL_MAX_FRAGMENT_INPUT_COMPONENTS", GL_MAX_FRAGMENT_INPUT_COMPONENTS},
    {"GL_CONTEXT_PROFILE_MASK", GL_CONTEXT_PROFILE_MASK},
    {"GL_UNPACK_COMPRESSED_BLOCK_WIDTH", GL_UNPACK_COMPRESSED_BLOCK_WIDTH},
    {"GL_UNPACK_COMPRESSED_BLOCK_HEIGHT", GL_UNPACK_COMPRESSED_BLOCK_HEIGHT},
    {"GL_UNPACK_COMPRESSED_BLOCK_DEPTH", GL_UNPACK_COMPRESSED_BLOCK_DEPTH},
    {"GL_UNPACK_COMPRESSED_BLOCK_SIZE", GL_UNPACK_COMPRESSED_BLOCK_SIZE},
    {"GL_PACK_COMPRESSED_BLOCK_WIDTH", GL_PACK_COMPRESSED_BLOCK_WIDTH},
    {"GL_PACK_COMPRESSED_BLOCK_HEIGHT", GL_PACK_COMPRESSED_BLOCK_HEIGHT},
    {"GL_PACK_COMPRESSED_BLOCK_DEPTH", GL_PACK_COMPRESSED_BLOCK_DEPTH},
    {"GL_PACK_COMPRESSED_BLOCK_SIZE", GL_PACK_COMPRESSED_BLOCK_SIZE},
    {"GL_TEXTURE_IMMUTABLE_FORMAT", GL_TEXTURE_IMMUTABLE_FORMAT},
    {"GL_SGX_PROGRAM_BINARY_IMG", GL_SGX_PROGRAM_BINARY_IMG},
    {"GL_RENDERBUFFER_SAMPLES_IMG", GL_RENDERBUFFER_SAMPLES_IMG},
    {"GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE_IMG", GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE_IMG},
    {"GL_MAX_SAMPLES_IMG", GL_MAX_SAMPLES_IMG},
    {"GL_TEXTURE_SAMPLES_IMG", GL_TEXTURE_SAMPLES_IMG},
    {"GL_COMPRESSED_RGBA_PVRTC_2BPPV2_IMG", GL_COMPRESSED_RGBA_PVRTC_2BPPV2_IMG},
    {"GL_COMPRESSED_RGBA_PVRTC_4BPPV2_IMG", GL_COMPRESSED_RGBA_PVRTC_4BPPV2_IMG},
    {"GL_CUBIC_IMG", GL_CUBIC_IMG},
    {"GL_CUBIC_MIPMAP_NEAREST_IMG", GL_CUBIC_MIPMAP_NEAREST_IMG},
    {"GL_CUBIC_MIPMAP_LINEAR_IMG", GL_CUBIC_MIPMAP_LINEAR_IMG},
    {"GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE_AND_DOWNSAMPLE_IMG", GL_FRAMEBUFFER_INCOMPLETE_MULTISAMPLE_AND_DOWNSAMPLE_IMG},
    {"GL_NUM_DOWNSAMPLE_SCALES_IMG", GL_NUM_DOWNSAMPLE_SCALES_IMG},
    {"GL_DOWNSAMPLE_SCALES_IMG", GL_DOWNSAMPLE_SCALES_IMG},
    {"GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_SCALE_IMG", GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_SCALE_IMG},
    {"GL_MAX_DEBUG_MESSAGE_LENGTH", GL_MAX_DEBUG_MESSAGE_LENGTH},
    {"GL_MAX_DEBUG_LOGGED_MESSAGES", GL_MAX_DEBUG_LOGGED_MESSAGES},
    {"GL_DEBUG_LOGGED_MESSAGES", GL_DEBUG_LOGGED_MESSAGES},
    {"GL_DEBUG_SEVERITY_HIGH", GL_DEBUG_SEVERITY_HIGH},
    {"GL_DEBUG_SEVERITY_MEDIUM", GL_DEBUG_SEVERITY_MEDIUM},
    {"GL_DEBUG_SEVERITY_LOW", GL_DEBUG_SEVERITY_LOW},
    {"GL_DEBUG_CATEGORY_API_ERROR_AMD", GL_DEBUG_CATEGORY_API_ERROR_AMD},
    {"GL_DEBUG_CATEGORY_WINDOW_SYSTEM_AMD", GL_DEBUG_CATEGORY_WINDOW_SYSTEM_AMD},
    {"GL_DEBUG_CATEGORY_DEPRECATION_AMD", GL_DEBUG_CATEGORY_DEPRECATION_AMD},
    {"GL_DEBUG_CATEGORY_UNDEFINED_BEHAVIOR_AMD", GL_DEBUG_CATEGORY_UNDEFINED_BEHAVIOR_AMD},
    {"GL_DEBUG_CATEGORY_PERFORMANCE_AMD", GL_DEBUG_CATEGORY_PERFORMANCE_AMD},
    {"GL_DEBUG_CATEGORY_SHADER_COMPILER_AMD", GL_DEBUG_CATEGORY_SHADER_COMPILER_AMD},
    {"GL_DEBUG_CATEGORY_APPLICATION_AMD", GL_DEBUG_CATEGORY_APPLICATION_AMD},
    {"GL_DEBUG_CATEGORY_OTHER_AMD", GL_DEBUG_CATEGORY_OTHER_AMD},
    {"GL_BUFFER_OBJECT_EXT", GL_BUFFER_OBJECT_EXT},
    {"GL_PERFORMANCE_MONITOR_AMD", GL_PERFORMANCE_MONITOR_AMD},
    {"GL_QUERY_OBJECT_EXT", GL_QUERY_OBJECT_EXT},
    {"GL_VERTEX_ARRAY_OBJECT_EXT", GL_VERTEX_ARRAY_OBJECT_EXT},
    {"GL_SAMPLER_OBJECT_AMD", GL_SAMPLER_OBJECT_AMD},
    {"GL_EXTERNAL_VIRTUAL_MEMORY_BUFFER_AMD", GL_EXTERNAL_VIRTUAL_MEMORY_BUFFER_AMD},
    {"GL_QUERY_BUFFER", GL_QUERY_BUFFER},
    {"GL_QUERY_BUFFER_BINDING", GL_QUERY_BUFFER_BINDING},
    {"GL_QUERY_RESULT_NO_WAIT", GL_QUERY_RESULT_NO_WAIT},
    {"GL_VIRTUAL_PAGE_SIZE_X_ARB", GL_VIRTUAL_PAGE_SIZE_X_ARB},
    {"GL_VIRTUAL_PAGE_SIZE_Y_ARB", GL_VIRTUAL_PAGE_SIZE_Y_ARB},
    {"GL_VIRTUAL_PAGE_SIZE_Z_ARB", GL_VIRTUAL_PAGE_SIZE_Z_ARB},
    {"GL_MAX_SPARSE_TEXTURE_SIZE_ARB", GL_MAX_SPARSE_TEXTURE_SIZE_ARB},
    {"GL_MAX_SPARSE_3D_TEXTURE_SIZE_ARB", GL_MAX_SPARSE_3D_TEXTURE_SIZE_ARB},
    {"GL_MAX_SPARSE_ARRAY_TEXTURE_LAYERS", GL_MAX_SPARSE_ARRAY_TEXTURE_LAYERS},
    {"GL_MIN_SPARSE_LEVEL_AMD", GL_MIN_SPARSE_LEVEL_AMD},
    {"GL_MIN_LOD_WARNING_AMD", GL_MIN_LOD_WARNING_AMD},
    {"GL_TEXTURE_BUFFER_OFFSET", GL_TEXTURE_BUFFER_OFFSET},
    {"GL_TEXTURE_BUFFER_SIZE", GL_TEXTURE_BUFFER_SIZE},
    {"GL_TEXTURE_BUFFER_OFFSET_ALIGNMENT", GL_TEXTURE_BUFFER_OFFSET_ALIGNMENT},
    {"GL_STREAM_RASTERIZATION_AMD", GL_STREAM_RASTERIZATION_AMD},
    {"GL_VERTEX_ELEMENT_SWIZZLE_AMD", GL_VERTEX_ELEMENT_SWIZZLE_AMD},
    {"GL_VERTEX_ID_SWIZZLE_AMD", GL_VERTEX_ID_SWIZZLE_AMD},
    {"GL_TEXTURE_SPARSE_ARB", GL_TEXTURE_SPARSE_ARB},
    {"GL_VIRTUAL_PAGE_SIZE_INDEX_ARB", GL_VIRTUAL_PAGE_SIZE_INDEX_ARB},
    {"GL_NUM_VIRTUAL_PAGE_SIZES_ARB", GL_NUM_VIRTUAL_PAGE_SIZES_ARB},
    {"GL_SPARSE_TEXTURE_FULL_ARRAY_CUBE_MIPMAPS_ARB", GL_SPARSE_TEXTURE_FULL_ARRAY_CUBE_MIPMAPS_ARB},
    {"GL_NUM_SPARSE_LEVELS_ARB", GL_NUM_SPARSE_LEVELS_ARB},
    {"GL_MAX_SHADER_COMPILER_THREADS_ARB", GL_MAX_SHADER_COMPILER_THREADS_ARB},
    {"GL_COMPLETION_STATUS_ARB", GL_COMPLETION_STATUS_ARB},
    {"GL_COMPUTE_SHADER", GL_COMPUTE_SHADER},
    {"GL_MAX_COMPUTE_UNIFORM_BLOCKS", GL_MAX_COMPUTE_UNIFORM_BLOCKS},
    {"GL_MAX_COMPUTE_TEXTURE_IMAGE_UNITS", GL_MAX_COMPUTE_TEXTURE_IMAGE_UNITS},
    {"GL_MAX_COMPUTE_IMAGE_UNIFORMS", GL_MAX_COMPUTE_IMAGE_UNIFORMS},
    {"GL_MAX_COMPUTE_WORK_GROUP_COUNT", GL_MAX_COMPUTE_WORK_GROUP_COUNT},
    {"GL_MAX_COMPUTE_WORK_GROUP_SIZE", GL_MAX_COMPUTE_WORK_GROUP_SIZE},
    {"GL_SHADER_BINARY_DMP", GL_SHADER_BINARY_DMP},
    {"GL_SMAPHS30_PROGRAM_BINARY_DMP", GL_SMAPHS30_PROGRAM_BINARY_DMP},
    {"GL_SMAPHS_PROGRAM_BINARY_DMP", GL_SMAPHS_PROGRAM_BINARY_DMP},
    {"GL_DMP_PROGRAM_BINARY_DMP", GL_DMP_PROGRAM_BINARY_DMP},
    {"GL_COMPRESSED_R11_EAC", GL_COMPRESSED_R11_EAC},
    {"GL_COMPRESSED_SIGNED_R11_EAC", GL_COMPRESSED_SIGNED_R11_EAC},
    {"GL_COMPRESSED_RG11_EAC", GL_COMPRESSED_RG11_EAC},
    {"GL_COMPRESSED_SIGNED_RG11_EAC", GL_COMPRESSED_SIGNED_RG11_EAC},
    {"GL_COMPRESSED_RGB8_ETC2", GL_COMPRESSED_RGB8_ETC2},
    {"GL_COMPRESSED_SRGB8_ETC2", GL_COMPRESSED_SRGB8_ETC2},
    {"GL_COMPRESSED_RGB8_PUNCHTHROUGH_ALPHA1_ETC2", GL_COMPRESSED_RGB8_PUNCHTHROUGH_ALPHA1_ETC2},
    {"GL_COMPRESSED_SRGB8_PUNCHTHROUGH_ALPHA1_ETC2", GL_COMPRESSED_SRGB8_PUNCHTHROUGH_ALPHA1_ETC2},
    {"GL_COMPRESSED_RGBA8_ETC2_EAC", GL_COMPRESSED_RGBA8_ETC2_EAC},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ETC2_EAC", GL_COMPRESSED_SRGB8_ALPHA8_ETC2_EAC},
    {"GL_BLEND_PREMULTIPLIED_SRC_NV", GL_BLEND_PREMULTIPLIED_SRC_NV},
    {"GL_BLEND_OVERLAP_NV", GL_BLEND_OVERLAP_NV},
    {"GL_UNCORRELATED_NV", GL_UNCORRELATED_NV},
    {"GL_DISJOINT_NV", GL_DISJOINT_NV},
    {"GL_CONJOINT_NV", GL_CONJOINT_NV},
    {"GL_BLEND_ADVANCED_COHERENT_KHR", GL_BLEND_ADVANCED_COHERENT_KHR},
    {"GL_SRC_NV", GL_SRC_NV},
    {"GL_DST_NV", GL_DST_NV},
    {"GL_SRC_OVER_NV", GL_SRC_OVER_NV},
    {"GL_DST_OVER_NV", GL_DST_OVER_NV},
    {"GL_SRC_IN_NV", GL_SRC_IN_NV},
    {"GL_DST_IN_NV", GL_DST_IN_NV},
    {"GL_SRC_OUT_NV", GL_SRC_OUT_NV},
    {"GL_DST_OUT_NV", GL_DST_OUT_NV},
    {"GL_SRC_ATOP_NV", GL_SRC_ATOP_NV},
    {"GL_DST_ATOP_NV", GL_DST_ATOP_NV},
    {"GL_PLUS_NV", GL_PLUS_NV},
    {"GL_PLUS_DARKER_NV", GL_PLUS_DARKER_NV},
    {"GL_MULTIPLY_KHR", GL_MULTIPLY_KHR},
    {"GL_SCREEN_KHR", GL_SCREEN_KHR},
    {"GL_OVERLAY_KHR", GL_OVERLAY_KHR},
    {"GL_DARKEN_KHR", GL_DARKEN_KHR},
    {"GL_LIGHTEN_KHR", GL_LIGHTEN_KHR},
    {"GL_COLORDODGE_KHR", GL_COLORDODGE_KHR},
    {"GL_COLORBURN_KHR", GL_COLORBURN_KHR},
    {"GL_HARDLIGHT_KHR", GL_HARDLIGHT_KHR},
    {"GL_SOFTLIGHT_KHR", GL_SOFTLIGHT_KHR},
    {"GL_DIFFERENCE_KHR", GL_DIFFERENCE_KHR},
    {"GL_MINUS_NV", GL_MINUS_NV},
    {"GL_EXCLUSION_KHR", GL_EXCLUSION_KHR},
    {"GL_CONTRAST_NV", GL_CONTRAST_NV},
    {"GL_INVERT_RGB_NV", GL_INVERT_RGB_NV},
    {"GL_LINEARDODGE_NV", GL_LINEARDODGE_NV},
    {"GL_LINEARBURN_NV", GL_LINEARBURN_NV},
    {"GL_VIVIDLIGHT_NV", GL_VIVIDLIGHT_NV},
    {"GL_LINEARLIGHT_NV", GL_LINEARLIGHT_NV},
    {"GL_PINLIGHT_NV", GL_PINLIGHT_NV},
    {"GL_HARDMIX_NV", GL_HARDMIX_NV},
    {"GL_HSL_HUE_KHR", GL_HSL_HUE_KHR},
    {"GL_HSL_SATURATION_KHR", GL_HSL_SATURATION_KHR},
    {"GL_HSL_COLOR_KHR", GL_HSL_COLOR_KHR},
    {"GL_HSL_LUMINOSITY_KHR", GL_HSL_LUMINOSITY_KHR},
    {"GL_PLUS_CLAMPED_NV", GL_PLUS_CLAMPED_NV},
    {"GL_PLUS_CLAMPED_ALPHA_NV", GL_PLUS_CLAMPED_ALPHA_NV},
    {"GL_MINUS_CLAMPED_NV", GL_MINUS_CLAMPED_NV},
    {"GL_INVERT_OVG_NV", GL_INVERT_OVG_NV},
    {"GL_PRIMITIVE_BOUNDING_BOX_EXT", GL_PRIMITIVE_BOUNDING_BOX_EXT},
    {"GL_ATOMIC_COUNTER_BUFFER", GL_ATOMIC_COUNTER_BUFFER},
    {"GL_ATOMIC_COUNTER_BUFFER_BINDING", GL_ATOMIC_COUNTER_BUFFER_BINDING},
    {"GL_ATOMIC_COUNTER_BUFFER_START", GL_ATOMIC_COUNTER_BUFFER_START},
    {"GL_ATOMIC_COUNTER_BUFFER_SIZE", GL_ATOMIC_COUNTER_BUFFER_SIZE},
    {"GL_ATOMIC_COUNTER_BUFFER_DATA_SIZE", GL_ATOMIC_COUNTER_BUFFER_DATA_SIZE},
    {"GL_ATOMIC_COUNTER_BUFFER_ACTIVE_ATOMIC_COUNTERS", GL_ATOMIC_COUNTER_BUFFER_ACTIVE_ATOMIC_COUNTERS},
    {"GL_ATOMIC_COUNTER_BUFFER_ACTIVE_ATOMIC_COUNTER_INDICES", GL_ATOMIC_COUNTER_BUFFER_ACTIVE_ATOMIC_COUNTER_INDICES},
    {"GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_VERTEX_SHADER", GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_VERTEX_SHADER},
    {"GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_TESS_CONTROL_SHADER", GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_TESS_CONTROL_SHADER},
    {"GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_TESS_EVALUATION_SHADER", GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_TESS_EVALUATION_SHADER},
    {"GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_GEOMETRY_SHADER", GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_GEOMETRY_SHADER},
    {"GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_FRAGMENT_SHADER", GL_ATOMIC_COUNTER_BUFFER_REFERENCED_BY_FRAGMENT_SHADER},
    {"GL_MAX_VERTEX_ATOMIC_COUNTER_BUFFERS", GL_MAX_VERTEX_ATOMIC_COUNTER_BUFFERS},
    {"GL_MAX_TESS_CONTROL_ATOMIC_COUNTER_BUFFERS", GL_MAX_TESS_CONTROL_ATOMIC_COUNTER_BUFFERS},
    {"GL_MAX_TESS_EVALUATION_ATOMIC_COUNTER_BUFFERS", GL_MAX_TESS_EVALUATION_ATOMIC_COUNTER_BUFFERS},
    {"GL_MAX_GEOMETRY_ATOMIC_COUNTER_BUFFERS", GL_MAX_GEOMETRY_ATOMIC_COUNTER_BUFFERS},
    {"GL_MAX_FRAGMENT_ATOMIC_COUNTER_BUFFERS", GL_MAX_FRAGMENT_ATOMIC_COUNTER_BUFFERS},
    {"GL_MAX_COMBINED_ATOMIC_COUNTER_BUFFERS", GL_MAX_COMBINED_ATOMIC_COUNTER_BUFFERS},
    {"GL_MAX_VERTEX_ATOMIC_COUNTERS", GL_MAX_VERTEX_ATOMIC_COUNTERS},
    {"GL_MAX_TESS_CONTROL_ATOMIC_COUNTERS", GL_MAX_TESS_CONTROL_ATOMIC_COUNTERS},
    {"GL_MAX_TESS_EVALUATION_ATOMIC_COUNTERS", GL_MAX_TESS_EVALUATION_ATOMIC_COUNTERS},
    {"GL_MAX_GEOMETRY_ATOMIC_COUNTERS", GL_MAX_GEOMETRY_ATOMIC_COUNTERS},
    {"GL_MAX_FRAGMENT_ATOMIC_COUNTERS", GL_MAX_FRAGMENT_ATOMIC_COUNTERS},
    {"GL_MAX_COMBINED_ATOMIC_COUNTERS", GL_MAX_COMBINED_ATOMIC_COUNTERS},
    {"GL_MAX_ATOMIC_COUNTER_BUFFER_SIZE", GL_MAX_ATOMIC_COUNTER_BUFFER_SIZE},
    {"GL_ACTIVE_ATOMIC_COUNTER_BUFFERS", GL_ACTIVE_ATOMIC_COUNTER_BUFFERS},
    {"GL_UNIFORM_ATOMIC_COUNTER_BUFFER_INDEX", GL_UNIFORM_ATOMIC_COUNTER_BUFFER_INDEX},
    {"GL_UNSIGNED_INT_ATOMIC_COUNTER", GL_UNSIGNED_INT_ATOMIC_COUNTER},
    {"GL_MAX_ATOMIC_COUNTER_BUFFER_BINDINGS", GL_MAX_ATOMIC_COUNTER_BUFFER_BINDINGS},
    {"GL_FRAGMENT_COVERAGE_TO_COLOR_NV", GL_FRAGMENT_COVERAGE_TO_COLOR_NV},
    {"GL_FRAGMENT_COVERAGE_COLOR_NV", GL_FRAGMENT_COVERAGE_COLOR_NV},
    {"GL_DEBUG_OUTPUT", GL_DEBUG_OUTPUT},
    {"GL_UNIFORM", GL_UNIFORM},
    {"GL_UNIFORM_BLOCK", GL_UNIFORM_BLOCK},
    {"GL_PROGRAM_INPUT", GL_PROGRAM_INPUT},
    {"GL_PROGRAM_OUTPUT", GL_PROGRAM_OUTPUT},
    {"GL_BUFFER_VARIABLE", GL_BUFFER_VARIABLE},
    {"GL_SHADER_STORAGE_BLOCK", GL_SHADER_STORAGE_BLOCK},
    {"GL_IS_PER_PATCH", GL_IS_PER_PATCH},
    {"GL_VERTEX_SUBROUTINE", GL_VERTEX_SUBROUTINE},
    {"GL_TESS_CONTROL_SUBROUTINE", GL_TESS_CONTROL_SUBROUTINE},
    {"GL_TESS_EVALUATION_SUBROUTINE", GL_TESS_EVALUATION_SUBROUTINE},
    {"GL_GEOMETRY_SUBROUTINE", GL_GEOMETRY_SUBROUTINE},
    {"GL_FRAGMENT_SUBROUTINE", GL_FRAGMENT_SUBROUTINE},
    {"GL_COMPUTE_SUBROUTINE", GL_COMPUTE_SUBROUTINE},
    {"GL_VERTEX_SUBROUTINE_UNIFORM", GL_VERTEX_SUBROUTINE_UNIFORM},
    {"GL_TESS_CONTROL_SUBROUTINE_UNIFORM", GL_TESS_CONTROL_SUBROUTINE_UNIFORM},
    {"GL_TESS_EVALUATION_SUBROUTINE_UNIFORM", GL_TESS_EVALUATION_SUBROUTINE_UNIFORM},
    {"GL_GEOMETRY_SUBROUTINE_UNIFORM", GL_GEOMETRY_SUBROUTINE_UNIFORM},
    {"GL_FRAGMENT_SUBROUTINE_UNIFORM", GL_FRAGMENT_SUBROUTINE_UNIFORM},
    {"GL_COMPUTE_SUBROUTINE_UNIFORM", GL_COMPUTE_SUBROUTINE_UNIFORM},
    {"GL_TRANSFORM_FEEDBACK_VARYING", GL_TRANSFORM_FEEDBACK_VARYING},
    {"GL_ACTIVE_RESOURCES", GL_ACTIVE_RESOURCES},
    {"GL_MAX_NAME_LENGTH", GL_MAX_NAME_LENGTH},
    {"GL_MAX_NUM_ACTIVE_VARIABLES", GL_MAX_NUM_ACTIVE_VARIABLES},
    {"GL_MAX_NUM_COMPATIBLE_SUBROUTINES", GL_MAX_NUM_COMPATIBLE_SUBROUTINES},
    {"GL_NAME_LENGTH", GL_NAME_LENGTH},
    {"GL_TYPE", GL_TYPE},
    {"GL_ARRAY_SIZE", GL_ARRAY_SIZE},
    {"GL_OFFSET", GL_OFFSET},
    {"GL_BLOCK_INDEX", GL_BLOCK_INDEX},
    {"GL_ARRAY_STRIDE", GL_ARRAY_STRIDE},
    {"GL_MATRIX_STRIDE", GL_MATRIX_STRIDE},
    {"GL_IS_ROW_MAJOR", GL_IS_ROW_MAJOR},
    {"GL_ATOMIC_COUNTER_BUFFER_INDEX", GL_ATOMIC_COUNTER_BUFFER_INDEX},
    {"GL_BUFFER_BINDING", GL_BUFFER_BINDING},
    {"GL_BUFFER_DATA_SIZE", GL_BUFFER_DATA_SIZE},
    {"GL_NUM_ACTIVE_VARIABLES", GL_NUM_ACTIVE_VARIABLES},
    {"GL_ACTIVE_VARIABLES", GL_ACTIVE_VARIABLES},
    {"GL_REFERENCED_BY_VERTEX_SHADER", GL_REFERENCED_BY_VERTEX_SHADER},
    {"GL_REFERENCED_BY_TESS_CONTROL_SHADER", GL_REFERENCED_BY_TESS_CONTROL_SHADER},
    {"GL_REFERENCED_BY_TESS_EVALUATION_SHADER", GL_REFERENCED_BY_TESS_EVALUATION_SHADER},
    {"GL_REFERENCED_BY_GEOMETRY_SHADER", GL_REFERENCED_BY_GEOMETRY_SHADER},
    {"GL_REFERENCED_BY_FRAGMENT_SHADER", GL_REFERENCED_BY_FRAGMENT_SHADER},
    {"GL_REFERENCED_BY_COMPUTE_SHADER", GL_REFERENCED_BY_COMPUTE_SHADER},
    {"GL_TOP_LEVEL_ARRAY_SIZE", GL_TOP_LEVEL_ARRAY_SIZE},
    {"GL_TOP_LEVEL_ARRAY_STRIDE", GL_TOP_LEVEL_ARRAY_STRIDE},
    {"GL_LOCATION", GL_LOCATION},
    {"GL_LOCATION_INDEX", GL_LOCATION_INDEX},
    {"GL_FRAMEBUFFER_DEFAULT_WIDTH", GL_FRAMEBUFFER_DEFAULT_WIDTH},
    {"GL_FRAMEBUFFER_DEFAULT_HEIGHT", GL_FRAMEBUFFER_DEFAULT_HEIGHT},
    {"GL_FRAMEBUFFER_DEFAULT_LAYERS", GL_FRAMEBUFFER_DEFAULT_LAYERS},
    {"GL_FRAMEBUFFER_DEFAULT_SAMPLES", GL_FRAMEBUFFER_DEFAULT_SAMPLES},
    {"GL_FRAMEBUFFER_DEFAULT_FIXED_SAMPLE_LOCATIONS", GL_FRAMEBUFFER_DEFAULT_FIXED_SAMPLE_LOCATIONS},
    {"GL_MAX_FRAMEBUFFER_WIDTH", GL_MAX_FRAMEBUFFER_WIDTH},
    {"GL_MAX_FRAMEBUFFER_HEIGHT", GL_MAX_FRAMEBUFFER_HEIGHT},
    {"GL_MAX_FRAMEBUFFER_LAYERS", GL_MAX_FRAMEBUFFER_LAYERS},
    {"GL_MAX_FRAMEBUFFER_SAMPLES", GL_MAX_FRAMEBUFFER_SAMPLES},
    {"GL_RASTER_MULTISAMPLE_EXT", GL_RASTER_MULTISAMPLE_EXT},
    {"GL_RASTER_SAMPLES_EXT", GL_RASTER_SAMPLES_EXT},
    {"GL_MAX_RASTER_SAMPLES_EXT", GL_MAX_RASTER_SAMPLES_EXT},
    {"GL_RASTER_FIXED_SAMPLE_LOCATIONS_EXT", GL_RASTER_FIXED_SAMPLE_LOCATIONS_EXT},
    {"GL_MULTISAMPLE_RASTERIZATION_ALLOWED_EXT", GL_MULTISAMPLE_RASTERIZATION_ALLOWED_EXT},
    {"GL_EFFECTIVE_RASTER_SAMPLES_EXT", GL_EFFECTIVE_RASTER_SAMPLES_EXT},
    {"GL_DEPTH_SAMPLES_NV", GL_DEPTH_SAMPLES_NV},
    {"GL_STENCIL_SAMPLES_NV", GL_STENCIL_SAMPLES_NV},
    {"GL_MIXED_DEPTH_SAMPLES_SUPPORTED_NV", GL_MIXED_DEPTH_SAMPLES_SUPPORTED_NV},
    {"GL_MIXED_STENCIL_SAMPLES_SUPPORTED_NV", GL_MIXED_STENCIL_SAMPLES_SUPPORTED_NV},
    {"GL_COVERAGE_MODULATION_TABLE_NV", GL_COVERAGE_MODULATION_TABLE_NV},
    {"GL_COVERAGE_MODULATION_NV", GL_COVERAGE_MODULATION_NV},
    {"GL_COVERAGE_MODULATION_TABLE_SIZE_NV", GL_COVERAGE_MODULATION_TABLE_SIZE_NV},
    {"GL_WARP_SIZE_NV", GL_WARP_SIZE_NV},
    {"GL_WARPS_PER_SM_NV", GL_WARPS_PER_SM_NV},
    {"GL_SM_COUNT_NV", GL_SM_COUNT_NV},
    {"GL_FILL_RECTANGLE_NV", GL_FILL_RECTANGLE_NV},
    {"GL_SAMPLE_LOCATION_SUBPIXEL_BITS_ARB", GL_SAMPLE_LOCATION_SUBPIXEL_BITS_ARB},
    {"GL_SAMPLE_LOCATION_PIXEL_GRID_WIDTH_ARB", GL_SAMPLE_LOCATION_PIXEL_GRID_WIDTH_ARB},
    {"GL_SAMPLE_LOCATION_PIXEL_GRID_HEIGHT_ARB", GL_SAMPLE_LOCATION_PIXEL_GRID_HEIGHT_ARB},
    {"GL_PROGRAMMABLE_SAMPLE_LOCATION_TABLE_SIZE_ARB", GL_PROGRAMMABLE_SAMPLE_LOCATION_TABLE_SIZE_ARB},
    {"GL_PROGRAMMABLE_SAMPLE_LOCATION_ARB", GL_PROGRAMMABLE_SAMPLE_LOCATION_ARB},
    {"GL_FRAMEBUFFER_PROGRAMMABLE_SAMPLE_LOCATIONS_ARB", GL_FRAMEBUFFER_PROGRAMMABLE_SAMPLE_LOCATIONS_ARB},
    {"GL_FRAMEBUFFER_SAMPLE_LOCATION_PIXEL_GRID_ARB", GL_FRAMEBUFFER_SAMPLE_LOCATION_PIXEL_GRID_ARB},
    {"GL_MAX_COMPUTE_VARIABLE_GROUP_INVOCATIONS_ARB", GL_MAX_COMPUTE_VARIABLE_GROUP_INVOCATIONS_ARB},
    {"GL_MAX_COMPUTE_VARIABLE_GROUP_SIZE_ARB", GL_MAX_COMPUTE_VARIABLE_GROUP_SIZE_ARB},
    {"GL_CONSERVATIVE_RASTERIZATION_NV", GL_CONSERVATIVE_RASTERIZATION_NV},
    {"GL_SUBPIXEL_PRECISION_BIAS_X_BITS_NV", GL_SUBPIXEL_PRECISION_BIAS_X_BITS_NV},
    {"GL_SUBPIXEL_PRECISION_BIAS_Y_BITS_NV", GL_SUBPIXEL_PRECISION_BIAS_Y_BITS_NV},
    {"GL_MAX_SUBPIXEL_PRECISION_BIAS_BITS_NV", GL_MAX_SUBPIXEL_PRECISION_BIAS_BITS_NV},
    {"GL_LOCATION_COMPONENT", GL_LOCATION_COMPONENT},
    {"GL_TRANSFORM_FEEDBACK_BUFFER_INDEX", GL_TRANSFORM_FEEDBACK_BUFFER_INDEX},
    {"GL_TRANSFORM_FEEDBACK_BUFFER_STRIDE", GL_TRANSFORM_FEEDBACK_BUFFER_STRIDE},
    {"GL_CLIP_ORIGIN", GL_CLIP_ORIGIN},
    {"GL_CLIP_DEPTH_MODE", GL_CLIP_DEPTH_MODE},
    {"GL_NEGATIVE_ONE_TO_ONE", GL_NEGATIVE_ONE_TO_ONE},
    {"GL_ZERO_TO_ONE", GL_ZERO_TO_ONE},
    {"GL_CLEAR_TEXTURE", GL_CLEAR_TEXTURE},
    {"GL_TEXTURE_REDUCTION_MODE_ARB", GL_TEXTURE_REDUCTION_MODE_ARB},
    {"GL_WEIGHTED_AVERAGE_ARB", GL_WEIGHTED_AVERAGE_ARB},
    {"GL_FONT_GLYPHS_AVAILABLE_NV", GL_FONT_GLYPHS_AVAILABLE_NV},
    {"GL_FONT_TARGET_UNAVAILABLE_NV", GL_FONT_TARGET_UNAVAILABLE_NV},
    {"GL_FONT_UNAVAILABLE_NV", GL_FONT_UNAVAILABLE_NV},
    {"GL_FONT_UNINTELLIGIBLE_NV", GL_FONT_UNINTELLIGIBLE_NV},
    {"GL_STANDARD_FONT_FORMAT_NV", GL_STANDARD_FONT_FORMAT_NV},
    {"GL_FRAGMENT_INPUT_NV", GL_FRAGMENT_INPUT_NV},
    {"GL_UNIFORM_BUFFER_UNIFIED_NV", GL_UNIFORM_BUFFER_UNIFIED_NV},
    {"GL_UNIFORM_BUFFER_ADDRESS_NV", GL_UNIFORM_BUFFER_ADDRESS_NV},
    {"GL_UNIFORM_BUFFER_LENGTH_NV", GL_UNIFORM_BUFFER_LENGTH_NV},
    {"GL_MULTISAMPLES_NV", GL_MULTISAMPLES_NV},
    {"GL_SUPERSAMPLE_SCALE_X_NV", GL_SUPERSAMPLE_SCALE_X_NV},
    {"GL_SUPERSAMPLE_SCALE_Y_NV", GL_SUPERSAMPLE_SCALE_Y_NV},
    {"GL_CONFORMANT_NV", GL_CONFORMANT_NV},
    {"GL_CONSERVATIVE_RASTER_DILATE_NV", GL_CONSERVATIVE_RASTER_DILATE_NV},
    {"GL_CONSERVATIVE_RASTER_DILATE_RANGE_NV", GL_CONSERVATIVE_RASTER_DILATE_RANGE_NV},
    {"GL_CONSERVATIVE_RASTER_DILATE_GRANULARITY_NV", GL_CONSERVATIVE_RASTER_DILATE_GRANULARITY_NV},
    {"GL_NUM_SAMPLE_COUNTS", GL_NUM_SAMPLE_COUNTS},
    {"GL_MULTISAMPLE_LINE_WIDTH_RANGE_ARB", GL_MULTISAMPLE_LINE_WIDTH_RANGE_ARB},
    {"GL_MULTISAMPLE_LINE_WIDTH_GRANULARITY_ARB", GL_MULTISAMPLE_LINE_WIDTH_GRANULARITY_ARB},
    {"GL_TRANSLATED_SHADER_SOURCE_LENGTH_ANGLE", GL_TRANSLATED_SHADER_SOURCE_LENGTH_ANGLE},
    {"GL_BGRA8_EXT", GL_BGRA8_EXT},
    {"GL_TEXTURE_USAGE_ANGLE", GL_TEXTURE_USAGE_ANGLE},
    {"GL_FRAMEBUFFER_ATTACHMENT_ANGLE", GL_FRAMEBUFFER_ATTACHMENT_ANGLE},
    {"GL_PACK_REVERSE_ROW_ORDER_ANGLE", GL_PACK_REVERSE_ROW_ORDER_ANGLE},
    {"GL_PROGRAM_BINARY_ANGLE", GL_PROGRAM_BINARY_ANGLE},
    {"GL_COMPRESSED_RGBA_ASTC_4x4_KHR", GL_COMPRESSED_RGBA_ASTC_4x4_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_5x4_KHR", GL_COMPRESSED_RGBA_ASTC_5x4_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_5x5_KHR", GL_COMPRESSED_RGBA_ASTC_5x5_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_6x5_KHR", GL_COMPRESSED_RGBA_ASTC_6x5_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_6x6_KHR", GL_COMPRESSED_RGBA_ASTC_6x6_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_8x5_KHR", GL_COMPRESSED_RGBA_ASTC_8x5_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_8x6_KHR", GL_COMPRESSED_RGBA_ASTC_8x6_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_8x8_KHR", GL_COMPRESSED_RGBA_ASTC_8x8_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_10x5_KHR", GL_COMPRESSED_RGBA_ASTC_10x5_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_10x6_KHR", GL_COMPRESSED_RGBA_ASTC_10x6_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_10x8_KHR", GL_COMPRESSED_RGBA_ASTC_10x8_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_10x10_KHR", GL_COMPRESSED_RGBA_ASTC_10x10_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_12x10_KHR", GL_COMPRESSED_RGBA_ASTC_12x10_KHR},
    {"GL_COMPRESSED_RGBA_ASTC_12x12_KHR", GL_COMPRESSED_RGBA_ASTC_12x12_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_4x4_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_4x4_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_5x4_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_5x4_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_5x5_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_5x5_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_6x5_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_6x5_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_6x6_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_6x6_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x5_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x5_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x6_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x6_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x8_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_8x8_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x5_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x5_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x6_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x6_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x8_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x8_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x10_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_10x10_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_12x10_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_12x10_KHR},
    {"GL_COMPRESSED_SRGB8_ALPHA8_ASTC_12x12_KHR", GL_COMPRESSED_SRGB8_ALPHA8_ASTC_12x12_KHR},
    {"GL_COMPRESSED_SRGB_ALPHA_PVRTC_2BPPV2_IMG", GL_COMPRESSED_SRGB_ALPHA_PVRTC_2BPPV2_IMG},
    {"GL_COMPRESSED_SRGB_ALPHA_PVRTC_4BPPV2_IMG", GL_COMPRESSED_SRGB_ALPHA_PVRTC_4BPPV2_IMG},
    {"GL_PERFQUERY_COUNTER_EVENT_INTEL", GL_PERFQUERY_COUNTER_EVENT_INTEL},
    {"GL_PERFQUERY_COUNTER_DURATION_NORM_INTEL", GL_PERFQUERY_COUNTER_DURATION_NORM_INTEL},
    {"GL_PERFQUERY_COUNTER_DURATION_RAW_INTEL", GL_PERFQUERY_COUNTER_DURATION_RAW_INTEL},
    {"GL_PERFQUERY_COUNTER_THROUGHPUT_INTEL", GL_PERFQUERY_COUNTER_THROUGHPUT_INTEL},
    {"GL_PERFQUERY_COUNTER_RAW_INTEL", GL_PERFQUERY_COUNTER_RAW_INTEL},
    {"GL_PERFQUERY_COUNTER_TIMESTAMP_INTEL", GL_PERFQUERY_COUNTER_TIMESTAMP_INTEL},
    {"GL_PERFQUERY_COUNTER_DATA_UINT32_INTEL", GL_PERFQUERY_COUNTER_DATA_UINT32_INTEL},
    {"GL_PERFQUERY_COUNTER_DATA_UINT64_INTEL", GL_PERFQUERY_COUNTER_DATA_UINT64_INTEL},
    {"GL_PERFQUERY_COUNTER_DATA_FLOAT_INTEL", GL_PERFQUERY_COUNTER_DATA_FLOAT_INTEL},
    {"GL_PERFQUERY_COUNTER_DATA_DOUBLE_INTEL", GL_PERFQUERY_COUNTER_DATA_DOUBLE_INTEL},
    {"GL_PERFQUERY_COUNTER_DATA_BOOL32_INTEL", GL_PERFQUERY_COUNTER_DATA_BOOL32_INTEL},
    {"GL_PERFQUERY_QUERY_NAME_LENGTH_MAX_INTEL", GL_PERFQUERY_QUERY_NAME_LENGTH_MAX_INTEL},
    {"GL_PERFQUERY_COUNTER_NAME_LENGTH_MAX_INTEL", GL_PERFQUERY_COUNTER_NAME_LENGTH_MAX_INTEL},
    {"GL_PERFQUERY_COUNTER_DESC_LENGTH_MAX_INTEL", GL_PERFQUERY_COUNTER_DESC_LENGTH_MAX_INTEL},
    {"GL_PERFQUERY_GPA_EXTENDED_COUNTERS_INTEL", GL_PERFQUERY_GPA_EXTENDED_COUNTERS_INTEL},
    {"GL_SHADER_BINARY_FORMAT_SPIR_V", GL_SHADER_BINARY_FORMAT_SPIR_V},
    {"GL_SPIR_V_BINARY", GL_SPIR_V_BINARY},
    {"GL_SPIR_V_EXTENSIONS", GL_SPIR_V_EXTENSIONS},
    {"GL_NUM_SPIR_V_EXTENSIONS", GL_NUM_SPIR_V_EXTENSIONS},
    {"GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_NUM_VIEWS_OVR", GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_NUM_VIEWS_OVR},
    {"GL_MAX_VIEWS_OVR", GL_MAX_VIEWS_OVR},
    {"GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_BASE_VIEW_INDEX_OVR", GL_FRAMEBUFFER_ATTACHMENT_TEXTURE_BASE_VIEW_INDEX_OVR},
    {"GL_FRAMEBUFFER_INCOMPLETE_VIEW_TARGETS_OVR", GL_FRAMEBUFFER_INCOMPLETE_VIEW_TARGETS_OVR},
    {"GL_MAX_SHADER_COMBINED_LOCAL_STORAGE_FAST_SIZE_EXT", GL_MAX_SHADER_COMBINED_LOCAL_STORAGE_FAST_SIZE_EXT},
    {"GL_MAX_SHADER_COMBINED_LOCAL_STORAGE_SIZE_EXT", GL_MAX_SHADER_COMBINED_LOCAL_STORAGE_SIZE_EXT},
    {"GL_FRAMEBUFFER_INCOMPLETE_INSUFFICIENT_SHADER_COMBINED_LOCAL_STORAGE_EXT", GL_FRAMEBUFFER_INCOMPLETE_INSUFFICIENT_SHADER_COMBINED_LOCAL_STORAGE_EXT},
    {"GL_SURFACE_COMPRESSION_EXT", GL_SURFACE_COMPRESSION_EXT},
    {"GL_RASTER_POSITION_UNCLIPPED_IBM", GL_RASTER_POSITION_UNCLIPPED_IBM},
    {"GL_CULL_VERTEX_IBM", GL_CULL_VERTEX_IBM},
    {"GL_ALL_STATIC_DATA_IBM", GL_ALL_STATIC_DATA_IBM},
    {"GL_STATIC_VERTEX_ARRAY_IBM", GL_STATIC_VERTEX_ARRAY_IBM},
    {"GL_VERTEX_ARRAY_LIST_IBM", GL_VERTEX_ARRAY_LIST_IBM},
    {"GL_NORMAL_ARRAY_LIST_IBM", GL_NORMAL_ARRAY_LIST_IBM},
    {"GL_COLOR_ARRAY_LIST_IBM", GL_COLOR_ARRAY_LIST_IBM},
    {"GL_INDEX_ARRAY_LIST_IBM", GL_INDEX_ARRAY_LIST_IBM},
    {"GL_TEXTURE_COORD_ARRAY_LIST_IBM", GL_TEXTURE_COORD_ARRAY_LIST_IBM},
    {"GL_EDGE_FLAG_ARRAY_LIST_IBM", GL_EDGE_FLAG_ARRAY_LIST_IBM},
    {"GL_FOG_COORDINATE_ARRAY_LIST_IBM", GL_FOG_COORDINATE_ARRAY_LIST_IBM},
    {"GL_SECONDARY_COLOR_ARRAY_LIST_IBM", GL_SECONDARY_COLOR_ARRAY_LIST_IBM},
    {"GL_VERTEX_ARRAY_LIST_STRIDE_IBM", GL_VERTEX_ARRAY_LIST_STRIDE_IBM},
    {"GL_NORMAL_ARRAY_LIST_STRIDE_IBM", GL_NORMAL_ARRAY_LIST_STRIDE_IBM},
    {"GL_COLOR_ARRAY_LIST_STRIDE_IBM", GL_COLOR_ARRAY_LIST_STRIDE_IBM},
    {"GL_INDEX_ARRAY_LIST_STRIDE_IBM", GL_INDEX_ARRAY_LIST_STRIDE_IBM},
    {"GL_TEXTURE_COORD_ARRAY_LIST_STRIDE_IBM", GL_TEXTURE_COORD_ARRAY_LIST_STRIDE_IBM},
    {"GL_EDGE_FLAG_ARRAY_LIST_STRIDE_IBM", GL_EDGE_FLAG_ARRAY_LIST_STRIDE_IBM},
    {"GL_FOG_COORDINATE_ARRAY_LIST_STRIDE_IBM", GL_FOG_COORDINATE_ARRAY_LIST_STRIDE_IBM},
    {"GL_SECONDARY_COLOR_ARRAY_LIST_STRIDE_IBM", GL_SECONDARY_COLOR_ARRAY_LIST_STRIDE_IBM},
    {"GL_PREFER_DOUBLEBUFFER_HINT_PGI", GL_PREFER_DOUBLEBUFFER_HINT_PGI},
    {"GL_CONSERVE_MEMORY_HINT_PGI", GL_CONSERVE_MEMORY_HINT_PGI},
    {"GL_RECLAIM_MEMORY_HINT_PGI", GL_RECLAIM_MEMORY_HINT_PGI},
    {"GL_NATIVE_GRAPHICS_HANDLE_PGI", GL_NATIVE_GRAPHICS_HANDLE_PGI},
    {"GL_NATIVE_GRAPHICS_BEGIN_HINT_PGI", GL_NATIVE_GRAPHICS_BEGIN_HINT_PGI},
    {"GL_NATIVE_GRAPHICS_END_HINT_PGI", GL_NATIVE_GRAPHICS_END_HINT_PGI},
    {"GL_ALWAYS_FAST_HINT_PGI", GL_ALWAYS_FAST_HINT_PGI},
    {"GL_ALWAYS_SOFT_HINT_PGI", GL_ALWAYS_SOFT_HINT_PGI},
    {"GL_ALLOW_DRAW_OBJ_HINT_PGI", GL_ALLOW_DRAW_OBJ_HINT_PGI},
    {"GL_ALLOW_DRAW_WIN_HINT_PGI", GL_ALLOW_DRAW_WIN_HINT_PGI},
    {"GL_ALLOW_DRAW_FRG_HINT_PGI", GL_ALLOW_DRAW_FRG_HINT_PGI},
    {"GL_ALLOW_DRAW_MEM_HINT_PGI", GL_ALLOW_DRAW_MEM_HINT_PGI},
    {"GL_STRICT_DEPTHFUNC_HINT_PGI", GL_STRICT_DEPTHFUNC_HINT_PGI},
    {"GL_STRICT_LIGHTING_HINT_PGI", GL_STRICT_LIGHTING_HINT_PGI},
    {"GL_STRICT_SCISSOR_HINT_PGI", GL_STRICT_SCISSOR_HINT_PGI},
    {"GL_FULL_STIPPLE_HINT_PGI", GL_FULL_STIPPLE_HINT_PGI},
    {"GL_CLIP_NEAR_HINT_PGI", GL_CLIP_NEAR_HINT_PGI},
    {"GL_CLIP_FAR_HINT_PGI", GL_CLIP_FAR_HINT_PGI},
    {"GL_WIDE_LINE_HINT_PGI", GL_WIDE_LINE_HINT_PGI},
    {"GL_BACK_NORMALS_HINT_PGI", GL_BACK_NORMALS_HINT_PGI},
    {"GL_VERTEX_DATA_HINT_PGI", GL_VERTEX_DATA_HINT_PGI},
    {"GL_VERTEX_CONSISTENT_HINT_PGI", GL_VERTEX_CONSISTENT_HINT_PGI},
    {"GL_MATERIAL_SIDE_HINT_PGI", GL_MATERIAL_SIDE_HINT_PGI},
    {"GL_MAX_VERTEX_HINT_PGI", GL_MAX_VERTEX_HINT_PGI},
    {"GL_INVALID_INDEX", GL_INVALID_INDEX},
};

static const trace::EnumSig _enumGLenum_sig = {
    1, 3522, _enumGLenum_values
};

static const trace::BitmaskFlag _bitmaskGLbitfield1_flags[] = {
    {"GL_ALL_ATTRIB_BITS", (unsigned long long)GL_ALL_ATTRIB_BITS},
    {"GL_CURRENT_BIT", (unsigned long long)GL_CURRENT_BIT},
    {"GL_POINT_BIT", (unsigned long long)GL_POINT_BIT},
    {"GL_LINE_BIT", (unsigned long long)GL_LINE_BIT},
    {"GL_POLYGON_BIT", (unsigned long long)GL_POLYGON_BIT},
    {"GL_POLYGON_STIPPLE_BIT", (unsigned long long)GL_POLYGON_STIPPLE_BIT},
    {"GL_PIXEL_MODE_BIT", (unsigned long long)GL_PIXEL_MODE_BIT},
    {"GL_LIGHTING_BIT", (unsigned long long)GL_LIGHTING_BIT},
    {"GL_FOG_BIT", (unsigned long long)GL_FOG_BIT},
    {"GL_DEPTH_BUFFER_BIT", (unsigned long long)GL_DEPTH_BUFFER_BIT},
    {"GL_ACCUM_BUFFER_BIT", (unsigned long long)GL_ACCUM_BUFFER_BIT},
    {"GL_STENCIL_BUFFER_BIT", (unsigned long long)GL_STENCIL_BUFFER_BIT},
    {"GL_VIEWPORT_BIT", (unsigned long long)GL_VIEWPORT_BIT},
    {"GL_TRANSFORM_BIT", (unsigned long long)GL_TRANSFORM_BIT},
    {"GL_ENABLE_BIT", (unsigned long long)GL_ENABLE_BIT},
    {"GL_COLOR_BUFFER_BIT", (unsigned long long)GL_COLOR_BUFFER_BIT},
    {"GL_HINT_BIT", (unsigned long long)GL_HINT_BIT},
    {"GL_EVAL_BIT", (unsigned long long)GL_EVAL_BIT},
    {"GL_LIST_BIT", (unsigned long long)GL_LIST_BIT},
    {"GL_TEXTURE_BIT", (unsigned long long)GL_TEXTURE_BIT},
    {"GL_SCISSOR_BIT", (unsigned long long)GL_SCISSOR_BIT},
    {"GL_MULTISAMPLE_BIT", (unsigned long long)GL_MULTISAMPLE_BIT},
};

static const trace::BitmaskSig _bitmaskGLbitfield1_sig = {
    0, 22, _bitmaskGLbitfield1_flags
};

static const trace::EnumValue _enumGLenum2_values[] = {
    {"GL_NO_ERROR", GL_NO_ERROR},
    {"GL_INVALID_ENUM", GL_INVALID_ENUM},
    {"GL_INVALID_VALUE", GL_INVALID_VALUE},
    {"GL_INVALID_OPERATION", GL_INVALID_OPERATION},
    {"GL_STACK_OVERFLOW", GL_STACK_OVERFLOW},
    {"GL_STACK_UNDERFLOW", GL_STACK_UNDERFLOW},
    {"GL_OUT_OF_MEMORY", GL_OUT_OF_MEMORY},
    {"GL_INVALID_FRAMEBUFFER_OPERATION", GL_INVALID_FRAMEBUFFER_OPERATION},
    {"GL_TABLE_TOO_LARGE", GL_TABLE_TOO_LARGE},
};

static const trace::EnumSig _enumGLenum2_sig = {
    3, 9, _enumGLenum2_values
};

static const trace::EnumValue _enumGLenum1_values[] = {
    {"GL_POINTS", GL_POINTS},
    {"GL_LINES", GL_LINES},
    {"GL_LINE_LOOP", GL_LINE_LOOP},
    {"GL_LINE_STRIP", GL_LINE_STRIP},
    {"GL_TRIANGLES", GL_TRIANGLES},
    {"GL_TRIANGLE_STRIP", GL_TRIANGLE_STRIP},
    {"GL_TRIANGLE_FAN", GL_TRIANGLE_FAN},
    {"GL_QUADS", GL_QUADS},
    {"GL_QUAD_STRIP", GL_QUAD_STRIP},
    {"GL_POLYGON", GL_POLYGON},
    {"GL_LINES_ADJACENCY", GL_LINES_ADJACENCY},
    {"GL_LINE_STRIP_ADJACENCY", GL_LINE_STRIP_ADJACENCY},
    {"GL_TRIANGLES_ADJACENCY", GL_TRIANGLES_ADJACENCY},
    {"GL_TRIANGLE_STRIP_ADJACENCY", GL_TRIANGLE_STRIP_ADJACENCY},
    {"GL_PATCHES", GL_PATCHES},
};

static const trace::EnumSig _enumGLenum1_sig = {
    2, 15, _enumGLenum1_values
};

static const trace::EnumValue _enumGLint2_values[] = {
    {"GL_BGRA", GL_BGRA},
};

static const trace::EnumSig _enumGLint2_sig = {
    5, 1, _enumGLint2_values
};

static const trace::BitmaskFlag _bitmaskGLbitfield2_flags[] = {
    {"GL_CLIENT_ALL_ATTRIB_BITS", (unsigned long long)GL_CLIENT_ALL_ATTRIB_BITS},
    {"GL_CLIENT_PIXEL_STORE_BIT", (unsigned long long)GL_CLIENT_PIXEL_STORE_BIT},
    {"GL_CLIENT_VERTEX_ARRAY_BIT", (unsigned long long)GL_CLIENT_VERTEX_ARRAY_BIT},
};

static const trace::BitmaskSig _bitmaskGLbitfield2_sig = {
    1, 3, _bitmaskGLbitfield2_flags
};

static void _write__GLvoid3(int selector, GLvoid * const & value) {
    switch (selector) {
    default:
    trace::localWriter.writePointer((uintptr_t)(GLvoid *)(value));
        break;
    case GL_COUNTER_TYPE_AMD:
    if ((GLenum *)(value)) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumGLenum_sig, *(GLenum *)(value));
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GL_PERCENTAGE_AMD:
    if ((float *)(value)) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeFloat(*(float *)(value));
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    case GL_COUNTER_RANGE_AMD:
    if ((float *)(value)) {
        size_t _cfloat = 2 > 0 ? 2 : 0;
        trace::localWriter.beginArray(_cfloat);
        for (size_t _ifloat = 0; _ifloat < _cfloat; ++_ifloat) {
            trace::localWriter.beginElement();
    trace::localWriter.writeFloat(((float *)(value))[_ifloat]);
            trace::localWriter.endElement();
        }
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
        break;
    }
}

static const trace::BitmaskFlag _bitmaskGLbitfield10_flags[] = {
    {"GL_TEXTURE_STORAGE_SPARSE_BIT_AMD", (unsigned long long)GL_TEXTURE_STORAGE_SPARSE_BIT_AMD},
};

static const trace::BitmaskSig _bitmaskGLbitfield10_sig = {
    9, 1, _bitmaskGLbitfield10_flags
};

static const trace::BitmaskFlag _bitmaskGLbitfield6_flags[] = {
    {"GL_SYNC_FLUSH_COMMANDS_BIT", (unsigned long long)GL_SYNC_FLUSH_COMMANDS_BIT},
};

static const trace::BitmaskSig _bitmaskGLbitfield6_sig = {
    5, 1, _bitmaskGLbitfield6_flags
};

static const trace::BitmaskFlag _bitmaskGLbitfield7_flags[] = {
    {"GL_ALL_BARRIER_BITS", (unsigned long long)GL_ALL_BARRIER_BITS},
    {"GL_VERTEX_ATTRIB_ARRAY_BARRIER_BIT", (unsigned long long)GL_VERTEX_ATTRIB_ARRAY_BARRIER_BIT},
    {"GL_ELEMENT_ARRAY_BARRIER_BIT", (unsigned long long)GL_ELEMENT_ARRAY_BARRIER_BIT},
    {"GL_UNIFORM_BARRIER_BIT", (unsigned long long)GL_UNIFORM_BARRIER_BIT},
    {"GL_TEXTURE_FETCH_BARRIER_BIT", (unsigned long long)GL_TEXTURE_FETCH_BARRIER_BIT},
    {"GL_SHADER_GLOBAL_ACCESS_BARRIER_BIT_NV", (unsigned long long)GL_SHADER_GLOBAL_ACCESS_BARRIER_BIT_NV},
    {"GL_SHADER_IMAGE_ACCESS_BARRIER_BIT", (unsigned long long)GL_SHADER_IMAGE_ACCESS_BARRIER_BIT},
    {"GL_COMMAND_BARRIER_BIT", (unsigned long long)GL_COMMAND_BARRIER_BIT},
    {"GL_PIXEL_BUFFER_BARRIER_BIT", (unsigned long long)GL_PIXEL_BUFFER_BARRIER_BIT},
    {"GL_TEXTURE_UPDATE_BARRIER_BIT", (unsigned long long)GL_TEXTURE_UPDATE_BARRIER_BIT},
    {"GL_BUFFER_UPDATE_BARRIER_BIT", (unsigned long long)GL_BUFFER_UPDATE_BARRIER_BIT},
    {"GL_FRAMEBUFFER_BARRIER_BIT", (unsigned long long)GL_FRAMEBUFFER_BARRIER_BIT},
    {"GL_TRANSFORM_FEEDBACK_BARRIER_BIT", (unsigned long long)GL_TRANSFORM_FEEDBACK_BARRIER_BIT},
    {"GL_ATOMIC_COUNTER_BARRIER_BIT", (unsigned long long)GL_ATOMIC_COUNTER_BARRIER_BIT},
    {"GL_SHADER_STORAGE_BARRIER_BIT", (unsigned long long)GL_SHADER_STORAGE_BARRIER_BIT},
    {"GL_CLIENT_MAPPED_BUFFER_BARRIER_BIT", (unsigned long long)GL_CLIENT_MAPPED_BUFFER_BARRIER_BIT},
    {"GL_QUERY_BUFFER_BARRIER_BIT", (unsigned long long)GL_QUERY_BUFFER_BARRIER_BIT},
};

static const trace::BitmaskSig _bitmaskGLbitfield7_sig = {
    6, 17, _bitmaskGLbitfield7_flags
};

static const trace::BitmaskFlag _bitmaskGLbitfield5_flags[] = {
    {"GL_MAP_READ_BIT", (unsigned long long)GL_MAP_READ_BIT},
    {"GL_MAP_WRITE_BIT", (unsigned long long)GL_MAP_WRITE_BIT},
    {"GL_MAP_PERSISTENT_BIT", (unsigned long long)GL_MAP_PERSISTENT_BIT},
    {"GL_MAP_COHERENT_BIT", (unsigned long long)GL_MAP_COHERENT_BIT},
    {"GL_DYNAMIC_STORAGE_BIT", (unsigned long long)GL_DYNAMIC_STORAGE_BIT},
    {"GL_CLIENT_STORAGE_BIT", (unsigned long long)GL_CLIENT_STORAGE_BIT},
};

static const trace::BitmaskSig _bitmaskGLbitfield5_sig = {
    4, 6, _bitmaskGLbitfield5_flags
};

static const trace::BitmaskFlag _bitmaskGLbitfield4_flags[] = {
    {"GL_MAP_READ_BIT", (unsigned long long)GL_MAP_READ_BIT},
    {"GL_MAP_WRITE_BIT", (unsigned long long)GL_MAP_WRITE_BIT},
    {"GL_MAP_INVALIDATE_RANGE_BIT", (unsigned long long)GL_MAP_INVALIDATE_RANGE_BIT},
    {"GL_MAP_INVALIDATE_BUFFER_BIT", (unsigned long long)GL_MAP_INVALIDATE_BUFFER_BIT},
    {"GL_MAP_FLUSH_EXPLICIT_BIT", (unsigned long long)GL_MAP_FLUSH_EXPLICIT_BIT},
    {"GL_MAP_UNSYNCHRONIZED_BIT", (unsigned long long)GL_MAP_UNSYNCHRONIZED_BIT},
    {"GL_MAP_PERSISTENT_BIT", (unsigned long long)GL_MAP_PERSISTENT_BIT},
    {"GL_MAP_COHERENT_BIT", (unsigned long long)GL_MAP_COHERENT_BIT},
};

static const trace::BitmaskSig _bitmaskGLbitfield4_sig = {
    3, 8, _bitmaskGLbitfield4_flags
};

static const trace::BitmaskFlag _bitmaskGLbitfield3_flags[] = {
    {"GL_ALL_SHADER_BITS", (unsigned long long)GL_ALL_SHADER_BITS},
    {"GL_VERTEX_SHADER_BIT", (unsigned long long)GL_VERTEX_SHADER_BIT},
    {"GL_FRAGMENT_SHADER_BIT", (unsigned long long)GL_FRAGMENT_SHADER_BIT},
    {"GL_GEOMETRY_SHADER_BIT", (unsigned long long)GL_GEOMETRY_SHADER_BIT},
    {"GL_TESS_CONTROL_SHADER_BIT", (unsigned long long)GL_TESS_CONTROL_SHADER_BIT},
    {"GL_TESS_EVALUATION_SHADER_BIT", (unsigned long long)GL_TESS_EVALUATION_SHADER_BIT},
    {"GL_COMPUTE_SHADER_BIT", (unsigned long long)GL_COMPUTE_SHADER_BIT},
};

static const trace::BitmaskSig _bitmaskGLbitfield3_sig = {
    2, 7, _bitmaskGLbitfield3_flags
};

static const trace::EnumValue _enumGLint3_values[] = {
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_NONE_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_NONE_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_DEFAULT_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_DEFAULT_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_1BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_1BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_2BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_2BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_3BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_3BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_4BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_4BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_5BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_5BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_6BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_6BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_7BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_7BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_8BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_8BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_9BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_9BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_10BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_10BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_11BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_11BPC_EXT},
    {"GL_SURFACE_COMPRESSION_FIXED_RATE_12BPC_EXT", GL_SURFACE_COMPRESSION_FIXED_RATE_12BPC_EXT},
};

static const trace::EnumSig _enumGLint3_sig = {
    6, 14, _enumGLint3_values
};

static const trace::EnumValue _enumGLenum3_values[] = {
    {"GL_KTX_FRONT_REGION", GL_KTX_FRONT_REGION},
    {"GL_KTX_BACK_REGION", GL_KTX_BACK_REGION},
    {"GL_KTX_Z_REGION", GL_KTX_Z_REGION},
    {"GL_KTX_STENCIL_REGION", GL_KTX_STENCIL_REGION},
};

static const trace::EnumSig _enumGLenum3_sig = {
    4, 4, _enumGLenum3_values
};

static const trace::BitmaskFlag _bitmaskGLbitfield9_flags[] = {
    {"GL_BOLD_BIT_NV", (unsigned long long)GL_BOLD_BIT_NV},
    {"GL_ITALIC_BIT_NV", (unsigned long long)GL_ITALIC_BIT_NV},
};

static const trace::BitmaskSig _bitmaskGLbitfield9_sig = {
    8, 2, _bitmaskGLbitfield9_flags
};

static const trace::BitmaskFlag _bitmaskGLbitfield8_flags[] = {
    {"GL_GLYPH_WIDTH_BIT_NV", (unsigned long long)GL_GLYPH_WIDTH_BIT_NV},
    {"GL_GLYPH_HEIGHT_BIT_NV", (unsigned long long)GL_GLYPH_HEIGHT_BIT_NV},
    {"GL_GLYPH_HORIZONTAL_BEARING_X_BIT_NV", (unsigned long long)GL_GLYPH_HORIZONTAL_BEARING_X_BIT_NV},
    {"GL_GLYPH_HORIZONTAL_BEARING_Y_BIT_NV", (unsigned long long)GL_GLYPH_HORIZONTAL_BEARING_Y_BIT_NV},
    {"GL_GLYPH_HORIZONTAL_BEARING_ADVANCE_BIT_NV", (unsigned long long)GL_GLYPH_HORIZONTAL_BEARING_ADVANCE_BIT_NV},
    {"GL_GLYPH_VERTICAL_BEARING_X_BIT_NV", (unsigned long long)GL_GLYPH_VERTICAL_BEARING_X_BIT_NV},
    {"GL_GLYPH_VERTICAL_BEARING_Y_BIT_NV", (unsigned long long)GL_GLYPH_VERTICAL_BEARING_Y_BIT_NV},
    {"GL_GLYPH_VERTICAL_BEARING_ADVANCE_BIT_NV", (unsigned long long)GL_GLYPH_VERTICAL_BEARING_ADVANCE_BIT_NV},
    {"GL_GLYPH_HAS_KERNING_BIT_NV", (unsigned long long)GL_GLYPH_HAS_KERNING_BIT_NV},
    {"GL_FONT_X_MIN_BOUNDS_BIT_NV", (unsigned long long)GL_FONT_X_MIN_BOUNDS_BIT_NV},
    {"GL_FONT_Y_MIN_BOUNDS_BIT_NV", (unsigned long long)GL_FONT_Y_MIN_BOUNDS_BIT_NV},
    {"GL_FONT_X_MAX_BOUNDS_BIT_NV", (unsigned long long)GL_FONT_X_MAX_BOUNDS_BIT_NV},
    {"GL_FONT_Y_MAX_BOUNDS_BIT_NV", (unsigned long long)GL_FONT_Y_MAX_BOUNDS_BIT_NV},
    {"GL_FONT_UNITS_PER_EM_BIT_NV", (unsigned long long)GL_FONT_UNITS_PER_EM_BIT_NV},
    {"GL_FONT_ASCENDER_BIT_NV", (unsigned long long)GL_FONT_ASCENDER_BIT_NV},
    {"GL_FONT_DESCENDER_BIT_NV", (unsigned long long)GL_FONT_DESCENDER_BIT_NV},
    {"GL_FONT_HEIGHT_BIT_NV", (unsigned long long)GL_FONT_HEIGHT_BIT_NV},
    {"GL_FONT_MAX_ADVANCE_WIDTH_BIT_NV", (unsigned long long)GL_FONT_MAX_ADVANCE_WIDTH_BIT_NV},
    {"GL_FONT_MAX_ADVANCE_HEIGHT_BIT_NV", (unsigned long long)GL_FONT_MAX_ADVANCE_HEIGHT_BIT_NV},
    {"GL_FONT_UNDERLINE_POSITION_BIT_NV", (unsigned long long)GL_FONT_UNDERLINE_POSITION_BIT_NV},
    {"GL_FONT_UNDERLINE_THICKNESS_BIT_NV", (unsigned long long)GL_FONT_UNDERLINE_THICKNESS_BIT_NV},
    {"GL_FONT_HAS_KERNING_BIT_NV", (unsigned long long)GL_FONT_HAS_KERNING_BIT_NV},
};

static const trace::BitmaskSig _bitmaskGLbitfield8_sig = {
    7, 22, _bitmaskGLbitfield8_flags
};


static const char * _glXChooseVisual_args[3] = {"dpy", "screen", "attribList"};
static const trace::FunctionSig _glXChooseVisual_sig = {4, "glXChooseVisual", 3, _glXChooseVisual_args};

static const char * _glXCreateContext_args[4] = {"dpy", "vis", "shareList", "direct"};
static const trace::FunctionSig _glXCreateContext_sig = {5, "glXCreateContext", 4, _glXCreateContext_args};

static const char * _glXDestroyContext_args[2] = {"dpy", "ctx"};
static const trace::FunctionSig _glXDestroyContext_sig = {6, "glXDestroyContext", 2, _glXDestroyContext_args};

static const char * _glXMakeCurrent_args[3] = {"dpy", "drawable", "ctx"};
static const trace::FunctionSig _glXMakeCurrent_sig = {7, "glXMakeCurrent", 3, _glXMakeCurrent_args};

static const char * _glXCopyContext_args[4] = {"dpy", "src", "dst", "mask"};
static const trace::FunctionSig _glXCopyContext_sig = {8, "glXCopyContext", 4, _glXCopyContext_args};

static const char * _glXSwapBuffers_args[2] = {"dpy", "drawable"};
static const trace::FunctionSig _glXSwapBuffers_sig = {9, "glXSwapBuffers", 2, _glXSwapBuffers_args};

static const char * _glXCreateGLXPixmap_args[3] = {"dpy", "visual", "pixmap"};
static const trace::FunctionSig _glXCreateGLXPixmap_sig = {10, "glXCreateGLXPixmap", 3, _glXCreateGLXPixmap_args};

static const char * _glXDestroyGLXPixmap_args[2] = {"dpy", "pixmap"};
static const trace::FunctionSig _glXDestroyGLXPixmap_sig = {11, "glXDestroyGLXPixmap", 2, _glXDestroyGLXPixmap_args};

static const char * _glXQueryExtension_args[3] = {"dpy", "errorb", "event"};
static const trace::FunctionSig _glXQueryExtension_sig = {12, "glXQueryExtension", 3, _glXQueryExtension_args};

static const char * _glXQueryVersion_args[3] = {"dpy", "maj", "min"};
static const trace::FunctionSig _glXQueryVersion_sig = {13, "glXQueryVersion", 3, _glXQueryVersion_args};

static const char * _glXIsDirect_args[2] = {"dpy", "ctx"};
static const trace::FunctionSig _glXIsDirect_sig = {14, "glXIsDirect", 2, _glXIsDirect_args};

static const char * _glXGetConfig_args[4] = {"dpy", "visual", "attrib", "value"};
static const trace::FunctionSig _glXGetConfig_sig = {15, "glXGetConfig", 4, _glXGetConfig_args};

static const char ** _glXGetCurrentContext_args = NULL;
static const trace::FunctionSig _glXGetCurrentContext_sig = {16, "glXGetCurrentContext", 0, _glXGetCurrentContext_args};

static const char ** _glXGetCurrentDrawable_args = NULL;
static const trace::FunctionSig _glXGetCurrentDrawable_sig = {17, "glXGetCurrentDrawable", 0, _glXGetCurrentDrawable_args};

static const char ** _glXWaitGL_args = NULL;
static const trace::FunctionSig _glXWaitGL_sig = {18, "glXWaitGL", 0, _glXWaitGL_args};

static const char ** _glXWaitX_args = NULL;
static const trace::FunctionSig _glXWaitX_sig = {19, "glXWaitX", 0, _glXWaitX_args};

static const char * _glXUseXFont_args[4] = {"font", "first", "count", "list"};
static const trace::FunctionSig _glXUseXFont_sig = {20, "glXUseXFont", 4, _glXUseXFont_args};

static const char * _glXQueryExtensionsString_args[2] = {"dpy", "screen"};
static const trace::FunctionSig _glXQueryExtensionsString_sig = {21, "glXQueryExtensionsString", 2, _glXQueryExtensionsString_args};

static const char * _glXQueryServerString_args[3] = {"dpy", "screen", "name"};
static const trace::FunctionSig _glXQueryServerString_sig = {22, "glXQueryServerString", 3, _glXQueryServerString_args};

static const char * _glXGetClientString_args[2] = {"dpy", "name"};
static const trace::FunctionSig _glXGetClientString_sig = {23, "glXGetClientString", 2, _glXGetClientString_args};

static const char ** _glXGetCurrentDisplay_args = NULL;
static const trace::FunctionSig _glXGetCurrentDisplay_sig = {24, "glXGetCurrentDisplay", 0, _glXGetCurrentDisplay_args};

static const char * _glXChooseFBConfig_args[4] = {"dpy", "screen", "attribList", "nitems"};
static const trace::FunctionSig _glXChooseFBConfig_sig = {25, "glXChooseFBConfig", 4, _glXChooseFBConfig_args};

static const char * _glXGetFBConfigAttrib_args[4] = {"dpy", "config", "attribute", "value"};
static const trace::FunctionSig _glXGetFBConfigAttrib_sig = {26, "glXGetFBConfigAttrib", 4, _glXGetFBConfigAttrib_args};

static const char * _glXGetFBConfigs_args[3] = {"dpy", "screen", "nelements"};
static const trace::FunctionSig _glXGetFBConfigs_sig = {27, "glXGetFBConfigs", 3, _glXGetFBConfigs_args};

static const char * _glXGetVisualFromFBConfig_args[2] = {"dpy", "config"};
static const trace::FunctionSig _glXGetVisualFromFBConfig_sig = {28, "glXGetVisualFromFBConfig", 2, _glXGetVisualFromFBConfig_args};

static const char * _glXCreateWindow_args[4] = {"dpy", "config", "win", "attribList"};
static const trace::FunctionSig _glXCreateWindow_sig = {29, "glXCreateWindow", 4, _glXCreateWindow_args};

static const char * _glXDestroyWindow_args[2] = {"dpy", "window"};
static const trace::FunctionSig _glXDestroyWindow_sig = {30, "glXDestroyWindow", 2, _glXDestroyWindow_args};

static const char * _glXCreatePixmap_args[4] = {"dpy", "config", "pixmap", "attribList"};
static const trace::FunctionSig _glXCreatePixmap_sig = {31, "glXCreatePixmap", 4, _glXCreatePixmap_args};

static const char * _glXDestroyPixmap_args[2] = {"dpy", "pixmap"};
static const trace::FunctionSig _glXDestroyPixmap_sig = {32, "glXDestroyPixmap", 2, _glXDestroyPixmap_args};

static const char * _glXCreatePbuffer_args[3] = {"dpy", "config", "attribList"};
static const trace::FunctionSig _glXCreatePbuffer_sig = {33, "glXCreatePbuffer", 3, _glXCreatePbuffer_args};

static const char * _glXDestroyPbuffer_args[2] = {"dpy", "pbuf"};
static const trace::FunctionSig _glXDestroyPbuffer_sig = {34, "glXDestroyPbuffer", 2, _glXDestroyPbuffer_args};

static const char * _glXQueryDrawable_args[4] = {"dpy", "draw", "attribute", "value"};
static const trace::FunctionSig _glXQueryDrawable_sig = {35, "glXQueryDrawable", 4, _glXQueryDrawable_args};

static const char * _glXCreateNewContext_args[5] = {"dpy", "config", "renderType", "shareList", "direct"};
static const trace::FunctionSig _glXCreateNewContext_sig = {36, "glXCreateNewContext", 5, _glXCreateNewContext_args};

static const char * _glXMakeContextCurrent_args[4] = {"dpy", "draw", "read", "ctx"};
static const trace::FunctionSig _glXMakeContextCurrent_sig = {37, "glXMakeContextCurrent", 4, _glXMakeContextCurrent_args};

static const char ** _glXGetCurrentReadDrawable_args = NULL;
static const trace::FunctionSig _glXGetCurrentReadDrawable_sig = {38, "glXGetCurrentReadDrawable", 0, _glXGetCurrentReadDrawable_args};

static const char * _glXQueryContext_args[4] = {"dpy", "ctx", "attribute", "value"};
static const trace::FunctionSig _glXQueryContext_sig = {39, "glXQueryContext", 4, _glXQueryContext_args};

static const char * _glXSelectEvent_args[3] = {"dpy", "drawable", "mask"};
static const trace::FunctionSig _glXSelectEvent_sig = {40, "glXSelectEvent", 3, _glXSelectEvent_args};

static const char * _glXGetSelectedEvent_args[3] = {"dpy", "drawable", "mask"};
static const trace::FunctionSig _glXGetSelectedEvent_sig = {41, "glXGetSelectedEvent", 3, _glXGetSelectedEvent_args};

static const char * _glXCreateContextAttribsARB_args[5] = {"dpy", "config", "share_context", "direct", "attrib_list"};
static const trace::FunctionSig _glXCreateContextAttribsARB_sig = {42, "glXCreateContextAttribsARB", 5, _glXCreateContextAttribsARB_args};

static const char ** _glXGetCurrentDisplayEXT_args = NULL;
static const trace::FunctionSig _glXGetCurrentDisplayEXT_sig = {43, "glXGetCurrentDisplayEXT", 0, _glXGetCurrentDisplayEXT_args};

static const char * _glXQueryContextInfoEXT_args[4] = {"dpy", "context", "attribute", "value"};
static const trace::FunctionSig _glXQueryContextInfoEXT_sig = {44, "glXQueryContextInfoEXT", 4, _glXQueryContextInfoEXT_args};

static const char * _glXGetContextIDEXT_args[1] = {"context"};
static const trace::FunctionSig _glXGetContextIDEXT_sig = {45, "glXGetContextIDEXT", 1, _glXGetContextIDEXT_args};

static const char * _glXImportContextEXT_args[2] = {"dpy", "contextID"};
static const trace::FunctionSig _glXImportContextEXT_sig = {46, "glXImportContextEXT", 2, _glXImportContextEXT_args};

static const char * _glXFreeContextEXT_args[2] = {"dpy", "context"};
static const trace::FunctionSig _glXFreeContextEXT_sig = {47, "glXFreeContextEXT", 2, _glXFreeContextEXT_args};

static const char * _glXSwapIntervalEXT_args[3] = {"dpy", "drawable", "interval"};
static const trace::FunctionSig _glXSwapIntervalEXT_sig = {48, "glXSwapIntervalEXT", 3, _glXSwapIntervalEXT_args};

static const char * _glXBindTexImageEXT_args[4] = {"display", "drawable", "buffer", "attrib_list"};
static const trace::FunctionSig _glXBindTexImageEXT_sig = {49, "glXBindTexImageEXT", 4, _glXBindTexImageEXT_args};

static const char * _glXReleaseTexImageEXT_args[3] = {"display", "drawable", "buffer"};
static const trace::FunctionSig _glXReleaseTexImageEXT_sig = {50, "glXReleaseTexImageEXT", 3, _glXReleaseTexImageEXT_args};

static const char * _glXGetAGPOffsetMESA_args[1] = {"pointer"};
static const trace::FunctionSig _glXGetAGPOffsetMESA_sig = {51, "glXGetAGPOffsetMESA", 1, _glXGetAGPOffsetMESA_args};

static const char * _glXCopySubBufferMESA_args[6] = {"dpy", "drawable", "x", "y", "width", "height"};
static const trace::FunctionSig _glXCopySubBufferMESA_sig = {52, "glXCopySubBufferMESA", 6, _glXCopySubBufferMESA_args};

static const char * _glXCreateGLXPixmapMESA_args[4] = {"dpy", "visual", "pixmap", "cmap"};
static const trace::FunctionSig _glXCreateGLXPixmapMESA_sig = {53, "glXCreateGLXPixmapMESA", 4, _glXCreateGLXPixmapMESA_args};

static const char * _glXQueryCurrentRendererIntegerMESA_args[2] = {"attribute", "value"};
static const trace::FunctionSig _glXQueryCurrentRendererIntegerMESA_sig = {54, "glXQueryCurrentRendererIntegerMESA", 2, _glXQueryCurrentRendererIntegerMESA_args};

static const char * _glXQueryCurrentRendererStringMESA_args[1] = {"attribute"};
static const trace::FunctionSig _glXQueryCurrentRendererStringMESA_sig = {55, "glXQueryCurrentRendererStringMESA", 1, _glXQueryCurrentRendererStringMESA_args};

static const char * _glXQueryRendererIntegerMESA_args[5] = {"dpy", "screen", "renderer", "attribute", "value"};
static const trace::FunctionSig _glXQueryRendererIntegerMESA_sig = {56, "glXQueryRendererIntegerMESA", 5, _glXQueryRendererIntegerMESA_args};

static const char * _glXQueryRendererStringMESA_args[4] = {"dpy", "screen", "renderer", "attribute"};
static const trace::FunctionSig _glXQueryRendererStringMESA_sig = {57, "glXQueryRendererStringMESA", 4, _glXQueryRendererStringMESA_args};

static const char * _glXReleaseBuffersMESA_args[2] = {"dpy", "drawable"};
static const trace::FunctionSig _glXReleaseBuffersMESA_sig = {58, "glXReleaseBuffersMESA", 2, _glXReleaseBuffersMESA_args};

static const char * _glXSet3DfxModeMESA_args[1] = {"mode"};
static const trace::FunctionSig _glXSet3DfxModeMESA_sig = {59, "glXSet3DfxModeMESA", 1, _glXSet3DfxModeMESA_args};

static const char * _glXSwapIntervalMESA_args[1] = {"interval"};
static const trace::FunctionSig _glXSwapIntervalMESA_sig = {60, "glXSwapIntervalMESA", 1, _glXSwapIntervalMESA_args};

static const char ** _glXGetSwapIntervalMESA_args = NULL;
static const trace::FunctionSig _glXGetSwapIntervalMESA_sig = {61, "glXGetSwapIntervalMESA", 0, _glXGetSwapIntervalMESA_args};

static const char * _glXCopyImageSubDataNV_args[18] = {"dpy", "srcCtx", "srcName", "srcTarget", "srcLevel", "srcX", "srcY", "srcZ", "dstCtx", "dstName", "dstTarget", "dstLevel", "dstX", "dstY", "dstZ", "width", "height", "depth"};
static const trace::FunctionSig _glXCopyImageSubDataNV_sig = {62, "glXCopyImageSubDataNV", 18, _glXCopyImageSubDataNV_args};

static const char * _glXEnumerateVideoDevicesNV_args[3] = {"dpy", "screen", "nelements"};
static const trace::FunctionSig _glXEnumerateVideoDevicesNV_sig = {63, "glXEnumerateVideoDevicesNV", 3, _glXEnumerateVideoDevicesNV_args};

static const char * _glXBindVideoDeviceNV_args[4] = {"dpy", "video_slot", "video_device", "attrib_list"};
static const trace::FunctionSig _glXBindVideoDeviceNV_sig = {64, "glXBindVideoDeviceNV", 4, _glXBindVideoDeviceNV_args};

static const char * _glXJoinSwapGroupNV_args[3] = {"dpy", "drawable", "group"};
static const trace::FunctionSig _glXJoinSwapGroupNV_sig = {65, "glXJoinSwapGroupNV", 3, _glXJoinSwapGroupNV_args};

static const char * _glXBindSwapBarrierNV_args[3] = {"dpy", "group", "barrier"};
static const trace::FunctionSig _glXBindSwapBarrierNV_sig = {66, "glXBindSwapBarrierNV", 3, _glXBindSwapBarrierNV_args};

static const char * _glXQuerySwapGroupNV_args[4] = {"dpy", "drawable", "group", "barrier"};
static const trace::FunctionSig _glXQuerySwapGroupNV_sig = {67, "glXQuerySwapGroupNV", 4, _glXQuerySwapGroupNV_args};

static const char * _glXQueryMaxSwapGroupsNV_args[4] = {"dpy", "screen", "maxGroups", "maxBarriers"};
static const trace::FunctionSig _glXQueryMaxSwapGroupsNV_sig = {68, "glXQueryMaxSwapGroupsNV", 4, _glXQueryMaxSwapGroupsNV_args};

static const char * _glXQueryFrameCountNV_args[3] = {"dpy", "screen", "count"};
static const trace::FunctionSig _glXQueryFrameCountNV_sig = {69, "glXQueryFrameCountNV", 3, _glXQueryFrameCountNV_args};

static const char * _glXResetFrameCountNV_args[2] = {"dpy", "screen"};
static const trace::FunctionSig _glXResetFrameCountNV_sig = {70, "glXResetFrameCountNV", 2, _glXResetFrameCountNV_args};

static const char * _glXAllocateMemoryNV_args[4] = {"size", "readfreq", "writefreq", "priority"};
static const trace::FunctionSig _glXAllocateMemoryNV_sig = {71, "glXAllocateMemoryNV", 4, _glXAllocateMemoryNV_args};

static const char * _glXFreeMemoryNV_args[1] = {"pointer"};
static const trace::FunctionSig _glXFreeMemoryNV_sig = {72, "glXFreeMemoryNV", 1, _glXFreeMemoryNV_args};

static const char * _glXBindVideoCaptureDeviceNV_args[3] = {"dpy", "video_capture_slot", "device"};
static const trace::FunctionSig _glXBindVideoCaptureDeviceNV_sig = {73, "glXBindVideoCaptureDeviceNV", 3, _glXBindVideoCaptureDeviceNV_args};

static const char * _glXEnumerateVideoCaptureDevicesNV_args[3] = {"dpy", "screen", "nelements"};
static const trace::FunctionSig _glXEnumerateVideoCaptureDevicesNV_sig = {74, "glXEnumerateVideoCaptureDevicesNV", 3, _glXEnumerateVideoCaptureDevicesNV_args};

static const char * _glXLockVideoCaptureDeviceNV_args[2] = {"dpy", "device"};
static const trace::FunctionSig _glXLockVideoCaptureDeviceNV_sig = {75, "glXLockVideoCaptureDeviceNV", 2, _glXLockVideoCaptureDeviceNV_args};

static const char * _glXQueryVideoCaptureDeviceNV_args[4] = {"dpy", "device", "attribute", "value"};
static const trace::FunctionSig _glXQueryVideoCaptureDeviceNV_sig = {76, "glXQueryVideoCaptureDeviceNV", 4, _glXQueryVideoCaptureDeviceNV_args};

static const char * _glXReleaseVideoCaptureDeviceNV_args[2] = {"dpy", "device"};
static const trace::FunctionSig _glXReleaseVideoCaptureDeviceNV_sig = {77, "glXReleaseVideoCaptureDeviceNV", 2, _glXReleaseVideoCaptureDeviceNV_args};

static const char * _glXGetVideoDeviceNV_args[4] = {"dpy", "screen", "numVideoDevices", "pVideoDevice"};
static const trace::FunctionSig _glXGetVideoDeviceNV_sig = {78, "glXGetVideoDeviceNV", 4, _glXGetVideoDeviceNV_args};

static const char * _glXReleaseVideoDeviceNV_args[3] = {"dpy", "screen", "VideoDevice"};
static const trace::FunctionSig _glXReleaseVideoDeviceNV_sig = {79, "glXReleaseVideoDeviceNV", 3, _glXReleaseVideoDeviceNV_args};

static const char * _glXBindVideoImageNV_args[4] = {"dpy", "VideoDevice", "pbuf", "iVideoBuffer"};
static const trace::FunctionSig _glXBindVideoImageNV_sig = {80, "glXBindVideoImageNV", 4, _glXBindVideoImageNV_args};

static const char * _glXReleaseVideoImageNV_args[2] = {"dpy", "pbuf"};
static const trace::FunctionSig _glXReleaseVideoImageNV_sig = {81, "glXReleaseVideoImageNV", 2, _glXReleaseVideoImageNV_args};

static const char * _glXSendPbufferToVideoNV_args[5] = {"dpy", "pbuf", "iBufferType", "pulCounterPbuffer", "bBlock"};
static const trace::FunctionSig _glXSendPbufferToVideoNV_sig = {82, "glXSendPbufferToVideoNV", 5, _glXSendPbufferToVideoNV_args};

static const char * _glXGetVideoInfoNV_args[5] = {"dpy", "screen", "VideoDevice", "pulCounterOutputPbuffer", "pulCounterOutputVideo"};
static const trace::FunctionSig _glXGetVideoInfoNV_sig = {83, "glXGetVideoInfoNV", 5, _glXGetVideoInfoNV_args};

static const char * _glXGetSyncValuesOML_args[5] = {"dpy", "drawable", "ust", "msc", "sbc"};
static const trace::FunctionSig _glXGetSyncValuesOML_sig = {84, "glXGetSyncValuesOML", 5, _glXGetSyncValuesOML_args};

static const char * _glXGetMscRateOML_args[4] = {"dpy", "drawable", "numerator", "denominator"};
static const trace::FunctionSig _glXGetMscRateOML_sig = {85, "glXGetMscRateOML", 4, _glXGetMscRateOML_args};

static const char * _glXSwapBuffersMscOML_args[5] = {"dpy", "drawable", "target_msc", "divisor", "remainder"};
static const trace::FunctionSig _glXSwapBuffersMscOML_sig = {86, "glXSwapBuffersMscOML", 5, _glXSwapBuffersMscOML_args};

static const char * _glXWaitForMscOML_args[8] = {"dpy", "drawable", "target_msc", "divisor", "remainder", "ust", "msc", "sbc"};
static const trace::FunctionSig _glXWaitForMscOML_sig = {87, "glXWaitForMscOML", 8, _glXWaitForMscOML_args};

static const char * _glXWaitForSbcOML_args[6] = {"dpy", "drawable", "target_sbc", "ust", "msc", "sbc"};
static const trace::FunctionSig _glXWaitForSbcOML_sig = {88, "glXWaitForSbcOML", 6, _glXWaitForSbcOML_args};

static const char * _glXCushionSGI_args[3] = {"dpy", "window", "cushion"};
static const trace::FunctionSig _glXCushionSGI_sig = {89, "glXCushionSGI", 3, _glXCushionSGI_args};

static const char * _glXMakeCurrentReadSGI_args[4] = {"dpy", "draw", "read", "ctx"};
static const trace::FunctionSig _glXMakeCurrentReadSGI_sig = {90, "glXMakeCurrentReadSGI", 4, _glXMakeCurrentReadSGI_args};

static const char ** _glXGetCurrentReadDrawableSGI_args = NULL;
static const trace::FunctionSig _glXGetCurrentReadDrawableSGI_sig = {91, "glXGetCurrentReadDrawableSGI", 0, _glXGetCurrentReadDrawableSGI_args};

static const char * _glXSwapIntervalSGI_args[1] = {"interval"};
static const trace::FunctionSig _glXSwapIntervalSGI_sig = {92, "glXSwapIntervalSGI", 1, _glXSwapIntervalSGI_args};

static const char * _glXGetVideoSyncSGI_args[1] = {"count"};
static const trace::FunctionSig _glXGetVideoSyncSGI_sig = {93, "glXGetVideoSyncSGI", 1, _glXGetVideoSyncSGI_args};

static const char * _glXWaitVideoSyncSGI_args[3] = {"divisor", "remainder", "count"};
static const trace::FunctionSig _glXWaitVideoSyncSGI_sig = {94, "glXWaitVideoSyncSGI", 3, _glXWaitVideoSyncSGI_args};

static const char * _glXGetFBConfigAttribSGIX_args[4] = {"dpy", "config", "attribute", "value"};
static const trace::FunctionSig _glXGetFBConfigAttribSGIX_sig = {95, "glXGetFBConfigAttribSGIX", 4, _glXGetFBConfigAttribSGIX_args};

static const char * _glXChooseFBConfigSGIX_args[4] = {"dpy", "screen", "attrib_list", "nelements"};
static const trace::FunctionSig _glXChooseFBConfigSGIX_sig = {96, "glXChooseFBConfigSGIX", 4, _glXChooseFBConfigSGIX_args};

static const char * _glXCreateGLXPixmapWithConfigSGIX_args[3] = {"dpy", "config", "pixmap"};
static const trace::FunctionSig _glXCreateGLXPixmapWithConfigSGIX_sig = {97, "glXCreateGLXPixmapWithConfigSGIX", 3, _glXCreateGLXPixmapWithConfigSGIX_args};

static const char * _glXCreateContextWithConfigSGIX_args[5] = {"dpy", "config", "render_type", "share_list", "direct"};
static const trace::FunctionSig _glXCreateContextWithConfigSGIX_sig = {98, "glXCreateContextWithConfigSGIX", 5, _glXCreateContextWithConfigSGIX_args};

static const char * _glXGetVisualFromFBConfigSGIX_args[2] = {"dpy", "config"};
static const trace::FunctionSig _glXGetVisualFromFBConfigSGIX_sig = {99, "glXGetVisualFromFBConfigSGIX", 2, _glXGetVisualFromFBConfigSGIX_args};

static const char * _glXGetFBConfigFromVisualSGIX_args[2] = {"dpy", "vis"};
static const trace::FunctionSig _glXGetFBConfigFromVisualSGIX_sig = {100, "glXGetFBConfigFromVisualSGIX", 2, _glXGetFBConfigFromVisualSGIX_args};

static const char * _glXCreateGLXPbufferSGIX_args[5] = {"dpy", "config", "width", "height", "attrib_list"};
static const trace::FunctionSig _glXCreateGLXPbufferSGIX_sig = {101, "glXCreateGLXPbufferSGIX", 5, _glXCreateGLXPbufferSGIX_args};

static const char * _glXDestroyGLXPbufferSGIX_args[2] = {"dpy", "pbuf"};
static const trace::FunctionSig _glXDestroyGLXPbufferSGIX_sig = {102, "glXDestroyGLXPbufferSGIX", 2, _glXDestroyGLXPbufferSGIX_args};

static const char * _glXQueryGLXPbufferSGIX_args[4] = {"dpy", "pbuf", "attribute", "value"};
static const trace::FunctionSig _glXQueryGLXPbufferSGIX_sig = {103, "glXQueryGLXPbufferSGIX", 4, _glXQueryGLXPbufferSGIX_args};

static const char * _glXSelectEventSGIX_args[3] = {"dpy", "drawable", "mask"};
static const trace::FunctionSig _glXSelectEventSGIX_sig = {104, "glXSelectEventSGIX", 3, _glXSelectEventSGIX_args};

static const char * _glXGetSelectedEventSGIX_args[3] = {"dpy", "drawable", "mask"};
static const trace::FunctionSig _glXGetSelectedEventSGIX_sig = {105, "glXGetSelectedEventSGIX", 3, _glXGetSelectedEventSGIX_args};

static const char * _glXBindSwapBarrierSGIX_args[3] = {"dpy", "drawable", "barrier"};
static const trace::FunctionSig _glXBindSwapBarrierSGIX_sig = {106, "glXBindSwapBarrierSGIX", 3, _glXBindSwapBarrierSGIX_args};

static const char * _glXQueryMaxSwapBarriersSGIX_args[3] = {"dpy", "screen", "max"};
static const trace::FunctionSig _glXQueryMaxSwapBarriersSGIX_sig = {107, "glXQueryMaxSwapBarriersSGIX", 3, _glXQueryMaxSwapBarriersSGIX_args};

static const char * _glXJoinSwapGroupSGIX_args[3] = {"dpy", "drawable", "member"};
static const trace::FunctionSig _glXJoinSwapGroupSGIX_sig = {108, "glXJoinSwapGroupSGIX", 3, _glXJoinSwapGroupSGIX_args};

static const char * _glXBindChannelToWindowSGIX_args[4] = {"display", "screen", "channel", "window"};
static const trace::FunctionSig _glXBindChannelToWindowSGIX_sig = {109, "glXBindChannelToWindowSGIX", 4, _glXBindChannelToWindowSGIX_args};

static const char * _glXChannelRectSGIX_args[7] = {"display", "screen", "channel", "x", "y", "w", "h"};
static const trace::FunctionSig _glXChannelRectSGIX_sig = {110, "glXChannelRectSGIX", 7, _glXChannelRectSGIX_args};

static const char * _glXQueryChannelRectSGIX_args[7] = {"display", "screen", "channel", "dx", "dy", "dw", "dh"};
static const trace::FunctionSig _glXQueryChannelRectSGIX_sig = {111, "glXQueryChannelRectSGIX", 7, _glXQueryChannelRectSGIX_args};

static const char * _glXQueryChannelDeltasSGIX_args[7] = {"display", "screen", "channel", "x", "y", "w", "h"};
static const trace::FunctionSig _glXQueryChannelDeltasSGIX_sig = {112, "glXQueryChannelDeltasSGIX", 7, _glXQueryChannelDeltasSGIX_args};

static const char * _glXChannelRectSyncSGIX_args[4] = {"display", "screen", "channel", "synctype"};
static const trace::FunctionSig _glXChannelRectSyncSGIX_sig = {113, "glXChannelRectSyncSGIX", 4, _glXChannelRectSyncSGIX_args};

static const char * _glXGetProcAddressARB_args[1] = {"procName"};
static const trace::FunctionSig _glXGetProcAddressARB_sig = {114, "glXGetProcAddressARB", 1, _glXGetProcAddressARB_args};

static const char * _glXGetProcAddress_args[1] = {"procName"};
static const trace::FunctionSig _glXGetProcAddress_sig = {115, "glXGetProcAddress", 1, _glXGetProcAddress_args};

static const char * _glCullFace_args[1] = {"mode"};
static const trace::FunctionSig _glCullFace_sig = {116, "glCullFace", 1, _glCullFace_args};

static const char * _glFrontFace_args[1] = {"mode"};
static const trace::FunctionSig _glFrontFace_sig = {117, "glFrontFace", 1, _glFrontFace_args};

static const char * _glHint_args[2] = {"target", "mode"};
static const trace::FunctionSig _glHint_sig = {118, "glHint", 2, _glHint_args};

static const char * _glLineWidth_args[1] = {"width"};
static const trace::FunctionSig _glLineWidth_sig = {119, "glLineWidth", 1, _glLineWidth_args};

static const char * _glPointSize_args[1] = {"size"};
static const trace::FunctionSig _glPointSize_sig = {120, "glPointSize", 1, _glPointSize_args};

static const char * _glPolygonMode_args[2] = {"face", "mode"};
static const trace::FunctionSig _glPolygonMode_sig = {121, "glPolygonMode", 2, _glPolygonMode_args};

static const char * _glScissor_args[4] = {"x", "y", "width", "height"};
static const trace::FunctionSig _glScissor_sig = {122, "glScissor", 4, _glScissor_args};

static const char * _glTexParameterf_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glTexParameterf_sig = {123, "glTexParameterf", 3, _glTexParameterf_args};

static const char * _glTexParameterfv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glTexParameterfv_sig = {124, "glTexParameterfv", 3, _glTexParameterfv_args};

static const char * _glTexParameteri_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glTexParameteri_sig = {125, "glTexParameteri", 3, _glTexParameteri_args};

static const char * _glTexParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glTexParameteriv_sig = {126, "glTexParameteriv", 3, _glTexParameteriv_args};

static const char * _glTexImage1D_args[8] = {"target", "level", "internalformat", "width", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glTexImage1D_sig = {127, "glTexImage1D", 8, _glTexImage1D_args};

static const char * _glTexImage2D_args[9] = {"target", "level", "internalformat", "width", "height", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glTexImage2D_sig = {128, "glTexImage2D", 9, _glTexImage2D_args};

static const char * _glDrawBuffer_args[1] = {"mode"};
static const trace::FunctionSig _glDrawBuffer_sig = {129, "glDrawBuffer", 1, _glDrawBuffer_args};

static const char * _glClear_args[1] = {"mask"};
static const trace::FunctionSig _glClear_sig = {130, "glClear", 1, _glClear_args};

static const char * _glClearColor_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glClearColor_sig = {131, "glClearColor", 4, _glClearColor_args};

static const char * _glClearStencil_args[1] = {"s"};
static const trace::FunctionSig _glClearStencil_sig = {132, "glClearStencil", 1, _glClearStencil_args};

static const char * _glClearDepth_args[1] = {"depth"};
static const trace::FunctionSig _glClearDepth_sig = {133, "glClearDepth", 1, _glClearDepth_args};

static const char * _glStencilMask_args[1] = {"mask"};
static const trace::FunctionSig _glStencilMask_sig = {134, "glStencilMask", 1, _glStencilMask_args};

static const char * _glColorMask_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColorMask_sig = {135, "glColorMask", 4, _glColorMask_args};

static const char * _glDepthMask_args[1] = {"flag"};
static const trace::FunctionSig _glDepthMask_sig = {136, "glDepthMask", 1, _glDepthMask_args};

static const char * _glDisable_args[1] = {"cap"};
static const trace::FunctionSig _glDisable_sig = {137, "glDisable", 1, _glDisable_args};

static const char * _glEnable_args[1] = {"cap"};
static const trace::FunctionSig _glEnable_sig = {138, "glEnable", 1, _glEnable_args};

static const char ** _glFinish_args = NULL;
static const trace::FunctionSig _glFinish_sig = {139, "glFinish", 0, _glFinish_args};

static const char ** _glFlush_args = NULL;
static const trace::FunctionSig _glFlush_sig = {140, "glFlush", 0, _glFlush_args};

static const char * _glBlendFunc_args[2] = {"sfactor", "dfactor"};
static const trace::FunctionSig _glBlendFunc_sig = {141, "glBlendFunc", 2, _glBlendFunc_args};

static const char * _glLogicOp_args[1] = {"opcode"};
static const trace::FunctionSig _glLogicOp_sig = {142, "glLogicOp", 1, _glLogicOp_args};

static const char * _glStencilFunc_args[3] = {"func", "ref", "mask"};
static const trace::FunctionSig _glStencilFunc_sig = {143, "glStencilFunc", 3, _glStencilFunc_args};

static const char * _glStencilOp_args[3] = {"fail", "zfail", "zpass"};
static const trace::FunctionSig _glStencilOp_sig = {144, "glStencilOp", 3, _glStencilOp_args};

static const char * _glDepthFunc_args[1] = {"func"};
static const trace::FunctionSig _glDepthFunc_sig = {145, "glDepthFunc", 1, _glDepthFunc_args};

static const char * _glPixelStoref_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPixelStoref_sig = {146, "glPixelStoref", 2, _glPixelStoref_args};

static const char * _glPixelStorei_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPixelStorei_sig = {147, "glPixelStorei", 2, _glPixelStorei_args};

static const char * _glReadBuffer_args[1] = {"mode"};
static const trace::FunctionSig _glReadBuffer_sig = {148, "glReadBuffer", 1, _glReadBuffer_args};

static const char * _glReadPixels_args[7] = {"x", "y", "width", "height", "format", "type", "pixels"};
static const trace::FunctionSig _glReadPixels_sig = {149, "glReadPixels", 7, _glReadPixels_args};

static const char * _glGetBooleanv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetBooleanv_sig = {150, "glGetBooleanv", 2, _glGetBooleanv_args};

static const char * _glGetDoublev_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetDoublev_sig = {151, "glGetDoublev", 2, _glGetDoublev_args};

static const char ** _glGetError_args = NULL;
static const trace::FunctionSig _glGetError_sig = {152, "glGetError", 0, _glGetError_args};

static const char * _glGetFloatv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetFloatv_sig = {153, "glGetFloatv", 2, _glGetFloatv_args};

static const char * _glGetIntegerv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetIntegerv_sig = {154, "glGetIntegerv", 2, _glGetIntegerv_args};

static const char * _glGetString_args[1] = {"name"};
static const trace::FunctionSig _glGetString_sig = {155, "glGetString", 1, _glGetString_args};

static const char * _glGetTexImage_args[5] = {"target", "level", "format", "type", "pixels"};
static const trace::FunctionSig _glGetTexImage_sig = {156, "glGetTexImage", 5, _glGetTexImage_args};

static const char * _glGetTexParameterfv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexParameterfv_sig = {157, "glGetTexParameterfv", 3, _glGetTexParameterfv_args};

static const char * _glGetTexParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexParameteriv_sig = {158, "glGetTexParameteriv", 3, _glGetTexParameteriv_args};

static const char * _glGetTexLevelParameterfv_args[4] = {"target", "level", "pname", "params"};
static const trace::FunctionSig _glGetTexLevelParameterfv_sig = {159, "glGetTexLevelParameterfv", 4, _glGetTexLevelParameterfv_args};

static const char * _glGetTexLevelParameteriv_args[4] = {"target", "level", "pname", "params"};
static const trace::FunctionSig _glGetTexLevelParameteriv_sig = {160, "glGetTexLevelParameteriv", 4, _glGetTexLevelParameteriv_args};

static const char * _glIsEnabled_args[1] = {"cap"};
static const trace::FunctionSig _glIsEnabled_sig = {161, "glIsEnabled", 1, _glIsEnabled_args};

static const char * _glDepthRange_args[2] = {"zNear", "zFar"};
static const trace::FunctionSig _glDepthRange_sig = {162, "glDepthRange", 2, _glDepthRange_args};

static const char * _glViewport_args[4] = {"x", "y", "width", "height"};
static const trace::FunctionSig _glViewport_sig = {163, "glViewport", 4, _glViewport_args};

static const char * _glNewList_args[2] = {"list", "mode"};
static const trace::FunctionSig _glNewList_sig = {164, "glNewList", 2, _glNewList_args};

static const char ** _glEndList_args = NULL;
static const trace::FunctionSig _glEndList_sig = {165, "glEndList", 0, _glEndList_args};

static const char * _glCallList_args[1] = {"list"};
static const trace::FunctionSig _glCallList_sig = {166, "glCallList", 1, _glCallList_args};

static const char * _glCallLists_args[3] = {"n", "type", "lists"};
static const trace::FunctionSig _glCallLists_sig = {167, "glCallLists", 3, _glCallLists_args};

static const char * _glDeleteLists_args[2] = {"list", "range"};
static const trace::FunctionSig _glDeleteLists_sig = {168, "glDeleteLists", 2, _glDeleteLists_args};

static const char * _glGenLists_args[1] = {"range"};
static const trace::FunctionSig _glGenLists_sig = {169, "glGenLists", 1, _glGenLists_args};

static const char * _glListBase_args[1] = {"base"};
static const trace::FunctionSig _glListBase_sig = {170, "glListBase", 1, _glListBase_args};

static const char * _glBegin_args[1] = {"mode"};
static const trace::FunctionSig _glBegin_sig = {171, "glBegin", 1, _glBegin_args};

static const char * _glBitmap_args[7] = {"width", "height", "xorig", "yorig", "xmove", "ymove", "bitmap"};
static const trace::FunctionSig _glBitmap_sig = {172, "glBitmap", 7, _glBitmap_args};

static const char * _glColor3b_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glColor3b_sig = {173, "glColor3b", 3, _glColor3b_args};

static const char * _glColor3bv_args[1] = {"v"};
static const trace::FunctionSig _glColor3bv_sig = {174, "glColor3bv", 1, _glColor3bv_args};

static const char * _glColor3d_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glColor3d_sig = {175, "glColor3d", 3, _glColor3d_args};

static const char * _glColor3dv_args[1] = {"v"};
static const trace::FunctionSig _glColor3dv_sig = {176, "glColor3dv", 1, _glColor3dv_args};

static const char * _glColor3f_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glColor3f_sig = {177, "glColor3f", 3, _glColor3f_args};

static const char * _glColor3fv_args[1] = {"v"};
static const trace::FunctionSig _glColor3fv_sig = {178, "glColor3fv", 1, _glColor3fv_args};

static const char * _glColor3i_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glColor3i_sig = {179, "glColor3i", 3, _glColor3i_args};

static const char * _glColor3iv_args[1] = {"v"};
static const trace::FunctionSig _glColor3iv_sig = {180, "glColor3iv", 1, _glColor3iv_args};

static const char * _glColor3s_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glColor3s_sig = {181, "glColor3s", 3, _glColor3s_args};

static const char * _glColor3sv_args[1] = {"v"};
static const trace::FunctionSig _glColor3sv_sig = {182, "glColor3sv", 1, _glColor3sv_args};

static const char * _glColor3ub_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glColor3ub_sig = {183, "glColor3ub", 3, _glColor3ub_args};

static const char * _glColor3ubv_args[1] = {"v"};
static const trace::FunctionSig _glColor3ubv_sig = {184, "glColor3ubv", 1, _glColor3ubv_args};

static const char * _glColor3ui_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glColor3ui_sig = {185, "glColor3ui", 3, _glColor3ui_args};

static const char * _glColor3uiv_args[1] = {"v"};
static const trace::FunctionSig _glColor3uiv_sig = {186, "glColor3uiv", 1, _glColor3uiv_args};

static const char * _glColor3us_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glColor3us_sig = {187, "glColor3us", 3, _glColor3us_args};

static const char * _glColor3usv_args[1] = {"v"};
static const trace::FunctionSig _glColor3usv_sig = {188, "glColor3usv", 1, _glColor3usv_args};

static const char * _glColor4b_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColor4b_sig = {189, "glColor4b", 4, _glColor4b_args};

static const char * _glColor4bv_args[1] = {"v"};
static const trace::FunctionSig _glColor4bv_sig = {190, "glColor4bv", 1, _glColor4bv_args};

static const char * _glColor4d_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColor4d_sig = {191, "glColor4d", 4, _glColor4d_args};

static const char * _glColor4dv_args[1] = {"v"};
static const trace::FunctionSig _glColor4dv_sig = {192, "glColor4dv", 1, _glColor4dv_args};

static const char * _glColor4f_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColor4f_sig = {193, "glColor4f", 4, _glColor4f_args};

static const char * _glColor4fv_args[1] = {"v"};
static const trace::FunctionSig _glColor4fv_sig = {194, "glColor4fv", 1, _glColor4fv_args};

static const char * _glColor4i_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColor4i_sig = {195, "glColor4i", 4, _glColor4i_args};

static const char * _glColor4iv_args[1] = {"v"};
static const trace::FunctionSig _glColor4iv_sig = {196, "glColor4iv", 1, _glColor4iv_args};

static const char * _glColor4s_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColor4s_sig = {197, "glColor4s", 4, _glColor4s_args};

static const char * _glColor4sv_args[1] = {"v"};
static const trace::FunctionSig _glColor4sv_sig = {198, "glColor4sv", 1, _glColor4sv_args};

static const char * _glColor4ub_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColor4ub_sig = {199, "glColor4ub", 4, _glColor4ub_args};

static const char * _glColor4ubv_args[1] = {"v"};
static const trace::FunctionSig _glColor4ubv_sig = {200, "glColor4ubv", 1, _glColor4ubv_args};

static const char * _glColor4ui_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColor4ui_sig = {201, "glColor4ui", 4, _glColor4ui_args};

static const char * _glColor4uiv_args[1] = {"v"};
static const trace::FunctionSig _glColor4uiv_sig = {202, "glColor4uiv", 1, _glColor4uiv_args};

static const char * _glColor4us_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColor4us_sig = {203, "glColor4us", 4, _glColor4us_args};

static const char * _glColor4usv_args[1] = {"v"};
static const trace::FunctionSig _glColor4usv_sig = {204, "glColor4usv", 1, _glColor4usv_args};

static const char * _glEdgeFlag_args[1] = {"flag"};
static const trace::FunctionSig _glEdgeFlag_sig = {205, "glEdgeFlag", 1, _glEdgeFlag_args};

static const char * _glEdgeFlagv_args[1] = {"flag"};
static const trace::FunctionSig _glEdgeFlagv_sig = {206, "glEdgeFlagv", 1, _glEdgeFlagv_args};

static const char ** _glEnd_args = NULL;
static const trace::FunctionSig _glEnd_sig = {207, "glEnd", 0, _glEnd_args};

static const char * _glIndexd_args[1] = {"c"};
static const trace::FunctionSig _glIndexd_sig = {208, "glIndexd", 1, _glIndexd_args};

static const char * _glIndexdv_args[1] = {"c"};
static const trace::FunctionSig _glIndexdv_sig = {209, "glIndexdv", 1, _glIndexdv_args};

static const char * _glIndexf_args[1] = {"c"};
static const trace::FunctionSig _glIndexf_sig = {210, "glIndexf", 1, _glIndexf_args};

static const char * _glIndexfv_args[1] = {"c"};
static const trace::FunctionSig _glIndexfv_sig = {211, "glIndexfv", 1, _glIndexfv_args};

static const char * _glIndexi_args[1] = {"c"};
static const trace::FunctionSig _glIndexi_sig = {212, "glIndexi", 1, _glIndexi_args};

static const char * _glIndexiv_args[1] = {"c"};
static const trace::FunctionSig _glIndexiv_sig = {213, "glIndexiv", 1, _glIndexiv_args};

static const char * _glIndexs_args[1] = {"c"};
static const trace::FunctionSig _glIndexs_sig = {214, "glIndexs", 1, _glIndexs_args};

static const char * _glIndexsv_args[1] = {"c"};
static const trace::FunctionSig _glIndexsv_sig = {215, "glIndexsv", 1, _glIndexsv_args};

static const char * _glNormal3b_args[3] = {"nx", "ny", "nz"};
static const trace::FunctionSig _glNormal3b_sig = {216, "glNormal3b", 3, _glNormal3b_args};

static const char * _glNormal3bv_args[1] = {"v"};
static const trace::FunctionSig _glNormal3bv_sig = {217, "glNormal3bv", 1, _glNormal3bv_args};

static const char * _glNormal3d_args[3] = {"nx", "ny", "nz"};
static const trace::FunctionSig _glNormal3d_sig = {218, "glNormal3d", 3, _glNormal3d_args};

static const char * _glNormal3dv_args[1] = {"v"};
static const trace::FunctionSig _glNormal3dv_sig = {219, "glNormal3dv", 1, _glNormal3dv_args};

static const char * _glNormal3f_args[3] = {"nx", "ny", "nz"};
static const trace::FunctionSig _glNormal3f_sig = {220, "glNormal3f", 3, _glNormal3f_args};

static const char * _glNormal3fv_args[1] = {"v"};
static const trace::FunctionSig _glNormal3fv_sig = {221, "glNormal3fv", 1, _glNormal3fv_args};

static const char * _glNormal3i_args[3] = {"nx", "ny", "nz"};
static const trace::FunctionSig _glNormal3i_sig = {222, "glNormal3i", 3, _glNormal3i_args};

static const char * _glNormal3iv_args[1] = {"v"};
static const trace::FunctionSig _glNormal3iv_sig = {223, "glNormal3iv", 1, _glNormal3iv_args};

static const char * _glNormal3s_args[3] = {"nx", "ny", "nz"};
static const trace::FunctionSig _glNormal3s_sig = {224, "glNormal3s", 3, _glNormal3s_args};

static const char * _glNormal3sv_args[1] = {"v"};
static const trace::FunctionSig _glNormal3sv_sig = {225, "glNormal3sv", 1, _glNormal3sv_args};

static const char * _glRasterPos2d_args[2] = {"x", "y"};
static const trace::FunctionSig _glRasterPos2d_sig = {226, "glRasterPos2d", 2, _glRasterPos2d_args};

static const char * _glRasterPos2dv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos2dv_sig = {227, "glRasterPos2dv", 1, _glRasterPos2dv_args};

static const char * _glRasterPos2f_args[2] = {"x", "y"};
static const trace::FunctionSig _glRasterPos2f_sig = {228, "glRasterPos2f", 2, _glRasterPos2f_args};

static const char * _glRasterPos2fv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos2fv_sig = {229, "glRasterPos2fv", 1, _glRasterPos2fv_args};

static const char * _glRasterPos2i_args[2] = {"x", "y"};
static const trace::FunctionSig _glRasterPos2i_sig = {230, "glRasterPos2i", 2, _glRasterPos2i_args};

static const char * _glRasterPos2iv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos2iv_sig = {231, "glRasterPos2iv", 1, _glRasterPos2iv_args};

static const char * _glRasterPos2s_args[2] = {"x", "y"};
static const trace::FunctionSig _glRasterPos2s_sig = {232, "glRasterPos2s", 2, _glRasterPos2s_args};

static const char * _glRasterPos2sv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos2sv_sig = {233, "glRasterPos2sv", 1, _glRasterPos2sv_args};

static const char * _glRasterPos3d_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glRasterPos3d_sig = {234, "glRasterPos3d", 3, _glRasterPos3d_args};

static const char * _glRasterPos3dv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos3dv_sig = {235, "glRasterPos3dv", 1, _glRasterPos3dv_args};

static const char * _glRasterPos3f_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glRasterPos3f_sig = {236, "glRasterPos3f", 3, _glRasterPos3f_args};

static const char * _glRasterPos3fv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos3fv_sig = {237, "glRasterPos3fv", 1, _glRasterPos3fv_args};

static const char * _glRasterPos3i_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glRasterPos3i_sig = {238, "glRasterPos3i", 3, _glRasterPos3i_args};

static const char * _glRasterPos3iv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos3iv_sig = {239, "glRasterPos3iv", 1, _glRasterPos3iv_args};

static const char * _glRasterPos3s_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glRasterPos3s_sig = {240, "glRasterPos3s", 3, _glRasterPos3s_args};

static const char * _glRasterPos3sv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos3sv_sig = {241, "glRasterPos3sv", 1, _glRasterPos3sv_args};

static const char * _glRasterPos4d_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glRasterPos4d_sig = {242, "glRasterPos4d", 4, _glRasterPos4d_args};

static const char * _glRasterPos4dv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos4dv_sig = {243, "glRasterPos4dv", 1, _glRasterPos4dv_args};

static const char * _glRasterPos4f_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glRasterPos4f_sig = {244, "glRasterPos4f", 4, _glRasterPos4f_args};

static const char * _glRasterPos4fv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos4fv_sig = {245, "glRasterPos4fv", 1, _glRasterPos4fv_args};

static const char * _glRasterPos4i_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glRasterPos4i_sig = {246, "glRasterPos4i", 4, _glRasterPos4i_args};

static const char * _glRasterPos4iv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos4iv_sig = {247, "glRasterPos4iv", 1, _glRasterPos4iv_args};

static const char * _glRasterPos4s_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glRasterPos4s_sig = {248, "glRasterPos4s", 4, _glRasterPos4s_args};

static const char * _glRasterPos4sv_args[1] = {"v"};
static const trace::FunctionSig _glRasterPos4sv_sig = {249, "glRasterPos4sv", 1, _glRasterPos4sv_args};

static const char * _glRectd_args[4] = {"x1", "y1", "x2", "y2"};
static const trace::FunctionSig _glRectd_sig = {250, "glRectd", 4, _glRectd_args};

static const char * _glRectdv_args[2] = {"v1", "v2"};
static const trace::FunctionSig _glRectdv_sig = {251, "glRectdv", 2, _glRectdv_args};

static const char * _glRectf_args[4] = {"x1", "y1", "x2", "y2"};
static const trace::FunctionSig _glRectf_sig = {252, "glRectf", 4, _glRectf_args};

static const char * _glRectfv_args[2] = {"v1", "v2"};
static const trace::FunctionSig _glRectfv_sig = {253, "glRectfv", 2, _glRectfv_args};

static const char * _glRecti_args[4] = {"x1", "y1", "x2", "y2"};
static const trace::FunctionSig _glRecti_sig = {254, "glRecti", 4, _glRecti_args};

static const char * _glRectiv_args[2] = {"v1", "v2"};
static const trace::FunctionSig _glRectiv_sig = {255, "glRectiv", 2, _glRectiv_args};

static const char * _glRects_args[4] = {"x1", "y1", "x2", "y2"};
static const trace::FunctionSig _glRects_sig = {256, "glRects", 4, _glRects_args};

static const char * _glRectsv_args[2] = {"v1", "v2"};
static const trace::FunctionSig _glRectsv_sig = {257, "glRectsv", 2, _glRectsv_args};

static const char * _glTexCoord1d_args[1] = {"s"};
static const trace::FunctionSig _glTexCoord1d_sig = {258, "glTexCoord1d", 1, _glTexCoord1d_args};

static const char * _glTexCoord1dv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord1dv_sig = {259, "glTexCoord1dv", 1, _glTexCoord1dv_args};

static const char * _glTexCoord1f_args[1] = {"s"};
static const trace::FunctionSig _glTexCoord1f_sig = {260, "glTexCoord1f", 1, _glTexCoord1f_args};

static const char * _glTexCoord1fv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord1fv_sig = {261, "glTexCoord1fv", 1, _glTexCoord1fv_args};

static const char * _glTexCoord1i_args[1] = {"s"};
static const trace::FunctionSig _glTexCoord1i_sig = {262, "glTexCoord1i", 1, _glTexCoord1i_args};

static const char * _glTexCoord1iv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord1iv_sig = {263, "glTexCoord1iv", 1, _glTexCoord1iv_args};

static const char * _glTexCoord1s_args[1] = {"s"};
static const trace::FunctionSig _glTexCoord1s_sig = {264, "glTexCoord1s", 1, _glTexCoord1s_args};

static const char * _glTexCoord1sv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord1sv_sig = {265, "glTexCoord1sv", 1, _glTexCoord1sv_args};

static const char * _glTexCoord2d_args[2] = {"s", "t"};
static const trace::FunctionSig _glTexCoord2d_sig = {266, "glTexCoord2d", 2, _glTexCoord2d_args};

static const char * _glTexCoord2dv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord2dv_sig = {267, "glTexCoord2dv", 1, _glTexCoord2dv_args};

static const char * _glTexCoord2f_args[2] = {"s", "t"};
static const trace::FunctionSig _glTexCoord2f_sig = {268, "glTexCoord2f", 2, _glTexCoord2f_args};

static const char * _glTexCoord2fv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord2fv_sig = {269, "glTexCoord2fv", 1, _glTexCoord2fv_args};

static const char * _glTexCoord2i_args[2] = {"s", "t"};
static const trace::FunctionSig _glTexCoord2i_sig = {270, "glTexCoord2i", 2, _glTexCoord2i_args};

static const char * _glTexCoord2iv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord2iv_sig = {271, "glTexCoord2iv", 1, _glTexCoord2iv_args};

static const char * _glTexCoord2s_args[2] = {"s", "t"};
static const trace::FunctionSig _glTexCoord2s_sig = {272, "glTexCoord2s", 2, _glTexCoord2s_args};

static const char * _glTexCoord2sv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord2sv_sig = {273, "glTexCoord2sv", 1, _glTexCoord2sv_args};

static const char * _glTexCoord3d_args[3] = {"s", "t", "r"};
static const trace::FunctionSig _glTexCoord3d_sig = {274, "glTexCoord3d", 3, _glTexCoord3d_args};

static const char * _glTexCoord3dv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord3dv_sig = {275, "glTexCoord3dv", 1, _glTexCoord3dv_args};

static const char * _glTexCoord3f_args[3] = {"s", "t", "r"};
static const trace::FunctionSig _glTexCoord3f_sig = {276, "glTexCoord3f", 3, _glTexCoord3f_args};

static const char * _glTexCoord3fv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord3fv_sig = {277, "glTexCoord3fv", 1, _glTexCoord3fv_args};

static const char * _glTexCoord3i_args[3] = {"s", "t", "r"};
static const trace::FunctionSig _glTexCoord3i_sig = {278, "glTexCoord3i", 3, _glTexCoord3i_args};

static const char * _glTexCoord3iv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord3iv_sig = {279, "glTexCoord3iv", 1, _glTexCoord3iv_args};

static const char * _glTexCoord3s_args[3] = {"s", "t", "r"};
static const trace::FunctionSig _glTexCoord3s_sig = {280, "glTexCoord3s", 3, _glTexCoord3s_args};

static const char * _glTexCoord3sv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord3sv_sig = {281, "glTexCoord3sv", 1, _glTexCoord3sv_args};

static const char * _glTexCoord4d_args[4] = {"s", "t", "r", "q"};
static const trace::FunctionSig _glTexCoord4d_sig = {282, "glTexCoord4d", 4, _glTexCoord4d_args};

static const char * _glTexCoord4dv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord4dv_sig = {283, "glTexCoord4dv", 1, _glTexCoord4dv_args};

static const char * _glTexCoord4f_args[4] = {"s", "t", "r", "q"};
static const trace::FunctionSig _glTexCoord4f_sig = {284, "glTexCoord4f", 4, _glTexCoord4f_args};

static const char * _glTexCoord4fv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord4fv_sig = {285, "glTexCoord4fv", 1, _glTexCoord4fv_args};

static const char * _glTexCoord4i_args[4] = {"s", "t", "r", "q"};
static const trace::FunctionSig _glTexCoord4i_sig = {286, "glTexCoord4i", 4, _glTexCoord4i_args};

static const char * _glTexCoord4iv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord4iv_sig = {287, "glTexCoord4iv", 1, _glTexCoord4iv_args};

static const char * _glTexCoord4s_args[4] = {"s", "t", "r", "q"};
static const trace::FunctionSig _glTexCoord4s_sig = {288, "glTexCoord4s", 4, _glTexCoord4s_args};

static const char * _glTexCoord4sv_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord4sv_sig = {289, "glTexCoord4sv", 1, _glTexCoord4sv_args};

static const char * _glVertex2d_args[2] = {"x", "y"};
static const trace::FunctionSig _glVertex2d_sig = {290, "glVertex2d", 2, _glVertex2d_args};

static const char * _glVertex2dv_args[1] = {"v"};
static const trace::FunctionSig _glVertex2dv_sig = {291, "glVertex2dv", 1, _glVertex2dv_args};

static const char * _glVertex2f_args[2] = {"x", "y"};
static const trace::FunctionSig _glVertex2f_sig = {292, "glVertex2f", 2, _glVertex2f_args};

static const char * _glVertex2fv_args[1] = {"v"};
static const trace::FunctionSig _glVertex2fv_sig = {293, "glVertex2fv", 1, _glVertex2fv_args};

static const char * _glVertex2i_args[2] = {"x", "y"};
static const trace::FunctionSig _glVertex2i_sig = {294, "glVertex2i", 2, _glVertex2i_args};

static const char * _glVertex2iv_args[1] = {"v"};
static const trace::FunctionSig _glVertex2iv_sig = {295, "glVertex2iv", 1, _glVertex2iv_args};

static const char * _glVertex2s_args[2] = {"x", "y"};
static const trace::FunctionSig _glVertex2s_sig = {296, "glVertex2s", 2, _glVertex2s_args};

static const char * _glVertex2sv_args[1] = {"v"};
static const trace::FunctionSig _glVertex2sv_sig = {297, "glVertex2sv", 1, _glVertex2sv_args};

static const char * _glVertex3d_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glVertex3d_sig = {298, "glVertex3d", 3, _glVertex3d_args};

static const char * _glVertex3dv_args[1] = {"v"};
static const trace::FunctionSig _glVertex3dv_sig = {299, "glVertex3dv", 1, _glVertex3dv_args};

static const char * _glVertex3f_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glVertex3f_sig = {300, "glVertex3f", 3, _glVertex3f_args};

static const char * _glVertex3fv_args[1] = {"v"};
static const trace::FunctionSig _glVertex3fv_sig = {301, "glVertex3fv", 1, _glVertex3fv_args};

static const char * _glVertex3i_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glVertex3i_sig = {302, "glVertex3i", 3, _glVertex3i_args};

static const char * _glVertex3iv_args[1] = {"v"};
static const trace::FunctionSig _glVertex3iv_sig = {303, "glVertex3iv", 1, _glVertex3iv_args};

static const char * _glVertex3s_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glVertex3s_sig = {304, "glVertex3s", 3, _glVertex3s_args};

static const char * _glVertex3sv_args[1] = {"v"};
static const trace::FunctionSig _glVertex3sv_sig = {305, "glVertex3sv", 1, _glVertex3sv_args};

static const char * _glVertex4d_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glVertex4d_sig = {306, "glVertex4d", 4, _glVertex4d_args};

static const char * _glVertex4dv_args[1] = {"v"};
static const trace::FunctionSig _glVertex4dv_sig = {307, "glVertex4dv", 1, _glVertex4dv_args};

static const char * _glVertex4f_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glVertex4f_sig = {308, "glVertex4f", 4, _glVertex4f_args};

static const char * _glVertex4fv_args[1] = {"v"};
static const trace::FunctionSig _glVertex4fv_sig = {309, "glVertex4fv", 1, _glVertex4fv_args};

static const char * _glVertex4i_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glVertex4i_sig = {310, "glVertex4i", 4, _glVertex4i_args};

static const char * _glVertex4iv_args[1] = {"v"};
static const trace::FunctionSig _glVertex4iv_sig = {311, "glVertex4iv", 1, _glVertex4iv_args};

static const char * _glVertex4s_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glVertex4s_sig = {312, "glVertex4s", 4, _glVertex4s_args};

static const char * _glVertex4sv_args[1] = {"v"};
static const trace::FunctionSig _glVertex4sv_sig = {313, "glVertex4sv", 1, _glVertex4sv_args};

static const char * _glClipPlane_args[2] = {"plane", "equation"};
static const trace::FunctionSig _glClipPlane_sig = {314, "glClipPlane", 2, _glClipPlane_args};

static const char * _glColorMaterial_args[2] = {"face", "mode"};
static const trace::FunctionSig _glColorMaterial_sig = {315, "glColorMaterial", 2, _glColorMaterial_args};

static const char * _glFogf_args[2] = {"pname", "param"};
static const trace::FunctionSig _glFogf_sig = {316, "glFogf", 2, _glFogf_args};

static const char * _glFogfv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glFogfv_sig = {317, "glFogfv", 2, _glFogfv_args};

static const char * _glFogi_args[2] = {"pname", "param"};
static const trace::FunctionSig _glFogi_sig = {318, "glFogi", 2, _glFogi_args};

static const char * _glFogiv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glFogiv_sig = {319, "glFogiv", 2, _glFogiv_args};

static const char * _glLightf_args[3] = {"light", "pname", "param"};
static const trace::FunctionSig _glLightf_sig = {320, "glLightf", 3, _glLightf_args};

static const char * _glLightfv_args[3] = {"light", "pname", "params"};
static const trace::FunctionSig _glLightfv_sig = {321, "glLightfv", 3, _glLightfv_args};

static const char * _glLighti_args[3] = {"light", "pname", "param"};
static const trace::FunctionSig _glLighti_sig = {322, "glLighti", 3, _glLighti_args};

static const char * _glLightiv_args[3] = {"light", "pname", "params"};
static const trace::FunctionSig _glLightiv_sig = {323, "glLightiv", 3, _glLightiv_args};

static const char * _glLightModelf_args[2] = {"pname", "param"};
static const trace::FunctionSig _glLightModelf_sig = {324, "glLightModelf", 2, _glLightModelf_args};

static const char * _glLightModelfv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glLightModelfv_sig = {325, "glLightModelfv", 2, _glLightModelfv_args};

static const char * _glLightModeli_args[2] = {"pname", "param"};
static const trace::FunctionSig _glLightModeli_sig = {326, "glLightModeli", 2, _glLightModeli_args};

static const char * _glLightModeliv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glLightModeliv_sig = {327, "glLightModeliv", 2, _glLightModeliv_args};

static const char * _glLineStipple_args[2] = {"factor", "pattern"};
static const trace::FunctionSig _glLineStipple_sig = {328, "glLineStipple", 2, _glLineStipple_args};

static const char * _glMaterialf_args[3] = {"face", "pname", "param"};
static const trace::FunctionSig _glMaterialf_sig = {329, "glMaterialf", 3, _glMaterialf_args};

static const char * _glMaterialfv_args[3] = {"face", "pname", "params"};
static const trace::FunctionSig _glMaterialfv_sig = {330, "glMaterialfv", 3, _glMaterialfv_args};

static const char * _glMateriali_args[3] = {"face", "pname", "param"};
static const trace::FunctionSig _glMateriali_sig = {331, "glMateriali", 3, _glMateriali_args};

static const char * _glMaterialiv_args[3] = {"face", "pname", "params"};
static const trace::FunctionSig _glMaterialiv_sig = {332, "glMaterialiv", 3, _glMaterialiv_args};

static const char * _glPolygonStipple_args[1] = {"mask"};
static const trace::FunctionSig _glPolygonStipple_sig = {333, "glPolygonStipple", 1, _glPolygonStipple_args};

static const char * _glShadeModel_args[1] = {"mode"};
static const trace::FunctionSig _glShadeModel_sig = {334, "glShadeModel", 1, _glShadeModel_args};

static const char * _glTexEnvf_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glTexEnvf_sig = {335, "glTexEnvf", 3, _glTexEnvf_args};

static const char * _glTexEnvfv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glTexEnvfv_sig = {336, "glTexEnvfv", 3, _glTexEnvfv_args};

static const char * _glTexEnvi_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glTexEnvi_sig = {337, "glTexEnvi", 3, _glTexEnvi_args};

static const char * _glTexEnviv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glTexEnviv_sig = {338, "glTexEnviv", 3, _glTexEnviv_args};

static const char * _glTexGend_args[3] = {"coord", "pname", "param"};
static const trace::FunctionSig _glTexGend_sig = {339, "glTexGend", 3, _glTexGend_args};

static const char * _glTexGendv_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glTexGendv_sig = {340, "glTexGendv", 3, _glTexGendv_args};

static const char * _glTexGenf_args[3] = {"coord", "pname", "param"};
static const trace::FunctionSig _glTexGenf_sig = {341, "glTexGenf", 3, _glTexGenf_args};

static const char * _glTexGenfv_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glTexGenfv_sig = {342, "glTexGenfv", 3, _glTexGenfv_args};

static const char * _glTexGeni_args[3] = {"coord", "pname", "param"};
static const trace::FunctionSig _glTexGeni_sig = {343, "glTexGeni", 3, _glTexGeni_args};

static const char * _glTexGeniv_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glTexGeniv_sig = {344, "glTexGeniv", 3, _glTexGeniv_args};

static const char * _glFeedbackBuffer_args[3] = {"size", "type", "buffer"};
static const trace::FunctionSig _glFeedbackBuffer_sig = {345, "glFeedbackBuffer", 3, _glFeedbackBuffer_args};

static const char * _glSelectBuffer_args[2] = {"size", "buffer"};
static const trace::FunctionSig _glSelectBuffer_sig = {346, "glSelectBuffer", 2, _glSelectBuffer_args};

static const char * _glRenderMode_args[1] = {"mode"};
static const trace::FunctionSig _glRenderMode_sig = {347, "glRenderMode", 1, _glRenderMode_args};

static const char ** _glInitNames_args = NULL;
static const trace::FunctionSig _glInitNames_sig = {348, "glInitNames", 0, _glInitNames_args};

static const char * _glLoadName_args[1] = {"name"};
static const trace::FunctionSig _glLoadName_sig = {349, "glLoadName", 1, _glLoadName_args};

static const char * _glPassThrough_args[1] = {"token"};
static const trace::FunctionSig _glPassThrough_sig = {350, "glPassThrough", 1, _glPassThrough_args};

static const char ** _glPopName_args = NULL;
static const trace::FunctionSig _glPopName_sig = {351, "glPopName", 0, _glPopName_args};

static const char * _glPushName_args[1] = {"name"};
static const trace::FunctionSig _glPushName_sig = {352, "glPushName", 1, _glPushName_args};

static const char * _glClearAccum_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glClearAccum_sig = {353, "glClearAccum", 4, _glClearAccum_args};

static const char * _glClearIndex_args[1] = {"c"};
static const trace::FunctionSig _glClearIndex_sig = {354, "glClearIndex", 1, _glClearIndex_args};

static const char * _glIndexMask_args[1] = {"mask"};
static const trace::FunctionSig _glIndexMask_sig = {355, "glIndexMask", 1, _glIndexMask_args};

static const char * _glAccum_args[2] = {"op", "value"};
static const trace::FunctionSig _glAccum_sig = {356, "glAccum", 2, _glAccum_args};

static const char ** _glPopAttrib_args = NULL;
static const trace::FunctionSig _glPopAttrib_sig = {357, "glPopAttrib", 0, _glPopAttrib_args};

static const char * _glPushAttrib_args[1] = {"mask"};
static const trace::FunctionSig _glPushAttrib_sig = {358, "glPushAttrib", 1, _glPushAttrib_args};

static const char * _glMap1d_args[6] = {"target", "u1", "u2", "stride", "order", "points"};
static const trace::FunctionSig _glMap1d_sig = {359, "glMap1d", 6, _glMap1d_args};

static const char * _glMap1f_args[6] = {"target", "u1", "u2", "stride", "order", "points"};
static const trace::FunctionSig _glMap1f_sig = {360, "glMap1f", 6, _glMap1f_args};

static const char * _glMap2d_args[10] = {"target", "u1", "u2", "ustride", "uorder", "v1", "v2", "vstride", "vorder", "points"};
static const trace::FunctionSig _glMap2d_sig = {361, "glMap2d", 10, _glMap2d_args};

static const char * _glMap2f_args[10] = {"target", "u1", "u2", "ustride", "uorder", "v1", "v2", "vstride", "vorder", "points"};
static const trace::FunctionSig _glMap2f_sig = {362, "glMap2f", 10, _glMap2f_args};

static const char * _glMapGrid1d_args[3] = {"un", "u1", "u2"};
static const trace::FunctionSig _glMapGrid1d_sig = {363, "glMapGrid1d", 3, _glMapGrid1d_args};

static const char * _glMapGrid1f_args[3] = {"un", "u1", "u2"};
static const trace::FunctionSig _glMapGrid1f_sig = {364, "glMapGrid1f", 3, _glMapGrid1f_args};

static const char * _glMapGrid2d_args[6] = {"un", "u1", "u2", "vn", "v1", "v2"};
static const trace::FunctionSig _glMapGrid2d_sig = {365, "glMapGrid2d", 6, _glMapGrid2d_args};

static const char * _glMapGrid2f_args[6] = {"un", "u1", "u2", "vn", "v1", "v2"};
static const trace::FunctionSig _glMapGrid2f_sig = {366, "glMapGrid2f", 6, _glMapGrid2f_args};

static const char * _glEvalCoord1d_args[1] = {"u"};
static const trace::FunctionSig _glEvalCoord1d_sig = {367, "glEvalCoord1d", 1, _glEvalCoord1d_args};

static const char * _glEvalCoord1dv_args[1] = {"u"};
static const trace::FunctionSig _glEvalCoord1dv_sig = {368, "glEvalCoord1dv", 1, _glEvalCoord1dv_args};

static const char * _glEvalCoord1f_args[1] = {"u"};
static const trace::FunctionSig _glEvalCoord1f_sig = {369, "glEvalCoord1f", 1, _glEvalCoord1f_args};

static const char * _glEvalCoord1fv_args[1] = {"u"};
static const trace::FunctionSig _glEvalCoord1fv_sig = {370, "glEvalCoord1fv", 1, _glEvalCoord1fv_args};

static const char * _glEvalCoord2d_args[2] = {"u", "v"};
static const trace::FunctionSig _glEvalCoord2d_sig = {371, "glEvalCoord2d", 2, _glEvalCoord2d_args};

static const char * _glEvalCoord2dv_args[1] = {"u"};
static const trace::FunctionSig _glEvalCoord2dv_sig = {372, "glEvalCoord2dv", 1, _glEvalCoord2dv_args};

static const char * _glEvalCoord2f_args[2] = {"u", "v"};
static const trace::FunctionSig _glEvalCoord2f_sig = {373, "glEvalCoord2f", 2, _glEvalCoord2f_args};

static const char * _glEvalCoord2fv_args[1] = {"u"};
static const trace::FunctionSig _glEvalCoord2fv_sig = {374, "glEvalCoord2fv", 1, _glEvalCoord2fv_args};

static const char * _glEvalMesh1_args[3] = {"mode", "i1", "i2"};
static const trace::FunctionSig _glEvalMesh1_sig = {375, "glEvalMesh1", 3, _glEvalMesh1_args};

static const char * _glEvalPoint1_args[1] = {"i"};
static const trace::FunctionSig _glEvalPoint1_sig = {376, "glEvalPoint1", 1, _glEvalPoint1_args};

static const char * _glEvalMesh2_args[5] = {"mode", "i1", "i2", "j1", "j2"};
static const trace::FunctionSig _glEvalMesh2_sig = {377, "glEvalMesh2", 5, _glEvalMesh2_args};

static const char * _glEvalPoint2_args[2] = {"i", "j"};
static const trace::FunctionSig _glEvalPoint2_sig = {378, "glEvalPoint2", 2, _glEvalPoint2_args};

static const char * _glAlphaFunc_args[2] = {"func", "ref"};
static const trace::FunctionSig _glAlphaFunc_sig = {379, "glAlphaFunc", 2, _glAlphaFunc_args};

static const char * _glPixelZoom_args[2] = {"xfactor", "yfactor"};
static const trace::FunctionSig _glPixelZoom_sig = {380, "glPixelZoom", 2, _glPixelZoom_args};

static const char * _glPixelTransferf_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPixelTransferf_sig = {381, "glPixelTransferf", 2, _glPixelTransferf_args};

static const char * _glPixelTransferi_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPixelTransferi_sig = {382, "glPixelTransferi", 2, _glPixelTransferi_args};

static const char * _glPixelMapfv_args[3] = {"map", "mapsize", "values"};
static const trace::FunctionSig _glPixelMapfv_sig = {383, "glPixelMapfv", 3, _glPixelMapfv_args};

static const char * _glPixelMapuiv_args[3] = {"map", "mapsize", "values"};
static const trace::FunctionSig _glPixelMapuiv_sig = {384, "glPixelMapuiv", 3, _glPixelMapuiv_args};

static const char * _glPixelMapusv_args[3] = {"map", "mapsize", "values"};
static const trace::FunctionSig _glPixelMapusv_sig = {385, "glPixelMapusv", 3, _glPixelMapusv_args};

static const char * _glCopyPixels_args[5] = {"x", "y", "width", "height", "type"};
static const trace::FunctionSig _glCopyPixels_sig = {386, "glCopyPixels", 5, _glCopyPixels_args};

static const char * _glDrawPixels_args[5] = {"width", "height", "format", "type", "pixels"};
static const trace::FunctionSig _glDrawPixels_sig = {387, "glDrawPixels", 5, _glDrawPixels_args};

static const char * _glGetClipPlane_args[2] = {"plane", "equation"};
static const trace::FunctionSig _glGetClipPlane_sig = {388, "glGetClipPlane", 2, _glGetClipPlane_args};

static const char * _glGetLightfv_args[3] = {"light", "pname", "params"};
static const trace::FunctionSig _glGetLightfv_sig = {389, "glGetLightfv", 3, _glGetLightfv_args};

static const char * _glGetLightiv_args[3] = {"light", "pname", "params"};
static const trace::FunctionSig _glGetLightiv_sig = {390, "glGetLightiv", 3, _glGetLightiv_args};

static const char * _glGetMapdv_args[3] = {"target", "query", "v"};
static const trace::FunctionSig _glGetMapdv_sig = {391, "glGetMapdv", 3, _glGetMapdv_args};

static const char * _glGetMapfv_args[3] = {"target", "query", "v"};
static const trace::FunctionSig _glGetMapfv_sig = {392, "glGetMapfv", 3, _glGetMapfv_args};

static const char * _glGetMapiv_args[3] = {"target", "query", "v"};
static const trace::FunctionSig _glGetMapiv_sig = {393, "glGetMapiv", 3, _glGetMapiv_args};

static const char * _glGetMaterialfv_args[3] = {"face", "pname", "params"};
static const trace::FunctionSig _glGetMaterialfv_sig = {394, "glGetMaterialfv", 3, _glGetMaterialfv_args};

static const char * _glGetMaterialiv_args[3] = {"face", "pname", "params"};
static const trace::FunctionSig _glGetMaterialiv_sig = {395, "glGetMaterialiv", 3, _glGetMaterialiv_args};

static const char * _glGetPixelMapfv_args[2] = {"map", "values"};
static const trace::FunctionSig _glGetPixelMapfv_sig = {396, "glGetPixelMapfv", 2, _glGetPixelMapfv_args};

static const char * _glGetPixelMapuiv_args[2] = {"map", "values"};
static const trace::FunctionSig _glGetPixelMapuiv_sig = {397, "glGetPixelMapuiv", 2, _glGetPixelMapuiv_args};

static const char * _glGetPixelMapusv_args[2] = {"map", "values"};
static const trace::FunctionSig _glGetPixelMapusv_sig = {398, "glGetPixelMapusv", 2, _glGetPixelMapusv_args};

static const char * _glGetPolygonStipple_args[1] = {"mask"};
static const trace::FunctionSig _glGetPolygonStipple_sig = {399, "glGetPolygonStipple", 1, _glGetPolygonStipple_args};

static const char * _glGetTexEnvfv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexEnvfv_sig = {400, "glGetTexEnvfv", 3, _glGetTexEnvfv_args};

static const char * _glGetTexEnviv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexEnviv_sig = {401, "glGetTexEnviv", 3, _glGetTexEnviv_args};

static const char * _glGetTexGendv_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glGetTexGendv_sig = {402, "glGetTexGendv", 3, _glGetTexGendv_args};

static const char * _glGetTexGenfv_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glGetTexGenfv_sig = {403, "glGetTexGenfv", 3, _glGetTexGenfv_args};

static const char * _glGetTexGeniv_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glGetTexGeniv_sig = {404, "glGetTexGeniv", 3, _glGetTexGeniv_args};

static const char * _glIsList_args[1] = {"list"};
static const trace::FunctionSig _glIsList_sig = {405, "glIsList", 1, _glIsList_args};

static const char * _glFrustum_args[6] = {"left", "right", "bottom", "top", "zNear", "zFar"};
static const trace::FunctionSig _glFrustum_sig = {406, "glFrustum", 6, _glFrustum_args};

static const char ** _glLoadIdentity_args = NULL;
static const trace::FunctionSig _glLoadIdentity_sig = {407, "glLoadIdentity", 0, _glLoadIdentity_args};

static const char * _glLoadMatrixf_args[1] = {"m"};
static const trace::FunctionSig _glLoadMatrixf_sig = {408, "glLoadMatrixf", 1, _glLoadMatrixf_args};

static const char * _glLoadMatrixd_args[1] = {"m"};
static const trace::FunctionSig _glLoadMatrixd_sig = {409, "glLoadMatrixd", 1, _glLoadMatrixd_args};

static const char * _glMatrixMode_args[1] = {"mode"};
static const trace::FunctionSig _glMatrixMode_sig = {410, "glMatrixMode", 1, _glMatrixMode_args};

static const char * _glMultMatrixf_args[1] = {"m"};
static const trace::FunctionSig _glMultMatrixf_sig = {411, "glMultMatrixf", 1, _glMultMatrixf_args};

static const char * _glMultMatrixd_args[1] = {"m"};
static const trace::FunctionSig _glMultMatrixd_sig = {412, "glMultMatrixd", 1, _glMultMatrixd_args};

static const char * _glOrtho_args[6] = {"left", "right", "bottom", "top", "zNear", "zFar"};
static const trace::FunctionSig _glOrtho_sig = {413, "glOrtho", 6, _glOrtho_args};

static const char ** _glPopMatrix_args = NULL;
static const trace::FunctionSig _glPopMatrix_sig = {414, "glPopMatrix", 0, _glPopMatrix_args};

static const char ** _glPushMatrix_args = NULL;
static const trace::FunctionSig _glPushMatrix_sig = {415, "glPushMatrix", 0, _glPushMatrix_args};

static const char * _glRotated_args[4] = {"angle", "x", "y", "z"};
static const trace::FunctionSig _glRotated_sig = {416, "glRotated", 4, _glRotated_args};

static const char * _glRotatef_args[4] = {"angle", "x", "y", "z"};
static const trace::FunctionSig _glRotatef_sig = {417, "glRotatef", 4, _glRotatef_args};

static const char * _glScaled_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glScaled_sig = {418, "glScaled", 3, _glScaled_args};

static const char * _glScalef_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glScalef_sig = {419, "glScalef", 3, _glScalef_args};

static const char * _glTranslated_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glTranslated_sig = {420, "glTranslated", 3, _glTranslated_args};

static const char * _glTranslatef_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glTranslatef_sig = {421, "glTranslatef", 3, _glTranslatef_args};

static const char * _glDrawArrays_args[3] = {"mode", "first", "count"};
static const trace::FunctionSig _glDrawArrays_sig = {422, "glDrawArrays", 3, _glDrawArrays_args};

static const char * _glDrawElements_args[4] = {"mode", "count", "type", "indices"};
static const trace::FunctionSig _glDrawElements_sig = {423, "glDrawElements", 4, _glDrawElements_args};

static const char * _glGetPointerv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetPointerv_sig = {424, "glGetPointerv", 2, _glGetPointerv_args};

static const char * _glPolygonOffset_args[2] = {"factor", "units"};
static const trace::FunctionSig _glPolygonOffset_sig = {425, "glPolygonOffset", 2, _glPolygonOffset_args};

static const char * _glCopyTexImage1D_args[7] = {"target", "level", "internalformat", "x", "y", "width", "border"};
static const trace::FunctionSig _glCopyTexImage1D_sig = {426, "glCopyTexImage1D", 7, _glCopyTexImage1D_args};

static const char * _glCopyTexImage2D_args[8] = {"target", "level", "internalformat", "x", "y", "width", "height", "border"};
static const trace::FunctionSig _glCopyTexImage2D_sig = {427, "glCopyTexImage2D", 8, _glCopyTexImage2D_args};

static const char * _glCopyTexSubImage1D_args[6] = {"target", "level", "xoffset", "x", "y", "width"};
static const trace::FunctionSig _glCopyTexSubImage1D_sig = {428, "glCopyTexSubImage1D", 6, _glCopyTexSubImage1D_args};

static const char * _glCopyTexSubImage2D_args[8] = {"target", "level", "xoffset", "yoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyTexSubImage2D_sig = {429, "glCopyTexSubImage2D", 8, _glCopyTexSubImage2D_args};

static const char * _glTexSubImage1D_args[7] = {"target", "level", "xoffset", "width", "format", "type", "pixels"};
static const trace::FunctionSig _glTexSubImage1D_sig = {430, "glTexSubImage1D", 7, _glTexSubImage1D_args};

static const char * _glTexSubImage2D_args[9] = {"target", "level", "xoffset", "yoffset", "width", "height", "format", "type", "pixels"};
static const trace::FunctionSig _glTexSubImage2D_sig = {431, "glTexSubImage2D", 9, _glTexSubImage2D_args};

static const char * _glBindTexture_args[2] = {"target", "texture"};
static const trace::FunctionSig _glBindTexture_sig = {432, "glBindTexture", 2, _glBindTexture_args};

static const char * _glDeleteTextures_args[2] = {"n", "textures"};
static const trace::FunctionSig _glDeleteTextures_sig = {433, "glDeleteTextures", 2, _glDeleteTextures_args};

static const char * _glGenTextures_args[2] = {"n", "textures"};
static const trace::FunctionSig _glGenTextures_sig = {434, "glGenTextures", 2, _glGenTextures_args};

static const char * _glIsTexture_args[1] = {"texture"};
static const trace::FunctionSig _glIsTexture_sig = {435, "glIsTexture", 1, _glIsTexture_args};

static const char * _glArrayElement_args[1] = {"i"};
static const trace::FunctionSig _glArrayElement_sig = {436, "glArrayElement", 1, _glArrayElement_args};

static const char * _glColorPointer_args[4] = {"size", "type", "stride", "pointer"};
static const trace::FunctionSig _glColorPointer_sig = {437, "glColorPointer", 4, _glColorPointer_args};

static const char * _glDisableClientState_args[1] = {"array"};
static const trace::FunctionSig _glDisableClientState_sig = {438, "glDisableClientState", 1, _glDisableClientState_args};

static const char * _glEdgeFlagPointer_args[2] = {"stride", "pointer"};
static const trace::FunctionSig _glEdgeFlagPointer_sig = {439, "glEdgeFlagPointer", 2, _glEdgeFlagPointer_args};

static const char * _glEnableClientState_args[1] = {"array"};
static const trace::FunctionSig _glEnableClientState_sig = {440, "glEnableClientState", 1, _glEnableClientState_args};

static const char * _glIndexPointer_args[3] = {"type", "stride", "pointer"};
static const trace::FunctionSig _glIndexPointer_sig = {441, "glIndexPointer", 3, _glIndexPointer_args};

static const char * _glInterleavedArrays_args[3] = {"format", "stride", "pointer"};
static const trace::FunctionSig _glInterleavedArrays_sig = {442, "glInterleavedArrays", 3, _glInterleavedArrays_args};

static const char * _glNormalPointer_args[3] = {"type", "stride", "pointer"};
static const trace::FunctionSig _glNormalPointer_sig = {443, "glNormalPointer", 3, _glNormalPointer_args};

static const char * _glTexCoordPointer_args[4] = {"size", "type", "stride", "pointer"};
static const trace::FunctionSig _glTexCoordPointer_sig = {444, "glTexCoordPointer", 4, _glTexCoordPointer_args};

static const char * _glVertexPointer_args[4] = {"size", "type", "stride", "pointer"};
static const trace::FunctionSig _glVertexPointer_sig = {445, "glVertexPointer", 4, _glVertexPointer_args};

static const char * _glAreTexturesResident_args[3] = {"n", "textures", "residences"};
static const trace::FunctionSig _glAreTexturesResident_sig = {446, "glAreTexturesResident", 3, _glAreTexturesResident_args};

static const char * _glPrioritizeTextures_args[3] = {"n", "textures", "priorities"};
static const trace::FunctionSig _glPrioritizeTextures_sig = {447, "glPrioritizeTextures", 3, _glPrioritizeTextures_args};

static const char * _glIndexub_args[1] = {"c"};
static const trace::FunctionSig _glIndexub_sig = {448, "glIndexub", 1, _glIndexub_args};

static const char * _glIndexubv_args[1] = {"c"};
static const trace::FunctionSig _glIndexubv_sig = {449, "glIndexubv", 1, _glIndexubv_args};

static const char ** _glPopClientAttrib_args = NULL;
static const trace::FunctionSig _glPopClientAttrib_sig = {450, "glPopClientAttrib", 0, _glPopClientAttrib_args};

static const char * _glPushClientAttrib_args[1] = {"mask"};
static const trace::FunctionSig _glPushClientAttrib_sig = {451, "glPushClientAttrib", 1, _glPushClientAttrib_args};

static const char * _glBlendColor_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glBlendColor_sig = {452, "glBlendColor", 4, _glBlendColor_args};

static const char * _glBlendEquation_args[1] = {"mode"};
static const trace::FunctionSig _glBlendEquation_sig = {453, "glBlendEquation", 1, _glBlendEquation_args};

static const char * _glDrawRangeElements_args[6] = {"mode", "start", "end", "count", "type", "indices"};
static const trace::FunctionSig _glDrawRangeElements_sig = {454, "glDrawRangeElements", 6, _glDrawRangeElements_args};

static const char * _glTexImage3D_args[10] = {"target", "level", "internalformat", "width", "height", "depth", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glTexImage3D_sig = {455, "glTexImage3D", 10, _glTexImage3D_args};

static const char * _glTexSubImage3D_args[11] = {"target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "type", "pixels"};
static const trace::FunctionSig _glTexSubImage3D_sig = {456, "glTexSubImage3D", 11, _glTexSubImage3D_args};

static const char * _glCopyTexSubImage3D_args[9] = {"target", "level", "xoffset", "yoffset", "zoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyTexSubImage3D_sig = {457, "glCopyTexSubImage3D", 9, _glCopyTexSubImage3D_args};

static const char * _glColorTable_args[6] = {"target", "internalformat", "width", "format", "type", "table"};
static const trace::FunctionSig _glColorTable_sig = {458, "glColorTable", 6, _glColorTable_args};

static const char * _glColorTableParameterfv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glColorTableParameterfv_sig = {459, "glColorTableParameterfv", 3, _glColorTableParameterfv_args};

static const char * _glColorTableParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glColorTableParameteriv_sig = {460, "glColorTableParameteriv", 3, _glColorTableParameteriv_args};

static const char * _glCopyColorTable_args[5] = {"target", "internalformat", "x", "y", "width"};
static const trace::FunctionSig _glCopyColorTable_sig = {461, "glCopyColorTable", 5, _glCopyColorTable_args};

static const char * _glGetColorTable_args[4] = {"target", "format", "type", "table"};
static const trace::FunctionSig _glGetColorTable_sig = {462, "glGetColorTable", 4, _glGetColorTable_args};

static const char * _glGetColorTableParameterfv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetColorTableParameterfv_sig = {463, "glGetColorTableParameterfv", 3, _glGetColorTableParameterfv_args};

static const char * _glGetColorTableParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetColorTableParameteriv_sig = {464, "glGetColorTableParameteriv", 3, _glGetColorTableParameteriv_args};

static const char * _glColorSubTable_args[6] = {"target", "start", "count", "format", "type", "data"};
static const trace::FunctionSig _glColorSubTable_sig = {465, "glColorSubTable", 6, _glColorSubTable_args};

static const char * _glCopyColorSubTable_args[5] = {"target", "start", "x", "y", "width"};
static const trace::FunctionSig _glCopyColorSubTable_sig = {466, "glCopyColorSubTable", 5, _glCopyColorSubTable_args};

static const char * _glConvolutionFilter1D_args[6] = {"target", "internalformat", "width", "format", "type", "image"};
static const trace::FunctionSig _glConvolutionFilter1D_sig = {467, "glConvolutionFilter1D", 6, _glConvolutionFilter1D_args};

static const char * _glConvolutionFilter2D_args[7] = {"target", "internalformat", "width", "height", "format", "type", "image"};
static const trace::FunctionSig _glConvolutionFilter2D_sig = {468, "glConvolutionFilter2D", 7, _glConvolutionFilter2D_args};

static const char * _glConvolutionParameterf_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glConvolutionParameterf_sig = {469, "glConvolutionParameterf", 3, _glConvolutionParameterf_args};

static const char * _glConvolutionParameterfv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glConvolutionParameterfv_sig = {470, "glConvolutionParameterfv", 3, _glConvolutionParameterfv_args};

static const char * _glConvolutionParameteri_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glConvolutionParameteri_sig = {471, "glConvolutionParameteri", 3, _glConvolutionParameteri_args};

static const char * _glConvolutionParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glConvolutionParameteriv_sig = {472, "glConvolutionParameteriv", 3, _glConvolutionParameteriv_args};

static const char * _glCopyConvolutionFilter1D_args[5] = {"target", "internalformat", "x", "y", "width"};
static const trace::FunctionSig _glCopyConvolutionFilter1D_sig = {473, "glCopyConvolutionFilter1D", 5, _glCopyConvolutionFilter1D_args};

static const char * _glCopyConvolutionFilter2D_args[6] = {"target", "internalformat", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyConvolutionFilter2D_sig = {474, "glCopyConvolutionFilter2D", 6, _glCopyConvolutionFilter2D_args};

static const char * _glGetConvolutionFilter_args[4] = {"target", "format", "type", "image"};
static const trace::FunctionSig _glGetConvolutionFilter_sig = {475, "glGetConvolutionFilter", 4, _glGetConvolutionFilter_args};

static const char * _glGetConvolutionParameterfv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetConvolutionParameterfv_sig = {476, "glGetConvolutionParameterfv", 3, _glGetConvolutionParameterfv_args};

static const char * _glGetConvolutionParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetConvolutionParameteriv_sig = {477, "glGetConvolutionParameteriv", 3, _glGetConvolutionParameteriv_args};

static const char * _glGetSeparableFilter_args[6] = {"target", "format", "type", "row", "column", "span"};
static const trace::FunctionSig _glGetSeparableFilter_sig = {478, "glGetSeparableFilter", 6, _glGetSeparableFilter_args};

static const char * _glSeparableFilter2D_args[8] = {"target", "internalformat", "width", "height", "format", "type", "row", "column"};
static const trace::FunctionSig _glSeparableFilter2D_sig = {479, "glSeparableFilter2D", 8, _glSeparableFilter2D_args};

static const char * _glGetHistogram_args[5] = {"target", "reset", "format", "type", "values"};
static const trace::FunctionSig _glGetHistogram_sig = {480, "glGetHistogram", 5, _glGetHistogram_args};

static const char * _glGetHistogramParameterfv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetHistogramParameterfv_sig = {481, "glGetHistogramParameterfv", 3, _glGetHistogramParameterfv_args};

static const char * _glGetHistogramParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetHistogramParameteriv_sig = {482, "glGetHistogramParameteriv", 3, _glGetHistogramParameteriv_args};

static const char * _glGetMinmax_args[5] = {"target", "reset", "format", "type", "values"};
static const trace::FunctionSig _glGetMinmax_sig = {483, "glGetMinmax", 5, _glGetMinmax_args};

static const char * _glGetMinmaxParameterfv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetMinmaxParameterfv_sig = {484, "glGetMinmaxParameterfv", 3, _glGetMinmaxParameterfv_args};

static const char * _glGetMinmaxParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetMinmaxParameteriv_sig = {485, "glGetMinmaxParameteriv", 3, _glGetMinmaxParameteriv_args};

static const char * _glHistogram_args[4] = {"target", "width", "internalformat", "sink"};
static const trace::FunctionSig _glHistogram_sig = {486, "glHistogram", 4, _glHistogram_args};

static const char * _glMinmax_args[3] = {"target", "internalformat", "sink"};
static const trace::FunctionSig _glMinmax_sig = {487, "glMinmax", 3, _glMinmax_args};

static const char * _glResetHistogram_args[1] = {"target"};
static const trace::FunctionSig _glResetHistogram_sig = {488, "glResetHistogram", 1, _glResetHistogram_args};

static const char * _glResetMinmax_args[1] = {"target"};
static const trace::FunctionSig _glResetMinmax_sig = {489, "glResetMinmax", 1, _glResetMinmax_args};

static const char * _glActiveTexture_args[1] = {"texture"};
static const trace::FunctionSig _glActiveTexture_sig = {490, "glActiveTexture", 1, _glActiveTexture_args};

static const char * _glSampleCoverage_args[2] = {"value", "invert"};
static const trace::FunctionSig _glSampleCoverage_sig = {491, "glSampleCoverage", 2, _glSampleCoverage_args};

static const char * _glCompressedTexImage3D_args[9] = {"target", "level", "internalformat", "width", "height", "depth", "border", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexImage3D_sig = {492, "glCompressedTexImage3D", 9, _glCompressedTexImage3D_args};

static const char * _glCompressedTexImage2D_args[8] = {"target", "level", "internalformat", "width", "height", "border", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexImage2D_sig = {493, "glCompressedTexImage2D", 8, _glCompressedTexImage2D_args};

static const char * _glCompressedTexImage1D_args[7] = {"target", "level", "internalformat", "width", "border", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexImage1D_sig = {494, "glCompressedTexImage1D", 7, _glCompressedTexImage1D_args};

static const char * _glCompressedTexSubImage3D_args[11] = {"target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexSubImage3D_sig = {495, "glCompressedTexSubImage3D", 11, _glCompressedTexSubImage3D_args};

static const char * _glCompressedTexSubImage2D_args[9] = {"target", "level", "xoffset", "yoffset", "width", "height", "format", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexSubImage2D_sig = {496, "glCompressedTexSubImage2D", 9, _glCompressedTexSubImage2D_args};

static const char * _glCompressedTexSubImage1D_args[7] = {"target", "level", "xoffset", "width", "format", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexSubImage1D_sig = {497, "glCompressedTexSubImage1D", 7, _glCompressedTexSubImage1D_args};

static const char * _glGetCompressedTexImage_args[3] = {"target", "level", "img"};
static const trace::FunctionSig _glGetCompressedTexImage_sig = {498, "glGetCompressedTexImage", 3, _glGetCompressedTexImage_args};

static const char * _glClientActiveTexture_args[1] = {"texture"};
static const trace::FunctionSig _glClientActiveTexture_sig = {499, "glClientActiveTexture", 1, _glClientActiveTexture_args};

static const char * _glMultiTexCoord1d_args[2] = {"target", "s"};
static const trace::FunctionSig _glMultiTexCoord1d_sig = {500, "glMultiTexCoord1d", 2, _glMultiTexCoord1d_args};

static const char * _glMultiTexCoord1dv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord1dv_sig = {501, "glMultiTexCoord1dv", 2, _glMultiTexCoord1dv_args};

static const char * _glMultiTexCoord1f_args[2] = {"target", "s"};
static const trace::FunctionSig _glMultiTexCoord1f_sig = {502, "glMultiTexCoord1f", 2, _glMultiTexCoord1f_args};

static const char * _glMultiTexCoord1fv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord1fv_sig = {503, "glMultiTexCoord1fv", 2, _glMultiTexCoord1fv_args};

static const char * _glMultiTexCoord1i_args[2] = {"target", "s"};
static const trace::FunctionSig _glMultiTexCoord1i_sig = {504, "glMultiTexCoord1i", 2, _glMultiTexCoord1i_args};

static const char * _glMultiTexCoord1iv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord1iv_sig = {505, "glMultiTexCoord1iv", 2, _glMultiTexCoord1iv_args};

static const char * _glMultiTexCoord1s_args[2] = {"target", "s"};
static const trace::FunctionSig _glMultiTexCoord1s_sig = {506, "glMultiTexCoord1s", 2, _glMultiTexCoord1s_args};

static const char * _glMultiTexCoord1sv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord1sv_sig = {507, "glMultiTexCoord1sv", 2, _glMultiTexCoord1sv_args};

static const char * _glMultiTexCoord2d_args[3] = {"target", "s", "t"};
static const trace::FunctionSig _glMultiTexCoord2d_sig = {508, "glMultiTexCoord2d", 3, _glMultiTexCoord2d_args};

static const char * _glMultiTexCoord2dv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord2dv_sig = {509, "glMultiTexCoord2dv", 2, _glMultiTexCoord2dv_args};

static const char * _glMultiTexCoord2f_args[3] = {"target", "s", "t"};
static const trace::FunctionSig _glMultiTexCoord2f_sig = {510, "glMultiTexCoord2f", 3, _glMultiTexCoord2f_args};

static const char * _glMultiTexCoord2fv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord2fv_sig = {511, "glMultiTexCoord2fv", 2, _glMultiTexCoord2fv_args};

static const char * _glMultiTexCoord2i_args[3] = {"target", "s", "t"};
static const trace::FunctionSig _glMultiTexCoord2i_sig = {512, "glMultiTexCoord2i", 3, _glMultiTexCoord2i_args};

static const char * _glMultiTexCoord2iv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord2iv_sig = {513, "glMultiTexCoord2iv", 2, _glMultiTexCoord2iv_args};

static const char * _glMultiTexCoord2s_args[3] = {"target", "s", "t"};
static const trace::FunctionSig _glMultiTexCoord2s_sig = {514, "glMultiTexCoord2s", 3, _glMultiTexCoord2s_args};

static const char * _glMultiTexCoord2sv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord2sv_sig = {515, "glMultiTexCoord2sv", 2, _glMultiTexCoord2sv_args};

static const char * _glMultiTexCoord3d_args[4] = {"target", "s", "t", "r"};
static const trace::FunctionSig _glMultiTexCoord3d_sig = {516, "glMultiTexCoord3d", 4, _glMultiTexCoord3d_args};

static const char * _glMultiTexCoord3dv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord3dv_sig = {517, "glMultiTexCoord3dv", 2, _glMultiTexCoord3dv_args};

static const char * _glMultiTexCoord3f_args[4] = {"target", "s", "t", "r"};
static const trace::FunctionSig _glMultiTexCoord3f_sig = {518, "glMultiTexCoord3f", 4, _glMultiTexCoord3f_args};

static const char * _glMultiTexCoord3fv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord3fv_sig = {519, "glMultiTexCoord3fv", 2, _glMultiTexCoord3fv_args};

static const char * _glMultiTexCoord3i_args[4] = {"target", "s", "t", "r"};
static const trace::FunctionSig _glMultiTexCoord3i_sig = {520, "glMultiTexCoord3i", 4, _glMultiTexCoord3i_args};

static const char * _glMultiTexCoord3iv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord3iv_sig = {521, "glMultiTexCoord3iv", 2, _glMultiTexCoord3iv_args};

static const char * _glMultiTexCoord3s_args[4] = {"target", "s", "t", "r"};
static const trace::FunctionSig _glMultiTexCoord3s_sig = {522, "glMultiTexCoord3s", 4, _glMultiTexCoord3s_args};

static const char * _glMultiTexCoord3sv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord3sv_sig = {523, "glMultiTexCoord3sv", 2, _glMultiTexCoord3sv_args};

static const char * _glMultiTexCoord4d_args[5] = {"target", "s", "t", "r", "q"};
static const trace::FunctionSig _glMultiTexCoord4d_sig = {524, "glMultiTexCoord4d", 5, _glMultiTexCoord4d_args};

static const char * _glMultiTexCoord4dv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord4dv_sig = {525, "glMultiTexCoord4dv", 2, _glMultiTexCoord4dv_args};

static const char * _glMultiTexCoord4f_args[5] = {"target", "s", "t", "r", "q"};
static const trace::FunctionSig _glMultiTexCoord4f_sig = {526, "glMultiTexCoord4f", 5, _glMultiTexCoord4f_args};

static const char * _glMultiTexCoord4fv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord4fv_sig = {527, "glMultiTexCoord4fv", 2, _glMultiTexCoord4fv_args};

static const char * _glMultiTexCoord4i_args[5] = {"target", "s", "t", "r", "q"};
static const trace::FunctionSig _glMultiTexCoord4i_sig = {528, "glMultiTexCoord4i", 5, _glMultiTexCoord4i_args};

static const char * _glMultiTexCoord4iv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord4iv_sig = {529, "glMultiTexCoord4iv", 2, _glMultiTexCoord4iv_args};

static const char * _glMultiTexCoord4s_args[5] = {"target", "s", "t", "r", "q"};
static const trace::FunctionSig _glMultiTexCoord4s_sig = {530, "glMultiTexCoord4s", 5, _glMultiTexCoord4s_args};

static const char * _glMultiTexCoord4sv_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord4sv_sig = {531, "glMultiTexCoord4sv", 2, _glMultiTexCoord4sv_args};

static const char * _glLoadTransposeMatrixf_args[1] = {"m"};
static const trace::FunctionSig _glLoadTransposeMatrixf_sig = {532, "glLoadTransposeMatrixf", 1, _glLoadTransposeMatrixf_args};

static const char * _glLoadTransposeMatrixd_args[1] = {"m"};
static const trace::FunctionSig _glLoadTransposeMatrixd_sig = {533, "glLoadTransposeMatrixd", 1, _glLoadTransposeMatrixd_args};

static const char * _glMultTransposeMatrixf_args[1] = {"m"};
static const trace::FunctionSig _glMultTransposeMatrixf_sig = {534, "glMultTransposeMatrixf", 1, _glMultTransposeMatrixf_args};

static const char * _glMultTransposeMatrixd_args[1] = {"m"};
static const trace::FunctionSig _glMultTransposeMatrixd_sig = {535, "glMultTransposeMatrixd", 1, _glMultTransposeMatrixd_args};

static const char * _glBlendFuncSeparate_args[4] = {"sfactorRGB", "dfactorRGB", "sfactorAlpha", "dfactorAlpha"};
static const trace::FunctionSig _glBlendFuncSeparate_sig = {536, "glBlendFuncSeparate", 4, _glBlendFuncSeparate_args};

static const char * _glMultiDrawArrays_args[4] = {"mode", "first", "count", "drawcount"};
static const trace::FunctionSig _glMultiDrawArrays_sig = {537, "glMultiDrawArrays", 4, _glMultiDrawArrays_args};

static const char * _glMultiDrawElements_args[5] = {"mode", "count", "type", "indices", "drawcount"};
static const trace::FunctionSig _glMultiDrawElements_sig = {538, "glMultiDrawElements", 5, _glMultiDrawElements_args};

static const char * _glPointParameterf_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPointParameterf_sig = {539, "glPointParameterf", 2, _glPointParameterf_args};

static const char * _glPointParameterfv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glPointParameterfv_sig = {540, "glPointParameterfv", 2, _glPointParameterfv_args};

static const char * _glPointParameteri_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPointParameteri_sig = {541, "glPointParameteri", 2, _glPointParameteri_args};

static const char * _glPointParameteriv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glPointParameteriv_sig = {542, "glPointParameteriv", 2, _glPointParameteriv_args};

static const char * _glFogCoordf_args[1] = {"coord"};
static const trace::FunctionSig _glFogCoordf_sig = {543, "glFogCoordf", 1, _glFogCoordf_args};

static const char * _glFogCoordfv_args[1] = {"coord"};
static const trace::FunctionSig _glFogCoordfv_sig = {544, "glFogCoordfv", 1, _glFogCoordfv_args};

static const char * _glFogCoordd_args[1] = {"coord"};
static const trace::FunctionSig _glFogCoordd_sig = {545, "glFogCoordd", 1, _glFogCoordd_args};

static const char * _glFogCoorddv_args[1] = {"coord"};
static const trace::FunctionSig _glFogCoorddv_sig = {546, "glFogCoorddv", 1, _glFogCoorddv_args};

static const char * _glFogCoordPointer_args[3] = {"type", "stride", "pointer"};
static const trace::FunctionSig _glFogCoordPointer_sig = {547, "glFogCoordPointer", 3, _glFogCoordPointer_args};

static const char * _glSecondaryColor3b_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3b_sig = {548, "glSecondaryColor3b", 3, _glSecondaryColor3b_args};

static const char * _glSecondaryColor3bv_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3bv_sig = {549, "glSecondaryColor3bv", 1, _glSecondaryColor3bv_args};

static const char * _glSecondaryColor3d_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3d_sig = {550, "glSecondaryColor3d", 3, _glSecondaryColor3d_args};

static const char * _glSecondaryColor3dv_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3dv_sig = {551, "glSecondaryColor3dv", 1, _glSecondaryColor3dv_args};

static const char * _glSecondaryColor3f_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3f_sig = {552, "glSecondaryColor3f", 3, _glSecondaryColor3f_args};

static const char * _glSecondaryColor3fv_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3fv_sig = {553, "glSecondaryColor3fv", 1, _glSecondaryColor3fv_args};

static const char * _glSecondaryColor3i_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3i_sig = {554, "glSecondaryColor3i", 3, _glSecondaryColor3i_args};

static const char * _glSecondaryColor3iv_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3iv_sig = {555, "glSecondaryColor3iv", 1, _glSecondaryColor3iv_args};

static const char * _glSecondaryColor3s_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3s_sig = {556, "glSecondaryColor3s", 3, _glSecondaryColor3s_args};

static const char * _glSecondaryColor3sv_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3sv_sig = {557, "glSecondaryColor3sv", 1, _glSecondaryColor3sv_args};

static const char * _glSecondaryColor3ub_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3ub_sig = {558, "glSecondaryColor3ub", 3, _glSecondaryColor3ub_args};

static const char * _glSecondaryColor3ubv_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3ubv_sig = {559, "glSecondaryColor3ubv", 1, _glSecondaryColor3ubv_args};

static const char * _glSecondaryColor3ui_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3ui_sig = {560, "glSecondaryColor3ui", 3, _glSecondaryColor3ui_args};

static const char * _glSecondaryColor3uiv_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3uiv_sig = {561, "glSecondaryColor3uiv", 1, _glSecondaryColor3uiv_args};

static const char * _glSecondaryColor3us_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3us_sig = {562, "glSecondaryColor3us", 3, _glSecondaryColor3us_args};

static const char * _glSecondaryColor3usv_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3usv_sig = {563, "glSecondaryColor3usv", 1, _glSecondaryColor3usv_args};

static const char * _glSecondaryColorPointer_args[4] = {"size", "type", "stride", "pointer"};
static const trace::FunctionSig _glSecondaryColorPointer_sig = {564, "glSecondaryColorPointer", 4, _glSecondaryColorPointer_args};

static const char * _glWindowPos2d_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2d_sig = {565, "glWindowPos2d", 2, _glWindowPos2d_args};

static const char * _glWindowPos2dv_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2dv_sig = {566, "glWindowPos2dv", 1, _glWindowPos2dv_args};

static const char * _glWindowPos2f_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2f_sig = {567, "glWindowPos2f", 2, _glWindowPos2f_args};

static const char * _glWindowPos2fv_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2fv_sig = {568, "glWindowPos2fv", 1, _glWindowPos2fv_args};

static const char * _glWindowPos2i_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2i_sig = {569, "glWindowPos2i", 2, _glWindowPos2i_args};

static const char * _glWindowPos2iv_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2iv_sig = {570, "glWindowPos2iv", 1, _glWindowPos2iv_args};

static const char * _glWindowPos2s_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2s_sig = {571, "glWindowPos2s", 2, _glWindowPos2s_args};

static const char * _glWindowPos2sv_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2sv_sig = {572, "glWindowPos2sv", 1, _glWindowPos2sv_args};

static const char * _glWindowPos3d_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3d_sig = {573, "glWindowPos3d", 3, _glWindowPos3d_args};

static const char * _glWindowPos3dv_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3dv_sig = {574, "glWindowPos3dv", 1, _glWindowPos3dv_args};

static const char * _glWindowPos3f_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3f_sig = {575, "glWindowPos3f", 3, _glWindowPos3f_args};

static const char * _glWindowPos3fv_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3fv_sig = {576, "glWindowPos3fv", 1, _glWindowPos3fv_args};

static const char * _glWindowPos3i_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3i_sig = {577, "glWindowPos3i", 3, _glWindowPos3i_args};

static const char * _glWindowPos3iv_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3iv_sig = {578, "glWindowPos3iv", 1, _glWindowPos3iv_args};

static const char * _glWindowPos3s_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3s_sig = {579, "glWindowPos3s", 3, _glWindowPos3s_args};

static const char * _glWindowPos3sv_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3sv_sig = {580, "glWindowPos3sv", 1, _glWindowPos3sv_args};

static const char * _glGenQueries_args[2] = {"n", "ids"};
static const trace::FunctionSig _glGenQueries_sig = {581, "glGenQueries", 2, _glGenQueries_args};

static const char * _glDeleteQueries_args[2] = {"n", "ids"};
static const trace::FunctionSig _glDeleteQueries_sig = {582, "glDeleteQueries", 2, _glDeleteQueries_args};

static const char * _glIsQuery_args[1] = {"id"};
static const trace::FunctionSig _glIsQuery_sig = {583, "glIsQuery", 1, _glIsQuery_args};

static const char * _glBeginQuery_args[2] = {"target", "id"};
static const trace::FunctionSig _glBeginQuery_sig = {584, "glBeginQuery", 2, _glBeginQuery_args};

static const char * _glEndQuery_args[1] = {"target"};
static const trace::FunctionSig _glEndQuery_sig = {585, "glEndQuery", 1, _glEndQuery_args};

static const char * _glGetQueryiv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetQueryiv_sig = {586, "glGetQueryiv", 3, _glGetQueryiv_args};

static const char * _glGetQueryObjectiv_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectiv_sig = {587, "glGetQueryObjectiv", 3, _glGetQueryObjectiv_args};

static const char * _glGetQueryObjectuiv_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectuiv_sig = {588, "glGetQueryObjectuiv", 3, _glGetQueryObjectuiv_args};

static const char * _glBindBuffer_args[2] = {"target", "buffer"};
static const trace::FunctionSig _glBindBuffer_sig = {589, "glBindBuffer", 2, _glBindBuffer_args};

static const char * _glDeleteBuffers_args[2] = {"n", "buffers"};
static const trace::FunctionSig _glDeleteBuffers_sig = {590, "glDeleteBuffers", 2, _glDeleteBuffers_args};

static const char * _glGenBuffers_args[2] = {"n", "buffers"};
static const trace::FunctionSig _glGenBuffers_sig = {591, "glGenBuffers", 2, _glGenBuffers_args};

static const char * _glIsBuffer_args[1] = {"buffer"};
static const trace::FunctionSig _glIsBuffer_sig = {592, "glIsBuffer", 1, _glIsBuffer_args};

static const char * _glBufferData_args[4] = {"target", "size", "data", "usage"};
static const trace::FunctionSig _glBufferData_sig = {593, "glBufferData", 4, _glBufferData_args};

static const char * _glBufferSubData_args[4] = {"target", "offset", "size", "data"};
static const trace::FunctionSig _glBufferSubData_sig = {594, "glBufferSubData", 4, _glBufferSubData_args};

static const char * _glGetBufferSubData_args[4] = {"target", "offset", "size", "data"};
static const trace::FunctionSig _glGetBufferSubData_sig = {595, "glGetBufferSubData", 4, _glGetBufferSubData_args};

static const char * _glMapBuffer_args[2] = {"target", "access"};
static const trace::FunctionSig _glMapBuffer_sig = {596, "glMapBuffer", 2, _glMapBuffer_args};

static const char * _glUnmapBuffer_args[1] = {"target"};
static const trace::FunctionSig _glUnmapBuffer_sig = {597, "glUnmapBuffer", 1, _glUnmapBuffer_args};

static const char * _glGetBufferParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetBufferParameteriv_sig = {598, "glGetBufferParameteriv", 3, _glGetBufferParameteriv_args};

static const char * _glGetBufferPointerv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetBufferPointerv_sig = {599, "glGetBufferPointerv", 3, _glGetBufferPointerv_args};

static const char * _glBlendEquationSeparate_args[2] = {"modeRGB", "modeAlpha"};
static const trace::FunctionSig _glBlendEquationSeparate_sig = {600, "glBlendEquationSeparate", 2, _glBlendEquationSeparate_args};

static const char * _glDrawBuffers_args[2] = {"n", "bufs"};
static const trace::FunctionSig _glDrawBuffers_sig = {601, "glDrawBuffers", 2, _glDrawBuffers_args};

static const char * _glStencilOpSeparate_args[4] = {"face", "sfail", "dpfail", "dppass"};
static const trace::FunctionSig _glStencilOpSeparate_sig = {602, "glStencilOpSeparate", 4, _glStencilOpSeparate_args};

static const char * _glStencilFuncSeparate_args[4] = {"face", "func", "ref", "mask"};
static const trace::FunctionSig _glStencilFuncSeparate_sig = {603, "glStencilFuncSeparate", 4, _glStencilFuncSeparate_args};

static const char * _glStencilMaskSeparate_args[2] = {"face", "mask"};
static const trace::FunctionSig _glStencilMaskSeparate_sig = {604, "glStencilMaskSeparate", 2, _glStencilMaskSeparate_args};

static const char * _glAttachShader_args[2] = {"program", "shader"};
static const trace::FunctionSig _glAttachShader_sig = {605, "glAttachShader", 2, _glAttachShader_args};

static const char * _glBindAttribLocation_args[3] = {"program", "index", "name"};
static const trace::FunctionSig _glBindAttribLocation_sig = {606, "glBindAttribLocation", 3, _glBindAttribLocation_args};

static const char * _glCompileShader_args[1] = {"shader"};
static const trace::FunctionSig _glCompileShader_sig = {607, "glCompileShader", 1, _glCompileShader_args};

static const char ** _glCreateProgram_args = NULL;
static const trace::FunctionSig _glCreateProgram_sig = {608, "glCreateProgram", 0, _glCreateProgram_args};

static const char * _glCreateShader_args[1] = {"type"};
static const trace::FunctionSig _glCreateShader_sig = {609, "glCreateShader", 1, _glCreateShader_args};

static const char * _glDeleteProgram_args[1] = {"program"};
static const trace::FunctionSig _glDeleteProgram_sig = {610, "glDeleteProgram", 1, _glDeleteProgram_args};

static const char * _glDeleteShader_args[1] = {"shader"};
static const trace::FunctionSig _glDeleteShader_sig = {611, "glDeleteShader", 1, _glDeleteShader_args};

static const char * _glDetachShader_args[2] = {"program", "shader"};
static const trace::FunctionSig _glDetachShader_sig = {612, "glDetachShader", 2, _glDetachShader_args};

static const char * _glDisableVertexAttribArray_args[1] = {"index"};
static const trace::FunctionSig _glDisableVertexAttribArray_sig = {613, "glDisableVertexAttribArray", 1, _glDisableVertexAttribArray_args};

static const char * _glEnableVertexAttribArray_args[1] = {"index"};
static const trace::FunctionSig _glEnableVertexAttribArray_sig = {614, "glEnableVertexAttribArray", 1, _glEnableVertexAttribArray_args};

static const char * _glGetActiveAttrib_args[7] = {"program", "index", "bufSize", "length", "size", "type", "name"};
static const trace::FunctionSig _glGetActiveAttrib_sig = {615, "glGetActiveAttrib", 7, _glGetActiveAttrib_args};

static const char * _glGetActiveUniform_args[7] = {"program", "index", "bufSize", "length", "size", "type", "name"};
static const trace::FunctionSig _glGetActiveUniform_sig = {616, "glGetActiveUniform", 7, _glGetActiveUniform_args};

static const char * _glGetAttachedShaders_args[4] = {"program", "maxCount", "count", "obj"};
static const trace::FunctionSig _glGetAttachedShaders_sig = {617, "glGetAttachedShaders", 4, _glGetAttachedShaders_args};

static const char * _glGetAttribLocation_args[2] = {"program", "name"};
static const trace::FunctionSig _glGetAttribLocation_sig = {618, "glGetAttribLocation", 2, _glGetAttribLocation_args};

static const char * _glGetProgramiv_args[3] = {"program", "pname", "params"};
static const trace::FunctionSig _glGetProgramiv_sig = {619, "glGetProgramiv", 3, _glGetProgramiv_args};

static const char * _glGetProgramInfoLog_args[4] = {"program", "bufSize", "length", "infoLog"};
static const trace::FunctionSig _glGetProgramInfoLog_sig = {620, "glGetProgramInfoLog", 4, _glGetProgramInfoLog_args};

static const char * _glGetShaderiv_args[3] = {"shader", "pname", "params"};
static const trace::FunctionSig _glGetShaderiv_sig = {621, "glGetShaderiv", 3, _glGetShaderiv_args};

static const char * _glGetShaderInfoLog_args[4] = {"shader", "bufSize", "length", "infoLog"};
static const trace::FunctionSig _glGetShaderInfoLog_sig = {622, "glGetShaderInfoLog", 4, _glGetShaderInfoLog_args};

static const char * _glGetShaderSource_args[4] = {"shader", "bufSize", "length", "source"};
static const trace::FunctionSig _glGetShaderSource_sig = {623, "glGetShaderSource", 4, _glGetShaderSource_args};

static const char * _glGetUniformLocation_args[2] = {"program", "name"};
static const trace::FunctionSig _glGetUniformLocation_sig = {624, "glGetUniformLocation", 2, _glGetUniformLocation_args};

static const char * _glGetUniformfv_args[3] = {"program", "location", "params"};
static const trace::FunctionSig _glGetUniformfv_sig = {625, "glGetUniformfv", 3, _glGetUniformfv_args};

static const char * _glGetUniformiv_args[3] = {"program", "location", "params"};
static const trace::FunctionSig _glGetUniformiv_sig = {626, "glGetUniformiv", 3, _glGetUniformiv_args};

static const char * _glGetVertexAttribdv_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribdv_sig = {627, "glGetVertexAttribdv", 3, _glGetVertexAttribdv_args};

static const char * _glGetVertexAttribfv_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribfv_sig = {628, "glGetVertexAttribfv", 3, _glGetVertexAttribfv_args};

static const char * _glGetVertexAttribiv_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribiv_sig = {629, "glGetVertexAttribiv", 3, _glGetVertexAttribiv_args};

static const char * _glGetVertexAttribPointerv_args[3] = {"index", "pname", "pointer"};
static const trace::FunctionSig _glGetVertexAttribPointerv_sig = {630, "glGetVertexAttribPointerv", 3, _glGetVertexAttribPointerv_args};

static const char * _glIsProgram_args[1] = {"program"};
static const trace::FunctionSig _glIsProgram_sig = {631, "glIsProgram", 1, _glIsProgram_args};

static const char * _glIsShader_args[1] = {"shader"};
static const trace::FunctionSig _glIsShader_sig = {632, "glIsShader", 1, _glIsShader_args};

static const char * _glLinkProgram_args[1] = {"program"};
static const trace::FunctionSig _glLinkProgram_sig = {633, "glLinkProgram", 1, _glLinkProgram_args};

static const char * _glShaderSource_args[4] = {"shader", "count", "string", "length"};
static const trace::FunctionSig _glShaderSource_sig = {634, "glShaderSource", 4, _glShaderSource_args};

static const char * _glUseProgram_args[1] = {"program"};
static const trace::FunctionSig _glUseProgram_sig = {635, "glUseProgram", 1, _glUseProgram_args};

static const char * _glUniform1f_args[2] = {"location", "v0"};
static const trace::FunctionSig _glUniform1f_sig = {636, "glUniform1f", 2, _glUniform1f_args};

static const char * _glUniform2f_args[3] = {"location", "v0", "v1"};
static const trace::FunctionSig _glUniform2f_sig = {637, "glUniform2f", 3, _glUniform2f_args};

static const char * _glUniform3f_args[4] = {"location", "v0", "v1", "v2"};
static const trace::FunctionSig _glUniform3f_sig = {638, "glUniform3f", 4, _glUniform3f_args};

static const char * _glUniform4f_args[5] = {"location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glUniform4f_sig = {639, "glUniform4f", 5, _glUniform4f_args};

static const char * _glUniform1i_args[2] = {"location", "v0"};
static const trace::FunctionSig _glUniform1i_sig = {640, "glUniform1i", 2, _glUniform1i_args};

static const char * _glUniform2i_args[3] = {"location", "v0", "v1"};
static const trace::FunctionSig _glUniform2i_sig = {641, "glUniform2i", 3, _glUniform2i_args};

static const char * _glUniform3i_args[4] = {"location", "v0", "v1", "v2"};
static const trace::FunctionSig _glUniform3i_sig = {642, "glUniform3i", 4, _glUniform3i_args};

static const char * _glUniform4i_args[5] = {"location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glUniform4i_sig = {643, "glUniform4i", 5, _glUniform4i_args};

static const char * _glUniform1fv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1fv_sig = {644, "glUniform1fv", 3, _glUniform1fv_args};

static const char * _glUniform2fv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2fv_sig = {645, "glUniform2fv", 3, _glUniform2fv_args};

static const char * _glUniform3fv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3fv_sig = {646, "glUniform3fv", 3, _glUniform3fv_args};

static const char * _glUniform4fv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4fv_sig = {647, "glUniform4fv", 3, _glUniform4fv_args};

static const char * _glUniform1iv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1iv_sig = {648, "glUniform1iv", 3, _glUniform1iv_args};

static const char * _glUniform2iv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2iv_sig = {649, "glUniform2iv", 3, _glUniform2iv_args};

static const char * _glUniform3iv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3iv_sig = {650, "glUniform3iv", 3, _glUniform3iv_args};

static const char * _glUniform4iv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4iv_sig = {651, "glUniform4iv", 3, _glUniform4iv_args};

static const char * _glUniformMatrix2fv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix2fv_sig = {652, "glUniformMatrix2fv", 4, _glUniformMatrix2fv_args};

static const char * _glUniformMatrix3fv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix3fv_sig = {653, "glUniformMatrix3fv", 4, _glUniformMatrix3fv_args};

static const char * _glUniformMatrix4fv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix4fv_sig = {654, "glUniformMatrix4fv", 4, _glUniformMatrix4fv_args};

static const char * _glValidateProgram_args[1] = {"program"};
static const trace::FunctionSig _glValidateProgram_sig = {655, "glValidateProgram", 1, _glValidateProgram_args};

static const char * _glVertexAttrib1d_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttrib1d_sig = {656, "glVertexAttrib1d", 2, _glVertexAttrib1d_args};

static const char * _glVertexAttrib1dv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib1dv_sig = {657, "glVertexAttrib1dv", 2, _glVertexAttrib1dv_args};

static const char * _glVertexAttrib1f_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttrib1f_sig = {658, "glVertexAttrib1f", 2, _glVertexAttrib1f_args};

static const char * _glVertexAttrib1fv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib1fv_sig = {659, "glVertexAttrib1fv", 2, _glVertexAttrib1fv_args};

static const char * _glVertexAttrib1s_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttrib1s_sig = {660, "glVertexAttrib1s", 2, _glVertexAttrib1s_args};

static const char * _glVertexAttrib1sv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib1sv_sig = {661, "glVertexAttrib1sv", 2, _glVertexAttrib1sv_args};

static const char * _glVertexAttrib2d_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttrib2d_sig = {662, "glVertexAttrib2d", 3, _glVertexAttrib2d_args};

static const char * _glVertexAttrib2dv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib2dv_sig = {663, "glVertexAttrib2dv", 2, _glVertexAttrib2dv_args};

static const char * _glVertexAttrib2f_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttrib2f_sig = {664, "glVertexAttrib2f", 3, _glVertexAttrib2f_args};

static const char * _glVertexAttrib2fv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib2fv_sig = {665, "glVertexAttrib2fv", 2, _glVertexAttrib2fv_args};

static const char * _glVertexAttrib2s_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttrib2s_sig = {666, "glVertexAttrib2s", 3, _glVertexAttrib2s_args};

static const char * _glVertexAttrib2sv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib2sv_sig = {667, "glVertexAttrib2sv", 2, _glVertexAttrib2sv_args};

static const char * _glVertexAttrib3d_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttrib3d_sig = {668, "glVertexAttrib3d", 4, _glVertexAttrib3d_args};

static const char * _glVertexAttrib3dv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib3dv_sig = {669, "glVertexAttrib3dv", 2, _glVertexAttrib3dv_args};

static const char * _glVertexAttrib3f_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttrib3f_sig = {670, "glVertexAttrib3f", 4, _glVertexAttrib3f_args};

static const char * _glVertexAttrib3fv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib3fv_sig = {671, "glVertexAttrib3fv", 2, _glVertexAttrib3fv_args};

static const char * _glVertexAttrib3s_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttrib3s_sig = {672, "glVertexAttrib3s", 4, _glVertexAttrib3s_args};

static const char * _glVertexAttrib3sv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib3sv_sig = {673, "glVertexAttrib3sv", 2, _glVertexAttrib3sv_args};

static const char * _glVertexAttrib4Nbv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4Nbv_sig = {674, "glVertexAttrib4Nbv", 2, _glVertexAttrib4Nbv_args};

static const char * _glVertexAttrib4Niv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4Niv_sig = {675, "glVertexAttrib4Niv", 2, _glVertexAttrib4Niv_args};

static const char * _glVertexAttrib4Nsv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4Nsv_sig = {676, "glVertexAttrib4Nsv", 2, _glVertexAttrib4Nsv_args};

static const char * _glVertexAttrib4Nub_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4Nub_sig = {677, "glVertexAttrib4Nub", 5, _glVertexAttrib4Nub_args};

static const char * _glVertexAttrib4Nubv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4Nubv_sig = {678, "glVertexAttrib4Nubv", 2, _glVertexAttrib4Nubv_args};

static const char * _glVertexAttrib4Nuiv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4Nuiv_sig = {679, "glVertexAttrib4Nuiv", 2, _glVertexAttrib4Nuiv_args};

static const char * _glVertexAttrib4Nusv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4Nusv_sig = {680, "glVertexAttrib4Nusv", 2, _glVertexAttrib4Nusv_args};

static const char * _glVertexAttrib4bv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4bv_sig = {681, "glVertexAttrib4bv", 2, _glVertexAttrib4bv_args};

static const char * _glVertexAttrib4d_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4d_sig = {682, "glVertexAttrib4d", 5, _glVertexAttrib4d_args};

static const char * _glVertexAttrib4dv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4dv_sig = {683, "glVertexAttrib4dv", 2, _glVertexAttrib4dv_args};

static const char * _glVertexAttrib4f_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4f_sig = {684, "glVertexAttrib4f", 5, _glVertexAttrib4f_args};

static const char * _glVertexAttrib4fv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4fv_sig = {685, "glVertexAttrib4fv", 2, _glVertexAttrib4fv_args};

static const char * _glVertexAttrib4iv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4iv_sig = {686, "glVertexAttrib4iv", 2, _glVertexAttrib4iv_args};

static const char * _glVertexAttrib4s_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4s_sig = {687, "glVertexAttrib4s", 5, _glVertexAttrib4s_args};

static const char * _glVertexAttrib4sv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4sv_sig = {688, "glVertexAttrib4sv", 2, _glVertexAttrib4sv_args};

static const char * _glVertexAttrib4ubv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4ubv_sig = {689, "glVertexAttrib4ubv", 2, _glVertexAttrib4ubv_args};

static const char * _glVertexAttrib4uiv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4uiv_sig = {690, "glVertexAttrib4uiv", 2, _glVertexAttrib4uiv_args};

static const char * _glVertexAttrib4usv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4usv_sig = {691, "glVertexAttrib4usv", 2, _glVertexAttrib4usv_args};

static const char * _glVertexAttribPointer_args[6] = {"index", "size", "type", "normalized", "stride", "pointer"};
static const trace::FunctionSig _glVertexAttribPointer_sig = {692, "glVertexAttribPointer", 6, _glVertexAttribPointer_args};

static const char * _glUniformMatrix2x3fv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix2x3fv_sig = {693, "glUniformMatrix2x3fv", 4, _glUniformMatrix2x3fv_args};

static const char * _glUniformMatrix3x2fv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix3x2fv_sig = {694, "glUniformMatrix3x2fv", 4, _glUniformMatrix3x2fv_args};

static const char * _glUniformMatrix2x4fv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix2x4fv_sig = {695, "glUniformMatrix2x4fv", 4, _glUniformMatrix2x4fv_args};

static const char * _glUniformMatrix4x2fv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix4x2fv_sig = {696, "glUniformMatrix4x2fv", 4, _glUniformMatrix4x2fv_args};

static const char * _glUniformMatrix3x4fv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix3x4fv_sig = {697, "glUniformMatrix3x4fv", 4, _glUniformMatrix3x4fv_args};

static const char * _glUniformMatrix4x3fv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix4x3fv_sig = {698, "glUniformMatrix4x3fv", 4, _glUniformMatrix4x3fv_args};

static const char * _glColorMaski_args[5] = {"index", "r", "g", "b", "a"};
static const trace::FunctionSig _glColorMaski_sig = {699, "glColorMaski", 5, _glColorMaski_args};

static const char * _glGetBooleani_v_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetBooleani_v_sig = {700, "glGetBooleani_v", 3, _glGetBooleani_v_args};

static const char * _glGetIntegeri_v_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetIntegeri_v_sig = {701, "glGetIntegeri_v", 3, _glGetIntegeri_v_args};

static const char * _glEnablei_args[2] = {"target", "index"};
static const trace::FunctionSig _glEnablei_sig = {702, "glEnablei", 2, _glEnablei_args};

static const char * _glDisablei_args[2] = {"target", "index"};
static const trace::FunctionSig _glDisablei_sig = {703, "glDisablei", 2, _glDisablei_args};

static const char * _glIsEnabledi_args[2] = {"target", "index"};
static const trace::FunctionSig _glIsEnabledi_sig = {704, "glIsEnabledi", 2, _glIsEnabledi_args};

static const char * _glBeginTransformFeedback_args[1] = {"primitiveMode"};
static const trace::FunctionSig _glBeginTransformFeedback_sig = {705, "glBeginTransformFeedback", 1, _glBeginTransformFeedback_args};

static const char ** _glEndTransformFeedback_args = NULL;
static const trace::FunctionSig _glEndTransformFeedback_sig = {706, "glEndTransformFeedback", 0, _glEndTransformFeedback_args};

static const char * _glBindBufferRange_args[5] = {"target", "index", "buffer", "offset", "size"};
static const trace::FunctionSig _glBindBufferRange_sig = {707, "glBindBufferRange", 5, _glBindBufferRange_args};

static const char * _glBindBufferBase_args[3] = {"target", "index", "buffer"};
static const trace::FunctionSig _glBindBufferBase_sig = {708, "glBindBufferBase", 3, _glBindBufferBase_args};

static const char * _glTransformFeedbackVaryings_args[4] = {"program", "count", "varyings", "bufferMode"};
static const trace::FunctionSig _glTransformFeedbackVaryings_sig = {709, "glTransformFeedbackVaryings", 4, _glTransformFeedbackVaryings_args};

static const char * _glGetTransformFeedbackVarying_args[7] = {"program", "index", "bufSize", "length", "size", "type", "name"};
static const trace::FunctionSig _glGetTransformFeedbackVarying_sig = {710, "glGetTransformFeedbackVarying", 7, _glGetTransformFeedbackVarying_args};

static const char * _glClampColor_args[2] = {"target", "clamp"};
static const trace::FunctionSig _glClampColor_sig = {711, "glClampColor", 2, _glClampColor_args};

static const char * _glBeginConditionalRender_args[2] = {"id", "mode"};
static const trace::FunctionSig _glBeginConditionalRender_sig = {712, "glBeginConditionalRender", 2, _glBeginConditionalRender_args};

static const char ** _glEndConditionalRender_args = NULL;
static const trace::FunctionSig _glEndConditionalRender_sig = {713, "glEndConditionalRender", 0, _glEndConditionalRender_args};

static const char * _glVertexAttribIPointer_args[5] = {"index", "size", "type", "stride", "pointer"};
static const trace::FunctionSig _glVertexAttribIPointer_sig = {714, "glVertexAttribIPointer", 5, _glVertexAttribIPointer_args};

static const char * _glGetVertexAttribIiv_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribIiv_sig = {715, "glGetVertexAttribIiv", 3, _glGetVertexAttribIiv_args};

static const char * _glGetVertexAttribIuiv_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribIuiv_sig = {716, "glGetVertexAttribIuiv", 3, _glGetVertexAttribIuiv_args};

static const char * _glVertexAttribI1i_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttribI1i_sig = {717, "glVertexAttribI1i", 2, _glVertexAttribI1i_args};

static const char * _glVertexAttribI2i_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttribI2i_sig = {718, "glVertexAttribI2i", 3, _glVertexAttribI2i_args};

static const char * _glVertexAttribI3i_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttribI3i_sig = {719, "glVertexAttribI3i", 4, _glVertexAttribI3i_args};

static const char * _glVertexAttribI4i_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttribI4i_sig = {720, "glVertexAttribI4i", 5, _glVertexAttribI4i_args};

static const char * _glVertexAttribI1ui_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttribI1ui_sig = {721, "glVertexAttribI1ui", 2, _glVertexAttribI1ui_args};

static const char * _glVertexAttribI2ui_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttribI2ui_sig = {722, "glVertexAttribI2ui", 3, _glVertexAttribI2ui_args};

static const char * _glVertexAttribI3ui_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttribI3ui_sig = {723, "glVertexAttribI3ui", 4, _glVertexAttribI3ui_args};

static const char * _glVertexAttribI4ui_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttribI4ui_sig = {724, "glVertexAttribI4ui", 5, _glVertexAttribI4ui_args};

static const char * _glVertexAttribI1iv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI1iv_sig = {725, "glVertexAttribI1iv", 2, _glVertexAttribI1iv_args};

static const char * _glVertexAttribI2iv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI2iv_sig = {726, "glVertexAttribI2iv", 2, _glVertexAttribI2iv_args};

static const char * _glVertexAttribI3iv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI3iv_sig = {727, "glVertexAttribI3iv", 2, _glVertexAttribI3iv_args};

static const char * _glVertexAttribI4iv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4iv_sig = {728, "glVertexAttribI4iv", 2, _glVertexAttribI4iv_args};

static const char * _glVertexAttribI1uiv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI1uiv_sig = {729, "glVertexAttribI1uiv", 2, _glVertexAttribI1uiv_args};

static const char * _glVertexAttribI2uiv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI2uiv_sig = {730, "glVertexAttribI2uiv", 2, _glVertexAttribI2uiv_args};

static const char * _glVertexAttribI3uiv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI3uiv_sig = {731, "glVertexAttribI3uiv", 2, _glVertexAttribI3uiv_args};

static const char * _glVertexAttribI4uiv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4uiv_sig = {732, "glVertexAttribI4uiv", 2, _glVertexAttribI4uiv_args};

static const char * _glVertexAttribI4bv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4bv_sig = {733, "glVertexAttribI4bv", 2, _glVertexAttribI4bv_args};

static const char * _glVertexAttribI4sv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4sv_sig = {734, "glVertexAttribI4sv", 2, _glVertexAttribI4sv_args};

static const char * _glVertexAttribI4ubv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4ubv_sig = {735, "glVertexAttribI4ubv", 2, _glVertexAttribI4ubv_args};

static const char * _glVertexAttribI4usv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4usv_sig = {736, "glVertexAttribI4usv", 2, _glVertexAttribI4usv_args};

static const char * _glGetUniformuiv_args[3] = {"program", "location", "params"};
static const trace::FunctionSig _glGetUniformuiv_sig = {737, "glGetUniformuiv", 3, _glGetUniformuiv_args};

static const char * _glBindFragDataLocation_args[3] = {"program", "color", "name"};
static const trace::FunctionSig _glBindFragDataLocation_sig = {738, "glBindFragDataLocation", 3, _glBindFragDataLocation_args};

static const char * _glGetFragDataLocation_args[2] = {"program", "name"};
static const trace::FunctionSig _glGetFragDataLocation_sig = {739, "glGetFragDataLocation", 2, _glGetFragDataLocation_args};

static const char * _glUniform1ui_args[2] = {"location", "v0"};
static const trace::FunctionSig _glUniform1ui_sig = {740, "glUniform1ui", 2, _glUniform1ui_args};

static const char * _glUniform2ui_args[3] = {"location", "v0", "v1"};
static const trace::FunctionSig _glUniform2ui_sig = {741, "glUniform2ui", 3, _glUniform2ui_args};

static const char * _glUniform3ui_args[4] = {"location", "v0", "v1", "v2"};
static const trace::FunctionSig _glUniform3ui_sig = {742, "glUniform3ui", 4, _glUniform3ui_args};

static const char * _glUniform4ui_args[5] = {"location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glUniform4ui_sig = {743, "glUniform4ui", 5, _glUniform4ui_args};

static const char * _glUniform1uiv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1uiv_sig = {744, "glUniform1uiv", 3, _glUniform1uiv_args};

static const char * _glUniform2uiv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2uiv_sig = {745, "glUniform2uiv", 3, _glUniform2uiv_args};

static const char * _glUniform3uiv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3uiv_sig = {746, "glUniform3uiv", 3, _glUniform3uiv_args};

static const char * _glUniform4uiv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4uiv_sig = {747, "glUniform4uiv", 3, _glUniform4uiv_args};

static const char * _glTexParameterIiv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glTexParameterIiv_sig = {748, "glTexParameterIiv", 3, _glTexParameterIiv_args};

static const char * _glTexParameterIuiv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glTexParameterIuiv_sig = {749, "glTexParameterIuiv", 3, _glTexParameterIuiv_args};

static const char * _glGetTexParameterIiv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexParameterIiv_sig = {750, "glGetTexParameterIiv", 3, _glGetTexParameterIiv_args};

static const char * _glGetTexParameterIuiv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexParameterIuiv_sig = {751, "glGetTexParameterIuiv", 3, _glGetTexParameterIuiv_args};

static const char * _glClearBufferiv_args[3] = {"buffer", "drawbuffer", "value"};
static const trace::FunctionSig _glClearBufferiv_sig = {752, "glClearBufferiv", 3, _glClearBufferiv_args};

static const char * _glClearBufferuiv_args[3] = {"buffer", "drawbuffer", "value"};
static const trace::FunctionSig _glClearBufferuiv_sig = {753, "glClearBufferuiv", 3, _glClearBufferuiv_args};

static const char * _glClearBufferfv_args[3] = {"buffer", "drawbuffer", "value"};
static const trace::FunctionSig _glClearBufferfv_sig = {754, "glClearBufferfv", 3, _glClearBufferfv_args};

static const char * _glClearBufferfi_args[4] = {"buffer", "drawbuffer", "depth", "stencil"};
static const trace::FunctionSig _glClearBufferfi_sig = {755, "glClearBufferfi", 4, _glClearBufferfi_args};

static const char * _glGetStringi_args[2] = {"name", "index"};
static const trace::FunctionSig _glGetStringi_sig = {756, "glGetStringi", 2, _glGetStringi_args};

static const char * _glDrawArraysInstanced_args[4] = {"mode", "first", "count", "instancecount"};
static const trace::FunctionSig _glDrawArraysInstanced_sig = {757, "glDrawArraysInstanced", 4, _glDrawArraysInstanced_args};

static const char * _glDrawElementsInstanced_args[5] = {"mode", "count", "type", "indices", "instancecount"};
static const trace::FunctionSig _glDrawElementsInstanced_sig = {758, "glDrawElementsInstanced", 5, _glDrawElementsInstanced_args};

static const char * _glTexBuffer_args[3] = {"target", "internalformat", "buffer"};
static const trace::FunctionSig _glTexBuffer_sig = {759, "glTexBuffer", 3, _glTexBuffer_args};

static const char * _glPrimitiveRestartIndex_args[1] = {"index"};
static const trace::FunctionSig _glPrimitiveRestartIndex_sig = {760, "glPrimitiveRestartIndex", 1, _glPrimitiveRestartIndex_args};

static const char * _glGetInteger64i_v_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetInteger64i_v_sig = {761, "glGetInteger64i_v", 3, _glGetInteger64i_v_args};

static const char * _glGetBufferParameteri64v_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetBufferParameteri64v_sig = {762, "glGetBufferParameteri64v", 3, _glGetBufferParameteri64v_args};

static const char * _glFramebufferTexture_args[4] = {"target", "attachment", "texture", "level"};
static const trace::FunctionSig _glFramebufferTexture_sig = {763, "glFramebufferTexture", 4, _glFramebufferTexture_args};

static const char * _glVertexAttribDivisor_args[2] = {"index", "divisor"};
static const trace::FunctionSig _glVertexAttribDivisor_sig = {764, "glVertexAttribDivisor", 2, _glVertexAttribDivisor_args};

static const char * _glMinSampleShading_args[1] = {"value"};
static const trace::FunctionSig _glMinSampleShading_sig = {765, "glMinSampleShading", 1, _glMinSampleShading_args};

static const char * _glBlendEquationi_args[2] = {"buf", "mode"};
static const trace::FunctionSig _glBlendEquationi_sig = {766, "glBlendEquationi", 2, _glBlendEquationi_args};

static const char * _glBlendEquationSeparatei_args[3] = {"buf", "modeRGB", "modeAlpha"};
static const trace::FunctionSig _glBlendEquationSeparatei_sig = {767, "glBlendEquationSeparatei", 3, _glBlendEquationSeparatei_args};

static const char * _glBlendFunci_args[3] = {"buf", "src", "dst"};
static const trace::FunctionSig _glBlendFunci_sig = {768, "glBlendFunci", 3, _glBlendFunci_args};

static const char * _glBlendFuncSeparatei_args[5] = {"buf", "srcRGB", "dstRGB", "srcAlpha", "dstAlpha"};
static const trace::FunctionSig _glBlendFuncSeparatei_sig = {769, "glBlendFuncSeparatei", 5, _glBlendFuncSeparatei_args};

static const char * _glGetnCompressedTexImage_args[4] = {"target", "lod", "bufSize", "pixels"};
static const trace::FunctionSig _glGetnCompressedTexImage_sig = {770, "glGetnCompressedTexImage", 4, _glGetnCompressedTexImage_args};

static const char * _glGetnTexImage_args[6] = {"target", "level", "format", "type", "bufSize", "pixels"};
static const trace::FunctionSig _glGetnTexImage_sig = {771, "glGetnTexImage", 6, _glGetnTexImage_args};

static const char * _glGetnUniformdv_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformdv_sig = {772, "glGetnUniformdv", 4, _glGetnUniformdv_args};

static const char * _glGetnMapdv_args[4] = {"target", "query", "bufSize", "v"};
static const trace::FunctionSig _glGetnMapdv_sig = {773, "glGetnMapdv", 4, _glGetnMapdv_args};

static const char * _glGetnMapfv_args[4] = {"target", "query", "bufSize", "v"};
static const trace::FunctionSig _glGetnMapfv_sig = {774, "glGetnMapfv", 4, _glGetnMapfv_args};

static const char * _glGetnMapiv_args[4] = {"target", "query", "bufSize", "v"};
static const trace::FunctionSig _glGetnMapiv_sig = {775, "glGetnMapiv", 4, _glGetnMapiv_args};

static const char * _glGetnPixelMapfv_args[3] = {"map", "bufSize", "values"};
static const trace::FunctionSig _glGetnPixelMapfv_sig = {776, "glGetnPixelMapfv", 3, _glGetnPixelMapfv_args};

static const char * _glGetnPixelMapuiv_args[3] = {"map", "bufSize", "values"};
static const trace::FunctionSig _glGetnPixelMapuiv_sig = {777, "glGetnPixelMapuiv", 3, _glGetnPixelMapuiv_args};

static const char * _glGetnPixelMapusv_args[3] = {"map", "bufSize", "values"};
static const trace::FunctionSig _glGetnPixelMapusv_sig = {778, "glGetnPixelMapusv", 3, _glGetnPixelMapusv_args};

static const char * _glGetnPolygonStipple_args[2] = {"bufSize", "pattern"};
static const trace::FunctionSig _glGetnPolygonStipple_sig = {779, "glGetnPolygonStipple", 2, _glGetnPolygonStipple_args};

static const char * _glGetnColorTable_args[5] = {"target", "format", "type", "bufSize", "table"};
static const trace::FunctionSig _glGetnColorTable_sig = {780, "glGetnColorTable", 5, _glGetnColorTable_args};

static const char * _glGetnConvolutionFilter_args[5] = {"target", "format", "type", "bufSize", "image"};
static const trace::FunctionSig _glGetnConvolutionFilter_sig = {781, "glGetnConvolutionFilter", 5, _glGetnConvolutionFilter_args};

static const char * _glGetnSeparableFilter_args[8] = {"target", "format", "type", "rowBufSize", "row", "columnBufSize", "column", "span"};
static const trace::FunctionSig _glGetnSeparableFilter_sig = {782, "glGetnSeparableFilter", 8, _glGetnSeparableFilter_args};

static const char * _glGetnHistogram_args[6] = {"target", "reset", "format", "type", "bufSize", "values"};
static const trace::FunctionSig _glGetnHistogram_sig = {783, "glGetnHistogram", 6, _glGetnHistogram_args};

static const char * _glGetnMinmax_args[6] = {"target", "reset", "format", "type", "bufSize", "values"};
static const trace::FunctionSig _glGetnMinmax_sig = {784, "glGetnMinmax", 6, _glGetnMinmax_args};

static const char * _glSpecializeShader_args[5] = {"shader", "pEntryPoint", "numSpecializationConstants", "pConstantIndex", "pConstantValue"};
static const trace::FunctionSig _glSpecializeShader_sig = {785, "glSpecializeShader", 5, _glSpecializeShader_args};

static const char * _glMultiDrawArraysIndirectCount_args[5] = {"mode", "indirect", "drawcount", "maxdrawcount", "stride"};
static const trace::FunctionSig _glMultiDrawArraysIndirectCount_sig = {786, "glMultiDrawArraysIndirectCount", 5, _glMultiDrawArraysIndirectCount_args};

static const char * _glMultiDrawElementsIndirectCount_args[6] = {"mode", "type", "indirect", "drawcount", "maxdrawcount", "stride"};
static const trace::FunctionSig _glMultiDrawElementsIndirectCount_sig = {787, "glMultiDrawElementsIndirectCount", 6, _glMultiDrawElementsIndirectCount_args};

static const char * _glClipPlanef_args[2] = {"plane", "equation"};
static const trace::FunctionSig _glClipPlanef_sig = {788, "glClipPlanef", 2, _glClipPlanef_args};

static const char * _glFrustumf_args[6] = {"left", "right", "bottom", "top", "zNear", "zFar"};
static const trace::FunctionSig _glFrustumf_sig = {789, "glFrustumf", 6, _glFrustumf_args};

static const char * _glGetClipPlanef_args[2] = {"plane", "equation"};
static const trace::FunctionSig _glGetClipPlanef_sig = {790, "glGetClipPlanef", 2, _glGetClipPlanef_args};

static const char * _glOrthof_args[6] = {"left", "right", "bottom", "top", "zNear", "zFar"};
static const trace::FunctionSig _glOrthof_sig = {791, "glOrthof", 6, _glOrthof_args};

static const char * _glAlphaFuncx_args[2] = {"func", "ref"};
static const trace::FunctionSig _glAlphaFuncx_sig = {792, "glAlphaFuncx", 2, _glAlphaFuncx_args};

static const char * _glClearColorx_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glClearColorx_sig = {793, "glClearColorx", 4, _glClearColorx_args};

static const char * _glClearDepthx_args[1] = {"depth"};
static const trace::FunctionSig _glClearDepthx_sig = {794, "glClearDepthx", 1, _glClearDepthx_args};

static const char * _glClipPlanex_args[2] = {"plane", "equation"};
static const trace::FunctionSig _glClipPlanex_sig = {795, "glClipPlanex", 2, _glClipPlanex_args};

static const char * _glColor4x_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColor4x_sig = {796, "glColor4x", 4, _glColor4x_args};

static const char * _glDepthRangex_args[2] = {"zNear", "zFar"};
static const trace::FunctionSig _glDepthRangex_sig = {797, "glDepthRangex", 2, _glDepthRangex_args};

static const char * _glFogx_args[2] = {"pname", "param"};
static const trace::FunctionSig _glFogx_sig = {798, "glFogx", 2, _glFogx_args};

static const char * _glFogxv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glFogxv_sig = {799, "glFogxv", 2, _glFogxv_args};

static const char * _glFrustumx_args[6] = {"left", "right", "bottom", "top", "zNear", "zFar"};
static const trace::FunctionSig _glFrustumx_sig = {800, "glFrustumx", 6, _glFrustumx_args};

static const char * _glGetClipPlanex_args[2] = {"plane", "equation"};
static const trace::FunctionSig _glGetClipPlanex_sig = {801, "glGetClipPlanex", 2, _glGetClipPlanex_args};

static const char * _glGetFixedv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetFixedv_sig = {802, "glGetFixedv", 2, _glGetFixedv_args};

static const char * _glGetLightxv_args[3] = {"light", "pname", "params"};
static const trace::FunctionSig _glGetLightxv_sig = {803, "glGetLightxv", 3, _glGetLightxv_args};

static const char * _glGetMaterialxv_args[3] = {"face", "pname", "params"};
static const trace::FunctionSig _glGetMaterialxv_sig = {804, "glGetMaterialxv", 3, _glGetMaterialxv_args};

static const char * _glGetTexEnvxv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexEnvxv_sig = {805, "glGetTexEnvxv", 3, _glGetTexEnvxv_args};

static const char * _glGetTexParameterxv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexParameterxv_sig = {806, "glGetTexParameterxv", 3, _glGetTexParameterxv_args};

static const char * _glLightModelx_args[2] = {"pname", "param"};
static const trace::FunctionSig _glLightModelx_sig = {807, "glLightModelx", 2, _glLightModelx_args};

static const char * _glLightModelxv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glLightModelxv_sig = {808, "glLightModelxv", 2, _glLightModelxv_args};

static const char * _glLightx_args[3] = {"light", "pname", "param"};
static const trace::FunctionSig _glLightx_sig = {809, "glLightx", 3, _glLightx_args};

static const char * _glLightxv_args[3] = {"light", "pname", "params"};
static const trace::FunctionSig _glLightxv_sig = {810, "glLightxv", 3, _glLightxv_args};

static const char * _glLineWidthx_args[1] = {"width"};
static const trace::FunctionSig _glLineWidthx_sig = {811, "glLineWidthx", 1, _glLineWidthx_args};

static const char * _glLoadMatrixx_args[1] = {"m"};
static const trace::FunctionSig _glLoadMatrixx_sig = {812, "glLoadMatrixx", 1, _glLoadMatrixx_args};

static const char * _glMaterialx_args[3] = {"face", "pname", "param"};
static const trace::FunctionSig _glMaterialx_sig = {813, "glMaterialx", 3, _glMaterialx_args};

static const char * _glMaterialxv_args[3] = {"face", "pname", "params"};
static const trace::FunctionSig _glMaterialxv_sig = {814, "glMaterialxv", 3, _glMaterialxv_args};

static const char * _glMultMatrixx_args[1] = {"m"};
static const trace::FunctionSig _glMultMatrixx_sig = {815, "glMultMatrixx", 1, _glMultMatrixx_args};

static const char * _glMultiTexCoord4x_args[5] = {"target", "s", "t", "r", "q"};
static const trace::FunctionSig _glMultiTexCoord4x_sig = {816, "glMultiTexCoord4x", 5, _glMultiTexCoord4x_args};

static const char * _glNormal3x_args[3] = {"nx", "ny", "nz"};
static const trace::FunctionSig _glNormal3x_sig = {817, "glNormal3x", 3, _glNormal3x_args};

static const char * _glOrthox_args[6] = {"left", "right", "bottom", "top", "zNear", "zFar"};
static const trace::FunctionSig _glOrthox_sig = {818, "glOrthox", 6, _glOrthox_args};

static const char * _glPointParameterx_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPointParameterx_sig = {819, "glPointParameterx", 2, _glPointParameterx_args};

static const char * _glPointParameterxv_args[2] = {"pname", "params"};
static const trace::FunctionSig _glPointParameterxv_sig = {820, "glPointParameterxv", 2, _glPointParameterxv_args};

static const char * _glPointSizex_args[1] = {"size"};
static const trace::FunctionSig _glPointSizex_sig = {821, "glPointSizex", 1, _glPointSizex_args};

static const char * _glPolygonOffsetx_args[2] = {"factor", "units"};
static const trace::FunctionSig _glPolygonOffsetx_sig = {822, "glPolygonOffsetx", 2, _glPolygonOffsetx_args};

static const char * _glRotatex_args[4] = {"angle", "x", "y", "z"};
static const trace::FunctionSig _glRotatex_sig = {823, "glRotatex", 4, _glRotatex_args};

static const char * _glSampleCoveragex_args[2] = {"value", "invert"};
static const trace::FunctionSig _glSampleCoveragex_sig = {824, "glSampleCoveragex", 2, _glSampleCoveragex_args};

static const char * _glScalex_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glScalex_sig = {825, "glScalex", 3, _glScalex_args};

static const char * _glTexEnvx_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glTexEnvx_sig = {826, "glTexEnvx", 3, _glTexEnvx_args};

static const char * _glTexEnvxv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glTexEnvxv_sig = {827, "glTexEnvxv", 3, _glTexEnvxv_args};

static const char * _glTexParameterx_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glTexParameterx_sig = {828, "glTexParameterx", 3, _glTexParameterx_args};

static const char * _glTexParameterxv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glTexParameterxv_sig = {829, "glTexParameterxv", 3, _glTexParameterxv_args};

static const char * _glTranslatex_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glTranslatex_sig = {830, "glTranslatex", 3, _glTranslatex_args};

static const char ** _glBlendBarrier_args = NULL;
static const trace::FunctionSig _glBlendBarrier_sig = {831, "glBlendBarrier", 0, _glBlendBarrier_args};

static const char * _glPrimitiveBoundingBox_args[8] = {"minX", "minY", "minZ", "minW", "maxX", "maxY", "maxZ", "maxW"};
static const trace::FunctionSig _glPrimitiveBoundingBox_sig = {832, "glPrimitiveBoundingBox", 8, _glPrimitiveBoundingBox_args};

static const char * _glTbufferMask3DFX_args[1] = {"mask"};
static const trace::FunctionSig _glTbufferMask3DFX_sig = {833, "glTbufferMask3DFX", 1, _glTbufferMask3DFX_args};

static const char * _glDebugMessageEnableAMD_args[5] = {"category", "severity", "count", "ids", "enabled"};
static const trace::FunctionSig _glDebugMessageEnableAMD_sig = {834, "glDebugMessageEnableAMD", 5, _glDebugMessageEnableAMD_args};

static const char * _glDebugMessageInsertAMD_args[5] = {"category", "severity", "id", "length", "buf"};
static const trace::FunctionSig _glDebugMessageInsertAMD_sig = {835, "glDebugMessageInsertAMD", 5, _glDebugMessageInsertAMD_args};

static const char * _glDebugMessageCallbackAMD_args[2] = {"callback", "userParam"};
static const trace::FunctionSig _glDebugMessageCallbackAMD_sig = {836, "glDebugMessageCallbackAMD", 2, _glDebugMessageCallbackAMD_args};

static const char * _glGetDebugMessageLogAMD_args[7] = {"count", "bufsize", "categories", "severities", "ids", "lengths", "message"};
static const trace::FunctionSig _glGetDebugMessageLogAMD_sig = {837, "glGetDebugMessageLogAMD", 7, _glGetDebugMessageLogAMD_args};

static const char * _glBlendFuncIndexedAMD_args[3] = {"buf", "src", "dst"};
static const trace::FunctionSig _glBlendFuncIndexedAMD_sig = {838, "glBlendFuncIndexedAMD", 3, _glBlendFuncIndexedAMD_args};

static const char * _glBlendFuncSeparateIndexedAMD_args[5] = {"buf", "srcRGB", "dstRGB", "srcAlpha", "dstAlpha"};
static const trace::FunctionSig _glBlendFuncSeparateIndexedAMD_sig = {839, "glBlendFuncSeparateIndexedAMD", 5, _glBlendFuncSeparateIndexedAMD_args};

static const char * _glBlendEquationIndexedAMD_args[2] = {"buf", "mode"};
static const trace::FunctionSig _glBlendEquationIndexedAMD_sig = {840, "glBlendEquationIndexedAMD", 2, _glBlendEquationIndexedAMD_args};

static const char * _glBlendEquationSeparateIndexedAMD_args[3] = {"buf", "modeRGB", "modeAlpha"};
static const trace::FunctionSig _glBlendEquationSeparateIndexedAMD_sig = {841, "glBlendEquationSeparateIndexedAMD", 3, _glBlendEquationSeparateIndexedAMD_args};

static const char * _glVertexAttribParameteriAMD_args[3] = {"index", "pname", "param"};
static const trace::FunctionSig _glVertexAttribParameteriAMD_sig = {842, "glVertexAttribParameteriAMD", 3, _glVertexAttribParameteriAMD_args};

static const char * _glMultiDrawArraysIndirectAMD_args[4] = {"mode", "indirect", "drawcount", "stride"};
static const trace::FunctionSig _glMultiDrawArraysIndirectAMD_sig = {843, "glMultiDrawArraysIndirectAMD", 4, _glMultiDrawArraysIndirectAMD_args};

static const char * _glMultiDrawElementsIndirectAMD_args[5] = {"mode", "type", "indirect", "drawcount", "stride"};
static const trace::FunctionSig _glMultiDrawElementsIndirectAMD_sig = {844, "glMultiDrawElementsIndirectAMD", 5, _glMultiDrawElementsIndirectAMD_args};

static const char * _glGenNamesAMD_args[3] = {"identifier", "num", "names"};
static const trace::FunctionSig _glGenNamesAMD_sig = {845, "glGenNamesAMD", 3, _glGenNamesAMD_args};

static const char * _glDeleteNamesAMD_args[3] = {"identifier", "num", "names"};
static const trace::FunctionSig _glDeleteNamesAMD_sig = {846, "glDeleteNamesAMD", 3, _glDeleteNamesAMD_args};

static const char * _glIsNameAMD_args[2] = {"identifier", "name"};
static const trace::FunctionSig _glIsNameAMD_sig = {847, "glIsNameAMD", 2, _glIsNameAMD_args};

static const char * _glQueryObjectParameteruiAMD_args[4] = {"target", "id", "pname", "param"};
static const trace::FunctionSig _glQueryObjectParameteruiAMD_sig = {848, "glQueryObjectParameteruiAMD", 4, _glQueryObjectParameteruiAMD_args};

static const char * _glGetPerfMonitorGroupsAMD_args[3] = {"numGroups", "groupsSize", "groups"};
static const trace::FunctionSig _glGetPerfMonitorGroupsAMD_sig = {849, "glGetPerfMonitorGroupsAMD", 3, _glGetPerfMonitorGroupsAMD_args};

static const char * _glGetPerfMonitorCountersAMD_args[5] = {"group", "numCounters", "maxActiveCounters", "counterSize", "counters"};
static const trace::FunctionSig _glGetPerfMonitorCountersAMD_sig = {850, "glGetPerfMonitorCountersAMD", 5, _glGetPerfMonitorCountersAMD_args};

static const char * _glGetPerfMonitorGroupStringAMD_args[4] = {"group", "bufSize", "length", "groupString"};
static const trace::FunctionSig _glGetPerfMonitorGroupStringAMD_sig = {851, "glGetPerfMonitorGroupStringAMD", 4, _glGetPerfMonitorGroupStringAMD_args};

static const char * _glGetPerfMonitorCounterStringAMD_args[5] = {"group", "counter", "bufSize", "length", "counterString"};
static const trace::FunctionSig _glGetPerfMonitorCounterStringAMD_sig = {852, "glGetPerfMonitorCounterStringAMD", 5, _glGetPerfMonitorCounterStringAMD_args};

static const char * _glGetPerfMonitorCounterInfoAMD_args[4] = {"group", "counter", "pname", "data"};
static const trace::FunctionSig _glGetPerfMonitorCounterInfoAMD_sig = {853, "glGetPerfMonitorCounterInfoAMD", 4, _glGetPerfMonitorCounterInfoAMD_args};

static const char * _glGenPerfMonitorsAMD_args[2] = {"n", "monitors"};
static const trace::FunctionSig _glGenPerfMonitorsAMD_sig = {854, "glGenPerfMonitorsAMD", 2, _glGenPerfMonitorsAMD_args};

static const char * _glDeletePerfMonitorsAMD_args[2] = {"n", "monitors"};
static const trace::FunctionSig _glDeletePerfMonitorsAMD_sig = {855, "glDeletePerfMonitorsAMD", 2, _glDeletePerfMonitorsAMD_args};

static const char * _glSelectPerfMonitorCountersAMD_args[5] = {"monitor", "enable", "group", "numCounters", "counterList"};
static const trace::FunctionSig _glSelectPerfMonitorCountersAMD_sig = {856, "glSelectPerfMonitorCountersAMD", 5, _glSelectPerfMonitorCountersAMD_args};

static const char * _glBeginPerfMonitorAMD_args[1] = {"monitor"};
static const trace::FunctionSig _glBeginPerfMonitorAMD_sig = {857, "glBeginPerfMonitorAMD", 1, _glBeginPerfMonitorAMD_args};

static const char * _glEndPerfMonitorAMD_args[1] = {"monitor"};
static const trace::FunctionSig _glEndPerfMonitorAMD_sig = {858, "glEndPerfMonitorAMD", 1, _glEndPerfMonitorAMD_args};

static const char * _glGetPerfMonitorCounterDataAMD_args[5] = {"monitor", "pname", "dataSize", "data", "bytesWritten"};
static const trace::FunctionSig _glGetPerfMonitorCounterDataAMD_sig = {859, "glGetPerfMonitorCounterDataAMD", 5, _glGetPerfMonitorCounterDataAMD_args};

static const char * _glSetMultisamplefvAMD_args[3] = {"pname", "index", "val"};
static const trace::FunctionSig _glSetMultisamplefvAMD_sig = {860, "glSetMultisamplefvAMD", 3, _glSetMultisamplefvAMD_args};

static const char * _glTexStorageSparseAMD_args[7] = {"target", "internalFormat", "width", "height", "depth", "layers", "flags"};
static const trace::FunctionSig _glTexStorageSparseAMD_sig = {861, "glTexStorageSparseAMD", 7, _glTexStorageSparseAMD_args};

static const char * _glTextureStorageSparseAMD_args[8] = {"texture", "target", "internalFormat", "width", "height", "depth", "layers", "flags"};
static const trace::FunctionSig _glTextureStorageSparseAMD_sig = {862, "glTextureStorageSparseAMD", 8, _glTextureStorageSparseAMD_args};

static const char * _glStencilOpValueAMD_args[2] = {"face", "value"};
static const trace::FunctionSig _glStencilOpValueAMD_sig = {863, "glStencilOpValueAMD", 2, _glStencilOpValueAMD_args};

static const char * _glTessellationFactorAMD_args[1] = {"factor"};
static const trace::FunctionSig _glTessellationFactorAMD_sig = {864, "glTessellationFactorAMD", 1, _glTessellationFactorAMD_args};

static const char * _glTessellationModeAMD_args[1] = {"mode"};
static const trace::FunctionSig _glTessellationModeAMD_sig = {865, "glTessellationModeAMD", 1, _glTessellationModeAMD_args};

static const char * _glBlitFramebufferANGLE_args[10] = {"srcX0", "srcY0", "srcX1", "srcY1", "dstX0", "dstY0", "dstX1", "dstY1", "mask", "filter"};
static const trace::FunctionSig _glBlitFramebufferANGLE_sig = {866, "glBlitFramebufferANGLE", 10, _glBlitFramebufferANGLE_args};

static const char * _glRenderbufferStorageMultisampleANGLE_args[5] = {"target", "samples", "internalformat", "width", "height"};
static const trace::FunctionSig _glRenderbufferStorageMultisampleANGLE_sig = {867, "glRenderbufferStorageMultisampleANGLE", 5, _glRenderbufferStorageMultisampleANGLE_args};

static const char * _glDrawArraysInstancedANGLE_args[4] = {"mode", "first", "count", "instancecount"};
static const trace::FunctionSig _glDrawArraysInstancedANGLE_sig = {868, "glDrawArraysInstancedANGLE", 4, _glDrawArraysInstancedANGLE_args};

static const char * _glDrawElementsInstancedANGLE_args[5] = {"mode", "count", "type", "indices", "instancecount"};
static const trace::FunctionSig _glDrawElementsInstancedANGLE_sig = {869, "glDrawElementsInstancedANGLE", 5, _glDrawElementsInstancedANGLE_args};

static const char * _glVertexAttribDivisorANGLE_args[2] = {"index", "divisor"};
static const trace::FunctionSig _glVertexAttribDivisorANGLE_sig = {870, "glVertexAttribDivisorANGLE", 2, _glVertexAttribDivisorANGLE_args};

static const char * _glGenQueriesANGLE_args[2] = {"n", "ids"};
static const trace::FunctionSig _glGenQueriesANGLE_sig = {871, "glGenQueriesANGLE", 2, _glGenQueriesANGLE_args};

static const char * _glDeleteQueriesANGLE_args[2] = {"n", "ids"};
static const trace::FunctionSig _glDeleteQueriesANGLE_sig = {872, "glDeleteQueriesANGLE", 2, _glDeleteQueriesANGLE_args};

static const char * _glIsQueryANGLE_args[1] = {"id"};
static const trace::FunctionSig _glIsQueryANGLE_sig = {873, "glIsQueryANGLE", 1, _glIsQueryANGLE_args};

static const char * _glBeginQueryANGLE_args[2] = {"target", "id"};
static const trace::FunctionSig _glBeginQueryANGLE_sig = {874, "glBeginQueryANGLE", 2, _glBeginQueryANGLE_args};

static const char * _glEndQueryANGLE_args[1] = {"target"};
static const trace::FunctionSig _glEndQueryANGLE_sig = {875, "glEndQueryANGLE", 1, _glEndQueryANGLE_args};

static const char * _glQueryCounterANGLE_args[2] = {"id", "target"};
static const trace::FunctionSig _glQueryCounterANGLE_sig = {876, "glQueryCounterANGLE", 2, _glQueryCounterANGLE_args};

static const char * _glGetQueryivANGLE_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetQueryivANGLE_sig = {877, "glGetQueryivANGLE", 3, _glGetQueryivANGLE_args};

static const char * _glGetQueryObjectivANGLE_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectivANGLE_sig = {878, "glGetQueryObjectivANGLE", 3, _glGetQueryObjectivANGLE_args};

static const char * _glGetQueryObjectuivANGLE_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectuivANGLE_sig = {879, "glGetQueryObjectuivANGLE", 3, _glGetQueryObjectuivANGLE_args};

static const char * _glGetQueryObjecti64vANGLE_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjecti64vANGLE_sig = {880, "glGetQueryObjecti64vANGLE", 3, _glGetQueryObjecti64vANGLE_args};

static const char * _glGetQueryObjectui64vANGLE_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectui64vANGLE_sig = {881, "glGetQueryObjectui64vANGLE", 3, _glGetQueryObjectui64vANGLE_args};

static const char * _glGetTranslatedShaderSourceANGLE_args[4] = {"shader", "bufsize", "length", "source"};
static const trace::FunctionSig _glGetTranslatedShaderSourceANGLE_sig = {882, "glGetTranslatedShaderSourceANGLE", 4, _glGetTranslatedShaderSourceANGLE_args};

static const char * _glCopyTextureLevelsAPPLE_args[4] = {"destinationTexture", "sourceTexture", "sourceBaseLevel", "sourceLevelCount"};
static const trace::FunctionSig _glCopyTextureLevelsAPPLE_sig = {883, "glCopyTextureLevelsAPPLE", 4, _glCopyTextureLevelsAPPLE_args};

static const char * _glElementPointerAPPLE_args[2] = {"type", "pointer"};
static const trace::FunctionSig _glElementPointerAPPLE_sig = {884, "glElementPointerAPPLE", 2, _glElementPointerAPPLE_args};

static const char * _glDrawElementArrayAPPLE_args[3] = {"mode", "first", "count"};
static const trace::FunctionSig _glDrawElementArrayAPPLE_sig = {885, "glDrawElementArrayAPPLE", 3, _glDrawElementArrayAPPLE_args};

static const char * _glDrawRangeElementArrayAPPLE_args[5] = {"mode", "start", "end", "first", "count"};
static const trace::FunctionSig _glDrawRangeElementArrayAPPLE_sig = {886, "glDrawRangeElementArrayAPPLE", 5, _glDrawRangeElementArrayAPPLE_args};

static const char * _glMultiDrawElementArrayAPPLE_args[4] = {"mode", "first", "count", "drawcount"};
static const trace::FunctionSig _glMultiDrawElementArrayAPPLE_sig = {887, "glMultiDrawElementArrayAPPLE", 4, _glMultiDrawElementArrayAPPLE_args};

static const char * _glMultiDrawRangeElementArrayAPPLE_args[6] = {"mode", "start", "end", "first", "count", "drawcount"};
static const trace::FunctionSig _glMultiDrawRangeElementArrayAPPLE_sig = {888, "glMultiDrawRangeElementArrayAPPLE", 6, _glMultiDrawRangeElementArrayAPPLE_args};

static const char * _glGenFencesAPPLE_args[2] = {"n", "fences"};
static const trace::FunctionSig _glGenFencesAPPLE_sig = {889, "glGenFencesAPPLE", 2, _glGenFencesAPPLE_args};

static const char * _glDeleteFencesAPPLE_args[2] = {"n", "fences"};
static const trace::FunctionSig _glDeleteFencesAPPLE_sig = {890, "glDeleteFencesAPPLE", 2, _glDeleteFencesAPPLE_args};

static const char * _glSetFenceAPPLE_args[1] = {"fence"};
static const trace::FunctionSig _glSetFenceAPPLE_sig = {891, "glSetFenceAPPLE", 1, _glSetFenceAPPLE_args};

static const char * _glIsFenceAPPLE_args[1] = {"fence"};
static const trace::FunctionSig _glIsFenceAPPLE_sig = {892, "glIsFenceAPPLE", 1, _glIsFenceAPPLE_args};

static const char * _glTestFenceAPPLE_args[1] = {"fence"};
static const trace::FunctionSig _glTestFenceAPPLE_sig = {893, "glTestFenceAPPLE", 1, _glTestFenceAPPLE_args};

static const char * _glFinishFenceAPPLE_args[1] = {"fence"};
static const trace::FunctionSig _glFinishFenceAPPLE_sig = {894, "glFinishFenceAPPLE", 1, _glFinishFenceAPPLE_args};

static const char * _glTestObjectAPPLE_args[2] = {"object", "name"};
static const trace::FunctionSig _glTestObjectAPPLE_sig = {895, "glTestObjectAPPLE", 2, _glTestObjectAPPLE_args};

static const char * _glFinishObjectAPPLE_args[2] = {"object", "name"};
static const trace::FunctionSig _glFinishObjectAPPLE_sig = {896, "glFinishObjectAPPLE", 2, _glFinishObjectAPPLE_args};

static const char * _glBufferParameteriAPPLE_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glBufferParameteriAPPLE_sig = {897, "glBufferParameteriAPPLE", 3, _glBufferParameteriAPPLE_args};

static const char * _glFlushMappedBufferRangeAPPLE_args[3] = {"target", "offset", "size"};
static const trace::FunctionSig _glFlushMappedBufferRangeAPPLE_sig = {898, "glFlushMappedBufferRangeAPPLE", 3, _glFlushMappedBufferRangeAPPLE_args};

static const char ** _glFlushRenderAPPLE_args = NULL;
static const trace::FunctionSig _glFlushRenderAPPLE_sig = {899, "glFlushRenderAPPLE", 0, _glFlushRenderAPPLE_args};

static const char ** _glFinishRenderAPPLE_args = NULL;
static const trace::FunctionSig _glFinishRenderAPPLE_sig = {900, "glFinishRenderAPPLE", 0, _glFinishRenderAPPLE_args};

static const char ** _glSwapAPPLE_args = NULL;
static const trace::FunctionSig _glSwapAPPLE_sig = {901, "glSwapAPPLE", 0, _glSwapAPPLE_args};

static const char * _glRenderbufferStorageMultisampleAPPLE_args[5] = {"target", "samples", "internalformat", "width", "height"};
static const trace::FunctionSig _glRenderbufferStorageMultisampleAPPLE_sig = {902, "glRenderbufferStorageMultisampleAPPLE", 5, _glRenderbufferStorageMultisampleAPPLE_args};

static const char ** _glResolveMultisampleFramebufferAPPLE_args = NULL;
static const trace::FunctionSig _glResolveMultisampleFramebufferAPPLE_sig = {903, "glResolveMultisampleFramebufferAPPLE", 0, _glResolveMultisampleFramebufferAPPLE_args};

static const char * _glObjectPurgeableAPPLE_args[3] = {"objectType", "name", "option"};
static const trace::FunctionSig _glObjectPurgeableAPPLE_sig = {904, "glObjectPurgeableAPPLE", 3, _glObjectPurgeableAPPLE_args};

static const char * _glObjectUnpurgeableAPPLE_args[3] = {"objectType", "name", "option"};
static const trace::FunctionSig _glObjectUnpurgeableAPPLE_sig = {905, "glObjectUnpurgeableAPPLE", 3, _glObjectUnpurgeableAPPLE_args};

static const char * _glGetObjectParameterivAPPLE_args[4] = {"objectType", "name", "pname", "params"};
static const trace::FunctionSig _glGetObjectParameterivAPPLE_sig = {906, "glGetObjectParameterivAPPLE", 4, _glGetObjectParameterivAPPLE_args};

static const char * _glPointSizePointerAPPLE_args[3] = {"type", "stride", "pointer"};
static const trace::FunctionSig _glPointSizePointerAPPLE_sig = {907, "glPointSizePointerAPPLE", 3, _glPointSizePointerAPPLE_args};

static const char * _glVertexPointSizefAPPLE_args[1] = {"size"};
static const trace::FunctionSig _glVertexPointSizefAPPLE_sig = {908, "glVertexPointSizefAPPLE", 1, _glVertexPointSizefAPPLE_args};

static const char * _glFenceSyncAPPLE_args[2] = {"condition", "flags"};
static const trace::FunctionSig _glFenceSyncAPPLE_sig = {909, "glFenceSyncAPPLE", 2, _glFenceSyncAPPLE_args};

static const char * _glIsSyncAPPLE_args[1] = {"sync"};
static const trace::FunctionSig _glIsSyncAPPLE_sig = {910, "glIsSyncAPPLE", 1, _glIsSyncAPPLE_args};

static const char * _glDeleteSyncAPPLE_args[1] = {"sync"};
static const trace::FunctionSig _glDeleteSyncAPPLE_sig = {911, "glDeleteSyncAPPLE", 1, _glDeleteSyncAPPLE_args};

static const char * _glClientWaitSyncAPPLE_args[3] = {"sync", "flags", "timeout"};
static const trace::FunctionSig _glClientWaitSyncAPPLE_sig = {912, "glClientWaitSyncAPPLE", 3, _glClientWaitSyncAPPLE_args};

static const char * _glWaitSyncAPPLE_args[3] = {"sync", "flags", "timeout"};
static const trace::FunctionSig _glWaitSyncAPPLE_sig = {913, "glWaitSyncAPPLE", 3, _glWaitSyncAPPLE_args};

static const char * _glGetInteger64vAPPLE_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetInteger64vAPPLE_sig = {914, "glGetInteger64vAPPLE", 2, _glGetInteger64vAPPLE_args};

static const char * _glGetSyncivAPPLE_args[5] = {"sync", "pname", "bufSize", "length", "values"};
static const trace::FunctionSig _glGetSyncivAPPLE_sig = {915, "glGetSyncivAPPLE", 5, _glGetSyncivAPPLE_args};

static const char * _glTextureRangeAPPLE_args[3] = {"target", "length", "pointer"};
static const trace::FunctionSig _glTextureRangeAPPLE_sig = {916, "glTextureRangeAPPLE", 3, _glTextureRangeAPPLE_args};

static const char * _glGetTexParameterPointervAPPLE_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexParameterPointervAPPLE_sig = {917, "glGetTexParameterPointervAPPLE", 3, _glGetTexParameterPointervAPPLE_args};

static const char * _glBindVertexArrayAPPLE_args[1] = {"array"};
static const trace::FunctionSig _glBindVertexArrayAPPLE_sig = {918, "glBindVertexArrayAPPLE", 1, _glBindVertexArrayAPPLE_args};

static const char * _glDeleteVertexArraysAPPLE_args[2] = {"n", "arrays"};
static const trace::FunctionSig _glDeleteVertexArraysAPPLE_sig = {919, "glDeleteVertexArraysAPPLE", 2, _glDeleteVertexArraysAPPLE_args};

static const char * _glGenVertexArraysAPPLE_args[2] = {"n", "arrays"};
static const trace::FunctionSig _glGenVertexArraysAPPLE_sig = {920, "glGenVertexArraysAPPLE", 2, _glGenVertexArraysAPPLE_args};

static const char * _glIsVertexArrayAPPLE_args[1] = {"array"};
static const trace::FunctionSig _glIsVertexArrayAPPLE_sig = {921, "glIsVertexArrayAPPLE", 1, _glIsVertexArrayAPPLE_args};

static const char * _glVertexArrayRangeAPPLE_args[2] = {"length", "pointer"};
static const trace::FunctionSig _glVertexArrayRangeAPPLE_sig = {922, "glVertexArrayRangeAPPLE", 2, _glVertexArrayRangeAPPLE_args};

static const char * _glFlushVertexArrayRangeAPPLE_args[2] = {"length", "pointer"};
static const trace::FunctionSig _glFlushVertexArrayRangeAPPLE_sig = {923, "glFlushVertexArrayRangeAPPLE", 2, _glFlushVertexArrayRangeAPPLE_args};

static const char * _glVertexArrayParameteriAPPLE_args[2] = {"pname", "param"};
static const trace::FunctionSig _glVertexArrayParameteriAPPLE_sig = {924, "glVertexArrayParameteriAPPLE", 2, _glVertexArrayParameteriAPPLE_args};

static const char * _glEnableVertexAttribAPPLE_args[2] = {"index", "pname"};
static const trace::FunctionSig _glEnableVertexAttribAPPLE_sig = {925, "glEnableVertexAttribAPPLE", 2, _glEnableVertexAttribAPPLE_args};

static const char * _glDisableVertexAttribAPPLE_args[2] = {"index", "pname"};
static const trace::FunctionSig _glDisableVertexAttribAPPLE_sig = {926, "glDisableVertexAttribAPPLE", 2, _glDisableVertexAttribAPPLE_args};

static const char * _glIsVertexAttribEnabledAPPLE_args[2] = {"index", "pname"};
static const trace::FunctionSig _glIsVertexAttribEnabledAPPLE_sig = {927, "glIsVertexAttribEnabledAPPLE", 2, _glIsVertexAttribEnabledAPPLE_args};

static const char * _glMapVertexAttrib1dAPPLE_args[7] = {"index", "size", "u1", "u2", "stride", "order", "points"};
static const trace::FunctionSig _glMapVertexAttrib1dAPPLE_sig = {928, "glMapVertexAttrib1dAPPLE", 7, _glMapVertexAttrib1dAPPLE_args};

static const char * _glMapVertexAttrib1fAPPLE_args[7] = {"index", "size", "u1", "u2", "stride", "order", "points"};
static const trace::FunctionSig _glMapVertexAttrib1fAPPLE_sig = {929, "glMapVertexAttrib1fAPPLE", 7, _glMapVertexAttrib1fAPPLE_args};

static const char * _glMapVertexAttrib2dAPPLE_args[11] = {"index", "size", "u1", "u2", "ustride", "uorder", "v1", "v2", "vstride", "vorder", "points"};
static const trace::FunctionSig _glMapVertexAttrib2dAPPLE_sig = {930, "glMapVertexAttrib2dAPPLE", 11, _glMapVertexAttrib2dAPPLE_args};

static const char * _glMapVertexAttrib2fAPPLE_args[11] = {"index", "size", "u1", "u2", "ustride", "uorder", "v1", "v2", "vstride", "vorder", "points"};
static const trace::FunctionSig _glMapVertexAttrib2fAPPLE_sig = {931, "glMapVertexAttrib2fAPPLE", 11, _glMapVertexAttrib2fAPPLE_args};

static const char ** _glReleaseShaderCompiler_args = NULL;
static const trace::FunctionSig _glReleaseShaderCompiler_sig = {932, "glReleaseShaderCompiler", 0, _glReleaseShaderCompiler_args};

static const char * _glShaderBinary_args[5] = {"count", "shaders", "binaryformat", "binary", "length"};
static const trace::FunctionSig _glShaderBinary_sig = {933, "glShaderBinary", 5, _glShaderBinary_args};

static const char * _glGetShaderPrecisionFormat_args[4] = {"shadertype", "precisiontype", "range", "precision"};
static const trace::FunctionSig _glGetShaderPrecisionFormat_sig = {934, "glGetShaderPrecisionFormat", 4, _glGetShaderPrecisionFormat_args};

static const char * _glDepthRangef_args[2] = {"n", "f"};
static const trace::FunctionSig _glDepthRangef_sig = {935, "glDepthRangef", 2, _glDepthRangef_args};

static const char * _glClearDepthf_args[1] = {"d"};
static const trace::FunctionSig _glClearDepthf_sig = {936, "glClearDepthf", 1, _glClearDepthf_args};

static const char * _glMemoryBarrierByRegion_args[1] = {"barriers"};
static const trace::FunctionSig _glMemoryBarrierByRegion_sig = {937, "glMemoryBarrierByRegion", 1, _glMemoryBarrierByRegion_args};

static const char * _glPrimitiveBoundingBoxARB_args[8] = {"minX", "minY", "minZ", "minW", "maxX", "maxY", "maxZ", "maxW"};
static const trace::FunctionSig _glPrimitiveBoundingBoxARB_sig = {938, "glPrimitiveBoundingBoxARB", 8, _glPrimitiveBoundingBoxARB_args};

static const char * _glDrawArraysInstancedBaseInstance_args[5] = {"mode", "first", "count", "instancecount", "baseinstance"};
static const trace::FunctionSig _glDrawArraysInstancedBaseInstance_sig = {939, "glDrawArraysInstancedBaseInstance", 5, _glDrawArraysInstancedBaseInstance_args};

static const char * _glDrawElementsInstancedBaseInstance_args[6] = {"mode", "count", "type", "indices", "instancecount", "baseinstance"};
static const trace::FunctionSig _glDrawElementsInstancedBaseInstance_sig = {940, "glDrawElementsInstancedBaseInstance", 6, _glDrawElementsInstancedBaseInstance_args};

static const char * _glDrawElementsInstancedBaseVertexBaseInstance_args[7] = {"mode", "count", "type", "indices", "instancecount", "basevertex", "baseinstance"};
static const trace::FunctionSig _glDrawElementsInstancedBaseVertexBaseInstance_sig = {941, "glDrawElementsInstancedBaseVertexBaseInstance", 7, _glDrawElementsInstancedBaseVertexBaseInstance_args};

static const char * _glGetTextureHandleARB_args[1] = {"texture"};
static const trace::FunctionSig _glGetTextureHandleARB_sig = {942, "glGetTextureHandleARB", 1, _glGetTextureHandleARB_args};

static const char * _glGetTextureSamplerHandleARB_args[2] = {"texture", "sampler"};
static const trace::FunctionSig _glGetTextureSamplerHandleARB_sig = {943, "glGetTextureSamplerHandleARB", 2, _glGetTextureSamplerHandleARB_args};

static const char * _glMakeTextureHandleResidentARB_args[1] = {"handle"};
static const trace::FunctionSig _glMakeTextureHandleResidentARB_sig = {944, "glMakeTextureHandleResidentARB", 1, _glMakeTextureHandleResidentARB_args};

static const char * _glMakeTextureHandleNonResidentARB_args[1] = {"handle"};
static const trace::FunctionSig _glMakeTextureHandleNonResidentARB_sig = {945, "glMakeTextureHandleNonResidentARB", 1, _glMakeTextureHandleNonResidentARB_args};

static const char * _glGetImageHandleARB_args[5] = {"texture", "level", "layered", "layer", "format"};
static const trace::FunctionSig _glGetImageHandleARB_sig = {946, "glGetImageHandleARB", 5, _glGetImageHandleARB_args};

static const char * _glMakeImageHandleResidentARB_args[2] = {"handle", "access"};
static const trace::FunctionSig _glMakeImageHandleResidentARB_sig = {947, "glMakeImageHandleResidentARB", 2, _glMakeImageHandleResidentARB_args};

static const char * _glMakeImageHandleNonResidentARB_args[1] = {"handle"};
static const trace::FunctionSig _glMakeImageHandleNonResidentARB_sig = {948, "glMakeImageHandleNonResidentARB", 1, _glMakeImageHandleNonResidentARB_args};

static const char * _glUniformHandleui64ARB_args[2] = {"location", "value"};
static const trace::FunctionSig _glUniformHandleui64ARB_sig = {949, "glUniformHandleui64ARB", 2, _glUniformHandleui64ARB_args};

static const char * _glUniformHandleui64vARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniformHandleui64vARB_sig = {950, "glUniformHandleui64vARB", 3, _glUniformHandleui64vARB_args};

static const char * _glProgramUniformHandleui64ARB_args[3] = {"program", "location", "value"};
static const trace::FunctionSig _glProgramUniformHandleui64ARB_sig = {951, "glProgramUniformHandleui64ARB", 3, _glProgramUniformHandleui64ARB_args};

static const char * _glProgramUniformHandleui64vARB_args[4] = {"program", "location", "count", "values"};
static const trace::FunctionSig _glProgramUniformHandleui64vARB_sig = {952, "glProgramUniformHandleui64vARB", 4, _glProgramUniformHandleui64vARB_args};

static const char * _glIsTextureHandleResidentARB_args[1] = {"handle"};
static const trace::FunctionSig _glIsTextureHandleResidentARB_sig = {953, "glIsTextureHandleResidentARB", 1, _glIsTextureHandleResidentARB_args};

static const char * _glIsImageHandleResidentARB_args[1] = {"handle"};
static const trace::FunctionSig _glIsImageHandleResidentARB_sig = {954, "glIsImageHandleResidentARB", 1, _glIsImageHandleResidentARB_args};

static const char * _glVertexAttribL1ui64ARB_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttribL1ui64ARB_sig = {955, "glVertexAttribL1ui64ARB", 2, _glVertexAttribL1ui64ARB_args};

static const char * _glVertexAttribL1ui64vARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL1ui64vARB_sig = {956, "glVertexAttribL1ui64vARB", 2, _glVertexAttribL1ui64vARB_args};

static const char * _glGetVertexAttribLui64vARB_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribLui64vARB_sig = {957, "glGetVertexAttribLui64vARB", 3, _glGetVertexAttribLui64vARB_args};

static const char * _glBindFragDataLocationIndexed_args[4] = {"program", "colorNumber", "index", "name"};
static const trace::FunctionSig _glBindFragDataLocationIndexed_sig = {958, "glBindFragDataLocationIndexed", 4, _glBindFragDataLocationIndexed_args};

static const char * _glGetFragDataIndex_args[2] = {"program", "name"};
static const trace::FunctionSig _glGetFragDataIndex_sig = {959, "glGetFragDataIndex", 2, _glGetFragDataIndex_args};

static const char * _glBufferStorage_args[4] = {"target", "size", "data", "flags"};
static const trace::FunctionSig _glBufferStorage_sig = {960, "glBufferStorage", 4, _glBufferStorage_args};

static const char * _glCreateSyncFromCLeventARB_args[3] = {"context", "event", "flags"};
static const trace::FunctionSig _glCreateSyncFromCLeventARB_sig = {961, "glCreateSyncFromCLeventARB", 3, _glCreateSyncFromCLeventARB_args};

static const char * _glClearBufferData_args[5] = {"target", "internalformat", "format", "type", "data"};
static const trace::FunctionSig _glClearBufferData_sig = {962, "glClearBufferData", 5, _glClearBufferData_args};

static const char * _glClearBufferSubData_args[7] = {"target", "internalformat", "offset", "size", "format", "type", "data"};
static const trace::FunctionSig _glClearBufferSubData_sig = {963, "glClearBufferSubData", 7, _glClearBufferSubData_args};

static const char * _glClearTexImage_args[5] = {"texture", "level", "format", "type", "data"};
static const trace::FunctionSig _glClearTexImage_sig = {964, "glClearTexImage", 5, _glClearTexImage_args};

static const char * _glClearTexSubImage_args[11] = {"texture", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "type", "data"};
static const trace::FunctionSig _glClearTexSubImage_sig = {965, "glClearTexSubImage", 11, _glClearTexSubImage_args};

static const char * _glClipControl_args[2] = {"origin", "depth"};
static const trace::FunctionSig _glClipControl_sig = {966, "glClipControl", 2, _glClipControl_args};

static const char * _glClampColorARB_args[2] = {"target", "clamp"};
static const trace::FunctionSig _glClampColorARB_sig = {967, "glClampColorARB", 2, _glClampColorARB_args};

static const char * _glDispatchCompute_args[3] = {"num_groups_x", "num_groups_y", "num_groups_z"};
static const trace::FunctionSig _glDispatchCompute_sig = {968, "glDispatchCompute", 3, _glDispatchCompute_args};

static const char * _glDispatchComputeIndirect_args[1] = {"indirect"};
static const trace::FunctionSig _glDispatchComputeIndirect_sig = {969, "glDispatchComputeIndirect", 1, _glDispatchComputeIndirect_args};

static const char * _glDispatchComputeGroupSizeARB_args[6] = {"num_groups_x", "num_groups_y", "num_groups_z", "group_size_x", "group_size_y", "group_size_z"};
static const trace::FunctionSig _glDispatchComputeGroupSizeARB_sig = {970, "glDispatchComputeGroupSizeARB", 6, _glDispatchComputeGroupSizeARB_args};

static const char * _glCopyBufferSubData_args[5] = {"readTarget", "writeTarget", "readOffset", "writeOffset", "size"};
static const trace::FunctionSig _glCopyBufferSubData_sig = {971, "glCopyBufferSubData", 5, _glCopyBufferSubData_args};

static const char * _glCopyImageSubData_args[15] = {"srcName", "srcTarget", "srcLevel", "srcX", "srcY", "srcZ", "dstName", "dstTarget", "dstLevel", "dstX", "dstY", "dstZ", "srcWidth", "srcHeight", "srcDepth"};
static const trace::FunctionSig _glCopyImageSubData_sig = {972, "glCopyImageSubData", 15, _glCopyImageSubData_args};

static const char * _glDebugMessageControlARB_args[6] = {"source", "type", "severity", "count", "ids", "enabled"};
static const trace::FunctionSig _glDebugMessageControlARB_sig = {973, "glDebugMessageControlARB", 6, _glDebugMessageControlARB_args};

static const char * _glDebugMessageInsertARB_args[6] = {"source", "type", "id", "severity", "length", "buf"};
static const trace::FunctionSig _glDebugMessageInsertARB_sig = {974, "glDebugMessageInsertARB", 6, _glDebugMessageInsertARB_args};

static const char * _glDebugMessageCallbackARB_args[2] = {"callback", "userParam"};
static const trace::FunctionSig _glDebugMessageCallbackARB_sig = {975, "glDebugMessageCallbackARB", 2, _glDebugMessageCallbackARB_args};

static const char * _glGetDebugMessageLogARB_args[8] = {"count", "bufsize", "sources", "types", "ids", "severities", "lengths", "messageLog"};
static const trace::FunctionSig _glGetDebugMessageLogARB_sig = {976, "glGetDebugMessageLogARB", 8, _glGetDebugMessageLogARB_args};

static const char * _glCreateTransformFeedbacks_args[2] = {"n", "ids"};
static const trace::FunctionSig _glCreateTransformFeedbacks_sig = {977, "glCreateTransformFeedbacks", 2, _glCreateTransformFeedbacks_args};

static const char * _glTransformFeedbackBufferBase_args[3] = {"xfb", "index", "buffer"};
static const trace::FunctionSig _glTransformFeedbackBufferBase_sig = {978, "glTransformFeedbackBufferBase", 3, _glTransformFeedbackBufferBase_args};

static const char * _glTransformFeedbackBufferRange_args[5] = {"xfb", "index", "buffer", "offset", "size"};
static const trace::FunctionSig _glTransformFeedbackBufferRange_sig = {979, "glTransformFeedbackBufferRange", 5, _glTransformFeedbackBufferRange_args};

static const char * _glGetTransformFeedbackiv_args[3] = {"xfb", "pname", "param"};
static const trace::FunctionSig _glGetTransformFeedbackiv_sig = {980, "glGetTransformFeedbackiv", 3, _glGetTransformFeedbackiv_args};

static const char * _glGetTransformFeedbacki_v_args[4] = {"xfb", "pname", "index", "param"};
static const trace::FunctionSig _glGetTransformFeedbacki_v_sig = {981, "glGetTransformFeedbacki_v", 4, _glGetTransformFeedbacki_v_args};

static const char * _glGetTransformFeedbacki64_v_args[4] = {"xfb", "pname", "index", "param"};
static const trace::FunctionSig _glGetTransformFeedbacki64_v_sig = {982, "glGetTransformFeedbacki64_v", 4, _glGetTransformFeedbacki64_v_args};

static const char * _glCreateBuffers_args[2] = {"n", "buffers"};
static const trace::FunctionSig _glCreateBuffers_sig = {983, "glCreateBuffers", 2, _glCreateBuffers_args};

static const char * _glNamedBufferStorage_args[4] = {"buffer", "size", "data", "flags"};
static const trace::FunctionSig _glNamedBufferStorage_sig = {984, "glNamedBufferStorage", 4, _glNamedBufferStorage_args};

static const char * _glNamedBufferData_args[4] = {"buffer", "size", "data", "usage"};
static const trace::FunctionSig _glNamedBufferData_sig = {985, "glNamedBufferData", 4, _glNamedBufferData_args};

static const char * _glNamedBufferSubData_args[4] = {"buffer", "offset", "size", "data"};
static const trace::FunctionSig _glNamedBufferSubData_sig = {986, "glNamedBufferSubData", 4, _glNamedBufferSubData_args};

static const char * _glCopyNamedBufferSubData_args[5] = {"readBuffer", "writeBuffer", "readOffset", "writeOffset", "size"};
static const trace::FunctionSig _glCopyNamedBufferSubData_sig = {987, "glCopyNamedBufferSubData", 5, _glCopyNamedBufferSubData_args};

static const char * _glClearNamedBufferData_args[5] = {"buffer", "internalformat", "format", "type", "data"};
static const trace::FunctionSig _glClearNamedBufferData_sig = {988, "glClearNamedBufferData", 5, _glClearNamedBufferData_args};

static const char * _glClearNamedBufferSubData_args[7] = {"buffer", "internalformat", "offset", "size", "format", "type", "data"};
static const trace::FunctionSig _glClearNamedBufferSubData_sig = {989, "glClearNamedBufferSubData", 7, _glClearNamedBufferSubData_args};

static const char * _glMapNamedBuffer_args[2] = {"buffer", "access"};
static const trace::FunctionSig _glMapNamedBuffer_sig = {990, "glMapNamedBuffer", 2, _glMapNamedBuffer_args};

static const char * _glMapNamedBufferRange_args[4] = {"buffer", "offset", "length", "access"};
static const trace::FunctionSig _glMapNamedBufferRange_sig = {991, "glMapNamedBufferRange", 4, _glMapNamedBufferRange_args};

static const char * _glUnmapNamedBuffer_args[1] = {"buffer"};
static const trace::FunctionSig _glUnmapNamedBuffer_sig = {992, "glUnmapNamedBuffer", 1, _glUnmapNamedBuffer_args};

static const char * _glFlushMappedNamedBufferRange_args[3] = {"buffer", "offset", "length"};
static const trace::FunctionSig _glFlushMappedNamedBufferRange_sig = {993, "glFlushMappedNamedBufferRange", 3, _glFlushMappedNamedBufferRange_args};

static const char * _glGetNamedBufferParameteriv_args[3] = {"buffer", "pname", "params"};
static const trace::FunctionSig _glGetNamedBufferParameteriv_sig = {994, "glGetNamedBufferParameteriv", 3, _glGetNamedBufferParameteriv_args};

static const char * _glGetNamedBufferParameteri64v_args[3] = {"buffer", "pname", "params"};
static const trace::FunctionSig _glGetNamedBufferParameteri64v_sig = {995, "glGetNamedBufferParameteri64v", 3, _glGetNamedBufferParameteri64v_args};

static const char * _glGetNamedBufferPointerv_args[3] = {"buffer", "pname", "params"};
static const trace::FunctionSig _glGetNamedBufferPointerv_sig = {996, "glGetNamedBufferPointerv", 3, _glGetNamedBufferPointerv_args};

static const char * _glGetNamedBufferSubData_args[4] = {"buffer", "offset", "size", "data"};
static const trace::FunctionSig _glGetNamedBufferSubData_sig = {997, "glGetNamedBufferSubData", 4, _glGetNamedBufferSubData_args};

static const char * _glCreateFramebuffers_args[2] = {"n", "framebuffers"};
static const trace::FunctionSig _glCreateFramebuffers_sig = {998, "glCreateFramebuffers", 2, _glCreateFramebuffers_args};

static const char * _glNamedFramebufferRenderbuffer_args[4] = {"framebuffer", "attachment", "renderbuffertarget", "renderbuffer"};
static const trace::FunctionSig _glNamedFramebufferRenderbuffer_sig = {999, "glNamedFramebufferRenderbuffer", 4, _glNamedFramebufferRenderbuffer_args};

static const char * _glNamedFramebufferParameteri_args[3] = {"framebuffer", "pname", "param"};
static const trace::FunctionSig _glNamedFramebufferParameteri_sig = {1000, "glNamedFramebufferParameteri", 3, _glNamedFramebufferParameteri_args};

static const char * _glNamedFramebufferTexture_args[4] = {"framebuffer", "attachment", "texture", "level"};
static const trace::FunctionSig _glNamedFramebufferTexture_sig = {1001, "glNamedFramebufferTexture", 4, _glNamedFramebufferTexture_args};

static const char * _glNamedFramebufferTextureLayer_args[5] = {"framebuffer", "attachment", "texture", "level", "layer"};
static const trace::FunctionSig _glNamedFramebufferTextureLayer_sig = {1002, "glNamedFramebufferTextureLayer", 5, _glNamedFramebufferTextureLayer_args};

static const char * _glNamedFramebufferDrawBuffer_args[2] = {"framebuffer", "buf"};
static const trace::FunctionSig _glNamedFramebufferDrawBuffer_sig = {1003, "glNamedFramebufferDrawBuffer", 2, _glNamedFramebufferDrawBuffer_args};

static const char * _glNamedFramebufferDrawBuffers_args[3] = {"framebuffer", "n", "bufs"};
static const trace::FunctionSig _glNamedFramebufferDrawBuffers_sig = {1004, "glNamedFramebufferDrawBuffers", 3, _glNamedFramebufferDrawBuffers_args};

static const char * _glNamedFramebufferReadBuffer_args[2] = {"framebuffer", "src"};
static const trace::FunctionSig _glNamedFramebufferReadBuffer_sig = {1005, "glNamedFramebufferReadBuffer", 2, _glNamedFramebufferReadBuffer_args};

static const char * _glInvalidateNamedFramebufferData_args[3] = {"framebuffer", "numAttachments", "attachments"};
static const trace::FunctionSig _glInvalidateNamedFramebufferData_sig = {1006, "glInvalidateNamedFramebufferData", 3, _glInvalidateNamedFramebufferData_args};

static const char * _glInvalidateNamedFramebufferSubData_args[7] = {"framebuffer", "numAttachments", "attachments", "x", "y", "width", "height"};
static const trace::FunctionSig _glInvalidateNamedFramebufferSubData_sig = {1007, "glInvalidateNamedFramebufferSubData", 7, _glInvalidateNamedFramebufferSubData_args};

static const char * _glClearNamedFramebufferiv_args[4] = {"framebuffer", "buffer", "drawbuffer", "value"};
static const trace::FunctionSig _glClearNamedFramebufferiv_sig = {1008, "glClearNamedFramebufferiv", 4, _glClearNamedFramebufferiv_args};

static const char * _glClearNamedFramebufferuiv_args[4] = {"framebuffer", "buffer", "drawbuffer", "value"};
static const trace::FunctionSig _glClearNamedFramebufferuiv_sig = {1009, "glClearNamedFramebufferuiv", 4, _glClearNamedFramebufferuiv_args};

static const char * _glClearNamedFramebufferfv_args[4] = {"framebuffer", "buffer", "drawbuffer", "value"};
static const trace::FunctionSig _glClearNamedFramebufferfv_sig = {1010, "glClearNamedFramebufferfv", 4, _glClearNamedFramebufferfv_args};

static const char * _glClearNamedFramebufferfi_args[5] = {"framebuffer", "buffer", "drawbuffer", "depth", "stencil"};
static const trace::FunctionSig _glClearNamedFramebufferfi_sig = {1011, "glClearNamedFramebufferfi", 5, _glClearNamedFramebufferfi_args};

static const char * _glBlitNamedFramebuffer_args[12] = {"readFramebuffer", "drawFramebuffer", "srcX0", "srcY0", "srcX1", "srcY1", "dstX0", "dstY0", "dstX1", "dstY1", "mask", "filter"};
static const trace::FunctionSig _glBlitNamedFramebuffer_sig = {1012, "glBlitNamedFramebuffer", 12, _glBlitNamedFramebuffer_args};

static const char * _glCheckNamedFramebufferStatus_args[2] = {"framebuffer", "target"};
static const trace::FunctionSig _glCheckNamedFramebufferStatus_sig = {1013, "glCheckNamedFramebufferStatus", 2, _glCheckNamedFramebufferStatus_args};

static const char * _glGetNamedFramebufferParameteriv_args[3] = {"framebuffer", "pname", "param"};
static const trace::FunctionSig _glGetNamedFramebufferParameteriv_sig = {1014, "glGetNamedFramebufferParameteriv", 3, _glGetNamedFramebufferParameteriv_args};

static const char * _glGetNamedFramebufferAttachmentParameteriv_args[4] = {"framebuffer", "attachment", "pname", "params"};
static const trace::FunctionSig _glGetNamedFramebufferAttachmentParameteriv_sig = {1015, "glGetNamedFramebufferAttachmentParameteriv", 4, _glGetNamedFramebufferAttachmentParameteriv_args};

static const char * _glCreateRenderbuffers_args[2] = {"n", "renderbuffers"};
static const trace::FunctionSig _glCreateRenderbuffers_sig = {1016, "glCreateRenderbuffers", 2, _glCreateRenderbuffers_args};

static const char * _glNamedRenderbufferStorage_args[4] = {"renderbuffer", "internalformat", "width", "height"};
static const trace::FunctionSig _glNamedRenderbufferStorage_sig = {1017, "glNamedRenderbufferStorage", 4, _glNamedRenderbufferStorage_args};

static const char * _glNamedRenderbufferStorageMultisample_args[5] = {"renderbuffer", "samples", "internalformat", "width", "height"};
static const trace::FunctionSig _glNamedRenderbufferStorageMultisample_sig = {1018, "glNamedRenderbufferStorageMultisample", 5, _glNamedRenderbufferStorageMultisample_args};

static const char * _glGetNamedRenderbufferParameteriv_args[3] = {"renderbuffer", "pname", "params"};
static const trace::FunctionSig _glGetNamedRenderbufferParameteriv_sig = {1019, "glGetNamedRenderbufferParameteriv", 3, _glGetNamedRenderbufferParameteriv_args};

static const char * _glCreateTextures_args[3] = {"target", "n", "textures"};
static const trace::FunctionSig _glCreateTextures_sig = {1020, "glCreateTextures", 3, _glCreateTextures_args};

static const char * _glTextureBuffer_args[3] = {"texture", "internalformat", "buffer"};
static const trace::FunctionSig _glTextureBuffer_sig = {1021, "glTextureBuffer", 3, _glTextureBuffer_args};

static const char * _glTextureBufferRange_args[5] = {"texture", "internalformat", "buffer", "offset", "size"};
static const trace::FunctionSig _glTextureBufferRange_sig = {1022, "glTextureBufferRange", 5, _glTextureBufferRange_args};

static const char * _glTextureStorage1D_args[4] = {"texture", "levels", "internalformat", "width"};
static const trace::FunctionSig _glTextureStorage1D_sig = {1023, "glTextureStorage1D", 4, _glTextureStorage1D_args};

static const char * _glTextureStorage2D_args[5] = {"texture", "levels", "internalformat", "width", "height"};
static const trace::FunctionSig _glTextureStorage2D_sig = {1024, "glTextureStorage2D", 5, _glTextureStorage2D_args};

static const char * _glTextureStorage3D_args[6] = {"texture", "levels", "internalformat", "width", "height", "depth"};
static const trace::FunctionSig _glTextureStorage3D_sig = {1025, "glTextureStorage3D", 6, _glTextureStorage3D_args};

static const char * _glTextureStorage2DMultisample_args[6] = {"texture", "samples", "internalformat", "width", "height", "fixedsamplelocations"};
static const trace::FunctionSig _glTextureStorage2DMultisample_sig = {1026, "glTextureStorage2DMultisample", 6, _glTextureStorage2DMultisample_args};

static const char * _glTextureStorage3DMultisample_args[7] = {"texture", "samples", "internalformat", "width", "height", "depth", "fixedsamplelocations"};
static const trace::FunctionSig _glTextureStorage3DMultisample_sig = {1027, "glTextureStorage3DMultisample", 7, _glTextureStorage3DMultisample_args};

static const char * _glTextureSubImage1D_args[7] = {"texture", "level", "xoffset", "width", "format", "type", "pixels"};
static const trace::FunctionSig _glTextureSubImage1D_sig = {1028, "glTextureSubImage1D", 7, _glTextureSubImage1D_args};

static const char * _glTextureSubImage2D_args[9] = {"texture", "level", "xoffset", "yoffset", "width", "height", "format", "type", "pixels"};
static const trace::FunctionSig _glTextureSubImage2D_sig = {1029, "glTextureSubImage2D", 9, _glTextureSubImage2D_args};

static const char * _glTextureSubImage3D_args[11] = {"texture", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "type", "pixels"};
static const trace::FunctionSig _glTextureSubImage3D_sig = {1030, "glTextureSubImage3D", 11, _glTextureSubImage3D_args};

static const char * _glCompressedTextureSubImage1D_args[7] = {"texture", "level", "xoffset", "width", "format", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTextureSubImage1D_sig = {1031, "glCompressedTextureSubImage1D", 7, _glCompressedTextureSubImage1D_args};

static const char * _glCompressedTextureSubImage2D_args[9] = {"texture", "level", "xoffset", "yoffset", "width", "height", "format", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTextureSubImage2D_sig = {1032, "glCompressedTextureSubImage2D", 9, _glCompressedTextureSubImage2D_args};

static const char * _glCompressedTextureSubImage3D_args[11] = {"texture", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTextureSubImage3D_sig = {1033, "glCompressedTextureSubImage3D", 11, _glCompressedTextureSubImage3D_args};

static const char * _glCopyTextureSubImage1D_args[6] = {"texture", "level", "xoffset", "x", "y", "width"};
static const trace::FunctionSig _glCopyTextureSubImage1D_sig = {1034, "glCopyTextureSubImage1D", 6, _glCopyTextureSubImage1D_args};

static const char * _glCopyTextureSubImage2D_args[8] = {"texture", "level", "xoffset", "yoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyTextureSubImage2D_sig = {1035, "glCopyTextureSubImage2D", 8, _glCopyTextureSubImage2D_args};

static const char * _glCopyTextureSubImage3D_args[9] = {"texture", "level", "xoffset", "yoffset", "zoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyTextureSubImage3D_sig = {1036, "glCopyTextureSubImage3D", 9, _glCopyTextureSubImage3D_args};

static const char * _glTextureParameterf_args[3] = {"texture", "pname", "param"};
static const trace::FunctionSig _glTextureParameterf_sig = {1037, "glTextureParameterf", 3, _glTextureParameterf_args};

static const char * _glTextureParameterfv_args[3] = {"texture", "pname", "params"};
static const trace::FunctionSig _glTextureParameterfv_sig = {1038, "glTextureParameterfv", 3, _glTextureParameterfv_args};

static const char * _glTextureParameteri_args[3] = {"texture", "pname", "param"};
static const trace::FunctionSig _glTextureParameteri_sig = {1039, "glTextureParameteri", 3, _glTextureParameteri_args};

static const char * _glTextureParameterIiv_args[3] = {"texture", "pname", "params"};
static const trace::FunctionSig _glTextureParameterIiv_sig = {1040, "glTextureParameterIiv", 3, _glTextureParameterIiv_args};

static const char * _glTextureParameterIuiv_args[3] = {"texture", "pname", "params"};
static const trace::FunctionSig _glTextureParameterIuiv_sig = {1041, "glTextureParameterIuiv", 3, _glTextureParameterIuiv_args};

static const char * _glTextureParameteriv_args[3] = {"texture", "pname", "params"};
static const trace::FunctionSig _glTextureParameteriv_sig = {1042, "glTextureParameteriv", 3, _glTextureParameteriv_args};

static const char * _glGenerateTextureMipmap_args[1] = {"texture"};
static const trace::FunctionSig _glGenerateTextureMipmap_sig = {1043, "glGenerateTextureMipmap", 1, _glGenerateTextureMipmap_args};

static const char * _glBindTextureUnit_args[2] = {"unit", "texture"};
static const trace::FunctionSig _glBindTextureUnit_sig = {1044, "glBindTextureUnit", 2, _glBindTextureUnit_args};

static const char * _glGetTextureImage_args[6] = {"texture", "level", "format", "type", "bufSize", "pixels"};
static const trace::FunctionSig _glGetTextureImage_sig = {1045, "glGetTextureImage", 6, _glGetTextureImage_args};

static const char * _glGetCompressedTextureImage_args[4] = {"texture", "level", "bufSize", "pixels"};
static const trace::FunctionSig _glGetCompressedTextureImage_sig = {1046, "glGetCompressedTextureImage", 4, _glGetCompressedTextureImage_args};

static const char * _glGetTextureLevelParameterfv_args[4] = {"texture", "level", "pname", "params"};
static const trace::FunctionSig _glGetTextureLevelParameterfv_sig = {1047, "glGetTextureLevelParameterfv", 4, _glGetTextureLevelParameterfv_args};

static const char * _glGetTextureLevelParameteriv_args[4] = {"texture", "level", "pname", "params"};
static const trace::FunctionSig _glGetTextureLevelParameteriv_sig = {1048, "glGetTextureLevelParameteriv", 4, _glGetTextureLevelParameteriv_args};

static const char * _glGetTextureParameterfv_args[3] = {"texture", "pname", "params"};
static const trace::FunctionSig _glGetTextureParameterfv_sig = {1049, "glGetTextureParameterfv", 3, _glGetTextureParameterfv_args};

static const char * _glGetTextureParameterIiv_args[3] = {"texture", "pname", "params"};
static const trace::FunctionSig _glGetTextureParameterIiv_sig = {1050, "glGetTextureParameterIiv", 3, _glGetTextureParameterIiv_args};

static const char * _glGetTextureParameterIuiv_args[3] = {"texture", "pname", "params"};
static const trace::FunctionSig _glGetTextureParameterIuiv_sig = {1051, "glGetTextureParameterIuiv", 3, _glGetTextureParameterIuiv_args};

static const char * _glGetTextureParameteriv_args[3] = {"texture", "pname", "params"};
static const trace::FunctionSig _glGetTextureParameteriv_sig = {1052, "glGetTextureParameteriv", 3, _glGetTextureParameteriv_args};

static const char * _glCreateVertexArrays_args[2] = {"n", "arrays"};
static const trace::FunctionSig _glCreateVertexArrays_sig = {1053, "glCreateVertexArrays", 2, _glCreateVertexArrays_args};

static const char * _glDisableVertexArrayAttrib_args[2] = {"vaobj", "index"};
static const trace::FunctionSig _glDisableVertexArrayAttrib_sig = {1054, "glDisableVertexArrayAttrib", 2, _glDisableVertexArrayAttrib_args};

static const char * _glEnableVertexArrayAttrib_args[2] = {"vaobj", "index"};
static const trace::FunctionSig _glEnableVertexArrayAttrib_sig = {1055, "glEnableVertexArrayAttrib", 2, _glEnableVertexArrayAttrib_args};

static const char * _glVertexArrayElementBuffer_args[2] = {"vaobj", "buffer"};
static const trace::FunctionSig _glVertexArrayElementBuffer_sig = {1056, "glVertexArrayElementBuffer", 2, _glVertexArrayElementBuffer_args};

static const char * _glVertexArrayVertexBuffer_args[5] = {"vaobj", "bindingindex", "buffer", "offset", "stride"};
static const trace::FunctionSig _glVertexArrayVertexBuffer_sig = {1057, "glVertexArrayVertexBuffer", 5, _glVertexArrayVertexBuffer_args};

static const char * _glVertexArrayVertexBuffers_args[6] = {"vaobj", "first", "count", "buffers", "offsets", "strides"};
static const trace::FunctionSig _glVertexArrayVertexBuffers_sig = {1058, "glVertexArrayVertexBuffers", 6, _glVertexArrayVertexBuffers_args};

static const char * _glVertexArrayAttribBinding_args[3] = {"vaobj", "attribindex", "bindingindex"};
static const trace::FunctionSig _glVertexArrayAttribBinding_sig = {1059, "glVertexArrayAttribBinding", 3, _glVertexArrayAttribBinding_args};

static const char * _glVertexArrayAttribFormat_args[6] = {"vaobj", "attribindex", "size", "type", "normalized", "relativeoffset"};
static const trace::FunctionSig _glVertexArrayAttribFormat_sig = {1060, "glVertexArrayAttribFormat", 6, _glVertexArrayAttribFormat_args};

static const char * _glVertexArrayAttribIFormat_args[5] = {"vaobj", "attribindex", "size", "type", "relativeoffset"};
static const trace::FunctionSig _glVertexArrayAttribIFormat_sig = {1061, "glVertexArrayAttribIFormat", 5, _glVertexArrayAttribIFormat_args};

static const char * _glVertexArrayAttribLFormat_args[5] = {"vaobj", "attribindex", "size", "type", "relativeoffset"};
static const trace::FunctionSig _glVertexArrayAttribLFormat_sig = {1062, "glVertexArrayAttribLFormat", 5, _glVertexArrayAttribLFormat_args};

static const char * _glVertexArrayBindingDivisor_args[3] = {"vaobj", "bindingindex", "divisor"};
static const trace::FunctionSig _glVertexArrayBindingDivisor_sig = {1063, "glVertexArrayBindingDivisor", 3, _glVertexArrayBindingDivisor_args};

static const char * _glGetVertexArrayiv_args[3] = {"vaobj", "pname", "param"};
static const trace::FunctionSig _glGetVertexArrayiv_sig = {1064, "glGetVertexArrayiv", 3, _glGetVertexArrayiv_args};

static const char * _glGetVertexArrayIndexediv_args[4] = {"vaobj", "index", "pname", "param"};
static const trace::FunctionSig _glGetVertexArrayIndexediv_sig = {1065, "glGetVertexArrayIndexediv", 4, _glGetVertexArrayIndexediv_args};

static const char * _glGetVertexArrayIndexed64iv_args[4] = {"vaobj", "index", "pname", "param"};
static const trace::FunctionSig _glGetVertexArrayIndexed64iv_sig = {1066, "glGetVertexArrayIndexed64iv", 4, _glGetVertexArrayIndexed64iv_args};

static const char * _glCreateSamplers_args[2] = {"n", "samplers"};
static const trace::FunctionSig _glCreateSamplers_sig = {1067, "glCreateSamplers", 2, _glCreateSamplers_args};

static const char * _glCreateProgramPipelines_args[2] = {"n", "pipelines"};
static const trace::FunctionSig _glCreateProgramPipelines_sig = {1068, "glCreateProgramPipelines", 2, _glCreateProgramPipelines_args};

static const char * _glCreateQueries_args[3] = {"target", "n", "ids"};
static const trace::FunctionSig _glCreateQueries_sig = {1069, "glCreateQueries", 3, _glCreateQueries_args};

static const char * _glGetQueryBufferObjecti64v_args[4] = {"id", "buffer", "pname", "offset"};
static const trace::FunctionSig _glGetQueryBufferObjecti64v_sig = {1070, "glGetQueryBufferObjecti64v", 4, _glGetQueryBufferObjecti64v_args};

static const char * _glGetQueryBufferObjectiv_args[4] = {"id", "buffer", "pname", "offset"};
static const trace::FunctionSig _glGetQueryBufferObjectiv_sig = {1071, "glGetQueryBufferObjectiv", 4, _glGetQueryBufferObjectiv_args};

static const char * _glGetQueryBufferObjectui64v_args[4] = {"id", "buffer", "pname", "offset"};
static const trace::FunctionSig _glGetQueryBufferObjectui64v_sig = {1072, "glGetQueryBufferObjectui64v", 4, _glGetQueryBufferObjectui64v_args};

static const char * _glGetQueryBufferObjectuiv_args[4] = {"id", "buffer", "pname", "offset"};
static const trace::FunctionSig _glGetQueryBufferObjectuiv_sig = {1073, "glGetQueryBufferObjectuiv", 4, _glGetQueryBufferObjectuiv_args};

static const char * _glDrawBuffersARB_args[2] = {"n", "bufs"};
static const trace::FunctionSig _glDrawBuffersARB_sig = {1074, "glDrawBuffersARB", 2, _glDrawBuffersARB_args};

static const char * _glBlendEquationiARB_args[2] = {"buf", "mode"};
static const trace::FunctionSig _glBlendEquationiARB_sig = {1075, "glBlendEquationiARB", 2, _glBlendEquationiARB_args};

static const char * _glBlendEquationSeparateiARB_args[3] = {"buf", "modeRGB", "modeAlpha"};
static const trace::FunctionSig _glBlendEquationSeparateiARB_sig = {1076, "glBlendEquationSeparateiARB", 3, _glBlendEquationSeparateiARB_args};

static const char * _glBlendFunciARB_args[3] = {"buf", "src", "dst"};
static const trace::FunctionSig _glBlendFunciARB_sig = {1077, "glBlendFunciARB", 3, _glBlendFunciARB_args};

static const char * _glBlendFuncSeparateiARB_args[5] = {"buf", "srcRGB", "dstRGB", "srcAlpha", "dstAlpha"};
static const trace::FunctionSig _glBlendFuncSeparateiARB_sig = {1078, "glBlendFuncSeparateiARB", 5, _glBlendFuncSeparateiARB_args};

static const char * _glDrawElementsBaseVertex_args[5] = {"mode", "count", "type", "indices", "basevertex"};
static const trace::FunctionSig _glDrawElementsBaseVertex_sig = {1079, "glDrawElementsBaseVertex", 5, _glDrawElementsBaseVertex_args};

static const char * _glDrawRangeElementsBaseVertex_args[7] = {"mode", "start", "end", "count", "type", "indices", "basevertex"};
static const trace::FunctionSig _glDrawRangeElementsBaseVertex_sig = {1080, "glDrawRangeElementsBaseVertex", 7, _glDrawRangeElementsBaseVertex_args};

static const char * _glDrawElementsInstancedBaseVertex_args[6] = {"mode", "count", "type", "indices", "instancecount", "basevertex"};
static const trace::FunctionSig _glDrawElementsInstancedBaseVertex_sig = {1081, "glDrawElementsInstancedBaseVertex", 6, _glDrawElementsInstancedBaseVertex_args};

static const char * _glMultiDrawElementsBaseVertex_args[6] = {"mode", "count", "type", "indices", "drawcount", "basevertex"};
static const trace::FunctionSig _glMultiDrawElementsBaseVertex_sig = {1082, "glMultiDrawElementsBaseVertex", 6, _glMultiDrawElementsBaseVertex_args};

static const char * _glDrawArraysIndirect_args[2] = {"mode", "indirect"};
static const trace::FunctionSig _glDrawArraysIndirect_sig = {1083, "glDrawArraysIndirect", 2, _glDrawArraysIndirect_args};

static const char * _glDrawElementsIndirect_args[3] = {"mode", "type", "indirect"};
static const trace::FunctionSig _glDrawElementsIndirect_sig = {1084, "glDrawElementsIndirect", 3, _glDrawElementsIndirect_args};

static const char * _glDrawArraysInstancedARB_args[4] = {"mode", "first", "count", "instancecount"};
static const trace::FunctionSig _glDrawArraysInstancedARB_sig = {1085, "glDrawArraysInstancedARB", 4, _glDrawArraysInstancedARB_args};

static const char * _glDrawElementsInstancedARB_args[5] = {"mode", "count", "type", "indices", "instancecount"};
static const trace::FunctionSig _glDrawElementsInstancedARB_sig = {1086, "glDrawElementsInstancedARB", 5, _glDrawElementsInstancedARB_args};

static const char * _glFramebufferParameteri_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glFramebufferParameteri_sig = {1087, "glFramebufferParameteri", 3, _glFramebufferParameteri_args};

static const char * _glGetFramebufferParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetFramebufferParameteriv_sig = {1088, "glGetFramebufferParameteriv", 3, _glGetFramebufferParameteriv_args};

static const char * _glIsRenderbuffer_args[1] = {"renderbuffer"};
static const trace::FunctionSig _glIsRenderbuffer_sig = {1089, "glIsRenderbuffer", 1, _glIsRenderbuffer_args};

static const char * _glBindRenderbuffer_args[2] = {"target", "renderbuffer"};
static const trace::FunctionSig _glBindRenderbuffer_sig = {1090, "glBindRenderbuffer", 2, _glBindRenderbuffer_args};

static const char * _glDeleteRenderbuffers_args[2] = {"n", "renderbuffers"};
static const trace::FunctionSig _glDeleteRenderbuffers_sig = {1091, "glDeleteRenderbuffers", 2, _glDeleteRenderbuffers_args};

static const char * _glGenRenderbuffers_args[2] = {"n", "renderbuffers"};
static const trace::FunctionSig _glGenRenderbuffers_sig = {1092, "glGenRenderbuffers", 2, _glGenRenderbuffers_args};

static const char * _glRenderbufferStorage_args[4] = {"target", "internalformat", "width", "height"};
static const trace::FunctionSig _glRenderbufferStorage_sig = {1093, "glRenderbufferStorage", 4, _glRenderbufferStorage_args};

static const char * _glGetRenderbufferParameteriv_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetRenderbufferParameteriv_sig = {1094, "glGetRenderbufferParameteriv", 3, _glGetRenderbufferParameteriv_args};

static const char * _glIsFramebuffer_args[1] = {"framebuffer"};
static const trace::FunctionSig _glIsFramebuffer_sig = {1095, "glIsFramebuffer", 1, _glIsFramebuffer_args};

static const char * _glBindFramebuffer_args[2] = {"target", "framebuffer"};
static const trace::FunctionSig _glBindFramebuffer_sig = {1096, "glBindFramebuffer", 2, _glBindFramebuffer_args};

static const char * _glDeleteFramebuffers_args[2] = {"n", "framebuffers"};
static const trace::FunctionSig _glDeleteFramebuffers_sig = {1097, "glDeleteFramebuffers", 2, _glDeleteFramebuffers_args};

static const char * _glGenFramebuffers_args[2] = {"n", "framebuffers"};
static const trace::FunctionSig _glGenFramebuffers_sig = {1098, "glGenFramebuffers", 2, _glGenFramebuffers_args};

static const char * _glCheckFramebufferStatus_args[1] = {"target"};
static const trace::FunctionSig _glCheckFramebufferStatus_sig = {1099, "glCheckFramebufferStatus", 1, _glCheckFramebufferStatus_args};

static const char * _glFramebufferTexture1D_args[5] = {"target", "attachment", "textarget", "texture", "level"};
static const trace::FunctionSig _glFramebufferTexture1D_sig = {1100, "glFramebufferTexture1D", 5, _glFramebufferTexture1D_args};

static const char * _glFramebufferTexture2D_args[5] = {"target", "attachment", "textarget", "texture", "level"};
static const trace::FunctionSig _glFramebufferTexture2D_sig = {1101, "glFramebufferTexture2D", 5, _glFramebufferTexture2D_args};

static const char * _glFramebufferTexture3D_args[6] = {"target", "attachment", "textarget", "texture", "level", "zoffset"};
static const trace::FunctionSig _glFramebufferTexture3D_sig = {1102, "glFramebufferTexture3D", 6, _glFramebufferTexture3D_args};

static const char * _glFramebufferRenderbuffer_args[4] = {"target", "attachment", "renderbuffertarget", "renderbuffer"};
static const trace::FunctionSig _glFramebufferRenderbuffer_sig = {1103, "glFramebufferRenderbuffer", 4, _glFramebufferRenderbuffer_args};

static const char * _glGetFramebufferAttachmentParameteriv_args[4] = {"target", "attachment", "pname", "params"};
static const trace::FunctionSig _glGetFramebufferAttachmentParameteriv_sig = {1104, "glGetFramebufferAttachmentParameteriv", 4, _glGetFramebufferAttachmentParameteriv_args};

static const char * _glGenerateMipmap_args[1] = {"target"};
static const trace::FunctionSig _glGenerateMipmap_sig = {1105, "glGenerateMipmap", 1, _glGenerateMipmap_args};

static const char * _glBlitFramebuffer_args[10] = {"srcX0", "srcY0", "srcX1", "srcY1", "dstX0", "dstY0", "dstX1", "dstY1", "mask", "filter"};
static const trace::FunctionSig _glBlitFramebuffer_sig = {1106, "glBlitFramebuffer", 10, _glBlitFramebuffer_args};

static const char * _glRenderbufferStorageMultisample_args[5] = {"target", "samples", "internalformat", "width", "height"};
static const trace::FunctionSig _glRenderbufferStorageMultisample_sig = {1107, "glRenderbufferStorageMultisample", 5, _glRenderbufferStorageMultisample_args};

static const char * _glFramebufferTextureLayer_args[5] = {"target", "attachment", "texture", "level", "layer"};
static const trace::FunctionSig _glFramebufferTextureLayer_sig = {1108, "glFramebufferTextureLayer", 5, _glFramebufferTextureLayer_args};

static const char * _glProgramParameteriARB_args[3] = {"program", "pname", "value"};
static const trace::FunctionSig _glProgramParameteriARB_sig = {1109, "glProgramParameteriARB", 3, _glProgramParameteriARB_args};

static const char * _glFramebufferTextureARB_args[4] = {"target", "attachment", "texture", "level"};
static const trace::FunctionSig _glFramebufferTextureARB_sig = {1110, "glFramebufferTextureARB", 4, _glFramebufferTextureARB_args};

static const char * _glFramebufferTextureLayerARB_args[5] = {"target", "attachment", "texture", "level", "layer"};
static const trace::FunctionSig _glFramebufferTextureLayerARB_sig = {1111, "glFramebufferTextureLayerARB", 5, _glFramebufferTextureLayerARB_args};

static const char * _glFramebufferTextureFaceARB_args[5] = {"target", "attachment", "texture", "level", "face"};
static const trace::FunctionSig _glFramebufferTextureFaceARB_sig = {1112, "glFramebufferTextureFaceARB", 5, _glFramebufferTextureFaceARB_args};

static const char * _glGetProgramBinary_args[5] = {"program", "bufSize", "length", "binaryFormat", "binary"};
static const trace::FunctionSig _glGetProgramBinary_sig = {1113, "glGetProgramBinary", 5, _glGetProgramBinary_args};

static const char * _glProgramBinary_args[4] = {"program", "binaryFormat", "binary", "length"};
static const trace::FunctionSig _glProgramBinary_sig = {1114, "glProgramBinary", 4, _glProgramBinary_args};

static const char * _glProgramParameteri_args[3] = {"program", "pname", "value"};
static const trace::FunctionSig _glProgramParameteri_sig = {1115, "glProgramParameteri", 3, _glProgramParameteri_args};

static const char * _glGetTextureSubImage_args[12] = {"texture", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "type", "bufSize", "pixels"};
static const trace::FunctionSig _glGetTextureSubImage_sig = {1116, "glGetTextureSubImage", 12, _glGetTextureSubImage_args};

static const char * _glGetCompressedTextureSubImage_args[10] = {"texture", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "bufSize", "pixels"};
static const trace::FunctionSig _glGetCompressedTextureSubImage_sig = {1117, "glGetCompressedTextureSubImage", 10, _glGetCompressedTextureSubImage_args};

static const char * _glSpecializeShaderARB_args[5] = {"shader", "pEntryPoint", "numSpecializationConstants", "pConstantIndex", "pConstantValue"};
static const trace::FunctionSig _glSpecializeShaderARB_sig = {1118, "glSpecializeShaderARB", 5, _glSpecializeShaderARB_args};

static const char * _glUniform1d_args[2] = {"location", "x"};
static const trace::FunctionSig _glUniform1d_sig = {1119, "glUniform1d", 2, _glUniform1d_args};

static const char * _glUniform2d_args[3] = {"location", "x", "y"};
static const trace::FunctionSig _glUniform2d_sig = {1120, "glUniform2d", 3, _glUniform2d_args};

static const char * _glUniform3d_args[4] = {"location", "x", "y", "z"};
static const trace::FunctionSig _glUniform3d_sig = {1121, "glUniform3d", 4, _glUniform3d_args};

static const char * _glUniform4d_args[5] = {"location", "x", "y", "z", "w"};
static const trace::FunctionSig _glUniform4d_sig = {1122, "glUniform4d", 5, _glUniform4d_args};

static const char * _glUniform1dv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1dv_sig = {1123, "glUniform1dv", 3, _glUniform1dv_args};

static const char * _glUniform2dv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2dv_sig = {1124, "glUniform2dv", 3, _glUniform2dv_args};

static const char * _glUniform3dv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3dv_sig = {1125, "glUniform3dv", 3, _glUniform3dv_args};

static const char * _glUniform4dv_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4dv_sig = {1126, "glUniform4dv", 3, _glUniform4dv_args};

static const char * _glUniformMatrix2dv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix2dv_sig = {1127, "glUniformMatrix2dv", 4, _glUniformMatrix2dv_args};

static const char * _glUniformMatrix3dv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix3dv_sig = {1128, "glUniformMatrix3dv", 4, _glUniformMatrix3dv_args};

static const char * _glUniformMatrix4dv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix4dv_sig = {1129, "glUniformMatrix4dv", 4, _glUniformMatrix4dv_args};

static const char * _glUniformMatrix2x3dv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix2x3dv_sig = {1130, "glUniformMatrix2x3dv", 4, _glUniformMatrix2x3dv_args};

static const char * _glUniformMatrix2x4dv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix2x4dv_sig = {1131, "glUniformMatrix2x4dv", 4, _glUniformMatrix2x4dv_args};

static const char * _glUniformMatrix3x2dv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix3x2dv_sig = {1132, "glUniformMatrix3x2dv", 4, _glUniformMatrix3x2dv_args};

static const char * _glUniformMatrix3x4dv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix3x4dv_sig = {1133, "glUniformMatrix3x4dv", 4, _glUniformMatrix3x4dv_args};

static const char * _glUniformMatrix4x2dv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix4x2dv_sig = {1134, "glUniformMatrix4x2dv", 4, _glUniformMatrix4x2dv_args};

static const char * _glUniformMatrix4x3dv_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix4x3dv_sig = {1135, "glUniformMatrix4x3dv", 4, _glUniformMatrix4x3dv_args};

static const char * _glGetUniformdv_args[3] = {"program", "location", "params"};
static const trace::FunctionSig _glGetUniformdv_sig = {1136, "glGetUniformdv", 3, _glGetUniformdv_args};

static const char * _glUniform1i64ARB_args[2] = {"location", "x"};
static const trace::FunctionSig _glUniform1i64ARB_sig = {1137, "glUniform1i64ARB", 2, _glUniform1i64ARB_args};

static const char * _glUniform2i64ARB_args[3] = {"location", "x", "y"};
static const trace::FunctionSig _glUniform2i64ARB_sig = {1138, "glUniform2i64ARB", 3, _glUniform2i64ARB_args};

static const char * _glUniform3i64ARB_args[4] = {"location", "x", "y", "z"};
static const trace::FunctionSig _glUniform3i64ARB_sig = {1139, "glUniform3i64ARB", 4, _glUniform3i64ARB_args};

static const char * _glUniform4i64ARB_args[5] = {"location", "x", "y", "z", "w"};
static const trace::FunctionSig _glUniform4i64ARB_sig = {1140, "glUniform4i64ARB", 5, _glUniform4i64ARB_args};

static const char * _glUniform1i64vARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1i64vARB_sig = {1141, "glUniform1i64vARB", 3, _glUniform1i64vARB_args};

static const char * _glUniform2i64vARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2i64vARB_sig = {1142, "glUniform2i64vARB", 3, _glUniform2i64vARB_args};

static const char * _glUniform3i64vARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3i64vARB_sig = {1143, "glUniform3i64vARB", 3, _glUniform3i64vARB_args};

static const char * _glUniform4i64vARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4i64vARB_sig = {1144, "glUniform4i64vARB", 3, _glUniform4i64vARB_args};

static const char * _glUniform1ui64ARB_args[2] = {"location", "x"};
static const trace::FunctionSig _glUniform1ui64ARB_sig = {1145, "glUniform1ui64ARB", 2, _glUniform1ui64ARB_args};

static const char * _glUniform2ui64ARB_args[3] = {"location", "x", "y"};
static const trace::FunctionSig _glUniform2ui64ARB_sig = {1146, "glUniform2ui64ARB", 3, _glUniform2ui64ARB_args};

static const char * _glUniform3ui64ARB_args[4] = {"location", "x", "y", "z"};
static const trace::FunctionSig _glUniform3ui64ARB_sig = {1147, "glUniform3ui64ARB", 4, _glUniform3ui64ARB_args};

static const char * _glUniform4ui64ARB_args[5] = {"location", "x", "y", "z", "w"};
static const trace::FunctionSig _glUniform4ui64ARB_sig = {1148, "glUniform4ui64ARB", 5, _glUniform4ui64ARB_args};

static const char * _glUniform1ui64vARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1ui64vARB_sig = {1149, "glUniform1ui64vARB", 3, _glUniform1ui64vARB_args};

static const char * _glUniform2ui64vARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2ui64vARB_sig = {1150, "glUniform2ui64vARB", 3, _glUniform2ui64vARB_args};

static const char * _glUniform3ui64vARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3ui64vARB_sig = {1151, "glUniform3ui64vARB", 3, _glUniform3ui64vARB_args};

static const char * _glUniform4ui64vARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4ui64vARB_sig = {1152, "glUniform4ui64vARB", 3, _glUniform4ui64vARB_args};

static const char * _glGetUniformi64vARB_args[3] = {"program", "location", "params"};
static const trace::FunctionSig _glGetUniformi64vARB_sig = {1153, "glGetUniformi64vARB", 3, _glGetUniformi64vARB_args};

static const char * _glGetUniformui64vARB_args[3] = {"program", "location", "params"};
static const trace::FunctionSig _glGetUniformui64vARB_sig = {1154, "glGetUniformui64vARB", 3, _glGetUniformui64vARB_args};

static const char * _glGetnUniformi64vARB_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformi64vARB_sig = {1155, "glGetnUniformi64vARB", 4, _glGetnUniformi64vARB_args};

static const char * _glGetnUniformui64vARB_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformui64vARB_sig = {1156, "glGetnUniformui64vARB", 4, _glGetnUniformui64vARB_args};

static const char * _glProgramUniform1i64ARB_args[3] = {"program", "location", "x"};
static const trace::FunctionSig _glProgramUniform1i64ARB_sig = {1157, "glProgramUniform1i64ARB", 3, _glProgramUniform1i64ARB_args};

static const char * _glProgramUniform2i64ARB_args[4] = {"program", "location", "x", "y"};
static const trace::FunctionSig _glProgramUniform2i64ARB_sig = {1158, "glProgramUniform2i64ARB", 4, _glProgramUniform2i64ARB_args};

static const char * _glProgramUniform3i64ARB_args[5] = {"program", "location", "x", "y", "z"};
static const trace::FunctionSig _glProgramUniform3i64ARB_sig = {1159, "glProgramUniform3i64ARB", 5, _glProgramUniform3i64ARB_args};

static const char * _glProgramUniform4i64ARB_args[6] = {"program", "location", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramUniform4i64ARB_sig = {1160, "glProgramUniform4i64ARB", 6, _glProgramUniform4i64ARB_args};

static const char * _glProgramUniform1i64vARB_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1i64vARB_sig = {1161, "glProgramUniform1i64vARB", 4, _glProgramUniform1i64vARB_args};

static const char * _glProgramUniform2i64vARB_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2i64vARB_sig = {1162, "glProgramUniform2i64vARB", 4, _glProgramUniform2i64vARB_args};

static const char * _glProgramUniform3i64vARB_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3i64vARB_sig = {1163, "glProgramUniform3i64vARB", 4, _glProgramUniform3i64vARB_args};

static const char * _glProgramUniform4i64vARB_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4i64vARB_sig = {1164, "glProgramUniform4i64vARB", 4, _glProgramUniform4i64vARB_args};

static const char * _glProgramUniform1ui64ARB_args[3] = {"program", "location", "x"};
static const trace::FunctionSig _glProgramUniform1ui64ARB_sig = {1165, "glProgramUniform1ui64ARB", 3, _glProgramUniform1ui64ARB_args};

static const char * _glProgramUniform2ui64ARB_args[4] = {"program", "location", "x", "y"};
static const trace::FunctionSig _glProgramUniform2ui64ARB_sig = {1166, "glProgramUniform2ui64ARB", 4, _glProgramUniform2ui64ARB_args};

static const char * _glProgramUniform3ui64ARB_args[5] = {"program", "location", "x", "y", "z"};
static const trace::FunctionSig _glProgramUniform3ui64ARB_sig = {1167, "glProgramUniform3ui64ARB", 5, _glProgramUniform3ui64ARB_args};

static const char * _glProgramUniform4ui64ARB_args[6] = {"program", "location", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramUniform4ui64ARB_sig = {1168, "glProgramUniform4ui64ARB", 6, _glProgramUniform4ui64ARB_args};

static const char * _glProgramUniform1ui64vARB_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1ui64vARB_sig = {1169, "glProgramUniform1ui64vARB", 4, _glProgramUniform1ui64vARB_args};

static const char * _glProgramUniform2ui64vARB_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2ui64vARB_sig = {1170, "glProgramUniform2ui64vARB", 4, _glProgramUniform2ui64vARB_args};

static const char * _glProgramUniform3ui64vARB_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3ui64vARB_sig = {1171, "glProgramUniform3ui64vARB", 4, _glProgramUniform3ui64vARB_args};

static const char * _glProgramUniform4ui64vARB_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4ui64vARB_sig = {1172, "glProgramUniform4ui64vARB", 4, _glProgramUniform4ui64vARB_args};

static const char * _glMultiDrawArraysIndirectCountARB_args[5] = {"mode", "indirect", "drawcount", "maxdrawcount", "stride"};
static const trace::FunctionSig _glMultiDrawArraysIndirectCountARB_sig = {1173, "glMultiDrawArraysIndirectCountARB", 5, _glMultiDrawArraysIndirectCountARB_args};

static const char * _glMultiDrawElementsIndirectCountARB_args[6] = {"mode", "type", "indirect", "drawcount", "maxdrawcount", "stride"};
static const trace::FunctionSig _glMultiDrawElementsIndirectCountARB_sig = {1174, "glMultiDrawElementsIndirectCountARB", 6, _glMultiDrawElementsIndirectCountARB_args};

static const char * _glVertexAttribDivisorARB_args[2] = {"index", "divisor"};
static const trace::FunctionSig _glVertexAttribDivisorARB_sig = {1175, "glVertexAttribDivisorARB", 2, _glVertexAttribDivisorARB_args};

static const char * _glGetInternalformativ_args[5] = {"target", "internalformat", "pname", "bufSize", "params"};
static const trace::FunctionSig _glGetInternalformativ_sig = {1176, "glGetInternalformativ", 5, _glGetInternalformativ_args};

static const char * _glGetInternalformati64v_args[5] = {"target", "internalformat", "pname", "bufSize", "params"};
static const trace::FunctionSig _glGetInternalformati64v_sig = {1177, "glGetInternalformati64v", 5, _glGetInternalformati64v_args};

static const char * _glInvalidateTexSubImage_args[8] = {"texture", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth"};
static const trace::FunctionSig _glInvalidateTexSubImage_sig = {1178, "glInvalidateTexSubImage", 8, _glInvalidateTexSubImage_args};

static const char * _glInvalidateTexImage_args[2] = {"texture", "level"};
static const trace::FunctionSig _glInvalidateTexImage_sig = {1179, "glInvalidateTexImage", 2, _glInvalidateTexImage_args};

static const char * _glInvalidateBufferSubData_args[3] = {"buffer", "offset", "length"};
static const trace::FunctionSig _glInvalidateBufferSubData_sig = {1180, "glInvalidateBufferSubData", 3, _glInvalidateBufferSubData_args};

static const char * _glInvalidateBufferData_args[1] = {"buffer"};
static const trace::FunctionSig _glInvalidateBufferData_sig = {1181, "glInvalidateBufferData", 1, _glInvalidateBufferData_args};

static const char * _glInvalidateFramebuffer_args[3] = {"target", "numAttachments", "attachments"};
static const trace::FunctionSig _glInvalidateFramebuffer_sig = {1182, "glInvalidateFramebuffer", 3, _glInvalidateFramebuffer_args};

static const char * _glInvalidateSubFramebuffer_args[7] = {"target", "numAttachments", "attachments", "x", "y", "width", "height"};
static const trace::FunctionSig _glInvalidateSubFramebuffer_sig = {1183, "glInvalidateSubFramebuffer", 7, _glInvalidateSubFramebuffer_args};

static const char * _glMapBufferRange_args[4] = {"target", "offset", "length", "access"};
static const trace::FunctionSig _glMapBufferRange_sig = {1184, "glMapBufferRange", 4, _glMapBufferRange_args};

static const char * _glFlushMappedBufferRange_args[3] = {"target", "offset", "length"};
static const trace::FunctionSig _glFlushMappedBufferRange_sig = {1185, "glFlushMappedBufferRange", 3, _glFlushMappedBufferRange_args};

static const char * _glCurrentPaletteMatrixARB_args[1] = {"index"};
static const trace::FunctionSig _glCurrentPaletteMatrixARB_sig = {1186, "glCurrentPaletteMatrixARB", 1, _glCurrentPaletteMatrixARB_args};

static const char * _glMatrixIndexubvARB_args[2] = {"size", "indices"};
static const trace::FunctionSig _glMatrixIndexubvARB_sig = {1187, "glMatrixIndexubvARB", 2, _glMatrixIndexubvARB_args};

static const char * _glMatrixIndexusvARB_args[2] = {"size", "indices"};
static const trace::FunctionSig _glMatrixIndexusvARB_sig = {1188, "glMatrixIndexusvARB", 2, _glMatrixIndexusvARB_args};

static const char * _glMatrixIndexuivARB_args[2] = {"size", "indices"};
static const trace::FunctionSig _glMatrixIndexuivARB_sig = {1189, "glMatrixIndexuivARB", 2, _glMatrixIndexuivARB_args};

static const char * _glMatrixIndexPointerARB_args[4] = {"size", "type", "stride", "pointer"};
static const trace::FunctionSig _glMatrixIndexPointerARB_sig = {1190, "glMatrixIndexPointerARB", 4, _glMatrixIndexPointerARB_args};

static const char * _glBindBuffersBase_args[4] = {"target", "first", "count", "buffers"};
static const trace::FunctionSig _glBindBuffersBase_sig = {1191, "glBindBuffersBase", 4, _glBindBuffersBase_args};

static const char * _glBindBuffersRange_args[6] = {"target", "first", "count", "buffers", "offsets", "sizes"};
static const trace::FunctionSig _glBindBuffersRange_sig = {1192, "glBindBuffersRange", 6, _glBindBuffersRange_args};

static const char * _glBindTextures_args[3] = {"first", "count", "textures"};
static const trace::FunctionSig _glBindTextures_sig = {1193, "glBindTextures", 3, _glBindTextures_args};

static const char * _glBindSamplers_args[3] = {"first", "count", "samplers"};
static const trace::FunctionSig _glBindSamplers_sig = {1194, "glBindSamplers", 3, _glBindSamplers_args};

static const char * _glBindImageTextures_args[3] = {"first", "count", "textures"};
static const trace::FunctionSig _glBindImageTextures_sig = {1195, "glBindImageTextures", 3, _glBindImageTextures_args};

static const char * _glBindVertexBuffers_args[5] = {"first", "count", "buffers", "offsets", "strides"};
static const trace::FunctionSig _glBindVertexBuffers_sig = {1196, "glBindVertexBuffers", 5, _glBindVertexBuffers_args};

static const char * _glMultiDrawArraysIndirect_args[4] = {"mode", "indirect", "drawcount", "stride"};
static const trace::FunctionSig _glMultiDrawArraysIndirect_sig = {1197, "glMultiDrawArraysIndirect", 4, _glMultiDrawArraysIndirect_args};

static const char * _glMultiDrawElementsIndirect_args[5] = {"mode", "type", "indirect", "drawcount", "stride"};
static const trace::FunctionSig _glMultiDrawElementsIndirect_sig = {1198, "glMultiDrawElementsIndirect", 5, _glMultiDrawElementsIndirect_args};

static const char * _glSampleCoverageARB_args[2] = {"value", "invert"};
static const trace::FunctionSig _glSampleCoverageARB_sig = {1199, "glSampleCoverageARB", 2, _glSampleCoverageARB_args};

static const char * _glActiveTextureARB_args[1] = {"texture"};
static const trace::FunctionSig _glActiveTextureARB_sig = {1200, "glActiveTextureARB", 1, _glActiveTextureARB_args};

static const char * _glClientActiveTextureARB_args[1] = {"texture"};
static const trace::FunctionSig _glClientActiveTextureARB_sig = {1201, "glClientActiveTextureARB", 1, _glClientActiveTextureARB_args};

static const char * _glMultiTexCoord1dARB_args[2] = {"target", "s"};
static const trace::FunctionSig _glMultiTexCoord1dARB_sig = {1202, "glMultiTexCoord1dARB", 2, _glMultiTexCoord1dARB_args};

static const char * _glMultiTexCoord1dvARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord1dvARB_sig = {1203, "glMultiTexCoord1dvARB", 2, _glMultiTexCoord1dvARB_args};

static const char * _glMultiTexCoord1fARB_args[2] = {"target", "s"};
static const trace::FunctionSig _glMultiTexCoord1fARB_sig = {1204, "glMultiTexCoord1fARB", 2, _glMultiTexCoord1fARB_args};

static const char * _glMultiTexCoord1fvARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord1fvARB_sig = {1205, "glMultiTexCoord1fvARB", 2, _glMultiTexCoord1fvARB_args};

static const char * _glMultiTexCoord1iARB_args[2] = {"target", "s"};
static const trace::FunctionSig _glMultiTexCoord1iARB_sig = {1206, "glMultiTexCoord1iARB", 2, _glMultiTexCoord1iARB_args};

static const char * _glMultiTexCoord1ivARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord1ivARB_sig = {1207, "glMultiTexCoord1ivARB", 2, _glMultiTexCoord1ivARB_args};

static const char * _glMultiTexCoord1sARB_args[2] = {"target", "s"};
static const trace::FunctionSig _glMultiTexCoord1sARB_sig = {1208, "glMultiTexCoord1sARB", 2, _glMultiTexCoord1sARB_args};

static const char * _glMultiTexCoord1svARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord1svARB_sig = {1209, "glMultiTexCoord1svARB", 2, _glMultiTexCoord1svARB_args};

static const char * _glMultiTexCoord2dARB_args[3] = {"target", "s", "t"};
static const trace::FunctionSig _glMultiTexCoord2dARB_sig = {1210, "glMultiTexCoord2dARB", 3, _glMultiTexCoord2dARB_args};

static const char * _glMultiTexCoord2dvARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord2dvARB_sig = {1211, "glMultiTexCoord2dvARB", 2, _glMultiTexCoord2dvARB_args};

static const char * _glMultiTexCoord2fARB_args[3] = {"target", "s", "t"};
static const trace::FunctionSig _glMultiTexCoord2fARB_sig = {1212, "glMultiTexCoord2fARB", 3, _glMultiTexCoord2fARB_args};

static const char * _glMultiTexCoord2fvARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord2fvARB_sig = {1213, "glMultiTexCoord2fvARB", 2, _glMultiTexCoord2fvARB_args};

static const char * _glMultiTexCoord2iARB_args[3] = {"target", "s", "t"};
static const trace::FunctionSig _glMultiTexCoord2iARB_sig = {1214, "glMultiTexCoord2iARB", 3, _glMultiTexCoord2iARB_args};

static const char * _glMultiTexCoord2ivARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord2ivARB_sig = {1215, "glMultiTexCoord2ivARB", 2, _glMultiTexCoord2ivARB_args};

static const char * _glMultiTexCoord2sARB_args[3] = {"target", "s", "t"};
static const trace::FunctionSig _glMultiTexCoord2sARB_sig = {1216, "glMultiTexCoord2sARB", 3, _glMultiTexCoord2sARB_args};

static const char * _glMultiTexCoord2svARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord2svARB_sig = {1217, "glMultiTexCoord2svARB", 2, _glMultiTexCoord2svARB_args};

static const char * _glMultiTexCoord3dARB_args[4] = {"target", "s", "t", "r"};
static const trace::FunctionSig _glMultiTexCoord3dARB_sig = {1218, "glMultiTexCoord3dARB", 4, _glMultiTexCoord3dARB_args};

static const char * _glMultiTexCoord3dvARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord3dvARB_sig = {1219, "glMultiTexCoord3dvARB", 2, _glMultiTexCoord3dvARB_args};

static const char * _glMultiTexCoord3fARB_args[4] = {"target", "s", "t", "r"};
static const trace::FunctionSig _glMultiTexCoord3fARB_sig = {1220, "glMultiTexCoord3fARB", 4, _glMultiTexCoord3fARB_args};

static const char * _glMultiTexCoord3fvARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord3fvARB_sig = {1221, "glMultiTexCoord3fvARB", 2, _glMultiTexCoord3fvARB_args};

static const char * _glMultiTexCoord3iARB_args[4] = {"target", "s", "t", "r"};
static const trace::FunctionSig _glMultiTexCoord3iARB_sig = {1222, "glMultiTexCoord3iARB", 4, _glMultiTexCoord3iARB_args};

static const char * _glMultiTexCoord3ivARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord3ivARB_sig = {1223, "glMultiTexCoord3ivARB", 2, _glMultiTexCoord3ivARB_args};

static const char * _glMultiTexCoord3sARB_args[4] = {"target", "s", "t", "r"};
static const trace::FunctionSig _glMultiTexCoord3sARB_sig = {1224, "glMultiTexCoord3sARB", 4, _glMultiTexCoord3sARB_args};

static const char * _glMultiTexCoord3svARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord3svARB_sig = {1225, "glMultiTexCoord3svARB", 2, _glMultiTexCoord3svARB_args};

static const char * _glMultiTexCoord4dARB_args[5] = {"target", "s", "t", "r", "q"};
static const trace::FunctionSig _glMultiTexCoord4dARB_sig = {1226, "glMultiTexCoord4dARB", 5, _glMultiTexCoord4dARB_args};

static const char * _glMultiTexCoord4dvARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord4dvARB_sig = {1227, "glMultiTexCoord4dvARB", 2, _glMultiTexCoord4dvARB_args};

static const char * _glMultiTexCoord4fARB_args[5] = {"target", "s", "t", "r", "q"};
static const trace::FunctionSig _glMultiTexCoord4fARB_sig = {1228, "glMultiTexCoord4fARB", 5, _glMultiTexCoord4fARB_args};

static const char * _glMultiTexCoord4fvARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord4fvARB_sig = {1229, "glMultiTexCoord4fvARB", 2, _glMultiTexCoord4fvARB_args};

static const char * _glMultiTexCoord4iARB_args[5] = {"target", "s", "t", "r", "q"};
static const trace::FunctionSig _glMultiTexCoord4iARB_sig = {1230, "glMultiTexCoord4iARB", 5, _glMultiTexCoord4iARB_args};

static const char * _glMultiTexCoord4ivARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord4ivARB_sig = {1231, "glMultiTexCoord4ivARB", 2, _glMultiTexCoord4ivARB_args};

static const char * _glMultiTexCoord4sARB_args[5] = {"target", "s", "t", "r", "q"};
static const trace::FunctionSig _glMultiTexCoord4sARB_sig = {1232, "glMultiTexCoord4sARB", 5, _glMultiTexCoord4sARB_args};

static const char * _glMultiTexCoord4svARB_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord4svARB_sig = {1233, "glMultiTexCoord4svARB", 2, _glMultiTexCoord4svARB_args};

static const char * _glGenQueriesARB_args[2] = {"n", "ids"};
static const trace::FunctionSig _glGenQueriesARB_sig = {1234, "glGenQueriesARB", 2, _glGenQueriesARB_args};

static const char * _glDeleteQueriesARB_args[2] = {"n", "ids"};
static const trace::FunctionSig _glDeleteQueriesARB_sig = {1235, "glDeleteQueriesARB", 2, _glDeleteQueriesARB_args};

static const char * _glIsQueryARB_args[1] = {"id"};
static const trace::FunctionSig _glIsQueryARB_sig = {1236, "glIsQueryARB", 1, _glIsQueryARB_args};

static const char * _glBeginQueryARB_args[2] = {"target", "id"};
static const trace::FunctionSig _glBeginQueryARB_sig = {1237, "glBeginQueryARB", 2, _glBeginQueryARB_args};

static const char * _glEndQueryARB_args[1] = {"target"};
static const trace::FunctionSig _glEndQueryARB_sig = {1238, "glEndQueryARB", 1, _glEndQueryARB_args};

static const char * _glGetQueryivARB_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetQueryivARB_sig = {1239, "glGetQueryivARB", 3, _glGetQueryivARB_args};

static const char * _glGetQueryObjectivARB_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectivARB_sig = {1240, "glGetQueryObjectivARB", 3, _glGetQueryObjectivARB_args};

static const char * _glGetQueryObjectuivARB_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectuivARB_sig = {1241, "glGetQueryObjectuivARB", 3, _glGetQueryObjectuivARB_args};

static const char * _glMaxShaderCompilerThreadsARB_args[1] = {"count"};
static const trace::FunctionSig _glMaxShaderCompilerThreadsARB_sig = {1242, "glMaxShaderCompilerThreadsARB", 1, _glMaxShaderCompilerThreadsARB_args};

static const char * _glPointParameterfARB_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPointParameterfARB_sig = {1243, "glPointParameterfARB", 2, _glPointParameterfARB_args};

static const char * _glPointParameterfvARB_args[2] = {"pname", "params"};
static const trace::FunctionSig _glPointParameterfvARB_sig = {1244, "glPointParameterfvARB", 2, _glPointParameterfvARB_args};

static const char * _glGetProgramInterfaceiv_args[4] = {"program", "programInterface", "pname", "params"};
static const trace::FunctionSig _glGetProgramInterfaceiv_sig = {1245, "glGetProgramInterfaceiv", 4, _glGetProgramInterfaceiv_args};

static const char * _glGetProgramResourceIndex_args[3] = {"program", "programInterface", "name"};
static const trace::FunctionSig _glGetProgramResourceIndex_sig = {1246, "glGetProgramResourceIndex", 3, _glGetProgramResourceIndex_args};

static const char * _glGetProgramResourceName_args[6] = {"program", "programInterface", "index", "bufSize", "length", "name"};
static const trace::FunctionSig _glGetProgramResourceName_sig = {1247, "glGetProgramResourceName", 6, _glGetProgramResourceName_args};

static const char * _glGetProgramResourceiv_args[8] = {"program", "programInterface", "index", "propCount", "props", "bufSize", "length", "params"};
static const trace::FunctionSig _glGetProgramResourceiv_sig = {1248, "glGetProgramResourceiv", 8, _glGetProgramResourceiv_args};

static const char * _glGetProgramResourceLocation_args[3] = {"program", "programInterface", "name"};
static const trace::FunctionSig _glGetProgramResourceLocation_sig = {1249, "glGetProgramResourceLocation", 3, _glGetProgramResourceLocation_args};

static const char * _glGetProgramResourceLocationIndex_args[3] = {"program", "programInterface", "name"};
static const trace::FunctionSig _glGetProgramResourceLocationIndex_sig = {1250, "glGetProgramResourceLocationIndex", 3, _glGetProgramResourceLocationIndex_args};

static const char * _glProvokingVertex_args[1] = {"mode"};
static const trace::FunctionSig _glProvokingVertex_sig = {1251, "glProvokingVertex", 1, _glProvokingVertex_args};

static const char ** _glGetGraphicsResetStatusARB_args = NULL;
static const trace::FunctionSig _glGetGraphicsResetStatusARB_sig = {1252, "glGetGraphicsResetStatusARB", 0, _glGetGraphicsResetStatusARB_args};

static const char * _glGetnMapdvARB_args[4] = {"target", "query", "bufSize", "v"};
static const trace::FunctionSig _glGetnMapdvARB_sig = {1253, "glGetnMapdvARB", 4, _glGetnMapdvARB_args};

static const char * _glGetnMapfvARB_args[4] = {"target", "query", "bufSize", "v"};
static const trace::FunctionSig _glGetnMapfvARB_sig = {1254, "glGetnMapfvARB", 4, _glGetnMapfvARB_args};

static const char * _glGetnMapivARB_args[4] = {"target", "query", "bufSize", "v"};
static const trace::FunctionSig _glGetnMapivARB_sig = {1255, "glGetnMapivARB", 4, _glGetnMapivARB_args};

static const char * _glGetnPixelMapfvARB_args[3] = {"map", "bufSize", "values"};
static const trace::FunctionSig _glGetnPixelMapfvARB_sig = {1256, "glGetnPixelMapfvARB", 3, _glGetnPixelMapfvARB_args};

static const char * _glGetnPixelMapuivARB_args[3] = {"map", "bufSize", "values"};
static const trace::FunctionSig _glGetnPixelMapuivARB_sig = {1257, "glGetnPixelMapuivARB", 3, _glGetnPixelMapuivARB_args};

static const char * _glGetnPixelMapusvARB_args[3] = {"map", "bufSize", "values"};
static const trace::FunctionSig _glGetnPixelMapusvARB_sig = {1258, "glGetnPixelMapusvARB", 3, _glGetnPixelMapusvARB_args};

static const char * _glGetnPolygonStippleARB_args[2] = {"bufSize", "pattern"};
static const trace::FunctionSig _glGetnPolygonStippleARB_sig = {1259, "glGetnPolygonStippleARB", 2, _glGetnPolygonStippleARB_args};

static const char * _glGetnColorTableARB_args[5] = {"target", "format", "type", "bufSize", "table"};
static const trace::FunctionSig _glGetnColorTableARB_sig = {1260, "glGetnColorTableARB", 5, _glGetnColorTableARB_args};

static const char * _glGetnConvolutionFilterARB_args[5] = {"target", "format", "type", "bufSize", "image"};
static const trace::FunctionSig _glGetnConvolutionFilterARB_sig = {1261, "glGetnConvolutionFilterARB", 5, _glGetnConvolutionFilterARB_args};

static const char * _glGetnSeparableFilterARB_args[8] = {"target", "format", "type", "rowBufSize", "row", "columnBufSize", "column", "span"};
static const trace::FunctionSig _glGetnSeparableFilterARB_sig = {1262, "glGetnSeparableFilterARB", 8, _glGetnSeparableFilterARB_args};

static const char * _glGetnHistogramARB_args[6] = {"target", "reset", "format", "type", "bufSize", "values"};
static const trace::FunctionSig _glGetnHistogramARB_sig = {1263, "glGetnHistogramARB", 6, _glGetnHistogramARB_args};

static const char * _glGetnMinmaxARB_args[6] = {"target", "reset", "format", "type", "bufSize", "values"};
static const trace::FunctionSig _glGetnMinmaxARB_sig = {1264, "glGetnMinmaxARB", 6, _glGetnMinmaxARB_args};

static const char * _glGetnTexImageARB_args[6] = {"target", "level", "format", "type", "bufSize", "img"};
static const trace::FunctionSig _glGetnTexImageARB_sig = {1265, "glGetnTexImageARB", 6, _glGetnTexImageARB_args};

static const char * _glReadnPixelsARB_args[8] = {"x", "y", "width", "height", "format", "type", "bufSize", "data"};
static const trace::FunctionSig _glReadnPixelsARB_sig = {1266, "glReadnPixelsARB", 8, _glReadnPixelsARB_args};

static const char * _glGetnCompressedTexImageARB_args[4] = {"target", "lod", "bufSize", "img"};
static const trace::FunctionSig _glGetnCompressedTexImageARB_sig = {1267, "glGetnCompressedTexImageARB", 4, _glGetnCompressedTexImageARB_args};

static const char * _glGetnUniformfvARB_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformfvARB_sig = {1268, "glGetnUniformfvARB", 4, _glGetnUniformfvARB_args};

static const char * _glGetnUniformivARB_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformivARB_sig = {1269, "glGetnUniformivARB", 4, _glGetnUniformivARB_args};

static const char * _glGetnUniformuivARB_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformuivARB_sig = {1270, "glGetnUniformuivARB", 4, _glGetnUniformuivARB_args};

static const char * _glGetnUniformdvARB_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformdvARB_sig = {1271, "glGetnUniformdvARB", 4, _glGetnUniformdvARB_args};

static const char * _glFramebufferSampleLocationsfvARB_args[4] = {"target", "start", "count", "v"};
static const trace::FunctionSig _glFramebufferSampleLocationsfvARB_sig = {1272, "glFramebufferSampleLocationsfvARB", 4, _glFramebufferSampleLocationsfvARB_args};

static const char * _glNamedFramebufferSampleLocationsfvARB_args[4] = {"framebuffer", "start", "count", "v"};
static const trace::FunctionSig _glNamedFramebufferSampleLocationsfvARB_sig = {1273, "glNamedFramebufferSampleLocationsfvARB", 4, _glNamedFramebufferSampleLocationsfvARB_args};

static const char ** _glEvaluateDepthValuesARB_args = NULL;
static const trace::FunctionSig _glEvaluateDepthValuesARB_sig = {1274, "glEvaluateDepthValuesARB", 0, _glEvaluateDepthValuesARB_args};

static const char * _glMinSampleShadingARB_args[1] = {"value"};
static const trace::FunctionSig _glMinSampleShadingARB_sig = {1275, "glMinSampleShadingARB", 1, _glMinSampleShadingARB_args};

static const char * _glGenSamplers_args[2] = {"count", "samplers"};
static const trace::FunctionSig _glGenSamplers_sig = {1276, "glGenSamplers", 2, _glGenSamplers_args};

static const char * _glDeleteSamplers_args[2] = {"count", "samplers"};
static const trace::FunctionSig _glDeleteSamplers_sig = {1277, "glDeleteSamplers", 2, _glDeleteSamplers_args};

static const char * _glIsSampler_args[1] = {"sampler"};
static const trace::FunctionSig _glIsSampler_sig = {1278, "glIsSampler", 1, _glIsSampler_args};

static const char * _glBindSampler_args[2] = {"unit", "sampler"};
static const trace::FunctionSig _glBindSampler_sig = {1279, "glBindSampler", 2, _glBindSampler_args};

static const char * _glSamplerParameteri_args[3] = {"sampler", "pname", "param"};
static const trace::FunctionSig _glSamplerParameteri_sig = {1280, "glSamplerParameteri", 3, _glSamplerParameteri_args};

static const char * _glSamplerParameteriv_args[3] = {"sampler", "pname", "param"};
static const trace::FunctionSig _glSamplerParameteriv_sig = {1281, "glSamplerParameteriv", 3, _glSamplerParameteriv_args};

static const char * _glSamplerParameterf_args[3] = {"sampler", "pname", "param"};
static const trace::FunctionSig _glSamplerParameterf_sig = {1282, "glSamplerParameterf", 3, _glSamplerParameterf_args};

static const char * _glSamplerParameterfv_args[3] = {"sampler", "pname", "param"};
static const trace::FunctionSig _glSamplerParameterfv_sig = {1283, "glSamplerParameterfv", 3, _glSamplerParameterfv_args};

static const char * _glSamplerParameterIiv_args[3] = {"sampler", "pname", "param"};
static const trace::FunctionSig _glSamplerParameterIiv_sig = {1284, "glSamplerParameterIiv", 3, _glSamplerParameterIiv_args};

static const char * _glSamplerParameterIuiv_args[3] = {"sampler", "pname", "param"};
static const trace::FunctionSig _glSamplerParameterIuiv_sig = {1285, "glSamplerParameterIuiv", 3, _glSamplerParameterIuiv_args};

static const char * _glGetSamplerParameteriv_args[3] = {"sampler", "pname", "params"};
static const trace::FunctionSig _glGetSamplerParameteriv_sig = {1286, "glGetSamplerParameteriv", 3, _glGetSamplerParameteriv_args};

static const char * _glGetSamplerParameterIiv_args[3] = {"sampler", "pname", "params"};
static const trace::FunctionSig _glGetSamplerParameterIiv_sig = {1287, "glGetSamplerParameterIiv", 3, _glGetSamplerParameterIiv_args};

static const char * _glGetSamplerParameterfv_args[3] = {"sampler", "pname", "params"};
static const trace::FunctionSig _glGetSamplerParameterfv_sig = {1288, "glGetSamplerParameterfv", 3, _glGetSamplerParameterfv_args};

static const char * _glGetSamplerParameterIuiv_args[3] = {"sampler", "pname", "params"};
static const trace::FunctionSig _glGetSamplerParameterIuiv_sig = {1289, "glGetSamplerParameterIuiv", 3, _glGetSamplerParameterIuiv_args};

static const char * _glUseProgramStages_args[3] = {"pipeline", "stages", "program"};
static const trace::FunctionSig _glUseProgramStages_sig = {1290, "glUseProgramStages", 3, _glUseProgramStages_args};

static const char * _glActiveShaderProgram_args[2] = {"pipeline", "program"};
static const trace::FunctionSig _glActiveShaderProgram_sig = {1291, "glActiveShaderProgram", 2, _glActiveShaderProgram_args};

static const char * _glCreateShaderProgramv_args[3] = {"type", "count", "strings"};
static const trace::FunctionSig _glCreateShaderProgramv_sig = {1292, "glCreateShaderProgramv", 3, _glCreateShaderProgramv_args};

static const char * _glBindProgramPipeline_args[1] = {"pipeline"};
static const trace::FunctionSig _glBindProgramPipeline_sig = {1293, "glBindProgramPipeline", 1, _glBindProgramPipeline_args};

static const char * _glDeleteProgramPipelines_args[2] = {"n", "pipelines"};
static const trace::FunctionSig _glDeleteProgramPipelines_sig = {1294, "glDeleteProgramPipelines", 2, _glDeleteProgramPipelines_args};

static const char * _glGenProgramPipelines_args[2] = {"n", "pipelines"};
static const trace::FunctionSig _glGenProgramPipelines_sig = {1295, "glGenProgramPipelines", 2, _glGenProgramPipelines_args};

static const char * _glIsProgramPipeline_args[1] = {"pipeline"};
static const trace::FunctionSig _glIsProgramPipeline_sig = {1296, "glIsProgramPipeline", 1, _glIsProgramPipeline_args};

static const char * _glGetProgramPipelineiv_args[3] = {"pipeline", "pname", "params"};
static const trace::FunctionSig _glGetProgramPipelineiv_sig = {1297, "glGetProgramPipelineiv", 3, _glGetProgramPipelineiv_args};

static const char * _glProgramUniform1i_args[3] = {"program", "location", "v0"};
static const trace::FunctionSig _glProgramUniform1i_sig = {1298, "glProgramUniform1i", 3, _glProgramUniform1i_args};

static const char * _glProgramUniform1iv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1iv_sig = {1299, "glProgramUniform1iv", 4, _glProgramUniform1iv_args};

static const char * _glProgramUniform1f_args[3] = {"program", "location", "v0"};
static const trace::FunctionSig _glProgramUniform1f_sig = {1300, "glProgramUniform1f", 3, _glProgramUniform1f_args};

static const char * _glProgramUniform1fv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1fv_sig = {1301, "glProgramUniform1fv", 4, _glProgramUniform1fv_args};

static const char * _glProgramUniform1d_args[3] = {"program", "location", "v0"};
static const trace::FunctionSig _glProgramUniform1d_sig = {1302, "glProgramUniform1d", 3, _glProgramUniform1d_args};

static const char * _glProgramUniform1dv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1dv_sig = {1303, "glProgramUniform1dv", 4, _glProgramUniform1dv_args};

static const char * _glProgramUniform1ui_args[3] = {"program", "location", "v0"};
static const trace::FunctionSig _glProgramUniform1ui_sig = {1304, "glProgramUniform1ui", 3, _glProgramUniform1ui_args};

static const char * _glProgramUniform1uiv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1uiv_sig = {1305, "glProgramUniform1uiv", 4, _glProgramUniform1uiv_args};

static const char * _glProgramUniform2i_args[4] = {"program", "location", "v0", "v1"};
static const trace::FunctionSig _glProgramUniform2i_sig = {1306, "glProgramUniform2i", 4, _glProgramUniform2i_args};

static const char * _glProgramUniform2iv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2iv_sig = {1307, "glProgramUniform2iv", 4, _glProgramUniform2iv_args};

static const char * _glProgramUniform2f_args[4] = {"program", "location", "v0", "v1"};
static const trace::FunctionSig _glProgramUniform2f_sig = {1308, "glProgramUniform2f", 4, _glProgramUniform2f_args};

static const char * _glProgramUniform2fv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2fv_sig = {1309, "glProgramUniform2fv", 4, _glProgramUniform2fv_args};

static const char * _glProgramUniform2d_args[4] = {"program", "location", "v0", "v1"};
static const trace::FunctionSig _glProgramUniform2d_sig = {1310, "glProgramUniform2d", 4, _glProgramUniform2d_args};

static const char * _glProgramUniform2dv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2dv_sig = {1311, "glProgramUniform2dv", 4, _glProgramUniform2dv_args};

static const char * _glProgramUniform2ui_args[4] = {"program", "location", "v0", "v1"};
static const trace::FunctionSig _glProgramUniform2ui_sig = {1312, "glProgramUniform2ui", 4, _glProgramUniform2ui_args};

static const char * _glProgramUniform2uiv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2uiv_sig = {1313, "glProgramUniform2uiv", 4, _glProgramUniform2uiv_args};

static const char * _glProgramUniform3i_args[5] = {"program", "location", "v0", "v1", "v2"};
static const trace::FunctionSig _glProgramUniform3i_sig = {1314, "glProgramUniform3i", 5, _glProgramUniform3i_args};

static const char * _glProgramUniform3iv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3iv_sig = {1315, "glProgramUniform3iv", 4, _glProgramUniform3iv_args};

static const char * _glProgramUniform3f_args[5] = {"program", "location", "v0", "v1", "v2"};
static const trace::FunctionSig _glProgramUniform3f_sig = {1316, "glProgramUniform3f", 5, _glProgramUniform3f_args};

static const char * _glProgramUniform3fv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3fv_sig = {1317, "glProgramUniform3fv", 4, _glProgramUniform3fv_args};

static const char * _glProgramUniform3d_args[5] = {"program", "location", "v0", "v1", "v2"};
static const trace::FunctionSig _glProgramUniform3d_sig = {1318, "glProgramUniform3d", 5, _glProgramUniform3d_args};

static const char * _glProgramUniform3dv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3dv_sig = {1319, "glProgramUniform3dv", 4, _glProgramUniform3dv_args};

static const char * _glProgramUniform3ui_args[5] = {"program", "location", "v0", "v1", "v2"};
static const trace::FunctionSig _glProgramUniform3ui_sig = {1320, "glProgramUniform3ui", 5, _glProgramUniform3ui_args};

static const char * _glProgramUniform3uiv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3uiv_sig = {1321, "glProgramUniform3uiv", 4, _glProgramUniform3uiv_args};

static const char * _glProgramUniform4i_args[6] = {"program", "location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glProgramUniform4i_sig = {1322, "glProgramUniform4i", 6, _glProgramUniform4i_args};

static const char * _glProgramUniform4iv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4iv_sig = {1323, "glProgramUniform4iv", 4, _glProgramUniform4iv_args};

static const char * _glProgramUniform4f_args[6] = {"program", "location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glProgramUniform4f_sig = {1324, "glProgramUniform4f", 6, _glProgramUniform4f_args};

static const char * _glProgramUniform4fv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4fv_sig = {1325, "glProgramUniform4fv", 4, _glProgramUniform4fv_args};

static const char * _glProgramUniform4d_args[6] = {"program", "location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glProgramUniform4d_sig = {1326, "glProgramUniform4d", 6, _glProgramUniform4d_args};

static const char * _glProgramUniform4dv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4dv_sig = {1327, "glProgramUniform4dv", 4, _glProgramUniform4dv_args};

static const char * _glProgramUniform4ui_args[6] = {"program", "location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glProgramUniform4ui_sig = {1328, "glProgramUniform4ui", 6, _glProgramUniform4ui_args};

static const char * _glProgramUniform4uiv_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4uiv_sig = {1329, "glProgramUniform4uiv", 4, _glProgramUniform4uiv_args};

static const char * _glProgramUniformMatrix2fv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2fv_sig = {1330, "glProgramUniformMatrix2fv", 5, _glProgramUniformMatrix2fv_args};

static const char * _glProgramUniformMatrix3fv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3fv_sig = {1331, "glProgramUniformMatrix3fv", 5, _glProgramUniformMatrix3fv_args};

static const char * _glProgramUniformMatrix4fv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4fv_sig = {1332, "glProgramUniformMatrix4fv", 5, _glProgramUniformMatrix4fv_args};

static const char * _glProgramUniformMatrix2dv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2dv_sig = {1333, "glProgramUniformMatrix2dv", 5, _glProgramUniformMatrix2dv_args};

static const char * _glProgramUniformMatrix3dv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3dv_sig = {1334, "glProgramUniformMatrix3dv", 5, _glProgramUniformMatrix3dv_args};

static const char * _glProgramUniformMatrix4dv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4dv_sig = {1335, "glProgramUniformMatrix4dv", 5, _glProgramUniformMatrix4dv_args};

static const char * _glProgramUniformMatrix2x3fv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2x3fv_sig = {1336, "glProgramUniformMatrix2x3fv", 5, _glProgramUniformMatrix2x3fv_args};

static const char * _glProgramUniformMatrix3x2fv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3x2fv_sig = {1337, "glProgramUniformMatrix3x2fv", 5, _glProgramUniformMatrix3x2fv_args};

static const char * _glProgramUniformMatrix2x4fv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2x4fv_sig = {1338, "glProgramUniformMatrix2x4fv", 5, _glProgramUniformMatrix2x4fv_args};

static const char * _glProgramUniformMatrix4x2fv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4x2fv_sig = {1339, "glProgramUniformMatrix4x2fv", 5, _glProgramUniformMatrix4x2fv_args};

static const char * _glProgramUniformMatrix3x4fv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3x4fv_sig = {1340, "glProgramUniformMatrix3x4fv", 5, _glProgramUniformMatrix3x4fv_args};

static const char * _glProgramUniformMatrix4x3fv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4x3fv_sig = {1341, "glProgramUniformMatrix4x3fv", 5, _glProgramUniformMatrix4x3fv_args};

static const char * _glProgramUniformMatrix2x3dv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2x3dv_sig = {1342, "glProgramUniformMatrix2x3dv", 5, _glProgramUniformMatrix2x3dv_args};

static const char * _glProgramUniformMatrix3x2dv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3x2dv_sig = {1343, "glProgramUniformMatrix3x2dv", 5, _glProgramUniformMatrix3x2dv_args};

static const char * _glProgramUniformMatrix2x4dv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2x4dv_sig = {1344, "glProgramUniformMatrix2x4dv", 5, _glProgramUniformMatrix2x4dv_args};

static const char * _glProgramUniformMatrix4x2dv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4x2dv_sig = {1345, "glProgramUniformMatrix4x2dv", 5, _glProgramUniformMatrix4x2dv_args};

static const char * _glProgramUniformMatrix3x4dv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3x4dv_sig = {1346, "glProgramUniformMatrix3x4dv", 5, _glProgramUniformMatrix3x4dv_args};

static const char * _glProgramUniformMatrix4x3dv_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4x3dv_sig = {1347, "glProgramUniformMatrix4x3dv", 5, _glProgramUniformMatrix4x3dv_args};

static const char * _glValidateProgramPipeline_args[1] = {"pipeline"};
static const trace::FunctionSig _glValidateProgramPipeline_sig = {1348, "glValidateProgramPipeline", 1, _glValidateProgramPipeline_args};

static const char * _glGetProgramPipelineInfoLog_args[4] = {"pipeline", "bufSize", "length", "infoLog"};
static const trace::FunctionSig _glGetProgramPipelineInfoLog_sig = {1349, "glGetProgramPipelineInfoLog", 4, _glGetProgramPipelineInfoLog_args};

static const char * _glGetActiveAtomicCounterBufferiv_args[4] = {"program", "bufferIndex", "pname", "params"};
static const trace::FunctionSig _glGetActiveAtomicCounterBufferiv_sig = {1350, "glGetActiveAtomicCounterBufferiv", 4, _glGetActiveAtomicCounterBufferiv_args};

static const char * _glBindImageTexture_args[7] = {"unit", "texture", "level", "layered", "layer", "access", "format"};
static const trace::FunctionSig _glBindImageTexture_sig = {1351, "glBindImageTexture", 7, _glBindImageTexture_args};

static const char * _glMemoryBarrier_args[1] = {"barriers"};
static const trace::FunctionSig _glMemoryBarrier_sig = {1352, "glMemoryBarrier", 1, _glMemoryBarrier_args};

static const char * _glDeleteObjectARB_args[1] = {"obj"};
static const trace::FunctionSig _glDeleteObjectARB_sig = {1353, "glDeleteObjectARB", 1, _glDeleteObjectARB_args};

static const char * _glGetHandleARB_args[1] = {"pname"};
static const trace::FunctionSig _glGetHandleARB_sig = {1354, "glGetHandleARB", 1, _glGetHandleARB_args};

static const char * _glDetachObjectARB_args[2] = {"containerObj", "attachedObj"};
static const trace::FunctionSig _glDetachObjectARB_sig = {1355, "glDetachObjectARB", 2, _glDetachObjectARB_args};

static const char * _glCreateShaderObjectARB_args[1] = {"shaderType"};
static const trace::FunctionSig _glCreateShaderObjectARB_sig = {1356, "glCreateShaderObjectARB", 1, _glCreateShaderObjectARB_args};

static const char * _glShaderSourceARB_args[4] = {"shaderObj", "count", "string", "length"};
static const trace::FunctionSig _glShaderSourceARB_sig = {1357, "glShaderSourceARB", 4, _glShaderSourceARB_args};

static const char * _glCompileShaderARB_args[1] = {"shaderObj"};
static const trace::FunctionSig _glCompileShaderARB_sig = {1358, "glCompileShaderARB", 1, _glCompileShaderARB_args};

static const char ** _glCreateProgramObjectARB_args = NULL;
static const trace::FunctionSig _glCreateProgramObjectARB_sig = {1359, "glCreateProgramObjectARB", 0, _glCreateProgramObjectARB_args};

static const char * _glAttachObjectARB_args[2] = {"containerObj", "obj"};
static const trace::FunctionSig _glAttachObjectARB_sig = {1360, "glAttachObjectARB", 2, _glAttachObjectARB_args};

static const char * _glLinkProgramARB_args[1] = {"programObj"};
static const trace::FunctionSig _glLinkProgramARB_sig = {1361, "glLinkProgramARB", 1, _glLinkProgramARB_args};

static const char * _glUseProgramObjectARB_args[1] = {"programObj"};
static const trace::FunctionSig _glUseProgramObjectARB_sig = {1362, "glUseProgramObjectARB", 1, _glUseProgramObjectARB_args};

static const char * _glValidateProgramARB_args[1] = {"programObj"};
static const trace::FunctionSig _glValidateProgramARB_sig = {1363, "glValidateProgramARB", 1, _glValidateProgramARB_args};

static const char * _glUniform1fARB_args[2] = {"location", "v0"};
static const trace::FunctionSig _glUniform1fARB_sig = {1364, "glUniform1fARB", 2, _glUniform1fARB_args};

static const char * _glUniform2fARB_args[3] = {"location", "v0", "v1"};
static const trace::FunctionSig _glUniform2fARB_sig = {1365, "glUniform2fARB", 3, _glUniform2fARB_args};

static const char * _glUniform3fARB_args[4] = {"location", "v0", "v1", "v2"};
static const trace::FunctionSig _glUniform3fARB_sig = {1366, "glUniform3fARB", 4, _glUniform3fARB_args};

static const char * _glUniform4fARB_args[5] = {"location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glUniform4fARB_sig = {1367, "glUniform4fARB", 5, _glUniform4fARB_args};

static const char * _glUniform1iARB_args[2] = {"location", "v0"};
static const trace::FunctionSig _glUniform1iARB_sig = {1368, "glUniform1iARB", 2, _glUniform1iARB_args};

static const char * _glUniform2iARB_args[3] = {"location", "v0", "v1"};
static const trace::FunctionSig _glUniform2iARB_sig = {1369, "glUniform2iARB", 3, _glUniform2iARB_args};

static const char * _glUniform3iARB_args[4] = {"location", "v0", "v1", "v2"};
static const trace::FunctionSig _glUniform3iARB_sig = {1370, "glUniform3iARB", 4, _glUniform3iARB_args};

static const char * _glUniform4iARB_args[5] = {"location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glUniform4iARB_sig = {1371, "glUniform4iARB", 5, _glUniform4iARB_args};

static const char * _glUniform1fvARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1fvARB_sig = {1372, "glUniform1fvARB", 3, _glUniform1fvARB_args};

static const char * _glUniform2fvARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2fvARB_sig = {1373, "glUniform2fvARB", 3, _glUniform2fvARB_args};

static const char * _glUniform3fvARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3fvARB_sig = {1374, "glUniform3fvARB", 3, _glUniform3fvARB_args};

static const char * _glUniform4fvARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4fvARB_sig = {1375, "glUniform4fvARB", 3, _glUniform4fvARB_args};

static const char * _glUniform1ivARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1ivARB_sig = {1376, "glUniform1ivARB", 3, _glUniform1ivARB_args};

static const char * _glUniform2ivARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2ivARB_sig = {1377, "glUniform2ivARB", 3, _glUniform2ivARB_args};

static const char * _glUniform3ivARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3ivARB_sig = {1378, "glUniform3ivARB", 3, _glUniform3ivARB_args};

static const char * _glUniform4ivARB_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4ivARB_sig = {1379, "glUniform4ivARB", 3, _glUniform4ivARB_args};

static const char * _glUniformMatrix2fvARB_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix2fvARB_sig = {1380, "glUniformMatrix2fvARB", 4, _glUniformMatrix2fvARB_args};

static const char * _glUniformMatrix3fvARB_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix3fvARB_sig = {1381, "glUniformMatrix3fvARB", 4, _glUniformMatrix3fvARB_args};

static const char * _glUniformMatrix4fvARB_args[4] = {"location", "count", "transpose", "value"};
static const trace::FunctionSig _glUniformMatrix4fvARB_sig = {1382, "glUniformMatrix4fvARB", 4, _glUniformMatrix4fvARB_args};

static const char * _glGetObjectParameterfvARB_args[3] = {"obj", "pname", "params"};
static const trace::FunctionSig _glGetObjectParameterfvARB_sig = {1383, "glGetObjectParameterfvARB", 3, _glGetObjectParameterfvARB_args};

static const char * _glGetObjectParameterivARB_args[3] = {"obj", "pname", "params"};
static const trace::FunctionSig _glGetObjectParameterivARB_sig = {1384, "glGetObjectParameterivARB", 3, _glGetObjectParameterivARB_args};

static const char * _glGetInfoLogARB_args[4] = {"obj", "maxLength", "length", "infoLog"};
static const trace::FunctionSig _glGetInfoLogARB_sig = {1385, "glGetInfoLogARB", 4, _glGetInfoLogARB_args};

static const char * _glGetAttachedObjectsARB_args[4] = {"containerObj", "maxCount", "count", "obj"};
static const trace::FunctionSig _glGetAttachedObjectsARB_sig = {1386, "glGetAttachedObjectsARB", 4, _glGetAttachedObjectsARB_args};

static const char * _glGetUniformLocationARB_args[2] = {"programObj", "name"};
static const trace::FunctionSig _glGetUniformLocationARB_sig = {1387, "glGetUniformLocationARB", 2, _glGetUniformLocationARB_args};

static const char * _glGetActiveUniformARB_args[7] = {"programObj", "index", "bufSize", "length", "size", "type", "name"};
static const trace::FunctionSig _glGetActiveUniformARB_sig = {1388, "glGetActiveUniformARB", 7, _glGetActiveUniformARB_args};

static const char * _glGetUniformfvARB_args[3] = {"programObj", "location", "params"};
static const trace::FunctionSig _glGetUniformfvARB_sig = {1389, "glGetUniformfvARB", 3, _glGetUniformfvARB_args};

static const char * _glGetUniformivARB_args[3] = {"programObj", "location", "params"};
static const trace::FunctionSig _glGetUniformivARB_sig = {1390, "glGetUniformivARB", 3, _glGetUniformivARB_args};

static const char * _glGetShaderSourceARB_args[4] = {"obj", "bufSize", "length", "source"};
static const trace::FunctionSig _glGetShaderSourceARB_sig = {1391, "glGetShaderSourceARB", 4, _glGetShaderSourceARB_args};

static const char * _glShaderStorageBlockBinding_args[3] = {"program", "storageBlockIndex", "storageBlockBinding"};
static const trace::FunctionSig _glShaderStorageBlockBinding_sig = {1392, "glShaderStorageBlockBinding", 3, _glShaderStorageBlockBinding_args};

static const char * _glGetSubroutineUniformLocation_args[3] = {"program", "shadertype", "name"};
static const trace::FunctionSig _glGetSubroutineUniformLocation_sig = {1393, "glGetSubroutineUniformLocation", 3, _glGetSubroutineUniformLocation_args};

static const char * _glGetSubroutineIndex_args[3] = {"program", "shadertype", "name"};
static const trace::FunctionSig _glGetSubroutineIndex_sig = {1394, "glGetSubroutineIndex", 3, _glGetSubroutineIndex_args};

static const char * _glGetActiveSubroutineUniformiv_args[5] = {"program", "shadertype", "index", "pname", "values"};
static const trace::FunctionSig _glGetActiveSubroutineUniformiv_sig = {1395, "glGetActiveSubroutineUniformiv", 5, _glGetActiveSubroutineUniformiv_args};

static const char * _glGetActiveSubroutineUniformName_args[6] = {"program", "shadertype", "index", "bufsize", "length", "name"};
static const trace::FunctionSig _glGetActiveSubroutineUniformName_sig = {1396, "glGetActiveSubroutineUniformName", 6, _glGetActiveSubroutineUniformName_args};

static const char * _glGetActiveSubroutineName_args[6] = {"program", "shadertype", "index", "bufsize", "length", "name"};
static const trace::FunctionSig _glGetActiveSubroutineName_sig = {1397, "glGetActiveSubroutineName", 6, _glGetActiveSubroutineName_args};

static const char * _glUniformSubroutinesuiv_args[3] = {"shadertype", "count", "indices"};
static const trace::FunctionSig _glUniformSubroutinesuiv_sig = {1398, "glUniformSubroutinesuiv", 3, _glUniformSubroutinesuiv_args};

static const char * _glGetUniformSubroutineuiv_args[3] = {"shadertype", "location", "params"};
static const trace::FunctionSig _glGetUniformSubroutineuiv_sig = {1399, "glGetUniformSubroutineuiv", 3, _glGetUniformSubroutineuiv_args};

static const char * _glGetProgramStageiv_args[4] = {"program", "shadertype", "pname", "values"};
static const trace::FunctionSig _glGetProgramStageiv_sig = {1400, "glGetProgramStageiv", 4, _glGetProgramStageiv_args};

static const char * _glNamedStringARB_args[5] = {"type", "namelen", "name", "stringlen", "string"};
static const trace::FunctionSig _glNamedStringARB_sig = {1401, "glNamedStringARB", 5, _glNamedStringARB_args};

static const char * _glDeleteNamedStringARB_args[2] = {"namelen", "name"};
static const trace::FunctionSig _glDeleteNamedStringARB_sig = {1402, "glDeleteNamedStringARB", 2, _glDeleteNamedStringARB_args};

static const char * _glCompileShaderIncludeARB_args[4] = {"shader", "count", "path", "length"};
static const trace::FunctionSig _glCompileShaderIncludeARB_sig = {1403, "glCompileShaderIncludeARB", 4, _glCompileShaderIncludeARB_args};

static const char * _glIsNamedStringARB_args[2] = {"namelen", "name"};
static const trace::FunctionSig _glIsNamedStringARB_sig = {1404, "glIsNamedStringARB", 2, _glIsNamedStringARB_args};

static const char * _glGetNamedStringARB_args[5] = {"namelen", "name", "bufSize", "stringlen", "string"};
static const trace::FunctionSig _glGetNamedStringARB_sig = {1405, "glGetNamedStringARB", 5, _glGetNamedStringARB_args};

static const char * _glGetNamedStringivARB_args[4] = {"namelen", "name", "pname", "params"};
static const trace::FunctionSig _glGetNamedStringivARB_sig = {1406, "glGetNamedStringivARB", 4, _glGetNamedStringivARB_args};

static const char * _glBufferPageCommitmentARB_args[4] = {"target", "offset", "size", "commit"};
static const trace::FunctionSig _glBufferPageCommitmentARB_sig = {1407, "glBufferPageCommitmentARB", 4, _glBufferPageCommitmentARB_args};

static const char * _glNamedBufferPageCommitmentEXT_args[4] = {"buffer", "offset", "size", "commit"};
static const trace::FunctionSig _glNamedBufferPageCommitmentEXT_sig = {1408, "glNamedBufferPageCommitmentEXT", 4, _glNamedBufferPageCommitmentEXT_args};

static const char * _glNamedBufferPageCommitmentARB_args[4] = {"buffer", "offset", "size", "commit"};
static const trace::FunctionSig _glNamedBufferPageCommitmentARB_sig = {1409, "glNamedBufferPageCommitmentARB", 4, _glNamedBufferPageCommitmentARB_args};

static const char * _glTexPageCommitmentARB_args[9] = {"target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "resident"};
static const trace::FunctionSig _glTexPageCommitmentARB_sig = {1410, "glTexPageCommitmentARB", 9, _glTexPageCommitmentARB_args};

static const char * _glFenceSync_args[2] = {"condition", "flags"};
static const trace::FunctionSig _glFenceSync_sig = {1411, "glFenceSync", 2, _glFenceSync_args};

static const char * _glIsSync_args[1] = {"sync"};
static const trace::FunctionSig _glIsSync_sig = {1412, "glIsSync", 1, _glIsSync_args};

static const char * _glDeleteSync_args[1] = {"sync"};
static const trace::FunctionSig _glDeleteSync_sig = {1413, "glDeleteSync", 1, _glDeleteSync_args};

static const char * _glClientWaitSync_args[3] = {"sync", "flags", "timeout"};
static const trace::FunctionSig _glClientWaitSync_sig = {1414, "glClientWaitSync", 3, _glClientWaitSync_args};

static const char * _glWaitSync_args[3] = {"sync", "flags", "timeout"};
static const trace::FunctionSig _glWaitSync_sig = {1415, "glWaitSync", 3, _glWaitSync_args};

static const char * _glGetInteger64v_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetInteger64v_sig = {1416, "glGetInteger64v", 2, _glGetInteger64v_args};

static const char * _glGetSynciv_args[5] = {"sync", "pname", "bufSize", "length", "values"};
static const trace::FunctionSig _glGetSynciv_sig = {1417, "glGetSynciv", 5, _glGetSynciv_args};

static const char * _glPatchParameteri_args[2] = {"pname", "value"};
static const trace::FunctionSig _glPatchParameteri_sig = {1418, "glPatchParameteri", 2, _glPatchParameteri_args};

static const char * _glPatchParameterfv_args[2] = {"pname", "values"};
static const trace::FunctionSig _glPatchParameterfv_sig = {1419, "glPatchParameterfv", 2, _glPatchParameterfv_args};

static const char ** _glTextureBarrier_args = NULL;
static const trace::FunctionSig _glTextureBarrier_sig = {1420, "glTextureBarrier", 0, _glTextureBarrier_args};

static const char * _glTexBufferARB_args[3] = {"target", "internalformat", "buffer"};
static const trace::FunctionSig _glTexBufferARB_sig = {1421, "glTexBufferARB", 3, _glTexBufferARB_args};

static const char * _glTexBufferRange_args[5] = {"target", "internalformat", "buffer", "offset", "size"};
static const trace::FunctionSig _glTexBufferRange_sig = {1422, "glTexBufferRange", 5, _glTexBufferRange_args};

static const char * _glCompressedTexImage3DARB_args[9] = {"target", "level", "internalformat", "width", "height", "depth", "border", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexImage3DARB_sig = {1423, "glCompressedTexImage3DARB", 9, _glCompressedTexImage3DARB_args};

static const char * _glCompressedTexImage2DARB_args[8] = {"target", "level", "internalformat", "width", "height", "border", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexImage2DARB_sig = {1424, "glCompressedTexImage2DARB", 8, _glCompressedTexImage2DARB_args};

static const char * _glCompressedTexImage1DARB_args[7] = {"target", "level", "internalformat", "width", "border", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexImage1DARB_sig = {1425, "glCompressedTexImage1DARB", 7, _glCompressedTexImage1DARB_args};

static const char * _glCompressedTexSubImage3DARB_args[11] = {"target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexSubImage3DARB_sig = {1426, "glCompressedTexSubImage3DARB", 11, _glCompressedTexSubImage3DARB_args};

static const char * _glCompressedTexSubImage2DARB_args[9] = {"target", "level", "xoffset", "yoffset", "width", "height", "format", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexSubImage2DARB_sig = {1427, "glCompressedTexSubImage2DARB", 9, _glCompressedTexSubImage2DARB_args};

static const char * _glCompressedTexSubImage1DARB_args[7] = {"target", "level", "xoffset", "width", "format", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexSubImage1DARB_sig = {1428, "glCompressedTexSubImage1DARB", 7, _glCompressedTexSubImage1DARB_args};

static const char * _glGetCompressedTexImageARB_args[3] = {"target", "level", "img"};
static const trace::FunctionSig _glGetCompressedTexImageARB_sig = {1429, "glGetCompressedTexImageARB", 3, _glGetCompressedTexImageARB_args};

static const char * _glTexImage2DMultisample_args[6] = {"target", "samples", "internalformat", "width", "height", "fixedsamplelocations"};
static const trace::FunctionSig _glTexImage2DMultisample_sig = {1430, "glTexImage2DMultisample", 6, _glTexImage2DMultisample_args};

static const char * _glTexImage3DMultisample_args[7] = {"target", "samples", "internalformat", "width", "height", "depth", "fixedsamplelocations"};
static const trace::FunctionSig _glTexImage3DMultisample_sig = {1431, "glTexImage3DMultisample", 7, _glTexImage3DMultisample_args};

static const char * _glGetMultisamplefv_args[3] = {"pname", "index", "val"};
static const trace::FunctionSig _glGetMultisamplefv_sig = {1432, "glGetMultisamplefv", 3, _glGetMultisamplefv_args};

static const char * _glSampleMaski_args[2] = {"index", "mask"};
static const trace::FunctionSig _glSampleMaski_sig = {1433, "glSampleMaski", 2, _glSampleMaski_args};

static const char * _glTexStorage1D_args[4] = {"target", "levels", "internalformat", "width"};
static const trace::FunctionSig _glTexStorage1D_sig = {1434, "glTexStorage1D", 4, _glTexStorage1D_args};

static const char * _glTexStorage2D_args[5] = {"target", "levels", "internalformat", "width", "height"};
static const trace::FunctionSig _glTexStorage2D_sig = {1435, "glTexStorage2D", 5, _glTexStorage2D_args};

static const char * _glTexStorage3D_args[6] = {"target", "levels", "internalformat", "width", "height", "depth"};
static const trace::FunctionSig _glTexStorage3D_sig = {1436, "glTexStorage3D", 6, _glTexStorage3D_args};

static const char * _glTexStorageAttribs2DEXT_args[6] = {"target", "levels", "internalformat", "width", "height", "attrib_list"};
static const trace::FunctionSig _glTexStorageAttribs2DEXT_sig = {1437, "glTexStorageAttribs2DEXT", 6, _glTexStorageAttribs2DEXT_args};

static const char * _glTexStorageAttribs3DEXT_args[7] = {"target", "levels", "internalformat", "width", "height", "depth", "attrib_list"};
static const trace::FunctionSig _glTexStorageAttribs3DEXT_sig = {1438, "glTexStorageAttribs3DEXT", 7, _glTexStorageAttribs3DEXT_args};

static const char * _glTexStorage2DMultisample_args[6] = {"target", "samples", "internalformat", "width", "height", "fixedsamplelocations"};
static const trace::FunctionSig _glTexStorage2DMultisample_sig = {1439, "glTexStorage2DMultisample", 6, _glTexStorage2DMultisample_args};

static const char * _glTexStorage3DMultisample_args[7] = {"target", "samples", "internalformat", "width", "height", "depth", "fixedsamplelocations"};
static const trace::FunctionSig _glTexStorage3DMultisample_sig = {1440, "glTexStorage3DMultisample", 7, _glTexStorage3DMultisample_args};

static const char * _glEGLImageTargetTexStorageEXT_args[3] = {"target", "image", "attrib_list"};
static const trace::FunctionSig _glEGLImageTargetTexStorageEXT_sig = {1441, "glEGLImageTargetTexStorageEXT", 3, _glEGLImageTargetTexStorageEXT_args};

static const char * _glEGLImageTargetTextureStorageEXT_args[3] = {"texture", "image", "attrib_list"};
static const trace::FunctionSig _glEGLImageTargetTextureStorageEXT_sig = {1442, "glEGLImageTargetTextureStorageEXT", 3, _glEGLImageTargetTextureStorageEXT_args};

static const char * _glTextureView_args[8] = {"texture", "target", "origtexture", "internalformat", "minlevel", "numlevels", "minlayer", "numlayers"};
static const trace::FunctionSig _glTextureView_sig = {1443, "glTextureView", 8, _glTextureView_args};

static const char * _glQueryCounter_args[2] = {"id", "target"};
static const trace::FunctionSig _glQueryCounter_sig = {1444, "glQueryCounter", 2, _glQueryCounter_args};

static const char * _glGetQueryObjecti64v_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjecti64v_sig = {1445, "glGetQueryObjecti64v", 3, _glGetQueryObjecti64v_args};

static const char * _glGetQueryObjectui64v_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectui64v_sig = {1446, "glGetQueryObjectui64v", 3, _glGetQueryObjectui64v_args};

static const char * _glBindTransformFeedback_args[2] = {"target", "id"};
static const trace::FunctionSig _glBindTransformFeedback_sig = {1447, "glBindTransformFeedback", 2, _glBindTransformFeedback_args};

static const char * _glDeleteTransformFeedbacks_args[2] = {"n", "ids"};
static const trace::FunctionSig _glDeleteTransformFeedbacks_sig = {1448, "glDeleteTransformFeedbacks", 2, _glDeleteTransformFeedbacks_args};

static const char * _glGenTransformFeedbacks_args[2] = {"n", "ids"};
static const trace::FunctionSig _glGenTransformFeedbacks_sig = {1449, "glGenTransformFeedbacks", 2, _glGenTransformFeedbacks_args};

static const char * _glIsTransformFeedback_args[1] = {"id"};
static const trace::FunctionSig _glIsTransformFeedback_sig = {1450, "glIsTransformFeedback", 1, _glIsTransformFeedback_args};

static const char ** _glPauseTransformFeedback_args = NULL;
static const trace::FunctionSig _glPauseTransformFeedback_sig = {1451, "glPauseTransformFeedback", 0, _glPauseTransformFeedback_args};

static const char ** _glResumeTransformFeedback_args = NULL;
static const trace::FunctionSig _glResumeTransformFeedback_sig = {1452, "glResumeTransformFeedback", 0, _glResumeTransformFeedback_args};

static const char * _glDrawTransformFeedback_args[2] = {"mode", "id"};
static const trace::FunctionSig _glDrawTransformFeedback_sig = {1453, "glDrawTransformFeedback", 2, _glDrawTransformFeedback_args};

static const char * _glDrawTransformFeedbackStream_args[3] = {"mode", "id", "stream"};
static const trace::FunctionSig _glDrawTransformFeedbackStream_sig = {1454, "glDrawTransformFeedbackStream", 3, _glDrawTransformFeedbackStream_args};

static const char * _glBeginQueryIndexed_args[3] = {"target", "index", "id"};
static const trace::FunctionSig _glBeginQueryIndexed_sig = {1455, "glBeginQueryIndexed", 3, _glBeginQueryIndexed_args};

static const char * _glEndQueryIndexed_args[2] = {"target", "index"};
static const trace::FunctionSig _glEndQueryIndexed_sig = {1456, "glEndQueryIndexed", 2, _glEndQueryIndexed_args};

static const char * _glGetQueryIndexediv_args[4] = {"target", "index", "pname", "params"};
static const trace::FunctionSig _glGetQueryIndexediv_sig = {1457, "glGetQueryIndexediv", 4, _glGetQueryIndexediv_args};

static const char * _glDrawTransformFeedbackInstanced_args[3] = {"mode", "id", "instancecount"};
static const trace::FunctionSig _glDrawTransformFeedbackInstanced_sig = {1458, "glDrawTransformFeedbackInstanced", 3, _glDrawTransformFeedbackInstanced_args};

static const char * _glDrawTransformFeedbackStreamInstanced_args[4] = {"mode", "id", "stream", "instancecount"};
static const trace::FunctionSig _glDrawTransformFeedbackStreamInstanced_sig = {1459, "glDrawTransformFeedbackStreamInstanced", 4, _glDrawTransformFeedbackStreamInstanced_args};

static const char * _glLoadTransposeMatrixfARB_args[1] = {"m"};
static const trace::FunctionSig _glLoadTransposeMatrixfARB_sig = {1460, "glLoadTransposeMatrixfARB", 1, _glLoadTransposeMatrixfARB_args};

static const char * _glLoadTransposeMatrixdARB_args[1] = {"m"};
static const trace::FunctionSig _glLoadTransposeMatrixdARB_sig = {1461, "glLoadTransposeMatrixdARB", 1, _glLoadTransposeMatrixdARB_args};

static const char * _glMultTransposeMatrixfARB_args[1] = {"m"};
static const trace::FunctionSig _glMultTransposeMatrixfARB_sig = {1462, "glMultTransposeMatrixfARB", 1, _glMultTransposeMatrixfARB_args};

static const char * _glMultTransposeMatrixdARB_args[1] = {"m"};
static const trace::FunctionSig _glMultTransposeMatrixdARB_sig = {1463, "glMultTransposeMatrixdARB", 1, _glMultTransposeMatrixdARB_args};

static const char * _glGetUniformIndices_args[4] = {"program", "uniformCount", "uniformNames", "uniformIndices"};
static const trace::FunctionSig _glGetUniformIndices_sig = {1464, "glGetUniformIndices", 4, _glGetUniformIndices_args};

static const char * _glGetActiveUniformsiv_args[5] = {"program", "uniformCount", "uniformIndices", "pname", "params"};
static const trace::FunctionSig _glGetActiveUniformsiv_sig = {1465, "glGetActiveUniformsiv", 5, _glGetActiveUniformsiv_args};

static const char * _glGetActiveUniformName_args[5] = {"program", "uniformIndex", "bufSize", "length", "uniformName"};
static const trace::FunctionSig _glGetActiveUniformName_sig = {1466, "glGetActiveUniformName", 5, _glGetActiveUniformName_args};

static const char * _glGetUniformBlockIndex_args[2] = {"program", "uniformBlockName"};
static const trace::FunctionSig _glGetUniformBlockIndex_sig = {1467, "glGetUniformBlockIndex", 2, _glGetUniformBlockIndex_args};

static const char * _glGetActiveUniformBlockiv_args[4] = {"program", "uniformBlockIndex", "pname", "params"};
static const trace::FunctionSig _glGetActiveUniformBlockiv_sig = {1468, "glGetActiveUniformBlockiv", 4, _glGetActiveUniformBlockiv_args};

static const char * _glGetActiveUniformBlockName_args[5] = {"program", "uniformBlockIndex", "bufSize", "length", "uniformBlockName"};
static const trace::FunctionSig _glGetActiveUniformBlockName_sig = {1469, "glGetActiveUniformBlockName", 5, _glGetActiveUniformBlockName_args};

static const char * _glUniformBlockBinding_args[3] = {"program", "uniformBlockIndex", "uniformBlockBinding"};
static const trace::FunctionSig _glUniformBlockBinding_sig = {1470, "glUniformBlockBinding", 3, _glUniformBlockBinding_args};

static const char * _glBindVertexArray_args[1] = {"array"};
static const trace::FunctionSig _glBindVertexArray_sig = {1471, "glBindVertexArray", 1, _glBindVertexArray_args};

static const char * _glDeleteVertexArrays_args[2] = {"n", "arrays"};
static const trace::FunctionSig _glDeleteVertexArrays_sig = {1472, "glDeleteVertexArrays", 2, _glDeleteVertexArrays_args};

static const char * _glGenVertexArrays_args[2] = {"n", "arrays"};
static const trace::FunctionSig _glGenVertexArrays_sig = {1473, "glGenVertexArrays", 2, _glGenVertexArrays_args};

static const char * _glIsVertexArray_args[1] = {"array"};
static const trace::FunctionSig _glIsVertexArray_sig = {1474, "glIsVertexArray", 1, _glIsVertexArray_args};

static const char * _glVertexAttribL1d_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttribL1d_sig = {1475, "glVertexAttribL1d", 2, _glVertexAttribL1d_args};

static const char * _glVertexAttribL2d_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttribL2d_sig = {1476, "glVertexAttribL2d", 3, _glVertexAttribL2d_args};

static const char * _glVertexAttribL3d_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttribL3d_sig = {1477, "glVertexAttribL3d", 4, _glVertexAttribL3d_args};

static const char * _glVertexAttribL4d_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttribL4d_sig = {1478, "glVertexAttribL4d", 5, _glVertexAttribL4d_args};

static const char * _glVertexAttribL1dv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL1dv_sig = {1479, "glVertexAttribL1dv", 2, _glVertexAttribL1dv_args};

static const char * _glVertexAttribL2dv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL2dv_sig = {1480, "glVertexAttribL2dv", 2, _glVertexAttribL2dv_args};

static const char * _glVertexAttribL3dv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL3dv_sig = {1481, "glVertexAttribL3dv", 2, _glVertexAttribL3dv_args};

static const char * _glVertexAttribL4dv_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL4dv_sig = {1482, "glVertexAttribL4dv", 2, _glVertexAttribL4dv_args};

static const char * _glVertexAttribLPointer_args[5] = {"index", "size", "type", "stride", "pointer"};
static const trace::FunctionSig _glVertexAttribLPointer_sig = {1483, "glVertexAttribLPointer", 5, _glVertexAttribLPointer_args};

static const char * _glGetVertexAttribLdv_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribLdv_sig = {1484, "glGetVertexAttribLdv", 3, _glGetVertexAttribLdv_args};

static const char * _glBindVertexBuffer_args[4] = {"bindingindex", "buffer", "offset", "stride"};
static const trace::FunctionSig _glBindVertexBuffer_sig = {1485, "glBindVertexBuffer", 4, _glBindVertexBuffer_args};

static const char * _glVertexAttribFormat_args[5] = {"attribindex", "size", "type", "normalized", "relativeoffset"};
static const trace::FunctionSig _glVertexAttribFormat_sig = {1486, "glVertexAttribFormat", 5, _glVertexAttribFormat_args};

static const char * _glVertexAttribIFormat_args[4] = {"attribindex", "size", "type", "relativeoffset"};
static const trace::FunctionSig _glVertexAttribIFormat_sig = {1487, "glVertexAttribIFormat", 4, _glVertexAttribIFormat_args};

static const char * _glVertexAttribLFormat_args[4] = {"attribindex", "size", "type", "relativeoffset"};
static const trace::FunctionSig _glVertexAttribLFormat_sig = {1488, "glVertexAttribLFormat", 4, _glVertexAttribLFormat_args};

static const char * _glVertexAttribBinding_args[2] = {"attribindex", "bindingindex"};
static const trace::FunctionSig _glVertexAttribBinding_sig = {1489, "glVertexAttribBinding", 2, _glVertexAttribBinding_args};

static const char * _glVertexBindingDivisor_args[2] = {"bindingindex", "divisor"};
static const trace::FunctionSig _glVertexBindingDivisor_sig = {1490, "glVertexBindingDivisor", 2, _glVertexBindingDivisor_args};

static const char * _glWeightbvARB_args[2] = {"size", "weights"};
static const trace::FunctionSig _glWeightbvARB_sig = {1491, "glWeightbvARB", 2, _glWeightbvARB_args};

static const char * _glWeightsvARB_args[2] = {"size", "weights"};
static const trace::FunctionSig _glWeightsvARB_sig = {1492, "glWeightsvARB", 2, _glWeightsvARB_args};

static const char * _glWeightivARB_args[2] = {"size", "weights"};
static const trace::FunctionSig _glWeightivARB_sig = {1493, "glWeightivARB", 2, _glWeightivARB_args};

static const char * _glWeightfvARB_args[2] = {"size", "weights"};
static const trace::FunctionSig _glWeightfvARB_sig = {1494, "glWeightfvARB", 2, _glWeightfvARB_args};

static const char * _glWeightdvARB_args[2] = {"size", "weights"};
static const trace::FunctionSig _glWeightdvARB_sig = {1495, "glWeightdvARB", 2, _glWeightdvARB_args};

static const char * _glWeightubvARB_args[2] = {"size", "weights"};
static const trace::FunctionSig _glWeightubvARB_sig = {1496, "glWeightubvARB", 2, _glWeightubvARB_args};

static const char * _glWeightusvARB_args[2] = {"size", "weights"};
static const trace::FunctionSig _glWeightusvARB_sig = {1497, "glWeightusvARB", 2, _glWeightusvARB_args};

static const char * _glWeightuivARB_args[2] = {"size", "weights"};
static const trace::FunctionSig _glWeightuivARB_sig = {1498, "glWeightuivARB", 2, _glWeightuivARB_args};

static const char * _glWeightPointerARB_args[4] = {"size", "type", "stride", "pointer"};
static const trace::FunctionSig _glWeightPointerARB_sig = {1499, "glWeightPointerARB", 4, _glWeightPointerARB_args};

static const char * _glVertexBlendARB_args[1] = {"count"};
static const trace::FunctionSig _glVertexBlendARB_sig = {1500, "glVertexBlendARB", 1, _glVertexBlendARB_args};

static const char * _glBindBufferARB_args[2] = {"target", "buffer"};
static const trace::FunctionSig _glBindBufferARB_sig = {1501, "glBindBufferARB", 2, _glBindBufferARB_args};

static const char * _glDeleteBuffersARB_args[2] = {"n", "buffers"};
static const trace::FunctionSig _glDeleteBuffersARB_sig = {1502, "glDeleteBuffersARB", 2, _glDeleteBuffersARB_args};

static const char * _glGenBuffersARB_args[2] = {"n", "buffers"};
static const trace::FunctionSig _glGenBuffersARB_sig = {1503, "glGenBuffersARB", 2, _glGenBuffersARB_args};

static const char * _glIsBufferARB_args[1] = {"buffer"};
static const trace::FunctionSig _glIsBufferARB_sig = {1504, "glIsBufferARB", 1, _glIsBufferARB_args};

static const char * _glBufferDataARB_args[4] = {"target", "size", "data", "usage"};
static const trace::FunctionSig _glBufferDataARB_sig = {1505, "glBufferDataARB", 4, _glBufferDataARB_args};

static const char * _glBufferSubDataARB_args[4] = {"target", "offset", "size", "data"};
static const trace::FunctionSig _glBufferSubDataARB_sig = {1506, "glBufferSubDataARB", 4, _glBufferSubDataARB_args};

static const char * _glGetBufferSubDataARB_args[4] = {"target", "offset", "size", "data"};
static const trace::FunctionSig _glGetBufferSubDataARB_sig = {1507, "glGetBufferSubDataARB", 4, _glGetBufferSubDataARB_args};

static const char * _glMapBufferARB_args[2] = {"target", "access"};
static const trace::FunctionSig _glMapBufferARB_sig = {1508, "glMapBufferARB", 2, _glMapBufferARB_args};

static const char * _glUnmapBufferARB_args[1] = {"target"};
static const trace::FunctionSig _glUnmapBufferARB_sig = {1509, "glUnmapBufferARB", 1, _glUnmapBufferARB_args};

static const char * _glGetBufferParameterivARB_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetBufferParameterivARB_sig = {1510, "glGetBufferParameterivARB", 3, _glGetBufferParameterivARB_args};

static const char * _glGetBufferPointervARB_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetBufferPointervARB_sig = {1511, "glGetBufferPointervARB", 3, _glGetBufferPointervARB_args};

static const char * _glVertexAttrib1dARB_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttrib1dARB_sig = {1512, "glVertexAttrib1dARB", 2, _glVertexAttrib1dARB_args};

static const char * _glVertexAttrib1dvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib1dvARB_sig = {1513, "glVertexAttrib1dvARB", 2, _glVertexAttrib1dvARB_args};

static const char * _glVertexAttrib1fARB_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttrib1fARB_sig = {1514, "glVertexAttrib1fARB", 2, _glVertexAttrib1fARB_args};

static const char * _glVertexAttrib1fvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib1fvARB_sig = {1515, "glVertexAttrib1fvARB", 2, _glVertexAttrib1fvARB_args};

static const char * _glVertexAttrib1sARB_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttrib1sARB_sig = {1516, "glVertexAttrib1sARB", 2, _glVertexAttrib1sARB_args};

static const char * _glVertexAttrib1svARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib1svARB_sig = {1517, "glVertexAttrib1svARB", 2, _glVertexAttrib1svARB_args};

static const char * _glVertexAttrib2dARB_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttrib2dARB_sig = {1518, "glVertexAttrib2dARB", 3, _glVertexAttrib2dARB_args};

static const char * _glVertexAttrib2dvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib2dvARB_sig = {1519, "glVertexAttrib2dvARB", 2, _glVertexAttrib2dvARB_args};

static const char * _glVertexAttrib2fARB_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttrib2fARB_sig = {1520, "glVertexAttrib2fARB", 3, _glVertexAttrib2fARB_args};

static const char * _glVertexAttrib2fvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib2fvARB_sig = {1521, "glVertexAttrib2fvARB", 2, _glVertexAttrib2fvARB_args};

static const char * _glVertexAttrib2sARB_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttrib2sARB_sig = {1522, "glVertexAttrib2sARB", 3, _glVertexAttrib2sARB_args};

static const char * _glVertexAttrib2svARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib2svARB_sig = {1523, "glVertexAttrib2svARB", 2, _glVertexAttrib2svARB_args};

static const char * _glVertexAttrib3dARB_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttrib3dARB_sig = {1524, "glVertexAttrib3dARB", 4, _glVertexAttrib3dARB_args};

static const char * _glVertexAttrib3dvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib3dvARB_sig = {1525, "glVertexAttrib3dvARB", 2, _glVertexAttrib3dvARB_args};

static const char * _glVertexAttrib3fARB_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttrib3fARB_sig = {1526, "glVertexAttrib3fARB", 4, _glVertexAttrib3fARB_args};

static const char * _glVertexAttrib3fvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib3fvARB_sig = {1527, "glVertexAttrib3fvARB", 2, _glVertexAttrib3fvARB_args};

static const char * _glVertexAttrib3sARB_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttrib3sARB_sig = {1528, "glVertexAttrib3sARB", 4, _glVertexAttrib3sARB_args};

static const char * _glVertexAttrib3svARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib3svARB_sig = {1529, "glVertexAttrib3svARB", 2, _glVertexAttrib3svARB_args};

static const char * _glVertexAttrib4NbvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4NbvARB_sig = {1530, "glVertexAttrib4NbvARB", 2, _glVertexAttrib4NbvARB_args};

static const char * _glVertexAttrib4NivARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4NivARB_sig = {1531, "glVertexAttrib4NivARB", 2, _glVertexAttrib4NivARB_args};

static const char * _glVertexAttrib4NsvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4NsvARB_sig = {1532, "glVertexAttrib4NsvARB", 2, _glVertexAttrib4NsvARB_args};

static const char * _glVertexAttrib4NubARB_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4NubARB_sig = {1533, "glVertexAttrib4NubARB", 5, _glVertexAttrib4NubARB_args};

static const char * _glVertexAttrib4NubvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4NubvARB_sig = {1534, "glVertexAttrib4NubvARB", 2, _glVertexAttrib4NubvARB_args};

static const char * _glVertexAttrib4NuivARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4NuivARB_sig = {1535, "glVertexAttrib4NuivARB", 2, _glVertexAttrib4NuivARB_args};

static const char * _glVertexAttrib4NusvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4NusvARB_sig = {1536, "glVertexAttrib4NusvARB", 2, _glVertexAttrib4NusvARB_args};

static const char * _glVertexAttrib4bvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4bvARB_sig = {1537, "glVertexAttrib4bvARB", 2, _glVertexAttrib4bvARB_args};

static const char * _glVertexAttrib4dARB_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4dARB_sig = {1538, "glVertexAttrib4dARB", 5, _glVertexAttrib4dARB_args};

static const char * _glVertexAttrib4dvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4dvARB_sig = {1539, "glVertexAttrib4dvARB", 2, _glVertexAttrib4dvARB_args};

static const char * _glVertexAttrib4fARB_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4fARB_sig = {1540, "glVertexAttrib4fARB", 5, _glVertexAttrib4fARB_args};

static const char * _glVertexAttrib4fvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4fvARB_sig = {1541, "glVertexAttrib4fvARB", 2, _glVertexAttrib4fvARB_args};

static const char * _glVertexAttrib4ivARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4ivARB_sig = {1542, "glVertexAttrib4ivARB", 2, _glVertexAttrib4ivARB_args};

static const char * _glVertexAttrib4sARB_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4sARB_sig = {1543, "glVertexAttrib4sARB", 5, _glVertexAttrib4sARB_args};

static const char * _glVertexAttrib4svARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4svARB_sig = {1544, "glVertexAttrib4svARB", 2, _glVertexAttrib4svARB_args};

static const char * _glVertexAttrib4ubvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4ubvARB_sig = {1545, "glVertexAttrib4ubvARB", 2, _glVertexAttrib4ubvARB_args};

static const char * _glVertexAttrib4uivARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4uivARB_sig = {1546, "glVertexAttrib4uivARB", 2, _glVertexAttrib4uivARB_args};

static const char * _glVertexAttrib4usvARB_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4usvARB_sig = {1547, "glVertexAttrib4usvARB", 2, _glVertexAttrib4usvARB_args};

static const char * _glVertexAttribPointerARB_args[6] = {"index", "size", "type", "normalized", "stride", "pointer"};
static const trace::FunctionSig _glVertexAttribPointerARB_sig = {1548, "glVertexAttribPointerARB", 6, _glVertexAttribPointerARB_args};

static const char * _glEnableVertexAttribArrayARB_args[1] = {"index"};
static const trace::FunctionSig _glEnableVertexAttribArrayARB_sig = {1549, "glEnableVertexAttribArrayARB", 1, _glEnableVertexAttribArrayARB_args};

static const char * _glDisableVertexAttribArrayARB_args[1] = {"index"};
static const trace::FunctionSig _glDisableVertexAttribArrayARB_sig = {1550, "glDisableVertexAttribArrayARB", 1, _glDisableVertexAttribArrayARB_args};

static const char * _glProgramStringARB_args[4] = {"target", "format", "len", "string"};
static const trace::FunctionSig _glProgramStringARB_sig = {1551, "glProgramStringARB", 4, _glProgramStringARB_args};

static const char * _glBindProgramARB_args[2] = {"target", "program"};
static const trace::FunctionSig _glBindProgramARB_sig = {1552, "glBindProgramARB", 2, _glBindProgramARB_args};

static const char * _glDeleteProgramsARB_args[2] = {"n", "programs"};
static const trace::FunctionSig _glDeleteProgramsARB_sig = {1553, "glDeleteProgramsARB", 2, _glDeleteProgramsARB_args};

static const char * _glGenProgramsARB_args[2] = {"n", "programs"};
static const trace::FunctionSig _glGenProgramsARB_sig = {1554, "glGenProgramsARB", 2, _glGenProgramsARB_args};

static const char * _glProgramEnvParameter4dARB_args[6] = {"target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramEnvParameter4dARB_sig = {1555, "glProgramEnvParameter4dARB", 6, _glProgramEnvParameter4dARB_args};

static const char * _glProgramEnvParameter4dvARB_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glProgramEnvParameter4dvARB_sig = {1556, "glProgramEnvParameter4dvARB", 3, _glProgramEnvParameter4dvARB_args};

static const char * _glProgramEnvParameter4fARB_args[6] = {"target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramEnvParameter4fARB_sig = {1557, "glProgramEnvParameter4fARB", 6, _glProgramEnvParameter4fARB_args};

static const char * _glProgramEnvParameter4fvARB_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glProgramEnvParameter4fvARB_sig = {1558, "glProgramEnvParameter4fvARB", 3, _glProgramEnvParameter4fvARB_args};

static const char * _glProgramLocalParameter4dARB_args[6] = {"target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramLocalParameter4dARB_sig = {1559, "glProgramLocalParameter4dARB", 6, _glProgramLocalParameter4dARB_args};

static const char * _glProgramLocalParameter4dvARB_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glProgramLocalParameter4dvARB_sig = {1560, "glProgramLocalParameter4dvARB", 3, _glProgramLocalParameter4dvARB_args};

static const char * _glProgramLocalParameter4fARB_args[6] = {"target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramLocalParameter4fARB_sig = {1561, "glProgramLocalParameter4fARB", 6, _glProgramLocalParameter4fARB_args};

static const char * _glProgramLocalParameter4fvARB_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glProgramLocalParameter4fvARB_sig = {1562, "glProgramLocalParameter4fvARB", 3, _glProgramLocalParameter4fvARB_args};

static const char * _glGetProgramEnvParameterdvARB_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glGetProgramEnvParameterdvARB_sig = {1563, "glGetProgramEnvParameterdvARB", 3, _glGetProgramEnvParameterdvARB_args};

static const char * _glGetProgramEnvParameterfvARB_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glGetProgramEnvParameterfvARB_sig = {1564, "glGetProgramEnvParameterfvARB", 3, _glGetProgramEnvParameterfvARB_args};

static const char * _glGetProgramLocalParameterdvARB_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glGetProgramLocalParameterdvARB_sig = {1565, "glGetProgramLocalParameterdvARB", 3, _glGetProgramLocalParameterdvARB_args};

static const char * _glGetProgramLocalParameterfvARB_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glGetProgramLocalParameterfvARB_sig = {1566, "glGetProgramLocalParameterfvARB", 3, _glGetProgramLocalParameterfvARB_args};

static const char * _glGetProgramivARB_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetProgramivARB_sig = {1567, "glGetProgramivARB", 3, _glGetProgramivARB_args};

static const char * _glGetProgramStringARB_args[3] = {"target", "pname", "string"};
static const trace::FunctionSig _glGetProgramStringARB_sig = {1568, "glGetProgramStringARB", 3, _glGetProgramStringARB_args};

static const char * _glGetVertexAttribdvARB_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribdvARB_sig = {1569, "glGetVertexAttribdvARB", 3, _glGetVertexAttribdvARB_args};

static const char * _glGetVertexAttribfvARB_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribfvARB_sig = {1570, "glGetVertexAttribfvARB", 3, _glGetVertexAttribfvARB_args};

static const char * _glGetVertexAttribivARB_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribivARB_sig = {1571, "glGetVertexAttribivARB", 3, _glGetVertexAttribivARB_args};

static const char * _glGetVertexAttribPointervARB_args[3] = {"index", "pname", "pointer"};
static const trace::FunctionSig _glGetVertexAttribPointervARB_sig = {1572, "glGetVertexAttribPointervARB", 3, _glGetVertexAttribPointervARB_args};

static const char * _glIsProgramARB_args[1] = {"program"};
static const trace::FunctionSig _glIsProgramARB_sig = {1573, "glIsProgramARB", 1, _glIsProgramARB_args};

static const char * _glBindAttribLocationARB_args[3] = {"programObj", "index", "name"};
static const trace::FunctionSig _glBindAttribLocationARB_sig = {1574, "glBindAttribLocationARB", 3, _glBindAttribLocationARB_args};

static const char * _glGetActiveAttribARB_args[7] = {"programObj", "index", "maxLength", "length", "size", "type", "name"};
static const trace::FunctionSig _glGetActiveAttribARB_sig = {1575, "glGetActiveAttribARB", 7, _glGetActiveAttribARB_args};

static const char * _glGetAttribLocationARB_args[2] = {"programObj", "name"};
static const trace::FunctionSig _glGetAttribLocationARB_sig = {1576, "glGetAttribLocationARB", 2, _glGetAttribLocationARB_args};

static const char * _glVertexAttribP1ui_args[4] = {"index", "type", "normalized", "value"};
static const trace::FunctionSig _glVertexAttribP1ui_sig = {1577, "glVertexAttribP1ui", 4, _glVertexAttribP1ui_args};

static const char * _glVertexAttribP1uiv_args[4] = {"index", "type", "normalized", "value"};
static const trace::FunctionSig _glVertexAttribP1uiv_sig = {1578, "glVertexAttribP1uiv", 4, _glVertexAttribP1uiv_args};

static const char * _glVertexAttribP2ui_args[4] = {"index", "type", "normalized", "value"};
static const trace::FunctionSig _glVertexAttribP2ui_sig = {1579, "glVertexAttribP2ui", 4, _glVertexAttribP2ui_args};

static const char * _glVertexAttribP2uiv_args[4] = {"index", "type", "normalized", "value"};
static const trace::FunctionSig _glVertexAttribP2uiv_sig = {1580, "glVertexAttribP2uiv", 4, _glVertexAttribP2uiv_args};

static const char * _glVertexAttribP3ui_args[4] = {"index", "type", "normalized", "value"};
static const trace::FunctionSig _glVertexAttribP3ui_sig = {1581, "glVertexAttribP3ui", 4, _glVertexAttribP3ui_args};

static const char * _glVertexAttribP3uiv_args[4] = {"index", "type", "normalized", "value"};
static const trace::FunctionSig _glVertexAttribP3uiv_sig = {1582, "glVertexAttribP3uiv", 4, _glVertexAttribP3uiv_args};

static const char * _glVertexAttribP4ui_args[4] = {"index", "type", "normalized", "value"};
static const trace::FunctionSig _glVertexAttribP4ui_sig = {1583, "glVertexAttribP4ui", 4, _glVertexAttribP4ui_args};

static const char * _glVertexAttribP4uiv_args[4] = {"index", "type", "normalized", "value"};
static const trace::FunctionSig _glVertexAttribP4uiv_sig = {1584, "glVertexAttribP4uiv", 4, _glVertexAttribP4uiv_args};

static const char * _glVertexP2ui_args[2] = {"type", "value"};
static const trace::FunctionSig _glVertexP2ui_sig = {1585, "glVertexP2ui", 2, _glVertexP2ui_args};

static const char * _glVertexP2uiv_args[2] = {"type", "value"};
static const trace::FunctionSig _glVertexP2uiv_sig = {1586, "glVertexP2uiv", 2, _glVertexP2uiv_args};

static const char * _glVertexP3ui_args[2] = {"type", "value"};
static const trace::FunctionSig _glVertexP3ui_sig = {1587, "glVertexP3ui", 2, _glVertexP3ui_args};

static const char * _glVertexP3uiv_args[2] = {"type", "value"};
static const trace::FunctionSig _glVertexP3uiv_sig = {1588, "glVertexP3uiv", 2, _glVertexP3uiv_args};

static const char * _glVertexP4ui_args[2] = {"type", "value"};
static const trace::FunctionSig _glVertexP4ui_sig = {1589, "glVertexP4ui", 2, _glVertexP4ui_args};

static const char * _glVertexP4uiv_args[2] = {"type", "value"};
static const trace::FunctionSig _glVertexP4uiv_sig = {1590, "glVertexP4uiv", 2, _glVertexP4uiv_args};

static const char * _glTexCoordP1ui_args[2] = {"type", "coords"};
static const trace::FunctionSig _glTexCoordP1ui_sig = {1591, "glTexCoordP1ui", 2, _glTexCoordP1ui_args};

static const char * _glTexCoordP1uiv_args[2] = {"type", "coords"};
static const trace::FunctionSig _glTexCoordP1uiv_sig = {1592, "glTexCoordP1uiv", 2, _glTexCoordP1uiv_args};

static const char * _glTexCoordP2ui_args[2] = {"type", "coords"};
static const trace::FunctionSig _glTexCoordP2ui_sig = {1593, "glTexCoordP2ui", 2, _glTexCoordP2ui_args};

static const char * _glTexCoordP2uiv_args[2] = {"type", "coords"};
static const trace::FunctionSig _glTexCoordP2uiv_sig = {1594, "glTexCoordP2uiv", 2, _glTexCoordP2uiv_args};

static const char * _glTexCoordP3ui_args[2] = {"type", "coords"};
static const trace::FunctionSig _glTexCoordP3ui_sig = {1595, "glTexCoordP3ui", 2, _glTexCoordP3ui_args};

static const char * _glTexCoordP3uiv_args[2] = {"type", "coords"};
static const trace::FunctionSig _glTexCoordP3uiv_sig = {1596, "glTexCoordP3uiv", 2, _glTexCoordP3uiv_args};

static const char * _glTexCoordP4ui_args[2] = {"type", "coords"};
static const trace::FunctionSig _glTexCoordP4ui_sig = {1597, "glTexCoordP4ui", 2, _glTexCoordP4ui_args};

static const char * _glTexCoordP4uiv_args[2] = {"type", "coords"};
static const trace::FunctionSig _glTexCoordP4uiv_sig = {1598, "glTexCoordP4uiv", 2, _glTexCoordP4uiv_args};

static const char * _glMultiTexCoordP1ui_args[3] = {"texture", "type", "coords"};
static const trace::FunctionSig _glMultiTexCoordP1ui_sig = {1599, "glMultiTexCoordP1ui", 3, _glMultiTexCoordP1ui_args};

static const char * _glMultiTexCoordP1uiv_args[3] = {"texture", "type", "coords"};
static const trace::FunctionSig _glMultiTexCoordP1uiv_sig = {1600, "glMultiTexCoordP1uiv", 3, _glMultiTexCoordP1uiv_args};

static const char * _glMultiTexCoordP2ui_args[3] = {"texture", "type", "coords"};
static const trace::FunctionSig _glMultiTexCoordP2ui_sig = {1601, "glMultiTexCoordP2ui", 3, _glMultiTexCoordP2ui_args};

static const char * _glMultiTexCoordP2uiv_args[3] = {"texture", "type", "coords"};
static const trace::FunctionSig _glMultiTexCoordP2uiv_sig = {1602, "glMultiTexCoordP2uiv", 3, _glMultiTexCoordP2uiv_args};

static const char * _glMultiTexCoordP3ui_args[3] = {"texture", "type", "coords"};
static const trace::FunctionSig _glMultiTexCoordP3ui_sig = {1603, "glMultiTexCoordP3ui", 3, _glMultiTexCoordP3ui_args};

static const char * _glMultiTexCoordP3uiv_args[3] = {"texture", "type", "coords"};
static const trace::FunctionSig _glMultiTexCoordP3uiv_sig = {1604, "glMultiTexCoordP3uiv", 3, _glMultiTexCoordP3uiv_args};

static const char * _glMultiTexCoordP4ui_args[3] = {"texture", "type", "coords"};
static const trace::FunctionSig _glMultiTexCoordP4ui_sig = {1605, "glMultiTexCoordP4ui", 3, _glMultiTexCoordP4ui_args};

static const char * _glMultiTexCoordP4uiv_args[3] = {"texture", "type", "coords"};
static const trace::FunctionSig _glMultiTexCoordP4uiv_sig = {1606, "glMultiTexCoordP4uiv", 3, _glMultiTexCoordP4uiv_args};

static const char * _glNormalP3ui_args[2] = {"type", "coords"};
static const trace::FunctionSig _glNormalP3ui_sig = {1607, "glNormalP3ui", 2, _glNormalP3ui_args};

static const char * _glNormalP3uiv_args[2] = {"type", "coords"};
static const trace::FunctionSig _glNormalP3uiv_sig = {1608, "glNormalP3uiv", 2, _glNormalP3uiv_args};

static const char * _glColorP3ui_args[2] = {"type", "color"};
static const trace::FunctionSig _glColorP3ui_sig = {1609, "glColorP3ui", 2, _glColorP3ui_args};

static const char * _glColorP3uiv_args[2] = {"type", "color"};
static const trace::FunctionSig _glColorP3uiv_sig = {1610, "glColorP3uiv", 2, _glColorP3uiv_args};

static const char * _glColorP4ui_args[2] = {"type", "color"};
static const trace::FunctionSig _glColorP4ui_sig = {1611, "glColorP4ui", 2, _glColorP4ui_args};

static const char * _glColorP4uiv_args[2] = {"type", "color"};
static const trace::FunctionSig _glColorP4uiv_sig = {1612, "glColorP4uiv", 2, _glColorP4uiv_args};

static const char * _glSecondaryColorP3ui_args[2] = {"type", "color"};
static const trace::FunctionSig _glSecondaryColorP3ui_sig = {1613, "glSecondaryColorP3ui", 2, _glSecondaryColorP3ui_args};

static const char * _glSecondaryColorP3uiv_args[2] = {"type", "color"};
static const trace::FunctionSig _glSecondaryColorP3uiv_sig = {1614, "glSecondaryColorP3uiv", 2, _glSecondaryColorP3uiv_args};

static const char * _glViewportArrayv_args[3] = {"first", "count", "v"};
static const trace::FunctionSig _glViewportArrayv_sig = {1615, "glViewportArrayv", 3, _glViewportArrayv_args};

static const char * _glViewportIndexedf_args[5] = {"index", "x", "y", "w", "h"};
static const trace::FunctionSig _glViewportIndexedf_sig = {1616, "glViewportIndexedf", 5, _glViewportIndexedf_args};

static const char * _glViewportIndexedfv_args[2] = {"index", "v"};
static const trace::FunctionSig _glViewportIndexedfv_sig = {1617, "glViewportIndexedfv", 2, _glViewportIndexedfv_args};

static const char * _glScissorArrayv_args[3] = {"first", "count", "v"};
static const trace::FunctionSig _glScissorArrayv_sig = {1618, "glScissorArrayv", 3, _glScissorArrayv_args};

static const char * _glScissorIndexed_args[5] = {"index", "left", "bottom", "width", "height"};
static const trace::FunctionSig _glScissorIndexed_sig = {1619, "glScissorIndexed", 5, _glScissorIndexed_args};

static const char * _glScissorIndexedv_args[2] = {"index", "v"};
static const trace::FunctionSig _glScissorIndexedv_sig = {1620, "glScissorIndexedv", 2, _glScissorIndexedv_args};

static const char * _glDepthRangeArrayv_args[3] = {"first", "count", "v"};
static const trace::FunctionSig _glDepthRangeArrayv_sig = {1621, "glDepthRangeArrayv", 3, _glDepthRangeArrayv_args};

static const char * _glDepthRangeIndexed_args[3] = {"index", "n", "f"};
static const trace::FunctionSig _glDepthRangeIndexed_sig = {1622, "glDepthRangeIndexed", 3, _glDepthRangeIndexed_args};

static const char * _glGetFloati_v_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetFloati_v_sig = {1623, "glGetFloati_v", 3, _glGetFloati_v_args};

static const char * _glGetDoublei_v_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetDoublei_v_sig = {1624, "glGetDoublei_v", 3, _glGetDoublei_v_args};

static const char * _glWindowPos2dARB_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2dARB_sig = {1625, "glWindowPos2dARB", 2, _glWindowPos2dARB_args};

static const char * _glWindowPos2dvARB_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2dvARB_sig = {1626, "glWindowPos2dvARB", 1, _glWindowPos2dvARB_args};

static const char * _glWindowPos2fARB_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2fARB_sig = {1627, "glWindowPos2fARB", 2, _glWindowPos2fARB_args};

static const char * _glWindowPos2fvARB_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2fvARB_sig = {1628, "glWindowPos2fvARB", 1, _glWindowPos2fvARB_args};

static const char * _glWindowPos2iARB_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2iARB_sig = {1629, "glWindowPos2iARB", 2, _glWindowPos2iARB_args};

static const char * _glWindowPos2ivARB_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2ivARB_sig = {1630, "glWindowPos2ivARB", 1, _glWindowPos2ivARB_args};

static const char * _glWindowPos2sARB_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2sARB_sig = {1631, "glWindowPos2sARB", 2, _glWindowPos2sARB_args};

static const char * _glWindowPos2svARB_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2svARB_sig = {1632, "glWindowPos2svARB", 1, _glWindowPos2svARB_args};

static const char * _glWindowPos3dARB_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3dARB_sig = {1633, "glWindowPos3dARB", 3, _glWindowPos3dARB_args};

static const char * _glWindowPos3dvARB_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3dvARB_sig = {1634, "glWindowPos3dvARB", 1, _glWindowPos3dvARB_args};

static const char * _glWindowPos3fARB_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3fARB_sig = {1635, "glWindowPos3fARB", 3, _glWindowPos3fARB_args};

static const char * _glWindowPos3fvARB_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3fvARB_sig = {1636, "glWindowPos3fvARB", 1, _glWindowPos3fvARB_args};

static const char * _glWindowPos3iARB_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3iARB_sig = {1637, "glWindowPos3iARB", 3, _glWindowPos3iARB_args};

static const char * _glWindowPos3ivARB_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3ivARB_sig = {1638, "glWindowPos3ivARB", 1, _glWindowPos3ivARB_args};

static const char * _glWindowPos3sARB_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3sARB_sig = {1639, "glWindowPos3sARB", 3, _glWindowPos3sARB_args};

static const char * _glWindowPos3svARB_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3svARB_sig = {1640, "glWindowPos3svARB", 1, _glWindowPos3svARB_args};

static const char * _glDrawBuffersATI_args[2] = {"n", "bufs"};
static const trace::FunctionSig _glDrawBuffersATI_sig = {1641, "glDrawBuffersATI", 2, _glDrawBuffersATI_args};

static const char * _glElementPointerATI_args[2] = {"type", "pointer"};
static const trace::FunctionSig _glElementPointerATI_sig = {1642, "glElementPointerATI", 2, _glElementPointerATI_args};

static const char * _glDrawElementArrayATI_args[2] = {"mode", "count"};
static const trace::FunctionSig _glDrawElementArrayATI_sig = {1643, "glDrawElementArrayATI", 2, _glDrawElementArrayATI_args};

static const char * _glDrawRangeElementArrayATI_args[4] = {"mode", "start", "end", "count"};
static const trace::FunctionSig _glDrawRangeElementArrayATI_sig = {1644, "glDrawRangeElementArrayATI", 4, _glDrawRangeElementArrayATI_args};

static const char * _glTexBumpParameterivATI_args[2] = {"pname", "param"};
static const trace::FunctionSig _glTexBumpParameterivATI_sig = {1645, "glTexBumpParameterivATI", 2, _glTexBumpParameterivATI_args};

static const char * _glTexBumpParameterfvATI_args[2] = {"pname", "param"};
static const trace::FunctionSig _glTexBumpParameterfvATI_sig = {1646, "glTexBumpParameterfvATI", 2, _glTexBumpParameterfvATI_args};

static const char * _glGetTexBumpParameterivATI_args[2] = {"pname", "param"};
static const trace::FunctionSig _glGetTexBumpParameterivATI_sig = {1647, "glGetTexBumpParameterivATI", 2, _glGetTexBumpParameterivATI_args};

static const char * _glGetTexBumpParameterfvATI_args[2] = {"pname", "param"};
static const trace::FunctionSig _glGetTexBumpParameterfvATI_sig = {1648, "glGetTexBumpParameterfvATI", 2, _glGetTexBumpParameterfvATI_args};

static const char * _glBlendEquationSeparateATI_args[2] = {"equationRGB", "equationAlpha"};
static const trace::FunctionSig _glBlendEquationSeparateATI_sig = {1649, "glBlendEquationSeparateATI", 2, _glBlendEquationSeparateATI_args};

static const char * _glGenFragmentShadersATI_args[1] = {"range"};
static const trace::FunctionSig _glGenFragmentShadersATI_sig = {1650, "glGenFragmentShadersATI", 1, _glGenFragmentShadersATI_args};

static const char * _glBindFragmentShaderATI_args[1] = {"id"};
static const trace::FunctionSig _glBindFragmentShaderATI_sig = {1651, "glBindFragmentShaderATI", 1, _glBindFragmentShaderATI_args};

static const char * _glDeleteFragmentShaderATI_args[1] = {"id"};
static const trace::FunctionSig _glDeleteFragmentShaderATI_sig = {1652, "glDeleteFragmentShaderATI", 1, _glDeleteFragmentShaderATI_args};

static const char ** _glBeginFragmentShaderATI_args = NULL;
static const trace::FunctionSig _glBeginFragmentShaderATI_sig = {1653, "glBeginFragmentShaderATI", 0, _glBeginFragmentShaderATI_args};

static const char ** _glEndFragmentShaderATI_args = NULL;
static const trace::FunctionSig _glEndFragmentShaderATI_sig = {1654, "glEndFragmentShaderATI", 0, _glEndFragmentShaderATI_args};

static const char * _glPassTexCoordATI_args[3] = {"dst", "coord", "swizzle"};
static const trace::FunctionSig _glPassTexCoordATI_sig = {1655, "glPassTexCoordATI", 3, _glPassTexCoordATI_args};

static const char * _glSampleMapATI_args[3] = {"dst", "interp", "swizzle"};
static const trace::FunctionSig _glSampleMapATI_sig = {1656, "glSampleMapATI", 3, _glSampleMapATI_args};

static const char * _glColorFragmentOp1ATI_args[7] = {"op", "dst", "dstMask", "dstMod", "arg1", "arg1Rep", "arg1Mod"};
static const trace::FunctionSig _glColorFragmentOp1ATI_sig = {1657, "glColorFragmentOp1ATI", 7, _glColorFragmentOp1ATI_args};

static const char * _glColorFragmentOp2ATI_args[10] = {"op", "dst", "dstMask", "dstMod", "arg1", "arg1Rep", "arg1Mod", "arg2", "arg2Rep", "arg2Mod"};
static const trace::FunctionSig _glColorFragmentOp2ATI_sig = {1658, "glColorFragmentOp2ATI", 10, _glColorFragmentOp2ATI_args};

static const char * _glColorFragmentOp3ATI_args[13] = {"op", "dst", "dstMask", "dstMod", "arg1", "arg1Rep", "arg1Mod", "arg2", "arg2Rep", "arg2Mod", "arg3", "arg3Rep", "arg3Mod"};
static const trace::FunctionSig _glColorFragmentOp3ATI_sig = {1659, "glColorFragmentOp3ATI", 13, _glColorFragmentOp3ATI_args};

static const char * _glAlphaFragmentOp1ATI_args[6] = {"op", "dst", "dstMod", "arg1", "arg1Rep", "arg1Mod"};
static const trace::FunctionSig _glAlphaFragmentOp1ATI_sig = {1660, "glAlphaFragmentOp1ATI", 6, _glAlphaFragmentOp1ATI_args};

static const char * _glAlphaFragmentOp2ATI_args[9] = {"op", "dst", "dstMod", "arg1", "arg1Rep", "arg1Mod", "arg2", "arg2Rep", "arg2Mod"};
static const trace::FunctionSig _glAlphaFragmentOp2ATI_sig = {1661, "glAlphaFragmentOp2ATI", 9, _glAlphaFragmentOp2ATI_args};

static const char * _glAlphaFragmentOp3ATI_args[12] = {"op", "dst", "dstMod", "arg1", "arg1Rep", "arg1Mod", "arg2", "arg2Rep", "arg2Mod", "arg3", "arg3Rep", "arg3Mod"};
static const trace::FunctionSig _glAlphaFragmentOp3ATI_sig = {1662, "glAlphaFragmentOp3ATI", 12, _glAlphaFragmentOp3ATI_args};

static const char * _glSetFragmentShaderConstantATI_args[2] = {"dst", "value"};
static const trace::FunctionSig _glSetFragmentShaderConstantATI_sig = {1663, "glSetFragmentShaderConstantATI", 2, _glSetFragmentShaderConstantATI_args};

static const char * _glMapObjectBufferATI_args[1] = {"buffer"};
static const trace::FunctionSig _glMapObjectBufferATI_sig = {1664, "glMapObjectBufferATI", 1, _glMapObjectBufferATI_args};

static const char * _glUnmapObjectBufferATI_args[1] = {"buffer"};
static const trace::FunctionSig _glUnmapObjectBufferATI_sig = {1665, "glUnmapObjectBufferATI", 1, _glUnmapObjectBufferATI_args};

static const char * _glPNTrianglesiATI_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPNTrianglesiATI_sig = {1666, "glPNTrianglesiATI", 2, _glPNTrianglesiATI_args};

static const char * _glPNTrianglesfATI_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPNTrianglesfATI_sig = {1667, "glPNTrianglesfATI", 2, _glPNTrianglesfATI_args};

static const char * _glPNTrianglesiATIX_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPNTrianglesiATIX_sig = {1668, "glPNTrianglesiATIX", 2, _glPNTrianglesiATIX_args};

static const char * _glPNTrianglesfATIX_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPNTrianglesfATIX_sig = {1669, "glPNTrianglesfATIX", 2, _glPNTrianglesfATIX_args};

static const char * _glStencilOpSeparateATI_args[4] = {"face", "sfail", "dpfail", "dppass"};
static const trace::FunctionSig _glStencilOpSeparateATI_sig = {1670, "glStencilOpSeparateATI", 4, _glStencilOpSeparateATI_args};

static const char * _glStencilFuncSeparateATI_args[4] = {"frontfunc", "backfunc", "ref", "mask"};
static const trace::FunctionSig _glStencilFuncSeparateATI_sig = {1671, "glStencilFuncSeparateATI", 4, _glStencilFuncSeparateATI_args};

static const char * _glNewObjectBufferATI_args[3] = {"size", "pointer", "usage"};
static const trace::FunctionSig _glNewObjectBufferATI_sig = {1672, "glNewObjectBufferATI", 3, _glNewObjectBufferATI_args};

static const char * _glIsObjectBufferATI_args[1] = {"buffer"};
static const trace::FunctionSig _glIsObjectBufferATI_sig = {1673, "glIsObjectBufferATI", 1, _glIsObjectBufferATI_args};

static const char * _glUpdateObjectBufferATI_args[5] = {"buffer", "offset", "size", "pointer", "preserve"};
static const trace::FunctionSig _glUpdateObjectBufferATI_sig = {1674, "glUpdateObjectBufferATI", 5, _glUpdateObjectBufferATI_args};

static const char * _glGetObjectBufferfvATI_args[3] = {"buffer", "pname", "params"};
static const trace::FunctionSig _glGetObjectBufferfvATI_sig = {1675, "glGetObjectBufferfvATI", 3, _glGetObjectBufferfvATI_args};

static const char * _glGetObjectBufferivATI_args[3] = {"buffer", "pname", "params"};
static const trace::FunctionSig _glGetObjectBufferivATI_sig = {1676, "glGetObjectBufferivATI", 3, _glGetObjectBufferivATI_args};

static const char * _glFreeObjectBufferATI_args[1] = {"buffer"};
static const trace::FunctionSig _glFreeObjectBufferATI_sig = {1677, "glFreeObjectBufferATI", 1, _glFreeObjectBufferATI_args};

static const char * _glArrayObjectATI_args[6] = {"array", "size", "type", "stride", "buffer", "offset"};
static const trace::FunctionSig _glArrayObjectATI_sig = {1678, "glArrayObjectATI", 6, _glArrayObjectATI_args};

static const char * _glGetArrayObjectfvATI_args[3] = {"array", "pname", "params"};
static const trace::FunctionSig _glGetArrayObjectfvATI_sig = {1679, "glGetArrayObjectfvATI", 3, _glGetArrayObjectfvATI_args};

static const char * _glGetArrayObjectivATI_args[3] = {"array", "pname", "params"};
static const trace::FunctionSig _glGetArrayObjectivATI_sig = {1680, "glGetArrayObjectivATI", 3, _glGetArrayObjectivATI_args};

static const char * _glVariantArrayObjectATI_args[5] = {"id", "type", "stride", "buffer", "offset"};
static const trace::FunctionSig _glVariantArrayObjectATI_sig = {1681, "glVariantArrayObjectATI", 5, _glVariantArrayObjectATI_args};

static const char * _glGetVariantArrayObjectfvATI_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetVariantArrayObjectfvATI_sig = {1682, "glGetVariantArrayObjectfvATI", 3, _glGetVariantArrayObjectfvATI_args};

static const char * _glGetVariantArrayObjectivATI_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetVariantArrayObjectivATI_sig = {1683, "glGetVariantArrayObjectivATI", 3, _glGetVariantArrayObjectivATI_args};

static const char * _glVertexAttribArrayObjectATI_args[7] = {"index", "size", "type", "normalized", "stride", "buffer", "offset"};
static const trace::FunctionSig _glVertexAttribArrayObjectATI_sig = {1684, "glVertexAttribArrayObjectATI", 7, _glVertexAttribArrayObjectATI_args};

static const char * _glGetVertexAttribArrayObjectfvATI_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribArrayObjectfvATI_sig = {1685, "glGetVertexAttribArrayObjectfvATI", 3, _glGetVertexAttribArrayObjectfvATI_args};

static const char * _glGetVertexAttribArrayObjectivATI_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribArrayObjectivATI_sig = {1686, "glGetVertexAttribArrayObjectivATI", 3, _glGetVertexAttribArrayObjectivATI_args};

static const char * _glVertexStream1sATI_args[2] = {"stream", "x"};
static const trace::FunctionSig _glVertexStream1sATI_sig = {1687, "glVertexStream1sATI", 2, _glVertexStream1sATI_args};

static const char * _glVertexStream1svATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream1svATI_sig = {1688, "glVertexStream1svATI", 2, _glVertexStream1svATI_args};

static const char * _glVertexStream1iATI_args[2] = {"stream", "x"};
static const trace::FunctionSig _glVertexStream1iATI_sig = {1689, "glVertexStream1iATI", 2, _glVertexStream1iATI_args};

static const char * _glVertexStream1ivATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream1ivATI_sig = {1690, "glVertexStream1ivATI", 2, _glVertexStream1ivATI_args};

static const char * _glVertexStream1fATI_args[2] = {"stream", "x"};
static const trace::FunctionSig _glVertexStream1fATI_sig = {1691, "glVertexStream1fATI", 2, _glVertexStream1fATI_args};

static const char * _glVertexStream1fvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream1fvATI_sig = {1692, "glVertexStream1fvATI", 2, _glVertexStream1fvATI_args};

static const char * _glVertexStream1dATI_args[2] = {"stream", "x"};
static const trace::FunctionSig _glVertexStream1dATI_sig = {1693, "glVertexStream1dATI", 2, _glVertexStream1dATI_args};

static const char * _glVertexStream1dvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream1dvATI_sig = {1694, "glVertexStream1dvATI", 2, _glVertexStream1dvATI_args};

static const char * _glVertexStream2sATI_args[3] = {"stream", "x", "y"};
static const trace::FunctionSig _glVertexStream2sATI_sig = {1695, "glVertexStream2sATI", 3, _glVertexStream2sATI_args};

static const char * _glVertexStream2svATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream2svATI_sig = {1696, "glVertexStream2svATI", 2, _glVertexStream2svATI_args};

static const char * _glVertexStream2iATI_args[3] = {"stream", "x", "y"};
static const trace::FunctionSig _glVertexStream2iATI_sig = {1697, "glVertexStream2iATI", 3, _glVertexStream2iATI_args};

static const char * _glVertexStream2ivATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream2ivATI_sig = {1698, "glVertexStream2ivATI", 2, _glVertexStream2ivATI_args};

static const char * _glVertexStream2fATI_args[3] = {"stream", "x", "y"};
static const trace::FunctionSig _glVertexStream2fATI_sig = {1699, "glVertexStream2fATI", 3, _glVertexStream2fATI_args};

static const char * _glVertexStream2fvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream2fvATI_sig = {1700, "glVertexStream2fvATI", 2, _glVertexStream2fvATI_args};

static const char * _glVertexStream2dATI_args[3] = {"stream", "x", "y"};
static const trace::FunctionSig _glVertexStream2dATI_sig = {1701, "glVertexStream2dATI", 3, _glVertexStream2dATI_args};

static const char * _glVertexStream2dvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream2dvATI_sig = {1702, "glVertexStream2dvATI", 2, _glVertexStream2dvATI_args};

static const char * _glVertexStream3sATI_args[4] = {"stream", "x", "y", "z"};
static const trace::FunctionSig _glVertexStream3sATI_sig = {1703, "glVertexStream3sATI", 4, _glVertexStream3sATI_args};

static const char * _glVertexStream3svATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream3svATI_sig = {1704, "glVertexStream3svATI", 2, _glVertexStream3svATI_args};

static const char * _glVertexStream3iATI_args[4] = {"stream", "x", "y", "z"};
static const trace::FunctionSig _glVertexStream3iATI_sig = {1705, "glVertexStream3iATI", 4, _glVertexStream3iATI_args};

static const char * _glVertexStream3ivATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream3ivATI_sig = {1706, "glVertexStream3ivATI", 2, _glVertexStream3ivATI_args};

static const char * _glVertexStream3fATI_args[4] = {"stream", "x", "y", "z"};
static const trace::FunctionSig _glVertexStream3fATI_sig = {1707, "glVertexStream3fATI", 4, _glVertexStream3fATI_args};

static const char * _glVertexStream3fvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream3fvATI_sig = {1708, "glVertexStream3fvATI", 2, _glVertexStream3fvATI_args};

static const char * _glVertexStream3dATI_args[4] = {"stream", "x", "y", "z"};
static const trace::FunctionSig _glVertexStream3dATI_sig = {1709, "glVertexStream3dATI", 4, _glVertexStream3dATI_args};

static const char * _glVertexStream3dvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream3dvATI_sig = {1710, "glVertexStream3dvATI", 2, _glVertexStream3dvATI_args};

static const char * _glVertexStream4sATI_args[5] = {"stream", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexStream4sATI_sig = {1711, "glVertexStream4sATI", 5, _glVertexStream4sATI_args};

static const char * _glVertexStream4svATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream4svATI_sig = {1712, "glVertexStream4svATI", 2, _glVertexStream4svATI_args};

static const char * _glVertexStream4iATI_args[5] = {"stream", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexStream4iATI_sig = {1713, "glVertexStream4iATI", 5, _glVertexStream4iATI_args};

static const char * _glVertexStream4ivATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream4ivATI_sig = {1714, "glVertexStream4ivATI", 2, _glVertexStream4ivATI_args};

static const char * _glVertexStream4fATI_args[5] = {"stream", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexStream4fATI_sig = {1715, "glVertexStream4fATI", 5, _glVertexStream4fATI_args};

static const char * _glVertexStream4fvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream4fvATI_sig = {1716, "glVertexStream4fvATI", 2, _glVertexStream4fvATI_args};

static const char * _glVertexStream4dATI_args[5] = {"stream", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexStream4dATI_sig = {1717, "glVertexStream4dATI", 5, _glVertexStream4dATI_args};

static const char * _glVertexStream4dvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glVertexStream4dvATI_sig = {1718, "glVertexStream4dvATI", 2, _glVertexStream4dvATI_args};

static const char * _glNormalStream3bATI_args[4] = {"stream", "nx", "ny", "nz"};
static const trace::FunctionSig _glNormalStream3bATI_sig = {1719, "glNormalStream3bATI", 4, _glNormalStream3bATI_args};

static const char * _glNormalStream3bvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glNormalStream3bvATI_sig = {1720, "glNormalStream3bvATI", 2, _glNormalStream3bvATI_args};

static const char * _glNormalStream3sATI_args[4] = {"stream", "nx", "ny", "nz"};
static const trace::FunctionSig _glNormalStream3sATI_sig = {1721, "glNormalStream3sATI", 4, _glNormalStream3sATI_args};

static const char * _glNormalStream3svATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glNormalStream3svATI_sig = {1722, "glNormalStream3svATI", 2, _glNormalStream3svATI_args};

static const char * _glNormalStream3iATI_args[4] = {"stream", "nx", "ny", "nz"};
static const trace::FunctionSig _glNormalStream3iATI_sig = {1723, "glNormalStream3iATI", 4, _glNormalStream3iATI_args};

static const char * _glNormalStream3ivATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glNormalStream3ivATI_sig = {1724, "glNormalStream3ivATI", 2, _glNormalStream3ivATI_args};

static const char * _glNormalStream3fATI_args[4] = {"stream", "nx", "ny", "nz"};
static const trace::FunctionSig _glNormalStream3fATI_sig = {1725, "glNormalStream3fATI", 4, _glNormalStream3fATI_args};

static const char * _glNormalStream3fvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glNormalStream3fvATI_sig = {1726, "glNormalStream3fvATI", 2, _glNormalStream3fvATI_args};

static const char * _glNormalStream3dATI_args[4] = {"stream", "nx", "ny", "nz"};
static const trace::FunctionSig _glNormalStream3dATI_sig = {1727, "glNormalStream3dATI", 4, _glNormalStream3dATI_args};

static const char * _glNormalStream3dvATI_args[2] = {"stream", "coords"};
static const trace::FunctionSig _glNormalStream3dvATI_sig = {1728, "glNormalStream3dvATI", 2, _glNormalStream3dvATI_args};

static const char * _glClientActiveVertexStreamATI_args[1] = {"stream"};
static const trace::FunctionSig _glClientActiveVertexStreamATI_sig = {1729, "glClientActiveVertexStreamATI", 1, _glClientActiveVertexStreamATI_args};

static const char * _glVertexBlendEnviATI_args[2] = {"pname", "param"};
static const trace::FunctionSig _glVertexBlendEnviATI_sig = {1730, "glVertexBlendEnviATI", 2, _glVertexBlendEnviATI_args};

static const char * _glVertexBlendEnvfATI_args[2] = {"pname", "param"};
static const trace::FunctionSig _glVertexBlendEnvfATI_sig = {1731, "glVertexBlendEnvfATI", 2, _glVertexBlendEnvfATI_args};

static const char * _glDrawArraysInstancedBaseInstanceEXT_args[5] = {"mode", "first", "count", "instancecount", "baseinstance"};
static const trace::FunctionSig _glDrawArraysInstancedBaseInstanceEXT_sig = {1732, "glDrawArraysInstancedBaseInstanceEXT", 5, _glDrawArraysInstancedBaseInstanceEXT_args};

static const char * _glDrawElementsInstancedBaseInstanceEXT_args[6] = {"mode", "count", "type", "indices", "instancecount", "baseinstance"};
static const trace::FunctionSig _glDrawElementsInstancedBaseInstanceEXT_sig = {1733, "glDrawElementsInstancedBaseInstanceEXT", 6, _glDrawElementsInstancedBaseInstanceEXT_args};

static const char * _glDrawElementsInstancedBaseVertexBaseInstanceEXT_args[7] = {"mode", "count", "type", "indices", "instancecount", "basevertex", "baseinstance"};
static const trace::FunctionSig _glDrawElementsInstancedBaseVertexBaseInstanceEXT_sig = {1734, "glDrawElementsInstancedBaseVertexBaseInstanceEXT", 7, _glDrawElementsInstancedBaseVertexBaseInstanceEXT_args};

static const char * _glUniformBufferEXT_args[3] = {"program", "location", "buffer"};
static const trace::FunctionSig _glUniformBufferEXT_sig = {1735, "glUniformBufferEXT", 3, _glUniformBufferEXT_args};

static const char * _glGetUniformBufferSizeEXT_args[2] = {"program", "location"};
static const trace::FunctionSig _glGetUniformBufferSizeEXT_sig = {1736, "glGetUniformBufferSizeEXT", 2, _glGetUniformBufferSizeEXT_args};

static const char * _glGetUniformOffsetEXT_args[2] = {"program", "location"};
static const trace::FunctionSig _glGetUniformOffsetEXT_sig = {1737, "glGetUniformOffsetEXT", 2, _glGetUniformOffsetEXT_args};

static const char * _glBlendColorEXT_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glBlendColorEXT_sig = {1738, "glBlendColorEXT", 4, _glBlendColorEXT_args};

static const char * _glBlendEquationSeparateEXT_args[2] = {"modeRGB", "modeAlpha"};
static const trace::FunctionSig _glBlendEquationSeparateEXT_sig = {1739, "glBlendEquationSeparateEXT", 2, _glBlendEquationSeparateEXT_args};

static const char * _glBindFragDataLocationIndexedEXT_args[4] = {"program", "colorNumber", "index", "name"};
static const trace::FunctionSig _glBindFragDataLocationIndexedEXT_sig = {1740, "glBindFragDataLocationIndexedEXT", 4, _glBindFragDataLocationIndexedEXT_args};

static const char * _glBindFragDataLocationEXT_args[3] = {"program", "color", "name"};
static const trace::FunctionSig _glBindFragDataLocationEXT_sig = {1741, "glBindFragDataLocationEXT", 3, _glBindFragDataLocationEXT_args};

static const char * _glGetProgramResourceLocationIndexEXT_args[3] = {"program", "programInterface", "name"};
static const trace::FunctionSig _glGetProgramResourceLocationIndexEXT_sig = {1742, "glGetProgramResourceLocationIndexEXT", 3, _glGetProgramResourceLocationIndexEXT_args};

static const char * _glGetFragDataIndexEXT_args[2] = {"program", "name"};
static const trace::FunctionSig _glGetFragDataIndexEXT_sig = {1743, "glGetFragDataIndexEXT", 2, _glGetFragDataIndexEXT_args};

static const char * _glBlendFuncSeparateEXT_args[4] = {"sfactorRGB", "dfactorRGB", "sfactorAlpha", "dfactorAlpha"};
static const trace::FunctionSig _glBlendFuncSeparateEXT_sig = {1744, "glBlendFuncSeparateEXT", 4, _glBlendFuncSeparateEXT_args};

static const char * _glBlendEquationEXT_args[1] = {"mode"};
static const trace::FunctionSig _glBlendEquationEXT_sig = {1745, "glBlendEquationEXT", 1, _glBlendEquationEXT_args};

static const char * _glBufferStorageEXT_args[4] = {"target", "size", "data", "flags"};
static const trace::FunctionSig _glBufferStorageEXT_sig = {1746, "glBufferStorageEXT", 4, _glBufferStorageEXT_args};

static const char * _glClearTexImageEXT_args[5] = {"texture", "level", "format", "type", "data"};
static const trace::FunctionSig _glClearTexImageEXT_sig = {1747, "glClearTexImageEXT", 5, _glClearTexImageEXT_args};

static const char * _glClearTexSubImageEXT_args[11] = {"texture", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "type", "data"};
static const trace::FunctionSig _glClearTexSubImageEXT_sig = {1748, "glClearTexSubImageEXT", 11, _glClearTexSubImageEXT_args};

static const char * _glClipControlEXT_args[2] = {"origin", "depth"};
static const trace::FunctionSig _glClipControlEXT_sig = {1749, "glClipControlEXT", 2, _glClipControlEXT_args};

static const char * _glColorSubTableEXT_args[6] = {"target", "start", "count", "format", "type", "data"};
static const trace::FunctionSig _glColorSubTableEXT_sig = {1750, "glColorSubTableEXT", 6, _glColorSubTableEXT_args};

static const char * _glCopyColorSubTableEXT_args[5] = {"target", "start", "x", "y", "width"};
static const trace::FunctionSig _glCopyColorSubTableEXT_sig = {1751, "glCopyColorSubTableEXT", 5, _glCopyColorSubTableEXT_args};

static const char * _glLockArraysEXT_args[2] = {"first", "count"};
static const trace::FunctionSig _glLockArraysEXT_sig = {1752, "glLockArraysEXT", 2, _glLockArraysEXT_args};

static const char ** _glUnlockArraysEXT_args = NULL;
static const trace::FunctionSig _glUnlockArraysEXT_sig = {1753, "glUnlockArraysEXT", 0, _glUnlockArraysEXT_args};

static const char * _glConvolutionFilter1DEXT_args[6] = {"target", "internalformat", "width", "format", "type", "image"};
static const trace::FunctionSig _glConvolutionFilter1DEXT_sig = {1754, "glConvolutionFilter1DEXT", 6, _glConvolutionFilter1DEXT_args};

static const char * _glConvolutionFilter2DEXT_args[7] = {"target", "internalformat", "width", "height", "format", "type", "image"};
static const trace::FunctionSig _glConvolutionFilter2DEXT_sig = {1755, "glConvolutionFilter2DEXT", 7, _glConvolutionFilter2DEXT_args};

static const char * _glConvolutionParameterfEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glConvolutionParameterfEXT_sig = {1756, "glConvolutionParameterfEXT", 3, _glConvolutionParameterfEXT_args};

static const char * _glConvolutionParameterfvEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glConvolutionParameterfvEXT_sig = {1757, "glConvolutionParameterfvEXT", 3, _glConvolutionParameterfvEXT_args};

static const char * _glConvolutionParameteriEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glConvolutionParameteriEXT_sig = {1758, "glConvolutionParameteriEXT", 3, _glConvolutionParameteriEXT_args};

static const char * _glConvolutionParameterivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glConvolutionParameterivEXT_sig = {1759, "glConvolutionParameterivEXT", 3, _glConvolutionParameterivEXT_args};

static const char * _glCopyConvolutionFilter1DEXT_args[5] = {"target", "internalformat", "x", "y", "width"};
static const trace::FunctionSig _glCopyConvolutionFilter1DEXT_sig = {1760, "glCopyConvolutionFilter1DEXT", 5, _glCopyConvolutionFilter1DEXT_args};

static const char * _glCopyConvolutionFilter2DEXT_args[6] = {"target", "internalformat", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyConvolutionFilter2DEXT_sig = {1761, "glCopyConvolutionFilter2DEXT", 6, _glCopyConvolutionFilter2DEXT_args};

static const char * _glGetConvolutionFilterEXT_args[4] = {"target", "format", "type", "image"};
static const trace::FunctionSig _glGetConvolutionFilterEXT_sig = {1762, "glGetConvolutionFilterEXT", 4, _glGetConvolutionFilterEXT_args};

static const char * _glGetConvolutionParameterfvEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetConvolutionParameterfvEXT_sig = {1763, "glGetConvolutionParameterfvEXT", 3, _glGetConvolutionParameterfvEXT_args};

static const char * _glGetConvolutionParameterivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetConvolutionParameterivEXT_sig = {1764, "glGetConvolutionParameterivEXT", 3, _glGetConvolutionParameterivEXT_args};

static const char * _glGetSeparableFilterEXT_args[6] = {"target", "format", "type", "row", "column", "span"};
static const trace::FunctionSig _glGetSeparableFilterEXT_sig = {1765, "glGetSeparableFilterEXT", 6, _glGetSeparableFilterEXT_args};

static const char * _glSeparableFilter2DEXT_args[8] = {"target", "internalformat", "width", "height", "format", "type", "row", "column"};
static const trace::FunctionSig _glSeparableFilter2DEXT_sig = {1766, "glSeparableFilter2DEXT", 8, _glSeparableFilter2DEXT_args};

static const char * _glTangent3bEXT_args[3] = {"tx", "ty", "tz"};
static const trace::FunctionSig _glTangent3bEXT_sig = {1767, "glTangent3bEXT", 3, _glTangent3bEXT_args};

static const char * _glTangent3bvEXT_args[1] = {"v"};
static const trace::FunctionSig _glTangent3bvEXT_sig = {1768, "glTangent3bvEXT", 1, _glTangent3bvEXT_args};

static const char * _glTangent3dEXT_args[3] = {"tx", "ty", "tz"};
static const trace::FunctionSig _glTangent3dEXT_sig = {1769, "glTangent3dEXT", 3, _glTangent3dEXT_args};

static const char * _glTangent3dvEXT_args[1] = {"v"};
static const trace::FunctionSig _glTangent3dvEXT_sig = {1770, "glTangent3dvEXT", 1, _glTangent3dvEXT_args};

static const char * _glTangent3fEXT_args[3] = {"tx", "ty", "tz"};
static const trace::FunctionSig _glTangent3fEXT_sig = {1771, "glTangent3fEXT", 3, _glTangent3fEXT_args};

static const char * _glTangent3fvEXT_args[1] = {"v"};
static const trace::FunctionSig _glTangent3fvEXT_sig = {1772, "glTangent3fvEXT", 1, _glTangent3fvEXT_args};

static const char * _glTangent3iEXT_args[3] = {"tx", "ty", "tz"};
static const trace::FunctionSig _glTangent3iEXT_sig = {1773, "glTangent3iEXT", 3, _glTangent3iEXT_args};

static const char * _glTangent3ivEXT_args[1] = {"v"};
static const trace::FunctionSig _glTangent3ivEXT_sig = {1774, "glTangent3ivEXT", 1, _glTangent3ivEXT_args};

static const char * _glTangent3sEXT_args[3] = {"tx", "ty", "tz"};
static const trace::FunctionSig _glTangent3sEXT_sig = {1775, "glTangent3sEXT", 3, _glTangent3sEXT_args};

static const char * _glTangent3svEXT_args[1] = {"v"};
static const trace::FunctionSig _glTangent3svEXT_sig = {1776, "glTangent3svEXT", 1, _glTangent3svEXT_args};

static const char * _glBinormal3bEXT_args[3] = {"bx", "by", "bz"};
static const trace::FunctionSig _glBinormal3bEXT_sig = {1777, "glBinormal3bEXT", 3, _glBinormal3bEXT_args};

static const char * _glBinormal3bvEXT_args[1] = {"v"};
static const trace::FunctionSig _glBinormal3bvEXT_sig = {1778, "glBinormal3bvEXT", 1, _glBinormal3bvEXT_args};

static const char * _glBinormal3dEXT_args[3] = {"bx", "by", "bz"};
static const trace::FunctionSig _glBinormal3dEXT_sig = {1779, "glBinormal3dEXT", 3, _glBinormal3dEXT_args};

static const char * _glBinormal3dvEXT_args[1] = {"v"};
static const trace::FunctionSig _glBinormal3dvEXT_sig = {1780, "glBinormal3dvEXT", 1, _glBinormal3dvEXT_args};

static const char * _glBinormal3fEXT_args[3] = {"bx", "by", "bz"};
static const trace::FunctionSig _glBinormal3fEXT_sig = {1781, "glBinormal3fEXT", 3, _glBinormal3fEXT_args};

static const char * _glBinormal3fvEXT_args[1] = {"v"};
static const trace::FunctionSig _glBinormal3fvEXT_sig = {1782, "glBinormal3fvEXT", 1, _glBinormal3fvEXT_args};

static const char * _glBinormal3iEXT_args[3] = {"bx", "by", "bz"};
static const trace::FunctionSig _glBinormal3iEXT_sig = {1783, "glBinormal3iEXT", 3, _glBinormal3iEXT_args};

static const char * _glBinormal3ivEXT_args[1] = {"v"};
static const trace::FunctionSig _glBinormal3ivEXT_sig = {1784, "glBinormal3ivEXT", 1, _glBinormal3ivEXT_args};

static const char * _glBinormal3sEXT_args[3] = {"bx", "by", "bz"};
static const trace::FunctionSig _glBinormal3sEXT_sig = {1785, "glBinormal3sEXT", 3, _glBinormal3sEXT_args};

static const char * _glBinormal3svEXT_args[1] = {"v"};
static const trace::FunctionSig _glBinormal3svEXT_sig = {1786, "glBinormal3svEXT", 1, _glBinormal3svEXT_args};

static const char * _glTangentPointerEXT_args[3] = {"type", "stride", "pointer"};
static const trace::FunctionSig _glTangentPointerEXT_sig = {1787, "glTangentPointerEXT", 3, _glTangentPointerEXT_args};

static const char * _glBinormalPointerEXT_args[3] = {"type", "stride", "pointer"};
static const trace::FunctionSig _glBinormalPointerEXT_sig = {1788, "glBinormalPointerEXT", 3, _glBinormalPointerEXT_args};

static const char * _glCopyImageSubDataEXT_args[15] = {"srcName", "srcTarget", "srcLevel", "srcX", "srcY", "srcZ", "dstName", "dstTarget", "dstLevel", "dstX", "dstY", "dstZ", "srcWidth", "srcHeight", "srcDepth"};
static const trace::FunctionSig _glCopyImageSubDataEXT_sig = {1789, "glCopyImageSubDataEXT", 15, _glCopyImageSubDataEXT_args};

static const char * _glCopyTexImage1DEXT_args[7] = {"target", "level", "internalformat", "x", "y", "width", "border"};
static const trace::FunctionSig _glCopyTexImage1DEXT_sig = {1790, "glCopyTexImage1DEXT", 7, _glCopyTexImage1DEXT_args};

static const char * _glCopyTexImage2DEXT_args[8] = {"target", "level", "internalformat", "x", "y", "width", "height", "border"};
static const trace::FunctionSig _glCopyTexImage2DEXT_sig = {1791, "glCopyTexImage2DEXT", 8, _glCopyTexImage2DEXT_args};

static const char * _glCopyTexSubImage1DEXT_args[6] = {"target", "level", "xoffset", "x", "y", "width"};
static const trace::FunctionSig _glCopyTexSubImage1DEXT_sig = {1792, "glCopyTexSubImage1DEXT", 6, _glCopyTexSubImage1DEXT_args};

static const char * _glCopyTexSubImage2DEXT_args[8] = {"target", "level", "xoffset", "yoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyTexSubImage2DEXT_sig = {1793, "glCopyTexSubImage2DEXT", 8, _glCopyTexSubImage2DEXT_args};

static const char * _glCopyTexSubImage3DEXT_args[9] = {"target", "level", "xoffset", "yoffset", "zoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyTexSubImage3DEXT_sig = {1794, "glCopyTexSubImage3DEXT", 9, _glCopyTexSubImage3DEXT_args};

static const char * _glCullParameterdvEXT_args[2] = {"pname", "params"};
static const trace::FunctionSig _glCullParameterdvEXT_sig = {1795, "glCullParameterdvEXT", 2, _glCullParameterdvEXT_args};

static const char * _glCullParameterfvEXT_args[2] = {"pname", "params"};
static const trace::FunctionSig _glCullParameterfvEXT_sig = {1796, "glCullParameterfvEXT", 2, _glCullParameterfvEXT_args};

static const char * _glLabelObjectEXT_args[4] = {"type", "object", "length", "label"};
static const trace::FunctionSig _glLabelObjectEXT_sig = {1797, "glLabelObjectEXT", 4, _glLabelObjectEXT_args};

static const char * _glGetObjectLabelEXT_args[5] = {"type", "object", "bufSize", "length", "label"};
static const trace::FunctionSig _glGetObjectLabelEXT_sig = {1798, "glGetObjectLabelEXT", 5, _glGetObjectLabelEXT_args};

static const char * _glInsertEventMarkerEXT_args[2] = {"length", "marker"};
static const trace::FunctionSig _glInsertEventMarkerEXT_sig = {1799, "glInsertEventMarkerEXT", 2, _glInsertEventMarkerEXT_args};

static const char * _glPushGroupMarkerEXT_args[2] = {"length", "marker"};
static const trace::FunctionSig _glPushGroupMarkerEXT_sig = {1800, "glPushGroupMarkerEXT", 2, _glPushGroupMarkerEXT_args};

static const char ** _glPopGroupMarkerEXT_args = NULL;
static const trace::FunctionSig _glPopGroupMarkerEXT_sig = {1801, "glPopGroupMarkerEXT", 0, _glPopGroupMarkerEXT_args};

static const char * _glDepthBoundsEXT_args[2] = {"zmin", "zmax"};
static const trace::FunctionSig _glDepthBoundsEXT_sig = {1802, "glDepthBoundsEXT", 2, _glDepthBoundsEXT_args};

static const char * _glBindMultiTextureEXT_args[3] = {"texunit", "target", "texture"};
static const trace::FunctionSig _glBindMultiTextureEXT_sig = {1803, "glBindMultiTextureEXT", 3, _glBindMultiTextureEXT_args};

static const char * _glCheckNamedFramebufferStatusEXT_args[2] = {"framebuffer", "target"};
static const trace::FunctionSig _glCheckNamedFramebufferStatusEXT_sig = {1804, "glCheckNamedFramebufferStatusEXT", 2, _glCheckNamedFramebufferStatusEXT_args};

static const char * _glClearNamedBufferDataEXT_args[5] = {"buffer", "internalformat", "format", "type", "data"};
static const trace::FunctionSig _glClearNamedBufferDataEXT_sig = {1805, "glClearNamedBufferDataEXT", 5, _glClearNamedBufferDataEXT_args};

static const char * _glClearNamedBufferSubDataEXT_args[7] = {"buffer", "internalformat", "offset", "size", "format", "type", "data"};
static const trace::FunctionSig _glClearNamedBufferSubDataEXT_sig = {1806, "glClearNamedBufferSubDataEXT", 7, _glClearNamedBufferSubDataEXT_args};

static const char * _glClientAttribDefaultEXT_args[1] = {"mask"};
static const trace::FunctionSig _glClientAttribDefaultEXT_sig = {1807, "glClientAttribDefaultEXT", 1, _glClientAttribDefaultEXT_args};

static const char * _glCompressedMultiTexImage1DEXT_args[8] = {"texunit", "target", "level", "internalformat", "width", "border", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedMultiTexImage1DEXT_sig = {1808, "glCompressedMultiTexImage1DEXT", 8, _glCompressedMultiTexImage1DEXT_args};

static const char * _glCompressedMultiTexImage2DEXT_args[9] = {"texunit", "target", "level", "internalformat", "width", "height", "border", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedMultiTexImage2DEXT_sig = {1809, "glCompressedMultiTexImage2DEXT", 9, _glCompressedMultiTexImage2DEXT_args};

static const char * _glCompressedMultiTexImage3DEXT_args[10] = {"texunit", "target", "level", "internalformat", "width", "height", "depth", "border", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedMultiTexImage3DEXT_sig = {1810, "glCompressedMultiTexImage3DEXT", 10, _glCompressedMultiTexImage3DEXT_args};

static const char * _glCompressedMultiTexSubImage1DEXT_args[8] = {"texunit", "target", "level", "xoffset", "width", "format", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedMultiTexSubImage1DEXT_sig = {1811, "glCompressedMultiTexSubImage1DEXT", 8, _glCompressedMultiTexSubImage1DEXT_args};

static const char * _glCompressedMultiTexSubImage2DEXT_args[10] = {"texunit", "target", "level", "xoffset", "yoffset", "width", "height", "format", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedMultiTexSubImage2DEXT_sig = {1812, "glCompressedMultiTexSubImage2DEXT", 10, _glCompressedMultiTexSubImage2DEXT_args};

static const char * _glCompressedMultiTexSubImage3DEXT_args[12] = {"texunit", "target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedMultiTexSubImage3DEXT_sig = {1813, "glCompressedMultiTexSubImage3DEXT", 12, _glCompressedMultiTexSubImage3DEXT_args};

static const char * _glCompressedTextureImage1DEXT_args[8] = {"texture", "target", "level", "internalformat", "width", "border", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedTextureImage1DEXT_sig = {1814, "glCompressedTextureImage1DEXT", 8, _glCompressedTextureImage1DEXT_args};

static const char * _glCompressedTextureImage2DEXT_args[9] = {"texture", "target", "level", "internalformat", "width", "height", "border", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedTextureImage2DEXT_sig = {1815, "glCompressedTextureImage2DEXT", 9, _glCompressedTextureImage2DEXT_args};

static const char * _glCompressedTextureImage3DEXT_args[10] = {"texture", "target", "level", "internalformat", "width", "height", "depth", "border", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedTextureImage3DEXT_sig = {1816, "glCompressedTextureImage3DEXT", 10, _glCompressedTextureImage3DEXT_args};

static const char * _glCompressedTextureSubImage1DEXT_args[8] = {"texture", "target", "level", "xoffset", "width", "format", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedTextureSubImage1DEXT_sig = {1817, "glCompressedTextureSubImage1DEXT", 8, _glCompressedTextureSubImage1DEXT_args};

static const char * _glCompressedTextureSubImage2DEXT_args[10] = {"texture", "target", "level", "xoffset", "yoffset", "width", "height", "format", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedTextureSubImage2DEXT_sig = {1818, "glCompressedTextureSubImage2DEXT", 10, _glCompressedTextureSubImage2DEXT_args};

static const char * _glCompressedTextureSubImage3DEXT_args[12] = {"texture", "target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "imageSize", "bits"};
static const trace::FunctionSig _glCompressedTextureSubImage3DEXT_sig = {1819, "glCompressedTextureSubImage3DEXT", 12, _glCompressedTextureSubImage3DEXT_args};

static const char * _glCopyMultiTexImage1DEXT_args[8] = {"texunit", "target", "level", "internalformat", "x", "y", "width", "border"};
static const trace::FunctionSig _glCopyMultiTexImage1DEXT_sig = {1820, "glCopyMultiTexImage1DEXT", 8, _glCopyMultiTexImage1DEXT_args};

static const char * _glCopyMultiTexImage2DEXT_args[9] = {"texunit", "target", "level", "internalformat", "x", "y", "width", "height", "border"};
static const trace::FunctionSig _glCopyMultiTexImage2DEXT_sig = {1821, "glCopyMultiTexImage2DEXT", 9, _glCopyMultiTexImage2DEXT_args};

static const char * _glCopyMultiTexSubImage1DEXT_args[7] = {"texunit", "target", "level", "xoffset", "x", "y", "width"};
static const trace::FunctionSig _glCopyMultiTexSubImage1DEXT_sig = {1822, "glCopyMultiTexSubImage1DEXT", 7, _glCopyMultiTexSubImage1DEXT_args};

static const char * _glCopyMultiTexSubImage2DEXT_args[9] = {"texunit", "target", "level", "xoffset", "yoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyMultiTexSubImage2DEXT_sig = {1823, "glCopyMultiTexSubImage2DEXT", 9, _glCopyMultiTexSubImage2DEXT_args};

static const char * _glCopyMultiTexSubImage3DEXT_args[10] = {"texunit", "target", "level", "xoffset", "yoffset", "zoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyMultiTexSubImage3DEXT_sig = {1824, "glCopyMultiTexSubImage3DEXT", 10, _glCopyMultiTexSubImage3DEXT_args};

static const char * _glCopyTextureImage1DEXT_args[8] = {"texture", "target", "level", "internalformat", "x", "y", "width", "border"};
static const trace::FunctionSig _glCopyTextureImage1DEXT_sig = {1825, "glCopyTextureImage1DEXT", 8, _glCopyTextureImage1DEXT_args};

static const char * _glCopyTextureImage2DEXT_args[9] = {"texture", "target", "level", "internalformat", "x", "y", "width", "height", "border"};
static const trace::FunctionSig _glCopyTextureImage2DEXT_sig = {1826, "glCopyTextureImage2DEXT", 9, _glCopyTextureImage2DEXT_args};

static const char * _glCopyTextureSubImage1DEXT_args[7] = {"texture", "target", "level", "xoffset", "x", "y", "width"};
static const trace::FunctionSig _glCopyTextureSubImage1DEXT_sig = {1827, "glCopyTextureSubImage1DEXT", 7, _glCopyTextureSubImage1DEXT_args};

static const char * _glCopyTextureSubImage2DEXT_args[9] = {"texture", "target", "level", "xoffset", "yoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyTextureSubImage2DEXT_sig = {1828, "glCopyTextureSubImage2DEXT", 9, _glCopyTextureSubImage2DEXT_args};

static const char * _glCopyTextureSubImage3DEXT_args[10] = {"texture", "target", "level", "xoffset", "yoffset", "zoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyTextureSubImage3DEXT_sig = {1829, "glCopyTextureSubImage3DEXT", 10, _glCopyTextureSubImage3DEXT_args};

static const char * _glDisableClientStateIndexedEXT_args[2] = {"array", "index"};
static const trace::FunctionSig _glDisableClientStateIndexedEXT_sig = {1830, "glDisableClientStateIndexedEXT", 2, _glDisableClientStateIndexedEXT_args};

static const char * _glDisableClientStateiEXT_args[2] = {"array", "index"};
static const trace::FunctionSig _glDisableClientStateiEXT_sig = {1831, "glDisableClientStateiEXT", 2, _glDisableClientStateiEXT_args};

static const char * _glDisableVertexArrayAttribEXT_args[2] = {"vaobj", "index"};
static const trace::FunctionSig _glDisableVertexArrayAttribEXT_sig = {1832, "glDisableVertexArrayAttribEXT", 2, _glDisableVertexArrayAttribEXT_args};

static const char * _glDisableVertexArrayEXT_args[2] = {"vaobj", "array"};
static const trace::FunctionSig _glDisableVertexArrayEXT_sig = {1833, "glDisableVertexArrayEXT", 2, _glDisableVertexArrayEXT_args};

static const char * _glEnableClientStateIndexedEXT_args[2] = {"array", "index"};
static const trace::FunctionSig _glEnableClientStateIndexedEXT_sig = {1834, "glEnableClientStateIndexedEXT", 2, _glEnableClientStateIndexedEXT_args};

static const char * _glEnableClientStateiEXT_args[2] = {"array", "index"};
static const trace::FunctionSig _glEnableClientStateiEXT_sig = {1835, "glEnableClientStateiEXT", 2, _glEnableClientStateiEXT_args};

static const char * _glEnableVertexArrayAttribEXT_args[2] = {"vaobj", "index"};
static const trace::FunctionSig _glEnableVertexArrayAttribEXT_sig = {1836, "glEnableVertexArrayAttribEXT", 2, _glEnableVertexArrayAttribEXT_args};

static const char * _glEnableVertexArrayEXT_args[2] = {"vaobj", "array"};
static const trace::FunctionSig _glEnableVertexArrayEXT_sig = {1837, "glEnableVertexArrayEXT", 2, _glEnableVertexArrayEXT_args};

static const char * _glFlushMappedNamedBufferRangeEXT_args[3] = {"buffer", "offset", "length"};
static const trace::FunctionSig _glFlushMappedNamedBufferRangeEXT_sig = {1838, "glFlushMappedNamedBufferRangeEXT", 3, _glFlushMappedNamedBufferRangeEXT_args};

static const char * _glFramebufferDrawBufferEXT_args[2] = {"framebuffer", "mode"};
static const trace::FunctionSig _glFramebufferDrawBufferEXT_sig = {1839, "glFramebufferDrawBufferEXT", 2, _glFramebufferDrawBufferEXT_args};

static const char * _glFramebufferDrawBuffersEXT_args[3] = {"framebuffer", "n", "bufs"};
static const trace::FunctionSig _glFramebufferDrawBuffersEXT_sig = {1840, "glFramebufferDrawBuffersEXT", 3, _glFramebufferDrawBuffersEXT_args};

static const char * _glFramebufferReadBufferEXT_args[2] = {"framebuffer", "mode"};
static const trace::FunctionSig _glFramebufferReadBufferEXT_sig = {1841, "glFramebufferReadBufferEXT", 2, _glFramebufferReadBufferEXT_args};

static const char * _glGenerateMultiTexMipmapEXT_args[2] = {"texunit", "target"};
static const trace::FunctionSig _glGenerateMultiTexMipmapEXT_sig = {1842, "glGenerateMultiTexMipmapEXT", 2, _glGenerateMultiTexMipmapEXT_args};

static const char * _glGenerateTextureMipmapEXT_args[2] = {"texture", "target"};
static const trace::FunctionSig _glGenerateTextureMipmapEXT_sig = {1843, "glGenerateTextureMipmapEXT", 2, _glGenerateTextureMipmapEXT_args};

static const char * _glGetCompressedMultiTexImageEXT_args[4] = {"texunit", "target", "lod", "img"};
static const trace::FunctionSig _glGetCompressedMultiTexImageEXT_sig = {1844, "glGetCompressedMultiTexImageEXT", 4, _glGetCompressedMultiTexImageEXT_args};

static const char * _glGetCompressedTextureImageEXT_args[4] = {"texture", "target", "lod", "img"};
static const trace::FunctionSig _glGetCompressedTextureImageEXT_sig = {1845, "glGetCompressedTextureImageEXT", 4, _glGetCompressedTextureImageEXT_args};

static const char * _glGetDoubleIndexedvEXT_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetDoubleIndexedvEXT_sig = {1846, "glGetDoubleIndexedvEXT", 3, _glGetDoubleIndexedvEXT_args};

static const char * _glGetDoublei_vEXT_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetDoublei_vEXT_sig = {1847, "glGetDoublei_vEXT", 3, _glGetDoublei_vEXT_args};

static const char * _glGetFloatIndexedvEXT_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetFloatIndexedvEXT_sig = {1848, "glGetFloatIndexedvEXT", 3, _glGetFloatIndexedvEXT_args};

static const char * _glGetFloati_vEXT_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetFloati_vEXT_sig = {1849, "glGetFloati_vEXT", 3, _glGetFloati_vEXT_args};

static const char * _glGetFramebufferParameterivEXT_args[3] = {"framebuffer", "pname", "params"};
static const trace::FunctionSig _glGetFramebufferParameterivEXT_sig = {1850, "glGetFramebufferParameterivEXT", 3, _glGetFramebufferParameterivEXT_args};

static const char * _glGetMultiTexEnvfvEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexEnvfvEXT_sig = {1851, "glGetMultiTexEnvfvEXT", 4, _glGetMultiTexEnvfvEXT_args};

static const char * _glGetMultiTexEnvivEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexEnvivEXT_sig = {1852, "glGetMultiTexEnvivEXT", 4, _glGetMultiTexEnvivEXT_args};

static const char * _glGetMultiTexGendvEXT_args[4] = {"texunit", "coord", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexGendvEXT_sig = {1853, "glGetMultiTexGendvEXT", 4, _glGetMultiTexGendvEXT_args};

static const char * _glGetMultiTexGenfvEXT_args[4] = {"texunit", "coord", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexGenfvEXT_sig = {1854, "glGetMultiTexGenfvEXT", 4, _glGetMultiTexGenfvEXT_args};

static const char * _glGetMultiTexGenivEXT_args[4] = {"texunit", "coord", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexGenivEXT_sig = {1855, "glGetMultiTexGenivEXT", 4, _glGetMultiTexGenivEXT_args};

static const char * _glGetMultiTexImageEXT_args[6] = {"texunit", "target", "level", "format", "type", "pixels"};
static const trace::FunctionSig _glGetMultiTexImageEXT_sig = {1856, "glGetMultiTexImageEXT", 6, _glGetMultiTexImageEXT_args};

static const char * _glGetMultiTexLevelParameterfvEXT_args[5] = {"texunit", "target", "level", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexLevelParameterfvEXT_sig = {1857, "glGetMultiTexLevelParameterfvEXT", 5, _glGetMultiTexLevelParameterfvEXT_args};

static const char * _glGetMultiTexLevelParameterivEXT_args[5] = {"texunit", "target", "level", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexLevelParameterivEXT_sig = {1858, "glGetMultiTexLevelParameterivEXT", 5, _glGetMultiTexLevelParameterivEXT_args};

static const char * _glGetMultiTexParameterIivEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexParameterIivEXT_sig = {1859, "glGetMultiTexParameterIivEXT", 4, _glGetMultiTexParameterIivEXT_args};

static const char * _glGetMultiTexParameterIuivEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexParameterIuivEXT_sig = {1860, "glGetMultiTexParameterIuivEXT", 4, _glGetMultiTexParameterIuivEXT_args};

static const char * _glGetMultiTexParameterfvEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexParameterfvEXT_sig = {1861, "glGetMultiTexParameterfvEXT", 4, _glGetMultiTexParameterfvEXT_args};

static const char * _glGetMultiTexParameterivEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glGetMultiTexParameterivEXT_sig = {1862, "glGetMultiTexParameterivEXT", 4, _glGetMultiTexParameterivEXT_args};

static const char * _glGetNamedBufferParameterivEXT_args[3] = {"buffer", "pname", "params"};
static const trace::FunctionSig _glGetNamedBufferParameterivEXT_sig = {1863, "glGetNamedBufferParameterivEXT", 3, _glGetNamedBufferParameterivEXT_args};

static const char * _glGetNamedBufferPointervEXT_args[3] = {"buffer", "pname", "params"};
static const trace::FunctionSig _glGetNamedBufferPointervEXT_sig = {1864, "glGetNamedBufferPointervEXT", 3, _glGetNamedBufferPointervEXT_args};

static const char * _glGetNamedBufferSubDataEXT_args[4] = {"buffer", "offset", "size", "data"};
static const trace::FunctionSig _glGetNamedBufferSubDataEXT_sig = {1865, "glGetNamedBufferSubDataEXT", 4, _glGetNamedBufferSubDataEXT_args};

static const char * _glGetNamedFramebufferAttachmentParameterivEXT_args[4] = {"framebuffer", "attachment", "pname", "params"};
static const trace::FunctionSig _glGetNamedFramebufferAttachmentParameterivEXT_sig = {1866, "glGetNamedFramebufferAttachmentParameterivEXT", 4, _glGetNamedFramebufferAttachmentParameterivEXT_args};

static const char * _glGetNamedFramebufferParameterivEXT_args[3] = {"framebuffer", "pname", "params"};
static const trace::FunctionSig _glGetNamedFramebufferParameterivEXT_sig = {1867, "glGetNamedFramebufferParameterivEXT", 3, _glGetNamedFramebufferParameterivEXT_args};

static const char * _glGetNamedProgramLocalParameterIivEXT_args[4] = {"program", "target", "index", "params"};
static const trace::FunctionSig _glGetNamedProgramLocalParameterIivEXT_sig = {1868, "glGetNamedProgramLocalParameterIivEXT", 4, _glGetNamedProgramLocalParameterIivEXT_args};

static const char * _glGetNamedProgramLocalParameterIuivEXT_args[4] = {"program", "target", "index", "params"};
static const trace::FunctionSig _glGetNamedProgramLocalParameterIuivEXT_sig = {1869, "glGetNamedProgramLocalParameterIuivEXT", 4, _glGetNamedProgramLocalParameterIuivEXT_args};

static const char * _glGetNamedProgramLocalParameterdvEXT_args[4] = {"program", "target", "index", "params"};
static const trace::FunctionSig _glGetNamedProgramLocalParameterdvEXT_sig = {1870, "glGetNamedProgramLocalParameterdvEXT", 4, _glGetNamedProgramLocalParameterdvEXT_args};

static const char * _glGetNamedProgramLocalParameterfvEXT_args[4] = {"program", "target", "index", "params"};
static const trace::FunctionSig _glGetNamedProgramLocalParameterfvEXT_sig = {1871, "glGetNamedProgramLocalParameterfvEXT", 4, _glGetNamedProgramLocalParameterfvEXT_args};

static const char * _glGetNamedProgramStringEXT_args[4] = {"program", "target", "pname", "string"};
static const trace::FunctionSig _glGetNamedProgramStringEXT_sig = {1872, "glGetNamedProgramStringEXT", 4, _glGetNamedProgramStringEXT_args};

static const char * _glGetNamedProgramivEXT_args[4] = {"program", "target", "pname", "params"};
static const trace::FunctionSig _glGetNamedProgramivEXT_sig = {1873, "glGetNamedProgramivEXT", 4, _glGetNamedProgramivEXT_args};

static const char * _glGetNamedRenderbufferParameterivEXT_args[3] = {"renderbuffer", "pname", "params"};
static const trace::FunctionSig _glGetNamedRenderbufferParameterivEXT_sig = {1874, "glGetNamedRenderbufferParameterivEXT", 3, _glGetNamedRenderbufferParameterivEXT_args};

static const char * _glGetPointerIndexedvEXT_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetPointerIndexedvEXT_sig = {1875, "glGetPointerIndexedvEXT", 3, _glGetPointerIndexedvEXT_args};

static const char * _glGetPointeri_vEXT_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetPointeri_vEXT_sig = {1876, "glGetPointeri_vEXT", 3, _glGetPointeri_vEXT_args};

static const char * _glGetTextureImageEXT_args[6] = {"texture", "target", "level", "format", "type", "pixels"};
static const trace::FunctionSig _glGetTextureImageEXT_sig = {1877, "glGetTextureImageEXT", 6, _glGetTextureImageEXT_args};

static const char * _glGetTextureLevelParameterfvEXT_args[5] = {"texture", "target", "level", "pname", "params"};
static const trace::FunctionSig _glGetTextureLevelParameterfvEXT_sig = {1878, "glGetTextureLevelParameterfvEXT", 5, _glGetTextureLevelParameterfvEXT_args};

static const char * _glGetTextureLevelParameterivEXT_args[5] = {"texture", "target", "level", "pname", "params"};
static const trace::FunctionSig _glGetTextureLevelParameterivEXT_sig = {1879, "glGetTextureLevelParameterivEXT", 5, _glGetTextureLevelParameterivEXT_args};

static const char * _glGetTextureParameterIivEXT_args[4] = {"texture", "target", "pname", "params"};
static const trace::FunctionSig _glGetTextureParameterIivEXT_sig = {1880, "glGetTextureParameterIivEXT", 4, _glGetTextureParameterIivEXT_args};

static const char * _glGetTextureParameterIuivEXT_args[4] = {"texture", "target", "pname", "params"};
static const trace::FunctionSig _glGetTextureParameterIuivEXT_sig = {1881, "glGetTextureParameterIuivEXT", 4, _glGetTextureParameterIuivEXT_args};

static const char * _glGetTextureParameterfvEXT_args[4] = {"texture", "target", "pname", "params"};
static const trace::FunctionSig _glGetTextureParameterfvEXT_sig = {1882, "glGetTextureParameterfvEXT", 4, _glGetTextureParameterfvEXT_args};

static const char * _glGetTextureParameterivEXT_args[4] = {"texture", "target", "pname", "params"};
static const trace::FunctionSig _glGetTextureParameterivEXT_sig = {1883, "glGetTextureParameterivEXT", 4, _glGetTextureParameterivEXT_args};

static const char * _glGetVertexArrayIntegeri_vEXT_args[4] = {"vaobj", "index", "pname", "param"};
static const trace::FunctionSig _glGetVertexArrayIntegeri_vEXT_sig = {1884, "glGetVertexArrayIntegeri_vEXT", 4, _glGetVertexArrayIntegeri_vEXT_args};

static const char * _glGetVertexArrayIntegervEXT_args[3] = {"vaobj", "pname", "param"};
static const trace::FunctionSig _glGetVertexArrayIntegervEXT_sig = {1885, "glGetVertexArrayIntegervEXT", 3, _glGetVertexArrayIntegervEXT_args};

static const char * _glGetVertexArrayPointeri_vEXT_args[4] = {"vaobj", "index", "pname", "param"};
static const trace::FunctionSig _glGetVertexArrayPointeri_vEXT_sig = {1886, "glGetVertexArrayPointeri_vEXT", 4, _glGetVertexArrayPointeri_vEXT_args};

static const char * _glGetVertexArrayPointervEXT_args[3] = {"vaobj", "pname", "param"};
static const trace::FunctionSig _glGetVertexArrayPointervEXT_sig = {1887, "glGetVertexArrayPointervEXT", 3, _glGetVertexArrayPointervEXT_args};

static const char * _glMapNamedBufferEXT_args[2] = {"buffer", "access"};
static const trace::FunctionSig _glMapNamedBufferEXT_sig = {1888, "glMapNamedBufferEXT", 2, _glMapNamedBufferEXT_args};

static const char * _glMapNamedBufferRangeEXT_args[4] = {"buffer", "offset", "length", "access"};
static const trace::FunctionSig _glMapNamedBufferRangeEXT_sig = {1889, "glMapNamedBufferRangeEXT", 4, _glMapNamedBufferRangeEXT_args};

static const char * _glMatrixFrustumEXT_args[7] = {"mode", "left", "right", "bottom", "top", "zNear", "zFar"};
static const trace::FunctionSig _glMatrixFrustumEXT_sig = {1890, "glMatrixFrustumEXT", 7, _glMatrixFrustumEXT_args};

static const char * _glMatrixLoadIdentityEXT_args[1] = {"mode"};
static const trace::FunctionSig _glMatrixLoadIdentityEXT_sig = {1891, "glMatrixLoadIdentityEXT", 1, _glMatrixLoadIdentityEXT_args};

static const char * _glMatrixLoadTransposedEXT_args[2] = {"mode", "m"};
static const trace::FunctionSig _glMatrixLoadTransposedEXT_sig = {1892, "glMatrixLoadTransposedEXT", 2, _glMatrixLoadTransposedEXT_args};

static const char * _glMatrixLoadTransposefEXT_args[2] = {"mode", "m"};
static const trace::FunctionSig _glMatrixLoadTransposefEXT_sig = {1893, "glMatrixLoadTransposefEXT", 2, _glMatrixLoadTransposefEXT_args};

static const char * _glMatrixLoaddEXT_args[2] = {"mode", "m"};
static const trace::FunctionSig _glMatrixLoaddEXT_sig = {1894, "glMatrixLoaddEXT", 2, _glMatrixLoaddEXT_args};

static const char * _glMatrixLoadfEXT_args[2] = {"mode", "m"};
static const trace::FunctionSig _glMatrixLoadfEXT_sig = {1895, "glMatrixLoadfEXT", 2, _glMatrixLoadfEXT_args};

static const char * _glMatrixMultTransposedEXT_args[2] = {"mode", "m"};
static const trace::FunctionSig _glMatrixMultTransposedEXT_sig = {1896, "glMatrixMultTransposedEXT", 2, _glMatrixMultTransposedEXT_args};

static const char * _glMatrixMultTransposefEXT_args[2] = {"mode", "m"};
static const trace::FunctionSig _glMatrixMultTransposefEXT_sig = {1897, "glMatrixMultTransposefEXT", 2, _glMatrixMultTransposefEXT_args};

static const char * _glMatrixMultdEXT_args[2] = {"mode", "m"};
static const trace::FunctionSig _glMatrixMultdEXT_sig = {1898, "glMatrixMultdEXT", 2, _glMatrixMultdEXT_args};

static const char * _glMatrixMultfEXT_args[2] = {"mode", "m"};
static const trace::FunctionSig _glMatrixMultfEXT_sig = {1899, "glMatrixMultfEXT", 2, _glMatrixMultfEXT_args};

static const char * _glMatrixOrthoEXT_args[7] = {"mode", "left", "right", "bottom", "top", "zNear", "zFar"};
static const trace::FunctionSig _glMatrixOrthoEXT_sig = {1900, "glMatrixOrthoEXT", 7, _glMatrixOrthoEXT_args};

static const char * _glMatrixPopEXT_args[1] = {"mode"};
static const trace::FunctionSig _glMatrixPopEXT_sig = {1901, "glMatrixPopEXT", 1, _glMatrixPopEXT_args};

static const char * _glMatrixPushEXT_args[1] = {"mode"};
static const trace::FunctionSig _glMatrixPushEXT_sig = {1902, "glMatrixPushEXT", 1, _glMatrixPushEXT_args};

static const char * _glMatrixRotatedEXT_args[5] = {"mode", "angle", "x", "y", "z"};
static const trace::FunctionSig _glMatrixRotatedEXT_sig = {1903, "glMatrixRotatedEXT", 5, _glMatrixRotatedEXT_args};

static const char * _glMatrixRotatefEXT_args[5] = {"mode", "angle", "x", "y", "z"};
static const trace::FunctionSig _glMatrixRotatefEXT_sig = {1904, "glMatrixRotatefEXT", 5, _glMatrixRotatefEXT_args};

static const char * _glMatrixScaledEXT_args[4] = {"mode", "x", "y", "z"};
static const trace::FunctionSig _glMatrixScaledEXT_sig = {1905, "glMatrixScaledEXT", 4, _glMatrixScaledEXT_args};

static const char * _glMatrixScalefEXT_args[4] = {"mode", "x", "y", "z"};
static const trace::FunctionSig _glMatrixScalefEXT_sig = {1906, "glMatrixScalefEXT", 4, _glMatrixScalefEXT_args};

static const char * _glMatrixTranslatedEXT_args[4] = {"mode", "x", "y", "z"};
static const trace::FunctionSig _glMatrixTranslatedEXT_sig = {1907, "glMatrixTranslatedEXT", 4, _glMatrixTranslatedEXT_args};

static const char * _glMatrixTranslatefEXT_args[4] = {"mode", "x", "y", "z"};
static const trace::FunctionSig _glMatrixTranslatefEXT_sig = {1908, "glMatrixTranslatefEXT", 4, _glMatrixTranslatefEXT_args};

static const char * _glMultiTexBufferEXT_args[4] = {"texunit", "target", "internalformat", "buffer"};
static const trace::FunctionSig _glMultiTexBufferEXT_sig = {1909, "glMultiTexBufferEXT", 4, _glMultiTexBufferEXT_args};

static const char * _glMultiTexCoordPointerEXT_args[5] = {"texunit", "size", "type", "stride", "pointer"};
static const trace::FunctionSig _glMultiTexCoordPointerEXT_sig = {1910, "glMultiTexCoordPointerEXT", 5, _glMultiTexCoordPointerEXT_args};

static const char * _glMultiTexEnvfEXT_args[4] = {"texunit", "target", "pname", "param"};
static const trace::FunctionSig _glMultiTexEnvfEXT_sig = {1911, "glMultiTexEnvfEXT", 4, _glMultiTexEnvfEXT_args};

static const char * _glMultiTexEnvfvEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glMultiTexEnvfvEXT_sig = {1912, "glMultiTexEnvfvEXT", 4, _glMultiTexEnvfvEXT_args};

static const char * _glMultiTexEnviEXT_args[4] = {"texunit", "target", "pname", "param"};
static const trace::FunctionSig _glMultiTexEnviEXT_sig = {1913, "glMultiTexEnviEXT", 4, _glMultiTexEnviEXT_args};

static const char * _glMultiTexEnvivEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glMultiTexEnvivEXT_sig = {1914, "glMultiTexEnvivEXT", 4, _glMultiTexEnvivEXT_args};

static const char * _glMultiTexGendEXT_args[4] = {"texunit", "coord", "pname", "param"};
static const trace::FunctionSig _glMultiTexGendEXT_sig = {1915, "glMultiTexGendEXT", 4, _glMultiTexGendEXT_args};

static const char * _glMultiTexGendvEXT_args[4] = {"texunit", "coord", "pname", "params"};
static const trace::FunctionSig _glMultiTexGendvEXT_sig = {1916, "glMultiTexGendvEXT", 4, _glMultiTexGendvEXT_args};

static const char * _glMultiTexGenfEXT_args[4] = {"texunit", "coord", "pname", "param"};
static const trace::FunctionSig _glMultiTexGenfEXT_sig = {1917, "glMultiTexGenfEXT", 4, _glMultiTexGenfEXT_args};

static const char * _glMultiTexGenfvEXT_args[4] = {"texunit", "coord", "pname", "params"};
static const trace::FunctionSig _glMultiTexGenfvEXT_sig = {1918, "glMultiTexGenfvEXT", 4, _glMultiTexGenfvEXT_args};

static const char * _glMultiTexGeniEXT_args[4] = {"texunit", "coord", "pname", "param"};
static const trace::FunctionSig _glMultiTexGeniEXT_sig = {1919, "glMultiTexGeniEXT", 4, _glMultiTexGeniEXT_args};

static const char * _glMultiTexGenivEXT_args[4] = {"texunit", "coord", "pname", "params"};
static const trace::FunctionSig _glMultiTexGenivEXT_sig = {1920, "glMultiTexGenivEXT", 4, _glMultiTexGenivEXT_args};

static const char * _glMultiTexImage1DEXT_args[9] = {"texunit", "target", "level", "internalformat", "width", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glMultiTexImage1DEXT_sig = {1921, "glMultiTexImage1DEXT", 9, _glMultiTexImage1DEXT_args};

static const char * _glMultiTexImage2DEXT_args[10] = {"texunit", "target", "level", "internalformat", "width", "height", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glMultiTexImage2DEXT_sig = {1922, "glMultiTexImage2DEXT", 10, _glMultiTexImage2DEXT_args};

static const char * _glMultiTexImage3DEXT_args[11] = {"texunit", "target", "level", "internalformat", "width", "height", "depth", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glMultiTexImage3DEXT_sig = {1923, "glMultiTexImage3DEXT", 11, _glMultiTexImage3DEXT_args};

static const char * _glMultiTexParameterIivEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glMultiTexParameterIivEXT_sig = {1924, "glMultiTexParameterIivEXT", 4, _glMultiTexParameterIivEXT_args};

static const char * _glMultiTexParameterIuivEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glMultiTexParameterIuivEXT_sig = {1925, "glMultiTexParameterIuivEXT", 4, _glMultiTexParameterIuivEXT_args};

static const char * _glMultiTexParameterfEXT_args[4] = {"texunit", "target", "pname", "param"};
static const trace::FunctionSig _glMultiTexParameterfEXT_sig = {1926, "glMultiTexParameterfEXT", 4, _glMultiTexParameterfEXT_args};

static const char * _glMultiTexParameterfvEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glMultiTexParameterfvEXT_sig = {1927, "glMultiTexParameterfvEXT", 4, _glMultiTexParameterfvEXT_args};

static const char * _glMultiTexParameteriEXT_args[4] = {"texunit", "target", "pname", "param"};
static const trace::FunctionSig _glMultiTexParameteriEXT_sig = {1928, "glMultiTexParameteriEXT", 4, _glMultiTexParameteriEXT_args};

static const char * _glMultiTexParameterivEXT_args[4] = {"texunit", "target", "pname", "params"};
static const trace::FunctionSig _glMultiTexParameterivEXT_sig = {1929, "glMultiTexParameterivEXT", 4, _glMultiTexParameterivEXT_args};

static const char * _glMultiTexRenderbufferEXT_args[3] = {"texunit", "target", "renderbuffer"};
static const trace::FunctionSig _glMultiTexRenderbufferEXT_sig = {1930, "glMultiTexRenderbufferEXT", 3, _glMultiTexRenderbufferEXT_args};

static const char * _glMultiTexSubImage1DEXT_args[8] = {"texunit", "target", "level", "xoffset", "width", "format", "type", "pixels"};
static const trace::FunctionSig _glMultiTexSubImage1DEXT_sig = {1931, "glMultiTexSubImage1DEXT", 8, _glMultiTexSubImage1DEXT_args};

static const char * _glMultiTexSubImage2DEXT_args[10] = {"texunit", "target", "level", "xoffset", "yoffset", "width", "height", "format", "type", "pixels"};
static const trace::FunctionSig _glMultiTexSubImage2DEXT_sig = {1932, "glMultiTexSubImage2DEXT", 10, _glMultiTexSubImage2DEXT_args};

static const char * _glMultiTexSubImage3DEXT_args[12] = {"texunit", "target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "type", "pixels"};
static const trace::FunctionSig _glMultiTexSubImage3DEXT_sig = {1933, "glMultiTexSubImage3DEXT", 12, _glMultiTexSubImage3DEXT_args};

static const char * _glNamedBufferDataEXT_args[4] = {"buffer", "size", "data", "usage"};
static const trace::FunctionSig _glNamedBufferDataEXT_sig = {1934, "glNamedBufferDataEXT", 4, _glNamedBufferDataEXT_args};

static const char * _glNamedBufferStorageEXT_args[4] = {"buffer", "size", "data", "flags"};
static const trace::FunctionSig _glNamedBufferStorageEXT_sig = {1935, "glNamedBufferStorageEXT", 4, _glNamedBufferStorageEXT_args};

static const char * _glNamedBufferSubDataEXT_args[4] = {"buffer", "offset", "size", "data"};
static const trace::FunctionSig _glNamedBufferSubDataEXT_sig = {1936, "glNamedBufferSubDataEXT", 4, _glNamedBufferSubDataEXT_args};

static const char * _glNamedCopyBufferSubDataEXT_args[5] = {"readBuffer", "writeBuffer", "readOffset", "writeOffset", "size"};
static const trace::FunctionSig _glNamedCopyBufferSubDataEXT_sig = {1937, "glNamedCopyBufferSubDataEXT", 5, _glNamedCopyBufferSubDataEXT_args};

static const char * _glNamedFramebufferParameteriEXT_args[3] = {"framebuffer", "pname", "param"};
static const trace::FunctionSig _glNamedFramebufferParameteriEXT_sig = {1938, "glNamedFramebufferParameteriEXT", 3, _glNamedFramebufferParameteriEXT_args};

static const char * _glNamedFramebufferRenderbufferEXT_args[4] = {"framebuffer", "attachment", "renderbuffertarget", "renderbuffer"};
static const trace::FunctionSig _glNamedFramebufferRenderbufferEXT_sig = {1939, "glNamedFramebufferRenderbufferEXT", 4, _glNamedFramebufferRenderbufferEXT_args};

static const char * _glNamedFramebufferTexture1DEXT_args[5] = {"framebuffer", "attachment", "textarget", "texture", "level"};
static const trace::FunctionSig _glNamedFramebufferTexture1DEXT_sig = {1940, "glNamedFramebufferTexture1DEXT", 5, _glNamedFramebufferTexture1DEXT_args};

static const char * _glNamedFramebufferTexture2DEXT_args[5] = {"framebuffer", "attachment", "textarget", "texture", "level"};
static const trace::FunctionSig _glNamedFramebufferTexture2DEXT_sig = {1941, "glNamedFramebufferTexture2DEXT", 5, _glNamedFramebufferTexture2DEXT_args};

static const char * _glNamedFramebufferTexture3DEXT_args[6] = {"framebuffer", "attachment", "textarget", "texture", "level", "zoffset"};
static const trace::FunctionSig _glNamedFramebufferTexture3DEXT_sig = {1942, "glNamedFramebufferTexture3DEXT", 6, _glNamedFramebufferTexture3DEXT_args};

static const char * _glNamedFramebufferTextureEXT_args[4] = {"framebuffer", "attachment", "texture", "level"};
static const trace::FunctionSig _glNamedFramebufferTextureEXT_sig = {1943, "glNamedFramebufferTextureEXT", 4, _glNamedFramebufferTextureEXT_args};

static const char * _glNamedFramebufferTextureFaceEXT_args[5] = {"framebuffer", "attachment", "texture", "level", "face"};
static const trace::FunctionSig _glNamedFramebufferTextureFaceEXT_sig = {1944, "glNamedFramebufferTextureFaceEXT", 5, _glNamedFramebufferTextureFaceEXT_args};

static const char * _glNamedFramebufferTextureLayerEXT_args[5] = {"framebuffer", "attachment", "texture", "level", "layer"};
static const trace::FunctionSig _glNamedFramebufferTextureLayerEXT_sig = {1945, "glNamedFramebufferTextureLayerEXT", 5, _glNamedFramebufferTextureLayerEXT_args};

static const char * _glNamedProgramLocalParameter4dEXT_args[7] = {"program", "target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glNamedProgramLocalParameter4dEXT_sig = {1946, "glNamedProgramLocalParameter4dEXT", 7, _glNamedProgramLocalParameter4dEXT_args};

static const char * _glNamedProgramLocalParameter4dvEXT_args[4] = {"program", "target", "index", "params"};
static const trace::FunctionSig _glNamedProgramLocalParameter4dvEXT_sig = {1947, "glNamedProgramLocalParameter4dvEXT", 4, _glNamedProgramLocalParameter4dvEXT_args};

static const char * _glNamedProgramLocalParameter4fEXT_args[7] = {"program", "target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glNamedProgramLocalParameter4fEXT_sig = {1948, "glNamedProgramLocalParameter4fEXT", 7, _glNamedProgramLocalParameter4fEXT_args};

static const char * _glNamedProgramLocalParameter4fvEXT_args[4] = {"program", "target", "index", "params"};
static const trace::FunctionSig _glNamedProgramLocalParameter4fvEXT_sig = {1949, "glNamedProgramLocalParameter4fvEXT", 4, _glNamedProgramLocalParameter4fvEXT_args};

static const char * _glNamedProgramLocalParameterI4iEXT_args[7] = {"program", "target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glNamedProgramLocalParameterI4iEXT_sig = {1950, "glNamedProgramLocalParameterI4iEXT", 7, _glNamedProgramLocalParameterI4iEXT_args};

static const char * _glNamedProgramLocalParameterI4ivEXT_args[4] = {"program", "target", "index", "params"};
static const trace::FunctionSig _glNamedProgramLocalParameterI4ivEXT_sig = {1951, "glNamedProgramLocalParameterI4ivEXT", 4, _glNamedProgramLocalParameterI4ivEXT_args};

static const char * _glNamedProgramLocalParameterI4uiEXT_args[7] = {"program", "target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glNamedProgramLocalParameterI4uiEXT_sig = {1952, "glNamedProgramLocalParameterI4uiEXT", 7, _glNamedProgramLocalParameterI4uiEXT_args};

static const char * _glNamedProgramLocalParameterI4uivEXT_args[4] = {"program", "target", "index", "params"};
static const trace::FunctionSig _glNamedProgramLocalParameterI4uivEXT_sig = {1953, "glNamedProgramLocalParameterI4uivEXT", 4, _glNamedProgramLocalParameterI4uivEXT_args};

static const char * _glNamedProgramLocalParameters4fvEXT_args[5] = {"program", "target", "index", "count", "params"};
static const trace::FunctionSig _glNamedProgramLocalParameters4fvEXT_sig = {1954, "glNamedProgramLocalParameters4fvEXT", 5, _glNamedProgramLocalParameters4fvEXT_args};

static const char * _glNamedProgramLocalParametersI4ivEXT_args[5] = {"program", "target", "index", "count", "params"};
static const trace::FunctionSig _glNamedProgramLocalParametersI4ivEXT_sig = {1955, "glNamedProgramLocalParametersI4ivEXT", 5, _glNamedProgramLocalParametersI4ivEXT_args};

static const char * _glNamedProgramLocalParametersI4uivEXT_args[5] = {"program", "target", "index", "count", "params"};
static const trace::FunctionSig _glNamedProgramLocalParametersI4uivEXT_sig = {1956, "glNamedProgramLocalParametersI4uivEXT", 5, _glNamedProgramLocalParametersI4uivEXT_args};

static const char * _glNamedProgramStringEXT_args[5] = {"program", "target", "format", "len", "string"};
static const trace::FunctionSig _glNamedProgramStringEXT_sig = {1957, "glNamedProgramStringEXT", 5, _glNamedProgramStringEXT_args};

static const char * _glNamedRenderbufferStorageEXT_args[4] = {"renderbuffer", "internalformat", "width", "height"};
static const trace::FunctionSig _glNamedRenderbufferStorageEXT_sig = {1958, "glNamedRenderbufferStorageEXT", 4, _glNamedRenderbufferStorageEXT_args};

static const char * _glNamedRenderbufferStorageMultisampleCoverageEXT_args[6] = {"renderbuffer", "coverageSamples", "colorSamples", "internalformat", "width", "height"};
static const trace::FunctionSig _glNamedRenderbufferStorageMultisampleCoverageEXT_sig = {1959, "glNamedRenderbufferStorageMultisampleCoverageEXT", 6, _glNamedRenderbufferStorageMultisampleCoverageEXT_args};

static const char * _glNamedRenderbufferStorageMultisampleEXT_args[5] = {"renderbuffer", "samples", "internalformat", "width", "height"};
static const trace::FunctionSig _glNamedRenderbufferStorageMultisampleEXT_sig = {1960, "glNamedRenderbufferStorageMultisampleEXT", 5, _glNamedRenderbufferStorageMultisampleEXT_args};

static const char * _glProgramUniform1dEXT_args[3] = {"program", "location", "x"};
static const trace::FunctionSig _glProgramUniform1dEXT_sig = {1961, "glProgramUniform1dEXT", 3, _glProgramUniform1dEXT_args};

static const char * _glProgramUniform1dvEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1dvEXT_sig = {1962, "glProgramUniform1dvEXT", 4, _glProgramUniform1dvEXT_args};

static const char * _glProgramUniform1fEXT_args[3] = {"program", "location", "v0"};
static const trace::FunctionSig _glProgramUniform1fEXT_sig = {1963, "glProgramUniform1fEXT", 3, _glProgramUniform1fEXT_args};

static const char * _glProgramUniform1fvEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1fvEXT_sig = {1964, "glProgramUniform1fvEXT", 4, _glProgramUniform1fvEXT_args};

static const char * _glProgramUniform1iEXT_args[3] = {"program", "location", "v0"};
static const trace::FunctionSig _glProgramUniform1iEXT_sig = {1965, "glProgramUniform1iEXT", 3, _glProgramUniform1iEXT_args};

static const char * _glProgramUniform1ivEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1ivEXT_sig = {1966, "glProgramUniform1ivEXT", 4, _glProgramUniform1ivEXT_args};

static const char * _glProgramUniform1uiEXT_args[3] = {"program", "location", "v0"};
static const trace::FunctionSig _glProgramUniform1uiEXT_sig = {1967, "glProgramUniform1uiEXT", 3, _glProgramUniform1uiEXT_args};

static const char * _glProgramUniform1uivEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1uivEXT_sig = {1968, "glProgramUniform1uivEXT", 4, _glProgramUniform1uivEXT_args};

static const char * _glProgramUniform2dEXT_args[4] = {"program", "location", "x", "y"};
static const trace::FunctionSig _glProgramUniform2dEXT_sig = {1969, "glProgramUniform2dEXT", 4, _glProgramUniform2dEXT_args};

static const char * _glProgramUniform2dvEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2dvEXT_sig = {1970, "glProgramUniform2dvEXT", 4, _glProgramUniform2dvEXT_args};

static const char * _glProgramUniform2fEXT_args[4] = {"program", "location", "v0", "v1"};
static const trace::FunctionSig _glProgramUniform2fEXT_sig = {1971, "glProgramUniform2fEXT", 4, _glProgramUniform2fEXT_args};

static const char * _glProgramUniform2fvEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2fvEXT_sig = {1972, "glProgramUniform2fvEXT", 4, _glProgramUniform2fvEXT_args};

static const char * _glProgramUniform2iEXT_args[4] = {"program", "location", "v0", "v1"};
static const trace::FunctionSig _glProgramUniform2iEXT_sig = {1973, "glProgramUniform2iEXT", 4, _glProgramUniform2iEXT_args};

static const char * _glProgramUniform2ivEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2ivEXT_sig = {1974, "glProgramUniform2ivEXT", 4, _glProgramUniform2ivEXT_args};

static const char * _glProgramUniform2uiEXT_args[4] = {"program", "location", "v0", "v1"};
static const trace::FunctionSig _glProgramUniform2uiEXT_sig = {1975, "glProgramUniform2uiEXT", 4, _glProgramUniform2uiEXT_args};

static const char * _glProgramUniform2uivEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2uivEXT_sig = {1976, "glProgramUniform2uivEXT", 4, _glProgramUniform2uivEXT_args};

static const char * _glProgramUniform3dEXT_args[5] = {"program", "location", "x", "y", "z"};
static const trace::FunctionSig _glProgramUniform3dEXT_sig = {1977, "glProgramUniform3dEXT", 5, _glProgramUniform3dEXT_args};

static const char * _glProgramUniform3dvEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3dvEXT_sig = {1978, "glProgramUniform3dvEXT", 4, _glProgramUniform3dvEXT_args};

static const char * _glProgramUniform3fEXT_args[5] = {"program", "location", "v0", "v1", "v2"};
static const trace::FunctionSig _glProgramUniform3fEXT_sig = {1979, "glProgramUniform3fEXT", 5, _glProgramUniform3fEXT_args};

static const char * _glProgramUniform3fvEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3fvEXT_sig = {1980, "glProgramUniform3fvEXT", 4, _glProgramUniform3fvEXT_args};

static const char * _glProgramUniform3iEXT_args[5] = {"program", "location", "v0", "v1", "v2"};
static const trace::FunctionSig _glProgramUniform3iEXT_sig = {1981, "glProgramUniform3iEXT", 5, _glProgramUniform3iEXT_args};

static const char * _glProgramUniform3ivEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3ivEXT_sig = {1982, "glProgramUniform3ivEXT", 4, _glProgramUniform3ivEXT_args};

static const char * _glProgramUniform3uiEXT_args[5] = {"program", "location", "v0", "v1", "v2"};
static const trace::FunctionSig _glProgramUniform3uiEXT_sig = {1983, "glProgramUniform3uiEXT", 5, _glProgramUniform3uiEXT_args};

static const char * _glProgramUniform3uivEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3uivEXT_sig = {1984, "glProgramUniform3uivEXT", 4, _glProgramUniform3uivEXT_args};

static const char * _glProgramUniform4dEXT_args[6] = {"program", "location", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramUniform4dEXT_sig = {1985, "glProgramUniform4dEXT", 6, _glProgramUniform4dEXT_args};

static const char * _glProgramUniform4dvEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4dvEXT_sig = {1986, "glProgramUniform4dvEXT", 4, _glProgramUniform4dvEXT_args};

static const char * _glProgramUniform4fEXT_args[6] = {"program", "location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glProgramUniform4fEXT_sig = {1987, "glProgramUniform4fEXT", 6, _glProgramUniform4fEXT_args};

static const char * _glProgramUniform4fvEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4fvEXT_sig = {1988, "glProgramUniform4fvEXT", 4, _glProgramUniform4fvEXT_args};

static const char * _glProgramUniform4iEXT_args[6] = {"program", "location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glProgramUniform4iEXT_sig = {1989, "glProgramUniform4iEXT", 6, _glProgramUniform4iEXT_args};

static const char * _glProgramUniform4ivEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4ivEXT_sig = {1990, "glProgramUniform4ivEXT", 4, _glProgramUniform4ivEXT_args};

static const char * _glProgramUniform4uiEXT_args[6] = {"program", "location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glProgramUniform4uiEXT_sig = {1991, "glProgramUniform4uiEXT", 6, _glProgramUniform4uiEXT_args};

static const char * _glProgramUniform4uivEXT_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4uivEXT_sig = {1992, "glProgramUniform4uivEXT", 4, _glProgramUniform4uivEXT_args};

static const char * _glProgramUniformMatrix2dvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2dvEXT_sig = {1993, "glProgramUniformMatrix2dvEXT", 5, _glProgramUniformMatrix2dvEXT_args};

static const char * _glProgramUniformMatrix2fvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2fvEXT_sig = {1994, "glProgramUniformMatrix2fvEXT", 5, _glProgramUniformMatrix2fvEXT_args};

static const char * _glProgramUniformMatrix2x3dvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2x3dvEXT_sig = {1995, "glProgramUniformMatrix2x3dvEXT", 5, _glProgramUniformMatrix2x3dvEXT_args};

static const char * _glProgramUniformMatrix2x3fvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2x3fvEXT_sig = {1996, "glProgramUniformMatrix2x3fvEXT", 5, _glProgramUniformMatrix2x3fvEXT_args};

static const char * _glProgramUniformMatrix2x4dvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2x4dvEXT_sig = {1997, "glProgramUniformMatrix2x4dvEXT", 5, _glProgramUniformMatrix2x4dvEXT_args};

static const char * _glProgramUniformMatrix2x4fvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix2x4fvEXT_sig = {1998, "glProgramUniformMatrix2x4fvEXT", 5, _glProgramUniformMatrix2x4fvEXT_args};

static const char * _glProgramUniformMatrix3dvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3dvEXT_sig = {1999, "glProgramUniformMatrix3dvEXT", 5, _glProgramUniformMatrix3dvEXT_args};

static const char * _glProgramUniformMatrix3fvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3fvEXT_sig = {2000, "glProgramUniformMatrix3fvEXT", 5, _glProgramUniformMatrix3fvEXT_args};

static const char * _glProgramUniformMatrix3x2dvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3x2dvEXT_sig = {2001, "glProgramUniformMatrix3x2dvEXT", 5, _glProgramUniformMatrix3x2dvEXT_args};

static const char * _glProgramUniformMatrix3x2fvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3x2fvEXT_sig = {2002, "glProgramUniformMatrix3x2fvEXT", 5, _glProgramUniformMatrix3x2fvEXT_args};

static const char * _glProgramUniformMatrix3x4dvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3x4dvEXT_sig = {2003, "glProgramUniformMatrix3x4dvEXT", 5, _glProgramUniformMatrix3x4dvEXT_args};

static const char * _glProgramUniformMatrix3x4fvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix3x4fvEXT_sig = {2004, "glProgramUniformMatrix3x4fvEXT", 5, _glProgramUniformMatrix3x4fvEXT_args};

static const char * _glProgramUniformMatrix4dvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4dvEXT_sig = {2005, "glProgramUniformMatrix4dvEXT", 5, _glProgramUniformMatrix4dvEXT_args};

static const char * _glProgramUniformMatrix4fvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4fvEXT_sig = {2006, "glProgramUniformMatrix4fvEXT", 5, _glProgramUniformMatrix4fvEXT_args};

static const char * _glProgramUniformMatrix4x2dvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4x2dvEXT_sig = {2007, "glProgramUniformMatrix4x2dvEXT", 5, _glProgramUniformMatrix4x2dvEXT_args};

static const char * _glProgramUniformMatrix4x2fvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4x2fvEXT_sig = {2008, "glProgramUniformMatrix4x2fvEXT", 5, _glProgramUniformMatrix4x2fvEXT_args};

static const char * _glProgramUniformMatrix4x3dvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4x3dvEXT_sig = {2009, "glProgramUniformMatrix4x3dvEXT", 5, _glProgramUniformMatrix4x3dvEXT_args};

static const char * _glProgramUniformMatrix4x3fvEXT_args[5] = {"program", "location", "count", "transpose", "value"};
static const trace::FunctionSig _glProgramUniformMatrix4x3fvEXT_sig = {2010, "glProgramUniformMatrix4x3fvEXT", 5, _glProgramUniformMatrix4x3fvEXT_args};

static const char * _glPushClientAttribDefaultEXT_args[1] = {"mask"};
static const trace::FunctionSig _glPushClientAttribDefaultEXT_sig = {2011, "glPushClientAttribDefaultEXT", 1, _glPushClientAttribDefaultEXT_args};

static const char * _glTextureBufferEXT_args[4] = {"texture", "target", "internalformat", "buffer"};
static const trace::FunctionSig _glTextureBufferEXT_sig = {2012, "glTextureBufferEXT", 4, _glTextureBufferEXT_args};

static const char * _glTextureBufferRangeEXT_args[6] = {"texture", "target", "internalformat", "buffer", "offset", "size"};
static const trace::FunctionSig _glTextureBufferRangeEXT_sig = {2013, "glTextureBufferRangeEXT", 6, _glTextureBufferRangeEXT_args};

static const char * _glTextureImage1DEXT_args[9] = {"texture", "target", "level", "internalformat", "width", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glTextureImage1DEXT_sig = {2014, "glTextureImage1DEXT", 9, _glTextureImage1DEXT_args};

static const char * _glTextureImage2DEXT_args[10] = {"texture", "target", "level", "internalformat", "width", "height", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glTextureImage2DEXT_sig = {2015, "glTextureImage2DEXT", 10, _glTextureImage2DEXT_args};

static const char * _glTextureImage3DEXT_args[11] = {"texture", "target", "level", "internalformat", "width", "height", "depth", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glTextureImage3DEXT_sig = {2016, "glTextureImage3DEXT", 11, _glTextureImage3DEXT_args};

static const char * _glTexturePageCommitmentEXT_args[9] = {"texture", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "commit"};
static const trace::FunctionSig _glTexturePageCommitmentEXT_sig = {2017, "glTexturePageCommitmentEXT", 9, _glTexturePageCommitmentEXT_args};

static const char * _glTextureParameterIivEXT_args[4] = {"texture", "target", "pname", "params"};
static const trace::FunctionSig _glTextureParameterIivEXT_sig = {2018, "glTextureParameterIivEXT", 4, _glTextureParameterIivEXT_args};

static const char * _glTextureParameterIuivEXT_args[4] = {"texture", "target", "pname", "params"};
static const trace::FunctionSig _glTextureParameterIuivEXT_sig = {2019, "glTextureParameterIuivEXT", 4, _glTextureParameterIuivEXT_args};

static const char * _glTextureParameterfEXT_args[4] = {"texture", "target", "pname", "param"};
static const trace::FunctionSig _glTextureParameterfEXT_sig = {2020, "glTextureParameterfEXT", 4, _glTextureParameterfEXT_args};

static const char * _glTextureParameterfvEXT_args[4] = {"texture", "target", "pname", "params"};
static const trace::FunctionSig _glTextureParameterfvEXT_sig = {2021, "glTextureParameterfvEXT", 4, _glTextureParameterfvEXT_args};

static const char * _glTextureParameteriEXT_args[4] = {"texture", "target", "pname", "param"};
static const trace::FunctionSig _glTextureParameteriEXT_sig = {2022, "glTextureParameteriEXT", 4, _glTextureParameteriEXT_args};

static const char * _glTextureParameterivEXT_args[4] = {"texture", "target", "pname", "params"};
static const trace::FunctionSig _glTextureParameterivEXT_sig = {2023, "glTextureParameterivEXT", 4, _glTextureParameterivEXT_args};

static const char * _glTextureRenderbufferEXT_args[3] = {"texture", "target", "renderbuffer"};
static const trace::FunctionSig _glTextureRenderbufferEXT_sig = {2024, "glTextureRenderbufferEXT", 3, _glTextureRenderbufferEXT_args};

static const char * _glTextureStorage1DEXT_args[5] = {"texture", "target", "levels", "internalformat", "width"};
static const trace::FunctionSig _glTextureStorage1DEXT_sig = {2025, "glTextureStorage1DEXT", 5, _glTextureStorage1DEXT_args};

static const char * _glTextureStorage2DEXT_args[6] = {"texture", "target", "levels", "internalformat", "width", "height"};
static const trace::FunctionSig _glTextureStorage2DEXT_sig = {2026, "glTextureStorage2DEXT", 6, _glTextureStorage2DEXT_args};

static const char * _glTextureStorage2DMultisampleEXT_args[7] = {"texture", "target", "samples", "internalformat", "width", "height", "fixedsamplelocations"};
static const trace::FunctionSig _glTextureStorage2DMultisampleEXT_sig = {2027, "glTextureStorage2DMultisampleEXT", 7, _glTextureStorage2DMultisampleEXT_args};

static const char * _glTextureStorage3DEXT_args[7] = {"texture", "target", "levels", "internalformat", "width", "height", "depth"};
static const trace::FunctionSig _glTextureStorage3DEXT_sig = {2028, "glTextureStorage3DEXT", 7, _glTextureStorage3DEXT_args};

static const char * _glTextureStorage3DMultisampleEXT_args[8] = {"texture", "target", "samples", "internalformat", "width", "height", "depth", "fixedsamplelocations"};
static const trace::FunctionSig _glTextureStorage3DMultisampleEXT_sig = {2029, "glTextureStorage3DMultisampleEXT", 8, _glTextureStorage3DMultisampleEXT_args};

static const char * _glTextureSubImage1DEXT_args[8] = {"texture", "target", "level", "xoffset", "width", "format", "type", "pixels"};
static const trace::FunctionSig _glTextureSubImage1DEXT_sig = {2030, "glTextureSubImage1DEXT", 8, _glTextureSubImage1DEXT_args};

static const char * _glTextureSubImage2DEXT_args[10] = {"texture", "target", "level", "xoffset", "yoffset", "width", "height", "format", "type", "pixels"};
static const trace::FunctionSig _glTextureSubImage2DEXT_sig = {2031, "glTextureSubImage2DEXT", 10, _glTextureSubImage2DEXT_args};

static const char * _glTextureSubImage3DEXT_args[12] = {"texture", "target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "type", "pixels"};
static const trace::FunctionSig _glTextureSubImage3DEXT_sig = {2032, "glTextureSubImage3DEXT", 12, _glTextureSubImage3DEXT_args};

static const char * _glUnmapNamedBufferEXT_args[1] = {"buffer"};
static const trace::FunctionSig _glUnmapNamedBufferEXT_sig = {2033, "glUnmapNamedBufferEXT", 1, _glUnmapNamedBufferEXT_args};

static const char * _glVertexArrayBindVertexBufferEXT_args[5] = {"vaobj", "bindingindex", "buffer", "offset", "stride"};
static const trace::FunctionSig _glVertexArrayBindVertexBufferEXT_sig = {2034, "glVertexArrayBindVertexBufferEXT", 5, _glVertexArrayBindVertexBufferEXT_args};

static const char * _glVertexArrayColorOffsetEXT_args[6] = {"vaobj", "buffer", "size", "type", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayColorOffsetEXT_sig = {2035, "glVertexArrayColorOffsetEXT", 6, _glVertexArrayColorOffsetEXT_args};

static const char * _glVertexArrayEdgeFlagOffsetEXT_args[4] = {"vaobj", "buffer", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayEdgeFlagOffsetEXT_sig = {2036, "glVertexArrayEdgeFlagOffsetEXT", 4, _glVertexArrayEdgeFlagOffsetEXT_args};

static const char * _glVertexArrayFogCoordOffsetEXT_args[5] = {"vaobj", "buffer", "type", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayFogCoordOffsetEXT_sig = {2037, "glVertexArrayFogCoordOffsetEXT", 5, _glVertexArrayFogCoordOffsetEXT_args};

static const char * _glVertexArrayIndexOffsetEXT_args[5] = {"vaobj", "buffer", "type", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayIndexOffsetEXT_sig = {2038, "glVertexArrayIndexOffsetEXT", 5, _glVertexArrayIndexOffsetEXT_args};

static const char * _glVertexArrayMultiTexCoordOffsetEXT_args[7] = {"vaobj", "buffer", "texunit", "size", "type", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayMultiTexCoordOffsetEXT_sig = {2039, "glVertexArrayMultiTexCoordOffsetEXT", 7, _glVertexArrayMultiTexCoordOffsetEXT_args};

static const char * _glVertexArrayNormalOffsetEXT_args[5] = {"vaobj", "buffer", "type", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayNormalOffsetEXT_sig = {2040, "glVertexArrayNormalOffsetEXT", 5, _glVertexArrayNormalOffsetEXT_args};

static const char * _glVertexArraySecondaryColorOffsetEXT_args[6] = {"vaobj", "buffer", "size", "type", "stride", "offset"};
static const trace::FunctionSig _glVertexArraySecondaryColorOffsetEXT_sig = {2041, "glVertexArraySecondaryColorOffsetEXT", 6, _glVertexArraySecondaryColorOffsetEXT_args};

static const char * _glVertexArrayTexCoordOffsetEXT_args[6] = {"vaobj", "buffer", "size", "type", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayTexCoordOffsetEXT_sig = {2042, "glVertexArrayTexCoordOffsetEXT", 6, _glVertexArrayTexCoordOffsetEXT_args};

static const char * _glVertexArrayVertexAttribBindingEXT_args[3] = {"vaobj", "attribindex", "bindingindex"};
static const trace::FunctionSig _glVertexArrayVertexAttribBindingEXT_sig = {2043, "glVertexArrayVertexAttribBindingEXT", 3, _glVertexArrayVertexAttribBindingEXT_args};

static const char * _glVertexArrayVertexAttribDivisorEXT_args[3] = {"vaobj", "index", "divisor"};
static const trace::FunctionSig _glVertexArrayVertexAttribDivisorEXT_sig = {2044, "glVertexArrayVertexAttribDivisorEXT", 3, _glVertexArrayVertexAttribDivisorEXT_args};

static const char * _glVertexArrayVertexAttribFormatEXT_args[6] = {"vaobj", "attribindex", "size", "type", "normalized", "relativeoffset"};
static const trace::FunctionSig _glVertexArrayVertexAttribFormatEXT_sig = {2045, "glVertexArrayVertexAttribFormatEXT", 6, _glVertexArrayVertexAttribFormatEXT_args};

static const char * _glVertexArrayVertexAttribIFormatEXT_args[5] = {"vaobj", "attribindex", "size", "type", "relativeoffset"};
static const trace::FunctionSig _glVertexArrayVertexAttribIFormatEXT_sig = {2046, "glVertexArrayVertexAttribIFormatEXT", 5, _glVertexArrayVertexAttribIFormatEXT_args};

static const char * _glVertexArrayVertexAttribIOffsetEXT_args[7] = {"vaobj", "buffer", "index", "size", "type", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayVertexAttribIOffsetEXT_sig = {2047, "glVertexArrayVertexAttribIOffsetEXT", 7, _glVertexArrayVertexAttribIOffsetEXT_args};

static const char * _glVertexArrayVertexAttribLFormatEXT_args[5] = {"vaobj", "attribindex", "size", "type", "relativeoffset"};
static const trace::FunctionSig _glVertexArrayVertexAttribLFormatEXT_sig = {2048, "glVertexArrayVertexAttribLFormatEXT", 5, _glVertexArrayVertexAttribLFormatEXT_args};

static const char * _glVertexArrayVertexAttribLOffsetEXT_args[7] = {"vaobj", "buffer", "index", "size", "type", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayVertexAttribLOffsetEXT_sig = {2049, "glVertexArrayVertexAttribLOffsetEXT", 7, _glVertexArrayVertexAttribLOffsetEXT_args};

static const char * _glVertexArrayVertexAttribOffsetEXT_args[8] = {"vaobj", "buffer", "index", "size", "type", "normalized", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayVertexAttribOffsetEXT_sig = {2050, "glVertexArrayVertexAttribOffsetEXT", 8, _glVertexArrayVertexAttribOffsetEXT_args};

static const char * _glVertexArrayVertexBindingDivisorEXT_args[3] = {"vaobj", "bindingindex", "divisor"};
static const trace::FunctionSig _glVertexArrayVertexBindingDivisorEXT_sig = {2051, "glVertexArrayVertexBindingDivisorEXT", 3, _glVertexArrayVertexBindingDivisorEXT_args};

static const char * _glVertexArrayVertexOffsetEXT_args[6] = {"vaobj", "buffer", "size", "type", "stride", "offset"};
static const trace::FunctionSig _glVertexArrayVertexOffsetEXT_sig = {2052, "glVertexArrayVertexOffsetEXT", 6, _glVertexArrayVertexOffsetEXT_args};

static const char * _glDiscardFramebufferEXT_args[3] = {"target", "numAttachments", "attachments"};
static const trace::FunctionSig _glDiscardFramebufferEXT_sig = {2053, "glDiscardFramebufferEXT", 3, _glDiscardFramebufferEXT_args};

static const char * _glGenQueriesEXT_args[2] = {"n", "ids"};
static const trace::FunctionSig _glGenQueriesEXT_sig = {2054, "glGenQueriesEXT", 2, _glGenQueriesEXT_args};

static const char * _glDeleteQueriesEXT_args[2] = {"n", "ids"};
static const trace::FunctionSig _glDeleteQueriesEXT_sig = {2055, "glDeleteQueriesEXT", 2, _glDeleteQueriesEXT_args};

static const char * _glIsQueryEXT_args[1] = {"id"};
static const trace::FunctionSig _glIsQueryEXT_sig = {2056, "glIsQueryEXT", 1, _glIsQueryEXT_args};

static const char * _glBeginQueryEXT_args[2] = {"target", "id"};
static const trace::FunctionSig _glBeginQueryEXT_sig = {2057, "glBeginQueryEXT", 2, _glBeginQueryEXT_args};

static const char * _glEndQueryEXT_args[1] = {"target"};
static const trace::FunctionSig _glEndQueryEXT_sig = {2058, "glEndQueryEXT", 1, _glEndQueryEXT_args};

static const char * _glQueryCounterEXT_args[2] = {"id", "target"};
static const trace::FunctionSig _glQueryCounterEXT_sig = {2059, "glQueryCounterEXT", 2, _glQueryCounterEXT_args};

static const char * _glGetQueryivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetQueryivEXT_sig = {2060, "glGetQueryivEXT", 3, _glGetQueryivEXT_args};

static const char * _glGetQueryObjectivEXT_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectivEXT_sig = {2061, "glGetQueryObjectivEXT", 3, _glGetQueryObjectivEXT_args};

static const char * _glGetQueryObjectuivEXT_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectuivEXT_sig = {2062, "glGetQueryObjectuivEXT", 3, _glGetQueryObjectuivEXT_args};

static const char * _glGetQueryObjecti64vEXT_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjecti64vEXT_sig = {2063, "glGetQueryObjecti64vEXT", 3, _glGetQueryObjecti64vEXT_args};

static const char * _glGetQueryObjectui64vEXT_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetQueryObjectui64vEXT_sig = {2064, "glGetQueryObjectui64vEXT", 3, _glGetQueryObjectui64vEXT_args};

static const char * _glDrawBuffersEXT_args[2] = {"n", "bufs"};
static const trace::FunctionSig _glDrawBuffersEXT_sig = {2065, "glDrawBuffersEXT", 2, _glDrawBuffersEXT_args};

static const char * _glColorMaskIndexedEXT_args[5] = {"index", "r", "g", "b", "a"};
static const trace::FunctionSig _glColorMaskIndexedEXT_sig = {2066, "glColorMaskIndexedEXT", 5, _glColorMaskIndexedEXT_args};

static const char * _glGetBooleanIndexedvEXT_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetBooleanIndexedvEXT_sig = {2067, "glGetBooleanIndexedvEXT", 3, _glGetBooleanIndexedvEXT_args};

static const char * _glGetIntegerIndexedvEXT_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetIntegerIndexedvEXT_sig = {2068, "glGetIntegerIndexedvEXT", 3, _glGetIntegerIndexedvEXT_args};

static const char * _glEnableIndexedEXT_args[2] = {"target", "index"};
static const trace::FunctionSig _glEnableIndexedEXT_sig = {2069, "glEnableIndexedEXT", 2, _glEnableIndexedEXT_args};

static const char * _glDisableIndexedEXT_args[2] = {"target", "index"};
static const trace::FunctionSig _glDisableIndexedEXT_sig = {2070, "glDisableIndexedEXT", 2, _glDisableIndexedEXT_args};

static const char * _glIsEnabledIndexedEXT_args[2] = {"target", "index"};
static const trace::FunctionSig _glIsEnabledIndexedEXT_sig = {2071, "glIsEnabledIndexedEXT", 2, _glIsEnabledIndexedEXT_args};

static const char * _glEnableiEXT_args[2] = {"target", "index"};
static const trace::FunctionSig _glEnableiEXT_sig = {2072, "glEnableiEXT", 2, _glEnableiEXT_args};

static const char * _glDisableiEXT_args[2] = {"target", "index"};
static const trace::FunctionSig _glDisableiEXT_sig = {2073, "glDisableiEXT", 2, _glDisableiEXT_args};

static const char * _glBlendEquationiEXT_args[2] = {"buf", "mode"};
static const trace::FunctionSig _glBlendEquationiEXT_sig = {2074, "glBlendEquationiEXT", 2, _glBlendEquationiEXT_args};

static const char * _glBlendEquationSeparateiEXT_args[3] = {"buf", "modeRGB", "modeAlpha"};
static const trace::FunctionSig _glBlendEquationSeparateiEXT_sig = {2075, "glBlendEquationSeparateiEXT", 3, _glBlendEquationSeparateiEXT_args};

static const char * _glBlendFunciEXT_args[3] = {"buf", "src", "dst"};
static const trace::FunctionSig _glBlendFunciEXT_sig = {2076, "glBlendFunciEXT", 3, _glBlendFunciEXT_args};

static const char * _glBlendFuncSeparateiEXT_args[5] = {"buf", "srcRGB", "dstRGB", "srcAlpha", "dstAlpha"};
static const trace::FunctionSig _glBlendFuncSeparateiEXT_sig = {2077, "glBlendFuncSeparateiEXT", 5, _glBlendFuncSeparateiEXT_args};

static const char * _glColorMaskiEXT_args[5] = {"index", "r", "g", "b", "a"};
static const trace::FunctionSig _glColorMaskiEXT_sig = {2078, "glColorMaskiEXT", 5, _glColorMaskiEXT_args};

static const char * _glIsEnablediEXT_args[2] = {"target", "index"};
static const trace::FunctionSig _glIsEnablediEXT_sig = {2079, "glIsEnablediEXT", 2, _glIsEnablediEXT_args};

static const char * _glDrawElementsBaseVertexEXT_args[5] = {"mode", "count", "type", "indices", "basevertex"};
static const trace::FunctionSig _glDrawElementsBaseVertexEXT_sig = {2080, "glDrawElementsBaseVertexEXT", 5, _glDrawElementsBaseVertexEXT_args};

static const char * _glDrawRangeElementsBaseVertexEXT_args[7] = {"mode", "start", "end", "count", "type", "indices", "basevertex"};
static const trace::FunctionSig _glDrawRangeElementsBaseVertexEXT_sig = {2081, "glDrawRangeElementsBaseVertexEXT", 7, _glDrawRangeElementsBaseVertexEXT_args};

static const char * _glDrawElementsInstancedBaseVertexEXT_args[6] = {"mode", "count", "type", "indices", "instancecount", "basevertex"};
static const trace::FunctionSig _glDrawElementsInstancedBaseVertexEXT_sig = {2082, "glDrawElementsInstancedBaseVertexEXT", 6, _glDrawElementsInstancedBaseVertexEXT_args};

static const char * _glMultiDrawElementsBaseVertexEXT_args[6] = {"mode", "count", "type", "indices", "drawcount", "basevertex"};
static const trace::FunctionSig _glMultiDrawElementsBaseVertexEXT_sig = {2083, "glMultiDrawElementsBaseVertexEXT", 6, _glMultiDrawElementsBaseVertexEXT_args};

static const char * _glDrawArraysInstancedEXT_args[4] = {"mode", "first", "count", "instancecount"};
static const trace::FunctionSig _glDrawArraysInstancedEXT_sig = {2084, "glDrawArraysInstancedEXT", 4, _glDrawArraysInstancedEXT_args};

static const char * _glDrawElementsInstancedEXT_args[5] = {"mode", "count", "type", "indices", "instancecount"};
static const trace::FunctionSig _glDrawElementsInstancedEXT_sig = {2085, "glDrawElementsInstancedEXT", 5, _glDrawElementsInstancedEXT_args};

static const char * _glDrawRangeElementsEXT_args[6] = {"mode", "start", "end", "count", "type", "indices"};
static const trace::FunctionSig _glDrawRangeElementsEXT_sig = {2086, "glDrawRangeElementsEXT", 6, _glDrawRangeElementsEXT_args};

static const char * _glFogCoordfEXT_args[1] = {"coord"};
static const trace::FunctionSig _glFogCoordfEXT_sig = {2087, "glFogCoordfEXT", 1, _glFogCoordfEXT_args};

static const char * _glFogCoordfvEXT_args[1] = {"coord"};
static const trace::FunctionSig _glFogCoordfvEXT_sig = {2088, "glFogCoordfvEXT", 1, _glFogCoordfvEXT_args};

static const char * _glFogCoorddEXT_args[1] = {"coord"};
static const trace::FunctionSig _glFogCoorddEXT_sig = {2089, "glFogCoorddEXT", 1, _glFogCoorddEXT_args};

static const char * _glFogCoorddvEXT_args[1] = {"coord"};
static const trace::FunctionSig _glFogCoorddvEXT_sig = {2090, "glFogCoorddvEXT", 1, _glFogCoorddvEXT_args};

static const char * _glFogCoordPointerEXT_args[3] = {"type", "stride", "pointer"};
static const trace::FunctionSig _glFogCoordPointerEXT_sig = {2091, "glFogCoordPointerEXT", 3, _glFogCoordPointerEXT_args};

static const char * _glBlitFramebufferEXT_args[10] = {"srcX0", "srcY0", "srcX1", "srcY1", "dstX0", "dstY0", "dstX1", "dstY1", "mask", "filter"};
static const trace::FunctionSig _glBlitFramebufferEXT_sig = {2092, "glBlitFramebufferEXT", 10, _glBlitFramebufferEXT_args};

static const char * _glRenderbufferStorageMultisampleEXT_args[5] = {"target", "samples", "internalformat", "width", "height"};
static const trace::FunctionSig _glRenderbufferStorageMultisampleEXT_sig = {2093, "glRenderbufferStorageMultisampleEXT", 5, _glRenderbufferStorageMultisampleEXT_args};

static const char * _glIsRenderbufferEXT_args[1] = {"renderbuffer"};
static const trace::FunctionSig _glIsRenderbufferEXT_sig = {2094, "glIsRenderbufferEXT", 1, _glIsRenderbufferEXT_args};

static const char * _glBindRenderbufferEXT_args[2] = {"target", "renderbuffer"};
static const trace::FunctionSig _glBindRenderbufferEXT_sig = {2095, "glBindRenderbufferEXT", 2, _glBindRenderbufferEXT_args};

static const char * _glDeleteRenderbuffersEXT_args[2] = {"n", "renderbuffers"};
static const trace::FunctionSig _glDeleteRenderbuffersEXT_sig = {2096, "glDeleteRenderbuffersEXT", 2, _glDeleteRenderbuffersEXT_args};

static const char * _glGenRenderbuffersEXT_args[2] = {"n", "renderbuffers"};
static const trace::FunctionSig _glGenRenderbuffersEXT_sig = {2097, "glGenRenderbuffersEXT", 2, _glGenRenderbuffersEXT_args};

static const char * _glRenderbufferStorageEXT_args[4] = {"target", "internalformat", "width", "height"};
static const trace::FunctionSig _glRenderbufferStorageEXT_sig = {2098, "glRenderbufferStorageEXT", 4, _glRenderbufferStorageEXT_args};

static const char * _glGetRenderbufferParameterivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetRenderbufferParameterivEXT_sig = {2099, "glGetRenderbufferParameterivEXT", 3, _glGetRenderbufferParameterivEXT_args};

static const char * _glIsFramebufferEXT_args[1] = {"framebuffer"};
static const trace::FunctionSig _glIsFramebufferEXT_sig = {2100, "glIsFramebufferEXT", 1, _glIsFramebufferEXT_args};

static const char * _glBindFramebufferEXT_args[2] = {"target", "framebuffer"};
static const trace::FunctionSig _glBindFramebufferEXT_sig = {2101, "glBindFramebufferEXT", 2, _glBindFramebufferEXT_args};

static const char * _glDeleteFramebuffersEXT_args[2] = {"n", "framebuffers"};
static const trace::FunctionSig _glDeleteFramebuffersEXT_sig = {2102, "glDeleteFramebuffersEXT", 2, _glDeleteFramebuffersEXT_args};

static const char * _glGenFramebuffersEXT_args[2] = {"n", "framebuffers"};
static const trace::FunctionSig _glGenFramebuffersEXT_sig = {2103, "glGenFramebuffersEXT", 2, _glGenFramebuffersEXT_args};

static const char * _glCheckFramebufferStatusEXT_args[1] = {"target"};
static const trace::FunctionSig _glCheckFramebufferStatusEXT_sig = {2104, "glCheckFramebufferStatusEXT", 1, _glCheckFramebufferStatusEXT_args};

static const char * _glFramebufferTexture1DEXT_args[5] = {"target", "attachment", "textarget", "texture", "level"};
static const trace::FunctionSig _glFramebufferTexture1DEXT_sig = {2105, "glFramebufferTexture1DEXT", 5, _glFramebufferTexture1DEXT_args};

static const char * _glFramebufferTexture2DEXT_args[5] = {"target", "attachment", "textarget", "texture", "level"};
static const trace::FunctionSig _glFramebufferTexture2DEXT_sig = {2106, "glFramebufferTexture2DEXT", 5, _glFramebufferTexture2DEXT_args};

static const char * _glFramebufferTexture3DEXT_args[6] = {"target", "attachment", "textarget", "texture", "level", "zoffset"};
static const trace::FunctionSig _glFramebufferTexture3DEXT_sig = {2107, "glFramebufferTexture3DEXT", 6, _glFramebufferTexture3DEXT_args};

static const char * _glFramebufferRenderbufferEXT_args[4] = {"target", "attachment", "renderbuffertarget", "renderbuffer"};
static const trace::FunctionSig _glFramebufferRenderbufferEXT_sig = {2108, "glFramebufferRenderbufferEXT", 4, _glFramebufferRenderbufferEXT_args};

static const char * _glGetFramebufferAttachmentParameterivEXT_args[4] = {"target", "attachment", "pname", "params"};
static const trace::FunctionSig _glGetFramebufferAttachmentParameterivEXT_sig = {2109, "glGetFramebufferAttachmentParameterivEXT", 4, _glGetFramebufferAttachmentParameterivEXT_args};

static const char * _glGenerateMipmapEXT_args[1] = {"target"};
static const trace::FunctionSig _glGenerateMipmapEXT_sig = {2110, "glGenerateMipmapEXT", 1, _glGenerateMipmapEXT_args};

static const char * _glFramebufferTextureEXT_args[4] = {"target", "attachment", "texture", "level"};
static const trace::FunctionSig _glFramebufferTextureEXT_sig = {2111, "glFramebufferTextureEXT", 4, _glFramebufferTextureEXT_args};

static const char * _glProgramParameteriEXT_args[3] = {"program", "pname", "value"};
static const trace::FunctionSig _glProgramParameteriEXT_sig = {2112, "glProgramParameteriEXT", 3, _glProgramParameteriEXT_args};

static const char * _glProgramEnvParameters4fvEXT_args[4] = {"target", "index", "count", "params"};
static const trace::FunctionSig _glProgramEnvParameters4fvEXT_sig = {2113, "glProgramEnvParameters4fvEXT", 4, _glProgramEnvParameters4fvEXT_args};

static const char * _glProgramLocalParameters4fvEXT_args[4] = {"target", "index", "count", "params"};
static const trace::FunctionSig _glProgramLocalParameters4fvEXT_sig = {2114, "glProgramLocalParameters4fvEXT", 4, _glProgramLocalParameters4fvEXT_args};

static const char * _glGetUniformuivEXT_args[3] = {"program", "location", "params"};
static const trace::FunctionSig _glGetUniformuivEXT_sig = {2115, "glGetUniformuivEXT", 3, _glGetUniformuivEXT_args};

static const char * _glGetFragDataLocationEXT_args[2] = {"program", "name"};
static const trace::FunctionSig _glGetFragDataLocationEXT_sig = {2116, "glGetFragDataLocationEXT", 2, _glGetFragDataLocationEXT_args};

static const char * _glUniform1uiEXT_args[2] = {"location", "v0"};
static const trace::FunctionSig _glUniform1uiEXT_sig = {2117, "glUniform1uiEXT", 2, _glUniform1uiEXT_args};

static const char * _glUniform2uiEXT_args[3] = {"location", "v0", "v1"};
static const trace::FunctionSig _glUniform2uiEXT_sig = {2118, "glUniform2uiEXT", 3, _glUniform2uiEXT_args};

static const char * _glUniform3uiEXT_args[4] = {"location", "v0", "v1", "v2"};
static const trace::FunctionSig _glUniform3uiEXT_sig = {2119, "glUniform3uiEXT", 4, _glUniform3uiEXT_args};

static const char * _glUniform4uiEXT_args[5] = {"location", "v0", "v1", "v2", "v3"};
static const trace::FunctionSig _glUniform4uiEXT_sig = {2120, "glUniform4uiEXT", 5, _glUniform4uiEXT_args};

static const char * _glUniform1uivEXT_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1uivEXT_sig = {2121, "glUniform1uivEXT", 3, _glUniform1uivEXT_args};

static const char * _glUniform2uivEXT_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2uivEXT_sig = {2122, "glUniform2uivEXT", 3, _glUniform2uivEXT_args};

static const char * _glUniform3uivEXT_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3uivEXT_sig = {2123, "glUniform3uivEXT", 3, _glUniform3uivEXT_args};

static const char * _glUniform4uivEXT_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4uivEXT_sig = {2124, "glUniform4uivEXT", 3, _glUniform4uivEXT_args};

static const char * _glGetHistogramEXT_args[5] = {"target", "reset", "format", "type", "values"};
static const trace::FunctionSig _glGetHistogramEXT_sig = {2125, "glGetHistogramEXT", 5, _glGetHistogramEXT_args};

static const char * _glGetHistogramParameterfvEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetHistogramParameterfvEXT_sig = {2126, "glGetHistogramParameterfvEXT", 3, _glGetHistogramParameterfvEXT_args};

static const char * _glGetHistogramParameterivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetHistogramParameterivEXT_sig = {2127, "glGetHistogramParameterivEXT", 3, _glGetHistogramParameterivEXT_args};

static const char * _glGetMinmaxEXT_args[5] = {"target", "reset", "format", "type", "values"};
static const trace::FunctionSig _glGetMinmaxEXT_sig = {2128, "glGetMinmaxEXT", 5, _glGetMinmaxEXT_args};

static const char * _glGetMinmaxParameterfvEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetMinmaxParameterfvEXT_sig = {2129, "glGetMinmaxParameterfvEXT", 3, _glGetMinmaxParameterfvEXT_args};

static const char * _glGetMinmaxParameterivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetMinmaxParameterivEXT_sig = {2130, "glGetMinmaxParameterivEXT", 3, _glGetMinmaxParameterivEXT_args};

static const char * _glHistogramEXT_args[4] = {"target", "width", "internalformat", "sink"};
static const trace::FunctionSig _glHistogramEXT_sig = {2131, "glHistogramEXT", 4, _glHistogramEXT_args};

static const char * _glMinmaxEXT_args[3] = {"target", "internalformat", "sink"};
static const trace::FunctionSig _glMinmaxEXT_sig = {2132, "glMinmaxEXT", 3, _glMinmaxEXT_args};

static const char * _glResetHistogramEXT_args[1] = {"target"};
static const trace::FunctionSig _glResetHistogramEXT_sig = {2133, "glResetHistogramEXT", 1, _glResetHistogramEXT_args};

static const char * _glResetMinmaxEXT_args[1] = {"target"};
static const trace::FunctionSig _glResetMinmaxEXT_sig = {2134, "glResetMinmaxEXT", 1, _glResetMinmaxEXT_args};

static const char * _glIndexFuncEXT_args[2] = {"func", "ref"};
static const trace::FunctionSig _glIndexFuncEXT_sig = {2135, "glIndexFuncEXT", 2, _glIndexFuncEXT_args};

static const char * _glIndexMaterialEXT_args[2] = {"face", "mode"};
static const trace::FunctionSig _glIndexMaterialEXT_sig = {2136, "glIndexMaterialEXT", 2, _glIndexMaterialEXT_args};

static const char * _glVertexAttribDivisorEXT_args[2] = {"index", "divisor"};
static const trace::FunctionSig _glVertexAttribDivisorEXT_sig = {2137, "glVertexAttribDivisorEXT", 2, _glVertexAttribDivisorEXT_args};

static const char * _glApplyTextureEXT_args[1] = {"mode"};
static const trace::FunctionSig _glApplyTextureEXT_sig = {2138, "glApplyTextureEXT", 1, _glApplyTextureEXT_args};

static const char * _glTextureLightEXT_args[1] = {"pname"};
static const trace::FunctionSig _glTextureLightEXT_sig = {2139, "glTextureLightEXT", 1, _glTextureLightEXT_args};

static const char * _glTextureMaterialEXT_args[2] = {"face", "mode"};
static const trace::FunctionSig _glTextureMaterialEXT_sig = {2140, "glTextureMaterialEXT", 2, _glTextureMaterialEXT_args};

static const char * _glMapBufferRangeEXT_args[4] = {"target", "offset", "length", "access"};
static const trace::FunctionSig _glMapBufferRangeEXT_sig = {2141, "glMapBufferRangeEXT", 4, _glMapBufferRangeEXT_args};

static const char * _glFlushMappedBufferRangeEXT_args[3] = {"target", "offset", "length"};
static const trace::FunctionSig _glFlushMappedBufferRangeEXT_sig = {2142, "glFlushMappedBufferRangeEXT", 3, _glFlushMappedBufferRangeEXT_args};

static const char * _glMultiDrawArraysEXT_args[4] = {"mode", "first", "count", "drawcount"};
static const trace::FunctionSig _glMultiDrawArraysEXT_sig = {2143, "glMultiDrawArraysEXT", 4, _glMultiDrawArraysEXT_args};

static const char * _glMultiDrawElementsEXT_args[5] = {"mode", "count", "type", "indices", "drawcount"};
static const trace::FunctionSig _glMultiDrawElementsEXT_sig = {2144, "glMultiDrawElementsEXT", 5, _glMultiDrawElementsEXT_args};

static const char * _glMultiDrawArraysIndirectEXT_args[4] = {"mode", "indirect", "drawcount", "stride"};
static const trace::FunctionSig _glMultiDrawArraysIndirectEXT_sig = {2145, "glMultiDrawArraysIndirectEXT", 4, _glMultiDrawArraysIndirectEXT_args};

static const char * _glMultiDrawElementsIndirectEXT_args[5] = {"mode", "type", "indirect", "drawcount", "stride"};
static const trace::FunctionSig _glMultiDrawElementsIndirectEXT_sig = {2146, "glMultiDrawElementsIndirectEXT", 5, _glMultiDrawElementsIndirectEXT_args};

static const char * _glSampleMaskEXT_args[2] = {"value", "invert"};
static const trace::FunctionSig _glSampleMaskEXT_sig = {2147, "glSampleMaskEXT", 2, _glSampleMaskEXT_args};

static const char * _glSamplePatternEXT_args[1] = {"pattern"};
static const trace::FunctionSig _glSamplePatternEXT_sig = {2148, "glSamplePatternEXT", 1, _glSamplePatternEXT_args};

static const char * _glFramebufferTexture2DMultisampleEXT_args[6] = {"target", "attachment", "textarget", "texture", "level", "samples"};
static const trace::FunctionSig _glFramebufferTexture2DMultisampleEXT_sig = {2149, "glFramebufferTexture2DMultisampleEXT", 6, _glFramebufferTexture2DMultisampleEXT_args};

static const char * _glReadBufferIndexedEXT_args[2] = {"src", "index"};
static const trace::FunctionSig _glReadBufferIndexedEXT_sig = {2150, "glReadBufferIndexedEXT", 2, _glReadBufferIndexedEXT_args};

static const char * _glDrawBuffersIndexedEXT_args[3] = {"n", "location", "indices"};
static const trace::FunctionSig _glDrawBuffersIndexedEXT_sig = {2151, "glDrawBuffersIndexedEXT", 3, _glDrawBuffersIndexedEXT_args};

static const char * _glGetIntegeri_vEXT_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetIntegeri_vEXT_sig = {2152, "glGetIntegeri_vEXT", 3, _glGetIntegeri_vEXT_args};

static const char * _glColorTableEXT_args[6] = {"target", "internalformat", "width", "format", "type", "table"};
static const trace::FunctionSig _glColorTableEXT_sig = {2153, "glColorTableEXT", 6, _glColorTableEXT_args};

static const char * _glGetColorTableEXT_args[4] = {"target", "format", "type", "data"};
static const trace::FunctionSig _glGetColorTableEXT_sig = {2154, "glGetColorTableEXT", 4, _glGetColorTableEXT_args};

static const char * _glGetColorTableParameterivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetColorTableParameterivEXT_sig = {2155, "glGetColorTableParameterivEXT", 3, _glGetColorTableParameterivEXT_args};

static const char * _glGetColorTableParameterfvEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetColorTableParameterfvEXT_sig = {2156, "glGetColorTableParameterfvEXT", 3, _glGetColorTableParameterfvEXT_args};

static const char * _glPixelTransformParameteriEXT_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glPixelTransformParameteriEXT_sig = {2157, "glPixelTransformParameteriEXT", 3, _glPixelTransformParameteriEXT_args};

static const char * _glPixelTransformParameterfEXT_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glPixelTransformParameterfEXT_sig = {2158, "glPixelTransformParameterfEXT", 3, _glPixelTransformParameterfEXT_args};

static const char * _glPixelTransformParameterivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glPixelTransformParameterivEXT_sig = {2159, "glPixelTransformParameterivEXT", 3, _glPixelTransformParameterivEXT_args};

static const char * _glPixelTransformParameterfvEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glPixelTransformParameterfvEXT_sig = {2160, "glPixelTransformParameterfvEXT", 3, _glPixelTransformParameterfvEXT_args};

static const char * _glGetPixelTransformParameterivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetPixelTransformParameterivEXT_sig = {2161, "glGetPixelTransformParameterivEXT", 3, _glGetPixelTransformParameterivEXT_args};

static const char * _glGetPixelTransformParameterfvEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetPixelTransformParameterfvEXT_sig = {2162, "glGetPixelTransformParameterfvEXT", 3, _glGetPixelTransformParameterfvEXT_args};

static const char * _glPointParameterfEXT_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPointParameterfEXT_sig = {2163, "glPointParameterfEXT", 2, _glPointParameterfEXT_args};

static const char * _glPointParameterfvEXT_args[2] = {"pname", "params"};
static const trace::FunctionSig _glPointParameterfvEXT_sig = {2164, "glPointParameterfvEXT", 2, _glPointParameterfvEXT_args};

static const char * _glPolygonOffsetEXT_args[2] = {"factor", "bias"};
static const trace::FunctionSig _glPolygonOffsetEXT_sig = {2165, "glPolygonOffsetEXT", 2, _glPolygonOffsetEXT_args};

static const char * _glPolygonOffsetClampEXT_args[3] = {"factor", "units", "clamp"};
static const trace::FunctionSig _glPolygonOffsetClampEXT_sig = {2166, "glPolygonOffsetClampEXT", 3, _glPolygonOffsetClampEXT_args};

static const char * _glPrimitiveBoundingBoxEXT_args[8] = {"minX", "minY", "minZ", "minW", "maxX", "maxY", "maxZ", "maxW"};
static const trace::FunctionSig _glPrimitiveBoundingBoxEXT_sig = {2167, "glPrimitiveBoundingBoxEXT", 8, _glPrimitiveBoundingBoxEXT_args};

static const char * _glProvokingVertexEXT_args[1] = {"mode"};
static const trace::FunctionSig _glProvokingVertexEXT_sig = {2168, "glProvokingVertexEXT", 1, _glProvokingVertexEXT_args};

static const char * _glRasterSamplesEXT_args[2] = {"samples", "fixedsamplelocations"};
static const trace::FunctionSig _glRasterSamplesEXT_sig = {2169, "glRasterSamplesEXT", 2, _glRasterSamplesEXT_args};

static const char ** _glGetGraphicsResetStatusEXT_args = NULL;
static const trace::FunctionSig _glGetGraphicsResetStatusEXT_sig = {2170, "glGetGraphicsResetStatusEXT", 0, _glGetGraphicsResetStatusEXT_args};

static const char * _glReadnPixelsEXT_args[8] = {"x", "y", "width", "height", "format", "type", "bufSize", "data"};
static const trace::FunctionSig _glReadnPixelsEXT_sig = {2171, "glReadnPixelsEXT", 8, _glReadnPixelsEXT_args};

static const char * _glGetnUniformfvEXT_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformfvEXT_sig = {2172, "glGetnUniformfvEXT", 4, _glGetnUniformfvEXT_args};

static const char * _glGetnUniformivEXT_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformivEXT_sig = {2173, "glGetnUniformivEXT", 4, _glGetnUniformivEXT_args};

static const char * _glSecondaryColor3bEXT_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3bEXT_sig = {2174, "glSecondaryColor3bEXT", 3, _glSecondaryColor3bEXT_args};

static const char * _glSecondaryColor3bvEXT_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3bvEXT_sig = {2175, "glSecondaryColor3bvEXT", 1, _glSecondaryColor3bvEXT_args};

static const char * _glSecondaryColor3dEXT_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3dEXT_sig = {2176, "glSecondaryColor3dEXT", 3, _glSecondaryColor3dEXT_args};

static const char * _glSecondaryColor3dvEXT_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3dvEXT_sig = {2177, "glSecondaryColor3dvEXT", 1, _glSecondaryColor3dvEXT_args};

static const char * _glSecondaryColor3fEXT_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3fEXT_sig = {2178, "glSecondaryColor3fEXT", 3, _glSecondaryColor3fEXT_args};

static const char * _glSecondaryColor3fvEXT_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3fvEXT_sig = {2179, "glSecondaryColor3fvEXT", 1, _glSecondaryColor3fvEXT_args};

static const char * _glSecondaryColor3iEXT_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3iEXT_sig = {2180, "glSecondaryColor3iEXT", 3, _glSecondaryColor3iEXT_args};

static const char * _glSecondaryColor3ivEXT_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3ivEXT_sig = {2181, "glSecondaryColor3ivEXT", 1, _glSecondaryColor3ivEXT_args};

static const char * _glSecondaryColor3sEXT_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3sEXT_sig = {2182, "glSecondaryColor3sEXT", 3, _glSecondaryColor3sEXT_args};

static const char * _glSecondaryColor3svEXT_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3svEXT_sig = {2183, "glSecondaryColor3svEXT", 1, _glSecondaryColor3svEXT_args};

static const char * _glSecondaryColor3ubEXT_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3ubEXT_sig = {2184, "glSecondaryColor3ubEXT", 3, _glSecondaryColor3ubEXT_args};

static const char * _glSecondaryColor3ubvEXT_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3ubvEXT_sig = {2185, "glSecondaryColor3ubvEXT", 1, _glSecondaryColor3ubvEXT_args};

static const char * _glSecondaryColor3uiEXT_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3uiEXT_sig = {2186, "glSecondaryColor3uiEXT", 3, _glSecondaryColor3uiEXT_args};

static const char * _glSecondaryColor3uivEXT_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3uivEXT_sig = {2187, "glSecondaryColor3uivEXT", 1, _glSecondaryColor3uivEXT_args};

static const char * _glSecondaryColor3usEXT_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3usEXT_sig = {2188, "glSecondaryColor3usEXT", 3, _glSecondaryColor3usEXT_args};

static const char * _glSecondaryColor3usvEXT_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3usvEXT_sig = {2189, "glSecondaryColor3usvEXT", 1, _glSecondaryColor3usvEXT_args};

static const char * _glSecondaryColorPointerEXT_args[4] = {"size", "type", "stride", "pointer"};
static const trace::FunctionSig _glSecondaryColorPointerEXT_sig = {2190, "glSecondaryColorPointerEXT", 4, _glSecondaryColorPointerEXT_args};

static const char * _glUseShaderProgramEXT_args[2] = {"type", "program"};
static const trace::FunctionSig _glUseShaderProgramEXT_sig = {2191, "glUseShaderProgramEXT", 2, _glUseShaderProgramEXT_args};

static const char * _glActiveProgramEXT_args[1] = {"program"};
static const trace::FunctionSig _glActiveProgramEXT_sig = {2192, "glActiveProgramEXT", 1, _glActiveProgramEXT_args};

static const char * _glCreateShaderProgramEXT_args[2] = {"type", "string"};
static const trace::FunctionSig _glCreateShaderProgramEXT_sig = {2193, "glCreateShaderProgramEXT", 2, _glCreateShaderProgramEXT_args};

static const char * _glActiveShaderProgramEXT_args[2] = {"pipeline", "program"};
static const trace::FunctionSig _glActiveShaderProgramEXT_sig = {2194, "glActiveShaderProgramEXT", 2, _glActiveShaderProgramEXT_args};

static const char * _glBindProgramPipelineEXT_args[1] = {"pipeline"};
static const trace::FunctionSig _glBindProgramPipelineEXT_sig = {2195, "glBindProgramPipelineEXT", 1, _glBindProgramPipelineEXT_args};

static const char * _glCreateShaderProgramvEXT_args[3] = {"type", "count", "strings"};
static const trace::FunctionSig _glCreateShaderProgramvEXT_sig = {2196, "glCreateShaderProgramvEXT", 3, _glCreateShaderProgramvEXT_args};

static const char * _glDeleteProgramPipelinesEXT_args[2] = {"n", "pipelines"};
static const trace::FunctionSig _glDeleteProgramPipelinesEXT_sig = {2197, "glDeleteProgramPipelinesEXT", 2, _glDeleteProgramPipelinesEXT_args};

static const char * _glGenProgramPipelinesEXT_args[2] = {"n", "pipelines"};
static const trace::FunctionSig _glGenProgramPipelinesEXT_sig = {2198, "glGenProgramPipelinesEXT", 2, _glGenProgramPipelinesEXT_args};

static const char * _glGetProgramPipelineInfoLogEXT_args[4] = {"pipeline", "bufSize", "length", "infoLog"};
static const trace::FunctionSig _glGetProgramPipelineInfoLogEXT_sig = {2199, "glGetProgramPipelineInfoLogEXT", 4, _glGetProgramPipelineInfoLogEXT_args};

static const char * _glGetProgramPipelineivEXT_args[3] = {"pipeline", "pname", "params"};
static const trace::FunctionSig _glGetProgramPipelineivEXT_sig = {2200, "glGetProgramPipelineivEXT", 3, _glGetProgramPipelineivEXT_args};

static const char * _glIsProgramPipelineEXT_args[1] = {"pipeline"};
static const trace::FunctionSig _glIsProgramPipelineEXT_sig = {2201, "glIsProgramPipelineEXT", 1, _glIsProgramPipelineEXT_args};

static const char * _glUseProgramStagesEXT_args[3] = {"pipeline", "stages", "program"};
static const trace::FunctionSig _glUseProgramStagesEXT_sig = {2202, "glUseProgramStagesEXT", 3, _glUseProgramStagesEXT_args};

static const char * _glValidateProgramPipelineEXT_args[1] = {"pipeline"};
static const trace::FunctionSig _glValidateProgramPipelineEXT_sig = {2203, "glValidateProgramPipelineEXT", 1, _glValidateProgramPipelineEXT_args};

static const char * _glBindImageTextureEXT_args[7] = {"index", "texture", "level", "layered", "layer", "access", "format"};
static const trace::FunctionSig _glBindImageTextureEXT_sig = {2204, "glBindImageTextureEXT", 7, _glBindImageTextureEXT_args};

static const char * _glMemoryBarrierEXT_args[1] = {"barriers"};
static const trace::FunctionSig _glMemoryBarrierEXT_sig = {2205, "glMemoryBarrierEXT", 1, _glMemoryBarrierEXT_args};

static const char * _glFramebufferPixelLocalStorageSizeEXT_args[2] = {"target", "size"};
static const trace::FunctionSig _glFramebufferPixelLocalStorageSizeEXT_sig = {2206, "glFramebufferPixelLocalStorageSizeEXT", 2, _glFramebufferPixelLocalStorageSizeEXT_args};

static const char * _glGetFramebufferPixelLocalStorageSizeEXT_args[1] = {"target"};
static const trace::FunctionSig _glGetFramebufferPixelLocalStorageSizeEXT_sig = {2207, "glGetFramebufferPixelLocalStorageSizeEXT", 1, _glGetFramebufferPixelLocalStorageSizeEXT_args};

static const char * _glClearPixelLocalStorageuiEXT_args[3] = {"offset", "n", "values"};
static const trace::FunctionSig _glClearPixelLocalStorageuiEXT_sig = {2208, "glClearPixelLocalStorageuiEXT", 3, _glClearPixelLocalStorageuiEXT_args};

static const char * _glTexPageCommitmentEXT_args[9] = {"target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "commit"};
static const trace::FunctionSig _glTexPageCommitmentEXT_sig = {2209, "glTexPageCommitmentEXT", 9, _glTexPageCommitmentEXT_args};

static const char * _glStencilClearTagEXT_args[2] = {"stencilTagBits", "stencilClearTag"};
static const trace::FunctionSig _glStencilClearTagEXT_sig = {2210, "glStencilClearTagEXT", 2, _glStencilClearTagEXT_args};

static const char * _glActiveStencilFaceEXT_args[1] = {"face"};
static const trace::FunctionSig _glActiveStencilFaceEXT_sig = {2211, "glActiveStencilFaceEXT", 1, _glActiveStencilFaceEXT_args};

static const char * _glTexSubImage1DEXT_args[7] = {"target", "level", "xoffset", "width", "format", "type", "pixels"};
static const trace::FunctionSig _glTexSubImage1DEXT_sig = {2212, "glTexSubImage1DEXT", 7, _glTexSubImage1DEXT_args};

static const char * _glTexSubImage2DEXT_args[9] = {"target", "level", "xoffset", "yoffset", "width", "height", "format", "type", "pixels"};
static const trace::FunctionSig _glTexSubImage2DEXT_sig = {2213, "glTexSubImage2DEXT", 9, _glTexSubImage2DEXT_args};

static const char * _glPatchParameteriEXT_args[2] = {"pname", "value"};
static const trace::FunctionSig _glPatchParameteriEXT_sig = {2214, "glPatchParameteriEXT", 2, _glPatchParameteriEXT_args};

static const char * _glTexImage3DEXT_args[10] = {"target", "level", "internalformat", "width", "height", "depth", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glTexImage3DEXT_sig = {2215, "glTexImage3DEXT", 10, _glTexImage3DEXT_args};

static const char * _glTexSubImage3DEXT_args[11] = {"target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "type", "pixels"};
static const trace::FunctionSig _glTexSubImage3DEXT_sig = {2216, "glTexSubImage3DEXT", 11, _glTexSubImage3DEXT_args};

static const char * _glFramebufferTextureLayerEXT_args[5] = {"target", "attachment", "texture", "level", "layer"};
static const trace::FunctionSig _glFramebufferTextureLayerEXT_sig = {2217, "glFramebufferTextureLayerEXT", 5, _glFramebufferTextureLayerEXT_args};

static const char * _glTexParameterIivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glTexParameterIivEXT_sig = {2218, "glTexParameterIivEXT", 3, _glTexParameterIivEXT_args};

static const char * _glTexParameterIuivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glTexParameterIuivEXT_sig = {2219, "glTexParameterIuivEXT", 3, _glTexParameterIuivEXT_args};

static const char * _glGetTexParameterIivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexParameterIivEXT_sig = {2220, "glGetTexParameterIivEXT", 3, _glGetTexParameterIivEXT_args};

static const char * _glGetTexParameterIuivEXT_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetTexParameterIuivEXT_sig = {2221, "glGetTexParameterIuivEXT", 3, _glGetTexParameterIuivEXT_args};

static const char * _glSamplerParameterIivEXT_args[3] = {"sampler", "pname", "param"};
static const trace::FunctionSig _glSamplerParameterIivEXT_sig = {2222, "glSamplerParameterIivEXT", 3, _glSamplerParameterIivEXT_args};

static const char * _glSamplerParameterIuivEXT_args[3] = {"sampler", "pname", "param"};
static const trace::FunctionSig _glSamplerParameterIuivEXT_sig = {2223, "glSamplerParameterIuivEXT", 3, _glSamplerParameterIuivEXT_args};

static const char * _glGetSamplerParameterIivEXT_args[3] = {"sampler", "pname", "params"};
static const trace::FunctionSig _glGetSamplerParameterIivEXT_sig = {2224, "glGetSamplerParameterIivEXT", 3, _glGetSamplerParameterIivEXT_args};

static const char * _glGetSamplerParameterIuivEXT_args[3] = {"sampler", "pname", "params"};
static const trace::FunctionSig _glGetSamplerParameterIuivEXT_sig = {2225, "glGetSamplerParameterIuivEXT", 3, _glGetSamplerParameterIuivEXT_args};

static const char * _glTexBufferEXT_args[3] = {"target", "internalformat", "buffer"};
static const trace::FunctionSig _glTexBufferEXT_sig = {2226, "glTexBufferEXT", 3, _glTexBufferEXT_args};

static const char * _glTexBufferRangeEXT_args[5] = {"target", "internalformat", "buffer", "offset", "size"};
static const trace::FunctionSig _glTexBufferRangeEXT_sig = {2227, "glTexBufferRangeEXT", 5, _glTexBufferRangeEXT_args};

static const char * _glClearColorIiEXT_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glClearColorIiEXT_sig = {2228, "glClearColorIiEXT", 4, _glClearColorIiEXT_args};

static const char * _glClearColorIuiEXT_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glClearColorIuiEXT_sig = {2229, "glClearColorIuiEXT", 4, _glClearColorIuiEXT_args};

static const char * _glAreTexturesResidentEXT_args[3] = {"n", "textures", "residences"};
static const trace::FunctionSig _glAreTexturesResidentEXT_sig = {2230, "glAreTexturesResidentEXT", 3, _glAreTexturesResidentEXT_args};

static const char * _glBindTextureEXT_args[2] = {"target", "texture"};
static const trace::FunctionSig _glBindTextureEXT_sig = {2231, "glBindTextureEXT", 2, _glBindTextureEXT_args};

static const char * _glDeleteTexturesEXT_args[2] = {"n", "textures"};
static const trace::FunctionSig _glDeleteTexturesEXT_sig = {2232, "glDeleteTexturesEXT", 2, _glDeleteTexturesEXT_args};

static const char * _glGenTexturesEXT_args[2] = {"n", "textures"};
static const trace::FunctionSig _glGenTexturesEXT_sig = {2233, "glGenTexturesEXT", 2, _glGenTexturesEXT_args};

static const char * _glIsTextureEXT_args[1] = {"texture"};
static const trace::FunctionSig _glIsTextureEXT_sig = {2234, "glIsTextureEXT", 1, _glIsTextureEXT_args};

static const char * _glPrioritizeTexturesEXT_args[3] = {"n", "textures", "priorities"};
static const trace::FunctionSig _glPrioritizeTexturesEXT_sig = {2235, "glPrioritizeTexturesEXT", 3, _glPrioritizeTexturesEXT_args};

static const char * _glTextureNormalEXT_args[1] = {"mode"};
static const trace::FunctionSig _glTextureNormalEXT_sig = {2236, "glTextureNormalEXT", 1, _glTextureNormalEXT_args};

static const char * _glTexStorage1DEXT_args[4] = {"target", "levels", "internalformat", "width"};
static const trace::FunctionSig _glTexStorage1DEXT_sig = {2237, "glTexStorage1DEXT", 4, _glTexStorage1DEXT_args};

static const char * _glTexStorage2DEXT_args[5] = {"target", "levels", "internalformat", "width", "height"};
static const trace::FunctionSig _glTexStorage2DEXT_sig = {2238, "glTexStorage2DEXT", 5, _glTexStorage2DEXT_args};

static const char * _glTexStorage3DEXT_args[6] = {"target", "levels", "internalformat", "width", "height", "depth"};
static const trace::FunctionSig _glTexStorage3DEXT_sig = {2239, "glTexStorage3DEXT", 6, _glTexStorage3DEXT_args};

static const char * _glTextureViewEXT_args[8] = {"texture", "target", "origtexture", "internalformat", "minlevel", "numlevels", "minlayer", "numlayers"};
static const trace::FunctionSig _glTextureViewEXT_sig = {2240, "glTextureViewEXT", 8, _glTextureViewEXT_args};

static const char * _glBeginTransformFeedbackEXT_args[1] = {"primitiveMode"};
static const trace::FunctionSig _glBeginTransformFeedbackEXT_sig = {2241, "glBeginTransformFeedbackEXT", 1, _glBeginTransformFeedbackEXT_args};

static const char ** _glEndTransformFeedbackEXT_args = NULL;
static const trace::FunctionSig _glEndTransformFeedbackEXT_sig = {2242, "glEndTransformFeedbackEXT", 0, _glEndTransformFeedbackEXT_args};

static const char * _glBindBufferRangeEXT_args[5] = {"target", "index", "buffer", "offset", "size"};
static const trace::FunctionSig _glBindBufferRangeEXT_sig = {2243, "glBindBufferRangeEXT", 5, _glBindBufferRangeEXT_args};

static const char * _glBindBufferOffsetEXT_args[4] = {"target", "index", "buffer", "offset"};
static const trace::FunctionSig _glBindBufferOffsetEXT_sig = {2244, "glBindBufferOffsetEXT", 4, _glBindBufferOffsetEXT_args};

static const char * _glBindBufferBaseEXT_args[3] = {"target", "index", "buffer"};
static const trace::FunctionSig _glBindBufferBaseEXT_sig = {2245, "glBindBufferBaseEXT", 3, _glBindBufferBaseEXT_args};

static const char * _glTransformFeedbackVaryingsEXT_args[4] = {"program", "count", "varyings", "bufferMode"};
static const trace::FunctionSig _glTransformFeedbackVaryingsEXT_sig = {2246, "glTransformFeedbackVaryingsEXT", 4, _glTransformFeedbackVaryingsEXT_args};

static const char * _glGetTransformFeedbackVaryingEXT_args[7] = {"program", "index", "bufSize", "length", "size", "type", "name"};
static const trace::FunctionSig _glGetTransformFeedbackVaryingEXT_sig = {2247, "glGetTransformFeedbackVaryingEXT", 7, _glGetTransformFeedbackVaryingEXT_args};

static const char * _glArrayElementEXT_args[1] = {"i"};
static const trace::FunctionSig _glArrayElementEXT_sig = {2248, "glArrayElementEXT", 1, _glArrayElementEXT_args};

static const char * _glColorPointerEXT_args[5] = {"size", "type", "stride", "count", "pointer"};
static const trace::FunctionSig _glColorPointerEXT_sig = {2249, "glColorPointerEXT", 5, _glColorPointerEXT_args};

static const char * _glDrawArraysEXT_args[3] = {"mode", "first", "count"};
static const trace::FunctionSig _glDrawArraysEXT_sig = {2250, "glDrawArraysEXT", 3, _glDrawArraysEXT_args};

static const char * _glEdgeFlagPointerEXT_args[3] = {"stride", "count", "pointer"};
static const trace::FunctionSig _glEdgeFlagPointerEXT_sig = {2251, "glEdgeFlagPointerEXT", 3, _glEdgeFlagPointerEXT_args};

static const char * _glGetPointervEXT_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetPointervEXT_sig = {2252, "glGetPointervEXT", 2, _glGetPointervEXT_args};

static const char * _glIndexPointerEXT_args[4] = {"type", "stride", "count", "pointer"};
static const trace::FunctionSig _glIndexPointerEXT_sig = {2253, "glIndexPointerEXT", 4, _glIndexPointerEXT_args};

static const char * _glNormalPointerEXT_args[4] = {"type", "stride", "count", "pointer"};
static const trace::FunctionSig _glNormalPointerEXT_sig = {2254, "glNormalPointerEXT", 4, _glNormalPointerEXT_args};

static const char * _glTexCoordPointerEXT_args[5] = {"size", "type", "stride", "count", "pointer"};
static const trace::FunctionSig _glTexCoordPointerEXT_sig = {2255, "glTexCoordPointerEXT", 5, _glTexCoordPointerEXT_args};

static const char * _glVertexPointerEXT_args[5] = {"size", "type", "stride", "count", "pointer"};
static const trace::FunctionSig _glVertexPointerEXT_sig = {2256, "glVertexPointerEXT", 5, _glVertexPointerEXT_args};

static const char * _glVertexAttribL1dEXT_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttribL1dEXT_sig = {2257, "glVertexAttribL1dEXT", 2, _glVertexAttribL1dEXT_args};

static const char * _glVertexAttribL2dEXT_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttribL2dEXT_sig = {2258, "glVertexAttribL2dEXT", 3, _glVertexAttribL2dEXT_args};

static const char * _glVertexAttribL3dEXT_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttribL3dEXT_sig = {2259, "glVertexAttribL3dEXT", 4, _glVertexAttribL3dEXT_args};

static const char * _glVertexAttribL4dEXT_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttribL4dEXT_sig = {2260, "glVertexAttribL4dEXT", 5, _glVertexAttribL4dEXT_args};

static const char * _glVertexAttribL1dvEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL1dvEXT_sig = {2261, "glVertexAttribL1dvEXT", 2, _glVertexAttribL1dvEXT_args};

static const char * _glVertexAttribL2dvEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL2dvEXT_sig = {2262, "glVertexAttribL2dvEXT", 2, _glVertexAttribL2dvEXT_args};

static const char * _glVertexAttribL3dvEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL3dvEXT_sig = {2263, "glVertexAttribL3dvEXT", 2, _glVertexAttribL3dvEXT_args};

static const char * _glVertexAttribL4dvEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL4dvEXT_sig = {2264, "glVertexAttribL4dvEXT", 2, _glVertexAttribL4dvEXT_args};

static const char * _glVertexAttribLPointerEXT_args[5] = {"index", "size", "type", "stride", "pointer"};
static const trace::FunctionSig _glVertexAttribLPointerEXT_sig = {2265, "glVertexAttribLPointerEXT", 5, _glVertexAttribLPointerEXT_args};

static const char * _glGetVertexAttribLdvEXT_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribLdvEXT_sig = {2266, "glGetVertexAttribLdvEXT", 3, _glGetVertexAttribLdvEXT_args};

static const char ** _glBeginVertexShaderEXT_args = NULL;
static const trace::FunctionSig _glBeginVertexShaderEXT_sig = {2267, "glBeginVertexShaderEXT", 0, _glBeginVertexShaderEXT_args};

static const char ** _glEndVertexShaderEXT_args = NULL;
static const trace::FunctionSig _glEndVertexShaderEXT_sig = {2268, "glEndVertexShaderEXT", 0, _glEndVertexShaderEXT_args};

static const char * _glBindVertexShaderEXT_args[1] = {"id"};
static const trace::FunctionSig _glBindVertexShaderEXT_sig = {2269, "glBindVertexShaderEXT", 1, _glBindVertexShaderEXT_args};

static const char * _glGenVertexShadersEXT_args[1] = {"range"};
static const trace::FunctionSig _glGenVertexShadersEXT_sig = {2270, "glGenVertexShadersEXT", 1, _glGenVertexShadersEXT_args};

static const char * _glDeleteVertexShaderEXT_args[1] = {"id"};
static const trace::FunctionSig _glDeleteVertexShaderEXT_sig = {2271, "glDeleteVertexShaderEXT", 1, _glDeleteVertexShaderEXT_args};

static const char * _glShaderOp1EXT_args[3] = {"op", "res", "arg1"};
static const trace::FunctionSig _glShaderOp1EXT_sig = {2272, "glShaderOp1EXT", 3, _glShaderOp1EXT_args};

static const char * _glShaderOp2EXT_args[4] = {"op", "res", "arg1", "arg2"};
static const trace::FunctionSig _glShaderOp2EXT_sig = {2273, "glShaderOp2EXT", 4, _glShaderOp2EXT_args};

static const char * _glShaderOp3EXT_args[5] = {"op", "res", "arg1", "arg2", "arg3"};
static const trace::FunctionSig _glShaderOp3EXT_sig = {2274, "glShaderOp3EXT", 5, _glShaderOp3EXT_args};

static const char * _glSwizzleEXT_args[6] = {"res", "in", "outX", "outY", "outZ", "outW"};
static const trace::FunctionSig _glSwizzleEXT_sig = {2275, "glSwizzleEXT", 6, _glSwizzleEXT_args};

static const char * _glWriteMaskEXT_args[6] = {"res", "in", "outX", "outY", "outZ", "outW"};
static const trace::FunctionSig _glWriteMaskEXT_sig = {2276, "glWriteMaskEXT", 6, _glWriteMaskEXT_args};

static const char * _glInsertComponentEXT_args[3] = {"res", "src", "num"};
static const trace::FunctionSig _glInsertComponentEXT_sig = {2277, "glInsertComponentEXT", 3, _glInsertComponentEXT_args};

static const char * _glExtractComponentEXT_args[3] = {"res", "src", "num"};
static const trace::FunctionSig _glExtractComponentEXT_sig = {2278, "glExtractComponentEXT", 3, _glExtractComponentEXT_args};

static const char * _glGenSymbolsEXT_args[4] = {"datatype", "storagetype", "range", "components"};
static const trace::FunctionSig _glGenSymbolsEXT_sig = {2279, "glGenSymbolsEXT", 4, _glGenSymbolsEXT_args};

static const char * _glSetInvariantEXT_args[3] = {"id", "type", "addr"};
static const trace::FunctionSig _glSetInvariantEXT_sig = {2280, "glSetInvariantEXT", 3, _glSetInvariantEXT_args};

static const char * _glSetLocalConstantEXT_args[3] = {"id", "type", "addr"};
static const trace::FunctionSig _glSetLocalConstantEXT_sig = {2281, "glSetLocalConstantEXT", 3, _glSetLocalConstantEXT_args};

static const char * _glVariantbvEXT_args[2] = {"id", "addr"};
static const trace::FunctionSig _glVariantbvEXT_sig = {2282, "glVariantbvEXT", 2, _glVariantbvEXT_args};

static const char * _glVariantsvEXT_args[2] = {"id", "addr"};
static const trace::FunctionSig _glVariantsvEXT_sig = {2283, "glVariantsvEXT", 2, _glVariantsvEXT_args};

static const char * _glVariantivEXT_args[2] = {"id", "addr"};
static const trace::FunctionSig _glVariantivEXT_sig = {2284, "glVariantivEXT", 2, _glVariantivEXT_args};

static const char * _glVariantfvEXT_args[2] = {"id", "addr"};
static const trace::FunctionSig _glVariantfvEXT_sig = {2285, "glVariantfvEXT", 2, _glVariantfvEXT_args};

static const char * _glVariantdvEXT_args[2] = {"id", "addr"};
static const trace::FunctionSig _glVariantdvEXT_sig = {2286, "glVariantdvEXT", 2, _glVariantdvEXT_args};

static const char * _glVariantubvEXT_args[2] = {"id", "addr"};
static const trace::FunctionSig _glVariantubvEXT_sig = {2287, "glVariantubvEXT", 2, _glVariantubvEXT_args};

static const char * _glVariantusvEXT_args[2] = {"id", "addr"};
static const trace::FunctionSig _glVariantusvEXT_sig = {2288, "glVariantusvEXT", 2, _glVariantusvEXT_args};

static const char * _glVariantuivEXT_args[2] = {"id", "addr"};
static const trace::FunctionSig _glVariantuivEXT_sig = {2289, "glVariantuivEXT", 2, _glVariantuivEXT_args};

static const char * _glVariantPointerEXT_args[4] = {"id", "type", "stride", "addr"};
static const trace::FunctionSig _glVariantPointerEXT_sig = {2290, "glVariantPointerEXT", 4, _glVariantPointerEXT_args};

static const char * _glEnableVariantClientStateEXT_args[1] = {"id"};
static const trace::FunctionSig _glEnableVariantClientStateEXT_sig = {2291, "glEnableVariantClientStateEXT", 1, _glEnableVariantClientStateEXT_args};

static const char * _glDisableVariantClientStateEXT_args[1] = {"id"};
static const trace::FunctionSig _glDisableVariantClientStateEXT_sig = {2292, "glDisableVariantClientStateEXT", 1, _glDisableVariantClientStateEXT_args};

static const char * _glBindLightParameterEXT_args[2] = {"light", "value"};
static const trace::FunctionSig _glBindLightParameterEXT_sig = {2293, "glBindLightParameterEXT", 2, _glBindLightParameterEXT_args};

static const char * _glBindMaterialParameterEXT_args[2] = {"face", "value"};
static const trace::FunctionSig _glBindMaterialParameterEXT_sig = {2294, "glBindMaterialParameterEXT", 2, _glBindMaterialParameterEXT_args};

static const char * _glBindTexGenParameterEXT_args[3] = {"unit", "coord", "value"};
static const trace::FunctionSig _glBindTexGenParameterEXT_sig = {2295, "glBindTexGenParameterEXT", 3, _glBindTexGenParameterEXT_args};

static const char * _glBindTextureUnitParameterEXT_args[2] = {"unit", "value"};
static const trace::FunctionSig _glBindTextureUnitParameterEXT_sig = {2296, "glBindTextureUnitParameterEXT", 2, _glBindTextureUnitParameterEXT_args};

static const char * _glBindParameterEXT_args[1] = {"value"};
static const trace::FunctionSig _glBindParameterEXT_sig = {2297, "glBindParameterEXT", 1, _glBindParameterEXT_args};

static const char * _glIsVariantEnabledEXT_args[2] = {"id", "cap"};
static const trace::FunctionSig _glIsVariantEnabledEXT_sig = {2298, "glIsVariantEnabledEXT", 2, _glIsVariantEnabledEXT_args};

static const char * _glGetVariantBooleanvEXT_args[3] = {"id", "value", "data"};
static const trace::FunctionSig _glGetVariantBooleanvEXT_sig = {2299, "glGetVariantBooleanvEXT", 3, _glGetVariantBooleanvEXT_args};

static const char * _glGetVariantIntegervEXT_args[3] = {"id", "value", "data"};
static const trace::FunctionSig _glGetVariantIntegervEXT_sig = {2300, "glGetVariantIntegervEXT", 3, _glGetVariantIntegervEXT_args};

static const char * _glGetVariantFloatvEXT_args[3] = {"id", "value", "data"};
static const trace::FunctionSig _glGetVariantFloatvEXT_sig = {2301, "glGetVariantFloatvEXT", 3, _glGetVariantFloatvEXT_args};

static const char * _glGetVariantPointervEXT_args[3] = {"id", "value", "data"};
static const trace::FunctionSig _glGetVariantPointervEXT_sig = {2302, "glGetVariantPointervEXT", 3, _glGetVariantPointervEXT_args};

static const char * _glGetInvariantBooleanvEXT_args[3] = {"id", "value", "data"};
static const trace::FunctionSig _glGetInvariantBooleanvEXT_sig = {2303, "glGetInvariantBooleanvEXT", 3, _glGetInvariantBooleanvEXT_args};

static const char * _glGetInvariantIntegervEXT_args[3] = {"id", "value", "data"};
static const trace::FunctionSig _glGetInvariantIntegervEXT_sig = {2304, "glGetInvariantIntegervEXT", 3, _glGetInvariantIntegervEXT_args};

static const char * _glGetInvariantFloatvEXT_args[3] = {"id", "value", "data"};
static const trace::FunctionSig _glGetInvariantFloatvEXT_sig = {2305, "glGetInvariantFloatvEXT", 3, _glGetInvariantFloatvEXT_args};

static const char * _glGetLocalConstantBooleanvEXT_args[3] = {"id", "value", "data"};
static const trace::FunctionSig _glGetLocalConstantBooleanvEXT_sig = {2306, "glGetLocalConstantBooleanvEXT", 3, _glGetLocalConstantBooleanvEXT_args};

static const char * _glGetLocalConstantIntegervEXT_args[3] = {"id", "value", "data"};
static const trace::FunctionSig _glGetLocalConstantIntegervEXT_sig = {2307, "glGetLocalConstantIntegervEXT", 3, _glGetLocalConstantIntegervEXT_args};

static const char * _glGetLocalConstantFloatvEXT_args[3] = {"id", "value", "data"};
static const trace::FunctionSig _glGetLocalConstantFloatvEXT_sig = {2308, "glGetLocalConstantFloatvEXT", 3, _glGetLocalConstantFloatvEXT_args};

static const char * _glVertexWeightfEXT_args[1] = {"weight"};
static const trace::FunctionSig _glVertexWeightfEXT_sig = {2309, "glVertexWeightfEXT", 1, _glVertexWeightfEXT_args};

static const char * _glVertexWeightfvEXT_args[1] = {"weight"};
static const trace::FunctionSig _glVertexWeightfvEXT_sig = {2310, "glVertexWeightfvEXT", 1, _glVertexWeightfvEXT_args};

static const char * _glVertexWeightPointerEXT_args[4] = {"size", "type", "stride", "pointer"};
static const trace::FunctionSig _glVertexWeightPointerEXT_sig = {2311, "glVertexWeightPointerEXT", 4, _glVertexWeightPointerEXT_args};

static const char * _glImportSyncEXT_args[3] = {"external_sync_type", "external_sync", "flags"};
static const trace::FunctionSig _glImportSyncEXT_sig = {2312, "glImportSyncEXT", 3, _glImportSyncEXT_args};

static const char ** _glFrameTerminatorGREMEDY_args = NULL;
static const trace::FunctionSig _glFrameTerminatorGREMEDY_sig = {2313, "glFrameTerminatorGREMEDY", 0, _glFrameTerminatorGREMEDY_args};

static const char * _glStringMarkerGREMEDY_args[2] = {"len", "string"};
static const trace::FunctionSig _glStringMarkerGREMEDY_sig = {2314, "glStringMarkerGREMEDY", 2, _glStringMarkerGREMEDY_args};

static const char * _glImageTransformParameteriHP_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glImageTransformParameteriHP_sig = {2315, "glImageTransformParameteriHP", 3, _glImageTransformParameteriHP_args};

static const char * _glImageTransformParameterfHP_args[3] = {"target", "pname", "param"};
static const trace::FunctionSig _glImageTransformParameterfHP_sig = {2316, "glImageTransformParameterfHP", 3, _glImageTransformParameterfHP_args};

static const char * _glImageTransformParameterivHP_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glImageTransformParameterivHP_sig = {2317, "glImageTransformParameterivHP", 3, _glImageTransformParameterivHP_args};

static const char * _glImageTransformParameterfvHP_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glImageTransformParameterfvHP_sig = {2318, "glImageTransformParameterfvHP", 3, _glImageTransformParameterfvHP_args};

static const char * _glGetImageTransformParameterivHP_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetImageTransformParameterivHP_sig = {2319, "glGetImageTransformParameterivHP", 3, _glGetImageTransformParameterivHP_args};

static const char * _glGetImageTransformParameterfvHP_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetImageTransformParameterfvHP_sig = {2320, "glGetImageTransformParameterfvHP", 3, _glGetImageTransformParameterfvHP_args};

static const char * _glMultiModeDrawArraysIBM_args[5] = {"mode", "first", "count", "drawcount", "modestride"};
static const trace::FunctionSig _glMultiModeDrawArraysIBM_sig = {2321, "glMultiModeDrawArraysIBM", 5, _glMultiModeDrawArraysIBM_args};

static const char * _glMultiModeDrawElementsIBM_args[6] = {"mode", "count", "type", "indices", "drawcount", "modestride"};
static const trace::FunctionSig _glMultiModeDrawElementsIBM_sig = {2322, "glMultiModeDrawElementsIBM", 6, _glMultiModeDrawElementsIBM_args};

static const char * _glColorPointerListIBM_args[5] = {"size", "type", "stride", "pointer", "ptrstride"};
static const trace::FunctionSig _glColorPointerListIBM_sig = {2323, "glColorPointerListIBM", 5, _glColorPointerListIBM_args};

static const char * _glSecondaryColorPointerListIBM_args[5] = {"size", "type", "stride", "pointer", "ptrstride"};
static const trace::FunctionSig _glSecondaryColorPointerListIBM_sig = {2324, "glSecondaryColorPointerListIBM", 5, _glSecondaryColorPointerListIBM_args};

static const char * _glEdgeFlagPointerListIBM_args[3] = {"stride", "pointer", "ptrstride"};
static const trace::FunctionSig _glEdgeFlagPointerListIBM_sig = {2325, "glEdgeFlagPointerListIBM", 3, _glEdgeFlagPointerListIBM_args};

static const char * _glFogCoordPointerListIBM_args[4] = {"type", "stride", "pointer", "ptrstride"};
static const trace::FunctionSig _glFogCoordPointerListIBM_sig = {2326, "glFogCoordPointerListIBM", 4, _glFogCoordPointerListIBM_args};

static const char * _glIndexPointerListIBM_args[4] = {"type", "stride", "pointer", "ptrstride"};
static const trace::FunctionSig _glIndexPointerListIBM_sig = {2327, "glIndexPointerListIBM", 4, _glIndexPointerListIBM_args};

static const char * _glNormalPointerListIBM_args[4] = {"type", "stride", "pointer", "ptrstride"};
static const trace::FunctionSig _glNormalPointerListIBM_sig = {2328, "glNormalPointerListIBM", 4, _glNormalPointerListIBM_args};

static const char * _glTexCoordPointerListIBM_args[5] = {"size", "type", "stride", "pointer", "ptrstride"};
static const trace::FunctionSig _glTexCoordPointerListIBM_sig = {2329, "glTexCoordPointerListIBM", 5, _glTexCoordPointerListIBM_args};

static const char * _glVertexPointerListIBM_args[5] = {"size", "type", "stride", "pointer", "ptrstride"};
static const trace::FunctionSig _glVertexPointerListIBM_sig = {2330, "glVertexPointerListIBM", 5, _glVertexPointerListIBM_args};

static const char * _glRenderbufferStorageMultisampleIMG_args[5] = {"target", "samples", "internalformat", "width", "height"};
static const trace::FunctionSig _glRenderbufferStorageMultisampleIMG_sig = {2331, "glRenderbufferStorageMultisampleIMG", 5, _glRenderbufferStorageMultisampleIMG_args};

static const char * _glFramebufferTexture2DMultisampleIMG_args[6] = {"target", "attachment", "textarget", "texture", "level", "samples"};
static const trace::FunctionSig _glFramebufferTexture2DMultisampleIMG_sig = {2332, "glFramebufferTexture2DMultisampleIMG", 6, _glFramebufferTexture2DMultisampleIMG_args};

static const char * _glBlendFuncSeparateINGR_args[4] = {"sfactorRGB", "dfactorRGB", "sfactorAlpha", "dfactorAlpha"};
static const trace::FunctionSig _glBlendFuncSeparateINGR_sig = {2333, "glBlendFuncSeparateINGR", 4, _glBlendFuncSeparateINGR_args};

static const char * _glSyncTextureINTEL_args[1] = {"texture"};
static const trace::FunctionSig _glSyncTextureINTEL_sig = {2334, "glSyncTextureINTEL", 1, _glSyncTextureINTEL_args};

static const char * _glUnmapTexture2DINTEL_args[2] = {"texture", "level"};
static const trace::FunctionSig _glUnmapTexture2DINTEL_sig = {2335, "glUnmapTexture2DINTEL", 2, _glUnmapTexture2DINTEL_args};

static const char * _glMapTexture2DINTEL_args[5] = {"texture", "level", "access", "stride", "layout"};
static const trace::FunctionSig _glMapTexture2DINTEL_sig = {2336, "glMapTexture2DINTEL", 5, _glMapTexture2DINTEL_args};

static const char * _glVertexPointervINTEL_args[3] = {"size", "type", "pointer"};
static const trace::FunctionSig _glVertexPointervINTEL_sig = {2337, "glVertexPointervINTEL", 3, _glVertexPointervINTEL_args};

static const char * _glNormalPointervINTEL_args[2] = {"type", "pointer"};
static const trace::FunctionSig _glNormalPointervINTEL_sig = {2338, "glNormalPointervINTEL", 2, _glNormalPointervINTEL_args};

static const char * _glColorPointervINTEL_args[3] = {"size", "type", "pointer"};
static const trace::FunctionSig _glColorPointervINTEL_sig = {2339, "glColorPointervINTEL", 3, _glColorPointervINTEL_args};

static const char * _glTexCoordPointervINTEL_args[3] = {"size", "type", "pointer"};
static const trace::FunctionSig _glTexCoordPointervINTEL_sig = {2340, "glTexCoordPointervINTEL", 3, _glTexCoordPointervINTEL_args};

static const char * _glBeginPerfQueryINTEL_args[1] = {"queryHandle"};
static const trace::FunctionSig _glBeginPerfQueryINTEL_sig = {2341, "glBeginPerfQueryINTEL", 1, _glBeginPerfQueryINTEL_args};

static const char * _glCreatePerfQueryINTEL_args[2] = {"queryId", "queryHandle"};
static const trace::FunctionSig _glCreatePerfQueryINTEL_sig = {2342, "glCreatePerfQueryINTEL", 2, _glCreatePerfQueryINTEL_args};

static const char * _glDeletePerfQueryINTEL_args[1] = {"queryHandle"};
static const trace::FunctionSig _glDeletePerfQueryINTEL_sig = {2343, "glDeletePerfQueryINTEL", 1, _glDeletePerfQueryINTEL_args};

static const char * _glEndPerfQueryINTEL_args[1] = {"queryHandle"};
static const trace::FunctionSig _glEndPerfQueryINTEL_sig = {2344, "glEndPerfQueryINTEL", 1, _glEndPerfQueryINTEL_args};

static const char * _glGetFirstPerfQueryIdINTEL_args[1] = {"queryId"};
static const trace::FunctionSig _glGetFirstPerfQueryIdINTEL_sig = {2345, "glGetFirstPerfQueryIdINTEL", 1, _glGetFirstPerfQueryIdINTEL_args};

static const char * _glGetNextPerfQueryIdINTEL_args[2] = {"queryId", "nextQueryId"};
static const trace::FunctionSig _glGetNextPerfQueryIdINTEL_sig = {2346, "glGetNextPerfQueryIdINTEL", 2, _glGetNextPerfQueryIdINTEL_args};

static const char * _glGetPerfCounterInfoINTEL_args[11] = {"queryId", "counterId", "counterNameLength", "counterName", "counterDescLength", "counterDesc", "counterOffset", "counterDataSize", "counterTypeEnum", "counterDataTypeEnum", "rawCounterMaxValue"};
static const trace::FunctionSig _glGetPerfCounterInfoINTEL_sig = {2347, "glGetPerfCounterInfoINTEL", 11, _glGetPerfCounterInfoINTEL_args};

static const char * _glGetPerfQueryDataINTEL_args[5] = {"queryHandle", "flags", "dataSize", "data", "bytesWritten"};
static const trace::FunctionSig _glGetPerfQueryDataINTEL_sig = {2348, "glGetPerfQueryDataINTEL", 5, _glGetPerfQueryDataINTEL_args};

static const char * _glGetPerfQueryIdByNameINTEL_args[2] = {"queryName", "queryId"};
static const trace::FunctionSig _glGetPerfQueryIdByNameINTEL_sig = {2349, "glGetPerfQueryIdByNameINTEL", 2, _glGetPerfQueryIdByNameINTEL_args};

static const char * _glGetPerfQueryInfoINTEL_args[7] = {"queryId", "queryNameLength", "queryName", "dataSize", "noCounters", "noInstances", "capsMask"};
static const trace::FunctionSig _glGetPerfQueryInfoINTEL_sig = {2350, "glGetPerfQueryInfoINTEL", 7, _glGetPerfQueryInfoINTEL_args};

static const char ** _glBlendBarrierKHR_args = NULL;
static const trace::FunctionSig _glBlendBarrierKHR_sig = {2351, "glBlendBarrierKHR", 0, _glBlendBarrierKHR_args};

static const char * _glDebugMessageControl_args[6] = {"source", "type", "severity", "count", "ids", "enabled"};
static const trace::FunctionSig _glDebugMessageControl_sig = {2352, "glDebugMessageControl", 6, _glDebugMessageControl_args};

static const char * _glDebugMessageInsert_args[6] = {"source", "type", "id", "severity", "length", "buf"};
static const trace::FunctionSig _glDebugMessageInsert_sig = {2353, "glDebugMessageInsert", 6, _glDebugMessageInsert_args};

static const char * _glDebugMessageCallback_args[2] = {"callback", "userParam"};
static const trace::FunctionSig _glDebugMessageCallback_sig = {2354, "glDebugMessageCallback", 2, _glDebugMessageCallback_args};

static const char * _glGetDebugMessageLog_args[8] = {"count", "bufsize", "sources", "types", "ids", "severities", "lengths", "messageLog"};
static const trace::FunctionSig _glGetDebugMessageLog_sig = {2355, "glGetDebugMessageLog", 8, _glGetDebugMessageLog_args};

static const char * _glPushDebugGroup_args[4] = {"source", "id", "length", "message"};
static const trace::FunctionSig _glPushDebugGroup_sig = {2356, "glPushDebugGroup", 4, _glPushDebugGroup_args};

static const char ** _glPopDebugGroup_args = NULL;
static const trace::FunctionSig _glPopDebugGroup_sig = {2357, "glPopDebugGroup", 0, _glPopDebugGroup_args};

static const char * _glObjectLabel_args[4] = {"identifier", "name", "length", "label"};
static const trace::FunctionSig _glObjectLabel_sig = {2358, "glObjectLabel", 4, _glObjectLabel_args};

static const char * _glGetObjectLabel_args[5] = {"identifier", "name", "bufSize", "length", "label"};
static const trace::FunctionSig _glGetObjectLabel_sig = {2359, "glGetObjectLabel", 5, _glGetObjectLabel_args};

static const char * _glObjectPtrLabel_args[3] = {"ptr", "length", "label"};
static const trace::FunctionSig _glObjectPtrLabel_sig = {2360, "glObjectPtrLabel", 3, _glObjectPtrLabel_args};

static const char * _glGetObjectPtrLabel_args[4] = {"ptr", "bufSize", "length", "label"};
static const trace::FunctionSig _glGetObjectPtrLabel_sig = {2361, "glGetObjectPtrLabel", 4, _glGetObjectPtrLabel_args};

static const char * _glDebugMessageControlKHR_args[6] = {"source", "type", "severity", "count", "ids", "enabled"};
static const trace::FunctionSig _glDebugMessageControlKHR_sig = {2362, "glDebugMessageControlKHR", 6, _glDebugMessageControlKHR_args};

static const char * _glDebugMessageInsertKHR_args[6] = {"source", "type", "id", "severity", "length", "buf"};
static const trace::FunctionSig _glDebugMessageInsertKHR_sig = {2363, "glDebugMessageInsertKHR", 6, _glDebugMessageInsertKHR_args};

static const char * _glDebugMessageCallbackKHR_args[2] = {"callback", "userParam"};
static const trace::FunctionSig _glDebugMessageCallbackKHR_sig = {2364, "glDebugMessageCallbackKHR", 2, _glDebugMessageCallbackKHR_args};

static const char * _glGetDebugMessageLogKHR_args[8] = {"count", "bufsize", "sources", "types", "ids", "severities", "lengths", "messageLog"};
static const trace::FunctionSig _glGetDebugMessageLogKHR_sig = {2365, "glGetDebugMessageLogKHR", 8, _glGetDebugMessageLogKHR_args};

static const char * _glPushDebugGroupKHR_args[4] = {"source", "id", "length", "message"};
static const trace::FunctionSig _glPushDebugGroupKHR_sig = {2366, "glPushDebugGroupKHR", 4, _glPushDebugGroupKHR_args};

static const char ** _glPopDebugGroupKHR_args = NULL;
static const trace::FunctionSig _glPopDebugGroupKHR_sig = {2367, "glPopDebugGroupKHR", 0, _glPopDebugGroupKHR_args};

static const char * _glObjectLabelKHR_args[4] = {"identifier", "name", "length", "label"};
static const trace::FunctionSig _glObjectLabelKHR_sig = {2368, "glObjectLabelKHR", 4, _glObjectLabelKHR_args};

static const char * _glGetObjectLabelKHR_args[5] = {"identifier", "name", "bufSize", "length", "label"};
static const trace::FunctionSig _glGetObjectLabelKHR_sig = {2369, "glGetObjectLabelKHR", 5, _glGetObjectLabelKHR_args};

static const char * _glObjectPtrLabelKHR_args[3] = {"ptr", "length", "label"};
static const trace::FunctionSig _glObjectPtrLabelKHR_sig = {2370, "glObjectPtrLabelKHR", 3, _glObjectPtrLabelKHR_args};

static const char * _glGetObjectPtrLabelKHR_args[4] = {"ptr", "bufSize", "length", "label"};
static const trace::FunctionSig _glGetObjectPtrLabelKHR_sig = {2371, "glGetObjectPtrLabelKHR", 4, _glGetObjectPtrLabelKHR_args};

static const char * _glGetPointervKHR_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetPointervKHR_sig = {2372, "glGetPointervKHR", 2, _glGetPointervKHR_args};

static const char ** _glGetGraphicsResetStatus_args = NULL;
static const trace::FunctionSig _glGetGraphicsResetStatus_sig = {2373, "glGetGraphicsResetStatus", 0, _glGetGraphicsResetStatus_args};

static const char * _glReadnPixels_args[8] = {"x", "y", "width", "height", "format", "type", "bufSize", "data"};
static const trace::FunctionSig _glReadnPixels_sig = {2374, "glReadnPixels", 8, _glReadnPixels_args};

static const char * _glGetnUniformfv_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformfv_sig = {2375, "glGetnUniformfv", 4, _glGetnUniformfv_args};

static const char * _glGetnUniformiv_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformiv_sig = {2376, "glGetnUniformiv", 4, _glGetnUniformiv_args};

static const char * _glGetnUniformuiv_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformuiv_sig = {2377, "glGetnUniformuiv", 4, _glGetnUniformuiv_args};

static const char ** _glGetGraphicsResetStatusKHR_args = NULL;
static const trace::FunctionSig _glGetGraphicsResetStatusKHR_sig = {2378, "glGetGraphicsResetStatusKHR", 0, _glGetGraphicsResetStatusKHR_args};

static const char * _glReadnPixelsKHR_args[8] = {"x", "y", "width", "height", "format", "type", "bufSize", "data"};
static const trace::FunctionSig _glReadnPixelsKHR_sig = {2379, "glReadnPixelsKHR", 8, _glReadnPixelsKHR_args};

static const char * _glGetnUniformfvKHR_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformfvKHR_sig = {2380, "glGetnUniformfvKHR", 4, _glGetnUniformfvKHR_args};

static const char * _glGetnUniformivKHR_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformivKHR_sig = {2381, "glGetnUniformivKHR", 4, _glGetnUniformivKHR_args};

static const char * _glGetnUniformuivKHR_args[4] = {"program", "location", "bufSize", "params"};
static const trace::FunctionSig _glGetnUniformuivKHR_sig = {2382, "glGetnUniformuivKHR", 4, _glGetnUniformuivKHR_args};

static const char * _glNewBufferRegion_args[1] = {"type"};
static const trace::FunctionSig _glNewBufferRegion_sig = {2383, "glNewBufferRegion", 1, _glNewBufferRegion_args};

static const char * _glDeleteBufferRegion_args[1] = {"region"};
static const trace::FunctionSig _glDeleteBufferRegion_sig = {2384, "glDeleteBufferRegion", 1, _glDeleteBufferRegion_args};

static const char * _glReadBufferRegion_args[5] = {"region", "x", "y", "width", "height"};
static const trace::FunctionSig _glReadBufferRegion_sig = {2385, "glReadBufferRegion", 5, _glReadBufferRegion_args};

static const char * _glDrawBufferRegion_args[7] = {"region", "x", "y", "width", "height", "xDest", "yDest"};
static const trace::FunctionSig _glDrawBufferRegion_sig = {2386, "glDrawBufferRegion", 7, _glDrawBufferRegion_args};

static const char ** _glBufferRegionEnabled_args = NULL;
static const trace::FunctionSig _glBufferRegionEnabled_sig = {2387, "glBufferRegionEnabled", 0, _glBufferRegionEnabled_args};

static const char ** _glResizeBuffersMESA_args = NULL;
static const trace::FunctionSig _glResizeBuffersMESA_sig = {2388, "glResizeBuffersMESA", 0, _glResizeBuffersMESA_args};

static const char * _glWindowPos2dMESA_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2dMESA_sig = {2389, "glWindowPos2dMESA", 2, _glWindowPos2dMESA_args};

static const char * _glWindowPos2dvMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2dvMESA_sig = {2390, "glWindowPos2dvMESA", 1, _glWindowPos2dvMESA_args};

static const char * _glWindowPos2fMESA_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2fMESA_sig = {2391, "glWindowPos2fMESA", 2, _glWindowPos2fMESA_args};

static const char * _glWindowPos2fvMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2fvMESA_sig = {2392, "glWindowPos2fvMESA", 1, _glWindowPos2fvMESA_args};

static const char * _glWindowPos2iMESA_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2iMESA_sig = {2393, "glWindowPos2iMESA", 2, _glWindowPos2iMESA_args};

static const char * _glWindowPos2ivMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2ivMESA_sig = {2394, "glWindowPos2ivMESA", 1, _glWindowPos2ivMESA_args};

static const char * _glWindowPos2sMESA_args[2] = {"x", "y"};
static const trace::FunctionSig _glWindowPos2sMESA_sig = {2395, "glWindowPos2sMESA", 2, _glWindowPos2sMESA_args};

static const char * _glWindowPos2svMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos2svMESA_sig = {2396, "glWindowPos2svMESA", 1, _glWindowPos2svMESA_args};

static const char * _glWindowPos3dMESA_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3dMESA_sig = {2397, "glWindowPos3dMESA", 3, _glWindowPos3dMESA_args};

static const char * _glWindowPos3dvMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3dvMESA_sig = {2398, "glWindowPos3dvMESA", 1, _glWindowPos3dvMESA_args};

static const char * _glWindowPos3fMESA_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3fMESA_sig = {2399, "glWindowPos3fMESA", 3, _glWindowPos3fMESA_args};

static const char * _glWindowPos3fvMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3fvMESA_sig = {2400, "glWindowPos3fvMESA", 1, _glWindowPos3fvMESA_args};

static const char * _glWindowPos3iMESA_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3iMESA_sig = {2401, "glWindowPos3iMESA", 3, _glWindowPos3iMESA_args};

static const char * _glWindowPos3ivMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3ivMESA_sig = {2402, "glWindowPos3ivMESA", 1, _glWindowPos3ivMESA_args};

static const char * _glWindowPos3sMESA_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glWindowPos3sMESA_sig = {2403, "glWindowPos3sMESA", 3, _glWindowPos3sMESA_args};

static const char * _glWindowPos3svMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos3svMESA_sig = {2404, "glWindowPos3svMESA", 1, _glWindowPos3svMESA_args};

static const char * _glWindowPos4dMESA_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glWindowPos4dMESA_sig = {2405, "glWindowPos4dMESA", 4, _glWindowPos4dMESA_args};

static const char * _glWindowPos4dvMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos4dvMESA_sig = {2406, "glWindowPos4dvMESA", 1, _glWindowPos4dvMESA_args};

static const char * _glWindowPos4fMESA_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glWindowPos4fMESA_sig = {2407, "glWindowPos4fMESA", 4, _glWindowPos4fMESA_args};

static const char * _glWindowPos4fvMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos4fvMESA_sig = {2408, "glWindowPos4fvMESA", 1, _glWindowPos4fvMESA_args};

static const char * _glWindowPos4iMESA_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glWindowPos4iMESA_sig = {2409, "glWindowPos4iMESA", 4, _glWindowPos4iMESA_args};

static const char * _glWindowPos4ivMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos4ivMESA_sig = {2410, "glWindowPos4ivMESA", 1, _glWindowPos4ivMESA_args};

static const char * _glWindowPos4sMESA_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glWindowPos4sMESA_sig = {2411, "glWindowPos4sMESA", 4, _glWindowPos4sMESA_args};

static const char * _glWindowPos4svMESA_args[1] = {"v"};
static const trace::FunctionSig _glWindowPos4svMESA_sig = {2412, "glWindowPos4svMESA", 1, _glWindowPos4svMESA_args};

static const char * _glBeginConditionalRenderNVX_args[1] = {"id"};
static const trace::FunctionSig _glBeginConditionalRenderNVX_sig = {2413, "glBeginConditionalRenderNVX", 1, _glBeginConditionalRenderNVX_args};

static const char ** _glEndConditionalRenderNVX_args = NULL;
static const trace::FunctionSig _glEndConditionalRenderNVX_sig = {2414, "glEndConditionalRenderNVX", 0, _glEndConditionalRenderNVX_args};

static const char * _glMultiDrawArraysIndirectBindlessNV_args[5] = {"mode", "indirect", "drawCount", "stride", "vertexBufferCount"};
static const trace::FunctionSig _glMultiDrawArraysIndirectBindlessNV_sig = {2415, "glMultiDrawArraysIndirectBindlessNV", 5, _glMultiDrawArraysIndirectBindlessNV_args};

static const char * _glMultiDrawElementsIndirectBindlessNV_args[6] = {"mode", "type", "indirect", "drawCount", "stride", "vertexBufferCount"};
static const trace::FunctionSig _glMultiDrawElementsIndirectBindlessNV_sig = {2416, "glMultiDrawElementsIndirectBindlessNV", 6, _glMultiDrawElementsIndirectBindlessNV_args};

static const char * _glGetTextureHandleNV_args[1] = {"texture"};
static const trace::FunctionSig _glGetTextureHandleNV_sig = {2417, "glGetTextureHandleNV", 1, _glGetTextureHandleNV_args};

static const char * _glGetTextureSamplerHandleNV_args[2] = {"texture", "sampler"};
static const trace::FunctionSig _glGetTextureSamplerHandleNV_sig = {2418, "glGetTextureSamplerHandleNV", 2, _glGetTextureSamplerHandleNV_args};

static const char * _glMakeTextureHandleResidentNV_args[1] = {"handle"};
static const trace::FunctionSig _glMakeTextureHandleResidentNV_sig = {2419, "glMakeTextureHandleResidentNV", 1, _glMakeTextureHandleResidentNV_args};

static const char * _glMakeTextureHandleNonResidentNV_args[1] = {"handle"};
static const trace::FunctionSig _glMakeTextureHandleNonResidentNV_sig = {2420, "glMakeTextureHandleNonResidentNV", 1, _glMakeTextureHandleNonResidentNV_args};

static const char * _glGetImageHandleNV_args[5] = {"texture", "level", "layered", "layer", "format"};
static const trace::FunctionSig _glGetImageHandleNV_sig = {2421, "glGetImageHandleNV", 5, _glGetImageHandleNV_args};

static const char * _glMakeImageHandleResidentNV_args[2] = {"handle", "access"};
static const trace::FunctionSig _glMakeImageHandleResidentNV_sig = {2422, "glMakeImageHandleResidentNV", 2, _glMakeImageHandleResidentNV_args};

static const char * _glMakeImageHandleNonResidentNV_args[1] = {"handle"};
static const trace::FunctionSig _glMakeImageHandleNonResidentNV_sig = {2423, "glMakeImageHandleNonResidentNV", 1, _glMakeImageHandleNonResidentNV_args};

static const char * _glUniformHandleui64NV_args[2] = {"location", "value"};
static const trace::FunctionSig _glUniformHandleui64NV_sig = {2424, "glUniformHandleui64NV", 2, _glUniformHandleui64NV_args};

static const char * _glUniformHandleui64vNV_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniformHandleui64vNV_sig = {2425, "glUniformHandleui64vNV", 3, _glUniformHandleui64vNV_args};

static const char * _glProgramUniformHandleui64NV_args[3] = {"program", "location", "value"};
static const trace::FunctionSig _glProgramUniformHandleui64NV_sig = {2426, "glProgramUniformHandleui64NV", 3, _glProgramUniformHandleui64NV_args};

static const char * _glProgramUniformHandleui64vNV_args[4] = {"program", "location", "count", "values"};
static const trace::FunctionSig _glProgramUniformHandleui64vNV_sig = {2427, "glProgramUniformHandleui64vNV", 4, _glProgramUniformHandleui64vNV_args};

static const char * _glIsTextureHandleResidentNV_args[1] = {"handle"};
static const trace::FunctionSig _glIsTextureHandleResidentNV_sig = {2428, "glIsTextureHandleResidentNV", 1, _glIsTextureHandleResidentNV_args};

static const char * _glIsImageHandleResidentNV_args[1] = {"handle"};
static const trace::FunctionSig _glIsImageHandleResidentNV_sig = {2429, "glIsImageHandleResidentNV", 1, _glIsImageHandleResidentNV_args};

static const char * _glBlendParameteriNV_args[2] = {"pname", "value"};
static const trace::FunctionSig _glBlendParameteriNV_sig = {2430, "glBlendParameteriNV", 2, _glBlendParameteriNV_args};

static const char ** _glBlendBarrierNV_args = NULL;
static const trace::FunctionSig _glBlendBarrierNV_sig = {2431, "glBlendBarrierNV", 0, _glBlendBarrierNV_args};

static const char * _glBeginConditionalRenderNV_args[2] = {"id", "mode"};
static const trace::FunctionSig _glBeginConditionalRenderNV_sig = {2432, "glBeginConditionalRenderNV", 2, _glBeginConditionalRenderNV_args};

static const char ** _glEndConditionalRenderNV_args = NULL;
static const trace::FunctionSig _glEndConditionalRenderNV_sig = {2433, "glEndConditionalRenderNV", 0, _glEndConditionalRenderNV_args};

static const char * _glCopyImageSubDataNV_args[15] = {"srcName", "srcTarget", "srcLevel", "srcX", "srcY", "srcZ", "dstName", "dstTarget", "dstLevel", "dstX", "dstY", "dstZ", "width", "height", "depth"};
static const trace::FunctionSig _glCopyImageSubDataNV_sig = {2434, "glCopyImageSubDataNV", 15, _glCopyImageSubDataNV_args};

static const char * _glCoverageMaskNV_args[1] = {"mask"};
static const trace::FunctionSig _glCoverageMaskNV_sig = {2435, "glCoverageMaskNV", 1, _glCoverageMaskNV_args};

static const char * _glCoverageOperationNV_args[1] = {"operation"};
static const trace::FunctionSig _glCoverageOperationNV_sig = {2436, "glCoverageOperationNV", 1, _glCoverageOperationNV_args};

static const char * _glDepthRangedNV_args[2] = {"zNear", "zFar"};
static const trace::FunctionSig _glDepthRangedNV_sig = {2437, "glDepthRangedNV", 2, _glDepthRangedNV_args};

static const char * _glClearDepthdNV_args[1] = {"depth"};
static const trace::FunctionSig _glClearDepthdNV_sig = {2438, "glClearDepthdNV", 1, _glClearDepthdNV_args};

static const char * _glDepthBoundsdNV_args[2] = {"zmin", "zmax"};
static const trace::FunctionSig _glDepthBoundsdNV_sig = {2439, "glDepthBoundsdNV", 2, _glDepthBoundsdNV_args};

static const char * _glDrawBuffersNV_args[2] = {"n", "bufs"};
static const trace::FunctionSig _glDrawBuffersNV_sig = {2440, "glDrawBuffersNV", 2, _glDrawBuffersNV_args};

static const char * _glDrawTextureNV_args[11] = {"texture", "sampler", "x0", "y0", "x1", "y1", "z", "s0", "t0", "s1", "t1"};
static const trace::FunctionSig _glDrawTextureNV_sig = {2441, "glDrawTextureNV", 11, _glDrawTextureNV_args};

static const char * _glMapControlPointsNV_args[9] = {"target", "index", "type", "ustride", "vstride", "uorder", "vorder", "packed", "points"};
static const trace::FunctionSig _glMapControlPointsNV_sig = {2442, "glMapControlPointsNV", 9, _glMapControlPointsNV_args};

static const char * _glMapParameterivNV_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glMapParameterivNV_sig = {2443, "glMapParameterivNV", 3, _glMapParameterivNV_args};

static const char * _glMapParameterfvNV_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glMapParameterfvNV_sig = {2444, "glMapParameterfvNV", 3, _glMapParameterfvNV_args};

static const char * _glGetMapControlPointsNV_args[7] = {"target", "index", "type", "ustride", "vstride", "packed", "points"};
static const trace::FunctionSig _glGetMapControlPointsNV_sig = {2445, "glGetMapControlPointsNV", 7, _glGetMapControlPointsNV_args};

static const char * _glGetMapParameterivNV_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetMapParameterivNV_sig = {2446, "glGetMapParameterivNV", 3, _glGetMapParameterivNV_args};

static const char * _glGetMapParameterfvNV_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetMapParameterfvNV_sig = {2447, "glGetMapParameterfvNV", 3, _glGetMapParameterfvNV_args};

static const char * _glGetMapAttribParameterivNV_args[4] = {"target", "index", "pname", "params"};
static const trace::FunctionSig _glGetMapAttribParameterivNV_sig = {2448, "glGetMapAttribParameterivNV", 4, _glGetMapAttribParameterivNV_args};

static const char * _glGetMapAttribParameterfvNV_args[4] = {"target", "index", "pname", "params"};
static const trace::FunctionSig _glGetMapAttribParameterfvNV_sig = {2449, "glGetMapAttribParameterfvNV", 4, _glGetMapAttribParameterfvNV_args};

static const char * _glEvalMapsNV_args[2] = {"target", "mode"};
static const trace::FunctionSig _glEvalMapsNV_sig = {2450, "glEvalMapsNV", 2, _glEvalMapsNV_args};

static const char * _glGetMultisamplefvNV_args[3] = {"pname", "index", "val"};
static const trace::FunctionSig _glGetMultisamplefvNV_sig = {2451, "glGetMultisamplefvNV", 3, _glGetMultisamplefvNV_args};

static const char * _glSampleMaskIndexedNV_args[2] = {"index", "mask"};
static const trace::FunctionSig _glSampleMaskIndexedNV_sig = {2452, "glSampleMaskIndexedNV", 2, _glSampleMaskIndexedNV_args};

static const char * _glTexRenderbufferNV_args[2] = {"target", "renderbuffer"};
static const trace::FunctionSig _glTexRenderbufferNV_sig = {2453, "glTexRenderbufferNV", 2, _glTexRenderbufferNV_args};

static const char * _glDeleteFencesNV_args[2] = {"n", "fences"};
static const trace::FunctionSig _glDeleteFencesNV_sig = {2454, "glDeleteFencesNV", 2, _glDeleteFencesNV_args};

static const char * _glGenFencesNV_args[2] = {"n", "fences"};
static const trace::FunctionSig _glGenFencesNV_sig = {2455, "glGenFencesNV", 2, _glGenFencesNV_args};

static const char * _glIsFenceNV_args[1] = {"fence"};
static const trace::FunctionSig _glIsFenceNV_sig = {2456, "glIsFenceNV", 1, _glIsFenceNV_args};

static const char * _glTestFenceNV_args[1] = {"fence"};
static const trace::FunctionSig _glTestFenceNV_sig = {2457, "glTestFenceNV", 1, _glTestFenceNV_args};

static const char * _glGetFenceivNV_args[3] = {"fence", "pname", "params"};
static const trace::FunctionSig _glGetFenceivNV_sig = {2458, "glGetFenceivNV", 3, _glGetFenceivNV_args};

static const char * _glFinishFenceNV_args[1] = {"fence"};
static const trace::FunctionSig _glFinishFenceNV_sig = {2459, "glFinishFenceNV", 1, _glFinishFenceNV_args};

static const char * _glSetFenceNV_args[2] = {"fence", "condition"};
static const trace::FunctionSig _glSetFenceNV_sig = {2460, "glSetFenceNV", 2, _glSetFenceNV_args};

static const char * _glProgramNamedParameter4fNV_args[7] = {"id", "len", "name", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramNamedParameter4fNV_sig = {2461, "glProgramNamedParameter4fNV", 7, _glProgramNamedParameter4fNV_args};

static const char * _glProgramNamedParameter4dNV_args[7] = {"id", "len", "name", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramNamedParameter4dNV_sig = {2462, "glProgramNamedParameter4dNV", 7, _glProgramNamedParameter4dNV_args};

static const char * _glProgramNamedParameter4fvNV_args[4] = {"id", "len", "name", "v"};
static const trace::FunctionSig _glProgramNamedParameter4fvNV_sig = {2463, "glProgramNamedParameter4fvNV", 4, _glProgramNamedParameter4fvNV_args};

static const char * _glProgramNamedParameter4dvNV_args[4] = {"id", "len", "name", "v"};
static const trace::FunctionSig _glProgramNamedParameter4dvNV_sig = {2464, "glProgramNamedParameter4dvNV", 4, _glProgramNamedParameter4dvNV_args};

static const char * _glGetProgramNamedParameterdvNV_args[4] = {"id", "len", "name", "params"};
static const trace::FunctionSig _glGetProgramNamedParameterdvNV_sig = {2465, "glGetProgramNamedParameterdvNV", 4, _glGetProgramNamedParameterdvNV_args};

static const char * _glGetProgramNamedParameterfvNV_args[4] = {"id", "len", "name", "params"};
static const trace::FunctionSig _glGetProgramNamedParameterfvNV_sig = {2466, "glGetProgramNamedParameterfvNV", 4, _glGetProgramNamedParameterfvNV_args};

static const char * _glBlitFramebufferNV_args[10] = {"srcX0", "srcY0", "srcX1", "srcY1", "dstX0", "dstY0", "dstX1", "dstY1", "mask", "filter"};
static const trace::FunctionSig _glBlitFramebufferNV_sig = {2467, "glBlitFramebufferNV", 10, _glBlitFramebufferNV_args};

static const char * _glCoverageModulationTableNV_args[2] = {"n", "v"};
static const trace::FunctionSig _glCoverageModulationTableNV_sig = {2468, "glCoverageModulationTableNV", 2, _glCoverageModulationTableNV_args};

static const char * _glGetCoverageModulationTableNV_args[2] = {"bufsize", "v"};
static const trace::FunctionSig _glGetCoverageModulationTableNV_sig = {2469, "glGetCoverageModulationTableNV", 2, _glGetCoverageModulationTableNV_args};

static const char * _glCoverageModulationNV_args[1] = {"components"};
static const trace::FunctionSig _glCoverageModulationNV_sig = {2470, "glCoverageModulationNV", 1, _glCoverageModulationNV_args};

static const char * _glRenderbufferStorageMultisampleCoverageNV_args[6] = {"target", "coverageSamples", "colorSamples", "internalformat", "width", "height"};
static const trace::FunctionSig _glRenderbufferStorageMultisampleCoverageNV_sig = {2471, "glRenderbufferStorageMultisampleCoverageNV", 6, _glRenderbufferStorageMultisampleCoverageNV_args};

static const char * _glProgramVertexLimitNV_args[2] = {"target", "limit"};
static const trace::FunctionSig _glProgramVertexLimitNV_sig = {2472, "glProgramVertexLimitNV", 2, _glProgramVertexLimitNV_args};

static const char * _glFramebufferTextureFaceEXT_args[5] = {"target", "attachment", "texture", "level", "face"};
static const trace::FunctionSig _glFramebufferTextureFaceEXT_sig = {2473, "glFramebufferTextureFaceEXT", 5, _glFramebufferTextureFaceEXT_args};

static const char * _glProgramLocalParameterI4iNV_args[6] = {"target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramLocalParameterI4iNV_sig = {2474, "glProgramLocalParameterI4iNV", 6, _glProgramLocalParameterI4iNV_args};

static const char * _glProgramLocalParameterI4ivNV_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glProgramLocalParameterI4ivNV_sig = {2475, "glProgramLocalParameterI4ivNV", 3, _glProgramLocalParameterI4ivNV_args};

static const char * _glProgramLocalParametersI4ivNV_args[4] = {"target", "index", "count", "params"};
static const trace::FunctionSig _glProgramLocalParametersI4ivNV_sig = {2476, "glProgramLocalParametersI4ivNV", 4, _glProgramLocalParametersI4ivNV_args};

static const char * _glProgramLocalParameterI4uiNV_args[6] = {"target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramLocalParameterI4uiNV_sig = {2477, "glProgramLocalParameterI4uiNV", 6, _glProgramLocalParameterI4uiNV_args};

static const char * _glProgramLocalParameterI4uivNV_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glProgramLocalParameterI4uivNV_sig = {2478, "glProgramLocalParameterI4uivNV", 3, _glProgramLocalParameterI4uivNV_args};

static const char * _glProgramLocalParametersI4uivNV_args[4] = {"target", "index", "count", "params"};
static const trace::FunctionSig _glProgramLocalParametersI4uivNV_sig = {2479, "glProgramLocalParametersI4uivNV", 4, _glProgramLocalParametersI4uivNV_args};

static const char * _glProgramEnvParameterI4iNV_args[6] = {"target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramEnvParameterI4iNV_sig = {2480, "glProgramEnvParameterI4iNV", 6, _glProgramEnvParameterI4iNV_args};

static const char * _glProgramEnvParameterI4ivNV_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glProgramEnvParameterI4ivNV_sig = {2481, "glProgramEnvParameterI4ivNV", 3, _glProgramEnvParameterI4ivNV_args};

static const char * _glProgramEnvParametersI4ivNV_args[4] = {"target", "index", "count", "params"};
static const trace::FunctionSig _glProgramEnvParametersI4ivNV_sig = {2482, "glProgramEnvParametersI4ivNV", 4, _glProgramEnvParametersI4ivNV_args};

static const char * _glProgramEnvParameterI4uiNV_args[6] = {"target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramEnvParameterI4uiNV_sig = {2483, "glProgramEnvParameterI4uiNV", 6, _glProgramEnvParameterI4uiNV_args};

static const char * _glProgramEnvParameterI4uivNV_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glProgramEnvParameterI4uivNV_sig = {2484, "glProgramEnvParameterI4uivNV", 3, _glProgramEnvParameterI4uivNV_args};

static const char * _glProgramEnvParametersI4uivNV_args[4] = {"target", "index", "count", "params"};
static const trace::FunctionSig _glProgramEnvParametersI4uivNV_sig = {2485, "glProgramEnvParametersI4uivNV", 4, _glProgramEnvParametersI4uivNV_args};

static const char * _glGetProgramLocalParameterIivNV_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glGetProgramLocalParameterIivNV_sig = {2486, "glGetProgramLocalParameterIivNV", 3, _glGetProgramLocalParameterIivNV_args};

static const char * _glGetProgramLocalParameterIuivNV_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glGetProgramLocalParameterIuivNV_sig = {2487, "glGetProgramLocalParameterIuivNV", 3, _glGetProgramLocalParameterIuivNV_args};

static const char * _glGetProgramEnvParameterIivNV_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glGetProgramEnvParameterIivNV_sig = {2488, "glGetProgramEnvParameterIivNV", 3, _glGetProgramEnvParameterIivNV_args};

static const char * _glGetProgramEnvParameterIuivNV_args[3] = {"target", "index", "params"};
static const trace::FunctionSig _glGetProgramEnvParameterIuivNV_sig = {2489, "glGetProgramEnvParameterIuivNV", 3, _glGetProgramEnvParameterIuivNV_args};

static const char * _glProgramSubroutineParametersuivNV_args[3] = {"target", "count", "params"};
static const trace::FunctionSig _glProgramSubroutineParametersuivNV_sig = {2490, "glProgramSubroutineParametersuivNV", 3, _glProgramSubroutineParametersuivNV_args};

static const char * _glGetProgramSubroutineParameteruivNV_args[3] = {"target", "index", "param"};
static const trace::FunctionSig _glGetProgramSubroutineParameteruivNV_sig = {2491, "glGetProgramSubroutineParameteruivNV", 3, _glGetProgramSubroutineParameteruivNV_args};

static const char * _glUniform1i64NV_args[2] = {"location", "x"};
static const trace::FunctionSig _glUniform1i64NV_sig = {2492, "glUniform1i64NV", 2, _glUniform1i64NV_args};

static const char * _glUniform2i64NV_args[3] = {"location", "x", "y"};
static const trace::FunctionSig _glUniform2i64NV_sig = {2493, "glUniform2i64NV", 3, _glUniform2i64NV_args};

static const char * _glUniform3i64NV_args[4] = {"location", "x", "y", "z"};
static const trace::FunctionSig _glUniform3i64NV_sig = {2494, "glUniform3i64NV", 4, _glUniform3i64NV_args};

static const char * _glUniform4i64NV_args[5] = {"location", "x", "y", "z", "w"};
static const trace::FunctionSig _glUniform4i64NV_sig = {2495, "glUniform4i64NV", 5, _glUniform4i64NV_args};

static const char * _glUniform1i64vNV_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1i64vNV_sig = {2496, "glUniform1i64vNV", 3, _glUniform1i64vNV_args};

static const char * _glUniform2i64vNV_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2i64vNV_sig = {2497, "glUniform2i64vNV", 3, _glUniform2i64vNV_args};

static const char * _glUniform3i64vNV_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3i64vNV_sig = {2498, "glUniform3i64vNV", 3, _glUniform3i64vNV_args};

static const char * _glUniform4i64vNV_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4i64vNV_sig = {2499, "glUniform4i64vNV", 3, _glUniform4i64vNV_args};

static const char * _glUniform1ui64NV_args[2] = {"location", "x"};
static const trace::FunctionSig _glUniform1ui64NV_sig = {2500, "glUniform1ui64NV", 2, _glUniform1ui64NV_args};

static const char * _glUniform2ui64NV_args[3] = {"location", "x", "y"};
static const trace::FunctionSig _glUniform2ui64NV_sig = {2501, "glUniform2ui64NV", 3, _glUniform2ui64NV_args};

static const char * _glUniform3ui64NV_args[4] = {"location", "x", "y", "z"};
static const trace::FunctionSig _glUniform3ui64NV_sig = {2502, "glUniform3ui64NV", 4, _glUniform3ui64NV_args};

static const char * _glUniform4ui64NV_args[5] = {"location", "x", "y", "z", "w"};
static const trace::FunctionSig _glUniform4ui64NV_sig = {2503, "glUniform4ui64NV", 5, _glUniform4ui64NV_args};

static const char * _glUniform1ui64vNV_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform1ui64vNV_sig = {2504, "glUniform1ui64vNV", 3, _glUniform1ui64vNV_args};

static const char * _glUniform2ui64vNV_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform2ui64vNV_sig = {2505, "glUniform2ui64vNV", 3, _glUniform2ui64vNV_args};

static const char * _glUniform3ui64vNV_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform3ui64vNV_sig = {2506, "glUniform3ui64vNV", 3, _glUniform3ui64vNV_args};

static const char * _glUniform4ui64vNV_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniform4ui64vNV_sig = {2507, "glUniform4ui64vNV", 3, _glUniform4ui64vNV_args};

static const char * _glGetUniformi64vNV_args[3] = {"program", "location", "params"};
static const trace::FunctionSig _glGetUniformi64vNV_sig = {2508, "glGetUniformi64vNV", 3, _glGetUniformi64vNV_args};

static const char * _glProgramUniform1i64NV_args[3] = {"program", "location", "x"};
static const trace::FunctionSig _glProgramUniform1i64NV_sig = {2509, "glProgramUniform1i64NV", 3, _glProgramUniform1i64NV_args};

static const char * _glProgramUniform2i64NV_args[4] = {"program", "location", "x", "y"};
static const trace::FunctionSig _glProgramUniform2i64NV_sig = {2510, "glProgramUniform2i64NV", 4, _glProgramUniform2i64NV_args};

static const char * _glProgramUniform3i64NV_args[5] = {"program", "location", "x", "y", "z"};
static const trace::FunctionSig _glProgramUniform3i64NV_sig = {2511, "glProgramUniform3i64NV", 5, _glProgramUniform3i64NV_args};

static const char * _glProgramUniform4i64NV_args[6] = {"program", "location", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramUniform4i64NV_sig = {2512, "glProgramUniform4i64NV", 6, _glProgramUniform4i64NV_args};

static const char * _glProgramUniform1i64vNV_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1i64vNV_sig = {2513, "glProgramUniform1i64vNV", 4, _glProgramUniform1i64vNV_args};

static const char * _glProgramUniform2i64vNV_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2i64vNV_sig = {2514, "glProgramUniform2i64vNV", 4, _glProgramUniform2i64vNV_args};

static const char * _glProgramUniform3i64vNV_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3i64vNV_sig = {2515, "glProgramUniform3i64vNV", 4, _glProgramUniform3i64vNV_args};

static const char * _glProgramUniform4i64vNV_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4i64vNV_sig = {2516, "glProgramUniform4i64vNV", 4, _glProgramUniform4i64vNV_args};

static const char * _glProgramUniform1ui64NV_args[3] = {"program", "location", "x"};
static const trace::FunctionSig _glProgramUniform1ui64NV_sig = {2517, "glProgramUniform1ui64NV", 3, _glProgramUniform1ui64NV_args};

static const char * _glProgramUniform2ui64NV_args[4] = {"program", "location", "x", "y"};
static const trace::FunctionSig _glProgramUniform2ui64NV_sig = {2518, "glProgramUniform2ui64NV", 4, _glProgramUniform2ui64NV_args};

static const char * _glProgramUniform3ui64NV_args[5] = {"program", "location", "x", "y", "z"};
static const trace::FunctionSig _glProgramUniform3ui64NV_sig = {2519, "glProgramUniform3ui64NV", 5, _glProgramUniform3ui64NV_args};

static const char * _glProgramUniform4ui64NV_args[6] = {"program", "location", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramUniform4ui64NV_sig = {2520, "glProgramUniform4ui64NV", 6, _glProgramUniform4ui64NV_args};

static const char * _glProgramUniform1ui64vNV_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform1ui64vNV_sig = {2521, "glProgramUniform1ui64vNV", 4, _glProgramUniform1ui64vNV_args};

static const char * _glProgramUniform2ui64vNV_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform2ui64vNV_sig = {2522, "glProgramUniform2ui64vNV", 4, _glProgramUniform2ui64vNV_args};

static const char * _glProgramUniform3ui64vNV_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform3ui64vNV_sig = {2523, "glProgramUniform3ui64vNV", 4, _glProgramUniform3ui64vNV_args};

static const char * _glProgramUniform4ui64vNV_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniform4ui64vNV_sig = {2524, "glProgramUniform4ui64vNV", 4, _glProgramUniform4ui64vNV_args};

static const char * _glVertex2hNV_args[2] = {"x", "y"};
static const trace::FunctionSig _glVertex2hNV_sig = {2525, "glVertex2hNV", 2, _glVertex2hNV_args};

static const char * _glVertex2hvNV_args[1] = {"v"};
static const trace::FunctionSig _glVertex2hvNV_sig = {2526, "glVertex2hvNV", 1, _glVertex2hvNV_args};

static const char * _glVertex3hNV_args[3] = {"x", "y", "z"};
static const trace::FunctionSig _glVertex3hNV_sig = {2527, "glVertex3hNV", 3, _glVertex3hNV_args};

static const char * _glVertex3hvNV_args[1] = {"v"};
static const trace::FunctionSig _glVertex3hvNV_sig = {2528, "glVertex3hvNV", 1, _glVertex3hvNV_args};

static const char * _glVertex4hNV_args[4] = {"x", "y", "z", "w"};
static const trace::FunctionSig _glVertex4hNV_sig = {2529, "glVertex4hNV", 4, _glVertex4hNV_args};

static const char * _glVertex4hvNV_args[1] = {"v"};
static const trace::FunctionSig _glVertex4hvNV_sig = {2530, "glVertex4hvNV", 1, _glVertex4hvNV_args};

static const char * _glNormal3hNV_args[3] = {"nx", "ny", "nz"};
static const trace::FunctionSig _glNormal3hNV_sig = {2531, "glNormal3hNV", 3, _glNormal3hNV_args};

static const char * _glNormal3hvNV_args[1] = {"v"};
static const trace::FunctionSig _glNormal3hvNV_sig = {2532, "glNormal3hvNV", 1, _glNormal3hvNV_args};

static const char * _glColor3hNV_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glColor3hNV_sig = {2533, "glColor3hNV", 3, _glColor3hNV_args};

static const char * _glColor3hvNV_args[1] = {"v"};
static const trace::FunctionSig _glColor3hvNV_sig = {2534, "glColor3hvNV", 1, _glColor3hvNV_args};

static const char * _glColor4hNV_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glColor4hNV_sig = {2535, "glColor4hNV", 4, _glColor4hNV_args};

static const char * _glColor4hvNV_args[1] = {"v"};
static const trace::FunctionSig _glColor4hvNV_sig = {2536, "glColor4hvNV", 1, _glColor4hvNV_args};

static const char * _glTexCoord1hNV_args[1] = {"s"};
static const trace::FunctionSig _glTexCoord1hNV_sig = {2537, "glTexCoord1hNV", 1, _glTexCoord1hNV_args};

static const char * _glTexCoord1hvNV_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord1hvNV_sig = {2538, "glTexCoord1hvNV", 1, _glTexCoord1hvNV_args};

static const char * _glTexCoord2hNV_args[2] = {"s", "t"};
static const trace::FunctionSig _glTexCoord2hNV_sig = {2539, "glTexCoord2hNV", 2, _glTexCoord2hNV_args};

static const char * _glTexCoord2hvNV_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord2hvNV_sig = {2540, "glTexCoord2hvNV", 1, _glTexCoord2hvNV_args};

static const char * _glTexCoord3hNV_args[3] = {"s", "t", "r"};
static const trace::FunctionSig _glTexCoord3hNV_sig = {2541, "glTexCoord3hNV", 3, _glTexCoord3hNV_args};

static const char * _glTexCoord3hvNV_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord3hvNV_sig = {2542, "glTexCoord3hvNV", 1, _glTexCoord3hvNV_args};

static const char * _glTexCoord4hNV_args[4] = {"s", "t", "r", "q"};
static const trace::FunctionSig _glTexCoord4hNV_sig = {2543, "glTexCoord4hNV", 4, _glTexCoord4hNV_args};

static const char * _glTexCoord4hvNV_args[1] = {"v"};
static const trace::FunctionSig _glTexCoord4hvNV_sig = {2544, "glTexCoord4hvNV", 1, _glTexCoord4hvNV_args};

static const char * _glMultiTexCoord1hNV_args[2] = {"target", "s"};
static const trace::FunctionSig _glMultiTexCoord1hNV_sig = {2545, "glMultiTexCoord1hNV", 2, _glMultiTexCoord1hNV_args};

static const char * _glMultiTexCoord1hvNV_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord1hvNV_sig = {2546, "glMultiTexCoord1hvNV", 2, _glMultiTexCoord1hvNV_args};

static const char * _glMultiTexCoord2hNV_args[3] = {"target", "s", "t"};
static const trace::FunctionSig _glMultiTexCoord2hNV_sig = {2547, "glMultiTexCoord2hNV", 3, _glMultiTexCoord2hNV_args};

static const char * _glMultiTexCoord2hvNV_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord2hvNV_sig = {2548, "glMultiTexCoord2hvNV", 2, _glMultiTexCoord2hvNV_args};

static const char * _glMultiTexCoord3hNV_args[4] = {"target", "s", "t", "r"};
static const trace::FunctionSig _glMultiTexCoord3hNV_sig = {2549, "glMultiTexCoord3hNV", 4, _glMultiTexCoord3hNV_args};

static const char * _glMultiTexCoord3hvNV_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord3hvNV_sig = {2550, "glMultiTexCoord3hvNV", 2, _glMultiTexCoord3hvNV_args};

static const char * _glMultiTexCoord4hNV_args[5] = {"target", "s", "t", "r", "q"};
static const trace::FunctionSig _glMultiTexCoord4hNV_sig = {2551, "glMultiTexCoord4hNV", 5, _glMultiTexCoord4hNV_args};

static const char * _glMultiTexCoord4hvNV_args[2] = {"target", "v"};
static const trace::FunctionSig _glMultiTexCoord4hvNV_sig = {2552, "glMultiTexCoord4hvNV", 2, _glMultiTexCoord4hvNV_args};

static const char * _glFogCoordhNV_args[1] = {"fog"};
static const trace::FunctionSig _glFogCoordhNV_sig = {2553, "glFogCoordhNV", 1, _glFogCoordhNV_args};

static const char * _glFogCoordhvNV_args[1] = {"fog"};
static const trace::FunctionSig _glFogCoordhvNV_sig = {2554, "glFogCoordhvNV", 1, _glFogCoordhvNV_args};

static const char * _glSecondaryColor3hNV_args[3] = {"red", "green", "blue"};
static const trace::FunctionSig _glSecondaryColor3hNV_sig = {2555, "glSecondaryColor3hNV", 3, _glSecondaryColor3hNV_args};

static const char * _glSecondaryColor3hvNV_args[1] = {"v"};
static const trace::FunctionSig _glSecondaryColor3hvNV_sig = {2556, "glSecondaryColor3hvNV", 1, _glSecondaryColor3hvNV_args};

static const char * _glVertexWeighthNV_args[1] = {"weight"};
static const trace::FunctionSig _glVertexWeighthNV_sig = {2557, "glVertexWeighthNV", 1, _glVertexWeighthNV_args};

static const char * _glVertexWeighthvNV_args[1] = {"weight"};
static const trace::FunctionSig _glVertexWeighthvNV_sig = {2558, "glVertexWeighthvNV", 1, _glVertexWeighthvNV_args};

static const char * _glVertexAttrib1hNV_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttrib1hNV_sig = {2559, "glVertexAttrib1hNV", 2, _glVertexAttrib1hNV_args};

static const char * _glVertexAttrib1hvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib1hvNV_sig = {2560, "glVertexAttrib1hvNV", 2, _glVertexAttrib1hvNV_args};

static const char * _glVertexAttrib2hNV_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttrib2hNV_sig = {2561, "glVertexAttrib2hNV", 3, _glVertexAttrib2hNV_args};

static const char * _glVertexAttrib2hvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib2hvNV_sig = {2562, "glVertexAttrib2hvNV", 2, _glVertexAttrib2hvNV_args};

static const char * _glVertexAttrib3hNV_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttrib3hNV_sig = {2563, "glVertexAttrib3hNV", 4, _glVertexAttrib3hNV_args};

static const char * _glVertexAttrib3hvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib3hvNV_sig = {2564, "glVertexAttrib3hvNV", 2, _glVertexAttrib3hvNV_args};

static const char * _glVertexAttrib4hNV_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4hNV_sig = {2565, "glVertexAttrib4hNV", 5, _glVertexAttrib4hNV_args};

static const char * _glVertexAttrib4hvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4hvNV_sig = {2566, "glVertexAttrib4hvNV", 2, _glVertexAttrib4hvNV_args};

static const char * _glVertexAttribs1hvNV_args[3] = {"index", "n", "v"};
static const trace::FunctionSig _glVertexAttribs1hvNV_sig = {2567, "glVertexAttribs1hvNV", 3, _glVertexAttribs1hvNV_args};

static const char * _glVertexAttribs2hvNV_args[3] = {"index", "n", "v"};
static const trace::FunctionSig _glVertexAttribs2hvNV_sig = {2568, "glVertexAttribs2hvNV", 3, _glVertexAttribs2hvNV_args};

static const char * _glVertexAttribs3hvNV_args[3] = {"index", "n", "v"};
static const trace::FunctionSig _glVertexAttribs3hvNV_sig = {2569, "glVertexAttribs3hvNV", 3, _glVertexAttribs3hvNV_args};

static const char * _glVertexAttribs4hvNV_args[3] = {"index", "n", "v"};
static const trace::FunctionSig _glVertexAttribs4hvNV_sig = {2570, "glVertexAttribs4hvNV", 3, _glVertexAttribs4hvNV_args};

static const char * _glGenOcclusionQueriesNV_args[2] = {"n", "ids"};
static const trace::FunctionSig _glGenOcclusionQueriesNV_sig = {2571, "glGenOcclusionQueriesNV", 2, _glGenOcclusionQueriesNV_args};

static const char * _glDeleteOcclusionQueriesNV_args[2] = {"n", "ids"};
static const trace::FunctionSig _glDeleteOcclusionQueriesNV_sig = {2572, "glDeleteOcclusionQueriesNV", 2, _glDeleteOcclusionQueriesNV_args};

static const char * _glIsOcclusionQueryNV_args[1] = {"id"};
static const trace::FunctionSig _glIsOcclusionQueryNV_sig = {2573, "glIsOcclusionQueryNV", 1, _glIsOcclusionQueryNV_args};

static const char * _glBeginOcclusionQueryNV_args[1] = {"id"};
static const trace::FunctionSig _glBeginOcclusionQueryNV_sig = {2574, "glBeginOcclusionQueryNV", 1, _glBeginOcclusionQueryNV_args};

static const char ** _glEndOcclusionQueryNV_args = NULL;
static const trace::FunctionSig _glEndOcclusionQueryNV_sig = {2575, "glEndOcclusionQueryNV", 0, _glEndOcclusionQueryNV_args};

static const char * _glGetOcclusionQueryivNV_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetOcclusionQueryivNV_sig = {2576, "glGetOcclusionQueryivNV", 3, _glGetOcclusionQueryivNV_args};

static const char * _glGetOcclusionQueryuivNV_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetOcclusionQueryuivNV_sig = {2577, "glGetOcclusionQueryuivNV", 3, _glGetOcclusionQueryuivNV_args};

static const char * _glProgramBufferParametersfvNV_args[5] = {"target", "buffer", "index", "count", "params"};
static const trace::FunctionSig _glProgramBufferParametersfvNV_sig = {2578, "glProgramBufferParametersfvNV", 5, _glProgramBufferParametersfvNV_args};

static const char * _glProgramBufferParametersIivNV_args[5] = {"target", "buffer", "index", "count", "params"};
static const trace::FunctionSig _glProgramBufferParametersIivNV_sig = {2579, "glProgramBufferParametersIivNV", 5, _glProgramBufferParametersIivNV_args};

static const char * _glProgramBufferParametersIuivNV_args[5] = {"target", "buffer", "index", "count", "params"};
static const trace::FunctionSig _glProgramBufferParametersIuivNV_sig = {2580, "glProgramBufferParametersIuivNV", 5, _glProgramBufferParametersIuivNV_args};

static const char * _glGenPathsNV_args[1] = {"range"};
static const trace::FunctionSig _glGenPathsNV_sig = {2581, "glGenPathsNV", 1, _glGenPathsNV_args};

static const char * _glDeletePathsNV_args[2] = {"path", "range"};
static const trace::FunctionSig _glDeletePathsNV_sig = {2582, "glDeletePathsNV", 2, _glDeletePathsNV_args};

static const char * _glIsPathNV_args[1] = {"path"};
static const trace::FunctionSig _glIsPathNV_sig = {2583, "glIsPathNV", 1, _glIsPathNV_args};

static const char * _glPathCommandsNV_args[6] = {"path", "numCommands", "commands", "numCoords", "coordType", "coords"};
static const trace::FunctionSig _glPathCommandsNV_sig = {2584, "glPathCommandsNV", 6, _glPathCommandsNV_args};

static const char * _glPathCoordsNV_args[4] = {"path", "numCoords", "coordType", "coords"};
static const trace::FunctionSig _glPathCoordsNV_sig = {2585, "glPathCoordsNV", 4, _glPathCoordsNV_args};

static const char * _glPathSubCommandsNV_args[8] = {"path", "commandStart", "commandsToDelete", "numCommands", "commands", "numCoords", "coordType", "coords"};
static const trace::FunctionSig _glPathSubCommandsNV_sig = {2586, "glPathSubCommandsNV", 8, _glPathSubCommandsNV_args};

static const char * _glPathSubCoordsNV_args[5] = {"path", "coordStart", "numCoords", "coordType", "coords"};
static const trace::FunctionSig _glPathSubCoordsNV_sig = {2587, "glPathSubCoordsNV", 5, _glPathSubCoordsNV_args};

static const char * _glPathStringNV_args[4] = {"path", "format", "length", "pathString"};
static const trace::FunctionSig _glPathStringNV_sig = {2588, "glPathStringNV", 4, _glPathStringNV_args};

static const char * _glPathGlyphsNV_args[10] = {"firstPathName", "fontTarget", "fontName", "fontStyle", "numGlyphs", "type", "charcodes", "handleMissingGlyphs", "pathParameterTemplate", "emScale"};
static const trace::FunctionSig _glPathGlyphsNV_sig = {2589, "glPathGlyphsNV", 10, _glPathGlyphsNV_args};

static const char * _glPathGlyphRangeNV_args[9] = {"firstPathName", "fontTarget", "fontName", "fontStyle", "firstGlyph", "numGlyphs", "handleMissingGlyphs", "pathParameterTemplate", "emScale"};
static const trace::FunctionSig _glPathGlyphRangeNV_sig = {2590, "glPathGlyphRangeNV", 9, _glPathGlyphRangeNV_args};

static const char * _glWeightPathsNV_args[4] = {"resultPath", "numPaths", "paths", "weights"};
static const trace::FunctionSig _glWeightPathsNV_sig = {2591, "glWeightPathsNV", 4, _glWeightPathsNV_args};

static const char * _glCopyPathNV_args[2] = {"resultPath", "srcPath"};
static const trace::FunctionSig _glCopyPathNV_sig = {2592, "glCopyPathNV", 2, _glCopyPathNV_args};

static const char * _glInterpolatePathsNV_args[4] = {"resultPath", "pathA", "pathB", "weight"};
static const trace::FunctionSig _glInterpolatePathsNV_sig = {2593, "glInterpolatePathsNV", 4, _glInterpolatePathsNV_args};

static const char * _glTransformPathNV_args[4] = {"resultPath", "srcPath", "transformType", "transformValues"};
static const trace::FunctionSig _glTransformPathNV_sig = {2594, "glTransformPathNV", 4, _glTransformPathNV_args};

static const char * _glPathParameterivNV_args[3] = {"path", "pname", "value"};
static const trace::FunctionSig _glPathParameterivNV_sig = {2595, "glPathParameterivNV", 3, _glPathParameterivNV_args};

static const char * _glPathParameteriNV_args[3] = {"path", "pname", "value"};
static const trace::FunctionSig _glPathParameteriNV_sig = {2596, "glPathParameteriNV", 3, _glPathParameteriNV_args};

static const char * _glPathParameterfvNV_args[3] = {"path", "pname", "value"};
static const trace::FunctionSig _glPathParameterfvNV_sig = {2597, "glPathParameterfvNV", 3, _glPathParameterfvNV_args};

static const char * _glPathParameterfNV_args[3] = {"path", "pname", "value"};
static const trace::FunctionSig _glPathParameterfNV_sig = {2598, "glPathParameterfNV", 3, _glPathParameterfNV_args};

static const char * _glPathDashArrayNV_args[3] = {"path", "dashCount", "dashArray"};
static const trace::FunctionSig _glPathDashArrayNV_sig = {2599, "glPathDashArrayNV", 3, _glPathDashArrayNV_args};

static const char * _glPathStencilFuncNV_args[3] = {"func", "ref", "mask"};
static const trace::FunctionSig _glPathStencilFuncNV_sig = {2600, "glPathStencilFuncNV", 3, _glPathStencilFuncNV_args};

static const char * _glPathStencilDepthOffsetNV_args[2] = {"factor", "units"};
static const trace::FunctionSig _glPathStencilDepthOffsetNV_sig = {2601, "glPathStencilDepthOffsetNV", 2, _glPathStencilDepthOffsetNV_args};

static const char * _glStencilFillPathNV_args[3] = {"path", "fillMode", "mask"};
static const trace::FunctionSig _glStencilFillPathNV_sig = {2602, "glStencilFillPathNV", 3, _glStencilFillPathNV_args};

static const char * _glStencilStrokePathNV_args[3] = {"path", "reference", "mask"};
static const trace::FunctionSig _glStencilStrokePathNV_sig = {2603, "glStencilStrokePathNV", 3, _glStencilStrokePathNV_args};

static const char * _glStencilFillPathInstancedNV_args[8] = {"numPaths", "pathNameType", "paths", "pathBase", "fillMode", "mask", "transformType", "transformValues"};
static const trace::FunctionSig _glStencilFillPathInstancedNV_sig = {2604, "glStencilFillPathInstancedNV", 8, _glStencilFillPathInstancedNV_args};

static const char * _glStencilStrokePathInstancedNV_args[8] = {"numPaths", "pathNameType", "paths", "pathBase", "reference", "mask", "transformType", "transformValues"};
static const trace::FunctionSig _glStencilStrokePathInstancedNV_sig = {2605, "glStencilStrokePathInstancedNV", 8, _glStencilStrokePathInstancedNV_args};

static const char * _glPathCoverDepthFuncNV_args[1] = {"func"};
static const trace::FunctionSig _glPathCoverDepthFuncNV_sig = {2606, "glPathCoverDepthFuncNV", 1, _glPathCoverDepthFuncNV_args};

static const char * _glPathColorGenNV_args[4] = {"color", "genMode", "colorFormat", "coeffs"};
static const trace::FunctionSig _glPathColorGenNV_sig = {2607, "glPathColorGenNV", 4, _glPathColorGenNV_args};

static const char * _glPathTexGenNV_args[4] = {"texCoordSet", "genMode", "components", "coeffs"};
static const trace::FunctionSig _glPathTexGenNV_sig = {2608, "glPathTexGenNV", 4, _glPathTexGenNV_args};

static const char * _glPathFogGenNV_args[1] = {"genMode"};
static const trace::FunctionSig _glPathFogGenNV_sig = {2609, "glPathFogGenNV", 1, _glPathFogGenNV_args};

static const char * _glCoverFillPathNV_args[2] = {"path", "coverMode"};
static const trace::FunctionSig _glCoverFillPathNV_sig = {2610, "glCoverFillPathNV", 2, _glCoverFillPathNV_args};

static const char * _glCoverStrokePathNV_args[2] = {"path", "coverMode"};
static const trace::FunctionSig _glCoverStrokePathNV_sig = {2611, "glCoverStrokePathNV", 2, _glCoverStrokePathNV_args};

static const char * _glCoverFillPathInstancedNV_args[7] = {"numPaths", "pathNameType", "paths", "pathBase", "coverMode", "transformType", "transformValues"};
static const trace::FunctionSig _glCoverFillPathInstancedNV_sig = {2612, "glCoverFillPathInstancedNV", 7, _glCoverFillPathInstancedNV_args};

static const char * _glCoverStrokePathInstancedNV_args[7] = {"numPaths", "pathNameType", "paths", "pathBase", "coverMode", "transformType", "transformValues"};
static const trace::FunctionSig _glCoverStrokePathInstancedNV_sig = {2613, "glCoverStrokePathInstancedNV", 7, _glCoverStrokePathInstancedNV_args};

static const char * _glGetPathParameterivNV_args[3] = {"path", "pname", "value"};
static const trace::FunctionSig _glGetPathParameterivNV_sig = {2614, "glGetPathParameterivNV", 3, _glGetPathParameterivNV_args};

static const char * _glGetPathParameterfvNV_args[3] = {"path", "pname", "value"};
static const trace::FunctionSig _glGetPathParameterfvNV_sig = {2615, "glGetPathParameterfvNV", 3, _glGetPathParameterfvNV_args};

static const char * _glGetPathCommandsNV_args[2] = {"path", "commands"};
static const trace::FunctionSig _glGetPathCommandsNV_sig = {2616, "glGetPathCommandsNV", 2, _glGetPathCommandsNV_args};

static const char * _glGetPathCoordsNV_args[2] = {"path", "coords"};
static const trace::FunctionSig _glGetPathCoordsNV_sig = {2617, "glGetPathCoordsNV", 2, _glGetPathCoordsNV_args};

static const char * _glGetPathDashArrayNV_args[2] = {"path", "dashArray"};
static const trace::FunctionSig _glGetPathDashArrayNV_sig = {2618, "glGetPathDashArrayNV", 2, _glGetPathDashArrayNV_args};

static const char * _glGetPathMetricsNV_args[7] = {"metricQueryMask", "numPaths", "pathNameType", "paths", "pathBase", "stride", "metrics"};
static const trace::FunctionSig _glGetPathMetricsNV_sig = {2619, "glGetPathMetricsNV", 7, _glGetPathMetricsNV_args};

static const char * _glGetPathMetricRangeNV_args[5] = {"metricQueryMask", "firstPathName", "numPaths", "stride", "metrics"};
static const trace::FunctionSig _glGetPathMetricRangeNV_sig = {2620, "glGetPathMetricRangeNV", 5, _glGetPathMetricRangeNV_args};

static const char * _glGetPathSpacingNV_args[9] = {"pathListMode", "numPaths", "pathNameType", "paths", "pathBase", "advanceScale", "kerningScale", "transformType", "returnedSpacing"};
static const trace::FunctionSig _glGetPathSpacingNV_sig = {2621, "glGetPathSpacingNV", 9, _glGetPathSpacingNV_args};

static const char * _glGetPathColorGenivNV_args[3] = {"color", "pname", "value"};
static const trace::FunctionSig _glGetPathColorGenivNV_sig = {2622, "glGetPathColorGenivNV", 3, _glGetPathColorGenivNV_args};

static const char * _glGetPathColorGenfvNV_args[3] = {"color", "pname", "value"};
static const trace::FunctionSig _glGetPathColorGenfvNV_sig = {2623, "glGetPathColorGenfvNV", 3, _glGetPathColorGenfvNV_args};

static const char * _glGetPathTexGenivNV_args[3] = {"texCoordSet", "pname", "value"};
static const trace::FunctionSig _glGetPathTexGenivNV_sig = {2624, "glGetPathTexGenivNV", 3, _glGetPathTexGenivNV_args};

static const char * _glGetPathTexGenfvNV_args[3] = {"texCoordSet", "pname", "value"};
static const trace::FunctionSig _glGetPathTexGenfvNV_sig = {2625, "glGetPathTexGenfvNV", 3, _glGetPathTexGenfvNV_args};

static const char * _glIsPointInFillPathNV_args[4] = {"path", "mask", "x", "y"};
static const trace::FunctionSig _glIsPointInFillPathNV_sig = {2626, "glIsPointInFillPathNV", 4, _glIsPointInFillPathNV_args};

static const char * _glIsPointInStrokePathNV_args[3] = {"path", "x", "y"};
static const trace::FunctionSig _glIsPointInStrokePathNV_sig = {2627, "glIsPointInStrokePathNV", 3, _glIsPointInStrokePathNV_args};

static const char * _glGetPathLengthNV_args[3] = {"path", "startSegment", "numSegments"};
static const trace::FunctionSig _glGetPathLengthNV_sig = {2628, "glGetPathLengthNV", 3, _glGetPathLengthNV_args};

static const char * _glPointAlongPathNV_args[8] = {"path", "startSegment", "numSegments", "distance", "x", "y", "tangentX", "tangentY"};
static const trace::FunctionSig _glPointAlongPathNV_sig = {2629, "glPointAlongPathNV", 8, _glPointAlongPathNV_args};

static const char * _glMatrixLoad3x2fNV_args[2] = {"matrixMode", "m"};
static const trace::FunctionSig _glMatrixLoad3x2fNV_sig = {2630, "glMatrixLoad3x2fNV", 2, _glMatrixLoad3x2fNV_args};

static const char * _glMatrixLoad3x3fNV_args[2] = {"matrixMode", "m"};
static const trace::FunctionSig _glMatrixLoad3x3fNV_sig = {2631, "glMatrixLoad3x3fNV", 2, _glMatrixLoad3x3fNV_args};

static const char * _glMatrixLoadTranspose3x3fNV_args[2] = {"matrixMode", "m"};
static const trace::FunctionSig _glMatrixLoadTranspose3x3fNV_sig = {2632, "glMatrixLoadTranspose3x3fNV", 2, _glMatrixLoadTranspose3x3fNV_args};

static const char * _glMatrixMult3x2fNV_args[2] = {"matrixMode", "m"};
static const trace::FunctionSig _glMatrixMult3x2fNV_sig = {2633, "glMatrixMult3x2fNV", 2, _glMatrixMult3x2fNV_args};

static const char * _glMatrixMult3x3fNV_args[2] = {"matrixMode", "m"};
static const trace::FunctionSig _glMatrixMult3x3fNV_sig = {2634, "glMatrixMult3x3fNV", 2, _glMatrixMult3x3fNV_args};

static const char * _glMatrixMultTranspose3x3fNV_args[2] = {"matrixMode", "m"};
static const trace::FunctionSig _glMatrixMultTranspose3x3fNV_sig = {2635, "glMatrixMultTranspose3x3fNV", 2, _glMatrixMultTranspose3x3fNV_args};

static const char * _glStencilThenCoverFillPathNV_args[4] = {"path", "fillMode", "mask", "coverMode"};
static const trace::FunctionSig _glStencilThenCoverFillPathNV_sig = {2636, "glStencilThenCoverFillPathNV", 4, _glStencilThenCoverFillPathNV_args};

static const char * _glStencilThenCoverStrokePathNV_args[4] = {"path", "reference", "mask", "coverMode"};
static const trace::FunctionSig _glStencilThenCoverStrokePathNV_sig = {2637, "glStencilThenCoverStrokePathNV", 4, _glStencilThenCoverStrokePathNV_args};

static const char * _glStencilThenCoverFillPathInstancedNV_args[9] = {"numPaths", "pathNameType", "paths", "pathBase", "fillMode", "mask", "coverMode", "transformType", "transformValues"};
static const trace::FunctionSig _glStencilThenCoverFillPathInstancedNV_sig = {2638, "glStencilThenCoverFillPathInstancedNV", 9, _glStencilThenCoverFillPathInstancedNV_args};

static const char * _glStencilThenCoverStrokePathInstancedNV_args[9] = {"numPaths", "pathNameType", "paths", "pathBase", "reference", "mask", "coverMode", "transformType", "transformValues"};
static const trace::FunctionSig _glStencilThenCoverStrokePathInstancedNV_sig = {2639, "glStencilThenCoverStrokePathInstancedNV", 9, _glStencilThenCoverStrokePathInstancedNV_args};

static const char * _glPathGlyphIndexRangeNV_args[6] = {"fontTarget", "fontName", "fontStyle", "pathParameterTemplate", "emScale", "baseAndCount"};
static const trace::FunctionSig _glPathGlyphIndexRangeNV_sig = {2640, "glPathGlyphIndexRangeNV", 6, _glPathGlyphIndexRangeNV_args};

static const char * _glPathGlyphIndexArrayNV_args[8] = {"firstPathName", "fontTarget", "fontName", "fontStyle", "firstGlyphIndex", "numGlyphs", "pathParameterTemplate", "emScale"};
static const trace::FunctionSig _glPathGlyphIndexArrayNV_sig = {2641, "glPathGlyphIndexArrayNV", 8, _glPathGlyphIndexArrayNV_args};

static const char * _glPathMemoryGlyphIndexArrayNV_args[9] = {"firstPathName", "fontTarget", "fontSize", "fontData", "faceIndex", "firstGlyphIndex", "numGlyphs", "pathParameterTemplate", "emScale"};
static const trace::FunctionSig _glPathMemoryGlyphIndexArrayNV_sig = {2642, "glPathMemoryGlyphIndexArrayNV", 9, _glPathMemoryGlyphIndexArrayNV_args};

static const char * _glProgramPathFragmentInputGenNV_args[5] = {"program", "location", "genMode", "components", "coeffs"};
static const trace::FunctionSig _glProgramPathFragmentInputGenNV_sig = {2643, "glProgramPathFragmentInputGenNV", 5, _glProgramPathFragmentInputGenNV_args};

static const char * _glGetProgramResourcefvNV_args[8] = {"program", "programInterface", "index", "propCount", "props", "bufSize", "length", "params"};
static const trace::FunctionSig _glGetProgramResourcefvNV_sig = {2644, "glGetProgramResourcefvNV", 8, _glGetProgramResourcefvNV_args};

static const char * _glPixelDataRangeNV_args[3] = {"target", "length", "pointer"};
static const trace::FunctionSig _glPixelDataRangeNV_sig = {2645, "glPixelDataRangeNV", 3, _glPixelDataRangeNV_args};

static const char * _glFlushPixelDataRangeNV_args[1] = {"target"};
static const trace::FunctionSig _glFlushPixelDataRangeNV_sig = {2646, "glFlushPixelDataRangeNV", 1, _glFlushPixelDataRangeNV_args};

static const char * _glPointParameteriNV_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPointParameteriNV_sig = {2647, "glPointParameteriNV", 2, _glPointParameteriNV_args};

static const char * _glPointParameterivNV_args[2] = {"pname", "params"};
static const trace::FunctionSig _glPointParameterivNV_sig = {2648, "glPointParameterivNV", 2, _glPointParameterivNV_args};

static const char * _glPresentFrameKeyedNV_args[11] = {"video_slot", "minPresentTime", "beginPresentTimeId", "presentDurationId", "type", "target0", "fill0", "key0", "target1", "fill1", "key1"};
static const trace::FunctionSig _glPresentFrameKeyedNV_sig = {2649, "glPresentFrameKeyedNV", 11, _glPresentFrameKeyedNV_args};

static const char * _glPresentFrameDualFillNV_args[13] = {"video_slot", "minPresentTime", "beginPresentTimeId", "presentDurationId", "type", "target0", "fill0", "target1", "fill1", "target2", "fill2", "target3", "fill3"};
static const trace::FunctionSig _glPresentFrameDualFillNV_sig = {2650, "glPresentFrameDualFillNV", 13, _glPresentFrameDualFillNV_args};

static const char * _glGetVideoivNV_args[3] = {"video_slot", "pname", "params"};
static const trace::FunctionSig _glGetVideoivNV_sig = {2651, "glGetVideoivNV", 3, _glGetVideoivNV_args};

static const char * _glGetVideouivNV_args[3] = {"video_slot", "pname", "params"};
static const trace::FunctionSig _glGetVideouivNV_sig = {2652, "glGetVideouivNV", 3, _glGetVideouivNV_args};

static const char * _glGetVideoi64vNV_args[3] = {"video_slot", "pname", "params"};
static const trace::FunctionSig _glGetVideoi64vNV_sig = {2653, "glGetVideoi64vNV", 3, _glGetVideoi64vNV_args};

static const char * _glGetVideoui64vNV_args[3] = {"video_slot", "pname", "params"};
static const trace::FunctionSig _glGetVideoui64vNV_sig = {2654, "glGetVideoui64vNV", 3, _glGetVideoui64vNV_args};

static const char ** _glPrimitiveRestartNV_args = NULL;
static const trace::FunctionSig _glPrimitiveRestartNV_sig = {2655, "glPrimitiveRestartNV", 0, _glPrimitiveRestartNV_args};

static const char * _glPrimitiveRestartIndexNV_args[1] = {"index"};
static const trace::FunctionSig _glPrimitiveRestartIndexNV_sig = {2656, "glPrimitiveRestartIndexNV", 1, _glPrimitiveRestartIndexNV_args};

static const char * _glReadBufferNV_args[1] = {"mode"};
static const trace::FunctionSig _glReadBufferNV_sig = {2657, "glReadBufferNV", 1, _glReadBufferNV_args};

static const char * _glCombinerParameterfvNV_args[2] = {"pname", "params"};
static const trace::FunctionSig _glCombinerParameterfvNV_sig = {2658, "glCombinerParameterfvNV", 2, _glCombinerParameterfvNV_args};

static const char * _glCombinerParameterfNV_args[2] = {"pname", "param"};
static const trace::FunctionSig _glCombinerParameterfNV_sig = {2659, "glCombinerParameterfNV", 2, _glCombinerParameterfNV_args};

static const char * _glCombinerParameterivNV_args[2] = {"pname", "params"};
static const trace::FunctionSig _glCombinerParameterivNV_sig = {2660, "glCombinerParameterivNV", 2, _glCombinerParameterivNV_args};

static const char * _glCombinerParameteriNV_args[2] = {"pname", "param"};
static const trace::FunctionSig _glCombinerParameteriNV_sig = {2661, "glCombinerParameteriNV", 2, _glCombinerParameteriNV_args};

static const char * _glCombinerInputNV_args[6] = {"stage", "portion", "variable", "input", "mapping", "componentUsage"};
static const trace::FunctionSig _glCombinerInputNV_sig = {2662, "glCombinerInputNV", 6, _glCombinerInputNV_args};

static const char * _glCombinerOutputNV_args[10] = {"stage", "portion", "abOutput", "cdOutput", "sumOutput", "scale", "bias", "abDotProduct", "cdDotProduct", "muxSum"};
static const trace::FunctionSig _glCombinerOutputNV_sig = {2663, "glCombinerOutputNV", 10, _glCombinerOutputNV_args};

static const char * _glFinalCombinerInputNV_args[4] = {"variable", "input", "mapping", "componentUsage"};
static const trace::FunctionSig _glFinalCombinerInputNV_sig = {2664, "glFinalCombinerInputNV", 4, _glFinalCombinerInputNV_args};

static const char * _glGetCombinerInputParameterfvNV_args[5] = {"stage", "portion", "variable", "pname", "params"};
static const trace::FunctionSig _glGetCombinerInputParameterfvNV_sig = {2665, "glGetCombinerInputParameterfvNV", 5, _glGetCombinerInputParameterfvNV_args};

static const char * _glGetCombinerInputParameterivNV_args[5] = {"stage", "portion", "variable", "pname", "params"};
static const trace::FunctionSig _glGetCombinerInputParameterivNV_sig = {2666, "glGetCombinerInputParameterivNV", 5, _glGetCombinerInputParameterivNV_args};

static const char * _glGetCombinerOutputParameterfvNV_args[4] = {"stage", "portion", "pname", "params"};
static const trace::FunctionSig _glGetCombinerOutputParameterfvNV_sig = {2667, "glGetCombinerOutputParameterfvNV", 4, _glGetCombinerOutputParameterfvNV_args};

static const char * _glGetCombinerOutputParameterivNV_args[4] = {"stage", "portion", "pname", "params"};
static const trace::FunctionSig _glGetCombinerOutputParameterivNV_sig = {2668, "glGetCombinerOutputParameterivNV", 4, _glGetCombinerOutputParameterivNV_args};

static const char * _glGetFinalCombinerInputParameterfvNV_args[3] = {"variable", "pname", "params"};
static const trace::FunctionSig _glGetFinalCombinerInputParameterfvNV_sig = {2669, "glGetFinalCombinerInputParameterfvNV", 3, _glGetFinalCombinerInputParameterfvNV_args};

static const char * _glGetFinalCombinerInputParameterivNV_args[3] = {"variable", "pname", "params"};
static const trace::FunctionSig _glGetFinalCombinerInputParameterivNV_sig = {2670, "glGetFinalCombinerInputParameterivNV", 3, _glGetFinalCombinerInputParameterivNV_args};

static const char * _glCombinerStageParameterfvNV_args[3] = {"stage", "pname", "params"};
static const trace::FunctionSig _glCombinerStageParameterfvNV_sig = {2671, "glCombinerStageParameterfvNV", 3, _glCombinerStageParameterfvNV_args};

static const char * _glGetCombinerStageParameterfvNV_args[3] = {"stage", "pname", "params"};
static const trace::FunctionSig _glGetCombinerStageParameterfvNV_sig = {2672, "glGetCombinerStageParameterfvNV", 3, _glGetCombinerStageParameterfvNV_args};

static const char * _glMakeBufferResidentNV_args[2] = {"target", "access"};
static const trace::FunctionSig _glMakeBufferResidentNV_sig = {2673, "glMakeBufferResidentNV", 2, _glMakeBufferResidentNV_args};

static const char * _glMakeBufferNonResidentNV_args[1] = {"target"};
static const trace::FunctionSig _glMakeBufferNonResidentNV_sig = {2674, "glMakeBufferNonResidentNV", 1, _glMakeBufferNonResidentNV_args};

static const char * _glIsBufferResidentNV_args[1] = {"target"};
static const trace::FunctionSig _glIsBufferResidentNV_sig = {2675, "glIsBufferResidentNV", 1, _glIsBufferResidentNV_args};

static const char * _glMakeNamedBufferResidentNV_args[2] = {"buffer", "access"};
static const trace::FunctionSig _glMakeNamedBufferResidentNV_sig = {2676, "glMakeNamedBufferResidentNV", 2, _glMakeNamedBufferResidentNV_args};

static const char * _glMakeNamedBufferNonResidentNV_args[1] = {"buffer"};
static const trace::FunctionSig _glMakeNamedBufferNonResidentNV_sig = {2677, "glMakeNamedBufferNonResidentNV", 1, _glMakeNamedBufferNonResidentNV_args};

static const char * _glIsNamedBufferResidentNV_args[1] = {"buffer"};
static const trace::FunctionSig _glIsNamedBufferResidentNV_sig = {2678, "glIsNamedBufferResidentNV", 1, _glIsNamedBufferResidentNV_args};

static const char * _glGetBufferParameterui64vNV_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetBufferParameterui64vNV_sig = {2679, "glGetBufferParameterui64vNV", 3, _glGetBufferParameterui64vNV_args};

static const char * _glGetNamedBufferParameterui64vNV_args[3] = {"buffer", "pname", "params"};
static const trace::FunctionSig _glGetNamedBufferParameterui64vNV_sig = {2680, "glGetNamedBufferParameterui64vNV", 3, _glGetNamedBufferParameterui64vNV_args};

static const char * _glGetIntegerui64vNV_args[2] = {"value", "result"};
static const trace::FunctionSig _glGetIntegerui64vNV_sig = {2681, "glGetIntegerui64vNV", 2, _glGetIntegerui64vNV_args};

static const char * _glUniformui64NV_args[2] = {"location", "value"};
static const trace::FunctionSig _glUniformui64NV_sig = {2682, "glUniformui64NV", 2, _glUniformui64NV_args};

static const char * _glUniformui64vNV_args[3] = {"location", "count", "value"};
static const trace::FunctionSig _glUniformui64vNV_sig = {2683, "glUniformui64vNV", 3, _glUniformui64vNV_args};

static const char * _glGetUniformui64vNV_args[3] = {"program", "location", "params"};
static const trace::FunctionSig _glGetUniformui64vNV_sig = {2684, "glGetUniformui64vNV", 3, _glGetUniformui64vNV_args};

static const char * _glProgramUniformui64NV_args[3] = {"program", "location", "value"};
static const trace::FunctionSig _glProgramUniformui64NV_sig = {2685, "glProgramUniformui64NV", 3, _glProgramUniformui64NV_args};

static const char * _glProgramUniformui64vNV_args[4] = {"program", "location", "count", "value"};
static const trace::FunctionSig _glProgramUniformui64vNV_sig = {2686, "glProgramUniformui64vNV", 4, _glProgramUniformui64vNV_args};

static const char ** _glTextureBarrierNV_args = NULL;
static const trace::FunctionSig _glTextureBarrierNV_sig = {2687, "glTextureBarrierNV", 0, _glTextureBarrierNV_args};

static const char * _glTexImage2DMultisampleCoverageNV_args[7] = {"target", "coverageSamples", "colorSamples", "internalFormat", "width", "height", "fixedSampleLocations"};
static const trace::FunctionSig _glTexImage2DMultisampleCoverageNV_sig = {2688, "glTexImage2DMultisampleCoverageNV", 7, _glTexImage2DMultisampleCoverageNV_args};

static const char * _glTexImage3DMultisampleCoverageNV_args[8] = {"target", "coverageSamples", "colorSamples", "internalFormat", "width", "height", "depth", "fixedSampleLocations"};
static const trace::FunctionSig _glTexImage3DMultisampleCoverageNV_sig = {2689, "glTexImage3DMultisampleCoverageNV", 8, _glTexImage3DMultisampleCoverageNV_args};

static const char * _glTextureImage2DMultisampleNV_args[7] = {"texture", "target", "samples", "internalFormat", "width", "height", "fixedSampleLocations"};
static const trace::FunctionSig _glTextureImage2DMultisampleNV_sig = {2690, "glTextureImage2DMultisampleNV", 7, _glTextureImage2DMultisampleNV_args};

static const char * _glTextureImage3DMultisampleNV_args[8] = {"texture", "target", "samples", "internalFormat", "width", "height", "depth", "fixedSampleLocations"};
static const trace::FunctionSig _glTextureImage3DMultisampleNV_sig = {2691, "glTextureImage3DMultisampleNV", 8, _glTextureImage3DMultisampleNV_args};

static const char * _glTextureImage2DMultisampleCoverageNV_args[8] = {"texture", "target", "coverageSamples", "colorSamples", "internalFormat", "width", "height", "fixedSampleLocations"};
static const trace::FunctionSig _glTextureImage2DMultisampleCoverageNV_sig = {2692, "glTextureImage2DMultisampleCoverageNV", 8, _glTextureImage2DMultisampleCoverageNV_args};

static const char * _glTextureImage3DMultisampleCoverageNV_args[9] = {"texture", "target", "coverageSamples", "colorSamples", "internalFormat", "width", "height", "depth", "fixedSampleLocations"};
static const trace::FunctionSig _glTextureImage3DMultisampleCoverageNV_sig = {2693, "glTextureImage3DMultisampleCoverageNV", 9, _glTextureImage3DMultisampleCoverageNV_args};

static const char * _glBeginTransformFeedbackNV_args[1] = {"primitiveMode"};
static const trace::FunctionSig _glBeginTransformFeedbackNV_sig = {2694, "glBeginTransformFeedbackNV", 1, _glBeginTransformFeedbackNV_args};

static const char ** _glEndTransformFeedbackNV_args = NULL;
static const trace::FunctionSig _glEndTransformFeedbackNV_sig = {2695, "glEndTransformFeedbackNV", 0, _glEndTransformFeedbackNV_args};

static const char * _glTransformFeedbackAttribsNV_args[3] = {"count", "attribs", "bufferMode"};
static const trace::FunctionSig _glTransformFeedbackAttribsNV_sig = {2696, "glTransformFeedbackAttribsNV", 3, _glTransformFeedbackAttribsNV_args};

static const char * _glBindBufferRangeNV_args[5] = {"target", "index", "buffer", "offset", "size"};
static const trace::FunctionSig _glBindBufferRangeNV_sig = {2697, "glBindBufferRangeNV", 5, _glBindBufferRangeNV_args};

static const char * _glBindBufferOffsetNV_args[4] = {"target", "index", "buffer", "offset"};
static const trace::FunctionSig _glBindBufferOffsetNV_sig = {2698, "glBindBufferOffsetNV", 4, _glBindBufferOffsetNV_args};

static const char * _glBindBufferBaseNV_args[3] = {"target", "index", "buffer"};
static const trace::FunctionSig _glBindBufferBaseNV_sig = {2699, "glBindBufferBaseNV", 3, _glBindBufferBaseNV_args};

static const char * _glTransformFeedbackVaryingsNV_args[4] = {"program", "count", "locations", "bufferMode"};
static const trace::FunctionSig _glTransformFeedbackVaryingsNV_sig = {2700, "glTransformFeedbackVaryingsNV", 4, _glTransformFeedbackVaryingsNV_args};

static const char * _glActiveVaryingNV_args[2] = {"program", "name"};
static const trace::FunctionSig _glActiveVaryingNV_sig = {2701, "glActiveVaryingNV", 2, _glActiveVaryingNV_args};

static const char * _glGetVaryingLocationNV_args[2] = {"program", "name"};
static const trace::FunctionSig _glGetVaryingLocationNV_sig = {2702, "glGetVaryingLocationNV", 2, _glGetVaryingLocationNV_args};

static const char * _glGetActiveVaryingNV_args[7] = {"program", "index", "bufSize", "length", "size", "type", "name"};
static const trace::FunctionSig _glGetActiveVaryingNV_sig = {2703, "glGetActiveVaryingNV", 7, _glGetActiveVaryingNV_args};

static const char * _glGetTransformFeedbackVaryingNV_args[3] = {"program", "index", "location"};
static const trace::FunctionSig _glGetTransformFeedbackVaryingNV_sig = {2704, "glGetTransformFeedbackVaryingNV", 3, _glGetTransformFeedbackVaryingNV_args};

static const char * _glTransformFeedbackStreamAttribsNV_args[5] = {"count", "attribs", "nbuffers", "bufstreams", "bufferMode"};
static const trace::FunctionSig _glTransformFeedbackStreamAttribsNV_sig = {2705, "glTransformFeedbackStreamAttribsNV", 5, _glTransformFeedbackStreamAttribsNV_args};

static const char * _glBindTransformFeedbackNV_args[2] = {"target", "id"};
static const trace::FunctionSig _glBindTransformFeedbackNV_sig = {2706, "glBindTransformFeedbackNV", 2, _glBindTransformFeedbackNV_args};

static const char * _glDeleteTransformFeedbacksNV_args[2] = {"n", "ids"};
static const trace::FunctionSig _glDeleteTransformFeedbacksNV_sig = {2707, "glDeleteTransformFeedbacksNV", 2, _glDeleteTransformFeedbacksNV_args};

static const char * _glGenTransformFeedbacksNV_args[2] = {"n", "ids"};
static const trace::FunctionSig _glGenTransformFeedbacksNV_sig = {2708, "glGenTransformFeedbacksNV", 2, _glGenTransformFeedbacksNV_args};

static const char * _glIsTransformFeedbackNV_args[1] = {"id"};
static const trace::FunctionSig _glIsTransformFeedbackNV_sig = {2709, "glIsTransformFeedbackNV", 1, _glIsTransformFeedbackNV_args};

static const char ** _glPauseTransformFeedbackNV_args = NULL;
static const trace::FunctionSig _glPauseTransformFeedbackNV_sig = {2710, "glPauseTransformFeedbackNV", 0, _glPauseTransformFeedbackNV_args};

static const char ** _glResumeTransformFeedbackNV_args = NULL;
static const trace::FunctionSig _glResumeTransformFeedbackNV_sig = {2711, "glResumeTransformFeedbackNV", 0, _glResumeTransformFeedbackNV_args};

static const char * _glDrawTransformFeedbackNV_args[2] = {"mode", "id"};
static const trace::FunctionSig _glDrawTransformFeedbackNV_sig = {2712, "glDrawTransformFeedbackNV", 2, _glDrawTransformFeedbackNV_args};

static const char * _glVDPAUInitNV_args[2] = {"vdpDevice", "getProcAddress"};
static const trace::FunctionSig _glVDPAUInitNV_sig = {2713, "glVDPAUInitNV", 2, _glVDPAUInitNV_args};

static const char ** _glVDPAUFiniNV_args = NULL;
static const trace::FunctionSig _glVDPAUFiniNV_sig = {2714, "glVDPAUFiniNV", 0, _glVDPAUFiniNV_args};

static const char * _glVDPAURegisterVideoSurfaceNV_args[4] = {"vdpSurface", "target", "numTextureNames", "textureNames"};
static const trace::FunctionSig _glVDPAURegisterVideoSurfaceNV_sig = {2715, "glVDPAURegisterVideoSurfaceNV", 4, _glVDPAURegisterVideoSurfaceNV_args};

static const char * _glVDPAURegisterOutputSurfaceNV_args[4] = {"vdpSurface", "target", "numTextureNames", "textureNames"};
static const trace::FunctionSig _glVDPAURegisterOutputSurfaceNV_sig = {2716, "glVDPAURegisterOutputSurfaceNV", 4, _glVDPAURegisterOutputSurfaceNV_args};

static const char * _glVDPAUIsSurfaceNV_args[1] = {"surface"};
static const trace::FunctionSig _glVDPAUIsSurfaceNV_sig = {2717, "glVDPAUIsSurfaceNV", 1, _glVDPAUIsSurfaceNV_args};

static const char * _glVDPAUUnregisterSurfaceNV_args[1] = {"surface"};
static const trace::FunctionSig _glVDPAUUnregisterSurfaceNV_sig = {2718, "glVDPAUUnregisterSurfaceNV", 1, _glVDPAUUnregisterSurfaceNV_args};

static const char * _glVDPAUGetSurfaceivNV_args[5] = {"surface", "pname", "bufSize", "length", "values"};
static const trace::FunctionSig _glVDPAUGetSurfaceivNV_sig = {2719, "glVDPAUGetSurfaceivNV", 5, _glVDPAUGetSurfaceivNV_args};

static const char * _glVDPAUSurfaceAccessNV_args[2] = {"surface", "access"};
static const trace::FunctionSig _glVDPAUSurfaceAccessNV_sig = {2720, "glVDPAUSurfaceAccessNV", 2, _glVDPAUSurfaceAccessNV_args};

static const char * _glVDPAUMapSurfacesNV_args[2] = {"numSurfaces", "surfaces"};
static const trace::FunctionSig _glVDPAUMapSurfacesNV_sig = {2721, "glVDPAUMapSurfacesNV", 2, _glVDPAUMapSurfacesNV_args};

static const char * _glVDPAUUnmapSurfacesNV_args[2] = {"numSurface", "surfaces"};
static const trace::FunctionSig _glVDPAUUnmapSurfacesNV_sig = {2722, "glVDPAUUnmapSurfacesNV", 2, _glVDPAUUnmapSurfacesNV_args};

static const char ** _glFlushVertexArrayRangeNV_args = NULL;
static const trace::FunctionSig _glFlushVertexArrayRangeNV_sig = {2723, "glFlushVertexArrayRangeNV", 0, _glFlushVertexArrayRangeNV_args};

static const char * _glVertexArrayRangeNV_args[2] = {"length", "pointer"};
static const trace::FunctionSig _glVertexArrayRangeNV_sig = {2724, "glVertexArrayRangeNV", 2, _glVertexArrayRangeNV_args};

static const char * _glVertexAttribL1i64NV_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttribL1i64NV_sig = {2725, "glVertexAttribL1i64NV", 2, _glVertexAttribL1i64NV_args};

static const char * _glVertexAttribL2i64NV_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttribL2i64NV_sig = {2726, "glVertexAttribL2i64NV", 3, _glVertexAttribL2i64NV_args};

static const char * _glVertexAttribL3i64NV_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttribL3i64NV_sig = {2727, "glVertexAttribL3i64NV", 4, _glVertexAttribL3i64NV_args};

static const char * _glVertexAttribL4i64NV_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttribL4i64NV_sig = {2728, "glVertexAttribL4i64NV", 5, _glVertexAttribL4i64NV_args};

static const char * _glVertexAttribL1i64vNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL1i64vNV_sig = {2729, "glVertexAttribL1i64vNV", 2, _glVertexAttribL1i64vNV_args};

static const char * _glVertexAttribL2i64vNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL2i64vNV_sig = {2730, "glVertexAttribL2i64vNV", 2, _glVertexAttribL2i64vNV_args};

static const char * _glVertexAttribL3i64vNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL3i64vNV_sig = {2731, "glVertexAttribL3i64vNV", 2, _glVertexAttribL3i64vNV_args};

static const char * _glVertexAttribL4i64vNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL4i64vNV_sig = {2732, "glVertexAttribL4i64vNV", 2, _glVertexAttribL4i64vNV_args};

static const char * _glVertexAttribL1ui64NV_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttribL1ui64NV_sig = {2733, "glVertexAttribL1ui64NV", 2, _glVertexAttribL1ui64NV_args};

static const char * _glVertexAttribL2ui64NV_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttribL2ui64NV_sig = {2734, "glVertexAttribL2ui64NV", 3, _glVertexAttribL2ui64NV_args};

static const char * _glVertexAttribL3ui64NV_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttribL3ui64NV_sig = {2735, "glVertexAttribL3ui64NV", 4, _glVertexAttribL3ui64NV_args};

static const char * _glVertexAttribL4ui64NV_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttribL4ui64NV_sig = {2736, "glVertexAttribL4ui64NV", 5, _glVertexAttribL4ui64NV_args};

static const char * _glVertexAttribL1ui64vNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL1ui64vNV_sig = {2737, "glVertexAttribL1ui64vNV", 2, _glVertexAttribL1ui64vNV_args};

static const char * _glVertexAttribL2ui64vNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL2ui64vNV_sig = {2738, "glVertexAttribL2ui64vNV", 2, _glVertexAttribL2ui64vNV_args};

static const char * _glVertexAttribL3ui64vNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL3ui64vNV_sig = {2739, "glVertexAttribL3ui64vNV", 2, _glVertexAttribL3ui64vNV_args};

static const char * _glVertexAttribL4ui64vNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribL4ui64vNV_sig = {2740, "glVertexAttribL4ui64vNV", 2, _glVertexAttribL4ui64vNV_args};

static const char * _glGetVertexAttribLi64vNV_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribLi64vNV_sig = {2741, "glGetVertexAttribLi64vNV", 3, _glGetVertexAttribLi64vNV_args};

static const char * _glGetVertexAttribLui64vNV_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribLui64vNV_sig = {2742, "glGetVertexAttribLui64vNV", 3, _glGetVertexAttribLui64vNV_args};

static const char * _glVertexAttribLFormatNV_args[4] = {"index", "size", "type", "stride"};
static const trace::FunctionSig _glVertexAttribLFormatNV_sig = {2743, "glVertexAttribLFormatNV", 4, _glVertexAttribLFormatNV_args};

static const char * _glBufferAddressRangeNV_args[4] = {"pname", "index", "address", "length"};
static const trace::FunctionSig _glBufferAddressRangeNV_sig = {2744, "glBufferAddressRangeNV", 4, _glBufferAddressRangeNV_args};

static const char * _glVertexFormatNV_args[3] = {"size", "type", "stride"};
static const trace::FunctionSig _glVertexFormatNV_sig = {2745, "glVertexFormatNV", 3, _glVertexFormatNV_args};

static const char * _glNormalFormatNV_args[2] = {"type", "stride"};
static const trace::FunctionSig _glNormalFormatNV_sig = {2746, "glNormalFormatNV", 2, _glNormalFormatNV_args};

static const char * _glColorFormatNV_args[3] = {"size", "type", "stride"};
static const trace::FunctionSig _glColorFormatNV_sig = {2747, "glColorFormatNV", 3, _glColorFormatNV_args};

static const char * _glIndexFormatNV_args[2] = {"type", "stride"};
static const trace::FunctionSig _glIndexFormatNV_sig = {2748, "glIndexFormatNV", 2, _glIndexFormatNV_args};

static const char * _glTexCoordFormatNV_args[3] = {"size", "type", "stride"};
static const trace::FunctionSig _glTexCoordFormatNV_sig = {2749, "glTexCoordFormatNV", 3, _glTexCoordFormatNV_args};

static const char * _glEdgeFlagFormatNV_args[1] = {"stride"};
static const trace::FunctionSig _glEdgeFlagFormatNV_sig = {2750, "glEdgeFlagFormatNV", 1, _glEdgeFlagFormatNV_args};

static const char * _glSecondaryColorFormatNV_args[3] = {"size", "type", "stride"};
static const trace::FunctionSig _glSecondaryColorFormatNV_sig = {2751, "glSecondaryColorFormatNV", 3, _glSecondaryColorFormatNV_args};

static const char * _glFogCoordFormatNV_args[2] = {"type", "stride"};
static const trace::FunctionSig _glFogCoordFormatNV_sig = {2752, "glFogCoordFormatNV", 2, _glFogCoordFormatNV_args};

static const char * _glVertexAttribFormatNV_args[5] = {"index", "size", "type", "normalized", "stride"};
static const trace::FunctionSig _glVertexAttribFormatNV_sig = {2753, "glVertexAttribFormatNV", 5, _glVertexAttribFormatNV_args};

static const char * _glVertexAttribIFormatNV_args[4] = {"index", "size", "type", "stride"};
static const trace::FunctionSig _glVertexAttribIFormatNV_sig = {2754, "glVertexAttribIFormatNV", 4, _glVertexAttribIFormatNV_args};

static const char * _glGetIntegerui64i_vNV_args[3] = {"value", "index", "result"};
static const trace::FunctionSig _glGetIntegerui64i_vNV_sig = {2755, "glGetIntegerui64i_vNV", 3, _glGetIntegerui64i_vNV_args};

static const char * _glAreProgramsResidentNV_args[3] = {"n", "ids", "residences"};
static const trace::FunctionSig _glAreProgramsResidentNV_sig = {2756, "glAreProgramsResidentNV", 3, _glAreProgramsResidentNV_args};

static const char * _glBindProgramNV_args[2] = {"target", "program"};
static const trace::FunctionSig _glBindProgramNV_sig = {2757, "glBindProgramNV", 2, _glBindProgramNV_args};

static const char * _glDeleteProgramsNV_args[2] = {"n", "programs"};
static const trace::FunctionSig _glDeleteProgramsNV_sig = {2758, "glDeleteProgramsNV", 2, _glDeleteProgramsNV_args};

static const char * _glExecuteProgramNV_args[3] = {"target", "id", "params"};
static const trace::FunctionSig _glExecuteProgramNV_sig = {2759, "glExecuteProgramNV", 3, _glExecuteProgramNV_args};

static const char * _glGenProgramsNV_args[2] = {"n", "programs"};
static const trace::FunctionSig _glGenProgramsNV_sig = {2760, "glGenProgramsNV", 2, _glGenProgramsNV_args};

static const char * _glGetProgramParameterdvNV_args[4] = {"target", "index", "pname", "params"};
static const trace::FunctionSig _glGetProgramParameterdvNV_sig = {2761, "glGetProgramParameterdvNV", 4, _glGetProgramParameterdvNV_args};

static const char * _glGetProgramParameterfvNV_args[4] = {"target", "index", "pname", "params"};
static const trace::FunctionSig _glGetProgramParameterfvNV_sig = {2762, "glGetProgramParameterfvNV", 4, _glGetProgramParameterfvNV_args};

static const char * _glGetProgramivNV_args[3] = {"id", "pname", "params"};
static const trace::FunctionSig _glGetProgramivNV_sig = {2763, "glGetProgramivNV", 3, _glGetProgramivNV_args};

static const char * _glGetProgramStringNV_args[3] = {"id", "pname", "program"};
static const trace::FunctionSig _glGetProgramStringNV_sig = {2764, "glGetProgramStringNV", 3, _glGetProgramStringNV_args};

static const char * _glGetTrackMatrixivNV_args[4] = {"target", "address", "pname", "params"};
static const trace::FunctionSig _glGetTrackMatrixivNV_sig = {2765, "glGetTrackMatrixivNV", 4, _glGetTrackMatrixivNV_args};

static const char * _glGetVertexAttribdvNV_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribdvNV_sig = {2766, "glGetVertexAttribdvNV", 3, _glGetVertexAttribdvNV_args};

static const char * _glGetVertexAttribfvNV_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribfvNV_sig = {2767, "glGetVertexAttribfvNV", 3, _glGetVertexAttribfvNV_args};

static const char * _glGetVertexAttribivNV_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribivNV_sig = {2768, "glGetVertexAttribivNV", 3, _glGetVertexAttribivNV_args};

static const char * _glGetVertexAttribPointervNV_args[3] = {"index", "pname", "pointer"};
static const trace::FunctionSig _glGetVertexAttribPointervNV_sig = {2769, "glGetVertexAttribPointervNV", 3, _glGetVertexAttribPointervNV_args};

static const char * _glIsProgramNV_args[1] = {"program"};
static const trace::FunctionSig _glIsProgramNV_sig = {2770, "glIsProgramNV", 1, _glIsProgramNV_args};

static const char * _glLoadProgramNV_args[4] = {"target", "id", "len", "program"};
static const trace::FunctionSig _glLoadProgramNV_sig = {2771, "glLoadProgramNV", 4, _glLoadProgramNV_args};

static const char * _glProgramParameter4dNV_args[6] = {"target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramParameter4dNV_sig = {2772, "glProgramParameter4dNV", 6, _glProgramParameter4dNV_args};

static const char * _glProgramParameter4dvNV_args[3] = {"target", "index", "v"};
static const trace::FunctionSig _glProgramParameter4dvNV_sig = {2773, "glProgramParameter4dvNV", 3, _glProgramParameter4dvNV_args};

static const char * _glProgramParameter4fNV_args[6] = {"target", "index", "x", "y", "z", "w"};
static const trace::FunctionSig _glProgramParameter4fNV_sig = {2774, "glProgramParameter4fNV", 6, _glProgramParameter4fNV_args};

static const char * _glProgramParameter4fvNV_args[3] = {"target", "index", "v"};
static const trace::FunctionSig _glProgramParameter4fvNV_sig = {2775, "glProgramParameter4fvNV", 3, _glProgramParameter4fvNV_args};

static const char * _glProgramParameters4dvNV_args[4] = {"target", "index", "count", "v"};
static const trace::FunctionSig _glProgramParameters4dvNV_sig = {2776, "glProgramParameters4dvNV", 4, _glProgramParameters4dvNV_args};

static const char * _glProgramParameters4fvNV_args[4] = {"target", "index", "count", "v"};
static const trace::FunctionSig _glProgramParameters4fvNV_sig = {2777, "glProgramParameters4fvNV", 4, _glProgramParameters4fvNV_args};

static const char * _glRequestResidentProgramsNV_args[2] = {"n", "programs"};
static const trace::FunctionSig _glRequestResidentProgramsNV_sig = {2778, "glRequestResidentProgramsNV", 2, _glRequestResidentProgramsNV_args};

static const char * _glTrackMatrixNV_args[4] = {"target", "address", "matrix", "transform"};
static const trace::FunctionSig _glTrackMatrixNV_sig = {2779, "glTrackMatrixNV", 4, _glTrackMatrixNV_args};

static const char * _glVertexAttribPointerNV_args[5] = {"index", "size", "type", "stride", "pointer"};
static const trace::FunctionSig _glVertexAttribPointerNV_sig = {2780, "glVertexAttribPointerNV", 5, _glVertexAttribPointerNV_args};

static const char * _glVertexAttrib1dNV_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttrib1dNV_sig = {2781, "glVertexAttrib1dNV", 2, _glVertexAttrib1dNV_args};

static const char * _glVertexAttrib1dvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib1dvNV_sig = {2782, "glVertexAttrib1dvNV", 2, _glVertexAttrib1dvNV_args};

static const char * _glVertexAttrib1fNV_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttrib1fNV_sig = {2783, "glVertexAttrib1fNV", 2, _glVertexAttrib1fNV_args};

static const char * _glVertexAttrib1fvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib1fvNV_sig = {2784, "glVertexAttrib1fvNV", 2, _glVertexAttrib1fvNV_args};

static const char * _glVertexAttrib1sNV_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttrib1sNV_sig = {2785, "glVertexAttrib1sNV", 2, _glVertexAttrib1sNV_args};

static const char * _glVertexAttrib1svNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib1svNV_sig = {2786, "glVertexAttrib1svNV", 2, _glVertexAttrib1svNV_args};

static const char * _glVertexAttrib2dNV_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttrib2dNV_sig = {2787, "glVertexAttrib2dNV", 3, _glVertexAttrib2dNV_args};

static const char * _glVertexAttrib2dvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib2dvNV_sig = {2788, "glVertexAttrib2dvNV", 2, _glVertexAttrib2dvNV_args};

static const char * _glVertexAttrib2fNV_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttrib2fNV_sig = {2789, "glVertexAttrib2fNV", 3, _glVertexAttrib2fNV_args};

static const char * _glVertexAttrib2fvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib2fvNV_sig = {2790, "glVertexAttrib2fvNV", 2, _glVertexAttrib2fvNV_args};

static const char * _glVertexAttrib2sNV_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttrib2sNV_sig = {2791, "glVertexAttrib2sNV", 3, _glVertexAttrib2sNV_args};

static const char * _glVertexAttrib2svNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib2svNV_sig = {2792, "glVertexAttrib2svNV", 2, _glVertexAttrib2svNV_args};

static const char * _glVertexAttrib3dNV_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttrib3dNV_sig = {2793, "glVertexAttrib3dNV", 4, _glVertexAttrib3dNV_args};

static const char * _glVertexAttrib3dvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib3dvNV_sig = {2794, "glVertexAttrib3dvNV", 2, _glVertexAttrib3dvNV_args};

static const char * _glVertexAttrib3fNV_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttrib3fNV_sig = {2795, "glVertexAttrib3fNV", 4, _glVertexAttrib3fNV_args};

static const char * _glVertexAttrib3fvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib3fvNV_sig = {2796, "glVertexAttrib3fvNV", 2, _glVertexAttrib3fvNV_args};

static const char * _glVertexAttrib3sNV_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttrib3sNV_sig = {2797, "glVertexAttrib3sNV", 4, _glVertexAttrib3sNV_args};

static const char * _glVertexAttrib3svNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib3svNV_sig = {2798, "glVertexAttrib3svNV", 2, _glVertexAttrib3svNV_args};

static const char * _glVertexAttrib4dNV_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4dNV_sig = {2799, "glVertexAttrib4dNV", 5, _glVertexAttrib4dNV_args};

static const char * _glVertexAttrib4dvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4dvNV_sig = {2800, "glVertexAttrib4dvNV", 2, _glVertexAttrib4dvNV_args};

static const char * _glVertexAttrib4fNV_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4fNV_sig = {2801, "glVertexAttrib4fNV", 5, _glVertexAttrib4fNV_args};

static const char * _glVertexAttrib4fvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4fvNV_sig = {2802, "glVertexAttrib4fvNV", 2, _glVertexAttrib4fvNV_args};

static const char * _glVertexAttrib4sNV_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4sNV_sig = {2803, "glVertexAttrib4sNV", 5, _glVertexAttrib4sNV_args};

static const char * _glVertexAttrib4svNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4svNV_sig = {2804, "glVertexAttrib4svNV", 2, _glVertexAttrib4svNV_args};

static const char * _glVertexAttrib4ubNV_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttrib4ubNV_sig = {2805, "glVertexAttrib4ubNV", 5, _glVertexAttrib4ubNV_args};

static const char * _glVertexAttrib4ubvNV_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttrib4ubvNV_sig = {2806, "glVertexAttrib4ubvNV", 2, _glVertexAttrib4ubvNV_args};

static const char * _glVertexAttribs1dvNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs1dvNV_sig = {2807, "glVertexAttribs1dvNV", 3, _glVertexAttribs1dvNV_args};

static const char * _glVertexAttribs1fvNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs1fvNV_sig = {2808, "glVertexAttribs1fvNV", 3, _glVertexAttribs1fvNV_args};

static const char * _glVertexAttribs1svNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs1svNV_sig = {2809, "glVertexAttribs1svNV", 3, _glVertexAttribs1svNV_args};

static const char * _glVertexAttribs2dvNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs2dvNV_sig = {2810, "glVertexAttribs2dvNV", 3, _glVertexAttribs2dvNV_args};

static const char * _glVertexAttribs2fvNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs2fvNV_sig = {2811, "glVertexAttribs2fvNV", 3, _glVertexAttribs2fvNV_args};

static const char * _glVertexAttribs2svNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs2svNV_sig = {2812, "glVertexAttribs2svNV", 3, _glVertexAttribs2svNV_args};

static const char * _glVertexAttribs3dvNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs3dvNV_sig = {2813, "glVertexAttribs3dvNV", 3, _glVertexAttribs3dvNV_args};

static const char * _glVertexAttribs3fvNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs3fvNV_sig = {2814, "glVertexAttribs3fvNV", 3, _glVertexAttribs3fvNV_args};

static const char * _glVertexAttribs3svNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs3svNV_sig = {2815, "glVertexAttribs3svNV", 3, _glVertexAttribs3svNV_args};

static const char * _glVertexAttribs4dvNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs4dvNV_sig = {2816, "glVertexAttribs4dvNV", 3, _glVertexAttribs4dvNV_args};

static const char * _glVertexAttribs4fvNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs4fvNV_sig = {2817, "glVertexAttribs4fvNV", 3, _glVertexAttribs4fvNV_args};

static const char * _glVertexAttribs4svNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs4svNV_sig = {2818, "glVertexAttribs4svNV", 3, _glVertexAttribs4svNV_args};

static const char * _glVertexAttribs4ubvNV_args[3] = {"index", "count", "v"};
static const trace::FunctionSig _glVertexAttribs4ubvNV_sig = {2819, "glVertexAttribs4ubvNV", 3, _glVertexAttribs4ubvNV_args};

static const char * _glVertexAttribI1iEXT_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttribI1iEXT_sig = {2820, "glVertexAttribI1iEXT", 2, _glVertexAttribI1iEXT_args};

static const char * _glVertexAttribI2iEXT_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttribI2iEXT_sig = {2821, "glVertexAttribI2iEXT", 3, _glVertexAttribI2iEXT_args};

static const char * _glVertexAttribI3iEXT_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttribI3iEXT_sig = {2822, "glVertexAttribI3iEXT", 4, _glVertexAttribI3iEXT_args};

static const char * _glVertexAttribI4iEXT_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttribI4iEXT_sig = {2823, "glVertexAttribI4iEXT", 5, _glVertexAttribI4iEXT_args};

static const char * _glVertexAttribI1uiEXT_args[2] = {"index", "x"};
static const trace::FunctionSig _glVertexAttribI1uiEXT_sig = {2824, "glVertexAttribI1uiEXT", 2, _glVertexAttribI1uiEXT_args};

static const char * _glVertexAttribI2uiEXT_args[3] = {"index", "x", "y"};
static const trace::FunctionSig _glVertexAttribI2uiEXT_sig = {2825, "glVertexAttribI2uiEXT", 3, _glVertexAttribI2uiEXT_args};

static const char * _glVertexAttribI3uiEXT_args[4] = {"index", "x", "y", "z"};
static const trace::FunctionSig _glVertexAttribI3uiEXT_sig = {2826, "glVertexAttribI3uiEXT", 4, _glVertexAttribI3uiEXT_args};

static const char * _glVertexAttribI4uiEXT_args[5] = {"index", "x", "y", "z", "w"};
static const trace::FunctionSig _glVertexAttribI4uiEXT_sig = {2827, "glVertexAttribI4uiEXT", 5, _glVertexAttribI4uiEXT_args};

static const char * _glVertexAttribI1ivEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI1ivEXT_sig = {2828, "glVertexAttribI1ivEXT", 2, _glVertexAttribI1ivEXT_args};

static const char * _glVertexAttribI2ivEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI2ivEXT_sig = {2829, "glVertexAttribI2ivEXT", 2, _glVertexAttribI2ivEXT_args};

static const char * _glVertexAttribI3ivEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI3ivEXT_sig = {2830, "glVertexAttribI3ivEXT", 2, _glVertexAttribI3ivEXT_args};

static const char * _glVertexAttribI4ivEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4ivEXT_sig = {2831, "glVertexAttribI4ivEXT", 2, _glVertexAttribI4ivEXT_args};

static const char * _glVertexAttribI1uivEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI1uivEXT_sig = {2832, "glVertexAttribI1uivEXT", 2, _glVertexAttribI1uivEXT_args};

static const char * _glVertexAttribI2uivEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI2uivEXT_sig = {2833, "glVertexAttribI2uivEXT", 2, _glVertexAttribI2uivEXT_args};

static const char * _glVertexAttribI3uivEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI3uivEXT_sig = {2834, "glVertexAttribI3uivEXT", 2, _glVertexAttribI3uivEXT_args};

static const char * _glVertexAttribI4uivEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4uivEXT_sig = {2835, "glVertexAttribI4uivEXT", 2, _glVertexAttribI4uivEXT_args};

static const char * _glVertexAttribI4bvEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4bvEXT_sig = {2836, "glVertexAttribI4bvEXT", 2, _glVertexAttribI4bvEXT_args};

static const char * _glVertexAttribI4svEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4svEXT_sig = {2837, "glVertexAttribI4svEXT", 2, _glVertexAttribI4svEXT_args};

static const char * _glVertexAttribI4ubvEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4ubvEXT_sig = {2838, "glVertexAttribI4ubvEXT", 2, _glVertexAttribI4ubvEXT_args};

static const char * _glVertexAttribI4usvEXT_args[2] = {"index", "v"};
static const trace::FunctionSig _glVertexAttribI4usvEXT_sig = {2839, "glVertexAttribI4usvEXT", 2, _glVertexAttribI4usvEXT_args};

static const char * _glVertexAttribIPointerEXT_args[5] = {"index", "size", "type", "stride", "pointer"};
static const trace::FunctionSig _glVertexAttribIPointerEXT_sig = {2840, "glVertexAttribIPointerEXT", 5, _glVertexAttribIPointerEXT_args};

static const char * _glGetVertexAttribIivEXT_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribIivEXT_sig = {2841, "glGetVertexAttribIivEXT", 3, _glGetVertexAttribIivEXT_args};

static const char * _glGetVertexAttribIuivEXT_args[3] = {"index", "pname", "params"};
static const trace::FunctionSig _glGetVertexAttribIuivEXT_sig = {2842, "glGetVertexAttribIuivEXT", 3, _glGetVertexAttribIuivEXT_args};

static const char * _glBeginVideoCaptureNV_args[1] = {"video_capture_slot"};
static const trace::FunctionSig _glBeginVideoCaptureNV_sig = {2843, "glBeginVideoCaptureNV", 1, _glBeginVideoCaptureNV_args};

static const char * _glBindVideoCaptureStreamBufferNV_args[4] = {"video_capture_slot", "stream", "frame_region", "offset"};
static const trace::FunctionSig _glBindVideoCaptureStreamBufferNV_sig = {2844, "glBindVideoCaptureStreamBufferNV", 4, _glBindVideoCaptureStreamBufferNV_args};

static const char * _glBindVideoCaptureStreamTextureNV_args[5] = {"video_capture_slot", "stream", "frame_region", "target", "texture"};
static const trace::FunctionSig _glBindVideoCaptureStreamTextureNV_sig = {2845, "glBindVideoCaptureStreamTextureNV", 5, _glBindVideoCaptureStreamTextureNV_args};

static const char * _glEndVideoCaptureNV_args[1] = {"video_capture_slot"};
static const trace::FunctionSig _glEndVideoCaptureNV_sig = {2846, "glEndVideoCaptureNV", 1, _glEndVideoCaptureNV_args};

static const char * _glGetVideoCaptureivNV_args[3] = {"video_capture_slot", "pname", "params"};
static const trace::FunctionSig _glGetVideoCaptureivNV_sig = {2847, "glGetVideoCaptureivNV", 3, _glGetVideoCaptureivNV_args};

static const char * _glGetVideoCaptureStreamivNV_args[4] = {"video_capture_slot", "stream", "pname", "params"};
static const trace::FunctionSig _glGetVideoCaptureStreamivNV_sig = {2848, "glGetVideoCaptureStreamivNV", 4, _glGetVideoCaptureStreamivNV_args};

static const char * _glGetVideoCaptureStreamfvNV_args[4] = {"video_capture_slot", "stream", "pname", "params"};
static const trace::FunctionSig _glGetVideoCaptureStreamfvNV_sig = {2849, "glGetVideoCaptureStreamfvNV", 4, _glGetVideoCaptureStreamfvNV_args};

static const char * _glGetVideoCaptureStreamdvNV_args[4] = {"video_capture_slot", "stream", "pname", "params"};
static const trace::FunctionSig _glGetVideoCaptureStreamdvNV_sig = {2850, "glGetVideoCaptureStreamdvNV", 4, _glGetVideoCaptureStreamdvNV_args};

static const char * _glVideoCaptureNV_args[3] = {"video_capture_slot", "sequence_num", "capture_time"};
static const trace::FunctionSig _glVideoCaptureNV_sig = {2851, "glVideoCaptureNV", 3, _glVideoCaptureNV_args};

static const char * _glVideoCaptureStreamParameterivNV_args[4] = {"video_capture_slot", "stream", "pname", "params"};
static const trace::FunctionSig _glVideoCaptureStreamParameterivNV_sig = {2852, "glVideoCaptureStreamParameterivNV", 4, _glVideoCaptureStreamParameterivNV_args};

static const char * _glVideoCaptureStreamParameterfvNV_args[4] = {"video_capture_slot", "stream", "pname", "params"};
static const trace::FunctionSig _glVideoCaptureStreamParameterfvNV_sig = {2853, "glVideoCaptureStreamParameterfvNV", 4, _glVideoCaptureStreamParameterfvNV_args};

static const char * _glVideoCaptureStreamParameterdvNV_args[4] = {"video_capture_slot", "stream", "pname", "params"};
static const trace::FunctionSig _glVideoCaptureStreamParameterdvNV_sig = {2854, "glVideoCaptureStreamParameterdvNV", 4, _glVideoCaptureStreamParameterdvNV_args};

static const char * _glBlendEquationSeparateOES_args[2] = {"modeRGB", "modeAlpha"};
static const trace::FunctionSig _glBlendEquationSeparateOES_sig = {2855, "glBlendEquationSeparateOES", 2, _glBlendEquationSeparateOES_args};

static const char * _glBlendFuncSeparateOES_args[4] = {"sfactorRGB", "dfactorRGB", "sfactorAlpha", "dfactorAlpha"};
static const trace::FunctionSig _glBlendFuncSeparateOES_sig = {2856, "glBlendFuncSeparateOES", 4, _glBlendFuncSeparateOES_args};

static const char * _glBlendEquationOES_args[1] = {"mode"};
static const trace::FunctionSig _glBlendEquationOES_sig = {2857, "glBlendEquationOES", 1, _glBlendEquationOES_args};

static const char * _glDrawTexfOES_args[5] = {"x", "y", "z", "width", "height"};
static const trace::FunctionSig _glDrawTexfOES_sig = {2858, "glDrawTexfOES", 5, _glDrawTexfOES_args};

static const char * _glDrawTexfvOES_args[1] = {"coords"};
static const trace::FunctionSig _glDrawTexfvOES_sig = {2859, "glDrawTexfvOES", 1, _glDrawTexfvOES_args};

static const char * _glDrawTexiOES_args[5] = {"x", "y", "z", "width", "height"};
static const trace::FunctionSig _glDrawTexiOES_sig = {2860, "glDrawTexiOES", 5, _glDrawTexiOES_args};

static const char * _glDrawTexivOES_args[1] = {"coords"};
static const trace::FunctionSig _glDrawTexivOES_sig = {2861, "glDrawTexivOES", 1, _glDrawTexivOES_args};

static const char * _glDrawTexsOES_args[5] = {"x", "y", "z", "width", "height"};
static const trace::FunctionSig _glDrawTexsOES_sig = {2862, "glDrawTexsOES", 5, _glDrawTexsOES_args};

static const char * _glDrawTexsvOES_args[1] = {"coords"};
static const trace::FunctionSig _glDrawTexsvOES_sig = {2863, "glDrawTexsvOES", 1, _glDrawTexsvOES_args};

static const char * _glIsRenderbufferOES_args[1] = {"renderbuffer"};
static const trace::FunctionSig _glIsRenderbufferOES_sig = {2864, "glIsRenderbufferOES", 1, _glIsRenderbufferOES_args};

static const char * _glBindRenderbufferOES_args[2] = {"target", "renderbuffer"};
static const trace::FunctionSig _glBindRenderbufferOES_sig = {2865, "glBindRenderbufferOES", 2, _glBindRenderbufferOES_args};

static const char * _glDeleteRenderbuffersOES_args[2] = {"n", "renderbuffers"};
static const trace::FunctionSig _glDeleteRenderbuffersOES_sig = {2866, "glDeleteRenderbuffersOES", 2, _glDeleteRenderbuffersOES_args};

static const char * _glGenRenderbuffersOES_args[2] = {"n", "renderbuffers"};
static const trace::FunctionSig _glGenRenderbuffersOES_sig = {2867, "glGenRenderbuffersOES", 2, _glGenRenderbuffersOES_args};

static const char * _glRenderbufferStorageOES_args[4] = {"target", "internalformat", "width", "height"};
static const trace::FunctionSig _glRenderbufferStorageOES_sig = {2868, "glRenderbufferStorageOES", 4, _glRenderbufferStorageOES_args};

static const char * _glGetRenderbufferParameterivOES_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetRenderbufferParameterivOES_sig = {2869, "glGetRenderbufferParameterivOES", 3, _glGetRenderbufferParameterivOES_args};

static const char * _glIsFramebufferOES_args[1] = {"framebuffer"};
static const trace::FunctionSig _glIsFramebufferOES_sig = {2870, "glIsFramebufferOES", 1, _glIsFramebufferOES_args};

static const char * _glBindFramebufferOES_args[2] = {"target", "framebuffer"};
static const trace::FunctionSig _glBindFramebufferOES_sig = {2871, "glBindFramebufferOES", 2, _glBindFramebufferOES_args};

static const char * _glDeleteFramebuffersOES_args[2] = {"n", "framebuffers"};
static const trace::FunctionSig _glDeleteFramebuffersOES_sig = {2872, "glDeleteFramebuffersOES", 2, _glDeleteFramebuffersOES_args};

static const char * _glGenFramebuffersOES_args[2] = {"n", "framebuffers"};
static const trace::FunctionSig _glGenFramebuffersOES_sig = {2873, "glGenFramebuffersOES", 2, _glGenFramebuffersOES_args};

static const char * _glCheckFramebufferStatusOES_args[1] = {"target"};
static const trace::FunctionSig _glCheckFramebufferStatusOES_sig = {2874, "glCheckFramebufferStatusOES", 1, _glCheckFramebufferStatusOES_args};

static const char * _glFramebufferTexture2DOES_args[5] = {"target", "attachment", "textarget", "texture", "level"};
static const trace::FunctionSig _glFramebufferTexture2DOES_sig = {2875, "glFramebufferTexture2DOES", 5, _glFramebufferTexture2DOES_args};

static const char * _glFramebufferRenderbufferOES_args[4] = {"target", "attachment", "renderbuffertarget", "renderbuffer"};
static const trace::FunctionSig _glFramebufferRenderbufferOES_sig = {2876, "glFramebufferRenderbufferOES", 4, _glFramebufferRenderbufferOES_args};

static const char * _glGetFramebufferAttachmentParameterivOES_args[4] = {"target", "attachment", "pname", "params"};
static const trace::FunctionSig _glGetFramebufferAttachmentParameterivOES_sig = {2877, "glGetFramebufferAttachmentParameterivOES", 4, _glGetFramebufferAttachmentParameterivOES_args};

static const char * _glGenerateMipmapOES_args[1] = {"target"};
static const trace::FunctionSig _glGenerateMipmapOES_sig = {2878, "glGenerateMipmapOES", 1, _glGenerateMipmapOES_args};

static const char * _glGetProgramBinaryOES_args[5] = {"program", "bufSize", "length", "binaryFormat", "binary"};
static const trace::FunctionSig _glGetProgramBinaryOES_sig = {2879, "glGetProgramBinaryOES", 5, _glGetProgramBinaryOES_args};

static const char * _glProgramBinaryOES_args[4] = {"program", "binaryFormat", "binary", "length"};
static const trace::FunctionSig _glProgramBinaryOES_sig = {2880, "glProgramBinaryOES", 4, _glProgramBinaryOES_args};

static const char * _glGetBufferPointervOES_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetBufferPointervOES_sig = {2881, "glGetBufferPointervOES", 3, _glGetBufferPointervOES_args};

static const char * _glMapBufferOES_args[2] = {"target", "access"};
static const trace::FunctionSig _glMapBufferOES_sig = {2882, "glMapBufferOES", 2, _glMapBufferOES_args};

static const char * _glUnmapBufferOES_args[1] = {"target"};
static const trace::FunctionSig _glUnmapBufferOES_sig = {2883, "glUnmapBufferOES", 1, _glUnmapBufferOES_args};

static const char * _glCurrentPaletteMatrixOES_args[1] = {"index"};
static const trace::FunctionSig _glCurrentPaletteMatrixOES_sig = {2884, "glCurrentPaletteMatrixOES", 1, _glCurrentPaletteMatrixOES_args};

static const char ** _glLoadPaletteFromModelViewMatrixOES_args = NULL;
static const trace::FunctionSig _glLoadPaletteFromModelViewMatrixOES_sig = {2885, "glLoadPaletteFromModelViewMatrixOES", 0, _glLoadPaletteFromModelViewMatrixOES_args};

static const char * _glMatrixIndexPointerOES_args[4] = {"size", "type", "stride", "pointer"};
static const trace::FunctionSig _glMatrixIndexPointerOES_sig = {2886, "glMatrixIndexPointerOES", 4, _glMatrixIndexPointerOES_args};

static const char * _glWeightPointerOES_args[4] = {"size", "type", "stride", "pointer"};
static const trace::FunctionSig _glWeightPointerOES_sig = {2887, "glWeightPointerOES", 4, _glWeightPointerOES_args};

static const char * _glPointSizePointerOES_args[3] = {"type", "stride", "ptr"};
static const trace::FunctionSig _glPointSizePointerOES_sig = {2888, "glPointSizePointerOES", 3, _glPointSizePointerOES_args};

static const char * _glQueryMatrixxOES_args[2] = {"mantissa", "exponent"};
static const trace::FunctionSig _glQueryMatrixxOES_sig = {2889, "glQueryMatrixxOES", 2, _glQueryMatrixxOES_args};

static const char * _glClearDepthfOES_args[1] = {"depth"};
static const trace::FunctionSig _glClearDepthfOES_sig = {2890, "glClearDepthfOES", 1, _glClearDepthfOES_args};

static const char * _glClipPlanefOES_args[2] = {"plane", "equation"};
static const trace::FunctionSig _glClipPlanefOES_sig = {2891, "glClipPlanefOES", 2, _glClipPlanefOES_args};

static const char * _glDepthRangefOES_args[2] = {"n", "f"};
static const trace::FunctionSig _glDepthRangefOES_sig = {2892, "glDepthRangefOES", 2, _glDepthRangefOES_args};

static const char * _glFrustumfOES_args[6] = {"l", "r", "b", "t", "n", "f"};
static const trace::FunctionSig _glFrustumfOES_sig = {2893, "glFrustumfOES", 6, _glFrustumfOES_args};

static const char * _glGetClipPlanefOES_args[2] = {"plane", "equation"};
static const trace::FunctionSig _glGetClipPlanefOES_sig = {2894, "glGetClipPlanefOES", 2, _glGetClipPlanefOES_args};

static const char * _glOrthofOES_args[6] = {"l", "r", "b", "t", "n", "f"};
static const trace::FunctionSig _glOrthofOES_sig = {2895, "glOrthofOES", 6, _glOrthofOES_args};

static const char * _glTexImage3DOES_args[10] = {"target", "level", "internalformat", "width", "height", "depth", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glTexImage3DOES_sig = {2896, "glTexImage3DOES", 10, _glTexImage3DOES_args};

static const char * _glTexSubImage3DOES_args[11] = {"target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "type", "pixels"};
static const trace::FunctionSig _glTexSubImage3DOES_sig = {2897, "glTexSubImage3DOES", 11, _glTexSubImage3DOES_args};

static const char * _glCopyTexSubImage3DOES_args[9] = {"target", "level", "xoffset", "yoffset", "zoffset", "x", "y", "width", "height"};
static const trace::FunctionSig _glCopyTexSubImage3DOES_sig = {2898, "glCopyTexSubImage3DOES", 9, _glCopyTexSubImage3DOES_args};

static const char * _glCompressedTexImage3DOES_args[9] = {"target", "level", "internalformat", "width", "height", "depth", "border", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexImage3DOES_sig = {2899, "glCompressedTexImage3DOES", 9, _glCompressedTexImage3DOES_args};

static const char * _glCompressedTexSubImage3DOES_args[11] = {"target", "level", "xoffset", "yoffset", "zoffset", "width", "height", "depth", "format", "imageSize", "data"};
static const trace::FunctionSig _glCompressedTexSubImage3DOES_sig = {2900, "glCompressedTexSubImage3DOES", 11, _glCompressedTexSubImage3DOES_args};

static const char * _glFramebufferTexture3DOES_args[6] = {"target", "attachment", "textarget", "texture", "level", "zoffset"};
static const trace::FunctionSig _glFramebufferTexture3DOES_sig = {2901, "glFramebufferTexture3DOES", 6, _glFramebufferTexture3DOES_args};

static const char * _glTexGenfOES_args[3] = {"coord", "pname", "param"};
static const trace::FunctionSig _glTexGenfOES_sig = {2902, "glTexGenfOES", 3, _glTexGenfOES_args};

static const char * _glTexGenfvOES_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glTexGenfvOES_sig = {2903, "glTexGenfvOES", 3, _glTexGenfvOES_args};

static const char * _glTexGeniOES_args[3] = {"coord", "pname", "param"};
static const trace::FunctionSig _glTexGeniOES_sig = {2904, "glTexGeniOES", 3, _glTexGeniOES_args};

static const char * _glTexGenivOES_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glTexGenivOES_sig = {2905, "glTexGenivOES", 3, _glTexGenivOES_args};

static const char * _glTexGenxOES_args[3] = {"coord", "pname", "param"};
static const trace::FunctionSig _glTexGenxOES_sig = {2906, "glTexGenxOES", 3, _glTexGenxOES_args};

static const char * _glTexGenxvOES_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glTexGenxvOES_sig = {2907, "glTexGenxvOES", 3, _glTexGenxvOES_args};

static const char * _glGetTexGenfvOES_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glGetTexGenfvOES_sig = {2908, "glGetTexGenfvOES", 3, _glGetTexGenfvOES_args};

static const char * _glGetTexGenivOES_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glGetTexGenivOES_sig = {2909, "glGetTexGenivOES", 3, _glGetTexGenivOES_args};

static const char * _glGetTexGenxvOES_args[3] = {"coord", "pname", "params"};
static const trace::FunctionSig _glGetTexGenxvOES_sig = {2910, "glGetTexGenxvOES", 3, _glGetTexGenxvOES_args};

static const char * _glBindVertexArrayOES_args[1] = {"array"};
static const trace::FunctionSig _glBindVertexArrayOES_sig = {2911, "glBindVertexArrayOES", 1, _glBindVertexArrayOES_args};

static const char * _glDeleteVertexArraysOES_args[2] = {"n", "arrays"};
static const trace::FunctionSig _glDeleteVertexArraysOES_sig = {2912, "glDeleteVertexArraysOES", 2, _glDeleteVertexArraysOES_args};

static const char * _glGenVertexArraysOES_args[2] = {"n", "arrays"};
static const trace::FunctionSig _glGenVertexArraysOES_sig = {2913, "glGenVertexArraysOES", 2, _glGenVertexArraysOES_args};

static const char * _glIsVertexArrayOES_args[1] = {"array"};
static const trace::FunctionSig _glIsVertexArrayOES_sig = {2914, "glIsVertexArrayOES", 1, _glIsVertexArrayOES_args};

static const char * _glViewportArrayvOES_args[3] = {"first", "count", "v"};
static const trace::FunctionSig _glViewportArrayvOES_sig = {2915, "glViewportArrayvOES", 3, _glViewportArrayvOES_args};

static const char * _glViewportIndexedfOES_args[5] = {"index", "x", "y", "w", "h"};
static const trace::FunctionSig _glViewportIndexedfOES_sig = {2916, "glViewportIndexedfOES", 5, _glViewportIndexedfOES_args};

static const char * _glViewportIndexedfvOES_args[2] = {"index", "v"};
static const trace::FunctionSig _glViewportIndexedfvOES_sig = {2917, "glViewportIndexedfvOES", 2, _glViewportIndexedfvOES_args};

static const char * _glScissorArrayvOES_args[3] = {"first", "count", "v"};
static const trace::FunctionSig _glScissorArrayvOES_sig = {2918, "glScissorArrayvOES", 3, _glScissorArrayvOES_args};

static const char * _glScissorIndexedOES_args[5] = {"index", "left", "bottom", "width", "height"};
static const trace::FunctionSig _glScissorIndexedOES_sig = {2919, "glScissorIndexedOES", 5, _glScissorIndexedOES_args};

static const char * _glScissorIndexedvOES_args[2] = {"index", "v"};
static const trace::FunctionSig _glScissorIndexedvOES_sig = {2920, "glScissorIndexedvOES", 2, _glScissorIndexedvOES_args};

static const char * _glDepthRangeArrayfvOES_args[3] = {"first", "count", "v"};
static const trace::FunctionSig _glDepthRangeArrayfvOES_sig = {2921, "glDepthRangeArrayfvOES", 3, _glDepthRangeArrayfvOES_args};

static const char * _glDepthRangeIndexedfOES_args[3] = {"index", "n", "f"};
static const trace::FunctionSig _glDepthRangeIndexedfOES_sig = {2922, "glDepthRangeIndexedfOES", 3, _glDepthRangeIndexedfOES_args};

static const char * _glGetFloati_vOES_args[3] = {"target", "index", "data"};
static const trace::FunctionSig _glGetFloati_vOES_sig = {2923, "glGetFloati_vOES", 3, _glGetFloati_vOES_args};

static const char * _glFramebufferTextureMultiviewOVR_args[6] = {"target", "attachment", "texture", "level", "baseViewIndex", "numViews"};
static const trace::FunctionSig _glFramebufferTextureMultiviewOVR_sig = {2924, "glFramebufferTextureMultiviewOVR", 6, _glFramebufferTextureMultiviewOVR_args};

static const char * _glNamedFramebufferTextureMultiviewOVR_args[6] = {"framebuffer", "attachment", "texture", "level", "baseViewIndex", "numViews"};
static const trace::FunctionSig _glNamedFramebufferTextureMultiviewOVR_sig = {2925, "glNamedFramebufferTextureMultiviewOVR", 6, _glNamedFramebufferTextureMultiviewOVR_args};

static const char * _glHintPGI_args[2] = {"target", "mode"};
static const trace::FunctionSig _glHintPGI_sig = {2926, "glHintPGI", 2, _glHintPGI_args};

static const char * _glDetailTexFuncSGIS_args[3] = {"target", "n", "points"};
static const trace::FunctionSig _glDetailTexFuncSGIS_sig = {2927, "glDetailTexFuncSGIS", 3, _glDetailTexFuncSGIS_args};

static const char * _glGetDetailTexFuncSGIS_args[2] = {"target", "points"};
static const trace::FunctionSig _glGetDetailTexFuncSGIS_sig = {2928, "glGetDetailTexFuncSGIS", 2, _glGetDetailTexFuncSGIS_args};

static const char * _glFogFuncSGIS_args[2] = {"n", "points"};
static const trace::FunctionSig _glFogFuncSGIS_sig = {2929, "glFogFuncSGIS", 2, _glFogFuncSGIS_args};

static const char * _glGetFogFuncSGIS_args[1] = {"points"};
static const trace::FunctionSig _glGetFogFuncSGIS_sig = {2930, "glGetFogFuncSGIS", 1, _glGetFogFuncSGIS_args};

static const char * _glSampleMaskSGIS_args[2] = {"value", "invert"};
static const trace::FunctionSig _glSampleMaskSGIS_sig = {2931, "glSampleMaskSGIS", 2, _glSampleMaskSGIS_args};

static const char * _glSamplePatternSGIS_args[1] = {"pattern"};
static const trace::FunctionSig _glSamplePatternSGIS_sig = {2932, "glSamplePatternSGIS", 1, _glSamplePatternSGIS_args};

static const char * _glPixelTexGenParameteriSGIS_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPixelTexGenParameteriSGIS_sig = {2933, "glPixelTexGenParameteriSGIS", 2, _glPixelTexGenParameteriSGIS_args};

static const char * _glPixelTexGenParameterivSGIS_args[2] = {"pname", "params"};
static const trace::FunctionSig _glPixelTexGenParameterivSGIS_sig = {2934, "glPixelTexGenParameterivSGIS", 2, _glPixelTexGenParameterivSGIS_args};

static const char * _glPixelTexGenParameterfSGIS_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPixelTexGenParameterfSGIS_sig = {2935, "glPixelTexGenParameterfSGIS", 2, _glPixelTexGenParameterfSGIS_args};

static const char * _glPixelTexGenParameterfvSGIS_args[2] = {"pname", "params"};
static const trace::FunctionSig _glPixelTexGenParameterfvSGIS_sig = {2936, "glPixelTexGenParameterfvSGIS", 2, _glPixelTexGenParameterfvSGIS_args};

static const char * _glGetPixelTexGenParameterivSGIS_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetPixelTexGenParameterivSGIS_sig = {2937, "glGetPixelTexGenParameterivSGIS", 2, _glGetPixelTexGenParameterivSGIS_args};

static const char * _glGetPixelTexGenParameterfvSGIS_args[2] = {"pname", "params"};
static const trace::FunctionSig _glGetPixelTexGenParameterfvSGIS_sig = {2938, "glGetPixelTexGenParameterfvSGIS", 2, _glGetPixelTexGenParameterfvSGIS_args};

static const char * _glPointParameterfSGIS_args[2] = {"pname", "param"};
static const trace::FunctionSig _glPointParameterfSGIS_sig = {2939, "glPointParameterfSGIS", 2, _glPointParameterfSGIS_args};

static const char * _glPointParameterfvSGIS_args[2] = {"pname", "params"};
static const trace::FunctionSig _glPointParameterfvSGIS_sig = {2940, "glPointParameterfvSGIS", 2, _glPointParameterfvSGIS_args};

static const char * _glSharpenTexFuncSGIS_args[3] = {"target", "n", "points"};
static const trace::FunctionSig _glSharpenTexFuncSGIS_sig = {2941, "glSharpenTexFuncSGIS", 3, _glSharpenTexFuncSGIS_args};

static const char * _glGetSharpenTexFuncSGIS_args[2] = {"target", "points"};
static const trace::FunctionSig _glGetSharpenTexFuncSGIS_sig = {2942, "glGetSharpenTexFuncSGIS", 2, _glGetSharpenTexFuncSGIS_args};

static const char * _glTexImage4DSGIS_args[11] = {"target", "level", "internalformat", "width", "height", "depth", "size4d", "border", "format", "type", "pixels"};
static const trace::FunctionSig _glTexImage4DSGIS_sig = {2943, "glTexImage4DSGIS", 11, _glTexImage4DSGIS_args};

static const char * _glTexSubImage4DSGIS_args[13] = {"target", "level", "xoffset", "yoffset", "zoffset", "woffset", "width", "height", "depth", "size4d", "format", "type", "pixels"};
static const trace::FunctionSig _glTexSubImage4DSGIS_sig = {2944, "glTexSubImage4DSGIS", 13, _glTexSubImage4DSGIS_args};

static const char * _glTextureColorMaskSGIS_args[4] = {"red", "green", "blue", "alpha"};
static const trace::FunctionSig _glTextureColorMaskSGIS_sig = {2945, "glTextureColorMaskSGIS", 4, _glTextureColorMaskSGIS_args};

static const char * _glGetTexFilterFuncSGIS_args[3] = {"target", "filter", "weights"};
static const trace::FunctionSig _glGetTexFilterFuncSGIS_sig = {2946, "glGetTexFilterFuncSGIS", 3, _glGetTexFilterFuncSGIS_args};

static const char * _glTexFilterFuncSGIS_args[4] = {"target", "filter", "n", "weights"};
static const trace::FunctionSig _glTexFilterFuncSGIS_sig = {2947, "glTexFilterFuncSGIS", 4, _glTexFilterFuncSGIS_args};

static const char * _glAsyncMarkerSGIX_args[1] = {"marker"};
static const trace::FunctionSig _glAsyncMarkerSGIX_sig = {2948, "glAsyncMarkerSGIX", 1, _glAsyncMarkerSGIX_args};

static const char * _glFinishAsyncSGIX_args[1] = {"markerp"};
static const trace::FunctionSig _glFinishAsyncSGIX_sig = {2949, "glFinishAsyncSGIX", 1, _glFinishAsyncSGIX_args};

static const char * _glPollAsyncSGIX_args[1] = {"markerp"};
static const trace::FunctionSig _glPollAsyncSGIX_sig = {2950, "glPollAsyncSGIX", 1, _glPollAsyncSGIX_args};

static const char * _glGenAsyncMarkersSGIX_args[1] = {"range"};
static const trace::FunctionSig _glGenAsyncMarkersSGIX_sig = {2951, "glGenAsyncMarkersSGIX", 1, _glGenAsyncMarkersSGIX_args};

static const char * _glDeleteAsyncMarkersSGIX_args[2] = {"marker", "range"};
static const trace::FunctionSig _glDeleteAsyncMarkersSGIX_sig = {2952, "glDeleteAsyncMarkersSGIX", 2, _glDeleteAsyncMarkersSGIX_args};

static const char * _glIsAsyncMarkerSGIX_args[1] = {"marker"};
static const trace::FunctionSig _glIsAsyncMarkerSGIX_sig = {2953, "glIsAsyncMarkerSGIX", 1, _glIsAsyncMarkerSGIX_args};

static const char ** _glFlushRasterSGIX_args = NULL;
static const trace::FunctionSig _glFlushRasterSGIX_sig = {2954, "glFlushRasterSGIX", 0, _glFlushRasterSGIX_args};

static const char * _glFragmentColorMaterialSGIX_args[2] = {"face", "mode"};
static const trace::FunctionSig _glFragmentColorMaterialSGIX_sig = {2955, "glFragmentColorMaterialSGIX", 2, _glFragmentColorMaterialSGIX_args};

static const char * _glFragmentLightfSGIX_args[3] = {"light", "pname", "param"};
static const trace::FunctionSig _glFragmentLightfSGIX_sig = {2956, "glFragmentLightfSGIX", 3, _glFragmentLightfSGIX_args};

static const char * _glFragmentLightfvSGIX_args[3] = {"light", "pname", "params"};
static const trace::FunctionSig _glFragmentLightfvSGIX_sig = {2957, "glFragmentLightfvSGIX", 3, _glFragmentLightfvSGIX_args};

static const char * _glFragmentLightiSGIX_args[3] = {"light", "pname", "param"};
static const trace::FunctionSig _glFragmentLightiSGIX_sig = {2958, "glFragmentLightiSGIX", 3, _glFragmentLightiSGIX_args};

static const char * _glFragmentLightivSGIX_args[3] = {"light", "pname", "params"};
static const trace::FunctionSig _glFragmentLightivSGIX_sig = {2959, "glFragmentLightivSGIX", 3, _glFragmentLightivSGIX_args};

static const char * _glFragmentLightModelfSGIX_args[2] = {"pname", "param"};
static const trace::FunctionSig _glFragmentLightModelfSGIX_sig = {2960, "glFragmentLightModelfSGIX", 2, _glFragmentLightModelfSGIX_args};

static const char * _glFragmentLightModelfvSGIX_args[2] = {"pname", "params"};
static const trace::FunctionSig _glFragmentLightModelfvSGIX_sig = {2961, "glFragmentLightModelfvSGIX", 2, _glFragmentLightModelfvSGIX_args};

static const char * _glFragmentLightModeliSGIX_args[2] = {"pname", "param"};
static const trace::FunctionSig _glFragmentLightModeliSGIX_sig = {2962, "glFragmentLightModeliSGIX", 2, _glFragmentLightModeliSGIX_args};

static const char * _glFragmentLightModelivSGIX_args[2] = {"pname", "params"};
static const trace::FunctionSig _glFragmentLightModelivSGIX_sig = {2963, "glFragmentLightModelivSGIX", 2, _glFragmentLightModelivSGIX_args};

static const char * _glFragmentMaterialfSGIX_args[3] = {"face", "pname", "param"};
static const trace::FunctionSig _glFragmentMaterialfSGIX_sig = {2964, "glFragmentMaterialfSGIX", 3, _glFragmentMaterialfSGIX_args};

static const char * _glFragmentMaterialfvSGIX_args[3] = {"face", "pname", "params"};
static const trace::FunctionSig _glFragmentMaterialfvSGIX_sig = {2965, "glFragmentMaterialfvSGIX", 3, _glFragmentMaterialfvSGIX_args};

static const char * _glFragmentMaterialiSGIX_args[3] = {"face", "pname", "param"};
static const trace::FunctionSig _glFragmentMaterialiSGIX_sig = {2966, "glFragmentMaterialiSGIX", 3, _glFragmentMaterialiSGIX_args};

static const char * _glFragmentMaterialivSGIX_args[3] = {"face", "pname", "params"};
static const trace::FunctionSig _glFragmentMaterialivSGIX_sig = {2967, "glFragmentMaterialivSGIX", 3, _glFragmentMaterialivSGIX_args};

static const char * _glGetFragmentLightfvSGIX_args[3] = {"light", "pname", "params"};
static const trace::FunctionSig _glGetFragmentLightfvSGIX_sig = {2968, "glGetFragmentLightfvSGIX", 3, _glGetFragmentLightfvSGIX_args};

static const char * _glGetFragmentLightivSGIX_args[3] = {"light", "pname", "params"};
static const trace::FunctionSig _glGetFragmentLightivSGIX_sig = {2969, "glGetFragmentLightivSGIX", 3, _glGetFragmentLightivSGIX_args};

static const char * _glGetFragmentMaterialfvSGIX_args[3] = {"face", "pname", "params"};
static const trace::FunctionSig _glGetFragmentMaterialfvSGIX_sig = {2970, "glGetFragmentMaterialfvSGIX", 3, _glGetFragmentMaterialfvSGIX_args};

static const char * _glGetFragmentMaterialivSGIX_args[3] = {"face", "pname", "params"};
static const trace::FunctionSig _glGetFragmentMaterialivSGIX_sig = {2971, "glGetFragmentMaterialivSGIX", 3, _glGetFragmentMaterialivSGIX_args};

static const char * _glLightEnviSGIX_args[2] = {"pname", "param"};
static const trace::FunctionSig _glLightEnviSGIX_sig = {2972, "glLightEnviSGIX", 2, _glLightEnviSGIX_args};

static const char * _glFrameZoomSGIX_args[1] = {"factor"};
static const trace::FunctionSig _glFrameZoomSGIX_sig = {2973, "glFrameZoomSGIX", 1, _glFrameZoomSGIX_args};

static const char * _glIglooInterfaceSGIX_args[2] = {"pname", "params"};
static const trace::FunctionSig _glIglooInterfaceSGIX_sig = {2974, "glIglooInterfaceSGIX", 2, _glIglooInterfaceSGIX_args};

static const char ** _glGetInstrumentsSGIX_args = NULL;
static const trace::FunctionSig _glGetInstrumentsSGIX_sig = {2975, "glGetInstrumentsSGIX", 0, _glGetInstrumentsSGIX_args};

static const char * _glInstrumentsBufferSGIX_args[2] = {"size", "buffer"};
static const trace::FunctionSig _glInstrumentsBufferSGIX_sig = {2976, "glInstrumentsBufferSGIX", 2, _glInstrumentsBufferSGIX_args};

static const char * _glPollInstrumentsSGIX_args[1] = {"marker_p"};
static const trace::FunctionSig _glPollInstrumentsSGIX_sig = {2977, "glPollInstrumentsSGIX", 1, _glPollInstrumentsSGIX_args};

static const char * _glReadInstrumentsSGIX_args[1] = {"marker"};
static const trace::FunctionSig _glReadInstrumentsSGIX_sig = {2978, "glReadInstrumentsSGIX", 1, _glReadInstrumentsSGIX_args};

static const char ** _glStartInstrumentsSGIX_args = NULL;
static const trace::FunctionSig _glStartInstrumentsSGIX_sig = {2979, "glStartInstrumentsSGIX", 0, _glStartInstrumentsSGIX_args};

static const char * _glStopInstrumentsSGIX_args[1] = {"marker"};
static const trace::FunctionSig _glStopInstrumentsSGIX_sig = {2980, "glStopInstrumentsSGIX", 1, _glStopInstrumentsSGIX_args};

static const char * _glGetListParameterfvSGIX_args[3] = {"list", "pname", "params"};
static const trace::FunctionSig _glGetListParameterfvSGIX_sig = {2981, "glGetListParameterfvSGIX", 3, _glGetListParameterfvSGIX_args};

static const char * _glGetListParameterivSGIX_args[3] = {"list", "pname", "params"};
static const trace::FunctionSig _glGetListParameterivSGIX_sig = {2982, "glGetListParameterivSGIX", 3, _glGetListParameterivSGIX_args};

static const char * _glListParameterfSGIX_args[3] = {"list", "pname", "param"};
static const trace::FunctionSig _glListParameterfSGIX_sig = {2983, "glListParameterfSGIX", 3, _glListParameterfSGIX_args};

static const char * _glListParameterfvSGIX_args[3] = {"list", "pname", "params"};
static const trace::FunctionSig _glListParameterfvSGIX_sig = {2984, "glListParameterfvSGIX", 3, _glListParameterfvSGIX_args};

static const char * _glListParameteriSGIX_args[3] = {"list", "pname", "param"};
static const trace::FunctionSig _glListParameteriSGIX_sig = {2985, "glListParameteriSGIX", 3, _glListParameteriSGIX_args};

static const char * _glListParameterivSGIX_args[3] = {"list", "pname", "params"};
static const trace::FunctionSig _glListParameterivSGIX_sig = {2986, "glListParameterivSGIX", 3, _glListParameterivSGIX_args};

static const char * _glPixelTexGenSGIX_args[1] = {"mode"};
static const trace::FunctionSig _glPixelTexGenSGIX_sig = {2987, "glPixelTexGenSGIX", 1, _glPixelTexGenSGIX_args};

static const char * _glDeformationMap3dSGIX_args[14] = {"target", "u1", "u2", "ustride", "uorder", "v1", "v2", "vstride", "vorder", "w1", "w2", "wstride", "worder", "points"};
static const trace::FunctionSig _glDeformationMap3dSGIX_sig = {2988, "glDeformationMap3dSGIX", 14, _glDeformationMap3dSGIX_args};

static const char * _glDeformationMap3fSGIX_args[14] = {"target", "u1", "u2", "ustride", "uorder", "v1", "v2", "vstride", "vorder", "w1", "w2", "wstride", "worder", "points"};
static const trace::FunctionSig _glDeformationMap3fSGIX_sig = {2989, "glDeformationMap3fSGIX", 14, _glDeformationMap3fSGIX_args};

static const char * _glDeformSGIX_args[1] = {"mask"};
static const trace::FunctionSig _glDeformSGIX_sig = {2990, "glDeformSGIX", 1, _glDeformSGIX_args};

static const char * _glLoadIdentityDeformationMapSGIX_args[1] = {"mask"};
static const trace::FunctionSig _glLoadIdentityDeformationMapSGIX_sig = {2991, "glLoadIdentityDeformationMapSGIX", 1, _glLoadIdentityDeformationMapSGIX_args};

static const char * _glReferencePlaneSGIX_args[1] = {"equation"};
static const trace::FunctionSig _glReferencePlaneSGIX_sig = {2992, "glReferencePlaneSGIX", 1, _glReferencePlaneSGIX_args};

static const char * _glSpriteParameterfSGIX_args[2] = {"pname", "param"};
static const trace::FunctionSig _glSpriteParameterfSGIX_sig = {2993, "glSpriteParameterfSGIX", 2, _glSpriteParameterfSGIX_args};

static const char * _glSpriteParameterfvSGIX_args[2] = {"pname", "params"};
static const trace::FunctionSig _glSpriteParameterfvSGIX_sig = {2994, "glSpriteParameterfvSGIX", 2, _glSpriteParameterfvSGIX_args};

static const char * _glSpriteParameteriSGIX_args[2] = {"pname", "param"};
static const trace::FunctionSig _glSpriteParameteriSGIX_sig = {2995, "glSpriteParameteriSGIX", 2, _glSpriteParameteriSGIX_args};

static const char * _glSpriteParameterivSGIX_args[2] = {"pname", "params"};
static const trace::FunctionSig _glSpriteParameterivSGIX_sig = {2996, "glSpriteParameterivSGIX", 2, _glSpriteParameterivSGIX_args};

static const char ** _glTagSampleBufferSGIX_args = NULL;
static const trace::FunctionSig _glTagSampleBufferSGIX_sig = {2997, "glTagSampleBufferSGIX", 0, _glTagSampleBufferSGIX_args};

static const char * _glColorTableSGI_args[6] = {"target", "internalformat", "width", "format", "type", "table"};
static const trace::FunctionSig _glColorTableSGI_sig = {2998, "glColorTableSGI", 6, _glColorTableSGI_args};

static const char * _glColorTableParameterfvSGI_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glColorTableParameterfvSGI_sig = {2999, "glColorTableParameterfvSGI", 3, _glColorTableParameterfvSGI_args};

static const char * _glColorTableParameterivSGI_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glColorTableParameterivSGI_sig = {3000, "glColorTableParameterivSGI", 3, _glColorTableParameterivSGI_args};

static const char * _glCopyColorTableSGI_args[5] = {"target", "internalformat", "x", "y", "width"};
static const trace::FunctionSig _glCopyColorTableSGI_sig = {3001, "glCopyColorTableSGI", 5, _glCopyColorTableSGI_args};

static const char * _glGetColorTableSGI_args[4] = {"target", "format", "type", "table"};
static const trace::FunctionSig _glGetColorTableSGI_sig = {3002, "glGetColorTableSGI", 4, _glGetColorTableSGI_args};

static const char * _glGetColorTableParameterfvSGI_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetColorTableParameterfvSGI_sig = {3003, "glGetColorTableParameterfvSGI", 3, _glGetColorTableParameterfvSGI_args};

static const char * _glGetColorTableParameterivSGI_args[3] = {"target", "pname", "params"};
static const trace::FunctionSig _glGetColorTableParameterivSGI_sig = {3004, "glGetColorTableParameterivSGI", 3, _glGetColorTableParameterivSGI_args};

static const char ** _glFinishTextureSUNX_args = NULL;
static const trace::FunctionSig _glFinishTextureSUNX_sig = {3005, "glFinishTextureSUNX", 0, _glFinishTextureSUNX_args};

static const char * _glGlobalAlphaFactorbSUN_args[1] = {"factor"};
static const trace::FunctionSig _glGlobalAlphaFactorbSUN_sig = {3006, "glGlobalAlphaFactorbSUN", 1, _glGlobalAlphaFactorbSUN_args};

static const char * _glGlobalAlphaFactorsSUN_args[1] = {"factor"};
static const trace::FunctionSig _glGlobalAlphaFactorsSUN_sig = {3007, "glGlobalAlphaFactorsSUN", 1, _glGlobalAlphaFactorsSUN_args};

static const char * _glGlobalAlphaFactoriSUN_args[1] = {"factor"};
static const trace::FunctionSig _glGlobalAlphaFactoriSUN_sig = {3008, "glGlobalAlphaFactoriSUN", 1, _glGlobalAlphaFactoriSUN_args};

static const char * _glGlobalAlphaFactorfSUN_args[1] = {"factor"};
static const trace::FunctionSig _glGlobalAlphaFactorfSUN_sig = {3009, "glGlobalAlphaFactorfSUN", 1, _glGlobalAlphaFactorfSUN_args};

static const char * _glGlobalAlphaFactordSUN_args[1] = {"factor"};
static const trace::FunctionSig _glGlobalAlphaFactordSUN_sig = {3010, "glGlobalAlphaFactordSUN", 1, _glGlobalAlphaFactordSUN_args};

static const char * _glGlobalAlphaFactorubSUN_args[1] = {"factor"};
static const trace::FunctionSig _glGlobalAlphaFactorubSUN_sig = {3011, "glGlobalAlphaFactorubSUN", 1, _glGlobalAlphaFactorubSUN_args};

static const char * _glGlobalAlphaFactorusSUN_args[1] = {"factor"};
static const trace::FunctionSig _glGlobalAlphaFactorusSUN_sig = {3012, "glGlobalAlphaFactorusSUN", 1, _glGlobalAlphaFactorusSUN_args};

static const char * _glGlobalAlphaFactoruiSUN_args[1] = {"factor"};
static const trace::FunctionSig _glGlobalAlphaFactoruiSUN_sig = {3013, "glGlobalAlphaFactoruiSUN", 1, _glGlobalAlphaFactoruiSUN_args};

static const char * _glDrawMeshArraysSUN_args[4] = {"mode", "first", "count", "width"};
static const trace::FunctionSig _glDrawMeshArraysSUN_sig = {3014, "glDrawMeshArraysSUN", 4, _glDrawMeshArraysSUN_args};

static const char * _glReplacementCodeuiSUN_args[1] = {"code"};
static const trace::FunctionSig _glReplacementCodeuiSUN_sig = {3015, "glReplacementCodeuiSUN", 1, _glReplacementCodeuiSUN_args};

static const char * _glReplacementCodeusSUN_args[1] = {"code"};
static const trace::FunctionSig _glReplacementCodeusSUN_sig = {3016, "glReplacementCodeusSUN", 1, _glReplacementCodeusSUN_args};

static const char * _glReplacementCodeubSUN_args[1] = {"code"};
static const trace::FunctionSig _glReplacementCodeubSUN_sig = {3017, "glReplacementCodeubSUN", 1, _glReplacementCodeubSUN_args};

static const char * _glReplacementCodeuivSUN_args[1] = {"code"};
static const trace::FunctionSig _glReplacementCodeuivSUN_sig = {3018, "glReplacementCodeuivSUN", 1, _glReplacementCodeuivSUN_args};

static const char * _glReplacementCodeusvSUN_args[1] = {"code"};
static const trace::FunctionSig _glReplacementCodeusvSUN_sig = {3019, "glReplacementCodeusvSUN", 1, _glReplacementCodeusvSUN_args};

static const char * _glReplacementCodeubvSUN_args[1] = {"code"};
static const trace::FunctionSig _glReplacementCodeubvSUN_sig = {3020, "glReplacementCodeubvSUN", 1, _glReplacementCodeubvSUN_args};

static const char * _glReplacementCodePointerSUN_args[3] = {"type", "stride", "pointer"};
static const trace::FunctionSig _glReplacementCodePointerSUN_sig = {3021, "glReplacementCodePointerSUN", 3, _glReplacementCodePointerSUN_args};

static const char * _glColor4ubVertex2fSUN_args[6] = {"r", "g", "b", "a", "x", "y"};
static const trace::FunctionSig _glColor4ubVertex2fSUN_sig = {3022, "glColor4ubVertex2fSUN", 6, _glColor4ubVertex2fSUN_args};

static const char * _glColor4ubVertex2fvSUN_args[2] = {"c", "v"};
static const trace::FunctionSig _glColor4ubVertex2fvSUN_sig = {3023, "glColor4ubVertex2fvSUN", 2, _glColor4ubVertex2fvSUN_args};

static const char * _glColor4ubVertex3fSUN_args[7] = {"r", "g", "b", "a", "x", "y", "z"};
static const trace::FunctionSig _glColor4ubVertex3fSUN_sig = {3024, "glColor4ubVertex3fSUN", 7, _glColor4ubVertex3fSUN_args};

static const char * _glColor4ubVertex3fvSUN_args[2] = {"c", "v"};
static const trace::FunctionSig _glColor4ubVertex3fvSUN_sig = {3025, "glColor4ubVertex3fvSUN", 2, _glColor4ubVertex3fvSUN_args};

static const char * _glColor3fVertex3fSUN_args[6] = {"r", "g", "b", "x", "y", "z"};
static const trace::FunctionSig _glColor3fVertex3fSUN_sig = {3026, "glColor3fVertex3fSUN", 6, _glColor3fVertex3fSUN_args};

static const char * _glColor3fVertex3fvSUN_args[2] = {"c", "v"};
static const trace::FunctionSig _glColor3fVertex3fvSUN_sig = {3027, "glColor3fVertex3fvSUN", 2, _glColor3fVertex3fvSUN_args};

static const char * _glNormal3fVertex3fSUN_args[6] = {"nx", "ny", "nz", "x", "y", "z"};
static const trace::FunctionSig _glNormal3fVertex3fSUN_sig = {3028, "glNormal3fVertex3fSUN", 6, _glNormal3fVertex3fSUN_args};

static const char * _glNormal3fVertex3fvSUN_args[2] = {"n", "v"};
static const trace::FunctionSig _glNormal3fVertex3fvSUN_sig = {3029, "glNormal3fVertex3fvSUN", 2, _glNormal3fVertex3fvSUN_args};

static const char * _glColor4fNormal3fVertex3fSUN_args[10] = {"r", "g", "b", "a", "nx", "ny", "nz", "x", "y", "z"};
static const trace::FunctionSig _glColor4fNormal3fVertex3fSUN_sig = {3030, "glColor4fNormal3fVertex3fSUN", 10, _glColor4fNormal3fVertex3fSUN_args};

static const char * _glColor4fNormal3fVertex3fvSUN_args[3] = {"c", "n", "v"};
static const trace::FunctionSig _glColor4fNormal3fVertex3fvSUN_sig = {3031, "glColor4fNormal3fVertex3fvSUN", 3, _glColor4fNormal3fVertex3fvSUN_args};

static const char * _glTexCoord2fVertex3fSUN_args[5] = {"s", "t", "x", "y", "z"};
static const trace::FunctionSig _glTexCoord2fVertex3fSUN_sig = {3032, "glTexCoord2fVertex3fSUN", 5, _glTexCoord2fVertex3fSUN_args};

static const char * _glTexCoord2fVertex3fvSUN_args[2] = {"tc", "v"};
static const trace::FunctionSig _glTexCoord2fVertex3fvSUN_sig = {3033, "glTexCoord2fVertex3fvSUN", 2, _glTexCoord2fVertex3fvSUN_args};

static const char * _glTexCoord4fVertex4fSUN_args[8] = {"s", "t", "p", "q", "x", "y", "z", "w"};
static const trace::FunctionSig _glTexCoord4fVertex4fSUN_sig = {3034, "glTexCoord4fVertex4fSUN", 8, _glTexCoord4fVertex4fSUN_args};

static const char * _glTexCoord4fVertex4fvSUN_args[2] = {"tc", "v"};
static const trace::FunctionSig _glTexCoord4fVertex4fvSUN_sig = {3035, "glTexCoord4fVertex4fvSUN", 2, _glTexCoord4fVertex4fvSUN_args};

static const char * _glTexCoord2fColor4ubVertex3fSUN_args[9] = {"s", "t", "r", "g", "b", "a", "x", "y", "z"};
static const trace::FunctionSig _glTexCoord2fColor4ubVertex3fSUN_sig = {3036, "glTexCoord2fColor4ubVertex3fSUN", 9, _glTexCoord2fColor4ubVertex3fSUN_args};

static const char * _glTexCoord2fColor4ubVertex3fvSUN_args[3] = {"tc", "c", "v"};
static const trace::FunctionSig _glTexCoord2fColor4ubVertex3fvSUN_sig = {3037, "glTexCoord2fColor4ubVertex3fvSUN", 3, _glTexCoord2fColor4ubVertex3fvSUN_args};

static const char * _glTexCoord2fColor3fVertex3fSUN_args[8] = {"s", "t", "r", "g", "b", "x", "y", "z"};
static const trace::FunctionSig _glTexCoord2fColor3fVertex3fSUN_sig = {3038, "glTexCoord2fColor3fVertex3fSUN", 8, _glTexCoord2fColor3fVertex3fSUN_args};

static const char * _glTexCoord2fColor3fVertex3fvSUN_args[3] = {"tc", "c", "v"};
static const trace::FunctionSig _glTexCoord2fColor3fVertex3fvSUN_sig = {3039, "glTexCoord2fColor3fVertex3fvSUN", 3, _glTexCoord2fColor3fVertex3fvSUN_args};

static const char * _glTexCoord2fNormal3fVertex3fSUN_args[8] = {"s", "t", "nx", "ny", "nz", "x", "y", "z"};
static const trace::FunctionSig _glTexCoord2fNormal3fVertex3fSUN_sig = {3040, "glTexCoord2fNormal3fVertex3fSUN", 8, _glTexCoord2fNormal3fVertex3fSUN_args};

static const char * _glTexCoord2fNormal3fVertex3fvSUN_args[3] = {"tc", "n", "v"};
static const trace::FunctionSig _glTexCoord2fNormal3fVertex3fvSUN_sig = {3041, "glTexCoord2fNormal3fVertex3fvSUN", 3, _glTexCoord2fNormal3fVertex3fvSUN_args};

static const char * _glTexCoord2fColor4fNormal3fVertex3fSUN_args[12] = {"s", "t", "r", "g", "b", "a", "nx", "ny", "nz", "x", "y", "z"};
static const trace::FunctionSig _glTexCoord2fColor4fNormal3fVertex3fSUN_sig = {3042, "glTexCoord2fColor4fNormal3fVertex3fSUN", 12, _glTexCoord2fColor4fNormal3fVertex3fSUN_args};

static const char * _glTexCoord2fColor4fNormal3fVertex3fvSUN_args[4] = {"tc", "c", "n", "v"};
static const trace::FunctionSig _glTexCoord2fColor4fNormal3fVertex3fvSUN_sig = {3043, "glTexCoord2fColor4fNormal3fVertex3fvSUN", 4, _glTexCoord2fColor4fNormal3fVertex3fvSUN_args};

static const char * _glTexCoord4fColor4fNormal3fVertex4fSUN_args[15] = {"s", "t", "p", "q", "r", "g", "b", "a", "nx", "ny", "nz", "x", "y", "z", "w"};
static const trace::FunctionSig _glTexCoord4fColor4fNormal3fVertex4fSUN_sig = {3044, "glTexCoord4fColor4fNormal3fVertex4fSUN", 15, _glTexCoord4fColor4fNormal3fVertex4fSUN_args};

static const char * _glTexCoord4fColor4fNormal3fVertex4fvSUN_args[4] = {"tc", "c", "n", "v"};
static const trace::FunctionSig _glTexCoord4fColor4fNormal3fVertex4fvSUN_sig = {3045, "glTexCoord4fColor4fNormal3fVertex4fvSUN", 4, _glTexCoord4fColor4fNormal3fVertex4fvSUN_args};

static const char * _glReplacementCodeuiVertex3fSUN_args[4] = {"rc", "x", "y", "z"};
static const trace::FunctionSig _glReplacementCodeuiVertex3fSUN_sig = {3046, "glReplacementCodeuiVertex3fSUN", 4, _glReplacementCodeuiVertex3fSUN_args};

static const char * _glReplacementCodeuiVertex3fvSUN_args[2] = {"rc", "v"};
static const trace::FunctionSig _glReplacementCodeuiVertex3fvSUN_sig = {3047, "glReplacementCodeuiVertex3fvSUN", 2, _glReplacementCodeuiVertex3fvSUN_args};

static const char * _glReplacementCodeuiColor4ubVertex3fSUN_args[8] = {"rc", "r", "g", "b", "a", "x", "y", "z"};
static const trace::FunctionSig _glReplacementCodeuiColor4ubVertex3fSUN_sig = {3048, "glReplacementCodeuiColor4ubVertex3fSUN", 8, _glReplacementCodeuiColor4ubVertex3fSUN_args};

static const char * _glReplacementCodeuiColor4ubVertex3fvSUN_args[3] = {"rc", "c", "v"};
static const trace::FunctionSig _glReplacementCodeuiColor4ubVertex3fvSUN_sig = {3049, "glReplacementCodeuiColor4ubVertex3fvSUN", 3, _glReplacementCodeuiColor4ubVertex3fvSUN_args};

static const char * _glReplacementCodeuiColor3fVertex3fSUN_args[7] = {"rc", "r", "g", "b", "x", "y", "z"};
static const trace::FunctionSig _glReplacementCodeuiColor3fVertex3fSUN_sig = {3050, "glReplacementCodeuiColor3fVertex3fSUN", 7, _glReplacementCodeuiColor3fVertex3fSUN_args};

static const char * _glReplacementCodeuiColor3fVertex3fvSUN_args[3] = {"rc", "c", "v"};
static const trace::FunctionSig _glReplacementCodeuiColor3fVertex3fvSUN_sig = {3051, "glReplacementCodeuiColor3fVertex3fvSUN", 3, _glReplacementCodeuiColor3fVertex3fvSUN_args};

static const char * _glReplacementCodeuiNormal3fVertex3fSUN_args[7] = {"rc", "nx", "ny", "nz", "x", "y", "z"};
static const trace::FunctionSig _glReplacementCodeuiNormal3fVertex3fSUN_sig = {3052, "glReplacementCodeuiNormal3fVertex3fSUN", 7, _glReplacementCodeuiNormal3fVertex3fSUN_args};

static const char * _glReplacementCodeuiNormal3fVertex3fvSUN_args[3] = {"rc", "n", "v"};
static const trace::FunctionSig _glReplacementCodeuiNormal3fVertex3fvSUN_sig = {3053, "glReplacementCodeuiNormal3fVertex3fvSUN", 3, _glReplacementCodeuiNormal3fVertex3fvSUN_args};

static const char * _glReplacementCodeuiColor4fNormal3fVertex3fSUN_args[11] = {"rc", "r", "g", "b", "a", "nx", "ny", "nz", "x", "y", "z"};
static const trace::FunctionSig _glReplacementCodeuiColor4fNormal3fVertex3fSUN_sig = {3054, "glReplacementCodeuiColor4fNormal3fVertex3fSUN", 11, _glReplacementCodeuiColor4fNormal3fVertex3fSUN_args};

static const char * _glReplacementCodeuiColor4fNormal3fVertex3fvSUN_args[4] = {"rc", "c", "n", "v"};
static const trace::FunctionSig _glReplacementCodeuiColor4fNormal3fVertex3fvSUN_sig = {3055, "glReplacementCodeuiColor4fNormal3fVertex3fvSUN", 4, _glReplacementCodeuiColor4fNormal3fVertex3fvSUN_args};

static const char * _glReplacementCodeuiTexCoord2fVertex3fSUN_args[6] = {"rc", "s", "t", "x", "y", "z"};
static const trace::FunctionSig _glReplacementCodeuiTexCoord2fVertex3fSUN_sig = {3056, "glReplacementCodeuiTexCoord2fVertex3fSUN", 6, _glReplacementCodeuiTexCoord2fVertex3fSUN_args};

static const char * _glReplacementCodeuiTexCoord2fVertex3fvSUN_args[3] = {"rc", "tc", "v"};
static const trace::FunctionSig _glReplacementCodeuiTexCoord2fVertex3fvSUN_sig = {3057, "glReplacementCodeuiTexCoord2fVertex3fvSUN", 3, _glReplacementCodeuiTexCoord2fVertex3fvSUN_args};

static const char * _glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN_args[9] = {"rc", "s", "t", "nx", "ny", "nz", "x", "y", "z"};
static const trace::FunctionSig _glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN_sig = {3058, "glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN", 9, _glReplacementCodeuiTexCoord2fNormal3fVertex3fSUN_args};

static const char * _glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN_args[4] = {"rc", "tc", "n", "v"};
static const trace::FunctionSig _glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN_sig = {3059, "glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN", 4, _glReplacementCodeuiTexCoord2fNormal3fVertex3fvSUN_args};

static const char * _glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN_args[13] = {"rc", "s", "t", "r", "g", "b", "a", "nx", "ny", "nz", "x", "y", "z"};
static const trace::FunctionSig _glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN_sig = {3060, "glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN", 13, _glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fSUN_args};

static const char * _glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN_args[5] = {"rc", "tc", "c", "n", "v"};
static const trace::FunctionSig _glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN_sig = {3061, "glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN", 5, _glReplacementCodeuiTexCoord2fColor4fNormal3fVertex3fvSUN_args};

static const char * _glAddSwapHintRectWIN_args[4] = {"x", "y", "width", "height"};
static const trace::FunctionSig _glAddSwapHintRectWIN_sig = {3062, "glAddSwapHintRectWIN", 4, _glAddSwapHintRectWIN_args};

extern "C" PUBLIC
XVisualInfo * glXChooseVisual(Display * dpy, int screen, int * attribList) {
    XVisualInfo * _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXChooseVisual_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeSInt(screen);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    {
    int _cint2;
    for (_cint2 = 0; attribList && attribList[_cint2] != 0; _cint2 += 2) {
        switch (int(attribList[_cint2])) {
        case GLX_USE_GL:
        case GLX_RGBA:
        case GLX_DOUBLEBUFFER:
        case GLX_STEREO:
        case GLX_FRAMEBUFFER_SRGB_CAPABLE_ARB:
            _cint2--;
            break;
        }
    }
    _cint2 += attribList ? 1 : 0;
    trace::localWriter.beginArray(_cint2);
    for (int _iint2 = 0; _iint2 < _cint2; _iint2++) {
        trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint2_sig, attribList[_iint2]);
        trace::localWriter.endElement();
        if (_iint2 + 1 >= _cint2) {
            break;
        }
        switch (int(attribList[_iint2++])) {
        case GLX_BUFFER_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_LEVEL:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_AUX_BUFFERS:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_RED_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_GREEN_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_BLUE_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_ALPHA_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_DEPTH_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_STENCIL_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_ACCUM_RED_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_ACCUM_GREEN_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_ACCUM_BLUE_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_ACCUM_ALPHA_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_CONFIG_CAVEAT:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint7_sig, (attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_X_VISUAL_TYPE:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint8_sig, (attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_TYPE:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint9_sig, (attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_INDEX_VALUE:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_RED_VALUE:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_GREEN_VALUE:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_BLUE_VALUE:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_ALPHA_VALUE:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_SAMPLE_BUFFERS:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_SAMPLES:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iint2]);
            trace::localWriter.endElement();
            break;
        case GLX_USE_GL:
        case GLX_RGBA:
        case GLX_DOUBLEBUFFER:
        case GLX_STEREO:
        case GLX_FRAMEBUFFER_SRGB_CAPABLE_ARB:
            _iint2--;
            break;
        default:
            trace::localWriter.beginElement();
            os::log("apitrace: warning: %s: unknown key 0x%04X, interpreting value as int\n", __FUNCTION__, int(attribList[_iint2 - 1]));
            trace::localWriter.writeSInt(attribList[_iint2]);
            trace::localWriter.endElement();
            break;
        }
    }
    trace::localWriter.endArray();
    }
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXChooseVisual(dpy, screen, attribList);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    if (_result) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.beginStruct(&_structXVisualInfo_sig);
    trace::localWriter.writePointer((uintptr_t)(*_result).visual);
    trace::localWriter.writeUInt((*_result).visualid);
    trace::localWriter.writeSInt((*_result).screen);
    trace::localWriter.writeSInt((*_result).depth);
    trace::localWriter.writeSInt((*_result).c_class);
    trace::localWriter.writeUInt((*_result).red_mask);
    trace::localWriter.writeUInt((*_result).green_mask);
    trace::localWriter.writeUInt((*_result).blue_mask);
    trace::localWriter.writeSInt((*_result).colormap_size);
    trace::localWriter.writeSInt((*_result).bits_per_rgb);
    trace::localWriter.endStruct();
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
GLXContext glXCreateContext(Display * dpy, XVisualInfo * vis, GLXContext shareList, int direct) {
    GLXContext _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXCreateContext_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    if (vis) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.beginStruct(&_structXVisualInfo_sig);
    trace::localWriter.writePointer((uintptr_t)(*vis).visual);
    trace::localWriter.writeUInt((*vis).visualid);
    trace::localWriter.writeSInt((*vis).screen);
    trace::localWriter.writeSInt((*vis).depth);
    trace::localWriter.writeSInt((*vis).c_class);
    trace::localWriter.writeUInt((*vis).red_mask);
    trace::localWriter.writeUInt((*vis).green_mask);
    trace::localWriter.writeUInt((*vis).blue_mask);
    trace::localWriter.writeSInt((*vis).colormap_size);
    trace::localWriter.writeSInt((*vis).bits_per_rgb);
    trace::localWriter.endStruct();
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writePointer((uintptr_t)shareList);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(3);
    trace::localWriter.writeEnum(&_enumint1_sig, direct);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXCreateContext(dpy, vis, shareList, direct);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writePointer((uintptr_t)_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    if (_result != NULL)
        gltrace::createContext((uintptr_t)_result, (uintptr_t)shareList);
    return _result;
}

extern "C" PUBLIC
void glXDestroyContext(Display * dpy, GLXContext ctx) {
    gltrace::releaseContext((uintptr_t)ctx);
    unsigned _call = trace::localWriter.beginEnter(&_glXDestroyContext_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writePointer((uintptr_t)ctx);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _glXDestroyContext(dpy, ctx);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
int glXMakeCurrent(Display * dpy, GLXDrawable drawable, GLXContext ctx) {
    int _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXMakeCurrent_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeUInt(drawable);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writePointer((uintptr_t)ctx);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXMakeCurrent(dpy, drawable, ctx);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeEnum(&_enumint1_sig, _result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    if (_result) {
        if (ctx != NULL)
            gltrace::setContext((uintptr_t)ctx);
        else
            gltrace::clearContext();
    }
    return _result;
}

extern "C" PUBLIC
void glXCopyContext(Display * dpy, GLXContext src, GLXContext dst, unsigned long mask) {
    unsigned _call = trace::localWriter.beginEnter(&_glXCopyContext_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writePointer((uintptr_t)src);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writePointer((uintptr_t)dst);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(3);
    trace::localWriter.writeUInt(mask);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _glXCopyContext(dpy, src, dst, mask);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
void glXSwapBuffers(Display * dpy, GLXDrawable drawable) {
    unsigned _call = trace::localWriter.beginEnter(&_glXSwapBuffers_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeUInt(drawable);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _glXSwapBuffers(dpy, drawable);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
GLXPixmap glXCreateGLXPixmap(Display * dpy, XVisualInfo * visual, Pixmap pixmap) {
    GLXPixmap _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXCreateGLXPixmap_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    if (visual) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.beginStruct(&_structXVisualInfo_sig);
    trace::localWriter.writePointer((uintptr_t)(*visual).visual);
    trace::localWriter.writeUInt((*visual).visualid);
    trace::localWriter.writeSInt((*visual).screen);
    trace::localWriter.writeSInt((*visual).depth);
    trace::localWriter.writeSInt((*visual).c_class);
    trace::localWriter.writeUInt((*visual).red_mask);
    trace::localWriter.writeUInt((*visual).green_mask);
    trace::localWriter.writeUInt((*visual).blue_mask);
    trace::localWriter.writeSInt((*visual).colormap_size);
    trace::localWriter.writeSInt((*visual).bits_per_rgb);
    trace::localWriter.endStruct();
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writeUInt(pixmap);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXCreateGLXPixmap(dpy, visual, pixmap);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeUInt(_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
void glXDestroyGLXPixmap(Display * dpy, GLXPixmap pixmap) {
    unsigned _call = trace::localWriter.beginEnter(&_glXDestroyGLXPixmap_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeUInt(pixmap);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _glXDestroyGLXPixmap(dpy, pixmap);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
int glXQueryExtension(Display * dpy, int * errorb, int * event) {
    int _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXQueryExtension_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXQueryExtension(dpy, errorb, event);
    trace::localWriter.beginLeave(_call);
    if (true) {
    trace::localWriter.beginArg(1);
    if (errorb) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeSInt(*errorb);
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    if (event) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeSInt(*event);
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeEnum(&_enumint1_sig, _result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
int glXQueryVersion(Display * dpy, int * maj, int * min) {
    int _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXQueryVersion_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXQueryVersion(dpy, maj, min);
    trace::localWriter.beginLeave(_call);
    if (true) {
    trace::localWriter.beginArg(1);
    if (maj) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeSInt(*maj);
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    if (min) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeSInt(*min);
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeEnum(&_enumint1_sig, _result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
int glXIsDirect(Display * dpy, GLXContext ctx) {
    int _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXIsDirect_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writePointer((uintptr_t)ctx);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXIsDirect(dpy, ctx);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeEnum(&_enumint1_sig, _result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
int glXGetConfig(Display * dpy, XVisualInfo * visual, int attrib, int * value) {
    int _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXGetConfig_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    if (visual) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.beginStruct(&_structXVisualInfo_sig);
    trace::localWriter.writePointer((uintptr_t)(*visual).visual);
    trace::localWriter.writeUInt((*visual).visualid);
    trace::localWriter.writeSInt((*visual).screen);
    trace::localWriter.writeSInt((*visual).depth);
    trace::localWriter.writeSInt((*visual).c_class);
    trace::localWriter.writeUInt((*visual).red_mask);
    trace::localWriter.writeUInt((*visual).green_mask);
    trace::localWriter.writeUInt((*visual).blue_mask);
    trace::localWriter.writeSInt((*visual).colormap_size);
    trace::localWriter.writeSInt((*visual).bits_per_rgb);
    trace::localWriter.endStruct();
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writeEnum(&_enumint2_sig, attrib);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXGetConfig(dpy, visual, attrib, value);
    trace::localWriter.beginLeave(_call);
    if (true) {
    trace::localWriter.beginArg(3);
    if (value) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeSInt(*value);
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeEnum(&_enumint3_sig, _result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
GLXContext glXGetCurrentContext(void) {
    GLXContext _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXGetCurrentContext_sig);
    trace::localWriter.endEnter();
    _result = _glXGetCurrentContext();
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writePointer((uintptr_t)_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
GLXDrawable glXGetCurrentDrawable(void) {
    GLXDrawable _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXGetCurrentDrawable_sig);
    trace::localWriter.endEnter();
    _result = _glXGetCurrentDrawable();
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeUInt(_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
void glXWaitGL(void) {
    unsigned _call = trace::localWriter.beginEnter(&_glXWaitGL_sig);
    trace::localWriter.endEnter();
    _glXWaitGL();
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
void glXWaitX(void) {
    unsigned _call = trace::localWriter.beginEnter(&_glXWaitX_sig);
    trace::localWriter.endEnter();
    _glXWaitX();
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
void glXUseXFont(Font font, int first, int count, int list) {
    unsigned _call = trace::localWriter.beginEnter(&_glXUseXFont_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writeUInt(font);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeSInt(first);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writeSInt(count);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(3);
    trace::localWriter.writeSInt(list);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _glXUseXFont(font, first, count, list);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
const char * glXQueryExtensionsString(Display * dpy, int screen) {
    const char * _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXQueryExtensionsString_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeSInt(screen);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXQueryExtensionsString(dpy, screen);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeString(_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
const char * glXQueryServerString(Display * dpy, int screen, int name) {
    const char * _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXQueryServerString_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeSInt(screen);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writeEnum(&_enumint4_sig, name);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXQueryServerString(dpy, screen, name);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeString(_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
const char * glXGetClientString(Display * dpy, int name) {
    const char * _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXGetClientString_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeEnum(&_enumint4_sig, name);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXGetClientString(dpy, name);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeString(_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
Display * glXGetCurrentDisplay(void) {
    Display * _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXGetCurrentDisplay_sig);
    trace::localWriter.endEnter();
    _result = _glXGetCurrentDisplay();
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writePointer((uintptr_t)_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
GLXFBConfig * glXChooseFBConfig(Display * dpy, int screen, const int * attribList, int * nitems) {
    GLXFBConfig * _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXChooseFBConfig_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeSInt(screen);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    {
    int _cCint21;
    for (_cCint21 = 0; attribList && attribList[_cCint21] != 0; _cCint21 += 2) {
    }
    _cCint21 += attribList ? 1 : 0;
    trace::localWriter.beginArray(_cCint21);
    for (int _iCint21 = 0; _iCint21 < _cCint21; _iCint21++) {
        trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint2_sig, attribList[_iCint21]);
        trace::localWriter.endElement();
        if (_iCint21 + 1 >= _cCint21) {
            break;
        }
        switch (int(attribList[_iCint21++])) {
        case GLX_BUFFER_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_LEVEL:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_AUX_BUFFERS:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_RED_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_GREEN_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_BLUE_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_ALPHA_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_DEPTH_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_STENCIL_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_ACCUM_RED_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_ACCUM_GREEN_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_ACCUM_BLUE_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_ACCUM_ALPHA_SIZE:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_CONFIG_CAVEAT:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint7_sig, (attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_X_VISUAL_TYPE:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint8_sig, (attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_TYPE:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint9_sig, (attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_INDEX_VALUE:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_RED_VALUE:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_GREEN_VALUE:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_BLUE_VALUE:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_TRANSPARENT_ALPHA_VALUE:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_SAMPLE_BUFFERS:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_SAMPLES:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_DOUBLEBUFFER:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint1_sig, (attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_STEREO:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint1_sig, (attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_RENDER_TYPE:
            trace::localWriter.beginElement();
    trace::localWriter.writeBitmask(&_bitmaskint11_sig, (attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_DRAWABLE_TYPE:
            trace::localWriter.beginElement();
    trace::localWriter.writeBitmask(&_bitmaskint12_sig, (attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_X_RENDERABLE:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint1_sig, (attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_FBCONFIG_ID:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_MAX_PBUFFER_WIDTH:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_MAX_PBUFFER_HEIGHT:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_MAX_PBUFFER_PIXELS:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_VISUAL_ID:
            trace::localWriter.beginElement();
    trace::localWriter.writeUInt((attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        case GLX_FRAMEBUFFER_SRGB_CAPABLE_ARB:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint1_sig, (attribList)[_iCint21]);
            trace::localWriter.endElement();
            break;
        default:
            trace::localWriter.beginElement();
            os::log("apitrace: warning: %s: unknown key 0x%04X, interpreting value as int\n", __FUNCTION__, int(attribList[_iCint21 - 1]));
            trace::localWriter.writeSInt(attribList[_iCint21]);
            trace::localWriter.endElement();
            break;
        }
    }
    trace::localWriter.endArray();
    }
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXChooseFBConfig(dpy, screen, attribList, nitems);
    trace::localWriter.beginLeave(_call);
    if (true) {
    trace::localWriter.beginArg(3);
    if (nitems) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeSInt(*nitems);
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    }
    trace::localWriter.beginReturn();
    if (_result) {
        size_t _cGLXFBConfig = *nitems > 0 ? *nitems : 0;
        trace::localWriter.beginArray(_cGLXFBConfig);
        for (size_t _iGLXFBConfig = 0; _iGLXFBConfig < _cGLXFBConfig; ++_iGLXFBConfig) {
            trace::localWriter.beginElement();
    trace::localWriter.writePointer((uintptr_t)(_result)[_iGLXFBConfig]);
            trace::localWriter.endElement();
        }
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
int glXGetFBConfigAttrib(Display * dpy, GLXFBConfig config, int attribute, int * value) {
    int _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXGetFBConfigAttrib_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writePointer((uintptr_t)config);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writeEnum(&_enumint2_sig, attribute);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXGetFBConfigAttrib(dpy, config, attribute, value);
    trace::localWriter.beginLeave(_call);
    if (true) {
    trace::localWriter.beginArg(3);
    if (value) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeSInt(*value);
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeSInt(_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
GLXFBConfig * glXGetFBConfigs(Display * dpy, int screen, int * nelements) {
    GLXFBConfig * _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXGetFBConfigs_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeSInt(screen);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXGetFBConfigs(dpy, screen, nelements);
    trace::localWriter.beginLeave(_call);
    if (true) {
    trace::localWriter.beginArg(2);
    if (nelements) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeSInt(*nelements);
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    }
    trace::localWriter.beginReturn();
    if (_result) {
        size_t _cGLXFBConfig = *nelements > 0 ? *nelements : 0;
        trace::localWriter.beginArray(_cGLXFBConfig);
        for (size_t _iGLXFBConfig = 0; _iGLXFBConfig < _cGLXFBConfig; ++_iGLXFBConfig) {
            trace::localWriter.beginElement();
    trace::localWriter.writePointer((uintptr_t)(_result)[_iGLXFBConfig]);
            trace::localWriter.endElement();
        }
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
XVisualInfo * glXGetVisualFromFBConfig(Display * dpy, GLXFBConfig config) {
    XVisualInfo * _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXGetVisualFromFBConfig_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writePointer((uintptr_t)config);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXGetVisualFromFBConfig(dpy, config);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    if (_result) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.beginStruct(&_structXVisualInfo_sig);
    trace::localWriter.writePointer((uintptr_t)(*_result).visual);
    trace::localWriter.writeUInt((*_result).visualid);
    trace::localWriter.writeSInt((*_result).screen);
    trace::localWriter.writeSInt((*_result).depth);
    trace::localWriter.writeSInt((*_result).c_class);
    trace::localWriter.writeUInt((*_result).red_mask);
    trace::localWriter.writeUInt((*_result).green_mask);
    trace::localWriter.writeUInt((*_result).blue_mask);
    trace::localWriter.writeSInt((*_result).colormap_size);
    trace::localWriter.writeSInt((*_result).bits_per_rgb);
    trace::localWriter.endStruct();
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
GLXWindow glXCreateWindow(Display * dpy, GLXFBConfig config, Window win, const int * attribList) {
    GLXWindow _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXCreateWindow_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writePointer((uintptr_t)config);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writeUInt(win);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(3);
    {
    int _cCint2;
    for (_cCint2 = 0; attribList && attribList[_cCint2] != 0; _cCint2 += 2) {
    }
    _cCint2 += attribList ? 1 : 0;
    trace::localWriter.beginArray(_cCint2);
    for (int _iCint2 = 0; _iCint2 < _cCint2; _iCint2++) {
        trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint2_sig, attribList[_iCint2]);
        trace::localWriter.endElement();
        if (_iCint2 + 1 >= _cCint2) {
            break;
        }
        switch (int(attribList[_iCint2++])) {
        default:
            trace::localWriter.beginElement();
            os::log("apitrace: warning: %s: unknown key 0x%04X, interpreting value as int\n", __FUNCTION__, int(attribList[_iCint2 - 1]));
            trace::localWriter.writeSInt(attribList[_iCint2]);
            trace::localWriter.endElement();
            break;
        }
    }
    trace::localWriter.endArray();
    }
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXCreateWindow(dpy, config, win, attribList);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeUInt(_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
void glXDestroyWindow(Display * dpy, GLXWindow window) {
    unsigned _call = trace::localWriter.beginEnter(&_glXDestroyWindow_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeUInt(window);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _glXDestroyWindow(dpy, window);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
GLXPixmap glXCreatePixmap(Display * dpy, GLXFBConfig config, Pixmap pixmap, const int * attribList) {
    GLXPixmap _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXCreatePixmap_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writePointer((uintptr_t)config);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writeUInt(pixmap);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(3);
    {
    int _cCint23;
    for (_cCint23 = 0; attribList && attribList[_cCint23] != 0; _cCint23 += 2) {
    }
    _cCint23 += attribList ? 1 : 0;
    trace::localWriter.beginArray(_cCint23);
    for (int _iCint23 = 0; _iCint23 < _cCint23; _iCint23++) {
        trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint2_sig, attribList[_iCint23]);
        trace::localWriter.endElement();
        if (_iCint23 + 1 >= _cCint23) {
            break;
        }
        switch (int(attribList[_iCint23++])) {
        case GLX_TEXTURE_FORMAT_EXT:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint2_sig, (attribList)[_iCint23]);
            trace::localWriter.endElement();
            break;
        case GLX_TEXTURE_TARGET_EXT:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint2_sig, (attribList)[_iCint23]);
            trace::localWriter.endElement();
            break;
        case GLX_MIPMAP_TEXTURE_EXT:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint1_sig, (attribList)[_iCint23]);
            trace::localWriter.endElement();
            break;
        default:
            trace::localWriter.beginElement();
            os::log("apitrace: warning: %s: unknown key 0x%04X, interpreting value as int\n", __FUNCTION__, int(attribList[_iCint23 - 1]));
            trace::localWriter.writeSInt(attribList[_iCint23]);
            trace::localWriter.endElement();
            break;
        }
    }
    trace::localWriter.endArray();
    }
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXCreatePixmap(dpy, config, pixmap, attribList);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeUInt(_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
void glXDestroyPixmap(Display * dpy, GLXPixmap pixmap) {
    unsigned _call = trace::localWriter.beginEnter(&_glXDestroyPixmap_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeUInt(pixmap);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _glXDestroyPixmap(dpy, pixmap);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
GLXPbuffer glXCreatePbuffer(Display * dpy, GLXFBConfig config, const int * attribList) {
    GLXPbuffer _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXCreatePbuffer_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writePointer((uintptr_t)config);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    {
    int _cCint24;
    for (_cCint24 = 0; attribList && attribList[_cCint24] != 0; _cCint24 += 2) {
    }
    _cCint24 += attribList ? 1 : 0;
    trace::localWriter.beginArray(_cCint24);
    for (int _iCint24 = 0; _iCint24 < _cCint24; _iCint24++) {
        trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint2_sig, attribList[_iCint24]);
        trace::localWriter.endElement();
        if (_iCint24 + 1 >= _cCint24) {
            break;
        }
        switch (int(attribList[_iCint24++])) {
        case GLX_PBUFFER_WIDTH:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint24]);
            trace::localWriter.endElement();
            break;
        case GLX_PBUFFER_HEIGHT:
            trace::localWriter.beginElement();
    trace::localWriter.writeSInt((attribList)[_iCint24]);
            trace::localWriter.endElement();
            break;
        case GLX_LARGEST_PBUFFER:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint1_sig, (attribList)[_iCint24]);
            trace::localWriter.endElement();
            break;
        case GLX_PRESERVED_CONTENTS:
            trace::localWriter.beginElement();
    trace::localWriter.writeEnum(&_enumint1_sig, (attribList)[_iCint24]);
            trace::localWriter.endElement();
            break;
        default:
            trace::localWriter.beginElement();
            os::log("apitrace: warning: %s: unknown key 0x%04X, interpreting value as int\n", __FUNCTION__, int(attribList[_iCint24 - 1]));
            trace::localWriter.writeSInt(attribList[_iCint24]);
            trace::localWriter.endElement();
            break;
        }
    }
    trace::localWriter.endArray();
    }
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXCreatePbuffer(dpy, config, attribList);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writeUInt(_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    return _result;
}

extern "C" PUBLIC
void glXDestroyPbuffer(Display * dpy, GLXPbuffer pbuf) {
    unsigned _call = trace::localWriter.beginEnter(&_glXDestroyPbuffer_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeUInt(pbuf);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _glXDestroyPbuffer(dpy, pbuf);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
void glXQueryDrawable(Display * dpy, GLXDrawable draw, int attribute, unsigned int * value) {
    unsigned _call = trace::localWriter.beginEnter(&_glXQueryDrawable_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writeUInt(draw);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writeEnum(&_enumint2_sig, attribute);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _glXQueryDrawable(dpy, draw, attribute, value);
    trace::localWriter.beginLeave(_call);
    if (true) {
    trace::localWriter.beginArg(3);
    if (value) {
        trace::localWriter.beginArray(1);
        trace::localWriter.beginElement();
    trace::localWriter.writeUInt(*value);
        trace::localWriter.endElement();
        trace::localWriter.endArray();
    } else {
        trace::localWriter.writeNull();
    }
    trace::localWriter.endArg();
    }
    trace::localWriter.endLeave();
}

extern "C" PUBLIC
GLXContext glXCreateNewContext(Display * dpy, GLXFBConfig config, int renderType, GLXContext shareList, int direct) {
    GLXContext _result;
    unsigned _call = trace::localWriter.beginEnter(&_glXCreateNewContext_sig);
    trace::localWriter.beginArg(0);
    trace::localWriter.writePointer((uintptr_t)dpy);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(1);
    trace::localWriter.writePointer((uintptr_t)config);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(2);
    trace::localWriter.writeEnum(&_enumint2_sig, renderType);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(3);
    trace::localWriter.writePointer((uintptr_t)shareList);
    trace::localWriter.endArg();
    trace::localWriter.beginArg(4);
    trace::localWriter.writeEnum(&_enumint1_sig, direct);
    trace::localWriter.endArg();
    trace::localWriter.endEnter();
    _result = _glXCreateNewContext(dpy, config, renderType, shareList, direct);
    trace::localWriter.beginLeave(_call);
    if (true) {
    }
    trace::localWriter.beginReturn();
    trace::localWriter.writePointer((uintptr_t)_result);
    trace::localWriter.endReturn();
    trace::localWriter.endLeave();
    if (_result != NULL)
        gltrace::createContext((uintptr_t)_result, (u