{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GSubprocess@ allows the creation of and interaction with child
-- processes.
-- 
-- Processes can be communicated with using standard GIO-style APIs (ie:
-- t'GI.Gio.Objects.InputStream.InputStream', t'GI.Gio.Objects.OutputStream.OutputStream'). There are GIO-style APIs
-- to wait for process termination (ie: cancellable and with an asynchronous
-- variant).
-- 
-- There is an API to force a process to terminate, as well as a
-- race-free API for sending UNIX signals to a subprocess.
-- 
-- One major advantage that GIO brings over the core GLib library is
-- comprehensive API for asynchronous I\/O, such
-- 'GI.Gio.Objects.OutputStream.outputStreamSpliceAsync'.  This makes @GSubprocess@
-- significantly more powerful and flexible than equivalent APIs in
-- some other languages such as the @subprocess.py@
-- included with Python.  For example, using @GSubprocess@ one could
-- create two child processes, reading standard output from the first,
-- processing it, and writing to the input stream of the second, all
-- without blocking the main loop.
-- 
-- A powerful 'GI.Gio.Objects.Subprocess.subprocessCommunicate' API is provided similar to the
-- @communicate()@ method of @subprocess.py@. This enables very easy
-- interaction with a subprocess that has been opened with pipes.
-- 
-- @GSubprocess@ defaults to tight control over the file descriptors open
-- in the child process, avoiding dangling-FD issues that are caused by
-- a simple @fork()@\/@exec()@.  The only open file descriptors in the
-- spawned process are ones that were explicitly specified by the
-- @GSubprocess@ API (unless @G_SUBPROCESS_FLAGS_INHERIT_FDS@ was
-- specified).
-- 
-- @GSubprocess@ will quickly reap all child processes as they exit,
-- avoiding ‘zombie processes’ remaining around for long periods of
-- time.  'GI.Gio.Objects.Subprocess.subprocessWait' can be used to wait for this to happen,
-- but it will happen even without the call being explicitly made.
-- 
-- As a matter of principle, @GSubprocess@ has no API that accepts
-- shell-style space-separated strings.  It will, however, match the
-- typical shell behaviour of searching the @PATH@ for executables that do
-- not contain a directory separator in their name. By default, the @PATH@
-- of the current process is used.  You can specify
-- @G_SUBPROCESS_FLAGS_SEARCH_PATH_FROM_ENVP@ to use the @PATH@ of the
-- launcher environment instead.
-- 
-- @GSubprocess@ attempts to have a very simple API for most uses (ie:
-- spawning a subprocess with arguments and support for most typical
-- kinds of input and output redirection).  See 'GI.Gio.Objects.Subprocess.subprocessNew'. The
-- t'GI.Gio.Objects.SubprocessLauncher.SubprocessLauncher' API is provided for more complicated cases
-- (advanced types of redirection, environment variable manipulation,
-- change of working directory, child setup functions, etc).
-- 
-- A typical use of @GSubprocess@ will involve calling
-- 'GI.Gio.Objects.Subprocess.subprocessNew', followed by 'GI.Gio.Objects.Subprocess.subprocessWaitAsync' or
-- 'GI.Gio.Objects.Subprocess.subprocessWait'.  After the process exits, the status can be
-- checked using functions such as 'GI.Gio.Objects.Subprocess.subprocessGetIfExited' (which
-- are similar to the familiar @WIFEXITED@-style POSIX macros).
-- 
-- Note that as of GLib 2.82, creating a @GSubprocess@ causes the signal
-- @SIGPIPE@ to be ignored for the remainder of the program. If you are writing
-- a command-line utility that uses @GSubprocess@, you may need to take into
-- account the fact that your program will not automatically be killed
-- if it tries to write to @stdout@ after it has been closed.
-- 
-- /Since: 2.40/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Objects.Subprocess
    ( 

-- * Exported types
    Subprocess(..)                          ,
    IsSubprocess                            ,
    toSubprocess                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [communicate]("GI.Gio.Objects.Subprocess#g:method:communicate"), [communicateAsync]("GI.Gio.Objects.Subprocess#g:method:communicateAsync"), [communicateFinish]("GI.Gio.Objects.Subprocess#g:method:communicateFinish"), [communicateUtf8]("GI.Gio.Objects.Subprocess#g:method:communicateUtf8"), [communicateUtf8Async]("GI.Gio.Objects.Subprocess#g:method:communicateUtf8Async"), [communicateUtf8Finish]("GI.Gio.Objects.Subprocess#g:method:communicateUtf8Finish"), [forceExit]("GI.Gio.Objects.Subprocess#g:method:forceExit"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [init]("GI.Gio.Interfaces.Initable#g:method:init"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [sendSignal]("GI.Gio.Objects.Subprocess#g:method:sendSignal"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [wait]("GI.Gio.Objects.Subprocess#g:method:wait"), [waitAsync]("GI.Gio.Objects.Subprocess#g:method:waitAsync"), [waitCheck]("GI.Gio.Objects.Subprocess#g:method:waitCheck"), [waitCheckAsync]("GI.Gio.Objects.Subprocess#g:method:waitCheckAsync"), [waitCheckFinish]("GI.Gio.Objects.Subprocess#g:method:waitCheckFinish"), [waitFinish]("GI.Gio.Objects.Subprocess#g:method:waitFinish"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getExitStatus]("GI.Gio.Objects.Subprocess#g:method:getExitStatus"), [getIdentifier]("GI.Gio.Objects.Subprocess#g:method:getIdentifier"), [getIfExited]("GI.Gio.Objects.Subprocess#g:method:getIfExited"), [getIfSignaled]("GI.Gio.Objects.Subprocess#g:method:getIfSignaled"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getStatus]("GI.Gio.Objects.Subprocess#g:method:getStatus"), [getStderrPipe]("GI.Gio.Objects.Subprocess#g:method:getStderrPipe"), [getStdinPipe]("GI.Gio.Objects.Subprocess#g:method:getStdinPipe"), [getStdoutPipe]("GI.Gio.Objects.Subprocess#g:method:getStdoutPipe"), [getSuccessful]("GI.Gio.Objects.Subprocess#g:method:getSuccessful"), [getTermSig]("GI.Gio.Objects.Subprocess#g:method:getTermSig").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveSubprocessMethod                 ,
#endif

-- ** communicate #method:communicate#

#if defined(ENABLE_OVERLOADING)
    SubprocessCommunicateMethodInfo         ,
#endif
    subprocessCommunicate                   ,


-- ** communicateAsync #method:communicateAsync#

#if defined(ENABLE_OVERLOADING)
    SubprocessCommunicateAsyncMethodInfo    ,
#endif
    subprocessCommunicateAsync              ,


-- ** communicateFinish #method:communicateFinish#

#if defined(ENABLE_OVERLOADING)
    SubprocessCommunicateFinishMethodInfo   ,
#endif
    subprocessCommunicateFinish             ,


-- ** communicateUtf8 #method:communicateUtf8#

#if defined(ENABLE_OVERLOADING)
    SubprocessCommunicateUtf8MethodInfo     ,
#endif
    subprocessCommunicateUtf8               ,


-- ** communicateUtf8Async #method:communicateUtf8Async#

#if defined(ENABLE_OVERLOADING)
    SubprocessCommunicateUtf8AsyncMethodInfo,
#endif
    subprocessCommunicateUtf8Async          ,


-- ** communicateUtf8Finish #method:communicateUtf8Finish#

#if defined(ENABLE_OVERLOADING)
    SubprocessCommunicateUtf8FinishMethodInfo,
#endif
    subprocessCommunicateUtf8Finish         ,


-- ** forceExit #method:forceExit#

#if defined(ENABLE_OVERLOADING)
    SubprocessForceExitMethodInfo           ,
#endif
    subprocessForceExit                     ,


-- ** getExitStatus #method:getExitStatus#

#if defined(ENABLE_OVERLOADING)
    SubprocessGetExitStatusMethodInfo       ,
#endif
    subprocessGetExitStatus                 ,


-- ** getIdentifier #method:getIdentifier#

#if defined(ENABLE_OVERLOADING)
    SubprocessGetIdentifierMethodInfo       ,
#endif
    subprocessGetIdentifier                 ,


-- ** getIfExited #method:getIfExited#

#if defined(ENABLE_OVERLOADING)
    SubprocessGetIfExitedMethodInfo         ,
#endif
    subprocessGetIfExited                   ,


-- ** getIfSignaled #method:getIfSignaled#

#if defined(ENABLE_OVERLOADING)
    SubprocessGetIfSignaledMethodInfo       ,
#endif
    subprocessGetIfSignaled                 ,


-- ** getStatus #method:getStatus#

#if defined(ENABLE_OVERLOADING)
    SubprocessGetStatusMethodInfo           ,
#endif
    subprocessGetStatus                     ,


-- ** getStderrPipe #method:getStderrPipe#

#if defined(ENABLE_OVERLOADING)
    SubprocessGetStderrPipeMethodInfo       ,
#endif
    subprocessGetStderrPipe                 ,


-- ** getStdinPipe #method:getStdinPipe#

#if defined(ENABLE_OVERLOADING)
    SubprocessGetStdinPipeMethodInfo        ,
#endif
    subprocessGetStdinPipe                  ,


-- ** getStdoutPipe #method:getStdoutPipe#

#if defined(ENABLE_OVERLOADING)
    SubprocessGetStdoutPipeMethodInfo       ,
#endif
    subprocessGetStdoutPipe                 ,


-- ** getSuccessful #method:getSuccessful#

#if defined(ENABLE_OVERLOADING)
    SubprocessGetSuccessfulMethodInfo       ,
#endif
    subprocessGetSuccessful                 ,


-- ** getTermSig #method:getTermSig#

#if defined(ENABLE_OVERLOADING)
    SubprocessGetTermSigMethodInfo          ,
#endif
    subprocessGetTermSig                    ,


-- ** new #method:new#

    subprocessNew                           ,


-- ** sendSignal #method:sendSignal#

#if defined(ENABLE_OVERLOADING)
    SubprocessSendSignalMethodInfo          ,
#endif
    subprocessSendSignal                    ,


-- ** wait #method:wait#

#if defined(ENABLE_OVERLOADING)
    SubprocessWaitMethodInfo                ,
#endif
    subprocessWait                          ,


-- ** waitAsync #method:waitAsync#

#if defined(ENABLE_OVERLOADING)
    SubprocessWaitAsyncMethodInfo           ,
#endif
    subprocessWaitAsync                     ,


-- ** waitCheck #method:waitCheck#

#if defined(ENABLE_OVERLOADING)
    SubprocessWaitCheckMethodInfo           ,
#endif
    subprocessWaitCheck                     ,


-- ** waitCheckAsync #method:waitCheckAsync#

#if defined(ENABLE_OVERLOADING)
    SubprocessWaitCheckAsyncMethodInfo      ,
#endif
    subprocessWaitCheckAsync                ,


-- ** waitCheckFinish #method:waitCheckFinish#

#if defined(ENABLE_OVERLOADING)
    SubprocessWaitCheckFinishMethodInfo     ,
#endif
    subprocessWaitCheckFinish               ,


-- ** waitFinish #method:waitFinish#

#if defined(ENABLE_OVERLOADING)
    SubprocessWaitFinishMethodInfo          ,
#endif
    subprocessWaitFinish                    ,




 -- * Properties


-- ** argv #attr:argv#
-- | Argument vector.
-- 
-- /Since: 2.40/

#if defined(ENABLE_OVERLOADING)
    SubprocessArgvPropertyInfo              ,
#endif
    constructSubprocessArgv                 ,
#if defined(ENABLE_OVERLOADING)
    subprocessArgv                          ,
#endif


-- ** flags #attr:flags#
-- | Subprocess flags.
-- 
-- /Since: 2.40/

#if defined(ENABLE_OVERLOADING)
    SubprocessFlagsPropertyInfo             ,
#endif
    constructSubprocessFlags                ,
#if defined(ENABLE_OVERLOADING)
    subprocessFlags                         ,
#endif




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.Parameter as GObject.Parameter
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream
import {-# SOURCE #-} qualified GI.Gio.Structs.OutputVector as Gio.OutputVector

#else
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Flags as Gio.Flags
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Interfaces.Initable as Gio.Initable
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable
import {-# SOURCE #-} qualified GI.Gio.Objects.InputStream as Gio.InputStream
import {-# SOURCE #-} qualified GI.Gio.Objects.OutputStream as Gio.OutputStream

#endif

-- | Memory-managed wrapper type.
newtype Subprocess = Subprocess (SP.ManagedPtr Subprocess)
    deriving (Eq)

instance SP.ManagedPtrNewtype Subprocess where
    toManagedPtr (Subprocess p) = p

foreign import ccall "g_subprocess_get_type"
    c_g_subprocess_get_type :: IO B.Types.GType

instance B.Types.TypedObject Subprocess where
    glibType = c_g_subprocess_get_type

instance B.Types.GObject Subprocess

-- | Type class for types which can be safely cast to t'Subprocess', for instance with `toSubprocess`.
class (SP.GObject o, O.IsDescendantOf Subprocess o) => IsSubprocess o
instance (SP.GObject o, O.IsDescendantOf Subprocess o) => IsSubprocess o

instance O.HasParentTypes Subprocess
type instance O.ParentTypes Subprocess = '[GObject.Object.Object, Gio.Initable.Initable]

-- | Cast to t'Subprocess', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toSubprocess :: (MIO.MonadIO m, IsSubprocess o) => o -> m Subprocess
toSubprocess = MIO.liftIO . B.ManagedPtr.unsafeCastTo Subprocess

-- | Convert t'Subprocess' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Subprocess) where
    gvalueGType_ = c_g_subprocess_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Subprocess)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Subprocess)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Subprocess ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveSubprocessMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveSubprocessMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveSubprocessMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveSubprocessMethod "communicate" o = SubprocessCommunicateMethodInfo
    ResolveSubprocessMethod "communicateAsync" o = SubprocessCommunicateAsyncMethodInfo
    ResolveSubprocessMethod "communicateFinish" o = SubprocessCommunicateFinishMethodInfo
    ResolveSubprocessMethod "communicateUtf8" o = SubprocessCommunicateUtf8MethodInfo
    ResolveSubprocessMethod "communicateUtf8Async" o = SubprocessCommunicateUtf8AsyncMethodInfo
    ResolveSubprocessMethod "communicateUtf8Finish" o = SubprocessCommunicateUtf8FinishMethodInfo
    ResolveSubprocessMethod "forceExit" o = SubprocessForceExitMethodInfo
    ResolveSubprocessMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveSubprocessMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveSubprocessMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveSubprocessMethod "init" o = Gio.Initable.InitableInitMethodInfo
    ResolveSubprocessMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveSubprocessMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveSubprocessMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveSubprocessMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveSubprocessMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveSubprocessMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveSubprocessMethod "sendSignal" o = SubprocessSendSignalMethodInfo
    ResolveSubprocessMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveSubprocessMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveSubprocessMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveSubprocessMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveSubprocessMethod "wait" o = SubprocessWaitMethodInfo
    ResolveSubprocessMethod "waitAsync" o = SubprocessWaitAsyncMethodInfo
    ResolveSubprocessMethod "waitCheck" o = SubprocessWaitCheckMethodInfo
    ResolveSubprocessMethod "waitCheckAsync" o = SubprocessWaitCheckAsyncMethodInfo
    ResolveSubprocessMethod "waitCheckFinish" o = SubprocessWaitCheckFinishMethodInfo
    ResolveSubprocessMethod "waitFinish" o = SubprocessWaitFinishMethodInfo
    ResolveSubprocessMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveSubprocessMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveSubprocessMethod "getExitStatus" o = SubprocessGetExitStatusMethodInfo
    ResolveSubprocessMethod "getIdentifier" o = SubprocessGetIdentifierMethodInfo
    ResolveSubprocessMethod "getIfExited" o = SubprocessGetIfExitedMethodInfo
    ResolveSubprocessMethod "getIfSignaled" o = SubprocessGetIfSignaledMethodInfo
    ResolveSubprocessMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveSubprocessMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveSubprocessMethod "getStatus" o = SubprocessGetStatusMethodInfo
    ResolveSubprocessMethod "getStderrPipe" o = SubprocessGetStderrPipeMethodInfo
    ResolveSubprocessMethod "getStdinPipe" o = SubprocessGetStdinPipeMethodInfo
    ResolveSubprocessMethod "getStdoutPipe" o = SubprocessGetStdoutPipeMethodInfo
    ResolveSubprocessMethod "getSuccessful" o = SubprocessGetSuccessfulMethodInfo
    ResolveSubprocessMethod "getTermSig" o = SubprocessGetTermSigMethodInfo
    ResolveSubprocessMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveSubprocessMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveSubprocessMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveSubprocessMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveSubprocessMethod t Subprocess, O.OverloadedMethod info Subprocess p) => OL.IsLabel t (Subprocess -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveSubprocessMethod t Subprocess, O.OverloadedMethod info Subprocess p, R.HasField t Subprocess p) => R.HasField t Subprocess p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveSubprocessMethod t Subprocess, O.OverloadedMethodInfo info Subprocess) => OL.IsLabel t (O.MethodProxy info Subprocess) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- VVV Prop "argv"
   -- Type: TCArray True (-1) (-1) (TBasicType TUTF8)
   -- Flags: [PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Construct a t'GValueConstruct' with valid value for the “@argv@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSubprocessArgv :: (IsSubprocess o, MIO.MonadIO m) => [T.Text] -> m (GValueConstruct o)
constructSubprocessArgv val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyStringArray "argv" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data SubprocessArgvPropertyInfo
instance AttrInfo SubprocessArgvPropertyInfo where
    type AttrAllowedOps SubprocessArgvPropertyInfo = '[ 'AttrConstruct, 'AttrClear]
    type AttrBaseTypeConstraint SubprocessArgvPropertyInfo = IsSubprocess
    type AttrSetTypeConstraint SubprocessArgvPropertyInfo = (~) [T.Text]
    type AttrTransferTypeConstraint SubprocessArgvPropertyInfo = (~) [T.Text]
    type AttrTransferType SubprocessArgvPropertyInfo = [T.Text]
    type AttrGetType SubprocessArgvPropertyInfo = ()
    type AttrLabel SubprocessArgvPropertyInfo = "argv"
    type AttrOrigin SubprocessArgvPropertyInfo = Subprocess
    attrGet = undefined
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSubprocessArgv
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.argv"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#g:attr:argv"
        })
#endif

-- VVV Prop "flags"
   -- Type: TInterface (Name {namespace = "Gio", name = "SubprocessFlags"})
   -- Flags: [PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Construct a t'GValueConstruct' with valid value for the “@flags@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructSubprocessFlags :: (IsSubprocess o, MIO.MonadIO m) => [Gio.Flags.SubprocessFlags] -> m (GValueConstruct o)
constructSubprocessFlags val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFlags "flags" val

#if defined(ENABLE_OVERLOADING)
data SubprocessFlagsPropertyInfo
instance AttrInfo SubprocessFlagsPropertyInfo where
    type AttrAllowedOps SubprocessFlagsPropertyInfo = '[ 'AttrConstruct]
    type AttrBaseTypeConstraint SubprocessFlagsPropertyInfo = IsSubprocess
    type AttrSetTypeConstraint SubprocessFlagsPropertyInfo = (~) [Gio.Flags.SubprocessFlags]
    type AttrTransferTypeConstraint SubprocessFlagsPropertyInfo = (~) [Gio.Flags.SubprocessFlags]
    type AttrTransferType SubprocessFlagsPropertyInfo = [Gio.Flags.SubprocessFlags]
    type AttrGetType SubprocessFlagsPropertyInfo = ()
    type AttrLabel SubprocessFlagsPropertyInfo = "flags"
    type AttrOrigin SubprocessFlagsPropertyInfo = Subprocess
    attrGet = undefined
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructSubprocessFlags
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.flags"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#g:attr:flags"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Subprocess
type instance O.AttributeList Subprocess = SubprocessAttributeList
type SubprocessAttributeList = ('[ '("argv", SubprocessArgvPropertyInfo), '("flags", SubprocessFlagsPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
subprocessArgv :: AttrLabelProxy "argv"
subprocessArgv = AttrLabelProxy

subprocessFlags :: AttrLabelProxy "flags"
subprocessFlags = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Subprocess = SubprocessSignalList
type SubprocessSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Subprocess::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "argv"
--           , argType = TCArray True (-1) (-1) (TBasicType TFileName)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "commandline arguments for the subprocess"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "SubprocessFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "flags that define the behaviour of the subprocess"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "Subprocess" })
-- throws : True
-- Skip return : False

foreign import ccall "g_subprocess_newv" g_subprocess_newv :: 
    Ptr CString ->                          -- argv : TCArray True (-1) (-1) (TBasicType TFileName)
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gio", name = "SubprocessFlags"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Subprocess)

-- | Create a new process with the given flags and argument list.
-- 
-- The argument list is expected to be 'P.Nothing'-terminated.
-- 
-- /Since: 2.40/
subprocessNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [[Char]]
    -- ^ /@argv@/: commandline arguments for the subprocess
    -> [Gio.Flags.SubprocessFlags]
    -- ^ /@flags@/: flags that define the behaviour of the subprocess
    -> m Subprocess
    -- ^ __Returns:__ A newly created t'GI.Gio.Objects.Subprocess.Subprocess', or 'P.Nothing' on error (and /@error@/
    --   will be set) /(Can throw 'Data.GI.Base.GError.GError')/
subprocessNew argv flags = liftIO $ do
    argv' <- packZeroTerminatedFileNameArray argv
    let flags' = gflagsToWord flags
    onException (do
        result <- propagateGError $ g_subprocess_newv argv' flags'
        checkUnexpectedReturnNULL "subprocessNew" result
        result' <- (wrapObject Subprocess) result
        mapZeroTerminatedCArray freeMem argv'
        freeMem argv'
        return result'
     ) (do
        mapZeroTerminatedCArray freeMem argv'
        freeMem argv'
     )

#if defined(ENABLE_OVERLOADING)
#endif

-- method Subprocess::communicate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stdin_buf"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "data to send to the stdin of the subprocess, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stdout_buf"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data read from the subprocess stdout"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "stderr_buf"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data read from the subprocess stderr"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_subprocess_communicate" g_subprocess_communicate :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Ptr GLib.Bytes.Bytes ->                 -- stdin_buf : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GLib.Bytes.Bytes) ->           -- stdout_buf : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Ptr (Ptr GLib.Bytes.Bytes) ->           -- stderr_buf : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Communicate with the subprocess until it terminates, and all input
-- and output has been completed.
-- 
-- If /@stdinBuf@/ is given, the subprocess must have been created with
-- 'GI.Gio.Flags.SubprocessFlagsStdinPipe'.  The given data is fed to the
-- stdin of the subprocess and the pipe is closed (ie: EOF).
-- 
-- At the same time (as not to cause blocking when dealing with large
-- amounts of data), if 'GI.Gio.Flags.SubprocessFlagsStdoutPipe' or
-- 'GI.Gio.Flags.SubprocessFlagsStderrPipe' were used, reads from those
-- streams.  The data that was read is returned in /@stdout@/ and\/or
-- the /@stderr@/.
-- 
-- If the subprocess was created with 'GI.Gio.Flags.SubprocessFlagsStdoutPipe',
-- /@stdoutBuf@/ will contain the data read from stdout.  Otherwise, for
-- subprocesses not created with 'GI.Gio.Flags.SubprocessFlagsStdoutPipe',
-- /@stdoutBuf@/ will be set to 'P.Nothing'.  Similar provisions apply to
-- /@stderrBuf@/ and 'GI.Gio.Flags.SubprocessFlagsStderrPipe'.
-- 
-- As usual, any output variable may be given as 'P.Nothing' to ignore it.
-- 
-- If you desire the stdout and stderr data to be interleaved, create
-- the subprocess with 'GI.Gio.Flags.SubprocessFlagsStdoutPipe' and
-- 'GI.Gio.Flags.SubprocessFlagsStderrMerge'.  The merged result will be returned
-- in /@stdoutBuf@/ and /@stderrBuf@/ will be set to 'P.Nothing'.
-- 
-- In case of any error (including cancellation), 'P.False' will be
-- returned with /@error@/ set.  Some or all of the stdin data may have
-- been written.  Any stdout or stderr data that has been read will be
-- discarded. None of the out variables (aside from /@error@/) will have
-- been set to anything in particular and should not be inspected.
-- 
-- In the case that 'P.True' is returned, the subprocess has exited and the
-- exit status inspection APIs (eg: 'GI.Gio.Objects.Subprocess.subprocessGetIfExited',
-- 'GI.Gio.Objects.Subprocess.subprocessGetExitStatus') may be used.
-- 
-- You should not attempt to use any of the subprocess pipes after
-- starting this function, since they may be left in strange states,
-- even if the operation was cancelled.  You should especially not
-- attempt to interact with the pipes while the operation is in progress
-- (either from another thread or if using the asynchronous version).
-- 
-- /Since: 2.40/
subprocessCommunicate ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> Maybe (GLib.Bytes.Bytes)
    -- ^ /@stdinBuf@/: data to send to the stdin of the subprocess, or 'P.Nothing'
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable'
    -> m ((Maybe GLib.Bytes.Bytes, Maybe GLib.Bytes.Bytes))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
subprocessCommunicate subprocess stdinBuf cancellable = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    maybeStdinBuf <- case stdinBuf of
        Nothing -> return FP.nullPtr
        Just jStdinBuf -> do
            jStdinBuf' <- unsafeManagedPtrGetPtr jStdinBuf
            return jStdinBuf'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    stdoutBuf <- callocMem :: IO (Ptr (Ptr GLib.Bytes.Bytes))
    stderrBuf <- callocMem :: IO (Ptr (Ptr GLib.Bytes.Bytes))
    onException (do
        _ <- propagateGError $ g_subprocess_communicate subprocess' maybeStdinBuf maybeCancellable stdoutBuf stderrBuf
        stdoutBuf' <- peek stdoutBuf
        maybeStdoutBuf' <- convertIfNonNull stdoutBuf' $ \stdoutBuf'' -> do
            stdoutBuf''' <- (wrapBoxed GLib.Bytes.Bytes) stdoutBuf''
            return stdoutBuf'''
        stderrBuf' <- peek stderrBuf
        maybeStderrBuf' <- convertIfNonNull stderrBuf' $ \stderrBuf'' -> do
            stderrBuf''' <- (wrapBoxed GLib.Bytes.Bytes) stderrBuf''
            return stderrBuf'''
        touchManagedPtr subprocess
        whenJust stdinBuf touchManagedPtr
        whenJust cancellable touchManagedPtr
        freeMem stdoutBuf
        freeMem stderrBuf
        return (maybeStdoutBuf', maybeStderrBuf')
     ) (do
        freeMem stdoutBuf
        freeMem stderrBuf
     )

#if defined(ENABLE_OVERLOADING)
data SubprocessCommunicateMethodInfo
instance (signature ~ (Maybe (GLib.Bytes.Bytes) -> Maybe (b) -> m ((Maybe GLib.Bytes.Bytes, Maybe GLib.Bytes.Bytes))), MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod SubprocessCommunicateMethodInfo a signature where
    overloadedMethod = subprocessCommunicate

instance O.OverloadedMethodInfo SubprocessCommunicateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessCommunicate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessCommunicate"
        })


#endif

-- method Subprocess::communicate_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation { rawDocText = Just "Self" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stdin_buf"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Input data, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Cancellable" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Callback" , sinceVersion = Nothing }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "User data" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_communicate_async" g_subprocess_communicate_async :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Ptr GLib.Bytes.Bytes ->                 -- stdin_buf : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronous version of 'GI.Gio.Objects.Subprocess.subprocessCommunicate'.  Complete
-- invocation with 'GI.Gio.Objects.Subprocess.subprocessCommunicateFinish'.
subprocessCommunicateAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@subprocess@/: Self
    -> Maybe (GLib.Bytes.Bytes)
    -- ^ /@stdinBuf@/: Input data, or 'P.Nothing'
    -> Maybe (b)
    -- ^ /@cancellable@/: Cancellable
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: Callback
    -> m ()
subprocessCommunicateAsync subprocess stdinBuf cancellable callback = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    maybeStdinBuf <- case stdinBuf of
        Nothing -> return FP.nullPtr
        Just jStdinBuf -> do
            jStdinBuf' <- unsafeManagedPtrGetPtr jStdinBuf
            return jStdinBuf'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_subprocess_communicate_async subprocess' maybeStdinBuf maybeCancellable maybeCallback userData
    touchManagedPtr subprocess
    whenJust stdinBuf touchManagedPtr
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data SubprocessCommunicateAsyncMethodInfo
instance (signature ~ (Maybe (GLib.Bytes.Bytes) -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod SubprocessCommunicateAsyncMethodInfo a signature where
    overloadedMethod = subprocessCommunicateAsync

instance O.OverloadedMethodInfo SubprocessCommunicateAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessCommunicateAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessCommunicateAsync"
        })


#endif

-- method Subprocess::communicate_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation { rawDocText = Just "Self" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Result" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stdout_buf"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Return location for stdout data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "stderr_buf"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Return location for stderr data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_subprocess_communicate_finish" g_subprocess_communicate_finish :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GLib.Bytes.Bytes) ->           -- stdout_buf : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Ptr (Ptr GLib.Bytes.Bytes) ->           -- stderr_buf : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Complete an invocation of 'GI.Gio.Objects.Subprocess.subprocessCommunicateAsync'.
subprocessCommunicateFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@subprocess@/: Self
    -> b
    -- ^ /@result@/: Result
    -> m ((Maybe GLib.Bytes.Bytes, Maybe GLib.Bytes.Bytes))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
subprocessCommunicateFinish subprocess result_ = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result_' <- unsafeManagedPtrCastPtr result_
    stdoutBuf <- callocMem :: IO (Ptr (Ptr GLib.Bytes.Bytes))
    stderrBuf <- callocMem :: IO (Ptr (Ptr GLib.Bytes.Bytes))
    onException (do
        _ <- propagateGError $ g_subprocess_communicate_finish subprocess' result_' stdoutBuf stderrBuf
        stdoutBuf' <- peek stdoutBuf
        maybeStdoutBuf' <- convertIfNonNull stdoutBuf' $ \stdoutBuf'' -> do
            stdoutBuf''' <- (wrapBoxed GLib.Bytes.Bytes) stdoutBuf''
            return stdoutBuf'''
        stderrBuf' <- peek stderrBuf
        maybeStderrBuf' <- convertIfNonNull stderrBuf' $ \stderrBuf'' -> do
            stderrBuf''' <- (wrapBoxed GLib.Bytes.Bytes) stderrBuf''
            return stderrBuf'''
        touchManagedPtr subprocess
        touchManagedPtr result_
        freeMem stdoutBuf
        freeMem stderrBuf
        return (maybeStdoutBuf', maybeStderrBuf')
     ) (do
        freeMem stdoutBuf
        freeMem stderrBuf
     )

#if defined(ENABLE_OVERLOADING)
data SubprocessCommunicateFinishMethodInfo
instance (signature ~ (b -> m ((Maybe GLib.Bytes.Bytes, Maybe GLib.Bytes.Bytes))), MonadIO m, IsSubprocess a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod SubprocessCommunicateFinishMethodInfo a signature where
    overloadedMethod = subprocessCommunicateFinish

instance O.OverloadedMethodInfo SubprocessCommunicateFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessCommunicateFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessCommunicateFinish"
        })


#endif

-- method Subprocess::communicate_utf8
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stdin_buf"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "data to send to the stdin of the subprocess, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stdout_buf"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data read from the subprocess stdout"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "stderr_buf"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data read from the subprocess stderr"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_subprocess_communicate_utf8" g_subprocess_communicate_utf8 :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    CString ->                              -- stdin_buf : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr CString ->                          -- stdout_buf : TBasicType TUTF8
    Ptr CString ->                          -- stderr_buf : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Like 'GI.Gio.Objects.Subprocess.subprocessCommunicate', but validates the output of the
-- process as UTF-8, and returns it as a regular NUL terminated string.
-- 
-- On error, /@stdoutBuf@/ and /@stderrBuf@/ will be set to undefined values and
-- should not be used.
subprocessCommunicateUtf8 ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> Maybe (T.Text)
    -- ^ /@stdinBuf@/: data to send to the stdin of the subprocess, or 'P.Nothing'
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable'
    -> m ((Maybe T.Text, Maybe T.Text))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
subprocessCommunicateUtf8 subprocess stdinBuf cancellable = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    maybeStdinBuf <- case stdinBuf of
        Nothing -> return FP.nullPtr
        Just jStdinBuf -> do
            jStdinBuf' <- textToCString jStdinBuf
            return jStdinBuf'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    stdoutBuf <- callocMem :: IO (Ptr CString)
    stderrBuf <- callocMem :: IO (Ptr CString)
    onException (do
        _ <- propagateGError $ g_subprocess_communicate_utf8 subprocess' maybeStdinBuf maybeCancellable stdoutBuf stderrBuf
        stdoutBuf' <- peek stdoutBuf
        maybeStdoutBuf' <- convertIfNonNull stdoutBuf' $ \stdoutBuf'' -> do
            stdoutBuf''' <- cstringToText stdoutBuf''
            return stdoutBuf'''
        freeMem stdoutBuf'
        stderrBuf' <- peek stderrBuf
        maybeStderrBuf' <- convertIfNonNull stderrBuf' $ \stderrBuf'' -> do
            stderrBuf''' <- cstringToText stderrBuf''
            return stderrBuf'''
        freeMem stderrBuf'
        touchManagedPtr subprocess
        whenJust cancellable touchManagedPtr
        freeMem maybeStdinBuf
        freeMem stdoutBuf
        freeMem stderrBuf
        return (maybeStdoutBuf', maybeStderrBuf')
     ) (do
        freeMem maybeStdinBuf
        freeMem stdoutBuf
        freeMem stderrBuf
     )

#if defined(ENABLE_OVERLOADING)
data SubprocessCommunicateUtf8MethodInfo
instance (signature ~ (Maybe (T.Text) -> Maybe (b) -> m ((Maybe T.Text, Maybe T.Text))), MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod SubprocessCommunicateUtf8MethodInfo a signature where
    overloadedMethod = subprocessCommunicateUtf8

instance O.OverloadedMethodInfo SubprocessCommunicateUtf8MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessCommunicateUtf8",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessCommunicateUtf8"
        })


#endif

-- method Subprocess::communicate_utf8_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation { rawDocText = Just "Self" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stdin_buf"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Input data, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Cancellable" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Callback" , sinceVersion = Nothing }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "User data" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_communicate_utf8_async" g_subprocess_communicate_utf8_async :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    CString ->                              -- stdin_buf : TBasicType TUTF8
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Asynchronous version of 'GI.Gio.Objects.Subprocess.subprocessCommunicateUtf8'.  Complete
-- invocation with 'GI.Gio.Objects.Subprocess.subprocessCommunicateUtf8Finish'.
subprocessCommunicateUtf8Async ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@subprocess@/: Self
    -> Maybe (T.Text)
    -- ^ /@stdinBuf@/: Input data, or 'P.Nothing'
    -> Maybe (b)
    -- ^ /@cancellable@/: Cancellable
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: Callback
    -> m ()
subprocessCommunicateUtf8Async subprocess stdinBuf cancellable callback = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    maybeStdinBuf <- case stdinBuf of
        Nothing -> return FP.nullPtr
        Just jStdinBuf -> do
            jStdinBuf' <- textToCString jStdinBuf
            return jStdinBuf'
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_subprocess_communicate_utf8_async subprocess' maybeStdinBuf maybeCancellable maybeCallback userData
    touchManagedPtr subprocess
    whenJust cancellable touchManagedPtr
    freeMem maybeStdinBuf
    return ()

#if defined(ENABLE_OVERLOADING)
data SubprocessCommunicateUtf8AsyncMethodInfo
instance (signature ~ (Maybe (T.Text) -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod SubprocessCommunicateUtf8AsyncMethodInfo a signature where
    overloadedMethod = subprocessCommunicateUtf8Async

instance O.OverloadedMethodInfo SubprocessCommunicateUtf8AsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessCommunicateUtf8Async",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessCommunicateUtf8Async"
        })


#endif

-- method Subprocess::communicate_utf8_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation { rawDocText = Just "Self" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Result" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stdout_buf"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Return location for stdout data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "stderr_buf"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Return location for stderr data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_subprocess_communicate_utf8_finish" g_subprocess_communicate_utf8_finish :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr CString ->                          -- stdout_buf : TBasicType TUTF8
    Ptr CString ->                          -- stderr_buf : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Complete an invocation of 'GI.Gio.Objects.Subprocess.subprocessCommunicateUtf8Async'.
subprocessCommunicateUtf8Finish ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@subprocess@/: Self
    -> b
    -- ^ /@result@/: Result
    -> m ((Maybe T.Text, Maybe T.Text))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
subprocessCommunicateUtf8Finish subprocess result_ = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result_' <- unsafeManagedPtrCastPtr result_
    stdoutBuf <- callocMem :: IO (Ptr CString)
    stderrBuf <- callocMem :: IO (Ptr CString)
    onException (do
        _ <- propagateGError $ g_subprocess_communicate_utf8_finish subprocess' result_' stdoutBuf stderrBuf
        stdoutBuf' <- peek stdoutBuf
        maybeStdoutBuf' <- convertIfNonNull stdoutBuf' $ \stdoutBuf'' -> do
            stdoutBuf''' <- cstringToText stdoutBuf''
            return stdoutBuf'''
        freeMem stdoutBuf'
        stderrBuf' <- peek stderrBuf
        maybeStderrBuf' <- convertIfNonNull stderrBuf' $ \stderrBuf'' -> do
            stderrBuf''' <- cstringToText stderrBuf''
            return stderrBuf'''
        freeMem stderrBuf'
        touchManagedPtr subprocess
        touchManagedPtr result_
        freeMem stdoutBuf
        freeMem stderrBuf
        return (maybeStdoutBuf', maybeStderrBuf')
     ) (do
        freeMem stdoutBuf
        freeMem stderrBuf
     )

#if defined(ENABLE_OVERLOADING)
data SubprocessCommunicateUtf8FinishMethodInfo
instance (signature ~ (b -> m ((Maybe T.Text, Maybe T.Text))), MonadIO m, IsSubprocess a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod SubprocessCommunicateUtf8FinishMethodInfo a signature where
    overloadedMethod = subprocessCommunicateUtf8Finish

instance O.OverloadedMethodInfo SubprocessCommunicateUtf8FinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessCommunicateUtf8Finish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessCommunicateUtf8Finish"
        })


#endif

-- method Subprocess::force_exit
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_force_exit" g_subprocess_force_exit :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO ()

-- | Use an operating-system specific method to attempt an immediate,
-- forceful termination of the process.  There is no mechanism to
-- determine whether or not the request itself was successful;
-- however, you can use 'GI.Gio.Objects.Subprocess.subprocessWait' to monitor the status of
-- the process after calling this function.
-- 
-- On Unix, this function sends @/SIGKILL/@.
-- 
-- /Since: 2.40/
subprocessForceExit ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m ()
subprocessForceExit subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    g_subprocess_force_exit subprocess'
    touchManagedPtr subprocess
    return ()

#if defined(ENABLE_OVERLOADING)
data SubprocessForceExitMethodInfo
instance (signature ~ (m ()), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessForceExitMethodInfo a signature where
    overloadedMethod = subprocessForceExit

instance O.OverloadedMethodInfo SubprocessForceExitMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessForceExit",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessForceExit"
        })


#endif

-- method Subprocess::get_exit_status
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_get_exit_status" g_subprocess_get_exit_status :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO Int32

-- | Check the exit status of the subprocess, given that it exited
-- normally.  This is the value passed to the @/exit()/@ system call or the
-- return value from main.
-- 
-- This is equivalent to the system WEXITSTATUS macro.
-- 
-- It is an error to call this function before 'GI.Gio.Objects.Subprocess.subprocessWait' and
-- unless 'GI.Gio.Objects.Subprocess.subprocessGetIfExited' returned 'P.True'.
-- 
-- /Since: 2.40/
subprocessGetExitStatus ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m Int32
    -- ^ __Returns:__ the exit status
subprocessGetExitStatus subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result <- g_subprocess_get_exit_status subprocess'
    touchManagedPtr subprocess
    return result

#if defined(ENABLE_OVERLOADING)
data SubprocessGetExitStatusMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessGetExitStatusMethodInfo a signature where
    overloadedMethod = subprocessGetExitStatus

instance O.OverloadedMethodInfo SubprocessGetExitStatusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessGetExitStatus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessGetExitStatus"
        })


#endif

-- method Subprocess::get_identifier
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_get_identifier" g_subprocess_get_identifier :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO CString

-- | On UNIX, returns the process ID as a decimal string.
-- On Windows, returns the result of @/GetProcessId()/@ also as a string.
-- If the subprocess has terminated, this will return 'P.Nothing'.
-- 
-- /Since: 2.40/
subprocessGetIdentifier ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the subprocess identifier, or 'P.Nothing' if the subprocess
    --    has terminated
subprocessGetIdentifier subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result <- g_subprocess_get_identifier subprocess'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr subprocess
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data SubprocessGetIdentifierMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessGetIdentifierMethodInfo a signature where
    overloadedMethod = subprocessGetIdentifier

instance O.OverloadedMethodInfo SubprocessGetIdentifierMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessGetIdentifier",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessGetIdentifier"
        })


#endif

-- method Subprocess::get_if_exited
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_get_if_exited" g_subprocess_get_if_exited :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO CInt

-- | Check if the given subprocess exited normally (ie: by way of @/exit()/@
-- or return from @/main()/@).
-- 
-- This is equivalent to the system WIFEXITED macro.
-- 
-- It is an error to call this function before 'GI.Gio.Objects.Subprocess.subprocessWait' has
-- returned.
-- 
-- /Since: 2.40/
subprocessGetIfExited ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the case of a normal exit
subprocessGetIfExited subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result <- g_subprocess_get_if_exited subprocess'
    let result' = (/= 0) result
    touchManagedPtr subprocess
    return result'

#if defined(ENABLE_OVERLOADING)
data SubprocessGetIfExitedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessGetIfExitedMethodInfo a signature where
    overloadedMethod = subprocessGetIfExited

instance O.OverloadedMethodInfo SubprocessGetIfExitedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessGetIfExited",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessGetIfExited"
        })


#endif

-- method Subprocess::get_if_signaled
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_get_if_signaled" g_subprocess_get_if_signaled :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO CInt

-- | Check if the given subprocess terminated in response to a signal.
-- 
-- This is equivalent to the system WIFSIGNALED macro.
-- 
-- It is an error to call this function before 'GI.Gio.Objects.Subprocess.subprocessWait' has
-- returned.
-- 
-- /Since: 2.40/
subprocessGetIfSignaled ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the case of termination due to a signal
subprocessGetIfSignaled subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result <- g_subprocess_get_if_signaled subprocess'
    let result' = (/= 0) result
    touchManagedPtr subprocess
    return result'

#if defined(ENABLE_OVERLOADING)
data SubprocessGetIfSignaledMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessGetIfSignaledMethodInfo a signature where
    overloadedMethod = subprocessGetIfSignaled

instance O.OverloadedMethodInfo SubprocessGetIfSignaledMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessGetIfSignaled",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessGetIfSignaled"
        })


#endif

-- method Subprocess::get_status
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_get_status" g_subprocess_get_status :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO Int32

-- | Gets the raw status code of the process, as from @/waitpid()/@.
-- 
-- This value has no particular meaning, but it can be used with the
-- macros defined by the system headers such as WIFEXITED.  It can also
-- be used with 'GI.GLib.Functions.spawnCheckWaitStatus'.
-- 
-- It is more likely that you want to use 'GI.Gio.Objects.Subprocess.subprocessGetIfExited'
-- followed by 'GI.Gio.Objects.Subprocess.subprocessGetExitStatus'.
-- 
-- It is an error to call this function before 'GI.Gio.Objects.Subprocess.subprocessWait' has
-- returned.
-- 
-- /Since: 2.40/
subprocessGetStatus ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m Int32
    -- ^ __Returns:__ the (meaningless) @/waitpid()/@ exit status from the kernel
subprocessGetStatus subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result <- g_subprocess_get_status subprocess'
    touchManagedPtr subprocess
    return result

#if defined(ENABLE_OVERLOADING)
data SubprocessGetStatusMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessGetStatusMethodInfo a signature where
    overloadedMethod = subprocessGetStatus

instance O.OverloadedMethodInfo SubprocessGetStatusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessGetStatus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessGetStatus"
        })


#endif

-- method Subprocess::get_stderr_pipe
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "InputStream" })
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_get_stderr_pipe" g_subprocess_get_stderr_pipe :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO (Ptr Gio.InputStream.InputStream)

-- | Gets the t'GI.Gio.Objects.InputStream.InputStream' from which to read the stderr output of
-- /@subprocess@/.
-- 
-- The process must have been created with 'GI.Gio.Flags.SubprocessFlagsStderrPipe',
-- otherwise 'P.Nothing' will be returned.
-- 
-- /Since: 2.40/
subprocessGetStderrPipe ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m (Maybe Gio.InputStream.InputStream)
    -- ^ __Returns:__ the stderr pipe
subprocessGetStderrPipe subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result <- g_subprocess_get_stderr_pipe subprocess'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gio.InputStream.InputStream) result'
        return result''
    touchManagedPtr subprocess
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data SubprocessGetStderrPipeMethodInfo
instance (signature ~ (m (Maybe Gio.InputStream.InputStream)), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessGetStderrPipeMethodInfo a signature where
    overloadedMethod = subprocessGetStderrPipe

instance O.OverloadedMethodInfo SubprocessGetStderrPipeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessGetStderrPipe",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessGetStderrPipe"
        })


#endif

-- method Subprocess::get_stdin_pipe
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gio" , name = "OutputStream" })
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_get_stdin_pipe" g_subprocess_get_stdin_pipe :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO (Ptr Gio.OutputStream.OutputStream)

-- | Gets the t'GI.Gio.Objects.OutputStream.OutputStream' that you can write to in order to give data
-- to the stdin of /@subprocess@/.
-- 
-- The process must have been created with 'GI.Gio.Flags.SubprocessFlagsStdinPipe' and
-- not 'GI.Gio.Flags.SubprocessFlagsStdinInherit', otherwise 'P.Nothing' will be returned.
-- 
-- /Since: 2.40/
subprocessGetStdinPipe ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m (Maybe Gio.OutputStream.OutputStream)
    -- ^ __Returns:__ the stdout pipe
subprocessGetStdinPipe subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result <- g_subprocess_get_stdin_pipe subprocess'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gio.OutputStream.OutputStream) result'
        return result''
    touchManagedPtr subprocess
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data SubprocessGetStdinPipeMethodInfo
instance (signature ~ (m (Maybe Gio.OutputStream.OutputStream)), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessGetStdinPipeMethodInfo a signature where
    overloadedMethod = subprocessGetStdinPipe

instance O.OverloadedMethodInfo SubprocessGetStdinPipeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessGetStdinPipe",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessGetStdinPipe"
        })


#endif

-- method Subprocess::get_stdout_pipe
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "InputStream" })
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_get_stdout_pipe" g_subprocess_get_stdout_pipe :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO (Ptr Gio.InputStream.InputStream)

-- | Gets the t'GI.Gio.Objects.InputStream.InputStream' from which to read the stdout output of
-- /@subprocess@/.
-- 
-- The process must have been created with 'GI.Gio.Flags.SubprocessFlagsStdoutPipe',
-- otherwise 'P.Nothing' will be returned.
-- 
-- /Since: 2.40/
subprocessGetStdoutPipe ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m (Maybe Gio.InputStream.InputStream)
    -- ^ __Returns:__ the stdout pipe
subprocessGetStdoutPipe subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result <- g_subprocess_get_stdout_pipe subprocess'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gio.InputStream.InputStream) result'
        return result''
    touchManagedPtr subprocess
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data SubprocessGetStdoutPipeMethodInfo
instance (signature ~ (m (Maybe Gio.InputStream.InputStream)), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessGetStdoutPipeMethodInfo a signature where
    overloadedMethod = subprocessGetStdoutPipe

instance O.OverloadedMethodInfo SubprocessGetStdoutPipeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessGetStdoutPipe",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessGetStdoutPipe"
        })


#endif

-- method Subprocess::get_successful
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_get_successful" g_subprocess_get_successful :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO CInt

-- | Checks if the process was \"successful\".  A process is considered
-- successful if it exited cleanly with an exit status of 0, either by
-- way of the @/exit()/@ system call or return from @/main()/@.
-- 
-- It is an error to call this function before 'GI.Gio.Objects.Subprocess.subprocessWait' has
-- returned.
-- 
-- /Since: 2.40/
subprocessGetSuccessful ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the process exited cleanly with a exit status of 0
subprocessGetSuccessful subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result <- g_subprocess_get_successful subprocess'
    let result' = (/= 0) result
    touchManagedPtr subprocess
    return result'

#if defined(ENABLE_OVERLOADING)
data SubprocessGetSuccessfulMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessGetSuccessfulMethodInfo a signature where
    overloadedMethod = subprocessGetSuccessful

instance O.OverloadedMethodInfo SubprocessGetSuccessfulMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessGetSuccessful",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessGetSuccessful"
        })


#endif

-- method Subprocess::get_term_sig
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_get_term_sig" g_subprocess_get_term_sig :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    IO Int32

-- | Get the signal number that caused the subprocess to terminate, given
-- that it terminated due to a signal.
-- 
-- This is equivalent to the system WTERMSIG macro.
-- 
-- It is an error to call this function before 'GI.Gio.Objects.Subprocess.subprocessWait' and
-- unless 'GI.Gio.Objects.Subprocess.subprocessGetIfSignaled' returned 'P.True'.
-- 
-- /Since: 2.40/
subprocessGetTermSig ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> m Int32
    -- ^ __Returns:__ the signal causing termination
subprocessGetTermSig subprocess = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result <- g_subprocess_get_term_sig subprocess'
    touchManagedPtr subprocess
    return result

#if defined(ENABLE_OVERLOADING)
data SubprocessGetTermSigMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessGetTermSigMethodInfo a signature where
    overloadedMethod = subprocessGetTermSig

instance O.OverloadedMethodInfo SubprocessGetTermSigMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessGetTermSig",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessGetTermSig"
        })


#endif

-- method Subprocess::send_signal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "signal_num"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the signal number to send"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_send_signal" g_subprocess_send_signal :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Int32 ->                                -- signal_num : TBasicType TInt
    IO ()

-- | Sends the UNIX signal /@signalNum@/ to the subprocess, if it is still
-- running.
-- 
-- This API is race-free.  If the subprocess has terminated, it will not
-- be signalled.
-- 
-- This API is not available on Windows.
-- 
-- /Since: 2.40/
subprocessSendSignal ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> Int32
    -- ^ /@signalNum@/: the signal number to send
    -> m ()
subprocessSendSignal subprocess signalNum = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    g_subprocess_send_signal subprocess' signalNum
    touchManagedPtr subprocess
    return ()

#if defined(ENABLE_OVERLOADING)
data SubprocessSendSignalMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsSubprocess a) => O.OverloadedMethod SubprocessSendSignalMethodInfo a signature where
    overloadedMethod = subprocessSendSignal

instance O.OverloadedMethodInfo SubprocessSendSignalMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessSendSignal",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessSendSignal"
        })


#endif

-- method Subprocess::wait
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_subprocess_wait" g_subprocess_wait :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Synchronously wait for the subprocess to terminate.
-- 
-- After the process terminates you can query its exit status with
-- functions such as 'GI.Gio.Objects.Subprocess.subprocessGetIfExited' and
-- 'GI.Gio.Objects.Subprocess.subprocessGetExitStatus'.
-- 
-- This function does not fail in the case of the subprocess having
-- abnormal termination.  See 'GI.Gio.Objects.Subprocess.subprocessWaitCheck' for that.
-- 
-- Cancelling /@cancellable@/ doesn\'t kill the subprocess.  Call
-- 'GI.Gio.Objects.Subprocess.subprocessForceExit' if it is desirable.
-- 
-- /Since: 2.40/
subprocessWait ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
subprocessWait subprocess cancellable = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_subprocess_wait subprocess' maybeCancellable
        touchManagedPtr subprocess
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data SubprocessWaitMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod SubprocessWaitMethodInfo a signature where
    overloadedMethod = subprocessWait

instance O.OverloadedMethodInfo SubprocessWaitMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessWait",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessWait"
        })


#endif

-- method Subprocess::wait_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call when the operation is complete"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 3
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user_data for @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_wait_async" g_subprocess_wait_async :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Wait for the subprocess to terminate.
-- 
-- This is the asynchronous version of 'GI.Gio.Objects.Subprocess.subprocessWait'.
-- 
-- /Since: 2.40/
subprocessWaitAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the operation is complete
    -> m ()
subprocessWaitAsync subprocess cancellable callback = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_subprocess_wait_async subprocess' maybeCancellable maybeCallback userData
    touchManagedPtr subprocess
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data SubprocessWaitAsyncMethodInfo
instance (signature ~ (Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod SubprocessWaitAsyncMethodInfo a signature where
    overloadedMethod = subprocessWaitAsync

instance O.OverloadedMethodInfo SubprocessWaitAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessWaitAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessWaitAsync"
        })


#endif

-- method Subprocess::wait_check
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_subprocess_wait_check" g_subprocess_wait_check :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Combines 'GI.Gio.Objects.Subprocess.subprocessWait' with 'GI.GLib.Functions.spawnCheckWaitStatus'.
-- 
-- /Since: 2.40/
subprocessWaitCheck ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
subprocessWaitCheck subprocess cancellable = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    onException (do
        _ <- propagateGError $ g_subprocess_wait_check subprocess' maybeCancellable
        touchManagedPtr subprocess
        whenJust cancellable touchManagedPtr
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data SubprocessWaitCheckMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod SubprocessWaitCheckMethodInfo a signature where
    overloadedMethod = subprocessWaitCheck

instance O.OverloadedMethodInfo SubprocessWaitCheckMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessWaitCheck",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessWaitCheck"
        })


#endif

-- method Subprocess::wait_check_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GCancellable, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call when the operation is complete"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 3
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "user_data for @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_subprocess_wait_check_async" g_subprocess_wait_check_async :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Combines 'GI.Gio.Objects.Subprocess.subprocessWaitAsync' with 'GI.GLib.Functions.spawnCheckWaitStatus'.
-- 
-- This is the asynchronous version of 'GI.Gio.Objects.Subprocess.subprocessWaitCheck'.
-- 
-- /Since: 2.40/
subprocessWaitCheckAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> Maybe (b)
    -- ^ /@cancellable@/: a t'GI.Gio.Objects.Cancellable.Cancellable', or 'P.Nothing'
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the operation is complete
    -> m ()
subprocessWaitCheckAsync subprocess cancellable callback = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_subprocess_wait_check_async subprocess' maybeCancellable maybeCallback userData
    touchManagedPtr subprocess
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data SubprocessWaitCheckAsyncMethodInfo
instance (signature ~ (Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsSubprocess a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod SubprocessWaitCheckAsyncMethodInfo a signature where
    overloadedMethod = subprocessWaitCheckAsync

instance O.OverloadedMethodInfo SubprocessWaitCheckAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessWaitCheckAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessWaitCheckAsync"
        })


#endif

-- method Subprocess::wait_check_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the #GAsyncResult passed to your #GAsyncReadyCallback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_subprocess_wait_check_finish" g_subprocess_wait_check_finish :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Collects the result of a previous call to
-- 'GI.Gio.Objects.Subprocess.subprocessWaitCheckAsync'.
-- 
-- /Since: 2.40/
subprocessWaitCheckFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> b
    -- ^ /@result@/: the t'GI.Gio.Interfaces.AsyncResult.AsyncResult' passed to your t'GI.Gio.Callbacks.AsyncReadyCallback'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
subprocessWaitCheckFinish subprocess result_ = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_subprocess_wait_check_finish subprocess' result_'
        touchManagedPtr subprocess
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data SubprocessWaitCheckFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsSubprocess a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod SubprocessWaitCheckFinishMethodInfo a signature where
    overloadedMethod = subprocessWaitCheckFinish

instance O.OverloadedMethodInfo SubprocessWaitCheckFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessWaitCheckFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessWaitCheckFinish"
        })


#endif

-- method Subprocess::wait_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "subprocess"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Subprocess" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GSubprocess" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "result"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the #GAsyncResult passed to your #GAsyncReadyCallback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_subprocess_wait_finish" g_subprocess_wait_finish :: 
    Ptr Subprocess ->                       -- subprocess : TInterface (Name {namespace = "Gio", name = "Subprocess"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- result : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Collects the result of a previous call to
-- 'GI.Gio.Objects.Subprocess.subprocessWaitAsync'.
-- 
-- /Since: 2.40/
subprocessWaitFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsSubprocess a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@subprocess@/: a t'GI.Gio.Objects.Subprocess.Subprocess'
    -> b
    -- ^ /@result@/: the t'GI.Gio.Interfaces.AsyncResult.AsyncResult' passed to your t'GI.Gio.Callbacks.AsyncReadyCallback'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
subprocessWaitFinish subprocess result_ = liftIO $ do
    subprocess' <- unsafeManagedPtrCastPtr subprocess
    result_' <- unsafeManagedPtrCastPtr result_
    onException (do
        _ <- propagateGError $ g_subprocess_wait_finish subprocess' result_'
        touchManagedPtr subprocess
        touchManagedPtr result_
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data SubprocessWaitFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsSubprocess a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod SubprocessWaitFinishMethodInfo a signature where
    overloadedMethod = subprocessWaitFinish

instance O.OverloadedMethodInfo SubprocessWaitFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Objects.Subprocess.subprocessWaitFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Objects-Subprocess.html#v:subprocessWaitFinish"
        })


#endif


