{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Reads XML descriptions of a user interface and instantiates the described objects.
-- 
-- To create a @GtkBuilder@ from a user interface description, call
-- 'GI.Gtk.Objects.Builder.builderNewFromFile', 'GI.Gtk.Objects.Builder.builderNewFromResource'
-- or 'GI.Gtk.Objects.Builder.builderNewFromString'.
-- 
-- In the (unusual) case that you want to add user interface
-- descriptions from multiple sources to the same @GtkBuilder@ you can
-- call 'GI.Gtk.Objects.Builder.builderNew' to get an empty builder and populate it by
-- (multiple) calls to 'GI.Gtk.Objects.Builder.builderAddFromFile',
-- 'GI.Gtk.Objects.Builder.builderAddFromResource' or
-- 'GI.Gtk.Objects.Builder.builderAddFromString'.
-- 
-- A @GtkBuilder@ holds a reference to all objects that it has constructed
-- and drops these references when it is finalized. This finalization can
-- cause the destruction of non-widget objects or widgets which are not
-- contained in a toplevel window. For toplevel windows constructed by a
-- builder, it is the responsibility of the user to call
-- 'GI.Gtk.Objects.Window.windowDestroy' to get rid of them and all the widgets
-- they contain.
-- 
-- The functions 'GI.Gtk.Objects.Builder.builderGetObject' and
-- 'GI.Gtk.Objects.Builder.builderGetObjects' can be used to access the widgets in
-- the interface by the names assigned to them inside the UI description.
-- Toplevel windows returned by these functions will stay around until the
-- user explicitly destroys them with 'GI.Gtk.Objects.Window.windowDestroy'. Other
-- widgets will either be part of a larger hierarchy constructed by the
-- builder (in which case you should not have to worry about their lifecycle),
-- or without a parent, in which case they have to be added to some container
-- to make use of them. Non-widget objects need to be reffed with
-- 'GI.GObject.Objects.Object.objectRef' to keep them beyond the lifespan of the builder.
-- 
-- == GtkBuilder UI Definitions
-- 
-- @GtkBuilder@ parses textual descriptions of user interfaces which are
-- specified in XML format. We refer to these descriptions as “GtkBuilder
-- UI definitions” or just “UI definitions” if the context is clear.
-- 
-- === Structure of UI definitions
-- 
-- UI definition files are always encoded in UTF-8.
-- 
-- The toplevel element is @\<interface>@. It optionally takes a “domain”
-- attribute, which will make the builder look for translated strings
-- using @dgettext()@ in the domain specified. This can also be done by
-- calling 'GI.Gtk.Objects.Builder.builderSetTranslationDomain' on the builder.
-- For example:
-- 
-- 
-- === /xml code/
-- ><?xml version="1.0" encoding="UTF-8"?>
-- ><interface domain="your-app">
-- >  ...
-- ></interface>
-- 
-- 
-- === Requirements
-- 
-- The target toolkit version(s) are described by @\<requires>@ elements,
-- the “lib” attribute specifies the widget library in question (currently
-- the only supported value is “gtk”) and the “version” attribute specifies
-- the target version in the form “@\<major>@.@\<minor>@”. @GtkBuilder@ will
-- error out if the version requirements are not met. For example:
-- 
-- 
-- === /xml code/
-- ><?xml version="1.0" encoding="UTF-8"?>
-- ><interface domain="your-app">
-- >  <requires lib="gtk" version="4.0" />
-- ></interface>
-- 
-- 
-- === Objects
-- 
-- Objects are defined as children of the @\<interface>@ element.
-- 
-- Objects are described by @\<object>@ elements, which can contain
-- @\<property>@ elements to set properties, @\<signal>@ elements which
-- connect signals to handlers, and @\<child>@ elements, which describe
-- child objects.
-- 
-- Typically, the specific kind of object represented by an @\<object>@
-- element is specified by the “class” attribute. If the type has not
-- been loaded yet, GTK tries to find the @get_type()@ function from the
-- class name by applying heuristics. This works in most cases, but if
-- necessary, it is possible to specify the name of the @get_type()@
-- function explicitly with the \"type-func\" attribute. If your UI definition
-- is referencing internal types, you should make sure to call
-- @g_type_ensure()@ for each object type before parsing the UI definition.
-- 
-- Objects may be given a name with the “id” attribute, which allows the
-- application to retrieve them from the builder with
-- 'GI.Gtk.Objects.Builder.builderGetObject'. An id is also necessary to use the
-- object as property value in other parts of the UI definition. GTK
-- reserves ids starting and ending with @___@ (three consecutive
-- underscores) for its own purposes.
-- 
-- === Properties
-- 
-- Setting properties of objects is pretty straightforward with the
-- @\<property>@ element: the “name” attribute specifies the name of the
-- property, and the content of the element specifies the value:
-- 
-- 
-- === /xml code/
-- ><object class="GtkButton">
-- >  <property name="label">Hello, world</property>
-- ></object>
-- 
-- 
-- If the “translatable” attribute is set to a true value, GTK uses
-- @gettext()@ (or @dgettext()@ if the builder has a translation domain set)
-- to find a translation for the value. This happens before the value
-- is parsed, so it can be used for properties of any type, but it is
-- probably most useful for string properties. It is also possible to
-- specify a context to disambiguate short strings, and comments which
-- may help the translators:
-- 
-- 
-- === /xml code/
-- ><object class="GtkButton">
-- >  <property name="label"
-- >            translatable="yes"
-- >            context="button"
-- >            comments="A classic">Hello, world</property>
-- ></object>
-- 
-- 
-- The xgettext tool that is part of gettext can extract these strings,
-- but note that it only looks for translatable=\"yes\".
-- 
-- @GtkBuilder@ can parse textual representations for the most common
-- property types:
-- 
-- * characters
-- * strings
-- * integers
-- * floating-point numbers
-- * booleans (strings like “TRUE”, “t”, “yes”, “y”, “1” are interpreted
--   as true values, strings like “FALSE”, “f”, “no”, “n”, “0” are interpreted
--   as false values)
-- * string lists (separated by newlines)
-- * enumeration types (can be specified by their full C identifier their short
--   name used when registering the enumeration type, or their integer value)
-- * flag types (can be specified by their C identifier or short name,
--   optionally combined with “|” for bitwise OR, or a single integer value
--   e.g., “GTK_INPUT_HINT_EMOJI|GTK_INPUT_HINT_LOWERCASE”, or “emoji|lowercase” or 520).
-- * colors (in the format understood by 'GI.Gdk.Structs.RGBA.rGBAParse')
-- * transforms (in the format understood by 'GI.Gsk.Structs.Transform.transformParse')
-- * Pango attribute lists (in the format understood by 'GI.Pango.Structs.AttrList.attrListToString')
-- * Pango tab arrays (in the format understood by 'GI.Pango.Structs.TabArray.tabArrayToString')
-- * Pango font descriptions (in the format understood by 'GI.Pango.Structs.FontDescription.fontDescriptionFromString')
-- * @GVariant@ (in the format understood by 'GI.GLib.Structs.Variant.variantParse')
-- * textures (can be specified as an object id, a resource path or a filename of an image file to load relative to the Builder file or the CWD if 'GI.Gtk.Objects.Builder.builderAddFromString' was used)
-- * GFile (like textures, can be specified as an object id, a URI or a filename of a file to load relative to the Builder file or the CWD if 'GI.Gtk.Objects.Builder.builderAddFromString' was used)
-- 
-- 
-- Objects can be referred to by their name and by default refer to
-- objects declared in the local XML fragment and objects exposed via
-- 'GI.Gtk.Objects.Builder.builderExposeObject'. In general, @GtkBuilder@ allows
-- forward references to objects declared in the local XML; an object
-- doesn’t have to be constructed before it can be referred to. The
-- exception to this rule is that an object has to be constructed before
-- it can be used as the value of a construct-only property.
-- 
-- === Child objects
-- 
-- Many widgets have properties for child widgets, such as
-- [Expander:child]("GI.Gtk.Objects.Expander#g:attr:child"). In this case, the preferred way to
-- specify the child widget in a ui file is to simply set the property:
-- 
-- 
-- === /xml code/
-- ><object class="GtkExpander">
-- >  <property name="child">
-- >    <object class="GtkLabel">
-- >    ...
-- >    </object>
-- >  </property>
-- ></object>
-- 
-- 
-- Generic containers that can contain an arbitrary number of children,
-- such as t'GI.Gtk.Objects.Box.Box' instead use the @\<child>@ element. A @\<child>@
-- element contains an @\<object>@ element which describes the child object.
-- Most often, child objects are widgets inside a container, but they can
-- also be, e.g., actions in an action group, or columns in a tree model.
-- 
-- Any object type that implements the t'GI.Gtk.Interfaces.Buildable.Buildable' interface can
-- specify how children may be added to it. Since many objects and widgets that
-- are included with GTK already implement the @GtkBuildable@ interface,
-- typically child objects can be added using the @\<child>@ element without
-- having to be concerned about the underlying implementation.
-- 
-- See the <https://docs.gtk.org/gtk4/class.Widget.html#gtkwidget-as-gtkbuildable `GtkWidget` documentation>
-- for many examples of using @GtkBuilder@ with widgets, including setting
-- child objects using the @\<child>@ element.
-- 
-- A noteworthy special case to the general rule that only objects implementing
-- @GtkBuildable@ may specify how to handle the @\<child>@ element is that
-- @GtkBuilder@ provides special support for adding objects to a
-- t'GI.Gio.Objects.ListStore.ListStore' by using the @\<child>@ element. For instance:
-- 
-- 
-- === /xml code/
-- ><object class="GListStore">
-- >  <property name="item-type">MyObject</property>
-- >  <child>
-- >    <object class="MyObject" />
-- >  </child>
-- >  ...
-- ></object>
-- 
-- 
-- === Property bindings
-- 
-- It is also possible to bind a property value to another object\'s
-- property value using the attributes \"bind-source\" to specify the
-- source object of the binding, and optionally, \"bind-property\" and
-- \"bind-flags\" to specify the source property and source binding flags
-- respectively. Internally, @GtkBuilder@ implements this using
-- t'GI.GObject.Objects.Binding.Binding' objects.
-- 
-- For instance, in the example below the “label” property of the
-- @bottom_label@ widget is bound to the “label” property of the
-- @top_button@ widget:
-- 
-- 
-- === /xml code/
-- ><object class="GtkBox">
-- >  <property name="orientation">vertical</property>
-- >  <child>
-- >    <object class="GtkButton" id="top_button">
-- >      <property name="label">Hello, world</property>
-- >    </object>
-- >  </child>
-- >  <child>
-- >    <object class="GtkLabel" id="bottom_label">
-- >      <property name="label"
-- >                bind-source="top_button"
-- >                bind-property="label"
-- >                bind-flags="sync-create" />
-- >    </object>
-- >  </child>
-- ></object>
-- 
-- 
-- For more information, see the documentation of the
-- 'GI.GObject.Objects.Object.objectBindProperty' method.
-- 
-- Please note that another way to set up bindings between objects in .ui files
-- is to use the @GtkExpression@ methodology. See the
-- <https://docs.gtk.org/gtk4/class.Expression.html#gtkexpression-in-ui-files `GtkExpression` documentation>
-- for more information.
-- 
-- === Internal children
-- 
-- Sometimes it is necessary to refer to widgets which have implicitly
-- been constructed by GTK as part of a composite widget, to set
-- properties on them or to add further children (e.g. the content area
-- of a @GtkDialog@). This can be achieved by setting the “internal-child”
-- property of the @\<child>@ element to a true value. Note that @GtkBuilder@
-- still requires an @\<object>@ element for the internal child, even if it
-- has already been constructed.
-- 
-- === Specialized children
-- 
-- A number of widgets have different places where a child can be added
-- (e.g. tabs vs. page content in notebooks). This can be reflected in
-- a UI definition by specifying the “type” attribute on a @\<child>@
-- The possible values for the “type” attribute are described in the
-- sections describing the widget-specific portions of UI definitions.
-- 
-- === Signal handlers and function pointers
-- 
-- Signal handlers are set up with the @\<signal>@ element. The “name”
-- attribute specifies the name of the signal, and the “handler” attribute
-- specifies the function to connect to the signal.
-- 
-- 
-- === /xml code/
-- ><object class="GtkButton" id="hello_button">
-- >  <signal name="clicked" handler="hello_button__clicked" />
-- ></object>
-- 
-- 
-- The remaining attributes, “after”, “swapped” and “object”, have the
-- same meaning as the corresponding parameters of the
-- @/GObject.signal_connect_object/@ or @/GObject.signal_connect_data/@
-- functions:
-- 
-- * “after” matches the @G_CONNECT_AFTER@ flag, and will ensure that the
--   handler is called after the default class closure for the signal
-- * “swapped” matches the @G_CONNECT_SWAPPED@ flag, and will swap the
--   instance and closure arguments when invoking the signal handler
-- * “object” will bind the signal handler to the lifetime of the object
--   referenced by the attribute
-- 
-- 
-- By default \"swapped\" will be set to \"yes\" if not specified otherwise, in
-- the case where \"object\" is set, for convenience. A “last_modification_time”
-- attribute is also allowed, but it does not have a meaning to the builder.
-- 
-- When compiling applications for Windows, you must declare signal callbacks
-- with the @G_MODULE_EXPORT@ decorator, or they will not be put in the symbol
-- table:
-- 
-- 
-- === /c code/
-- >G_MODULE_EXPORT void
-- >hello_button__clicked (GtkButton *button,
-- >                       gpointer data)
-- >{
-- >  // ...
-- >}
-- 
-- 
-- On Linux and Unix, this is not necessary; applications should instead
-- be compiled with the @-Wl,--export-dynamic@ argument inside their compiler
-- flags, and linked against @gmodule-export-2.0@.
-- 
-- == Example UI Definition
-- 
-- 
-- === /xml code/
-- ><interface>
-- >  <object class="GtkDialog" id="dialog1">
-- >    <child internal-child="content_area">
-- >      <object class="GtkBox">
-- >        <child internal-child="action_area">
-- >          <object class="GtkBox">
-- >            <child>
-- >              <object class="GtkButton" id="ok_button">
-- >                <property name="label" translatable="yes">_Ok</property>
-- >                <property name="use-underline">True</property>
-- >                <signal name="clicked" handler="ok_button_clicked"/>
-- >              </object>
-- >            </child>
-- >          </object>
-- >        </child>
-- >      </object>
-- >    </child>
-- >  </object>
-- ></interface>
-- 
-- 
-- == Using GtkBuildable for extending UI definitions
-- 
-- Objects can implement the t'GI.Gtk.Interfaces.Buildable.Buildable' interface to add custom
-- elements and attributes to the XML. Typically, any extension will be
-- documented in each type that implements the interface.
-- 
-- == Menus
-- 
-- In addition to objects with properties that are created with @\<object>@ and
-- @\<property>@ elements, @GtkBuilder@ also allows to parse XML menu definitions
-- as used by t'GI.Gio.Objects.Menu.Menu' when exporting menu models over D-Bus, and as
-- described in the t'GI.Gtk.Objects.PopoverMenu.PopoverMenu' documentation. Menus can be defined
-- as toplevel elements, or as property values for properties of type @GMenuModel@.
-- 
-- == Templates
-- 
-- When describing a t'GI.Gtk.Objects.Widget.Widget', you can use the @\<template>@ tag to
-- describe a UI bound to a specific widget type. GTK will automatically load
-- the UI definition when instantiating the type, and bind children and
-- signal handlers to instance fields and function symbols.
-- 
-- For more information, see the <https://docs.gtk.org/gtk4/class.Widget.html#building-composite-widgets-from-template-xml `GtkWidget` documentation>
-- for details.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.Builder
    ( 

-- * Exported types
    Builder(..)                             ,
    IsBuilder                               ,
    toBuilder                               ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [addFromFile]("GI.Gtk.Objects.Builder#g:method:addFromFile"), [addFromResource]("GI.Gtk.Objects.Builder#g:method:addFromResource"), [addFromString]("GI.Gtk.Objects.Builder#g:method:addFromString"), [addObjectsFromFile]("GI.Gtk.Objects.Builder#g:method:addObjectsFromFile"), [addObjectsFromResource]("GI.Gtk.Objects.Builder#g:method:addObjectsFromResource"), [addObjectsFromString]("GI.Gtk.Objects.Builder#g:method:addObjectsFromString"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [createClosure]("GI.Gtk.Objects.Builder#g:method:createClosure"), [exposeObject]("GI.Gtk.Objects.Builder#g:method:exposeObject"), [extendWithTemplate]("GI.Gtk.Objects.Builder#g:method:extendWithTemplate"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [valueFromString]("GI.Gtk.Objects.Builder#g:method:valueFromString"), [valueFromStringType]("GI.Gtk.Objects.Builder#g:method:valueFromStringType"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getCurrentObject]("GI.Gtk.Objects.Builder#g:method:getCurrentObject"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getObject]("GI.Gtk.Objects.Builder#g:method:getObject"), [getObjects]("GI.Gtk.Objects.Builder#g:method:getObjects"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getScope]("GI.Gtk.Objects.Builder#g:method:getScope"), [getTranslationDomain]("GI.Gtk.Objects.Builder#g:method:getTranslationDomain"), [getTypeFromName]("GI.Gtk.Objects.Builder#g:method:getTypeFromName").
-- 
-- ==== Setters
-- [setCurrentObject]("GI.Gtk.Objects.Builder#g:method:setCurrentObject"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setScope]("GI.Gtk.Objects.Builder#g:method:setScope"), [setTranslationDomain]("GI.Gtk.Objects.Builder#g:method:setTranslationDomain").

#if defined(ENABLE_OVERLOADING)
    ResolveBuilderMethod                    ,
#endif

-- ** addFromFile #method:addFromFile#

#if defined(ENABLE_OVERLOADING)
    BuilderAddFromFileMethodInfo            ,
#endif
    builderAddFromFile                      ,


-- ** addFromResource #method:addFromResource#

#if defined(ENABLE_OVERLOADING)
    BuilderAddFromResourceMethodInfo        ,
#endif
    builderAddFromResource                  ,


-- ** addFromString #method:addFromString#

#if defined(ENABLE_OVERLOADING)
    BuilderAddFromStringMethodInfo          ,
#endif
    builderAddFromString                    ,


-- ** addObjectsFromFile #method:addObjectsFromFile#

#if defined(ENABLE_OVERLOADING)
    BuilderAddObjectsFromFileMethodInfo     ,
#endif
    builderAddObjectsFromFile               ,


-- ** addObjectsFromResource #method:addObjectsFromResource#

#if defined(ENABLE_OVERLOADING)
    BuilderAddObjectsFromResourceMethodInfo ,
#endif
    builderAddObjectsFromResource           ,


-- ** addObjectsFromString #method:addObjectsFromString#

#if defined(ENABLE_OVERLOADING)
    BuilderAddObjectsFromStringMethodInfo   ,
#endif
    builderAddObjectsFromString             ,


-- ** createClosure #method:createClosure#

#if defined(ENABLE_OVERLOADING)
    BuilderCreateClosureMethodInfo          ,
#endif
    builderCreateClosure                    ,


-- ** exposeObject #method:exposeObject#

#if defined(ENABLE_OVERLOADING)
    BuilderExposeObjectMethodInfo           ,
#endif
    builderExposeObject                     ,


-- ** extendWithTemplate #method:extendWithTemplate#

#if defined(ENABLE_OVERLOADING)
    BuilderExtendWithTemplateMethodInfo     ,
#endif
    builderExtendWithTemplate               ,


-- ** getCurrentObject #method:getCurrentObject#

#if defined(ENABLE_OVERLOADING)
    BuilderGetCurrentObjectMethodInfo       ,
#endif
    builderGetCurrentObject                 ,


-- ** getObject #method:getObject#

#if defined(ENABLE_OVERLOADING)
    BuilderGetObjectMethodInfo              ,
#endif
    builderGetObject                        ,


-- ** getObjects #method:getObjects#

#if defined(ENABLE_OVERLOADING)
    BuilderGetObjectsMethodInfo             ,
#endif
    builderGetObjects                       ,


-- ** getScope #method:getScope#

#if defined(ENABLE_OVERLOADING)
    BuilderGetScopeMethodInfo               ,
#endif
    builderGetScope                         ,


-- ** getTranslationDomain #method:getTranslationDomain#

#if defined(ENABLE_OVERLOADING)
    BuilderGetTranslationDomainMethodInfo   ,
#endif
    builderGetTranslationDomain             ,


-- ** getTypeFromName #method:getTypeFromName#

#if defined(ENABLE_OVERLOADING)
    BuilderGetTypeFromNameMethodInfo        ,
#endif
    builderGetTypeFromName                  ,


-- ** new #method:new#

    builderNew                              ,


-- ** newFromFile #method:newFromFile#

    builderNewFromFile                      ,


-- ** newFromResource #method:newFromResource#

    builderNewFromResource                  ,


-- ** newFromString #method:newFromString#

    builderNewFromString                    ,


-- ** setCurrentObject #method:setCurrentObject#

#if defined(ENABLE_OVERLOADING)
    BuilderSetCurrentObjectMethodInfo       ,
#endif
    builderSetCurrentObject                 ,


-- ** setScope #method:setScope#

#if defined(ENABLE_OVERLOADING)
    BuilderSetScopeMethodInfo               ,
#endif
    builderSetScope                         ,


-- ** setTranslationDomain #method:setTranslationDomain#

#if defined(ENABLE_OVERLOADING)
    BuilderSetTranslationDomainMethodInfo   ,
#endif
    builderSetTranslationDomain             ,


-- ** valueFromString #method:valueFromString#

#if defined(ENABLE_OVERLOADING)
    BuilderValueFromStringMethodInfo        ,
#endif
    builderValueFromString                  ,


-- ** valueFromStringType #method:valueFromStringType#

#if defined(ENABLE_OVERLOADING)
    BuilderValueFromStringTypeMethodInfo    ,
#endif
    builderValueFromStringType              ,




 -- * Properties


-- ** currentObject #attr:currentObject#
-- | The object the builder is evaluating for.

#if defined(ENABLE_OVERLOADING)
    BuilderCurrentObjectPropertyInfo        ,
#endif
#if defined(ENABLE_OVERLOADING)
    builderCurrentObject                    ,
#endif
    clearBuilderCurrentObject               ,
    constructBuilderCurrentObject           ,
    getBuilderCurrentObject                 ,
    setBuilderCurrentObject                 ,


-- ** scope #attr:scope#
-- | The scope the builder is operating in

#if defined(ENABLE_OVERLOADING)
    BuilderScopePropertyInfo                ,
#endif
#if defined(ENABLE_OVERLOADING)
    builderScope                            ,
#endif
    clearBuilderScope                       ,
    constructBuilderScope                   ,
    getBuilderScope                         ,
    setBuilderScope                         ,


-- ** translationDomain #attr:translationDomain#
-- | The translation domain used when translating property values that
-- have been marked as translatable.
-- 
-- If the translation domain is 'P.Nothing', @GtkBuilder@ uses @/gettext()/@,
-- otherwise 'GI.GLib.Functions.dgettext'.

#if defined(ENABLE_OVERLOADING)
    BuilderTranslationDomainPropertyInfo    ,
#endif
#if defined(ENABLE_OVERLOADING)
    builderTranslationDomain                ,
#endif
    clearBuilderTranslationDomain           ,
    constructBuilderTranslationDomain       ,
    getBuilderTranslationDomain             ,
    setBuilderTranslationDomain             ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.BuilderScope as Gtk.BuilderScope

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.BuilderScope as Gtk.BuilderScope

#endif

-- | Memory-managed wrapper type.
newtype Builder = Builder (SP.ManagedPtr Builder)
    deriving (Eq)

instance SP.ManagedPtrNewtype Builder where
    toManagedPtr (Builder p) = p

foreign import ccall "gtk_builder_get_type"
    c_gtk_builder_get_type :: IO B.Types.GType

instance B.Types.TypedObject Builder where
    glibType = c_gtk_builder_get_type

instance B.Types.GObject Builder

-- | Type class for types which can be safely cast to t'Builder', for instance with `toBuilder`.
class (SP.GObject o, O.IsDescendantOf Builder o) => IsBuilder o
instance (SP.GObject o, O.IsDescendantOf Builder o) => IsBuilder o

instance O.HasParentTypes Builder
type instance O.ParentTypes Builder = '[GObject.Object.Object]

-- | Cast to t'Builder', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toBuilder :: (MIO.MonadIO m, IsBuilder o) => o -> m Builder
toBuilder = MIO.liftIO . B.ManagedPtr.unsafeCastTo Builder

-- | Convert t'Builder' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Builder) where
    gvalueGType_ = c_gtk_builder_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Builder)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Builder)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Builder ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveBuilderMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveBuilderMethod "addFromFile" o = BuilderAddFromFileMethodInfo
    ResolveBuilderMethod "addFromResource" o = BuilderAddFromResourceMethodInfo
    ResolveBuilderMethod "addFromString" o = BuilderAddFromStringMethodInfo
    ResolveBuilderMethod "addObjectsFromFile" o = BuilderAddObjectsFromFileMethodInfo
    ResolveBuilderMethod "addObjectsFromResource" o = BuilderAddObjectsFromResourceMethodInfo
    ResolveBuilderMethod "addObjectsFromString" o = BuilderAddObjectsFromStringMethodInfo
    ResolveBuilderMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveBuilderMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveBuilderMethod "createClosure" o = BuilderCreateClosureMethodInfo
    ResolveBuilderMethod "exposeObject" o = BuilderExposeObjectMethodInfo
    ResolveBuilderMethod "extendWithTemplate" o = BuilderExtendWithTemplateMethodInfo
    ResolveBuilderMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveBuilderMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveBuilderMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveBuilderMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveBuilderMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveBuilderMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveBuilderMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveBuilderMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveBuilderMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveBuilderMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveBuilderMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveBuilderMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveBuilderMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveBuilderMethod "valueFromString" o = BuilderValueFromStringMethodInfo
    ResolveBuilderMethod "valueFromStringType" o = BuilderValueFromStringTypeMethodInfo
    ResolveBuilderMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveBuilderMethod "getCurrentObject" o = BuilderGetCurrentObjectMethodInfo
    ResolveBuilderMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveBuilderMethod "getObject" o = BuilderGetObjectMethodInfo
    ResolveBuilderMethod "getObjects" o = BuilderGetObjectsMethodInfo
    ResolveBuilderMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveBuilderMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveBuilderMethod "getScope" o = BuilderGetScopeMethodInfo
    ResolveBuilderMethod "getTranslationDomain" o = BuilderGetTranslationDomainMethodInfo
    ResolveBuilderMethod "getTypeFromName" o = BuilderGetTypeFromNameMethodInfo
    ResolveBuilderMethod "setCurrentObject" o = BuilderSetCurrentObjectMethodInfo
    ResolveBuilderMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveBuilderMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveBuilderMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveBuilderMethod "setScope" o = BuilderSetScopeMethodInfo
    ResolveBuilderMethod "setTranslationDomain" o = BuilderSetTranslationDomainMethodInfo
    ResolveBuilderMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveBuilderMethod t Builder, O.OverloadedMethod info Builder p) => OL.IsLabel t (Builder -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveBuilderMethod t Builder, O.OverloadedMethod info Builder p, R.HasField t Builder p) => R.HasField t Builder p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveBuilderMethod t Builder, O.OverloadedMethodInfo info Builder) => OL.IsLabel t (O.MethodProxy info Builder) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- VVV Prop "current-object"
   -- Type: TInterface (Name {namespace = "GObject", name = "Object"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@current-object@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' builder #currentObject
-- @
getBuilderCurrentObject :: (MonadIO m, IsBuilder o) => o -> m (Maybe GObject.Object.Object)
getBuilderCurrentObject obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "current-object" GObject.Object.Object

-- | Set the value of the “@current-object@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' builder [ #currentObject 'Data.GI.Base.Attributes.:=' value ]
-- @
setBuilderCurrentObject :: (MonadIO m, IsBuilder o, GObject.Object.IsObject a) => o -> a -> m ()
setBuilderCurrentObject obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyObject obj "current-object" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@current-object@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructBuilderCurrentObject :: (IsBuilder o, MIO.MonadIO m, GObject.Object.IsObject a) => a -> m (GValueConstruct o)
constructBuilderCurrentObject val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "current-object" (P.Just val)

-- | Set the value of the “@current-object@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #currentObject
-- @
clearBuilderCurrentObject :: (MonadIO m, IsBuilder o) => o -> m ()
clearBuilderCurrentObject obj = liftIO $ B.Properties.setObjectPropertyObject obj "current-object" (Nothing :: Maybe GObject.Object.Object)

#if defined(ENABLE_OVERLOADING)
data BuilderCurrentObjectPropertyInfo
instance AttrInfo BuilderCurrentObjectPropertyInfo where
    type AttrAllowedOps BuilderCurrentObjectPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint BuilderCurrentObjectPropertyInfo = IsBuilder
    type AttrSetTypeConstraint BuilderCurrentObjectPropertyInfo = GObject.Object.IsObject
    type AttrTransferTypeConstraint BuilderCurrentObjectPropertyInfo = GObject.Object.IsObject
    type AttrTransferType BuilderCurrentObjectPropertyInfo = GObject.Object.Object
    type AttrGetType BuilderCurrentObjectPropertyInfo = (Maybe GObject.Object.Object)
    type AttrLabel BuilderCurrentObjectPropertyInfo = "current-object"
    type AttrOrigin BuilderCurrentObjectPropertyInfo = Builder
    attrGet = getBuilderCurrentObject
    attrSet = setBuilderCurrentObject
    attrPut = undefined
    attrTransfer _ v = do
        unsafeCastTo GObject.Object.Object v
    attrConstruct = constructBuilderCurrentObject
    attrClear = clearBuilderCurrentObject
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.currentObject"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#g:attr:currentObject"
        })
#endif

-- VVV Prop "scope"
   -- Type: TInterface (Name {namespace = "Gtk", name = "BuilderScope"})
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstruct]
   -- Nullable: (Just False,Just True)

-- | Get the value of the “@scope@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' builder #scope
-- @
getBuilderScope :: (MonadIO m, IsBuilder o) => o -> m Gtk.BuilderScope.BuilderScope
getBuilderScope obj = MIO.liftIO $ checkUnexpectedNothing "getBuilderScope" $ B.Properties.getObjectPropertyObject obj "scope" Gtk.BuilderScope.BuilderScope

-- | Set the value of the “@scope@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' builder [ #scope 'Data.GI.Base.Attributes.:=' value ]
-- @
setBuilderScope :: (MonadIO m, IsBuilder o, Gtk.BuilderScope.IsBuilderScope a) => o -> a -> m ()
setBuilderScope obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyObject obj "scope" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@scope@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructBuilderScope :: (IsBuilder o, MIO.MonadIO m, Gtk.BuilderScope.IsBuilderScope a) => a -> m (GValueConstruct o)
constructBuilderScope val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "scope" (P.Just val)

-- | Set the value of the “@scope@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #scope
-- @
clearBuilderScope :: (MonadIO m, IsBuilder o) => o -> m ()
clearBuilderScope obj = liftIO $ B.Properties.setObjectPropertyObject obj "scope" (Nothing :: Maybe Gtk.BuilderScope.BuilderScope)

#if defined(ENABLE_OVERLOADING)
data BuilderScopePropertyInfo
instance AttrInfo BuilderScopePropertyInfo where
    type AttrAllowedOps BuilderScopePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint BuilderScopePropertyInfo = IsBuilder
    type AttrSetTypeConstraint BuilderScopePropertyInfo = Gtk.BuilderScope.IsBuilderScope
    type AttrTransferTypeConstraint BuilderScopePropertyInfo = Gtk.BuilderScope.IsBuilderScope
    type AttrTransferType BuilderScopePropertyInfo = Gtk.BuilderScope.BuilderScope
    type AttrGetType BuilderScopePropertyInfo = Gtk.BuilderScope.BuilderScope
    type AttrLabel BuilderScopePropertyInfo = "scope"
    type AttrOrigin BuilderScopePropertyInfo = Builder
    attrGet = getBuilderScope
    attrSet = setBuilderScope
    attrPut = undefined
    attrTransfer _ v = do
        unsafeCastTo Gtk.BuilderScope.BuilderScope v
    attrConstruct = constructBuilderScope
    attrClear = clearBuilderScope
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.scope"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#g:attr:scope"
        })
#endif

-- VVV Prop "translation-domain"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@translation-domain@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' builder #translationDomain
-- @
getBuilderTranslationDomain :: (MonadIO m, IsBuilder o) => o -> m (Maybe T.Text)
getBuilderTranslationDomain obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "translation-domain"

-- | Set the value of the “@translation-domain@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' builder [ #translationDomain 'Data.GI.Base.Attributes.:=' value ]
-- @
setBuilderTranslationDomain :: (MonadIO m, IsBuilder o) => o -> T.Text -> m ()
setBuilderTranslationDomain obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "translation-domain" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@translation-domain@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructBuilderTranslationDomain :: (IsBuilder o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructBuilderTranslationDomain val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "translation-domain" (P.Just val)

-- | Set the value of the “@translation-domain@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #translationDomain
-- @
clearBuilderTranslationDomain :: (MonadIO m, IsBuilder o) => o -> m ()
clearBuilderTranslationDomain obj = liftIO $ B.Properties.setObjectPropertyString obj "translation-domain" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data BuilderTranslationDomainPropertyInfo
instance AttrInfo BuilderTranslationDomainPropertyInfo where
    type AttrAllowedOps BuilderTranslationDomainPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint BuilderTranslationDomainPropertyInfo = IsBuilder
    type AttrSetTypeConstraint BuilderTranslationDomainPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint BuilderTranslationDomainPropertyInfo = (~) T.Text
    type AttrTransferType BuilderTranslationDomainPropertyInfo = T.Text
    type AttrGetType BuilderTranslationDomainPropertyInfo = (Maybe T.Text)
    type AttrLabel BuilderTranslationDomainPropertyInfo = "translation-domain"
    type AttrOrigin BuilderTranslationDomainPropertyInfo = Builder
    attrGet = getBuilderTranslationDomain
    attrSet = setBuilderTranslationDomain
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructBuilderTranslationDomain
    attrClear = clearBuilderTranslationDomain
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.translationDomain"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#g:attr:translationDomain"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Builder
type instance O.AttributeList Builder = BuilderAttributeList
type BuilderAttributeList = ('[ '("currentObject", BuilderCurrentObjectPropertyInfo), '("scope", BuilderScopePropertyInfo), '("translationDomain", BuilderTranslationDomainPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
builderCurrentObject :: AttrLabelProxy "currentObject"
builderCurrentObject = AttrLabelProxy

builderScope :: AttrLabelProxy "scope"
builderScope = AttrLabelProxy

builderTranslationDomain :: AttrLabelProxy "translationDomain"
builderTranslationDomain = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Builder = BuilderSignalList
type BuilderSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Builder::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Builder" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_new" gtk_builder_new :: 
    IO (Ptr Builder)

-- | Creates a new empty builder object.
-- 
-- This function is only useful if you intend to make multiple calls
-- to 'GI.Gtk.Objects.Builder.builderAddFromFile', 'GI.Gtk.Objects.Builder.builderAddFromResource'
-- or 'GI.Gtk.Objects.Builder.builderAddFromString' in order to merge multiple UI
-- descriptions into a single builder.
builderNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Builder
    -- ^ __Returns:__ a new (empty) @GtkBuilder@ object
builderNew  = liftIO $ do
    result <- gtk_builder_new
    checkUnexpectedReturnNULL "builderNew" result
    result' <- (wrapObject Builder) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Builder::new_from_file
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "filename"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "filename of user interface description file"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Builder" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_new_from_file" gtk_builder_new_from_file :: 
    CString ->                              -- filename : TBasicType TFileName
    IO (Ptr Builder)

-- | Parses the UI definition in the file /@filename@/.
-- 
-- If there is an error opening the file or parsing the description then
-- the program will be aborted. You should only ever attempt to parse
-- user interface descriptions that are shipped as part of your program.
builderNewFromFile ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    [Char]
    -- ^ /@filename@/: filename of user interface description file
    -> m Builder
    -- ^ __Returns:__ a @GtkBuilder@ containing the described interface
builderNewFromFile filename = liftIO $ do
    filename' <- stringToCString filename
    result <- gtk_builder_new_from_file filename'
    checkUnexpectedReturnNULL "builderNewFromFile" result
    result' <- (wrapObject Builder) result
    freeMem filename'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Builder::new_from_resource
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "resource_path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GResource` resource path"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Builder" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_new_from_resource" gtk_builder_new_from_resource :: 
    CString ->                              -- resource_path : TBasicType TUTF8
    IO (Ptr Builder)

-- | Parses the UI definition at /@resourcePath@/.
-- 
-- If there is an error locating the resource or parsing the
-- description, then the program will be aborted.
builderNewFromResource ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@resourcePath@/: a @GResource@ resource path
    -> m Builder
    -- ^ __Returns:__ a @GtkBuilder@ containing the described interface
builderNewFromResource resourcePath = liftIO $ do
    resourcePath' <- textToCString resourcePath
    result <- gtk_builder_new_from_resource resourcePath'
    checkUnexpectedReturnNULL "builderNewFromResource" result
    result' <- (wrapObject Builder) result
    freeMem resourcePath'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Builder::new_from_string
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a user interface (XML) description"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of @string, or -1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Builder" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_new_from_string" gtk_builder_new_from_string :: 
    CString ->                              -- string : TBasicType TUTF8
    DI.Int64 ->                             -- length : TBasicType TSSize
    IO (Ptr Builder)

-- | Parses the UI definition in /@string@/.
-- 
-- If /@string@/ is 'P.Nothing'-terminated, then /@length@/ should be -1.
-- If /@length@/ is not -1, then it is the length of /@string@/.
-- 
-- If there is an error parsing /@string@/ then the program will be
-- aborted. You should not attempt to parse user interface description
-- from untrusted sources.
builderNewFromString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@string@/: a user interface (XML) description
    -> DI.Int64
    -- ^ /@length@/: the length of /@string@/, or -1
    -> m Builder
    -- ^ __Returns:__ a @GtkBuilder@ containing the interface described by /@string@/
builderNewFromString string length_ = liftIO $ do
    string' <- textToCString string
    result <- gtk_builder_new_from_string string' length_
    checkUnexpectedReturnNULL "builderNewFromString" result
    result' <- (wrapObject Builder) result
    freeMem string'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Builder::add_from_file
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "filename"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the file to parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gtk_builder_add_from_file" gtk_builder_add_from_file :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- filename : TBasicType TFileName
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses a file containing a UI definition and merges it with
-- the current contents of /@builder@/.
-- 
-- This function is useful if you need to call
-- 'GI.Gtk.Objects.Builder.builderSetCurrentObject') to add user data to
-- callbacks before loading GtkBuilder UI. Otherwise, you probably
-- want 'GI.Gtk.Objects.Builder.builderNewFromFile' instead.
-- 
-- If an error occurs, 0 will be returned and /@error@/ will be assigned a
-- @GError@ from the @GTK_BUILDER_ERROR@, @G_MARKUP_ERROR@ or @G_FILE_ERROR@
-- domains.
-- 
-- It’s not really reasonable to attempt to handle failures of this
-- call. You should not use this function with untrusted files (ie:
-- files that are not part of your application). Broken @GtkBuilder@
-- files can easily crash your program, and it’s possible that memory
-- was leaked leading up to the reported failure. The only reasonable
-- thing to do when an error is detected is to call @g_error()@.
builderAddFromFile ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> [Char]
    -- ^ /@filename@/: the name of the file to parse
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
builderAddFromFile builder filename = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    filename' <- stringToCString filename
    onException (do
        _ <- propagateGError $ gtk_builder_add_from_file builder' filename'
        touchManagedPtr builder
        freeMem filename'
        return ()
     ) (do
        freeMem filename'
     )

#if defined(ENABLE_OVERLOADING)
data BuilderAddFromFileMethodInfo
instance (signature ~ ([Char] -> m ()), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderAddFromFileMethodInfo a signature where
    overloadedMethod = builderAddFromFile

instance O.OverloadedMethodInfo BuilderAddFromFileMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderAddFromFile",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderAddFromFile"
        })


#endif

-- method Builder::add_from_resource
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "resource_path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path of the resource file to parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gtk_builder_add_from_resource" gtk_builder_add_from_resource :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- resource_path : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses a resource file containing a UI definition
-- and merges it with the current contents of /@builder@/.
-- 
-- This function is useful if you need to call
-- 'GI.Gtk.Objects.Builder.builderSetCurrentObject' to add user data to
-- callbacks before loading GtkBuilder UI. Otherwise, you probably
-- want 'GI.Gtk.Objects.Builder.builderNewFromResource' instead.
-- 
-- If an error occurs, 0 will be returned and /@error@/ will be assigned a
-- @GError@ from the @/GTK_BUILDER_ERROR/@, @/G_MARKUP_ERROR/@ or @/G_RESOURCE_ERROR/@
-- domain.
-- 
-- It’s not really reasonable to attempt to handle failures of this
-- call.  The only reasonable thing to do when an error is detected is
-- to call @/g_error()/@.
builderAddFromResource ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> T.Text
    -- ^ /@resourcePath@/: the path of the resource file to parse
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
builderAddFromResource builder resourcePath = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    resourcePath' <- textToCString resourcePath
    onException (do
        _ <- propagateGError $ gtk_builder_add_from_resource builder' resourcePath'
        touchManagedPtr builder
        freeMem resourcePath'
        return ()
     ) (do
        freeMem resourcePath'
     )

#if defined(ENABLE_OVERLOADING)
data BuilderAddFromResourceMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderAddFromResourceMethodInfo a signature where
    overloadedMethod = builderAddFromResource

instance O.OverloadedMethodInfo BuilderAddFromResourceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderAddFromResource",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderAddFromResource"
        })


#endif

-- method Builder::add_from_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "buffer"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @buffer (may be -1 if @buffer is nul-terminated)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gtk_builder_add_from_string" gtk_builder_add_from_string :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- buffer : TBasicType TUTF8
    DI.Int64 ->                             -- length : TBasicType TSSize
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses a string containing a UI definition and merges it
-- with the current contents of /@builder@/.
-- 
-- This function is useful if you need to call
-- 'GI.Gtk.Objects.Builder.builderSetCurrentObject' to add user data to
-- callbacks before loading @GtkBuilder@ UI. Otherwise, you probably
-- want 'GI.Gtk.Objects.Builder.builderNewFromString' instead.
-- 
-- Upon errors 'P.False' will be returned and /@error@/ will be assigned a
-- @GError@ from the @/GTK_BUILDER_ERROR/@, @/G_MARKUP_ERROR/@ or
-- @/G_VARIANT_PARSE_ERROR/@ domain.
-- 
-- It’s not really reasonable to attempt to handle failures of this
-- call.  The only reasonable thing to do when an error is detected is
-- to call @/g_error()/@.
builderAddFromString ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> T.Text
    -- ^ /@buffer@/: the string to parse
    -> DI.Int64
    -- ^ /@length@/: the length of /@buffer@/ (may be -1 if /@buffer@/ is nul-terminated)
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
builderAddFromString builder buffer length_ = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    buffer' <- textToCString buffer
    onException (do
        _ <- propagateGError $ gtk_builder_add_from_string builder' buffer' length_
        touchManagedPtr builder
        freeMem buffer'
        return ()
     ) (do
        freeMem buffer'
     )

#if defined(ENABLE_OVERLOADING)
data BuilderAddFromStringMethodInfo
instance (signature ~ (T.Text -> DI.Int64 -> m ()), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderAddFromStringMethodInfo a signature where
    overloadedMethod = builderAddFromString

instance O.OverloadedMethodInfo BuilderAddFromStringMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderAddFromString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderAddFromString"
        })


#endif

-- method Builder::add_objects_from_file
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "filename"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the file to parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object_ids"
--           , argType = TCArray True (-1) (-1) (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "nul-terminated array of objects to build"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gtk_builder_add_objects_from_file" gtk_builder_add_objects_from_file :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- filename : TBasicType TFileName
    Ptr CString ->                          -- object_ids : TCArray True (-1) (-1) (TBasicType TUTF8)
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses a file containing a UI definition building only the
-- requested objects and merges them with the current contents
-- of /@builder@/.
-- 
-- Upon errors, 0 will be returned and /@error@/ will be assigned a
-- @GError@ from the @/GTK_BUILDER_ERROR/@, @/G_MARKUP_ERROR/@ or @/G_FILE_ERROR/@
-- domain.
-- 
-- If you are adding an object that depends on an object that is not
-- its child (for instance a @GtkTreeView@ that depends on its
-- @GtkTreeModel@), you have to explicitly list all of them in /@objectIds@/.
builderAddObjectsFromFile ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> [Char]
    -- ^ /@filename@/: the name of the file to parse
    -> [T.Text]
    -- ^ /@objectIds@/: nul-terminated array of objects to build
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
builderAddObjectsFromFile builder filename objectIds = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    filename' <- stringToCString filename
    objectIds' <- packZeroTerminatedUTF8CArray objectIds
    onException (do
        _ <- propagateGError $ gtk_builder_add_objects_from_file builder' filename' objectIds'
        touchManagedPtr builder
        freeMem filename'
        mapZeroTerminatedCArray freeMem objectIds'
        freeMem objectIds'
        return ()
     ) (do
        freeMem filename'
        mapZeroTerminatedCArray freeMem objectIds'
        freeMem objectIds'
     )

#if defined(ENABLE_OVERLOADING)
data BuilderAddObjectsFromFileMethodInfo
instance (signature ~ ([Char] -> [T.Text] -> m ()), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderAddObjectsFromFileMethodInfo a signature where
    overloadedMethod = builderAddObjectsFromFile

instance O.OverloadedMethodInfo BuilderAddObjectsFromFileMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderAddObjectsFromFile",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderAddObjectsFromFile"
        })


#endif

-- method Builder::add_objects_from_resource
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "resource_path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the path of the resource file to parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object_ids"
--           , argType = TCArray True (-1) (-1) (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "nul-terminated array of objects to build"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gtk_builder_add_objects_from_resource" gtk_builder_add_objects_from_resource :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- resource_path : TBasicType TUTF8
    Ptr CString ->                          -- object_ids : TCArray True (-1) (-1) (TBasicType TUTF8)
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses a resource file containing a UI definition, building
-- only the requested objects and merges them with the current
-- contents of /@builder@/.
-- 
-- Upon errors, 0 will be returned and /@error@/ will be assigned a
-- @GError@ from the @/GTK_BUILDER_ERROR/@, @/G_MARKUP_ERROR/@ or @/G_RESOURCE_ERROR/@
-- domain.
-- 
-- If you are adding an object that depends on an object that is not
-- its child (for instance a @GtkTreeView@ that depends on its
-- @GtkTreeModel@), you have to explicitly list all of them in /@objectIds@/.
builderAddObjectsFromResource ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> T.Text
    -- ^ /@resourcePath@/: the path of the resource file to parse
    -> [T.Text]
    -- ^ /@objectIds@/: nul-terminated array of objects to build
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
builderAddObjectsFromResource builder resourcePath objectIds = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    resourcePath' <- textToCString resourcePath
    objectIds' <- packZeroTerminatedUTF8CArray objectIds
    onException (do
        _ <- propagateGError $ gtk_builder_add_objects_from_resource builder' resourcePath' objectIds'
        touchManagedPtr builder
        freeMem resourcePath'
        mapZeroTerminatedCArray freeMem objectIds'
        freeMem objectIds'
        return ()
     ) (do
        freeMem resourcePath'
        mapZeroTerminatedCArray freeMem objectIds'
        freeMem objectIds'
     )

#if defined(ENABLE_OVERLOADING)
data BuilderAddObjectsFromResourceMethodInfo
instance (signature ~ (T.Text -> [T.Text] -> m ()), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderAddObjectsFromResourceMethodInfo a signature where
    overloadedMethod = builderAddObjectsFromResource

instance O.OverloadedMethodInfo BuilderAddObjectsFromResourceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderAddObjectsFromResource",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderAddObjectsFromResource"
        })


#endif

-- method Builder::add_objects_from_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "buffer"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @buffer (may be -1 if @buffer is nul-terminated)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object_ids"
--           , argType = TCArray True (-1) (-1) (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "nul-terminated array of objects to build"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gtk_builder_add_objects_from_string" gtk_builder_add_objects_from_string :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- buffer : TBasicType TUTF8
    DI.Int64 ->                             -- length : TBasicType TSSize
    Ptr CString ->                          -- object_ids : TCArray True (-1) (-1) (TBasicType TUTF8)
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Parses a string containing a UI definition, building only the
-- requested objects and merges them with the current contents of
-- /@builder@/.
-- 
-- Upon errors 'P.False' will be returned and /@error@/ will be assigned a
-- @GError@ from the @/GTK_BUILDER_ERROR/@ or @/G_MARKUP_ERROR/@ domain.
-- 
-- If you are adding an object that depends on an object that is not
-- its child (for instance a @GtkTreeView@ that depends on its
-- @GtkTreeModel@), you have to explicitly list all of them in /@objectIds@/.
builderAddObjectsFromString ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> T.Text
    -- ^ /@buffer@/: the string to parse
    -> DI.Int64
    -- ^ /@length@/: the length of /@buffer@/ (may be -1 if /@buffer@/ is nul-terminated)
    -> [T.Text]
    -- ^ /@objectIds@/: nul-terminated array of objects to build
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
builderAddObjectsFromString builder buffer length_ objectIds = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    buffer' <- textToCString buffer
    objectIds' <- packZeroTerminatedUTF8CArray objectIds
    onException (do
        _ <- propagateGError $ gtk_builder_add_objects_from_string builder' buffer' length_ objectIds'
        touchManagedPtr builder
        freeMem buffer'
        mapZeroTerminatedCArray freeMem objectIds'
        freeMem objectIds'
        return ()
     ) (do
        freeMem buffer'
        mapZeroTerminatedCArray freeMem objectIds'
        freeMem objectIds'
     )

#if defined(ENABLE_OVERLOADING)
data BuilderAddObjectsFromStringMethodInfo
instance (signature ~ (T.Text -> DI.Int64 -> [T.Text] -> m ()), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderAddObjectsFromStringMethodInfo a signature where
    overloadedMethod = builderAddObjectsFromString

instance O.OverloadedMethodInfo BuilderAddObjectsFromStringMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderAddObjectsFromString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderAddObjectsFromString"
        })


#endif

-- method Builder::create_closure
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "function_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "name of the function to look up"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface
--                 Name { namespace = "Gtk" , name = "BuilderClosureFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "closure creation flags"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Object to create the closure with"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TGClosure Nothing)
-- throws : True
-- Skip return : False

foreign import ccall "gtk_builder_create_closure" gtk_builder_create_closure :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- function_name : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gtk", name = "BuilderClosureFlags"})
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr (GClosure ()))

-- | Creates a closure to invoke the function called /@functionName@/.
-- 
-- This is using the @/create_closure()/@ implementation of /@builder@/\'s
-- t'GI.Gtk.Interfaces.BuilderScope.BuilderScope'.
-- 
-- If no closure could be created, 'P.Nothing' will be returned and /@error@/
-- will be set.
builderCreateClosure ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a, GObject.Object.IsObject b) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> T.Text
    -- ^ /@functionName@/: name of the function to look up
    -> [Gtk.Flags.BuilderClosureFlags]
    -- ^ /@flags@/: closure creation flags
    -> Maybe (b)
    -- ^ /@object@/: Object to create the closure with
    -> m (Maybe (GClosure c))
    -- ^ __Returns:__ A new closure for invoking /@functionName@/ /(Can throw 'Data.GI.Base.GError.GError')/
builderCreateClosure builder functionName flags object = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    functionName' <- textToCString functionName
    let flags' = gflagsToWord flags
    maybeObject <- case object of
        Nothing -> return FP.nullPtr
        Just jObject -> do
            jObject' <- unsafeManagedPtrCastPtr jObject
            return jObject'
    onException (do
        result <- propagateGError $ gtk_builder_create_closure builder' functionName' flags' maybeObject
        maybeResult <- convertIfNonNull result $ \result' -> do
            result'' <- (B.GClosure.wrapGClosurePtr . FP.castPtr) result'
            return result''
        touchManagedPtr builder
        whenJust object touchManagedPtr
        freeMem functionName'
        return maybeResult
     ) (do
        freeMem functionName'
     )

#if defined(ENABLE_OVERLOADING)
data BuilderCreateClosureMethodInfo
instance (signature ~ (T.Text -> [Gtk.Flags.BuilderClosureFlags] -> Maybe (b) -> m (Maybe (GClosure c))), MonadIO m, IsBuilder a, GObject.Object.IsObject b) => O.OverloadedMethod BuilderCreateClosureMethodInfo a signature where
    overloadedMethod = builderCreateClosure

instance O.OverloadedMethodInfo BuilderCreateClosureMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderCreateClosure",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderCreateClosure"
        })


#endif

-- method Builder::expose_object
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the object exposed to the builder"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the object to expose"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_expose_object" gtk_builder_expose_object :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- name : TBasicType TUTF8
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    IO ()

-- | Add /@object@/ to the /@builder@/ object pool so it can be
-- referenced just like any other object built by builder.
-- 
-- Only a single object may be added using /@name@/. However,
-- it is not an error to expose the same object under multiple
-- names. @gtk_builder_get_object()@ may be used to determine
-- if an object has already been added with /@name@/.
builderExposeObject ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a, GObject.Object.IsObject b) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> T.Text
    -- ^ /@name@/: the name of the object exposed to the builder
    -> b
    -- ^ /@object@/: the object to expose
    -> m ()
builderExposeObject builder name object = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    name' <- textToCString name
    object' <- unsafeManagedPtrCastPtr object
    gtk_builder_expose_object builder' name' object'
    touchManagedPtr builder
    touchManagedPtr object
    freeMem name'
    return ()

#if defined(ENABLE_OVERLOADING)
data BuilderExposeObjectMethodInfo
instance (signature ~ (T.Text -> b -> m ()), MonadIO m, IsBuilder a, GObject.Object.IsObject b) => O.OverloadedMethod BuilderExposeObjectMethodInfo a signature where
    overloadedMethod = builderExposeObject

instance O.OverloadedMethodInfo BuilderExposeObjectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderExposeObject",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderExposeObject"
        })


#endif

-- method Builder::extend_with_template
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the object that is being extended"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "template_type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the type that the template is for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "buffer"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the length of @buffer (may be -1 if @buffer is nul-terminated)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gtk_builder_extend_with_template" gtk_builder_extend_with_template :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CGType ->                               -- template_type : TBasicType TGType
    CString ->                              -- buffer : TBasicType TUTF8
    DI.Int64 ->                             -- length : TBasicType TSSize
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Main private entry point for building composite components
-- from template XML.
-- 
-- Most likely you do not need to call this function in applications as
-- templates are handled by @GtkWidget@.
builderExtendWithTemplate ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a, GObject.Object.IsObject b) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> b
    -- ^ /@object@/: the object that is being extended
    -> GType
    -- ^ /@templateType@/: the type that the template is for
    -> T.Text
    -- ^ /@buffer@/: the string to parse
    -> DI.Int64
    -- ^ /@length@/: the length of /@buffer@/ (may be -1 if /@buffer@/ is nul-terminated)
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
builderExtendWithTemplate builder object templateType buffer length_ = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    object' <- unsafeManagedPtrCastPtr object
    let templateType' = gtypeToCGType templateType
    buffer' <- textToCString buffer
    onException (do
        _ <- propagateGError $ gtk_builder_extend_with_template builder' object' templateType' buffer' length_
        touchManagedPtr builder
        touchManagedPtr object
        freeMem buffer'
        return ()
     ) (do
        freeMem buffer'
     )

#if defined(ENABLE_OVERLOADING)
data BuilderExtendWithTemplateMethodInfo
instance (signature ~ (b -> GType -> T.Text -> DI.Int64 -> m ()), MonadIO m, IsBuilder a, GObject.Object.IsObject b) => O.OverloadedMethod BuilderExtendWithTemplateMethodInfo a signature where
    overloadedMethod = builderExtendWithTemplate

instance O.OverloadedMethodInfo BuilderExtendWithTemplateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderExtendWithTemplate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderExtendWithTemplate"
        })


#endif

-- method Builder::get_current_object
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GObject" , name = "Object" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_get_current_object" gtk_builder_get_current_object :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    IO (Ptr GObject.Object.Object)

-- | Gets the current object set via 'GI.Gtk.Objects.Builder.builderSetCurrentObject'.
builderGetCurrentObject ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> m (Maybe GObject.Object.Object)
    -- ^ __Returns:__ the current object
builderGetCurrentObject builder = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    result <- gtk_builder_get_current_object builder'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject GObject.Object.Object) result'
        return result''
    touchManagedPtr builder
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data BuilderGetCurrentObjectMethodInfo
instance (signature ~ (m (Maybe GObject.Object.Object)), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderGetCurrentObjectMethodInfo a signature where
    overloadedMethod = builderGetCurrentObject

instance O.OverloadedMethodInfo BuilderGetCurrentObjectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderGetCurrentObject",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderGetCurrentObject"
        })


#endif

-- method Builder::get_object
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "name of object to get"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GObject" , name = "Object" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_get_object" gtk_builder_get_object :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- name : TBasicType TUTF8
    IO (Ptr GObject.Object.Object)

-- | Gets the object named /@name@/.
-- 
-- Note that this function does not increment the reference count
-- of the returned object.
builderGetObject ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> T.Text
    -- ^ /@name@/: name of object to get
    -> m (Maybe GObject.Object.Object)
    -- ^ __Returns:__ the object named /@name@/
builderGetObject builder name = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    name' <- textToCString name
    result <- gtk_builder_get_object builder' name'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject GObject.Object.Object) result'
        return result''
    touchManagedPtr builder
    freeMem name'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data BuilderGetObjectMethodInfo
instance (signature ~ (T.Text -> m (Maybe GObject.Object.Object)), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderGetObjectMethodInfo a signature where
    overloadedMethod = builderGetObject

instance O.OverloadedMethodInfo BuilderGetObjectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderGetObject",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderGetObject"
        })


#endif

-- method Builder::get_objects
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TGSList
--                  (TInterface Name { namespace = "GObject" , name = "Object" }))
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_get_objects" gtk_builder_get_objects :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    IO (Ptr (GSList (Ptr GObject.Object.Object)))

-- | Gets all objects that have been constructed by /@builder@/.
-- 
-- Note that this function does not increment the reference
-- counts of the returned objects.
builderGetObjects ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> m [GObject.Object.Object]
    -- ^ __Returns:__ a
    --   newly-allocated @GSList@ containing all the objects
    --   constructed by the @GtkBuilder instance@. It should be
    --   freed by @/g_slist_free()/@
builderGetObjects builder = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    result <- gtk_builder_get_objects builder'
    result' <- unpackGSList result
    result'' <- mapM (newObject GObject.Object.Object) result'
    g_slist_free result
    touchManagedPtr builder
    return result''

#if defined(ENABLE_OVERLOADING)
data BuilderGetObjectsMethodInfo
instance (signature ~ (m [GObject.Object.Object]), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderGetObjectsMethodInfo a signature where
    overloadedMethod = builderGetObjects

instance O.OverloadedMethodInfo BuilderGetObjectsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderGetObjects",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderGetObjects"
        })


#endif

-- method Builder::get_scope
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "BuilderScope" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_get_scope" gtk_builder_get_scope :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    IO (Ptr Gtk.BuilderScope.BuilderScope)

-- | Gets the scope in use that was set via 'GI.Gtk.Objects.Builder.builderSetScope'.
builderGetScope ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> m Gtk.BuilderScope.BuilderScope
    -- ^ __Returns:__ the current scope
builderGetScope builder = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    result <- gtk_builder_get_scope builder'
    checkUnexpectedReturnNULL "builderGetScope" result
    result' <- (newObject Gtk.BuilderScope.BuilderScope) result
    touchManagedPtr builder
    return result'

#if defined(ENABLE_OVERLOADING)
data BuilderGetScopeMethodInfo
instance (signature ~ (m Gtk.BuilderScope.BuilderScope), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderGetScopeMethodInfo a signature where
    overloadedMethod = builderGetScope

instance O.OverloadedMethodInfo BuilderGetScopeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderGetScope",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderGetScope"
        })


#endif

-- method Builder::get_translation_domain
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_get_translation_domain" gtk_builder_get_translation_domain :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    IO CString

-- | Gets the translation domain of /@builder@/.
builderGetTranslationDomain ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the translation domain
builderGetTranslationDomain builder = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    result <- gtk_builder_get_translation_domain builder'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr builder
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data BuilderGetTranslationDomainMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderGetTranslationDomainMethodInfo a signature where
    overloadedMethod = builderGetTranslationDomain

instance O.OverloadedMethodInfo BuilderGetTranslationDomainMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderGetTranslationDomain",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderGetTranslationDomain"
        })


#endif

-- method Builder::get_type_from_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "type_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "type name to lookup"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TGType)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_get_type_from_name" gtk_builder_get_type_from_name :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- type_name : TBasicType TUTF8
    IO CGType

-- | Looks up a type by name.
-- 
-- This is using the virtual function that @GtkBuilder@ has
-- for that purpose. This is mainly used when implementing
-- the @GtkBuildable@ interface on a type.
builderGetTypeFromName ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> T.Text
    -- ^ /@typeName@/: type name to lookup
    -> m GType
    -- ^ __Returns:__ the @GType@ found for /@typeName@/ or @/G_TYPE_INVALID/@
    --   if no type was found
builderGetTypeFromName builder typeName = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    typeName' <- textToCString typeName
    result <- gtk_builder_get_type_from_name builder' typeName'
    let result' = GType result
    touchManagedPtr builder
    freeMem typeName'
    return result'

#if defined(ENABLE_OVERLOADING)
data BuilderGetTypeFromNameMethodInfo
instance (signature ~ (T.Text -> m GType), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderGetTypeFromNameMethodInfo a signature where
    overloadedMethod = builderGetTypeFromName

instance O.OverloadedMethodInfo BuilderGetTypeFromNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderGetTypeFromName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderGetTypeFromName"
        })


#endif

-- method Builder::set_current_object
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "current_object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new current object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_set_current_object" gtk_builder_set_current_object :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    Ptr GObject.Object.Object ->            -- current_object : TInterface (Name {namespace = "GObject", name = "Object"})
    IO ()

-- | Sets the current object for the /@builder@/.
-- 
-- The current object can be thought of as the @this@ object that the
-- builder is working for and will often be used as the default object
-- when an object is optional.
-- 
-- 'GI.Gtk.Objects.Widget.widgetInitTemplate' for example will set the current
-- object to the widget the template is inited for. For functions like
-- 'GI.Gtk.Objects.Builder.builderNewFromResource', the current object will be 'P.Nothing'.
builderSetCurrentObject ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a, GObject.Object.IsObject b) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> Maybe (b)
    -- ^ /@currentObject@/: the new current object
    -> m ()
builderSetCurrentObject builder currentObject = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    maybeCurrentObject <- case currentObject of
        Nothing -> return FP.nullPtr
        Just jCurrentObject -> do
            jCurrentObject' <- unsafeManagedPtrCastPtr jCurrentObject
            return jCurrentObject'
    gtk_builder_set_current_object builder' maybeCurrentObject
    touchManagedPtr builder
    whenJust currentObject touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data BuilderSetCurrentObjectMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsBuilder a, GObject.Object.IsObject b) => O.OverloadedMethod BuilderSetCurrentObjectMethodInfo a signature where
    overloadedMethod = builderSetCurrentObject

instance O.OverloadedMethodInfo BuilderSetCurrentObjectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderSetCurrentObject",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderSetCurrentObject"
        })


#endif

-- method Builder::set_scope
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "scope"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "BuilderScope" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the scope to use" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_set_scope" gtk_builder_set_scope :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    Ptr Gtk.BuilderScope.BuilderScope ->    -- scope : TInterface (Name {namespace = "Gtk", name = "BuilderScope"})
    IO ()

-- | Sets the scope the builder should operate in.
-- 
-- If /@scope@/ is 'P.Nothing', a new t'GI.Gtk.Objects.BuilderCScope.BuilderCScope' will be created.
builderSetScope ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a, Gtk.BuilderScope.IsBuilderScope b) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> Maybe (b)
    -- ^ /@scope@/: the scope to use
    -> m ()
builderSetScope builder scope = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    maybeScope <- case scope of
        Nothing -> return FP.nullPtr
        Just jScope -> do
            jScope' <- unsafeManagedPtrCastPtr jScope
            return jScope'
    gtk_builder_set_scope builder' maybeScope
    touchManagedPtr builder
    whenJust scope touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data BuilderSetScopeMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsBuilder a, Gtk.BuilderScope.IsBuilderScope b) => O.OverloadedMethod BuilderSetScopeMethodInfo a signature where
    overloadedMethod = builderSetScope

instance O.OverloadedMethodInfo BuilderSetScopeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderSetScope",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderSetScope"
        })


#endif

-- method Builder::set_translation_domain
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "domain"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the translation domain"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_builder_set_translation_domain" gtk_builder_set_translation_domain :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CString ->                              -- domain : TBasicType TUTF8
    IO ()

-- | Sets the translation domain of /@builder@/.
builderSetTranslationDomain ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> Maybe (T.Text)
    -- ^ /@domain@/: the translation domain
    -> m ()
builderSetTranslationDomain builder domain = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    maybeDomain <- case domain of
        Nothing -> return FP.nullPtr
        Just jDomain -> do
            jDomain' <- textToCString jDomain
            return jDomain'
    gtk_builder_set_translation_domain builder' maybeDomain
    touchManagedPtr builder
    freeMem maybeDomain
    return ()

#if defined(ENABLE_OVERLOADING)
data BuilderSetTranslationDomainMethodInfo
instance (signature ~ (Maybe (T.Text) -> m ()), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderSetTranslationDomainMethodInfo a signature where
    overloadedMethod = builderSetTranslationDomain

instance O.OverloadedMethodInfo BuilderSetTranslationDomainMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderSetTranslationDomain",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderSetTranslationDomain"
        })


#endif

-- method Builder::value_from_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "pspec"
--           , argType = TParamSpec
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `GParamSpec` for the property"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string representation of the value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TGValue
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `GValue` to store the result in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gtk_builder_value_from_string" gtk_builder_value_from_string :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    Ptr GParamSpec ->                       -- pspec : TParamSpec
    CString ->                              -- string : TBasicType TUTF8
    Ptr GValue ->                           -- value : TGValue
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Demarshals a value from a string.
-- 
-- This function calls 'GI.GObject.Structs.Value.valueInit' on the /@value@/ argument,
-- so it need not be initialised beforehand.
-- 
-- Can handle char, uchar, boolean, int, uint, long,
-- ulong, enum, flags, float, double, string, @GdkRGBA@ and
-- @GtkAdjustment@ type values.
-- 
-- Upon errors 'P.False' will be returned and /@error@/ will be
-- assigned a @GError@ from the @/GTK_BUILDER_ERROR/@ domain.
builderValueFromString ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> GParamSpec
    -- ^ /@pspec@/: the @GParamSpec@ for the property
    -> T.Text
    -- ^ /@string@/: the string representation of the value
    -> m (GValue)
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
builderValueFromString builder pspec string = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    pspec' <- unsafeManagedPtrGetPtr pspec
    string' <- textToCString string
    value <- SP.callocBytes 24 :: IO (Ptr GValue)
    onException (do
        _ <- propagateGError $ gtk_builder_value_from_string builder' pspec' string' value
        value' <- B.GValue.wrapGValuePtr value
        touchManagedPtr builder
        touchManagedPtr pspec
        freeMem string'
        return value'
     ) (do
        freeMem string'
        freeMem value
     )

#if defined(ENABLE_OVERLOADING)
data BuilderValueFromStringMethodInfo
instance (signature ~ (GParamSpec -> T.Text -> m (GValue)), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderValueFromStringMethodInfo a signature where
    overloadedMethod = builderValueFromString

instance O.OverloadedMethodInfo BuilderValueFromStringMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderValueFromString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderValueFromString"
        })


#endif

-- method Builder::value_from_string_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "builder"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Builder" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkBuilder`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `GType` of the value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string representation of the value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TGValue
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `GValue` to store the result in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gtk_builder_value_from_string_type" gtk_builder_value_from_string_type :: 
    Ptr Builder ->                          -- builder : TInterface (Name {namespace = "Gtk", name = "Builder"})
    CGType ->                               -- type : TBasicType TGType
    CString ->                              -- string : TBasicType TUTF8
    Ptr GValue ->                           -- value : TGValue
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Demarshals a value from a string.
-- 
-- Unlike 'GI.Gtk.Objects.Builder.builderValueFromString', this function
-- takes a @GType@ instead of @GParamSpec@.
-- 
-- Calls 'GI.GObject.Structs.Value.valueInit' on the /@value@/ argument, so it
-- need not be initialised beforehand.
-- 
-- Upon errors 'P.False' will be returned and /@error@/ will be
-- assigned a @GError@ from the @/GTK_BUILDER_ERROR/@ domain.
builderValueFromStringType ::
    (B.CallStack.HasCallStack, MonadIO m, IsBuilder a) =>
    a
    -- ^ /@builder@/: a @GtkBuilder@
    -> GType
    -- ^ /@type@/: the @GType@ of the value
    -> T.Text
    -- ^ /@string@/: the string representation of the value
    -> m (GValue)
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
builderValueFromStringType builder type_ string = liftIO $ do
    builder' <- unsafeManagedPtrCastPtr builder
    let type_' = gtypeToCGType type_
    string' <- textToCString string
    value <- SP.callocBytes 24 :: IO (Ptr GValue)
    onException (do
        _ <- propagateGError $ gtk_builder_value_from_string_type builder' type_' string' value
        value' <- B.GValue.wrapGValuePtr value
        touchManagedPtr builder
        freeMem string'
        return value'
     ) (do
        freeMem string'
        freeMem value
     )

#if defined(ENABLE_OVERLOADING)
data BuilderValueFromStringTypeMethodInfo
instance (signature ~ (GType -> T.Text -> m (GValue)), MonadIO m, IsBuilder a) => O.OverloadedMethod BuilderValueFromStringTypeMethodInfo a signature where
    overloadedMethod = builderValueFromStringType

instance O.OverloadedMethodInfo BuilderValueFromStringTypeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Builder.builderValueFromStringType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Builder.html#v:builderValueFromStringType"
        })


#endif


