{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Displays a small amount of text.
-- 
-- Most labels are used to label another widget (such as an [class/@entry@/]).
-- 
-- \<picture>
--   \<source srcset=\"label-dark.png\" media=\"(prefers-color-scheme: dark)\">
--   \<img alt=\"An example GtkLabel\" src=\"label.png\">
-- \<\/picture>
-- 
-- == Shortcuts and Gestures
-- 
-- @GtkLabel@ supports the following keyboard shortcuts, when the cursor is
-- visible:
-- 
-- * \<kbd>Shift\<\/kbd>+\<kbd>F10\<\/kbd> or \<kbd>Menu\<\/kbd> opens the context menu.
-- * \<kbd>Ctrl\<\/kbd>+\<kbd>A\<\/kbd> or \<kbd>Ctrl\<\/kbd>+\<kbd>&sol;\<\/kbd>
--   selects all.
-- * \<kbd>Ctrl\<\/kbd>+\<kbd>Shift\<\/kbd>+\<kbd>A\<\/kbd> or
--   \<kbd>Ctrl\<\/kbd>+\<kbd>&bsol;\<\/kbd> unselects all.
-- 
-- 
-- Additionally, the following signals have default keybindings:
-- 
-- * [Label::activateCurrentLink]("GI.Gtk.Objects.Label#g:signal:activateCurrentLink")
-- * [Label::copyClipboard]("GI.Gtk.Objects.Label#g:signal:copyClipboard")
-- * [Label::moveCursor]("GI.Gtk.Objects.Label#g:signal:moveCursor")
-- 
-- 
-- == Actions
-- 
-- @GtkLabel@ defines a set of built-in actions:
-- 
-- * @clipboard.copy@ copies the text to the clipboard.
-- * @clipboard.cut@ doesn\'t do anything, since text in labels can\'t be deleted.
-- * @clipboard.paste@ doesn\'t do anything, since text in labels can\'t be
--   edited.
-- * @link.open@ opens the link, when activated on a link inside the label.
-- * @link.copy@ copies the link to the clipboard, when activated on a link
--   inside the label.
-- * @menu.popup@ opens the context menu.
-- * @selection.delete@ doesn\'t do anything, since text in labels can\'t be
--   deleted.
-- * @selection.select-all@ selects all of the text, if the label allows
--   selection.
-- 
-- 
-- == CSS nodes
-- 
-- 
-- 	
-- >label
-- >├── [selection]
-- >├── [link]
-- >┊
-- >╰── [link]
-- 
-- 
-- @GtkLabel@ has a single CSS node with the name label. A wide variety
-- of style classes may be applied to labels, such as .title, .subtitle,
-- .dim-label, etc. In the @GtkShortcutsWindow@, labels are used with the
-- .keycap style class.
-- 
-- If the label has a selection, it gets a subnode with name selection.
-- 
-- If the label has links, there is one subnode per link. These subnodes
-- carry the link or visited state depending on whether they have been
-- visited. In this case, label node also gets a .link style class.
-- 
-- == GtkLabel as GtkBuildable
-- 
-- The GtkLabel implementation of the GtkBuildable interface supports a
-- custom @\<attributes>@ element, which supports any number of @\<attribute>@
-- elements. The @\<attribute>@ element has attributes named “name“, “value“,
-- “start“ and “end“ and allows you to specify t'GI.Pango.Structs.Attribute.Attribute'
-- values for this label.
-- 
-- An example of a UI definition fragment specifying Pango attributes:
-- 
-- 
-- === /xml code/
-- ><object class="GtkLabel">
-- >  <attributes>
-- >    <attribute name="weight" value="PANGO_WEIGHT_BOLD"/>
-- >    <attribute name="background" value="red" start="5" end="10"/>
-- >  </attributes>
-- ></object>
-- 
-- 
-- The start and end attributes specify the range of characters to which the
-- Pango attribute applies. If start and end are not specified, the attribute is
-- applied to the whole text. Note that specifying ranges does not make much
-- sense with translatable attributes. Use markup embedded in the translatable
-- content instead.
-- 
-- == Accessibility
-- 
-- @GtkLabel@ uses the 'GI.Gtk.Enums.AccessibleRoleLabel' role.
-- 
-- == Mnemonics
-- 
-- Labels may contain “mnemonics”. Mnemonics are underlined characters in the
-- label, used for keyboard navigation. Mnemonics are created by providing a
-- string with an underscore before the mnemonic character, such as @\"_File\"@,
-- to the functions 'GI.Gtk.Objects.Label.labelNewWithMnemonic' or
-- 'GI.Gtk.Objects.Label.labelSetTextWithMnemonic'.
-- 
-- Mnemonics automatically activate any activatable widget the label is
-- inside, such as a t'GI.Gtk.Objects.Button.Button'; if the label is not inside the
-- mnemonic’s target widget, you have to tell the label about the target
-- using 'GI.Gtk.Objects.Label.labelSetMnemonicWidget'.
-- 
-- Here’s a simple example where the label is inside a button:
-- 
-- 
-- === /c code/
-- >// Pressing Alt+H will activate this button
-- >GtkWidget *button = gtk_button_new ();
-- >GtkWidget *label = gtk_label_new_with_mnemonic ("_Hello");
-- >gtk_button_set_child (GTK_BUTTON (button), label);
-- 
-- 
-- There’s a convenience function to create buttons with a mnemonic label
-- already inside:
-- 
-- 
-- === /c code/
-- >// Pressing Alt+H will activate this button
-- >GtkWidget *button = gtk_button_new_with_mnemonic ("_Hello");
-- 
-- 
-- To create a mnemonic for a widget alongside the label, such as a
-- t'GI.Gtk.Objects.Entry.Entry', you have to point the label at the entry with
-- 'GI.Gtk.Objects.Label.labelSetMnemonicWidget':
-- 
-- 
-- === /c code/
-- >// Pressing Alt+H will focus the entry
-- >GtkWidget *entry = gtk_entry_new ();
-- >GtkWidget *label = gtk_label_new_with_mnemonic ("_Hello");
-- >gtk_label_set_mnemonic_widget (GTK_LABEL (label), entry);
-- 
-- 
-- == Markup (styled text)
-- 
-- To make it easy to format text in a label (changing colors, fonts, etc.),
-- label text can be provided in a simple markup format:
-- 
-- Here’s how to create a label with a small font:
-- 
-- === /c code/
-- >GtkWidget *label = gtk_label_new (NULL);
-- >gtk_label_set_markup (GTK_LABEL (label), "<small>Small text</small>");
-- 
-- 
-- (See the Pango manual for complete documentation] of available
-- tags, 'GI.Pango.Functions.parseMarkup')
-- 
-- The markup passed to 'GI.Gtk.Objects.Label.labelSetMarkup' must be valid XML; for example,
-- literal @\<@, @>@ and @&@ characters must be escaped as @&lt;@, @&gt;@, and @&amp;@.
-- If you pass text obtained from the user, file, or a network to
-- 'GI.Gtk.Objects.Label.labelSetMarkup', you’ll want to escape it with
-- 'GI.GLib.Functions.markupEscapeText' or @/GLib.markup_printf_escaped/@.
-- 
-- Markup strings are just a convenient way to set the t'GI.Pango.Structs.AttrList.AttrList'
-- on a label; 'GI.Gtk.Objects.Label.labelSetAttributes' may be a simpler way to set
-- attributes in some cases. Be careful though; t'GI.Pango.Structs.AttrList.AttrList' tends
-- to cause internationalization problems, unless you’re applying attributes
-- to the entire string (i.e. unless you set the range of each attribute
-- to [0, @G_MAXINT@)). The reason is that specifying the @start_index@ and
-- @end_index@ for a t'GI.Pango.Structs.Attribute.Attribute' requires knowledge of the exact
-- string being displayed, so translations will cause problems.
-- 
-- == Selectable labels
-- 
-- Labels can be made selectable with 'GI.Gtk.Objects.Label.labelSetSelectable'.
-- Selectable labels allow the user to copy the label contents to the
-- clipboard. Only labels that contain useful-to-copy information — such
-- as error messages — should be made selectable.
-- 
-- == Text layout
-- 
-- A label can contain any number of paragraphs, but will have
-- performance problems if it contains more than a small number.
-- Paragraphs are separated by newlines or other paragraph separators
-- understood by Pango.
-- 
-- Labels can automatically wrap text if you call 'GI.Gtk.Objects.Label.labelSetWrap'.
-- 
-- 'GI.Gtk.Objects.Label.labelSetJustify' sets how the lines in a label align
-- with one another. If you want to set how the label as a whole aligns
-- in its available space, see the [Widget:halign]("GI.Gtk.Objects.Widget#g:attr:halign") and
-- [Widget:valign]("GI.Gtk.Objects.Widget#g:attr:valign") properties.
-- 
-- The [Label:widthChars]("GI.Gtk.Objects.Label#g:attr:widthChars") and [Label:maxWidthChars]("GI.Gtk.Objects.Label#g:attr:maxWidthChars")
-- properties can be used to control the size allocation of ellipsized or
-- wrapped labels. For ellipsizing labels, if either is specified (and less
-- than the actual text size), it is used as the minimum width, and the actual
-- text size is used as the natural width of the label. For wrapping labels,
-- width-chars is used as the minimum width, if specified, and max-width-chars
-- is used as the natural width. Even if max-width-chars specified, wrapping
-- labels will be rewrapped to use all of the available width.
-- 
-- == Links
-- 
-- GTK supports markup for clickable hyperlinks in addition to regular Pango
-- markup. The markup for links is borrowed from HTML, using the @\<a>@ tag
-- with “href“, “title“ and “class“ attributes. GTK renders links similar to
-- the way they appear in web browsers, with colored, underlined text. The
-- “title“ attribute is displayed as a tooltip on the link. The “class“
-- attribute is used as style class on the CSS node for the link.
-- 
-- An example of inline links looks like this:
-- 
-- 
-- === /c code/
-- >const char *text =
-- >"Go to the "
-- >"<a href=\"https://www.gtk.org\" title=\"&lt;i&gt;Our&lt;/i&gt; website\">"
-- >"GTK website</a> for more...";
-- >GtkWidget *label = gtk_label_new (NULL);
-- >gtk_label_set_markup (GTK_LABEL (label), text);
-- 
-- 
-- It is possible to implement custom handling for links and their tooltips
-- with the [Label::activateLink]("GI.Gtk.Objects.Label#g:signal:activateLink") signal and the
-- 'GI.Gtk.Objects.Label.labelGetCurrentUri' function.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.Label
    ( 

-- * Exported types
    Label(..)                               ,
    IsLabel                                 ,
    toLabel                                 ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [actionSetEnabled]("GI.Gtk.Objects.Widget#g:method:actionSetEnabled"), [activate]("GI.Gtk.Objects.Widget#g:method:activate"), [activateAction]("GI.Gtk.Objects.Widget#g:method:activateAction"), [activateDefault]("GI.Gtk.Objects.Widget#g:method:activateDefault"), [addController]("GI.Gtk.Objects.Widget#g:method:addController"), [addCssClass]("GI.Gtk.Objects.Widget#g:method:addCssClass"), [addMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:addMnemonicLabel"), [addTickCallback]("GI.Gtk.Objects.Widget#g:method:addTickCallback"), [allocate]("GI.Gtk.Objects.Widget#g:method:allocate"), [announce]("GI.Gtk.Interfaces.Accessible#g:method:announce"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [childFocus]("GI.Gtk.Objects.Widget#g:method:childFocus"), [computeBounds]("GI.Gtk.Objects.Widget#g:method:computeBounds"), [computeExpand]("GI.Gtk.Objects.Widget#g:method:computeExpand"), [computePoint]("GI.Gtk.Objects.Widget#g:method:computePoint"), [computeTransform]("GI.Gtk.Objects.Widget#g:method:computeTransform"), [contains]("GI.Gtk.Objects.Widget#g:method:contains"), [createPangoContext]("GI.Gtk.Objects.Widget#g:method:createPangoContext"), [createPangoLayout]("GI.Gtk.Objects.Widget#g:method:createPangoLayout"), [disposeTemplate]("GI.Gtk.Objects.Widget#g:method:disposeTemplate"), [dragCheckThreshold]("GI.Gtk.Objects.Widget#g:method:dragCheckThreshold"), [errorBell]("GI.Gtk.Objects.Widget#g:method:errorBell"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [grabFocus]("GI.Gtk.Objects.Widget#g:method:grabFocus"), [hasCssClass]("GI.Gtk.Objects.Widget#g:method:hasCssClass"), [hasDefault]("GI.Gtk.Objects.Widget#g:method:hasDefault"), [hasFocus]("GI.Gtk.Objects.Widget#g:method:hasFocus"), [hasVisibleFocus]("GI.Gtk.Objects.Widget#g:method:hasVisibleFocus"), [hide]("GI.Gtk.Objects.Widget#g:method:hide"), [inDestruction]("GI.Gtk.Objects.Widget#g:method:inDestruction"), [initTemplate]("GI.Gtk.Objects.Widget#g:method:initTemplate"), [insertActionGroup]("GI.Gtk.Objects.Widget#g:method:insertActionGroup"), [insertAfter]("GI.Gtk.Objects.Widget#g:method:insertAfter"), [insertBefore]("GI.Gtk.Objects.Widget#g:method:insertBefore"), [isAncestor]("GI.Gtk.Objects.Widget#g:method:isAncestor"), [isDrawable]("GI.Gtk.Objects.Widget#g:method:isDrawable"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isFocus]("GI.Gtk.Objects.Widget#g:method:isFocus"), [isSensitive]("GI.Gtk.Objects.Widget#g:method:isSensitive"), [isVisible]("GI.Gtk.Objects.Widget#g:method:isVisible"), [keynavFailed]("GI.Gtk.Objects.Widget#g:method:keynavFailed"), [listMnemonicLabels]("GI.Gtk.Objects.Widget#g:method:listMnemonicLabels"), [map]("GI.Gtk.Objects.Widget#g:method:map"), [measure]("GI.Gtk.Objects.Widget#g:method:measure"), [mnemonicActivate]("GI.Gtk.Objects.Widget#g:method:mnemonicActivate"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [observeChildren]("GI.Gtk.Objects.Widget#g:method:observeChildren"), [observeControllers]("GI.Gtk.Objects.Widget#g:method:observeControllers"), [pick]("GI.Gtk.Objects.Widget#g:method:pick"), [queueAllocate]("GI.Gtk.Objects.Widget#g:method:queueAllocate"), [queueDraw]("GI.Gtk.Objects.Widget#g:method:queueDraw"), [queueResize]("GI.Gtk.Objects.Widget#g:method:queueResize"), [realize]("GI.Gtk.Objects.Widget#g:method:realize"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [removeController]("GI.Gtk.Objects.Widget#g:method:removeController"), [removeCssClass]("GI.Gtk.Objects.Widget#g:method:removeCssClass"), [removeMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:removeMnemonicLabel"), [removeTickCallback]("GI.Gtk.Objects.Widget#g:method:removeTickCallback"), [resetProperty]("GI.Gtk.Interfaces.Accessible#g:method:resetProperty"), [resetRelation]("GI.Gtk.Interfaces.Accessible#g:method:resetRelation"), [resetState]("GI.Gtk.Interfaces.Accessible#g:method:resetState"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [selectRegion]("GI.Gtk.Objects.Label#g:method:selectRegion"), [shouldLayout]("GI.Gtk.Objects.Widget#g:method:shouldLayout"), [show]("GI.Gtk.Objects.Widget#g:method:show"), [sizeAllocate]("GI.Gtk.Objects.Widget#g:method:sizeAllocate"), [snapshotChild]("GI.Gtk.Objects.Widget#g:method:snapshotChild"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [translateCoordinates]("GI.Gtk.Objects.Widget#g:method:translateCoordinates"), [triggerTooltipQuery]("GI.Gtk.Objects.Widget#g:method:triggerTooltipQuery"), [unmap]("GI.Gtk.Objects.Widget#g:method:unmap"), [unparent]("GI.Gtk.Objects.Widget#g:method:unparent"), [unrealize]("GI.Gtk.Objects.Widget#g:method:unrealize"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [unsetStateFlags]("GI.Gtk.Objects.Widget#g:method:unsetStateFlags"), [updateCaretPosition]("GI.Gtk.Interfaces.AccessibleText#g:method:updateCaretPosition"), [updateContents]("GI.Gtk.Interfaces.AccessibleText#g:method:updateContents"), [updateNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:updateNextAccessibleSibling"), [updatePlatformState]("GI.Gtk.Interfaces.Accessible#g:method:updatePlatformState"), [updateProperty]("GI.Gtk.Interfaces.Accessible#g:method:updateProperty"), [updateRelation]("GI.Gtk.Interfaces.Accessible#g:method:updateRelation"), [updateSelectionBound]("GI.Gtk.Interfaces.AccessibleText#g:method:updateSelectionBound"), [updateState]("GI.Gtk.Interfaces.Accessible#g:method:updateState"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleParent"), [getAccessibleRole]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleRole"), [getAllocatedBaseline]("GI.Gtk.Objects.Widget#g:method:getAllocatedBaseline"), [getAllocatedHeight]("GI.Gtk.Objects.Widget#g:method:getAllocatedHeight"), [getAllocatedWidth]("GI.Gtk.Objects.Widget#g:method:getAllocatedWidth"), [getAllocation]("GI.Gtk.Objects.Widget#g:method:getAllocation"), [getAncestor]("GI.Gtk.Objects.Widget#g:method:getAncestor"), [getAtContext]("GI.Gtk.Interfaces.Accessible#g:method:getAtContext"), [getAttributes]("GI.Gtk.Objects.Label#g:method:getAttributes"), [getBaseline]("GI.Gtk.Objects.Widget#g:method:getBaseline"), [getBounds]("GI.Gtk.Interfaces.Accessible#g:method:getBounds"), [getBuildableId]("GI.Gtk.Interfaces.Buildable#g:method:getBuildableId"), [getCanFocus]("GI.Gtk.Objects.Widget#g:method:getCanFocus"), [getCanTarget]("GI.Gtk.Objects.Widget#g:method:getCanTarget"), [getChildVisible]("GI.Gtk.Objects.Widget#g:method:getChildVisible"), [getClipboard]("GI.Gtk.Objects.Widget#g:method:getClipboard"), [getColor]("GI.Gtk.Objects.Widget#g:method:getColor"), [getCssClasses]("GI.Gtk.Objects.Widget#g:method:getCssClasses"), [getCssName]("GI.Gtk.Objects.Widget#g:method:getCssName"), [getCurrentUri]("GI.Gtk.Objects.Label#g:method:getCurrentUri"), [getCursor]("GI.Gtk.Objects.Widget#g:method:getCursor"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDirection]("GI.Gtk.Objects.Widget#g:method:getDirection"), [getDisplay]("GI.Gtk.Objects.Widget#g:method:getDisplay"), [getEllipsize]("GI.Gtk.Objects.Label#g:method:getEllipsize"), [getExtraMenu]("GI.Gtk.Objects.Label#g:method:getExtraMenu"), [getFirstAccessibleChild]("GI.Gtk.Interfaces.Accessible#g:method:getFirstAccessibleChild"), [getFirstChild]("GI.Gtk.Objects.Widget#g:method:getFirstChild"), [getFocusChild]("GI.Gtk.Objects.Widget#g:method:getFocusChild"), [getFocusOnClick]("GI.Gtk.Objects.Widget#g:method:getFocusOnClick"), [getFocusable]("GI.Gtk.Objects.Widget#g:method:getFocusable"), [getFontMap]("GI.Gtk.Objects.Widget#g:method:getFontMap"), [getFontOptions]("GI.Gtk.Objects.Widget#g:method:getFontOptions"), [getFrameClock]("GI.Gtk.Objects.Widget#g:method:getFrameClock"), [getHalign]("GI.Gtk.Objects.Widget#g:method:getHalign"), [getHasTooltip]("GI.Gtk.Objects.Widget#g:method:getHasTooltip"), [getHeight]("GI.Gtk.Objects.Widget#g:method:getHeight"), [getHexpand]("GI.Gtk.Objects.Widget#g:method:getHexpand"), [getHexpandSet]("GI.Gtk.Objects.Widget#g:method:getHexpandSet"), [getJustify]("GI.Gtk.Objects.Label#g:method:getJustify"), [getLabel]("GI.Gtk.Objects.Label#g:method:getLabel"), [getLastChild]("GI.Gtk.Objects.Widget#g:method:getLastChild"), [getLayout]("GI.Gtk.Objects.Label#g:method:getLayout"), [getLayoutManager]("GI.Gtk.Objects.Widget#g:method:getLayoutManager"), [getLayoutOffsets]("GI.Gtk.Objects.Label#g:method:getLayoutOffsets"), [getLimitEvents]("GI.Gtk.Objects.Widget#g:method:getLimitEvents"), [getLines]("GI.Gtk.Objects.Label#g:method:getLines"), [getMapped]("GI.Gtk.Objects.Widget#g:method:getMapped"), [getMarginBottom]("GI.Gtk.Objects.Widget#g:method:getMarginBottom"), [getMarginEnd]("GI.Gtk.Objects.Widget#g:method:getMarginEnd"), [getMarginStart]("GI.Gtk.Objects.Widget#g:method:getMarginStart"), [getMarginTop]("GI.Gtk.Objects.Widget#g:method:getMarginTop"), [getMaxWidthChars]("GI.Gtk.Objects.Label#g:method:getMaxWidthChars"), [getMnemonicKeyval]("GI.Gtk.Objects.Label#g:method:getMnemonicKeyval"), [getMnemonicWidget]("GI.Gtk.Objects.Label#g:method:getMnemonicWidget"), [getName]("GI.Gtk.Objects.Widget#g:method:getName"), [getNative]("GI.Gtk.Objects.Widget#g:method:getNative"), [getNaturalWrapMode]("GI.Gtk.Objects.Label#g:method:getNaturalWrapMode"), [getNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:getNextAccessibleSibling"), [getNextSibling]("GI.Gtk.Objects.Widget#g:method:getNextSibling"), [getOpacity]("GI.Gtk.Objects.Widget#g:method:getOpacity"), [getOverflow]("GI.Gtk.Objects.Widget#g:method:getOverflow"), [getPangoContext]("GI.Gtk.Objects.Widget#g:method:getPangoContext"), [getParent]("GI.Gtk.Objects.Widget#g:method:getParent"), [getPlatformState]("GI.Gtk.Interfaces.Accessible#g:method:getPlatformState"), [getPreferredSize]("GI.Gtk.Objects.Widget#g:method:getPreferredSize"), [getPrevSibling]("GI.Gtk.Objects.Widget#g:method:getPrevSibling"), [getPrimaryClipboard]("GI.Gtk.Objects.Widget#g:method:getPrimaryClipboard"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRealized]("GI.Gtk.Objects.Widget#g:method:getRealized"), [getReceivesDefault]("GI.Gtk.Objects.Widget#g:method:getReceivesDefault"), [getRequestMode]("GI.Gtk.Objects.Widget#g:method:getRequestMode"), [getRoot]("GI.Gtk.Objects.Widget#g:method:getRoot"), [getScaleFactor]("GI.Gtk.Objects.Widget#g:method:getScaleFactor"), [getSelectable]("GI.Gtk.Objects.Label#g:method:getSelectable"), [getSelectionBounds]("GI.Gtk.Objects.Label#g:method:getSelectionBounds"), [getSensitive]("GI.Gtk.Objects.Widget#g:method:getSensitive"), [getSettings]("GI.Gtk.Objects.Widget#g:method:getSettings"), [getSingleLineMode]("GI.Gtk.Objects.Label#g:method:getSingleLineMode"), [getSize]("GI.Gtk.Objects.Widget#g:method:getSize"), [getSizeRequest]("GI.Gtk.Objects.Widget#g:method:getSizeRequest"), [getStateFlags]("GI.Gtk.Objects.Widget#g:method:getStateFlags"), [getStyleContext]("GI.Gtk.Objects.Widget#g:method:getStyleContext"), [getTabs]("GI.Gtk.Objects.Label#g:method:getTabs"), [getTemplateChild]("GI.Gtk.Objects.Widget#g:method:getTemplateChild"), [getText]("GI.Gtk.Objects.Label#g:method:getText"), [getTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:getTooltipMarkup"), [getTooltipText]("GI.Gtk.Objects.Widget#g:method:getTooltipText"), [getUseMarkup]("GI.Gtk.Objects.Label#g:method:getUseMarkup"), [getUseUnderline]("GI.Gtk.Objects.Label#g:method:getUseUnderline"), [getValign]("GI.Gtk.Objects.Widget#g:method:getValign"), [getVexpand]("GI.Gtk.Objects.Widget#g:method:getVexpand"), [getVexpandSet]("GI.Gtk.Objects.Widget#g:method:getVexpandSet"), [getVisible]("GI.Gtk.Objects.Widget#g:method:getVisible"), [getWidth]("GI.Gtk.Objects.Widget#g:method:getWidth"), [getWidthChars]("GI.Gtk.Objects.Label#g:method:getWidthChars"), [getWrap]("GI.Gtk.Objects.Label#g:method:getWrap"), [getWrapMode]("GI.Gtk.Objects.Label#g:method:getWrapMode"), [getXalign]("GI.Gtk.Objects.Label#g:method:getXalign"), [getYalign]("GI.Gtk.Objects.Label#g:method:getYalign").
-- 
-- ==== Setters
-- [setAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:setAccessibleParent"), [setAttributes]("GI.Gtk.Objects.Label#g:method:setAttributes"), [setCanFocus]("GI.Gtk.Objects.Widget#g:method:setCanFocus"), [setCanTarget]("GI.Gtk.Objects.Widget#g:method:setCanTarget"), [setChildVisible]("GI.Gtk.Objects.Widget#g:method:setChildVisible"), [setCssClasses]("GI.Gtk.Objects.Widget#g:method:setCssClasses"), [setCursor]("GI.Gtk.Objects.Widget#g:method:setCursor"), [setCursorFromName]("GI.Gtk.Objects.Widget#g:method:setCursorFromName"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDirection]("GI.Gtk.Objects.Widget#g:method:setDirection"), [setEllipsize]("GI.Gtk.Objects.Label#g:method:setEllipsize"), [setExtraMenu]("GI.Gtk.Objects.Label#g:method:setExtraMenu"), [setFocusChild]("GI.Gtk.Objects.Widget#g:method:setFocusChild"), [setFocusOnClick]("GI.Gtk.Objects.Widget#g:method:setFocusOnClick"), [setFocusable]("GI.Gtk.Objects.Widget#g:method:setFocusable"), [setFontMap]("GI.Gtk.Objects.Widget#g:method:setFontMap"), [setFontOptions]("GI.Gtk.Objects.Widget#g:method:setFontOptions"), [setHalign]("GI.Gtk.Objects.Widget#g:method:setHalign"), [setHasTooltip]("GI.Gtk.Objects.Widget#g:method:setHasTooltip"), [setHexpand]("GI.Gtk.Objects.Widget#g:method:setHexpand"), [setHexpandSet]("GI.Gtk.Objects.Widget#g:method:setHexpandSet"), [setJustify]("GI.Gtk.Objects.Label#g:method:setJustify"), [setLabel]("GI.Gtk.Objects.Label#g:method:setLabel"), [setLayoutManager]("GI.Gtk.Objects.Widget#g:method:setLayoutManager"), [setLimitEvents]("GI.Gtk.Objects.Widget#g:method:setLimitEvents"), [setLines]("GI.Gtk.Objects.Label#g:method:setLines"), [setMarginBottom]("GI.Gtk.Objects.Widget#g:method:setMarginBottom"), [setMarginEnd]("GI.Gtk.Objects.Widget#g:method:setMarginEnd"), [setMarginStart]("GI.Gtk.Objects.Widget#g:method:setMarginStart"), [setMarginTop]("GI.Gtk.Objects.Widget#g:method:setMarginTop"), [setMarkup]("GI.Gtk.Objects.Label#g:method:setMarkup"), [setMarkupWithMnemonic]("GI.Gtk.Objects.Label#g:method:setMarkupWithMnemonic"), [setMaxWidthChars]("GI.Gtk.Objects.Label#g:method:setMaxWidthChars"), [setMnemonicWidget]("GI.Gtk.Objects.Label#g:method:setMnemonicWidget"), [setName]("GI.Gtk.Objects.Widget#g:method:setName"), [setNaturalWrapMode]("GI.Gtk.Objects.Label#g:method:setNaturalWrapMode"), [setOpacity]("GI.Gtk.Objects.Widget#g:method:setOpacity"), [setOverflow]("GI.Gtk.Objects.Widget#g:method:setOverflow"), [setParent]("GI.Gtk.Objects.Widget#g:method:setParent"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setReceivesDefault]("GI.Gtk.Objects.Widget#g:method:setReceivesDefault"), [setSelectable]("GI.Gtk.Objects.Label#g:method:setSelectable"), [setSensitive]("GI.Gtk.Objects.Widget#g:method:setSensitive"), [setSingleLineMode]("GI.Gtk.Objects.Label#g:method:setSingleLineMode"), [setSizeRequest]("GI.Gtk.Objects.Widget#g:method:setSizeRequest"), [setStateFlags]("GI.Gtk.Objects.Widget#g:method:setStateFlags"), [setTabs]("GI.Gtk.Objects.Label#g:method:setTabs"), [setText]("GI.Gtk.Objects.Label#g:method:setText"), [setTextWithMnemonic]("GI.Gtk.Objects.Label#g:method:setTextWithMnemonic"), [setTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:setTooltipMarkup"), [setTooltipText]("GI.Gtk.Objects.Widget#g:method:setTooltipText"), [setUseMarkup]("GI.Gtk.Objects.Label#g:method:setUseMarkup"), [setUseUnderline]("GI.Gtk.Objects.Label#g:method:setUseUnderline"), [setValign]("GI.Gtk.Objects.Widget#g:method:setValign"), [setVexpand]("GI.Gtk.Objects.Widget#g:method:setVexpand"), [setVexpandSet]("GI.Gtk.Objects.Widget#g:method:setVexpandSet"), [setVisible]("GI.Gtk.Objects.Widget#g:method:setVisible"), [setWidthChars]("GI.Gtk.Objects.Label#g:method:setWidthChars"), [setWrap]("GI.Gtk.Objects.Label#g:method:setWrap"), [setWrapMode]("GI.Gtk.Objects.Label#g:method:setWrapMode"), [setXalign]("GI.Gtk.Objects.Label#g:method:setXalign"), [setYalign]("GI.Gtk.Objects.Label#g:method:setYalign").

#if defined(ENABLE_OVERLOADING)
    ResolveLabelMethod                      ,
#endif

-- ** getAttributes #method:getAttributes#

#if defined(ENABLE_OVERLOADING)
    LabelGetAttributesMethodInfo            ,
#endif
    labelGetAttributes                      ,


-- ** getCurrentUri #method:getCurrentUri#

#if defined(ENABLE_OVERLOADING)
    LabelGetCurrentUriMethodInfo            ,
#endif
    labelGetCurrentUri                      ,


-- ** getEllipsize #method:getEllipsize#

#if defined(ENABLE_OVERLOADING)
    LabelGetEllipsizeMethodInfo             ,
#endif
    labelGetEllipsize                       ,


-- ** getExtraMenu #method:getExtraMenu#

#if defined(ENABLE_OVERLOADING)
    LabelGetExtraMenuMethodInfo             ,
#endif
    labelGetExtraMenu                       ,


-- ** getJustify #method:getJustify#

#if defined(ENABLE_OVERLOADING)
    LabelGetJustifyMethodInfo               ,
#endif
    labelGetJustify                         ,


-- ** getLabel #method:getLabel#

#if defined(ENABLE_OVERLOADING)
    LabelGetLabelMethodInfo                 ,
#endif
    labelGetLabel                           ,


-- ** getLayout #method:getLayout#

#if defined(ENABLE_OVERLOADING)
    LabelGetLayoutMethodInfo                ,
#endif
    labelGetLayout                          ,


-- ** getLayoutOffsets #method:getLayoutOffsets#

#if defined(ENABLE_OVERLOADING)
    LabelGetLayoutOffsetsMethodInfo         ,
#endif
    labelGetLayoutOffsets                   ,


-- ** getLines #method:getLines#

#if defined(ENABLE_OVERLOADING)
    LabelGetLinesMethodInfo                 ,
#endif
    labelGetLines                           ,


-- ** getMaxWidthChars #method:getMaxWidthChars#

#if defined(ENABLE_OVERLOADING)
    LabelGetMaxWidthCharsMethodInfo         ,
#endif
    labelGetMaxWidthChars                   ,


-- ** getMnemonicKeyval #method:getMnemonicKeyval#

#if defined(ENABLE_OVERLOADING)
    LabelGetMnemonicKeyvalMethodInfo        ,
#endif
    labelGetMnemonicKeyval                  ,


-- ** getMnemonicWidget #method:getMnemonicWidget#

#if defined(ENABLE_OVERLOADING)
    LabelGetMnemonicWidgetMethodInfo        ,
#endif
    labelGetMnemonicWidget                  ,


-- ** getNaturalWrapMode #method:getNaturalWrapMode#

#if defined(ENABLE_OVERLOADING)
    LabelGetNaturalWrapModeMethodInfo       ,
#endif
    labelGetNaturalWrapMode                 ,


-- ** getSelectable #method:getSelectable#

#if defined(ENABLE_OVERLOADING)
    LabelGetSelectableMethodInfo            ,
#endif
    labelGetSelectable                      ,


-- ** getSelectionBounds #method:getSelectionBounds#

#if defined(ENABLE_OVERLOADING)
    LabelGetSelectionBoundsMethodInfo       ,
#endif
    labelGetSelectionBounds                 ,


-- ** getSingleLineMode #method:getSingleLineMode#

#if defined(ENABLE_OVERLOADING)
    LabelGetSingleLineModeMethodInfo        ,
#endif
    labelGetSingleLineMode                  ,


-- ** getTabs #method:getTabs#

#if defined(ENABLE_OVERLOADING)
    LabelGetTabsMethodInfo                  ,
#endif
    labelGetTabs                            ,


-- ** getText #method:getText#

#if defined(ENABLE_OVERLOADING)
    LabelGetTextMethodInfo                  ,
#endif
    labelGetText                            ,


-- ** getUseMarkup #method:getUseMarkup#

#if defined(ENABLE_OVERLOADING)
    LabelGetUseMarkupMethodInfo             ,
#endif
    labelGetUseMarkup                       ,


-- ** getUseUnderline #method:getUseUnderline#

#if defined(ENABLE_OVERLOADING)
    LabelGetUseUnderlineMethodInfo          ,
#endif
    labelGetUseUnderline                    ,


-- ** getWidthChars #method:getWidthChars#

#if defined(ENABLE_OVERLOADING)
    LabelGetWidthCharsMethodInfo            ,
#endif
    labelGetWidthChars                      ,


-- ** getWrap #method:getWrap#

#if defined(ENABLE_OVERLOADING)
    LabelGetWrapMethodInfo                  ,
#endif
    labelGetWrap                            ,


-- ** getWrapMode #method:getWrapMode#

#if defined(ENABLE_OVERLOADING)
    LabelGetWrapModeMethodInfo              ,
#endif
    labelGetWrapMode                        ,


-- ** getXalign #method:getXalign#

#if defined(ENABLE_OVERLOADING)
    LabelGetXalignMethodInfo                ,
#endif
    labelGetXalign                          ,


-- ** getYalign #method:getYalign#

#if defined(ENABLE_OVERLOADING)
    LabelGetYalignMethodInfo                ,
#endif
    labelGetYalign                          ,


-- ** new #method:new#

    labelNew                                ,


-- ** newWithMnemonic #method:newWithMnemonic#

    labelNewWithMnemonic                    ,


-- ** selectRegion #method:selectRegion#

#if defined(ENABLE_OVERLOADING)
    LabelSelectRegionMethodInfo             ,
#endif
    labelSelectRegion                       ,


-- ** setAttributes #method:setAttributes#

#if defined(ENABLE_OVERLOADING)
    LabelSetAttributesMethodInfo            ,
#endif
    labelSetAttributes                      ,


-- ** setEllipsize #method:setEllipsize#

#if defined(ENABLE_OVERLOADING)
    LabelSetEllipsizeMethodInfo             ,
#endif
    labelSetEllipsize                       ,


-- ** setExtraMenu #method:setExtraMenu#

#if defined(ENABLE_OVERLOADING)
    LabelSetExtraMenuMethodInfo             ,
#endif
    labelSetExtraMenu                       ,


-- ** setJustify #method:setJustify#

#if defined(ENABLE_OVERLOADING)
    LabelSetJustifyMethodInfo               ,
#endif
    labelSetJustify                         ,


-- ** setLabel #method:setLabel#

#if defined(ENABLE_OVERLOADING)
    LabelSetLabelMethodInfo                 ,
#endif
    labelSetLabel                           ,


-- ** setLines #method:setLines#

#if defined(ENABLE_OVERLOADING)
    LabelSetLinesMethodInfo                 ,
#endif
    labelSetLines                           ,


-- ** setMarkup #method:setMarkup#

#if defined(ENABLE_OVERLOADING)
    LabelSetMarkupMethodInfo                ,
#endif
    labelSetMarkup                          ,


-- ** setMarkupWithMnemonic #method:setMarkupWithMnemonic#

#if defined(ENABLE_OVERLOADING)
    LabelSetMarkupWithMnemonicMethodInfo    ,
#endif
    labelSetMarkupWithMnemonic              ,


-- ** setMaxWidthChars #method:setMaxWidthChars#

#if defined(ENABLE_OVERLOADING)
    LabelSetMaxWidthCharsMethodInfo         ,
#endif
    labelSetMaxWidthChars                   ,


-- ** setMnemonicWidget #method:setMnemonicWidget#

#if defined(ENABLE_OVERLOADING)
    LabelSetMnemonicWidgetMethodInfo        ,
#endif
    labelSetMnemonicWidget                  ,


-- ** setNaturalWrapMode #method:setNaturalWrapMode#

#if defined(ENABLE_OVERLOADING)
    LabelSetNaturalWrapModeMethodInfo       ,
#endif
    labelSetNaturalWrapMode                 ,


-- ** setSelectable #method:setSelectable#

#if defined(ENABLE_OVERLOADING)
    LabelSetSelectableMethodInfo            ,
#endif
    labelSetSelectable                      ,


-- ** setSingleLineMode #method:setSingleLineMode#

#if defined(ENABLE_OVERLOADING)
    LabelSetSingleLineModeMethodInfo        ,
#endif
    labelSetSingleLineMode                  ,


-- ** setTabs #method:setTabs#

#if defined(ENABLE_OVERLOADING)
    LabelSetTabsMethodInfo                  ,
#endif
    labelSetTabs                            ,


-- ** setText #method:setText#

#if defined(ENABLE_OVERLOADING)
    LabelSetTextMethodInfo                  ,
#endif
    labelSetText                            ,


-- ** setTextWithMnemonic #method:setTextWithMnemonic#

#if defined(ENABLE_OVERLOADING)
    LabelSetTextWithMnemonicMethodInfo      ,
#endif
    labelSetTextWithMnemonic                ,


-- ** setUseMarkup #method:setUseMarkup#

#if defined(ENABLE_OVERLOADING)
    LabelSetUseMarkupMethodInfo             ,
#endif
    labelSetUseMarkup                       ,


-- ** setUseUnderline #method:setUseUnderline#

#if defined(ENABLE_OVERLOADING)
    LabelSetUseUnderlineMethodInfo          ,
#endif
    labelSetUseUnderline                    ,


-- ** setWidthChars #method:setWidthChars#

#if defined(ENABLE_OVERLOADING)
    LabelSetWidthCharsMethodInfo            ,
#endif
    labelSetWidthChars                      ,


-- ** setWrap #method:setWrap#

#if defined(ENABLE_OVERLOADING)
    LabelSetWrapMethodInfo                  ,
#endif
    labelSetWrap                            ,


-- ** setWrapMode #method:setWrapMode#

#if defined(ENABLE_OVERLOADING)
    LabelSetWrapModeMethodInfo              ,
#endif
    labelSetWrapMode                        ,


-- ** setXalign #method:setXalign#

#if defined(ENABLE_OVERLOADING)
    LabelSetXalignMethodInfo                ,
#endif
    labelSetXalign                          ,


-- ** setYalign #method:setYalign#

#if defined(ENABLE_OVERLOADING)
    LabelSetYalignMethodInfo                ,
#endif
    labelSetYalign                          ,




 -- * Properties


-- ** attributes #attr:attributes#
-- | A list of style attributes to apply to the text of the label.

#if defined(ENABLE_OVERLOADING)
    LabelAttributesPropertyInfo             ,
#endif
    clearLabelAttributes                    ,
    constructLabelAttributes                ,
    getLabelAttributes                      ,
#if defined(ENABLE_OVERLOADING)
    labelAttributes                         ,
#endif
    setLabelAttributes                      ,


-- ** ellipsize #attr:ellipsize#
-- | The preferred place to ellipsize the string, if the label does
-- not have enough room to display the entire string.
-- 
-- Note that setting this property to a value other than
-- [enum.Pango.EllipsizeMode.none] has the side-effect that the label requests
-- only enough space to display the ellipsis \"...\". In particular, this
-- means that ellipsizing labels do not work well in notebook tabs, unless
-- the [NotebookPage:tabExpand]("GI.Gtk.Objects.NotebookPage#g:attr:tabExpand") child property is set to true.
-- 
-- Other ways to set a label\'s width are 'GI.Gtk.Objects.Widget.widgetSetSizeRequest'
-- and 'GI.Gtk.Objects.Label.labelSetWidthChars'.

#if defined(ENABLE_OVERLOADING)
    LabelEllipsizePropertyInfo              ,
#endif
    constructLabelEllipsize                 ,
    getLabelEllipsize                       ,
#if defined(ENABLE_OVERLOADING)
    labelEllipsize                          ,
#endif
    setLabelEllipsize                       ,


-- ** extraMenu #attr:extraMenu#
-- | A menu model whose contents will be appended to the context menu.

#if defined(ENABLE_OVERLOADING)
    LabelExtraMenuPropertyInfo              ,
#endif
    clearLabelExtraMenu                     ,
    constructLabelExtraMenu                 ,
    getLabelExtraMenu                       ,
#if defined(ENABLE_OVERLOADING)
    labelExtraMenu                          ,
#endif
    setLabelExtraMenu                       ,


-- ** justify #attr:justify#
-- | The alignment of the lines in the text of the label, relative to each other.
-- 
-- This does *not* affect the alignment of the label within its allocation.
-- See [Label:xalign]("GI.Gtk.Objects.Label#g:attr:xalign") for that.

#if defined(ENABLE_OVERLOADING)
    LabelJustifyPropertyInfo                ,
#endif
    constructLabelJustify                   ,
    getLabelJustify                         ,
#if defined(ENABLE_OVERLOADING)
    labelJustify                            ,
#endif
    setLabelJustify                         ,


-- ** label #attr:label#
-- | The contents of the label.
-- 
-- If the string contains Pango markup (see 'GI.Pango.Functions.parseMarkup'),
-- you will have to set the [Label:useMarkup]("GI.Gtk.Objects.Label#g:attr:useMarkup") property to
-- true in order for the label to display the markup attributes. See also
-- 'GI.Gtk.Objects.Label.labelSetMarkup' for a convenience function that sets both
-- this property and the [Label:useMarkup]("GI.Gtk.Objects.Label#g:attr:useMarkup") property at the
-- same time.
-- 
-- If the string contains underlines acting as mnemonics, you will have to
-- set the [Label:useUnderline]("GI.Gtk.Objects.Label#g:attr:useUnderline") property to true in order
-- for the label to display them.

#if defined(ENABLE_OVERLOADING)
    LabelLabelPropertyInfo                  ,
#endif
    constructLabelLabel                     ,
    getLabelLabel                           ,
#if defined(ENABLE_OVERLOADING)
    labelLabel                              ,
#endif
    setLabelLabel                           ,


-- ** lines #attr:lines#
-- | The number of lines to which an ellipsized, wrapping label
-- should display before it gets ellipsized. This both prevents the label
-- from ellipsizing before this many lines are displayed, and limits the
-- height request of the label to this many lines.
-- 
-- ::: warning
--     Setting this property has unintuitive and unfortunate consequences
--     for the minimum _width_ of the label. Specifically, if the height
--     of the label is such that it fits a smaller number of lines than
--     the value of this property, the label can not be ellipsized at all,
--     which means it must be wide enough to fit all the text fully.
-- 
-- This property has no effect if the label is not wrapping or ellipsized.
-- 
-- Set this property to -1 if you don\'t want to limit the number of lines.

#if defined(ENABLE_OVERLOADING)
    LabelLinesPropertyInfo                  ,
#endif
    constructLabelLines                     ,
    getLabelLines                           ,
#if defined(ENABLE_OVERLOADING)
    labelLines                              ,
#endif
    setLabelLines                           ,


-- ** maxWidthChars #attr:maxWidthChars#
-- | The desired maximum width of the label, in characters.
-- 
-- If this property is set to -1, the width will be calculated automatically.
-- 
-- See the section on <https://docs.gtk.org/gtk4/class.Label.html#text-layout text layout> for details
-- of how [Label:widthChars]("GI.Gtk.Objects.Label#g:attr:widthChars") and [Label:maxWidthChars]("GI.Gtk.Objects.Label#g:attr:maxWidthChars")
-- determine the width of ellipsized and wrapped labels.

#if defined(ENABLE_OVERLOADING)
    LabelMaxWidthCharsPropertyInfo          ,
#endif
    constructLabelMaxWidthChars             ,
    getLabelMaxWidthChars                   ,
#if defined(ENABLE_OVERLOADING)
    labelMaxWidthChars                      ,
#endif
    setLabelMaxWidthChars                   ,


-- ** mnemonicKeyval #attr:mnemonicKeyval#
-- | The mnemonic accelerator key for the label.

#if defined(ENABLE_OVERLOADING)
    LabelMnemonicKeyvalPropertyInfo         ,
#endif
    getLabelMnemonicKeyval                  ,
#if defined(ENABLE_OVERLOADING)
    labelMnemonicKeyval                     ,
#endif


-- ** mnemonicWidget #attr:mnemonicWidget#
-- | The widget to be activated when the labels mnemonic key is pressed.

#if defined(ENABLE_OVERLOADING)
    LabelMnemonicWidgetPropertyInfo         ,
#endif
    clearLabelMnemonicWidget                ,
    constructLabelMnemonicWidget            ,
    getLabelMnemonicWidget                  ,
#if defined(ENABLE_OVERLOADING)
    labelMnemonicWidget                     ,
#endif
    setLabelMnemonicWidget                  ,


-- ** naturalWrapMode #attr:naturalWrapMode#
-- | Select the line wrapping for the natural size request.
-- 
-- This only affects the natural size requested. For the actual wrapping
-- used, see the [Label:wrapMode]("GI.Gtk.Objects.Label#g:attr:wrapMode") property.
-- 
-- The default is 'GI.Gtk.Enums.NaturalWrapModeInherit', which inherits
-- the behavior of the [Label:wrapMode]("GI.Gtk.Objects.Label#g:attr:wrapMode") property.
-- 
-- /Since: 4.6/

#if defined(ENABLE_OVERLOADING)
    LabelNaturalWrapModePropertyInfo        ,
#endif
    constructLabelNaturalWrapMode           ,
    getLabelNaturalWrapMode                 ,
#if defined(ENABLE_OVERLOADING)
    labelNaturalWrapMode                    ,
#endif
    setLabelNaturalWrapMode                 ,


-- ** selectable #attr:selectable#
-- | Whether the label text can be selected with the mouse.

#if defined(ENABLE_OVERLOADING)
    LabelSelectablePropertyInfo             ,
#endif
    constructLabelSelectable                ,
    getLabelSelectable                      ,
#if defined(ENABLE_OVERLOADING)
    labelSelectable                         ,
#endif
    setLabelSelectable                      ,


-- ** singleLineMode #attr:singleLineMode#
-- | Whether the label is in single line mode.
-- 
-- In single line mode, the height of the label does not depend on the
-- actual text, it is always set to ascent + descent of the font. This
-- can be an advantage in situations where resizing the label because
-- of text changes would be distracting, e.g. in a statusbar.

#if defined(ENABLE_OVERLOADING)
    LabelSingleLineModePropertyInfo         ,
#endif
    constructLabelSingleLineMode            ,
    getLabelSingleLineMode                  ,
#if defined(ENABLE_OVERLOADING)
    labelSingleLineMode                     ,
#endif
    setLabelSingleLineMode                  ,


-- ** tabs #attr:tabs#
-- | Custom tabs for this label.
-- 
-- /Since: 4.8/

#if defined(ENABLE_OVERLOADING)
    LabelTabsPropertyInfo                   ,
#endif
    clearLabelTabs                          ,
    constructLabelTabs                      ,
    getLabelTabs                            ,
#if defined(ENABLE_OVERLOADING)
    labelTabs                               ,
#endif
    setLabelTabs                            ,


-- ** useMarkup #attr:useMarkup#
-- | True if the text of the label includes Pango markup.
-- 
-- See 'GI.Pango.Functions.parseMarkup'.

#if defined(ENABLE_OVERLOADING)
    LabelUseMarkupPropertyInfo              ,
#endif
    constructLabelUseMarkup                 ,
    getLabelUseMarkup                       ,
#if defined(ENABLE_OVERLOADING)
    labelUseMarkup                          ,
#endif
    setLabelUseMarkup                       ,


-- ** useUnderline #attr:useUnderline#
-- | True if the text of the label indicates a mnemonic with an @_@
-- before the mnemonic character.

#if defined(ENABLE_OVERLOADING)
    LabelUseUnderlinePropertyInfo           ,
#endif
    constructLabelUseUnderline              ,
    getLabelUseUnderline                    ,
#if defined(ENABLE_OVERLOADING)
    labelUseUnderline                       ,
#endif
    setLabelUseUnderline                    ,


-- ** widthChars #attr:widthChars#
-- | The desired width of the label, in characters.
-- 
-- If this property is set to -1, the width will be calculated automatically.
-- 
-- See the section on <https://docs.gtk.org/gtk4/class.Label.html#text-layout text layout> for details
-- of how [Label:widthChars]("GI.Gtk.Objects.Label#g:attr:widthChars") and [Label:maxWidthChars]("GI.Gtk.Objects.Label#g:attr:maxWidthChars")
-- determine the width of ellipsized and wrapped labels.

#if defined(ENABLE_OVERLOADING)
    LabelWidthCharsPropertyInfo             ,
#endif
    constructLabelWidthChars                ,
    getLabelWidthChars                      ,
#if defined(ENABLE_OVERLOADING)
    labelWidthChars                         ,
#endif
    setLabelWidthChars                      ,


-- ** wrap #attr:wrap#
-- | True if the label text will wrap if it gets too wide.

#if defined(ENABLE_OVERLOADING)
    LabelWrapPropertyInfo                   ,
#endif
    constructLabelWrap                      ,
    getLabelWrap                            ,
#if defined(ENABLE_OVERLOADING)
    labelWrap                               ,
#endif
    setLabelWrap                            ,


-- ** wrapMode #attr:wrapMode#
-- | Controls how the line wrapping is done.
-- 
-- This only affects the formatting if line wrapping is on (see the
-- [Label:wrap]("GI.Gtk.Objects.Label#g:attr:wrap") property). The default is 'GI.Pango.Enums.WrapModeWord',
-- which means wrap on word boundaries.
-- 
-- For sizing behavior, also consider the [Label:naturalWrapMode]("GI.Gtk.Objects.Label#g:attr:naturalWrapMode")
-- property.

#if defined(ENABLE_OVERLOADING)
    LabelWrapModePropertyInfo               ,
#endif
    constructLabelWrapMode                  ,
    getLabelWrapMode                        ,
#if defined(ENABLE_OVERLOADING)
    labelWrapMode                           ,
#endif
    setLabelWrapMode                        ,


-- ** xalign #attr:xalign#
-- | The horizontal alignment of the label text inside its size allocation.
-- 
-- Compare this to [Widget:halign]("GI.Gtk.Objects.Widget#g:attr:halign"), which determines how the
-- labels size allocation is positioned in the space available for the label.

#if defined(ENABLE_OVERLOADING)
    LabelXalignPropertyInfo                 ,
#endif
    constructLabelXalign                    ,
    getLabelXalign                          ,
#if defined(ENABLE_OVERLOADING)
    labelXalign                             ,
#endif
    setLabelXalign                          ,


-- ** yalign #attr:yalign#
-- | The vertical alignment of the label text inside its size allocation.
-- 
-- Compare this to [Widget:valign]("GI.Gtk.Objects.Widget#g:attr:valign"), which determines how the
-- labels size allocation is positioned in the space available for the label.

#if defined(ENABLE_OVERLOADING)
    LabelYalignPropertyInfo                 ,
#endif
    constructLabelYalign                    ,
    getLabelYalign                          ,
#if defined(ENABLE_OVERLOADING)
    labelYalign                             ,
#endif
    setLabelYalign                          ,




 -- * Signals


-- ** activateCurrentLink #signal:activateCurrentLink#

    LabelActivateCurrentLinkCallback        ,
#if defined(ENABLE_OVERLOADING)
    LabelActivateCurrentLinkSignalInfo      ,
#endif
    afterLabelActivateCurrentLink           ,
    onLabelActivateCurrentLink              ,


-- ** activateLink #signal:activateLink#

    LabelActivateLinkCallback               ,
#if defined(ENABLE_OVERLOADING)
    LabelActivateLinkSignalInfo             ,
#endif
    afterLabelActivateLink                  ,
    onLabelActivateLink                     ,


-- ** copyClipboard #signal:copyClipboard#

    LabelCopyClipboardCallback              ,
#if defined(ENABLE_OVERLOADING)
    LabelCopyClipboardSignalInfo            ,
#endif
    afterLabelCopyClipboard                 ,
    onLabelCopyClipboard                    ,


-- ** moveCursor #signal:moveCursor#

    LabelMoveCursorCallback                 ,
#if defined(ENABLE_OVERLOADING)
    LabelMoveCursorSignalInfo               ,
#endif
    afterLabelMoveCursor                    ,
    onLabelMoveCursor                       ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Gio.Objects.MenuModel as Gio.MenuModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.ComponentTransfer as Gsk.ComponentTransfer
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.AccessibleText as Gtk.AccessibleText
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout
import qualified GI.Pango.Structs.AttrList as Pango.AttrList
import qualified GI.Pango.Structs.TabArray as Pango.TabArray

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gio.Objects.MenuModel as Gio.MenuModel
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.AccessibleText as Gtk.AccessibleText
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Layout as Pango.Layout
import qualified GI.Pango.Structs.AttrList as Pango.AttrList
import qualified GI.Pango.Structs.TabArray as Pango.TabArray

#endif

-- | Memory-managed wrapper type.
newtype Label = Label (SP.ManagedPtr Label)
    deriving (Eq)

instance SP.ManagedPtrNewtype Label where
    toManagedPtr (Label p) = p

foreign import ccall "gtk_label_get_type"
    c_gtk_label_get_type :: IO B.Types.GType

instance B.Types.TypedObject Label where
    glibType = c_gtk_label_get_type

instance B.Types.GObject Label

-- | Type class for types which can be safely cast to t'Label', for instance with `toLabel`.
class (SP.GObject o, O.IsDescendantOf Label o) => IsLabel o
instance (SP.GObject o, O.IsDescendantOf Label o) => IsLabel o

instance O.HasParentTypes Label
type instance O.ParentTypes Label = '[Gtk.Widget.Widget, GObject.Object.Object, Gtk.Accessible.Accessible, Gtk.AccessibleText.AccessibleText, Gtk.Buildable.Buildable, Gtk.ConstraintTarget.ConstraintTarget]

-- | Cast to t'Label', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toLabel :: (MIO.MonadIO m, IsLabel o) => o -> m Label
toLabel = MIO.liftIO . B.ManagedPtr.unsafeCastTo Label

-- | Convert t'Label' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Label) where
    gvalueGType_ = c_gtk_label_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Label)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Label)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Label ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveLabelMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveLabelMethod "actionSetEnabled" o = Gtk.Widget.WidgetActionSetEnabledMethodInfo
    ResolveLabelMethod "activate" o = Gtk.Widget.WidgetActivateMethodInfo
    ResolveLabelMethod "activateAction" o = Gtk.Widget.WidgetActivateActionMethodInfo
    ResolveLabelMethod "activateDefault" o = Gtk.Widget.WidgetActivateDefaultMethodInfo
    ResolveLabelMethod "addController" o = Gtk.Widget.WidgetAddControllerMethodInfo
    ResolveLabelMethod "addCssClass" o = Gtk.Widget.WidgetAddCssClassMethodInfo
    ResolveLabelMethod "addMnemonicLabel" o = Gtk.Widget.WidgetAddMnemonicLabelMethodInfo
    ResolveLabelMethod "addTickCallback" o = Gtk.Widget.WidgetAddTickCallbackMethodInfo
    ResolveLabelMethod "allocate" o = Gtk.Widget.WidgetAllocateMethodInfo
    ResolveLabelMethod "announce" o = Gtk.Accessible.AccessibleAnnounceMethodInfo
    ResolveLabelMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveLabelMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveLabelMethod "childFocus" o = Gtk.Widget.WidgetChildFocusMethodInfo
    ResolveLabelMethod "computeBounds" o = Gtk.Widget.WidgetComputeBoundsMethodInfo
    ResolveLabelMethod "computeExpand" o = Gtk.Widget.WidgetComputeExpandMethodInfo
    ResolveLabelMethod "computePoint" o = Gtk.Widget.WidgetComputePointMethodInfo
    ResolveLabelMethod "computeTransform" o = Gtk.Widget.WidgetComputeTransformMethodInfo
    ResolveLabelMethod "contains" o = Gtk.Widget.WidgetContainsMethodInfo
    ResolveLabelMethod "createPangoContext" o = Gtk.Widget.WidgetCreatePangoContextMethodInfo
    ResolveLabelMethod "createPangoLayout" o = Gtk.Widget.WidgetCreatePangoLayoutMethodInfo
    ResolveLabelMethod "disposeTemplate" o = Gtk.Widget.WidgetDisposeTemplateMethodInfo
    ResolveLabelMethod "dragCheckThreshold" o = Gtk.Widget.WidgetDragCheckThresholdMethodInfo
    ResolveLabelMethod "errorBell" o = Gtk.Widget.WidgetErrorBellMethodInfo
    ResolveLabelMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveLabelMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveLabelMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveLabelMethod "grabFocus" o = Gtk.Widget.WidgetGrabFocusMethodInfo
    ResolveLabelMethod "hasCssClass" o = Gtk.Widget.WidgetHasCssClassMethodInfo
    ResolveLabelMethod "hasDefault" o = Gtk.Widget.WidgetHasDefaultMethodInfo
    ResolveLabelMethod "hasFocus" o = Gtk.Widget.WidgetHasFocusMethodInfo
    ResolveLabelMethod "hasVisibleFocus" o = Gtk.Widget.WidgetHasVisibleFocusMethodInfo
    ResolveLabelMethod "hide" o = Gtk.Widget.WidgetHideMethodInfo
    ResolveLabelMethod "inDestruction" o = Gtk.Widget.WidgetInDestructionMethodInfo
    ResolveLabelMethod "initTemplate" o = Gtk.Widget.WidgetInitTemplateMethodInfo
    ResolveLabelMethod "insertActionGroup" o = Gtk.Widget.WidgetInsertActionGroupMethodInfo
    ResolveLabelMethod "insertAfter" o = Gtk.Widget.WidgetInsertAfterMethodInfo
    ResolveLabelMethod "insertBefore" o = Gtk.Widget.WidgetInsertBeforeMethodInfo
    ResolveLabelMethod "isAncestor" o = Gtk.Widget.WidgetIsAncestorMethodInfo
    ResolveLabelMethod "isDrawable" o = Gtk.Widget.WidgetIsDrawableMethodInfo
    ResolveLabelMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveLabelMethod "isFocus" o = Gtk.Widget.WidgetIsFocusMethodInfo
    ResolveLabelMethod "isSensitive" o = Gtk.Widget.WidgetIsSensitiveMethodInfo
    ResolveLabelMethod "isVisible" o = Gtk.Widget.WidgetIsVisibleMethodInfo
    ResolveLabelMethod "keynavFailed" o = Gtk.Widget.WidgetKeynavFailedMethodInfo
    ResolveLabelMethod "listMnemonicLabels" o = Gtk.Widget.WidgetListMnemonicLabelsMethodInfo
    ResolveLabelMethod "map" o = Gtk.Widget.WidgetMapMethodInfo
    ResolveLabelMethod "measure" o = Gtk.Widget.WidgetMeasureMethodInfo
    ResolveLabelMethod "mnemonicActivate" o = Gtk.Widget.WidgetMnemonicActivateMethodInfo
    ResolveLabelMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveLabelMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveLabelMethod "observeChildren" o = Gtk.Widget.WidgetObserveChildrenMethodInfo
    ResolveLabelMethod "observeControllers" o = Gtk.Widget.WidgetObserveControllersMethodInfo
    ResolveLabelMethod "pick" o = Gtk.Widget.WidgetPickMethodInfo
    ResolveLabelMethod "queueAllocate" o = Gtk.Widget.WidgetQueueAllocateMethodInfo
    ResolveLabelMethod "queueDraw" o = Gtk.Widget.WidgetQueueDrawMethodInfo
    ResolveLabelMethod "queueResize" o = Gtk.Widget.WidgetQueueResizeMethodInfo
    ResolveLabelMethod "realize" o = Gtk.Widget.WidgetRealizeMethodInfo
    ResolveLabelMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveLabelMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveLabelMethod "removeController" o = Gtk.Widget.WidgetRemoveControllerMethodInfo
    ResolveLabelMethod "removeCssClass" o = Gtk.Widget.WidgetRemoveCssClassMethodInfo
    ResolveLabelMethod "removeMnemonicLabel" o = Gtk.Widget.WidgetRemoveMnemonicLabelMethodInfo
    ResolveLabelMethod "removeTickCallback" o = Gtk.Widget.WidgetRemoveTickCallbackMethodInfo
    ResolveLabelMethod "resetProperty" o = Gtk.Accessible.AccessibleResetPropertyMethodInfo
    ResolveLabelMethod "resetRelation" o = Gtk.Accessible.AccessibleResetRelationMethodInfo
    ResolveLabelMethod "resetState" o = Gtk.Accessible.AccessibleResetStateMethodInfo
    ResolveLabelMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveLabelMethod "selectRegion" o = LabelSelectRegionMethodInfo
    ResolveLabelMethod "shouldLayout" o = Gtk.Widget.WidgetShouldLayoutMethodInfo
    ResolveLabelMethod "show" o = Gtk.Widget.WidgetShowMethodInfo
    ResolveLabelMethod "sizeAllocate" o = Gtk.Widget.WidgetSizeAllocateMethodInfo
    ResolveLabelMethod "snapshotChild" o = Gtk.Widget.WidgetSnapshotChildMethodInfo
    ResolveLabelMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveLabelMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveLabelMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveLabelMethod "translateCoordinates" o = Gtk.Widget.WidgetTranslateCoordinatesMethodInfo
    ResolveLabelMethod "triggerTooltipQuery" o = Gtk.Widget.WidgetTriggerTooltipQueryMethodInfo
    ResolveLabelMethod "unmap" o = Gtk.Widget.WidgetUnmapMethodInfo
    ResolveLabelMethod "unparent" o = Gtk.Widget.WidgetUnparentMethodInfo
    ResolveLabelMethod "unrealize" o = Gtk.Widget.WidgetUnrealizeMethodInfo
    ResolveLabelMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveLabelMethod "unsetStateFlags" o = Gtk.Widget.WidgetUnsetStateFlagsMethodInfo
    ResolveLabelMethod "updateCaretPosition" o = Gtk.AccessibleText.AccessibleTextUpdateCaretPositionMethodInfo
    ResolveLabelMethod "updateContents" o = Gtk.AccessibleText.AccessibleTextUpdateContentsMethodInfo
    ResolveLabelMethod "updateNextAccessibleSibling" o = Gtk.Accessible.AccessibleUpdateNextAccessibleSiblingMethodInfo
    ResolveLabelMethod "updatePlatformState" o = Gtk.Accessible.AccessibleUpdatePlatformStateMethodInfo
    ResolveLabelMethod "updateProperty" o = Gtk.Accessible.AccessibleUpdatePropertyMethodInfo
    ResolveLabelMethod "updateRelation" o = Gtk.Accessible.AccessibleUpdateRelationMethodInfo
    ResolveLabelMethod "updateSelectionBound" o = Gtk.AccessibleText.AccessibleTextUpdateSelectionBoundMethodInfo
    ResolveLabelMethod "updateState" o = Gtk.Accessible.AccessibleUpdateStateMethodInfo
    ResolveLabelMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveLabelMethod "getAccessibleParent" o = Gtk.Accessible.AccessibleGetAccessibleParentMethodInfo
    ResolveLabelMethod "getAccessibleRole" o = Gtk.Accessible.AccessibleGetAccessibleRoleMethodInfo
    ResolveLabelMethod "getAllocatedBaseline" o = Gtk.Widget.WidgetGetAllocatedBaselineMethodInfo
    ResolveLabelMethod "getAllocatedHeight" o = Gtk.Widget.WidgetGetAllocatedHeightMethodInfo
    ResolveLabelMethod "getAllocatedWidth" o = Gtk.Widget.WidgetGetAllocatedWidthMethodInfo
    ResolveLabelMethod "getAllocation" o = Gtk.Widget.WidgetGetAllocationMethodInfo
    ResolveLabelMethod "getAncestor" o = Gtk.Widget.WidgetGetAncestorMethodInfo
    ResolveLabelMethod "getAtContext" o = Gtk.Accessible.AccessibleGetAtContextMethodInfo
    ResolveLabelMethod "getAttributes" o = LabelGetAttributesMethodInfo
    ResolveLabelMethod "getBaseline" o = Gtk.Widget.WidgetGetBaselineMethodInfo
    ResolveLabelMethod "getBounds" o = Gtk.Accessible.AccessibleGetBoundsMethodInfo
    ResolveLabelMethod "getBuildableId" o = Gtk.Buildable.BuildableGetBuildableIdMethodInfo
    ResolveLabelMethod "getCanFocus" o = Gtk.Widget.WidgetGetCanFocusMethodInfo
    ResolveLabelMethod "getCanTarget" o = Gtk.Widget.WidgetGetCanTargetMethodInfo
    ResolveLabelMethod "getChildVisible" o = Gtk.Widget.WidgetGetChildVisibleMethodInfo
    ResolveLabelMethod "getClipboard" o = Gtk.Widget.WidgetGetClipboardMethodInfo
    ResolveLabelMethod "getColor" o = Gtk.Widget.WidgetGetColorMethodInfo
    ResolveLabelMethod "getCssClasses" o = Gtk.Widget.WidgetGetCssClassesMethodInfo
    ResolveLabelMethod "getCssName" o = Gtk.Widget.WidgetGetCssNameMethodInfo
    ResolveLabelMethod "getCurrentUri" o = LabelGetCurrentUriMethodInfo
    ResolveLabelMethod "getCursor" o = Gtk.Widget.WidgetGetCursorMethodInfo
    ResolveLabelMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveLabelMethod "getDirection" o = Gtk.Widget.WidgetGetDirectionMethodInfo
    ResolveLabelMethod "getDisplay" o = Gtk.Widget.WidgetGetDisplayMethodInfo
    ResolveLabelMethod "getEllipsize" o = LabelGetEllipsizeMethodInfo
    ResolveLabelMethod "getExtraMenu" o = LabelGetExtraMenuMethodInfo
    ResolveLabelMethod "getFirstAccessibleChild" o = Gtk.Accessible.AccessibleGetFirstAccessibleChildMethodInfo
    ResolveLabelMethod "getFirstChild" o = Gtk.Widget.WidgetGetFirstChildMethodInfo
    ResolveLabelMethod "getFocusChild" o = Gtk.Widget.WidgetGetFocusChildMethodInfo
    ResolveLabelMethod "getFocusOnClick" o = Gtk.Widget.WidgetGetFocusOnClickMethodInfo
    ResolveLabelMethod "getFocusable" o = Gtk.Widget.WidgetGetFocusableMethodInfo
    ResolveLabelMethod "getFontMap" o = Gtk.Widget.WidgetGetFontMapMethodInfo
    ResolveLabelMethod "getFontOptions" o = Gtk.Widget.WidgetGetFontOptionsMethodInfo
    ResolveLabelMethod "getFrameClock" o = Gtk.Widget.WidgetGetFrameClockMethodInfo
    ResolveLabelMethod "getHalign" o = Gtk.Widget.WidgetGetHalignMethodInfo
    ResolveLabelMethod "getHasTooltip" o = Gtk.Widget.WidgetGetHasTooltipMethodInfo
    ResolveLabelMethod "getHeight" o = Gtk.Widget.WidgetGetHeightMethodInfo
    ResolveLabelMethod "getHexpand" o = Gtk.Widget.WidgetGetHexpandMethodInfo
    ResolveLabelMethod "getHexpandSet" o = Gtk.Widget.WidgetGetHexpandSetMethodInfo
    ResolveLabelMethod "getJustify" o = LabelGetJustifyMethodInfo
    ResolveLabelMethod "getLabel" o = LabelGetLabelMethodInfo
    ResolveLabelMethod "getLastChild" o = Gtk.Widget.WidgetGetLastChildMethodInfo
    ResolveLabelMethod "getLayout" o = LabelGetLayoutMethodInfo
    ResolveLabelMethod "getLayoutManager" o = Gtk.Widget.WidgetGetLayoutManagerMethodInfo
    ResolveLabelMethod "getLayoutOffsets" o = LabelGetLayoutOffsetsMethodInfo
    ResolveLabelMethod "getLimitEvents" o = Gtk.Widget.WidgetGetLimitEventsMethodInfo
    ResolveLabelMethod "getLines" o = LabelGetLinesMethodInfo
    ResolveLabelMethod "getMapped" o = Gtk.Widget.WidgetGetMappedMethodInfo
    ResolveLabelMethod "getMarginBottom" o = Gtk.Widget.WidgetGetMarginBottomMethodInfo
    ResolveLabelMethod "getMarginEnd" o = Gtk.Widget.WidgetGetMarginEndMethodInfo
    ResolveLabelMethod "getMarginStart" o = Gtk.Widget.WidgetGetMarginStartMethodInfo
    ResolveLabelMethod "getMarginTop" o = Gtk.Widget.WidgetGetMarginTopMethodInfo
    ResolveLabelMethod "getMaxWidthChars" o = LabelGetMaxWidthCharsMethodInfo
    ResolveLabelMethod "getMnemonicKeyval" o = LabelGetMnemonicKeyvalMethodInfo
    ResolveLabelMethod "getMnemonicWidget" o = LabelGetMnemonicWidgetMethodInfo
    ResolveLabelMethod "getName" o = Gtk.Widget.WidgetGetNameMethodInfo
    ResolveLabelMethod "getNative" o = Gtk.Widget.WidgetGetNativeMethodInfo
    ResolveLabelMethod "getNaturalWrapMode" o = LabelGetNaturalWrapModeMethodInfo
    ResolveLabelMethod "getNextAccessibleSibling" o = Gtk.Accessible.AccessibleGetNextAccessibleSiblingMethodInfo
    ResolveLabelMethod "getNextSibling" o = Gtk.Widget.WidgetGetNextSiblingMethodInfo
    ResolveLabelMethod "getOpacity" o = Gtk.Widget.WidgetGetOpacityMethodInfo
    ResolveLabelMethod "getOverflow" o = Gtk.Widget.WidgetGetOverflowMethodInfo
    ResolveLabelMethod "getPangoContext" o = Gtk.Widget.WidgetGetPangoContextMethodInfo
    ResolveLabelMethod "getParent" o = Gtk.Widget.WidgetGetParentMethodInfo
    ResolveLabelMethod "getPlatformState" o = Gtk.Accessible.AccessibleGetPlatformStateMethodInfo
    ResolveLabelMethod "getPreferredSize" o = Gtk.Widget.WidgetGetPreferredSizeMethodInfo
    ResolveLabelMethod "getPrevSibling" o = Gtk.Widget.WidgetGetPrevSiblingMethodInfo
    ResolveLabelMethod "getPrimaryClipboard" o = Gtk.Widget.WidgetGetPrimaryClipboardMethodInfo
    ResolveLabelMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveLabelMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveLabelMethod "getRealized" o = Gtk.Widget.WidgetGetRealizedMethodInfo
    ResolveLabelMethod "getReceivesDefault" o = Gtk.Widget.WidgetGetReceivesDefaultMethodInfo
    ResolveLabelMethod "getRequestMode" o = Gtk.Widget.WidgetGetRequestModeMethodInfo
    ResolveLabelMethod "getRoot" o = Gtk.Widget.WidgetGetRootMethodInfo
    ResolveLabelMethod "getScaleFactor" o = Gtk.Widget.WidgetGetScaleFactorMethodInfo
    ResolveLabelMethod "getSelectable" o = LabelGetSelectableMethodInfo
    ResolveLabelMethod "getSelectionBounds" o = LabelGetSelectionBoundsMethodInfo
    ResolveLabelMethod "getSensitive" o = Gtk.Widget.WidgetGetSensitiveMethodInfo
    ResolveLabelMethod "getSettings" o = Gtk.Widget.WidgetGetSettingsMethodInfo
    ResolveLabelMethod "getSingleLineMode" o = LabelGetSingleLineModeMethodInfo
    ResolveLabelMethod "getSize" o = Gtk.Widget.WidgetGetSizeMethodInfo
    ResolveLabelMethod "getSizeRequest" o = Gtk.Widget.WidgetGetSizeRequestMethodInfo
    ResolveLabelMethod "getStateFlags" o = Gtk.Widget.WidgetGetStateFlagsMethodInfo
    ResolveLabelMethod "getStyleContext" o = Gtk.Widget.WidgetGetStyleContextMethodInfo
    ResolveLabelMethod "getTabs" o = LabelGetTabsMethodInfo
    ResolveLabelMethod "getTemplateChild" o = Gtk.Widget.WidgetGetTemplateChildMethodInfo
    ResolveLabelMethod "getText" o = LabelGetTextMethodInfo
    ResolveLabelMethod "getTooltipMarkup" o = Gtk.Widget.WidgetGetTooltipMarkupMethodInfo
    ResolveLabelMethod "getTooltipText" o = Gtk.Widget.WidgetGetTooltipTextMethodInfo
    ResolveLabelMethod "getUseMarkup" o = LabelGetUseMarkupMethodInfo
    ResolveLabelMethod "getUseUnderline" o = LabelGetUseUnderlineMethodInfo
    ResolveLabelMethod "getValign" o = Gtk.Widget.WidgetGetValignMethodInfo
    ResolveLabelMethod "getVexpand" o = Gtk.Widget.WidgetGetVexpandMethodInfo
    ResolveLabelMethod "getVexpandSet" o = Gtk.Widget.WidgetGetVexpandSetMethodInfo
    ResolveLabelMethod "getVisible" o = Gtk.Widget.WidgetGetVisibleMethodInfo
    ResolveLabelMethod "getWidth" o = Gtk.Widget.WidgetGetWidthMethodInfo
    ResolveLabelMethod "getWidthChars" o = LabelGetWidthCharsMethodInfo
    ResolveLabelMethod "getWrap" o = LabelGetWrapMethodInfo
    ResolveLabelMethod "getWrapMode" o = LabelGetWrapModeMethodInfo
    ResolveLabelMethod "getXalign" o = LabelGetXalignMethodInfo
    ResolveLabelMethod "getYalign" o = LabelGetYalignMethodInfo
    ResolveLabelMethod "setAccessibleParent" o = Gtk.Accessible.AccessibleSetAccessibleParentMethodInfo
    ResolveLabelMethod "setAttributes" o = LabelSetAttributesMethodInfo
    ResolveLabelMethod "setCanFocus" o = Gtk.Widget.WidgetSetCanFocusMethodInfo
    ResolveLabelMethod "setCanTarget" o = Gtk.Widget.WidgetSetCanTargetMethodInfo
    ResolveLabelMethod "setChildVisible" o = Gtk.Widget.WidgetSetChildVisibleMethodInfo
    ResolveLabelMethod "setCssClasses" o = Gtk.Widget.WidgetSetCssClassesMethodInfo
    ResolveLabelMethod "setCursor" o = Gtk.Widget.WidgetSetCursorMethodInfo
    ResolveLabelMethod "setCursorFromName" o = Gtk.Widget.WidgetSetCursorFromNameMethodInfo
    ResolveLabelMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveLabelMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveLabelMethod "setDirection" o = Gtk.Widget.WidgetSetDirectionMethodInfo
    ResolveLabelMethod "setEllipsize" o = LabelSetEllipsizeMethodInfo
    ResolveLabelMethod "setExtraMenu" o = LabelSetExtraMenuMethodInfo
    ResolveLabelMethod "setFocusChild" o = Gtk.Widget.WidgetSetFocusChildMethodInfo
    ResolveLabelMethod "setFocusOnClick" o = Gtk.Widget.WidgetSetFocusOnClickMethodInfo
    ResolveLabelMethod "setFocusable" o = Gtk.Widget.WidgetSetFocusableMethodInfo
    ResolveLabelMethod "setFontMap" o = Gtk.Widget.WidgetSetFontMapMethodInfo
    ResolveLabelMethod "setFontOptions" o = Gtk.Widget.WidgetSetFontOptionsMethodInfo
    ResolveLabelMethod "setHalign" o = Gtk.Widget.WidgetSetHalignMethodInfo
    ResolveLabelMethod "setHasTooltip" o = Gtk.Widget.WidgetSetHasTooltipMethodInfo
    ResolveLabelMethod "setHexpand" o = Gtk.Widget.WidgetSetHexpandMethodInfo
    ResolveLabelMethod "setHexpandSet" o = Gtk.Widget.WidgetSetHexpandSetMethodInfo
    ResolveLabelMethod "setJustify" o = LabelSetJustifyMethodInfo
    ResolveLabelMethod "setLabel" o = LabelSetLabelMethodInfo
    ResolveLabelMethod "setLayoutManager" o = Gtk.Widget.WidgetSetLayoutManagerMethodInfo
    ResolveLabelMethod "setLimitEvents" o = Gtk.Widget.WidgetSetLimitEventsMethodInfo
    ResolveLabelMethod "setLines" o = LabelSetLinesMethodInfo
    ResolveLabelMethod "setMarginBottom" o = Gtk.Widget.WidgetSetMarginBottomMethodInfo
    ResolveLabelMethod "setMarginEnd" o = Gtk.Widget.WidgetSetMarginEndMethodInfo
    ResolveLabelMethod "setMarginStart" o = Gtk.Widget.WidgetSetMarginStartMethodInfo
    ResolveLabelMethod "setMarginTop" o = Gtk.Widget.WidgetSetMarginTopMethodInfo
    ResolveLabelMethod "setMarkup" o = LabelSetMarkupMethodInfo
    ResolveLabelMethod "setMarkupWithMnemonic" o = LabelSetMarkupWithMnemonicMethodInfo
    ResolveLabelMethod "setMaxWidthChars" o = LabelSetMaxWidthCharsMethodInfo
    ResolveLabelMethod "setMnemonicWidget" o = LabelSetMnemonicWidgetMethodInfo
    ResolveLabelMethod "setName" o = Gtk.Widget.WidgetSetNameMethodInfo
    ResolveLabelMethod "setNaturalWrapMode" o = LabelSetNaturalWrapModeMethodInfo
    ResolveLabelMethod "setOpacity" o = Gtk.Widget.WidgetSetOpacityMethodInfo
    ResolveLabelMethod "setOverflow" o = Gtk.Widget.WidgetSetOverflowMethodInfo
    ResolveLabelMethod "setParent" o = Gtk.Widget.WidgetSetParentMethodInfo
    ResolveLabelMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveLabelMethod "setReceivesDefault" o = Gtk.Widget.WidgetSetReceivesDefaultMethodInfo
    ResolveLabelMethod "setSelectable" o = LabelSetSelectableMethodInfo
    ResolveLabelMethod "setSensitive" o = Gtk.Widget.WidgetSetSensitiveMethodInfo
    ResolveLabelMethod "setSingleLineMode" o = LabelSetSingleLineModeMethodInfo
    ResolveLabelMethod "setSizeRequest" o = Gtk.Widget.WidgetSetSizeRequestMethodInfo
    ResolveLabelMethod "setStateFlags" o = Gtk.Widget.WidgetSetStateFlagsMethodInfo
    ResolveLabelMethod "setTabs" o = LabelSetTabsMethodInfo
    ResolveLabelMethod "setText" o = LabelSetTextMethodInfo
    ResolveLabelMethod "setTextWithMnemonic" o = LabelSetTextWithMnemonicMethodInfo
    ResolveLabelMethod "setTooltipMarkup" o = Gtk.Widget.WidgetSetTooltipMarkupMethodInfo
    ResolveLabelMethod "setTooltipText" o = Gtk.Widget.WidgetSetTooltipTextMethodInfo
    ResolveLabelMethod "setUseMarkup" o = LabelSetUseMarkupMethodInfo
    ResolveLabelMethod "setUseUnderline" o = LabelSetUseUnderlineMethodInfo
    ResolveLabelMethod "setValign" o = Gtk.Widget.WidgetSetValignMethodInfo
    ResolveLabelMethod "setVexpand" o = Gtk.Widget.WidgetSetVexpandMethodInfo
    ResolveLabelMethod "setVexpandSet" o = Gtk.Widget.WidgetSetVexpandSetMethodInfo
    ResolveLabelMethod "setVisible" o = Gtk.Widget.WidgetSetVisibleMethodInfo
    ResolveLabelMethod "setWidthChars" o = LabelSetWidthCharsMethodInfo
    ResolveLabelMethod "setWrap" o = LabelSetWrapMethodInfo
    ResolveLabelMethod "setWrapMode" o = LabelSetWrapModeMethodInfo
    ResolveLabelMethod "setXalign" o = LabelSetXalignMethodInfo
    ResolveLabelMethod "setYalign" o = LabelSetYalignMethodInfo
    ResolveLabelMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveLabelMethod t Label, O.OverloadedMethod info Label p) => OL.IsLabel t (Label -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveLabelMethod t Label, O.OverloadedMethod info Label p, R.HasField t Label p) => R.HasField t Label p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveLabelMethod t Label, O.OverloadedMethodInfo info Label) => OL.IsLabel t (O.MethodProxy info Label) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal Label::activate-current-link
-- | Gets emitted when the user activates a link in the label.
-- 
-- The @::activate-current-link@ is a <https://docs.gtk.org/gtk4/class.SignalAction.html keybinding signal>.
-- 
-- Applications may also emit the signal with @/g_signal_emit_by_name()/@
-- if they need to control activation of URIs programmatically.
-- 
-- The default bindings for this signal are all forms of the \<kbd>Enter\<\/kbd> key.
type LabelActivateCurrentLinkCallback =
    IO ()

type C_LabelActivateCurrentLinkCallback =
    Ptr Label ->                            -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_LabelActivateCurrentLinkCallback`.
foreign import ccall "wrapper"
    mk_LabelActivateCurrentLinkCallback :: C_LabelActivateCurrentLinkCallback -> IO (FunPtr C_LabelActivateCurrentLinkCallback)

wrap_LabelActivateCurrentLinkCallback :: 
    GObject a => (a -> LabelActivateCurrentLinkCallback) ->
    C_LabelActivateCurrentLinkCallback
wrap_LabelActivateCurrentLinkCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [activateCurrentLink](#signal:activateCurrentLink) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' label #activateCurrentLink callback
-- @
-- 
-- 
onLabelActivateCurrentLink :: (IsLabel a, MonadIO m) => a -> ((?self :: a) => LabelActivateCurrentLinkCallback) -> m SignalHandlerId
onLabelActivateCurrentLink obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_LabelActivateCurrentLinkCallback wrapped
    wrapped'' <- mk_LabelActivateCurrentLinkCallback wrapped'
    connectSignalFunPtr obj "activate-current-link" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [activateCurrentLink](#signal:activateCurrentLink) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' label #activateCurrentLink callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterLabelActivateCurrentLink :: (IsLabel a, MonadIO m) => a -> ((?self :: a) => LabelActivateCurrentLinkCallback) -> m SignalHandlerId
afterLabelActivateCurrentLink obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_LabelActivateCurrentLinkCallback wrapped
    wrapped'' <- mk_LabelActivateCurrentLinkCallback wrapped'
    connectSignalFunPtr obj "activate-current-link" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data LabelActivateCurrentLinkSignalInfo
instance SignalInfo LabelActivateCurrentLinkSignalInfo where
    type HaskellCallbackType LabelActivateCurrentLinkSignalInfo = LabelActivateCurrentLinkCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_LabelActivateCurrentLinkCallback cb
        cb'' <- mk_LabelActivateCurrentLinkCallback cb'
        connectSignalFunPtr obj "activate-current-link" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label::activate-current-link"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:signal:activateCurrentLink"})

#endif

-- signal Label::activate-link
-- | Gets emitted to activate a URI.
-- 
-- Applications may connect to it to override the default behaviour,
-- which is to call 'GI.Gtk.Objects.FileLauncher.fileLauncherLaunch'.
type LabelActivateLinkCallback =
    T.Text
    -- ^ /@uri@/: the URI that is activated
    -> IO Bool
    -- ^ __Returns:__ true if the link has been activated

type C_LabelActivateLinkCallback =
    Ptr Label ->                            -- object
    CString ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_LabelActivateLinkCallback`.
foreign import ccall "wrapper"
    mk_LabelActivateLinkCallback :: C_LabelActivateLinkCallback -> IO (FunPtr C_LabelActivateLinkCallback)

wrap_LabelActivateLinkCallback :: 
    GObject a => (a -> LabelActivateLinkCallback) ->
    C_LabelActivateLinkCallback
wrap_LabelActivateLinkCallback gi'cb gi'selfPtr uri _ = do
    uri' <- cstringToText uri
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  uri'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [activateLink](#signal:activateLink) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' label #activateLink callback
-- @
-- 
-- 
onLabelActivateLink :: (IsLabel a, MonadIO m) => a -> ((?self :: a) => LabelActivateLinkCallback) -> m SignalHandlerId
onLabelActivateLink obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_LabelActivateLinkCallback wrapped
    wrapped'' <- mk_LabelActivateLinkCallback wrapped'
    connectSignalFunPtr obj "activate-link" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [activateLink](#signal:activateLink) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' label #activateLink callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterLabelActivateLink :: (IsLabel a, MonadIO m) => a -> ((?self :: a) => LabelActivateLinkCallback) -> m SignalHandlerId
afterLabelActivateLink obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_LabelActivateLinkCallback wrapped
    wrapped'' <- mk_LabelActivateLinkCallback wrapped'
    connectSignalFunPtr obj "activate-link" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data LabelActivateLinkSignalInfo
instance SignalInfo LabelActivateLinkSignalInfo where
    type HaskellCallbackType LabelActivateLinkSignalInfo = LabelActivateLinkCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_LabelActivateLinkCallback cb
        cb'' <- mk_LabelActivateLinkCallback cb'
        connectSignalFunPtr obj "activate-link" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label::activate-link"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:signal:activateLink"})

#endif

-- signal Label::copy-clipboard
-- | Gets emitted to copy the selection to the clipboard.
-- 
-- The @::copy-clipboard@ signal is a <https://docs.gtk.org/gtk4/class.SignalAction.html keybinding signal>.
-- 
-- The default binding for this signal is \<kbd>Ctrl\<\/kbd>+\<kbd>c\<\/kbd>.
type LabelCopyClipboardCallback =
    IO ()

type C_LabelCopyClipboardCallback =
    Ptr Label ->                            -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_LabelCopyClipboardCallback`.
foreign import ccall "wrapper"
    mk_LabelCopyClipboardCallback :: C_LabelCopyClipboardCallback -> IO (FunPtr C_LabelCopyClipboardCallback)

wrap_LabelCopyClipboardCallback :: 
    GObject a => (a -> LabelCopyClipboardCallback) ->
    C_LabelCopyClipboardCallback
wrap_LabelCopyClipboardCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [copyClipboard](#signal:copyClipboard) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' label #copyClipboard callback
-- @
-- 
-- 
onLabelCopyClipboard :: (IsLabel a, MonadIO m) => a -> ((?self :: a) => LabelCopyClipboardCallback) -> m SignalHandlerId
onLabelCopyClipboard obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_LabelCopyClipboardCallback wrapped
    wrapped'' <- mk_LabelCopyClipboardCallback wrapped'
    connectSignalFunPtr obj "copy-clipboard" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [copyClipboard](#signal:copyClipboard) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' label #copyClipboard callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterLabelCopyClipboard :: (IsLabel a, MonadIO m) => a -> ((?self :: a) => LabelCopyClipboardCallback) -> m SignalHandlerId
afterLabelCopyClipboard obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_LabelCopyClipboardCallback wrapped
    wrapped'' <- mk_LabelCopyClipboardCallback wrapped'
    connectSignalFunPtr obj "copy-clipboard" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data LabelCopyClipboardSignalInfo
instance SignalInfo LabelCopyClipboardSignalInfo where
    type HaskellCallbackType LabelCopyClipboardSignalInfo = LabelCopyClipboardCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_LabelCopyClipboardCallback cb
        cb'' <- mk_LabelCopyClipboardCallback cb'
        connectSignalFunPtr obj "copy-clipboard" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label::copy-clipboard"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:signal:copyClipboard"})

#endif

-- signal Label::move-cursor
-- | Gets emitted when the user initiates a cursor movement.
-- 
-- The @::move-cursor@ signal is a <https://docs.gtk.org/gtk4/class.SignalAction.html keybinding signal>.
-- If the cursor is not visible in /@entry@/, this signal causes the viewport to
-- be moved instead.
-- 
-- Applications should not connect to it, but may emit it with
-- @/GObject.signal_emit_by_name/@ if they need to control
-- the cursor programmatically.
-- 
-- The default bindings for this signal come in two variants, the
-- variant with the \<kbd>Shift\<\/kbd> modifier extends the selection,
-- the variant without the \<kbd>Shift\<\/kbd> modifier does not.
-- There are too many key combinations to list them all here.
-- 
-- * \<kbd>←\<\/kbd>, \<kbd>→\<\/kbd>, \<kbd>↑\<\/kbd>, \<kbd>↓\<\/kbd>
--   move by individual characters\/lines
-- * \<kbd>Ctrl\<\/kbd>+\<kbd>←\<\/kbd>, etc. move by words\/paragraphs
-- * \<kbd>Home\<\/kbd> and \<kbd>End\<\/kbd> move to the ends of the buffer
type LabelMoveCursorCallback =
    Gtk.Enums.MovementStep
    -- ^ /@step@/: the granularity of the move, as a @GtkMovementStep@
    -> Int32
    -- ^ /@count@/: the number of /@step@/ units to move
    -> Bool
    -- ^ /@extendSelection@/: true if the move should extend the selection
    -> IO ()

type C_LabelMoveCursorCallback =
    Ptr Label ->                            -- object
    CUInt ->
    Int32 ->
    CInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_LabelMoveCursorCallback`.
foreign import ccall "wrapper"
    mk_LabelMoveCursorCallback :: C_LabelMoveCursorCallback -> IO (FunPtr C_LabelMoveCursorCallback)

wrap_LabelMoveCursorCallback :: 
    GObject a => (a -> LabelMoveCursorCallback) ->
    C_LabelMoveCursorCallback
wrap_LabelMoveCursorCallback gi'cb gi'selfPtr step count extendSelection _ = do
    let step' = (toEnum . fromIntegral) step
    let extendSelection' = (/= 0) extendSelection
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  step' count extendSelection'


-- | Connect a signal handler for the [moveCursor](#signal:moveCursor) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' label #moveCursor callback
-- @
-- 
-- 
onLabelMoveCursor :: (IsLabel a, MonadIO m) => a -> ((?self :: a) => LabelMoveCursorCallback) -> m SignalHandlerId
onLabelMoveCursor obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_LabelMoveCursorCallback wrapped
    wrapped'' <- mk_LabelMoveCursorCallback wrapped'
    connectSignalFunPtr obj "move-cursor" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [moveCursor](#signal:moveCursor) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' label #moveCursor callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterLabelMoveCursor :: (IsLabel a, MonadIO m) => a -> ((?self :: a) => LabelMoveCursorCallback) -> m SignalHandlerId
afterLabelMoveCursor obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_LabelMoveCursorCallback wrapped
    wrapped'' <- mk_LabelMoveCursorCallback wrapped'
    connectSignalFunPtr obj "move-cursor" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data LabelMoveCursorSignalInfo
instance SignalInfo LabelMoveCursorSignalInfo where
    type HaskellCallbackType LabelMoveCursorSignalInfo = LabelMoveCursorCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_LabelMoveCursorCallback cb
        cb'' <- mk_LabelMoveCursorCallback cb'
        connectSignalFunPtr obj "move-cursor" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label::move-cursor"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:signal:moveCursor"})

#endif

-- VVV Prop "attributes"
   -- Type: TInterface (Name {namespace = "Pango", name = "AttrList"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@attributes@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #attributes
-- @
getLabelAttributes :: (MonadIO m, IsLabel o) => o -> m (Maybe Pango.AttrList.AttrList)
getLabelAttributes obj = MIO.liftIO $ B.Properties.getObjectPropertyBoxed obj "attributes" Pango.AttrList.AttrList

-- | Set the value of the “@attributes@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #attributes 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelAttributes :: (MonadIO m, IsLabel o) => o -> Pango.AttrList.AttrList -> m ()
setLabelAttributes obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBoxed obj "attributes" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@attributes@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelAttributes :: (IsLabel o, MIO.MonadIO m) => Pango.AttrList.AttrList -> m (GValueConstruct o)
constructLabelAttributes val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBoxed "attributes" (P.Just val)

-- | Set the value of the “@attributes@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #attributes
-- @
clearLabelAttributes :: (MonadIO m, IsLabel o) => o -> m ()
clearLabelAttributes obj = liftIO $ B.Properties.setObjectPropertyBoxed obj "attributes" (Nothing :: Maybe Pango.AttrList.AttrList)

#if defined(ENABLE_OVERLOADING)
data LabelAttributesPropertyInfo
instance AttrInfo LabelAttributesPropertyInfo where
    type AttrAllowedOps LabelAttributesPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint LabelAttributesPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelAttributesPropertyInfo = (~) Pango.AttrList.AttrList
    type AttrTransferTypeConstraint LabelAttributesPropertyInfo = (~) Pango.AttrList.AttrList
    type AttrTransferType LabelAttributesPropertyInfo = Pango.AttrList.AttrList
    type AttrGetType LabelAttributesPropertyInfo = (Maybe Pango.AttrList.AttrList)
    type AttrLabel LabelAttributesPropertyInfo = "attributes"
    type AttrOrigin LabelAttributesPropertyInfo = Label
    attrGet = getLabelAttributes
    attrSet = setLabelAttributes
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelAttributes
    attrClear = clearLabelAttributes
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.attributes"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:attributes"
        })
#endif

-- VVV Prop "ellipsize"
   -- Type: TInterface (Name {namespace = "Pango", name = "EllipsizeMode"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@ellipsize@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #ellipsize
-- @
getLabelEllipsize :: (MonadIO m, IsLabel o) => o -> m Pango.Enums.EllipsizeMode
getLabelEllipsize obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "ellipsize"

-- | Set the value of the “@ellipsize@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #ellipsize 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelEllipsize :: (MonadIO m, IsLabel o) => o -> Pango.Enums.EllipsizeMode -> m ()
setLabelEllipsize obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "ellipsize" val

-- | Construct a t'GValueConstruct' with valid value for the “@ellipsize@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelEllipsize :: (IsLabel o, MIO.MonadIO m) => Pango.Enums.EllipsizeMode -> m (GValueConstruct o)
constructLabelEllipsize val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "ellipsize" val

#if defined(ENABLE_OVERLOADING)
data LabelEllipsizePropertyInfo
instance AttrInfo LabelEllipsizePropertyInfo where
    type AttrAllowedOps LabelEllipsizePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelEllipsizePropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelEllipsizePropertyInfo = (~) Pango.Enums.EllipsizeMode
    type AttrTransferTypeConstraint LabelEllipsizePropertyInfo = (~) Pango.Enums.EllipsizeMode
    type AttrTransferType LabelEllipsizePropertyInfo = Pango.Enums.EllipsizeMode
    type AttrGetType LabelEllipsizePropertyInfo = Pango.Enums.EllipsizeMode
    type AttrLabel LabelEllipsizePropertyInfo = "ellipsize"
    type AttrOrigin LabelEllipsizePropertyInfo = Label
    attrGet = getLabelEllipsize
    attrSet = setLabelEllipsize
    attrPut = setLabelEllipsize
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelEllipsize
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.ellipsize"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:ellipsize"
        })
#endif

-- VVV Prop "extra-menu"
   -- Type: TInterface (Name {namespace = "Gio", name = "MenuModel"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@extra-menu@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #extraMenu
-- @
getLabelExtraMenu :: (MonadIO m, IsLabel o) => o -> m (Maybe Gio.MenuModel.MenuModel)
getLabelExtraMenu obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "extra-menu" Gio.MenuModel.MenuModel

-- | Set the value of the “@extra-menu@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #extraMenu 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelExtraMenu :: (MonadIO m, IsLabel o, Gio.MenuModel.IsMenuModel a) => o -> a -> m ()
setLabelExtraMenu obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyObject obj "extra-menu" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@extra-menu@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelExtraMenu :: (IsLabel o, MIO.MonadIO m, Gio.MenuModel.IsMenuModel a) => a -> m (GValueConstruct o)
constructLabelExtraMenu val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "extra-menu" (P.Just val)

-- | Set the value of the “@extra-menu@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #extraMenu
-- @
clearLabelExtraMenu :: (MonadIO m, IsLabel o) => o -> m ()
clearLabelExtraMenu obj = liftIO $ B.Properties.setObjectPropertyObject obj "extra-menu" (Nothing :: Maybe Gio.MenuModel.MenuModel)

#if defined(ENABLE_OVERLOADING)
data LabelExtraMenuPropertyInfo
instance AttrInfo LabelExtraMenuPropertyInfo where
    type AttrAllowedOps LabelExtraMenuPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint LabelExtraMenuPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelExtraMenuPropertyInfo = Gio.MenuModel.IsMenuModel
    type AttrTransferTypeConstraint LabelExtraMenuPropertyInfo = Gio.MenuModel.IsMenuModel
    type AttrTransferType LabelExtraMenuPropertyInfo = Gio.MenuModel.MenuModel
    type AttrGetType LabelExtraMenuPropertyInfo = (Maybe Gio.MenuModel.MenuModel)
    type AttrLabel LabelExtraMenuPropertyInfo = "extra-menu"
    type AttrOrigin LabelExtraMenuPropertyInfo = Label
    attrGet = getLabelExtraMenu
    attrSet = setLabelExtraMenu
    attrPut = undefined
    attrTransfer _ v = do
        unsafeCastTo Gio.MenuModel.MenuModel v
    attrConstruct = constructLabelExtraMenu
    attrClear = clearLabelExtraMenu
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.extraMenu"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:extraMenu"
        })
#endif

-- VVV Prop "justify"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Justification"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@justify@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #justify
-- @
getLabelJustify :: (MonadIO m, IsLabel o) => o -> m Gtk.Enums.Justification
getLabelJustify obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "justify"

-- | Set the value of the “@justify@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #justify 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelJustify :: (MonadIO m, IsLabel o) => o -> Gtk.Enums.Justification -> m ()
setLabelJustify obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "justify" val

-- | Construct a t'GValueConstruct' with valid value for the “@justify@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelJustify :: (IsLabel o, MIO.MonadIO m) => Gtk.Enums.Justification -> m (GValueConstruct o)
constructLabelJustify val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "justify" val

#if defined(ENABLE_OVERLOADING)
data LabelJustifyPropertyInfo
instance AttrInfo LabelJustifyPropertyInfo where
    type AttrAllowedOps LabelJustifyPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelJustifyPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelJustifyPropertyInfo = (~) Gtk.Enums.Justification
    type AttrTransferTypeConstraint LabelJustifyPropertyInfo = (~) Gtk.Enums.Justification
    type AttrTransferType LabelJustifyPropertyInfo = Gtk.Enums.Justification
    type AttrGetType LabelJustifyPropertyInfo = Gtk.Enums.Justification
    type AttrLabel LabelJustifyPropertyInfo = "justify"
    type AttrOrigin LabelJustifyPropertyInfo = Label
    attrGet = getLabelJustify
    attrSet = setLabelJustify
    attrPut = setLabelJustify
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelJustify
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.justify"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:justify"
        })
#endif

-- VVV Prop "label"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@label@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #label
-- @
getLabelLabel :: (MonadIO m, IsLabel o) => o -> m T.Text
getLabelLabel obj = MIO.liftIO $ checkUnexpectedNothing "getLabelLabel" $ B.Properties.getObjectPropertyString obj "label"

-- | Set the value of the “@label@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #label 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelLabel :: (MonadIO m, IsLabel o) => o -> T.Text -> m ()
setLabelLabel obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "label" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@label@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelLabel :: (IsLabel o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructLabelLabel val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "label" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data LabelLabelPropertyInfo
instance AttrInfo LabelLabelPropertyInfo where
    type AttrAllowedOps LabelLabelPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelLabelPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelLabelPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint LabelLabelPropertyInfo = (~) T.Text
    type AttrTransferType LabelLabelPropertyInfo = T.Text
    type AttrGetType LabelLabelPropertyInfo = T.Text
    type AttrLabel LabelLabelPropertyInfo = "label"
    type AttrOrigin LabelLabelPropertyInfo = Label
    attrGet = getLabelLabel
    attrSet = setLabelLabel
    attrPut = setLabelLabel
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelLabel
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.label"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:label"
        })
#endif

-- VVV Prop "lines"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@lines@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #lines
-- @
getLabelLines :: (MonadIO m, IsLabel o) => o -> m Int32
getLabelLines obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "lines"

-- | Set the value of the “@lines@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #lines 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelLines :: (MonadIO m, IsLabel o) => o -> Int32 -> m ()
setLabelLines obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "lines" val

-- | Construct a t'GValueConstruct' with valid value for the “@lines@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelLines :: (IsLabel o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructLabelLines val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "lines" val

#if defined(ENABLE_OVERLOADING)
data LabelLinesPropertyInfo
instance AttrInfo LabelLinesPropertyInfo where
    type AttrAllowedOps LabelLinesPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelLinesPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelLinesPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint LabelLinesPropertyInfo = (~) Int32
    type AttrTransferType LabelLinesPropertyInfo = Int32
    type AttrGetType LabelLinesPropertyInfo = Int32
    type AttrLabel LabelLinesPropertyInfo = "lines"
    type AttrOrigin LabelLinesPropertyInfo = Label
    attrGet = getLabelLines
    attrSet = setLabelLines
    attrPut = setLabelLines
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelLines
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.lines"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:lines"
        })
#endif

-- VVV Prop "max-width-chars"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@max-width-chars@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #maxWidthChars
-- @
getLabelMaxWidthChars :: (MonadIO m, IsLabel o) => o -> m Int32
getLabelMaxWidthChars obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "max-width-chars"

-- | Set the value of the “@max-width-chars@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #maxWidthChars 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelMaxWidthChars :: (MonadIO m, IsLabel o) => o -> Int32 -> m ()
setLabelMaxWidthChars obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "max-width-chars" val

-- | Construct a t'GValueConstruct' with valid value for the “@max-width-chars@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelMaxWidthChars :: (IsLabel o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructLabelMaxWidthChars val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "max-width-chars" val

#if defined(ENABLE_OVERLOADING)
data LabelMaxWidthCharsPropertyInfo
instance AttrInfo LabelMaxWidthCharsPropertyInfo where
    type AttrAllowedOps LabelMaxWidthCharsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelMaxWidthCharsPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelMaxWidthCharsPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint LabelMaxWidthCharsPropertyInfo = (~) Int32
    type AttrTransferType LabelMaxWidthCharsPropertyInfo = Int32
    type AttrGetType LabelMaxWidthCharsPropertyInfo = Int32
    type AttrLabel LabelMaxWidthCharsPropertyInfo = "max-width-chars"
    type AttrOrigin LabelMaxWidthCharsPropertyInfo = Label
    attrGet = getLabelMaxWidthChars
    attrSet = setLabelMaxWidthChars
    attrPut = setLabelMaxWidthChars
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelMaxWidthChars
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.maxWidthChars"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:maxWidthChars"
        })
#endif

-- VVV Prop "mnemonic-keyval"
   -- Type: TBasicType TUInt
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@mnemonic-keyval@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #mnemonicKeyval
-- @
getLabelMnemonicKeyval :: (MonadIO m, IsLabel o) => o -> m Word32
getLabelMnemonicKeyval obj = MIO.liftIO $ B.Properties.getObjectPropertyUInt32 obj "mnemonic-keyval"

#if defined(ENABLE_OVERLOADING)
data LabelMnemonicKeyvalPropertyInfo
instance AttrInfo LabelMnemonicKeyvalPropertyInfo where
    type AttrAllowedOps LabelMnemonicKeyvalPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint LabelMnemonicKeyvalPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelMnemonicKeyvalPropertyInfo = (~) ()
    type AttrTransferTypeConstraint LabelMnemonicKeyvalPropertyInfo = (~) ()
    type AttrTransferType LabelMnemonicKeyvalPropertyInfo = ()
    type AttrGetType LabelMnemonicKeyvalPropertyInfo = Word32
    type AttrLabel LabelMnemonicKeyvalPropertyInfo = "mnemonic-keyval"
    type AttrOrigin LabelMnemonicKeyvalPropertyInfo = Label
    attrGet = getLabelMnemonicKeyval
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.mnemonicKeyval"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:mnemonicKeyval"
        })
#endif

-- VVV Prop "mnemonic-widget"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Widget"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just True,Just True)

-- | Get the value of the “@mnemonic-widget@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #mnemonicWidget
-- @
getLabelMnemonicWidget :: (MonadIO m, IsLabel o) => o -> m (Maybe Gtk.Widget.Widget)
getLabelMnemonicWidget obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "mnemonic-widget" Gtk.Widget.Widget

-- | Set the value of the “@mnemonic-widget@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #mnemonicWidget 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelMnemonicWidget :: (MonadIO m, IsLabel o, Gtk.Widget.IsWidget a) => o -> a -> m ()
setLabelMnemonicWidget obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyObject obj "mnemonic-widget" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@mnemonic-widget@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelMnemonicWidget :: (IsLabel o, MIO.MonadIO m, Gtk.Widget.IsWidget a) => a -> m (GValueConstruct o)
constructLabelMnemonicWidget val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "mnemonic-widget" (P.Just val)

-- | Set the value of the “@mnemonic-widget@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #mnemonicWidget
-- @
clearLabelMnemonicWidget :: (MonadIO m, IsLabel o) => o -> m ()
clearLabelMnemonicWidget obj = liftIO $ B.Properties.setObjectPropertyObject obj "mnemonic-widget" (Nothing :: Maybe Gtk.Widget.Widget)

#if defined(ENABLE_OVERLOADING)
data LabelMnemonicWidgetPropertyInfo
instance AttrInfo LabelMnemonicWidgetPropertyInfo where
    type AttrAllowedOps LabelMnemonicWidgetPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint LabelMnemonicWidgetPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelMnemonicWidgetPropertyInfo = Gtk.Widget.IsWidget
    type AttrTransferTypeConstraint LabelMnemonicWidgetPropertyInfo = Gtk.Widget.IsWidget
    type AttrTransferType LabelMnemonicWidgetPropertyInfo = Gtk.Widget.Widget
    type AttrGetType LabelMnemonicWidgetPropertyInfo = (Maybe Gtk.Widget.Widget)
    type AttrLabel LabelMnemonicWidgetPropertyInfo = "mnemonic-widget"
    type AttrOrigin LabelMnemonicWidgetPropertyInfo = Label
    attrGet = getLabelMnemonicWidget
    attrSet = setLabelMnemonicWidget
    attrPut = undefined
    attrTransfer _ v = do
        unsafeCastTo Gtk.Widget.Widget v
    attrConstruct = constructLabelMnemonicWidget
    attrClear = clearLabelMnemonicWidget
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.mnemonicWidget"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:mnemonicWidget"
        })
#endif

-- VVV Prop "natural-wrap-mode"
   -- Type: TInterface (Name {namespace = "Gtk", name = "NaturalWrapMode"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@natural-wrap-mode@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #naturalWrapMode
-- @
getLabelNaturalWrapMode :: (MonadIO m, IsLabel o) => o -> m Gtk.Enums.NaturalWrapMode
getLabelNaturalWrapMode obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "natural-wrap-mode"

-- | Set the value of the “@natural-wrap-mode@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #naturalWrapMode 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelNaturalWrapMode :: (MonadIO m, IsLabel o) => o -> Gtk.Enums.NaturalWrapMode -> m ()
setLabelNaturalWrapMode obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "natural-wrap-mode" val

-- | Construct a t'GValueConstruct' with valid value for the “@natural-wrap-mode@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelNaturalWrapMode :: (IsLabel o, MIO.MonadIO m) => Gtk.Enums.NaturalWrapMode -> m (GValueConstruct o)
constructLabelNaturalWrapMode val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "natural-wrap-mode" val

#if defined(ENABLE_OVERLOADING)
data LabelNaturalWrapModePropertyInfo
instance AttrInfo LabelNaturalWrapModePropertyInfo where
    type AttrAllowedOps LabelNaturalWrapModePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelNaturalWrapModePropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelNaturalWrapModePropertyInfo = (~) Gtk.Enums.NaturalWrapMode
    type AttrTransferTypeConstraint LabelNaturalWrapModePropertyInfo = (~) Gtk.Enums.NaturalWrapMode
    type AttrTransferType LabelNaturalWrapModePropertyInfo = Gtk.Enums.NaturalWrapMode
    type AttrGetType LabelNaturalWrapModePropertyInfo = Gtk.Enums.NaturalWrapMode
    type AttrLabel LabelNaturalWrapModePropertyInfo = "natural-wrap-mode"
    type AttrOrigin LabelNaturalWrapModePropertyInfo = Label
    attrGet = getLabelNaturalWrapMode
    attrSet = setLabelNaturalWrapMode
    attrPut = setLabelNaturalWrapMode
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelNaturalWrapMode
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.naturalWrapMode"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:naturalWrapMode"
        })
#endif

-- VVV Prop "selectable"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@selectable@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #selectable
-- @
getLabelSelectable :: (MonadIO m, IsLabel o) => o -> m Bool
getLabelSelectable obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "selectable"

-- | Set the value of the “@selectable@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #selectable 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelSelectable :: (MonadIO m, IsLabel o) => o -> Bool -> m ()
setLabelSelectable obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "selectable" val

-- | Construct a t'GValueConstruct' with valid value for the “@selectable@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelSelectable :: (IsLabel o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructLabelSelectable val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "selectable" val

#if defined(ENABLE_OVERLOADING)
data LabelSelectablePropertyInfo
instance AttrInfo LabelSelectablePropertyInfo where
    type AttrAllowedOps LabelSelectablePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelSelectablePropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelSelectablePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint LabelSelectablePropertyInfo = (~) Bool
    type AttrTransferType LabelSelectablePropertyInfo = Bool
    type AttrGetType LabelSelectablePropertyInfo = Bool
    type AttrLabel LabelSelectablePropertyInfo = "selectable"
    type AttrOrigin LabelSelectablePropertyInfo = Label
    attrGet = getLabelSelectable
    attrSet = setLabelSelectable
    attrPut = setLabelSelectable
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelSelectable
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.selectable"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:selectable"
        })
#endif

-- VVV Prop "single-line-mode"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@single-line-mode@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #singleLineMode
-- @
getLabelSingleLineMode :: (MonadIO m, IsLabel o) => o -> m Bool
getLabelSingleLineMode obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "single-line-mode"

-- | Set the value of the “@single-line-mode@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #singleLineMode 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelSingleLineMode :: (MonadIO m, IsLabel o) => o -> Bool -> m ()
setLabelSingleLineMode obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "single-line-mode" val

-- | Construct a t'GValueConstruct' with valid value for the “@single-line-mode@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelSingleLineMode :: (IsLabel o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructLabelSingleLineMode val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "single-line-mode" val

#if defined(ENABLE_OVERLOADING)
data LabelSingleLineModePropertyInfo
instance AttrInfo LabelSingleLineModePropertyInfo where
    type AttrAllowedOps LabelSingleLineModePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelSingleLineModePropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelSingleLineModePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint LabelSingleLineModePropertyInfo = (~) Bool
    type AttrTransferType LabelSingleLineModePropertyInfo = Bool
    type AttrGetType LabelSingleLineModePropertyInfo = Bool
    type AttrLabel LabelSingleLineModePropertyInfo = "single-line-mode"
    type AttrOrigin LabelSingleLineModePropertyInfo = Label
    attrGet = getLabelSingleLineMode
    attrSet = setLabelSingleLineMode
    attrPut = setLabelSingleLineMode
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelSingleLineMode
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.singleLineMode"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:singleLineMode"
        })
#endif

-- VVV Prop "tabs"
   -- Type: TInterface (Name {namespace = "Pango", name = "TabArray"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Just True)

-- | Get the value of the “@tabs@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #tabs
-- @
getLabelTabs :: (MonadIO m, IsLabel o) => o -> m (Maybe Pango.TabArray.TabArray)
getLabelTabs obj = MIO.liftIO $ B.Properties.getObjectPropertyBoxed obj "tabs" Pango.TabArray.TabArray

-- | Set the value of the “@tabs@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #tabs 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelTabs :: (MonadIO m, IsLabel o) => o -> Pango.TabArray.TabArray -> m ()
setLabelTabs obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBoxed obj "tabs" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@tabs@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelTabs :: (IsLabel o, MIO.MonadIO m) => Pango.TabArray.TabArray -> m (GValueConstruct o)
constructLabelTabs val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBoxed "tabs" (P.Just val)

-- | Set the value of the “@tabs@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #tabs
-- @
clearLabelTabs :: (MonadIO m, IsLabel o) => o -> m ()
clearLabelTabs obj = liftIO $ B.Properties.setObjectPropertyBoxed obj "tabs" (Nothing :: Maybe Pango.TabArray.TabArray)

#if defined(ENABLE_OVERLOADING)
data LabelTabsPropertyInfo
instance AttrInfo LabelTabsPropertyInfo where
    type AttrAllowedOps LabelTabsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint LabelTabsPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelTabsPropertyInfo = (~) Pango.TabArray.TabArray
    type AttrTransferTypeConstraint LabelTabsPropertyInfo = (~) Pango.TabArray.TabArray
    type AttrTransferType LabelTabsPropertyInfo = Pango.TabArray.TabArray
    type AttrGetType LabelTabsPropertyInfo = (Maybe Pango.TabArray.TabArray)
    type AttrLabel LabelTabsPropertyInfo = "tabs"
    type AttrOrigin LabelTabsPropertyInfo = Label
    attrGet = getLabelTabs
    attrSet = setLabelTabs
    attrPut = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelTabs
    attrClear = clearLabelTabs
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.tabs"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:tabs"
        })
#endif

-- VVV Prop "use-markup"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@use-markup@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #useMarkup
-- @
getLabelUseMarkup :: (MonadIO m, IsLabel o) => o -> m Bool
getLabelUseMarkup obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "use-markup"

-- | Set the value of the “@use-markup@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #useMarkup 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelUseMarkup :: (MonadIO m, IsLabel o) => o -> Bool -> m ()
setLabelUseMarkup obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "use-markup" val

-- | Construct a t'GValueConstruct' with valid value for the “@use-markup@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelUseMarkup :: (IsLabel o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructLabelUseMarkup val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "use-markup" val

#if defined(ENABLE_OVERLOADING)
data LabelUseMarkupPropertyInfo
instance AttrInfo LabelUseMarkupPropertyInfo where
    type AttrAllowedOps LabelUseMarkupPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelUseMarkupPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelUseMarkupPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint LabelUseMarkupPropertyInfo = (~) Bool
    type AttrTransferType LabelUseMarkupPropertyInfo = Bool
    type AttrGetType LabelUseMarkupPropertyInfo = Bool
    type AttrLabel LabelUseMarkupPropertyInfo = "use-markup"
    type AttrOrigin LabelUseMarkupPropertyInfo = Label
    attrGet = getLabelUseMarkup
    attrSet = setLabelUseMarkup
    attrPut = setLabelUseMarkup
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelUseMarkup
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.useMarkup"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:useMarkup"
        })
#endif

-- VVV Prop "use-underline"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@use-underline@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #useUnderline
-- @
getLabelUseUnderline :: (MonadIO m, IsLabel o) => o -> m Bool
getLabelUseUnderline obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "use-underline"

-- | Set the value of the “@use-underline@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #useUnderline 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelUseUnderline :: (MonadIO m, IsLabel o) => o -> Bool -> m ()
setLabelUseUnderline obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "use-underline" val

-- | Construct a t'GValueConstruct' with valid value for the “@use-underline@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelUseUnderline :: (IsLabel o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructLabelUseUnderline val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "use-underline" val

#if defined(ENABLE_OVERLOADING)
data LabelUseUnderlinePropertyInfo
instance AttrInfo LabelUseUnderlinePropertyInfo where
    type AttrAllowedOps LabelUseUnderlinePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelUseUnderlinePropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelUseUnderlinePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint LabelUseUnderlinePropertyInfo = (~) Bool
    type AttrTransferType LabelUseUnderlinePropertyInfo = Bool
    type AttrGetType LabelUseUnderlinePropertyInfo = Bool
    type AttrLabel LabelUseUnderlinePropertyInfo = "use-underline"
    type AttrOrigin LabelUseUnderlinePropertyInfo = Label
    attrGet = getLabelUseUnderline
    attrSet = setLabelUseUnderline
    attrPut = setLabelUseUnderline
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelUseUnderline
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.useUnderline"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:useUnderline"
        })
#endif

-- VVV Prop "width-chars"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@width-chars@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #widthChars
-- @
getLabelWidthChars :: (MonadIO m, IsLabel o) => o -> m Int32
getLabelWidthChars obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "width-chars"

-- | Set the value of the “@width-chars@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #widthChars 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelWidthChars :: (MonadIO m, IsLabel o) => o -> Int32 -> m ()
setLabelWidthChars obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "width-chars" val

-- | Construct a t'GValueConstruct' with valid value for the “@width-chars@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelWidthChars :: (IsLabel o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructLabelWidthChars val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "width-chars" val

#if defined(ENABLE_OVERLOADING)
data LabelWidthCharsPropertyInfo
instance AttrInfo LabelWidthCharsPropertyInfo where
    type AttrAllowedOps LabelWidthCharsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelWidthCharsPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelWidthCharsPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint LabelWidthCharsPropertyInfo = (~) Int32
    type AttrTransferType LabelWidthCharsPropertyInfo = Int32
    type AttrGetType LabelWidthCharsPropertyInfo = Int32
    type AttrLabel LabelWidthCharsPropertyInfo = "width-chars"
    type AttrOrigin LabelWidthCharsPropertyInfo = Label
    attrGet = getLabelWidthChars
    attrSet = setLabelWidthChars
    attrPut = setLabelWidthChars
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelWidthChars
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.widthChars"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:widthChars"
        })
#endif

-- VVV Prop "wrap"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@wrap@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #wrap
-- @
getLabelWrap :: (MonadIO m, IsLabel o) => o -> m Bool
getLabelWrap obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "wrap"

-- | Set the value of the “@wrap@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #wrap 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelWrap :: (MonadIO m, IsLabel o) => o -> Bool -> m ()
setLabelWrap obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "wrap" val

-- | Construct a t'GValueConstruct' with valid value for the “@wrap@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelWrap :: (IsLabel o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructLabelWrap val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "wrap" val

#if defined(ENABLE_OVERLOADING)
data LabelWrapPropertyInfo
instance AttrInfo LabelWrapPropertyInfo where
    type AttrAllowedOps LabelWrapPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelWrapPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelWrapPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint LabelWrapPropertyInfo = (~) Bool
    type AttrTransferType LabelWrapPropertyInfo = Bool
    type AttrGetType LabelWrapPropertyInfo = Bool
    type AttrLabel LabelWrapPropertyInfo = "wrap"
    type AttrOrigin LabelWrapPropertyInfo = Label
    attrGet = getLabelWrap
    attrSet = setLabelWrap
    attrPut = setLabelWrap
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelWrap
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.wrap"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:wrap"
        })
#endif

-- VVV Prop "wrap-mode"
   -- Type: TInterface (Name {namespace = "Pango", name = "WrapMode"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@wrap-mode@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #wrapMode
-- @
getLabelWrapMode :: (MonadIO m, IsLabel o) => o -> m Pango.Enums.WrapMode
getLabelWrapMode obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "wrap-mode"

-- | Set the value of the “@wrap-mode@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #wrapMode 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelWrapMode :: (MonadIO m, IsLabel o) => o -> Pango.Enums.WrapMode -> m ()
setLabelWrapMode obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "wrap-mode" val

-- | Construct a t'GValueConstruct' with valid value for the “@wrap-mode@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelWrapMode :: (IsLabel o, MIO.MonadIO m) => Pango.Enums.WrapMode -> m (GValueConstruct o)
constructLabelWrapMode val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "wrap-mode" val

#if defined(ENABLE_OVERLOADING)
data LabelWrapModePropertyInfo
instance AttrInfo LabelWrapModePropertyInfo where
    type AttrAllowedOps LabelWrapModePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelWrapModePropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelWrapModePropertyInfo = (~) Pango.Enums.WrapMode
    type AttrTransferTypeConstraint LabelWrapModePropertyInfo = (~) Pango.Enums.WrapMode
    type AttrTransferType LabelWrapModePropertyInfo = Pango.Enums.WrapMode
    type AttrGetType LabelWrapModePropertyInfo = Pango.Enums.WrapMode
    type AttrLabel LabelWrapModePropertyInfo = "wrap-mode"
    type AttrOrigin LabelWrapModePropertyInfo = Label
    attrGet = getLabelWrapMode
    attrSet = setLabelWrapMode
    attrPut = setLabelWrapMode
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelWrapMode
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.wrapMode"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:wrapMode"
        })
#endif

-- VVV Prop "xalign"
   -- Type: TBasicType TFloat
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@xalign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #xalign
-- @
getLabelXalign :: (MonadIO m, IsLabel o) => o -> m Float
getLabelXalign obj = MIO.liftIO $ B.Properties.getObjectPropertyFloat obj "xalign"

-- | Set the value of the “@xalign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #xalign 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelXalign :: (MonadIO m, IsLabel o) => o -> Float -> m ()
setLabelXalign obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyFloat obj "xalign" val

-- | Construct a t'GValueConstruct' with valid value for the “@xalign@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelXalign :: (IsLabel o, MIO.MonadIO m) => Float -> m (GValueConstruct o)
constructLabelXalign val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFloat "xalign" val

#if defined(ENABLE_OVERLOADING)
data LabelXalignPropertyInfo
instance AttrInfo LabelXalignPropertyInfo where
    type AttrAllowedOps LabelXalignPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelXalignPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelXalignPropertyInfo = (~) Float
    type AttrTransferTypeConstraint LabelXalignPropertyInfo = (~) Float
    type AttrTransferType LabelXalignPropertyInfo = Float
    type AttrGetType LabelXalignPropertyInfo = Float
    type AttrLabel LabelXalignPropertyInfo = "xalign"
    type AttrOrigin LabelXalignPropertyInfo = Label
    attrGet = getLabelXalign
    attrSet = setLabelXalign
    attrPut = setLabelXalign
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelXalign
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.xalign"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:xalign"
        })
#endif

-- VVV Prop "yalign"
   -- Type: TBasicType TFloat
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@yalign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' label #yalign
-- @
getLabelYalign :: (MonadIO m, IsLabel o) => o -> m Float
getLabelYalign obj = MIO.liftIO $ B.Properties.getObjectPropertyFloat obj "yalign"

-- | Set the value of the “@yalign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' label [ #yalign 'Data.GI.Base.Attributes.:=' value ]
-- @
setLabelYalign :: (MonadIO m, IsLabel o) => o -> Float -> m ()
setLabelYalign obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyFloat obj "yalign" val

-- | Construct a t'GValueConstruct' with valid value for the “@yalign@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructLabelYalign :: (IsLabel o, MIO.MonadIO m) => Float -> m (GValueConstruct o)
constructLabelYalign val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFloat "yalign" val

#if defined(ENABLE_OVERLOADING)
data LabelYalignPropertyInfo
instance AttrInfo LabelYalignPropertyInfo where
    type AttrAllowedOps LabelYalignPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint LabelYalignPropertyInfo = IsLabel
    type AttrSetTypeConstraint LabelYalignPropertyInfo = (~) Float
    type AttrTransferTypeConstraint LabelYalignPropertyInfo = (~) Float
    type AttrTransferType LabelYalignPropertyInfo = Float
    type AttrGetType LabelYalignPropertyInfo = Float
    type AttrLabel LabelYalignPropertyInfo = "yalign"
    type AttrOrigin LabelYalignPropertyInfo = Label
    attrGet = getLabelYalign
    attrSet = setLabelYalign
    attrPut = setLabelYalign
    attrTransfer _ v = do
        return v
    attrConstruct = constructLabelYalign
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.yalign"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#g:attr:yalign"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Label
type instance O.AttributeList Label = LabelAttributeList
type LabelAttributeList = ('[ '("accessibleRole", Gtk.Accessible.AccessibleAccessibleRolePropertyInfo), '("attributes", LabelAttributesPropertyInfo), '("canFocus", Gtk.Widget.WidgetCanFocusPropertyInfo), '("canTarget", Gtk.Widget.WidgetCanTargetPropertyInfo), '("cssClasses", Gtk.Widget.WidgetCssClassesPropertyInfo), '("cssName", Gtk.Widget.WidgetCssNamePropertyInfo), '("cursor", Gtk.Widget.WidgetCursorPropertyInfo), '("ellipsize", LabelEllipsizePropertyInfo), '("extraMenu", LabelExtraMenuPropertyInfo), '("focusOnClick", Gtk.Widget.WidgetFocusOnClickPropertyInfo), '("focusable", Gtk.Widget.WidgetFocusablePropertyInfo), '("halign", Gtk.Widget.WidgetHalignPropertyInfo), '("hasDefault", Gtk.Widget.WidgetHasDefaultPropertyInfo), '("hasFocus", Gtk.Widget.WidgetHasFocusPropertyInfo), '("hasTooltip", Gtk.Widget.WidgetHasTooltipPropertyInfo), '("heightRequest", Gtk.Widget.WidgetHeightRequestPropertyInfo), '("hexpand", Gtk.Widget.WidgetHexpandPropertyInfo), '("hexpandSet", Gtk.Widget.WidgetHexpandSetPropertyInfo), '("justify", LabelJustifyPropertyInfo), '("label", LabelLabelPropertyInfo), '("layoutManager", Gtk.Widget.WidgetLayoutManagerPropertyInfo), '("limitEvents", Gtk.Widget.WidgetLimitEventsPropertyInfo), '("lines", LabelLinesPropertyInfo), '("marginBottom", Gtk.Widget.WidgetMarginBottomPropertyInfo), '("marginEnd", Gtk.Widget.WidgetMarginEndPropertyInfo), '("marginStart", Gtk.Widget.WidgetMarginStartPropertyInfo), '("marginTop", Gtk.Widget.WidgetMarginTopPropertyInfo), '("maxWidthChars", LabelMaxWidthCharsPropertyInfo), '("mnemonicKeyval", LabelMnemonicKeyvalPropertyInfo), '("mnemonicWidget", LabelMnemonicWidgetPropertyInfo), '("name", Gtk.Widget.WidgetNamePropertyInfo), '("naturalWrapMode", LabelNaturalWrapModePropertyInfo), '("opacity", Gtk.Widget.WidgetOpacityPropertyInfo), '("overflow", Gtk.Widget.WidgetOverflowPropertyInfo), '("parent", Gtk.Widget.WidgetParentPropertyInfo), '("receivesDefault", Gtk.Widget.WidgetReceivesDefaultPropertyInfo), '("root", Gtk.Widget.WidgetRootPropertyInfo), '("scaleFactor", Gtk.Widget.WidgetScaleFactorPropertyInfo), '("selectable", LabelSelectablePropertyInfo), '("sensitive", Gtk.Widget.WidgetSensitivePropertyInfo), '("singleLineMode", LabelSingleLineModePropertyInfo), '("tabs", LabelTabsPropertyInfo), '("tooltipMarkup", Gtk.Widget.WidgetTooltipMarkupPropertyInfo), '("tooltipText", Gtk.Widget.WidgetTooltipTextPropertyInfo), '("useMarkup", LabelUseMarkupPropertyInfo), '("useUnderline", LabelUseUnderlinePropertyInfo), '("valign", Gtk.Widget.WidgetValignPropertyInfo), '("vexpand", Gtk.Widget.WidgetVexpandPropertyInfo), '("vexpandSet", Gtk.Widget.WidgetVexpandSetPropertyInfo), '("visible", Gtk.Widget.WidgetVisiblePropertyInfo), '("widthChars", LabelWidthCharsPropertyInfo), '("widthRequest", Gtk.Widget.WidgetWidthRequestPropertyInfo), '("wrap", LabelWrapPropertyInfo), '("wrapMode", LabelWrapModePropertyInfo), '("xalign", LabelXalignPropertyInfo), '("yalign", LabelYalignPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
labelAttributes :: AttrLabelProxy "attributes"
labelAttributes = AttrLabelProxy

labelEllipsize :: AttrLabelProxy "ellipsize"
labelEllipsize = AttrLabelProxy

labelExtraMenu :: AttrLabelProxy "extraMenu"
labelExtraMenu = AttrLabelProxy

labelJustify :: AttrLabelProxy "justify"
labelJustify = AttrLabelProxy

labelLabel :: AttrLabelProxy "label"
labelLabel = AttrLabelProxy

labelLines :: AttrLabelProxy "lines"
labelLines = AttrLabelProxy

labelMaxWidthChars :: AttrLabelProxy "maxWidthChars"
labelMaxWidthChars = AttrLabelProxy

labelMnemonicKeyval :: AttrLabelProxy "mnemonicKeyval"
labelMnemonicKeyval = AttrLabelProxy

labelMnemonicWidget :: AttrLabelProxy "mnemonicWidget"
labelMnemonicWidget = AttrLabelProxy

labelNaturalWrapMode :: AttrLabelProxy "naturalWrapMode"
labelNaturalWrapMode = AttrLabelProxy

labelSelectable :: AttrLabelProxy "selectable"
labelSelectable = AttrLabelProxy

labelSingleLineMode :: AttrLabelProxy "singleLineMode"
labelSingleLineMode = AttrLabelProxy

labelTabs :: AttrLabelProxy "tabs"
labelTabs = AttrLabelProxy

labelUseMarkup :: AttrLabelProxy "useMarkup"
labelUseMarkup = AttrLabelProxy

labelUseUnderline :: AttrLabelProxy "useUnderline"
labelUseUnderline = AttrLabelProxy

labelWidthChars :: AttrLabelProxy "widthChars"
labelWidthChars = AttrLabelProxy

labelWrap :: AttrLabelProxy "wrap"
labelWrap = AttrLabelProxy

labelWrapMode :: AttrLabelProxy "wrapMode"
labelWrapMode = AttrLabelProxy

labelXalign :: AttrLabelProxy "xalign"
labelXalign = AttrLabelProxy

labelYalign :: AttrLabelProxy "yalign"
labelYalign = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Label = LabelSignalList
type LabelSignalList = ('[ '("activateCurrentLink", LabelActivateCurrentLinkSignalInfo), '("activateLink", LabelActivateLinkSignalInfo), '("copyClipboard", LabelCopyClipboardSignalInfo), '("destroy", Gtk.Widget.WidgetDestroySignalInfo), '("directionChanged", Gtk.Widget.WidgetDirectionChangedSignalInfo), '("hide", Gtk.Widget.WidgetHideSignalInfo), '("keynavFailed", Gtk.Widget.WidgetKeynavFailedSignalInfo), '("map", Gtk.Widget.WidgetMapSignalInfo), '("mnemonicActivate", Gtk.Widget.WidgetMnemonicActivateSignalInfo), '("moveCursor", LabelMoveCursorSignalInfo), '("moveFocus", Gtk.Widget.WidgetMoveFocusSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("queryTooltip", Gtk.Widget.WidgetQueryTooltipSignalInfo), '("realize", Gtk.Widget.WidgetRealizeSignalInfo), '("show", Gtk.Widget.WidgetShowSignalInfo), '("stateFlagsChanged", Gtk.Widget.WidgetStateFlagsChangedSignalInfo), '("unmap", Gtk.Widget.WidgetUnmapSignalInfo), '("unrealize", Gtk.Widget.WidgetUnrealizeSignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Label::new
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the text of the label"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Label" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_new" gtk_label_new :: 
    CString ->                              -- str : TBasicType TUTF8
    IO (Ptr Label)

-- | Creates a new label with the given text inside it.
-- 
-- You can pass @NULL@ to get an empty label widget.
labelNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (T.Text)
    -- ^ /@str@/: the text of the label
    -> m Label
    -- ^ __Returns:__ the new label
labelNew str = liftIO $ do
    maybeStr <- case str of
        Nothing -> return FP.nullPtr
        Just jStr -> do
            jStr' <- textToCString jStr
            return jStr'
    result <- gtk_label_new maybeStr
    checkUnexpectedReturnNULL "labelNew" result
    result' <- (newObject Label) result
    freeMem maybeStr
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Label::new_with_mnemonic
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the text of the label, with an underscore in front of the\n  mnemonic character"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Label" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_new_with_mnemonic" gtk_label_new_with_mnemonic :: 
    CString ->                              -- str : TBasicType TUTF8
    IO (Ptr Label)

-- | Creates a new label with the given text inside it, and a mnemonic.
-- 
-- If characters in /@str@/ are preceded by an underscore, they are
-- underlined. If you need a literal underscore character in a label, use
-- \'__\' (two underscores). The first underlined character represents a
-- keyboard accelerator called a mnemonic. The mnemonic key can be used
-- to activate another widget, chosen automatically, or explicitly using
-- 'GI.Gtk.Objects.Label.labelSetMnemonicWidget'.
-- 
-- If 'GI.Gtk.Objects.Label.labelSetMnemonicWidget' is not called, then the first
-- activatable ancestor of the label will be chosen as the mnemonic
-- widget. For instance, if the label is inside a button or menu item,
-- the button or menu item will automatically become the mnemonic widget
-- and be activated by the mnemonic.
labelNewWithMnemonic ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (T.Text)
    -- ^ /@str@/: the text of the label, with an underscore in front of the
    --   mnemonic character
    -> m Label
    -- ^ __Returns:__ the new label
labelNewWithMnemonic str = liftIO $ do
    maybeStr <- case str of
        Nothing -> return FP.nullPtr
        Just jStr -> do
            jStr' <- textToCString jStr
            return jStr'
    result <- gtk_label_new_with_mnemonic maybeStr
    checkUnexpectedReturnNULL "labelNewWithMnemonic" result
    result' <- (newObject Label) result
    freeMem maybeStr
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Label::get_attributes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "AttrList" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_attributes" gtk_label_get_attributes :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO (Ptr Pango.AttrList.AttrList)

-- | Gets the label\'s attribute list.
-- 
-- This is the t'GI.Pango.Structs.AttrList.AttrList' that was set on the label using
-- 'GI.Gtk.Objects.Label.labelSetAttributes', if any. This function does not
-- reflect attributes that come from the label\'s markup (see
-- 'GI.Gtk.Objects.Label.labelSetMarkup'). If you want to get the effective
-- attributes for the label, use
-- @pango_layout_get_attributes (gtk_label_get_layout (self))@.
labelGetAttributes ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m (Maybe Pango.AttrList.AttrList)
    -- ^ __Returns:__ the attribute list
labelGetAttributes self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_attributes self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Pango.AttrList.AttrList) result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data LabelGetAttributesMethodInfo
instance (signature ~ (m (Maybe Pango.AttrList.AttrList)), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetAttributesMethodInfo a signature where
    overloadedMethod = labelGetAttributes

instance O.OverloadedMethodInfo LabelGetAttributesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetAttributes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetAttributes"
        })


#endif

-- method Label::get_current_uri
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_current_uri" gtk_label_get_current_uri :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CString

-- | Returns the URI for the active link in the label.
-- 
-- The active link is the one under the mouse pointer or, in a
-- selectable label, the link in which the text cursor is currently
-- positioned.
-- 
-- This function is intended for use in a [Label::activateLink]("GI.Gtk.Objects.Label#g:signal:activateLink")
-- handler or for use in a [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip") handler.
labelGetCurrentUri ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the active URI
labelGetCurrentUri self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_current_uri self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data LabelGetCurrentUriMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetCurrentUriMethodInfo a signature where
    overloadedMethod = labelGetCurrentUri

instance O.OverloadedMethodInfo LabelGetCurrentUriMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetCurrentUri",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetCurrentUri"
        })


#endif

-- method Label::get_ellipsize
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Pango" , name = "EllipsizeMode" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_ellipsize" gtk_label_get_ellipsize :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CUInt

-- | Returns the ellipsization mode of the label.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetEllipsize'.
labelGetEllipsize ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Pango.Enums.EllipsizeMode
    -- ^ __Returns:__ the ellipsization mode
labelGetEllipsize self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_ellipsize self'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetEllipsizeMethodInfo
instance (signature ~ (m Pango.Enums.EllipsizeMode), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetEllipsizeMethodInfo a signature where
    overloadedMethod = labelGetEllipsize

instance O.OverloadedMethodInfo LabelGetEllipsizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetEllipsize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetEllipsize"
        })


#endif

-- method Label::get_extra_menu
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "MenuModel" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_extra_menu" gtk_label_get_extra_menu :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO (Ptr Gio.MenuModel.MenuModel)

-- | Gets the extra menu model of the label.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetExtraMenu'.
labelGetExtraMenu ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m (Maybe Gio.MenuModel.MenuModel)
    -- ^ __Returns:__ the menu model
labelGetExtraMenu self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_extra_menu self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gio.MenuModel.MenuModel) result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data LabelGetExtraMenuMethodInfo
instance (signature ~ (m (Maybe Gio.MenuModel.MenuModel)), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetExtraMenuMethodInfo a signature where
    overloadedMethod = labelGetExtraMenu

instance O.OverloadedMethodInfo LabelGetExtraMenuMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetExtraMenu",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetExtraMenu"
        })


#endif

-- method Label::get_justify
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "Justification" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_justify" gtk_label_get_justify :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CUInt

-- | Returns the justification of the label.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetJustify'.
labelGetJustify ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Gtk.Enums.Justification
    -- ^ __Returns:__ the justification value
labelGetJustify self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_justify self'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetJustifyMethodInfo
instance (signature ~ (m Gtk.Enums.Justification), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetJustifyMethodInfo a signature where
    overloadedMethod = labelGetJustify

instance O.OverloadedMethodInfo LabelGetJustifyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetJustify",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetJustify"
        })


#endif

-- method Label::get_label
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_label" gtk_label_get_label :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CString

-- | Fetches the text from a label.
-- 
-- The returned text includes any embedded underlines indicating
-- mnemonics and Pango markup. (See 'GI.Gtk.Objects.Label.labelGetText').
labelGetLabel ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m T.Text
    -- ^ __Returns:__ the text of the label widget
labelGetLabel self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_label self'
    checkUnexpectedReturnNULL "labelGetLabel" result
    result' <- cstringToText result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetLabelMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetLabelMethodInfo a signature where
    overloadedMethod = labelGetLabel

instance O.OverloadedMethodInfo LabelGetLabelMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetLabel",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetLabel"
        })


#endif

-- method Label::get_layout
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Layout" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_layout" gtk_label_get_layout :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO (Ptr Pango.Layout.Layout)

-- | Gets the Pango layout used to display the label.
-- 
-- The layout is useful to e.g. convert text positions to pixel
-- positions, in combination with 'GI.Gtk.Objects.Label.labelGetLayoutOffsets'.
-- The returned layout is owned by the /@label@/ so need not be
-- freed by the caller. The /@label@/ is free to recreate its layout
-- at any time, so it should be considered read-only.
labelGetLayout ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Pango.Layout.Layout
    -- ^ __Returns:__ the t'GI.Pango.Objects.Layout.Layout' for this label
labelGetLayout self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_layout self'
    checkUnexpectedReturnNULL "labelGetLayout" result
    result' <- (newObject Pango.Layout.Layout) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetLayoutMethodInfo
instance (signature ~ (m Pango.Layout.Layout), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetLayoutMethodInfo a signature where
    overloadedMethod = labelGetLayout

instance O.OverloadedMethodInfo LabelGetLayoutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetLayout",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetLayout"
        })


#endif

-- method Label::get_layout_offsets
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store X offset of layout"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store Y offset of layout"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_layout_offsets" gtk_label_get_layout_offsets :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    Ptr Int32 ->                            -- x : TBasicType TInt
    Ptr Int32 ->                            -- y : TBasicType TInt
    IO ()

-- | Obtains the coordinates where the label will draw its Pango layout.
-- 
-- The coordinates are useful to convert mouse events into coordinates
-- inside the t'GI.Pango.Objects.Layout.Layout', e.g. to take some action if some part
-- of the label is clicked. Remember when using the t'GI.Pango.Objects.Layout.Layout'
-- functions you need to convert to and from pixels using @PANGO_PIXELS()@
-- or [const/@pango@/.SCALE].
labelGetLayoutOffsets ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m ((Int32, Int32))
labelGetLayoutOffsets self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    x <- allocMem :: IO (Ptr Int32)
    y <- allocMem :: IO (Ptr Int32)
    gtk_label_get_layout_offsets self' x y
    x' <- peek x
    y' <- peek y
    touchManagedPtr self
    freeMem x
    freeMem y
    return (x', y')

#if defined(ENABLE_OVERLOADING)
data LabelGetLayoutOffsetsMethodInfo
instance (signature ~ (m ((Int32, Int32))), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetLayoutOffsetsMethodInfo a signature where
    overloadedMethod = labelGetLayoutOffsets

instance O.OverloadedMethodInfo LabelGetLayoutOffsetsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetLayoutOffsets",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetLayoutOffsets"
        })


#endif

-- method Label::get_lines
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_lines" gtk_label_get_lines :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO Int32

-- | Gets the number of lines to which an ellipsized, wrapping
-- label should be limited.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetLines'.
labelGetLines ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Int32
    -- ^ __Returns:__ the number of lines
labelGetLines self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_lines self'
    touchManagedPtr self
    return result

#if defined(ENABLE_OVERLOADING)
data LabelGetLinesMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetLinesMethodInfo a signature where
    overloadedMethod = labelGetLines

instance O.OverloadedMethodInfo LabelGetLinesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetLines",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetLines"
        })


#endif

-- method Label::get_max_width_chars
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_max_width_chars" gtk_label_get_max_width_chars :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO Int32

-- | Retrieves the maximum width of the label in characters.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetWidthChars'.
labelGetMaxWidthChars ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Int32
    -- ^ __Returns:__ the maximum width of the label, in characters
labelGetMaxWidthChars self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_max_width_chars self'
    touchManagedPtr self
    return result

#if defined(ENABLE_OVERLOADING)
data LabelGetMaxWidthCharsMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetMaxWidthCharsMethodInfo a signature where
    overloadedMethod = labelGetMaxWidthChars

instance O.OverloadedMethodInfo LabelGetMaxWidthCharsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetMaxWidthChars",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetMaxWidthChars"
        })


#endif

-- method Label::get_mnemonic_keyval
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_mnemonic_keyval" gtk_label_get_mnemonic_keyval :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO Word32

-- | Return the mnemonic accelerator.
-- 
-- If the label has been set so that it has a mnemonic key this function
-- returns the keyval used for the mnemonic accelerator. If there is no
-- mnemonic set up it returns @GDK_KEY_VoidSymbol@.
labelGetMnemonicKeyval ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Word32
    -- ^ __Returns:__ GDK keyval usable for accelerators, or @GDK_KEY_VoidSymbol@
labelGetMnemonicKeyval self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_mnemonic_keyval self'
    touchManagedPtr self
    return result

#if defined(ENABLE_OVERLOADING)
data LabelGetMnemonicKeyvalMethodInfo
instance (signature ~ (m Word32), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetMnemonicKeyvalMethodInfo a signature where
    overloadedMethod = labelGetMnemonicKeyval

instance O.OverloadedMethodInfo LabelGetMnemonicKeyvalMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetMnemonicKeyval",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetMnemonicKeyval"
        })


#endif

-- method Label::get_mnemonic_widget
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_mnemonic_widget" gtk_label_get_mnemonic_widget :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO (Ptr Gtk.Widget.Widget)

-- | Retrieves the mnemonic target of this label.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetMnemonicWidget'.
labelGetMnemonicWidget ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m (Maybe Gtk.Widget.Widget)
    -- ^ __Returns:__ the target of the label’s mnemonic,
    --   or @NULL@ if none has been set and the default algorithm will be used.
labelGetMnemonicWidget self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_mnemonic_widget self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gtk.Widget.Widget) result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data LabelGetMnemonicWidgetMethodInfo
instance (signature ~ (m (Maybe Gtk.Widget.Widget)), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetMnemonicWidgetMethodInfo a signature where
    overloadedMethod = labelGetMnemonicWidget

instance O.OverloadedMethodInfo LabelGetMnemonicWidgetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetMnemonicWidget",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetMnemonicWidget"
        })


#endif

-- method Label::get_natural_wrap_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "NaturalWrapMode" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_natural_wrap_mode" gtk_label_get_natural_wrap_mode :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CUInt

-- | Returns natural line wrap mode used by the label.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetNaturalWrapMode'.
-- 
-- /Since: 4.6/
labelGetNaturalWrapMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Gtk.Enums.NaturalWrapMode
    -- ^ __Returns:__ the natural line wrap mode
labelGetNaturalWrapMode self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_natural_wrap_mode self'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetNaturalWrapModeMethodInfo
instance (signature ~ (m Gtk.Enums.NaturalWrapMode), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetNaturalWrapModeMethodInfo a signature where
    overloadedMethod = labelGetNaturalWrapMode

instance O.OverloadedMethodInfo LabelGetNaturalWrapModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetNaturalWrapMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetNaturalWrapMode"
        })


#endif

-- method Label::get_selectable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_selectable" gtk_label_get_selectable :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CInt

-- | Returns whether the label is selectable.
labelGetSelectable ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Bool
    -- ^ __Returns:__ true if the user can copy text from the label
labelGetSelectable self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_selectable self'
    let result' = (/= 0) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetSelectableMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetSelectableMethodInfo a signature where
    overloadedMethod = labelGetSelectable

instance O.OverloadedMethodInfo LabelGetSelectableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetSelectable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetSelectable"
        })


#endif

-- method Label::get_selection_bounds
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for start of selection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "end"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for end of selection"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_selection_bounds" gtk_label_get_selection_bounds :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    Ptr Int32 ->                            -- start : TBasicType TInt
    Ptr Int32 ->                            -- end : TBasicType TInt
    IO CInt

-- | Gets the selected range of characters in the label.
-- 
-- The returned /@start@/ and /@end@/ positions are in characters.
labelGetSelectionBounds ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m ((Bool, Int32, Int32))
    -- ^ __Returns:__ true if selection is non-empty
labelGetSelectionBounds self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    start <- allocMem :: IO (Ptr Int32)
    end <- allocMem :: IO (Ptr Int32)
    result <- gtk_label_get_selection_bounds self' start end
    let result' = (/= 0) result
    start' <- peek start
    end' <- peek end
    touchManagedPtr self
    freeMem start
    freeMem end
    return (result', start', end')

#if defined(ENABLE_OVERLOADING)
data LabelGetSelectionBoundsMethodInfo
instance (signature ~ (m ((Bool, Int32, Int32))), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetSelectionBoundsMethodInfo a signature where
    overloadedMethod = labelGetSelectionBounds

instance O.OverloadedMethodInfo LabelGetSelectionBoundsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetSelectionBounds",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetSelectionBounds"
        })


#endif

-- method Label::get_single_line_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_single_line_mode" gtk_label_get_single_line_mode :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CInt

-- | Returns whether the label is in single line mode.
labelGetSingleLineMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Bool
    -- ^ __Returns:__ true if the label is in single line mode
labelGetSingleLineMode self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_single_line_mode self'
    let result' = (/= 0) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetSingleLineModeMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetSingleLineModeMethodInfo a signature where
    overloadedMethod = labelGetSingleLineMode

instance O.OverloadedMethodInfo LabelGetSingleLineModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetSingleLineMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetSingleLineMode"
        })


#endif

-- method Label::get_tabs
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "TabArray" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_tabs" gtk_label_get_tabs :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO (Ptr Pango.TabArray.TabArray)

-- | Gets the tab stops for the label.
-- 
-- The returned array will be @NULL@ if “standard” (8-space) tabs are used.
-- 
-- /Since: 4.8/
labelGetTabs ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m (Maybe Pango.TabArray.TabArray)
    -- ^ __Returns:__ copy of default tab array,
    --   or @NULL@ if standard tabs are used
labelGetTabs self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_tabs self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Pango.TabArray.TabArray) result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data LabelGetTabsMethodInfo
instance (signature ~ (m (Maybe Pango.TabArray.TabArray)), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetTabsMethodInfo a signature where
    overloadedMethod = labelGetTabs

instance O.OverloadedMethodInfo LabelGetTabsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetTabs",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetTabs"
        })


#endif

-- method Label::get_text
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_text" gtk_label_get_text :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CString

-- | Gets the text of the label.
-- 
-- The returned text is as it appears on screen. This does not include
-- any embedded underlines indicating mnemonics or Pango markup. (See
-- 'GI.Gtk.Objects.Label.labelGetLabel')
labelGetText ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m T.Text
    -- ^ __Returns:__ the text in the label widget
labelGetText self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_text self'
    checkUnexpectedReturnNULL "labelGetText" result
    result' <- cstringToText result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetTextMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetTextMethodInfo a signature where
    overloadedMethod = labelGetText

instance O.OverloadedMethodInfo LabelGetTextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetText",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetText"
        })


#endif

-- method Label::get_use_markup
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_use_markup" gtk_label_get_use_markup :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CInt

-- | Returns whether the label’s text is interpreted as Pango markup.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetUseMarkup'.
labelGetUseMarkup ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Bool
    -- ^ __Returns:__ true if the label’s text will be parsed for markup
labelGetUseMarkup self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_use_markup self'
    let result' = (/= 0) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetUseMarkupMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetUseMarkupMethodInfo a signature where
    overloadedMethod = labelGetUseMarkup

instance O.OverloadedMethodInfo LabelGetUseMarkupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetUseMarkup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetUseMarkup"
        })


#endif

-- method Label::get_use_underline
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_use_underline" gtk_label_get_use_underline :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CInt

-- | Returns whether underlines in the label indicate mnemonics.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetUseUnderline'.
labelGetUseUnderline ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Bool
    -- ^ __Returns:__ true if underlines in the label indicate mnemonics
labelGetUseUnderline self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_use_underline self'
    let result' = (/= 0) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetUseUnderlineMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetUseUnderlineMethodInfo a signature where
    overloadedMethod = labelGetUseUnderline

instance O.OverloadedMethodInfo LabelGetUseUnderlineMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetUseUnderline",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetUseUnderline"
        })


#endif

-- method Label::get_width_chars
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_width_chars" gtk_label_get_width_chars :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO Int32

-- | Retrieves the desired width of the label in characters.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetWidthChars'.
labelGetWidthChars ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Int32
    -- ^ __Returns:__ the desired width of the label, in characters
labelGetWidthChars self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_width_chars self'
    touchManagedPtr self
    return result

#if defined(ENABLE_OVERLOADING)
data LabelGetWidthCharsMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetWidthCharsMethodInfo a signature where
    overloadedMethod = labelGetWidthChars

instance O.OverloadedMethodInfo LabelGetWidthCharsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetWidthChars",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetWidthChars"
        })


#endif

-- method Label::get_wrap
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_wrap" gtk_label_get_wrap :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CInt

-- | Returns whether lines in the label are automatically wrapped.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetWrap'.
labelGetWrap ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Bool
    -- ^ __Returns:__ true if the lines of the label are automatically wrapped
labelGetWrap self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_wrap self'
    let result' = (/= 0) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetWrapMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetWrapMethodInfo a signature where
    overloadedMethod = labelGetWrap

instance O.OverloadedMethodInfo LabelGetWrapMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetWrap",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetWrap"
        })


#endif

-- method Label::get_wrap_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "WrapMode" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_wrap_mode" gtk_label_get_wrap_mode :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CUInt

-- | Returns line wrap mode used by the label.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetWrapMode'.
labelGetWrapMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Pango.Enums.WrapMode
    -- ^ __Returns:__ the line wrap mode
labelGetWrapMode self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_wrap_mode self'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetWrapModeMethodInfo
instance (signature ~ (m Pango.Enums.WrapMode), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetWrapModeMethodInfo a signature where
    overloadedMethod = labelGetWrapMode

instance O.OverloadedMethodInfo LabelGetWrapModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetWrapMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetWrapMode"
        })


#endif

-- method Label::get_xalign
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFloat)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_xalign" gtk_label_get_xalign :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CFloat

-- | Gets the @xalign@ of the label.
-- 
-- See the [Label:xalign]("GI.Gtk.Objects.Label#g:attr:xalign") property.
labelGetXalign ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Float
    -- ^ __Returns:__ the xalign value
labelGetXalign self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_xalign self'
    let result' = realToFrac result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetXalignMethodInfo
instance (signature ~ (m Float), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetXalignMethodInfo a signature where
    overloadedMethod = labelGetXalign

instance O.OverloadedMethodInfo LabelGetXalignMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetXalign",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetXalign"
        })


#endif

-- method Label::get_yalign
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFloat)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_get_yalign" gtk_label_get_yalign :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    IO CFloat

-- | Gets the @yalign@ of the label.
-- 
-- See the [Label:yalign]("GI.Gtk.Objects.Label#g:attr:yalign") property.
labelGetYalign ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> m Float
    -- ^ __Returns:__ the yalign value
labelGetYalign self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_label_get_yalign self'
    let result' = realToFrac result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data LabelGetYalignMethodInfo
instance (signature ~ (m Float), MonadIO m, IsLabel a) => O.OverloadedMethod LabelGetYalignMethodInfo a signature where
    overloadedMethod = labelGetYalign

instance O.OverloadedMethodInfo LabelGetYalignMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelGetYalign",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelGetYalign"
        })


#endif

-- method Label::select_region
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "start offset, in characters"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "end_offset"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "end offset, in characters"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_select_region" gtk_label_select_region :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    Int32 ->                                -- start_offset : TBasicType TInt
    Int32 ->                                -- end_offset : TBasicType TInt
    IO ()

-- | Selects a range of characters in the label, if the label is selectable.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetSelectable'. If the label is not selectable,
-- this function has no effect. If /@startOffset@/ or
-- /@endOffset@/ are -1, then the end of the label will be substituted.
labelSelectRegion ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Int32
    -- ^ /@startOffset@/: start offset, in characters
    -> Int32
    -- ^ /@endOffset@/: end offset, in characters
    -> m ()
labelSelectRegion self startOffset endOffset = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    gtk_label_select_region self' startOffset endOffset
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSelectRegionMethodInfo
instance (signature ~ (Int32 -> Int32 -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSelectRegionMethodInfo a signature where
    overloadedMethod = labelSelectRegion

instance O.OverloadedMethodInfo LabelSelectRegionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSelectRegion",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSelectRegion"
        })


#endif

-- method Label::set_attributes
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "attrs"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "AttrList" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a list of style attributes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_attributes" gtk_label_set_attributes :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    Ptr Pango.AttrList.AttrList ->          -- attrs : TInterface (Name {namespace = "Pango", name = "AttrList"})
    IO ()

-- | Apply attributes to the label text.
-- 
-- The attributes set with this function will be applied and merged with
-- any other attributes previously effected by way of the
-- [Label:useUnderline]("GI.Gtk.Objects.Label#g:attr:useUnderline") or [Label:useMarkup]("GI.Gtk.Objects.Label#g:attr:useMarkup")
-- properties
-- 
-- While it is not recommended to mix markup strings with manually set
-- attributes, if you must; know that the attributes will be applied
-- to the label after the markup string is parsed.
labelSetAttributes ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Maybe (Pango.AttrList.AttrList)
    -- ^ /@attrs@/: a list of style attributes
    -> m ()
labelSetAttributes self attrs = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    maybeAttrs <- case attrs of
        Nothing -> return FP.nullPtr
        Just jAttrs -> do
            jAttrs' <- unsafeManagedPtrGetPtr jAttrs
            return jAttrs'
    gtk_label_set_attributes self' maybeAttrs
    touchManagedPtr self
    whenJust attrs touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetAttributesMethodInfo
instance (signature ~ (Maybe (Pango.AttrList.AttrList) -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetAttributesMethodInfo a signature where
    overloadedMethod = labelSetAttributes

instance O.OverloadedMethodInfo LabelSetAttributesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetAttributes",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetAttributes"
        })


#endif

-- method Label::set_ellipsize
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mode"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "EllipsizeMode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the ellipsization mode"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_ellipsize" gtk_label_set_ellipsize :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CUInt ->                                -- mode : TInterface (Name {namespace = "Pango", name = "EllipsizeMode"})
    IO ()

-- | Sets the mode used to ellipsize the text.
-- 
-- The text will be ellipsized if there is not
-- enough space to render the entire string.
labelSetEllipsize ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Pango.Enums.EllipsizeMode
    -- ^ /@mode@/: the ellipsization mode
    -> m ()
labelSetEllipsize self mode = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let mode' = (fromIntegral . fromEnum) mode
    gtk_label_set_ellipsize self' mode'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetEllipsizeMethodInfo
instance (signature ~ (Pango.Enums.EllipsizeMode -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetEllipsizeMethodInfo a signature where
    overloadedMethod = labelSetEllipsize

instance O.OverloadedMethodInfo LabelSetEllipsizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetEllipsize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetEllipsize"
        })


#endif

-- method Label::set_extra_menu
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "model"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "MenuModel" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a menu model" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_extra_menu" gtk_label_set_extra_menu :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    Ptr Gio.MenuModel.MenuModel ->          -- model : TInterface (Name {namespace = "Gio", name = "MenuModel"})
    IO ()

-- | Sets a menu model to add to the context menu of the label.
labelSetExtraMenu ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a, Gio.MenuModel.IsMenuModel b) =>
    a
    -- ^ /@self@/: a label
    -> Maybe (b)
    -- ^ /@model@/: a menu model
    -> m ()
labelSetExtraMenu self model = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    maybeModel <- case model of
        Nothing -> return FP.nullPtr
        Just jModel -> do
            jModel' <- unsafeManagedPtrCastPtr jModel
            return jModel'
    gtk_label_set_extra_menu self' maybeModel
    touchManagedPtr self
    whenJust model touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetExtraMenuMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsLabel a, Gio.MenuModel.IsMenuModel b) => O.OverloadedMethod LabelSetExtraMenuMethodInfo a signature where
    overloadedMethod = labelSetExtraMenu

instance O.OverloadedMethodInfo LabelSetExtraMenuMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetExtraMenu",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetExtraMenu"
        })


#endif

-- method Label::set_justify
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "jtype"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Justification" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new justification"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_justify" gtk_label_set_justify :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CUInt ->                                -- jtype : TInterface (Name {namespace = "Gtk", name = "Justification"})
    IO ()

-- | Sets the alignment of lines in the label relative to each other.
-- 
-- This function has no effect on labels containing only a single line.
-- 
-- 'GI.Gtk.Enums.JustificationLeft' is the default value when the widget
-- is first created with 'GI.Gtk.Objects.Label.labelNew'.
-- 
-- If you instead want to set the alignment of the label as a whole,
-- use 'GI.Gtk.Objects.Widget.widgetSetHalign' instead.
labelSetJustify ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Gtk.Enums.Justification
    -- ^ /@jtype@/: the new justification
    -> m ()
labelSetJustify self jtype = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let jtype' = (fromIntegral . fromEnum) jtype
    gtk_label_set_justify self' jtype'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetJustifyMethodInfo
instance (signature ~ (Gtk.Enums.Justification -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetJustifyMethodInfo a signature where
    overloadedMethod = labelSetJustify

instance O.OverloadedMethodInfo LabelSetJustifyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetJustify",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetJustify"
        })


#endif

-- method Label::set_label
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new text to set for the label"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_label" gtk_label_set_label :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CString ->                              -- str : TBasicType TUTF8
    IO ()

-- | Sets the text of the label.
-- 
-- The label is interpreted as including embedded underlines and\/or Pango
-- markup depending on the values of the [Label:useUnderline]("GI.Gtk.Objects.Label#g:attr:useUnderline")
-- and [Label:useMarkup]("GI.Gtk.Objects.Label#g:attr:useMarkup") properties.
labelSetLabel ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> T.Text
    -- ^ /@str@/: the new text to set for the label
    -> m ()
labelSetLabel self str = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    str' <- textToCString str
    gtk_label_set_label self' str'
    touchManagedPtr self
    freeMem str'
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetLabelMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetLabelMethodInfo a signature where
    overloadedMethod = labelSetLabel

instance O.OverloadedMethodInfo LabelSetLabelMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetLabel",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetLabel"
        })


#endif

-- method Label::set_lines
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "lines"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the desired number of lines, or -1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_lines" gtk_label_set_lines :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    Int32 ->                                -- lines : TBasicType TInt
    IO ()

-- | Sets the number of lines to which an ellipsized, wrapping label
-- should be limited.
-- 
-- This has no effect if the label is not wrapping or ellipsized.
-- Set this to -1 if you don’t want to limit the number of lines.
labelSetLines ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Int32
    -- ^ /@lines@/: the desired number of lines, or -1
    -> m ()
labelSetLines self lines = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    gtk_label_set_lines self' lines
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetLinesMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetLinesMethodInfo a signature where
    overloadedMethod = labelSetLines

instance O.OverloadedMethodInfo LabelSetLinesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetLines",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetLines"
        })


#endif

-- method Label::set_markup
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the markup string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_markup" gtk_label_set_markup :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CString ->                              -- str : TBasicType TUTF8
    IO ()

-- | Sets the labels text and attributes from markup.
-- 
-- The string must be marked up with Pango markup
-- (see 'GI.Pango.Functions.parseMarkup').
-- 
-- If /@str@/ is external data, you may need to escape it
-- with 'GI.GLib.Functions.markupEscapeText' or @/GLib.markup_printf_escaped/@:
-- 
-- 
-- === /c code/
-- >GtkWidget *self = gtk_label_new (NULL);
-- >const char *str = "...";
-- >const char *format = "<span style=\"italic\">\%s</span>";
-- >char *markup;
-- >
-- >markup = g_markup_printf_escaped (format, str);
-- >gtk_label_set_markup (GTK_LABEL (self), markup);
-- >g_free (markup);
-- 
-- 
-- This function sets the [Label:useMarkup]("GI.Gtk.Objects.Label#g:attr:useMarkup") property
-- to true.
-- 
-- Also see 'GI.Gtk.Objects.Label.labelSetText'.
labelSetMarkup ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> T.Text
    -- ^ /@str@/: the markup string
    -> m ()
labelSetMarkup self str = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    str' <- textToCString str
    gtk_label_set_markup self' str'
    touchManagedPtr self
    freeMem str'
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetMarkupMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetMarkupMethodInfo a signature where
    overloadedMethod = labelSetMarkup

instance O.OverloadedMethodInfo LabelSetMarkupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetMarkup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetMarkup"
        })


#endif

-- method Label::set_markup_with_mnemonic
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the markup string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_markup_with_mnemonic" gtk_label_set_markup_with_mnemonic :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CString ->                              -- str : TBasicType TUTF8
    IO ()

-- | Sets the labels text, attributes and mnemonic from markup.
-- 
-- Parses /@str@/ which is marked up with Pango markup (see 'GI.Pango.Functions.parseMarkup'),
-- setting the label’s text and attribute list based on the parse results.
-- If characters in /@str@/ are preceded by an underscore, they are underlined
-- indicating that they represent a keyboard accelerator called a mnemonic.
-- 
-- The mnemonic key can be used to activate another widget, chosen
-- automatically, or explicitly using 'GI.Gtk.Objects.Label.labelSetMnemonicWidget'.
labelSetMarkupWithMnemonic ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> T.Text
    -- ^ /@str@/: the markup string
    -> m ()
labelSetMarkupWithMnemonic self str = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    str' <- textToCString str
    gtk_label_set_markup_with_mnemonic self' str'
    touchManagedPtr self
    freeMem str'
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetMarkupWithMnemonicMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetMarkupWithMnemonicMethodInfo a signature where
    overloadedMethod = labelSetMarkupWithMnemonic

instance O.OverloadedMethodInfo LabelSetMarkupWithMnemonicMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetMarkupWithMnemonic",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetMarkupWithMnemonic"
        })


#endif

-- method Label::set_max_width_chars
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_chars"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new maximum width, in characters."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_max_width_chars" gtk_label_set_max_width_chars :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    Int32 ->                                -- n_chars : TBasicType TInt
    IO ()

-- | Sets the maximum width of the label in characters.
labelSetMaxWidthChars ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Int32
    -- ^ /@nChars@/: the new maximum width, in characters.
    -> m ()
labelSetMaxWidthChars self nChars = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    gtk_label_set_max_width_chars self' nChars
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetMaxWidthCharsMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetMaxWidthCharsMethodInfo a signature where
    overloadedMethod = labelSetMaxWidthChars

instance O.OverloadedMethodInfo LabelSetMaxWidthCharsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetMaxWidthChars",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetMaxWidthChars"
        })


#endif

-- method Label::set_mnemonic_widget
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the target widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_mnemonic_widget" gtk_label_set_mnemonic_widget :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    Ptr Gtk.Widget.Widget ->                -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Associate the label with its mnemonic target.
-- 
-- If the label has been set so that it has a mnemonic key (using
-- i.e. 'GI.Gtk.Objects.Label.labelSetMarkupWithMnemonic',
-- 'GI.Gtk.Objects.Label.labelSetTextWithMnemonic',
-- 'GI.Gtk.Objects.Label.labelNewWithMnemonic'
-- or the [property/@gtk@/.Label:use_underline] property) the label can
-- be associated with a widget that is the target of the mnemonic.
-- When the label is inside a widget (like a t'GI.Gtk.Objects.Button.Button' or a
-- t'GI.Gtk.Objects.Notebook.Notebook' tab) it is automatically associated with the
-- correct widget, but sometimes (i.e. when the target is a t'GI.Gtk.Objects.Entry.Entry'
-- next to the label) you need to set it explicitly using this function.
-- 
-- The target widget will be accelerated by emitting the
-- [Widget::mnemonicActivate]("GI.Gtk.Objects.Widget#g:signal:mnemonicActivate") signal on it. The default handler
-- for this signal will activate the widget if there are no mnemonic
-- collisions and toggle focus between the colliding widgets otherwise.
labelSetMnemonicWidget ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a, Gtk.Widget.IsWidget b) =>
    a
    -- ^ /@self@/: a label
    -> Maybe (b)
    -- ^ /@widget@/: the target widget
    -> m ()
labelSetMnemonicWidget self widget = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    maybeWidget <- case widget of
        Nothing -> return FP.nullPtr
        Just jWidget -> do
            jWidget' <- unsafeManagedPtrCastPtr jWidget
            return jWidget'
    gtk_label_set_mnemonic_widget self' maybeWidget
    touchManagedPtr self
    whenJust widget touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetMnemonicWidgetMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsLabel a, Gtk.Widget.IsWidget b) => O.OverloadedMethod LabelSetMnemonicWidgetMethodInfo a signature where
    overloadedMethod = labelSetMnemonicWidget

instance O.OverloadedMethodInfo LabelSetMnemonicWidgetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetMnemonicWidget",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetMnemonicWidget"
        })


#endif

-- method Label::set_natural_wrap_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "wrap_mode"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "NaturalWrapMode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the line wrapping mode"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_natural_wrap_mode" gtk_label_set_natural_wrap_mode :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CUInt ->                                -- wrap_mode : TInterface (Name {namespace = "Gtk", name = "NaturalWrapMode"})
    IO ()

-- | Selects the line wrapping for the natural size request.
-- 
-- This only affects the natural size requested, for the actual wrapping used,
-- see the [Label:wrapMode]("GI.Gtk.Objects.Label#g:attr:wrapMode") property.
-- 
-- /Since: 4.6/
labelSetNaturalWrapMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Gtk.Enums.NaturalWrapMode
    -- ^ /@wrapMode@/: the line wrapping mode
    -> m ()
labelSetNaturalWrapMode self wrapMode = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let wrapMode' = (fromIntegral . fromEnum) wrapMode
    gtk_label_set_natural_wrap_mode self' wrapMode'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetNaturalWrapModeMethodInfo
instance (signature ~ (Gtk.Enums.NaturalWrapMode -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetNaturalWrapModeMethodInfo a signature where
    overloadedMethod = labelSetNaturalWrapMode

instance O.OverloadedMethodInfo LabelSetNaturalWrapModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetNaturalWrapMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetNaturalWrapMode"
        })


#endif

-- method Label::set_selectable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "setting"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "true to allow selecting text in the label"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_selectable" gtk_label_set_selectable :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CInt ->                                 -- setting : TBasicType TBoolean
    IO ()

-- | Makes text in the label selectable.
-- 
-- Selectable labels allow the user to select text from the label,
-- for copy-and-paste.
labelSetSelectable ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Bool
    -- ^ /@setting@/: true to allow selecting text in the label
    -> m ()
labelSetSelectable self setting = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let setting' = (P.fromIntegral . P.fromEnum) setting
    gtk_label_set_selectable self' setting'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetSelectableMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetSelectableMethodInfo a signature where
    overloadedMethod = labelSetSelectable

instance O.OverloadedMethodInfo LabelSetSelectableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetSelectable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetSelectable"
        })


#endif

-- method Label::set_single_line_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "single_line_mode"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "true to enable single line mode"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_single_line_mode" gtk_label_set_single_line_mode :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CInt ->                                 -- single_line_mode : TBasicType TBoolean
    IO ()

-- | Sets whether the label is in single line mode.
labelSetSingleLineMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Bool
    -- ^ /@singleLineMode@/: true to enable single line mode
    -> m ()
labelSetSingleLineMode self singleLineMode = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let singleLineMode' = (P.fromIntegral . P.fromEnum) singleLineMode
    gtk_label_set_single_line_mode self' singleLineMode'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetSingleLineModeMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetSingleLineModeMethodInfo a signature where
    overloadedMethod = labelSetSingleLineMode

instance O.OverloadedMethodInfo LabelSetSingleLineModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetSingleLineMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetSingleLineMode"
        })


#endif

-- method Label::set_tabs
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "tabs"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "TabArray" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "tab stops" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_tabs" gtk_label_set_tabs :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    Ptr Pango.TabArray.TabArray ->          -- tabs : TInterface (Name {namespace = "Pango", name = "TabArray"})
    IO ()

-- | Sets tab stops for the label.
-- 
-- /Since: 4.8/
labelSetTabs ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Maybe (Pango.TabArray.TabArray)
    -- ^ /@tabs@/: tab stops
    -> m ()
labelSetTabs self tabs = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    maybeTabs <- case tabs of
        Nothing -> return FP.nullPtr
        Just jTabs -> do
            jTabs' <- unsafeManagedPtrGetPtr jTabs
            return jTabs'
    gtk_label_set_tabs self' maybeTabs
    touchManagedPtr self
    whenJust tabs touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetTabsMethodInfo
instance (signature ~ (Maybe (Pango.TabArray.TabArray) -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetTabsMethodInfo a signature where
    overloadedMethod = labelSetTabs

instance O.OverloadedMethodInfo LabelSetTabsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetTabs",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetTabs"
        })


#endif

-- method Label::set_text
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the text to show in @self"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_text" gtk_label_set_text :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CString ->                              -- str : TBasicType TUTF8
    IO ()

-- | Sets the text for the label.
-- 
-- It overwrites any text that was there before and clears any
-- previously set mnemonic accelerators, and sets the
-- [Label:useUnderline]("GI.Gtk.Objects.Label#g:attr:useUnderline") and
-- [Label:useMarkup]("GI.Gtk.Objects.Label#g:attr:useMarkup") properties to false.
-- 
-- Also see 'GI.Gtk.Objects.Label.labelSetMarkup'.
labelSetText ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> T.Text
    -- ^ /@str@/: the text to show in /@self@/
    -> m ()
labelSetText self str = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    str' <- textToCString str
    gtk_label_set_text self' str'
    touchManagedPtr self
    freeMem str'
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetTextMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetTextMethodInfo a signature where
    overloadedMethod = labelSetText

instance O.OverloadedMethodInfo LabelSetTextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetText",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetText"
        })


#endif

-- method Label::set_text_with_mnemonic
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "str"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the text" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_text_with_mnemonic" gtk_label_set_text_with_mnemonic :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CString ->                              -- str : TBasicType TUTF8
    IO ()

-- | Sets the text for the label, with mnemonics.
-- 
-- If characters in /@str@/ are preceded by an underscore, they are underlined
-- indicating that they represent a keyboard accelerator called a mnemonic.
-- The mnemonic key can be used to activate another widget, chosen
-- automatically, or explicitly using 'GI.Gtk.Objects.Label.labelSetMnemonicWidget'.
labelSetTextWithMnemonic ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> T.Text
    -- ^ /@str@/: the text
    -> m ()
labelSetTextWithMnemonic self str = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    str' <- textToCString str
    gtk_label_set_text_with_mnemonic self' str'
    touchManagedPtr self
    freeMem str'
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetTextWithMnemonicMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetTextWithMnemonicMethodInfo a signature where
    overloadedMethod = labelSetTextWithMnemonic

instance O.OverloadedMethodInfo LabelSetTextWithMnemonicMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetTextWithMnemonic",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetTextWithMnemonic"
        })


#endif

-- method Label::set_use_markup
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "setting"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "true if the label\8217s text should be parsed for markup."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_use_markup" gtk_label_set_use_markup :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CInt ->                                 -- setting : TBasicType TBoolean
    IO ()

-- | Sets whether the text of the label contains markup.
-- 
-- See 'GI.Gtk.Objects.Label.labelSetMarkup'.
labelSetUseMarkup ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Bool
    -- ^ /@setting@/: true if the label’s text should be parsed for markup.
    -> m ()
labelSetUseMarkup self setting = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let setting' = (P.fromIntegral . P.fromEnum) setting
    gtk_label_set_use_markup self' setting'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetUseMarkupMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetUseMarkupMethodInfo a signature where
    overloadedMethod = labelSetUseMarkup

instance O.OverloadedMethodInfo LabelSetUseMarkupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetUseMarkup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetUseMarkup"
        })


#endif

-- method Label::set_use_underline
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "setting"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "true if underlines in the text indicate mnemonics"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_use_underline" gtk_label_set_use_underline :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CInt ->                                 -- setting : TBasicType TBoolean
    IO ()

-- | Sets whether underlines in the text indicate mnemonics.
labelSetUseUnderline ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Bool
    -- ^ /@setting@/: true if underlines in the text indicate mnemonics
    -> m ()
labelSetUseUnderline self setting = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let setting' = (P.fromIntegral . P.fromEnum) setting
    gtk_label_set_use_underline self' setting'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetUseUnderlineMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetUseUnderlineMethodInfo a signature where
    overloadedMethod = labelSetUseUnderline

instance O.OverloadedMethodInfo LabelSetUseUnderlineMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetUseUnderline",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetUseUnderline"
        })


#endif

-- method Label::set_width_chars
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_chars"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new desired width, in characters."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_width_chars" gtk_label_set_width_chars :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    Int32 ->                                -- n_chars : TBasicType TInt
    IO ()

-- | Sets the desired width in characters of the label.
labelSetWidthChars ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Int32
    -- ^ /@nChars@/: the new desired width, in characters.
    -> m ()
labelSetWidthChars self nChars = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    gtk_label_set_width_chars self' nChars
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetWidthCharsMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetWidthCharsMethodInfo a signature where
    overloadedMethod = labelSetWidthChars

instance O.OverloadedMethodInfo LabelSetWidthCharsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetWidthChars",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetWidthChars"
        })


#endif

-- method Label::set_wrap
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "wrap"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether to wrap lines"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_wrap" gtk_label_set_wrap :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CInt ->                                 -- wrap : TBasicType TBoolean
    IO ()

-- | Toggles line wrapping within the label.
-- 
-- True makes it break lines if text exceeds the widget’s size.
-- false lets the text get cut off by the edge of the widget if
-- it exceeds the widget size.
-- 
-- Note that setting line wrapping to true does not make the label
-- wrap at its parent widget’s width, because GTK widgets conceptually
-- can’t make their requisition depend on the parent  widget’s size.
-- For a label that wraps at a specific position, set the label’s width
-- using 'GI.Gtk.Objects.Widget.widgetSetSizeRequest'.
labelSetWrap ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Bool
    -- ^ /@wrap@/: whether to wrap lines
    -> m ()
labelSetWrap self wrap = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let wrap' = (P.fromIntegral . P.fromEnum) wrap
    gtk_label_set_wrap self' wrap'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetWrapMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetWrapMethodInfo a signature where
    overloadedMethod = labelSetWrap

instance O.OverloadedMethodInfo LabelSetWrapMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetWrap",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetWrap"
        })


#endif

-- method Label::set_wrap_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "wrap_mode"
--           , argType =
--               TInterface Name { namespace = "Pango" , name = "WrapMode" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the line wrapping mode"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_wrap_mode" gtk_label_set_wrap_mode :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CUInt ->                                -- wrap_mode : TInterface (Name {namespace = "Pango", name = "WrapMode"})
    IO ()

-- | Controls how line wrapping is done.
-- 
-- This only affects the label if line wrapping is on. (See
-- 'GI.Gtk.Objects.Label.labelSetWrap')
-- 
-- The default is 'GI.Pango.Enums.WrapModeWord', which means
-- wrap on word boundaries.
-- 
-- For sizing behavior, also consider the
-- [Label:naturalWrapMode]("GI.Gtk.Objects.Label#g:attr:naturalWrapMode") property.
labelSetWrapMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Pango.Enums.WrapMode
    -- ^ /@wrapMode@/: the line wrapping mode
    -> m ()
labelSetWrapMode self wrapMode = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let wrapMode' = (fromIntegral . fromEnum) wrapMode
    gtk_label_set_wrap_mode self' wrapMode'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetWrapModeMethodInfo
instance (signature ~ (Pango.Enums.WrapMode -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetWrapModeMethodInfo a signature where
    overloadedMethod = labelSetWrapMode

instance O.OverloadedMethodInfo LabelSetWrapModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetWrapMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetWrapMode"
        })


#endif

-- method Label::set_xalign
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "xalign"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new xalign value, between 0 and 1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_xalign" gtk_label_set_xalign :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CFloat ->                               -- xalign : TBasicType TFloat
    IO ()

-- | Sets the @xalign@ of the label.
-- 
-- See the [Label:xalign]("GI.Gtk.Objects.Label#g:attr:xalign") property.
labelSetXalign ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Float
    -- ^ /@xalign@/: the new xalign value, between 0 and 1
    -> m ()
labelSetXalign self xalign = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let xalign' = realToFrac xalign
    gtk_label_set_xalign self' xalign'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetXalignMethodInfo
instance (signature ~ (Float -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetXalignMethodInfo a signature where
    overloadedMethod = labelSetXalign

instance O.OverloadedMethodInfo LabelSetXalignMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetXalign",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetXalign"
        })


#endif

-- method Label::set_yalign
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Label" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a label" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "yalign"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new yalign value, between 0 and 1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_label_set_yalign" gtk_label_set_yalign :: 
    Ptr Label ->                            -- self : TInterface (Name {namespace = "Gtk", name = "Label"})
    CFloat ->                               -- yalign : TBasicType TFloat
    IO ()

-- | Sets the @yalign@ of the label.
-- 
-- See the [Label:yalign]("GI.Gtk.Objects.Label#g:attr:yalign") property.
labelSetYalign ::
    (B.CallStack.HasCallStack, MonadIO m, IsLabel a) =>
    a
    -- ^ /@self@/: a label
    -> Float
    -- ^ /@yalign@/: the new yalign value, between 0 and 1
    -> m ()
labelSetYalign self yalign = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let yalign' = realToFrac yalign
    gtk_label_set_yalign self' yalign'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data LabelSetYalignMethodInfo
instance (signature ~ (Float -> m ()), MonadIO m, IsLabel a) => O.OverloadedMethod LabelSetYalignMethodInfo a signature where
    overloadedMethod = labelSetYalign

instance O.OverloadedMethodInfo LabelSetYalignMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Label.labelSetYalign",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Label.html#v:labelSetYalign"
        })


#endif


