/* -*- Mode: C++; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * This file is part of the LibreOffice project.
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/.
 */

#include "rtftokenizer.hxx"
#include <o3tl/string_view.hxx>
#include <oox/token/namespaces.hxx>
#include <tools/stream.hxx>
#include <svx/dialmgr.hxx>
#include <svx/strings.hrc>
#include <rtl/strbuf.hxx>
#include <rtl/character.hxx>
#include <sal/log.hxx>
#include "rtfskipdestination.hxx"
#include <com/sun/star/io/BufferSizeExceededException.hpp>
#include <com/sun/star/task/XStatusIndicator.hpp>
#include <filter/msfilter/rtfutil.hxx>

#include <frozen/bits/defines.h>
#include <frozen/bits/elsa_std.h>
#include <frozen/unordered_map.h>

using namespace com::sun::star;

namespace writerfilter::rtftok
{
namespace
{
/// Types of an RTF Control Word
enum class RTFControlType
{
    FLAG, // eg \sbknone takes no parameter
    DESTINATION, // eg \fonttbl, if ignored, the whole group should be skipped
    SYMBOL, // eg \tab
    TOGGLE, // eg \b (between on and off)
    VALUE // eg \fs (requires parameter)
};

/// Represents an RTF Control Word
struct RTFSymbol
{
    RTFControlType controlType;
    RTFKeyword index;
    int defValue = 0; ///< Most of the control words default to 0.
};

constexpr std::pair<const OString, RTFSymbol> controlWords[]{
    { "'"_ostr, { RTFControlType::SYMBOL, RTFKeyword::HEXCHAR } },
    { "-"_ostr, { RTFControlType::SYMBOL, RTFKeyword::OPTHYPH } },
    { "*"_ostr, { RTFControlType::SYMBOL, RTFKeyword::IGNORE } },
    { ":"_ostr, { RTFControlType::SYMBOL, RTFKeyword::SUBENTRY } },
    { "\\"_ostr, { RTFControlType::SYMBOL, RTFKeyword::BACKSLASH } },
    { "\n"_ostr, { RTFControlType::SYMBOL, RTFKeyword::PAR } },
    { "\r"_ostr, { RTFControlType::SYMBOL, RTFKeyword::PAR } },
    { "\r\n"_ostr, { RTFControlType::SYMBOL, RTFKeyword::PAR } },
    { "_"_ostr, { RTFControlType::SYMBOL, RTFKeyword::NOBRKHYPH } },
    { "{"_ostr, { RTFControlType::SYMBOL, RTFKeyword::LBRACE } },
    { "|"_ostr, { RTFControlType::SYMBOL, RTFKeyword::FORMULA } },
    { "}"_ostr, { RTFControlType::SYMBOL, RTFKeyword::RBRACE } },
    { "~"_ostr, { RTFControlType::SYMBOL, RTFKeyword::NOBREAK } },
    { "ab"_ostr, { RTFControlType::TOGGLE, RTFKeyword::AB, 1 } },
    { "absh"_ostr, { RTFControlType::VALUE, RTFKeyword::ABSH } },
    { "abslock"_ostr, { RTFControlType::FLAG, RTFKeyword::ABSLOCK } },
    { "absnoovrlp"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ABSNOOVRLP, 1 } },
    { "absw"_ostr, { RTFControlType::VALUE, RTFKeyword::ABSW } },
    { "acaps"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ACAPS, 1 } },
    { "acccircle"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ACCCIRCLE, 1 } },
    { "acccomma"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ACCCOMMA, 1 } },
    { "accdot"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ACCDOT, 1 } },
    { "accnone"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ACCNONE, 1 } },
    { "accunderdot"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ACCUNDERDOT, 1 } },
    { "acf"_ostr, { RTFControlType::VALUE, RTFKeyword::ACF } },
    { "adeff"_ostr, { RTFControlType::VALUE, RTFKeyword::ADEFF } },
    { "additive"_ostr, { RTFControlType::FLAG, RTFKeyword::ADDITIVE } },
    { "adeflang"_ostr, { RTFControlType::VALUE, RTFKeyword::ADEFLANG } },
    { "adjustright"_ostr, { RTFControlType::FLAG, RTFKeyword::ADJUSTRIGHT } },
    { "adn"_ostr, { RTFControlType::VALUE, RTFKeyword::ADN, 6 } },
    { "aenddoc"_ostr, { RTFControlType::FLAG, RTFKeyword::AENDDOC } },
    { "aendnotes"_ostr, { RTFControlType::FLAG, RTFKeyword::AENDNOTES } },
    { "aexpnd"_ostr, { RTFControlType::VALUE, RTFKeyword::AEXPND } },
    { "af"_ostr, { RTFControlType::VALUE, RTFKeyword::AF } },
    { "afelev"_ostr, { RTFControlType::FLAG, RTFKeyword::AFELEV } },
    { "afs"_ostr, { RTFControlType::VALUE, RTFKeyword::AFS, 24 } },
    { "aftnbj"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNBJ } },
    { "aftncn"_ostr, { RTFControlType::DESTINATION, RTFKeyword::AFTNCN } },
    { "aftnnalc"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNALC } },
    { "aftnnar"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNAR } },
    { "aftnnauc"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNAUC } },
    { "aftnnchi"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNCHI } },
    { "aftnnchosung"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNCHOSUNG } },
    { "aftnncnum"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNCNUM } },
    { "aftnndbar"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNDBAR } },
    { "aftnndbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNDBNUM } },
    { "aftnndbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNDBNUMD } },
    { "aftnndbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNDBNUMK } },
    { "aftnndbnumt"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNDBNUMT } },
    { "aftnnganada"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNGANADA } },
    { "aftnngbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNGBNUM } },
    { "aftnngbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNGBNUMD } },
    { "aftnngbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNGBNUMK } },
    { "aftnngbnuml"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNGBNUML } },
    { "aftnnrlc"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNRLC } },
    { "aftnnruc"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNRUC } },
    { "aftnnzodiac"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNZODIAC } },
    { "aftnnzodiacd"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNZODIACD } },
    { "aftnnzodiacl"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNNZODIACL } },
    { "aftnrestart"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNRESTART } },
    { "aftnrstcont"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNRSTCONT } },
    { "aftnsep"_ostr, { RTFControlType::DESTINATION, RTFKeyword::AFTNSEP } },
    { "aftnsepc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::AFTNSEPC } },
    { "aftnstart"_ostr, { RTFControlType::VALUE, RTFKeyword::AFTNSTART, 1 } },
    { "aftntj"_ostr, { RTFControlType::FLAG, RTFKeyword::AFTNTJ } },
    { "ai"_ostr, { RTFControlType::TOGGLE, RTFKeyword::AI, 1 } },
    { "alang"_ostr, { RTFControlType::VALUE, RTFKeyword::ALANG } },
    { "allowfieldendsel"_ostr, { RTFControlType::FLAG, RTFKeyword::ALLOWFIELDENDSEL } },
    { "allprot"_ostr, { RTFControlType::FLAG, RTFKeyword::ALLPROT } },
    { "alntblind"_ostr, { RTFControlType::FLAG, RTFKeyword::ALNTBLIND } },
    { "alt"_ostr, { RTFControlType::FLAG, RTFKeyword::ALT } },
    { "animtext"_ostr, { RTFControlType::VALUE, RTFKeyword::ANIMTEXT } },
    { "annotation"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ANNOTATION } },
    { "annotprot"_ostr, { RTFControlType::FLAG, RTFKeyword::ANNOTPROT } },
    { "ansi"_ostr, { RTFControlType::FLAG, RTFKeyword::ANSI } },
    { "ansicpg"_ostr, { RTFControlType::VALUE, RTFKeyword::ANSICPG } },
    { "aoutl"_ostr, { RTFControlType::TOGGLE, RTFKeyword::AOUTL, 1 } },
    { "ApplyBrkRules"_ostr, { RTFControlType::FLAG, RTFKeyword::APPLYBRKRULES } },
    { "ascaps"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ASCAPS, 1 } },
    { "ashad"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ASHAD, 1 } },
    { "asianbrkrule"_ostr, { RTFControlType::FLAG, RTFKeyword::ASIANBRKRULE } },
    { "aspalpha"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ASPALPHA, 1 } },
    { "aspnum"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ASPNUM, 1 } },
    { "astrike"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ASTRIKE, 1 } },
    { "atnauthor"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ATNAUTHOR } },
    { "atndate"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ATNDATE } },
    { "atnicn"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ATNICN } },
    { "atnid"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ATNID } },
    { "atnparent"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ATNPARENT } },
    { "atnref"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ATNREF } },
    { "atntime"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ATNTIME } },
    { "atrfend"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ATRFEND } },
    { "atrfstart"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ATRFSTART } },
    { "aul"_ostr, { RTFControlType::TOGGLE, RTFKeyword::AUL, 1 } },
    { "auld"_ostr, { RTFControlType::TOGGLE, RTFKeyword::AULD, 1 } },
    { "auldb"_ostr, { RTFControlType::TOGGLE, RTFKeyword::AULDB, 1 } },
    { "aulnone"_ostr, { RTFControlType::TOGGLE, RTFKeyword::AULNONE, 1 } },
    { "aulw"_ostr, { RTFControlType::TOGGLE, RTFKeyword::AULW, 1 } },
    { "aup"_ostr, { RTFControlType::VALUE, RTFKeyword::AUP, 6 } },
    { "author"_ostr, { RTFControlType::DESTINATION, RTFKeyword::AUTHOR } },
    { "autofmtoverride"_ostr, { RTFControlType::FLAG, RTFKeyword::AUTOFMTOVERRIDE } },
    { "b"_ostr, { RTFControlType::TOGGLE, RTFKeyword::B, 1 } },
    { "background"_ostr, { RTFControlType::DESTINATION, RTFKeyword::BACKGROUND } },
    { "bdbfhdr"_ostr, { RTFControlType::FLAG, RTFKeyword::BDBFHDR } },
    { "bdrrlswsix"_ostr, { RTFControlType::FLAG, RTFKeyword::BDRRLSWSIX } },
    { "bgbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::BGBDIAG } },
    { "bgcross"_ostr, { RTFControlType::FLAG, RTFKeyword::BGCROSS } },
    { "bgdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::BGDCROSS } },
    { "bgdkbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::BGDKBDIAG } },
    { "bgdkcross"_ostr, { RTFControlType::FLAG, RTFKeyword::BGDKCROSS } },
    { "bgdkdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::BGDKDCROSS } },
    { "bgdkfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::BGDKFDIAG } },
    { "bgdkhoriz"_ostr, { RTFControlType::FLAG, RTFKeyword::BGDKHORIZ } },
    { "bgdkvert"_ostr, { RTFControlType::FLAG, RTFKeyword::BGDKVERT } },
    { "bgfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::BGFDIAG } },
    { "bghoriz"_ostr, { RTFControlType::FLAG, RTFKeyword::BGHORIZ } },
    { "bgvert"_ostr, { RTFControlType::FLAG, RTFKeyword::BGVERT } },
    { "bin"_ostr, { RTFControlType::VALUE, RTFKeyword::BIN } },
    { "binfsxn"_ostr, { RTFControlType::VALUE, RTFKeyword::BINFSXN } },
    { "binsxn"_ostr, { RTFControlType::VALUE, RTFKeyword::BINSXN } },
    { "bkmkcolf"_ostr, { RTFControlType::VALUE, RTFKeyword::BKMKCOLF } },
    { "bkmkcoll"_ostr, { RTFControlType::VALUE, RTFKeyword::BKMKCOLL } },
    { "bkmkend"_ostr, { RTFControlType::DESTINATION, RTFKeyword::BKMKEND } },
    { "bkmkpub"_ostr, { RTFControlType::FLAG, RTFKeyword::BKMKPUB } },
    { "bkmkstart"_ostr, { RTFControlType::DESTINATION, RTFKeyword::BKMKSTART } },
    { "bliptag"_ostr, { RTFControlType::VALUE, RTFKeyword::BLIPTAG } },
    { "blipuid"_ostr, { RTFControlType::DESTINATION, RTFKeyword::BLIPUID } },
    { "blipupi"_ostr, { RTFControlType::VALUE, RTFKeyword::BLIPUPI } },
    { "blue"_ostr, { RTFControlType::VALUE, RTFKeyword::BLUE } },
    { "bookfold"_ostr, { RTFControlType::FLAG, RTFKeyword::BOOKFOLD } },
    { "bookfoldrev"_ostr, { RTFControlType::FLAG, RTFKeyword::BOOKFOLDREV } },
    { "bookfoldsheets"_ostr, { RTFControlType::VALUE, RTFKeyword::BOOKFOLDSHEETS } },
    { "box"_ostr, { RTFControlType::FLAG, RTFKeyword::BOX } },
    { "brdrart"_ostr, { RTFControlType::VALUE, RTFKeyword::BRDRART } },
    { "brdrb"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRB } },
    { "brdrbar"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRBAR } },
    { "brdrbtw"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRBTW } },
    { "brdrcf"_ostr, { RTFControlType::VALUE, RTFKeyword::BRDRCF } },
    { "brdrdash"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRDASH } },
    { "brdrdashd"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRDASHD } },
    { "brdrdashdd"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRDASHDD } },
    { "brdrdashdotstr"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRDASHDOTSTR } },
    { "brdrdashsm"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRDASHSM } },
    { "brdrdb"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRDB } },
    { "brdrdot"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRDOT } },
    { "brdremboss"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDREMBOSS } },
    { "brdrengrave"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRENGRAVE } },
    { "brdrframe"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRFRAME } },
    { "brdrhair"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRHAIR } },
    { "brdrinset"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRINSET } },
    { "brdrl"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRL } },
    { "brdrnil"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRNIL } },
    { "brdrnone"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRNONE } },
    { "brdroutset"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDROUTSET } },
    { "brdrr"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRR } },
    { "brdrs"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRS } },
    { "brdrsh"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRSH } },
    { "brdrt"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRT } },
    { "brdrtbl"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTBL } },
    { "brdrth"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTH } },
    { "brdrthtnlg"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTHTNLG } },
    { "brdrthtnmg"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTHTNMG } },
    { "brdrthtnsg"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTHTNSG } },
    { "brdrtnthlg"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTNTHLG } },
    { "brdrtnthmg"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTNTHMG } },
    { "brdrtnthsg"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTNTHSG } },
    { "brdrtnthtnlg"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTNTHTNLG } },
    { "brdrtnthtnmg"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTNTHTNMG } },
    { "brdrtnthtnsg"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTNTHTNSG } },
    { "brdrtriple"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRTRIPLE } },
    { "brdrw"_ostr, { RTFControlType::VALUE, RTFKeyword::BRDRW } },
    { "brdrwavy"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRWAVY } },
    { "brdrwavydb"_ostr, { RTFControlType::FLAG, RTFKeyword::BRDRWAVYDB } },
    { "brkfrm"_ostr, { RTFControlType::FLAG, RTFKeyword::BRKFRM } },
    { "brsp"_ostr, { RTFControlType::VALUE, RTFKeyword::BRSP } },
    { "bullet"_ostr, { RTFControlType::SYMBOL, RTFKeyword::BULLET } },
    { "buptim"_ostr, { RTFControlType::DESTINATION, RTFKeyword::BUPTIM } },
    { "bxe"_ostr, { RTFControlType::FLAG, RTFKeyword::BXE } },
    { "caccentfive"_ostr, { RTFControlType::FLAG, RTFKeyword::CACCENTFIVE } },
    { "caccentfour"_ostr, { RTFControlType::FLAG, RTFKeyword::CACCENTFOUR } },
    { "caccentone"_ostr, { RTFControlType::FLAG, RTFKeyword::CACCENTONE } },
    { "caccentsix"_ostr, { RTFControlType::FLAG, RTFKeyword::CACCENTSIX } },
    { "caccentthree"_ostr, { RTFControlType::FLAG, RTFKeyword::CACCENTTHREE } },
    { "caccenttwo"_ostr, { RTFControlType::FLAG, RTFKeyword::CACCENTTWO } },
    { "cachedcolbal"_ostr, { RTFControlType::FLAG, RTFKeyword::CACHEDCOLBAL } },
    { "caps"_ostr, { RTFControlType::TOGGLE, RTFKeyword::CAPS, 1 } },
    { "category"_ostr, { RTFControlType::DESTINATION, RTFKeyword::CATEGORY } },
    { "cb"_ostr, { RTFControlType::VALUE, RTFKeyword::CB } },
    { "cbackgroundone"_ostr, { RTFControlType::FLAG, RTFKeyword::CBACKGROUNDONE } },
    { "cbackgroundtwo"_ostr, { RTFControlType::FLAG, RTFKeyword::CBACKGROUNDTWO } },
    { "cbpat"_ostr, { RTFControlType::VALUE, RTFKeyword::CBPAT } },
    { "cchs"_ostr, { RTFControlType::VALUE, RTFKeyword::CCHS } },
    { "cell"_ostr, { RTFControlType::SYMBOL, RTFKeyword::CELL } },
    { "cellx"_ostr, { RTFControlType::VALUE, RTFKeyword::CELLX } },
    { "cf"_ostr, { RTFControlType::VALUE, RTFKeyword::CF } },
    { "cfollowedhyperlink"_ostr, { RTFControlType::FLAG, RTFKeyword::CFOLLOWEDHYPERLINK } },
    { "cfpat"_ostr, { RTFControlType::VALUE, RTFKeyword::CFPAT } },
    { "cgrid"_ostr, { RTFControlType::VALUE, RTFKeyword::CGRID } },
    { "charrsid"_ostr, { RTFControlType::VALUE, RTFKeyword::CHARRSID } },
    { "charscalex"_ostr, { RTFControlType::VALUE, RTFKeyword::CHARSCALEX, 100 } },
    { "chatn"_ostr, { RTFControlType::SYMBOL, RTFKeyword::CHATN } },
    { "chbgbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGBDIAG } },
    { "chbgcross"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGCROSS } },
    { "chbgdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGDCROSS } },
    { "chbgdkbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGDKBDIAG } },
    { "chbgdkcross"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGDKCROSS } },
    { "chbgdkdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGDKDCROSS } },
    { "chbgdkfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGDKFDIAG } },
    { "chbgdkhoriz"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGDKHORIZ } },
    { "chbgdkvert"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGDKVERT } },
    { "chbgfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGFDIAG } },
    { "chbghoriz"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGHORIZ } },
    { "chbgvert"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBGVERT } },
    { "chbrdr"_ostr, { RTFControlType::FLAG, RTFKeyword::CHBRDR } },
    { "chcbpat"_ostr, { RTFControlType::VALUE, RTFKeyword::CHCBPAT } },
    { "chcfpat"_ostr, { RTFControlType::VALUE, RTFKeyword::CHCFPAT } },
    { "chdate"_ostr, { RTFControlType::SYMBOL, RTFKeyword::CHDATE } },
    { "chdpa"_ostr, { RTFControlType::SYMBOL, RTFKeyword::CHDPA } },
    { "chdpl"_ostr, { RTFControlType::SYMBOL, RTFKeyword::CHDPL } },
    { "chftn"_ostr, { RTFControlType::SYMBOL, RTFKeyword::CHFTN } },
    { "chftnsep"_ostr, { RTFControlType::SYMBOL, RTFKeyword::CHFTNSEP } },
    { "chftnsepc"_ostr, { RTFControlType::SYMBOL, RTFKeyword::CHFTNSEPC } },
    { "chpgn"_ostr, { RTFControlType::SYMBOL, RTFKeyword::CHPGN } },
    { "chhres"_ostr, { RTFControlType::VALUE, RTFKeyword::CHHRES } },
    { "chshdng"_ostr, { RTFControlType::VALUE, RTFKeyword::CHSHDNG } },
    { "chtime"_ostr, { RTFControlType::SYMBOL, RTFKeyword::CHTIME } },
    { "chyperlink"_ostr, { RTFControlType::FLAG, RTFKeyword::CHYPERLINK } },
    { "clbgbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGBDIAG } },
    { "clbgcross"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGCROSS } },
    { "clbgdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGDCROSS } },
    { "clbgdkbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGDKBDIAG } },
    { "clbgdkcross"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGDKCROSS } },
    { "clbgdkdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGDKDCROSS } },
    { "clbgdkfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGDKFDIAG } },
    { "clbgdkhor"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGDKHOR } },
    { "clbgdkvert"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGDKVERT } },
    { "clbgfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGFDIAG } },
    { "clbghoriz"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGHORIZ } },
    { "clbgvert"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBGVERT } },
    { "clbrdrb"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBRDRB } },
    { "clbrdrl"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBRDRL } },
    { "clbrdrr"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBRDRR } },
    { "clbrdrt"_ostr, { RTFControlType::FLAG, RTFKeyword::CLBRDRT } },
    { "clcbpat"_ostr, { RTFControlType::VALUE, RTFKeyword::CLCBPAT } },
    { "clcbpatraw"_ostr, { RTFControlType::VALUE, RTFKeyword::CLCBPATRAW } },
    { "clcfpat"_ostr, { RTFControlType::VALUE, RTFKeyword::CLCFPAT } },
    { "clcfpatraw"_ostr, { RTFControlType::VALUE, RTFKeyword::CLCFPATRAW } },
    { "cldel"_ostr, { RTFControlType::FLAG, RTFKeyword::CLDEL } },
    { "cldelauth"_ostr, { RTFControlType::VALUE, RTFKeyword::CLDELAUTH } },
    { "cldeldttm"_ostr, { RTFControlType::VALUE, RTFKeyword::CLDELDTTM } },
    { "cldgll"_ostr, { RTFControlType::FLAG, RTFKeyword::CLDGLL } },
    { "cldglu"_ostr, { RTFControlType::FLAG, RTFKeyword::CLDGLU } },
    { "clFitText"_ostr, { RTFControlType::FLAG, RTFKeyword::CLFITTEXT } },
    { "clftsWidth"_ostr, { RTFControlType::VALUE, RTFKeyword::CLFTSWIDTH } },
    { "clhidemark"_ostr, { RTFControlType::FLAG, RTFKeyword::CLHIDEMARK } },
    { "clins"_ostr, { RTFControlType::FLAG, RTFKeyword::CLINS } },
    { "clinsauth"_ostr, { RTFControlType::VALUE, RTFKeyword::CLINSAUTH } },
    { "clinsdttm"_ostr, { RTFControlType::VALUE, RTFKeyword::CLINSDTTM } },
    { "clmgf"_ostr, { RTFControlType::FLAG, RTFKeyword::CLMGF } },
    { "clmrg"_ostr, { RTFControlType::FLAG, RTFKeyword::CLMRG } },
    { "clmrgd"_ostr, { RTFControlType::FLAG, RTFKeyword::CLMRGD } },
    { "clmrgdauth"_ostr, { RTFControlType::VALUE, RTFKeyword::CLMRGDAUTH } },
    { "clmrgddttm"_ostr, { RTFControlType::VALUE, RTFKeyword::CLMRGDDTTM } },
    { "clmrgdr"_ostr, { RTFControlType::FLAG, RTFKeyword::CLMRGDR } },
    { "clNoWrap"_ostr, { RTFControlType::FLAG, RTFKeyword::CLNOWRAP } },
    { "clpadb"_ostr, { RTFControlType::VALUE, RTFKeyword::CLPADB } },
    { "clpadfb"_ostr, { RTFControlType::VALUE, RTFKeyword::CLPADFB } },
    { "clpadfl"_ostr, { RTFControlType::VALUE, RTFKeyword::CLPADFL } },
    { "clpadfr"_ostr, { RTFControlType::VALUE, RTFKeyword::CLPADFR } },
    { "clpadft"_ostr, { RTFControlType::VALUE, RTFKeyword::CLPADFT } },
    { "clpadl"_ostr, { RTFControlType::VALUE, RTFKeyword::CLPADL } },
    { "clpadr"_ostr, { RTFControlType::VALUE, RTFKeyword::CLPADR } },
    { "clpadt"_ostr, { RTFControlType::VALUE, RTFKeyword::CLPADT } },
    { "clspb"_ostr, { RTFControlType::VALUE, RTFKeyword::CLSPB } },
    { "clspfb"_ostr, { RTFControlType::VALUE, RTFKeyword::CLSPFB } },
    { "clspfl"_ostr, { RTFControlType::VALUE, RTFKeyword::CLSPFL } },
    { "clspfr"_ostr, { RTFControlType::VALUE, RTFKeyword::CLSPFR } },
    { "clspft"_ostr, { RTFControlType::VALUE, RTFKeyword::CLSPFT } },
    { "clspl"_ostr, { RTFControlType::VALUE, RTFKeyword::CLSPL } },
    { "clspr"_ostr, { RTFControlType::VALUE, RTFKeyword::CLSPR } },
    { "clspt"_ostr, { RTFControlType::VALUE, RTFKeyword::CLSPT } },
    { "clshdng"_ostr, { RTFControlType::VALUE, RTFKeyword::CLSHDNG } },
    { "clshdngraw"_ostr, { RTFControlType::VALUE, RTFKeyword::CLSHDNGRAW } },
    { "clshdrawnil"_ostr, { RTFControlType::FLAG, RTFKeyword::CLSHDRAWNIL } },
    { "clsplit"_ostr, { RTFControlType::FLAG, RTFKeyword::CLSPLIT } },
    { "clsplitr"_ostr, { RTFControlType::FLAG, RTFKeyword::CLSPLITR } },
    { "cltxbtlr"_ostr, { RTFControlType::FLAG, RTFKeyword::CLTXBTLR } },
    { "cltxlrtb"_ostr, { RTFControlType::FLAG, RTFKeyword::CLTXLRTB } },
    { "cltxlrtbv"_ostr, { RTFControlType::FLAG, RTFKeyword::CLTXLRTBV } },
    { "cltxtbrl"_ostr, { RTFControlType::FLAG, RTFKeyword::CLTXTBRL } },
    { "cltxtbrlv"_ostr, { RTFControlType::FLAG, RTFKeyword::CLTXTBRLV } },
    { "clvertalb"_ostr, { RTFControlType::FLAG, RTFKeyword::CLVERTALB } },
    { "clvertalc"_ostr, { RTFControlType::FLAG, RTFKeyword::CLVERTALC } },
    { "clvertalt"_ostr, { RTFControlType::FLAG, RTFKeyword::CLVERTALT } },
    { "clvmgf"_ostr, { RTFControlType::FLAG, RTFKeyword::CLVMGF } },
    { "clvmrg"_ostr, { RTFControlType::FLAG, RTFKeyword::CLVMRG } },
    { "clwWidth"_ostr, { RTFControlType::VALUE, RTFKeyword::CLWWIDTH } },
    { "cmaindarkone"_ostr, { RTFControlType::FLAG, RTFKeyword::CMAINDARKONE } },
    { "cmaindarktwo"_ostr, { RTFControlType::FLAG, RTFKeyword::CMAINDARKTWO } },
    { "cmainlightone"_ostr, { RTFControlType::FLAG, RTFKeyword::CMAINLIGHTONE } },
    { "cmainlighttwo"_ostr, { RTFControlType::FLAG, RTFKeyword::CMAINLIGHTTWO } },
    { "collapsed"_ostr, { RTFControlType::FLAG, RTFKeyword::COLLAPSED } },
    { "colno"_ostr, { RTFControlType::VALUE, RTFKeyword::COLNO } },
    { "colorschememapping"_ostr, { RTFControlType::DESTINATION, RTFKeyword::COLORSCHEMEMAPPING } },
    { "colortbl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::COLORTBL } },
    { "cols"_ostr, { RTFControlType::VALUE, RTFKeyword::COLS, 1 } },
    { "colsr"_ostr, { RTFControlType::VALUE, RTFKeyword::COLSR } },
    { "colsx"_ostr, { RTFControlType::VALUE, RTFKeyword::COLSX, 720 } },
    { "column"_ostr, { RTFControlType::SYMBOL, RTFKeyword::COLUMN } },
    { "colw"_ostr, { RTFControlType::VALUE, RTFKeyword::COLW } },
    { "comment"_ostr, { RTFControlType::DESTINATION, RTFKeyword::COMMENT } },
    { "company"_ostr, { RTFControlType::DESTINATION, RTFKeyword::COMPANY } },
    { "contextualspace"_ostr, { RTFControlType::FLAG, RTFKeyword::CONTEXTUALSPACE } },
    { "cpg"_ostr, { RTFControlType::VALUE, RTFKeyword::CPG } },
    { "crauth"_ostr, { RTFControlType::VALUE, RTFKeyword::CRAUTH } },
    { "crdate"_ostr, { RTFControlType::VALUE, RTFKeyword::CRDATE } },
    { "creatim"_ostr, { RTFControlType::DESTINATION, RTFKeyword::CREATIM } },
    { "cs"_ostr, { RTFControlType::VALUE, RTFKeyword::CS } },
    { "cshade"_ostr, { RTFControlType::VALUE, RTFKeyword::CSHADE } },
    { "ctextone"_ostr, { RTFControlType::FLAG, RTFKeyword::CTEXTONE } },
    { "ctexttwo"_ostr, { RTFControlType::FLAG, RTFKeyword::CTEXTTWO } },
    { "ctint"_ostr, { RTFControlType::VALUE, RTFKeyword::CTINT } },
    { "ctrl"_ostr, { RTFControlType::FLAG, RTFKeyword::CTRL } },
    { "cts"_ostr, { RTFControlType::VALUE, RTFKeyword::CTS } },
    { "cufi"_ostr, { RTFControlType::VALUE, RTFKeyword::CUFI } },
    { "culi"_ostr, { RTFControlType::VALUE, RTFKeyword::CULI } },
    { "curi"_ostr, { RTFControlType::VALUE, RTFKeyword::CURI } },
    { "cvmme"_ostr, { RTFControlType::FLAG, RTFKeyword::CVMME } },
    { "datafield"_ostr, { RTFControlType::DESTINATION, RTFKeyword::DATAFIELD } },
    { "datastore"_ostr, { RTFControlType::DESTINATION, RTFKeyword::DATASTORE } },
    { "date"_ostr, { RTFControlType::FLAG, RTFKeyword::DATE } },
    { "dbch"_ostr, { RTFControlType::FLAG, RTFKeyword::DBCH } },
    { "defchp"_ostr, { RTFControlType::DESTINATION, RTFKeyword::DEFCHP } },
    { "deff"_ostr, { RTFControlType::VALUE, RTFKeyword::DEFF } },
    { "defformat"_ostr, { RTFControlType::FLAG, RTFKeyword::DEFFORMAT } },
    { "deflang"_ostr, { RTFControlType::VALUE, RTFKeyword::DEFLANG } },
    { "deflangfe"_ostr, { RTFControlType::VALUE, RTFKeyword::DEFLANGFE } },
    { "defpap"_ostr, { RTFControlType::DESTINATION, RTFKeyword::DEFPAP } },
    { "defshp"_ostr, { RTFControlType::FLAG, RTFKeyword::DEFSHP } },
    { "deftab"_ostr, { RTFControlType::VALUE, RTFKeyword::DEFTAB, 720 } },
    { "deleted"_ostr, { RTFControlType::TOGGLE, RTFKeyword::DELETED, 1 } },
    { "delrsid"_ostr, { RTFControlType::VALUE, RTFKeyword::DELRSID } },
    { "dfrauth"_ostr, { RTFControlType::VALUE, RTFKeyword::DFRAUTH } },
    { "dfrdate"_ostr, { RTFControlType::VALUE, RTFKeyword::DFRDATE } },
    { "dfrmtxtx"_ostr, { RTFControlType::VALUE, RTFKeyword::DFRMTXTX } },
    { "dfrmtxty"_ostr, { RTFControlType::VALUE, RTFKeyword::DFRMTXTY } },
    { "dfrstart"_ostr, { RTFControlType::VALUE, RTFKeyword::DFRSTART } },
    { "dfrstop"_ostr, { RTFControlType::VALUE, RTFKeyword::DFRSTOP } },
    { "dfrxst"_ostr, { RTFControlType::VALUE, RTFKeyword::DFRXST } },
    { "dghorigin"_ostr, { RTFControlType::VALUE, RTFKeyword::DGHORIGIN, 1701 } },
    { "dghshow"_ostr, { RTFControlType::VALUE, RTFKeyword::DGHSHOW, 3 } },
    { "dghspace"_ostr, { RTFControlType::VALUE, RTFKeyword::DGHSPACE, 120 } },
    { "dgmargin"_ostr, { RTFControlType::FLAG, RTFKeyword::DGMARGIN } },
    { "dgsnap"_ostr, { RTFControlType::FLAG, RTFKeyword::DGSNAP } },
    { "dgvorigin"_ostr, { RTFControlType::VALUE, RTFKeyword::DGVORIGIN, 1984 } },
    { "dgvshow"_ostr, { RTFControlType::VALUE, RTFKeyword::DGVSHOW } },
    { "dgvspace"_ostr, { RTFControlType::VALUE, RTFKeyword::DGVSPACE, 120 } },
    { "dibitmap"_ostr, { RTFControlType::VALUE, RTFKeyword::DIBITMAP } },
    { "disabled"_ostr, { RTFControlType::TOGGLE, RTFKeyword::DISABLED, 1 } },
    { "dn"_ostr, { RTFControlType::VALUE, RTFKeyword::DN, 6 } },
    { "dntblnsbdb"_ostr, { RTFControlType::FLAG, RTFKeyword::DNTBLNSBDB } },
    { "do"_ostr, { RTFControlType::DESTINATION, RTFKeyword::DO } },
    { "dobxcolumn"_ostr, { RTFControlType::FLAG, RTFKeyword::DOBXCOLUMN } },
    { "dobxmargin"_ostr, { RTFControlType::FLAG, RTFKeyword::DOBXMARGIN } },
    { "dobxpage"_ostr, { RTFControlType::FLAG, RTFKeyword::DOBXPAGE } },
    { "dobymargin"_ostr, { RTFControlType::FLAG, RTFKeyword::DOBYMARGIN } },
    { "dobypage"_ostr, { RTFControlType::FLAG, RTFKeyword::DOBYPAGE } },
    { "dobypara"_ostr, { RTFControlType::FLAG, RTFKeyword::DOBYPARA } },
    { "doccomm"_ostr, { RTFControlType::DESTINATION, RTFKeyword::DOCCOMM } },
    { "doctemp"_ostr, { RTFControlType::FLAG, RTFKeyword::DOCTEMP } },
    { "doctype"_ostr, { RTFControlType::VALUE, RTFKeyword::DOCTYPE } },
    { "docvar"_ostr, { RTFControlType::DESTINATION, RTFKeyword::DOCVAR } },
    { "dodhgt"_ostr, { RTFControlType::VALUE, RTFKeyword::DODHGT } },
    { "dolock"_ostr, { RTFControlType::FLAG, RTFKeyword::DOLOCK } },
    { "donotembedlingdata"_ostr, { RTFControlType::VALUE, RTFKeyword::DONOTEMBEDLINGDATA } },
    { "donotembedsysfont"_ostr, { RTFControlType::VALUE, RTFKeyword::DONOTEMBEDSYSFONT } },
    { "donotshowcomments"_ostr, { RTFControlType::FLAG, RTFKeyword::DONOTSHOWCOMMENTS } },
    { "donotshowinsdel"_ostr, { RTFControlType::FLAG, RTFKeyword::DONOTSHOWINSDEL } },
    { "donotshowmarkup"_ostr, { RTFControlType::FLAG, RTFKeyword::DONOTSHOWMARKUP } },
    { "donotshowprops"_ostr, { RTFControlType::FLAG, RTFKeyword::DONOTSHOWPROPS } },
    { "dpaendhol"_ostr, { RTFControlType::FLAG, RTFKeyword::DPAENDHOL } },
    { "dpaendl"_ostr, { RTFControlType::VALUE, RTFKeyword::DPAENDL } },
    { "dpaendsol"_ostr, { RTFControlType::FLAG, RTFKeyword::DPAENDSOL } },
    { "dpaendw"_ostr, { RTFControlType::VALUE, RTFKeyword::DPAENDW } },
    { "dparc"_ostr, { RTFControlType::FLAG, RTFKeyword::DPARC } },
    { "dparcflipx"_ostr, { RTFControlType::FLAG, RTFKeyword::DPARCFLIPX } },
    { "dparcflipy"_ostr, { RTFControlType::FLAG, RTFKeyword::DPARCFLIPY } },
    { "dpastarthol"_ostr, { RTFControlType::FLAG, RTFKeyword::DPASTARTHOL } },
    { "dpastartl"_ostr, { RTFControlType::VALUE, RTFKeyword::DPASTARTL } },
    { "dpastartsol"_ostr, { RTFControlType::FLAG, RTFKeyword::DPASTARTSOL } },
    { "dpastartw"_ostr, { RTFControlType::VALUE, RTFKeyword::DPASTARTW } },
    { "dpcallout"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCALLOUT } },
    { "dpcoa"_ostr, { RTFControlType::VALUE, RTFKeyword::DPCOA } },
    { "dpcoaccent"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCOACCENT } },
    { "dpcobestfit"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCOBESTFIT } },
    { "dpcoborder"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCOBORDER } },
    { "dpcodabs"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCODABS } },
    { "dpcodbottom"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCODBOTTOM } },
    { "dpcodcenter"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCODCENTER } },
    { "dpcodescent"_ostr, { RTFControlType::VALUE, RTFKeyword::DPCODESCENT } },
    { "dpcodtop"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCODTOP } },
    { "dpcolength"_ostr, { RTFControlType::VALUE, RTFKeyword::DPCOLENGTH } },
    { "dpcominusx"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCOMINUSX } },
    { "dpcominusy"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCOMINUSY } },
    { "dpcooffset"_ostr, { RTFControlType::VALUE, RTFKeyword::DPCOOFFSET } },
    { "dpcosmarta"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCOSMARTA } },
    { "dpcotdouble"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCOTDOUBLE } },
    { "dpcotright"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCOTRIGHT } },
    { "dpcotsingle"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCOTSINGLE } },
    { "dpcottriple"_ostr, { RTFControlType::FLAG, RTFKeyword::DPCOTTRIPLE } },
    { "dpcount"_ostr, { RTFControlType::VALUE, RTFKeyword::DPCOUNT } },
    { "dpellipse"_ostr, { RTFControlType::FLAG, RTFKeyword::DPELLIPSE } },
    { "dpendgroup"_ostr, { RTFControlType::FLAG, RTFKeyword::DPENDGROUP } },
    { "dpfillbgcb"_ostr, { RTFControlType::VALUE, RTFKeyword::DPFILLBGCB } },
    { "dpfillbgcg"_ostr, { RTFControlType::VALUE, RTFKeyword::DPFILLBGCG } },
    { "dpfillbgcr"_ostr, { RTFControlType::VALUE, RTFKeyword::DPFILLBGCR } },
    { "dpfillbggray"_ostr, { RTFControlType::VALUE, RTFKeyword::DPFILLBGGRAY } },
    { "dpfillbgpal"_ostr, { RTFControlType::FLAG, RTFKeyword::DPFILLBGPAL } },
    { "dpfillfgcb"_ostr, { RTFControlType::VALUE, RTFKeyword::DPFILLFGCB } },
    { "dpfillfgcg"_ostr, { RTFControlType::VALUE, RTFKeyword::DPFILLFGCG } },
    { "dpfillfgcr"_ostr, { RTFControlType::VALUE, RTFKeyword::DPFILLFGCR } },
    { "dpfillfggray"_ostr, { RTFControlType::VALUE, RTFKeyword::DPFILLFGGRAY } },
    { "dpfillfgpal"_ostr, { RTFControlType::FLAG, RTFKeyword::DPFILLFGPAL } },
    { "dpfillpat"_ostr, { RTFControlType::VALUE, RTFKeyword::DPFILLPAT } },
    { "dpgroup"_ostr, { RTFControlType::FLAG, RTFKeyword::DPGROUP } },
    { "dpline"_ostr, { RTFControlType::FLAG, RTFKeyword::DPLINE } },
    { "dplinecob"_ostr, { RTFControlType::VALUE, RTFKeyword::DPLINECOB } },
    { "dplinecog"_ostr, { RTFControlType::VALUE, RTFKeyword::DPLINECOG } },
    { "dplinecor"_ostr, { RTFControlType::VALUE, RTFKeyword::DPLINECOR } },
    { "dplinedado"_ostr, { RTFControlType::FLAG, RTFKeyword::DPLINEDADO } },
    { "dplinedadodo"_ostr, { RTFControlType::FLAG, RTFKeyword::DPLINEDADODO } },
    { "dplinedash"_ostr, { RTFControlType::FLAG, RTFKeyword::DPLINEDASH } },
    { "dplinedot"_ostr, { RTFControlType::FLAG, RTFKeyword::DPLINEDOT } },
    { "dplinegray"_ostr, { RTFControlType::VALUE, RTFKeyword::DPLINEGRAY } },
    { "dplinehollow"_ostr, { RTFControlType::FLAG, RTFKeyword::DPLINEHOLLOW } },
    { "dplinepal"_ostr, { RTFControlType::FLAG, RTFKeyword::DPLINEPAL } },
    { "dplinesolid"_ostr, { RTFControlType::FLAG, RTFKeyword::DPLINESOLID } },
    { "dplinew"_ostr, { RTFControlType::VALUE, RTFKeyword::DPLINEW } },
    { "dppolycount"_ostr, { RTFControlType::VALUE, RTFKeyword::DPPOLYCOUNT } },
    { "dppolygon"_ostr, { RTFControlType::FLAG, RTFKeyword::DPPOLYGON } },
    { "dppolyline"_ostr, { RTFControlType::FLAG, RTFKeyword::DPPOLYLINE } },
    { "dpptx"_ostr, { RTFControlType::VALUE, RTFKeyword::DPPTX } },
    { "dppty"_ostr, { RTFControlType::VALUE, RTFKeyword::DPPTY } },
    { "dprect"_ostr, { RTFControlType::FLAG, RTFKeyword::DPRECT } },
    { "dproundr"_ostr, { RTFControlType::FLAG, RTFKeyword::DPROUNDR } },
    { "dpshadow"_ostr, { RTFControlType::FLAG, RTFKeyword::DPSHADOW } },
    { "dpshadx"_ostr, { RTFControlType::VALUE, RTFKeyword::DPSHADX } },
    { "dpshady"_ostr, { RTFControlType::VALUE, RTFKeyword::DPSHADY } },
    { "dptxbtlr"_ostr, { RTFControlType::FLAG, RTFKeyword::DPTXBTLR } },
    { "dptxbx"_ostr, { RTFControlType::FLAG, RTFKeyword::DPTXBX } },
    { "dptxbxmar"_ostr, { RTFControlType::VALUE, RTFKeyword::DPTXBXMAR } },
    { "dptxbxtext"_ostr, { RTFControlType::DESTINATION, RTFKeyword::DPTXBXTEXT } },
    { "dptxlrtb"_ostr, { RTFControlType::FLAG, RTFKeyword::DPTXLRTB } },
    { "dptxlrtbv"_ostr, { RTFControlType::FLAG, RTFKeyword::DPTXLRTBV } },
    { "dptxtbrl"_ostr, { RTFControlType::FLAG, RTFKeyword::DPTXTBRL } },
    { "dptxtbrlv"_ostr, { RTFControlType::FLAG, RTFKeyword::DPTXTBRLV } },
    { "dpx"_ostr, { RTFControlType::VALUE, RTFKeyword::DPX } },
    { "dpxsize"_ostr, { RTFControlType::VALUE, RTFKeyword::DPXSIZE } },
    { "dpy"_ostr, { RTFControlType::VALUE, RTFKeyword::DPY } },
    { "dpysize"_ostr, { RTFControlType::VALUE, RTFKeyword::DPYSIZE } },
    { "dropcapli"_ostr, { RTFControlType::VALUE, RTFKeyword::DROPCAPLI } },
    { "dropcapt"_ostr, { RTFControlType::VALUE, RTFKeyword::DROPCAPT } },
    { "ds"_ostr, { RTFControlType::VALUE, RTFKeyword::DS } },
    { "dxfrtext"_ostr, { RTFControlType::VALUE, RTFKeyword::DXFRTEXT } },
    { "dy"_ostr, { RTFControlType::VALUE, RTFKeyword::DY } },
    { "ebcend"_ostr, { RTFControlType::DESTINATION, RTFKeyword::EBCEND } },
    { "ebcstart"_ostr, { RTFControlType::DESTINATION, RTFKeyword::EBCSTART } },
    { "edmins"_ostr, { RTFControlType::VALUE, RTFKeyword::EDMINS } },
    { "embo"_ostr, { RTFControlType::TOGGLE, RTFKeyword::EMBO, 1 } },
    { "emdash"_ostr, { RTFControlType::SYMBOL, RTFKeyword::EMDASH } },
    { "emfblip"_ostr, { RTFControlType::FLAG, RTFKeyword::EMFBLIP } },
    { "emspace"_ostr, { RTFControlType::SYMBOL, RTFKeyword::EMSPACE } },
    { "endash"_ostr, { RTFControlType::SYMBOL, RTFKeyword::ENDASH } },
    { "enddoc"_ostr, { RTFControlType::FLAG, RTFKeyword::ENDDOC } },
    { "endnhere"_ostr, { RTFControlType::FLAG, RTFKeyword::ENDNHERE } },
    { "endnotes"_ostr, { RTFControlType::FLAG, RTFKeyword::ENDNOTES } },
    { "enforceprot"_ostr, { RTFControlType::VALUE, RTFKeyword::ENFORCEPROT } },
    { "enspace"_ostr, { RTFControlType::SYMBOL, RTFKeyword::ENSPACE } },
    { "expnd"_ostr, { RTFControlType::VALUE, RTFKeyword::EXPND } },
    { "expndtw"_ostr, { RTFControlType::VALUE, RTFKeyword::EXPNDTW } },
    { "expshrtn"_ostr, { RTFControlType::FLAG, RTFKeyword::EXPSHRTN } },
    { "f"_ostr, { RTFControlType::VALUE, RTFKeyword::F } },
    { "faauto"_ostr, { RTFControlType::FLAG, RTFKeyword::FAAUTO } },
    { "facenter"_ostr, { RTFControlType::FLAG, RTFKeyword::FACENTER } },
    { "facingp"_ostr, { RTFControlType::TOGGLE, RTFKeyword::FACINGP, 1 } },
    { "factoidname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FACTOIDNAME } },
    { "fafixed"_ostr, { RTFControlType::FLAG, RTFKeyword::FAFIXED } },
    { "fahang"_ostr, { RTFControlType::FLAG, RTFKeyword::FAHANG } },
    { "falt"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FALT } },
    { "faroman"_ostr, { RTFControlType::FLAG, RTFKeyword::FAROMAN } },
    { "favar"_ostr, { RTFControlType::FLAG, RTFKeyword::FAVAR } },
    { "fbias"_ostr, { RTFControlType::VALUE, RTFKeyword::FBIAS } },
    { "fbidi"_ostr, { RTFControlType::FLAG, RTFKeyword::FBIDI } },
    { "fbidis"_ostr, { RTFControlType::FLAG, RTFKeyword::FBIDIS } },
    { "fbimajor"_ostr, { RTFControlType::FLAG, RTFKeyword::FBIMAJOR } },
    { "fbiminor"_ostr, { RTFControlType::FLAG, RTFKeyword::FBIMINOR } },
    { "fchars"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FCHARS } },
    { "fcharset"_ostr, { RTFControlType::VALUE, RTFKeyword::FCHARSET } },
    { "fcs"_ostr, { RTFControlType::VALUE, RTFKeyword::FCS } },
    { "fdbmajor"_ostr, { RTFControlType::FLAG, RTFKeyword::FDBMAJOR } },
    { "fdbminor"_ostr, { RTFControlType::FLAG, RTFKeyword::FDBMINOR } },
    { "fdecor"_ostr, { RTFControlType::FLAG, RTFKeyword::FDECOR } },
    { "felnbrelev"_ostr, { RTFControlType::FLAG, RTFKeyword::FELNBRELEV } },
    { "fet"_ostr, { RTFControlType::VALUE, RTFKeyword::FET } },
    { "fetch"_ostr, { RTFControlType::FLAG, RTFKeyword::FETCH } },
    { "ffdefres"_ostr, { RTFControlType::VALUE, RTFKeyword::FFDEFRES } },
    { "ffdeftext"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FFDEFTEXT } },
    { "ffentrymcr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FFENTRYMCR } },
    { "ffexitmcr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FFEXITMCR } },
    { "ffformat"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FFFORMAT } },
    { "ffhaslistbox"_ostr, { RTFControlType::VALUE, RTFKeyword::FFHASLISTBOX } },
    { "ffhelptext"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FFHELPTEXT } },
    { "ffhps"_ostr, { RTFControlType::VALUE, RTFKeyword::FFHPS } },
    { "ffl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FFL } },
    { "ffmaxlen"_ostr, { RTFControlType::VALUE, RTFKeyword::FFMAXLEN } },
    { "ffname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FFNAME } },
    { "ffownhelp"_ostr, { RTFControlType::VALUE, RTFKeyword::FFOWNHELP } },
    { "ffownstat"_ostr, { RTFControlType::VALUE, RTFKeyword::FFOWNSTAT } },
    { "ffprot"_ostr, { RTFControlType::VALUE, RTFKeyword::FFPROT } },
    { "ffrecalc"_ostr, { RTFControlType::VALUE, RTFKeyword::FFRECALC } },
    { "ffres"_ostr, { RTFControlType::VALUE, RTFKeyword::FFRES } },
    { "ffsize"_ostr, { RTFControlType::VALUE, RTFKeyword::FFSIZE } },
    { "ffstattext"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FFSTATTEXT } },
    { "fftype"_ostr, { RTFControlType::VALUE, RTFKeyword::FFTYPE } },
    { "fftypetxt"_ostr, { RTFControlType::VALUE, RTFKeyword::FFTYPETXT } },
    { "fhimajor"_ostr, { RTFControlType::FLAG, RTFKeyword::FHIMAJOR } },
    { "fhiminor"_ostr, { RTFControlType::FLAG, RTFKeyword::FHIMINOR } },
    { "fi"_ostr, { RTFControlType::VALUE, RTFKeyword::FI } },
    { "fid"_ostr, { RTFControlType::VALUE, RTFKeyword::FID } },
    { "field"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FIELD } },
    { "file"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FILE } },
    { "filetbl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FILETBL } },
    { "fittext"_ostr, { RTFControlType::VALUE, RTFKeyword::FITTEXT } },
    { "fjgothic"_ostr, { RTFControlType::FLAG, RTFKeyword::FJGOTHIC } },
    { "fjminchou"_ostr, { RTFControlType::FLAG, RTFKeyword::FJMINCHOU } },
    { "fldalt"_ostr, { RTFControlType::FLAG, RTFKeyword::FLDALT } },
    { "flddirty"_ostr, { RTFControlType::FLAG, RTFKeyword::FLDDIRTY } },
    { "fldedit"_ostr, { RTFControlType::FLAG, RTFKeyword::FLDEDIT } },
    { "fldinst"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FLDINST } },
    { "fldlock"_ostr, { RTFControlType::FLAG, RTFKeyword::FLDLOCK } },
    { "fldpriv"_ostr, { RTFControlType::FLAG, RTFKeyword::FLDPRIV } },
    { "fldrslt"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FLDRSLT } },
    { "fldtype"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FLDTYPE } },
    { "flomajor"_ostr, { RTFControlType::FLAG, RTFKeyword::FLOMAJOR } },
    { "flominor"_ostr, { RTFControlType::FLAG, RTFKeyword::FLOMINOR } },
    { "fmodern"_ostr, { RTFControlType::FLAG, RTFKeyword::FMODERN } },
    { "fn"_ostr, { RTFControlType::VALUE, RTFKeyword::FN } },
    { "fname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FNAME } },
    { "fnetwork"_ostr, { RTFControlType::FLAG, RTFKeyword::FNETWORK } },
    { "fnil"_ostr, { RTFControlType::FLAG, RTFKeyword::FNIL } },
    { "fnonfilesys"_ostr, { RTFControlType::FLAG, RTFKeyword::FNONFILESYS } },
    { "fontemb"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FONTEMB } },
    { "fontfile"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FONTFILE } },
    { "fonttbl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FONTTBL } },
    { "footer"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FOOTER } },
    { "footerf"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FOOTERF } },
    { "footerl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FOOTERL } },
    { "footerr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FOOTERR } },
    { "footery"_ostr, { RTFControlType::VALUE, RTFKeyword::FOOTERY, 720 } },
    { "footnote"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FOOTNOTE } },
    { "forceupgrade"_ostr, { RTFControlType::FLAG, RTFKeyword::FORCEUPGRADE } },
    { "formdisp"_ostr, { RTFControlType::FLAG, RTFKeyword::FORMDISP } },
    { "formfield"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FORMFIELD } },
    { "formprot"_ostr, { RTFControlType::FLAG, RTFKeyword::FORMPROT } },
    { "formshade"_ostr, { RTFControlType::FLAG, RTFKeyword::FORMSHADE } },
    { "fosnum"_ostr, { RTFControlType::VALUE, RTFKeyword::FOSNUM } },
    { "fprq"_ostr, { RTFControlType::VALUE, RTFKeyword::FPRQ } },
    { "fracwidth"_ostr, { RTFControlType::FLAG, RTFKeyword::FRACWIDTH } },
    { "frelative"_ostr, { RTFControlType::VALUE, RTFKeyword::FRELATIVE } },
    { "frmtxbtlr"_ostr, { RTFControlType::FLAG, RTFKeyword::FRMTXBTLR } },
    { "frmtxlrtb"_ostr, { RTFControlType::FLAG, RTFKeyword::FRMTXLRTB } },
    { "frmtxlrtbv"_ostr, { RTFControlType::FLAG, RTFKeyword::FRMTXLRTBV } },
    { "frmtxtbrl"_ostr, { RTFControlType::FLAG, RTFKeyword::FRMTXTBRL } },
    { "frmtxtbrlv"_ostr, { RTFControlType::FLAG, RTFKeyword::FRMTXTBRLV } },
    { "froman"_ostr, { RTFControlType::FLAG, RTFKeyword::FROMAN } },
    { "fromhtml"_ostr, { RTFControlType::VALUE, RTFKeyword::FROMHTML } },
    { "fromtext"_ostr, { RTFControlType::FLAG, RTFKeyword::FROMTEXT } },
    { "fs"_ostr, { RTFControlType::VALUE, RTFKeyword::FS, 24 } },
    { "fscript"_ostr, { RTFControlType::FLAG, RTFKeyword::FSCRIPT } },
    { "fswiss"_ostr, { RTFControlType::FLAG, RTFKeyword::FSWISS } },
    { "ftech"_ostr, { RTFControlType::FLAG, RTFKeyword::FTECH } },
    { "ftnalt"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNALT } },
    { "ftnbj"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNBJ } },
    { "ftncn"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FTNCN } },
    { "ftnil"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNIL } },
    { "ftnlytwnine"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNLYTWNINE } },
    { "ftnnalc"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNALC } },
    { "ftnnar"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNAR } },
    { "ftnnauc"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNAUC } },
    { "ftnnchi"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNCHI } },
    { "ftnnchosung"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNCHOSUNG } },
    { "ftnncnum"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNCNUM } },
    { "ftnndbar"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNDBAR } },
    { "ftnndbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNDBNUM } },
    { "ftnndbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNDBNUMD } },
    { "ftnndbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNDBNUMK } },
    { "ftnndbnumt"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNDBNUMT } },
    { "ftnnganada"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNGANADA } },
    { "ftnngbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNGBNUM } },
    { "ftnngbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNGBNUMD } },
    { "ftnngbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNGBNUMK } },
    { "ftnngbnuml"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNGBNUML } },
    { "ftnnrlc"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNRLC } },
    { "ftnnruc"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNRUC } },
    { "ftnnzodiac"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNZODIAC } },
    { "ftnnzodiacd"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNZODIACD } },
    { "ftnnzodiacl"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNNZODIACL } },
    { "ftnrestart"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNRESTART } },
    { "ftnrstcont"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNRSTCONT } },
    { "ftnrstpg"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNRSTPG } },
    { "ftnsep"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FTNSEP } },
    { "ftnsepc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FTNSEPC } },
    { "ftnstart"_ostr, { RTFControlType::VALUE, RTFKeyword::FTNSTART, 1 } },
    { "ftntj"_ostr, { RTFControlType::FLAG, RTFKeyword::FTNTJ } },
    { "fttruetype"_ostr, { RTFControlType::FLAG, RTFKeyword::FTTRUETYPE } },
    { "fvaliddos"_ostr, { RTFControlType::FLAG, RTFKeyword::FVALIDDOS } },
    { "fvalidhpfs"_ostr, { RTFControlType::FLAG, RTFKeyword::FVALIDHPFS } },
    { "fvalidmac"_ostr, { RTFControlType::FLAG, RTFKeyword::FVALIDMAC } },
    { "fvalidntfs"_ostr, { RTFControlType::FLAG, RTFKeyword::FVALIDNTFS } },
    { "g"_ostr, { RTFControlType::DESTINATION, RTFKeyword::G } },
    { "gcw"_ostr, { RTFControlType::VALUE, RTFKeyword::GCW } },
    { "generator"_ostr, { RTFControlType::DESTINATION, RTFKeyword::GENERATOR } },
    { "green"_ostr, { RTFControlType::VALUE, RTFKeyword::GREEN } },
    { "grfdocevents"_ostr, { RTFControlType::VALUE, RTFKeyword::GRFDOCEVENTS } },
    { "gridtbl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::GRIDTBL } },
    { "gutter"_ostr, { RTFControlType::VALUE, RTFKeyword::GUTTER } },
    { "gutterprl"_ostr, { RTFControlType::FLAG, RTFKeyword::GUTTERPRL } },
    { "guttersxn"_ostr, { RTFControlType::VALUE, RTFKeyword::GUTTERSXN } },
    { "header"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HEADER } },
    { "headerf"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HEADERF } },
    { "headerl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HEADERL } },
    { "headerr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HEADERR } },
    { "headery"_ostr, { RTFControlType::VALUE, RTFKeyword::HEADERY, 720 } },
    { "hich"_ostr, { RTFControlType::FLAG, RTFKeyword::HICH } },
    { "highlight"_ostr, { RTFControlType::VALUE, RTFKeyword::HIGHLIGHT } },
    { "hl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HL } },
    { "hlfr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HLFR } },
    { "hlinkbase"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HLINKBASE } },
    { "hlloc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HLLOC } },
    { "hlsrc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HLSRC } },
    { "horzdoc"_ostr, { RTFControlType::FLAG, RTFKeyword::HORZDOC } },
    { "horzsect"_ostr, { RTFControlType::FLAG, RTFKeyword::HORZSECT } },
    { "horzvert"_ostr, { RTFControlType::VALUE, RTFKeyword::HORZVERT } },
    { "hr"_ostr, { RTFControlType::VALUE, RTFKeyword::HR } },
    { "hres"_ostr, { RTFControlType::VALUE, RTFKeyword::HRES } },
    { "hrule"_ostr, { RTFControlType::FLAG, RTFKeyword::HRULE } },
    { "hsv"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HSV } },
    { "htmautsp"_ostr, { RTFControlType::FLAG, RTFKeyword::HTMAUTSP } },
    { "htmlbase"_ostr, { RTFControlType::FLAG, RTFKeyword::HTMLBASE } },
    { "htmlrtf"_ostr, { RTFControlType::TOGGLE, RTFKeyword::HTMLRTF, 1 } },
    { "htmltag"_ostr, { RTFControlType::DESTINATION, RTFKeyword::HTMLTAG } },
    { "hwelev"_ostr, { RTFControlType::FLAG, RTFKeyword::HWELEV } },
    { "hyphauto"_ostr, { RTFControlType::TOGGLE, RTFKeyword::HYPHAUTO, 1 } },
    { "hyphcaps"_ostr, { RTFControlType::TOGGLE, RTFKeyword::HYPHCAPS, 1 } },
    { "hyphconsec"_ostr, { RTFControlType::VALUE, RTFKeyword::HYPHCONSEC } },
    { "hyphhotz"_ostr, { RTFControlType::VALUE, RTFKeyword::HYPHHOTZ } },
    { "hyphpar"_ostr, { RTFControlType::TOGGLE, RTFKeyword::HYPHPAR, 1 } },
    { "i"_ostr, { RTFControlType::TOGGLE, RTFKeyword::I, 1 } },
    { "id"_ostr, { RTFControlType::VALUE, RTFKeyword::ID } },
    { "ignoremixedcontent"_ostr, { RTFControlType::VALUE, RTFKeyword::IGNOREMIXEDCONTENT } },
    { "ilfomacatclnup"_ostr, { RTFControlType::VALUE, RTFKeyword::ILFOMACATCLNUP } },
    { "ilvl"_ostr, { RTFControlType::VALUE, RTFKeyword::ILVL } },
    { "impr"_ostr, { RTFControlType::TOGGLE, RTFKeyword::IMPR, 1 } },
    { "indmirror"_ostr, { RTFControlType::FLAG, RTFKeyword::INDMIRROR } },
    { "indrlsweleven"_ostr, { RTFControlType::FLAG, RTFKeyword::INDRLSWELEVEN } },
    { "info"_ostr, { RTFControlType::DESTINATION, RTFKeyword::INFO } },
    { "insrsid"_ostr, { RTFControlType::VALUE, RTFKeyword::INSRSID } },
    { "intbl"_ostr, { RTFControlType::FLAG, RTFKeyword::INTBL } },
    { "ipgp"_ostr, { RTFControlType::VALUE, RTFKeyword::IPGP } },
    { "irowband"_ostr, { RTFControlType::VALUE, RTFKeyword::IROWBAND } },
    { "irow"_ostr, { RTFControlType::VALUE, RTFKeyword::IROW } },
    { "itap"_ostr, { RTFControlType::VALUE, RTFKeyword::ITAP, 1 } },
    { "ixe"_ostr, { RTFControlType::FLAG, RTFKeyword::IXE } },
    { "jcompress"_ostr, { RTFControlType::FLAG, RTFKeyword::JCOMPRESS } },
    { "jexpand"_ostr, { RTFControlType::FLAG, RTFKeyword::JEXPAND } },
    { "jis"_ostr, { RTFControlType::FLAG, RTFKeyword::JIS } },
    { "jpegblip"_ostr, { RTFControlType::FLAG, RTFKeyword::JPEGBLIP } },
    { "jsksu"_ostr, { RTFControlType::FLAG, RTFKeyword::JSKSU } },
    { "keep"_ostr, { RTFControlType::FLAG, RTFKeyword::KEEP } },
    { "keepn"_ostr, { RTFControlType::FLAG, RTFKeyword::KEEPN } },
    { "kerning"_ostr, { RTFControlType::VALUE, RTFKeyword::KERNING } },
    { "keycode"_ostr, { RTFControlType::DESTINATION, RTFKeyword::KEYCODE } },
    { "keywords"_ostr, { RTFControlType::DESTINATION, RTFKeyword::KEYWORDS } },
    { "krnprsnet"_ostr, { RTFControlType::FLAG, RTFKeyword::KRNPRSNET } },
    { "ksulang"_ostr, { RTFControlType::VALUE, RTFKeyword::KSULANG } },
    { "jclisttab"_ostr, { RTFControlType::FLAG, RTFKeyword::JCLISTTAB } },
    { "landscape"_ostr, { RTFControlType::FLAG, RTFKeyword::LANDSCAPE } },
    { "lang"_ostr, { RTFControlType::VALUE, RTFKeyword::LANG } },
    { "langfe"_ostr, { RTFControlType::VALUE, RTFKeyword::LANGFE } },
    { "langfenp"_ostr, { RTFControlType::VALUE, RTFKeyword::LANGFENP } },
    { "langnp"_ostr, { RTFControlType::VALUE, RTFKeyword::LANGNP } },
    { "lastrow"_ostr, { RTFControlType::FLAG, RTFKeyword::LASTROW } },
    { "latentstyles"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LATENTSTYLES } },
    { "lbr"_ostr, { RTFControlType::VALUE, RTFKeyword::LBR } },
    { "lchars"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LCHARS } },
    { "ldblquote"_ostr, { RTFControlType::SYMBOL, RTFKeyword::LDBLQUOTE } },
    { "level"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVEL } },
    { "levelfollow"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELFOLLOW } },
    { "levelindent"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELINDENT } },
    { "leveljc"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELJC } },
    { "leveljcn"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELJCN } },
    { "levellegal"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELLEGAL, 1 } },
    { "levelnfc"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELNFC } },
    { "levelnfcn"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELNFCN } },
    { "levelnorestart"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELNORESTART } },
    { "levelnumbers"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LEVELNUMBERS } },
    { "levelold"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELOLD } },
    { "levelpicture"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELPICTURE } },
    { "levelpicturenosize"_ostr, { RTFControlType::FLAG, RTFKeyword::LEVELPICTURENOSIZE } },
    { "levelprev"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELPREV } },
    { "levelprevspace"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELPREVSPACE } },
    { "levelspace"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELSPACE } },
    { "levelstartat"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELSTARTAT } },
    { "leveltemplateid"_ostr, { RTFControlType::VALUE, RTFKeyword::LEVELTEMPLATEID } },
    { "leveltext"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LEVELTEXT } },
    { "lfolevel"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LFOLEVEL } },
    { "li"_ostr, { RTFControlType::VALUE, RTFKeyword::LI } },
    { "line"_ostr, { RTFControlType::SYMBOL, RTFKeyword::LINE } },
    { "linebetcol"_ostr, { RTFControlType::FLAG, RTFKeyword::LINEBETCOL } },
    { "linecont"_ostr, { RTFControlType::FLAG, RTFKeyword::LINECONT } },
    { "linemod"_ostr, { RTFControlType::VALUE, RTFKeyword::LINEMOD, 1 } },
    { "lineppage"_ostr, { RTFControlType::FLAG, RTFKeyword::LINEPPAGE } },
    { "linerestart"_ostr, { RTFControlType::FLAG, RTFKeyword::LINERESTART } },
    { "linestart"_ostr, { RTFControlType::VALUE, RTFKeyword::LINESTART, 1 } },
    { "linestarts"_ostr, { RTFControlType::VALUE, RTFKeyword::LINESTARTS, 1 } },
    { "linex"_ostr, { RTFControlType::VALUE, RTFKeyword::LINEX, 360 } },
    { "linkself"_ostr, { RTFControlType::FLAG, RTFKeyword::LINKSELF } },
    { "linkstyles"_ostr, { RTFControlType::FLAG, RTFKeyword::LINKSTYLES } },
    { "linkval"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LINKVAL } },
    { "lin"_ostr, { RTFControlType::VALUE, RTFKeyword::LIN } },
    { "lisa"_ostr, { RTFControlType::VALUE, RTFKeyword::LISA } },
    { "lisb"_ostr, { RTFControlType::VALUE, RTFKeyword::LISB } },
    { "list"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LIST } },
    { "listhybrid"_ostr, { RTFControlType::FLAG, RTFKeyword::LISTHYBRID } },
    { "listid"_ostr, { RTFControlType::VALUE, RTFKeyword::LISTID } },
    { "listlevel"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LISTLEVEL } },
    { "listname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LISTNAME } },
    { "listoverride"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LISTOVERRIDE } },
    { "listoverridecount"_ostr, { RTFControlType::VALUE, RTFKeyword::LISTOVERRIDECOUNT } },
    { "listoverrideformat"_ostr, { RTFControlType::VALUE, RTFKeyword::LISTOVERRIDEFORMAT } },
    { "listoverridestartat"_ostr, { RTFControlType::FLAG, RTFKeyword::LISTOVERRIDESTARTAT } },
    { "listoverridetable"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LISTOVERRIDETABLE } },
    { "listpicture"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LISTPICTURE } },
    { "listrestarthdn"_ostr, { RTFControlType::VALUE, RTFKeyword::LISTRESTARTHDN } },
    { "listsimple"_ostr, { RTFControlType::VALUE, RTFKeyword::LISTSIMPLE } },
    { "liststyleid"_ostr, { RTFControlType::VALUE, RTFKeyword::LISTSTYLEID } },
    { "liststylename"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LISTSTYLENAME } },
    { "listtable"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LISTTABLE } },
    { "listtemplateid"_ostr, { RTFControlType::VALUE, RTFKeyword::LISTTEMPLATEID } },
    { "listtext"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LISTTEXT } },
    { "lnbrkrule"_ostr, { RTFControlType::FLAG, RTFKeyword::LNBRKRULE } },
    { "lndscpsxn"_ostr, { RTFControlType::FLAG, RTFKeyword::LNDSCPSXN } },
    { "lnongrid"_ostr, { RTFControlType::FLAG, RTFKeyword::LNONGRID } },
    { "loch"_ostr, { RTFControlType::FLAG, RTFKeyword::LOCH } },
    { "lquote"_ostr, { RTFControlType::SYMBOL, RTFKeyword::LQUOTE } },
    { "ls"_ostr, { RTFControlType::VALUE, RTFKeyword::LS } },
    { "lsdlocked"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDLOCKED } },
    { "lsdlockeddef"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDLOCKEDDEF } },
    { "lsdlockedexcept"_ostr, { RTFControlType::DESTINATION, RTFKeyword::LSDLOCKEDEXCEPT } },
    { "lsdpriority"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDPRIORITY } },
    { "lsdprioritydef"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDPRIORITYDEF } },
    { "lsdqformat"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDQFORMAT } },
    { "lsdqformatdef"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDQFORMATDEF } },
    { "lsdsemihidden"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDSEMIHIDDEN } },
    { "lsdsemihiddendef"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDSEMIHIDDENDEF } },
    { "lsdstimax"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDSTIMAX } },
    { "lsdunhideused"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDUNHIDEUSED } },
    { "lsdunhideuseddef"_ostr, { RTFControlType::VALUE, RTFKeyword::LSDUNHIDEUSEDDEF } },
    { "ltrch"_ostr, { RTFControlType::FLAG, RTFKeyword::LTRCH } },
    { "ltrdoc"_ostr, { RTFControlType::FLAG, RTFKeyword::LTRDOC } },
    { "ltrmark"_ostr, { RTFControlType::SYMBOL, RTFKeyword::LTRMARK } },
    { "ltrpar"_ostr, { RTFControlType::FLAG, RTFKeyword::LTRPAR } },
    { "ltrrow"_ostr, { RTFControlType::FLAG, RTFKeyword::LTRROW } },
    { "ltrsect"_ostr, { RTFControlType::FLAG, RTFKeyword::LTRSECT } },
    { "lvltentative"_ostr, { RTFControlType::FLAG, RTFKeyword::LVLTENTATIVE } },
    { "lytcalctblwd"_ostr, { RTFControlType::FLAG, RTFKeyword::LYTCALCTBLWD } },
    { "lytexcttp"_ostr, { RTFControlType::FLAG, RTFKeyword::LYTEXCTTP } },
    { "lytprtmet"_ostr, { RTFControlType::FLAG, RTFKeyword::LYTPRTMET } },
    { "lyttblrtgr"_ostr, { RTFControlType::FLAG, RTFKeyword::LYTTBLRTGR } },
    { "mac"_ostr, { RTFControlType::FLAG, RTFKeyword::MAC } },
    { "macc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MACC } },
    { "maccPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MACCPR } },
    { "macpict"_ostr, { RTFControlType::FLAG, RTFKeyword::MACPICT } },
    { "mailmerge"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MAILMERGE } },
    { "makebackup"_ostr, { RTFControlType::FLAG, RTFKeyword::MAKEBACKUP } },
    { "maln"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MALN } },
    { "malnScr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MALNSCR } },
    { "manager"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MANAGER } },
    { "margb"_ostr, { RTFControlType::VALUE, RTFKeyword::MARGB, 1440 } },
    { "margbsxn"_ostr, { RTFControlType::VALUE, RTFKeyword::MARGBSXN } },
    { "margl"_ostr, { RTFControlType::VALUE, RTFKeyword::MARGL, 1800 } },
    { "marglsxn"_ostr, { RTFControlType::VALUE, RTFKeyword::MARGLSXN } },
    { "margmirror"_ostr, { RTFControlType::FLAG, RTFKeyword::MARGMIRROR } },
    { "margmirsxn"_ostr, { RTFControlType::FLAG, RTFKeyword::MARGMIRSXN } },
    { "margPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MARGPR } },
    { "margr"_ostr, { RTFControlType::VALUE, RTFKeyword::MARGR, 1800 } },
    { "margrsxn"_ostr, { RTFControlType::VALUE, RTFKeyword::MARGRSXN } },
    { "margSz"_ostr, { RTFControlType::VALUE, RTFKeyword::MARGSZ } },
    { "margt"_ostr, { RTFControlType::VALUE, RTFKeyword::MARGT, 1440 } },
    { "margtsxn"_ostr, { RTFControlType::VALUE, RTFKeyword::MARGTSXN } },
    { "mbar"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MBAR } },
    { "mbarPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MBARPR } },
    { "mbaseJc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MBASEJC } },
    { "mbegChr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MBEGCHR } },
    { "mborderBox"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MBORDERBOX } },
    { "mborderBoxPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MBORDERBOXPR } },
    { "mbox"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MBOX } },
    { "mboxPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MBOXPR } },
    { "mbrk"_ostr, { RTFControlType::VALUE, RTFKeyword::MBRK } },
    { "mbrkBin"_ostr, { RTFControlType::VALUE, RTFKeyword::MBRKBIN } },
    { "mbrkBinSub"_ostr, { RTFControlType::VALUE, RTFKeyword::MBRKBINSUB } },
    { "mcGp"_ostr, { RTFControlType::VALUE, RTFKeyword::MCGP } },
    { "mcGpRule"_ostr, { RTFControlType::VALUE, RTFKeyword::MCGPRULE } },
    { "mchr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MCHR } },
    { "mcount"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MCOUNT } },
    { "mcSp"_ostr, { RTFControlType::VALUE, RTFKeyword::MCSP } },
    { "mctrlPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MCTRLPR } },
    { "md"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MD } },
    { "mdefJc"_ostr, { RTFControlType::VALUE, RTFKeyword::MDEFJC } },
    { "mdeg"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MDEG } },
    { "mdegHide"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MDEGHIDE } },
    { "mden"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MDEN } },
    { "mdiff"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MDIFF } },
    { "mdiffSty"_ostr, { RTFControlType::VALUE, RTFKeyword::MDIFFSTY } },
    { "mdispdef"_ostr, { RTFControlType::VALUE, RTFKeyword::MDISPDEF, 1 } },
    { "mdPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MDPR } },
    { "me"_ostr, { RTFControlType::DESTINATION, RTFKeyword::ME } },
    { "mendChr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MENDCHR } },
    { "meqArr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MEQARR } },
    { "meqArrPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MEQARRPR } },
    { "mf"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MF } },
    { "mfName"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MFNAME } },
    { "mfPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MFPR } },
    { "mfunc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MFUNC } },
    { "mfuncPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MFUNCPR } },
    { "mgroupChr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MGROUPCHR } },
    { "mgroupChrPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MGROUPCHRPR } },
    { "mgrow"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MGROW } },
    { "mhideBot"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MHIDEBOT } },
    { "mhideLeft"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MHIDELEFT } },
    { "mhideRight"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MHIDERIGHT } },
    { "mhideTop"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MHIDETOP } },
    { "mhtmltag"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MHTMLTAG } },
    { "min"_ostr, { RTFControlType::VALUE, RTFKeyword::MIN } },
    { "minterSp"_ostr, { RTFControlType::VALUE, RTFKeyword::MINTERSP } },
    { "mintLim"_ostr, { RTFControlType::VALUE, RTFKeyword::MINTLIM } },
    { "mintraSp"_ostr, { RTFControlType::VALUE, RTFKeyword::MINTRASP } },
    { "mjc"_ostr, { RTFControlType::VALUE, RTFKeyword::MJC } },
    { "mlim"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIM } },
    { "mlimloc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIMLOC } },
    { "mlimLoc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIMLOC } },
    { "mlimlow"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIMLOW } },
    { "mlimLow"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIMLOW } },
    { "mlimlowPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIMLOWPR } },
    { "mlimLowPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIMLOWPR } },
    { "mlimupp"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIMUPP } },
    { "mlimUpp"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIMUPP } },
    { "mlimuppPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIMUPPPR } },
    { "mlimUppPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MLIMUPPPR } },
    { "mlit"_ostr, { RTFControlType::FLAG, RTFKeyword::MLIT } },
    { "mlMargin"_ostr, { RTFControlType::VALUE, RTFKeyword::MLMARGIN } },
    { "mm"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MM } },
    { "mmaddfieldname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMADDFIELDNAME } },
    { "mmath"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMATH } },
    { "mmathFont"_ostr, { RTFControlType::VALUE, RTFKeyword::MMATHFONT } },
    { "mmathPict"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMATHPICT } },
    { "mmathPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMATHPR } },
    { "mmattach"_ostr, { RTFControlType::FLAG, RTFKeyword::MMATTACH } },
    { "mmaxdist"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMAXDIST } },
    { "mmblanklines"_ostr, { RTFControlType::FLAG, RTFKeyword::MMBLANKLINES } },
    { "mmc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMC } },
    { "mmcJc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMCJC } },
    { "mmconnectstr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMCONNECTSTR } },
    { "mmconnectstrdata"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMCONNECTSTRDATA } },
    { "mmcPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMCPR } },
    { "mmcs"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMCS } },
    { "mmdatasource"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMDATASOURCE } },
    { "mmdatatypeaccess"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDATATYPEACCESS } },
    { "mmdatatypeexcel"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDATATYPEEXCEL } },
    { "mmdatatypefile"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDATATYPEFILE } },
    { "mmdatatypeodbc"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDATATYPEODBC } },
    { "mmdatatypeodso"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDATATYPEODSO } },
    { "mmdatatypeqt"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDATATYPEQT } },
    { "mmdefaultsql"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDEFAULTSQL } },
    { "mmdestemail"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDESTEMAIL } },
    { "mmdestfax"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDESTFAX } },
    { "mmdestnewdoc"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDESTNEWDOC } },
    { "mmdestprinter"_ostr, { RTFControlType::FLAG, RTFKeyword::MMDESTPRINTER } },
    { "mmerrors"_ostr, { RTFControlType::VALUE, RTFKeyword::MMERRORS } },
    { "mmfttypeaddress"_ostr, { RTFControlType::FLAG, RTFKeyword::MMFTTYPEADDRESS } },
    { "mmfttypebarcode"_ostr, { RTFControlType::FLAG, RTFKeyword::MMFTTYPEBARCODE } },
    { "mmfttypedbcolumn"_ostr, { RTFControlType::FLAG, RTFKeyword::MMFTTYPEDBCOLUMN } },
    { "mmfttypemapped"_ostr, { RTFControlType::FLAG, RTFKeyword::MMFTTYPEMAPPED } },
    { "mmfttypenull"_ostr, { RTFControlType::FLAG, RTFKeyword::MMFTTYPENULL } },
    { "mmfttypesalutation"_ostr, { RTFControlType::FLAG, RTFKeyword::MMFTTYPESALUTATION } },
    { "mmheadersource"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMHEADERSOURCE } },
    { "mmjdsotype"_ostr, { RTFControlType::VALUE, RTFKeyword::MMJDSOTYPE } },
    { "mmlinktoquery"_ostr, { RTFControlType::FLAG, RTFKeyword::MMLINKTOQUERY } },
    { "mmmailsubject"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMMAILSUBJECT } },
    { "mmmaintypecatalog"_ostr, { RTFControlType::FLAG, RTFKeyword::MMMAINTYPECATALOG } },
    { "mmmaintypeemail"_ostr, { RTFControlType::FLAG, RTFKeyword::MMMAINTYPEEMAIL } },
    { "mmmaintypeenvelopes"_ostr, { RTFControlType::FLAG, RTFKeyword::MMMAINTYPEENVELOPES } },
    { "mmmaintypefax"_ostr, { RTFControlType::FLAG, RTFKeyword::MMMAINTYPEFAX } },
    { "mmmaintypelabels"_ostr, { RTFControlType::FLAG, RTFKeyword::MMMAINTYPELABELS } },
    { "mmmaintypeletters"_ostr, { RTFControlType::FLAG, RTFKeyword::MMMAINTYPELETTERS } },
    { "mmodso"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSO } },
    { "mmodsoactive"_ostr, { RTFControlType::VALUE, RTFKeyword::MMODSOACTIVE } },
    { "mmodsocoldelim"_ostr, { RTFControlType::VALUE, RTFKeyword::MMODSOCOLDELIM } },
    { "mmodsocolumn"_ostr, { RTFControlType::VALUE, RTFKeyword::MMODSOCOLUMN } },
    { "mmodsodynaddr"_ostr, { RTFControlType::VALUE, RTFKeyword::MMODSODYNADDR } },
    { "mmodsofhdr"_ostr, { RTFControlType::VALUE, RTFKeyword::MMODSOFHDR } },
    { "mmodsofilter"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSOFILTER } },
    { "mmodsofldmpdata"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSOFLDMPDATA } },
    { "mmodsofmcolumn"_ostr, { RTFControlType::VALUE, RTFKeyword::MMODSOFMCOLUMN } },
    { "mmodsohash"_ostr, { RTFControlType::VALUE, RTFKeyword::MMODSOHASH } },
    { "mmodsolid"_ostr, { RTFControlType::VALUE, RTFKeyword::MMODSOLID } },
    { "mmodsomappedname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSOMAPPEDNAME } },
    { "mmodsoname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSONAME } },
    { "mmodsorecipdata"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSORECIPDATA } },
    { "mmodsosort"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSOSORT } },
    { "mmodsosrc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSOSRC } },
    { "mmodsotable"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSOTABLE } },
    { "mmodsoudl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSOUDL } },
    { "mmodsoudldata"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSOUDLDATA } },
    { "mmodsouniquetag"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMODSOUNIQUETAG } },
    { "mmPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMPR } },
    { "mmquery"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMQUERY } },
    { "mmr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MMR } },
    { "mmreccur"_ostr, { RTFControlType::VALUE, RTFKeyword::MMRECCUR } },
    { "mmshowdata"_ostr, { RTFControlType::FLAG, RTFKeyword::MMSHOWDATA } },
    { "mnary"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MNARY } },
    { "mnaryLim"_ostr, { RTFControlType::VALUE, RTFKeyword::MNARYLIM } },
    { "mnaryPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MNARYPR } },
    { "mnoBreak"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MNOBREAK } },
    { "mnor"_ostr, { RTFControlType::FLAG, RTFKeyword::MNOR } },
    { "mnum"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MNUM } },
    { "mo"_ostr, { RTFControlType::VALUE, RTFKeyword::MO } },
    { "mobjDist"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MOBJDIST } },
    { "moMath"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MOMATH } },
    { "moMathPara"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MOMATHPARA } },
    { "moMathParaPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MOMATHPARAPR } },
    { "mopEmu"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MOPEMU } },
    { "mphant"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MPHANT } },
    { "mphantPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MPHANTPR } },
    { "mplcHide"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MPLCHIDE } },
    { "mpos"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MPOS } },
    { "mpostSp"_ostr, { RTFControlType::VALUE, RTFKeyword::MPOSTSP } },
    { "mpreSp"_ostr, { RTFControlType::VALUE, RTFKeyword::MPRESP } },
    { "mr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MR } },
    { "mrad"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MRAD } },
    { "mradPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MRADPR } },
    { "mrMargin"_ostr, { RTFControlType::VALUE, RTFKeyword::MRMARGIN } },
    { "mrPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MRPR } },
    { "mrSp"_ostr, { RTFControlType::VALUE, RTFKeyword::MRSP } },
    { "mrSpRule"_ostr, { RTFControlType::VALUE, RTFKeyword::MRSPRULE } },
    { "mscr"_ostr, { RTFControlType::VALUE, RTFKeyword::MSCR } },
    { "msepChr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSEPCHR } },
    { "mshow"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSHOW } },
    { "mshp"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSHP } },
    { "msmallFrac"_ostr, { RTFControlType::VALUE, RTFKeyword::MSMALLFRAC } },
    { "msmcap"_ostr, { RTFControlType::FLAG, RTFKeyword::MSMCAP } },
    { "msPre"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSPRE } },
    { "msPrePr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSPREPR } },
    { "msSub"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSSUB } },
    { "msSubPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSSUBPR } },
    { "msSubSup"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSSUBSUP } },
    { "msSubSupPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSSUBSUPPR } },
    { "msSup"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSSUP } },
    { "msSupPr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSSUPPR } },
    { "mstrikeBLTR"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSTRIKEBLTR } },
    { "mstrikeH"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSTRIKEH } },
    { "mstrikeTLBR"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSTRIKETLBR } },
    { "mstrikeV"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSTRIKEV } },
    { "msty"_ostr, { RTFControlType::VALUE, RTFKeyword::MSTY } },
    { "msub"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSUB } },
    { "msubHide"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSUBHIDE } },
    { "msup"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSUP } },
    { "msupHide"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MSUPHIDE } },
    { "mtransp"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MTRANSP } },
    { "mtype"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MTYPE } },
    { "muser"_ostr, { RTFControlType::FLAG, RTFKeyword::MUSER } },
    { "mvauth"_ostr, { RTFControlType::VALUE, RTFKeyword::MVAUTH } },
    { "mvdate"_ostr, { RTFControlType::VALUE, RTFKeyword::MVDATE } },
    { "mvertJc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MVERTJC } },
    { "mvf"_ostr, { RTFControlType::FLAG, RTFKeyword::MVF } },
    { "mvfmf"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MVFMF } },
    { "mvfml"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MVFML } },
    { "mvt"_ostr, { RTFControlType::FLAG, RTFKeyword::MVT } },
    { "mvtof"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MVTOF } },
    { "mvtol"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MVTOL } },
    { "mwrapIndent"_ostr, { RTFControlType::VALUE, RTFKeyword::MWRAPINDENT, 1440 } },
    { "mwrapRight"_ostr, { RTFControlType::VALUE, RTFKeyword::MWRAPRIGHT } },
    { "mzeroAsc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MZEROASC } },
    { "mzeroDesc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MZERODESC } },
    { "mzeroWid"_ostr, { RTFControlType::DESTINATION, RTFKeyword::MZEROWID } },
    { "nestcell"_ostr, { RTFControlType::SYMBOL, RTFKeyword::NESTCELL } },
    { "nestrow"_ostr, { RTFControlType::SYMBOL, RTFKeyword::NESTROW } },
    { "nesttableprops"_ostr, { RTFControlType::DESTINATION, RTFKeyword::NESTTABLEPROPS } },
    { "newtblstyruls"_ostr, { RTFControlType::FLAG, RTFKeyword::NEWTBLSTYRULS } },
    { "nextfile"_ostr, { RTFControlType::DESTINATION, RTFKeyword::NEXTFILE } },
    { "noafcnsttbl"_ostr, { RTFControlType::FLAG, RTFKeyword::NOAFCNSTTBL } },
    { "nobrkwrptbl"_ostr, { RTFControlType::FLAG, RTFKeyword::NOBRKWRPTBL } },
    { "nocolbal"_ostr, { RTFControlType::FLAG, RTFKeyword::NOCOLBAL } },
    { "nocompatoptions"_ostr, { RTFControlType::FLAG, RTFKeyword::NOCOMPATOPTIONS } },
    { "nocwrap"_ostr, { RTFControlType::FLAG, RTFKeyword::NOCWRAP } },
    { "nocxsptable"_ostr, { RTFControlType::FLAG, RTFKeyword::NOCXSPTABLE } },
    { "noextrasprl"_ostr, { RTFControlType::FLAG, RTFKeyword::NOEXTRASPRL } },
    { "nofchars"_ostr, { RTFControlType::VALUE, RTFKeyword::NOFCHARS } },
    { "nofcharsws"_ostr, { RTFControlType::VALUE, RTFKeyword::NOFCHARSWS } },
    { "nofeaturethrottle"_ostr, { RTFControlType::FLAG, RTFKeyword::NOFEATURETHROTTLE } },
    { "nofpages"_ostr, { RTFControlType::VALUE, RTFKeyword::NOFPAGES } },
    { "nofwords"_ostr, { RTFControlType::VALUE, RTFKeyword::NOFWORDS } },
    { "nogrowautofit"_ostr, { RTFControlType::FLAG, RTFKeyword::NOGROWAUTOFIT } },
    { "noindnmbrts"_ostr, { RTFControlType::FLAG, RTFKeyword::NOINDNMBRTS } },
    { "nojkernpunct"_ostr, { RTFControlType::FLAG, RTFKeyword::NOJKERNPUNCT } },
    { "nolead"_ostr, { RTFControlType::FLAG, RTFKeyword::NOLEAD } },
    { "noline"_ostr, { RTFControlType::FLAG, RTFKeyword::NOLINE } },
    { "nolnhtadjtbl"_ostr, { RTFControlType::FLAG, RTFKeyword::NOLNHTADJTBL } },
    { "nonesttables"_ostr, { RTFControlType::DESTINATION, RTFKeyword::NONESTTABLES } },
    { "nonshppict"_ostr, { RTFControlType::FLAG, RTFKeyword::NONSHPPICT } },
    { "nooverflow"_ostr, { RTFControlType::FLAG, RTFKeyword::NOOVERFLOW } },
    { "noproof"_ostr, { RTFControlType::FLAG, RTFKeyword::NOPROOF } },
    { "noqfpromote"_ostr, { RTFControlType::FLAG, RTFKeyword::NOQFPROMOTE } },
    { "nosectexpand"_ostr, { RTFControlType::FLAG, RTFKeyword::NOSECTEXPAND } },
    { "nosnaplinegrid"_ostr, { RTFControlType::FLAG, RTFKeyword::NOSNAPLINEGRID } },
    { "nospaceforul"_ostr, { RTFControlType::FLAG, RTFKeyword::NOSPACEFORUL } },
    { "nosupersub"_ostr, { RTFControlType::FLAG, RTFKeyword::NOSUPERSUB } },
    { "notabind"_ostr, { RTFControlType::FLAG, RTFKeyword::NOTABIND } },
    { "notbrkcnstfrctbl"_ostr, { RTFControlType::FLAG, RTFKeyword::NOTBRKCNSTFRCTBL } },
    { "notcvasp"_ostr, { RTFControlType::FLAG, RTFKeyword::NOTCVASP } },
    { "notvatxbx"_ostr, { RTFControlType::FLAG, RTFKeyword::NOTVATXBX } },
    { "nouicompat"_ostr, { RTFControlType::FLAG, RTFKeyword::NOUICOMPAT } },
    { "noultrlspc"_ostr, { RTFControlType::FLAG, RTFKeyword::NOULTRLSPC } },
    { "nowidctlpar"_ostr, { RTFControlType::FLAG, RTFKeyword::NOWIDCTLPAR } },
    { "nowrap"_ostr, { RTFControlType::FLAG, RTFKeyword::NOWRAP } },
    { "nowwrap"_ostr, { RTFControlType::FLAG, RTFKeyword::NOWWRAP } },
    { "noxlattoyen"_ostr, { RTFControlType::FLAG, RTFKeyword::NOXLATTOYEN } },
    { "objalias"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OBJALIAS } },
    { "objalign"_ostr, { RTFControlType::VALUE, RTFKeyword::OBJALIGN } },
    { "objattph"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJATTPH } },
    { "objautlink"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJAUTLINK } },
    { "objclass"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OBJCLASS } },
    { "objcropb"_ostr, { RTFControlType::VALUE, RTFKeyword::OBJCROPB } },
    { "objcropl"_ostr, { RTFControlType::VALUE, RTFKeyword::OBJCROPL } },
    { "objcropr"_ostr, { RTFControlType::VALUE, RTFKeyword::OBJCROPR } },
    { "objcropt"_ostr, { RTFControlType::VALUE, RTFKeyword::OBJCROPT } },
    { "objdata"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OBJDATA } },
    { "object"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OBJECT } },
    { "objemb"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJEMB } },
    { "objh"_ostr, { RTFControlType::VALUE, RTFKeyword::OBJH } },
    { "objhtml"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJHTML } },
    { "objicemb"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJICEMB } },
    { "objlink"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJLINK } },
    { "objlock"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJLOCK } },
    { "objname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OBJNAME } },
    { "objocx"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJOCX } },
    { "objpub"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJPUB } },
    { "objscalex"_ostr, { RTFControlType::VALUE, RTFKeyword::OBJSCALEX } },
    { "objscaley"_ostr, { RTFControlType::VALUE, RTFKeyword::OBJSCALEY } },
    { "objsect"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OBJSECT } },
    { "objsetsize"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJSETSIZE } },
    { "objsub"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJSUB } },
    { "objtime"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OBJTIME } },
    { "objtransy"_ostr, { RTFControlType::VALUE, RTFKeyword::OBJTRANSY } },
    { "objupdate"_ostr, { RTFControlType::FLAG, RTFKeyword::OBJUPDATE } },
    { "objw"_ostr, { RTFControlType::VALUE, RTFKeyword::OBJW } },
    { "ogutter"_ostr, { RTFControlType::VALUE, RTFKeyword::OGUTTER } },
    { "oldas"_ostr, { RTFControlType::FLAG, RTFKeyword::OLDAS } },
    { "oldcprops"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OLDCPROPS } },
    { "oldlinewrap"_ostr, { RTFControlType::FLAG, RTFKeyword::OLDLINEWRAP } },
    { "oldpprops"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OLDPPROPS } },
    { "oldsprops"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OLDSPROPS } },
    { "oldtprops"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OLDTPROPS } },
    { "oleclsid"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OLECLSID } },
    { "operator"_ostr, { RTFControlType::DESTINATION, RTFKeyword::OPERATOR } },
    { "otblrul"_ostr, { RTFControlType::FLAG, RTFKeyword::OTBLRUL } },
    { "outl"_ostr, { RTFControlType::TOGGLE, RTFKeyword::OUTL, 1 } },
    { "outlinelevel"_ostr, { RTFControlType::VALUE, RTFKeyword::OUTLINELEVEL } },
    { "overlay"_ostr, { RTFControlType::FLAG, RTFKeyword::OVERLAY } },
    { "page"_ostr, { RTFControlType::SYMBOL, RTFKeyword::PAGE } },
    { "pagebb"_ostr, { RTFControlType::FLAG, RTFKeyword::PAGEBB } },
    { "panose"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PANOSE } },
    { "paperh"_ostr, { RTFControlType::VALUE, RTFKeyword::PAPERH, 15840 } },
    { "paperw"_ostr, { RTFControlType::VALUE, RTFKeyword::PAPERW, 12240 } },
    { "par"_ostr, { RTFControlType::SYMBOL, RTFKeyword::PAR } },
    { "pararsid"_ostr, { RTFControlType::VALUE, RTFKeyword::PARARSID } },
    { "pard"_ostr, { RTFControlType::FLAG, RTFKeyword::PARD } },
    { "password"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PASSWORD } },
    { "passwordhash"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PASSWORDHASH } },
    { "pc"_ostr, { RTFControlType::FLAG, RTFKeyword::PC } },
    { "pca"_ostr, { RTFControlType::FLAG, RTFKeyword::PCA } },
    { "pgbrdrb"_ostr, { RTFControlType::FLAG, RTFKeyword::PGBRDRB } },
    { "pgbrdrfoot"_ostr, { RTFControlType::FLAG, RTFKeyword::PGBRDRFOOT } },
    { "pgbrdrhead"_ostr, { RTFControlType::FLAG, RTFKeyword::PGBRDRHEAD } },
    { "pgbrdrl"_ostr, { RTFControlType::FLAG, RTFKeyword::PGBRDRL } },
    { "pgbrdropt"_ostr, { RTFControlType::VALUE, RTFKeyword::PGBRDROPT } },
    { "pgbrdrr"_ostr, { RTFControlType::FLAG, RTFKeyword::PGBRDRR } },
    { "pgbrdrsnap"_ostr, { RTFControlType::FLAG, RTFKeyword::PGBRDRSNAP } },
    { "pgbrdrt"_ostr, { RTFControlType::FLAG, RTFKeyword::PGBRDRT } },
    { "pghsxn"_ostr, { RTFControlType::VALUE, RTFKeyword::PGHSXN } },
    { "pgnbidia"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNBIDIA } },
    { "pgnbidib"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNBIDIB } },
    { "pgnchosung"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNCHOSUNG } },
    { "pgncnum"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNCNUM } },
    { "pgncont"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNCONT } },
    { "pgndbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNDBNUM } },
    { "pgndbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNDBNUMD } },
    { "pgndbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNDBNUMK } },
    { "pgndbnumt"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNDBNUMT } },
    { "pgndec"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNDEC } },
    { "pgndecd"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNDECD } },
    { "pgnganada"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNGANADA } },
    { "pgngbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNGBNUM } },
    { "pgngbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNGBNUMD } },
    { "pgngbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNGBNUMK } },
    { "pgngbnuml"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNGBNUML } },
    { "pgnhindia"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNHINDIA } },
    { "pgnhindib"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNHINDIB } },
    { "pgnhindic"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNHINDIC } },
    { "pgnhindid"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNHINDID } },
    { "pgnhn"_ostr, { RTFControlType::VALUE, RTFKeyword::PGNHN } },
    { "pgnhnsc"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNHNSC } },
    { "pgnhnsh"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNHNSH } },
    { "pgnhnsm"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNHNSM } },
    { "pgnhnsn"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNHNSN } },
    { "pgnhnsp"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNHNSP } },
    { "pgnid"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNID } },
    { "pgnlcltr"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNLCLTR } },
    { "pgnlcrm"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNLCRM } },
    { "pgnrestart"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNRESTART } },
    { "pgnstart"_ostr, { RTFControlType::VALUE, RTFKeyword::PGNSTART, 1 } },
    { "pgnstarts"_ostr, { RTFControlType::VALUE, RTFKeyword::PGNSTARTS, 1 } },
    { "pgnthaia"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNTHAIA } },
    { "pgnthaib"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNTHAIB } },
    { "pgnthaic"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNTHAIC } },
    { "pgnucltr"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNUCLTR } },
    { "pgnucrm"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNUCRM } },
    { "pgnvieta"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNVIETA } },
    { "pgnx"_ostr, { RTFControlType::VALUE, RTFKeyword::PGNX, 720 } },
    { "pgny"_ostr, { RTFControlType::VALUE, RTFKeyword::PGNY, 720 } },
    { "pgnzodiac"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNZODIAC } },
    { "pgnzodiacd"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNZODIACD } },
    { "pgnzodiacl"_ostr, { RTFControlType::FLAG, RTFKeyword::PGNZODIACL } },
    { "pgp"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PGP } },
    { "pgptbl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PGPTBL } },
    { "pgwsxn"_ostr, { RTFControlType::VALUE, RTFKeyword::PGWSXN } },
    { "phcol"_ostr, { RTFControlType::FLAG, RTFKeyword::PHCOL } },
    { "phmrg"_ostr, { RTFControlType::FLAG, RTFKeyword::PHMRG } },
    { "phpg"_ostr, { RTFControlType::FLAG, RTFKeyword::PHPG } },
    { "picbmp"_ostr, { RTFControlType::FLAG, RTFKeyword::PICBMP } },
    { "picbpp"_ostr, { RTFControlType::VALUE, RTFKeyword::PICBPP } },
    { "piccropb"_ostr, { RTFControlType::VALUE, RTFKeyword::PICCROPB } },
    { "piccropl"_ostr, { RTFControlType::VALUE, RTFKeyword::PICCROPL } },
    { "piccropr"_ostr, { RTFControlType::VALUE, RTFKeyword::PICCROPR } },
    { "piccropt"_ostr, { RTFControlType::VALUE, RTFKeyword::PICCROPT } },
    { "pich"_ostr, { RTFControlType::VALUE, RTFKeyword::PICH } },
    { "pichgoal"_ostr, { RTFControlType::VALUE, RTFKeyword::PICHGOAL } },
    { "pichGoal"_ostr, { RTFControlType::VALUE, RTFKeyword::PICHGOAL } },
    { "picprop"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PICPROP } },
    { "picscaled"_ostr, { RTFControlType::FLAG, RTFKeyword::PICSCALED } },
    { "picscalex"_ostr, { RTFControlType::VALUE, RTFKeyword::PICSCALEX, 100 } },
    { "picscaley"_ostr, { RTFControlType::VALUE, RTFKeyword::PICSCALEY, 100 } },
    { "pict"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PICT } },
    { "picw"_ostr, { RTFControlType::VALUE, RTFKeyword::PICW } },
    { "picwgoal"_ostr, { RTFControlType::VALUE, RTFKeyword::PICWGOAL } },
    { "picwGoal"_ostr, { RTFControlType::VALUE, RTFKeyword::PICWGOAL } },
    { "pindtabqc"_ostr, { RTFControlType::FLAG, RTFKeyword::PINDTABQC } },
    { "pindtabql"_ostr, { RTFControlType::FLAG, RTFKeyword::PINDTABQL } },
    { "pindtabqr"_ostr, { RTFControlType::FLAG, RTFKeyword::PINDTABQR } },
    { "plain"_ostr, { RTFControlType::FLAG, RTFKeyword::PLAIN } },
    { "pmartabqc"_ostr, { RTFControlType::FLAG, RTFKeyword::PMARTABQC } },
    { "pmartabql"_ostr, { RTFControlType::FLAG, RTFKeyword::PMARTABQL } },
    { "pmartabqr"_ostr, { RTFControlType::FLAG, RTFKeyword::PMARTABQR } },
    { "pmmetafile"_ostr, { RTFControlType::VALUE, RTFKeyword::PMMETAFILE } },
    { "pn"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PN } },
    { "pnacross"_ostr, { RTFControlType::FLAG, RTFKeyword::PNACROSS } },
    { "pnaiu"_ostr, { RTFControlType::FLAG, RTFKeyword::PNAIU } },
    { "pnaiud"_ostr, { RTFControlType::FLAG, RTFKeyword::PNAIUD } },
    { "pnaiueo"_ostr, { RTFControlType::FLAG, RTFKeyword::PNAIUEO } },
    { "pnaiueod"_ostr, { RTFControlType::FLAG, RTFKeyword::PNAIUEOD } },
    { "pnb"_ostr, { RTFControlType::TOGGLE, RTFKeyword::PNB, 1 } },
    { "pnbidia"_ostr, { RTFControlType::FLAG, RTFKeyword::PNBIDIA } },
    { "pnbidib"_ostr, { RTFControlType::FLAG, RTFKeyword::PNBIDIB } },
    { "pncaps"_ostr, { RTFControlType::TOGGLE, RTFKeyword::PNCAPS, 1 } },
    { "pncard"_ostr, { RTFControlType::FLAG, RTFKeyword::PNCARD } },
    { "pncf"_ostr, { RTFControlType::VALUE, RTFKeyword::PNCF } },
    { "pnchosung"_ostr, { RTFControlType::FLAG, RTFKeyword::PNCHOSUNG } },
    { "pncnum"_ostr, { RTFControlType::FLAG, RTFKeyword::PNCNUM } },
    { "pndbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::PNDBNUM } },
    { "pndbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::PNDBNUMD } },
    { "pndbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::PNDBNUMK } },
    { "pndbnuml"_ostr, { RTFControlType::FLAG, RTFKeyword::PNDBNUML } },
    { "pndbnumt"_ostr, { RTFControlType::FLAG, RTFKeyword::PNDBNUMT } },
    { "pndec"_ostr, { RTFControlType::FLAG, RTFKeyword::PNDEC } },
    { "pndecd"_ostr, { RTFControlType::FLAG, RTFKeyword::PNDECD } },
    { "pnf"_ostr, { RTFControlType::VALUE, RTFKeyword::PNF } },
    { "pnfs"_ostr, { RTFControlType::VALUE, RTFKeyword::PNFS } },
    { "pnganada"_ostr, { RTFControlType::FLAG, RTFKeyword::PNGANADA } },
    { "pngblip"_ostr, { RTFControlType::FLAG, RTFKeyword::PNGBLIP } },
    { "pngbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::PNGBNUM } },
    { "pngbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::PNGBNUMD } },
    { "pngbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::PNGBNUMK } },
    { "pngbnuml"_ostr, { RTFControlType::FLAG, RTFKeyword::PNGBNUML } },
    { "pnhang"_ostr, { RTFControlType::FLAG, RTFKeyword::PNHANG } },
    { "pni"_ostr, { RTFControlType::TOGGLE, RTFKeyword::PNI, 1 } },
    { "pnindent"_ostr, { RTFControlType::VALUE, RTFKeyword::PNINDENT } },
    { "pniroha"_ostr, { RTFControlType::FLAG, RTFKeyword::PNIROHA } },
    { "pnirohad"_ostr, { RTFControlType::FLAG, RTFKeyword::PNIROHAD } },
    { "pnlcltr"_ostr, { RTFControlType::FLAG, RTFKeyword::PNLCLTR } },
    { "pnlcrm"_ostr, { RTFControlType::FLAG, RTFKeyword::PNLCRM } },
    { "pnlvl"_ostr, { RTFControlType::VALUE, RTFKeyword::PNLVL } },
    { "pnlvlblt"_ostr, { RTFControlType::FLAG, RTFKeyword::PNLVLBLT } },
    { "pnlvlbody"_ostr, { RTFControlType::FLAG, RTFKeyword::PNLVLBODY } },
    { "pnlvlcont"_ostr, { RTFControlType::FLAG, RTFKeyword::PNLVLCONT } },
    { "pnnumonce"_ostr, { RTFControlType::FLAG, RTFKeyword::PNNUMONCE } },
    { "pnord"_ostr, { RTFControlType::FLAG, RTFKeyword::PNORD } },
    { "pnordt"_ostr, { RTFControlType::FLAG, RTFKeyword::PNORDT } },
    { "pnprev"_ostr, { RTFControlType::FLAG, RTFKeyword::PNPREV } },
    { "pnqc"_ostr, { RTFControlType::FLAG, RTFKeyword::PNQC } },
    { "pnql"_ostr, { RTFControlType::FLAG, RTFKeyword::PNQL } },
    { "pnqr"_ostr, { RTFControlType::FLAG, RTFKeyword::PNQR } },
    { "pnrauth"_ostr, { RTFControlType::VALUE, RTFKeyword::PNRAUTH } },
    { "pnrdate"_ostr, { RTFControlType::VALUE, RTFKeyword::PNRDATE } },
    { "pnrestart"_ostr, { RTFControlType::FLAG, RTFKeyword::PNRESTART } },
    { "pnrnfc"_ostr, { RTFControlType::VALUE, RTFKeyword::PNRNFC } },
    { "pnrnot"_ostr, { RTFControlType::FLAG, RTFKeyword::PNRNOT } },
    { "pnrpnbr"_ostr, { RTFControlType::VALUE, RTFKeyword::PNRPNBR } },
    { "pnrrgb"_ostr, { RTFControlType::VALUE, RTFKeyword::PNRRGB } },
    { "pnrstart"_ostr, { RTFControlType::VALUE, RTFKeyword::PNRSTART } },
    { "pnrstop"_ostr, { RTFControlType::VALUE, RTFKeyword::PNRSTOP } },
    { "pnrxst"_ostr, { RTFControlType::VALUE, RTFKeyword::PNRXST } },
    { "pnscaps"_ostr, { RTFControlType::TOGGLE, RTFKeyword::PNSCAPS, 1 } },
    { "pnseclvl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PNSECLVL } },
    { "pnsp"_ostr, { RTFControlType::VALUE, RTFKeyword::PNSP } },
    { "pnstart"_ostr, { RTFControlType::VALUE, RTFKeyword::PNSTART } },
    { "pnstrike"_ostr, { RTFControlType::TOGGLE, RTFKeyword::PNSTRIKE, 1 } },
    { "pntext"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PNTEXT } },
    { "pntxta"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PNTXTA } },
    { "pntxtb"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PNTXTB } },
    { "pnucltr"_ostr, { RTFControlType::FLAG, RTFKeyword::PNUCLTR } },
    { "pnucrm"_ostr, { RTFControlType::FLAG, RTFKeyword::PNUCRM } },
    { "pnul"_ostr, { RTFControlType::TOGGLE, RTFKeyword::PNUL, 1 } },
    { "pnuld"_ostr, { RTFControlType::FLAG, RTFKeyword::PNULD } },
    { "pnuldash"_ostr, { RTFControlType::FLAG, RTFKeyword::PNULDASH } },
    { "pnuldashd"_ostr, { RTFControlType::FLAG, RTFKeyword::PNULDASHD } },
    { "pnuldashdd"_ostr, { RTFControlType::FLAG, RTFKeyword::PNULDASHDD } },
    { "pnuldb"_ostr, { RTFControlType::FLAG, RTFKeyword::PNULDB } },
    { "pnulhair"_ostr, { RTFControlType::FLAG, RTFKeyword::PNULHAIR } },
    { "pnulnone"_ostr, { RTFControlType::FLAG, RTFKeyword::PNULNONE } },
    { "pnulth"_ostr, { RTFControlType::FLAG, RTFKeyword::PNULTH } },
    { "pnulw"_ostr, { RTFControlType::FLAG, RTFKeyword::PNULW } },
    { "pnulwave"_ostr, { RTFControlType::FLAG, RTFKeyword::PNULWAVE } },
    { "pnzodiac"_ostr, { RTFControlType::FLAG, RTFKeyword::PNZODIAC } },
    { "pnzodiacd"_ostr, { RTFControlType::FLAG, RTFKeyword::PNZODIACD } },
    { "pnzodiacl"_ostr, { RTFControlType::FLAG, RTFKeyword::PNZODIACL } },
    { "posnegx"_ostr, { RTFControlType::VALUE, RTFKeyword::POSNEGX } },
    { "posnegy"_ostr, { RTFControlType::VALUE, RTFKeyword::POSNEGY } },
    { "posx"_ostr, { RTFControlType::VALUE, RTFKeyword::POSX } },
    { "posxc"_ostr, { RTFControlType::FLAG, RTFKeyword::POSXC } },
    { "posxi"_ostr, { RTFControlType::FLAG, RTFKeyword::POSXI } },
    { "posxl"_ostr, { RTFControlType::FLAG, RTFKeyword::POSXL } },
    { "posxo"_ostr, { RTFControlType::FLAG, RTFKeyword::POSXO } },
    { "posxr"_ostr, { RTFControlType::FLAG, RTFKeyword::POSXR } },
    { "posy"_ostr, { RTFControlType::VALUE, RTFKeyword::POSY } },
    { "posyb"_ostr, { RTFControlType::FLAG, RTFKeyword::POSYB } },
    { "posyc"_ostr, { RTFControlType::FLAG, RTFKeyword::POSYC } },
    { "posyil"_ostr, { RTFControlType::FLAG, RTFKeyword::POSYIL } },
    { "posyin"_ostr, { RTFControlType::FLAG, RTFKeyword::POSYIN } },
    { "posyout"_ostr, { RTFControlType::FLAG, RTFKeyword::POSYOUT } },
    { "posyt"_ostr, { RTFControlType::FLAG, RTFKeyword::POSYT } },
    { "prauth"_ostr, { RTFControlType::VALUE, RTFKeyword::PRAUTH } },
    { "prcolbl"_ostr, { RTFControlType::FLAG, RTFKeyword::PRCOLBL } },
    { "prdate"_ostr, { RTFControlType::VALUE, RTFKeyword::PRDATE } },
    { "printdata"_ostr, { RTFControlType::FLAG, RTFKeyword::PRINTDATA } },
    { "printim"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PRINTIM } },
    { "private"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PRIVATE } },
    { "propname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PROPNAME } },
    { "proptype"_ostr, { RTFControlType::VALUE, RTFKeyword::PROPTYPE } },
    { "protect"_ostr, { RTFControlType::TOGGLE, RTFKeyword::PROTECT, 1 } },
    { "protend"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PROTEND } },
    { "protlevel"_ostr, { RTFControlType::VALUE, RTFKeyword::PROTLEVEL } },
    { "protstart"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PROTSTART } },
    { "protusertbl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PROTUSERTBL } },
    { "psover"_ostr, { RTFControlType::FLAG, RTFKeyword::PSOVER } },
    { "psz"_ostr, { RTFControlType::VALUE, RTFKeyword::PSZ } },
    { "ptabldot"_ostr, { RTFControlType::FLAG, RTFKeyword::PTABLDOT } },
    { "ptablmdot"_ostr, { RTFControlType::FLAG, RTFKeyword::PTABLMDOT } },
    { "ptablminus"_ostr, { RTFControlType::FLAG, RTFKeyword::PTABLMINUS } },
    { "ptablnone"_ostr, { RTFControlType::FLAG, RTFKeyword::PTABLNONE } },
    { "ptabluscore"_ostr, { RTFControlType::FLAG, RTFKeyword::PTABLUSCORE } },
    { "pubauto"_ostr, { RTFControlType::FLAG, RTFKeyword::PUBAUTO } },
    { "pvmrg"_ostr, { RTFControlType::FLAG, RTFKeyword::PVMRG } },
    { "pvpara"_ostr, { RTFControlType::FLAG, RTFKeyword::PVPARA } },
    { "pvpg"_ostr, { RTFControlType::FLAG, RTFKeyword::PVPG } },
    { "pwd"_ostr, { RTFControlType::VALUE, RTFKeyword::PWD } },
    { "pxe"_ostr, { RTFControlType::DESTINATION, RTFKeyword::PXE } },
    { "qc"_ostr, { RTFControlType::FLAG, RTFKeyword::QC } },
    { "qd"_ostr, { RTFControlType::FLAG, RTFKeyword::QD } },
    { "qj"_ostr, { RTFControlType::FLAG, RTFKeyword::QJ } },
    { "qk"_ostr, { RTFControlType::VALUE, RTFKeyword::QK } },
    { "ql"_ostr, { RTFControlType::FLAG, RTFKeyword::QL } },
    { "qmspace"_ostr, { RTFControlType::SYMBOL, RTFKeyword::QMSPACE } },
    { "qr"_ostr, { RTFControlType::FLAG, RTFKeyword::QR } },
    { "qt"_ostr, { RTFControlType::FLAG, RTFKeyword::QT } },
    { "rawclbgdkbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGDKBDIAG } },
    { "rawclbgbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGBDIAG } },
    { "rawclbgcross"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGCROSS } },
    { "rawclbgdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGDCROSS } },
    { "rawclbgdkcross"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGDKCROSS } },
    { "rawclbgdkdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGDKDCROSS } },
    { "rawclbgdkfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGDKFDIAG } },
    { "rawclbgdkhor"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGDKHOR } },
    { "rawclbgdkvert"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGDKVERT } },
    { "rawclbgfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGFDIAG } },
    { "rawclbghoriz"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGHORIZ } },
    { "rawclbgvert"_ostr, { RTFControlType::FLAG, RTFKeyword::RAWCLBGVERT } },
    { "rdblquote"_ostr, { RTFControlType::SYMBOL, RTFKeyword::RDBLQUOTE } },
    { "readonlyrecommended"_ostr, { RTFControlType::FLAG, RTFKeyword::READONLYRECOMMENDED } },
    { "readprot"_ostr, { RTFControlType::FLAG, RTFKeyword::READPROT } },
    { "red"_ostr, { RTFControlType::VALUE, RTFKeyword::RED } },
    { "relyonvml"_ostr, { RTFControlType::VALUE, RTFKeyword::RELYONVML } },
    { "remdttm"_ostr, { RTFControlType::FLAG, RTFKeyword::REMDTTM } },
    { "rempersonalinfo"_ostr, { RTFControlType::FLAG, RTFKeyword::REMPERSONALINFO } },
    { "result"_ostr, { RTFControlType::DESTINATION, RTFKeyword::RESULT } },
    { "revauth"_ostr, { RTFControlType::VALUE, RTFKeyword::REVAUTH } },
    { "revauthdel"_ostr, { RTFControlType::VALUE, RTFKeyword::REVAUTHDEL } },
    { "revbar"_ostr, { RTFControlType::VALUE, RTFKeyword::REVBAR, 3 } },
    { "revdttm"_ostr, { RTFControlType::VALUE, RTFKeyword::REVDTTM } },
    { "revdttmdel"_ostr, { RTFControlType::VALUE, RTFKeyword::REVDTTMDEL } },
    { "revised"_ostr, { RTFControlType::TOGGLE, RTFKeyword::REVISED, 1 } },
    { "revisions"_ostr, { RTFControlType::FLAG, RTFKeyword::REVISIONS } },
    { "revprop"_ostr, { RTFControlType::VALUE, RTFKeyword::REVPROP, 3 } },
    { "revprot"_ostr, { RTFControlType::FLAG, RTFKeyword::REVPROT } },
    { "revtbl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::REVTBL } },
    { "revtim"_ostr, { RTFControlType::DESTINATION, RTFKeyword::REVTIM } },
    { "ri"_ostr, { RTFControlType::VALUE, RTFKeyword::RI } },
    { "rin"_ostr, { RTFControlType::VALUE, RTFKeyword::RIN } },
    { "row"_ostr, { RTFControlType::SYMBOL, RTFKeyword::ROW } },
    { "rquote"_ostr, { RTFControlType::SYMBOL, RTFKeyword::RQUOTE } },
    { "rsid"_ostr, { RTFControlType::VALUE, RTFKeyword::RSID } },
    { "rsidroot"_ostr, { RTFControlType::VALUE, RTFKeyword::RSIDROOT } },
    { "rsidtbl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::RSIDTBL } },
    { "rsltbmp"_ostr, { RTFControlType::FLAG, RTFKeyword::RSLTBMP } },
    { "rslthtml"_ostr, { RTFControlType::FLAG, RTFKeyword::RSLTHTML } },
    { "rsltmerge"_ostr, { RTFControlType::FLAG, RTFKeyword::RSLTMERGE } },
    { "rsltpict"_ostr, { RTFControlType::FLAG, RTFKeyword::RSLTPICT } },
    { "rsltrtf"_ostr, { RTFControlType::FLAG, RTFKeyword::RSLTRTF } },
    { "rslttxt"_ostr, { RTFControlType::FLAG, RTFKeyword::RSLTTXT } },
    { "rtf"_ostr, { RTFControlType::DESTINATION, RTFKeyword::RTF } },
    { "rtlch"_ostr, { RTFControlType::FLAG, RTFKeyword::RTLCH } },
    { "rtldoc"_ostr, { RTFControlType::FLAG, RTFKeyword::RTLDOC } },
    { "rtlgutter"_ostr, { RTFControlType::FLAG, RTFKeyword::RTLGUTTER } },
    { "rtlmark"_ostr, { RTFControlType::SYMBOL, RTFKeyword::RTLMARK } },
    { "rtlpar"_ostr, { RTFControlType::FLAG, RTFKeyword::RTLPAR } },
    { "rtlrow"_ostr, { RTFControlType::FLAG, RTFKeyword::RTLROW } },
    { "rtlsect"_ostr, { RTFControlType::FLAG, RTFKeyword::RTLSECT } },
    { "rxe"_ostr, { RTFControlType::DESTINATION, RTFKeyword::RXE } },
    { "s"_ostr, { RTFControlType::VALUE, RTFKeyword::S } },
    { "sa"_ostr, { RTFControlType::VALUE, RTFKeyword::SA } },
    { "saauto"_ostr, { RTFControlType::TOGGLE, RTFKeyword::SAAUTO, 1 } },
    { "saftnnalc"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNALC } },
    { "saftnnar"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNAR } },
    { "saftnnauc"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNAUC } },
    { "saftnnchi"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNCHI } },
    { "saftnnchosung"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNCHOSUNG } },
    { "saftnncnum"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNCNUM } },
    { "saftnndbar"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNDBAR } },
    { "saftnndbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNDBNUM } },
    { "saftnndbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNDBNUMD } },
    { "saftnndbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNDBNUMK } },
    { "saftnndbnumt"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNDBNUMT } },
    { "saftnnganada"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNGANADA } },
    { "saftnngbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNGBNUM } },
    { "saftnngbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNGBNUMD } },
    { "saftnngbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNGBNUMK } },
    { "saftnngbnuml"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNGBNUML } },
    { "saftnnrlc"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNRLC } },
    { "saftnnruc"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNRUC } },
    { "saftnnzodiac"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNZODIAC } },
    { "saftnnzodiacd"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNZODIACD } },
    { "saftnnzodiacl"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNNZODIACL } },
    { "saftnrestart"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNRESTART } },
    { "saftnrstcont"_ostr, { RTFControlType::FLAG, RTFKeyword::SAFTNRSTCONT } },
    { "saftnstart"_ostr, { RTFControlType::VALUE, RTFKeyword::SAFTNSTART, 1 } },
    { "sautoupd"_ostr, { RTFControlType::FLAG, RTFKeyword::SAUTOUPD } },
    { "saveinvalidxml"_ostr, { RTFControlType::FLAG, RTFKeyword::SAVEINVALIDXML } },
    { "saveprevpict"_ostr, { RTFControlType::FLAG, RTFKeyword::SAVEPREVPICT } },
    { "sb"_ostr, { RTFControlType::VALUE, RTFKeyword::SB } },
    { "sbasedon"_ostr, { RTFControlType::VALUE, RTFKeyword::SBASEDON, 222 } },
    { "sbauto"_ostr, { RTFControlType::TOGGLE, RTFKeyword::SBAUTO, 1 } },
    { "sbkcol"_ostr, { RTFControlType::FLAG, RTFKeyword::SBKCOL } },
    { "sbkeven"_ostr, { RTFControlType::FLAG, RTFKeyword::SBKEVEN } },
    { "sbknone"_ostr, { RTFControlType::FLAG, RTFKeyword::SBKNONE } },
    { "sbkodd"_ostr, { RTFControlType::FLAG, RTFKeyword::SBKODD } },
    { "sbkpage"_ostr, { RTFControlType::FLAG, RTFKeyword::SBKPAGE } },
    { "sbys"_ostr, { RTFControlType::FLAG, RTFKeyword::SBYS } },
    { "scaps"_ostr, { RTFControlType::TOGGLE, RTFKeyword::SCAPS, 1 } },
    { "scompose"_ostr, { RTFControlType::FLAG, RTFKeyword::SCOMPOSE } },
    { "sec"_ostr, { RTFControlType::VALUE, RTFKeyword::SEC } },
    { "sect"_ostr, { RTFControlType::SYMBOL, RTFKeyword::SECT } },
    { "sectd"_ostr, { RTFControlType::FLAG, RTFKeyword::SECTD } },
    { "sectdefaultcl"_ostr, { RTFControlType::FLAG, RTFKeyword::SECTDEFAULTCL } },
    { "sectexpand"_ostr, { RTFControlType::VALUE, RTFKeyword::SECTEXPAND } },
    { "sectlinegrid"_ostr, { RTFControlType::VALUE, RTFKeyword::SECTLINEGRID } },
    { "sectnum"_ostr, { RTFControlType::SYMBOL, RTFKeyword::SECTNUM } },
    { "sectrsid"_ostr, { RTFControlType::VALUE, RTFKeyword::SECTRSID } },
    { "sectspecifycl"_ostr, { RTFControlType::FLAG, RTFKeyword::SECTSPECIFYCL } },
    { "sectspecifygenN"_ostr, { RTFControlType::FLAG, RTFKeyword::SECTSPECIFYGENN } },
    { "sectspecifyl"_ostr, { RTFControlType::FLAG, RTFKeyword::SECTSPECIFYL } },
    { "sectunlocked"_ostr, { RTFControlType::FLAG, RTFKeyword::SECTUNLOCKED } },
    { "sftnbj"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNBJ } },
    { "sftnnalc"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNALC } },
    { "sftnnar"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNAR } },
    { "sftnnauc"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNAUC } },
    { "sftnnchi"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNCHI } },
    { "sftnnchosung"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNCHOSUNG } },
    { "sftnncnum"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNCNUM } },
    { "sftnndbar"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNDBAR } },
    { "sftnndbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNDBNUM } },
    { "sftnndbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNDBNUMD } },
    { "sftnndbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNDBNUMK } },
    { "sftnndbnumt"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNDBNUMT } },
    { "sftnnganada"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNGANADA } },
    { "sftnngbnum"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNGBNUM } },
    { "sftnngbnumd"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNGBNUMD } },
    { "sftnngbnumk"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNGBNUMK } },
    { "sftnngbnuml"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNGBNUML } },
    { "sftnnrlc"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNRLC } },
    { "sftnnruc"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNRUC } },
    { "sftnnzodiac"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNZODIAC } },
    { "sftnnzodiacd"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNZODIACD } },
    { "sftnnzodiacl"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNNZODIACL } },
    { "sftnrestart"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNRESTART } },
    { "sftnrstcont"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNRSTCONT } },
    { "sftnrstpg"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNRSTPG } },
    { "sftnstart"_ostr, { RTFControlType::VALUE, RTFKeyword::SFTNSTART, 1 } },
    { "sftntj"_ostr, { RTFControlType::FLAG, RTFKeyword::SFTNTJ } },
    { "shad"_ostr, { RTFControlType::TOGGLE, RTFKeyword::SHAD, 1 } },
    { "shading"_ostr, { RTFControlType::VALUE, RTFKeyword::SHADING } },
    { "shidden"_ostr, { RTFControlType::FLAG, RTFKeyword::SHIDDEN } },
    { "shift"_ostr, { RTFControlType::FLAG, RTFKeyword::SHIFT } },
    { "showplaceholdtext"_ostr, { RTFControlType::VALUE, RTFKeyword::SHOWPLACEHOLDTEXT } },
    { "showxmlerrors"_ostr, { RTFControlType::VALUE, RTFKeyword::SHOWXMLERRORS } },
    { "shp"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SHP } },
    { "shpbottom"_ostr, { RTFControlType::VALUE, RTFKeyword::SHPBOTTOM } },
    { "shpbxcolumn"_ostr, { RTFControlType::FLAG, RTFKeyword::SHPBXCOLUMN } },
    { "shpbxignore"_ostr, { RTFControlType::FLAG, RTFKeyword::SHPBXIGNORE } },
    { "shpbxmargin"_ostr, { RTFControlType::FLAG, RTFKeyword::SHPBXMARGIN } },
    { "shpbxpage"_ostr, { RTFControlType::FLAG, RTFKeyword::SHPBXPAGE } },
    { "shpbyignore"_ostr, { RTFControlType::FLAG, RTFKeyword::SHPBYIGNORE } },
    { "shpbymargin"_ostr, { RTFControlType::FLAG, RTFKeyword::SHPBYMARGIN } },
    { "shpbypage"_ostr, { RTFControlType::FLAG, RTFKeyword::SHPBYPAGE } },
    { "shpbypara"_ostr, { RTFControlType::FLAG, RTFKeyword::SHPBYPARA } },
    { "shpfblwtxt"_ostr, { RTFControlType::VALUE, RTFKeyword::SHPFBLWTXT } },
    { "shpfhdr"_ostr, { RTFControlType::VALUE, RTFKeyword::SHPFHDR } },
    { "shpgrp"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SHPGRP } },
    { "shpinst"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SHPINST } },
    { "shpleft"_ostr, { RTFControlType::VALUE, RTFKeyword::SHPLEFT } },
    { "shplid"_ostr, { RTFControlType::VALUE, RTFKeyword::SHPLID } },
    { "shplockanchor"_ostr, { RTFControlType::FLAG, RTFKeyword::SHPLOCKANCHOR } },
    { "shppict"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SHPPICT } },
    { "shpright"_ostr, { RTFControlType::VALUE, RTFKeyword::SHPRIGHT } },
    { "shprslt"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SHPRSLT } },
    { "shptop"_ostr, { RTFControlType::VALUE, RTFKeyword::SHPTOP } },
    { "shptxt"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SHPTXT } },
    { "shpwrk"_ostr, { RTFControlType::VALUE, RTFKeyword::SHPWRK } },
    { "shpwr"_ostr, { RTFControlType::VALUE, RTFKeyword::SHPWR } },
    { "shpz"_ostr, { RTFControlType::VALUE, RTFKeyword::SHPZ } },
    { "sl"_ostr, { RTFControlType::VALUE, RTFKeyword::SL } },
    { "slink"_ostr, { RTFControlType::VALUE, RTFKeyword::SLINK } },
    { "slmult"_ostr, { RTFControlType::VALUE, RTFKeyword::SLMULT } },
    { "slocked"_ostr, { RTFControlType::FLAG, RTFKeyword::SLOCKED } },
    { "sn"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SN } },
    { "snaptogridincell"_ostr, { RTFControlType::FLAG, RTFKeyword::SNAPTOGRIDINCELL } },
    { "snext"_ostr, { RTFControlType::VALUE, RTFKeyword::SNEXT } },
    { "softcol"_ostr, { RTFControlType::FLAG, RTFKeyword::SOFTCOL } },
    { "softlheight"_ostr, { RTFControlType::VALUE, RTFKeyword::SOFTLHEIGHT } },
    { "softline"_ostr, { RTFControlType::FLAG, RTFKeyword::SOFTLINE } },
    { "softpage"_ostr, { RTFControlType::FLAG, RTFKeyword::SOFTPAGE } },
    { "sp"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SP } },
    { "spersonal"_ostr, { RTFControlType::FLAG, RTFKeyword::SPERSONAL } },
    { "spltpgpar"_ostr, { RTFControlType::FLAG, RTFKeyword::SPLTPGPAR } },
    { "splytwnine"_ostr, { RTFControlType::FLAG, RTFKeyword::SPLYTWNINE } },
    { "spriority"_ostr, { RTFControlType::VALUE, RTFKeyword::SPRIORITY } },
    { "sprsbsp"_ostr, { RTFControlType::FLAG, RTFKeyword::SPRSBSP } },
    { "sprslnsp"_ostr, { RTFControlType::FLAG, RTFKeyword::SPRSLNSP } },
    { "sprsspbf"_ostr, { RTFControlType::FLAG, RTFKeyword::SPRSSPBF } },
    { "sprstsm"_ostr, { RTFControlType::FLAG, RTFKeyword::SPRSTSM } },
    { "sprstsp"_ostr, { RTFControlType::FLAG, RTFKeyword::SPRSTSP } },
    { "spv"_ostr, { RTFControlType::FLAG, RTFKeyword::SPV } },
    { "sqformat"_ostr, { RTFControlType::FLAG, RTFKeyword::SQFORMAT } },
    { "srauth"_ostr, { RTFControlType::VALUE, RTFKeyword::SRAUTH } },
    { "srdate"_ostr, { RTFControlType::VALUE, RTFKeyword::SRDATE } },
    { "sreply"_ostr, { RTFControlType::FLAG, RTFKeyword::SREPLY } },
    { "ssemihidden"_ostr, { RTFControlType::VALUE, RTFKeyword::SSEMIHIDDEN } },
    { "staticval"_ostr, { RTFControlType::DESTINATION, RTFKeyword::STATICVAL } },
    { "stextflow"_ostr, { RTFControlType::VALUE, RTFKeyword::STEXTFLOW } },
    { "strike"_ostr, { RTFControlType::TOGGLE, RTFKeyword::STRIKE, 1 } },
    { "striked"_ostr, { RTFControlType::TOGGLE, RTFKeyword::STRIKED, 1 } },
    { "stshfbi"_ostr, { RTFControlType::VALUE, RTFKeyword::STSHFBI } },
    { "stshfdbch"_ostr, { RTFControlType::VALUE, RTFKeyword::STSHFDBCH } },
    { "stshfhich"_ostr, { RTFControlType::VALUE, RTFKeyword::STSHFHICH } },
    { "stshfloch"_ostr, { RTFControlType::VALUE, RTFKeyword::STSHFLOCH } },
    { "stylelock"_ostr, { RTFControlType::FLAG, RTFKeyword::STYLELOCK } },
    { "stylelockbackcomp"_ostr, { RTFControlType::FLAG, RTFKeyword::STYLELOCKBACKCOMP } },
    { "stylelockenforced"_ostr, { RTFControlType::FLAG, RTFKeyword::STYLELOCKENFORCED } },
    { "stylelockqfset"_ostr, { RTFControlType::FLAG, RTFKeyword::STYLELOCKQFSET } },
    { "stylelocktheme"_ostr, { RTFControlType::FLAG, RTFKeyword::STYLELOCKTHEME } },
    { "stylesheet"_ostr, { RTFControlType::DESTINATION, RTFKeyword::STYLESHEET } },
    { "stylesortmethod"_ostr, { RTFControlType::VALUE, RTFKeyword::STYLESORTMETHOD, 1 } },
    { "styrsid"_ostr, { RTFControlType::VALUE, RTFKeyword::STYRSID } },
    { "sub"_ostr, { RTFControlType::FLAG, RTFKeyword::SUB } },
    { "subdocument"_ostr, { RTFControlType::VALUE, RTFKeyword::SUBDOCUMENT } },
    { "subfontbysize"_ostr, { RTFControlType::FLAG, RTFKeyword::SUBFONTBYSIZE } },
    { "subject"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SUBJECT } },
    { "sunhideused"_ostr, { RTFControlType::VALUE, RTFKeyword::SUNHIDEUSED } },
    { "super"_ostr, { RTFControlType::FLAG, RTFKeyword::SUPER } },
    { "sv"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SV } },
    { "svb"_ostr, { RTFControlType::DESTINATION, RTFKeyword::SVB } },
    { "swpbdr"_ostr, { RTFControlType::FLAG, RTFKeyword::SWPBDR } },
    { "tab"_ostr, { RTFControlType::SYMBOL, RTFKeyword::TAB } },
    { "tabsnoovrlp"_ostr, { RTFControlType::FLAG, RTFKeyword::TABSNOOVRLP } },
    { "taprtl"_ostr, { RTFControlType::FLAG, RTFKeyword::TAPRTL } },
    { "tb"_ostr, { RTFControlType::VALUE, RTFKeyword::TB } },
    { "tblind"_ostr, { RTFControlType::VALUE, RTFKeyword::TBLIND } },
    { "tblindtype"_ostr, { RTFControlType::VALUE, RTFKeyword::TBLINDTYPE } },
    { "tbllkbestfit"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKBESTFIT } },
    { "tbllkborder"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKBORDER } },
    { "tbllkcolor"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKCOLOR } },
    { "tbllkfont"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKFONT } },
    { "tbllkhdrcols"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKHDRCOLS } },
    { "tbllkhdrrows"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKHDRROWS } },
    { "tbllklastcol"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKLASTCOL } },
    { "tbllklastrow"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKLASTROW } },
    { "tbllknocolband"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKNOCOLBAND } },
    { "tbllknorowband"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKNOROWBAND } },
    { "tbllkshading"_ostr, { RTFControlType::FLAG, RTFKeyword::TBLLKSHADING } },
    { "tblrsid"_ostr, { RTFControlType::VALUE, RTFKeyword::TBLRSID } },
    { "tc"_ostr, { RTFControlType::DESTINATION, RTFKeyword::TC } },
    { "tcelld"_ostr, { RTFControlType::FLAG, RTFKeyword::TCELLD } },
    { "tcf"_ostr, { RTFControlType::VALUE, RTFKeyword::TCF, 67 } },
    { "tcl"_ostr, { RTFControlType::VALUE, RTFKeyword::TCL } },
    { "tcn"_ostr, { RTFControlType::FLAG, RTFKeyword::TCN } },
    { "tdfrmtxtBottom"_ostr, { RTFControlType::VALUE, RTFKeyword::TDFRMTXTBOTTOM } },
    { "tdfrmtxtLeft"_ostr, { RTFControlType::VALUE, RTFKeyword::TDFRMTXTLEFT } },
    { "tdfrmtxtRight"_ostr, { RTFControlType::VALUE, RTFKeyword::TDFRMTXTRIGHT } },
    { "tdfrmtxtTop"_ostr, { RTFControlType::VALUE, RTFKeyword::TDFRMTXTTOP } },
    { "template"_ostr, { RTFControlType::DESTINATION, RTFKeyword::TEMPLATE } },
    { "themedata"_ostr, { RTFControlType::DESTINATION, RTFKeyword::THEMEDATA } },
    { "themelang"_ostr, { RTFControlType::VALUE, RTFKeyword::THEMELANG } },
    { "themelangcs"_ostr, { RTFControlType::VALUE, RTFKeyword::THEMELANGCS } },
    { "themelangfe"_ostr, { RTFControlType::VALUE, RTFKeyword::THEMELANGFE } },
    { "time"_ostr, { RTFControlType::FLAG, RTFKeyword::TIME } },
    { "title"_ostr, { RTFControlType::DESTINATION, RTFKeyword::TITLE } },
    { "titlepg"_ostr, { RTFControlType::FLAG, RTFKeyword::TITLEPG } },
    { "tldot"_ostr, { RTFControlType::FLAG, RTFKeyword::TLDOT } },
    { "tleq"_ostr, { RTFControlType::FLAG, RTFKeyword::TLEQ } },
    { "tlhyph"_ostr, { RTFControlType::FLAG, RTFKeyword::TLHYPH } },
    { "tlmdot"_ostr, { RTFControlType::FLAG, RTFKeyword::TLMDOT } },
    { "tlth"_ostr, { RTFControlType::FLAG, RTFKeyword::TLTH } },
    { "tlul"_ostr, { RTFControlType::FLAG, RTFKeyword::TLUL } },
    { "toplinepunct"_ostr, { RTFControlType::FLAG, RTFKeyword::TOPLINEPUNCT } },
    { "tphcol"_ostr, { RTFControlType::FLAG, RTFKeyword::TPHCOL } },
    { "tphmrg"_ostr, { RTFControlType::FLAG, RTFKeyword::TPHMRG } },
    { "tphpg"_ostr, { RTFControlType::FLAG, RTFKeyword::TPHPG } },
    { "tposnegx"_ostr, { RTFControlType::VALUE, RTFKeyword::TPOSNEGX } },
    { "tposnegy"_ostr, { RTFControlType::VALUE, RTFKeyword::TPOSNEGY } },
    { "tposxc"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSXC } },
    { "tposxi"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSXI } },
    { "tposxl"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSXL } },
    { "tposx"_ostr, { RTFControlType::VALUE, RTFKeyword::TPOSX } },
    { "tposxo"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSXO } },
    { "tposxr"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSXR } },
    { "tposy"_ostr, { RTFControlType::VALUE, RTFKeyword::TPOSY } },
    { "tposyb"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSYB } },
    { "tposyc"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSYC } },
    { "tposyil"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSYIL } },
    { "tposyin"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSYIN } },
    { "tposyout"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSYOUT } },
    { "tposyt"_ostr, { RTFControlType::FLAG, RTFKeyword::TPOSYT } },
    { "tpvmrg"_ostr, { RTFControlType::FLAG, RTFKeyword::TPVMRG } },
    { "tpvpara"_ostr, { RTFControlType::FLAG, RTFKeyword::TPVPARA } },
    { "tpvpg"_ostr, { RTFControlType::FLAG, RTFKeyword::TPVPG } },
    { "tqc"_ostr, { RTFControlType::FLAG, RTFKeyword::TQC } },
    { "tqdec"_ostr, { RTFControlType::FLAG, RTFKeyword::TQDEC } },
    { "tqr"_ostr, { RTFControlType::FLAG, RTFKeyword::TQR } },
    { "trackformatting"_ostr, { RTFControlType::VALUE, RTFKeyword::TRACKFORMATTING } },
    { "trackmoves"_ostr, { RTFControlType::VALUE, RTFKeyword::TRACKMOVES } },
    { "transmf"_ostr, { RTFControlType::FLAG, RTFKeyword::TRANSMF } },
    { "trauth"_ostr, { RTFControlType::VALUE, RTFKeyword::TRAUTH } },
    { "trautofit"_ostr, { RTFControlType::TOGGLE, RTFKeyword::TRAUTOFIT, 1 } },
    { "trbgbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGBDIAG } },
    { "trbgcross"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGCROSS } },
    { "trbgdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGDCROSS } },
    { "trbgdkbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGDKBDIAG } },
    { "trbgdkcross"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGDKCROSS } },
    { "trbgdkdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGDKDCROSS } },
    { "trbgdkfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGDKFDIAG } },
    { "trbgdkhor"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGDKHOR } },
    { "trbgdkvert"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGDKVERT } },
    { "trbgfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGFDIAG } },
    { "trbghoriz"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGHORIZ } },
    { "trbgvert"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBGVERT } },
    { "trbrdrb"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBRDRB } },
    { "trbrdrh"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBRDRH } },
    { "trbrdrl"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBRDRL } },
    { "trbrdrr"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBRDRR } },
    { "trbrdrt"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBRDRT } },
    { "trbrdrv"_ostr, { RTFControlType::FLAG, RTFKeyword::TRBRDRV } },
    { "trcbpat"_ostr, { RTFControlType::VALUE, RTFKeyword::TRCBPAT } },
    { "trcfpat"_ostr, { RTFControlType::VALUE, RTFKeyword::TRCFPAT } },
    { "trdate"_ostr, { RTFControlType::VALUE, RTFKeyword::TRDATE } },
    { "trftsWidthA"_ostr, { RTFControlType::VALUE, RTFKeyword::TRFTSWIDTHA } },
    { "trftsWidthB"_ostr, { RTFControlType::VALUE, RTFKeyword::TRFTSWIDTHB } },
    { "trftsWidth"_ostr, { RTFControlType::VALUE, RTFKeyword::TRFTSWIDTH } },
    { "trgaph"_ostr, { RTFControlType::VALUE, RTFKeyword::TRGAPH } },
    { "trhdr"_ostr, { RTFControlType::FLAG, RTFKeyword::TRHDR } },
    { "trkeep"_ostr, { RTFControlType::FLAG, RTFKeyword::TRKEEP } },
    { "trkeepfollow"_ostr, { RTFControlType::FLAG, RTFKeyword::TRKEEPFOLLOW } },
    { "trleft"_ostr, { RTFControlType::VALUE, RTFKeyword::TRLEFT } },
    { "trowd"_ostr, { RTFControlType::FLAG, RTFKeyword::TROWD } },
    { "trpaddb"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADDB } },
    { "trpaddfb"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADDFB } },
    { "trpaddfl"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADDFL } },
    { "trpaddfr"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADDFR } },
    { "trpaddft"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADDFT } },
    { "trpaddl"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADDL } },
    { "trpaddr"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADDR } },
    { "trpaddt"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADDT } },
    { "trpadob"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADOB } },
    { "trpadofb"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADOFB } },
    { "trpadofl"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADOFL } },
    { "trpadofr"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADOFR } },
    { "trpadoft"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADOFT } },
    { "trpadol"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADOL } },
    { "trpador"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADOR } },
    { "trpadot"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPADOT } },
    { "trpat"_ostr, { RTFControlType::VALUE, RTFKeyword::TRPAT } },
    { "trqc"_ostr, { RTFControlType::FLAG, RTFKeyword::TRQC } },
    { "trql"_ostr, { RTFControlType::FLAG, RTFKeyword::TRQL } },
    { "trqr"_ostr, { RTFControlType::FLAG, RTFKeyword::TRQR } },
    { "trrh"_ostr, { RTFControlType::VALUE, RTFKeyword::TRRH } },
    { "trshdng"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSHDNG } },
    { "trspdb"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPDB } },
    { "trspdfb"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPDFB } },
    { "trspdfl"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPDFL } },
    { "trspdfr"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPDFR } },
    { "trspdft"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPDFT } },
    { "trspdl"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPDL } },
    { "trspdr"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPDR } },
    { "trspdt"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPDT } },
    { "trspob"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPOB } },
    { "trspofb"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPOFB } },
    { "trspofl"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPOFL } },
    { "trspofr"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPOFR } },
    { "trspoft"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPOFT } },
    { "trspol"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPOL } },
    { "trspor"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPOR } },
    { "trspot"_ostr, { RTFControlType::VALUE, RTFKeyword::TRSPOT } },
    { "truncatefontheight"_ostr, { RTFControlType::FLAG, RTFKeyword::TRUNCATEFONTHEIGHT } },
    { "truncex"_ostr, { RTFControlType::FLAG, RTFKeyword::TRUNCEX } },
    { "trwWidthA"_ostr, { RTFControlType::VALUE, RTFKeyword::TRWWIDTHA } },
    { "trwWidthB"_ostr, { RTFControlType::VALUE, RTFKeyword::TRWWIDTHB } },
    { "trwWidth"_ostr, { RTFControlType::VALUE, RTFKeyword::TRWWIDTH } },
    { "ts"_ostr, { RTFControlType::VALUE, RTFKeyword::TS } },
    { "tsbgbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGBDIAG } },
    { "tsbgcross"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGCROSS } },
    { "tsbgdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGDCROSS } },
    { "tsbgdkbdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGDKBDIAG } },
    { "tsbgdkcross"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGDKCROSS } },
    { "tsbgdkdcross"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGDKDCROSS } },
    { "tsbgdkfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGDKFDIAG } },
    { "tsbgdkhor"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGDKHOR } },
    { "tsbgdkvert"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGDKVERT } },
    { "tsbgfdiag"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGFDIAG } },
    { "tsbghoriz"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGHORIZ } },
    { "tsbgvert"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBGVERT } },
    { "tsbrdrb"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBRDRB } },
    { "tsbrdrdgl"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBRDRDGL } },
    { "tsbrdrdgr"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBRDRDGR } },
    { "tsbrdrh"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBRDRH } },
    { "tsbrdrl"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBRDRL } },
    { "tsbrdrr"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBRDRR } },
    { "tsbrdrt"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBRDRT } },
    { "tsbrdrv"_ostr, { RTFControlType::FLAG, RTFKeyword::TSBRDRV } },
    { "tscbandhorzeven"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCBANDHORZEVEN } },
    { "tscbandhorzodd"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCBANDHORZODD } },
    { "tscbandsh"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCBANDSH } },
    { "tscbandsv"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCBANDSV } },
    { "tscbandverteven"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCBANDVERTEVEN } },
    { "tscbandvertodd"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCBANDVERTODD } },
    { "tscellcbpat"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLCBPAT } },
    { "tscellcfpat"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLCFPAT } },
    { "tscellpaddb"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLPADDB } },
    { "tscellpaddfb"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLPADDFB } },
    { "tscellpaddfl"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLPADDFL } },
    { "tscellpaddfr"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLPADDFR } },
    { "tscellpaddft"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLPADDFT } },
    { "tscellpaddl"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLPADDL } },
    { "tscellpaddr"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLPADDR } },
    { "tscellpaddt"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLPADDT } },
    { "tscellpct"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLPCT } },
    { "tscellwidth"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLWIDTH } },
    { "tscellwidthfts"_ostr, { RTFControlType::VALUE, RTFKeyword::TSCELLWIDTHFTS } },
    { "tscfirstcol"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCFIRSTCOL } },
    { "tscfirstrow"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCFIRSTROW } },
    { "tsclastcol"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCLASTCOL } },
    { "tsclastrow"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCLASTROW } },
    { "tscnecell"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCNECELL } },
    { "tscnwcell"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCNWCELL } },
    { "tscsecell"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCSECELL } },
    { "tscswcell"_ostr, { RTFControlType::FLAG, RTFKeyword::TSCSWCELL } },
    { "tsd"_ostr, { RTFControlType::FLAG, RTFKeyword::TSD } },
    { "tsnowrap"_ostr, { RTFControlType::FLAG, RTFKeyword::TSNOWRAP } },
    { "tsrowd"_ostr, { RTFControlType::FLAG, RTFKeyword::TSROWD } },
    { "tsvertalb"_ostr, { RTFControlType::FLAG, RTFKeyword::TSVERTALB } },
    { "tsvertalc"_ostr, { RTFControlType::FLAG, RTFKeyword::TSVERTALC } },
    { "tsvertalt"_ostr, { RTFControlType::FLAG, RTFKeyword::TSVERTALT } },
    { "twoinone"_ostr, { RTFControlType::VALUE, RTFKeyword::TWOINONE } },
    { "twoonone"_ostr, { RTFControlType::FLAG, RTFKeyword::TWOONONE } },
    { "tx"_ostr, { RTFControlType::VALUE, RTFKeyword::TX } },
    { "txbxtwalways"_ostr, { RTFControlType::FLAG, RTFKeyword::TXBXTWALWAYS } },
    { "txbxtwfirst"_ostr, { RTFControlType::FLAG, RTFKeyword::TXBXTWFIRST } },
    { "txbxtwfirstlast"_ostr, { RTFControlType::FLAG, RTFKeyword::TXBXTWFIRSTLAST } },
    { "txbxtwlast"_ostr, { RTFControlType::FLAG, RTFKeyword::TXBXTWLAST } },
    { "txbxtwno"_ostr, { RTFControlType::FLAG, RTFKeyword::TXBXTWNO } },
    { "txe"_ostr, { RTFControlType::DESTINATION, RTFKeyword::TXE } },
    { "u"_ostr, { RTFControlType::VALUE, RTFKeyword::U } },
    { "uc"_ostr, { RTFControlType::VALUE, RTFKeyword::UC, 1 } },
    { "ud"_ostr, { RTFControlType::DESTINATION, RTFKeyword::UD } },
    { "ul"_ostr, { RTFControlType::TOGGLE, RTFKeyword::UL, 1 } },
    { "ulc"_ostr, { RTFControlType::VALUE, RTFKeyword::ULC } },
    { "uld"_ostr, { RTFControlType::FLAG, RTFKeyword::ULD } },
    { "uldash"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULDASH, 1 } },
    { "uldashd"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULDASHD, 1 } },
    { "uldashdd"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULDASHDD, 1 } },
    { "uldb"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULDB, 1 } },
    { "ulhair"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULHAIR, 1 } },
    { "ulhwave"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULHWAVE, 1 } },
    { "ulldash"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULLDASH, 1 } },
    { "ulnone"_ostr, { RTFControlType::FLAG, RTFKeyword::ULNONE } },
    { "ulth"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULTH, 1 } },
    { "ulthd"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULTHD, 1 } },
    { "ulthdash"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULTHDASH, 1 } },
    { "ulthdashd"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULTHDASHD, 1 } },
    { "ulthdashdd"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULTHDASHDD, 1 } },
    { "ulthldash"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULTHLDASH, 1 } },
    { "ululdbwave"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULULDBWAVE, 1 } },
    { "ulw"_ostr, { RTFControlType::FLAG, RTFKeyword::ULW } },
    { "ulwave"_ostr, { RTFControlType::TOGGLE, RTFKeyword::ULWAVE, 1 } },
    { "up"_ostr, { RTFControlType::VALUE, RTFKeyword::UP, 6 } },
    { "upr"_ostr, { RTFControlType::DESTINATION, RTFKeyword::UPR } },
    { "urtf"_ostr, { RTFControlType::VALUE, RTFKeyword::URTF } },
    { "useltbaln"_ostr, { RTFControlType::FLAG, RTFKeyword::USELTBALN } },
    { "usenormstyforlist"_ostr, { RTFControlType::FLAG, RTFKeyword::USENORMSTYFORLIST } },
    { "userprops"_ostr, { RTFControlType::DESTINATION, RTFKeyword::USERPROPS } },
    { "usexform"_ostr, { RTFControlType::FLAG, RTFKeyword::USEXFORM } },
    { "utinl"_ostr, { RTFControlType::FLAG, RTFKeyword::UTINL } },
    { "v"_ostr, { RTFControlType::TOGGLE, RTFKeyword::V, 1 } },
    { "validatexml"_ostr, { RTFControlType::VALUE, RTFKeyword::VALIDATEXML } },
    { "vern"_ostr, { RTFControlType::VALUE, RTFKeyword::VERN } },
    { "version"_ostr, { RTFControlType::VALUE, RTFKeyword::VERSION } },
    { "vertal"_ostr, { RTFControlType::FLAG, RTFKeyword::VERTAL } },
    { "vertalb"_ostr, { RTFControlType::FLAG, RTFKeyword::VERTALB } },
    { "vertalc"_ostr, { RTFControlType::FLAG, RTFKeyword::VERTALC } },
    { "vertalj"_ostr, { RTFControlType::FLAG, RTFKeyword::VERTALJ } },
    { "vertalt"_ostr, { RTFControlType::FLAG, RTFKeyword::VERTALT } },
    { "vertdoc"_ostr, { RTFControlType::FLAG, RTFKeyword::VERTDOC } },
    { "vertsect"_ostr, { RTFControlType::FLAG, RTFKeyword::VERTSECT } },
    { "viewbksp"_ostr, { RTFControlType::VALUE, RTFKeyword::VIEWBKSP } },
    { "viewkind"_ostr, { RTFControlType::VALUE, RTFKeyword::VIEWKIND } },
    { "viewnobound"_ostr, { RTFControlType::FLAG, RTFKeyword::VIEWNOBOUND } },
    { "viewscale"_ostr, { RTFControlType::VALUE, RTFKeyword::VIEWSCALE, 100 } },
    { "viewzk"_ostr, { RTFControlType::VALUE, RTFKeyword::VIEWZK } },
    { "wbitmap"_ostr, { RTFControlType::VALUE, RTFKeyword::WBITMAP } },
    { "wbmbitspixel"_ostr, { RTFControlType::VALUE, RTFKeyword::WBMBITSPIXEL, 1 } },
    { "wbmplanes"_ostr, { RTFControlType::VALUE, RTFKeyword::WBMPLANES } },
    { "wbmwidthbyte"_ostr, { RTFControlType::VALUE, RTFKeyword::WBMWIDTHBYTE } },
    { "webhidden"_ostr, { RTFControlType::FLAG, RTFKeyword::WEBHIDDEN } },
    { "wgrffmtfilter"_ostr, { RTFControlType::DESTINATION, RTFKeyword::WGRFFMTFILTER } },
    { "widctlpar"_ostr, { RTFControlType::FLAG, RTFKeyword::WIDCTLPAR } },
    { "widowctrl"_ostr, { RTFControlType::FLAG, RTFKeyword::WIDOWCTRL } },
    { "windowcaption"_ostr, { RTFControlType::DESTINATION, RTFKeyword::WINDOWCAPTION } },
    { "wmetafile"_ostr, { RTFControlType::VALUE, RTFKeyword::WMETAFILE, 1 } },
    { "wpeqn"_ostr, { RTFControlType::FLAG, RTFKeyword::WPEQN } },
    { "wpjst"_ostr, { RTFControlType::FLAG, RTFKeyword::WPJST } },
    { "wpsp"_ostr, { RTFControlType::FLAG, RTFKeyword::WPSP } },
    { "wraparound"_ostr, { RTFControlType::FLAG, RTFKeyword::WRAPAROUND } },
    { "wrapdefault"_ostr, { RTFControlType::FLAG, RTFKeyword::WRAPDEFAULT } },
    { "wrapthrough"_ostr, { RTFControlType::FLAG, RTFKeyword::WRAPTHROUGH } },
    { "wraptight"_ostr, { RTFControlType::FLAG, RTFKeyword::WRAPTIGHT } },
    { "wraptrsp"_ostr, { RTFControlType::FLAG, RTFKeyword::WRAPTRSP } },
    { "writereservation"_ostr, { RTFControlType::DESTINATION, RTFKeyword::WRITERESERVATION } },
    { "writereservhash"_ostr, { RTFControlType::DESTINATION, RTFKeyword::WRITERESERVHASH } },
    { "wrppunct"_ostr, { RTFControlType::FLAG, RTFKeyword::WRPPUNCT } },
    { "xe"_ostr, { RTFControlType::DESTINATION, RTFKeyword::XE } },
    { "xef"_ostr, { RTFControlType::VALUE, RTFKeyword::XEF } },
    { "xform"_ostr, { RTFControlType::DESTINATION, RTFKeyword::XFORM } },
    { "xmlattr"_ostr, { RTFControlType::FLAG, RTFKeyword::XMLATTR } },
    { "xmlattrname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::XMLATTRNAME } },
    { "xmlattrns"_ostr, { RTFControlType::VALUE, RTFKeyword::XMLATTRNS } },
    { "xmlattrvalue"_ostr, { RTFControlType::DESTINATION, RTFKeyword::XMLATTRVALUE } },
    { "xmlclose"_ostr, { RTFControlType::DESTINATION, RTFKeyword::XMLCLOSE } },
    { "xmlname"_ostr, { RTFControlType::DESTINATION, RTFKeyword::XMLNAME } },
    { "xmlns"_ostr, { RTFControlType::VALUE, RTFKeyword::XMLNS } },
    { "xmlnstbl"_ostr, { RTFControlType::DESTINATION, RTFKeyword::XMLNSTBL } },
    { "xmlopen"_ostr, { RTFControlType::DESTINATION, RTFKeyword::XMLOPEN } },
    { "xmlsdttcell"_ostr, { RTFControlType::FLAG, RTFKeyword::XMLSDTTCELL } },
    { "xmlsdttpara"_ostr, { RTFControlType::FLAG, RTFKeyword::XMLSDTTPARA } },
    { "xmlsdttregular"_ostr, { RTFControlType::FLAG, RTFKeyword::XMLSDTTREGULAR } },
    { "xmlsdttrow"_ostr, { RTFControlType::FLAG, RTFKeyword::XMLSDTTROW } },
    { "xmlsdttunknown"_ostr, { RTFControlType::FLAG, RTFKeyword::XMLSDTTUNKNOWN } },
    { "yr"_ostr, { RTFControlType::VALUE, RTFKeyword::YR } },
    { "yts"_ostr, { RTFControlType::VALUE, RTFKeyword::YTS } },
    { "yxe"_ostr, { RTFControlType::FLAG, RTFKeyword::YXE } },
    { "zwbo"_ostr, { RTFControlType::SYMBOL, RTFKeyword::ZWBO } },
    { "zwj"_ostr, { RTFControlType::SYMBOL, RTFKeyword::ZWJ } },
    { "zwnbo"_ostr, { RTFControlType::SYMBOL, RTFKeyword::ZWNBO } },
    { "zwnj"_ostr, { RTFControlType::SYMBOL, RTFKeyword::ZWNJ } },
    { "flymaincnt"_ostr, { RTFControlType::DESTINATION, RTFKeyword::FLYMAINCNT } },
    { "flyvert"_ostr, { RTFControlType::VALUE, RTFKeyword::FLYVERT } },
    { "flyhorz"_ostr, { RTFControlType::VALUE, RTFKeyword::FLYHORZ } },
    { "flyanchor"_ostr, { RTFControlType::VALUE, RTFKeyword::FLYANCHOR } },
};

const RTFSymbol* getSymbolData(const OString& keyword)
{
    static const std::unordered_map map(std::begin(controlWords), std::end(controlWords));
    if (auto it = map.find(keyword); it != map.end())
        return &it->second;
    return nullptr;
}

constexpr auto mathControlWords = frozen::make_unordered_map<RTFKeyword, RTFMathSymbolData>(
    { { RTFKeyword::MOMATH, { M_TOKEN(oMath), Destination::MOMATH } },
      { RTFKeyword::MF, { M_TOKEN(f), Destination::MF } },
      { RTFKeyword::MFPR, { M_TOKEN(fPr), Destination::MFPR } },
      { RTFKeyword::MCTRLPR, { M_TOKEN(ctrlPr), Destination::MCTRLPR } },
      { RTFKeyword::MNUM, { M_TOKEN(num), Destination::MNUM } },
      { RTFKeyword::MDEN, { M_TOKEN(den), Destination::MDEN } },
      { RTFKeyword::MACC, { M_TOKEN(acc), Destination::MACC } },
      { RTFKeyword::MACCPR, { M_TOKEN(accPr), Destination::MACCPR } },
      { RTFKeyword::MBAR, { M_TOKEN(bar), Destination::MBAR } },
      { RTFKeyword::MBARPR, { M_TOKEN(barPr), Destination::MBARPR } },
      { RTFKeyword::ME, { M_TOKEN(e), Destination::ME } },
      { RTFKeyword::MD, { M_TOKEN(d), Destination::MD } },
      { RTFKeyword::MDPR, { M_TOKEN(dPr), Destination::MDPR } },
      { RTFKeyword::MFUNC, { M_TOKEN(func), Destination::MFUNC } },
      { RTFKeyword::MFUNCPR, { M_TOKEN(funcPr), Destination::MFUNCPR } },
      { RTFKeyword::MFNAME, { M_TOKEN(fName), Destination::MFNAME } },
      { RTFKeyword::MLIMLOW, { M_TOKEN(limLow), Destination::MLIMLOW } },
      { RTFKeyword::MLIMLOWPR, { M_TOKEN(limLowPr), Destination::MLIMLOWPR } },
      { RTFKeyword::MLIM, { M_TOKEN(lim), Destination::MLIM } },
      { RTFKeyword::MM, { M_TOKEN(m), Destination::MM } },
      { RTFKeyword::MMPR, { M_TOKEN(mPr), Destination::MMPR } },
      { RTFKeyword::MMR, { M_TOKEN(mr), Destination::MMR } },
      { RTFKeyword::MNARY, { M_TOKEN(nary), Destination::MNARY } },
      { RTFKeyword::MNARYPR, { M_TOKEN(naryPr), Destination::MNARYPR } },
      { RTFKeyword::MSUB, { M_TOKEN(sub), Destination::MSUB } },
      { RTFKeyword::MSUP, { M_TOKEN(sup), Destination::MSUP } },
      { RTFKeyword::MLIMUPP, { M_TOKEN(limUpp), Destination::MLIMUPP } },
      { RTFKeyword::MLIMUPPPR, { M_TOKEN(limUppPr), Destination::MLIMUPPPR } },
      { RTFKeyword::MGROUPCHR, { M_TOKEN(groupChr), Destination::MGROUPCHR } },
      { RTFKeyword::MGROUPCHRPR, { M_TOKEN(groupChrPr), Destination::MGROUPCHRPR } },
      { RTFKeyword::MBORDERBOX, { M_TOKEN(borderBox), Destination::MBORDERBOX } },
      { RTFKeyword::MBORDERBOXPR, { M_TOKEN(borderBoxPr), Destination::MBORDERBOXPR } },
      { RTFKeyword::MRAD, { M_TOKEN(rad), Destination::MRAD } },
      { RTFKeyword::MRADPR, { M_TOKEN(radPr), Destination::MRADPR } },
      { RTFKeyword::MDEG, { M_TOKEN(deg), Destination::MDEG } },
      { RTFKeyword::MSSUB, { M_TOKEN(sSub), Destination::MSSUB } },
      { RTFKeyword::MSSUBPR, { M_TOKEN(sSubPr), Destination::MSSUBPR } },
      { RTFKeyword::MSSUP, { M_TOKEN(sSup), Destination::MSSUP } },
      { RTFKeyword::MSSUPPR, { M_TOKEN(sSupPr), Destination::MSSUPPR } },
      { RTFKeyword::MSSUBSUP, { M_TOKEN(sSubSup), Destination::MSSUBSUP } },
      { RTFKeyword::MSSUBSUPPR, { M_TOKEN(sSubSupPr), Destination::MSSUBSUPPR } },
      { RTFKeyword::MSPRE, { M_TOKEN(sPre), Destination::MSPRE } },
      { RTFKeyword::MSPREPR, { M_TOKEN(sPrePr), Destination::MSPREPR } },
      { RTFKeyword::MBOX, { M_TOKEN(box), Destination::MBOX } },
      { RTFKeyword::MEQARR, { M_TOKEN(eqArr), Destination::MEQARR } } });
}

// static
std::string_view RTFTokenizer::toString(RTFKeyword nKeyword)
{
    for (auto & [ kw, data ] : controlWords)
        if (data.index == nKeyword)
            return kw;
    return {};
}

RTFTokenizer::RTFTokenizer(RTFListener& rImport, SvStream* pInStream,
                           uno::Reference<task::XStatusIndicator> const& xStatusIndicator)
    : m_rImport(rImport)
    , m_pInStream(pInStream)
    , m_xStatusIndicator(xStatusIndicator)
    , m_nGroup(0)
    , m_nLineNumber(0)
    , m_nLineStartPos(0)
    , m_nGroupStart(0)
{
}

RTFTokenizer::~RTFTokenizer() = default;

RTFError RTFTokenizer::resolveParse()
{
    SAL_INFO("writerfilter.rtf", __func__);
    char ch;
    RTFError ret;
    // for hex chars
    int b = 0;
    int count = 2;
    std::size_t nPercentSize = 0;
    sal_uInt64 nLastPos = 0;

    if (m_xStatusIndicator.is())
    {
        OUString sDocLoad(SvxResId(RID_SVXSTR_DOC_LOAD));

        sal_uInt64 const nCurrentPos = Strm().Tell();
        sal_uInt64 const nEndPos = nCurrentPos + Strm().remainingSize();
        m_xStatusIndicator->start(sDocLoad, nEndPos);
        nPercentSize = nEndPos / 100;

        nLastPos = nCurrentPos;
        m_xStatusIndicator->setValue(nLastPos);
    }

    while (Strm().ReadChar(ch), !Strm().eof())
    {
        //SAL_INFO("writerfilter", __func__ << ": parsing character '" << ch << "'");

        sal_uInt64 const nCurrentPos = Strm().Tell();
        if (m_xStatusIndicator.is() && nCurrentPos > (nLastPos + nPercentSize))
        {
            nLastPos = nCurrentPos;
            m_xStatusIndicator->setValue(nLastPos);
        }

        if (m_nGroup < 0)
            return RTFError::GROUP_UNDER;
        if (m_nGroup > 0 && m_rImport.getInternalState() == RTFInternalState::BIN)
        {
            ret = m_rImport.resolveChars(ch);
            if (ret != RTFError::OK)
                return ret;
        }
        else
        {
            switch (ch)
            {
                case '{':
                    m_nGroupStart = Strm().Tell() - 1;
                    ret = m_rImport.pushState();
                    if (ret != RTFError::OK)
                        return ret;
                    break;
                case '}':
                    ret = m_rImport.popState();
                    if (ret != RTFError::OK)
                        return ret;
                    if (m_nGroup == 0)
                    {
                        if (m_rImport.isSubstream())
                            m_rImport.finishSubstream();
                        return RTFError::OK;
                    }
                    break;
                case '\\':
                    ret = resolveKeyword();
                    if (ret != RTFError::OK)
                        return ret;
                    break;
                case 0x0d:
                    break; // ignore this
                case 0x0a:
                    m_nLineNumber++;
                    m_nLineStartPos = nCurrentPos;
                    break;
                default:
                    if (m_nGroup == 0)
                        return RTFError::CHAR_OVER;
                    if (m_rImport.getInternalState() == RTFInternalState::NORMAL)
                    {
                        ret = m_rImport.resolveChars(ch);
                        if (ret != RTFError::OK)
                            return ret;
                    }
                    else
                    {
                        SAL_INFO("writerfilter.rtf", __func__ << ": hex internal state");
                        // Assume that \'<number><junk> means \'0<number>.
                        if (rtl::isAsciiDigit(static_cast<unsigned char>(ch))
                            || (ch >= 'a' && ch <= 'f') || (ch >= 'A' && ch <= 'F'))
                        {
                            b = b << 4;
                            sal_Int8 parsed = msfilter::rtfutil::AsHex(ch);
                            if (parsed == -1)
                                return RTFError::HEX_INVALID;
                            b += parsed;
                        }
                        count--;
                        if (!count)
                        {
                            ret = m_rImport.resolveChars(b);
                            if (ret != RTFError::OK)
                                return ret;
                            count = 2;
                            b = 0;
                            m_rImport.setInternalState(RTFInternalState::NORMAL);
                        }
                    }
                    break;
            }
        }
    }

    if (m_nGroup < 0)
        return RTFError::GROUP_UNDER;
    if (m_nGroup > 0)
        return RTFError::GROUP_OVER;
    return RTFError::OK;
}

void RTFTokenizer::pushGroup() { m_nGroup++; }

void RTFTokenizer::popGroup() { m_nGroup--; }

RTFError RTFTokenizer::resolveKeyword()
{
    char ch;

    Strm().ReadChar(ch);
    if (Strm().eof())
        return RTFError::UNEXPECTED_EOF;

    if (!rtl::isAsciiAlpha(static_cast<unsigned char>(ch)))
    {
        // control symbols aren't followed by a space, so we can return here
        // without doing any SeekRel()
        return dispatchKeyword(OString(ch), false, 0);
    }
    OStringBuffer aBuf(32);
    while (rtl::isAsciiAlpha(static_cast<unsigned char>(ch)))
    {
        aBuf.append(ch);
        if (aBuf.getLength() > 32)
            // See RTF spec v1.9.1, page 7
            // A control word's name cannot be longer than 32 letters.
            throw io::BufferSizeExceededException();
        Strm().ReadChar(ch);
        if (Strm().eof())
        {
            ch = ' ';
            break;
        }
    }

    bool bNeg = false;
    if (ch == '-')
    {
        // in case we'll have a parameter, that will be negative
        bNeg = true;
        Strm().ReadChar(ch);
        if (Strm().eof())
            return RTFError::UNEXPECTED_EOF;
    }
    bool bParam = false;
    int nParam = 0;
    if (rtl::isAsciiDigit(static_cast<unsigned char>(ch)))
    {
        OStringBuffer aParameter;

        // we have a parameter
        bParam = true;
        while (rtl::isAsciiDigit(static_cast<unsigned char>(ch)))
        {
            aParameter.append(ch);
            Strm().ReadChar(ch);
            if (Strm().eof())
            {
                ch = ' ';
                break;
            }
        }
        nParam = o3tl::toInt32(aParameter);
        if (bNeg)
            nParam = -nParam;
    }
    if (ch != ' ')
        Strm().SeekRel(-1);
    OString aKeyword = aBuf.makeStringAndClear();
    return dispatchKeyword(aKeyword, bParam, nParam);
}

// static
const RTFMathSymbolData* RTFTokenizer::lookupMathKeyword(RTFKeyword nKeyword)
{
    if (auto it = mathControlWords.find(nKeyword); it != mathControlWords.end())
        return &it->second;
    return nullptr;
}

RTFError RTFTokenizer::dispatchKeyword(OString const& rKeyword, bool bParam, int nParam)
{
    if (m_rImport.getDestination() == Destination::SKIP)
    {
        // skip binary data explicitly, to not trip over rtf markup
        // control characters
        if (rKeyword == "bin" && nParam > 0)
            Strm().SeekRel(nParam);
        return RTFError::OK;
    }
    SAL_INFO("writerfilter.rtf", __func__ << ": keyword '\\" << rKeyword << "' with param? "
                                          << (bParam ? 1 : 0) << " param val: '"
                                          << (bParam ? nParam : 0) << "'");
    const RTFSymbol* pSymbol = getSymbolData(rKeyword);
    if (!pSymbol)
    {
        SAL_INFO("writerfilter.rtf", __func__ << ": unknown keyword '\\" << rKeyword << "'");
        RTFSkipDestination aSkip(m_rImport);
        aSkip.setParsed(false);
        return RTFError::OK;
    }

    RTFError ret;
    switch (pSymbol->controlType)
    {
        case RTFControlType::FLAG:
            // flags ignore any parameter by definition
            ret = m_rImport.dispatchFlag(pSymbol->index);
            if (ret != RTFError::OK)
                return ret;
            break;
        case RTFControlType::DESTINATION:
            // same for destinations
            ret = m_rImport.dispatchDestination(pSymbol->index);
            if (ret != RTFError::OK)
                return ret;
            break;
        case RTFControlType::SYMBOL:
            // and symbols
            ret = m_rImport.dispatchSymbol(pSymbol->index);
            if (ret != RTFError::OK)
                return ret;
            break;
        case RTFControlType::TOGGLE:
            ret = m_rImport.dispatchToggle(pSymbol->index, bParam, nParam);
            if (ret != RTFError::OK)
                return ret;
            break;
        case RTFControlType::VALUE:
            if (!bParam)
                nParam = pSymbol->defValue;
            ret = m_rImport.dispatchValue(pSymbol->index, nParam);
            if (ret != RTFError::OK)
                return ret;
            break;
    }

    return RTFError::OK;
}

OUString RTFTokenizer::getPosition()
{
    return OUString::number(m_nLineNumber + 1) + ","
           + OUString::number(Strm().Tell() - m_nLineStartPos + 1);
}

} // namespace writerfilter::rtftok

/* vim:set shiftwidth=4 softtabstop=4 expandtab: */
