#ifndef GEN_UI_GFX_X_DAMAGE_H_
#define GEN_UI_GFX_X_DAMAGE_H_

#include <array>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <vector>

#include "base/component_export.h"
#include "base/memory/ref_counted_memory.h"
#include "base/memory/scoped_refptr.h"
#include "base/optional.h"
#include "base/files/scoped_file.h"
#include "ui/gfx/x/xproto_types.h"
#include "xfixes.h"
#include "xproto.h"

namespace x11 {

class Connection;

class COMPONENT_EXPORT(X11) Damage {
  public:
  static constexpr unsigned major_version = 1;
  static constexpr unsigned minor_version = 1;

  Damage(Connection* connection,
      const x11::QueryExtensionReply& info);

  uint8_t present() const {
    return info_.present;
  }
  uint8_t major_opcode() const {
    return info_.major_opcode;
  }
  uint8_t first_event() const {
    return info_.first_event;
  }
  uint8_t first_error() const {
    return info_.first_error;
  }

  Connection* connection() const { return connection_; }

  enum class DamageId : uint32_t {};

  enum class ReportLevel : int {
    RawRectangles = 0,
    DeltaRectangles = 1,
    BoundingBox = 2,
    NonEmpty = 3,
  };

  struct BadDamageError {
    uint16_t sequence{};
  };

  struct NotifyEvent {
    static constexpr int type_id = 1;
    static constexpr uint8_t opcode = 0;
    bool send_event{};
    ReportLevel level{};
    uint16_t sequence{};
    Drawable drawable{};
    DamageId damage{};
    Time timestamp{};
    Rectangle area{};
    Rectangle geometry{};

    x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&drawable); }
  };

  struct QueryVersionRequest {
    uint32_t client_major_version{};
    uint32_t client_minor_version{};
  };

  struct QueryVersionReply {
    uint16_t sequence{};
    uint32_t major_version{};
    uint32_t minor_version{};
  };

  using QueryVersionResponse = Response<QueryVersionReply>;

  Future<QueryVersionReply> QueryVersion(
      const QueryVersionRequest& request);

  struct CreateRequest {
    DamageId damage{};
    Drawable drawable{};
    ReportLevel level{};
  };

  using CreateResponse = Response<void>;

  Future<void> Create(
      const CreateRequest& request);

  struct DestroyRequest {
    DamageId damage{};
  };

  using DestroyResponse = Response<void>;

  Future<void> Destroy(
      const DestroyRequest& request);

  struct SubtractRequest {
    DamageId damage{};
    XFixes::Region repair{};
    XFixes::Region parts{};
  };

  using SubtractResponse = Response<void>;

  Future<void> Subtract(
      const SubtractRequest& request);

  struct AddRequest {
    Drawable drawable{};
    XFixes::Region region{};
  };

  using AddResponse = Response<void>;

  Future<void> Add(
      const AddRequest& request);

  private:
  x11::Connection* const connection_;
  x11::QueryExtensionReply info_{};
};

}  // namespace x11

inline constexpr x11::Damage::ReportLevel operator|(
    x11::Damage::ReportLevel l, x11::Damage::ReportLevel r) {
  using T = std::underlying_type_t<x11::Damage::ReportLevel>;
  return static_cast<x11::Damage::ReportLevel>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Damage::ReportLevel operator&(
    x11::Damage::ReportLevel l, x11::Damage::ReportLevel r) {
  using T = std::underlying_type_t<x11::Damage::ReportLevel>;
  return static_cast<x11::Damage::ReportLevel>(
      static_cast<T>(l) & static_cast<T>(r));
}


#endif  // GEN_UI_GFX_X_DAMAGE_H_
