// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
#include "webglRenderer.h"

const char *webglRenderer =
"/**\n"
" * Create a renderer object working fully in WebGL\n"
" * Here is a sample set of command to illustrate how to use this renderer\n"
" *\n"
" * var renderer = new WebGLRenderer('rendererId','http://localhost:8080/ParaViewWebService')\n"
" * renderer.init(sessionId, viewId);\n"
" * renderer.bindToElementId('containerID'); // => Add a WebGL canvas inside a div tag id 'containerID'\n"
" * renderer.start();\n"
" *\n"
" * renderer.init(otherSessionId, otherViewId);\n"
" * renderer.view.width = '100';\n"
" * renderer.view.height = '400';\n"
" * renderer.setSize('100', '400');\n"
" *\n"
" * renderer.unbindToElementId('containerID');\n"
" */\n"
"\n"
"// Global object to keep track of WebGL renderers\n"
"var webglRenderers = new Object();\n"
"\n"
"window.requestAnimFrame = (function(){\n"
"  return  window.requestAnimationFrame       ||\n"
"          window.webkitRequestAnimationFrame ||\n"
"          window.mozRequestAnimationFrame    ||\n"
"          window.oRequestAnimationFrame      ||\n"
"          window.msRequestAnimationFrame     ||\n"
"          function(/* function */ callback, /* DOMElement */ element){\n"
"            window.setTimeout(callback, 1000 / 60);\n"
"          };\n"
"})();\n"
"\n"
"function WebGLRenderer(rendererId, coreServiceURL) {\n"
"    this.baseURL = coreServiceURL + \"/WebGL\";\n"
"    this.rendererId = rendererId;\n"
"    this.sessionId = \"\";\n"
"    this.viewId = \"\";\n"
"    this.nbError = 0;\n"
"    this.localTimeStamp = 0;\n"
"    this.offlineMode = false;\n"
"    this.setServerMode(false);\n"
"    this.forceSquareSize = false;\n"
"\n"
"    this.view = new Object();\n"
"    this.view.width = 100;\n"
"    this.view.height = 100;\n"
"    this.view.id = rendererId;\n"
"    this.view.alt = \"ParaView Renderer\";\n"
"\n"
"    //Default Shaders\n"
"    this.view.shaderfs = document.createElement(\"script\");\n"
"    this.view.shaderfs.id = \"shader-fs\";\n"
"    this.view.shaderfs.type = \"x-shader/x-fragment\";\n"
"    this.view.shaderfs.innerHTML = \"\\\n"
"    #ifdef GL_ES\\n\\\n"
"    precision highp float;\\n\\\n"
"    #endif\\n\\\n"
"    uniform bool uIsLine;\\\n"
"    varying vec4 vColor;\\\n"
"    varying vec4 vTransformedNormal;\\\n"
"    varying vec4 vPosition;\\\n"
"    void main(void) {\\\n"
"        float directionalLightWeighting1 = max(dot(normalize(vTransformedNormal.xyz), vec3(0.0, 0.0, 1.0)), 0.0); \\\n"
"        float directionalLightWeighting2 = max(dot(normalize(vTransformedNormal.xyz), vec3(0.0, 0.0, -1.0)), 0.0);\\\n"
"        vec3 lightWeighting = max(vec3(1.0, 1.0, 1.0) * directionalLightWeighting1, vec3(1.0, 1.0, 1.0) * directionalLightWeighting2);\\\n"
"        if (uIsLine == false){\\\n"
"          gl_FragColor = vec4(vColor.rgb * lightWeighting, vColor.a);\\\n"
"        } else {\\\n"
"          gl_FragColor = vColor*vec4(1.0, 1.0, 1.0, 1.0);\\\n"
"        }\\\n"
"    }\";\n"
"    this.view.shadervs = document.createElement(\"script\");\n"
"    this.view.shadervs.id = \"shader-vs\";\n"
"    this.view.shadervs.type = \"x-shader/x-vertex\";\n"
"    this.view.shadervs.innerHTML = \"\\\n"
"    attribute vec3 aVertexPosition;\\\n"
"    attribute vec4 aVertexColor;\\\n"
"    attribute vec3 aVertexNormal;\\\n"
"    uniform mat4 uMVMatrix;\\\n"
"    uniform mat4 uPMatrix;\\\n"
"    uniform mat4 uNMatrix;\\\n"
"    varying vec4 vColor;\\\n"
"    varying vec4 vPosition;\\\n"
"    varying vec4 vTransformedNormal;\\\n"
"    void main(void) {\\\n"
"        vPosition = uMVMatrix * vec4(aVertexPosition, 1.0);\\\n"
"        gl_Position = uPMatrix * vPosition;\\\n"
"        vTransformedNormal = uNMatrix * vec4(aVertexNormal, 1.0);\\\n"
"        vColor = aVertexColor;\\\n"
"    }\";\n"
"\n"
"    // Point Shaders\n"
"    this.view.shaderfsPoint = document.createElement(\"script\");\n"
"    this.view.shaderfsPoint.id = \"shader-fs-Point\";\n"
"    this.view.shaderfsPoint.type = \"x-shader/x-fragment\";\n"
"    this.view.shaderfsPoint.innerHTML = \"\\\n"
"    #ifdef GL_ES\\n\\\n"
"    precision highp float;\\n\\\n"
"    #endif\\n\\\n"
"    varying vec4 vColor;\\\n"
"    void main(void) {\\\n"
"        gl_FragColor = vColor;\\\n"
"    }\";\n"
"    this.view.shadervsPoint = document.createElement(\"script\");\n"
"    this.view.shadervsPoint.id = \"shader-vs-Point\";\n"
"    this.view.shadervsPoint.type = \"x-shader/x-vertex\";\n"
"    this.view.shadervsPoint.innerHTML = \"\\\n"
"    attribute vec3 aVertexPosition;\\\n"
"    attribute vec4 aVertexColor;\\\n"
"    uniform mat4 uMVMatrix;\\\n"
"    uniform mat4 uPMatrix;\\\n"
"    uniform mat4 uNMatrix;\\\n"
"    uniform float uPointSize;\\\n"
"    varying vec4 vColor;\\\n"
"    void main(void) {\\\n"
"        vec4 pos = uMVMatrix * vec4(aVertexPosition, 1.0);\\\n"
"        gl_Position = uPMatrix * pos;\\\n"
"        vColor = aVertexColor*vec4(1.0, 1.0, 1.0, 1.0);\\\n"
"        gl_PointSize = uPointSize;\\\n"
"    }\";\n"
"\n"
"    //\n"
"    this.canvasName = \"glcanvas\" + rendererId;\n"
"    this.view.html = '<div><canvas id=\"' + this.canvasName + '\" style=\"border: none; overflow: hidden;';\n"
"    if (this.forceSquareSize == true) this.view.html += ' position: absolute;';\n"
"    this.view.html += ' left:-1px; top:-1px; right:0px; z-index=0;\" width=\"' + this.view.width\n"
"    + '\" height=\"' + this.view.height + '\"onmousedown=\"handleMouseDown(event,\\''+rendererId+'\\')\"\\\n"
"     onmousemove=\"handleMouseMove(event,\\''+rendererId+'\\')\" onmouseup=\"handleMouseUp(event,\\'' + rendererId\n"
"     + '\\')\" oncontextmenu=\"consumeEvent(event)\"> Your browser doesn\\'t appear to support the HTML5 <code>\\\n"
"     &lt;canvas&gt;</code> element.</canvas>';\n"
"\n"
"     this.view.html += '<canvas id=\"' + this.canvasName + 'Widget\" style=\"position: absolute; left:-1px; top:-1px; z-index:1;';\n"
"     if(this.forceSquareSize == true) this.view.html += 'position: absolute;';\n"
"     this.view.html += '\" width=\"' + this.view.width + '\" height=\"' + this.view.height +\n"
"     '\"onmousedown=\"handleMouseDown(event,\\''+rendererId+'\\')\" onmousemove=\"handleMouseMove(event,\\''+rendererId+'\\')\"\\\n"
"     onmouseup=\"handleMouseUp(event,\\'' + rendererId + '\\')\" oncontextmenu=\"consumeEvent(event)\"></canvas></div>';\n"
"    this.fps = 0;\n"
"\n"
"    // Register in global var\n"
"    webglRenderers[rendererId] = this;\n"
"}\n"
"\n"
"WebGLRenderer.prototype.bindToElementId = function (elementId) {\n"
"    this.oldInnerHTML = document.getElementById(elementId).innerHTML;\n"
"    document.getElementById(elementId).innerHTML = this.view.html;\n"
"\n"
"    document.getElementById(elementId).appendChild(this.view.shaderfs);\n"
"    document.getElementById(elementId).appendChild(this.view.shadervs);\n"
"    document.getElementById(elementId).appendChild(this.view.shaderfsPoint);\n"
"    document.getElementById(elementId).appendChild(this.view.shadervsPoint);\n"
"}\n"
"\n"
"WebGLRenderer.prototype.unbindToElementId = function (elementId) {\n"
"  document.getElementById(elementId).innerHTML = this.oldInnerHTML;\n"
"  clearTimeout(this.drawInterval);\n"
"  if (typeof(paraview) != \"undefined\") paraview.updateConfiguration(true, \"JPEG\", \"NO\");\n"
"}\n"
"\n"
"WebGLRenderer.prototype.setOfflineMode = function (mode) {\n"
"  this.offlineMode = mode;\n"
"  this.requestMetaData();\n"
"}\n"
"\n"
"WebGLRenderer.prototype.bindToElement = function (element) {\n"
"    this.oldInnerHTML = element.innerHTML;\n"
"    element.innerHTML = this.view.html;\n"
"\n"
"    element.appendChild(this.view.shaderfs);\n"
"    element.appendChild(this.view.shadervs);\n"
"    element.appendChild(this.view.shaderfsPoint);\n"
"    element.appendChild(this.view.shadervsPoint);\n"
"}\n"
"\n"
"WebGLRenderer.prototype.unbindToElement = function (element) {\n"
"  element.innerHTML = this.oldInnerHTML;\n"
"  clearTimeout(this.drawInterval);\n"
"  if (typeof(paraview) != \"undefined\") paraview.updateConfiguration(true, \"JPEG\", \"NO\");\n"
"}\n"
"\n"
"WebGLRenderer.prototype.init = function (sessionId, viewId) {\n"
"    this.sessionId = sessionId;\n"
"    this.viewId = viewId;\n"
"}\n"
"\n"
"WebGLRenderer.prototype.start = function(metadata, objects) {\n"
"    if (typeof(renderers) == \"undefined\"){\n"
"      renderers = Object();\n"
"      renderers.current = this;\n"
"    }\n"
"    if (typeof(paraview) != \"undefined\") paraview.updateConfiguration(true, \"JPEG\", \"WebGL\");\n"
"    canvas = document.getElementById(this.canvasName);\n"
"    canvas.width = this.view.width;\n"
"    canvas.height = this.view.height;\n"
"\n"
"    this.hasSceneChanged = true;        //Scene Graph Has Changed\n"
"    this.oldCamPos = null;              //Last Known Camera Position\n"
"    this.sceneJSON = null;              //Current Scene Graph\n"
"    this.up = [];\n"
"    this.right = [];\n"
"    this.z_dir = [];\n"
"    this.objects = [];                  //List of objects\n"
"    this.nbErrors = 0;                  //Number of Errors\n"
"    this.background = null;             //Background object: mesh, normals, colors, render\n"
"    this.interactionRatio = 2;\n"
"    this.requestInterval = 250;         //Frequency it request new data from the server\n"
"    this.requestOldInterval = 250;      //\n"
"    this.updateInterval = 100;          //Frequency the server will be updated\n"
"    this.fps = 0;\n"
"    this.frames = 0;\n"
"    this.lastTime = new Date().getTime();\n"
"    this.view.aspectRatio = 1;\n"
"    this.lookAt = [0,0,0,0,1,0,0,0,1];\n"
"    this.offlineMode = !(typeof(metadata)==\"undefined\" || typeof(objects)==\"undefined\");\n"
"\n"
"    this.cachedObjects = [];            //List of Cached Objects\n"
"    this.isCaching = false;             //Is Caching or Not\n"
"\n"
"    this.processQueue = [];             //List of process to be executed\n"
"\n"
"    this.objScale = 1.0;                //Scale applied locally in the scene\n"
"    this.translation = [0.0, 0.0, 0.0]; //Translation\n"
"    this.rotMatrix = mat4.create();     //Rotation Matrix\n"
"    mat4.identity(this.rotMatrix);\n"
"    this.rotMatrix2 = mat4.create(this.rotMatrix);\n"
"\n"
"    this.mouseDown = false;\n"
"    this.lastMouseX = 0;\n"
"    this.lastMouseY = 0;\n"
"\n"
"    this.mvMatrix = mat4.create(this.rotMatrix);\n"
"    this.pMatrix = mat4.create(this.rotMatrix);\n"
"\n"
"    // Initialize the GL context\n"
"    this.gl = null;\n"
"    try {\n"
"      this.gl = canvas.getContext(\"experimental-webgl\");\n"
"      this.gl.viewportWidth = this.view.width;\n"
"      this.gl.viewportHeight = this.view.height;\n"
"    } catch(e) {}\n"
"\n"
"    if (this.gl) {\n"
"      this.gl.clearColor(0.0, 0.0, 0.0, 1.0);\n"
"      this.gl.clearDepth(1.0);\n"
"      this.gl.enable(this.gl.DEPTH_TEST);\n"
"      this.gl.depthFunc(this.gl.LEQUAL);\n"
"\n"
"      this.gl.blendFunc(this.gl.SRC_ALPHA, this.gl.ONE_MINUS_SRC_ALPHA);\n"
"\n"
"      this.initShaders();\n"
"\n"
"      this.ctx2d = document.getElementById(this.canvasName + \"Widget\").getContext('2d');\n"
"      // Set up to draw the scene periodically.\n"
"      this.drawInterval = requestAnimFrame(new Function(\"webglRenderers['\" + this.view.id + \"'].drawScene();\"));\n"
"\n"
"      if (!this.offlineMode){\n"
"        this.requestMetaData();\n"
"        this.updateCamera();\n"
"      } else {\n"
"        this.sceneJSON = JSON.parse(metadata);\n"
"\n"
"        for(aw=0; aw<objects.length-1; aw++){\n"
"          obj = new Object();\n"
"          obj.data = objects[aw];\n"
"          obj.hasTransparency = this.sceneJSON.Objects[aw].transparency;\n"
"          obj.layer = this.sceneJSON.Objects[aw].layer;\n"
"          obj.render = function(){};\n"
"          this.processQueue[aw] = obj;\n"
"          this.objects[this.objects.length] = obj;\n"
"        }\n"
"      }\n"
"    } else {\n"
"      canvas.parentNode.innerHTML = \"<table width=100% height=100%><tr><td align=center>\\\n"
"      Sorry, your browser do not support WebGL.<br> For more information visit the website\\\n"
"      <a href='http://get.webgl.org/' target='_blank'>http://get.webgl.org/</a></td></tr></table>\";\n"
"    }\n"
"}\n"
"\n"
"WebGLRenderer.prototype.setForceSquareSize = function(b){\n"
"this.forceSquareSize = b;\n"
"}\n"
"\n"
"WebGLRenderer.prototype.getPageX = function(){\n"
"    var location = 0;\n"
"    var node = document.getElementById(this.canvasName);\n"
"    while(node) {\n"
"        location += node.offsetLeft;\n"
"        node = node.offsetParent;\n"
"    }\n"
"    return location;\n"
"}\n"
"\n"
"WebGLRenderer.prototype.getPageY = function(){\n"
"    var location = 0;\n"
"    var node = document.getElementById(this.canvasName);\n"
"    while(node) {\n"
"        location += node.offsetTop;\n"
"        node = node.offsetParent;\n"
"    }\n"
"    return location;\n"
"}\n"
"\n"
"WebGLRenderer.prototype.setServerMode = function(mode){\n"
"  if (typeof(this.interaction) == \"undefined\"){\n"
"    this.interaction = new Object();\n"
"    this.interaction.lastRealEvent = 0;\n"
"    this.interaction.needUp = false;\n"
"    this.interaction.lastEvent = 0;\n"
"    this.interaction.isDragging = false;\n"
"    this.interaction.action = \" \";\n"
"    this.interaction.keys = \" \";\n"
"    this.interaction.button = 0;\n"
"    this.interaction.x = 0;\n"
"    this.interaction.y = 0;\n"
"    this.interaction.xOrigin = 0;\n"
"    this.interaction.yOrigin = 0;\n"
"    this.serverMode = false;\n"
"  }\n"
"  if (this.serverMode == mode) return;\n"
"  this.serverMode = mode;\n"
"  if (!this.serverMode){\n"
"    this.updateId = setTimeout(\"webglRenderers[\\'\" + this.view.id + \"\\'].updateCamera()\", this.updateInterval);\n"
"  }\n"
"  canvas = document.getElementById(this.canvasName);\n"
"  canvasWidget = document.getElementById(this.canvasName + \"Widget\");\n"
"  if (this.serverMode){\n"
"    this.requestOldInterval = this.requestInterval;\n"
"    this.requestInterval = 50;\n"
"    canvas.setAttribute(\"onmousedown\", \"mouseServerInt('\"+this.view.id+\"','\"+this.sessionId+\"','\"+this.viewId+\"','down',event)\");\n"
"    canvas.setAttribute(\"onmousemove\", \"mouseServerInt('\"+this.view.id+\"','\"+this.sessionId+\"','\"+this.viewId+\"','move',event)\");\n"
"    canvas.setAttribute(\"onmouseup\"  , \"mouseServerInt('\"+this.view.id+\"','\"+this.sessionId+\"','\"+this.viewId+\"','up',event)\");\n"
"    canvasWidget.setAttribute(\"onmousedown\", \"mouseServerInt('\"+this.view.id+\"','\"+this.sessionId+\"','\"+this.viewId+\"','down',event)\");\n"
"    canvasWidget.setAttribute(\"onmousemove\", \"mouseServerInt('\"+this.view.id+\"','\"+this.sessionId+\"','\"+this.viewId+\"','move',event)\");\n"
"    canvasWidget.setAttribute(\"onmouseup\"  , \"mouseServerInt('\"+this.view.id+\"','\"+this.sessionId+\"','\"+this.viewId+\"','up',event)\");\n"
"  } else {\n"
"    this.requestInterval = this.requestOldInterval;\n"
"    canvas.setAttribute(\"onmousedown\", \"handleMouseDown(event,'\" + this.rendererId + \"')\");\n"
"    canvas.setAttribute(\"onmousemove\", \"handleMouseMove(event,'\" + this.rendererId + \"')\");\n"
"    canvas.setAttribute(\"onmouseup\"  , \"handleMouseUp(event,'\" + this.rendererId + \"')\");\n"
"    canvasWidget.setAttribute(\"onmousedown\", \"handleMouseDown(event,'\" + this.rendererId + \"')\");\n"
"    canvasWidget.setAttribute(\"onmousemove\", \"handleMouseMove(event,'\" + this.rendererId + \"')\");\n"
"    canvasWidget.setAttribute(\"onmouseup\"  , \"handleMouseUp(event,'\" + this.rendererId + \"')\");\n"
"  }\n"
"  canvas.setAttribute(\"oncontextmenu\", \"consumeEvent(event)\");\n"
"}\n"
"\n"
"WebGLRenderer.prototype.setSize = function(width, height) {\n"
"    width = parseFloat(width);\n"
"    height = parseFloat(height);\n"
"    w = width;\n"
"    h = height;\n"
"    this.view.aspectRatio = width/height;\n"
"    if(this.forceSquareSize){\n"
"      if (width > height) height = width;\n"
"      else width = height;\n"
"    }\n"
"    this.view.width = width;\n"
"    this.view.height = height;\n"
"    canvas = document.getElementById(this.canvasName);\n"
"    canvasWidget = document.getElementById(this.canvasName + \"Widget\");\n"
"    if (canvas){\n"
"      canvas.width = this.view.width;\n"
"      canvas.height = this.view.height;\n"
"      canvasWidget.width = this.view.width;\n"
"      canvasWidget.height = this.view.height;\n"
"      if (typeof(this.gl) != \"undefined\" && this.gl != null){\n"
"        if (!this.offlineMode) updateRendererSize(this.sessionId, this.viewId, width, height);\n"
"          this.gl.viewportWidth = this.view.width;\n"
"          this.gl.viewportHeight = this.view.height;\n"
"        }\n"
"        left = 0; tt = 0;\n"
"        if (this.forceSquareSize){\n"
"          left = Math.round((w-this.view.width)/2);\n"
"          tt = Math.round((h-this.view.height)/2);\n"
"        }\n"
"        this.view.left = left;\n"
"        this.view.top = top;\n"
"        if(this.forceSquareSize == true){\n"
"          canvas.setAttribute(\"style\", \"position: absolute; overflow: hidden; left: \" + left + \"px; top: \" + tt + \"px; right: 0px; z-index:0;\");\n"
"          canvasWidget.setAttribute(\"style\", \"position: absolute; overflow: hidden; left: \" + left + \"px; top: \" + tt + \"px; right: 0px; z-index:1;\");\n"
"        } else {\n"
"          canvas.setAttribute(\"style\", \"overflow: hidden; left: \" + left + \"px; top: \" + tt + \"px; right: 0px; z-index:0;\");\n"
"          canvasWidget.setAttribute(\"style\", \"position: absolute; overflow: hidden; left: \" + left + \"px; top: \" + tt + \"px; right: 0px; z-index:1;\");\n"
"        }\n"
"    }\n"
"}\n"
"\n"
"WebGLRenderer.prototype.requestMetaData = function() {\n"
"  if (this.mouseDown || renderers.current != this) return;\n"
"  if (this.offlineMode) return;\n"
"\n"
"  interval = this.requestInterval;\n"
"  if (this.serverMode) interval = interval/2;\n"
"  this.timer = setTimeout(\"webglRenderers[\\'\" + this.view.id + \"\\'].requestMetaData()\", interval);\n"
"  var request = new XMLHttpRequest();\n"
"  request.requester = this;\n"
"  filename = this.baseURL + \"?sid=\" + this.sessionId + \"&vid=\" + this.viewId + \"&q=meta\";\n"
"  try {\n"
"    request.open(\"GET\", filename, false);\n"
"    request.overrideMimeType('text/plain; charset=x-user-defined');\n"
"    request.onreadystatechange = function() {\n"
"      if(this.requester.mouseDown) return;\n"
"      if (request.status != 200) this.requester.nbErrors++\n"
"      else if (request.readyState == 4) {\n"
"        aux = JSON.parse(request.responseText);\n"
"        this.requester.hasSceneChanged = JSON.stringify(aux)!=JSON.stringify(this.requester.sceneJSON);\n"
"        this.requester.sceneJSON = JSON.parse(request.responseText);\n"
"        if (this.requester.hasSceneChanged) this.requester.updateScene();\n"
"      }\n"
"    }\n"
"  request.send();\n"
"  } catch (e) {\n"
"    this.nbErrors++;\n"
"  }\n"
"}\n"
"\n"
"WebGLRenderer.prototype.updateScene = function(){\n"
"  if (typeof(this.sceneJSON) == \"undefined\" || this.sceneJSON == null) return;\n"
"  c1 = [0,0,0];\n"
"  c2 = [0,0,0];\n"
"  for(l=0; l<this.sceneJSON.Renderers.length; l++){\n"
"    if(this.sceneJSON.Renderers[l].layer==0){\n"
"      this.lookAt = this.sceneJSON.Renderers[l].LookAt;\n"
"      c1 = this.sceneJSON.Renderers[l].Background1;\n"
"      if (typeof(this.sceneJSON.Renderers[l].Background2) != \"undefined\") c2 = this.sceneJSON.Renderers[l].Background2;\n"
"    }\n"
"  }\n"
"  this.initBackground(c1, c2);\n"
"  if (JSON.stringify(this.oldCamPos)!=JSON.stringify(this.lookAt)){\n"
"    this.translation = [0.0, 0.0, 0.0];\n"
"    this.objScale = 1.0;\n"
"    mat4.identity(this.rotMatrix);\n"
"\n"
"    this.up = [this.lookAt[4], this.lookAt[5], this.lookAt[6]];\n"
"    this.z_dir = [this.lookAt[1]-this.lookAt[7],\n"
"                  this.lookAt[2]-this.lookAt[8],\n"
"                  this.lookAt[3]-this.lookAt[9]];\n"
"    vec3.normalize(this.z_dir, this.z_dir);\n"
"    vec3.cross(this.z_dir, this.up, this.right);\n"
"  }\n"
"  this.oldCamPos = this.lookAt;\n"
"  var aux = [];\n"
"  intAtServer = false;\n"
"  if (!this.offlineMode){\n"
"    for(w=0; w<this.objects.length; w++){\n"
"      for(j=0; j<this.sceneJSON.Objects.length; j++){\n"
"        if (this.objects[w].md5 == this.sceneJSON.Objects[j].md5 && this.objects[w].id == this.sceneJSON.Objects[j].id){\n"
"          aux[aux.length] = this.objects[w];\n"
"        }\n"
"      }\n"
"    }\n"
"    this.objects = aux;\n"
"\n"
"    for(w=0; w<this.sceneJSON.Objects.length; w++){\n"
"      foundit = false;\n"
"\n"
"      if (this.isCaching){\n"
"        for(j=0; j<this.cachedObjects.length; j++)\n"
"          if (this.cachedObjects[j].md5==this.sceneJSON.Objects[w].md5 &&\n"
"              this.cachedObjects[j].id==this.sceneJSON.Objects[w].id){\n"
"            this.objects[this.objects.length] = this.cachedObjects[j];\n"
"            foundit = true;\n"
"          }\n"
"      }\n"
"      if (!foundit){\n"
"        for(k=0; k<this.sceneJSON.Objects[w].parts; k++){\n"
"          foundit = false;\n"
"          for(j=0; j<this.objects.length; j++){\n"
"            if (this.objects[j].md5==this.sceneJSON.Objects[w].md5 &&\n"
"              this.objects[j].id==this.sceneJSON.Objects[w].id && this.objects[j].part==k+1 )\n"
"              foundit=true;\n"
"            }\n"
"            if(!foundit) this.requestObject(this.sessionId, this.sceneJSON.id, this.sceneJSON.Objects[w].md5,\n"
"                                      k+1, this.sceneJSON.Objects[w].id, this.sceneJSON.Objects[w].transparency, this.sceneJSON.Objects[w].layer);\n"
"          }\n"
"      }\n"
"      if (this.sceneJSON.Objects[w].interactAtServer==1) intAtServer = true;\n"
"    }\n"
"  }\n"
"  this.hasSceneChanged = false;\n"
"  this.setServerMode(intAtServer);\n"
"}\n"
"\n"
"WebGLRenderer.prototype.requestObject = function(sid, vid, md5, part, id, hastransparency, layer){\n"
"  if (this.offlineMode) return;\n"
"  var request = new XMLHttpRequest();\n"
"  request.requester = this;\n"
"  filename = this.baseURL + \"?sid=\" + sid + \"&vid=\" + vid + \"&hash=\" + md5 + \"&part=\" + part + \"&q=mesh&id=\" + id;\n"
"  try {\n"
"    request.open(\"GET\", filename, false);\n"
"    request.overrideMimeType('text/plain; charset=x-user-defined');\n"
"    request.onreadystatechange = function() {\n"
"      if (request.status != 200) this.requester.nbErrors++\n"
"      else if (request.readyState == 4) {\n"
"        foundit = -1;\n"
"        for (i=0; i<this.requester.objects.length; i++)\n"
"          if (this.requester.objects[i].md5 == md5 && this.requester.objects[i].part == part\n"
"              && this.requester.objects[i].id == id) foundit = i;\n"
"        if (foundit == -1){\n"
"          foundit = this.requester.objects.length;\n"
"          this.requester.objects.length++;\n"
"        }\n"
"        this.requester.objects[foundit] = new Object();\n"
"        this.requester.objects[foundit].md5 = md5;    //hash\n"
"        this.requester.objects[foundit].part = part;  //part\n"
"        this.requester.objects[foundit].sid = sid;    //scene id\n"
"        this.requester.objects[foundit].vid = vid;    //view id\n"
"        this.requester.objects[foundit].id = id;      //object id\n"
"        this.requester.objects[foundit].data = request.responseText;\n"
"        this.requester.objects[foundit].hasTransparency = hastransparency;\n"
"        this.requester.objects[foundit].layer = layer;\n"
"        this.requester.objects[foundit].render = function(){};\n"
"        this.requester.processQueue[this.requester.processQueue.length] = this.requester.objects[foundit];\n"
"        this.requester.cachedObjects[this.requester.cachedObjects.length] = this.requester.objects[foundit];\n"
"      }\n"
"    }\n"
"    request.send();\n"
"  } catch (e){\n"
"    this.nbErrors++;\n"
"  }\n"
"}\n"
"\n"
"WebGLRenderer.prototype.parseObject = function(obj){\n"
"  var ss = []; pos = 0;\n"
"  for(i=0; i<obj.data.length; i++) ss[i] = obj.data.charCodeAt(i) & 0xff;\n"
"\n"
"  size = (ss[pos++]) + (ss[pos++] << 8) + (ss[pos++] << 16) + (ss[pos++] << 24);\n"
"  type = String.fromCharCode(ss[pos++]);\n"
"  obj.type = type;\n"
"  obj.father = this;\n"
"\n"
"  if (type == 'L'){\n"
"    obj.numberOfPoints = (ss[pos++]) + (ss[pos++] << 8) + (ss[pos++] << 16) + (ss[pos++] << 24);\n"
"    //Getting Points\n"
"    test = new Int8Array(obj.numberOfPoints*4*3); for(i=0; i<obj.numberOfPoints*4*3; i++) test[i] = ss[pos++];\n"
"    obj.points = new Float32Array(test.buffer);\n"
"    //Generating Normals\n"
"    test = new Array(obj.numberOfPoints*3); for(i=0; i<obj.numberOfPoints*3; i++) test[i] = 0.0;\n"
"    obj.normals = new Float32Array(test);\n"
"    //Getting Colors\n"
"    test = []; for(i=0; i<obj.numberOfPoints*4; i++) test[i] = ss[pos++]/255.0;\n"
"    obj.colors = new Float32Array(test);\n"
"\n"
"    obj.numberOfIndex = (ss[pos++]) + (ss[pos++] << 8) + (ss[pos++] << 16) + (ss[pos++] << 24);\n"
"    //Getting Index\n"
"    test = new Int8Array(obj.numberOfIndex*2); for(i=0; i<obj.numberOfIndex*2; i++) test[i] = ss[pos++];\n"
"    obj.index = new Uint16Array(test.buffer);\n"
"    //Getting Matrix\n"
"    test = new Int8Array(16*4); for(i=0; i<16*4; i++) test[i] = ss[pos++];\n"
"    obj.matrix = new Float32Array(test.buffer);\n"
"\n"
"    //Creating Buffers\n"
"    obj.lbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, obj.lbuff);\n"
"    this.gl.bufferData(this.gl.ARRAY_BUFFER, obj.points, this.gl.STATIC_DRAW); obj.lbuff.itemSize = 3;\n"
"\n"
"    obj.nbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, obj.nbuff);\n"
"    this.gl.bufferData(this.gl.ARRAY_BUFFER, obj.normals, this.gl.STATIC_DRAW);  obj.nbuff.itemSize = 3;\n"
"\n"
"    obj.cbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, obj.cbuff);\n"
"    this.gl.bufferData(this.gl.ARRAY_BUFFER, obj.colors, this.gl.STATIC_DRAW);   obj.cbuff.itemSize = 4;\n"
"\n"
"    obj.ibuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ELEMENT_ARRAY_BUFFER, obj.ibuff);\n"
"    this.gl.bufferData(this.gl.ELEMENT_ARRAY_BUFFER, obj.index, this.gl.STREAM_DRAW);\n"
"\n"
"    obj.render = this.renderLine;\n"
"  }\n"
"\n"
"  //-=-=-=-=-=[ MESH ]=-=-=-=-=-\n"
"  else if (type == 'M'){\n"
"    obj.numberOfVertices = (ss[pos++]) + (ss[pos++] << 8) + (ss[pos++] << 16) + (ss[pos++] << 24);\n"
"    //Getting Vertices\n"
"    test = new Int8Array(obj.numberOfVertices*4*3); for(i=0; i<obj.numberOfVertices*4*3; i++) test[i] = ss[pos++];\n"
"    obj.vertices = new Float32Array(test.buffer);\n"
"    //Getting Normals\n"
"    test = new Int8Array(obj.numberOfVertices*4*3); for(i=0; i<obj.numberOfVertices*4*3; i++) test[i] = ss[pos++];\n"
"    obj.normals = new Float32Array(test.buffer);\n"
"    //Getting Colors\n"
"    test = []; for(i=0; i<obj.numberOfVertices*4; i++) test[i] = ss[pos++]/255.0;\n"
"    obj.colors = new Float32Array(test);\n"
"\n"
"    obj.numberOfIndex = (ss[pos++]) + (ss[pos++] << 8) + (ss[pos++] << 16) + (ss[pos++] << 24);\n"
"    //Getting Index\n"
"    test = new Int8Array(obj.numberOfIndex*2); for(i=0; i<obj.numberOfIndex*2; i++) test[i] = ss[pos++];\n"
"    obj.index = new Uint16Array(test.buffer);\n"
"    //Getting Matrix\n"
"    test = new Int8Array(16*4); for(i=0; i<16*4; i++) test[i] = ss[pos++];\n"
"    obj.matrix = new Float32Array(test.buffer);\n"
"    //Getting TCoord\n"
"    obj.tcoord = null;\n"
"\n"
"    //Create Buffers\n"
"    obj.vbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, obj.vbuff);\n"
"    this.gl.bufferData(this.gl.ARRAY_BUFFER, obj.vertices, this.gl.STATIC_DRAW); obj.vbuff.itemSize = 3;\n"
"\n"
"    obj.nbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, obj.nbuff);\n"
"    this.gl.bufferData(this.gl.ARRAY_BUFFER, obj.normals, this.gl.STATIC_DRAW);  obj.nbuff.itemSize = 3;\n"
"\n"
"    obj.cbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, obj.cbuff);\n"
"    this.gl.bufferData(this.gl.ARRAY_BUFFER, obj.colors, this.gl.STATIC_DRAW);   obj.cbuff.itemSize = 4;\n"
"\n"
"    obj.ibuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ELEMENT_ARRAY_BUFFER, obj.ibuff);\n"
"    this.gl.bufferData(this.gl.ELEMENT_ARRAY_BUFFER, obj.index, this.gl.STREAM_DRAW);\n"
"\n"
"    obj.render = this.renderMesh;\n"
"  }\n"
"\n"
"  // ColorMap Widget\n"
"  else if (type == 'C'){\n"
"    obj.numOfColors = size;\n"
"\n"
"    //Getting Position\n"
"    test = new Int8Array(2*4); for(i=0; i<2*4; i++) test[i] = ss[pos++];\n"
"    obj.position = new Float32Array(test.buffer);\n"
"\n"
"    //Getting Size\n"
"    test = new Int8Array(2*4); for(i=0; i<2*4; i++) test[i] = ss[pos++];\n"
"    obj.size = new Float32Array(test.buffer);\n"
"\n"
"    //Getting Colors\n"
"    obj.colors = [];\n"
"    for(c=0; c<obj.numOfColors; c++){\n"
"      test = new Int8Array(4); for(i=0; i<4; i++) test[i] = ss[pos++];\n"
"      v = new Float32Array(test.buffer);\n"
"      xrgb = [v[0], ss[pos++], ss[pos++], ss[pos++]];\n"
"      obj.colors[c] = xrgb;\n"
"    }\n"
"\n"
"    obj.orientation = ss[pos++];\n"
"    obj.numOfLabels = ss[pos++];\n"
"    tt = \"\";\n"
"    for(jj=0; jj<(ss.length-pos); jj++) tt = tt + String.fromCharCode(ss[pos+jj]);\n"
"    obj.title = tt;\n"
"\n"
"    obj.render = this.renderColorMap;\n"
"  }\n"
"\n"
"  // Points\n"
"  else if (type == 'P'){\n"
"    obj.numberOfPoints = (ss[pos++]) + (ss[pos++] << 8) + (ss[pos++] << 16) + (ss[pos++] << 24);\n"
"    //Getting Points\n"
"    test = new Int8Array(obj.numberOfPoints*4*3); for(i=0; i<obj.numberOfPoints*4*3; i++) test[i] = ss[pos++];\n"
"    obj.points = new Float32Array(test.buffer);\n"
"\n"
"    //Getting Colors\n"
"    test = []; for(i=0; i<obj.numberOfPoints*4; i++) test[i] = ss[pos++]/255.0;\n"
"    obj.colors = new Float32Array(test);\n"
"\n"
"    //Getting Matrix //Wendel\n"
"    test = new Int8Array(16*4); for(i=0; i<16*4; i++) test[i] = ss[pos++];\n"
"    obj.matrix = new Float32Array(test.buffer);\n"
"\n"
"    //Creating Buffers\n"
"    obj.pbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, obj.pbuff);\n"
"    this.gl.bufferData(this.gl.ARRAY_BUFFER, obj.points, this.gl.STATIC_DRAW); obj.pbuff.itemSize = 3;\n"
"\n"
"    obj.cbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, obj.cbuff);\n"
"    this.gl.bufferData(this.gl.ARRAY_BUFFER, obj.colors, this.gl.STATIC_DRAW);   obj.cbuff.itemSize = 4;\n"
"\n"
"    obj.render = this.renderPoints;\n"
"  }\n"
"}\n"
"\n"
"WebGLRenderer.prototype.renderColorMap = function(){\n"
"  obj = this;\n"
"  render = this.father;\n"
"\n"
"  range = [obj.colors[0][0], obj.colors[obj.colors.length-1][0]];\n"
"  size = [obj.size[0]*render.view.width, obj.size[1]*render.view.height];\n"
"  pos = [obj.position[0]*render.view.width, (1-obj.position[1])*render.view.height];\n"
"  pos[1] = pos[1]-size[1];\n"
"  dx = size[0]/size[1];\n"
"  dy = size[1]/size[0];\n"
"  realSize = size;\n"
"\n"
"  textSizeX = Math.round(render.view.height/35);\n"
"  textSizeY = Math.round(render.view.height/23);\n"
"  if (obj.orientation == 1){\n"
"    size[0] = size[0]*dy/25;\n"
"    size[1] = size[1]-(2*textSizeY);\n"
"  } else {\n"
"    size[0] = size[0];\n"
"    size[1] = size[1]*dx/25;\n"
"  }\n"
"\n"
"  // Draw Gradient\n"
"  ctx = this.father.ctx2d;\n"
"  if(obj.orientation == 1){\n"
"    pos[1] += 2*textSizeY;\n"
"    grad = ctx.createLinearGradient(pos[0], pos[1], pos[0], pos[1]+size[1]);\n"
"  } else {\n"
"    pos[1] += 2*textSizeY;\n"
"    grad = ctx.createLinearGradient(pos[0], pos[1], pos[0]+size[0], pos[1]);\n"
"  }\n"
"  if ((range[1]-range[0]) == 0){\n"
"      color = 'rgba(' + obj.colors[0][1] + ',' + obj.colors[0][2] + ',' + obj.colors[0][3] + ',1)';\n"
"      grad.addColorStop(0, color);\n"
"      grad.addColorStop(1, color);\n"
"  } else {\n"
"    for(c=0; c<obj.colors.length; c++){\n"
"      v = ((obj.colors[c][0]-range[0])/(range[1]-range[0]));\n"
"      if (obj.orientation == 1) v=1-v;\n"
"      color = 'rgba(' + obj.colors[c][1] + ',' + obj.colors[c][2] + ',' + obj.colors[c][3] + ',1)';\n"
"      grad.addColorStop(v, color);\n"
"    }\n"
"  }\n"
"  ctx.fillStyle = grad;\n"
"  ctx.fillRect(pos[0], pos[1], size[0], size[1]);\n"
"  // Draw Range Labels\n"
"  range[0] = Math.round(range[0]*1000)/1000;\n"
"  range[1] = Math.round(range[1]*1000)/1000;\n"
"  ctx.fillStyle = 'white';\n"
"  ctx.font = textSizeY + 'px sans-serif';\n"
"  ctx.txtBaseline = 'ideographic';\n"
"  if (obj.orientation == 1){\n"
"    ctx.fillText(range[1], pos[0], pos[1]-5);\n"
"    ctx.fillText(range[0], pos[0], pos[1]+size[1]+textSizeY);\n"
"  } else {\n"
"    ctx.fillText(range[0], pos[0], pos[1]+size[1]+textSizeY);\n"
"    txt = range[1].toString();\n"
"    ctx.fillText(range[1], pos[0]+size[0]-((txt.length-1)*textSizeX), pos[1]+size[1]+textSizeY);\n"
"  }\n"
"  // Draw Title\n"
"  ctx.fillStyle = 'white';\n"
"  ctx.font = textSizeY + 'px sans-serif';\n"
"  ctx.txtBaseline = 'ideographic';\n"
"  if (obj.orientation == 1) ctx.fillText(obj.title, pos[0]+(obj.size[0]*render.view.width)/2-(obj.title.length*textSizeX/2), pos[1]-textSizeY-5);\n"
"  else ctx.fillText(obj.title, pos[0]+size[0]/2-(obj.title.length*textSizeX/2), pos[1]-textSizeY-5);\n"
"  // Draw Intervals' line\n"
"  //Draw Interval make the render process slow\n"
"  /*\n"
"  interval = obj.numOfLabels-1;\n"
"  if (obj.orientation == 1){\n"
"    diff = size[1]/(interval-1);\n"
"    y = pos[1]+size[1];\n"
"    x = size[0]/2;\n"
"    for(ii=0; ii<interval; ii++){\n"
"      y = Math.floor(y) + 0.5;\n"
"      if (ii%5) ctx.moveTo(pos[0]+2*x, y);\n"
"      else ctx.moveTo(pos[0]+x, y);\n"
"      ctx.lineTo(pos[0]+x*3, y);\n"
"      ctx.lineWidth = 1;\n"
"      ctx.strokeStyle = \"white\";\n"
"      ctx.stroke();\n"
"      y -= diff;\n"
"    }\n"
"  } else {\n"
"    diff = size[0]/(interval-1);\n"
"    y = size[1]/2;\n"
"    x = pos[0];\n"
"    for(ii=0; ii<interval; ii++){\n"
"      x = Math.floor(x) + 0.5;\n"
"      if (ii%5) ctx.moveTo(x, pos[1]);\n"
"      else ctx.moveTo(x, pos[1]+y);\n"
"      ctx.lineTo(x, pos[1]-y);\n"
"      ctx.lineWidth = 1;\n"
"      ctx.strokeStyle = \"white\";\n"
"      ctx.stroke();\n"
"      x += diff;\n"
"    }\n"
"  }/**/\n"
"}\n"
"\n"
"WebGLRenderer.prototype.initBackground = function(c1, c2){\n"
"  if (typeof(this.gl) == \"undefined\") return;\n"
"  if (typeof(this.sceneJSON) == \"undefined\") return;\n"
"\n"
"  this.background = new Object();\n"
"  this.background.vertices = new Float32Array([-1.0, -1.0, 0.0, 1.0, -1.0, 0.0, 1.0, 1.0, 0.0, -1.0, 1.0, 0.0]);\n"
"  this.background.colors = new Float32Array([c1[0], c1[1], c1[2], 1.0,\n"
"                                             c1[0], c1[1], c1[2], 1.0,\n"
"                                             c2[0], c2[1], c2[2], 1.0,\n"
"                                             c2[0], c2[1], c2[2], 1.0]);\n"
"  this.background.index = new Uint16Array([0, 1, 2, 0, 2, 3]);\n"
"  this.background.normals = new Float32Array([0.0, 0.0, -1.0, 0.0, 0.0, -1.0, 0.0, 0.0, -1.0, 0.0, 0.0, -1.0]);\n"
"\n"
"  this.background.numberOfIndex = 6;\n"
"\n"
"  //Create Buffers\n"
"  this.background.vbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, this.background.vbuff);\n"
"  this.gl.bufferData(this.gl.ARRAY_BUFFER, this.background.vertices, this.gl.STATIC_DRAW); this.background.vbuff.itemSize = 3;\n"
"  this.background.nbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, this.background.nbuff);\n"
"  this.gl.bufferData(this.gl.ARRAY_BUFFER, this.background.normals, this.gl.STATIC_DRAW);  this.background.nbuff.itemSize = 3;\n"
"  this.background.cbuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ARRAY_BUFFER, this.background.cbuff);\n"
"  this.gl.bufferData(this.gl.ARRAY_BUFFER, this.background.colors, this.gl.STATIC_DRAW);   this.background.cbuff.itemSize = 4;\n"
"  this.background.ibuff = this.gl.createBuffer(); this.gl.bindBuffer(this.gl.ELEMENT_ARRAY_BUFFER, this.background.ibuff);\n"
"  this.gl.bufferData(this.gl.ELEMENT_ARRAY_BUFFER, this.background.index, this.gl.STREAM_DRAW);\n"
"}\n"
"\n"
"WebGLRenderer.prototype.renderBackground = function(){\n"
"  if (this.background == null) return;\n"
"\n"
"  this.gl.useProgram(this.shaderProgram);\n"
"  this.gl.uniform1i(this.shaderProgram.uIsLine, false);\n"
"\n"
"  mat4.translate(this.mvMatrix, [0.0, 0.0, -1.0]);\n"
"  this.gl.bindBuffer(this.gl.ARRAY_BUFFER, this.background.vbuff);\n"
"  this.gl.vertexAttribPointer(this.shaderProgram.vertexPositionAttribute, this.background.vbuff.itemSize, this.gl.FLOAT, false, 0, 0);\n"
"  this.gl.bindBuffer(this.gl.ARRAY_BUFFER, this.background.nbuff);\n"
"  this.gl.vertexAttribPointer(this.shaderProgram.vertexNormalAttribute, this.background.nbuff.itemSize, this.gl.FLOAT, false, 0, 0);\n"
"  this.gl.bindBuffer(this.gl.ARRAY_BUFFER, this.background.cbuff);\n"
"  this.gl.vertexAttribPointer(this.shaderProgram.vertexColorAttribute, this.background.cbuff.itemSize, this.gl.FLOAT, false, 0, 0);\n"
"  this.gl.bindBuffer(this.gl.ELEMENT_ARRAY_BUFFER, this.background.ibuff);\n"
"  this.setMatrixUniforms(this.shaderProgram);\n"
"  this.gl.drawElements(this.gl.TRIANGLES, this.background.numberOfIndex, this.gl.UNSIGNED_SHORT, 0);\n"
"}\n"
"\n"
"WebGLRenderer.prototype.renderMesh = function(){\n"
"  obj = this;\n"
"  render = this.father;\n"
"  render.gl.useProgram(render.shaderProgram);\n"
"\n"
"  render.gl.uniform1i(render.shaderProgram.uIsLine, false);\n"
"\n"
"  cameraRot = mat4.toRotationMat(render.mvMatrix);\n"
"  mat4.transpose(cameraRot);\n"
"  inverse = mat4.create(); mat4.inverse(cameraRot, inverse);\n"
"  test = mat4.create(obj.matrix);\n"
"  mat4.transpose(test);\n"
"\n"
"  icenter = [-render.sceneJSON.Center[0], -render.sceneJSON.Center[1], -render.sceneJSON.Center[2]];\n"
"\n"
"  mvPushMatrix(render.mvMatrix);\n"
"  mat4.multiply(render.mvMatrix, cameraRot, render.mvMatrix);\n"
"  if(obj.layer == 0) mat4.translate(render.mvMatrix, render.translation);\n"
"  mat4.multiply(render.mvMatrix, inverse, render.mvMatrix);\n"
"\n"
"  if(obj.layer == 0) mat4.translate(render.mvMatrix, render.sceneJSON.Center);\n"
"  mat4.multiply(render.mvMatrix, cameraRot, render.mvMatrix);\n"
"  if(obj.layer == 0) mat4.scale(render.mvMatrix, [render.objScale, render.objScale, render.objScale], render.mvMatrix);\n"
"  mat4.multiply(render.mvMatrix, render.rotMatrix, render.mvMatrix);\n"
"  mat4.multiply(render.mvMatrix, inverse, render.mvMatrix);\n"
"  if(obj.layer == 0) mat4.translate(render.mvMatrix, icenter);\n"
"\n"
"  render.rotMatrix2 = render.mvMatrix;\n"
"\n"
"  mat4.multiply(render.mvMatrix, test, render.mvMatrix);\n"
"\n"
"  render.gl.bindBuffer(render.gl.ARRAY_BUFFER, obj.vbuff);\n"
"  render.gl.vertexAttribPointer(render.shaderProgram.vertexPositionAttribute, obj.vbuff.itemSize, render.gl.FLOAT, false, 0, 0);\n"
"  render.gl.bindBuffer(render.gl.ARRAY_BUFFER, obj.nbuff);\n"
"  render.gl.vertexAttribPointer(render.shaderProgram.vertexNormalAttribute, obj.nbuff.itemSize, render.gl.FLOAT, false, 0, 0);\n"
"  render.gl.bindBuffer(render.gl.ARRAY_BUFFER, obj.cbuff);\n"
"  render.gl.vertexAttribPointer(render.shaderProgram.vertexColorAttribute, obj.cbuff.itemSize, render.gl.FLOAT, false, 0, 0);\n"
"  render.gl.bindBuffer(render.gl.ELEMENT_ARRAY_BUFFER, obj.ibuff);\n"
"  render.setMatrixUniforms(render.shaderProgram);\n"
"  render.gl.drawElements(render.gl.TRIANGLES, obj.numberOfIndex, render.gl.UNSIGNED_SHORT, 0);\n"
"  render.mvMatrix = mvPopMatrix();\n"
"}\n"
"\n"
"WebGLRenderer.prototype.renderLine = function(){\n"
"  obj = this;\n"
"  render = this.father;\n"
"  render.gl.useProgram(render.shaderProgram);\n"
"\n"
"  render.gl.enable(render.gl.POLYGON_OFFSET_FILL);  //Avoid zfighting\n"
"  render.gl.polygonOffset(-1.0, -1.0);\n"
"\n"
"  render.gl.uniform1i(render.shaderProgram.uIsLine, true);\n"
"\n"
"  cameraRot = mat4.toRotationMat(render.mvMatrix);\n"
"  mat4.transpose(cameraRot);\n"
"  inverse = mat4.create(); mat4.inverse(cameraRot, inverse);\n"
"  test = mat4.create(obj.matrix);\n"
"  mat4.transpose(test);\n"
"\n"
"  icenter = [-render.sceneJSON.Center[0], -render.sceneJSON.Center[1], -render.sceneJSON.Center[2]];\n"
"\n"
"  mvPushMatrix(render.mvMatrix);\n"
"  mat4.multiply(render.mvMatrix, cameraRot, render.mvMatrix);\n"
"  if(obj.layer == 0) mat4.translate(render.mvMatrix, render.translation);\n"
"  mat4.multiply(render.mvMatrix, inverse, render.mvMatrix);\n"
"\n"
"  if(obj.layer == 0) mat4.translate(render.mvMatrix, render.sceneJSON.Center);\n"
"  mat4.multiply(render.mvMatrix, cameraRot, render.mvMatrix);\n"
"  if(obj.layer == 0) mat4.scale(render.mvMatrix, [render.objScale, render.objScale, render.objScale], render.mvMatrix);\n"
"  mat4.multiply(render.mvMatrix, render.rotMatrix, render.mvMatrix);\n"
"  mat4.multiply(render.mvMatrix, inverse, render.mvMatrix);\n"
"  if(obj.layer == 0) mat4.translate(render.mvMatrix, icenter);\n"
"\n"
"  render.rotMatrix2 = render.mvMatrix;\n"
"\n"
"  mat4.multiply(render.mvMatrix, test, render.mvMatrix);\n"
"\n"
"  render.gl.bindBuffer(render.gl.ARRAY_BUFFER, obj.lbuff);\n"
"  render.gl.vertexAttribPointer(render.shaderProgram.vertexPositionAttribute, obj.lbuff.itemSize, render.gl.FLOAT, false, 0, 0);\n"
"  render.gl.bindBuffer(render.gl.ARRAY_BUFFER, obj.nbuff);\n"
"  render.gl.vertexAttribPointer(render.shaderProgram.vertexNormalAttribute, obj.nbuff.itemSize, render.gl.FLOAT, false, 0, 0);\n"
"  render.gl.bindBuffer(render.gl.ARRAY_BUFFER, obj.cbuff);\n"
"  render.gl.vertexAttribPointer(render.shaderProgram.vertexColorAttribute, obj.cbuff.itemSize, render.gl.FLOAT, false, 0, 0);\n"
"  render.gl.bindBuffer(render.gl.ELEMENT_ARRAY_BUFFER, obj.ibuff);\n"
"  render.setMatrixUniforms(render.shaderProgram);\n"
"  render.gl.drawElements(render.gl.LINES, obj.numberOfIndex, render.gl.UNSIGNED_SHORT, 0);\n"
"  render.mvMatrix = mvPopMatrix();\n"
"\n"
"  render.gl.disable(render.gl.POLYGON_OFFSET_FILL);\n"
"}\n"
"\n"
"WebGLRenderer.prototype.renderPoints = function(){\n"
"  obj = this;\n"
"  render = this.father;\n"
"  render.gl.useProgram(render.pointShaderProgram);\n"
"\n"
"  render.gl.enable(render.gl.POLYGON_OFFSET_FILL);  //Avoid zfighting\n"
"  render.gl.polygonOffset(-1.0, -1.0);\n"
"\n"
"  render.gl.uniform1f(render.pointShaderProgram.uPointSize, 2.0);//Wendel\n"
"\n"
"  cameraRot = mat4.toRotationMat(render.mvMatrix);\n"
"  mat4.transpose(cameraRot);\n"
"  inverse = mat4.create(); mat4.inverse(cameraRot, inverse);\n"
"  test = mat4.create(obj.matrix);\n"
"  mat4.transpose(test);\n"
"\n"
"  icenter = [-render.sceneJSON.Center[0], -render.sceneJSON.Center[1], -render.sceneJSON.Center[2]];\n"
"\n"
"  mvPushMatrix(render.mvMatrix);\n"
"  mat4.multiply(render.mvMatrix, cameraRot, render.mvMatrix);\n"
"  if(obj.layer == 0) mat4.translate(render.mvMatrix, render.translation);\n"
"  mat4.multiply(render.mvMatrix, inverse, render.mvMatrix);\n"
"\n"
"  if(obj.layer == 0) mat4.translate(render.mvMatrix, render.sceneJSON.Center);\n"
"  mat4.multiply(render.mvMatrix, cameraRot, render.mvMatrix);\n"
"  if(obj.layer == 0) mat4.scale(render.mvMatrix, [render.objScale, render.objScale, render.objScale], render.mvMatrix);\n"
"  mat4.multiply(render.mvMatrix, render.rotMatrix, render.mvMatrix);\n"
"  mat4.multiply(render.mvMatrix, inverse, render.mvMatrix);\n"
"  if(obj.layer == 0) mat4.translate(render.mvMatrix, icenter);\n"
"\n"
"  render.rotMatrix2 = render.mvMatrix;\n"
"\n"
"  mat4.multiply(render.mvMatrix, test, render.mvMatrix);\n"
"\n"
"  render.gl.bindBuffer(render.gl.ARRAY_BUFFER, obj.pbuff);\n"
"  render.gl.vertexAttribPointer(render.pointShaderProgram.vertexPositionAttribute, obj.pbuff.itemSize, render.gl.FLOAT, false, 0, 0);\n"
"  render.gl.bindBuffer(render.gl.ARRAY_BUFFER, obj.cbuff);\n"
"  render.gl.vertexAttribPointer(render.pointShaderProgram.vertexColorAttribute, obj.cbuff.itemSize, render.gl.FLOAT, false, 0, 0);\n"
"  render.setMatrixUniforms(render.pointShaderProgram);\n"
"  render.gl.drawArrays(render.gl.POINTS, 0, obj.numberOfPoints);//Wendel\n"
"  render.mvMatrix = mvPopMatrix();\n"
"\n"
"  render.gl.disable(render.gl.POLYGON_OFFSET_FILL);\n"
"}\n"
"\n"
"WebGLRenderer.prototype.setMatrixUniforms = function(s) {\n"
"  mvMatrixInv = mat4.create();\n"
"  normal = mat4.create();\n"
"  mat4.inverse(this.mvMatrix, mvMatrixInv);\n"
"  mat4.transpose(mvMatrixInv, normal);\n"
"\n"
"  this.gl.uniformMatrix4fv(s.pMatrixUniform, false, this.pMatrix);\n"
"  this.gl.uniformMatrix4fv(s.mvMatrixUniform, false, this.mvMatrix);\n"
"  if(s.nMatrixUniform != null) this.gl.uniformMatrix4fv(s.nMatrixUniform, false, normal);\n"
"}\n"
"\n"
"WebGLRenderer.prototype.processObject = function() {\n"
"  if (this.processQueue.length != 0){\n"
"    obj = this.processQueue[this.processQueue.length-1];\n"
"    this.processQueue.length -= 1;\n"
"    this.parseObject(obj);\n"
"  }\n"
"}\n"
"\n"
"WebGLRenderer.prototype.drawScene = function() {\n"
"  this.drawInterval = requestAnimFrame(new Function(\"webglRenderers['\" + this.view.id + \"'].drawScene();\"));\n"
"  if (this.hasSceneChanged){\n"
"    this.updateScene();\n"
"  }\n"
"  if (this.sceneJSON == null){\n"
"    return;\n"
"  }\n"
"  this.frames++;\n"
"\n"
"  if(this.frames >= 50 && this.nbErrors < 5){\n"
"    this.frames = 0;\n"
"    ko = new Date();\n"
"    currTime = ko.getTime();\n"
"    diff = currTime - this.lastTime;\n"
"    this.lastTime = currTime;\n"
"    this.fps = 50000/diff;\n"
"  }\n"
"  this.processObject();\n"
"\n"
"  this.gl.viewport(0, 0, this.gl.viewportWidth, this.gl.viewportHeight);\n"
"  this.gl.clear(this.gl.COLOR_BUFFER_BIT | this.gl.DEPTH_BUFFER_BIT);\n"
"\n"
"  mat4.ortho(-1.0, 1.0, -1.0, 1.0, 1.0, 1000000.0, this.pMatrix);\n"
"  mat4.identity(this.mvMatrix);\n"
"  this.gl.disable(this.gl.DEPTH_TEST);\n"
"  this.renderBackground();\n"
"  this.gl.enable(this.gl.DEPTH_TEST);\n"
"\n"
"  this.ctx2d.clearRect(0, 0, this.view.width, this.view.height);\n"
"  for(rr=this.sceneJSON.Renderers.length-1; rr>=0 ; rr--){\n"
"    renderer = this.sceneJSON.Renderers[rr];\n"
"    width  = renderer.size[0]-renderer.origin[0];\n"
"    height = renderer.size[1]-renderer.origin[1];\n"
"    width = width*this.view.width;\n"
"    height = height*this.view.height;\n"
"    x = renderer.origin[0]*this.view.width;\n"
"    y = renderer.origin[1]*this.view.height;\n"
"    if (y < 0) y = 0;\n"
"    this.gl.viewport(x, y, width, height);\n"
"    //this.gl.clear(this.gl.DEPTH_BUFFER_BIT);\n"
"    mat4.perspective(renderer.LookAt[0], width/height, 0.1, 1000000.0, this.pMatrix);\n"
"    mat4.identity(this.mvMatrix);\n"
"    mat4.lookAt([renderer.LookAt[7], renderer.LookAt[8], renderer.LookAt[9]],\n"
"                [renderer.LookAt[1], renderer.LookAt[2], renderer.LookAt[3]],\n"
"                [renderer.LookAt[4], renderer.LookAt[5], renderer.LookAt[6]],\n"
"                this.mvMatrix);\n"
"\n"
"    for(r=0; r<this.objects.length; r++){\n"
"      if (!this.objects[r].hasTransparency && this.objects[r].layer == rr) this.objects[r].render();\n"
"    }\n"
"    //Render Objects with Transparency\n"
"    this.gl.enable(this.gl.BLEND);                //Enable transparency\n"
"    this.gl.enable(this.gl.POLYGON_OFFSET_FILL);  //Avoid zfighting\n"
"    this.gl.polygonOffset(-1.0, -1.0);\n"
"    for(r=0; r<this.objects.length; r++){\n"
"      if (this.objects[r].hasTransparency && this.objects[r].layer == rr) this.objects[r].render();\n"
"    }\n"
"    this.gl.disable(this.gl.POLYGON_OFFSET_FILL);\n"
"    this.gl.disable(this.gl.BLEND);\n"
"  }\n"
"}\n"
"\n"
"WebGLRenderer.prototype.initShaders = function() {\n"
"  this.shaderProgram;\n"
"  this.pointShaderProgram;\n"
"\n"
"  var fragmentShader = this.getShader(\"shader-fs\");\n"
"  var vertexShader = this.getShader(\"shader-vs\");\n"
"  var pointFragShader = this.getShader(\"shader-fs-Point\");\n"
"  var pointVertShader = this.getShader(\"shader-vs-Point\");\n"
"\n"
"  this.shaderProgram = this.gl.createProgram();\n"
"  this.gl.attachShader(this.shaderProgram, vertexShader);\n"
"  this.gl.attachShader(this.shaderProgram, fragmentShader);\n"
"  this.gl.linkProgram(this.shaderProgram);\n"
"  if (!this.gl.getProgramParameter(this.shaderProgram, this.gl.LINK_STATUS)) {\n"
"      alert(\"Could not initialise shaders\");\n"
"  }\n"
"  this.pointShaderProgram = this.gl.createProgram();\n"
"  this.gl.attachShader(this.pointShaderProgram, pointVertShader);\n"
"  this.gl.attachShader(this.pointShaderProgram, pointFragShader);\n"
"  this.gl.linkProgram(this.pointShaderProgram);\n"
"  if (!this.gl.getProgramParameter(this.pointShaderProgram, this.gl.LINK_STATUS)) {\n"
"      alert(\"Could not initialise the point shaders\");\n"
"  }\n"
"\n"
"  this.gl.useProgram(this.pointShaderProgram);\n"
"  this.pointShaderProgram.vertexPositionAttribute = this.gl.getAttribLocation(this.pointShaderProgram, \"aVertexPosition\");\n"
"  this.gl.enableVertexAttribArray(this.pointShaderProgram.vertexPositionAttribute);\n"
"  this.pointShaderProgram.vertexColorAttribute = this.gl.getAttribLocation(this.pointShaderProgram, \"aVertexColor\");\n"
"  this.gl.enableVertexAttribArray(this.pointShaderProgram.vertexColorAttribute);\n"
"  this.pointShaderProgram.pMatrixUniform = this.gl.getUniformLocation(this.pointShaderProgram, \"uPMatrix\");\n"
"  this.pointShaderProgram.mvMatrixUniform = this.gl.getUniformLocation(this.pointShaderProgram, \"uMVMatrix\");\n"
"  this.pointShaderProgram.nMatrixUniform = this.gl.getUniformLocation(this.pointShaderProgram, \"uNMatrix\");\n"
"  this.pointShaderProgram.uPointSize = this.gl.getUniformLocation(this.pointShaderProgram, \"uPointSize\");\n"
"\n"
"  this.gl.useProgram(this.shaderProgram);\n"
"  this.shaderProgram.vertexPositionAttribute = this.gl.getAttribLocation(this.shaderProgram, \"aVertexPosition\");\n"
"  this.gl.enableVertexAttribArray(this.shaderProgram.vertexPositionAttribute);\n"
"  this.shaderProgram.vertexColorAttribute = this.gl.getAttribLocation(this.shaderProgram, \"aVertexColor\");\n"
"  this.gl.enableVertexAttribArray(this.shaderProgram.vertexColorAttribute);\n"
"  this.shaderProgram.vertexNormalAttribute = this.gl.getAttribLocation(this.shaderProgram, \"aVertexNormal\");\n"
"  this.gl.enableVertexAttribArray(this.shaderProgram.vertexNormalAttribute);\n"
"  this.shaderProgram.pMatrixUniform = this.gl.getUniformLocation(this.shaderProgram, \"uPMatrix\");\n"
"  this.shaderProgram.mvMatrixUniform = this.gl.getUniformLocation(this.shaderProgram, \"uMVMatrix\");\n"
"  this.shaderProgram.nMatrixUniform = this.gl.getUniformLocation(this.shaderProgram, \"uNMatrix\");\n"
"  this.shaderProgram.uIsLine = this.gl.getUniformLocation(this.shaderProgram, \"uIsLine\");\n"
"}\n"
"\n"
"WebGLRenderer.prototype.getShader = function(id) {\n"
"    var shaderScript = document.getElementById(id);\n"
"    if (!shaderScript) {\n"
"        return null;\n"
"    }\n"
"\n"
"    var str = \"\";\n"
"    var k = shaderScript.firstChild;\n"
"    while (k) {\n"
"        if (k.nodeType == 3) {\n"
"            str += k.textContent;\n"
"        }\n"
"        k = k.nextSibling;\n"
"    }\n"
"\n"
"    var shader;\n"
"    if (shaderScript.type == \"x-shader/x-fragment\") {\n"
"        shader = this.gl.createShader(this.gl.FRAGMENT_SHADER);\n"
"    } else if (shaderScript.type == \"x-shader/x-vertex\") {\n"
"        shader = this.gl.createShader(this.gl.VERTEX_SHADER);\n"
"    } else {\n"
"        return null;\n"
"    }\n"
"\n"
"    this.gl.shaderSource(shader, str);\n"
"    this.gl.compileShader(shader);\n"
"\n"
"    if (!this.gl.getShaderParameter(shader, this.gl.COMPILE_STATUS)) {\n"
"        alert(this.gl.getShaderInfoLog(shader));\n"
"        return null;\n"
"    }\n"
"\n"
"    return shader;\n"
"}\n"
"\n"
"WebGLRenderer.prototype.forceUpdateCamera = function(){\n"
"  if (typeof(this.gl) == \"undefined\" || this.gl == null || renderers.current != this) return;\n"
"  if (this.offlineMode) return;\n"
"  render = this;\n"
"\n"
"  pos = [render.lookAt[7], render.lookAt[8], render.lookAt[9]];\n"
"  up  = [render.lookAt[4], render.lookAt[5], render.lookAt[6]];\n"
"  fp  = [render.lookAt[1], render.lookAt[2], render.lookAt[3]];\n"
"  tt  = [render.translation[0], render.translation[1], 0.0];\n"
"  center = [render.sceneJSON.Center[0], render.sceneJSON.Center[1], render.sceneJSON.Center[2]];\n"
"\n"
"  cameraRot = mat4.toRotationMat(render.mvMatrix);\n"
"  mat4.transpose(cameraRot);\n"
"  inverse = mat4.create(); mat4.inverse(cameraRot, inverse);\n"
"\n"
"  inv = mat4.create();\n"
"  mat4.identity(inv);\n"
"  mat4.multiply(inv, cameraRot, inv);\n"
"  mat4.scale(inv, [render.objScale, render.objScale, render.objScale], inv);\n"
"  mat4.multiply(inv, render.rotMatrix, inv);\n"
"  mat4.multiply(inv, inverse, inv);\n"
"\n"
"  mat4.inverse(inv, inv);\n"
"  fp = vec3.subtract(fp, center, fp);\n"
"  pos = vec3.subtract(pos, center, pos);\n"
"  mat4.multiplyVec3(inv, fp, fp);\n"
"  mat4.multiplyVec3(inv, pos, pos);\n"
"  mat4.multiplyVec3(inv, up, up);\n"
"  fp = vec3.add(fp, center, fp);\n"
"  pos = vec3.add(pos, center, pos);\n"
"  vec3.normalize(up, up);\n"
"\n"
"  tt2 = [0, 0, 0];\n"
"  tt2[0] += tt[0]*render.right[0];\n"
"  tt2[1] += tt[0]*render.right[1];\n"
"  tt2[2] += tt[0]*render.right[2];\n"
"  tt2[0] += tt[1]*render.up[0];\n"
"  tt2[1] += tt[1]*render.up[1];\n"
"  tt2[2] += tt[1]*render.up[2];\n"
"\n"
"  vec3.subtract(pos, tt2, pos);\n"
"  vec3.subtract(fp , tt2, fp);\n"
"\n"
"  paraviewObjects[parseInt(render.sessionId)].sendEvent(\"UpdateCamera\", render.viewId + \" \" + fp[0] + \" \" + fp[1] + \" \" + fp[2]\n"
"  + \" \" + up[0] + \" \" + up[1] + \" \" + up[2] + \" \" + pos[0] + \" \" + pos[1] + \" \" + pos[2]);\n"
"  clearTimeout(this.updateId);\n"
"  this.updateId = setTimeout(\"webglRenderers[\\'\" + this.view.id + \"\\'].updateCamera()\", this.updateInterval);\n"
"}\n"
"\n"
"WebGLRenderer.prototype.updateCamera = function(){\n"
"  if (!this.mouseDown){\n"
"    this.updateId = setTimeout(\"webglRenderers[\\'\" + this.view.id + \"\\'].updateCamera()\", this.updateInterval);\n"
"    return;\n"
"  }\n"
"  if (this.serverMode) return;\n"
"  this.forceUpdateCamera();\n"
"}\n"
"\n"
"/**********************************************************************************/\n"
"\n"
"var mvMatrixStack = [];\n"
"function mvPushMatrix(m) {\n"
"    var copy = mat4.create();\n"
"    mat4.set(m, copy);\n"
"    mvMatrixStack.push(copy);\n"
"}\n"
"\n"
"function mvPopMatrix() {\n"
"  if (mvMatrixStack.length == 0) {\n"
"    throw \"Invalid popMatrix!\";\n"
"  }\n"
"  return mvMatrixStack.pop();\n"
"}\n"
"\n"
"\n"
"function handleMouseDown(event, id) {\n"
"  render = webglRenderers[id];\n"
"  render.mouseDown = true;\n"
"  render.lastMouseX = event.clientX;\n"
"  render.lastMouseY = event.clientY;\n"
"  if (!render.offlineMode){\n"
"    paraviewObjects[render.sessionId].sendEvent('MouseEvent', render.viewId + ' 0 0');\n"
"    updateRendererSize(render.sessionId, render.viewId, render.view.width/render.interactionRatio, render.view.height/render.interactionRatio);\n"
"  }\n"
"  event.preventDefault();\n"
"  return false;\n"
"}\n"
"\n"
"function handleMouseUp(event, id) {\n"
"  render = webglRenderers[id];\n"
"  render.mouseDown = false;\n"
"  if (!render.offlineMode){\n"
"    paraviewObjects[render.sessionId].sendEvent('MouseEvent', render.viewId + ' 2 0');\n"
"    updateRendererSize(render.sessionId, render.viewId, render.view.width, render.view.height);\n"
"    render.forceUpdateCamera();\n"
"    render.requestMetaData();\n"
"  }\n"
"  event.preventDefault();\n"
"}\n"
"\n"
"function handleMouseMove(event, id) {\n"
"  render = webglRenderers[id];\n"
"  if (!render.mouseDown) {\n"
"    return;\n"
"  }\n"
"  var newX = event.clientX;\n"
"  var newY = event.clientY;\n"
"  var deltaX = newX - render.lastMouseX;\n"
"  var deltaY = newY - render.lastMouseY;\n"
"\n"
"  if (event.button == 0){\n"
"    var rX = deltaX/50.0;\n"
"    var rY = deltaY/50.0;\n"
"    var mx = mat4.create(); mat4.identity(mx); mat4.rotate(mx, rX, [0, 1, 0]);\n"
"    var my = mat4.create(); mat4.identity(my); mat4.rotate(my, rY, [1, 0, 0]);\n"
"    mat4.multiply(mx, my, mx);\n"
"    mat4.multiply(mx, render.rotMatrix, render.rotMatrix);\n"
"  } else if (event.button == 1){\n"
"    z = Math.abs(render.sceneJSON.Renderers[0].LookAt[9]-render.sceneJSON.Renderers[0].LookAt[3]);\n"
"    aux = z/render.objScale;\n"
"    render.translation[0] += aux*deltaX/1500.0;\n"
"    render.translation[1] -= aux*deltaY/1500.0;\n"
"  } else if (event.button == 2){\n"
"    render.objScale += render.objScale*(deltaY)/200.0;\n"
"  } else {\n"
"    render.objScale += render.objScale*(deltaY)/200.0;\n"
"  }\n"
"\n"
"  render.lastMouseX = newX;\n"
"  render.lastMouseY = newY;\n"
"\n"
"  event.preventDefault();\n"
"}\n"
"\n"
"function mouseServerInt(rendererId, sessionId, viewId, action, event){\n"
"    consumeEvent(event);\n"
"    render = webglRenderers[rendererId];\n"
"    render.interaction.lastRealEvent = event;\n"
"    var width = render.view.width;\n"
"    var height = render.view.height;\n"
"\n"
"    if(action == 'down') {\n"
"        if(render.interaction.needUp) {\n"
"            paraviewObjects[sessionId].sendEvent('MouseEvent', viewId + ' 2 ' + render.interaction.lastEvent);\n"
"        }\n"
"        render.interaction.isDragging = true;\n"
"        render.interaction.needUp = true;\n"
"        switch(event.button){\n"
"            case 1:\n"
"                render.interaction.button =  '0 ';\n"
"                break;\n"
"            case 4:\n"
"                render.interaction.button =  '1 ';\n"
"                break;\n"
"            case 2:\n"
"                render.interaction.button =  '2 ';\n"
"                break;\n"
"        }\n"
"        render.interaction.action = \" 0 \";\n"
"        render.interaction.keys = \"\";\n"
"        if(event.ctrlKey) {\n"
"            render.interaction.keys += \"1\";\n"
"        } else {\n"
"            render.interaction.keys += \"0\";\n"
"        }\n"
"        if(event.shiftKey) {\n"
"            render.interaction.keys += \" 1\";\n"
"        } else {\n"
"            render.interaction.keys += \" 0\";\n"
"        }\n"
"        render.interaction.x = event.screenX;\n"
"        render.interaction.y = event.screenY;\n"
"\n"
"        // Keep relative origin\n"
"        var docX = event.pageX;\n"
"        var docY = event.pageY;\n"
"        render.interaction.xOrigin = docX - render.getPageX();\n"
"        render.interaction.yOrigin = docY - render.getPageY();\n"
"    } else if (action == 'move') {\n"
"        render.interaction.action = \" 1 \";\n"
"    } else if ( action == 'up' || action == 'click') {\n"
"        render.interaction.isDragging = false;\n"
"        render.interaction.needUp = false;\n"
"        render.interaction.action = \" 2 \";\n"
"        //\n"
"        var mouseInfo = ((event.screenX-render.interaction.x + render.interaction.xOrigin)/height) + \" \" + (1-(event.screenY-render.interaction.y + render.interaction.yOrigin)/height) + \" \" + render.interaction.keys ;\n"
"        render.interaction.lastEvent = render.interaction.button + mouseInfo;\n"
"        paraviewObjects[sessionId].sendEvent('MouseEvent', viewId + render.interaction.action + render.interaction.lastEvent);\n"
"        render.interaction.scale = 1;\n"
"        render.interaction.button = event.button + ' ';\n"
"        render.interaction.keys = \"0 0\"\n"
"    }\n"
"    if(render.interaction.isDragging ){\n"
"        var mouseInfoDrag = ((event.screenX-render.interaction.x + render.interaction.xOrigin)/height) + \" \" + (1-(event.screenY-render.interaction.y + render.interaction.yOrigin)/height) + \" \" + render.interaction.keys ;\n"
"        var mouseAction = 'MouseEvent';\n"
"        if(action == 'move') {\n"
"            mouseAction = 'MouseMove';\n"
"        }\n"
"        render.interaction.lastEvent = render.interaction.button + mouseInfoDrag;\n"
"        paraviewObjects[sessionId].sendEvent(mouseAction, viewId + render.interaction.action + render.interaction.lastEvent);\n"
"    }\n"
"    return false;\n"
"}\n"
"";
