/*
 * Copyright (C) 2021-2023 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1.  Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 * 2.  Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#pragma once

#if ENABLE(GPU_PROCESS)

#include "ArgumentCoders.h"
#include "Connection.h"
#include "MessageNames.h"
#include <WebCore/DestinationColorSpace.h>
#include <WebCore/WebGPUTextureFormat.h>
#include <wtf/Forward.h>
#include <wtf/MachSendRight.h>
#include <wtf/RuntimeApplicationChecks.h>
#include <wtf/ThreadSafeRefCounted.h>
#include <wtf/Vector.h>

namespace WebCore {
struct RenderingResourceIdentifierType;
using RenderingResourceIdentifier = AtomicObjectIdentifier<RenderingResourceIdentifierType>;
enum class AlphaPremultiplication : uint8_t;
}

namespace WebKit {
struct WebGPUIdentifierType;
using WebGPUIdentifier = AtomicObjectIdentifier<WebGPUIdentifierType>;
}

namespace Messages {
namespace RemoteCompositorIntegration {

static inline IPC::ReceiverName messageReceiverName()
{
#if ASSERT_ENABLED
    static std::once_flag onceFlag;
    std::call_once(
        onceFlag,
        [&] {
            ASSERT(isInGPUProcess());
        }
    );
#endif
    return IPC::ReceiverName::RemoteCompositorIntegration;
}

#if PLATFORM(COCOA)
class RecreateRenderBuffers {
public:
    using Arguments = std::tuple<int, int, WebCore::DestinationColorSpace, WebCore::AlphaPremultiplication, WebCore::WebGPU::TextureFormat, WebKit::WebGPUIdentifier>;

    static IPC::MessageName name() { return IPC::MessageName::RemoteCompositorIntegration_RecreateRenderBuffers; }
    static constexpr bool isSync = true;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;
    static constexpr bool isStreamEncodable = true;
    static constexpr bool isReplyStreamEncodable = false;
    static constexpr bool isStreamBatched = false;

    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<Vector<MachSendRight>>;
    using Reply = CompletionHandler<void(Vector<MachSendRight>&&)>;
    RecreateRenderBuffers(const int& width, const int& height, const WebCore::DestinationColorSpace& destinationColorSpace, WebCore::AlphaPremultiplication alphaMode, const WebCore::WebGPU::TextureFormat& textureFormat, const WebKit::WebGPUIdentifier& deviceIdentifier)
        : m_width(width)
        , m_height(height)
        , m_destinationColorSpace(destinationColorSpace)
        , m_alphaMode(alphaMode)
        , m_textureFormat(textureFormat)
        , m_deviceIdentifier(deviceIdentifier)
    {
        ASSERT(isInWebProcess());
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
        SUPPRESS_FORWARD_DECL_ARG encoder << m_width;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_height;
        encoder << m_destinationColorSpace;
        encoder << m_alphaMode;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_textureFormat;
        SUPPRESS_FORWARD_DECL_ARG encoder << m_deviceIdentifier;
    }

private:
    SUPPRESS_FORWARD_DECL_MEMBER const int& m_width;
    SUPPRESS_FORWARD_DECL_MEMBER const int& m_height;
    const WebCore::DestinationColorSpace& m_destinationColorSpace;
    WebCore::AlphaPremultiplication m_alphaMode;
    SUPPRESS_FORWARD_DECL_MEMBER const WebCore::WebGPU::TextureFormat& m_textureFormat;
    SUPPRESS_FORWARD_DECL_MEMBER const WebKit::WebGPUIdentifier& m_deviceIdentifier;
};
#endif

class PrepareForDisplay {
public:
    using Arguments = std::tuple<uint32_t>;

    static IPC::MessageName name() { return IPC::MessageName::RemoteCompositorIntegration_PrepareForDisplay; }
    static constexpr bool isSync = true;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;
    static constexpr bool isStreamEncodable = true;
    static constexpr bool isReplyStreamEncodable = false;
    static constexpr bool isStreamBatched = false;

    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<bool>;
    using Reply = CompletionHandler<void(bool)>;
    explicit PrepareForDisplay(uint32_t frameIndex)
        : m_frameIndex(frameIndex)
    {
        ASSERT(isInWebProcess());
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
        encoder << m_frameIndex;
    }

private:
    uint32_t m_frameIndex;
};

class PaintCompositedResultsToCanvas {
public:
    using Arguments = std::tuple<WebCore::RenderingResourceIdentifier, uint32_t>;

    static IPC::MessageName name() { return IPC::MessageName::RemoteCompositorIntegration_PaintCompositedResultsToCanvas; }
    static constexpr bool isSync = true;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;
    static constexpr bool isStreamEncodable = true;
    static constexpr bool isReplyStreamEncodable = true;
    static constexpr bool isStreamBatched = false;

    static constexpr auto callbackThread = WTF::CompletionHandlerCallThread::ConstructionThread;
    using ReplyArguments = std::tuple<>;
    using Reply = CompletionHandler<void()>;
    PaintCompositedResultsToCanvas(const WebCore::RenderingResourceIdentifier& imageBuffer, uint32_t bufferIndex)
        : m_imageBuffer(imageBuffer)
        , m_bufferIndex(bufferIndex)
    {
        ASSERT(isInWebProcess());
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
        SUPPRESS_FORWARD_DECL_ARG encoder << m_imageBuffer;
        encoder << m_bufferIndex;
    }

private:
    SUPPRESS_FORWARD_DECL_MEMBER const WebCore::RenderingResourceIdentifier& m_imageBuffer;
    uint32_t m_bufferIndex;
};

class Destruct {
public:
    using Arguments = std::tuple<>;

    static IPC::MessageName name() { return IPC::MessageName::RemoteCompositorIntegration_Destruct; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;
    static constexpr bool isStreamEncodable = true;
    static constexpr bool isStreamBatched = false;

    Destruct()
    {
        ASSERT(isInWebProcess());
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
    }

private:
};

class UpdateContentsHeadroom {
public:
    using Arguments = std::tuple<float>;

    static IPC::MessageName name() { return IPC::MessageName::RemoteCompositorIntegration_UpdateContentsHeadroom; }
    static constexpr bool isSync = false;
    static constexpr bool canDispatchOutOfOrder = false;
    static constexpr bool replyCanDispatchOutOfOrder = false;
    static constexpr bool deferSendingIfSuspended = false;
    static constexpr bool isStreamEncodable = true;
    static constexpr bool isStreamBatched = false;

    explicit UpdateContentsHeadroom(float headroom)
        : m_headroom(headroom)
    {
        ASSERT(isInWebProcess());
    }

    template<typename Encoder>
    void encode(Encoder& encoder)
    {
        encoder << m_headroom;
    }

private:
    float m_headroom;
};

} // namespace RemoteCompositorIntegration
} // namespace Messages

#endif // ENABLE(GPU_PROCESS)
