<?php

namespace Drupal\Core\Access;

use Drupal\Core\Routing\Access\AccessInterface as RoutingAccessInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Symfony\Component\Routing\Route;
use Symfony\Component\HttpFoundation\Request;

/**
 * Access protection against CSRF attacks.
 *
 * The CsrfAccessCheck is added to any route with the '_csrf_token' route
 * requirement. If a link/url to a protected route is generated using the
 * url_generator service, a valid token will be added automatically. Otherwise,
 * a valid token can be generated by the csrf_token service using the route's
 * path (without leading slash) as the argument when generating the token. This
 * token can then be added as the 'token' query parameter when accessing the
 * protected route.
 *
 * @see \Drupal\Core\Access\RouteProcessorCsrf
 * @see \Drupal\Core\Access\CsrfTokenGenerator
 * @see https://www.drupal.org/docs/8/api/routing-system/access-checking-on-routes/csrf-access-checking
 */
class CsrfAccessCheck implements RoutingAccessInterface {

  use RoutePathGenerationTrait;

  /**
   * The CSRF token generator.
   */
  protected CsrfTokenGenerator $csrfToken;

  /**
   * Constructs a CsrfAccessCheck object.
   *
   * @param \Drupal\Core\Access\CsrfTokenGenerator $csrf_token
   *   The CSRF token generator.
   */
  public function __construct(CsrfTokenGenerator $csrf_token) {
    $this->csrfToken = $csrf_token;
  }

  /**
   * Checks access based on a CSRF token for the request.
   *
   * @param \Symfony\Component\Routing\Route $route
   *   The route to check against.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match object.
   *
   * @return \Drupal\Core\Access\AccessResultInterface
   *   The access result.
   */
  public function access(Route $route, Request $request, RouteMatchInterface $route_match) {
    $path = $this->generateRoutePath($route, $route_match->getRawParameters()->all());
    if ($this->csrfToken->validate($request->query->get('token', ''), $path)) {
      $result = AccessResult::allowed();
    }
    else {
      $result = AccessResult::forbidden($request->query->has('token') ? "'csrf_token' URL query argument is invalid." : "'csrf_token' URL query argument is missing.");
    }
    // Not cacheable because the CSRF token is highly dynamic.
    return $result->setCacheMaxAge(0);
  }

}
