// Copyright 2019 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "chrome/browser/local_search_service/local_search_service_proxy_factory.h"

#include "chrome/browser/local_search_service/local_search_service_proxy.h"
#include "chrome/browser/profiles/profile.h"
#include "components/keyed_service/content/browser_context_dependency_manager.h"

#if defined(OS_CHROMEOS)
#include "chrome/browser/chromeos/profiles/profile_helper.h"
#endif  // OS_CHROMEOS

namespace local_search_service {

LocalSearchServiceProxy* LocalSearchServiceProxyFactory::GetForProfile(
    Profile* profile) {
  return static_cast<LocalSearchServiceProxy*>(
      LocalSearchServiceProxyFactory::GetInstance()
          ->GetServiceForBrowserContext(profile, true /* create */));
}

LocalSearchServiceProxyFactory* LocalSearchServiceProxyFactory::GetInstance() {
  return base::Singleton<LocalSearchServiceProxyFactory>::get();
}

LocalSearchServiceProxyFactory::LocalSearchServiceProxyFactory()
    : BrowserContextKeyedServiceFactory(
          "LocalSearchServiceProxy",
          BrowserContextDependencyManager::GetInstance()) {}

LocalSearchServiceProxyFactory::~LocalSearchServiceProxyFactory() = default;

content::BrowserContext* LocalSearchServiceProxyFactory::GetBrowserContextToUse(
    content::BrowserContext* context) const {
  Profile* const profile = Profile::FromBrowserContext(context);
  if (!profile || profile->IsSystemProfile()) {
    // A system profile is a non-browsing profile and is not associated with a
    // user. Currently all search requests are generated by users, hence we do
    // not support system profiles now.
    return nullptr;
  }

#if defined(OS_CHROMEOS)
  if (chromeos::ProfileHelper::IsSigninProfile(profile)) {
    // We do not expect any search requests during user sign-in. Hence we do not
    // support sign-in profiles now.
    return nullptr;
  }
#endif  // OS_CHROMEOS

  // The service should exist in incognito mode.
  return context;
}

KeyedService* LocalSearchServiceProxyFactory::BuildServiceInstanceFor(
    content::BrowserContext* context) const {
  return new LocalSearchServiceProxy(Profile::FromBrowserContext(context));
}

}  // namespace local_search_service
