// SPDX-FileCopyrightText: 2011 - 2022 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: LGPL-3.0-or-later

#include "floatingpreview.h"
#include "appsnapshot.h"
#include "previewcontainer.h"

#include <DStyle>
#include <DGuiApplicationHelper>

#include <QGraphicsEffect>
#include <QPainter>
#include <QVBoxLayout>
#include <QPainterPath>

DWIDGET_USE_NAMESPACE

FloatingPreview::FloatingPreview(QWidget *parent)
    : QWidget(parent)
    , m_closeBtn3D(new DIconButton(DStyle::StandardPixmap::SP_CloseButton, this))
    , m_titleLabel(new DLabel(this))
{
    m_closeBtn3D->setObjectName("closebutton-3d");
    m_closeBtn3D->setFixedSize(24, 24);
    m_closeBtn3D->setIconSize(QSize(24, 24));
    m_closeBtn3D->setFlat(true);

    m_titleLabel->setBackgroundRole(QPalette::Base);
    m_titleLabel->setForegroundRole(QPalette::Text);
    m_titleLabel->setFocusPolicy(Qt::NoFocus);
    m_titleLabel->setAttribute(Qt::WA_TransparentForMouseEvents);

    QVBoxLayout *centralLayout = new QVBoxLayout;
    centralLayout->addWidget(m_closeBtn3D);
    centralLayout->setAlignment(m_closeBtn3D, Qt::AlignRight | Qt::AlignTop);
    centralLayout->addWidget(m_titleLabel);
    centralLayout->setAlignment(m_titleLabel, Qt::AlignCenter | Qt::AlignBottom);
    centralLayout->addSpacing(TITLE_MARGIN);
    centralLayout->setMargin(0);
    centralLayout->setSpacing(0);

    setLayout(centralLayout);

    connect(m_closeBtn3D, &DIconButton::clicked, this, &FloatingPreview::onCloseBtnClicked);
}

WId FloatingPreview::trackedWid() const
{
    Q_ASSERT(!m_tracked.isNull());

    return m_tracked->wid();
}

AppSnapshot *FloatingPreview::trackedWindow()
{
    return m_tracked;
}

void FloatingPreview::setFloatingTitleVisible(bool bVisible)
{
    m_titleLabel->setVisible(bVisible);
}

void FloatingPreview::trackWindow(AppSnapshot *const snap)
{
    if (!snap)
        return;

    if (!m_tracked.isNull())
        m_tracked->removeEventFilter(this);

    snap->installEventFilter(this);
    m_tracked = snap;

    m_closeBtn3D->setVisible(m_tracked->closeAble());

    // 显示此标题的前提条件：配置了标题跟随鼠标显示
    // 此对象是共用的，鼠标移动到哪个预览图，title就移动到哪里显示，所以他的text统一snap获取，不再重复计算显示长度
    m_titleLabel->setText(snap->appTitle());
    update();
}

void FloatingPreview::paintEvent(QPaintEvent *e)
{
    QWidget::paintEvent(e);

    if (m_tracked.isNull())
        return;

    const QImage &snapshot = m_tracked->snapshot();

    if (snapshot.isNull())
        return;

    QPainter painter(this);
    QColor color;

    if (DGuiApplicationHelper::instance()->themeType() == DGuiApplicationHelper::ColorType::DarkType) {
        color = Qt::white;
        color.setAlphaF(0.1);
        painter.setPen(color);
        color = Qt::black;
    } else {
        color = Qt::black;
        color.setAlphaF(0.1);
        painter.setPen(color);
        color = Qt::white;
    }

    color.setAlphaF(0.8);
    painter.setBrush(color);
    painter.setRenderHint(QPainter::Antialiasing);
    painter.drawRoundedRect(m_titleLabel->geometry().marginsAdded(QMargins(BTN_TITLE_MARGIN, BTN_TITLE_MARGIN, BTN_TITLE_MARGIN, BTN_TITLE_MARGIN))
                            , 10, 10, Qt::SizeMode::AbsoluteSize);

    const QRectF r = rect().marginsRemoved(QMargins(BORDER_MARGIN, BORDER_MARGIN, BORDER_MARGIN, BORDER_MARGIN));

    DStyleHelper dstyle(style());
    const int radius = dstyle.pixelMetric(DStyle::PM_FrameRadius);

    // 选中外框
    QPen pen;
    pen.setColor(palette().highlight().color());
    pen.setWidth(dstyle.pixelMetric(DStyle::PM_FocusBorderWidth));
    painter.setPen(pen);
    painter.setBrush(Qt::NoBrush);
    painter.drawRoundedRect(r, radius, radius);

    //绘制标题背景模糊效果
    color = QColor(Qt::black);
    const qreal initAlpha = 0.05;
    color.setAlphaF(initAlpha);
    QPen curPen;
    for (int i = 1; i < 6; i++) {
        QPainterPath path;
        QRect r(m_titleLabel->x() - BTN_TITLE_MARGIN - i, m_titleLabel->y() - BTN_TITLE_MARGIN - i, m_titleLabel->size().width() + 2 * BTN_TITLE_MARGIN + 2 * i
                , m_titleLabel->size().height() + 2 * BTN_TITLE_MARGIN + 2 * i);
        path.addRoundedRect(r, 10, 10);
        curPen.setColor(color);
        curPen.setWidth(0);
        painter.setPen(curPen);
        painter.drawPath(path);
        qreal alpha = initAlpha - 0.01 * i;
        color.setAlphaF(alpha);
    }
}

void FloatingPreview::mouseReleaseEvent(QMouseEvent *e)
{
    QWidget::mouseReleaseEvent(e);

    if (m_tracked) {
        emit m_tracked->clicked(m_tracked->wid());
    }
}

bool FloatingPreview::eventFilter(QObject *watched, QEvent *event)
{
    if (watched == m_tracked) {
        if (event->type() == QEvent::Destroy) {
            // 此处需要置空，否则当Destroy事件响应结束后，会在FloatingPreview::hideEvent使用m_tracked野指针
            m_tracked = nullptr;
            hide();
        }
    }

    return QWidget::eventFilter(watched, event);
}

void FloatingPreview::hideEvent(QHideEvent *event)
{
    if (m_tracked) {
        m_tracked->setContentsMargins(0, 0, 0, 0);
        m_tracked->setWindowState();
    }

    QWidget::hideEvent(event);
}

void FloatingPreview::onCloseBtnClicked()
{
    Q_ASSERT(!m_tracked.isNull());

    m_tracked->closeWindow();
}
