package modules

import (
	"encoding/json"

	Logger "pkg.deepin.io/daemon/sync/infrastructure/log"
	"pkg.deepin.io/daemon/sync/infrastructure/storage"
	"pkg.deepin.io/daemon/sync/infrastructure/utils"
	"pkg.deepin.io/daemon/sync/modules/background"
	. "pkg.deepin.io/daemon/sync/modules/model"
	"pkg.deepin.io/daemon/sync/modules/network"
)

func (m *Manager) SyncModule(cloud *utils.CloudStorage, item DataIFC,
	version string) (string, *SyncError) {
	var valid = true
	var cloudVersion string
	var cacheVersion string
	var upData []byte
	var cacheData []byte
	var rIFC, curIFC, cacheIFC DataIFC
	var err error
	var name = item.Name()

	// got current data
	curIFC, curData, serr := m.getModuleInfo(item)
	if serr != nil {
		return "", serr
	}
	if curIFC == nil || len(curData) == 0 {
		// disabled or no register
		Logger.Debug("not get current data: disabled or no register for ", name)
		return "", nil
	}

	// got cloud data
	if len(version) != 0 {
		rIFC, cloudVersion, serr = m.getModuleCloudInfo(item, version, cloud)
		if serr != nil {
			return "", serr
		}
	}
	if rIFC == nil {
		// never uploaded, uploading now
		err := m.handleBackgroundFiles(name, curIFC, true)
		if err != nil {
			return "", NewSyncError(CodeModuleDownloadErr, name, err.Error())
		}
		// background changed after parse
		if name == background.Name {
			curData, _ = json.Marshal(curIFC)
		}
		upData = curData
		cacheData = curData
		goto upload
	}

	// got cache data
	cacheIFC, cacheVersion, valid, serr = m.getModuleCacheInfo(item, *m.userID)
	if serr != nil {
		return "", serr
	}
	if !valid {
		err := m.handleBackgroundFiles(name, rIFC, false)
		if err != nil {
			return "", NewSyncError(CodeModuleDownloadErr, name, err.Error())
		}
		// TODO(jouyouyun): simple background data
		// merge network wireless data directly, then upload
		netChanged := m.workaroundNetwork(rIFC, curIFC)
		if !netChanged {
			return "", m.setModuleAndSaveCache(name,
				jsonMarshal(rIFC.GenCache(curIFC)), cloudVersion)
		}
		Logger.Debug("network connections changed, upload and set")
		upData = jsonMarshal(rIFC)
		cacheData = jsonMarshal(rIFC.GenCache(curIFC))
		goto upload
	}

	Logger.Debug("[Sync Module] version compare:", cloudVersion, cacheVersion)
	// compare remote, current and cache data
	if curIFC.Equal(cacheIFC) {
		Logger.Debug("[Sync Module] cur  equal cache:", name)
		// the local data no changes
		if cloudVersion <= cacheVersion {
			return "", nil
		}
		err := m.handleBackgroundFiles(name, rIFC, false)
		if err != nil {
			return "", NewSyncError(CodeModuleDownloadErr, name, err.Error())
		}
		return "", m.setModuleAndSaveCache(name,
			jsonMarshal(rIFC.GenCache(curIFC)), cloudVersion)
	}

	// the local data has changes
	// firstly merge the no changes options, then upload data
	rIFC.Merge(curIFC, cacheIFC)
	// upload backfround files
	err = m.handleBackgroundFiles(name, rIFC, true)
	if err != nil {
		return "", NewSyncError(CodeModuleDownloadErr, name, err.Error())
	}
	// TODO(jouyouyun): simple background data
	upData = jsonMarshal(rIFC)
	cacheData = jsonMarshal(rIFC.GenCache(curIFC))
upload:
	// Logger.Info("\t[Sync Module] upload data:", string(upData))
	ver, err := cloud.Set(name, upData, &storage.SetOptions{NewVersion: true})
	Logger.Debug("[Sync Module] upload module:", name, ver, err)
	if err != nil {
		return "", NewSyncError(CodeModuleUploadErr, name, err.Error())
	}
	m.setModuleAndSaveCache(name, cacheData, ver)
	return ver, nil
}

func (m *Manager) handleBackgroundFiles(name string, ifc DataIFC, upload bool) error {
	if name != background.Name {
		return nil
	}
	info := ifc.(*background.Data)
	if len(info.SlideShow) != 0 {
		// auto change background
		return nil
	}
	cloud, err := m.getCloudStorage()
	if err != nil {
		return err
	}
	err = info.Download(cloud)
	if err != nil {
		return err
	}
	// Logger.Info("[DEBUG] handle background files:", string(jsonMarshal(info)))
	if !upload {
		return nil
	}
	return info.Upload(cloud)
}

func (m *Manager) setModuleAndSaveCache(name string, data []byte,
	version string) *SyncError {
	err := m.setModuleData(name, data)
	if err != nil {
		return NewSyncError(CodeModuleSetErr,
			name, err.Error())
	}
	utils.SaveLocalCache(name, *m.userID, &utils.LocalCache{
		Data:    data,
		Version: version,
	})
	return nil
}

func (m *Manager) getIndexCloudInfo(cloud *utils.CloudStorage) (*ModuleIndex, *SyncError) {
	idxData, err := cloud.GetLatest(cloudKeyIndex)
	if err != nil {
		if err != storage.ErrNotFound {
			return nil, NewSyncError(CodeIndexDownloadErr,
				cloudKeyIndex, err.Error())
		}
		idxData = &storage.GetResult{}
	}
	var idxInfo ModuleIndex
	if len(idxData.Content) != 0 {
		err = json.Unmarshal(idxData.Content, &idxInfo)
		if err != nil {
			return nil, NewSyncError(CodeIndexFormatErr,
				cloudKeyIndex, err.Error())
		}
	}
	return &idxInfo, nil
}

func (m *Manager) getModuleInfo(item DataIFC) (DataIFC, []byte, *SyncError) {
	name := item.Name()
	data, err := m.getModuleData(name)
	if err != nil {
		return nil, nil, NewSyncError(CodeModuleGetErr, name, err.Error())
	}
	if len(data) == 0 {
		// disabled or no register
		return nil, nil, nil
	}
	// Logger.Debug("[Module Current] get:", name, string(data))
	var info = item.Model()
	err = json.Unmarshal(data, info)
	if err != nil {
		return nil, nil, NewSyncError(CodeModuleFormatErr, name, err.Error())
	}
	ifc := item.ToIFC(info)
	if ifc.Name() == background.Name {
		bg := ifc.(*background.Data)
		bg.Parse()
	}
	return ifc, data, nil
}

func (m *Manager) getModuleCloudInfo(item DataIFC, version string,
	cloud *utils.CloudStorage) (DataIFC, string, *SyncError) {
	name := item.Name()
	ret, err := cloud.Get(name, &storage.GetOptions{Version: version})
	if err != nil {
		if err != storage.ErrNotFound {
			return nil, "", NewSyncError(CodeModuleDownloadErr,
				name, err.Error())
		}
		ret = &storage.GetResult{}
	}
	if len(ret.Content) == 0 {
		return nil, "", nil
	}
	// Logger.Debug("[Module Cloud] get:", name, string(ret.Content))
	var info = item.Model()
	err = json.Unmarshal(ret.Content, info)
	if err != nil {
		return nil, "", NewSyncError(CodeModuleFormatErr, name, err.Error())
	}
	ifc := item.ToIFC(info)
	return ifc, ret.Version, nil
}

func (m *Manager) getModuleCacheInfo(item DataIFC, uid int64) (DataIFC, string, bool, *SyncError) {
	var info = item.Model()
	name := item.Name()
	valid := true
	ret, _ := utils.LoadLocalCache(name, *m.userID)
	if ret == nil || len(ret.Data) == 0 {
		// never synchronized, synchronizing now and save cache
		valid = false
	} else {
		// Logger.Info("[Modules] [Sync] cache data:", name, string(ret.Data))
		err := json.Unmarshal(ret.Data, info)
		if err != nil {
			// local cache invalid, replace it with cloud data
			valid = false
		}
	}
	if !valid {
		return nil, "", false, nil
	}
	// Logger.Debug("[Module Cahce] get:", name, string(ret.Data))
	return item.ToIFC(info), ret.Version, true, nil
}

func (m *Manager) workaroundNetwork(rIFC, curIFC DataIFC) bool {
	// Logger.Debug("[WORKSROUND Network] name:", rIFC.Name())
	if rIFC.Name() != network.Name {
		return false
	}
	rinfo := rIFC.(*network.Data)
	cinfo := curIFC.(*network.Data)
	// Logger.Debug("[WORKSROUND NETWORK] rlen, clen:", len(rinfo.Connections), len(cinfo.Connections))
	if len(rinfo.Connections) == 0 {
		rinfo.Connections = cinfo.Connections
		return true
	}
	if len(cinfo.Connections) == 0 {
		return false
	}
	ret := false
	for _, conn := range cinfo.Connections {
		tmp := rinfo.Connections.Get(conn.Type, conn.Filename)
		if tmp != nil {
			continue
		}
		rinfo.Connections = append(rinfo.Connections, conn)
		ret = true
	}
	// Logger.Debug("[WORKSROUND NETWORK] rlen after changed:", len(rinfo.Connections))
	return ret
}
