/*
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <stdio.h>

#include "glue.h"

#include "sig_match.h"

static void
sig_match_send_event(
	const struct sig_match* m, 
	void* s, 
	bool visible, 
	uint16_t x, uint16_t y,
	uint16_t w, uint16_t h
)
{
	unsigned int nr;

	for (nr = 0; ; nr++) {
		if (nr == m->nmembers) {
			return;
		}
		if (m->member[nr].s == s) {
			continue;
		}
		if (m->member[nr].f->event) {
			m->member[nr].f->event(
				m->member[nr].s, 
				visible, 
				x, y, 
				w, h);
		}
	}
}

void 
sig_match_event(
	const struct sig_match* m, 
	void *s, 
	uint16_t x, uint16_t y,
	uint16_t w, uint16_t h
)
{
	sig_match_send_event(m, s, true, x, y, w, h);
}

void
sig_match_invisible(const struct sig_match* m, void *s)
{
	sig_match_send_event(m, s, false, 0, 0, 0, 0);
}

void
sig_match_add_match(const struct sig_match* m, void *s, const char* image)
{
	unsigned int nr;
	for (nr = 0; ; nr++) {
		if (nr == m->nmembers) {
			return;
		}

		if (m->member[nr].s == s) {
			continue;
		}

		if (m->member[nr].f->add_match) {
			m->member[nr].f->add_match(m->member[nr].s, image);
		}
	}
}

void
sig_match_add_match_partial(
	const struct sig_match* m,
	void *s,
	const char* image,
	uint16_t x,
	uint16_t y,
	uint16_t w,
	uint16_t h
)
{
	unsigned int nr;
	for (nr = 0; ; nr++) {
		if (nr == m->nmembers) {
			return;
		}

		if (m->member[nr].s == s) {
			continue;
		}

		if (m->member[nr].f->add_match_partial) {
			m->member[nr].f->add_match_partial(m->member[nr].s,
								image, 
								x, y, 
								w, h);
		}
	}

}

void
sig_match_remove_match(const struct sig_match* m, void *s)
{
	unsigned int nr;
	for (nr = 0; ; nr++) {
		if (nr == m->nmembers) {
			return;
		}

		if (m->member[nr].s == s) {
			continue;
		}

		if (m->member[nr].f->remove_match) {
			m->member[nr].f->remove_match(m->member[nr].s);
		}
	}
}

void
sig_match_connect(
	struct sig_match* m, 
	void *s, 
	const struct sig_match_funcs *f
)
{
	assert(m);
	assert(m->type == SIG_GEN_MATCH);
	assert(m->nmembers < sizeof(m->member) / sizeof(m->member[0]));

	m->member[m->nmembers].f = f;
	m->member[m->nmembers].s = s;
	m->nmembers++;
}

struct sig_match* 
sig_match_create(const char *name)
{
	struct sig_match* m;

	m = shm_alloc(sizeof(*m));
	assert(m);

	m->type = SIG_GEN_MATCH;
	m->nmembers = 0;

	return m;
}

void
sig_match_destroy(struct sig_match *sig)
{
	assert(sig);
	assert(sig->type == SIG_GEN_MATCH);

	shm_free(sig);
}

void
sig_match_suspend(struct sig_match *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_suspend(b, size, fSig);
}

void
sig_match_resume(struct sig_match *b, FILE *fSig)
{
	size_t size = sizeof(*b);
	
	generic_resume(b, size, fSig);
}
