/*
 * Copyright (c) 2007, intarsys consulting GmbH
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * - Redistributions of source code must retain the above copyright notice,
 *   this list of conditions and the following disclaimer.
 *
 * - Redistributions in binary form must reproduce the above copyright notice,
 *   this list of conditions and the following disclaimer in the documentation
 *   and/or other materials provided with the distribution.
 *
 * - Neither the name of intarsys nor the names of its contributors may be used
 *   to endorse or promote products derived from this software without specific
 *   prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */
package de.intarsys.tools.locator;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;

import de.intarsys.tools.stream.StreamTools;

/**
 * Tool methods for dealing with {@link ILocator}.
 * 
 */
public class LocatorTools {

	public static final String PATH_SEPARATOR = ";"; //$NON-NLS-1$

	public static ILocator[] createLocators(String paths,
			ILocatorFactory factory) throws IOException {
		if (factory == null) {
			factory = LocatorFactory.get();
		}
		List<ILocator> locators = new ArrayList<ILocator>();
		for (StringTokenizer t = new StringTokenizer(paths, PATH_SEPARATOR); t
				.hasMoreTokens();) {
			String path = t.nextToken();
			locators.add(factory.createLocator(path));
		}
		ILocator[] result = new ILocator[locators.size()];
		return locators.toArray(result);
	}

	public static final byte[] getBytes(ILocator locator) throws IOException {
		if (locator instanceof ByteArrayLocator) {
			return ((ByteArrayLocator) locator).getBytes();
		}
		InputStream is = null;
		try {
			is = locator.getInputStream();
			return StreamTools.toByteArray(is);
		} finally {
			StreamTools.close(is);
		}
	}

	public static final File getFile(ILocator locator) throws IOException {
		if (locator instanceof FileLocator) {
			return ((FileLocator) locator).getFile();
		}
		// add conversion (temp file) code
		throw new IOException("can't create file for '" + locator.getFullName() //$NON-NLS-1$
				+ "'"); //$NON-NLS-1$
	}

	private LocatorTools() {
	}
}
