// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System;
using System.Diagnostics;

using Internal.Text;
using Internal.TypeSystem;

namespace ILCompiler.DependencyAnalysis
{
    public class EETypeOptionalFieldsNode : ObjectNode, ISymbolDefinitionNode
    {
        private EETypeNode _owner;

        public EETypeOptionalFieldsNode(EETypeNode owner)
        {
            _owner = owner;
        }

        public override ObjectNodeSection Section
        {
            get
            {
                if (_owner.Type.Context.Target.IsWindows)
                    return ObjectNodeSection.FoldableReadOnlyDataSection;
                else
                    return ObjectNodeSection.DataSection;
            }
        }

        public override bool StaticDependenciesAreComputed => true;

        public void AppendMangledName(NameMangler nameMangler, Utf8StringBuilder sb)
        {
            sb.Append("__optionalfields_");
            _owner.AppendMangledName(nameMangler, sb);
        }
        public int Offset => 0;
        public override bool IsShareable => true;

        protected override string GetName(NodeFactory factory) => this.GetMangledName(factory.NameMangler);

        public override bool ShouldSkipEmittingObjectNode(NodeFactory factory)
        {
            _owner.ComputeOptionalEETypeFields(factory, relocsOnly: false);
            return _owner.ShouldSkipEmittingObjectNode(factory) || !_owner.HasOptionalFields;
        }

        public override ObjectData GetData(NodeFactory factory, bool relocsOnly = false)
        {
            ObjectDataBuilder objData = new ObjectDataBuilder(factory, relocsOnly);
            objData.RequireInitialPointerAlignment();
            objData.AddSymbol(this);

            if (!relocsOnly)
            {
                _owner.ComputeOptionalEETypeFields(factory, relocsOnly: false);
                objData.EmitBytes(_owner.GetOptionalFieldsData());
            }
            
            return objData.ToObjectData();
        }

        protected internal override int ClassCode => 821718028;

        protected internal override int CompareToImpl(SortableDependencyNode other, CompilerComparer comparer)
        {
            return SortableDependencyNode.CompareImpl(_owner, ((EETypeOptionalFieldsNode)other)._owner, comparer);
        }
    }
}
