/*---------------------------------------------------------------------------*\
  =========                 |
  \\      /  F ield         | OpenFOAM: The Open Source CFD Toolbox
   \\    /   O peration     |
    \\  /    A nd           | Copyright (C) 2016 OpenCFD Ltd
     \\/     M anipulation  |
-------------------------------------------------------------------------------
License

    OpenFOAM is free software: you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    OpenFOAM is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
    FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
    for more details.

    You should have received a copy of the GNU General Public License
    along with OpenFOAM.  If not, see <http://www.gnu.org/licenses/>.

\*---------------------------------------------------------------------------*/

#include "eddyDissipationDiffusionModel.H"

namespace Foam
{
namespace combustionModels
{

// * * * * * * * * * * * * * * * * Constructors  * * * * * * * * * * * * * * //

template<class ReactionThermo, class ThermoType>
eddyDissipationDiffusionModel<ReactionThermo, ThermoType>::
eddyDissipationDiffusionModel
(
    const word& modelType,
    ReactionThermo& thermo,
    const compressibleTurbulenceModel& turb,
    const word& combustionProperties
)
:
    eddyDissipationModelBase<ReactionThermo, ThermoType>
    (
        modelType,
        thermo,
        turb,
        combustionProperties
    ),
    Cd_(this->coeffs().getScalar("Cd"))
{}


// * * * * * * * * * * * * * * * * Destructor  * * * * * * * * * * * * * * * //

template<class ReactionThermo, class ThermoType>
eddyDissipationDiffusionModel<ReactionThermo, ThermoType>::
~eddyDissipationDiffusionModel()
{}


// * * * * * * * * * * * * * * Member Functions  * * * * * * * * * * * * * * //


template<class ReactionThermo, class ThermoType>
Foam::tmp<Foam::volScalarField>
eddyDissipationDiffusionModel<ReactionThermo, ThermoType>::timeScale()
{
    return (max(this->rtTurb(), this->rtDiff()));
}


template<class ReactionThermo, class ThermoType>
Foam::tmp<Foam::volScalarField>
eddyDissipationDiffusionModel<ReactionThermo, ThermoType>::rtDiff() const
{
    tmp<volScalarField> tdelta
    (
        new volScalarField
        (
            IOobject
            (
                "tdelta",
                this->mesh().time().timeName(),
                this->mesh(),
                IOobject::NO_READ,
                IOobject::NO_WRITE
            ),
            this->mesh(),
            dimensionedScalar(dimLength, Zero),
            zeroGradientFvPatchScalarField::typeName
        )
    );

    volScalarField& delta = tdelta.ref();
    delta.ref() = pow(this->mesh().V(), 1.0/3.0);
    delta.correctBoundaryConditions();

    // NOTE: Assume Prt = 1
    return Cd_*this->turbulence().nuEff()/sqr(delta);
}


template<class ReactionThermo, class ThermoType>
bool eddyDissipationDiffusionModel<ReactionThermo, ThermoType>::read()
{
    if (eddyDissipationModelBase<ReactionThermo, ThermoType>::read())
    {
        this->coeffs().readEntry("Cd", Cd_);
        return true;
    }
    else
    {
        return false;
    }
}


// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //

} // End namespace combustionModels
} // End namespace Foam

// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * //
