///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
/*Prog:robin
NAME: @code{robin} - Robin boundary conditions 
@cindex Robin boundary conditions
@cindex RHEOPATH environment variable
@apindex P1
@apindex P2

SYNOPSIS:
@example
robin @{@var{options}@}* [-app @var{approx}] @var{mesh}[.geo[.gz]] @{@var{boundary}@}*
@end example

EXAMPLE:
@example
robin -I../data -app P1 full_carre_0.1 wall | field -I../data -
@end example

DESCRIPTION:       
@iftex
@tex
 find $u\in H^1(\Omega)$ such that:
   $$
   \left\{ 
     \matrix{
        \hfill                                                   -\Delta u & = & f  \hbox{ in } \Omega \hfill \cr
        \hfill {\displaystyle {{\partial u}\over {\partial n}}} + \alpha u & = & g  \hbox{ on } \Gamma_R \hfill \cr
	\hfill {\displaystyle {{\partial u}\over {\partial n}}}            & = & 0  \hbox{ on } \Gamma_N \hfill 
     }
   \right. \phantom{@}}
   $$
 where $\Gamma_R \cup \Gamma_N = \partial \Omega$
 and $\Gamma_R \cap \Gamma_N = \emptyset$,
 $\alpha > 0$ is constant, and
 $f$ and $g$ are given.
 By default, $\alpha=1$, $f=1$ and $g=0$.
@end tex
@end iftex

@ifinfo
find u in H1(Omega) such that:
@example
     - Laplacian u = f  in  Omega

               d u
     alpha u + --- = g  on  Gamma_R
               d n

               d u 
               --- = 0  on  Gamma_N
               d n
@end example
where Gamma_R and Gamma_N are a partition of the
boundary Gamma of Omega, alpha > 0 is constant
and f and g are given.
By defaut, alpha=1, f=1 and g=0.
@end ifinfo

@noindent
This default could be changed by a suitable
choice of the command-line options.

OPTIONS:
@table @code
  @itemx @var{filename}
  	specifies the name of the file containing
      	the input mesh. The @file{.geo} extension is assumed.
  @itemx -
  	read mesh on standard input instead on a file.
  @itemx -I@var{geodir} 
      	add @var{geodir} to the mesh search path.
      	This mechanism initializes a search path given by the environment variable
      	@samp{RHEOPATH}. If the environment variable 
	@samp{RHEOPATH} is not set, the
      	default value is the current directory
	(@pxref{geo class}).
  @itemx @var{domain}
  	specifies the name of the domain where Robin boundary conditions
	are used. 
        The rest of the boundary will use a homogeneous Neumann boundary
	condition.
	At least, one Robin boundary condition domain is required.
  @itemx -app @var{approx}
	Set the finite element approximation.
	Only @code{P1} and @code{P2} approximation are yet supported here.
  @itemx -alpha @var{float}
	Set the value for the @code{alpha} parameter related to the
	Robin boundary condition. Note that alpha may be strictly positive.
@end table

OPTIONS FOR TESTS:
When the domain is the square ]0,1[x]0,1[
error is also computed. In that case, 
@var{f} and @var{g} are chosen so that the exact solution is
explicitely known. This computation is usefull for non-regression test purpose.
These cases are specified by using some suitable command-line options.

@table @code
    @itemx -Dir
    @itemx -reg	
    @itemx -cos	
	Select respectively the @dfn{Dirichlet},
	the @dfn{regular}
	or the @dfn{cosinus} test case.
	These test cases are described in
	the next section.
    @itemx -exact
        Output on stdout the exact solution interpolated on the mesh, using
  	the specified element. By default, the approximate solution
	is printed to stdout.
    @itemx -diff
        Output on stdout the difference @code{uh-pi_h(u)} between
	the approximate and the interpolated exact solutions.
    @itemx -round
	Round the output, in order for the non-regression tests to
	be portable across platforms. Floating point are machine-dependent.
@end table
AUTHOR: Pierre.Saramito@@imag.fr
End:
*/

// ============================================================================
//  Includes
// ============================================================================

#include "rheolef/rheolef.h"
#include "rheolef/trace.h"
using namespace rheolef;
using namespace std;

Float epsilon = numeric_limits<Float>::epsilon();

// ============================================================================
//  The exact solution
// ============================================================================

Float alpha = 1;

// ----------------------------------------------------------------------------
//  1. Dirichlet case
// ----------------------------------------------------------------------------

Float u_Dir_2d (const point& x) {
  return (x[0]*(1-x[0]) + x[1]*(1-x[1]))/4 ;
}
Float g_Dir_2d (const point& x) {
  // warning: fit g(x,y) on the unit square only !
  return alpha*u_Dir_2d(x) - 1./4;
}

/*Prog:robin
DIRICHLET TEST CASE IN A SQUARE:

@c ifnottex is not recognized yet by texi2html
@ifinfo
When Omega=]0,1[x][0,1[, f=1, and
@example
	   | (alpha x (1-x) - 1)/4  for x in ]0,1[ and y = 0, 1
 g(x,y) = < 
	   | (alpha y (1-y) - 1)/4  for y in ]0,1[ and x = 0, 1
@end example
the solution is known explicitely
@example
           x (1-x) + y (1-y) 
  u(x,y) = -----------------
	           4
@end example
@end ifinfo

@iftex
@tex
When $\Omega=]0,1[^2$, $f=1$ and
  $$
  g(x,y) = \left\{ \matrix{
    (\alpha x(1-x) - 1)/4  & \hbox{ for } x \in ]0,1[ \hbox{ and } y = 0, 1 \cr
    (\alpha y(1-y) - 1)/4  & \hbox{ for } y \in ]0,1[ \hbox{ and } x = 0, 1
  } \right. \phantom{@}}
  $$
then, the solution is known explicitely
  $$
  u(x,y) = {{x(1-x) + y(1-y)}\over {4}}
  $$
@end tex
@end iftex

@noindent
Not that u is a second-order polynomial
with respect to each variable x and y.

@noindent
With alpha=1 and the @code{P1} approximation on a uniform mesh, we obtain
whith e = (u - u_ex)
@example
# h    |e|_(2;omega) |e|_(2;gamma) |e|_(2;1;omega) |e|_(inf;omega) |e|_(infty;gamma)
0.5      0.00557069    0.0111029    0.0567045      0.0112179       0.00961538
0.25     0.00131993    0.00292313   0.0201551      0.00373274      0.00365063
0.125    0.000233789   0.000630589  0.0058884      0.00099301      0.00099301
0.0625   4.99283e-05   0.00011468   0.00257175     0.000269382     0.00025238
0.03125  1.21721e-05   2.86123e-05  0.00109265     6.81078e-05     6.81078e-05
0.015625 3.11578e-06   5.37344e-06  0.000584626    1.86111e-05     1.62315e-05
@end example
and thus we observe a O(h^2) convergence for
both L^2 and L^infty norms, and a O(h) convergence for the H^1 norm.

When switching to the @code{P2} approximation, we obtain
the exact solution, up to the machine precision: indeed, the
solution is a second order polynom.
END: */

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
//  2. Regular square case
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
static const Float pi = acos(Float(-1.)); // pi
static const Float omega = pi;

Float u_reg_2d (const point& x) {
  return sin(omega*(x[0]*(1-x[0]) + x[1]*(1-x[1])));
}
Float f_reg_2d (const point& x) {
  Float z = omega*(x[0]*(1-x[0]) + x[1]*(1-x[1]));
  Float t = sqr(1-2*x[0]) + sqr(1-2*x[1]);
  return 4*omega*cos(z) + sqr(omega)*t*sin(z);
}
Float g_reg_2d (const point& x) {
  // warning: fit g(x,y) on the unit square only !
  Float z = omega*(x[0]*(1-x[0]) + x[1]*(1-x[1]));
  return alpha*sin(z) - omega*cos(z);
}

/*Prog:robin
REGULAR TEST CASE IN A SQUARE:
Now we switch to a non-polynomial regular solution.

@c ifnottex is not recognized yet by texi2html
@ifinfo
When Omega=]0,1[x][0,1[, we choose the exact solution
@example
  u(x,y) = sin((x(1-x)+y(1-y))pi)
@end example
and deduce the right-hand sides f and g.
@end ifinfo

@iftex
@tex
When $\Omega=]0,1[^2$, we choose the exact solution
  $$
  u(x,y) = \sin(\pi(x(1-x)+y(1-y)))
  $$
and deduce the right-hand sides $f$ and $g$.
@end tex
@end iftex

@noindent
With alpha=1 and the @code{P1} approximation on a uniform mesh, we obtain
whith e = (u - u_ex)
@example
# h    |e|_(2;omega) |e|_(2;gamma) |e|_(2;1;omega) |e|_(inf;omega) |e|_(infty;gamma)
0.5      0.609297        0.991235        0.6709          0.690474        0.652854
0.25     0.18116         0.282936        0.2677          0.255665        0.17842
0.125    0.0245253       0.0348991       0.0765863       0.0353861       0.0285902
0.0625   0.00458893      0.00559696      0.0317949       0.0072988       0.00528781
0.03125  0.00101987      0.00120666      0.0145257       0.00166761      0.00149711
0.015625 0.000263283     0.000306343     0.0079968       0.000429066     0.000420724
@end example
and thus we observe a O(h^2) convergence for
both L^2 and L^infty norms, and a O(h) convergence for the H^1 norm.

@noindent
With the @code{P1} approximation on a uniform mesh, we obtain
@example
# h    |e|_(2;omega) |e|_(2;gamma) |e|_(2;1;omega) |e|_(inf;omega) |e|_(infty;gamma)
0.5      0.00488447      0.0171008       0.066621        0.0236969       0.0162823
0.25     0.00139653      0.00342286      0.0274055       0.00452714      0.00452714
0.125    0.000128517     0.000384319     0.00529753      0.000859435     0.000859435
0.0625   1.25517e-05     4.27244e-05     0.00122707      0.00011549      0.00011549
0.03125  1.23824e-06     4.2217e-06      0.000255257     1.48647e-05     1.48647e-05
0.015625 5.99753e-07     1.18159e-06     6.95599e-05     2.36498e-06     2.36498e-06
@end example
and thus we observe a O(h^3) convergence for
both L^2 and L^infty norms, and a O(h^2) convergence for the H^1 norm.

END: */

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
//  3. Cosinus square case
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

Float f_cos_2d (const point& x)
{
  return 2*pi*pi*sin(pi*(x[0]+x[1])); 
}
Float u_cos_2d (const point& x)
{
  return sin(pi*(x[0]+x[1])); 
}
Float g_cos_2d (const point& x)
{
  // warning: fit g(x,y) on the unit square only !
  if (fabs(x[0]*(1-x[0])) < sqrt(epsilon)) {
     // on x = 0 or 1 boundaries
     // warning_macro ("g_cos_2d("<<x<<"): case 1");
     return alpha*u_cos_2d(x) - pi*cos(pi*x[1]);
  } else if (fabs(x[1]*(1-x[1])) < sqrt(epsilon)) {
     // warning_macro ("g_cos_2d("<<x<<"): case 2");
     return alpha*u_cos_2d(x) - pi*cos(pi*x[0]);
  } else {
     // warning_macro ("g_cos_2d("<<x<<"): case 3");
     return alpha*u_cos_2d(x);
  }
}

/*Prog:robin
COSINUS TEST CASE IN A SQUARE:
@c ifnottex is not recognized yet by texi2html
@ifinfo
When Omega=]0,1[x][0,1[, we choose
@example
  u(x,y) = sin(pi (x+y))
@end example
@end ifinfo

@iftex
@tex
When $\Omega=]0,1[^2$, we choose
  $$
  u(x,y) = \sin(\pi(x+y))
  $$
@end tex
@end iftex

@noindent
With alpha=1 and the @code{P1} approximation on a uniform mesh of size h, we obtain
whith e = (u - u_ex):
@example
# h    |e|_(2;omega) |e|_(2;gamma) |e|_(2;1;omega) |e|_(inf;omega) |e|_(infty;gamma)
0.5      0.325815     0.988468        1.42289         0.963001        0.963001
0.25     0.178941     0.60088         0.865343        0.714539        0.714539
0.125    0.0914071    0.329367        0.492801        0.493485        0.493485
0.0625   0.0461357    0.177045        0.288302        0.31872         0.31872
0.03125  0.0231568    0.0924567       0.165937        0.198851        0.198851
0.015625 0.0115533    0.0474078       0.0935944       0.120042        0.120042
@end example
and we observe rougthly a O(h) convergence for
both L^2, H^1 and L^infty norms. Note that the convergence is more
difficult than for the polynomial solution.

When switching to the @code{P2} approximation, we obtain
@example
# h    |e|_(2;omega) |e|_(2;gamma) |e|_(2;1;omega) |e|_(inf;omega) |e|_(infty;gamma)
0.5      0.132522       0.483261      0.855237        0.636449        0.636449
0.25     0.0632219      0.24686       0.41132         0.420735        0.420735
0.125    0.031043       0.126734      0.228482        0.264078        0.264078
0.0625   0.015418       0.0643841     0.128311        0.159761        0.159761
0.03125  0.00768954     0.0324566     0.0708668       0.0940824       0.0940824
0.015625 0.00384121     0.0162958     0.0385581       0.0541965       0.0541965
@end example
and we observe also rougthly a O(h) convergence for
both L^2, H^1 and L^infty norms. Using the P2 element
leads to the same precision. This effect is due 
to a loss of regularity of the right-hand
side g on the boundary.
END: */

/*
@toindex @code{gnuplot}
 The previous figure was producted with @code{gnuplot}
 by using the following commands:
@example
 set logscale
 set terminal dumb 60 30
 set xlabel "h"
 set ylabel "Err"
 plot \\
	"p1.dat" title "P1" w linespoints, \\
	"p2.dat" title "P2" w linespoints
@end example
*/

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
//  4. Dirichlet circular case
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

Float u_circle_Dir_2d (const point& x)
{
  Float rr = x[0]*x[0] + x[1]*x[1] ;
  return (1.-rr)/4. ;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 
//  5. Cosinus circular case
// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - 

Float u_circle_cos_2d (const point& x)
{
  Float rr = x[0]*x[0] + x[1]*x[1] ; 
  return sin(omega*rr) ;
}
Float f_circle_cos_2d (const point& x)
{
  Float rr = x[0]*x[0] + x[1]*x[1] ; 
  return 4.*omega*(omega*rr*sin(omega*rr)-cos(omega*rr)) ;
}
Float g_circle_cos_2d (const point& x)
{
  return Float(2.)*omega*cos(omega) ;
}

// ============================================================================
//  Usage
// ============================================================================

void usage()
{
  cerr << "robin: usage: robin"
       << " [-app {P1 | P2}]"
       << " [-alpha float]"
       << " [-cos | -Dir | -reg]"
       << " [-exact|-diff]"
       << " [-round]"
       << " {-Igeodir}*"
       << " -|mesh[.geo]"
       << " {domain}+"
       << endl;
  exit (1);
}
// ============================================================================
//  Utility
// ============================================================================

// round error, for non-regression tests,
// since P2 error is eps mach
inline Float my_round (const Float& x) { return (fabs(x) < Float(1e-10)) ? Float(0) : x; }

// get trace(u) on W(gamma)
field
get_trace (
    const field& u,	// in V(omega)
    const space& W)	// W(gamma)
{
    space V = u.get_space();
    form  m_gamma_h (W, W, "mass") ;
    form  m_trace_h (V, W, "mass") ;
    //
    // deduce field on W by solving
    //   m_gamma_h*weight_bdr = m_trace_h*weight
    //
    field u_bdr(W);
    ssk<Float> fact = ldlt(m_gamma_h.uu);
    u_bdr.b = 0;
    u_bdr.u = fact.solve(m_trace_h.uu*u.u + m_trace_h.ub*u.b - m_gamma_h.ub*u_bdr.b);
    return u_bdr;
}
// ============================================================================
//  Main
// ============================================================================

int main(int argc, char**argv)
{

  // --------------------------------------------------------------------------
  //  Get parameters
  // --------------------------------------------------------------------------
  
  bool verbose = true ;
  const char* approx = "P1";

  bool do_round = false ;

  bool do_cos = false ;
  bool do_Dir = false ;
  bool do_reg = false ;


  bool put_exact = false ;
  bool put_diff  = false ;

  geo Omega_h ;

  bool         mesh_done = false ;
  geo::size_type n_dom     = 0     ;
  vector<geo::size_type> io_dom(10) ;

  if (argc <= 1) usage() ;

  for (int io=1 ; io < argc ; io++ ) {

      if (argv [io][0] == '-' && argv [io][1] == 'I')
	append_dir_to_rheo_path (argv[io]+2) ;
      else if (strcmp(argv[io], "-app") == 0)   approx = argv[++io]       ;
      else if (strcmp(argv[io], "-alpha")==0)   alpha  = atof(argv[++io]) ;
      else if (strcmp(argv[io], "-cos")==0)     do_cos = true ;
      else if (strcmp(argv[io], "-Dir")==0)     do_Dir = true ;
      else if (strcmp(argv[io], "-reg")==0)     do_reg = true ;
      else if (strcmp(argv[io], "-exact")==0)   put_exact = true ;
      else if (strcmp(argv[io], "-diff")==0)    put_diff = true ;
      else if (strcmp(argv[io], "-round")==0)   do_round = true ;
      else if (strcmp(argv[io], "-") == 0) {

	  // input geo on standard input
	  if (mesh_done) usage() ;
	  cerr << " ! robin: read geo on stdin" << endl ;
	  cin >> Omega_h ;
	  mesh_done = true ;

      } else if (!mesh_done) {

	  // input geo on file
	  Omega_h = geo(argv[io]);
	  mesh_done = true ;
 
      } else {

	  // add a domain
	  cerr << "! robin condition on domain `" << argv[io] << "'" << endl;
	  io_dom[n_dom]=io ;
	  n_dom++ ;
      }
  }
  if (!mesh_done) {
      warning_macro("no mesh specified.");
      usage();
  }
  if (alpha <= Float(0)) {
      error_macro("alpha may be strictly positive: " << alpha);
  }
  if (Omega_h.dimension() != 2) {
      error_macro(Omega_h.dimension() << "D not ready yet -- sorry");
  }
  if (n_dom==0) {
      error_macro("no robin domain; hill-posed problem.");
  }
  domain Gamma_h(Omega_h[argv[io_dom[0]]]) ;
  for (geo::size_type i_dom=1 ; i_dom < n_dom ; i_dom++) {
      Gamma_h.cat(Omega_h[argv[io_dom[i_dom]]]) ;
  }
  // --------------------------------------------------------------------------
  //  Build spaces & forms
  // --------------------------------------------------------------------------
  
  space V_h(Omega_h, approx)          ; V_h.freeze() ;
  space S_h(Omega_h, Gamma_h, approx) ; S_h.freeze() ;

  form M_h       (V_h, V_h, "mass") ;
  form A_Omega_h (V_h, V_h, "grad_grad") ;   
  form M_Gamma_h (V_h, V_h, "mass", Gamma_h) ;

  form M_trace_h (V_h, S_h, "mass") ;

  form A_h = A_Omega_h + alpha*M_Gamma_h ;

  // --------------------------------------------------------------------------
  //  output forms -- COMMENTED BECAUSE OF 64bits PB
  // --------------------------------------------------------------------------
  // ofstream A_Omega_str ("A_Omega_h.hb");
  // A_Omega_str << A_Omega_h;

  // ofstream M_Gamma_str ("M_Gamma_h.hb");
  // M_Gamma_str << M_Gamma_h;

  // --------------------------------------------------------------------------
  //  Build exact solution and right-hand-side
  // --------------------------------------------------------------------------

  field u_ex(V_h) ;
  field f__h(V_h) ;
  field k__h(V_h) ;
  field u__h(V_h) ;

  field g__h(S_h) ;

  if (do_cos) {

	      warning_macro ("cos on a unit square");
	      u_ex = interpolate(V_h, u_cos_2d);
	      f__h = interpolate(V_h, f_cos_2d);
	      g__h = interpolate(S_h, g_cos_2d);

  } else if (do_Dir) {

	      warning_macro ("Dirichlet on a unit square");
	      u_ex = interpolate(V_h, u_Dir_2d);
	      f__h = 1.;
	      g__h = interpolate(S_h, g_Dir_2d);

  } else if (do_reg) {

	      warning_macro ("Regular on a unit square");
	      u_ex = interpolate(V_h, u_reg_2d);
	      f__h = interpolate(V_h, f_reg_2d);
	      g__h = interpolate(S_h, g_reg_2d);

  } else {
	      warning_macro ("exact solution unknown");
	      f__h = 1. ;
	      g__h = 0. ;
	      u_ex = 0. ; // unknown.
  }
  // --------------------------------------------------------------------------
  //  Solve: build the approximate solution
  // --------------------------------------------------------------------------

  k__h = M_h*f__h + M_trace_h.trans_mult(g__h);
  ssk<Float> MA = ldlt(A_h.uu) ;
  MA.solve(k__h.u, u__h.u) ;

  // --------------------------------------------------------------------------
  //  Output: solution and scalar products
  // --------------------------------------------------------------------------
  
  // force conversion to double for hight prec classes
  // for non-regression test purpose
  numeric_flags<Float>::output_as_double(true);

  if (!put_exact && !put_diff) {
      if (do_round) {
          cout << rheo << transform(u__h,my_round) << endl << endl;
      } else {
          cout << rheo << u__h << endl << endl;
      }
  }
  Float ps1 = dot( M_h*u__h, u__h) ;
  Float ps2 = dot( M_Gamma_h*u__h, u__h) ;

  cerr << "dot(M_h*u__h,u__h)        = " <<  ps1 << endl
       << "dot(M_Gamma_h*u__h, u__h) = " <<  ps2 << endl;

  // --------------------------------------------------------------------------
  //  Output: boundary-mesh
  // --------------------------------------------------------------------------

  string geo_file_name = S_h.get_geo().name() + ".geo";
  ofstream bdr_geo(geo_file_name.c_str()) ;
  bdr_geo << rheo << S_h.get_geo() ;
  cerr << "! robin: file `" << geo_file_name << "' created.\n" ;
  
  // --------------------------------------------------------------------------
  //  Output: errors L2
  // --------------------------------------------------------------------------

  if (!do_reg && !do_Dir && !do_cos) {
      // exact solution is unknown
      return 0;
  }
  field u_diff(V_h) ;
  u_diff = u_ex - u__h ;
  if (put_exact) {
      cout << rheo << u_ex << endl << endl;
  } else if (put_diff) {
      cout << rheo << u_diff << endl << endl;
  }
  string diff_file_name = Omega_h.name() + "-diff.field";
  ofstream diff(diff_file_name.c_str()) ;
  cerr << "! robin: file `" << diff_file_name << "' created.\n" ;
  diff << rheo << u_diff;

  Float err1 = sqrt(dot((M_h*u_diff).u, u_diff.u));
  Float err2 = sqrt(dot((M_Gamma_h*u_diff).u, u_diff.u));
  Float err3 = sqrt(dot((A_Omega_h*u_diff).u, u_diff.u));

  field gamma_u_diff = get_trace(u_diff, S_h);

  Float err_max1 = u_diff.u.max_abs();
  Float err_max2 = gamma_u_diff.u.max_abs();

  cerr << "# " << Omega_h.name() << endl;

  cerr << "# |u-u_ex|_{2;\\Omega} |u-u_ex|_{2;\\Gamma}"
       << " |u-u_ex|_{2;1;\\Omega}"
       << " |u-u_ex|_{infty;\\Omega} |u-u_ex|_{infty;\\Gamma}" << endl
       <<  err1 << "\t" << err2 << "\t" << err3 << "\t" 
       <<  err_max1 << "\t" << err_max2 << endl;
  return 0;
}
