# ifndef _RHEOLEF_CG_H
# define _RHEOLEF_CG_H
///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

#include "rheolef/solver_option.h"

namespace rheolef {

/*D:cg
NAME: @code{cg} -- conjugate gradient algorithm.
@findex cg
@cindex conjugate gradient algorithm
@cindex iterative solver
@cindex preconditioner
SYNOPSIS:
@example
  template <class Matrix, class Vector, class Preconditioner, class Real>
  int cg (const Matrix &A, Vector &x, const Vector &b,
    const solver_option& sopt = solver_option())
@end example

EXAMPLE:
  @noindent
  The simplest call to @code{cg} has the folling form:
  @example
    solver_option sopt;
    sopt.max_iter = 100;
    sopt.tol = 1e-7;
    int status = cg(a, x, b, eye(), sopt);
  @end example

DESCRIPTION:       
  @noindent
  @code{cg} solves the symmetric positive definite linear
  system Ax=b using the conjugate gradient method.
  The return value indicates convergence within max_iter (input)
  iterations (0), or no convergence within max_iter iterations (1).
  Upon successful return, output arguments have the following values:
  @table @code
    @item x
	approximate solution to Ax = b

    @item sopt.n_iter
	the number of iterations performed before the tolerance was reached

    @item sopt.residue
	the residual after the final iteration
  @end table
  See also the @ref{solver_option class}.

NOTE: 

  @noindent
  @code{cg} is an iterative template routine.

  @noindent
  @code{cg} follows the algorithm described on p. 15 in

  @emph{Templates for the solution of linear systems: building blocks for iterative methods}, 
	2nd Edition, 
        R. Barrett, M. Berry, T. F. Chan, J. Demmel, J. Donato, J. Dongarra, V. Eijkhout,
	R. Pozo, C. Romine, H. Van der Vorst,
        SIAM, 1994, 
	@url{ftp.netlib.org/templates/templates.ps}.

  @noindent
  The present implementation is inspired from 
  @code{IML++ 1.2} iterative method library,
  @url{http://math.nist.gov/iml++}.
AUTHOR: 
    Pierre Saramito
    | Pierre.Saramito@imag.fr
    LJK-IMAG, 38041 Grenoble cedex 9, France
DATE: 
    20 april 2009
METHODS: @cg
End:
*/

//<cg:
template <class Matrix, class Vector, class Vector2, class Preconditioner>
int cg (const Matrix &A, Vector &x, const Vector2 &Mb, const Preconditioner &M,
	const solver_option& sopt = solver_option())
{
  typedef typename Vector::size_type  Size;
  typedef typename Vector::float_type Real;
  std::string label = (sopt.label != "" ? sopt.label : "cg");
  Vector b = M.solve(Mb);
  Real norm2_b = dot(Mb,b);
  if (sopt.absolute_stopping || norm2_b == Real(0)) norm2_b = 1;
  Vector Mr = Mb - A*x;
  Real  norm2_r = 0;
  if (sopt.p_err) (*sopt.p_err) << "[" << label << "] #iteration residue" << std::endl;
  Vector p;
  for (sopt.n_iter = 0; sopt.n_iter <= sopt.max_iter; sopt.n_iter++) {
    Vector r = M.solve(Mr);
    Real prev_norm2_r = norm2_r;
    norm2_r = dot(Mr, r);
    sopt.residue = sqrt(norm2_r/norm2_b);
    if (sopt.p_err) (*sopt.p_err) << "[" << label << "] " << sopt.n_iter << " " << sopt.residue << std::endl;
    if (sopt.residue <= sopt.tol) return 0;     
    if (sopt.n_iter == 0) {
      p = r;
    } else {
      Real beta = norm2_r/prev_norm2_r;
      p = r + beta*p;
    }
    Vector Mq = A*p;
    Real alpha = norm2_r/dot(Mq, p);
    x  += alpha*p;
    Mr -= alpha*Mq;
  }
  return 1;
}
//>cg:
}// namespace rheolef
# endif // _RHEOLEF_CG_H
