/*
  funcalls.hh, copyright (c) 2006 by Vincent Fourmond: 
  The gory details of the function implementation...
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details (in the COPYING file).
  
*/

namespace SCalc {
  /// The class to deal with an arglist. It is derived from Expression * 
  /// because it is much simpler this way. It should never be called
  /// directly.
  class Arglist : public Expression {
    std::vector<Expression *> arguments;
  public:

    virtual int is_valid() {return 0;};
    virtual double evaluate(const double *, const double * s = NULL)
    { return 0;};

    Arglist(Session * s, Expression * start);
    void append(Expression * e) 
    { arguments.push_back(e);};
    
    int nb_args() { return arguments.size();};
    std::vector<Expression *> args() { return arguments;};
    
    virtual std::string pretty_print();
  };

  /// The class representing a call to a function.
  class Funcall : public Expression {
    std::vector<Expression *> args;
    FuncDef * function;
    int nb_args;

    /// Returns a copy of the arguments.
    std::vector<Expression *> copy_args() const;

    /// returns the expression of the derivation with regards
    /// to argument number n
    Expression * derive_arg_n(int id, int n);
  public:

    Funcall(Session * s, Arglist * argl, FuncDef * f) : Expression(s)
    { args = argl->args(); function = f; nb_args = args.size();};
    Funcall(Session * s, std::vector<Expression *> argl,
	    FuncDef * f) : Expression(s), args(argl)
    { function = f;nb_args = args.size();};

    virtual ~Funcall();
      
    virtual int is_valid() {return 1;};

    virtual std::string pretty_print();
    virtual double evaluate(const double *, const double *);

    virtual Expression * copy() 
    { return new Funcall(session(), copy_args(), function);};
    virtual Expression * simplify();

    virtual std::set<int> used_variables();

    /// derivation with reg
    virtual Expression * derive(int id);

  };

  /// A node refers to one argument of a function from within
  /// this function.
  class Node : public Expression {
    int id;
  public:
    Node(Session * s, int id) : Expression(s) { this->id = id;};
    virtual ~Node() {;};

    virtual std::string pretty_print();
    virtual double evaluate(const double *, const double * args)
    { return args[id]; };

    virtual Expression * copy() { return new Node(session(), id); };

    /// The derivation: the derivation with regards to the argument
    /// number 0 has to hold number -1, 1 -> -2 and so on.
    virtual Expression * derive(int i) {
      if(i == -(id + 1))
	return new Const(session(), 1);
      else
	return new Null(session());
    };
    
  };
};
