/*
 * Copyright (C) 2019 ~ 2020 Uniontech Software Technology Co., Ltd.
 *
 * Author:     yelei <yelei@uniontech.com>
 * Maintainer: yelei <yelei@uniontech.com>
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
#ifndef PHONEFILETHREAD_H
#define PHONEFILETHREAD_H

#include <QThread>
#include <QPixmap>
#include <QMap>
#include <QFileInfo>
#include <QMutex>

#include "defines.h"

#define PFERROR_NOTFIND_DCIM 1001
#define PFERROR_DELETE_FILE 1002
#define PEERROR_NO_FILE 1003
#define PEERROR_END 2001 //标识位

class PhoneFileThread : public QThread
{
    Q_OBJECT
public:
    PhoneFileThread(QObject *parent = nullptr);
    virtual ~PhoneFileThread();

    enum RunMode {
        DelFile = 0, //删除文件
        ExportFile, //导出文件
        ImportFile, //导入文件
        ReadPhoto, //获取照片
        ReadVideo //获取视频
    };

    bool copyFile(const QStringList &listFileNames, const QString &strDesPath, const RunMode &mode,
                  const PhoneFileType &type, const QString &strSysVersion, const QString &strRootPath,
                  const QString strPhoneID); //导入/导出
    bool delFile(QStringList &listFileNames, PhoneFileType type);
    bool readPhoto(const QString &strPath, const QString &strPhoneID, const QSize &sMaxSize, DEVICE_TYPE type,
                   bool bOnlyDir = false, bool bKeepAspectRatio = false);
    bool readVideo(const QString &strPath, const QString &strPhoneID, const QSize &sMaxSize, DEVICE_TYPE type,
                   bool bOnlyDir = false, bool bKeepAspectRatio = false);
    // 停止线程
    void quitImmediately();
    void stopImmediately();
    //清空数据
    void clear();
    static void clearCache(const QString &strPhoneID);
    static void clearCache(const RunMode &mode, const QString &strPhoneID);
    //线程锁住
    bool lock();
    bool unlock();

public:
    static QStringList getImageFilters();
    static QStringList getVideoFilters();
    static bool _readVideo_ffmpeg_one(const char *input, QIcon &pixmap, int &nTime);
    static bool refreshDirInfo(const RunMode &runMode, const QString &strDesPath,
                               const int &nFileCount, const quint64 &nFileSize); //刷新缓存信息
    QStringList getListFileNames() const;

private:
    static QVector<PhoneFileInfo> m_mapDirFileInfo_photo; //照片文件夹信息缓存
    static QVector<PhoneFileInfo> m_mapDirFileInfo_video; //视频文件夹信息缓存

signals:
    void sigProgress(int nRunMode, int nCur, int nTotal, QString strFileName); // 操作进度(拷贝)
    void sigFindPicture(PhoneFileInfo &info); // 图标查找
    void sigResult(int nRunMode, int nCount, int nSuccess, int nFailed, bool bIsStop); // 结果
    void error(int nErrorCode, QString strErrorMsg);

protected:
    void run();

private slots:
    PhoneFileInfo onVideoFindPixmap(QIcon &icon, QString path, int nTime);

private:
    bool getPhotoPath(QString strPhoneID, QStringList &listPath);
    QStringList getAndroidPhotoPath(QString strBasePath);

    bool _copyFile();
    bool _copyFile_one_auto(const QFileInfo &source, const QString &desPath);
    bool _copyFile_one_qfile(const QString &source, const QString &destination);
    bool _copyFile_one_iof(const QString &source, const QString &destination);
    bool _copyFile_one_f(const QString &source, const QString &destination);
    bool _copyFile_one_adb(const QString &source, const QString &destination);
    bool _delFile();
    static bool _delFile_one(const QString &strFile);
    bool _readPhoto();
    bool _readPhoto_one(const QString &path);
    bool _readVideo_ffmpeg();
    //    bool _readVideo_ffmpegthumbnailer();

    //    PhoneFileType getFileType(QString strFilePath);
    void setFile(QString strDesPath); // 根据路径自动筛选文件
    bool setDir(const QString &strDesPath, QString &strDesItemPath); //根据目录获取其中一张图片
    bool autoFilterSet(); // 自动判断应该查找的文件
    QPixmap resizePicture(const QPixmap &icon);
    QStringList getFileFilters();

    bool processCache();
    //    void videoFormatSupport(); //查看支持格式

private:
    static QMutex m_mutex_photo;
    static QMutex m_mutex_video;
    QMutex m_mutex_read;
    bool m_bLocked;

    QString m_strSrcFileName; //源路径
    QString m_strDesFileName; //指定路径
    QStringList m_listFileNames; //操作多个文件
    QString m_strDesPath; //多个文件指向目录
    QString m_strRootPath; // 安卓用 rootpath
    QString m_strPhoneID; //查询某个手机里的数据
    QString m_strSysVersion; //手机系统版本
    DEVICE_TYPE m_phoneType; //查询手机的类型
    RunMode m_mode;
    QSize m_sMaxSize; //图片最大尺寸
    bool m_bKeepAspectRatio; //是否保持原图比例 共用此参数会导致混乱
    bool m_bIsDir; //文件夹
    QMap<QString, QString> m_mapFileToDir; //文件对应的目录
    QMap<QString, int> m_mapDirToFileCount;
    QMap<QString, quint64> m_mapDirToFileSize;
    QStringList m_listFilters;
    bool m_isCanRun; //标识位
};

#endif // PHONEFILETHREAD_H
