% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cnd.R
\name{cnd_signal}
\alias{cnd_signal}
\title{Signal a condition}
\usage{
cnd_signal(cnd, .cnd, .mufflable)
}
\arguments{
\item{cnd}{A condition object (see \code{\link[=cnd]{cnd()}}).}

\item{.cnd, .mufflable}{These arguments are retired. \code{.cnd} has been
renamed to \code{cnd} and \code{.mufflable} no longer has any effect as
non-critical conditions are always signalled with a muffling
restart.}
}
\description{
Signal a condition to handlers that have been established on the
stack. Conditions signalled with \code{cnd_signal()} are assumed to be
benign. Control flow can resume normally once the condition has
been signalled (if no handler jumped somewhere else on the
evaluation stack). On the other hand, \code{cnd_abort()} treats the
condition as critical and will jump out of the distressed call
frame (see \code{\link[=rst_abort]{rst_abort()}}), unless a handler can deal with the
condition.
}
\details{
If \code{.critical} is \code{FALSE}, this function has no side effects beyond
calling handlers. In particular, execution will continue normally
after signalling the condition (unless a handler jumped somewhere
else via \code{\link[=rst_jump]{rst_jump()}} or by being \code{\link[=exiting]{exiting()}}). If \code{.critical} is
\code{TRUE}, the condition is signalled via \code{\link[base:stop]{base::stop()}} and the
program will terminate if no handler dealt with the condition by
jumping out of the distressed call frame.

\code{\link[=calling]{calling()}} handlers are called in turn when they decline to handle
the condition by returning normally. However, it is sometimes
useful for a calling handler to produce a side effect (signalling
another condition, displaying a message, logging something, etc),
prevent the condition from being passed to other handlers, and
resume execution from the place where the condition was
signalled. The easiest way to accomplish this is by jumping to a
restart point (see \code{\link[=with_restarts]{with_restarts()}}) established by the signalling
function. \code{cnd_signal()} always installs a muffle restart (see
\code{\link[=cnd_muffle]{cnd_muffle()}}).
}
\section{Lifecycle}{

\itemize{
\item Modifying a condition object with \code{cnd_signal()} is defunct.
Consequently the \code{.msg} and \code{.call} arguments are retired and
defunct as of rlang 0.3.0.  In addition \code{.cnd} is renamed to
\code{cnd} and soft-deprecated.
\item The \code{.mufflable} argument is soft-deprecated and no longer has
any effect. Non-critical conditions are always signalled with a
muffle restart.
\item Creating a condition object with \code{\link[=cnd_signal]{cnd_signal()}} is
soft-deprecated. Please use \code{\link[=signal]{signal()}} instead.
}
}

\examples{
# Creating a condition of type "foo"
cnd <- cnd("foo")

# If no handler capable of dealing with "foo" is established on the
# stack, signalling the condition has no effect:
cnd_signal(cnd)

# To learn more about establishing condition handlers, see
# documentation for with_handlers(), exiting() and calling():
with_handlers(cnd_signal(cnd),
  foo = calling(function(c) cat("side effect!\\n"))
)


# By default, cnd_signal() creates a muffling restart which allows
# calling handlers to prevent a condition from being passed on to
# other handlers and to resume execution:
undesirable_handler <- calling(function(c) cat("please don't call me\\n"))
muffling_handler <- calling(function(c) {
  cat("muffling foo...\\n")
  cnd_muffle(c)
})

with_handlers(foo = undesirable_handler,
  with_handlers(foo = muffling_handler, {
    cnd_signal(cnd("foo"))
    "return value"
  }))
}
\seealso{
\code{\link[=abort]{abort()}}, \code{\link[=warn]{warn()}} and \code{\link[=inform]{inform()}} for signalling typical
R conditions. See \code{\link[=with_handlers]{with_handlers()}} for establishing condition
handlers.
}
