/*
  funcalls.cc, copyright (c) 2006 by Vincent Fourmond: 
  The other side of gory details of the function implementation...
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details (in the COPYING file).
  
*/

#include <calc_internals.hh>
#include <sstream>


namespace SCalc {
  Arglist::Arglist(Session * s, Expression * start) 
    : Expression(s)
  {
    append(start);
  }

  std::string Arglist::pretty_print()
  {
    return std::string("Biniou");
  }

  std::string Funcall::pretty_print()
  {
    std::string arglist = args[0]->pretty_print();
    std::string retval;
    for(int i = 1; i < nb_args; i++)
      {
	arglist += ", ";
	arglist += args[i]->pretty_print();
      }
    if(function->name().empty())
      {
	retval = "[";
	retval += function->pretty_print() + "]";
      }
    else
      retval = function->name();
    retval += "(";
    retval += arglist;
    retval += ")";
    return retval;
  }
  
  double Funcall::evaluate(const double * vars, const double * a)
  {
    // it looks like it works fine...
    double params[nb_args];
    for(int i = 0; i < nb_args; i++)
      params[i] = args[i]->evaluate(vars, a);
    // This is the thing: the params are evaluated with args
    // but the function with params !
    return function->evaluate(vars, params);
  }

  Expression * Funcall::simplify()
  {
    std::vector<Expression *> newargs;
    for(int i =0; i < nb_args; i++)
      newargs.push_back(args[i]->simplify());
    return new Funcall(session(), newargs, function);
  }

  std::set<int> Funcall::used_variables()
  {
    std::set<int> l,r;
    l = args[0]->used_variables();
    for(int j = 1; j< nb_args; j++) 
      {
	r = args[j]->used_variables();
	for(std::set<int>::iterator i = r.begin(); i != r.end(); i++)
	  l.insert(*i);
      }
    return l;
  }


  Expression * Funcall::derive(int id)
  {
    Expression * cur = derive_arg_n(id, 0);
    // we keep on adding the derivatives until we've finished
    for(int i = 1; i<nb_args; i++)
      cur = new Plus(session(), cur, derive_arg_n(id, i));
    return cur;
  }
  
  std::vector<Expression *> Funcall::copy_args() const 
  {
    std::vector<Expression *> retval;
    for(std::vector<Expression *>::const_iterator i = args.begin();
	i != args.end(); i++)
      retval.push_back((*i)->copy());
    return retval;
  }

  Expression * Funcall::derive_arg_n(int id, int n)
  {
    // here, we build the expression u' * f'(u)
    FuncDef * derivative = function->derivative(n);
    if(derivative) {
      Funcall * f = new Funcall(session(), copy_args(), derivative);
      Expression * arg_d = args[n]->derive(id);
      return new Times(session(), arg_d, f);
    }
    else
      /// This really, really, really, should throw an appropriate
      /// exception rather than stupidly returning a wrong
      return new Null(session()); // returns 0 on failure.
  }

  std::string Node::pretty_print()
  {
    std::ostringstream str;
    str << "arg[";
    str << id;
    str << "]";
    return str.str();
  }
  
  Funcall::~Funcall()
  {
    for(int i = 0; i < args.size(); i++)
      delete args[i];
    args.clear();
  }

};
