/*
  Copyright (C) 2009, 2010 and 2012 Chris Vine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include <string>
#include <fstream>
#include <ios>
#include <ostream>

#include "preferences.h"

#include <c++-gtk-utils/shared_handle.h>

#ifdef ENABLE_NLS
#include <libintl.h>
#endif


void PreferencesCB::preferences_selected(GtkWidget* widget, void* data) {
  Preferences* instance = static_cast<Preferences*>(data);
  
  if (widget == instance->ok_button) {
    instance->save_prefs();
    instance->close();
  }
  else if (widget == instance->cancel_button) {
    instance->close();
  }
}

Preferences::Preferences(GtkWindow* parent): WinBase(gettext("Mount-gtk: preferences"),
						     prog_config.window_icon,
						     true, parent) {

  GtkWidget* window_box = gtk_vbox_new(false, 2);
  gtk_container_add(GTK_CONTAINER(get_win()), window_box);

  GtkWidget* alignment = gtk_alignment_new(0.5, 0.5, 0, 0);
  gtk_box_pack_start(GTK_BOX(window_box), alignment,
		     true, true, 12);

  GtkWidget* vbox = gtk_vbox_new(false, 12);
  gtk_container_add(GTK_CONTAINER(alignment), vbox);

  alignment = gtk_alignment_new(0, 0.5, 0, 0);
  gtk_box_pack_start(GTK_BOX(vbox), alignment,
		     false, false, 0);
  beep_button = gtk_check_button_new_with_label(gettext("Beep on error notification"));
  gtk_container_add(GTK_CONTAINER(alignment), beep_button);

  alignment = gtk_alignment_new(0, 0.5, 0, 0);
  gtk_box_pack_start(GTK_BOX(vbox), alignment,
		     false, false, 0);
  GtkWidget* hbox = gtk_hbox_new(false, 10);
  gtk_container_add(GTK_CONTAINER(alignment), hbox);
  GtkWidget* label = gtk_label_new(gettext("Visual error report:"));
  gtk_box_pack_start(GTK_BOX(hbox), label,
		     false, false, 0);
  GtkWidget* frame = gtk_frame_new(0);
  gtk_box_pack_start(GTK_BOX(hbox), frame,
		     false, false, 0);
  hbox = gtk_hbox_new(false, 10);
  gtk_container_add(GTK_CONTAINER(frame), hbox);
  no_notify_button = gtk_radio_button_new_with_label(0, gettext("None"));
  gtk_box_pack_start(GTK_BOX(hbox), no_notify_button,
		     false, false, 0);
  dialog_notify_button = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(no_notify_button),
								     gettext("Dialog"));
  gtk_box_pack_start(GTK_BOX(hbox), dialog_notify_button,
		     false, false, 0);
#ifdef HAVE_LIBNOTIFY
  lib_notify_button = gtk_radio_button_new_with_label_from_widget(GTK_RADIO_BUTTON(no_notify_button),
								  gettext("Libnotify"));
  gtk_box_pack_start(GTK_BOX(hbox), lib_notify_button,
		     false, false, 0);
#else
  lib_notify_button = 0;
#endif  

  alignment = gtk_alignment_new(0, 0.5, 0, 0);
  gtk_box_pack_start(GTK_BOX(vbox), alignment,
		     false, false, 0);
  tray_icon_button = gtk_check_button_new_with_label(gettext("Put program in system tray"));
  gtk_container_add(GTK_CONTAINER(alignment), tray_icon_button);

  alignment = gtk_alignment_new(0, 0.5, 0, 0);
  gtk_box_pack_start(GTK_BOX(vbox), alignment,
		     false, false, 0);
  tooltip_button = gtk_check_button_new_with_label(gettext("Show mount point as tooltip"));
  gtk_container_add(GTK_CONTAINER(alignment), tooltip_button);


  GtkWidget* button_box = gtk_hbutton_box_new();
  gtk_box_pack_start(GTK_BOX(window_box), button_box,
		     false, false, 0);
  gtk_button_box_set_layout(GTK_BUTTON_BOX(button_box), GTK_BUTTONBOX_END);
  gtk_box_set_spacing(GTK_BOX(button_box), 12);
  cancel_button = gtk_button_new_from_stock(GTK_STOCK_CANCEL);
  gtk_container_add(GTK_CONTAINER(button_box), cancel_button);
  ok_button = gtk_button_new_from_stock(GTK_STOCK_OK);
  gtk_container_add(GTK_CONTAINER(button_box), ok_button);

  g_signal_connect(G_OBJECT(ok_button), "clicked",
		   G_CALLBACK(PreferencesCB::preferences_selected), this);
  g_signal_connect(G_OBJECT(cancel_button), "clicked",
		   G_CALLBACK(PreferencesCB::preferences_selected), this);

#if GTK_CHECK_VERSION(2,20,0)
  gtk_widget_set_can_default(ok_button, true);
  gtk_widget_set_can_default(cancel_button, true);
#else
  GTK_WIDGET_SET_FLAGS(ok_button, GTK_CAN_DEFAULT);
  GTK_WIDGET_SET_FLAGS(cancel_button, GTK_CAN_DEFAULT);
#endif

  gtk_window_set_type_hint(get_win(), GDK_WINDOW_TYPE_HINT_DIALOG);
  gtk_container_set_border_width(GTK_CONTAINER(get_win()), 12);
  gtk_widget_grab_focus(ok_button);
  gtk_window_set_position(get_win(), GTK_WIN_POS_CENTER_ON_PARENT);

  get_prefs();

  gtk_widget_show_all(GTK_WIDGET(get_win()));
}

void Preferences::get_prefs() {

  if (prog_config.beep) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(beep_button), true);

  if (prog_config.error_mode == Error::none)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(no_notify_button), true);
  else if (prog_config.error_mode == Error::dialog)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog_notify_button), true);
#ifdef HAVE_LIBNOTIFY
  else {
    if (prog_config.libnotify_fail) {
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog_notify_button), true);
      gtk_widget_set_sensitive(GTK_WIDGET(lib_notify_button), false);
    }
    else gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(lib_notify_button), true);
  }
#endif

  if (prog_config.tray_icon) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tray_icon_button), true);

  if (prog_config.tooltip) gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(tooltip_button), true);
}


void Preferences::save_prefs() const {

  prog_config.beep = get_beep();
  prog_config.error_mode = get_error_mode();
  prog_config.tray_icon = get_tray_icon();
  prog_config.tooltip = get_tooltip();


  g_key_file_set_integer(prog_config.key_file,
			 "Preferences",
			 "Beep",
			 prog_config.beep ? 2: 1);

  g_key_file_set_integer(prog_config.key_file,
			 "Preferences",
			 "ReportMode",
			 prog_config.error_mode);
  
  g_key_file_set_integer(prog_config.key_file,
			 "Preferences",
			 "TrayIcon",
			 prog_config.tray_icon ? 2: 1);

  g_key_file_set_integer(prog_config.key_file,
			 "Preferences",
			 "Tooltip",
			 prog_config.tooltip ? 2: 1);


  std::ofstream fileout{prog_config.rc_filename.c_str(), std::ios::out};

  if (fileout) {
    GcharScopedHandle data{g_key_file_to_data(prog_config.key_file, 0, 0)};
    fileout << (char*)data.get();
  }
    
  else {
    std::string message{gettext("Can't open the following file for saving preferences: ")};
    message += prog_config.rc_filename;
    message += "\n";
    write_error(message.c_str(), true);
  }     
}

bool Preferences::get_beep() const {
  return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(beep_button));
}

Error::ReportMode Preferences::get_error_mode() const {
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(no_notify_button))) return Error::none;
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog_notify_button))) return Error::dialog;
  return Error::libnotify;
}

bool Preferences::get_tray_icon() const {
  return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tray_icon_button));
}

bool Preferences::get_tooltip() const {
  return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(tooltip_button));
}
