/*
  Copyright (C) 2009 to 2013, 2015 and 2020 Chris Vine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
*/

#include "prog_defs.h"

#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>

#include <string>
#include <iostream>
#include <fstream>
#include <exception>
#include <ios>
#include <ostream>
#include <locale>

#include <gdk/gdk.h>   // for gdk_beep()
#include <gtk/gtk.h>

#ifdef ENABLE_NLS
#include <libintl.h>
#endif

#ifdef HAVE_LIBNOTIFY
#include <libnotify/notify.h>
#endif

#include "mainwindow.h"
#include "window_icon.h"
#include "dialogs.h"
#include "utils/icon_info_handle.h"

#include <c++-gtk-utils/gobj_handle.h>
#include <c++-gtk-utils/shared_handle.h>
#include <c++-gtk-utils/gerror_handle.h>
#include <c++-gtk-utils/callback.h>
#include <c++-gtk-utils/prog_present.h>


void get_window_icon();
bool is_arg(const char*, int, char*[]);
void get_preferences();
void display_dbus_error_dialog();
extern "C" void atexit_cleanup();
gboolean present_prog(void*, const char**);

static MainWindow* main_win;

ProgConfig prog_config;

int main(int argc, char* argv[]) {

  // set up the locale for gettext() and base locale domain name of this program
#ifdef ENABLE_NLS
  bindtextdomain("mount-gtk", DATADIR "/locale");
  bind_textdomain_codeset("mount-gtk", "UTF-8");
  textdomain("mount-gtk");
#endif
  setlocale(LC_ALL,"");
  // make sure that the locale is set for C++ objects also
  // (this should also set it for C functions but calling setlocale()
  // explicitly above will also ensure POSIX compliance)
  try {
    std::locale::global(std::locale(""));
  }
  catch (std::exception& except) {
    write_error(except.what(), false);
  }

  if (is_arg("--version", argc, argv)) {
    std::string message("mount-gtk-" VERSION "\n");
    message += gettext("Copyright (C) 2009 - 2020 Chris Vine\n"
		       "This program is released under the "
		       "GNU General Public License, version 2\n");
    std::cout << message;
    return 0;
  }

  if (is_arg("--help", argc, argv)) {
    std::string message("mount-gtk-" VERSION "\n");
    message += gettext("Usage: mount-gtk [options]\n"
		       "Options:\n"
		       "\t-s  Start the program hidden in the system tray\n");
    std::cout << message;
    return 0;
  }

  // although we use the Thread::Thread class based on pthreads,
  // we need to call g_thread_init() in order to make glib thread safe
  // (we do not use the GDK global thread lock, so there is no need to
  // call gdk_threads_init())
  g_thread_init(0);

  gtk_init(&argc, &argv);

  if (register_prog("mount_gtk",
		    present_prog,
		    0)) {

    atexit(atexit_cleanup);
    get_window_icon();
    get_preferences();

    bool start_hidden = false;
    int opt_result;
    while ((opt_result = getopt(argc, argv, "s-:")) != -1) {
      switch(opt_result) {
      case 's':
	start_hidden = true;
	break;
      case '-':
	std::string message(argv[0]);
	message += ": ";
	message += gettext("Invalid option.  Options are:\n");
	message += "  -s\n"
	  "  --help\n"
	  "  --version\n";
	std::cerr << message;
	break;
      }
    }

    MainWindow main_window(start_hidden);
    main_win = &main_window;
    main_window.exec();

    return 0;
  }
  else {
    // for debugging
    //const char* args[3] = {"Instance already exists - ", "invoking via dbus\n", 0};
    // normal use
    const char** args = 0;
    int ret = present_instance(args);
    if (!ret) gdk_notify_startup_complete();
    else display_dbus_error_dialog();
    return ret;
  }
}

void get_window_icon() {

  bool have_icon = false;

  GtkIconTheme* icon_theme = gtk_icon_theme_get_default();

  IconInfoScopedHandle icon_info{gtk_icon_theme_lookup_icon(icon_theme,
							    "drive-removable-media",
							    24, GtkIconLookupFlags(0))};
  if (icon_info.get()) {
      
    const gchar* icon_path = gtk_icon_info_get_filename(icon_info.get());
    if (icon_path) {
      GError* error = 0;
      prog_config.window_icon =
        GobjHandle<GdkPixbuf>{gdk_pixbuf_new_from_file(icon_path, &error)};
      if (prog_config.window_icon.get()) have_icon = true;

      else {
        write_error("Pixbuf error in get_window_icon()\n", false);
        if (error) {
          GerrorScopedHandle handle{error};
          write_error(error->message, false);
          write_error("\n", false);
        }
      }
    }
  }
  if (!have_icon) {
    prog_config.window_icon =
      GobjHandle<GdkPixbuf>{gdk_pixbuf_new_from_xpm_data(window_icon_xpm)};
  }
}

bool is_arg(const char* arg, int argc, char* argv[]) {

  bool return_val;
  int count;
  for (return_val = false, count = 1; !return_val && count < argc; count++) {
    if (!strcmp(argv[count], arg)) return_val = true;
  }
  return return_val;
}

void get_preferences() {

  prog_config.libnotify_fail = false;

  char* home = getenv("HOME");
  if (!home) {
    write_error("Fatal error: HOME environmental variable is not defined!\n", false);
    exit(CONFIG_ERROR);
  }
  prog_config.rc_filename = home;
  prog_config.rc_filename += "/." RC_FILE;
  prog_config.key_file = g_key_file_new();

  // load mount-gtkrc, if it exists
  g_key_file_load_from_file(prog_config.key_file,
			    prog_config.rc_filename.c_str(),
			    G_KEY_FILE_NONE,
			    0);

  int result;
  bool default_applied = false;
  // key file values: 0 = default, 1 = no beep, 2 = beep
  result = g_key_file_get_integer(prog_config.key_file,
				  "Preferences",
				  "Beep",
				  0);
  if (!result) {
    prog_config.beep = true;
    default_applied = true;
  }
  else prog_config.beep = result != 1;


  // key file values: 0 = default, 1 = no tray icon, 2 = tray icon
  result = g_key_file_get_integer(prog_config.key_file,
				  "Preferences",
				  "TrayIcon",
				  0);
  if (!result) {
    prog_config.tray_icon = true;
    default_applied = true;
  }
  else prog_config.tray_icon = result != 1;


  // key file values: 0 = default, 1 = no tooltip, 2 = tooltip
  result = g_key_file_get_integer(prog_config.key_file,
				  "Preferences",
				  "Tooltip",
				  0);
  if (!result) {
    prog_config.tooltip = true;
    default_applied = true;
  }
  else prog_config.tooltip = result != 1;


  // key file values: 0 = default, 1 = none, 2 = dialog, 3 = libnotify
  result = g_key_file_get_integer(prog_config.key_file,
				  "Preferences",
				  "ReportMode",
				  0);
#ifdef HAVE_LIBNOTIFY
  if (!notify_init("mount-gtk")) {
    write_error(gettext("Cannot initialize libnotify - will not use it\n"), false);
    prog_config.libnotify_fail = true;
  }

  if (result <= 0 || result >= 3) {
    if (!prog_config.libnotify_fail) {
      prog_config.error_mode = Error::libnotify;
      if (result != 3) default_applied = true;
    }
    else {
      prog_config.error_mode = Error::dialog;
      default_applied = true;
    }
  }
  else prog_config.error_mode = Error::ReportMode(result);

#else
  if (result <= 0 || result >= 2) {
    prog_config.error_mode = Error::dialog;
    if (result != 2) default_applied = true;
  }
  else prog_config.error_mode = Error::none;
#endif

  // we have used defaults, so save a preferences file
  if (default_applied) { 
    g_key_file_set_integer(prog_config.key_file,
			   "Preferences",
			   "Beep",
			   prog_config.beep ? 2: 1);

    g_key_file_set_integer(prog_config.key_file,
			   "Preferences",
			   "ReportMode",
			   prog_config.error_mode);
  
    g_key_file_set_integer(prog_config.key_file,
			   "Preferences",
			   "TrayIcon",
			   prog_config.tray_icon ? 2: 1);

    g_key_file_set_integer(prog_config.key_file,
			   "Preferences",
			   "Tooltip",
			   prog_config.tooltip ? 2: 1);

    std::ofstream fileout{prog_config.rc_filename.c_str(), std::ios::out};

    if (fileout) {
      GcharScopedHandle data{g_key_file_to_data(prog_config.key_file, 0, 0)};
      fileout << (char*)data.get();
    }
    
    else {
      std::string message{gettext("Can't open the following file for saving preferences: ")};
      message += prog_config.rc_filename;
      message += "\n";
      write_error(message.c_str(), false);
    }     
  }
}

void display_dbus_error_dialog() {
  InfoDialog dialog{gettext("DBUS error: Cannot start mount-gtk.\n\n"
			    "Make sure the DBUS session message bus is running"),
		    gettext("mount-gtk: DBUS error"),
		    GTK_MESSAGE_ERROR,
		    0,
                    false};
  dialog.exec();
}

void atexit_cleanup() {

#ifdef HAVE_LIBNOTIFY
  if (notify_is_initted()) notify_uninit();
#endif
  // keep valgrind quiet
  g_key_file_free(prog_config.key_file);
}

void beep() {
  gdk_beep();
}

gboolean present_prog(void*, const char** args) {
  // we do not need to check main_win for validity to avoid timing
  // races.  If we enter this function via a remote call, the main
  // loop must be running, which means that the main_win pointer
  // must have been assigned to
  main_win->present();
  // just use args for debugging here
  if (args) {
    for (; *args != 0; ++args) {
      std::cout << *args;
    }
  }
  return TRUE;
}

void write_error(const char* message, bool notify) {
  std::cerr << message;
  if (notify) {
    if (prog_config.error_mode == Error::dialog) {
      new InfoDialog{message, gettext("Mount-gtk: error"),
	             GTK_MESSAGE_WARNING, 0, false};
    // there is no memory leak - the InfoDialog object
    // will delete itself when it is closed by the user
    }
#ifdef HAVE_LIBNOTIFY
    else if (prog_config.error_mode == Error::libnotify) { 
      GobjHandle<NotifyNotification> notification{notify_notification_new(
							  gettext("Mount-gtk: error"),
							  message,
							  0)};
      notify_notification_set_image_from_pixbuf(notification, prog_config.window_icon);

      GError* error = 0;
      if (!notify_notification_show(notification, &error)) {
	GerrorScopedHandle handle{error};
	std::cerr << gettext("Failed to send error notification\n")
		  << error->message << std::endl;
      }
    }
#endif
    if (prog_config.beep) beep();
  }
}

void strip(std::string& text) {
  // erase any trailing space or tab
  while (!text.empty() && text.find_last_of(" \t") == text.size() - 1) {
    text.resize(text.size() - 1);
  }
  // erase any leading space or tab
  while (!text.empty() && (text[0] == ' ' || text[0] == '\t')) {
    text.erase(0, 1);
  }
}
