/******************************************************************************
*******************************************************************************
**
**    Copyright 1998 David C. Teigland <teigland@lcse.umn.edu>
**              1999 Grant M. Erickson <gerickson@brocade.com>
**
**    University of Minnesota
** 
**    Module name: fwdl-linux.c
**    Date:        1998/10/08
**
**    Description:
**      This modules implements code which downloads firmware to a Seagate
**      drive attached to a Linux-based computer via the SCSI Generic driver.
**
*******************************************************************************
******************************************************************************/

/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *  
 *    This program is distributed in the hope that it will be useful,  
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the   
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <asm/param.h>

#include <scsi/scsi.h>
#include <scsi/sg.h>

#include <sys/stat.h>

#include "fwdl.h"
#include "fwdl-linux.h"


/* Preprocessor Defines */

/* NOTE: FWCHUNK + sizeof(cdb) + sizeof(sg_header) < SG_BIG_BUFF */

#define FWCHUNK			16384

#define	SCSI_OFF 		sizeof(struct sg_header)
#define	TESTUNITREADY_CMD_LEN	6


/* Function Macros */

#define	Free(p)		if ((p) != NULL) free (p)


/* Type Definitions */

typedef struct scsi_inquiry_s {
  unsigned char	 opcode;
  unsigned char	 evdp:1;
  unsigned char	 reserved:4;
  unsigned char	 lun:3;
  unsigned char	 page_code;
  unsigned char	 reserved2;
  unsigned char	 allocation_length;
  unsigned char	 control_byte;
} scsi_inquiry_t;

typedef struct scsi_wrbuf_s {
  unsigned char	 opcode;
  unsigned char	 mode:3;
  unsigned char	 reserved:2;
  unsigned char	 lun:3;
  unsigned char	 bufferid;
  unsigned char	 bufferoffset[3];
  unsigned char	 paramlistlen[3];
  unsigned char	 controlbyte;
} scsi_wrbuf_t;


/* Global Variables */


/* Function Prototypes */

static int	 openDevice(const char *device, int flags);
static int	 closeDevice(int fd);
static int	 scsiTestUnitReady(int fd);
static int	 scsiInquiry(int fd, void *buffer);
static int	 scsiWriteFW(int fd, void *buffer, unsigned int size);
static int	 handle_SCSI_cmd(int fd, unsigned cmd_len, unsigned in_sz,
				 void *i_buff, unsigned out_sz, void *o_buff);
static void	*get_SCSI_buf(unsigned bufsiz);
static void	 free_SCSI_buf(void *b);

scsi_ops_t linuxOps = {
	openDevice,
	closeDevice,
	scsiTestUnitReady,
	scsiInquiry,
	scsiWriteFW,
};


/******************************************************************************
*******************************************************************************
**
** static int openDevice()
**
** Description:
**   This routine opens a Linux generic SCSI disk device.
**
** Input(s):
**  *device - Name of the target device to open.
**   flags  - Flags passed to the open system call.
**
** Returns:
**   A non-negative integer representing the lowest numbered unused file
**   descriptor. Otherwise, -1 on error.
**
*******************************************************************************
******************************************************************************/

static int
openDevice(const char *device, int flags)
{
  struct stat statbuf;
  int fd = -1;
  int x;

  /* Check to see if the device actually exists first. */

  if (stat(device, &statbuf) < 0)
    {
      fprintf(stderr, "Could not open \"%s\": %s\n", device, strerror(errno));
      exit(EXIT_FAILURE);
    }
  else if (!S_ISCHR(statbuf.st_mode) && !S_ISBLK(statbuf.st_mode))
    {
      fprintf(stderr, "The device file \"%s\" is not a character or block "
	      "device.\n", device);
      exit(EXIT_FAILURE);
    }

  /*
   * We need to have exclusive, non-blocking access to the device in order to
   * download the firmware.
   */

  if ((fd = open(device, O_RDWR)) < 0)
    {
      fprintf(stderr,
	      "Could not open \"%s\" for read/write/exclusive access.\n", 
	      device);
      perror("");
      exit(EXIT_FAILURE);
    }

  /*  Set the timeout to 10 minutes  */

  x = 600 * HZ;
  if (ioctl(fd, SG_SET_TIMEOUT, &x) < 0)
  {
    perror("Can't set timeout");
    exit(EXIT_FAILURE);
  }

  return (fd);
}


/******************************************************************************
*******************************************************************************
**
** static int closeDevice()
**
** Description:
**   This routine closes the device associated with the file descriptor 'fd'.
**
** Input(s):
**   fd - A file descriptor for the device to be closed.
**
** Returns:
**   0 if OK, -1 on error.
**
*******************************************************************************
******************************************************************************/

static int
closeDevice(int fd)
{
  return (close(fd));
}


/******************************************************************************
*******************************************************************************
**
** static int scsiTestUnitReady()
**
** Description:
**   This routine closes the device associated with the file descriptor.
**
** Input(s):
**   fd - File descriptor of the disk target.
**
** Returns:
**   SCSI command status.
**
*******************************************************************************
******************************************************************************/

static int
scsiTestUnitReady(int fd)
{
  static unsigned char cmd[SCSI_OFF + TESTUNITREADY_CMD_LEN];

  bzero(&cmd, SCSI_OFF + TESTUNITREADY_CMD_LEN);
  cmd[SCSI_OFF] = TEST_UNIT_READY;

  if (handle_SCSI_cmd(fd, TESTUNITREADY_CMD_LEN, 0, cmd + SCSI_OFF, 0, NULL)) {
    fprintf(stderr, "TestUnitReady failed\n");
    return(-1);
  }

  return (0);
}


/******************************************************************************
*******************************************************************************
**
** static int scsiInquiry()
**
** Description:
**   This routine...
**
** Input(s):
**   fd     - File descriptor of the disk target.
**  *buffer - Pointer to a buffer sufficiently large enough to store SCSI
**            inquiry data.
**
** Output(s):
**  *buffer - Pointer to the SCSI inquiry data.
**
** Returns:
**   SCSI command status.
**
*******************************************************************************
******************************************************************************/

static int
scsiInquiry(int fd, void *buffer)
{
  scsi_inquiry_t	*si;
  inquiryData_t		*sid;


  si = (scsi_inquiry_t *)get_SCSI_buf(sizeof(scsi_inquiry_t));
  sid = (inquiryData_t *)get_SCSI_buf(sizeof(inquiryData_t));

  si->opcode		= INQUIRY;
  si->allocation_length = 96;

  if (handle_SCSI_cmd(fd, sizeof(scsi_inquiry_t), 0, si, 
		      sizeof(inquiryData_t), sid))
  {
    fprintf( stderr, "Inquiry failed\n");
    Free(si);
    Free(sid);

    return(-1);
  }

  memcpy(buffer, sid, sizeof(inquiryData_t));
  free_SCSI_buf(si);
  free_SCSI_buf(sid);

  return(0);
}


/******************************************************************************
*******************************************************************************
**
** static int scsiWriteFW()
**
** Description:
**   This routine...
**
** Input(s):
**   fd     - 
**  *buffer - 
**   size   - 
**
** Output(s):
**   N/A
**
** Returns:
**   0 if OK, -1 on error.
**
*******************************************************************************
******************************************************************************/

static int
scsiWriteFW(int fd, void *buffer, unsigned int size)
{
  scsi_wrbuf_t	*cmd;
  unsigned int	 n = 0, rem = 0;
  unsigned int	 i, numwr;


  if (size < FWCHUNK) { 
    cmd = (scsi_wrbuf_t *)get_SCSI_buf(sizeof(scsi_wrbuf_t) + size);
    cmd->opcode		 = WRITE_BUFFER;
    cmd->mode		 = 0x05;
    cmd->bufferid	 = 0x00;
    cmd->bufferoffset[0] = 0x00;
    cmd->bufferoffset[1] = 0x00;
    cmd->bufferoffset[2] = 0x00;
    cmd->paramlistlen[0] = (unsigned char)((size >> 16) & 0xff);
    cmd->paramlistlen[1] = (unsigned char)((size >>  8) & 0xff);
    cmd->paramlistlen[2] = (unsigned char)(size & 0xff); 
    cmd->controlbyte	 = 0;

    memcpy(cmd + sizeof(scsi_wrbuf_t), buffer, size );
  
    if (handle_SCSI_cmd(fd, sizeof(scsi_wrbuf_t), size, cmd, 0, NULL)) {
      fprintf(stderr, "WriteBuffer failed\n");
      if (cmd)
	free_SCSI_buf(cmd);
      return(-1);
    }
  }
  else
  {				/* use multiple commands to download */
    numwr = size / FWCHUNK;
  
    if (size % FWCHUNK) {
      numwr++;
      rem = (size % FWCHUNK);
    }

#ifdef DEBUG  
    if (debug) {
      fprintf(stderr, "Number of writes:   %d\n", numwr);
      fprintf(stderr, "Size of final write: %d\n", rem);
    }
#endif

    for(i = 0; i < numwr; i++) {
      if (i < numwr - 1)
	      n = FWCHUNK;
      else if (rem)
	      n = rem;
           
      cmd = (scsi_wrbuf_t *)get_SCSI_buf(sizeof(scsi_wrbuf_t) + n);           
      cmd->opcode	   = WRITE_BUFFER;
      cmd->mode		   = 0x07;
      cmd->bufferoffset[0] = (unsigned char)(((FWCHUNK * i) >> 16) & 0xff);
      cmd->bufferoffset[1] = (unsigned char)(((FWCHUNK * i) >>  8) & 0xff);
      cmd->bufferoffset[3] = (unsigned char)((FWCHUNK * i) & 0xff);
      cmd->paramlistlen[0] = (unsigned char)((n >> 16) & 0xff); 
      cmd->paramlistlen[1] = (unsigned char)((n >>  8) & 0xff); 
      cmd->paramlistlen[2] = (unsigned char)(n & 0xff); 
      
      memcpy((char *)cmd + sizeof(scsi_wrbuf_t),
	     (char *)buffer + FWCHUNK * i, n);
      
      if (handle_SCSI_cmd(fd, sizeof(scsi_wrbuf_t), n, cmd, 0, NULL)) {
        fprintf(stderr, "WriteBuffer(%d) failed\n", i);

        if (cmd)
	  free_SCSI_buf(cmd);

        return(-1);
      }
    
      if (cmd)
	free_SCSI_buf(cmd);
    }       
  }

  return(0);    
}


/******************************************************************************
*******************************************************************************
**
** static int handle_SCSI_cmd()
**
** Description:
**
**
** Input(s):
**   fd       -
**   cmd_len  -
**   in_size  -
**  *vi_buff  -
**   out_size -
**  *vo_buff  -
**
** Output(s):
**  *vo_buff  -
**
** Returns:
**
**
*******************************************************************************
******************************************************************************/

static int
handle_SCSI_cmd(int fd, unsigned int cmd_len, unsigned int in_size,
		void *vi_buff, unsigned int out_size, void *vo_buff)
{
  int			 i, status = 0;
  unsigned char		*tmp;
  struct sg_header	*sg_hd;
  unsigned char		*i_buff = (unsigned char *)vi_buff;
  unsigned char		*o_buff = (unsigned char *)vo_buff;


  if(!cmd_len)
	  return (-1);

  if(!i_buff)
	  return (-1);
  
#ifdef SG_BIG_BUFF
  if(SCSI_OFF + cmd_len + in_size > SG_BIG_BUFF)
	  return (-1);

  if(SCSI_OFF + out_size > SG_BIG_BUFF)
	  return (-1);
#else
  if(SCSI_OFF + cmd_len + in_size > 4096)
	  return (-1);

  if(SCSI_OFF + out_size > 4096)
	  return (-1);
#endif

  if(!o_buff) 
	  out_size = 0;
  else 
	  o_buff -= SCSI_OFF;

  if(!i_buff)
	  return (-1);
  else
	  i_buff -= SCSI_OFF;

#ifdef DEBUG
  if (debug) {
    tmp = (unsigned char *)vi_buff;
    fprintf(stderr, "handleSCSIcmd: CDB: ");
    for (i = 0; i < cmd_len; i++) {
      fprintf(stderr, "%02x ", (unsigned int)tmp[i]);
    }
    fprintf(stderr, "\n");
  }
#endif

  /* Generic SCSI device header construction */

  sg_hd = (struct sg_header *)i_buff;
  sg_hd->reply_len = SCSI_OFF + out_size;
  sg_hd->twelve_byte = cmd_len == 12;
  sg_hd->result = 0;

  /* Send command */

  status = write(fd, i_buff, SCSI_OFF + cmd_len + in_size); 

  if (status < 0 || status != SCSI_OFF + cmd_len + in_size || sg_hd->result)
  {
    /* Some error happened */

    fprintf(stderr, "handleSCSIcmd: write result = 0x%x cmd = 0x%x\n",
            sg_hd->result, i_buff[SCSI_OFF]); 

    if (status < 0) {
      perror("write");
    }
  }

  /* Use the sg_hd space from i_buff */
  if(!o_buff)
	  o_buff = i_buff;
  
  sg_hd = (struct sg_header *)o_buff;

  /* Retreive the result */

  status = read(fd, o_buff, SCSI_OFF + out_size); 

  if(status < 0 || status != SCSI_OFF + out_size || 
     sg_hd->result || sg_hd->sense_buffer[0])
  {
    int i;

    fprintf(stderr, "handleSCSIcmd: read status = 0x%x, result = 0x%x\n",
            status, sg_hd->result);

    fprintf(stderr, "handleSCSIcmd: read sense");

    for (i = 0; i < 16; i++) {
      fprintf(stderr, " %x", sg_hd->sense_buffer[i]);
    }
    fprintf(stderr, "\n");

    if(status < 0)
      perror("read");
  }

  if (status == SCSI_OFF + out_size)
	  status = 0;

  if (sg_hd->sense_buffer[0] != 0x0)
	  status = 1;  /* not GOOD */

  return (status);
}


/******************************************************************************
*******************************************************************************
**
** static void get_SCSI_buf()
**
** Description:
**   This routine...
**
** Input(s):
**   size - 
**
** Returns:
**   N/A
**
*******************************************************************************
******************************************************************************/

static void *
get_SCSI_buf(unsigned size)
{
  char	*buffer = (char *)malloc(size + SCSI_OFF);


  if (!buffer) {
    perror("malloc");
    exit(-1);
  }

  memset(buffer, 0, size + SCSI_OFF);

  return(buffer + SCSI_OFF);
}


/******************************************************************************
*******************************************************************************
**
** static void free_SCSI_buf()
**
** Description:
**   This routine...
**
** Input(s):
**  *buffer - 
**
** Returns:
**   N/A
**
*******************************************************************************
******************************************************************************/

static void
free_SCSI_buf(void *buffer)
{
  Free((char *)buffer - SCSI_OFF);
}
