/*
    Falling Block Game
    Copyright (C) 1999-2002 Jared Krinke <http://derajdezine.vze.com/>


    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    version 2 as published by the Free Software Foundation.

    This application is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this distribution; if not, write to:
    Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA02111-1307  USA

    Jared Krinke

    Deraj DeZine
    http://derajdezine.vze.com/
*/

#ifndef FBG_H
#define FBG_H
#include "glTGAImage.h"
#include <SDL/SDL.h>
#include <SDL/SDL_thread.h>
#include <vector>
#include <string>
#include <physfs.h>
#ifdef ENABLE_SOUND
#include <mikmod.h>
#endif
using namespace std;

class fbgLineGain;	// What was gained
class fbgBlock;	// A block (including the matrix)
class fbgGame;	// The main playing field matrix and such

// In milliseconds, the delay between a block's downward movement for every level (accurately copied from handheld game)
// DELAY = 920/(1+0.05*e^(0.44*LEVEL)) + 30
const int fbgLevelDelays[] = {1000, 900, 830, 800, 735, 655, 580, 500, 400, 300, 190, 160, 140, 125, 115, 100, 85, 70, 60, 55, 42};
#define LINESPEED	500
#define LIGHTLINEGAINREDRAWSPEED	100

// When line(s) have been gained
class fbgLineGain {
public:
	unsigned char lineCount; // 1-4 lines removed
	short linePos[4];
	short lineMatrix[4][10];
	int levelGained;	// Level that was attained
	fbgLineGain() {levelGained = -1;lineCount = 0;linePos[0] = -1;linePos[1] = -1;linePos[2] = -1;linePos[3] = -1;}
};

// Holds block's matrix and can perform rotating and such functions
class fbgBlock {
protected:
	bool matrix[4][4];
	short posX,posY;	// Position within game
	fbgGame* game;
	short index;	// Blocks color/pattern

	// Matrix Management
	void shiftLeft();
	void shiftRight();
	void shiftUp();
	void shiftDown();
	void shift(int targetX, int targetY);	// Center horizontally and move to top

	void rotateLeft();	// Actually moves bricks in matrix
	void rotateRight();
public:
	// Block Management
	bool checkBlockPosition() const;
	bool moveBlockDown();	// Returns if block was actually moved
	bool moveBlockRight();
	bool moveBlockLeft();
	bool moveBlockUp();
	void rotateBlockRight();
	void rotateBlockLeft();

	fbgGame* getGame() const {return game;}
	short getX();	// From left where block starts
	short getY();	// From top where block starts
	short getX2();	// From right, where block stats
	short getY2();	// From bottom, where block starts
	short getWidth() {return getX2() - getX() + 1;}
	short getHeight() {return getY2() - getY() + 1;}
	short getPosX() const {return posX;}
	short getPosY() const {return posY;}
	void setPosX(short newX) {posX = newX;}
	void setPosY(short newY) {posY = newY;}
	short getIndex() const {return index;}

	fbgBlock(fbgGame* newGame, short newIndex, bool newMatrix[16]);	// Constructor
	fbgBlock();
	void operator=(const fbgBlock&);	// Copy

	bool getMatrix(int row, int col) const {return matrix[row][col];}
};

class fbgOpenGLRenderer {
protected:
	fbgGame* game;
	// Textures
	glTGAImage *bg,*ml,*mb,*mt,*nb, *gameover;
	glTGAImage nr;
	glTGAImage *blocks;
	glTGAImage *digits;
	// Display Lists
	GLuint cube;
	GLuint cubeWithoutBack;
	GLuint digitsBase;
	GLfloat lineZSpeeds[4][10];

	void loadTextures();
	void killTextures();
	void buildLists();	// Build OpenGL display lists
	void killLists();
	void drawDigits(const string& str);
	void drawGameplay();	// Normal rendering during gameplay
	void drawLineGain();	// Rendering during a line gain
	void drawGameOver();	// When the game ends
	void drawGameMatrix();	// Code-saving function for drawing the game matrix
	void drawNextBlock(fbgBlock* theBlock);
public:
	bool init(bool fs=false, int width=640, int height=480, int bpp=24);
	void draw();
	void exit();

	void setGame(fbgGame* newGame) {game = newGame;}
};

// Board and game management
enum {EVT_REDRAW=1, EVT_DROP, EVT_REMOVELINEGAIN};
enum fbgGameState {GAMEPLAY=0, GAMEOVER, QUIT, BEGINWAIT};
const bool fbgGameStatePauseMask[] = {false, true, false, true};
class fbgGame {
protected:
	// Board-related
	short matrix[18][10];	// Game board
	vector<fbgBlock> blockSet;	// Different pieces
	fbgBlock curBlock;	// Needed in case we rotate/nudge
	short nextBlockIndex;	// Next block's type
	Uint32 lastDrop;	// Last time the block was lowered
	Uint32 nextDrop;	// When the piece should be moved down
	Uint32 lastLine;	// When the last line when gained
	fbgLineGain* lineGain;	// If we're currently animating a line gain this will != NULL
	bool paused;	// If gameplay is paused (ie during a line gain)
	fbgGameState state;	// The current game state
	string prefix;	// Prefix for loading files
	short btype;	// Height of B-Type garbage
	SDL_TimerID dropBlockID;	// ID of dropBlock callback thread
	SDL_Thread* removeLineGainID;	// ID of removeLineGain thread
	SDL_Thread* lightLineGainRedrawID;	// ID of lightLineGainRedraw thread
	fbgOpenGLRenderer renderer;	// Pointer to the renderer
	bool light;	// Use "light" rendering/processing?
	string theme;	// Name of the selected theme

	// Game-related
	short downStart;	// Where block started moving down
	bool downHeld, leftHeld, rightHeld;	// If the keys are held
	bool wantsToMove;	// If the block should be pushed under another
	int lines;	// Number of lines removed
	unsigned char level;	// Current level
	long score;	// Current score
	float curBlockAdjust;	// Maximum positive adjustment for smooth block falling

	// Music
#ifdef ENABLE_SOUND
	void* readerAddress;
	MODULE* music;
	SDL_Thread* musicID;
#endif

	// Block Management
	void cycleBlocks();
	void putBlockInMatrix(const fbgBlock& theBlock);
	void setupBlocks();

	// Timing functions
	void resetDropTimer();

	// Static Functions
	static Uint32 dropBlock(Uint32 interval, void* param);	// Block-drop timing callback
	static int removeLineGain(void* param);
	static int lightLineGainRedraw(void* param);	// Automatic redraw thread
#ifdef ENABLE_SOUND
	static int musicThread(void *param);	// Music thread
#endif
public:
	// Event Functions
	static inline void issue(const int& eventCode);
	void processEvent(const SDL_Event& event);
	void gameplayLoop();

	// PhysicsFS
	PHYSFS_file* loadThemeFile(const string& file);

	// Block Management
	void finishCurBlock();

	short getMatrix(int row, int col) {return matrix[row][col];}
	fbgBlock* getCurBlock() {return &curBlock;}
	void verifyCurBlockAdjust();

	// Music
#ifdef ENABLE_SOUND
	void initSound();
	void initMusic(const string& musicFile);
	void stopMusic();
	void stopSound();
#endif

	// Accessors
	fbgLineGain* getLineGain() {return lineGain;}
	fbgBlock* getBlockSet(int index) {return &blockSet[index];}
	short getNextBlockIndex() {return nextBlockIndex;}
	unsigned char getLevel() {return level;}
	int getLines() {return lines;}
	long getScore() {return score;}
	bool getDownHeld() {return downHeld;}
	bool getRightHeld() {return rightHeld;}
	bool getLeftHeld() {return leftHeld;}
	void setDownHeld(bool newHeld) {
		downHeld = newHeld;
		if (downHeld == true) downStart = getCurBlock()->getPosY();
		else downStart = -1;
	}
	void setRightHeld(bool newHeld) {rightHeld = newHeld;}
	void setLeftHeld(bool newHeld) {leftHeld = newHeld;}
	void setWantsToMove(bool newMove) {wantsToMove = newMove;}
	bool getWantsToMove() {return wantsToMove;}
	Uint32 getLastDrop() {return lastDrop;}
	Uint32 getNextDrop() {return nextDrop;}
	void setLastDrop(const Uint32& newDrop);
	void setNextDrop(const Uint32& newDrop);
	Uint32 getLastLine() {return lastLine;}
	void setLastLine(const Uint32& newLine) {lastLine = newLine;}
	bool getPaused() {return paused;}
	void setPaused(bool newP) {paused = newP;resetDropTimer();}
	float getCurBlockAdjust() {return curBlockAdjust;}
	void setCurBlockAdjust(float newAdjust) {curBlockAdjust = newAdjust;}
	fbgGameState getState() {return state;}
	void setState(fbgGameState newState) {state = newState; setPaused(fbgGameStatePauseMask[newState]);}
	fbgOpenGLRenderer* getRenderer() {return &renderer;}
	bool getLight() {return light;}
	const string& getTheme() {return theme;}
	// For smooth piece descent
	float getSmoothAdjust();

	// Constructors
	fbgGame(const char* argv0, const string& newTheme, bool newLight = false, int startLevel=1, int newbtype=-1, const string& prefix = "./", const char* musicFile = NULL);
	// Destructor
	~fbgGame();
};

#endif
