/*---------------------------------------------------------------------+
| Module-name:                                                         |
|        callmt2.c (IBM Storage Protect Authorized Version)         |
|                                                                      |
| Function:                                                            |
|        Invokes the series of IBM Storage Protect APIs necessary to|
|        backup an object.                                             |
|                                                                      |
|        o Thread1 will backup 420 bytes of data, query the backup,    |
|          and then restore the data.                                  |
|        o Thread2 will backup 440 bytes of data, query the backup,    |
|          and then restore the data.                                  |
|        o Thread3 will backup 460 bytes of data to a different server |
|          specified by the 'ALTSERVER' variable in the program, query |
|          the session options, and then restore the data.             |
|        o Thread4 will archive 480 bytes of data to a different       |
|          server specified by the 'ALTSERVER' variable in the         |
|          program, query the session options, and then restore the    |
|          data.                                                       |
|                                                                      |
| Setup:                                                               |
|        You should first setup the environment variables to point to  |
|        the correct directories and dsm.opt file.  To do this, make   |
|        sure that DSMI_CONFIG points to your dsm.opt file, and        |
|        DSMI_DIR and DSMI_LOG point to your working directory.  This  |
|        is a sample setenv script...                                  | 
|                                                                      |
|           #!/bin/ksh                                                 |
|           export DSMI_CONFIG=$PWD/dsm.opt                            |
|           export DSMI_DIR=$PWD                                       |
|           export DSMI_LOG=$PWD                                       |
|           export DSM_CONFIG=$PWD/dsm.opt                             |
|           export DSM_DIR=$PWD                                        |
|                                                                      |
|        The next thing that should be done is the creation of the     |
|        dsm.opt and dsm.sys files.  The line in your dsm.opt should   |
|        be...                                                         |
|                                                                      |
|           SERVERNAME server2                                         |
|                                                                      |
|        With the exception of the tcpserver addresses, the contents   |
|        of dsm.sys should be...                                       |
|                                                                      |
|           defaultserver server1                                      |
|                                                                      |
|           servername server1                                         |
|              commmethod     tcpip                                    |
|              tcpport        1500                                     |
|              tcpserver      1.111.111.111                            |
|                                                                      |
|           servername altserver1                                      |
|              commmethod     tcpip                                    |
|              tcpport        1500                                     |
|              tcpserver      2.222.222.22                             |
|                                                                      |
|           servername server2                                         |
|              commmethod     tcpip                                    |
|              tcpport        1500                                     |
|              tcpserver      1.111.111.111                            |
|              passwordaccess generate                                 |
|              nodename       nodegen                                  |
|              passworddir    /mydir                                   |
|                                                                      |
|           servername altserver2                                      |
|              commmethod     tcpip                                    |
|              tcpport        1500                                     |
|              tcpserver      2.222.222.22                             |
|              passwordaccess generate                                 |
|              nodename       nodegen                                  |
|              passworddir    /mydir                                   |
|                                                                      |
|        Next, in the two servers specified in your dsm.sys file, you  |
|        will have to register the node nodegen.  To do this, use      |
|        'dsmadmc' to logon to your IBM Storage Protect server, and |
|        type 'reg node nodegen nodegen'. Do this for the other server,|
|        and you are almost ready to begin. Compile the source code by |
|        using the supplied make file, and run the executable.         |
|                                                                      |
|           AIX: make -f makemt.aix                                    |
|           SUN: make -f makemt.sol                                    |
|                                                                      |
| Note:                                                                |
|        The password files for password generate should be            | 
|        created by running the regular sample (dapismp) as            | 
|        ADSM_AUTHORIZED (main menu 8, 0 then 0,0 password nodegen).   |
|        See the README.API for detailed explanation.                  |
|        Also, the 's' bit on the 'callmt2' executable                 |
|        should be on, as well as the owenership of the file set to    |
|        the non-root user that will be using it.  This will allow     |
|        that user to be IBM Storage Protect Authorized.            |
|                                                                      |
+---------------------------------------------------------------------*/

#include <pthread.h>

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>

#include "dsmapitd.h"
#include "dsmapifp.h"
#include "dsmrc.h"

#define ALTSERVER "altserver2"

#define MTAPI     316

void *Thread1(void *string);
void *Thread2(void *string);
void *Thread3(void *string);
void *Thread4(void *string);

int NumberOfThreads;

int main(int argc, char *argv[]) 
{
   pthread_t       thrd1, thrd2, thrd3, thrd4;
   pthread_attr_t  attr;
   dsInt16_t       rc = 0, apiver =0;
   dsUint32_t      apiVersion, applVersion;
   dsmApiVersionEx apiLibVer;
   envSetUp        dsmEnvSetUp;
   char            *rcMsg;
   int		       detached;
   char            uniStr[20];

   /* set attr to detached state so resources will be released
      when thread exists */
   pthread_attr_init(&attr);

#if defined(_HPUX)
   pthread_attr_setstacksize (&attr, 1024 * 1024);
#endif

#if defined(_OPEN_THREADS)
   detached = __DETACHED;
   pthread_attr_setdetachstate(&attr, &detached);
#else
   pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
#endif
   NumberOfThreads = 1;

   memset(&apiLibVer, 0x00, sizeof(dsmApiVersionEx));
   apiLibVer.stVersion = apiVersionExVer;
   dsmQueryApiVersionEx(&apiLibVer);

   applVersion = (10000 * DSM_API_VERSION) + (1000 * DSM_API_RELEASE) + (100 * DSM_API_LEVEL)
                 + DSM_API_SUBLEVEL;

   apiVersion = (10000 * apiLibVer.version) + (1000 * apiLibVer.release) + (100 * apiLibVer.level)
                 + apiLibVer.subLevel;

   /* check for compatibility problems */
   if (apiVersion < applVersion)
   { printf("The IBM Storage Protect API library Version = %d.%d.%d.%d is at a lower version\n",
        apiLibVer.version,
        apiLibVer.release,
        apiLibVer.level,
        apiLibVer.subLevel);
     printf("than the application version = %d.%d.%d.%d.\n",
        DSM_API_VERSION,
        DSM_API_RELEASE,
        DSM_API_LEVEL,
        DSM_API_SUBLEVEL);
     printf("Please upgrade the API accordingly.\n");
     return -1;
   }

   apiver = (apiLibVer.version * 100) + (apiLibVer.release * 10) + apiLibVer.level;
   if (apiver < MTAPI)
   {
      printf("\n***********************************************************\n");
      printf(  "The IBM Storage Protect API library is lower than 3.1.6 \n");
      printf(  "(multithreaded API)                                        \n");
      printf(  "Install the current level.                                 \n");
      printf(  "***********************************************************\n");
      return 0;
   }

   if (apiver >= 412)
   {
      if(apiLibVer.unicode)
         strcpy(uniStr,"(unicode)");
      else
         strcpy(uniStr,"         ");
   }

   printf("\n*************************************************************\n");
   printf(  "* Welcome to the sample application for                     *\n");
   printf(  "* IBM Storage Protect API.                                 *\n");
   printf(  "* API Library Version = %d.%d.%d.%d %s                   *\n",
   apiLibVer.version,
   apiLibVer.release,
   apiLibVer.level,
   apiLibVer.subLevel,
   uniStr);
   printf(  "*************************************************************\n");

   memset(&dsmEnvSetUp, 0x00, sizeof(envSetUp));

   dsmEnvSetUp.stVersion = envSetUpVersion;
   strcpy(dsmEnvSetUp.dsmiDir, ""); 
   strcpy(dsmEnvSetUp.dsmiConfig, ""); 
   strcpy(dsmEnvSetUp.dsmiLog, "");
   strcpy(dsmEnvSetUp.logName, "");
   dsmEnvSetUp.argv = argv;

   if ((rc = dsmSetUp(DSM_MULTITHREAD, &dsmEnvSetUp)) != DSM_RC_OK) 
   {
      rcMsg = (char *)malloc(DSM_MAX_RC_MSG_LENGTH);
      dsmRCMsg(0,rc,rcMsg);
      printf("Error in dsmSetUp rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmCleanUp(DSM_MULTITHREAD);
      return 0;
   }

   NumberOfThreads++;
   rc = pthread_create(&thrd1, &attr, Thread1, NULL);
   if(rc) {
     printf("Thread1 rc is %d\n",rc);
     return 0;
   }

   NumberOfThreads++;
   rc = pthread_create(&thrd2, &attr, Thread2, NULL);
   if(rc) {
     printf("Thread2 rc is %d\n",rc);
     return 0;
   }

   NumberOfThreads++;
   rc = pthread_create(&thrd3, &attr, Thread3, NULL);
   if(rc) {
     printf("Thread3 rc is %d\n",rc);
     return 0;
   }

   NumberOfThreads++;
   rc = pthread_create(&thrd4, &attr, Thread4, NULL);
   if(rc) {
     printf("Thread4 rc is %d\n",rc);
     return 0;
   }

   while(NumberOfThreads > 1)
      sleep(1);

   printf("Number of threads is >%d<  \n",NumberOfThreads);

   dsmCleanUp(DSM_MULTITHREAD);
   sleep(1);

   pthread_attr_destroy(&attr);

   return 0;
}

void *Thread1(void *string) 
{
   dsInt16_t          rc;
   int                i;
   char               options[100];
   char               configfile[10];
   mcBindKey          mcBindKey;
   ObjAttr            objAttrArea;
   DataBlk            dataBlkArea, qDataBlkArea, getBlk;
   qryBackupData      queryBuffer;
   qryRespBackupData  qbDataArea;
   dsUint16_t         reason;
   dsmObjName         objName;
   dsmQueryType       queryType;
   dsmGetList         dsmObjList;
   regFSData          regFilespace;
   dsmApiVersion      dsmApiVer;
   char               *rcMsg;
   dsUint32_t         handle = 0;

   dsmInitExIn_t      initIn;
   dsmInitExOut_t     initOut;
   dsmApiVersionEx    apiApplVer;

   memset(&initIn,     0x00, sizeof(dsmInitExIn_t));
   memset(&initOut,    0x00, sizeof(dsmInitExOut_t));
   memset(&apiApplVer, 0x00, sizeof(dsmApiVersionEx));

   printf("Thread1 started\n");

   dsmApiVer.version = DSM_API_VERSION;
   dsmApiVer.release = DSM_API_RELEASE;
   dsmApiVer.level   = DSM_API_LEVEL  ;

   apiApplVer.version  = DSM_API_VERSION;
   apiApplVer.release  = DSM_API_RELEASE;
   apiApplVer.level    = DSM_API_LEVEL  ;
   apiApplVer.subLevel = DSM_API_SUBLEVEL;

   /************** Initialize the API session ******************************/
   rcMsg = (char *)malloc(DSM_MAX_RC_MSG_LENGTH);

   /************ The following 2 can be updated optionally *****************/
   configfile[0]= '\0';    /* API config file for dsmInit */
   options[0] = '\0';

   initIn.stVersion        = dsmInitExInVersion;
   initIn.apiVersionExP    = &apiApplVer;
   initIn.clientNodeNameP  = NULL;
   initIn.clientOwnerNameP = NULL;
   initIn.clientPasswordP  = NULL;
   initIn.applicationTypeP = "API Test1";
   initIn.configfile       = configfile;
   initIn.options          = options;
   initIn.userNameP        = NULL;
   initIn.userPasswordP    = NULL;

   if ((rc = dsmInitEx(&handle,&initIn,&initOut)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("thread1 error in dsmInit. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   printf("Thread1 after dsmInit, session handle is %d\n",handle);

   strcpy(objName.fs,"/t1fs");
   strcpy(objName.hl,"/testhl");
   strcpy(objName.ll,"/testll");
   objName.objType = DSM_OBJ_FILE;

   sprintf(objName.ll, "%s%d",objName.ll,handle);

   /************ Build File Space ****************************************/
   regFilespace.fsName = (char *)malloc(100);
   regFilespace.fsType = (char *)malloc(100);

   strcpy(regFilespace.fsName,"/t1fs");
   strcpy(regFilespace.fsType,"CALLER1_FS");
   regFilespace.capacity.lo = 0;
   regFilespace.capacity.hi = 0;
   regFilespace.occupancy.lo = 0;
   regFilespace.occupancy.hi = 0;
   regFilespace.stVersion = regFSDataVersion;
   regFilespace.fsAttr.unixFSAttr.fsInfoLength = 6;
   strcpy(regFilespace.fsAttr.unixFSAttr.fsInfo,"fsinfo");

   /******************** Register the file space ************************/
   rc = dsmRegisterFS(handle, &regFilespace);

   if ((rc != DSM_RC_OK) && (rc != DSM_RC_FS_ALREADY_REGED)) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmRegisterFS. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmBeginTxn. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   mcBindKey.stVersion = mcBindKeyVersion;

   if ((rc = dsmBindMC(handle, &objName, stBackup, &mcBindKey)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmBindMC. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;      
      return NULL;
   }

   /****************** Backup the data ******************************/
   strcpy(objAttrArea.owner,"");
   objAttrArea.sizeEstimate.hi = 0;           /* size estimate of object */
   objAttrArea.sizeEstimate.lo = 420;         /* size estimate of object */
   objAttrArea.objCompressed = bFalse;
   objAttrArea.objInfoLength = 10;            /* length of objInfo */
   objAttrArea.objInfo = (char *)malloc(objAttrArea.objInfoLength + 1);
   strcpy(objAttrArea.objInfo,"abcdefghij");  /* object-dependent info */
   objAttrArea.stVersion = ObjAttrVersion;
   objAttrArea.mcNameP = NULL;

   if ((rc = dsmSendObj(handle, stBackup, NULL, &objName, &objAttrArea, 
        NULL)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmBackup. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;     
      return NULL;
   }

   dataBlkArea.bufferPtr = (char *)malloc(20);
   dataBlkArea.stVersion = DataBlkVersion;
   dataBlkArea.bufferLen = 20;              /* buffer length in bytes */
   /* datastream to backup */
   memcpy(dataBlkArea.bufferPtr,"T1T1T1T1T1T1T1T1T1T1", 20); 

   for (i=1; i<=21; i++) 
   {
      if ((rc = dsmSendData(handle,&dataBlkArea)) != DSM_RC_OK) 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread1 error in dsmSendData. rcMsg=%s\n",rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--;        
         return NULL;
      }

      if(i == 20)
         memcpy(dataBlkArea.bufferPtr,"T1T1T1T1T1T1T1T1TlT\0", 20);
   }

   dsmEndSendObj(handle);

   if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmEndTxn. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;     
      return NULL;
   }

   queryType = qtBackup;

   qbDataArea.stVersion = qryRespBackupDataVersion;
   qDataBlkArea.stVersion = DataBlkVersion;
   qDataBlkArea.bufferPtr = (char *)&qbDataArea;
   qDataBlkArea.bufferLen = sizeof(qryRespBackupData);

   queryBuffer.objName = (dsmObjName *)malloc(sizeof(dsmObjName));
   queryBuffer.owner = (char *)malloc(100);
 
   strcpy(queryBuffer.objName->fs, objName.fs);
   strcpy(queryBuffer.objName->hl, objName.hl);
   strcpy(queryBuffer.objName->ll, objName.ll);
   queryBuffer.objName->objType = objName.objType;
   strcpy(queryBuffer.owner, "");
   queryBuffer.objState = DSM_ACTIVE;
   queryBuffer.stVersion = qryBackupDataVersion;

   if ((rc=dsmBeginQuery(handle,queryType, (void *)&queryBuffer )) 
        != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error in dsmBeginQuery. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;     
      return NULL;
   }

   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
     ;

   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA)) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error from dsmGetNextQObj. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;     
      return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error from dsmEndQuery. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;     
      return NULL;
   }
 
   printf("\n*** Thread1 query response ***\n*** %s%s%s ***\n",
      qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll);

   dsmObjList.stVersion = dsmGetListVersion;
   dsmObjList.numObjId = 1;
   dsmObjList.objId = (ObjID *)malloc(sizeof(ObjID) * dsmObjList.numObjId);

   dsmObjList.objId[0].hi = qbDataArea.objId.hi;
   dsmObjList.objId[0].lo = qbDataArea.objId.lo;

   getBlk.bufferPtr = (char *)malloc(500);
   memset(getBlk.bufferPtr,0x00,sizeof(getBlk.bufferPtr));
   getBlk.bufferLen = 500;
   getBlk.stVersion = DataBlkVersion;

   memset(getBlk.bufferPtr,0x00,sizeof(getBlk.bufferPtr));

   if ((rc = dsmBeginGetData(handle,bTrue,gtBackup,&dsmObjList)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error from dsmBeginGetData. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   rc = dsmGetObj(handle, &(dsmObjList.objId[0]), &getBlk);

   if (rc == DSM_RC_FINISHED) 
   {
      if ((rc = dsmEndGetObj(handle)) != DSM_RC_OK) 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread1 error from dsmEndGetObj. rcMsg=%s\n", rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--;  
         return NULL;
      }
   }
   else if (rc == DSM_RC_MORE_DATA) 
   {
      while ((rc = dsmGetData(handle, &getBlk)) == DSM_RC_MORE_DATA)
        ;
      if (rc == DSM_RC_FINISHED) 
      {
         if ((rc = dsmEndGetObj(handle)) != DSM_RC_OK) 
         {
            dsmRCMsg(handle,rc,rcMsg);
            printf("Thread1 error from dsmEndGetObj. rcMsg=%s\n", rcMsg);
            free(rcMsg);
            dsmTerminate(handle);
            NumberOfThreads--;
            return NULL;
         }
      }
      else  /* dsmGetData did not finish */ 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread1 error from dsmGetData. rcMsg=%s\n", rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--; 
         return NULL;
      }
   }
   else  /* dsmGetObj did not finish */
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error from dsmGetObj. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   if ((rc = dsmEndGetData(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error from dsmEndGetData. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   printf("Thread1 bytes restored >%d<\n >%s<\n",getBlk.numBytes, 
      (char *)getBlk.bufferPtr);

   if ((rc = dsmTerminate(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread1 error from dsmTerminate. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      NumberOfThreads--;
      return NULL;  
   }

   free(regFilespace.fsName);
   free(regFilespace.fsType);
   free(objAttrArea.objInfo);
   free(dsmObjList.objId);
   free(getBlk.bufferPtr);
   free(dataBlkArea.bufferPtr);
   free(queryBuffer.objName);
   free(queryBuffer.owner);
   free(rcMsg);

   printf("Thread 1 called dsmTerminate\n");
   printf("Number of threads = >%d< \n",NumberOfThreads);
   NumberOfThreads--;  
   return NULL;
}

void *Thread2(void *string) 
{
   dsInt16_t          rc;
   int i;
   char               options[100];
   char               configfile[10];
   mcBindKey          mcBindKey;
   ObjAttr            objAttrArea;
   DataBlk            dataBlkArea, qDataBlkArea, getBlk;
   qryBackupData      queryBuffer;
   qryRespBackupData  qbDataArea;
   dsUint16_t         reason;
   dsmObjName         objName;
   dsmQueryType       queryType;
   dsmGetList         dsmObjList;
   regFSData          regFilespace;
   dsmApiVersion      dsmApiVer;
   char               *rcMsg;
   dsUint32_t         handle = 0;

   dsmInitExIn_t      initIn;
   dsmInitExOut_t     initOut;
   dsmApiVersionEx    apiApplVer;

   memset(&initIn,     0x00, sizeof(dsmInitExIn_t));
   memset(&initOut,    0x00, sizeof(dsmInitExOut_t));
   memset(&apiApplVer, 0x00, sizeof(dsmApiVersionEx));

   printf("Thread2 started\n");

   dsmApiVer.version = DSM_API_VERSION;
   dsmApiVer.release = DSM_API_RELEASE;
   dsmApiVer.level   = DSM_API_LEVEL  ;

   apiApplVer.version  = DSM_API_VERSION;
   apiApplVer.release  = DSM_API_RELEASE;
   apiApplVer.level    = DSM_API_LEVEL  ;
   apiApplVer.subLevel = DSM_API_SUBLEVEL;

   /************ Initialize the API session *******************************/
   rcMsg = (char *)malloc(DSM_MAX_RC_MSG_LENGTH);

   /************ The following 2 can be updated optionally ****************/
   configfile[0]= '\0';         /* API config file for dsInit */
   options[0]= '\0';            /* options string */

   initIn.stVersion        = dsmInitExInVersion;
   initIn.apiVersionExP    = &apiApplVer;
   initIn.clientNodeNameP  = NULL;
   initIn.clientOwnerNameP = NULL;
   initIn.clientPasswordP  = NULL;
   initIn.applicationTypeP = "API Test2";
   initIn.configfile       = NULL;
   initIn.options          = NULL;
   initIn.userNameP        = NULL;
   initIn.userPasswordP    = NULL;

   if ((rc = dsmInitEx(&handle,&initIn,&initOut)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error in dsmInit. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;   
      return NULL;
   }

   printf("Thread2 after dsmInit, session handle is %d\n",handle);
   
   strcpy(objName.fs,"/t2fs");
   strcpy(objName.hl,"/testhl");
   strcpy(objName.ll,"/testll");
   objName.objType = DSM_OBJ_FILE;

   sprintf(objName.ll, "%s%d",objName.ll,handle);

   /************ Build File Space ****************************************/
   regFilespace.fsName = (char *)malloc(100);
   regFilespace.fsType = (char *)malloc(100);

   strcpy(regFilespace.fsName,"/t2fs");
   strcpy(regFilespace.fsType,"CALLER2_FS");
   regFilespace.capacity.lo = 0;
   regFilespace.capacity.hi = 0;
   regFilespace.occupancy.lo = 0;
   regFilespace.occupancy.hi = 0;
   regFilespace.stVersion = regFSDataVersion;
   regFilespace.fsAttr.unixFSAttr.fsInfoLength = 6;
   strcpy(regFilespace.fsAttr.unixFSAttr.fsInfo,"fsinfo");

   /*************** Register the file space *****************************/
   rc = dsmRegisterFS(handle, &regFilespace);

   if ((rc != DSM_RC_OK) && (rc != DSM_RC_FS_ALREADY_REGED)) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error in dsmRegisterFS. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;     
      return NULL;
   }

   if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error in dsmBeginTxn. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   mcBindKey.stVersion = mcBindKeyVersion;

   if ((rc = dsmBindMC(handle, &objName, stBackup, &mcBindKey)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error in dsmBindMC. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   strcpy(objAttrArea.owner,"");
   objAttrArea.sizeEstimate.hi = 0;       /* size estimate of object */
   objAttrArea.sizeEstimate.lo = 440;       /* size estimate of object */
   objAttrArea.objCompressed = bFalse;
   objAttrArea.objInfoLength = 10;          /* length of objInfo */
   objAttrArea.objInfo = (char *)malloc(objAttrArea.objInfoLength + 1);
   strcpy(objAttrArea.objInfo,"abcdefghij");      /* object-dependent info */
   objAttrArea.stVersion = ObjAttrVersion;
   objAttrArea.mcNameP = NULL;

   /**************** Backup the data  *******************************/
   if ((rc = dsmSendObj(handle, stBackup, NULL, &objName, &objAttrArea, 
        NULL)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error in dsmBackup. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   dataBlkArea.bufferPtr = (char *)malloc(20);
   dataBlkArea.stVersion = DataBlkVersion;
   dataBlkArea.bufferLen = 20;              /* buffer length in bytes */
   /* datastream to backup */
   memcpy(dataBlkArea.bufferPtr,"T2T2T2T2T2T2T2T2T2T2", 20); 

   for (i=1; i<=22; i++) 
   {
      if ((rc = dsmSendData(handle,&dataBlkArea)) != DSM_RC_OK) 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread2 error in dsmSendData. rcMsg=%s\n",rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--; 
         return NULL;
      }

      if(i == 21)
         memcpy(dataBlkArea.bufferPtr,"T2T2T2T2T2T2T2T2T2T\0", 20);
   }

   dsmEndSendObj(handle);

   if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error in dsmEndTxn. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   queryType = qtBackup;

   qbDataArea.stVersion = qryRespBackupDataVersion;
   qDataBlkArea.stVersion = DataBlkVersion;
   qDataBlkArea.bufferPtr = (char *)&qbDataArea;
   qDataBlkArea.bufferLen = sizeof(qryRespBackupData);

   queryBuffer.objName = (dsmObjName *)malloc(sizeof(dsmObjName));
   queryBuffer.owner = (char *)malloc(100);

   strcpy(queryBuffer.objName->fs, objName.fs);
   strcpy(queryBuffer.objName->hl, objName.hl);
   strcpy(queryBuffer.objName->ll, objName.ll);
   queryBuffer.objName->objType = objName.objType;
   strcpy(queryBuffer.owner, "");
   queryBuffer.objState = DSM_ACTIVE;
   queryBuffer.stVersion = qryBackupDataVersion;

   if ((rc=dsmBeginQuery(handle,queryType, (void *)&queryBuffer )) 
        != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error in dsmBeginQuery. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
      ;

   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA)) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error from dsmGetNextQObj. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error from dsmEndQuery. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   printf("\n*** Thread2 query response ***\n*** %s%s%s ***\n", 
	  qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll);

   dsmObjList.stVersion = dsmGetListVersion;
   dsmObjList.numObjId = 1;
   dsmObjList.objId = (ObjID *)malloc(sizeof(ObjID) * dsmObjList.numObjId);

   dsmObjList.objId[0].hi = qbDataArea.objId.hi;
   dsmObjList.objId[0].lo = qbDataArea.objId.lo;

   if ((rc = dsmBeginGetData(handle,bTrue,gtBackup,&dsmObjList)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error from dsmBeginGetData. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   getBlk.bufferPtr = (char *)malloc(500);
   memset(getBlk.bufferPtr,0x00,sizeof(getBlk.bufferPtr));
   getBlk.bufferLen = 500;
   getBlk.stVersion = DataBlkVersion;

   rc = dsmGetObj(handle, &(dsmObjList.objId[0]), &getBlk);

   if (rc == DSM_RC_FINISHED) 
   {
      if ((rc = dsmEndGetObj(handle)) != DSM_RC_OK) 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread2 error from dsmEndGetObj. rcMsg=%s\n", rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--;        
      }
   }
   else if (rc == DSM_RC_MORE_DATA) 
   {
      while ((rc = dsmGetData(handle, &getBlk)) == DSM_RC_MORE_DATA)
         ;
      if (rc == DSM_RC_FINISHED) 
      {
         if ((rc = dsmEndGetObj(handle)) != DSM_RC_OK) 
         {
            dsmRCMsg(handle,rc,rcMsg);
            printf("Thread2 error from dsmEndGetObj. rcMsg=%s\n", rcMsg);
            free(rcMsg);
            dsmTerminate(handle);
            NumberOfThreads--; 
            return NULL;
         }
      }
      else  /* dsmGetData did not finish */
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread2 error from dsmGetData. rcMsg=%s\n", rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--; 
         return NULL;
      }
   }
   else  /* dsmGetObj did not finish */
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error from dsmGetObj. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   if ((rc = dsmEndGetData(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread2 error from dsmEndGetData. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   printf("Thread2 bytes restored >%d<\n >%s<\n",
      getBlk.numBytes, (char *)getBlk.bufferPtr);

   dsmTerminate(handle);

   free(regFilespace.fsName);
   free(regFilespace.fsType);
   free(objAttrArea.objInfo);
   free(dsmObjList.objId);
   free(getBlk.bufferPtr);
   free(dataBlkArea.bufferPtr);
   free(queryBuffer.objName);
   free(queryBuffer.owner);
   free(rcMsg);

   printf("Thread 2 called dsmTerminate\n");
   printf("Number of threads = >%d< \n",NumberOfThreads);
   NumberOfThreads--;
   return NULL;
}

void *Thread3(void *string) 
{
   dsInt16_t          rc;
   int                i;
   char               options[100];
   char               configfile[10];
   mcBindKey          mcBindKey;
   ObjAttr            objAttrArea;
   DataBlk            dataBlkArea, qDataBlkArea, getBlk;
   qryBackupData      queryBuffer;
   qryRespBackupData  qbDataArea;
   dsUint16_t         reason;
   dsmObjName         objName;
   dsmQueryType       queryType;
   dsmGetList         dsmObjList;
   regFSData          regFilespace;
   dsmApiVersion      dsmApiVer;
   char               *rcMsg;
   dsUint32_t         handle = 0;
   optStruct dsmOpt1;

   dsmInitExIn_t      initIn;
   dsmInitExOut_t     initOut;
   dsmApiVersionEx    apiApplVer;

   memset(&initIn,     0x00, sizeof(dsmInitExIn_t));
   memset(&initOut,    0x00, sizeof(dsmInitExOut_t));
   memset(&apiApplVer, 0x00, sizeof(dsmApiVersionEx));

   printf("Thread3 started\n");

   dsmApiVer.version = DSM_API_VERSION;
   dsmApiVer.release = DSM_API_RELEASE;
   dsmApiVer.level   = DSM_API_LEVEL  ;

   apiApplVer.version  = DSM_API_VERSION;
   apiApplVer.release  = DSM_API_RELEASE;
   apiApplVer.level    = DSM_API_LEVEL  ;
   apiApplVer.subLevel = DSM_API_SUBLEVEL;

   /*********** Initialize the API session ***********************************/
   rcMsg = (char *)malloc(DSM_MAX_RC_MSG_LENGTH);

   /********** The following 2 can be updated optionally *********************/
   configfile[0]= '\0';       /* API config file for dsInit */
   /* options[0]= '\0';*/     /* options string */
   sprintf(options, "-se=%s",ALTSERVER);

   initIn.stVersion        = dsmInitExInVersion;
   initIn.apiVersionExP    = &apiApplVer;
   initIn.clientNodeNameP  = NULL;
   initIn.clientOwnerNameP = NULL;
   initIn.clientPasswordP  = NULL;
   initIn.applicationTypeP = "API Test3";
   initIn.configfile       = configfile;
   initIn.options          = options;
   initIn.userNameP        = NULL;
   initIn.userPasswordP    = NULL;

   if ((rc = dsmInitEx(&handle,&initIn,&initOut)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error in dsmInit. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   printf("Thread3 after dsmInit, session handle is %d\n",handle);

   rc = dsmQuerySessOptions(handle, &dsmOpt1); 

   if (rc) 
      printf("*** Thread3 Error on issueing Query Session Options.  Rc = %d\n",rc);
   else 
   {
      printf("\n");
      printf("DSMI_DIR               >%s<\n", dsmOpt1.dsmiDir);
      printf("DSMI_CONFIG            >%s<\n", dsmOpt1.dsmiConfig);
      printf("serverName             >%s<\n", dsmOpt1.serverName);
      printf("commMethod              %d\n",  dsmOpt1.commMethod);
      printf("serverAddress          >%s<\n", dsmOpt1.serverAddress);
      printf("nodeName               >%s<\n", dsmOpt1.nodeName);
      printf("compress                %d\n",   dsmOpt1.compression);
      printf("compressalways          %d\n",   dsmOpt1.compressalways);
      printf("passwordAccess          %d\n",   dsmOpt1.passwordAccess);
   }

   /************ Build File Space ****************************************/
   regFilespace.fsName = (char *)malloc(100);
   regFilespace.fsType = (char *)malloc(100);

   strcpy(regFilespace.fsName,"/t3fs");
   strcpy(regFilespace.fsType,"CALLER3_FS");
   regFilespace.capacity.lo = 0;
   regFilespace.capacity.hi = 0;
   regFilespace.occupancy.lo = 0;
   regFilespace.occupancy.hi = 0;
   regFilespace.stVersion = regFSDataVersion;
   regFilespace.fsAttr.unixFSAttr.fsInfoLength = 6;
   strcpy(regFilespace.fsAttr.unixFSAttr.fsInfo,"fsinfo");

   /*********** Register the file space **********************************/
   rc = dsmRegisterFS(handle, &regFilespace);

   if ((rc != DSM_RC_OK) && (rc != DSM_RC_FS_ALREADY_REGED)) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error in dsmRegisterFS. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;  
      return NULL;
   }

   if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error in dsmBeginTxn. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   mcBindKey.stVersion = mcBindKeyVersion;

   strcpy(objName.fs,"/t3fs");
   strcpy(objName.hl,"/testhl");
   strcpy(objName.ll,"/testll");
   objName.objType = DSM_OBJ_FILE;

   if ((rc = dsmBindMC(handle, &objName, stBackup, &mcBindKey)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error in dsmBindMC. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;  
      return NULL;
   }

   strcpy(objAttrArea.owner,"");
   objAttrArea.sizeEstimate.hi = 0;       /* size estimate of object */
   objAttrArea.sizeEstimate.lo = 460;       /* size estimate of object */
   objAttrArea.objCompressed = bFalse;
   objAttrArea.objInfoLength = 10;          /* length of objInfo */
   objAttrArea.objInfo = (char *)malloc(objAttrArea.objInfoLength + 1);
   strcpy(objAttrArea.objInfo,"abcdefghij");      /* object-dependent info */
   objAttrArea.stVersion = ObjAttrVersion;
   objAttrArea.mcNameP = NULL;

   /***************** Backup the data  *****************************/
   if ((rc = dsmSendObj(handle, stBackup, NULL, &objName, &objAttrArea,
        NULL)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread 3 error in dsmBackup. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   dataBlkArea.bufferPtr = (char *)malloc(20);
   dataBlkArea.stVersion = DataBlkVersion;
   dataBlkArea.bufferLen = 20;              
   memcpy(dataBlkArea.bufferPtr,"T3T3T3T3T3T3T3T3T3T3", 20); 

   for (i=1; i<=23; i++) 
   {
      if ((rc = dsmSendData(handle,&dataBlkArea)) != DSM_RC_OK) 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread3 error in dsmSendData. rcMsg=%s\n",rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--; 
         return NULL;
      }

      if(i == 22)
         memcpy(dataBlkArea.bufferPtr,"T3T3T3T3T3T3T3T3T3T\0", 20);
   }

   dsmEndSendObj(handle);

   if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error in dsmEndTxn. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   queryType = qtBackup;

   qbDataArea.stVersion = qryRespBackupDataVersion;
   qDataBlkArea.stVersion = DataBlkVersion;
   qDataBlkArea.bufferPtr = (char *)&qbDataArea;
   qDataBlkArea.bufferLen = sizeof(qryRespBackupData);

   queryBuffer.objName = (dsmObjName *)malloc(sizeof(dsmObjName));
   queryBuffer.owner = (char *)malloc(100);

   strcpy(queryBuffer.objName->fs, objName.fs);
   strcpy(queryBuffer.objName->hl, objName.hl);
   strcpy(queryBuffer.objName->ll, objName.ll);
   queryBuffer.objName->objType = objName.objType;
   strcpy(queryBuffer.owner, "");
   queryBuffer.objState = DSM_ACTIVE;
   queryBuffer.stVersion = qryBackupDataVersion;

   if ((rc=dsmBeginQuery(handle,queryType, (void *)&queryBuffer )) 
        != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error in dsmBeginQuery. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
      ;

   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA)) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error from dsmGetNextQObj. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error from dsmEndQuery. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   printf("\n*** Thread3 query response ***\n*** %s%s%s *** \n",
      qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll);

   dsmObjList.stVersion = dsmGetListVersion;
   dsmObjList.numObjId = 1;
   dsmObjList.objId = (ObjID *)malloc(sizeof(ObjID) * dsmObjList.numObjId);

   dsmObjList.objId[0].hi = qbDataArea.objId.hi;
   dsmObjList.objId[0].lo = qbDataArea.objId.lo;

   if ((rc = dsmBeginGetData(handle,bTrue,gtBackup,&dsmObjList)) 
        != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error from dsmBeginGetData. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   getBlk.bufferPtr = (char *)malloc(500);
   memset(getBlk.bufferPtr,0x00,sizeof(getBlk.bufferPtr));
   getBlk.bufferLen = 500;
   getBlk.stVersion = DataBlkVersion;

   rc = dsmGetObj(handle, &(dsmObjList.objId[0]), &getBlk);

   if (rc == DSM_RC_FINISHED) 
   {
      if ((rc = dsmEndGetObj(handle)) != DSM_RC_OK) 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread3 error from dsmEndGetObj. rcMsg=%s\n", rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--;
         return NULL;
      }
   }
   else if (rc == DSM_RC_MORE_DATA) 
   {
      while ((rc = dsmGetData(handle, &getBlk)) == DSM_RC_MORE_DATA)
         ;
      if (rc == DSM_RC_FINISHED) 
      {
         if ((rc = dsmEndGetObj(handle)) != DSM_RC_OK) 
	 {
            dsmRCMsg(handle,rc,rcMsg);
            printf("Thread3 error from dsmEndGetObj. rcMsg=%s\n", rcMsg);
            free(rcMsg);
            dsmTerminate(handle);
            NumberOfThreads--;
      	    return NULL;
         }
      }
      else  /* dsmGetData did not finish */
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread3 error from dsmGetData. rcMsg=%s\n", rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--;
	 return NULL;
      }
   }
   else  /* dsmGetObj did not finish */
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error from dsmGetObj. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   if ((rc = dsmEndGetData(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread3 error from dsmEndGetData. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   printf("Thread3 bytes restored >%d<\n >%s<\n",getBlk.numBytes, 
      (char *)getBlk.bufferPtr);

   dsmTerminate(handle);

   free(regFilespace.fsName);
   free(regFilespace.fsType);
   free(objAttrArea.objInfo);
   free(dsmObjList.objId);
   free(getBlk.bufferPtr);
   free(dataBlkArea.bufferPtr);
   free(queryBuffer.objName);
   free(queryBuffer.owner);
   free(rcMsg);
      
   printf("Thread 3 called dsmTerminate\n");
   printf("Number of threads = >%d< \n",NumberOfThreads);
   NumberOfThreads--; 
   return NULL;
}

void *Thread4(void *string) 
{
   dsInt16_t          rc;
   int                i;
   char               options[100];
   char               configfile[10];
   mcBindKey          mcBindKey;
   ObjAttr            objAttrArea;
   DataBlk            dataBlkArea, qDataBlkArea, getBlk;
   qryArchiveData     queryBuffer;
   qryRespArchiveData qbDataArea;
   sndArchiveData     archData;
   dsmObjName         objName;
   dsUint16_t         reason;
   dsmQueryType       queryType;
   dsmGetList         dsmObjList;
   regFSData          regFilespace;
   dsmApiVersion      dsmApiVer;
   char               *rcMsg;
   dsUint32_t         handle = 0;
   optStruct          dsmOpt1;
   char desc[2] ={'*','\0'};

   dsmInitExIn_t      initIn;
   dsmInitExOut_t     initOut;
   dsmApiVersionEx    apiApplVer;

   memset(&initIn,     0x00, sizeof(dsmInitExIn_t));
   memset(&initOut,    0x00, sizeof(dsmInitExOut_t));
   memset(&apiApplVer, 0x00, sizeof(dsmApiVersionEx));

   printf("Thread4 started\n");

   dsmApiVer.version = DSM_API_VERSION;
   dsmApiVer.release = DSM_API_RELEASE;
   dsmApiVer.level   = DSM_API_LEVEL  ;

   apiApplVer.version  = DSM_API_VERSION;
   apiApplVer.release  = DSM_API_RELEASE;
   apiApplVer.level    = DSM_API_LEVEL  ;
   apiApplVer.subLevel = DSM_API_SUBLEVEL;

   /************ Initialize the API session **********************************/
   rcMsg = (char *)malloc(DSM_MAX_RC_MSG_LENGTH);

   /*************** The following 2 can be updated optionally ****************/
   configfile[0]= '\0';
   /* options[0]= '\0'; */
   sprintf(options, "-se=%s",ALTSERVER);

   initIn.stVersion        = dsmInitExInVersion;
   initIn.apiVersionExP    = &apiApplVer;
   initIn.clientNodeNameP  = NULL;
   initIn.clientOwnerNameP = NULL;
   initIn.clientPasswordP  = NULL;
   initIn.applicationTypeP = "API Test4";
   initIn.configfile       = configfile;
   initIn.options          = options;
   initIn.userNameP        = NULL;
   initIn.userPasswordP    = NULL;

   if ((rc = dsmInitEx(&handle,&initIn,&initOut)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error in dsmInit. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   printf("Thread4 after dsmInit, session handle is %d\n",handle);

   rc = dsmQuerySessOptions(handle, &dsmOpt1);

   if (rc) 
      printf("*** Thread4 Error on issueing Query Session Options.  Rc = %d\n",rc);
   else 
   {
      printf("\n");
      printf("DSMI_DIR               >%s<\n", dsmOpt1.dsmiDir);
      printf("DSMI_CONFIG            >%s<\n", dsmOpt1.dsmiConfig);
      printf("serverName             >%s<\n", dsmOpt1.serverName);
      printf("commMethod              %d\n",  dsmOpt1.commMethod);
      printf("serverAddress          >%s<\n", dsmOpt1.serverAddress);
      printf("nodeName               >%s<\n", dsmOpt1.nodeName);
      printf("compress                %d\n",   dsmOpt1.compression);
      printf("compressalways          %d\n",   dsmOpt1.compressalways);
      printf("passwordAccess          %d\n",   dsmOpt1.passwordAccess);
   }

   /************ Build File Space ****************************************/
   regFilespace.fsName = (char *)malloc(100);
   regFilespace.fsType = (char *)malloc(100);

   strcpy(regFilespace.fsName,"/t4fs");
   strcpy(regFilespace.fsType,"CALLER4_FS");
   regFilespace.capacity.lo = 0;
   regFilespace.capacity.hi = 0;
   regFilespace.occupancy.lo = 0;
   regFilespace.occupancy.hi = 0;
   regFilespace.stVersion = regFSDataVersion;
   regFilespace.fsAttr.unixFSAttr.fsInfoLength = 6;
   strcpy(regFilespace.fsAttr.unixFSAttr.fsInfo,"fsinfo");

   /************ Register the file space ********************************/
   rc = dsmRegisterFS(handle, &regFilespace);

   if ((rc != DSM_RC_OK) && (rc != DSM_RC_FS_ALREADY_REGED)) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error in dsmRegisterFS. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
	  return NULL;
   }

   if ((rc = dsmBeginTxn(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error in dsmBeginTxn. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   mcBindKey.stVersion = mcBindKeyVersion;

   strcpy(objName.fs,"/t4fs");
   strcpy(objName.hl,"/testhl");
   strcpy(objName.ll,"/testll");
   objName.objType = DSM_OBJ_FILE;

   if ((rc = dsmBindMC(handle, &objName, stArchive, &mcBindKey)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error in dsmBindMC. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   archData.stVersion = sndArchiveDataVersion;
   archData.descr = desc;

   strcpy(objAttrArea.owner,"");
   objAttrArea.sizeEstimate.hi = 0;         /* size estimate of object */
   objAttrArea.sizeEstimate.lo = 460;       /* size estimate of object */
   objAttrArea.objCompressed = bFalse;
   objAttrArea.objInfoLength = 10;          /* length of objInfo */
   objAttrArea.objInfo = (char *)malloc(objAttrArea.objInfoLength + 1);
   strcpy(objAttrArea.objInfo,"abcdefghij");      /* object-dependent info */
   objAttrArea.stVersion = ObjAttrVersion;
   objAttrArea.mcNameP = NULL;

   /***************** Backup the data  *************************************/
   if ((rc = dsmSendObj(handle, stArchive, &archData, &objName, &objAttrArea,
        NULL)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread 4 error in dsmBackup. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   dataBlkArea.bufferPtr = (char *)malloc(20);
   dataBlkArea.stVersion = DataBlkVersion;
   dataBlkArea.bufferLen = 20;
   memcpy(dataBlkArea.bufferPtr,"T4T4T4T4T4T4T4T4T4T4", 20); 

   for (i=1; i<=24; i++) 
   {
      if ((rc = dsmSendData(handle,&dataBlkArea)) != DSM_RC_OK) 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread4 error in dsmSendData. rcMsg=%s\n",rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--;
         return NULL;
      }

      if(i == 23)
         memcpy(dataBlkArea.bufferPtr,"T4T4T4T4T4T4T4T4T4T\0", 20);
   }

   dsmEndSendObj(handle);

   if ((rc = dsmEndTxn(handle,DSM_VOTE_COMMIT, &reason)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error in dsmEndTxn. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   queryType = qtArchive;

   qbDataArea.stVersion = qryRespArchiveDataVersion;
   qDataBlkArea.stVersion = DataBlkVersion;
   qDataBlkArea.bufferPtr = (char *)&qbDataArea;
   qDataBlkArea.bufferLen = sizeof(qryRespArchiveData); 

   queryBuffer.objName = (dsmObjName *)malloc(sizeof(dsmObjName));
   queryBuffer.owner = (char *)malloc(100);

   strcpy(queryBuffer.objName->fs, objName.fs);
   strcpy(queryBuffer.objName->hl, objName.hl);
   strcpy(queryBuffer.objName->ll, objName.ll);
   queryBuffer.objName->objType = objName.objType;
   strcpy(queryBuffer.owner, "");
   queryBuffer.stVersion = qryArchiveDataVersion;
   queryBuffer.descr = desc;

   queryBuffer.insDateLowerBound.year = DATE_MINUS_INFINITE;
   queryBuffer.insDateUpperBound.year = DATE_PLUS_INFINITE;
   queryBuffer.expDateLowerBound.year = DATE_MINUS_INFINITE;
   queryBuffer.expDateUpperBound.year = DATE_PLUS_INFINITE;

   if ((rc=dsmBeginQuery(handle,queryType, (void *)&queryBuffer )) 
        != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error in dsmBeginQuery. rcMsg=%s\n",rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--; 
      return NULL;
   }

   while ((rc = dsmGetNextQObj(handle, &qDataBlkArea)) == DSM_RC_MORE_DATA)
      ;

   if ((rc != DSM_RC_FINISHED) && (rc != DSM_RC_MORE_DATA)) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error from dsmGetNextQObj. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   if ((rc = dsmEndQuery(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error from dsmEndQuery. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   printf("\n*** Thread4 query response ***\n*** %s%s%s ***\n",
      qbDataArea.objName.fs, qbDataArea.objName.hl, qbDataArea.objName.ll);

   dsmObjList.stVersion = dsmGetListVersion;
   dsmObjList.numObjId = 1;
   dsmObjList.objId = (ObjID *)malloc(sizeof(ObjID) * dsmObjList.numObjId);

   dsmObjList.objId[0].hi = qbDataArea.objId.hi;
   dsmObjList.objId[0].lo = qbDataArea.objId.lo;

   if ((rc = dsmBeginGetData(handle,bTrue,gtArchive,&dsmObjList)) 
        != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error from dsmBeginGetData. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   getBlk.bufferPtr = (char *)malloc(500);
   memset(getBlk.bufferPtr,0x00,sizeof(getBlk.bufferPtr));
   getBlk.bufferLen = 500;
   getBlk.stVersion = DataBlkVersion;

   rc = dsmGetObj(handle, &(dsmObjList.objId[0]), &getBlk);

   if (rc == DSM_RC_FINISHED) 
   {
      if ((rc = dsmEndGetObj(handle)) != DSM_RC_OK) 
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread4 error from dsmEndGetObj. rcMsg=%s\n", rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--;
         return NULL;
      }
   }
   else if (rc == DSM_RC_MORE_DATA) 
   {
      while ((rc = dsmGetData(handle, &getBlk)) == DSM_RC_MORE_DATA)
         ;
      if (rc == DSM_RC_FINISHED) 
      {
         if ((rc = dsmEndGetObj(handle)) != DSM_RC_OK) 
		 {
            dsmRCMsg(handle,rc,rcMsg);
            printf("Thread4 error from dsmEndGetObj. rcMsg=%s\n", rcMsg);
            free(rcMsg);
            dsmTerminate(handle);
            NumberOfThreads--;
            return NULL;
         }
      }
      else  /* dsmGetData did not finish */
      {
         dsmRCMsg(handle,rc,rcMsg);
         printf("Thread4 error from dsmGetData. rcMsg=%s\n", rcMsg);
         free(rcMsg);
         dsmTerminate(handle);
         NumberOfThreads--; 
         return NULL;
      }
   }
   else  /* dsmGetObj did not finish */
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error from dsmGetObj. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   if ((rc = dsmEndGetData(handle)) != DSM_RC_OK) 
   {
      dsmRCMsg(handle,rc,rcMsg);
      printf("Thread4 error from dsmEndGetData. rcMsg=%s\n", rcMsg);
      free(rcMsg);
      dsmTerminate(handle);
      NumberOfThreads--;
      return NULL;
   }

   printf("Thread4 bytes restored >%d<\n >%s<\n",getBlk.numBytes, 
      (char *)getBlk.bufferPtr);

   dsmTerminate(handle);

   free(regFilespace.fsName);
   free(regFilespace.fsType);
   free(objAttrArea.objInfo);
   free(dsmObjList.objId);
   free(getBlk.bufferPtr);
   free(dataBlkArea.bufferPtr);
   free(queryBuffer.objName);
   free(queryBuffer.owner);
   free(rcMsg);
      
   printf("Thread 4 called dsmTerminate\n");
   printf("Number of threads = >%d< \n",NumberOfThreads);
   NumberOfThreads--; 
   return NULL;
}
