/*
 * This file is part of roccat-tools.
 *
 * roccat-tools is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Macro Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * roccat-tools is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Macro Public License for more details.
 *
 * You should have received a copy of the GNU Macro Public License
 * along with roccat-tools. If not, see <http://www.gnu.org/licenses/>.
 */

#include "ryos_keys_easyzone.h"
#include "roccat_helper.h"
#include "ryos_device.h"
#include <string.h>

static guint8 const default_keys_easyzone[sizeof(RyosKeysEasyzone)] = {
	0x0b, 0x26, 0x01, 0x00, 0x35, 0x00, 0x00, 0x1e, 0x00, 0x00, 0x1f, 0x00, 0x00, 0x20, 0x00, 0x00,
	0x21, 0x00, 0x00, 0x22, 0x00, 0x00, 0x23, 0x00, 0x00, 0x24, 0x00, 0x00, 0x25, 0x00, 0x00, 0x26,
	0x00, 0x00, 0x27, 0x00, 0x00, 0x2d, 0x00, 0x00, 0x2e, 0x00, 0x00, 0x89, 0x00, 0x00, 0x2a, 0x00,
	0x00, 0x2b, 0x00, 0x00, 0x14, 0x00, 0x00, 0x1a, 0x00, 0x00, 0x08, 0x00, 0x00, 0x15, 0x00, 0x00,
	0x17, 0x00, 0x00, 0x1c, 0x00, 0x00, 0x18, 0x00, 0x00, 0x0c, 0x00, 0x00, 0x12, 0x00, 0x00, 0x13,
	0x00, 0x00, 0x2f, 0x00, 0x00, 0x30, 0x00, 0x00, 0x31, 0x00, 0x00, 0x04, 0x00, 0x00, 0x16, 0x00,
	0x00, 0x07, 0x00, 0x00, 0x09, 0x00, 0x00, 0x0a, 0x00, 0x00, 0x0b, 0x00, 0x00, 0x0d, 0x00, 0x00,
	0x0e, 0x00, 0x00, 0x0f, 0x00, 0x00, 0x33, 0x00, 0x00, 0x34, 0x00, 0x00, 0x32, 0x00, 0x00, 0x28,
	0x00, 0x00, 0xe1, 0x00, 0x00, 0x64, 0x00, 0x00, 0x1d, 0x00, 0x00, 0x1b, 0x00, 0x00, 0x06, 0x00,
	0x00, 0x19, 0x00, 0x00, 0x05, 0x00, 0x00, 0x11, 0x00, 0x00, 0x10, 0x00, 0x00, 0x36, 0x00, 0x00,
	0x37, 0x00, 0x00, 0x38, 0x00, 0x00, 0x87, 0x00, 0x00, 0xe5, 0x00, 0x00, 0xe0, 0x00, 0x00, 0xe3,
	0x00, 0x00, 0xe2, 0x00, 0x00, 0x8b, 0x00, 0x00, 0x91, 0x00, 0x00, 0x2c, 0x00, 0x00, 0x90, 0x00,
	0x00, 0x8a, 0x00, 0x00, 0x88, 0x00, 0x00, 0xe6, 0x00, 0x00, 0x65, 0x00, 0x00, 0xe4, 0x00, 0x00,
	0x49, 0x00, 0x00, 0x4a, 0x00, 0x00, 0x4b, 0x00, 0x00, 0x4c, 0x00, 0x00, 0x4d, 0x00, 0x00, 0x4e,
	0x00, 0x00, 0x52, 0x00, 0x00, 0x50, 0x00, 0x00, 0x51, 0x00, 0x00, 0x4f, 0x00, 0x00, 0x53, 0x00,
	0x00, 0x54, 0x00, 0x00, 0x55, 0x00, 0x00, 0x56, 0x00, 0x00, 0x5f, 0x00, 0x00, 0x60, 0x00, 0x00,
	0x61, 0x00, 0x00, 0x57, 0x00, 0x00, 0x5c, 0x00, 0x00, 0x5d, 0x00, 0x00, 0x5e, 0x00, 0x00, 0x85,
	0x00, 0x00, 0x59, 0x00, 0x00, 0x5a, 0x00, 0x00, 0x5b, 0x00, 0x00, 0x58, 0x00, 0x00, 0x62, 0x00,
	0x00, 0x63, 0x00, 0x00, 0xa2, 0x1a,
};

RyosKeysEasyzone const *ryos_keys_easyzone_default(void) {
	return (RyosKeysEasyzone const *)default_keys_easyzone;
}

static guint16 ryos_keys_easyzone_calc_checksum(RyosKeysEasyzone const *keys_easyzone) {
	return ROCCAT_BYTESUM_PARTIALLY(keys_easyzone, RyosKeysEasyzone, report_id, checksum);
}

static void ryos_keys_easyzone_set_checksum(RyosKeysEasyzone *keys_easyzone, guint16 new_value) {
	keys_easyzone->checksum = GUINT16_TO_LE(new_value);
}

static void ryos_keys_easyzone_finalize(RyosKeysEasyzone *keys_easyzone, guint profile_index) {
	keys_easyzone->report_id = RYOS_REPORT_ID_KEYS_EASYZONE;
	keys_easyzone->size = sizeof(RyosKeysEasyzone);
	keys_easyzone->profile_index = profile_index;
	ryos_keys_easyzone_set_checksum(keys_easyzone, ryos_keys_easyzone_calc_checksum(keys_easyzone));
}

gboolean ryos_keys_easyzone_write(RoccatDevice *ryos, guint profile_index, RyosKeysEasyzone *keys_easyzone, GError **error) {
	g_assert(profile_index < RYOS_PROFILE_NUM);
	ryos_keys_easyzone_finalize(keys_easyzone, profile_index);
	return ryos_device_write(ryos, (gchar const *)keys_easyzone, sizeof(RyosKeysEasyzone), error);
}

RyosKeysEasyzone *ryos_keys_easyzone_read(RoccatDevice *ryos, guint profile_index, GError **error) {
	RyosKeysEasyzone *keys_easyzone;

	g_assert(profile_index < RYOS_PROFILE_NUM);

	gaminggear_device_lock(GAMINGGEAR_DEVICE(ryos));

	if (!ryos_select(ryos, profile_index, RYOS_CONTROL_REQUEST_KEYS_EASYZONE, error)) {
		gaminggear_device_unlock(GAMINGGEAR_DEVICE(ryos));
		return NULL;
	}

	keys_easyzone = (RyosKeysEasyzone *)ryos_device_read(ryos, RYOS_REPORT_ID_KEYS_EASYZONE, sizeof(RyosKeysEasyzone), error);

	gaminggear_device_unlock(GAMINGGEAR_DEVICE(ryos));

	return keys_easyzone;
}

gboolean ryos_keys_easyzone_equal(RyosKeysEasyzone const *left, RyosKeysEasyzone const *right) {
	gboolean equal;
	equal = ROCCAT_MEMCMP_PARTIALLY(left, right, RyosKeysEasyzone, keys[0], checksum);
	return equal ? FALSE : TRUE;
}

void ryos_keys_easyzone_copy(RyosKeysEasyzone *destination, RyosKeysEasyzone const *source) {
	memcpy(destination, source, sizeof(RyosKeysEasyzone));
}

guint ryos_keys_easyzone_index_to_macro_index(guint easyzone_index) {
	return easyzone_index + 0x10;
}

guint ryos_macro_index_to_keys_easyzone_index(guint macro_index) {
	return macro_index - 0x10;
}

gboolean ryos_macro_index_is_keys_easyzone(guint macro_index) {
	return macro_index >= 0x10 && macro_index <= 0x6f;
}
