// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

// +build !oss

package machine

import (
	"bytes"
	"encoding/json"
	"io"
	"io/ioutil"
	"strings"
)

// Config provides the Docker machine configuration.
type Config struct {
	Name   string
	Driver struct {
		IPAddress   string
		MachineName string
	}
	HostOptions struct {
		EngineOptions struct {
			TLSVerify bool `json:"TlsVerify"`
		}
		AuthOptions struct {
			CertDir          string
			CaCertPath       string
			CaPrivateKeyPath string
			ServerCertPath   string
			ServerKeyPath    string
			ClientKeyPath    string
			ClientCertPath   string
			StorePath        string
		}
	}
}

// helper function reads and unmarshalls the docker-machine
// configuration from a reader.
func parseReader(r io.Reader) (*Config, error) {
	out := new(Config)
	err := json.NewDecoder(r).Decode(out)
	return out, err
}

// heper function parses the docker-machine configuration
// from a json string.
func parseString(s string) (*Config, error) {
	r := strings.NewReader(s)
	return parseReader(r)
}

// heper function parses the docker-machine configuration
// from a json file.
func parseFile(path string) (*Config, error) {
	d, err := ioutil.ReadFile(path)
	if err != nil {
		return nil, err
	}
	r := bytes.NewReader(d)
	return parseReader(r)
}
