/***************************************************************************
 *                                                                         *
 *   brent.cpp   (begin: Feb 20 2003)                                      *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   Copyright (C) 2003-2004 by Le Sy Vinh, Arndt von Haeseler             *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   Some source code here taken from TREE-PUZZLE package               
 * (c) 2003-2004 by Heiko A. Schmidt, Korbinian Strimmer, Arndt von Haeseler
 * (c) 1999-2003 by Heiko A. Schmidt, Korbinian Strimmer,
 *                  M. Vingron, and Arndt von Haeseler
 * (c) 1995-1999 by Korbinian Strimmer and Arndt von Haeseler
 
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <iostream>
#include <stdlib.h>
#include <math.h>
#include "constant.h"
#include "brent.h"
#include "rannum.h"
//#include "math.h"

using namespace std;
//int Brent::isOptedPam_ = 0;

//constructor function
Brent::Brent() {
	isOptedPam_ = 0;
	for (int cnt = 1; cnt < MAX_DIMENSION; cnt++) {
		direction_set[cnt] = new double[MAX_DIMENSION];
		memset(direction_set[cnt], 0, sizeof(double) * MAX_DIMENSION);
		direction_set[cnt][cnt] = 1.0;
	}
}

int Brent::isOptedPam () {
	return isOptedPam_;
}

void Brent::turnOnOptedPam () {
	isOptedPam_ = 1;
}

void Brent::turnOffOptedPam () {
	isOptedPam_ = 0;
}

/******************************************************************************/
/* minimization of a function by Brents method (Numerical Recipes)            */
/******************************************************************************/


//-------------------------------------------------------------------------------------------------------
#define ITMAX 100
#define CGOLD 0.3819660
#define GOLD 1.618034
#define GLIMIT 100.0
#define TINY 1.0e-20
#define ZEPS 1.0e-10
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);
#define SIGN(a,b) ((b) >= 0.0 ? fabs(a) : -fabs(a))

/* Brents method in one dimension */
inline double Brent::opt (double ax, double bx, double cx, double tol,
                          double *foptx, double *f2optx, double fax, double fbx, double fcx) {
	int iter;
	double a,b,d=0,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm;
	double xw,wv,vx;
	double e=0.0;

	a=(ax < cx ? ax : cx);
	b=(ax > cx ? ax : cx);
	x=bx;
	fx=fbx;
	if (fax < fcx) {
		w=ax;
		fw=fax;
		v=cx;
		fv=fcx;
	} else {
		w=cx;
		fw=fcx;
		v=ax;
		fv=fax;
	}

	for (iter=1;iter<=ITMAX;iter++) {
		xm=0.5*(a+b);
		tol2=2.0*(tol1=tol*fabs(x)+ZEPS);
		if (fabs(x-xm) <= (tol2-0.5*(b-a))) {
			*foptx = fx;
			xw = x-w;
			wv = w-v;
			vx = v-x;
			*f2optx = 2.0*(fv*xw + fx*wv + fw*vx)/
			          (v*v*xw + x*x*wv + w*w*vx);
			return x;
		}

		if (fabs(e) > tol1) {
			r=(x-w)*(fx-fv);
			q=(x-v)*(fx-fw);
			p=(x-v)*q-(x-w)*r;
			q=2.0*(q-r);
			if (q > 0.0)
				p = -p;
			q=fabs(q);
			etemp=e;
			e=d;
			if (fabs(p) >= fabs(0.5*q*etemp) || p <= q*(a-x) || p >= q*(b-x))
				d=CGOLD*(e=(x >= xm ? a-x : b-x));
			else {
				d=p/q;
				u=x+d;
				if (u-a < tol2 || b-u < tol2)
					d=SIGN(tol1,xm-x);
			}
		} else {
			d=CGOLD*(e=(x >= xm ? a-x : b-x));
		}

		u=(fabs(d) >= tol1 ? x+d : x+SIGN(tol1,d));
		fu=cmpNegLogLi(u);
		if (fu <= fx) {
			if (u >= x)
				a=x;
			else
				b=x;

			SHFT(v,w,x,u)
			SHFT(fv,fw,fx,fu)
		} else {
			if (u < x)
				a=u;
			else
				b=u;
			if (fu <= fw || w == x) {
				v=w;
				w=u;
				fv=fw;
				fw=fu;
			} else
				if (fu <= fv || v == x || v == w) {
					v=u;
					fv=fu;
				}
		}
	}

	*foptx = fx;
	xw = x-w;
	wv = w-v;
	vx = v-x;
	*f2optx = 2.0*(fv*xw + fx*wv + fw*vx)/(v*v*xw + x*x*wv + w*w*vx);

	return x;
}

#undef ITMAX
#undef CGOLD
#undef ZEPS
#undef SHFT
#undef SIGN
#undef GOLD
#undef GLIMIT
#undef TINY


//-------------------------------------------------------------------------------------------------------
/* one-dimensional minimization - as input a lower and an upper limit and a trial
value for the minimum is needed: xmin < xguess < xmax
the function and a fractional tolerance has to be specified
onedimenmin returns the optimal x value and the value of the function
and its second derivative at this point
*/

double Brent::optOneDim(double xmin, double xguess, double xmax,
                        double tol, double *fx, double *f2x) {
	double eps, optx, ax, bx, cx, fa, fb, fc;
	//int    converged;	/* not converged error flag */
		
	/* first attempt to bracketize minimum */
	eps = xguess*tol*50.0;
	ax = xguess - eps;
	if (ax < xmin) ax = xmin;
	bx = xguess;
	cx = xguess + eps;
	if (cx > xmax) cx = xmax;
	
	/* check if this works */
	fa = cmpNegLogLi(ax);
	fb = cmpNegLogLi(bx);
	fc = cmpNegLogLi(cx);

	/* if it works use these borders else be conservative */
	if ((fa < fb) || (fc < fb)) {
		if (ax != xmin) fa = cmpNegLogLi(xmin);
		if (cx != xmax) fc = cmpNegLogLi(xmax);
		optx = opt(xmin, xguess, xmax, tol, fx, f2x, fa, fb, fc);
	} else
		optx = opt(ax, bx, cx, tol, fx, f2x, fa, fb, fc);

	return optx; /* return optimal x */
}

double Brent::optOneDim_old(double xmin, double xguess, double xmax,
                        double tol, double *fx, double *f2x) {
	double eps, optx, ax, bx, cx, fa, fb, fc;

	/* first attempt to bracketize minimum */
	/*	
	if (isOptedPam_ == 1) {
		bx = xguess;
		fb = cmpNegLogLi(bx);

		eps = xguess * 0.5;
		ax = xguess - eps;
		if (ax < xmin)
			ax = xmin;
		fa = cmpNegLogLi(ax);
		if (fa < fb) {
			ax = xmin;
			fa = cmpNegLogLi(ax);
		}

		cx = xguess + eps;
		if (cx > xmax)
			cx = xmax;
		fc = cmpNegLogLi(cx);
		if (fc < fb) {
			cx = xmax;
			fc = cmpNegLogLi(cx);
		}
		double optx_ = opt(ax, bx, cx, tol, fx, f2x, fa, fb, fc);
		return optx_;
	} //end of isOptedPam_
	*/
	if (xguess < xmin) xguess = xmin;
	if (xguess > xmax) xguess = xmax;
	
	eps = tol;

	ax = xguess - eps;
	if (ax < xmin)
		ax = xmin;
	bx = xguess;
	cx = xguess + eps;
	if (cx > xmax)
		cx = xmax;
	/* check if this works */
	fa = cmpNegLogLi(ax);
	fb = cmpNegLogLi(bx);
	fc = cmpNegLogLi(cx);
	if (fa >= fb && fc >= fb)
		return bx;

	eps = xguess * 0.5;

	if (fa <= fb) {
		ax = xguess - eps;
		if (ax < xmin)
			ax = xmin;
		fa = cmpNegLogLi (ax);
	}

	if (fc <= fb) {
		cx = xguess + eps;
		if (cx > xmax)
			cx = xmax;
		fc = cmpNegLogLi (cx);
	}


	/* if it works use these borders else be conservative */
	if ((fa < fb) || (fc < fb)) {
		if (ax != xmin) {
			ax = xmin;
			fa = cmpNegLogLi(ax);
		}

		if (ax == xmin) {
			double fa1 = cmpNegLogLi(ax + 0.5 * tol);
			if ( fa1 >  fa) {
				double fa2 = cmpNegLogLi(ax + tol);
				if (fa2 > fa1)
					return xmin;
			}
		}

		if (cx != xmax)
			fc = cmpNegLogLi(xmax);
		optx = opt(xmin, xguess, xmax, tol, fx, f2x, fa, fb, fc);
	} else {
		optx = opt(ax, bx, cx, tol, fx, f2x, fa, fb, fc);
	}

	return optx; /* return optimal x */
}


#define ITMAX 200
static double sqrarg;
#define SQR(a) ((sqrarg=(a)) == 0.0 ? 0.0 : sqrarg*sqrarg)

void Brent::powell(double p[], double **xi, int n, double ftol, int *iter, double *fret) {
	int i,ibig,j;
	double del,fp,fptt,t,*pt,*ptt,*xit;

	pt=vector(1,n);
	ptt=vector(1,n);
	xit=vector(1,n);
	*fret=targetFunk(p);
	for (j=1;j<=n;j++) pt[j]=p[j];
	for (*iter=1;;++(*iter)) {
		fp=(*fret);
		ibig=0;
		del=0.0;
		for (i=1;i<=n;i++) {
			for (j=1;j<=n;j++) xit[j]=xi[j][i];
			fptt=(*fret);
			linmin(p,xit,n,fret);
			if (fabs(fptt-(*fret)) > del) {
				del=fabs(fptt-(*fret));
				ibig=i;
			}
		}
		if (2.0*fabs(fp-(*fret)) <= ftol*(fabs(fp)+fabs(*fret))) {
			free_vector(xit,1,n);
			free_vector(ptt,1,n);
			free_vector(pt,1,n);
			return;
		}
		if (*iter == ITMAX) nrerror("powell exceeding maximum iterations.");
		for (j=1;j<=n;j++) {
			ptt[j]=2.0*p[j]-pt[j];
			xit[j]=p[j]-pt[j];
			pt[j]=p[j];
		}
		fptt=targetFunk(ptt);
		if (fptt < fp) {
			t=2.0*(fp-2.0*(*fret)+fptt)*SQR(fp-(*fret)-del)-del*SQR(fp-fptt);
			if (t < 0.0) {
				linmin(p,xit,n,fret);
				for (j=1;j<=n;j++) {
					xi[j][ibig]=xi[j][n];
					xi[j][n]=xit[j];
				}
			}
		}
	}
}
//#undef SQR
#undef ITMAX


#define TOL 2.0e-4

int ncom;
double *pcom,*xicom;

void Brent::linmin(double p[], double xi[], int n, double *fret) {
	int j;
	double xx,xmin,fx,fb,fa,bx,ax;

	ncom=n;
	pcom=vector(1,n);
	xicom=vector(1,n);
	for (j=1;j<=n;j++) {
		pcom[j]=p[j];
		xicom[j]=xi[j];
	}
	ax=0.0;
	xx=1.0;
	mnbrak(&ax,&xx,&bx,&fa,&fx,&fb);
	*fret=brent(ax,xx,bx,TOL,&xmin);
	for (j=1;j<=n;j++) {
		xi[j] *= xmin;
		p[j] += xi[j];
	}
	free_vector(xicom,1,n);
	free_vector(pcom,1,n);
}
#undef TOL

double Brent::f1dim(double x) {
	int j;
	double f,*xt;

	xt=vector(1,ncom);
	for (j=1;j<=ncom;j++) xt[j]=pcom[j]+x*xicom[j];
	f=targetFunk(xt);
	free_vector(xt,1,ncom);
	return f;
}

#define GOLD 1.618034
#define GLIMIT 100.0
#define TINY 1.0e-20
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);
static double maxarg1,maxarg2;
#define FMAX(a,b) (maxarg1=(a),maxarg2=(b),(maxarg1) > (maxarg2) ?\
        (maxarg1) : (maxarg2))
#define SIGN(a,b) ((b) >= 0.0 ? fabs(a) : -fabs(a))

void Brent::mnbrak(double *ax, double *bx, double *cx, double *fa, double *fb, double *fc) {
	double ulim,u,r,q,fu,dum;

	*fa=f1dim(*ax);
	*fb=f1dim(*bx);
	if (*fb > *fa) {
		SHFT(dum,*ax,*bx,dum)
		SHFT(dum,*fb,*fa,dum)
	}
	*cx=(*bx)+GOLD*(*bx-*ax);
	*fc=f1dim(*cx);
	while (*fb > *fc) {
		r=(*bx-*ax)*(*fb-*fc);
		q=(*bx-*cx)*(*fb-*fa);
		u=(*bx)-((*bx-*cx)*q-(*bx-*ax)*r)/
		  (2.0*SIGN(FMAX(fabs(q-r),TINY),q-r));
		ulim=(*bx)+GLIMIT*(*cx-*bx);
		if ((*bx-u)*(u-*cx) > 0.0) {
			fu=f1dim(u);
			if (fu < *fc) {
				*ax=(*bx);
				*bx=u;
				*fa=(*fb);
				*fb=fu;
				return;
			} else if (fu > *fb) {
				*cx=u;
				*fc=fu;
				return;
			}
			u=(*cx)+GOLD*(*cx-*bx);
			fu=f1dim(u);
		} else if ((*cx-u)*(u-ulim) > 0.0) {
			fu=f1dim(u);
			if (fu < *fc) {
				SHFT(*bx,*cx,u,*cx+GOLD*(*cx-*bx))
				SHFT(*fb,*fc,fu,f1dim(u))
			}
		} else if ((u-ulim)*(ulim-*cx) >= 0.0) {
			u=ulim;
			fu=f1dim(u);
		} else {
			u=(*cx)+GOLD*(*cx-*bx);
			fu=f1dim(u);
		}
		SHFT(*ax,*bx,*cx,u)
		SHFT(*fa,*fb,*fc,fu)
	}
}
#undef GOLD
#undef GLIMIT
#undef TINY
#undef SHFT
#undef FMAX

#define ITMAX 100
#define CGOLD 0.3819660
#define ZEPS 1.0e-10
#define SHFT(a,b,c,d) (a)=(b);(b)=(c);(c)=(d);

double Brent::brent(double ax, double bx, double cx, double tol,
                    double *xmin) {
	int iter;
	double a,b,d=0,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm;
	double e=0.0;

	a=(ax < cx ? ax : cx);
	b=(ax > cx ? ax : cx);
	x=w=v=bx;
	fw=fv=fx=f1dim(x);
	for (iter=1;iter<=ITMAX;iter++) {
		xm=0.5*(a+b);
		tol2=2.0*(tol1=tol*fabs(x)+ZEPS);
		if (fabs(x-xm) <= (tol2-0.5*(b-a))) {
			*xmin=x;
			return fx;
		}
		if (fabs(e) > tol1) {
			r=(x-w)*(fx-fv);
			q=(x-v)*(fx-fw);
			p=(x-v)*q-(x-w)*r;
			q=2.0*(q-r);
			if (q > 0.0) p = -p;
			q=fabs(q);
			etemp=e;
			e=d;
			if (fabs(p) >= fabs(0.5*q*etemp) || p <= q*(a-x) || p >= q*(b-x))
				d=CGOLD*(e=(x >= xm ? a-x : b-x));
			else {
				d=p/q;
				u=x+d;
				if (u-a < tol2 || b-u < tol2)
					d=SIGN(tol1,xm-x);
			}
		} else {
			d=CGOLD*(e=(x >= xm ? a-x : b-x));
		}
		u=(fabs(d) >= tol1 ? x+d : x+SIGN(tol1,d));
		fu=f1dim(u);
		if (fu <= fx) {
			if (u >= x) a=x; else b=x;
			SHFT(v,w,x,u)
			SHFT(fv,fw,fx,fu)
		} else {
			if (u < x) a=u; else b=u;
			if (fu <= fw || w == x) {
				v=w;
				w=u;
				fv=fw;
				fw=fu;
			} else if (fu <= fv || v == x || v == w) {
				v=u;
				fv=fu;
			}
		}
	}
	nrerror("Too many iterations in brent");
	*xmin=x;
	return fx;
}
#undef ITMAX
#undef CGOLD
#undef ZEPS
#undef SHFT
#undef SIGN

void Brent::initOptMulti(int ndim) {
	int cnt;
	for (cnt = 1; cnt <= ndim; cnt++) {
		memset(direction_set[cnt], 0, sizeof(double) * (ndim+1));
		direction_set[cnt][cnt] = 1.0;
	}
}


double Brent::optMultiDim(double guess[], int ndim, double ftol) {
	// call the power method
	int iterations;
	double freturn;
	powell(guess, direction_set, ndim, ftol, &iterations, &freturn);
	return freturn;
}

#define ALF 1.0e-4
#define TOLX 1.0e-7
#define FMAX(a,b) (maxarg1=(a),maxarg2=(b),(maxarg1) > (maxarg2) ?\
        (maxarg1) : (maxarg2))

void Brent::lnsrch(int n, double xold[], double fold, double g[], double p[], double x[],
                   double *f, double stpmax, int *check, double lower[], double upper[]) {
	int i;
	double a,alam,alam2=0,alamin,b,disc,f2=0,fold2=0,rhs1,rhs2,slope,sum,temp,
	test,tmplam;

	*check=0;
	for (sum=0.0,i=1;i<=n;i++) sum += p[i]*p[i];
	sum=sqrt(sum);
	if (sum > stpmax)
		for (i=1;i<=n;i++) p[i] *= stpmax/sum;
	for (slope=0.0,i=1;i<=n;i++)
		slope += g[i]*p[i];
	test=0.0;
	for (i=1;i<=n;i++) {
		temp=fabs(p[i])/FMAX(fabs(xold[i]),1.0);
		if (temp > test) test=temp;
	}
	alamin=TOLX/test;
	alam=1.0;
	/*
	int rep = 0;
	do {
		for (i=1;i<=n;i++) x[i]=xold[i]+alam*p[i];
		if (!checkRange(x))
			alam *= 0.5;
		else
			break;
		rep++;
	} while (rep < 10);
	*/
	bool first_time = true;
	for (;;) {
		for (i=1;i<=n;i++) x[i]=xold[i]+alam*p[i];
		fixBound(x, lower, upper, n);
		checkRange(x);
		*f=targetFunk(x);
		if (alam < alamin) {
			for (i=1;i<=n;i++) x[i]=xold[i];
			*check=1;
			return;
		} else if (*f <= fold+ALF*alam*slope) return;
		else {
			if (first_time)
				tmplam = -slope/(2.0*(*f-fold-slope));
			else {
				rhs1 = *f-fold-alam*slope;
				rhs2=f2-fold2-alam2*slope;
				a=(rhs1/(alam*alam)-rhs2/(alam2*alam2))/(alam-alam2);
				b=(-alam2*rhs1/(alam*alam)+alam*rhs2/(alam2*alam2))/(alam-alam2);
				if (a == 0.0) tmplam = -slope/(2.0*b);
				else {
					disc=b*b-3.0*a*slope;
					if (disc<0.0) //nrerror("Roundoff problem in lnsrch.");
						tmplam = 0.5 * alam;
					else if (b <= 0.0) tmplam=(-b+sqrt(disc))/(3.0*a);
					else tmplam = -slope/(b+sqrt(disc));
				}
				if (tmplam>0.5*alam)
					tmplam=0.5*alam;
			}
		}
		alam2=alam;
		f2 = *f;
		fold2=fold;
		alam=FMAX(tmplam,0.1*alam);
		first_time = false;
	}
}
#undef ALF
#undef TOLX

/*
void checkBound(double x[], double dir[], double lower[], double upper[], int n) {
	for (int i = 1; i <= n; i++) {
		if (x[i] + dir[i] < lower[i])
			dir[i] = lower[i] - x[i];
		else if (x[i] + dir[i] > upper[i])
			dir[i] = upper[i] - x[i];

	}
}
*/
const int MAX_ITER = 3;

double Brent::optMultiDim(double guess[], int ndim, double lower[], double upper[], bool bound_check[], double gtol) {
	int i, iter;
	double fret, minf = INFINITIVE;
	double *minx = new double [ndim+1];
	int count = 0;
	bool restart;
	do {
		dfpmin(guess, ndim, lower, upper, gtol, &iter, &fret);
		if (fret < minf) {
 			minf = fret;
			for (i = 1; i <= ndim; i++)
				minx[i] = guess[i];
		}
		count++;
		// restart the search if at the boundary
		// it's likely to end at a local optimum at the boundary
		restart = false;
		
		
		for (i = 1; i <= ndim; i++)
			if (bound_check[i])
			if (fabs(guess[i]-lower[i]) < EPS_BOUND || fabs(guess[i]-upper[i]) < EPS_BOUND) {
				restart = true;
				break;
			}
		
		if (!restart)
			restart = atBoundary(guess);
		if (!restart)
			break;

		if (count == MAX_ITER)
			break;
			
		do {
			for (i = 1; i <= ndim; i++) {
				guess[i] = ((double)RanNum::getRanInt(100)/100.0) * (upper[i] - lower[i])/3 + lower[i];
			}
		} while (!checkRange(guess));
		if (isMasterProc()) {
			cout << "Restart estimation at the boundary... ";
			//for (i = 1; i <= ndim; i++)
				//cout << guess[i] << " ";
			cout << endl;
		}
	} while (count < MAX_ITER);
	if (count > 1) {
		for (i = 1; i <= ndim; i++)
			guess[i] = minx[i];
		fret = minf;
	}
	delete minx;
	
	return fret;
}


//#define ITMAX 200
#define ITMAX 500
#define EPS 3.0e-8
#define TOLX (4*EPS)
#define STPMX 100.0

#define FREEALL free_vector(xi,1,n);free_vector(pnew,1,n); \
free_matrix(hessin,1,n,1,n);free_vector(hdg,1,n);free_vector(g,1,n); \
free_vector(dg,1,n);



void Brent::dfpmin(double p[], int n, double lower[], double upper[], double gtol, int *iter, double *fret) {
	int check,i,its,j;
	double den,fac,fad,fae,fp,stpmax,sum=0.0,sumdg,sumxi,temp,test;
	double *dg,*g,*hdg,**hessin,*pnew,*xi;

	dg=vector(1,n);
	g=vector(1,n);
	hdg=vector(1,n);
	hessin=matrix(1,n,1,n);
	pnew=vector(1,n);
	xi=vector(1,n);
	fp = derivativeFunk(p,g);
	for (i=1;i<=n;i++) {
		for (j=1;j<=n;j++) hessin[i][j]=0.0;
		hessin[i][i]=1.0;
		xi[i] = -g[i];
		sum += p[i]*p[i];
	}
	//checkBound(p, xi, lower, upper, n);
	//checkDirection(p, xi);

	stpmax=STPMX*FMAX(sqrt(sum),(double)n);
	for (its=1;its<=ITMAX;its++) {
		*iter=its;
		lnsrch(n,p,fp,g,xi,pnew,fret,stpmax,&check, lower, upper);
		fp = *fret;
		for (i=1;i<=n;i++) {
			xi[i]=pnew[i]-p[i];
			p[i]=pnew[i];
		}
		test=0.0;
		for (i=1;i<=n;i++) {
			temp=fabs(xi[i])/FMAX(fabs(p[i]),1.0);
			if (temp > test) test=temp;
		}
		if (test < TOLX) {
			FREEALL
			return;
		}
		for (i=1;i<=n;i++) dg[i]=g[i];
		derivativeFunk(p,g);
		test=0.0;
		den=FMAX(*fret,1.0);
		for (i=1;i<=n;i++) {
			temp=fabs(g[i])*FMAX(fabs(p[i]),1.0)/den;
			if (temp > test) test=temp;
		}
		if (test < gtol) {
			FREEALL
			return;
		}
		for (i=1;i<=n;i++) dg[i]=g[i]-dg[i];
		for (i=1;i<=n;i++) {
			hdg[i]=0.0;
			for (j=1;j<=n;j++) hdg[i] += hessin[i][j]*dg[j];
		}
		fac=fae=sumdg=sumxi=0.0;
		for (i=1;i<=n;i++) {
			fac += dg[i]*xi[i];
			fae += dg[i]*hdg[i];
			sumdg += SQR(dg[i]);
			sumxi += SQR(xi[i]);
		}
		if (fac*fac > EPS*sumdg*sumxi) {
			fac=1.0/fac;
			fad=1.0/fae;
			for (i=1;i<=n;i++) dg[i]=fac*xi[i]-fad*hdg[i];
			for (i=1;i<=n;i++) {
				for (j=1;j<=n;j++) {
					hessin[i][j] += fac*xi[i]*xi[j]
					                -fad*hdg[i]*hdg[j]+fae*dg[i]*dg[j];
				}
			}
		}
		for (i=1;i<=n;i++) {
			xi[i]=0.0;
			for (j=1;j<=n;j++) xi[i] -= hessin[i][j]*g[j];
		}
		//checkBound(p, xi, lower, upper, n);
		//checkDirection(p, xi);
		//if (*iter > 200) cout << "iteration=" << *iter << endl;
	}
	nrerror("too many iterations in dfpmin");
	FREEALL
}
#undef ITMAX
#undef EPS
#undef TOLX
#undef STPMX
#undef FREEALL
#undef FMAX

//--------------------------------------------------------------------
//release the memory of this class
void Brent::release () {
	for (int cnt = MAX_DIMENSION - 1; cnt >= 1; cnt--)
		delete direction_set[cnt];
}

//-------------------------------------------------------------------------------------------------------
//the destructor function
Brent::~Brent() {
	release ();
	//std::cout << "the constructor of Brent class " << endl;
}
