/***************************************************************************
 *                                                                         *
 *                  (begin: Feb 15 2005)                                   *
 *                                                                         *
 *   Parallel IQPNNI - Important Quartet Puzzle with NNI                   *
 *                                                                         *
 *   Copyright (C) 2005 by Le Sy Vinh, Bui Quang Minh, Arndt von Haeseler  *
 *   {vinh,minh}@cs.uni-duesseldorf.de                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef URTREEPAR_H
#define URTREEPAR_H

#include "urtree.h"

/**
The unrooted tree class used for communication the tree among processes. <br>
Use MPI non-blocking communication to overlap communication and computation.<br>
@author <a href="mailto:minh@cs.uni-duesseldorf.de">Bui Quang Minh</a>
*/
class UrTreePar : public UrTree {
public:
	UrTreePar();

	/**
		copy from the existed tree
		@param tree the tree to be cloned
	*/
	UrTreePar(UrTree &tree);


#ifdef PARALLEL
	/**
		pack the information of the tree, preprare to send to the master
	*/
	void packData();

	/**
		return TRUE if the data for sending is already packed
		@return TRUE if the data for sending is already packed
	*/
	bool isDataPacked();

	/**
		send the packed data to a specific process
		@param process the process rank to send to
		@param nonblock_comm TRUE if you want to non-blocking send, FALSE otherwise
	*/
	void sendToProcess(int process, bool nonblock_comm = true);

	/**
		send the packed data to the master
		@param nonblock_comm TRUE if you want to non-blocking send, FALSE otherwise
	*/
	void sendToMaster(bool nonblock_comm = true);

	/**
		receive the packed data from a specific process
		@param process the process rank to receive from
		@param nonblock_comm TRUE if you want to non-blocking send, FALSE otherwise
	*/
	void receiveFromProcess(int process, bool nonblock_comm = true);

	/**
		receive the packed data from master
		@param nonblock_comm TRUE if you want to non-blocking send, FALSE otherwise
	*/
	void receiveFromMaster(bool nonblock_comm = true);


	/**
		return TRUE if master already send something, ready to unpack data
		@return TRUE if master already send something, ready to unpack data
	*/
	bool isDataArrived();

	/**
		return TRUE if master already sent something, but process may not received completely
		@return TRUE if master already sent something, but process may not received completely
	*/
	bool isReadyToReceive();
	
	/**
		unpack information from the packet to rebuild the tree
	*/
	void unpackData();

	/**
		return the send request
		@return the send request
	*/
	MPI_Request *getSendRequest();

	/**
		return the receive request
		@return the receive request
	*/
	MPI_Request *getReceiveRequest();

	/**
		cancel all the attached receive or send
	*/
	void cancelCommunications();

	/**
		cancel the send operation
	*/
	void cancelSend();

	/**
		cancel the receive operation
	*/
	void cancelReceive();

	/**
		wait for the completion of receive request
	*/
	void waitReceive();

#endif // PARALLEL


	virtual ~UrTreePar();



protected:

#ifdef PARALLEL

	/** the packed data type */
	MPI_Datatype pack_data_type;

	/** the sending request, used for non blocking send */
	MPI_Request send_request;

	/** the receiving request, used for non blocking receive */
	MPI_Request recv_request;


#endif // PARALLEL
};

#endif
