//! Errors that can be generated by H.263 operations

use std::io::{Error as IoError, ErrorKind as IoErrorKind};
use thiserror::Error;

#[derive(Debug, Error)]
pub enum Error {
    #[error("the H.263 decoder failed internally, this is a bug")]
    InternalDecoderError,

    #[error("the H.263 bitstream doesn't start with a picture")]
    MiddleOfBitstream,

    #[error("the H.263 bitstream contains an invalid macroblock header")]
    InvalidMacroblockHeader,

    #[error("the H.263 bitstream contains invalid macroblock coded bits")]
    InvalidMacroblockCodedBits,

    #[error("the H.263 bitstream contains an invalid intra-dc coefficient")]
    InvalidIntraDc,

    #[error("the H.263 bitstream contains an invalid short ac coefficient")]
    InvalidShortCoefficient,

    #[error("the H.263 bitstream contains an invalid long ac coefficient")]
    InvalidLongCoefficient,

    #[error("the H.263 bitstream contains an invalid motion vector")]
    InvalidMvd,

    #[error("the H.263 bitstream has an invalid picture type")]
    InvalidPType,

    #[error("the H.263 bitstream has an invalid extension picture type")]
    InvalidPlusPType,

    #[error("the H.263 bitstream has an invalid group-of-blocks header")]
    InvalidGobHeader,

    #[error("the H.263 bitstream could not be decoded")]
    InvalidBitstream,

    #[error("the decoded H.263 bitstream is missing it's picture format")]
    PictureFormatMissing,

    #[error("the decoded H.263 bitstream has an invalid picture format")]
    PictureFormatInvalid,

    #[error("the decoded H.263 bitstream has uncoded iframe blocks")]
    UncodedIFrameBlocks,

    #[error("an I/O error occured: {0}")]
    UnhandledIoError(IoError),

    #[error("a feature in the H.263 bitstream being decoded is not yet supported")]
    UnimplementedDecoding,
}

impl From<IoError> for Error {
    fn from(ioe: IoError) -> Self {
        Error::UnhandledIoError(ioe)
    }
}

impl Error {
    /// Determines if this is an end-of-data error.
    ///
    /// EOF errors end the current picture.
    pub fn is_eof_error(&self) -> bool {
        if let Self::UnhandledIoError(ioe) = self {
            matches!(ioe.kind(), IoErrorKind::UnexpectedEof)
        } else {
            false
        }
    }

    /// Determines if this is an error caused by macroblock parsing.
    ///
    /// Macroblock parsing errors can be recovered from by searching for the
    /// next group of blocks, if possible.
    pub fn is_macroblock_error(&self) -> bool {
        matches!(self, Error::InvalidMacroblockHeader)
            || matches!(self, Error::InvalidMacroblockCodedBits)
    }

    /// Determines if this is an error caused by GOB parsing.
    ///
    /// GOB parsing errors end the current picture.
    pub fn is_gob_error(&self) -> bool {
        matches!(self, Error::InvalidGobHeader)
    }
}

pub type Result<T> = std::result::Result<T, Error>;
