/*
 * Copyright (C) 2003 Jean-Philippe Hergott <hergott@free.fr>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "callbacks.h"

#define MY_ENCODING "ISO-8859-1"

/**
 * xcave_xmlWrite:
 * 
 *
 * the xmlWriter ginterface for writing to a xcave file
 */
void
xcave_xmlWrite(const gchar *pDBPATH)
{
    gint rc;
    xmlTextWriterPtr writer;
	
    /*
     * this initialize the library and check potential ABI mismatches
     * between the version it was compiled for and the actual shared
     * library used.
     */
    LIBXML_TEST_VERSION

	/* Create a new XmlWriter for DBPATH, with no compression. */
    writer = xmlNewTextWriterFilename(pDBPATH, 0);
    if (writer == NULL) {
        printf("xcave_xmlWrite: Error creating the xml writer\n");
        return;
    }

    /* Start the document with the xml default for the version,
     * encoding ISO 8859-1 and the default for the standalone
     * declaration. */
    rc = xmlTextWriterStartDocument(writer, NULL, MY_ENCODING, NULL);
    if (rc < 0) {
        printf("xcave_xmlWrite: Error at xmlTextWriterStartDocument\n");
        return;
    }

	rc = xmlTextWriterSetIndent(writer, 1);
    if (rc < 0) {
        printf("xcave_xmlWrite: Error at xmlTextWriterSetIndent\n");
        return;
    }

    /*
     * 2.2.6
     * warning at compilation time
     * rc = xmlTextWriterSetIndentString (writer, "  ");
     */
	rc = xmlTextWriterSetIndentString (writer, BAD_CAST "  ");
    if (rc < 0) {
        printf("xcave_xmlWrite: Error at xmlTextWriterSetIndentString	\n");
        return;
    }

    /* Start an element named "xcave". Since thist is the first
     * element, this will be the root element of the document. */
    rc = xmlTextWriterStartElement(writer, BAD_CAST "xcave");
    if (rc < 0) {
        printf("xcave_xmlWrite: Error at xmlTextWriterStartElement <xcave>\n");
        return;
    }

    /* Add an attribute with name "xmlns" and value "http://xcave.free.fr" to xcave. */
    rc = xmlTextWriterWriteAttribute(writer, BAD_CAST "xmlns",BAD_CAST "http://xcave.free.fr/xmlwine");
    if (rc < 0) {
        printf("xcave_xmlWrite: Error at xmlTextWriterWriteAttribute\n");
        return;
    }
	
    /* Start an element named "wines" as child of xcave. */
    rc = xmlTextWriterStartElement(writer, BAD_CAST "wines");
    if (rc < 0) {
        printf("xcave_xmlWrite: Error at xmlTextWriterStartElement <wines>\n");
        return;
    }

    /*
     * first of all the stock
     */
    g_list_foreach(list_stock, (GFunc)xcave_xmlWriteSingleWine, writer);

    /*
     * then the degustation sheet
     */
    g_list_foreach(list_full_ds, (GFunc)xcave_xmlWriteDegustationSheet, writer);

    /* Close the element named wines. */
    rc = xmlTextWriterEndElement(writer);
    if (rc < 0) {
        printf("xcave_xmlWrite: Error at xmlTextWriterEndElement\n");
        return;
    }

    /* Here we could close the elements ORDER and EXAMPLE using the
     * function xmlTextWriterEndElement, but since we do not want to
     * write any other elements, we simply call xmlTextWriterEndDocument,
     * which will do all the work. */
    rc = xmlTextWriterEndDocument(writer);
    if (rc < 0) {
        printf("xcave_xmlWrite: Error at xmlTextWriterEndDocument\n");
        return;
    }

    xmlFreeTextWriter(writer);

	/*
     * Cleanup function for the XML library.
     */
    xmlCleanupParser();

	return;

}

int
xcave_xmlTextWriterWriteElement(const gchar* field, xmlChar	*tmp, xmlTextWriterPtr writer) {


    if (tmp != NULL) {
       if (xmlTextWriterWriteElement(writer, BAD_CAST field, BAD_CAST tmp) < 0) {
          printf("xcave_xmlWriteSingle: Error at xmlTextWriterWriteElement %s\n",field);
          return (-1);
       }
       xmlFree(tmp);
    } 
    else {
	 /*
	  * 2.2.6
	  * warning at compilation time
	  * if (xmlTextWriterWriteElement(writer, BAD_CAST field, field) < 0) {
	  */
         if (xmlTextWriterWriteElement(writer, BAD_CAST field, BAD_CAST field) < 0) {
            printf("xcave_xmlWriteSingle: Error at xmlTextWriterWriteElement %s\n",field);
            return(-1);
         }
    }

    return(0);
}

/*
 * wine xml line
 */
void 
xcave_xmlWriteSingleWine(gchar *vindetail, xmlTextWriterPtr writer) {
    gint	        rc;
    xmlChar    *tmp;
    ptrVin      wineDetail;


    /* Start an element named "wine" as child of wine. */
    rc = xmlTextWriterStartElement(writer, BAD_CAST "wine");
	if (rc < 0) {
        return;
    }

    wineDetail = (ptrVin) vindetail;

    /* Write an element named "COUNTRY" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->country, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("country", tmp, writer);

    /* Write an element named "REGION" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->Region, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("region", tmp, writer);

    /* Write an element named "APPELLATION" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->Appellation, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("appellation", tmp, writer);

	/* Start an element named "domain" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->Nom, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("domain", tmp, writer);

    /* Write an element named "vintage" as child of wine. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "vintage","%4d", wineDetail->vintage);

    /* Write an element named "color" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->color, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("color", tmp, writer);

	/* Write an element named "wine maker" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->supplier, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("wineMaker", tmp, writer);

    /* Write an element named "wine maker address" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->wineMakerAdr, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("wineMakerAddress", tmp, writer);

    /* Write an element named "cepage" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->Cepage, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("cepage", tmp, writer);

    /* Write an element named "POTENTIEL DE GARDE" as child of wine. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "potentielDeGarde","%d", wineDetail->Conserv);

    /* Write an element named "TEMP.DE DEGUSTATION" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->temperatureDeDegustation, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("temperatureDeDegustation", tmp, writer);

	/*********************
	 * writing rack list information
	 */
	GList            *tmp_list_emplacement;
	ptrEmplacement    tmpEmplacement;
	
	tmpEmplacement = g_malloc (sizeof(Emplacement));
	
	for(tmp_list_emplacement = wineDetail->list_emplacement; 
		tmp_list_emplacement != NULL; 
		tmp_list_emplacement = tmp_list_emplacement->next) {

		memset(tmpEmplacement,'\0',sizeof(Emplacement));
		memcpy(tmpEmplacement,tmp_list_emplacement->data,sizeof(Emplacement));

		/* Start an element named "rack" as child of wine. */
		rc = xmlTextWriterStartElement(writer, BAD_CAST "rack");
		if (rc < 0) {
			printf("xcave_xmlWriteSingle: Error at xmlTextWriterStartElement <rack>\n");
			return;
		}
			
		/* Add an attribute with name "rackNb" and rackNb value. */
		rc = xmlTextWriterWriteFormatAttribute(writer, BAD_CAST "rackNb","%.1d", tmpEmplacement->Rack);
			
		/* Add an attribute with name "row" and row value. */
		rc = xmlTextWriterWriteFormatAttribute(writer, BAD_CAST "row","%d", tmpEmplacement->Row);
																								
		/* Add an attribute with name "column" and column value  */
		rc = xmlTextWriterWriteFormatAttribute(writer, BAD_CAST "column","%d", tmpEmplacement->Column);
			
		/* Close the element named rack. */
		rc = xmlTextWriterEndElement(writer);
		if (rc < 0) {
			printf("xcave_xmlWriteSingle: Error at xmlTextWriterEndElement\n");
			return;
		}
	}
	free(tmpEmplacement);

	/*********************
	 * writing purchase list informations
	 */
	ptrPurchase      tmpPurchase;
	GList            *tmp_list_purchase;

	tmpPurchase = g_malloc (sizeof(Purchase));
	
	for(tmp_list_purchase = wineDetail->list_purchase; 
		tmp_list_purchase != NULL; 
		tmp_list_purchase = tmp_list_purchase->next) {
	
		memset(tmpPurchase,'\0',sizeof(Purchase));
		memcpy(tmpPurchase,tmp_list_purchase->data,sizeof(Purchase));
		
		/* Start an element named "purchase" as child of wine. */
		rc = xmlTextWriterStartElement(writer, BAD_CAST "purchase");	
		if (rc < 0) {
			printf("xcave_xmlWriteSingle: Error at xmlTextWriterStartElement <purchase>\n");
			return;
		}
		
		/* Add an attribute with name "purchase date" and purchase date value. */
		rc = xmlTextWriterWriteFormatAttribute(writer, BAD_CAST "date","%4d", tmpPurchase->Achat);
			
		/* Add an attribute with name "quantity" and qty value. */
		rc = xmlTextWriterWriteFormatAttribute(writer, BAD_CAST "qty","%d", tmpPurchase->qty);
			
		/* Add an attribute with name "price" and price value. */
		rc = xmlTextWriterWriteFormatAttribute(writer, BAD_CAST "price","%.2f", tmpPurchase->Prix);
			
		/* Add an attribute with name "supplier" and value "carrefour" to purchase. */
		tmp = xcave_utf8ToIsolat1(tmpPurchase->Supplier, MY_ENCODING);
		rc = xmlTextWriterWriteAttribute(writer,BAD_CAST "supplier", BAD_CAST tmp);
			
		/* Add an attribute with name "supplier address" and value "bassens" to purchase. */
		tmp = xcave_utf8ToIsolat1(tmpPurchase->SupplierAdr, MY_ENCODING);
		rc = xmlTextWriterWriteAttribute(writer, BAD_CAST "supplierAdr", BAD_CAST tmp);

		/* Close the element named purchase date. */
		rc = xmlTextWriterEndElement(writer);
		if (rc < 0) {
			printf("xcave_xmlWriteSingle: Error at xmlTextWriterEndElement\n");
			return;
		}
	}
	free(tmpPurchase);


	/*********************
	 * writing drinking list information
	 */
	ptrDrinking       tmpDrinking;
	GList            *tmp_list_drinking;
	
	tmpDrinking = g_malloc (sizeof(Drinking));

	for(tmp_list_drinking = wineDetail->list_drinking; 
		tmp_list_drinking != NULL; 
		tmp_list_drinking = tmp_list_drinking->next) {

		memset(tmpDrinking,'\0',sizeof(Drinking));
		memcpy(tmpDrinking,tmp_list_drinking->data,sizeof(Drinking));

		/* Start an element named "drinking" as child of wine. */
		rc = xmlTextWriterStartElement(writer, BAD_CAST "drinking");
		if (rc < 0) {
			printf("xcave_xmlWriteSingle: Error at xmlTextWriterStartElement <drinking>\n");
			return;
		}
	
		/* Add an attribute with name "date" and drinking date value. */
		rc = xmlTextWriterWriteFormatAttribute(writer, BAD_CAST "date","%4d", tmpDrinking->Conso);
			
		/* Add an attribute with name "qty" and drinking qty value. */
		rc = xmlTextWriterWriteFormatAttribute(writer, BAD_CAST "qty","%d", tmpDrinking->qty);
			
		/* Add an attribute with name "rated" and drinking rated value. */
		tmp = xcave_utf8ToIsolat1(tmpDrinking->Note, MY_ENCODING);
		rc = xmlTextWriterWriteAttribute(writer, BAD_CAST "rated",BAD_CAST tmp);
			
		/* degustation sheet */
		rc = xmlTextWriterWriteFormatAttribute(writer, BAD_CAST "degustationSheetId","%d", (gint)tmpDrinking->degustation_sheet_id);

		/* Add an attribute with name "comment" and drinking comment value. */
		tmp = xcave_utf8ToIsolat1(tmpDrinking->Commentaire, MY_ENCODING);
		rc = xmlTextWriterWriteAttribute(writer , BAD_CAST "comment", BAD_CAST tmp);
			
		/* Close the element named drinking. */
		rc = xmlTextWriterEndElement(writer);
		if (rc < 0) {
			printf("xcave_xmlWriteSingle: Error at xmlTextWriterEndElement\n");
			return;
		}

	}
	free(tmpDrinking);

    /* Write an element named "NOTE" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->Note, MY_ENCODING);
	rc = xcave_xmlTextWriterWriteElement("rated", tmp, writer);

    /* Write an element named "COMMENTAIRE" as child of wine. */
    tmp = xcave_utf8ToIsolat1(wineDetail->Commentaire, MY_ENCODING);
	rc = xcave_xmlTextWriterWriteElement("comment", tmp, writer);

    /* Close the element named wine. */
    rc = xmlTextWriterEndElement(writer);
    if (rc < 0) {
        printf("xcave_xmlWriteSingle: Error at xmlTextWriterEndElement\n");
        return;
    }

    return;
}


/*
 * wine xml line
 */
void 
xcave_xmlWriteDegustationSheet(gchar *DSdetail, xmlTextWriterPtr writer) {
    gint			rc;
    xmlChar		*tmp;
    ptrDegustation_sheet   ds;

    /* Start an element named "degustationSheet" as child of wine. */
    if ( xmlTextWriterStartElement(writer, BAD_CAST "degustationSheet") < 0) return;

    ds = (ptrDegustation_sheet) DSdetail;

    /* Write an element named "degustationSheetId" as child of degustation sheet. */
    // rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "degustationSheetId","%s", ctime((time_t *)ds->degustation_sheet_id));
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "degustationSheetId","%d", (gint)ds->degustation_sheet_id);

    /* Write an element named "domain" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->domain, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("domain", tmp, writer);

    /* Write an element named "vintage" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "vintage","%d", ds->vintage);

    /* Write an element named "appellation" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->appellation, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("appellation", tmp, writer);

    /* Write an element named "region" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->region, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("region", tmp, writer);

    /* Write an element named "color" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->color, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("color", tmp, writer);

    /* Write an element named "degree" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->degree, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("degree", tmp, writer);

    /* Write an element named "producer" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->producer, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("producer", tmp, writer);

    /* Write an element named "comment" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->comment, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("comment", tmp, writer);

    /* Write an element named "depthHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "depthHscale","%d", ds->depth_hscale);

    /* Write an element named "depthText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->depth_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("depthText", tmp, writer);

    /* Write an element named "tintHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "tintHscale","%d", ds->tint_hscale);

    /* Write an element named "tintText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->tint_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("tintText", tmp, writer);

    /* Write an element named "sosHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "sosHscale","%d", ds->sos_hscale);

    /* Write an element named "sosText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->sos_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("sosText", tmp, writer);

    /* Write an element named "floralHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "floralHscale","%d", ds->floral_hscale);

    /* Write an element named "floralText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->floral_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("floralText", tmp, writer);

    /* Write an element named "fruityHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "fruityHscale","%d", ds->fruity_hscale);

    /* Write an element named "fruityText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->fruity_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("fruityText", tmp, writer);

    /* Write an element named "vegetableHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "vegetableHscale","%d", ds->vegetable_hscale);

    /* Write an element named "vegetableText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->vegetable_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("vegetableText", tmp, writer);

    /* Write an element named "chemicalHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "chemicalHscale","%d", ds->chemical_hscale);

    /* Write an element named "chemicalText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->chemical_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("chemicalText", tmp, writer);

    /* Write an element named "balsamHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "balsamHscale","%d", ds->balsam_hscale);

    /* Write an element named "balsamText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->balsam_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("balsamText", tmp, writer);

    /* Write an element named "spicyHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "spicyHscale","%d", ds->spicy_hscale);

    /* Write an element named "spicyText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->spicy_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("spicyText", tmp, writer);

    /* Write an element named "woodedHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "woodedHscale","%d", ds->wooded_hscale);

    /* Write an element named "woodedText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->wooded_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("woodedText", tmp, writer);

    /* Write an element named "animalHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "animalHscale","%d", ds->animal_hscale);

    /* Write an element named "animalText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->animal_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("animalText", tmp, writer);

    /* Write an element named "otherHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "otherHscale","%d", ds->other_hscale);

    /* Write an element named "otherText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->other_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("otherText", tmp, writer);

    /* Write an element named "complexityHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "complexityHscale","%d", ds->complexity_hscale);

    /* Write an element named "acidityHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "acidityHscale","%d", ds->acidity_hscale);

    /* Write an element named "acidityText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->acidity_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("acidityText", tmp, writer);

    /* Write an element named "mellowHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "mellowHscale","%d", ds->mellow_hscale);

    /* Write an element named "mellowText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->mellow_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("mellowText", tmp, writer);

    /* Write an element named "bitternessHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "bitternessHscale","%d", ds->bitterness_hscale);

    /* Write an element named "bitternessText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->bitterness_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("bitternessText", tmp, writer);

    /* Write an element named "alcoholHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "alcoholHscale","%d", ds->alcohol_hscale);

    /* Write an element named "alcoholText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->alcohol_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("alcoholText", tmp, writer);

    /* Write an element named "chewingHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "chewingHscale","%d", ds->chewing_hscale);

    /* Write an element named "chewingText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->chewing_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("chewingText", tmp, writer);

    /* Write an element named "astringencyHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "astringencyHscale","%d", ds->astringency_hscale);

    /* Write an element named "astringencyText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->astringency_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("astringencyText", tmp, writer);

    /* Write an element named "dbnText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->dbn_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("dbnText", tmp, writer);

    /* Write an element named "persistencyHscale" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "persistencyHscale","%d", ds->persistency_hscale);

    /* Write an element named "persistencyText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->persistency_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("persistencyText", tmp, writer);

    /* Write an element named "defaultText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->default_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("defaultText", tmp, writer);

    /* Write an element named "globalRateSpinbutton" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "globalRateSpinbutton","%d", ds->global_rate_spinbutton);

    /* Write an element named "maturityrSpinbutton" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "maturitySpinbutton","%d", ds->maturity_spinbutton);

    /* Write an element named "worthKeepingSpinbutton" as child of degustation sheet. */
    rc = xmlTextWriterWriteFormatElement(writer, BAD_CAST "worthKeepingSpinbutton","%d", ds->worth_keeping_spinbutton);

    /* Write an element named "dishText" as child of degustation sheet. */
    tmp = xcave_utf8ToIsolat1(ds->dish_text, MY_ENCODING);
    rc = xcave_xmlTextWriterWriteElement("dishText", tmp, writer);

    /* Close the element named "degustation sheet". */
    rc = xmlTextWriterEndElement(writer);
    if (rc < 0) {
        printf("xcave_xmlWriteSingle: Error at xmlTextWriterEndElement\n");
        return;
    }

    return;
}

/**
 * xcave_utf8ToIsolat1:
 * @in: string in a given encoding
 * @encoding: the encoding used
 *
 * Converts @in ginto UTF-8 for processing with libxml2 APIs
 *
 * Returns the converted UTF-8 string, or NULL in case of error.
 */
xmlChar *
xcave_utf8ToIsolat1 ( const gchar *in, const gchar *encoding)
{
    guchar *out;
    gint ret,size,out_size,temp;
    xmlCharEncodingHandlerPtr handler = NULL; /* 2.3.2 */

    size = (gint)strlen(in)+1; 
    out_size = size*2-1; 
    out = g_malloc((size_t)out_size); 

    if (out) {	
       handler = xmlFindCharEncodingHandler(encoding);
       if (!handler) {
          g_free(out);
          return(NULL);
       } 
    }
    temp=size-1;
		
    /*
     * 2.2.6
     * warning at compilation time
     * ret = handler->input(out, &out_size, in, &temp);
     */
    ret = handler->input(out, &out_size, (guchar *)in, &temp);
    if (ret == -1 || temp-size+1) {
       if (ret) {
          printf("conversion wasn't successful.\n");
       } else {
            printf("conversion wasn't successful. converted: %i octets.\n",temp);
       }
       g_free(out);
       out = NULL;
    } else {
       out = realloc(out,out_size+1); 
       out[out_size]=0;
    }

   return (out);
}
