/*
 * // Copyright (c) Radzivon Bartoshyk 8/2025. All rights reserved.
 * //
 * // Redistribution and use in source and binary forms, with or without modification,
 * // are permitted provided that the following conditions are met:
 * //
 * // 1.  Redistributions of source code must retain the above copyright notice, this
 * // list of conditions and the following disclaimer.
 * //
 * // 2.  Redistributions in binary form must reproduce the above copyright notice,
 * // this list of conditions and the following disclaimer in the documentation
 * // and/or other materials provided with the distribution.
 * //
 * // 3.  Neither the name of the copyright holder nor the names of its
 * // contributors may be used to endorse or promote products derived from
 * // this software without specific prior written permission.
 * //
 * // THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * // AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * // IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * // DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * // FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * // DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * // SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * // CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * // OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * // OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
use crate::common::f_fmla;
use crate::polyeval::f_polyeval5;
use crate::sin_cosf::sincosf_eval::sincosf_eval;

#[inline(always)]
fn cosf_gen_impl(x: f32) -> f32 {
    let x_abs = x.to_bits() & 0x7fff_ffffu32;
    let x = f32::from_bits(x_abs);
    let xd = x as f64;

    // |x| <= pi/16
    if x_abs <= 0x3e49_0fdbu32 {
        // |x| < 0.000244141
        if x_abs < 0x3980_0000u32 {
            #[cfg(any(
                all(
                    any(target_arch = "x86", target_arch = "x86_64"),
                    target_feature = "fma"
                ),
                target_arch = "aarch64"
            ))]
            {
                use crate::common::f_fmlaf;
                return f_fmlaf(x, f32::from_bits(0xb3000000), 1.);
            }
            #[cfg(not(any(
                all(
                    any(target_arch = "x86", target_arch = "x86_64"),
                    target_feature = "fma"
                ),
                target_arch = "aarch64"
            )))]
            {
                return f_fmla(xd, f64::from_bits(0xbe60000000000000), 1.) as f32;
            }
        }

        // Cosine
        // Generated poly by Sollya:
        // f_cos_16 = cos(x);
        //
        // Q = fpminimax(f_cos_16, [|0, 2, 4, 6, 8|], [|1, D...|], [0, pi/16]);
        // See ./notes/cosf.sollya

        let x2 = xd * xd;
        let p = f_polyeval5(
            x2,
            f64::from_bits(0x3ff0000000000000),
            f64::from_bits(0xbfdffffffffffcea),
            f64::from_bits(0x3fa55555553d611a),
            f64::from_bits(0xbf56c16b2e26561a),
            f64::from_bits(0x3ef9faa67b9da80b),
        );
        return p as f32;
    }

    if x_abs >= 0x7f80_0000u32 {
        return x + f32::NAN;
    }

    // Formula:
    //   cos(x) = cos((k + y)*pi/32)
    //          = cos(y*pi/32) * cos(k*pi/32) - sin(y*pi/32) * sin(k*pi/32)
    // The values of sin(k*pi/32) and cos(k*pi/32) for k = 0..63 are precomputed
    // and stored using a vector of 32 doubles. Sin(y*pi/32) and cos(y*pi/32) are
    // computed using degree-7 and degree-6 minimax polynomials generated by
    // Sollya respectively.
    // Combine the results with the sine of sum formula:
    //   cos(x) = cos((k + y)*pi/32)
    //          = cos(y*pi/32) * cos(k*pi/32) - sin(y*pi/32) * sin(k*pi/32)
    //          = cosm1_y * cos_k + sin_y * sin_k
    //          = (cosm1_y * cos_k + cos_k) + sin_y * sin_k

    let rs = sincosf_eval(xd, x_abs);
    f_fmla(rs.sin_y, -rs.sin_k, f_fmla(rs.cosm1_y, rs.cos_k, rs.cos_k)) as f32
}

#[cfg(any(target_arch = "x86", target_arch = "x86_64"))]
#[target_feature(enable = "avx", enable = "fma")]
unsafe fn cosf_fma_impl(x: f32) -> f32 {
    let x_abs = x.to_bits() & 0x7fff_ffffu32;
    let x = f32::from_bits(x_abs);
    let xd = x as f64;

    // |x| <= pi/16
    if x_abs <= 0x3e49_0fdbu32 {
        // |x| < 0.000244141
        if x_abs < 0x3980_0000u32 {
            return f32::mul_add(x, f32::from_bits(0xb3000000), 1.);
        }

        // Cosine
        // Generated poly by Sollya:
        // f_cos_16 = cos(x);
        //
        // Q = fpminimax(f_cos_16, [|0, 2, 4, 6, 8|], [|1, D...|], [0, pi/16]);
        // See ./notes/cosf.sollya

        let x2 = xd * xd;
        use crate::polyeval::d_polyeval5;
        let p = d_polyeval5(
            x2,
            f64::from_bits(0x3ff0000000000000),
            f64::from_bits(0xbfdffffffffffcea),
            f64::from_bits(0x3fa55555553d611a),
            f64::from_bits(0xbf56c16b2e26561a),
            f64::from_bits(0x3ef9faa67b9da80b),
        );
        return p as f32;
    }

    if x_abs >= 0x7f80_0000u32 {
        return x + f32::NAN;
    }

    // Formula:
    //   cos(x) = cos((k + y)*pi/32)
    //          = cos(y*pi/32) * cos(k*pi/32) - sin(y*pi/32) * sin(k*pi/32)
    // The values of sin(k*pi/32) and cos(k*pi/32) for k = 0..63 are precomputed
    // and stored using a vector of 32 doubles. Sin(y*pi/32) and cos(y*pi/32) are
    // computed using degree-7 and degree-6 minimax polynomials generated by
    // Sollya respectively.
    // Combine the results with the sine of sum formula:
    //   cos(x) = cos((k + y)*pi/32)
    //          = cos(y*pi/32) * cos(k*pi/32) - sin(y*pi/32) * sin(k*pi/32)
    //          = cosm1_y * cos_k + sin_y * sin_k
    //          = (cosm1_y * cos_k + cos_k) + sin_y * sin_k
    use crate::sin_cosf::sincosf_eval::sincosf_eval_fma;
    let rs = sincosf_eval_fma(xd, x_abs);
    f64::mul_add(
        rs.sin_y,
        -rs.sin_k,
        f64::mul_add(rs.cosm1_y, rs.cos_k, rs.cos_k),
    ) as f32
}

/// Computes cosine function
///
/// ULP 0.5
///
#[inline]
pub fn f_cosf(x: f32) -> f32 {
    #[cfg(not(any(target_arch = "x86", target_arch = "x86_64")))]
    {
        cosf_gen_impl(x)
    }
    #[cfg(any(target_arch = "x86", target_arch = "x86_64"))]
    {
        use std::sync::OnceLock;
        static EXECUTOR: OnceLock<unsafe fn(f32) -> f32> = OnceLock::new();
        let q = EXECUTOR.get_or_init(|| {
            if std::arch::is_x86_feature_detected!("avx")
                && std::arch::is_x86_feature_detected!("fma")
            {
                cosf_fma_impl
            } else {
                cosf_gen_impl
            }
        });
        unsafe { q(x) }
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn f_cosf_test() {
        assert_eq!(f_cosf(0.0), 1.0);
        assert_eq!(f_cosf(std::f32::consts::PI), -1f32);
        assert_eq!(f_cosf(0.5), 0.87758255);
        assert_eq!(f_cosf(0.7), 0.7648422);
        assert_eq!(f_cosf(1.7), -0.12884454);
        assert!(f_cosf(f32::INFINITY).is_nan());
        assert!(f_cosf(f32::NEG_INFINITY).is_nan());
        assert!(f_cosf(f32::NAN).is_nan());
        assert_eq!(f_cosf(0.0002480338), 0.9999999692396206);
    }
}
