/* $Id: dp_stream_stat.c,v 1.6 2004/02/20 01:59:48 andrewbaker Exp $ */
/*
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
** Copyright (C) 2001 Martin Roesch <roesch@sourcefire.com>
**
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
*/

/*  I N C L U D E S  *****************************************************/
#include "config.h"

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef SOLARIS
    #include <strings.h>
#endif
#include <errno.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "dp_plugbase.h"
#include "spool.h"
#include "dp_stream_stat.h"
#include "util.h"
#include "barnyard.h"

/*  P R O T O T Y P E S  ************************************************/
void StreamStatDpSetup(char *, DpFunctionalNode *);
int StreamStatDpReadFileHeader(DpFunctionalNode *, SpoolFileHandle *sph);
int StreamStatDpReadRecord(SpoolFileHandle *sph);
int StreamStatDpProcessRecord(void *, DpFunctionalNode *dpfn);
void StreamStatDpPrintRecord(void *);

/*-----------------------Local Global Variables -----------------------*/
/* this is safe since we are single threaded and can only run one instance
 * of a data processor
 */
static void *buffer;


void StreamStatDpInit()
{
    PluginInfo pi;

    pi.author = strdup("Andrew R. Baker <andrewb@snort.org>");
    pi.version = strdup("1.0");
    pi.type = strdup("stream input processor");
    pi.copyright = strdup("(C) Copyright 2001, Andrew R. Baker");
    pi.description = 
        strdup("reads the stats output by stream4");
    pi.usage = strdup("dp_stream_stat");
    
    /* register the name, setup function and data type */
    RegisterDp("dp_stream_stat", StreamStatDpSetup, "stream_stat", &pi);
    
        
    free(pi.author);
    free(pi.version);
    free(pi.type);
    free(pi.copyright);
    free(pi.description);
    free(pi.usage);

    if(pv.verbose)
        LogMessage("dp_stream_stat loaded\n");
    return;
}

/* Add the input plugin to the DataProcessors list */
void StreamStatDpSetup(char *config, DpFunctionalNode *dpfn)
{
    dpfn->type = strdup("stream_stat");
	dpfn->magic = STREAM_STAT_MAGIC;
	RegisterDpReadRecord(StreamStatDpReadRecord, dpfn);
	RegisterDpReadFileHeader(StreamStatDpReadFileHeader, dpfn);
	RegisterDpProcessRecord(StreamStatDpProcessRecord, dpfn);
    /* allocate the global buffer */
    buffer = SafeAlloc(sizeof(StreamStatRecord));
}


int StreamStatDpReadFileHeader(DpFunctionalNode *dpfn, SpoolFileHandle *sph)
{
    StreamStatFileHeader file_header;
    ssize_t bytes_read;

    bzero(&file_header, sizeof(StreamStatFileHeader));

    if((bytes_read = read(sph->filedes, &file_header, 
                    sizeof(StreamStatFileHeader) ))
            != sizeof(StreamStatFileHeader))
    {
        printf("ERROR => Bad header in stream stat spool file \"%s\": %s "
                "[size: %d should be %d]\n", sph->filepath, strerror(errno), 
                bytes_read, sizeof(StreamStatFileHeader));

        return 1;
    }

    /* validate the header */
#ifdef DEBUG
    printf("Opened StreamStat File \"%s\", header:\n", sph->filepath);
    printf(" Magic          = 0x%X\n", file_header.magic);
    printf(" Version.major  = %d\n", file_header.version_major);
    printf(" Version.minor  = %d\n", file_header.version_minor);
    printf(" timezone       = %d\n", file_header.timezone);
    printf("====================================="
            "===============================\n");   
#endif

    /* copy the header into the dp plugin */
    if(dpfn->context.file_header != NULL)
        free(dpfn->context.file_header);
    dpfn->context.file_header = SafeAlloc(sizeof(file_header));
    memcpy(dpfn->context.file_header, &file_header, sizeof(file_header));
    /* copy the header into the Spool File Handle */
    sph->header = SafeAlloc(sizeof(file_header));
    memcpy(sph->header, &file_header, sizeof(file_header));

    return 0;
}
            
/* Partial reads should rarely, if ever, happen.  Thus we should not actually
   call lseek very often 
 */
   
int StreamStatDpReadRecord(SpoolFileHandle *sph)
{
    int fd;
    Record *record;
    ssize_t bytes_read;

    if(!sph)
        return -1;  /* Invalid args */

    fd = sph->filedes;
    record = &sph->record;

    if((bytes_read = read(fd, buffer, sizeof(StreamStatRecord))) == -1)
    {
        /* XXX: Poor error handling */
        perror("Error reading UnifiedAlert");
        exit(1);
    }
    if(bytes_read != sizeof(StreamStatRecord))
    {
        if(bytes_read == 0) /* EOF */
            return N_READ_EOF;
        /* partial read */
        if(lseek(fd, 0 - bytes_read, SEEK_CUR) == -1)
        {
            /* XXX: Poor error handling*/
            perror("lseek error");
            exit(1);
        }
        return N_READ_PARTIAL;
    }
    record->data = buffer;
    return 0;
}

int StreamStatDpProcessRecord(void *data, DpFunctionalNode *dpfn)
{
    if(data == NULL)
    {
#ifdef DEBUG
        printf("NULL Argument to StreamStatDpProcessRecord\n");
#endif
        return 1;
    }


    /* Call the output plugins */
    CallOutputPlugins(dpfn->oList, data);

#ifdef DEBUG
    StreamStatDpPrintRecord(data);
#endif 
    return 0;
}

void StreamStatDpPrintRecord(void *data)
{
    struct in_addr in;
    StreamStatRecord *record;

    if(data == NULL)
        return;
    
    record = (StreamStatRecord *)data;
    printf("Start Time     = %u\n", record->start_time);
    printf("End Time       = %u\n", record->end_time);
    in.s_addr = record->server_ip;
    printf("Server         = %s:%u\n", inet_ntoa(in), record->server_port);
    in.s_addr = record->client_ip;
    printf("Client         = %s:%u\n", inet_ntoa(in), record->client_port);
    printf("Server Traffic = %u bytes\t %u packets\n", record->server_bytes, 
            record->server_bytes);
    printf("Client Traffic = %u bytes\t %u packets\n", record->client_bytes, 
            record->client_bytes);
    printf("------------------------------------------------------\n");
}
