/*
** Copyright (C) 2003 Sourcefire, Inc.  (www.sourcefire.com)
** Copyright (C) 2001-2002 Andrew R. Baker <andrewb@snort.org>
** 
** This program is distributed under the terms of version 1.0 of the 
** Q Public License.  See LICENSE.QPL for further details.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
**
** Author(s):   Andrew R. Baker <andrewb@sourcefire.com>
**
** Based on op_alert_syslog.c
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/types.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <netinet/in.h>
#include <errno.h>
#include <netdb.h>
extern int h_errno;
#include <sys/socket.h>
#include <arpa/inet.h>

#include "output-plugins/op_plugbase.h"
#include "util.h"
#include "barnyard.h"
#include "input-plugins/dp_alert.h"
#include "mstring.h"
#include "classification.h"
#include "sid.h"

#define MODULE_NAME "alert_syslog2"

typedef struct _OpAlertSyslog2_Data 
{
    /* Runtime configuration */
    char *message_buffer;
    int header_length;
    int month_offset;
    int timestamp_offset;
    int socket;
    struct sockaddr_in sin;
    u_int8_t priority;

    /* Configuration arguments */
    char *tag;
    char *hostname;
    char *syslog_host;
    int facility;
    int severity;
    int syslog_port;
    u_int8_t pid_flag;
} OpAlertSyslog2_Data;

#define MESSAGE_LENGTH  1024
#define MONTH_LENGTH    3
#define TIMESTAMP_LENGTH 11

/* Message Template:
 *
 * <PRI>xxx xx xx:xx:xx xxxxxxxx  xxxxx[xxxxx]: 
 *
 * template len     - the length of the template
 * month offset     - the offset where the month begins
 * month length     - the length of bytes for the month
 * timestamp offset - the offset where the timestamp begins
 * timestamp length - the length of bytes for the timestamp
 * msg offset
 */

/* Values to use for the month portion of the message.  We cannot use
 * strftime for this
 */
char *month_values[] = 
{
    "Jan",
    "Feb",
    "Mar",
    "Apr",
    "May",
    "Jun",
    "Jul",
    "Aug",
    "Sep",
    "Oct",
    "Nov",
    "Dec"
};

/* Plugin entry functions */
static int OpAlertSyslog2_Setup(OutputPlugin *, char *args);
static int OpAlertSyslog2_Exit(OutputPlugin *);
static int OpAlertSyslog2_Start(OutputPlugin *, void *);
static int OpAlertSyslog2_Stop(OutputPlugin *);
static int OpAlertSyslog2(void *, void *);
static int OpAlertSyslog2_LogConfig(OutputPlugin *);

static OpAlertSyslog2_Data *OpAlertSyslog2_ParseArgs(char *args);

static int IsValidHostname(char *hostname);
static int IsValidSyslogTag(char *syslog_tag);

struct keyword_value
{
    char *keyword;
    int value;
};

struct keyword_value facility_map[] = 
{
    { "KERN",        0 },
    { "USER",        1 },
    { "MAIL",        2 },
    { "DAEMON",      3 },
    { "AUTH",        4 },
    { "SYSLOG",      5 },
    { "LPR",         6 },
    { "NEWS",        7 },
    { "UUCP",        8 },
    { "CRON",        9 },
    { "AUTHPRIV",   10 },
    { "FTP",        11 },
    { "NTP",        12 },
    { "AUDIT",      13 },
    { "ALERT",      14 },
    { "CLOCK",      15 },
    { "LOCAL0",     16 },
    { "LOCAL1",     17 },
    { "LOCAL2",     18 },
    { "LOCAL3",     19 },
    { "LOCAL4",     20 },
    { "LOCAL5",     21 },
    { "LOCAL6",     22 },
    { "LOCAL7",     23 },
    { NULL,         -1 }
};

#define DEFAULT_FACILITY    23
#define MAX_FACILITY        23

struct keyword_value severity_map[] = 
{
    { "EMERG",   0 },
    { "ALERT",   1 },
    { "CRIT",    2 },
    { "ERROR",   3 },
    { "WARN",    4 },
    { "NOTICE",  5 },
    { "INFO",    6 },
    { "DEBUG",   7 },
    { NULL,     -1 }
};

#define DEFAULT_SEVERITY    5
#define MAX_SEVERITY        7

#define DEFAULT_SYSLOG_PORT 514
#define DEFAULT_SYSLOG_HOST "localhost"

/* init routine makes this processor available for dataprocessor directives */
void OpAlertSyslog2_Init()
{
    OutputPlugin *outputPlugin;

    /* Register the output plugin */
    outputPlugin = RegisterOutputPlugin(MODULE_NAME, "alert");
    
    /* Set the functions */
    outputPlugin->setupFunc = OpAlertSyslog2_Setup;
    outputPlugin->exitFunc = OpAlertSyslog2_Exit;
    outputPlugin->startFunc = OpAlertSyslog2_Start;
    outputPlugin->stopFunc = OpAlertSyslog2_Stop;
    outputPlugin->outputFunc = OpAlertSyslog2;
    outputPlugin->logConfigFunc = OpAlertSyslog2_LogConfig;
}

static int OpAlertSyslog2_LogConfig(OutputPlugin *outputPlugin)
{
    OpAlertSyslog2_Data *data = NULL;
    
    if(!outputPlugin || !outputPlugin->data)
        return -1;

    data = (OpAlertSyslog2_Data *)outputPlugin->data;

    LogMessage("OpAlertSyslog2 configured\n");
    LogMessage("  Syslog Host/Port: %s:%u/udp\n", data->syslog_host,
            data->syslog_port);
    LogMessage("  Syslog Facility:  %s(%i)\n", facility_map[data->facility],
            data->facility);
    LogMessage("  Syslog Severity:  %s(%i)\n", severity_map[data->severity],
            data->severity);
    LogMessage("  Hostname: %s\n", data->hostname);
    if(data->pid_flag)
        LogMessage("  Tag: %s[%u]\n", data->tag, getpid());
    else
        LogMessage("  Tag: %s\n", data->tag);

    return 0;
}

/* Instantiate the output plugin */
static int OpAlertSyslog2_Setup(OutputPlugin *outputPlugin, char *args)
{
    /* setup the run time context for this output plugin */
    if(!(outputPlugin->data = OpAlertSyslog2_ParseArgs(args)))
    {
        FatalError("Failed to setup %s output plugin\n", MODULE_NAME);
        return -1;  /* XXX Eventually we will look at these result codes */
    }
    
    return 0;
}

/* Destructor function */
static int OpAlertSyslog2_Exit(OutputPlugin *outputPlugin)
{
    OpAlertSyslog2_Data *data = (OpAlertSyslog2_Data *)outputPlugin->data;
    
    /* free that context data */
    if(data)
    {
        if(data->message_buffer)
            free(data->message_buffer);
        data->message_buffer = NULL;

        if(data->tag)
            free(data->tag);
        data->tag = NULL;
        
        if(data->hostname)
            free(data->hostname);
        data->hostname = NULL;
        
        if(data->syslog_host)
            free(data->syslog_host);
        data->syslog_host = NULL;

        free(data);
    }
    data = NULL;
    outputPlugin->data = NULL;

    return 0;
}

/* Start function */
static int OpAlertSyslog2_Start(OutputPlugin *outputPlugin, void *spool_header)
{
    OpAlertSyslog2_Data *data = (OpAlertSyslog2_Data *)outputPlugin->data;
    struct hostent *host_entry = NULL;
    
    if(data == NULL)
        FatalError("ERROR: Unable to find context for %s\n", MODULE_NAME);
    
    if(pv.verbose >= 2)
        OpAlertSyslog2_LogConfig(outputPlugin);

    
    /* Lookup the remote host */
    if(inet_aton(data->syslog_host, &data->sin.sin_addr) == 0)
    {
        if(!(host_entry = gethostbyname(data->syslog_host)))
            FatalError("%s: Unable to lookup remote host '%s': %s\n", 
                    MODULE_NAME, data->syslog_host, hstrerror(h_errno));
        memcpy(&data->sin.sin_addr, host_entry->h_addr, 
                sizeof(data->sin.sin_addr));
    }
    data->sin.sin_family = AF_INET;
    data->sin.sin_port = htons(data->syslog_port);
    
    /* open a socket to the remote syslog host */
    if((data->socket = socket(PF_INET, SOCK_DGRAM, 0)) == -1)
        FatalError("%s: Unable to create socket: %s\n", MODULE_NAME, 
            strerror(errno));

    return 0;
}


/* Stop function */
static int OpAlertSyslog2_Stop(OutputPlugin *outputPlugin)
{
    OpAlertSyslog2_Data *data = (OpAlertSyslog2_Data *)outputPlugin->data;

    if(data == NULL)
        FatalError("ERROR: Unable to find context for %s\n", MODULE_NAME);

    /* close the socket */
    if(data->socket != -1)
        close(data->socket);
    data->socket = -1;
    
    return 0;
}


/* output function */
static int OpAlertSyslog2(void *context, void *data)
{
    UnifiedAlertRecord *record;
    OpAlertSyslog2_Data *op_data;
    struct tm *tm;
    char src[16];
    char dest[16];
    ClassType *class_type = NULL;
    Sid *sid = NULL;
    int length;
    int rval;


    if(!data || !context)
        return -1;

    record = (UnifiedAlertRecord *)data;
    op_data = (OpAlertSyslog2_Data *)context;

    sid = GetSid(record->event.sig_generator, record->event.sig_id);
    class_type = GetClassType(record->event.classification);

    /* convert alert timestamp to struct tm using gmtime */
    if(pv.localtime)
        tm = localtime(&record->ts.tv_sec);
    else
        tm = gmtime(&record->ts.tv_sec);

    memcpy(op_data->message_buffer + op_data->month_offset, 
            month_values[tm->tm_mon], MONTH_LENGTH);

    strftime(op_data->message_buffer + op_data->timestamp_offset, 
            TIMESTAMP_LENGTH + 1, "%e %H:%M:%S", tm);

    op_data->message_buffer[op_data->timestamp_offset + TIMESTAMP_LENGTH] = ' ';


    snprintf(src, 16, "%u.%u.%u.%u", (record->sip & 0xff000000) >> 24,
            (record->sip & 0x00ff0000) >> 16, (record->sip & 0x0000ff00) >> 8,
            record->sip & 0x000000ff);
    snprintf(dest, 16, "%u.%u.%u.%u", (record->dip & 0xff000000) >> 24,
            (record->dip & 0x00ff0000) >> 16, (record->dip & 0x0000ff00) >> 8,
            record->dip & 0x000000ff);

    switch(record->protocol)
    {
        case IPPROTO_TCP:
        case IPPROTO_UDP:
            length = snprintf(op_data->message_buffer + op_data->header_length,
                    MESSAGE_LENGTH - op_data->header_length, 
                    "[%d:%d:%d] %s [Classification: %s] "
                    "[Priority: %d] {%s} %s:%d -> %s:%d",
                    record->event.sig_generator, record->event.sig_id,
                    record->event.sig_rev, sid != NULL ? sid->msg : "ALERT",
                    class_type != NULL ? class_type->name : "Unknown",
                    record->event.priority, protocol_names[record->protocol],
                    src, record->sp, dest, record->dp);
            break;
        case IPPROTO_ICMP:
            length = snprintf(op_data->message_buffer + op_data->header_length,
                    MESSAGE_LENGTH - op_data->header_length, 
                    "[%d:%d:%d] %s [Classification: %s] "
                    "[Priority: %d] {%s} %s -> %s",
                    record->event.sig_generator, record->event.sig_id,
                    record->event.sig_rev, sid != NULL ? sid->msg : "ALERT",
                    class_type != NULL ? class_type->name : "Unknown",
                    record->event.priority, protocol_names[record->protocol],
                    src, dest);
            break;
        default:
            length = snprintf(op_data->message_buffer + op_data->header_length,
                    MESSAGE_LENGTH - op_data->header_length, 
                    "[%d:%d:%d] %s [Classification: %s] "
                    "[Priority: %d] {%s} %s -> %s",
                    record->event.sig_generator, record->event.sig_id,
                    record->event.sig_rev, sid != NULL ? sid->msg : "ALERT",
                    class_type != NULL ? class_type->name : "Unknown",
                    record->event.priority, protocol_names[record->protocol],
                    src, dest);
            break;
    }

    if((rval = sendto(op_data->socket, op_data->message_buffer, 
                    op_data->header_length + length, 0, 
                    (struct sockaddr *)&op_data->sin, 
                    sizeof(op_data->sin))) == -1)
        LogMessage("%s: sendto error %u: %s\n", errno, strerror(errno));

    return 0;
}

/* initialize the output processor for this particular instantiation */
OpAlertSyslog2_Data *OpAlertSyslog2_ParseArgs(char *args)
{
    OpAlertSyslog2_Data *data;
    char **toks;
    int num_toks;
    int i;
    int header_length = -1;
    char *index;

    if(pv.verbose)
        LogMessage("Parsing %s arguments: %s\n", MODULE_NAME, args);
    
    if(!(data = (OpAlertSyslog2_Data *)calloc(1, sizeof(OpAlertSyslog2_Data))))
    {
        FatalError("Out of memory creating %s configuration\n", MODULE_NAME);
        return NULL;
    }
    data->facility = -1;
    data->severity = -1;
    data->socket = -1;

    if(args)
    {
        toks = mSplit(args, ";", 8, &num_toks, 0);  
        /* XXX error check */

        for(i = 0; i < num_toks; i++)
        {
            char *token = toks[i];
            char **subtoks;
            int num_subtoks;
            long value;
            StripWhitespace(&token);
            if(*token == '\0')
                continue;
            /* split the token on ':' */
            subtoks = mSplit(token, ":", 2, &num_subtoks, 0);  
            /* XXX error check */

            if(strcasecmp("facility", subtoks[0]) == 0)
            {
                if(data->facility >= 0)
                {
                    FatalError("%s: Multiple %s arguments\n", 
                            MODULE_NAME, subtoks[0]);
                }
                /* Process facility Argument */
                if(num_subtoks != 2)
                {
                    FatalError("%s: Invalid %s argument: %s\n",
                            MODULE_NAME, subtoks[0], subtoks[1]);
                }

                if(String2Long(subtoks[1], &value) == 0)
                {
                    if(value > MAX_FACILITY)
                        FatalError("%s: Invalid %s argument: %s\n"
                                MODULE_NAME, subtoks[0], subtoks[1]);
                    else
                        data->facility = value;
                }
                else
                {
                    /* search for match in facility map */
                    int j = 0;
                    while(facility_map[j].keyword)
                    {
                        if(strcasecmp(facility_map[j].keyword, subtoks[1]) == 0)
                        {
                            data->facility = facility_map[j].value;
                            break;
                        }
                        j++;
                    }
                    if(data->facility < 0)
                    {
                        FatalError("%s: Invalid %s argument: %s\n",
                                MODULE_NAME, subtoks[0], subtoks[1]);
                    }
                }
            }
            else if(strcasecmp("severity", subtoks[0]) == 0)
            {
                if(data->severity >= 0)
                {
                    FatalError("%s: Multiple %s arguments\n", 
                            MODULE_NAME, subtoks[0]);
                }
                /* Process severity Argument */
                if(num_subtoks != 2)
                {
                    FatalError("%s: Invalid %s argument: %s\n",
                            MODULE_NAME, subtoks[0], subtoks[1]);
                }

                if(String2Long(subtoks[1], &value) == 0)
                {
                    if(value > MAX_FACILITY)
                        FatalError("%s: Invalid %s argument: %s\n"
                                MODULE_NAME, subtoks[0], subtoks[1]);
                    else
                        data->severity = value;
                }
                else
                {
                    /* search for match in severity map */
                    int j = 0;
                    while(severity_map[j].keyword)
                    {
                        if(strcasecmp(severity_map[j].keyword, subtoks[1]) == 0)
                        {
                            data->severity = severity_map[j].value;
                            break;
                        }
                        j++;
                    }
                    if(data->severity < 0)
                    {
                        FatalError("%s: Invalid %s argument: %s\n",
                                MODULE_NAME, subtoks[0], subtoks[1]);
                    }
                }
            }
            else if(strcasecmp("hostname", subtoks[0]) == 0)
            {
                if(data->hostname)
                {
                    FatalError("%s: Multiple %s arguments\n", 
                            MODULE_NAME, subtoks[0]);
                }
                /* Must be < 255 bytes and must contain only alphanumeric
                 * names and embedded '-'s */
                if(IsValidHostname(subtoks[1]) != 1)
                    FatalError("%s: %s argument is not a valid hostname: %s\n",
                            MODULE_NAME, subtoks[0], subtoks[1]);

                if(!(data->hostname = strdup(subtoks[1])))
                    FatalError("%s: Out of memory processing config\n");
            }
            else if(strcasecmp("tag", subtoks[0]) == 0)
            {
                if(data->tag)
                {
                    FatalError("%s: Multiple %s arguments\n", 
                            MODULE_NAME, subtoks[0]);
                }
                if(IsValidSyslogTag(subtoks[1]) != 1)
                    FatalError("%s: %s argument is not a valid syslog tag: "
                            "%s\n", MODULE_NAME, subtoks[0], subtoks[1]);
                
                if(!(data->tag = strdup(subtoks[1])))
                    FatalError("%s: Out of memory processing config\n");
            }
            else if(strcasecmp("syslog_host", subtoks[0]) == 0)
            {
                if(data->syslog_host)
                {
                    FatalError("%s: Multiple %s arguments\n", 
                            MODULE_NAME, subtoks[0]);
                }
                if(!(data->syslog_host = strdup(subtoks[1])))
                    FatalError("%s: Out of memory processing config\n");
            }
            else if(strcasecmp("syslog_port", subtoks[0]) == 0)
            {
                if(data->syslog_port > 0)
                {
                    FatalError("%s: Multiple %s arguments\n", 
                            MODULE_NAME, subtoks[0]);
                }
                if(String2Long(subtoks[1], &value) != 0)
                    FatalError("%s: Invalid %s argument: %s\n",
                            MODULE_NAME, subtoks[0], subtoks[1]);
                
                if(value < 1 || value > 65535)
                    FatalError("%s: Invalid %s argument: %s\n",
                            MODULE_NAME, subtoks[0], subtoks[1]);

                data->syslog_port = value;

            }
            else if(strcasecmp("withpid", subtoks[0]) == 0)
            {   
                if(num_subtoks != 1)
                {
                    FatalError("%s: %s does not take arguments\n",
                            MODULE_NAME, subtoks[0]);
                }
                data->pid_flag = 1;
            }
            else
            {
                FatalError("%s: Unknown argument: %s\n",
                        MODULE_NAME, subtoks[0]);
            }

            FreeToks(subtoks, num_subtoks);
        }

        FreeToks(toks, num_toks);
    }

    if(data->facility == -1)
        data->facility = DEFAULT_FACILITY;
    if(data->severity == -1)
        data->severity = DEFAULT_SEVERITY;
    if(!data->tag)
    {
        if(!(data->tag = strdup(PROGRAM_NAME)))
            FatalError("%s: Out of memory processing config\n");
    }
    if(!data->hostname)
    {
        char hostname[255];
        char *index;
        if(gethostname(hostname, 255) != 0)
            FatalError("%s: Unable to get hostname\n");
        /* since we may get a FQDN, munge the hostname */
        if((index = strchr(hostname, '.')))
            *index = '\0';

        if(!(data->hostname = strdup(hostname)))
            FatalError("%s: Out of memory processing config\n");
    }
    if(!data->syslog_host)
    {
        if(!(data->syslog_host = strdup(DEFAULT_SYSLOG_HOST)))
            FatalError("%s: Out of memory processing config\n");
    }
    if(data->syslog_port == 0)
        data->syslog_port = DEFAULT_SYSLOG_PORT;
    
    /* calculate the syslog priority */
    data->priority = data->facility * 8 + data->severity;

    /* allocate the message buffer */
    if(!(data->message_buffer = calloc(MESSAGE_LENGTH, sizeof(char))))
        FatalError("%s: Out of memory starting output plugin\n");

    /* copy in the basic string */
    if(data->pid_flag)
        header_length = snprintf(data->message_buffer, MESSAGE_LENGTH, 
                "<%u>XXX XX XX:XX:XX %s %s[%u]: ",
                data->priority, data->hostname, data->tag, getpid());
    else
        header_length = snprintf(data->message_buffer, MESSAGE_LENGTH, 
                "<%u>XXX XX XX:XX:XX %s %s: ", 
                data->priority, data->hostname, data->tag);

    if(header_length > MESSAGE_LENGTH)
        FatalError("%s: Message header length is too long: %i\n", 
                header_length);

    data->header_length = header_length;

    if(!(index = strchr(data->message_buffer, '>')))
        FatalError("%s: Error calculating priority field length\n");

    data->month_offset = index - data->message_buffer + 1;
    data->timestamp_offset = data->month_offset + 4;
    
        
    if(pv.verbose)
    {
    }
    
    return data;
}

static int IsValidHostname(char *hostname)
{
    char *index;
    int firstchar = 1;
    int lastdash = 0;
       
    
    if(!hostname)
        return 0;

    if(strlen(hostname) > 254)
        return 0;

    /* check characters */
    index = hostname;
    while(*index)
    {
        if(!isalnum(*index))
        {
            /* check for '-' */
            if(*index == '-')
            {
                if(firstchar)
                    return 0;
                lastdash = 1;
            }
            else
                return 0;
        }
        else
        {
            lastdash = 0;
            firstchar = 0;
        }
        index++;
    }

    if(firstchar || lastdash)
        return 0;
    
    return 1;
}

static int IsValidSyslogTag(char *syslog_tag)
{
    char *index;
    int firstchar = 1;
    int lastdash = 0;
       
    if(!syslog_tag)
        return 0;

    if(strlen(syslog_tag) > 254)
        return 0;

    /* check characters */
    index = syslog_tag;
    while(*index)
    {
        if(!isalnum(*index))
        {
            /* check for '-' */
            if(*index == '-')
            {
                if(firstchar)
                    return 0;
                lastdash = 1;
            }
            else
                return 0;
        }
        else
        {
            lastdash = 0;
            firstchar = 0;
        }
        index++;
    }

    if(firstchar || lastdash)
        return 0;
    
    return 1;
}
