/***************************************************************************
 *   Copyright (C) 2001 by Ludovic LANGE                                   *
 *   ludovic.lange@free.fr                                                 *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
/*
 * rapi.c
 * (c) 2001 Ludovic LANGE,
 * except SockOpen, (c) Unknown ? from socket.c / socket.h
 */

#include "config.h"

#undef VERSION

#define MIN(a,b) ((a)<(b)?(a):(b))

#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <errno.h>

#include "little_endian.h"

#include "chunk.h"
#include "rapi.h"

#define BUFSIZE 16384


/*
 * Push parameter information to the buffer
 *
 * DWORD parameter values must be 4 byte little-endian
 * WORD parameter values must be 2 byte little-endian
 * WCHAR parameter values must be 2-byte UNICODE (is it endian stuff on these too?) 
 */
static void pushParameter(RAPIHandleP rhnd,long size, void * parameterData, size_t parameterSize, int pushValue)
{
	if (parameterData)
	{
		pushLong( (rhnd->buffer), size, 1 );
		pushLong( (rhnd->buffer), size, parameterSize);

		if (pushValue)	/* pushValue is boolean */
		{
			long old_length;
			long new_length;

			pushLong( (rhnd->buffer), size, 1 );	/* data follows */

			old_length = _getbufferlen((rhnd->buffer));
			new_length = old_length + parameterSize;

			if (new_length < size)
			{
				memcpy(((unsigned char*)&((rhnd->buffer)->data)) + old_length, parameterData, parameterSize);
				_setbufferlen((rhnd->buffer), new_length);
			}
			else
			{
				DBG_printf( "rapibuffer too small, this will fail\n");
			}
		}
		else
		{
			pushLong( (rhnd->buffer), size, 0 );	/* no data follows */
		}
	}
	else
	{
		pushLong( (rhnd->buffer), size, 0 ); /* no parameter info */
	}
}

/*
 * Push parameter information to the buffer, omitting size information (needed for CeRecordWriteProps
 */
static void pushParameter_no_size(RAPIHandleP rhnd,long size, void * parameterData, size_t parameterSize)
{
	long old_length;
	long new_length;

	if (parameterData)
	{
		pushLong( (rhnd->buffer), size, 1 );	/* data follows */
	
		old_length = _getbufferlen((rhnd->buffer));
		new_length = old_length + parameterSize;
	
		if (new_length < size)
		{
			memcpy(((unsigned char*)&((rhnd->buffer)->data)) + old_length, parameterData, parameterSize);
			_setbufferlen((rhnd->buffer), new_length);
		}
		else
		{
			DBG_printf( "rapibuffer too small, this will fail\n");
		}
	}
	else
	{
		pushLong( (rhnd->buffer), size, 0 );	/* no data follows */
	}	
}




/**
 * Convert parameter to little endian before call to pushParameter
 */
#define pushParameterInt16(rhnd,size, parameterData, pushValue) \
if (pushValue && parameterData) \
	{ *(u_int16_t*)parameterData = htole16(*(u_int16_t*)parameterData); } \
pushParameter(rhnd,size, parameterData, sizeof(u_int16_t), pushValue);

#define pushParameterInt32(rhnd,size, parameterData, pushValue) \
if (pushValue && parameterData) \
	{ *(u_int32_t*)parameterData = htole32(*(u_int32_t*)parameterData); } \
pushParameter(rhnd,size, parameterData, sizeof(u_int32_t), pushValue)


/**
 * Pop parameter info from the buffer
 */
static void popParameter(RAPIHandleP rhnd,long* size, void * parameterData, size_t parameterMaxSize)
{
	long lng;
	
	lng = getLong((rhnd->sock), size);	/* parameter info available? */

	if (1 == lng)
	{
		size_t parameterRealSize = getLong((rhnd->sock), size);	/* parameter value size in buffer */

		lng = getLong((rhnd->sock), size); /* parameter value available? */

		if (1 == lng)
		{
			int overflow; 

			if (parameterData)
			{
				/* read directly to buffer */
				getbufferchunk((rhnd->sock), size, parameterData, MIN(parameterRealSize, parameterMaxSize));
				overflow = parameterRealSize - parameterMaxSize;
			}
			else
			{
				/* throw it all away */
				overflow = parameterRealSize;
			}
				
			if (overflow > 0)
			{
				if (parameterData)
					DBG_printf("Overflow by %i bytes. Parameter size is %i bytes but max %i bytes was expected. (%i bytes remaining)\n",
							overflow, parameterRealSize, parameterMaxSize , *size);

				{
					/* read overflowed buffer */
					void* tmp = malloc(overflow);
					getbufferchunk((rhnd->sock), size, tmp, overflow);
					free(tmp);
				}

			}
		}
		else if (0 != lng)
		{
			DBG_printf("popParameter (a): Expected 0 or 1 but got %i=0x%x (%i bytes remaining)\n", lng, lng, *size);
		}
	}
	else if (0 != lng)
	{
		DBG_printf("popParameter (b): Expected 0 or 1 but got %i=0x%x (%i bytes remaining)\n", lng, lng, *size);
	}
}

/**
 * Convert parameter data from little endian after call to popParameter
 */
#define popParameterInt16(rhnd,size, parameterData) \
popParameter(rhnd,size, parameterData, sizeof(u_int16_t)); \
if (parameterData) \
	{ *(u_int16_t*)parameterData = letoh16(*(u_int16_t*)parameterData); }

#define popParameterInt32(rhnd,size, parameterData) \
popParameter(rhnd,size, parameterData, sizeof(u_int32_t)); \
if (parameterData) \
	{ *(u_int32_t*)parameterData = letoh32(*(u_int32_t*)parameterData); }

void checkpassword(RAPIHandleP rhnd)
{
	char buf[4];
	/* Is the device protected ? */
	/* ------------------------- */
	if( (rhnd->lockbuffersize) != 0 )
	{
		safe_write( (rhnd->sock), (rhnd->lockbuffer), (rhnd->lockbuffersize) );
		read( (rhnd->sock), buf, 1 );
	}
}
/*=================================================================================================================*
 *=================================================================================================================*
 * RAPI - Global
 *=================================================================================================================*
 *=================================================================================================================*/

STDAPI_( DWORD ) CeGetSpecialFolderPath(RAPIHandleP rhnd, int nFolder, DWORD nBufferLength, LPWSTR lpBuffer )
{
	long size = BUFSIZE;
	long lng;
	WCHAR * str;

//	checkpassword();

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x44 ); 		/* Command */
	pushLong( (rhnd->buffer), size, nFolder ); 		/* Parameter1 : the folder */
	pushLong( (rhnd->buffer), size, nBufferLength ); 	/* Parameter2 : Buffer size that'll get the string */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );

	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 : %ld (0x%08lx)\n", lng, lng );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "string size : %ld (0x%08lx)\n", lng, lng );
	str = getString( (rhnd->sock), &size, lng+1 );
	DBG_printf( "string1 : %s\n", str );
	if ( lpBuffer )
	{
		memcpy( lpBuffer, str, MIN( ((lng+1)*sizeof(WCHAR) ), nBufferLength*sizeof(WCHAR) ) );
	}
	free(str);
	return lng;
}

/* ================================================================================================================= */
/* ================================================================================================================= */
/*  RAPI - Registry */
/* ================================================================================================================= */
/* ================================================================================================================= */

/*
STDAPI_(LONG) CeRegDeleteKey( HKEY, LPCWSTR );
STDAPI_(LONG) CeRegDeleteValue( HKEY, LPCWSTR );
STDAPI_(LONG) CeRegQueryValueEx( HKEY, LPCWSTR, LPDWORD, LPDWORD, LPBYTE, LPDWORD );
STDAPI_(LONG) CeRegSetValueEx( HKEY, LPCWSTR, DWORD, DWORD, LPBYTE, DWORD );
*/

STDAPI_( LONG ) CeRegCreateKeyEx(RAPIHandleP rhnd, HKEY hKey, LPCWSTR lpszSubKey, DWORD Reserved, LPWSTR lpszClass, DWORD ulOptions, REGSAM samDesired, LPSECURITY_ATTRIBUTES lpSecurityAttributes, PHKEY phkResult, LPDWORD lpdwDisposition )
{
	long size = BUFSIZE;
	long lng;
	long errcode;
	LONG result = ERROR_SUCCESS; /* May be result is really errcode. */

	DBG_printf( "CeRegCreatKeyEx( hKey = 0x%08X, lpszSubKey = 0x%08X, Reserved = 0x%08X, lpszClass = 0x%08X, ulOptions = 0x%08X, samDesired = 0x%08X, lpSecurityAttributes = 0x%08X, phkResult = 0x%08X, lpdwDisposition = 0x%08X )\n",
	            hKey, lpszSubKey, Reserved, lpszClass, ulOptions, samDesired, lpSecurityAttributes, phkResult, lpdwDisposition );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x20 ); 			/* Command */
	pushLong( (rhnd->buffer), size, hKey ); 			/* Parameter1 : */
	pushLong( (rhnd->buffer), size, 0x01 ); 			/* Parameter2 : */
	pushLong( (rhnd->buffer), size, 1 + wcslen( lpszSubKey ) ); 	/* Parameter3 : */
	pushString( (rhnd->buffer), size, lpszSubKey ); 		/* Parameter4 : the path */
	pushLong( (rhnd->buffer), size, 0x01 ); 			/* Parameter5 : */
	pushLong( (rhnd->buffer), size, 0x01 ); 			/* Parameter6 : */
	pushShort( (rhnd->buffer), size, 0x00 );		 	/* Parameter7 : */

	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );

	errcode = getLong( (rhnd->sock), &size );
	DBG_printf( "errpresent : %ld (0x%08lx)\n", errcode, errcode );
	if ( errcode != 0 )
	{
		errcode = getLong( (rhnd->sock), &size );
		DBG_printf( "errcode : %ld (0x%08lx)\n", errcode, errcode );
	}
	else
	{
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long 2 : %ld (0x%08lx)\n", lng, lng );
		*phkResult = ( HKEY ) getLong( (rhnd->sock), &size );
		DBG_printf( "pHkey : %ld (0x%08lx)\n", *phkResult, *phkResult );
		*lpdwDisposition = ( DWORD ) getLong( (rhnd->sock), &size );
		DBG_printf( "lpdwDisposition : %ld (0x%08lx)\n", *lpdwDisposition, *lpdwDisposition );
	}
	return result;
}

STDAPI_( LONG ) CeRegOpenKeyEx(RAPIHandleP rhnd, HKEY hKey, LPCWSTR lpszSubKey, DWORD ulOptions, REGSAM samDesired, PHKEY phkResult )
{
	long size = BUFSIZE;
	long errcode;

	DBG_printf( "CeRegOpenKeyEx( hKey = 0x%08X, lpszSubKey = 0x%08X, ulOptions = 0x%08X, samDesired = 0x%08X, phkResult = 0x%08X )\n",
	            hKey, lpszSubKey, ulOptions, samDesired, phkResult );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x1E ); 	/* Command */
	pushLong( (rhnd->buffer), size, hKey ); 		/* Parameter1 : */
#if 0
	pushLong( (rhnd->buffer), size, 0x01 ); 	/* Parameter2 : */
	pushLong( (rhnd->buffer), size, 1 + wcslen( lpszSubKey ) ); 	/* Parameter3 : */
	pushString( (rhnd->buffer), size, lpszSubKey ); 	/* Parameter4 : the path */
#else
	pushParameter(rhnd,size, lpszSubKey, lpszSubKey ? (wcslen(lpszSubKey) + 1) * sizeof(WCHAR) : 0, 1);
	pushLong( (rhnd->buffer), size, ulOptions);
	pushLong( (rhnd->buffer), size, samDesired);
	pushParameterInt32(rhnd,size, phkResult, 0);
#endif
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );

	errcode = getLong( (rhnd->sock), &size );
	DBG_printf( "errcode : %ld (0x%08lx)\n", errcode, errcode );

	errcode = getLong( (rhnd->sock), &size );
	DBG_printf( "errpresent : %ld (0x%08lx)\n", errcode, errcode );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long : %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );
	if ( errcode == 0 )
	{
		*phkResult = ( HKEY ) getLong( (rhnd->sock), &size );
		DBG_printf( "pHkey : %ld (0x%08lx)\n", *phkResult, *phkResult );
	}
	return errcode;
}

STDAPI_( LONG ) CeRegCloseKey( RAPIHandleP rhnd,HKEY hKey )
{
	long size = BUFSIZE;
	long lng;
	long result;

	DBG_printf( "CeRegCloseKey( hKey = 0x%08X )\n",
	            hKey );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x21 ); 	/* Command */
	pushLong( (rhnd->buffer), size, hKey ); 		/* Parameter1 : */

	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );
	size = getbufferlen( (rhnd->sock) );

	result = getLong( (rhnd->sock), &size );
	DBG_printf( "result : %ld (0x%08lx)\n", result, result );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 : %ld (0x%08lx)\n", lng, lng );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 3 : %ld (0x%08lx)\n", lng, lng );
	return result;
}

STDAPI_( LONG ) CeRegQueryInfoKey(RAPIHandleP rhnd, HKEY hKey, LPWSTR lpClass, LPDWORD lpcbClass, LPDWORD lpReserved, LPDWORD lpcSubKeys, LPDWORD lpcbMaxSubKeyLen, LPDWORD lpcbMaxClassLen, LPDWORD lpcValues, LPDWORD lpcbMaxValueNameLen, LPDWORD lpcbMaxValueLen, LPDWORD lpcbSecurityDescriptor, PFILETIME lpftLastWriteTime )
{
	long size = BUFSIZE;
	long lng;
	LONG result = ERROR_SUCCESS;

#if 0
	DBG_printf( "CeRegQueryInfoKey( hKey = 0x%08X, lpClass = 0x%08X, lpcbClass = 0x%08X, lpReserved = 0x%08X, lpcSubKeys = 0x%08X, lpcbMaxSubKeyLen = 0x%08X, lpcbMaxClassLen = 0x%08X, lpcValues = 0x%08X, lpcbMaxValueNameLen = 0x%08X, lpcbMaxValueLen = 0x%08X, lpcbSecurityDescriptor = 0x%08X, lpftLastWriteTime = 0x%08X )\n",
	            hKey, lpClass, lpcbClass, lpReserved, lpcSubKeys, lpcbMaxSubKeyLen, lpcbMaxClassLen, lpcValues, lpcbMaxValueNameLen, lpcbMaxValueLen, lpcbSecurityDescriptor, lpftLastWriteTime );
#endif

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x25 ); 	/* Command */
	pushLong( (rhnd->buffer), size, hKey ); 	/* hKey */

	pushParameter(rhnd,size, lpClass, lpcbClass ? *lpcbClass * sizeof(WCHAR): 0, 0);
	pushParameterInt32(rhnd,size, lpcbClass, 1);
	pushParameterInt32(rhnd,size, lpReserved, 0);
	pushParameterInt32(rhnd,size, lpcSubKeys, 0);
	pushParameterInt32(rhnd,size, lpcbMaxSubKeyLen, 0);
	pushParameterInt32(rhnd,size, lpcbMaxClassLen, 0);
	pushParameterInt32(rhnd,size, lpcValues, 0);
	pushParameterInt32(rhnd,size, lpcbMaxValueNameLen, 0);
	pushParameterInt32(rhnd,size, lpcbMaxValueLen, 0);
	pushParameterInt32(rhnd,size, lpcbSecurityDescriptor, 1);
	if (lpftLastWriteTime)
	{
		lpftLastWriteTime->dwLowDateTime = (DWORD)htole32(lpftLastWriteTime->dwLowDateTime);
		lpftLastWriteTime->dwHighDateTime = (DWORD)htole32(lpftLastWriteTime->dwHighDateTime);
	}
	pushParameter(rhnd,size, lpftLastWriteTime, sizeof(FILETIME), 0);

/*	DBG_printbuf( (rhnd->buffer) );*/
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );

	result = lng = getLong( (rhnd->sock), &size );
	(rhnd->_lasterror) = lng = getLong( (rhnd->sock), &size );

	if (ERROR_SUCCESS == result)
	{
		popParameter(rhnd,&size, lpClass, lpcbClass ? *lpcbClass: 0);
		popParameterInt32(rhnd,&size, lpcbClass);
		popParameterInt32(rhnd,&size, lpReserved);
		popParameterInt32(rhnd,&size, lpcSubKeys);
		popParameterInt32(rhnd,&size, lpcbMaxSubKeyLen);
		popParameterInt32(rhnd,&size, lpcbMaxClassLen);
		popParameterInt32(rhnd,&size, lpcValues);
		popParameterInt32(rhnd,&size, lpcbMaxValueNameLen);
		popParameterInt32(rhnd,&size, lpcbMaxValueLen);
		popParameterInt32(rhnd,&size, lpcbSecurityDescriptor);
		popParameter(rhnd,&size, lpftLastWriteTime, sizeof(FILETIME));
		if (lpftLastWriteTime)
		{
			lpftLastWriteTime->dwLowDateTime = (DWORD)letoh32(lpftLastWriteTime->dwLowDateTime);
			lpftLastWriteTime->dwHighDateTime = (DWORD)letoh32(lpftLastWriteTime->dwHighDateTime);
		}
	}

	if ( size > 0 )
	{
		DBG_printf( "size : %d\n", size );
/*		flushbuffer( (rhnd->sock) );*/
	}
	return result;
}

STDAPI_( LONG ) CeRegEnumValue(RAPIHandleP rhnd, HKEY hKey, DWORD dwIndex, LPWSTR lpszValueName, LPDWORD lpcbValueName, LPDWORD lpReserved, LPDWORD lpType, LPBYTE lpData, LPDWORD lpcbData )
{
	LONG result = ERROR_SUCCESS;
	long size = BUFSIZE;

#if 0
	DBG_printf( "CeRegEnumValue( hKey = 0x%08X, dwIndex = 0x%08X, lpszValueName = 0x%08X, lpcbValueName = 0x%08X, lpReserved = 0x%08X, lpType = 0x%08X, lpData = 0x%08X, lpcbData = 0x%08X )\n",
	            hKey, dwIndex, lpszValueName, ( *lpcbValueName ), lpReserved, ( *lpType ), ( *lpData ), ( *lpcbData ) );
#endif

	/* maybe check more parameters */
	if (!lpcbValueName || lpReserved)
		return 87; /* ERROR_INVALID_PARAMETER */
		
	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x23 ); 	/* Command */
	pushLong( (rhnd->buffer), size, hKey ); 	/* Parameter1 : */
	pushLong( (rhnd->buffer), size, dwIndex ); 	/* Parameter2 : */

	pushParameter(rhnd,size, lpszValueName, lpcbValueName ? *lpcbValueName * sizeof(WCHAR): 0, 0);
	pushParameterInt32(rhnd,size, lpcbValueName, 1);
	pushParameterInt32(rhnd,size, lpReserved, 1);
	pushParameterInt32(rhnd,size, lpType, 1);
	pushParameter(rhnd,size, lpData, lpcbData ? *lpcbData : 0, 0);
	pushParameterInt32(rhnd,size, lpcbData, 1);

/*	DBG_printbuf( (rhnd->buffer) );*/
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );
	
	result = getLong( (rhnd->sock), &size ); 
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );

	if (ERROR_SUCCESS == result)
	{
		/* here comes a result again! */
		result = getLong( (rhnd->sock), &size );
	}
		
	if (ERROR_SUCCESS == result)
	{
		popParameter(rhnd,&size, lpszValueName, lpcbValueName ? *lpcbValueName : 0);
		popParameterInt32(rhnd,&size, lpcbValueName);
		popParameterInt32(rhnd,&size, lpType);
		popParameter(rhnd,&size, lpData, lpcbData ? *lpcbData : 0);
		popParameterInt32(rhnd,&size, lpcbData);
	}

	if ( size > 0 )
	{
		DBG_printf( "size : %d\n", size );
/*		flushbuffer( (rhnd->sock) );*/
	}
	return result;
}

STDAPI_( LONG ) CeRegEnumKeyEx(RAPIHandleP rhnd, HKEY hKey, DWORD dwIndex, LPWSTR lpName, LPDWORD lpcbName, LPDWORD lpReserved, LPWSTR lpClass, LPDWORD lpcbClass, PFILETIME lpftLastWriteTime )
{
	long size = BUFSIZE;
	long lng;
	WCHAR * str;
	int i;
	long errcode;
	long strsz;
	long maxsz;

	DBG_printf( "CeRegEnumKeyEx( hKey = 0x%08X, dwIndex = 0x%08X, lpcbName = 0x%08X, lpReserved = 0x%08X, lpClass = 0x%08X, lpcbClass = 0x%08X, lpftLastWriteTime = 0x%08X )\n",
	            hKey, dwIndex, lpName, lpcbName ? ( (void*)*lpcbName ) : lpcbName, lpReserved, lpClass, lpcbClass ? ( (void*)*lpcbClass ) : lpcbClass, lpftLastWriteTime );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x1F ); 	/* Command */
	pushLong( (rhnd->buffer), size, hKey ); 	/* Parameter1 : */
	pushLong( (rhnd->buffer), size, dwIndex ); 	/* Parameter2 : */
	pushLong( (rhnd->buffer), size, 0x01 ); 	/* Parameter3 : */
	pushLong( (rhnd->buffer), size, 0x0202 ); 	/* Parameter4 : */
	pushLong( (rhnd->buffer), size, 0x00 ); 	/* Parameter5 : */
	pushLong( (rhnd->buffer), size, 0x01 ); 	/* Parameter6 : */
	pushLong( (rhnd->buffer), size, 0x04 ); 	/* Parameter7 : */
	pushLong( (rhnd->buffer), size, 0x01 ); 	/* Parameter8 : */
	pushLong( (rhnd->buffer), size, lpcbName ? ( *lpcbName ) : 0 ); 	/* Parameter9 : */
	pushLong( (rhnd->buffer), size, 0x00 ); 	/* Parameter10 : */
	pushLong( (rhnd->buffer), size, 0x00 ); 	/* Parameter11 : */
	pushLong( (rhnd->buffer), size, 0x00 ); 	/* Parameter12 : */
	pushLong( (rhnd->buffer), size, 0x00 ); 	/* Parameter13 : */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long : %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long : %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );
	errcode = getLong( (rhnd->sock), &size );
	DBG_printf( "errcode : %ld (0x%08lx)\n", errcode, errcode );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long : %ld (0x%08lx)\n", lng, lng );
	if ( errcode == 0 )
	{
		strsz = getLong( (rhnd->sock), &size );
		DBG_printf( "long : %ld (0x%08lx)\n", strsz, strsz );
		maxsz = 0;
		if ( lpcbName != NULL )
		{
			maxsz = ( *lpcbName );
			( *lpcbName ) = strsz - 1;
		}
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long : %ld (0x%08lx)\n", lng, lng );
		str = getString( (rhnd->sock), &size, strsz >> 1 );
		DBG_printf( "string1 : %s\n", str );
		if ( ( lpName != NULL ) )
		{
			memcpy( lpName, str, MIN( strsz, maxsz ) );
		}
		free(str);
		for ( i = 0; i < 7; i++ )
		{
			lng = getLong( (rhnd->sock), &size );
			DBG_printf( "long %d : %ld (0x%08lx)\n", i, lng, lng );
		}
	}

	return errcode;
}

/* ================================================================================================================= */
/* ================================================================================================================= */
/*  RAPI - Tests */
/* ================================================================================================================= */
/* ================================================================================================================= */

#if 0
DWORD CeTestSec1()
{
	long size=BUFSIZE;
	long lng;
	int result;
	long index;
 
	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x27 );	/* Command */
	pushLong( (rhnd->buffer), size, 0x80000002 );		/* Parameter1 : */
	pushLong( (rhnd->buffer), size, 0x01 );	/* Parameter2 : */
	pushLong( (rhnd->buffer), size, 0x20 );	/* Parameter3 : */
	pushString( (rhnd->buffer), size, "Comm\\SecurityProviders\\SCHANNEL" );	/* Parameter4 : the path */
	pushLong( (rhnd->buffer), size, 0x01 );	/* Parameter2 : */
	pushLong( (rhnd->buffer), size, 0x01 );	/* Parameter2 : */
	pushShort( (rhnd->buffer), size, 0x00 );	/* Parameter2 : */
 
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );
	result = readbuffer( (rhnd->sock), (rhnd->buffer), size );
	DBG_printbuf( (rhnd->buffer) );
	index = 0;
	return 2*lng;
}

DWORD CeTest0()
{
	long size=BUFSIZE;
	long lng;
	int result;
	long index;
	
	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x1E );	/* Command */
	pushLong( (rhnd->buffer), size, 0x80000000 );		/* Parameter1 : */
	pushLong( (rhnd->buffer), size, 0x01 );	/* Parameter2 : */
	pushLong( (rhnd->buffer), size, 0x27 );	/* Parameter3 : */
	pushString( (rhnd->buffer), size, "{000214A0-0000-0000-C000-000000000046}" );	/* Parameter3 : the path */
 
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );
	result = readbuffer( (rhnd->sock), (rhnd->buffer), size );
	DBG_printbuf( (rhnd->buffer) );
	index = 0;
	return 2*lng;
}
#endif

/* ================================================================================================================= */
/* ================================================================================================================= */
/*  RAPI - Files */
/* ================================================================================================================= */
/* ================================================================================================================= */

STDAPI_( BOOL ) CeFindAllFiles(RAPIHandleP rhnd, LPCWSTR szPath, DWORD dwFlags, LPDWORD lpdwFoundCount, LPLPCE_FIND_DATA ppFindDataArray )
{
	long size = BUFSIZE;
	long lng;
	WCHAR * str;
	long stlen = 0;
	long i;
	CE_FIND_DATA *ptr;

	DBG_printf( "CeFindAllFiles : szPath = 0x%08X, dwFlags = 0x%08X, lpdwFoundCount = 0x%08X, ppFindDataArray = 0x%08X\n", szPath, dwFlags, lpdwFoundCount, ppFindDataArray );

//	checkpassword();

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x09 ); 			/* Command */
	pushLong( (rhnd->buffer), size, 0x01 ); 			/* Parameter1 : */
	pushLong( (rhnd->buffer), size, 1 + wcslen( szPath ) ); 	/* Parameter2 : */
	pushString( (rhnd->buffer), size, szPath ); 			/* Parameter3 : the path */
	pushLong( (rhnd->buffer), size, dwFlags ); 			/* Parameter4 : Flags ? */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 (nb answer) : %ld (0x%08lx)\n", lng, lng );
	if ( lpdwFoundCount )
	{
		*lpdwFoundCount = lng;
	}
	if ( ppFindDataArray )
	{
		DBG_printf( "Before allocation\n" );
		*ppFindDataArray = ( LPCE_FIND_DATA ) calloc( lng, sizeof( CE_FIND_DATA ) );
		DBG_printf( "After allocation : *ppFindDataArray = %08X\n", *ppFindDataArray );
		if ( ( *ppFindDataArray ) )
		{
			for ( i = 0; i < lng; i++ )
			{
				ptr = &( ( *ppFindDataArray ) [ i ] );
				DBG_printf( "i=%d : ptr=%08X\n", i, ptr );
				if ( dwFlags & FAF_NAME )
				{
					stlen = getLong( (rhnd->sock), &size );
					DBG_printf( "string size : %ld (0x%08lx)\n", stlen, stlen );
				}
				if ( dwFlags & FAF_ATTRIBUTES )
				{
					ptr->dwFileAttributes = getLong( (rhnd->sock), &size );
					DBG_printf( "ptr->dwFileAttributes : %ld (0x%08lx)\n", ptr->dwFileAttributes, ptr->dwFileAttributes );
				}
				if ( dwFlags & FAF_CREATION_TIME )
				{
					ptr->ftCreationTime.dwLowDateTime = getLong( (rhnd->sock), &size );
					DBG_printf( "ptr->ftCreationTime.dwLowDateTime : %ld (0x%08lx)\n", ptr->ftCreationTime.dwLowDateTime, ptr->ftCreationTime.dwLowDateTime );
					ptr->ftCreationTime.dwHighDateTime = getLong( (rhnd->sock), &size );
					DBG_printf( "ptr->ftCreationTime.dwHighDateTime : %ld (0x%08lx)\n", ptr->ftCreationTime.dwHighDateTime, ptr->ftCreationTime.dwHighDateTime );
				}
				if ( dwFlags & FAF_LASTACCESS_TIME )
				{
					ptr->ftLastAccessTime.dwLowDateTime = getLong( (rhnd->sock), &size );
					DBG_printf( "ptr->ftLastAccessTime.dwLowDateTime : %ld (0x%08lx)\n", ptr->ftLastAccessTime.dwLowDateTime, ptr->ftLastAccessTime.dwLowDateTime );
					ptr->ftLastAccessTime.dwHighDateTime = getLong( (rhnd->sock), &size );
					DBG_printf( "ptr->ftLastAccessTime.dwHighDateTime : %ld (0x%08lx)\n", ptr->ftLastAccessTime.dwHighDateTime, ptr->ftLastAccessTime.dwHighDateTime );
				}
				if ( dwFlags & FAF_LASTWRITE_TIME )
				{
					ptr->ftLastWriteTime.dwLowDateTime = getLong( (rhnd->sock), &size );
					DBG_printf( "ptr->ftLastWriteTime.dwLowDateTime : %ld (0x%08lx)\n", ptr->ftLastWriteTime.dwLowDateTime, ptr->ftLastWriteTime.dwLowDateTime );
					ptr->ftLastWriteTime.dwHighDateTime = getLong( (rhnd->sock), &size );
					DBG_printf( "ptr->ftLastWriteTime.dwHighDateTime : %ld (0x%08lx)\n", ptr->ftLastWriteTime.dwHighDateTime, ptr->ftLastWriteTime.dwHighDateTime );
				}
				if ( dwFlags & FAF_SIZE_HIGH )
				{
					ptr->nFileSizeHigh = getLong( (rhnd->sock), &size );
					DBG_printf( "ptr->nFileSizeHigh : %ld (0x%08lx)\n", ptr->nFileSizeHigh, ptr->nFileSizeHigh );
				}
				if ( dwFlags & FAF_SIZE_LOW )
				{
					ptr->nFileSizeLow = getLong( (rhnd->sock), &size );
					DBG_printf( "ptr->nFileSizeLow : %ld (0x%08lx)\n", ptr->nFileSizeLow, ptr->nFileSizeLow );
				}
				if ( dwFlags & FAF_OID )
				{
					ptr->dwOID = getLong( (rhnd->sock), &size );
					DBG_printf( "ptr->dwOID : %ld (0x%08lx)\n", ptr->dwOID, ptr->dwOID );
				}
				if ( dwFlags & FAF_NAME )
				{
					str = getString( (rhnd->sock), &size, stlen );
					memcpy( ptr->cFileName, str, MAX_PATH );
					DBG_printf( "ptr->cFileName : %s\n", ptr->cFileName );
					free(str);
				}
			}
		}
	}
	return TRUE;
}

STDAPI_( HANDLE ) CeFindFirstFile(RAPIHandleP rhnd, LPCWSTR lpFileName, LPCE_FIND_DATA lpFileFindData )
{
	long size = BUFSIZE;
	long lng;
	HANDLE result;
	size_t stlen;
	/* CEDB_FIND_DATA *ptr; */

	DBG_printf( "CeFindFirstFile( lpFileName = 0x%08X, lpFileFindData = 0x%08X )\n",
	            lpFileName, lpFileFindData );

	initBuf( (rhnd->buffer), size );
        DBG_printf(" line : %d\n", __LINE__ );
	pushLong( (rhnd->buffer), size, 0x00 ); 		/* Command */
        DBG_printf(" line : %d\n", __LINE__ );
	pushLong( (rhnd->buffer), size, 0x01 ); 		/* Parameter1 : */
        DBG_printf(" line : %d\n", __LINE__ );
	if ( lpFileName )
	{
		stlen = wcslen( lpFileName );
	}
	else
	{
		stlen = -1;
	}
	DBG_printf( "size : %d\n", stlen );
	pushLong( (rhnd->buffer), size, 1 + stlen ); 		/* Parameter2 : Flags ? */
	pushString( (rhnd->buffer), size, lpFileName ); 	/* Parameter3 : the path */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );

	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 (errorcode?): %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );
	result = ( HANDLE ) getLong( (rhnd->sock), &size );
	DBG_printf( "long 3 : (size=%d) HANDLE %ld (0x%08lx)\n", size, result, result );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 4 : %ld (0x%08lx)\n", lng, lng );
	if ( lpFileFindData )
	{
		lpFileFindData->dwFileAttributes = getLong( (rhnd->sock), &size );
		getbufferchunk( (rhnd->sock), &size, &( lpFileFindData->ftCreationTime ), sizeof( FILETIME ) );
		getbufferchunk( (rhnd->sock), &size, &( lpFileFindData->ftLastAccessTime ), sizeof( FILETIME ) );
		getbufferchunk( (rhnd->sock), &size, &( lpFileFindData->ftLastWriteTime ), sizeof( FILETIME ) );
		lpFileFindData->nFileSizeHigh = getLong( (rhnd->sock), &size );
		lpFileFindData->nFileSizeLow = getLong( (rhnd->sock), &size );
		lpFileFindData->dwOID = getLong( (rhnd->sock), &size );
		memset( &( lpFileFindData->cFileName ), 0, MAX_PATH );
		getbufferchunk( (rhnd->sock), &size, &( lpFileFindData->cFileName ), size );
	}

	DBG_printf( "size : %d\n", size );
	if ( size > 0 )
	{
		flushbuffer( (rhnd->sock) );
	}
	DBG_printf( "long 5 : %ld (0x%08lx)\n", lpFileFindData->dwFileAttributes, lpFileFindData->dwFileAttributes );
	return result;
}

STDAPI_( BOOL ) CeFindNextFile(RAPIHandleP rhnd, HANDLE hFindFile,  /*LPWIN32_FIND_DATA ?*/LPCE_FIND_DATA lpFileFindData )
{
	long size = BUFSIZE;
	long lng;
	int result;
	/* CEDB_FIND_DATA *ptr; */

	DBG_printf( "CeFindNextFile( hFindFile = 0x%08X, lpFileFindData = 0x%08X )\n",
	            hFindFile, lpFileFindData );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x01 ); 		/* Command */
	pushLong( (rhnd->buffer), size, ( long ) hFindFile ); 	/* Parameter1 : */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );

	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 (errorcode?): %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 3 (size=%d): %ld (0x%08lx)\n", size, lng, lng );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 4 : %ld (0x%08lx)\n", lng, lng );
	if ( lpFileFindData )
	{
		lpFileFindData->dwFileAttributes = getLong( (rhnd->sock), &size );
		getbufferchunk( (rhnd->sock), &size, &( lpFileFindData->ftCreationTime ), sizeof( FILETIME ) );
		getbufferchunk( (rhnd->sock), &size, &( lpFileFindData->ftLastAccessTime ), sizeof( FILETIME ) );
		getbufferchunk( (rhnd->sock), &size, &( lpFileFindData->ftLastWriteTime ), sizeof( FILETIME ) );
		lpFileFindData->nFileSizeHigh = getLong( (rhnd->sock), &size );
		lpFileFindData->nFileSizeLow = getLong( (rhnd->sock), &size );
		lpFileFindData->dwOID = getLong( (rhnd->sock), &size );
		memset( &( lpFileFindData->cFileName ), 0, MAX_PATH );
		getbufferchunk( (rhnd->sock), &size, &( lpFileFindData->cFileName ), size );
	}
	DBG_printf( "size : %d\n", size );

	if ( size > 0 )
	{
		flushbuffer( (rhnd->sock) );
	}
	result = ( (rhnd->_lasterror) == 0 );
	return result;
}

STDAPI_( BOOL ) CeFindClose(RAPIHandleP rhnd, HANDLE hFindFile )
{
	long size = BUFSIZE;
	long lng;

	DBG_printf( "CeFindClose()\n" );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x02 ); 		/* Command */
	pushLong( (rhnd->buffer), size, ( long ) hFindFile );	/* Parameter1 : */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 (lasterror): %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 3 : %ld (0x%08lx)\n", lng, lng );

	if ( size > 0 )
	{
		DBG_printf( "size : %d\n", size );
		flushbuffer( (rhnd->sock) );
	}

	return lng;
}

STDAPI_( HANDLE ) CeCreateFile(RAPIHandleP rhnd, LPCWSTR lpFileName, DWORD dwDesiredAccess, DWORD dwShareMode, LPSECURITY_ATTRIBUTES lpSecurityAttributes, DWORD dwCreationDisposition, DWORD dwFlagsAndAttributes, HANDLE hTemplateFile )
{
	long size = BUFSIZE;
	long lng;
	HANDLE result;
	long stlen;

	DBG_printf( "CeCreateFile( lpFileName = 0x%08X, dwDesiredAccess = 0x%08X, dwShareMode = 0x%08X )\n",
	            lpFileName, dwDesiredAccess, dwShareMode );

//	checkpassword();

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x05 ); 			/* Command */
	pushLong( (rhnd->buffer), size, dwDesiredAccess ); 		/* Parameter1 */
	pushLong( (rhnd->buffer), size, dwShareMode ); 			/* Parameter2 */
	pushLong( (rhnd->buffer), size, dwCreationDisposition ); 	/* Parameter3 */
	pushLong( (rhnd->buffer), size, dwFlagsAndAttributes ); 	/* Parameter4 */
	pushLong( (rhnd->buffer), size, ( long ) hTemplateFile ); 	/* Parameter5 */
	pushLong( (rhnd->buffer), size, 1 ); 				/* Parameter6 */
	stlen = wcslen( lpFileName );
	pushLong( (rhnd->buffer), size, 1 + stlen ); 			/* Parameter7 */
	pushString( (rhnd->buffer), size, lpFileName ); 		/* Parameter8 */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );

	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 (errorcode?): %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );
	result = ( HANDLE ) getLong( (rhnd->sock), &size );
	DBG_printf( "long 3 (HANDLE): %ld (0x%08lx)\n", result, result );

	if ( size > 0 )
	{
		flushbuffer( (rhnd->sock) );
	}

	return result;
}

STDAPI_( BOOL ) CeReadFile(RAPIHandleP rhnd, HANDLE hFile, LPVOID lpBuffer, DWORD nNumberOfBytesToRead, LPDWORD lpNumberOfBytesRead, LPOVERLAPPED lpOverlapped )
{
	long size = BUFSIZE;
	long lng;
	BOOL result;
	/* CEDB_FIND_DATA *ptr; */

	DBG_printf( "CeReadFile( hFile = 0x%08X, nNumberOfBytesToRead = 0x%08X )\n",
	            hFile, nNumberOfBytesToRead );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x06 ); 		/* Command */
	pushLong( (rhnd->buffer), size, ( long ) hFile ); 	/* Parameter1 */
	pushLong( (rhnd->buffer), size, 0x01 ); 		/* Parameter2 */
	pushLong( (rhnd->buffer), size, nNumberOfBytesToRead ); /* Parameter3 */
	pushLong( (rhnd->buffer), size, 0x00 ); 		/* Parameter4 */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );

	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 (errorcode?): %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );
	result = ( BOOL ) getLong( (rhnd->sock), &size );
	DBG_printf( "long 3 (BOOL): %ld (0x%08lx)\n", result, result );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 4 (size): %ld (0x%08lx)\n", lng, lng );

	if ( lpNumberOfBytesRead )
	{
		*lpNumberOfBytesRead = lng;
	}
	if ( ( lpBuffer ) && ( lng > 0 ) )
	{
		getbufferchunk( (rhnd->sock), &size, lpBuffer, lng );
	}

	if ( size > 0 )
	{
		DBG_printf( "Size != 0 : size=%d\n", size );
		flushbuffer( (rhnd->sock) );
	}

	return result;
}

STDAPI_( BOOL ) CeWriteFile(RAPIHandleP rhnd, HANDLE hFile, LPVOID lpBuffer,
			     DWORD nNumberOfBytesToWrite,
			     LPDWORD lpNumberOfBytesWritten, 
			     LPOVERLAPPED lpOverlapped )
{
	long size = BUFSIZE;
	long lng;
	BOOL result;
	unsigned long buflen;

	
	/* CEDB_FIND_DATA *ptr; */

	DBG_printf( "CeWriteFile( hFile = 0x%08X, nNumberOfBytesToWrite = 0x%08X )\n",
	            hFile, nNumberOfBytesToWrite );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x07 ); 			/* Command */
	pushLong( (rhnd->buffer), size, ( long ) hFile ); 		/* Parameter1 */
	pushLong( (rhnd->buffer), size, 0x01 ); 			/* Parameter2 */
	pushLong( (rhnd->buffer), size, nNumberOfBytesToWrite ); 	/* Parameter3 */
	
	buflen=_getbufferlen((rhnd->buffer));
	_setbufferlen( (rhnd->buffer), buflen + nNumberOfBytesToWrite + 4);
	
	(void) safe_write((rhnd->sock), (void *) (rhnd->buffer), buflen + 4);

	(void) safe_write((rhnd->sock), lpBuffer, nNumberOfBytesToWrite);
	lng=0;
	(void) safe_write((rhnd->sock), (void *) &lng, sizeof(lng)); /*  ??? continue ??? */
	
	size = getbufferlen( (rhnd->sock) );

	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 (errorcode?): %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );

	result = ( BOOL ) getLong( (rhnd->sock), &size );
	DBG_printf( "long 3 (BOOL): %ld (0x%08lx)\n", result, result );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 4 (size): %ld (0x%08lx)\n", lng, lng );

	if ( lpNumberOfBytesWritten )
	{
		*lpNumberOfBytesWritten = lng;
	}
	if ( size > 0 )
	{
		flushbuffer( (rhnd->sock) );
	}

	return result;
}

/* ================================================================================================================= */
/* ================================================================================================================= */
/*  RAPI - Databases */
/* ================================================================================================================= */
/* ================================================================================================================= */


STDAPI_( BOOL ) CeFindAllDatabases(RAPIHandleP rhnd, DWORD dwDbaseType, WORD wFlags, LPWORD cFindData, LPLPCEDB_FIND_DATA ppFindData )
{
	long size = BUFSIZE;
	long lng;
	WCHAR * str;
	long stlen = 0;
	long i, j;
	WORD wrd;
	CEDB_FIND_DATA *ptr;

//	checkpassword();

	DBG_printf( "CeFindAllDatabases( dwDbaseType = 0x%08X, wFlags = 0x%04X, cFindData = 0x%08X, ppFindData = 0x%08X )\n",
	            dwDbaseType, wFlags, cFindData, ppFindData );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x2C ); 	/* Command */
	pushLong( (rhnd->buffer), size, dwDbaseType ); 	/* Parameter1 : */
	pushShort( (rhnd->buffer), size, wFlags ); 	/* Parameter2 : Flags ? */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 (errorcode?): %ld (0x%08lx)\n", lng, lng );
	wrd = getShort( (rhnd->sock), &size );
	DBG_printf( "word 1 : %d (0x%08lx)\n", wrd, wrd );
	if ( cFindData )
	{
		*cFindData = wrd;
	}
	if ( ppFindData && ( wrd > 0 ) )
	{
		DBG_printf( "Before allocation\n" );
		*ppFindData = ( LPCEDB_FIND_DATA ) calloc( wrd, sizeof( CEDB_FIND_DATA ) );
		DBG_printf( "After allocation : *ppFindData = %08X\n", *ppFindData );
		if ( ( *ppFindData ) )
		{
			for ( i = 0; i < wrd; i++ )
			{
				ptr = &( ( *ppFindData ) [ i ] );
				DBG_printf( "i=%d : ptr=%08X\n", i, ptr );
				if ( wFlags & FAD_OID )
				{
					ptr->OidDb = getLong( (rhnd->sock), &size );
					DBG_printf( "OidDb : %ld (0x%08lx)\n", ptr->OidDb, ptr->OidDb );
				}
				if ( wFlags & FAD_NAME )
				{
					stlen = getLong( (rhnd->sock), &size );
					DBG_printf( "string size : %ld (0x%08lx)\n", stlen, stlen );
				}
				if ( wFlags & FAD_FLAGS )
				{
					ptr->DbInfo.dwFlags = getLong( (rhnd->sock), &size );
					DBG_printf( "dwFlags : %ld (0x%08lx)\n", ptr->DbInfo.dwFlags, ptr->DbInfo.dwFlags );
				}
				if ( wFlags & FAD_NAME )
				{
					str = getString( (rhnd->sock), &size, stlen );
					memcpy( ptr->DbInfo.szDbaseName, str, sizeof( WCHAR ) * ( CEDB_MAXDBASENAMELEN > ( stlen ) ? CEDB_MAXDBASENAMELEN : ( stlen ) ) );
					DBG_printf( "ptr->DbInfo.szDbaseName : %s\n", ptr->DbInfo.szDbaseName );
					free(str);
				}
				if ( wFlags & FAD_TYPE )
				{
					ptr->DbInfo.dwDbaseType = getLong( (rhnd->sock), &size );
					DBG_printf( "dwDbaseType : %ld (0x%08lx)\n", ptr->DbInfo.dwDbaseType, ptr->DbInfo.dwDbaseType );
				}
				if ( wFlags & FAD_NUM_RECORDS )
				{
					ptr->DbInfo.wNumRecords = getShort( (rhnd->sock), &size );
					DBG_printf( "wNumRecords : %ld (0x%08lx)\n", ptr->DbInfo.wNumRecords, ptr->DbInfo.wNumRecords );
				}
				if ( wFlags & FAD_NUM_SORT_ORDER )
				{
					ptr->DbInfo.wNumSortOrder = getShort( (rhnd->sock), &size );
					DBG_printf( "wNumSortOrder : %ld (0x%08lx)\n", ptr->DbInfo.wNumSortOrder, ptr->DbInfo.wNumSortOrder );
				}
				if ( wFlags & FAD_SIZE )
				{
					ptr->DbInfo.dwSize = getLong( (rhnd->sock), &size );
					DBG_printf( "dwSize : %ld (0x%08lx)\n", ptr->DbInfo.dwSize, ptr->DbInfo.dwSize );
				}
				if ( wFlags & FAD_LAST_MODIFIED )
				{
					getFileTime( (rhnd->sock), &size, &( ptr->DbInfo.ftLastModified ) );
					DBG_printf( "ftLastModified : %ld (0x%08lx)\n", ptr->DbInfo.ftLastModified, ptr->DbInfo.ftLastModified );
				}
				if ( wFlags & FAD_SORT_SPECS )
				{
					for ( j = 0; j < CEDB_MAXSORTORDER; j++ )
					{
						getbufferchunk( (rhnd->sock), &size, &( ( ptr->DbInfo.rgSortSpecs ) [ j ] ), sizeof( SORTORDERSPEC ) );
						DBG_printf( "sortOrder[%d] : %ld (0x%08lx)\n", j, ( ptr->DbInfo.rgSortSpecs ) [ j ].propid, ( ptr->DbInfo.rgSortSpecs ) [ j ].propid );
						DBG_printf( "sortOrder[%d] : %ld (0x%08lx)\n", j, ( ptr->DbInfo.rgSortSpecs ) [ j ].dwFlags, ( ptr->DbInfo.rgSortSpecs ) [ j ].dwFlags );
					}
				}
			}
		}
	}
	return TRUE;
}

STDAPI_( HANDLE ) CeOpenDatabase(RAPIHandleP rhnd, PCEOID poid, LPWSTR lpszName, CEPROPID propid, DWORD dwFlags, HWND hwndNotify )
{
	long size = BUFSIZE;
	long lng;
	HANDLE result;

	DBG_printf( "CeOpenDatabase( poid = 0x%08X, lpszName = 0x%08X, propid = 0x%08X, dwFlags = 0x%08X, hwndNotify = 0x%08X )\n",
	            poid ? ( (void*)*poid ) : NULL, lpszName, propid, dwFlags, hwndNotify );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x0E ); 	/* Command */
	pushLong( (rhnd->buffer), size, poid ? ( ( long ) ( *poid ) ) : ( ( long ) NULL ) ); 		/* Parameter1 : */
	pushLong( (rhnd->buffer), size, propid ); 	/* Parameter2 : */
	pushLong( (rhnd->buffer), size, dwFlags ); 	/* Parameter3 : */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 (errorcode?): %ld (0x%08lx)\n", lng, lng );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 (errorcode?): %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = lng;
	result = ( HANDLE ) getLong( (rhnd->sock), &size );
	DBG_printf( "long 3 (errorcode?): %ld (0x%08lx)\n", result, result );

	return result;
}

STDAPI_( CEOID ) CeReadRecordProps(RAPIHandleP rhnd, HANDLE hDbase, DWORD dwFlags, LPWORD lpcPropID, CEPROPID* rgPropID, LPBYTE* lplpBuffer, LPDWORD lpcbBuffer )
{
	long size = BUFSIZE;
	long lng;
	long index;
	long i;
	WORD wrd;
	CEPROPVAL *ptr;
	LPBYTE ptrbyte;
	CEOID result;

	DBG_printf( "CeReadRecordProps( hDbase = 0x%08X, dwFlags = 0x%08X, lpcPropID = 0x%08X, rgPropID = 0x%08X, lplpBuffer = 0x%08X, lpcbBuffer = 0x%08X )\n",
	            hDbase, dwFlags, lpcPropID, rgPropID, lplpBuffer, lpcbBuffer );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x10 ); 		/* Command */
	pushLong( (rhnd->buffer), size, ( long ) hDbase ); 	/* Parameter1 : */
	pushLong( (rhnd->buffer), size, dwFlags ); 		/* Parameter2 : Flags ? */
	pushLong( (rhnd->buffer), size, 0 ); 			/* Parameter3 */
	pushLong( (rhnd->buffer), size, 0 ); 			/* Parameter4 */
	pushShort( (rhnd->buffer), size, 0 ); 			/* Parameter5 */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 (errorcode?): %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2: lasterror %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );
	result = getLong( (rhnd->sock), &size );
	DBG_printf( "long 3: CEOID %ld (0x%08lx)\n", result, result );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 4: (taille buffer) %ld (0x%08lx)\n", lng, lng );
	if ( lpcbBuffer && ( ( *lpcbBuffer ) == 0 ) )
	{
		*lpcbBuffer = lng;
	}
	wrd = getShort( (rhnd->sock), &size );
	DBG_printf( "word 1: (nbprops) %ld (0x%08lx)\n", wrd, wrd );
	if ( !rgPropID )
	{
		if ( lpcPropID )
		{
			*lpcPropID = wrd;
		}
	}

	/*	if( cFindData )
		{
			*cFindData = wrd;
		}*/
	if ( lplpBuffer && ( wrd > 0 ) )
	{
		if ( *lplpBuffer == NULL )
		{
			DBG_printf( "Before allocation\n" );
			*lplpBuffer = ( LPBYTE ) calloc( 1, lng );
			DBG_printf( "After allocation : *ppFindData = %08X\n", *lplpBuffer );
		}
		ptr = ( CEPROPVAL * ) ( *lplpBuffer );
		for ( i = 0; i < wrd; i++ )
		{
			DBG_printf( "i=%d : ptr=%08X\n", i, ptr );
			getbufferchunk( (rhnd->sock), &size, &( ptr[ i ] ), sizeof( CEPROPVAL ) );
			DBG_printf( "propval: propid %ld (0x%08lx) \n", ptr[ i ].propid, ptr[ i ].propid );
			DBG_printf( "propval: wLenData %d (0x%04x) \n", ptr[ i ].wLenData, ptr[ i ].wLenData );
			DBG_printf( "propval: wFlags %d (0x%04x) \n", ptr[ i ].wFlags, ptr[ i ].wFlags );
			switch ( ( ptr[ i ].propid ) & 0xFFFF )
			{
				case CEVT_BLOB:
					ptr[ i ].val.blob.lpb = ( LPBYTE ) ( ( DWORD ) ptr + ( DWORD ) ptr[ i ].val.blob.lpb );
					DBG_printf( "propval: BLOB (size : %d, ptr %-8X)\n", ptr[ i ].val.blob.dwCount, ptr[ i ].val.blob.lpb );
					break;
				case CEVT_LPWSTR:
					ptr[ i ].val.lpwstr = ( LPWSTR ) ( ( DWORD ) ptr + ( DWORD ) ptr[ i ].val.lpwstr );
					DBG_printf( "propval: LPWSTR %ld (0x%08lx) \n", ptr[ i ].val.lpwstr, ptr[ i ].val.lpwstr );
					break;
				case CEVT_I2:
					DBG_printf( "propval: I2 %d (0x%04x) \n", ptr[ i ].val.iVal, ptr[ i ].val.iVal );
					break;
				case CEVT_UI2:
					DBG_printf( "propval: UI2 %d (0x%04x) \n", ptr[ i ].val.uiVal, ptr[ i ].val.uiVal );
					break;
				case CEVT_I4:
					DBG_printf( "propval: I4 %ld (0x%08lx) \n", ptr[ i ].val.lVal, ptr[ i ].val.lVal );
					break;
				case CEVT_UI4:
					DBG_printf( "propval: UI4 %ld (0x%08lx) \n", ptr[ i ].val.ulVal, ptr[ i ].val.ulVal );
					break;
				case CEVT_BOOL:
					DBG_printf( "propval: BOOL %s\n", ptr[ i ].val.boolVal ? "true" : "false" );
					break;
				case CEVT_FILETIME:
					DBG_printf( "propval: FILETIME 0x%-8x\n", ptr[ i ].val.filetime );
					break;
				case CEVT_R8:
					DBG_printf( "propval: R8 %d ", ptr[ i ].val.dblVal );
					break;
				default:
					break;
			}
			/* 	                DBG_printf( "propval: sz:%d %ld (0x%08lx) \n", sizeof(CEVALUNION), valunion, valunion ); */
		}

		ptrbyte = ( LPBYTE ) & ( ptr[ wrd ] );
		index = ( long ) ( lng - ( ptrbyte - ( *lplpBuffer ) ) );
		DBG_printf( " getchunk : index = %d, size = %d\n", index, size );
		getbufferchunk( (rhnd->sock), &size, ptrbyte, ( ( long ) ptrbyte > index ) ? index : ( long ) ptrbyte );
	}
	return result;
}

STDAPI_(CEOID) CeWriteRecordProps(RAPIHandleP rhnd, HANDLE hDbase, CEOID oidRecord, WORD cPropID, CEPROPVAL* rgPropVal)
{
	long size = BUFSIZE;
	long lng;
	long datalen, buflen;
	long i;
	CEOID result;

	DBG_printf( "CeWriteRecordProps( hDbase = 0x%08X, oidRecord = 0x%08X, cPropID = 0x%08X, rgPropVal = 0x%08X )\n",
	            hDbase, oidRecord, cPropID, rgPropVal );

	/*
	*	Format of the CeWriteRecordProps packet - primitives are encoded in the CEPROPVAL structures, lpwstr and blob properties are
	*	attached to the end of the buffer and referenced by offset pointers
	*
	*		long hDBase | long oidRecord | long cPropID | long datalen (of following data) | n * CEPROPVAL | char[] data
	*
	*	Because CEPROPVAL is a union, the format is different for every type of prop:
	*
	*	long or short (iVal, uiVal, lVal, ulVal, boolVal): long propid | short wFlags | short wLenData (unused, set to 0) | short iVal or boolVal | short uiVal | long lVal or ulVal
	*
	*	FILETIME or double: long propid | short wFlags | short wLenData (unused) | DWORD FILETIME or double
	*
	*	lpwstr: long propid | short wFlags | short wLenData (unused) | long offset ( points to string data in data buffer, counted from beginning of CEPROPVALs)
	*
	*	blob: long propid | short wFlags | short wLenData (unused) | long blobsize | long offset (same as lpwstr)
	*/
	
	
	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x11 ); 		/* Command */
	pushLong( (rhnd->buffer), size, ( long ) hDbase ); 	/* Parameter1 : */
	pushLong( (rhnd->buffer), size, oidRecord ); 		/* Parameter2 : Flags ? */
	pushShort( (rhnd->buffer), size, cPropID ); 			/* Parameter3 */


/*
	* we have to go through the rgPropVals array three times:
	*		1. to determine the size of the whole buffer, including data
	*		2. to write out the CEPROPVAL array
	*		3. to write the data segment
	*/
		
	/* calculate the length of the whole buffer, including the data segment at the end */
	
	buflen = cPropID * sizeof( CEPROPVAL ); /* length of all cepropvals */
	
	for ( i = 0; i < cPropID; i++ )
	{	
		switch ( ( rgPropVal[i].propid ) & 0xFFFF )
		{
			case CEVT_BLOB:
					buflen += rgPropVal[i].val.blob.dwCount;
				break;
			case CEVT_LPWSTR:
					buflen += 2* ( wcslen( rgPropVal[i].val.lpwstr ) + 1 );
				break;
			default:
				break;
		}
	}
	
	pushLong( (rhnd->buffer), size, buflen);				
	
	/*
	 second time: write n * CEPROPVAL. Can't do it in one block, as we have to adjust the buffer offsets
	 */
	
	datalen = cPropID * sizeof( CEPROPVAL ); /*  holds the offset to the end of the data buffer	*/
	
	for ( i = 0; i < cPropID; i++ )
	{
		pushLong( (rhnd->buffer), size,  rgPropVal[i].propid );
		pushShort( (rhnd->buffer), size,  rgPropVal[i].wLenData );
		pushShort( (rhnd->buffer), size,  rgPropVal[i].wFlags );
		
		switch ( ( rgPropVal[i].propid ) & 0xFFFF )
		{
			case CEVT_BLOB:
					pushLong( (rhnd->buffer), size, rgPropVal[i].val.blob.dwCount );
					datalen += rgPropVal[i].val.blob.dwCount;
					pushLong( (rhnd->buffer), size, datalen );			
				break;
			case CEVT_LPWSTR:
					datalen += 2* ( wcslen( rgPropVal[i].val.lpwstr ) + 1 );
					pushLong( (rhnd->buffer), size, datalen );
					pushLong( (rhnd->buffer), size, 0 );			
				break;
			case CEVT_I2:
			case CEVT_UI2:
			case CEVT_I4:
			case CEVT_UI4:
				pushShort( (rhnd->buffer), size, rgPropVal[i].val.iVal );
				pushShort( (rhnd->buffer), size, rgPropVal[i].val.uiVal );
				pushLong( (rhnd->buffer), size, rgPropVal[i].val.lVal );
				break;
			case CEVT_BOOL:
				pushShort( (rhnd->buffer), size, rgPropVal[i].val.boolVal  );
				pushShort( (rhnd->buffer), size, 0 );
				pushLong( (rhnd->buffer), size, 0 );
				break;
			case CEVT_FILETIME:
				/* this assumes that the FILETIME is already in ole32 format! Is this a problem? */
				pushLong( (rhnd->buffer), size, rgPropVal[i].val.filetime.dwLowDateTime );
				pushLong( (rhnd->buffer), size, rgPropVal[i].val.filetime.dwHighDateTime );
				break;
			case CEVT_R8:
				pushParameter(rhnd, size, &(rgPropVal[i].val.dblVal), 4, 1 );
				break;
			default:
				break;
		}
	}	
	
	/* 3. write the data segment */
	
	for ( i = 0; i < cPropID; i++ )
	{	
		switch ( ( rgPropVal[i].propid ) & 0xFFFF )
		{
			case CEVT_BLOB:
					pushParameter_no_size(rhnd, size, rgPropVal[i].val.blob.lpb, rgPropVal[i].val.blob.dwCount );					
				break;
			case CEVT_LPWSTR:
					pushParameter_no_size(rhnd, size, rgPropVal[i].val.lpwstr, 2* ( wcslen( rgPropVal[i].val.lpwstr ) + 1) );
				break;
			default:
				break;
		}
	}	
	
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );

	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 (errorcode?): %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );
	result = ( HANDLE ) getLong( (rhnd->sock), &size );
	DBG_printf( "long 3 (HANDLE): %ld (0x%08lx)\n", result, result );

	if ( size > 0 )
	{
		flushbuffer( (rhnd->sock) );
	}

	return result;
}


/* ================================================================================================================= */
/* ================================================================================================================= */
/*  RAPI - Processes */
/* ================================================================================================================= */
/* ================================================================================================================= */

STDAPI_( BOOL ) CeCreateProcess(RAPIHandleP rhnd, LPCWSTR lpApplicationName,
				 LPCWSTR lpCommandLine, 
				 LPSECURITY_ATTRIBUTES lpProcessAttributes,
				 LPSECURITY_ATTRIBUTES lpThreadAttributes,
				 BOOL bInheritHandles, 
				 DWORD dwCreationFlags,
                                 LPVOID lpEnvironment, 
				 LPWSTR lpCurrentDirectory, 
				 LPSTARTUPINFO lpStartupInfo,
                                 LPPROCESS_INFORMATION lpProcessInformation )
{
	long size = BUFSIZE;
	long lng;
	long stlen;
	BOOL result = FALSE;

	DBG_printf( "CeCreateProcess( lpApplication = 0x%08X, lpCommandLine = 0x%08X )\n",
	            lpApplicationName, lpCommandLine );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x19 ); 	/* Command */
	if(lpApplicationName!=NULL)
	{
		pushLong( (rhnd->buffer), size, 0x01 );
		stlen = wcslen( lpApplicationName );
		pushLong( (rhnd->buffer), size, (stlen+1)*2 );
		pushLong( (rhnd->buffer), size, 0x01 );
		pushString( (rhnd->buffer), size, lpApplicationName );
	}
	else
	{
		pushLong( (rhnd->buffer), size, 0x00 );
	}
	if(lpCommandLine!=NULL)
	{
		pushLong( (rhnd->buffer), size, 0x01 );
		stlen = wcslen( lpCommandLine );
		pushLong( (rhnd->buffer), size, (stlen+1)*2 );
		pushLong( (rhnd->buffer), size, 0x01 );
		pushString( (rhnd->buffer), size, lpCommandLine );
	}
	else
	{
		pushLong( (rhnd->buffer), size, 0x00 );
	}
	
	pushLong( (rhnd->buffer), size, 0x00 ); /* Parameter3 : lpProcessAttibutes ? */
	pushLong( (rhnd->buffer), size, 0x00 ); /* Parameter4 : lpThreadAttributes ? */
	pushLong( (rhnd->buffer), size, 0x00 ); /* Parameter5 : bInheritHandles ? */
	pushLong( (rhnd->buffer), size, dwCreationFlags ); 
	pushLong( (rhnd->buffer), size, 0x00 ); /* Parameter7 : lpEnvironment ? */
	pushLong( (rhnd->buffer), size, 0x00 ); /* Parameter8 : lpCurrentDirectory ? */
	pushLong( (rhnd->buffer), size, 0x00 ); /* Parameter9 : lpStartupInfo ? */
	pushLong( (rhnd->buffer), size, 0x00 ); /* Parametera : lpProcessInformation ? */

	DBG_printbuf( (rhnd->buffer) );

	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 (errorcode?): %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2: lasterror %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );

	result = getLong( (rhnd->sock), &size );
	DBG_printf( "long 3: %ld (0x%08lx)\n", result, result );
	if(result)
	{
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long 4: %ld (0x%08lx)\n", lng, lng );
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long 5: %ld (0x%08lx)\n", lng, lng );
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long 6: %ld (0x%08lx)\n", lng, lng );
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long 7: %ld (0x%08lx)\n", lng, lng );
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long 8: %ld (0x%08lx)\n", lng, lng );
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long 9: %ld (0x%08lx)\n", lng, lng );
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long a: %ld (0x%08lx)\n", lng, lng );
	}
	else
	{
		lng = getLong( (rhnd->sock), &size );
		DBG_printf( "long 4: %ld (0x%08lx)\n", lng, lng );
	}

	if ( size > 0 )
	{
		flushbuffer( (rhnd->sock) );
	}

	return result;
}

/* ================================================================================================================= */
/* ================================================================================================================= */
/*  RAPI - General */
/* ================================================================================================================= */
/* ================================================================================================================= */

STDAPI_( BOOL ) CeCloseHandle(RAPIHandleP rhnd, HANDLE hObject )
{
	long size = BUFSIZE;
	long lng;

	DBG_printf( "CeCloseHandle()\n" );

	initBuf( (rhnd->buffer), size );
	pushLong( (rhnd->buffer), size, 0x08 ); 		/* Command */
	pushLong( (rhnd->buffer), size, ( long ) hObject ); 	/* Parameter1 : */
	DBG_printbuf( (rhnd->buffer) );
	sendbuffer( (rhnd->sock), (rhnd->buffer) );

	size = getbufferlen( (rhnd->sock) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 1 : %ld (0x%08lx)\n", lng, lng );
	(rhnd->_lasterror) = getLong( (rhnd->sock), &size );
	DBG_printf( "long 2 (lasterror): %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror) );
	lng = getLong( (rhnd->sock), &size );
	DBG_printf( "long 3 : %ld (0x%08lx)\n", lng, lng );

	return lng;
}

STDAPI_( DWORD ) CeGetLastError(RAPIHandleP rhnd)
{
	return (rhnd->_lasterror);
}

/* ================================================================================================================= */
/* ================================================================================================================= */
/*  RAPI - Library */
/* ================================================================================================================= */
/* ================================================================================================================= */

void findquotes( char * *ptrstart, char * *ptrstop )
{
	/* Go to the first double quote */
	/* ---------------------------- */
	while( ( *(*ptrstart) != 0 ) && ( *(*ptrstart) != '"' ) )
	{
		(*ptrstart) ++;
	}
	if( *(*ptrstart) != 0 )
	{
		(*ptrstart) ++;
		*ptrstop = *ptrstart;
		/* Reach the next double quote */
		/* --------------------------- */
		while( ( *(*ptrstop) != 0 ) && ( *(*ptrstop) != '"' ) )
		{
			(*ptrstop) ++;
		}
	}
}

int parseinfofile( RAPIHandleP rhnd );

int parseinfofile( RAPIHandleP rhnd )
{
	BOOL result = FALSE;
	struct stat st;
	int res;
	FILE * f;
	char line[ 1024 ];
	char * ptrstart;
	char * ptrstop;
	BOOL devicefound;
	int i;

	devicefound = FALSE;

	DBG_printf( "Before stat...\n" );
	res = stat( INFOFILE, &st );
	DBG_printf( "stat(), result = %d...\n", res );
	if ( res != -1 )
	{
		f = fopen( INFOFILE, "r" );
		DBG_printf( "fopen(), result = 0x%08X...\n", f );
		if ( f )
		{
			while ( ( !feof( f ) ) && ( result == FALSE ) )
			{
				line[ 0 ] = '\0';
				if ( fgets( line, 1024, f ) != NULL )
				{
					DBG_printf( "fscanf(), line = '%s'...\n", line );
					if ( line[ 0 ] != '#' )
					{
						/* Beginning of block : device <IP> { */
						/* ---------------------------------- */
						if( ((rhnd->hostname)==NULL) && ( ( ptrstart = strstr( line, "device" ) ) != NULL ) )
						{	
							devicefound = TRUE;
							ptrstart += 7;
							for ( ptrstop = ptrstart; ( ( *ptrstop ) != 0 ) && ( ( *ptrstop ) != '{' ); ptrstop++ );
							( *ptrstop ) = 0;
							(rhnd->hostname) = strdup( ptrstart );
							DBG_printf( "strdup(), hostname = '%s'...\n", (rhnd->hostname) );
						}

						if( devicefound )
						{

							/* End of block : } */
							/* ---------------- */
							if( strstr( line, "}" ) != NULL )
							{

								devicefound = FALSE;
								result = TRUE;

							} else {

								/* let's check for the passphrase */
								/* ------------------------------ */
								if( ( ((rhnd->lockbuffersize)!=0) || ((rhnd->lockbuffer)==NULL)) && ( ( ptrstart = strstr( line, "passphrase" ) ) != NULL ) )
								{
									DBG_printf( "found string passphrase = %s\n", line );
									findquotes( &ptrstart, &ptrstop );

									if( (*ptrstart) != 0 )
									{
										(*ptrstop) = '\0';
										if( NULL != strstr( line, "size" ) )
										{
											DBG_printf( "found string passphrase-size = %s\n", ptrstart );
											if( 1 == sscanf( ptrstart, "%d", &(rhnd->lockbuffersize) ) )
											{
												DBG_printf( "found passphrase-size = %d\n", (rhnd->lockbuffersize) );
												(rhnd->lockbuffer) = (unsigned char *) malloc( 1+ (rhnd->lockbuffersize) );
											}
										} else {
											for( i=0; i<(rhnd->lockbuffersize); i++ )
											{
												sscanf( ptrstart + (3*i), "%02X", (unsigned int *)&((rhnd->lockbuffer)[i]) );
												DBG_printf("%02X ", (rhnd->lockbuffer)[i] );
											}
											DBG_printf( "\nfound passphrase-data = %s\n", ptrstart );
										}
									}
								}
							}
						}
					}
				}
			}
			fclose( f );
		}
	}
	return result;
}

STDAPI_( HRESULT ) CeRapiInit(RAPIHandleP rhnd)
{
	if ( (rhnd->sock) == 0 )
	{
		parseinfofile(rhnd);
		if ( (rhnd->hostname) )
		{
			DBG_printf( "CeRapiInit(%s)\n", (rhnd->hostname) );
			(rhnd->sock) = SockOpen( (rhnd->hostname), 990 );
			if ( (rhnd->sock) > 0 )
			{
				DBG_printf( "CeRapiInit(%s) ok\n", (rhnd->hostname) );
				free( (rhnd->hostname) );
                                (rhnd->buffer) = (rapibuffer *)malloc( 4 + BUFSIZE );

	checkpassword(rhnd);

				return E_SUCCESS;
			}
			else
			{
				DBG_printf( "CeRapiInit(%s) Nok\n", (rhnd->hostname) );
				free( (rhnd->hostname) );
				return E_FAIL;
			}
			free( (rhnd->hostname) );
		}
		else
		{
			DBG_printf( "No CE Device found !?!\n" );
			return E_FAIL;
		}
	}
	else
	{
		DBG_printf( "CeRapiInit() already done\n" );
		return CERAPI_E_ALREADYINITIALIZED;
	}
}
/* STDAPI CeRapiInitEx (RAPIINIT*); */

STDAPI_( HRESULT ) CeRapiUninit(RAPIHandleP rhnd)
{
	HRESULT result = FALSE;
        if( (rhnd->buffer) )
        {
                free( (rhnd->buffer) );
                (rhnd->buffer) = NULL;
        }
	if ( (rhnd->sock) > 0 )
	{
		result = (close( (rhnd->sock) )==0) ? TRUE : FALSE;
		DBG_printf( "CeRapiUninit() %s\n", (result ? "ok" : "failed") );
		
		/* the following is needed to make subsequent init/uninit calls to RAPI
		 * (e.g. device disconnect/reconnect) */
	
		if (result) {
			(rhnd->sock) = 0;
			(rhnd->hostname) = NULL;
		}
			
	}
	else
	{
		DBG_printf( "CeRapiUninit() no need\n" );
	}
	return result;
}

STDAPI_( HRESULT ) CeRapiFreeBuffer(RAPIHandleP rhnd, LPVOID Buffer )
{
	if ( Buffer )
	{
		free( Buffer );
	}
	return ( HRESULT ) NULL;
}

STDAPI_(BOOL) CeGetVersionEx(RAPIHandleP rhnd,LPCEOSVERSIONINFO lpVersion)
{
	BOOL result = FALSE;
	long size = BUFSIZE;
	LONG lng;

//	checkpassword();
	
	initBuf((rhnd->buffer), size);
	
	pushLong((rhnd->buffer), size, 0x3B); 	/* Command */
	pushParameter(rhnd,size, lpVersion, lpVersion->dwOSVersionInfoSize, 0);
	
	sendbuffer( (rhnd->sock), (rhnd->buffer) );
	size = getbufferlen( (rhnd->sock) );

	/*
	 * The return package looks like this
	 *
	 * Offset  Size  Value
	 * 00      4     0
	 * 04      4     0
	 * 08      4     1
	 * 0c      4     n = real size of buffer
	 * 10      n     first n bytes of struct
	 */
	
	lng = getLong((rhnd->sock), &size); 
	(rhnd->_lasterror) = getLong((rhnd->sock), &size);

	if (0 == lng)
	{
		result = getLong((rhnd->sock), &size);
		if (1 == result)
		{
		 	long real_length = getLong((rhnd->sock), &size);
			getbufferchunk((rhnd->sock), &size, lpVersion, real_length);
		}
	}
	else
	{
		DBG_printf("Warning: expected 0 but got %i=0x%x\n", lng, lng);
	}
	
	if ( size > 0 )
	{
		DBG_printf( "size : %d\n", size );
		flushbuffer( (rhnd->sock) );
	}	
	
	return result;
}

STDAPI_(BOOL) CeCreateDirectory(RAPIHandleP rhnd,LPCWSTR lpDirName, LPSECURITY_ATTRIBUTES lpSecAttr)
{
	BOOL result = FALSE;
	long size = BUFSIZE;
	LONG lng;
	
	initBuf((rhnd->buffer), size);
	
	pushLong((rhnd->buffer), size, 0x17); 	/* Command */
	pushParameter(rhnd,size, (void*)lpDirName, (wcslen(lpDirName) + 1) * sizeof(WCHAR), 1);
	pushLong((rhnd->buffer), size, 0); 	/* lpSecAttr not used */
	
	/*DBG_printbuf( (rhnd->buffer) );*/
	sendbuffer( (rhnd->sock), (rhnd->buffer) );
	size = getbufferlen( (rhnd->sock) );

	/*
	 * The return package looks like this
	 *
	 * Offset  Size  Value
	 * 00      4     0
	 * 04      4     error code if the value below is 0
	 * 08      4     0/1
	*/

	lng = getLong((rhnd->sock), &size); 
	(rhnd->_lasterror) = getLong((rhnd->sock), &size);

	if (0 == lng)
	{
		result = getLong((rhnd->sock), &size);
	}
	else
	{
		DBG_printf("Warning: expected 0 but got %i=0x%x\n", lng, lng);
	}
	
	if ( size > 0 )
	{
		DBG_printf( "size : %d\n", size );
		flushbuffer( (rhnd->sock) );
	}	
	
	return result;
}

STDAPI_(BOOL) CeRemoveDirectory(RAPIHandleP rhnd,LPCWSTR lpPathName)
{
	BOOL result = FALSE;
	long size = BUFSIZE;
	LONG lng;
	
	initBuf((rhnd->buffer), size);
	
	pushLong((rhnd->buffer), size, 0x18); 	/* Command */
	pushParameter(rhnd,size, (void*)lpPathName, (wcslen(lpPathName) + 1) * sizeof(WCHAR), 1);
	
	/*DBG_printbuf( (rhnd->buffer) );*/
	sendbuffer( (rhnd->sock), (rhnd->buffer) );
	size = getbufferlen( (rhnd->sock) );

	/*
	 * The return package looks like this
	 *
	 * Offset  Size  Value
	 * 00      4     0
	 * 04      4     error code if the value below is 0
	 * 08      4     0/1
	*/

	lng = getLong((rhnd->sock), &size); 
	(rhnd->_lasterror) = getLong((rhnd->sock), &size);

	if (0 == lng)
	{
		result = getLong((rhnd->sock), &size);
	}
	else
	{
		DBG_printf("Warning: expected 0 but got %i=0x%x\n", lng, lng);
	}
	
	if ( size > 0 )
	{
		DBG_printf( "size : %d\n", size );
		flushbuffer( (rhnd->sock) );
	}	
	
	return result;
}

STDAPI_(BOOL) CeDeleteFile(RAPIHandleP rhnd,LPCWSTR lpFileName)
{
	BOOL result = FALSE;
	long size = BUFSIZE;
	LONG lng;
	
	initBuf((rhnd->buffer), size);
	
	pushLong((rhnd->buffer), size, 0x1c); 	/* Command */
	pushParameter(rhnd,size, (void*)lpFileName, (wcslen(lpFileName) + 1) * sizeof(WCHAR), 1);
	
	/*DBG_printbuf( (rhnd->buffer) );*/
	sendbuffer( (rhnd->sock), (rhnd->buffer) );
	size = getbufferlen( (rhnd->sock) );

	/*
	 * The return package looks like this
	 *
	 * Offset  Size  Value
	 * 00      4     0
	 * 04      4     error code if the value below is 0
	 * 08      4     0/1
	*/

	lng = getLong((rhnd->sock), &size); 
	DBG_printf("long 1 : %ld (0x%08lx)\n", lng, lng);
	(rhnd->_lasterror) = getLong((rhnd->sock), &size);
	DBG_printf("long 2 : %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror));

	if (0 == lng)
	{
		result = getLong((rhnd->sock), &size);
		DBG_printf("long 3 : %ld (0x%08lx)\n", result, result);
	}
	else
	{
		DBG_printf("Warning: expected 0 but got %i=0x%x\n", lng, lng);
	}
	
	if ( size > 0 )
	{
		DBG_printf( "size : %d\n", size );
		flushbuffer( (rhnd->sock) );
	}	
	
	return result;
}

STDAPI_( BOOL ) CeMoveFile(RAPIHandleP rhnd, LPCWSTR lpExistingFileName, LPCWSTR lpNewFileName )
{
	BOOL result = FALSE;
	long size = BUFSIZE;
	LONG lng;
	
	initBuf((rhnd->buffer), size);
	
	pushLong((rhnd->buffer), size, 0x1a); 	/* Command */
	pushParameter(rhnd,size, (void*)lpExistingFileName, (wcslen(lpExistingFileName) + 1) * sizeof(WCHAR), 1);
	pushParameter(rhnd,size, (void*)lpNewFileName, (wcslen(lpNewFileName) + 1) * sizeof(WCHAR), 1);
	
	/*DBG_printbuf( (rhnd->buffer) );*/
	sendbuffer( (rhnd->sock), (rhnd->buffer) );
	size = getbufferlen( (rhnd->sock) );

	/*
	 * The return package looks like this
	 *
	 * Offset  Size  Value
	 * 00      4     0
	 * 04      4     error code if the value below is 0
	 * 08      4     0/1
	*/

	lng = getLong((rhnd->sock), &size); 
	DBG_printf("long 1 : %ld (0x%08lx)\n", lng, lng);
	(rhnd->_lasterror) = getLong((rhnd->sock), &size);
	DBG_printf("long 2 : %ld (0x%08lx)\n", (rhnd->_lasterror), (rhnd->_lasterror));

	if (0 == lng)
	{
		result = getLong((rhnd->sock), &size);
		DBG_printf("long 3 : %ld (0x%08lx)\n", result, result);
	}
	else
	{
		DBG_printf("Warning: expected 0 but got %i=0x%x\n", lng, lng);
	}
	
	if ( size > 0 )
	{
		DBG_printf( "size : %d\n", size );
		flushbuffer( (rhnd->sock) );
	}	
	
	return result;
}

void CeGetSystemInfo(RAPIHandleP rhnd,LPSYSTEM_INFO lpSystemInfo)
{
	BOOL result = FALSE;
	long size = BUFSIZE;
	LONG lng;
	
	initBuf((rhnd->buffer), size);
	
	pushLong((rhnd->buffer), size, 0x2f); 	/* Command */
	pushParameter(rhnd,size, lpSystemInfo, sizeof(SYSTEM_INFO), 0);
	
	sendbuffer( (rhnd->sock), (rhnd->buffer) );
	size = getbufferlen( (rhnd->sock) );

	/*
	 * The return package looks like this
	 *
	 * Offset  Size  Value
	 * 00      4     0
	 * 04      4     0
	 * 08      4     1
	 * 0c      4     n = real size of buffer
	 * 10      n     first n bytes of struct
	 */
	
	lng = getLong((rhnd->sock), &size); 
	(rhnd->_lasterror) = getLong((rhnd->sock), &size);

	/*
	 * Note: On my system, 36 bytes are returned but sizeof(SYSTEM_INFO) is only 32!
	 *
	 * "Overflow by 4 bytes. Parameter size is 36 bytes but max 32 bytes was expected."
	 */
	
	if (0 == lng)
	{
		popParameter(rhnd,&size, lpSystemInfo, sizeof(SYSTEM_INFO));
	}
	else
	{
		DBG_printf("Warning: expected 0 but got %i=0x%x\n", lng, lng);
	}
	
	if ( size > 0 )
	{
		DBG_printf( "size : %d\n", size );
		flushbuffer( (rhnd->sock) );
	}	
}


