/*
 * gvfs.cpp
 * Copyright (C) 2002 Florin Malita <mali@go.ro>
 *
 * This file is part of LUFS, a free userspace filesystem implementation.
 * See http://lufs.sourceforge.net/ for updates.
 *
 * LUFS is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * LUFS is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#include <stdio.h>
#include <utime.h>

#include <sys/types.h>
#include <sys/stat.h>

#include <vector>
#include <string>

#include <lufs/proto.h>
#include <lufs/fs.h>

#include "gvfs.h"

#include <gnome.h>
#include <libgnomevfs/gnome-vfs.h>

extern "C" { 

void*
gvfs_init(struct list_head *cfg, struct dir_cache *cache, struct credentials *cred, void **global_ctx){
    return (void*)new GVFS(cfg, cache, cred);
}

void
gvfs_free(void *ctx){
    GVFS *p = (GVFS*)ctx;

    delete p;
}

int 	
gvfs_mount(void *ctx){
    return ((GVFS*)ctx)->do_mount();
}

void 	
gvfs_umount(void *ctx){

}

int 	
gvfs_readdir(void *ctx, char *dir_name, struct directory *dir){
    return ((GVFS*)ctx)->do_readdir(dir_name, dir);
}

int 	
gvfs_stat(void *ctx, char *name, struct lufs_fattr *fattr){
    return ((GVFS*)ctx)->do_stat(name, fattr);
}

int 	
gvfs_mkdir(void *ctx, char *dir, int mode){
    return ((GVFS*)ctx)->do_mkdir(dir, mode);
}

int 	
gvfs_rmdir(void *ctx, char *dir){
    return ((GVFS*)ctx)->do_rmdir(dir);
}

int 	
gvfs_create(void *ctx, char *file, int mode){
    return ((GVFS*)ctx)->do_create(file, mode);
}

int 	
gvfs_unlink(void *ctx, char *file){
    return ((GVFS*)ctx)->do_unlink(file);
}

int 	
gvfs_rename(void *ctx, char *old_name, char *new_name){
    return ((GVFS*)ctx)->do_rename(old_name, new_name);
}

int 	
gvfs_open(void *ctx, char *file, unsigned mode){
    return ((GVFS*)ctx)->do_open(file, mode);
}

int 	
gvfs_release(void *ctx, char *file){
    return ((GVFS*)ctx)->do_release(file);
}

int 	
gvfs_read(void *ctx, char *file, long long offset, unsigned long count, char *buf){
    return ((GVFS*)ctx)->do_read(file, offset, count, buf);
}

int	
gvfs_write(void *ctx, char *file, long long offset, unsigned long count, char *buf){
    return ((GVFS*)ctx)->do_write(file, offset, count, buf);
}

int 	
gvfs_readlink(void *ctx, char *link, char *buf, int buflen){
    return ((GVFS*)ctx)->do_readlink(link, buf, buflen);
}

int 	
gvfs_link(void *ctx, char *target, char *link){
    return ((GVFS*)ctx)->do_link(target, link);
}

int 	
gvfs_symlink(void *ctx, char *target, char *link){
    return ((GVFS*)ctx)->do_symlink(target, link);
}

int 	
gvfs_setattr(void *ctx, char *file, struct lufs_fattr *fattr){
    return ((GVFS*)ctx)->do_setattr(file, fattr);
}

} /* extern "C" */

static void
info2fattr(GnomeVFSFileInfo *info, struct lufs_fattr *fattr){
    memset(fattr, 0, sizeof(struct lufs_fattr));

    TRACE("cooking " << info->name);

    if(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_TYPE){
	TRACE("got the type");
	
	switch(info->type){
	case GNOME_VFS_FILE_TYPE_DIRECTORY:
	    fattr->f_mode = S_IFDIR;
	    break;
	case GNOME_VFS_FILE_TYPE_FIFO:
	    fattr->f_mode = S_IFIFO;
	    break;
	case GNOME_VFS_FILE_TYPE_SOCKET:
	    fattr->f_mode = S_IFSOCK;
	    break;
	case GNOME_VFS_FILE_TYPE_CHARACTER_DEVICE:
	    fattr->f_mode = S_IFCHR;
	    break;
	case GNOME_VFS_FILE_TYPE_BLOCK_DEVICE:
	    fattr->f_mode = S_IFBLK;
	    break;
	case GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK:
	    fattr->f_mode = S_IFLNK;
	    break;
	case GNOME_VFS_FILE_TYPE_REGULAR:
	case GNOME_VFS_FILE_TYPE_UNKNOWN:
	default:
	    fattr->f_mode = S_IFREG;
	}
    }else{
	TRACE("hmmm, no file type...");
	fattr->f_mode = S_IFREG;
    }

    if(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS){
	TRACE("got the permissions");
	fattr->f_mode |= info->permissions;
    }else{
	TRACE("hmmm, no file permissions?!?!");
	fattr->f_mode |= S_IRUSR;
    }

    if(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_LINK_COUNT){
	TRACE("got the link count");
	fattr->f_nlink = info->link_count;
    }else{
	TRACE("hmmm, no link count...");
	fattr->f_nlink = 1;
    }

    if(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SIZE){
	TRACE("got the size");
	fattr->f_size = info->size;
    }else{
	TRACE("hmmm, no size?!");
	fattr->f_size = 2048;
    }
 
    if(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_ATIME){
	TRACE("got the atime");
	fattr->f_atime = info->atime;
	if(!(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MTIME))
	    fattr->f_mtime = info->atime;
	if(!(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_CTIME))
	    fattr->f_ctime = info->atime;
    }

    if(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MTIME){
	TRACE("got the mtime");
	fattr->f_mtime = info->mtime;
	if(!(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_ATIME))
	    fattr->f_atime = info->mtime;
	if(!(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_CTIME))
	    fattr->f_ctime = info->mtime;
    }

    if(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_CTIME){
	TRACE("got the ctime");
	fattr->f_ctime = info->ctime;
	if(!(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_MTIME))
	    fattr->f_mtime = info->ctime;
	if(!(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_ATIME))
	    fattr->f_atime = info->ctime;
    }

    fattr->f_uid = 1;
    fattr->f_gid = 1;

}

GVFS::GVFS(struct list_head *c, struct dir_cache *cache, struct credentials *cred){
    TRACE("in GVFS constructor");

    cfg = c;
    this->cache = cache;
    this->cred = cred;
    
    if(!gnome_vfs_initialized())
	gnome_vfs_init();
}

GVFS::~GVFS(){
    TRACE("in GVFS destructor");
}

int
GVFS::do_mount(){
    TRACE("mounting a gvfs.");
    return 1;
}

int
GVFS::do_readdir(char* d, struct directory *ddir){
    struct lufs_fattr fattr;
    GnomeVFSDirectoryHandle *dir;
    GnomeVFSURI *uri;
    GnomeVFSFileInfo *info;
    string s;
    int res = -1;

    TRACE("dir: "<<d);

    if(!(uri = gnome_vfs_uri_new(d+1))){
	goto out;
    }

    if(!(info = gnome_vfs_file_info_new())){
	goto out_uri;
    }

    if(gnome_vfs_directory_open_from_uri(&dir, uri, GNOME_VFS_FILE_INFO_DEFAULT, NULL) != GNOME_VFS_OK){
	WARN("could not open directory!");
	goto out_info;
    }

    while(gnome_vfs_directory_read_next(dir, info) == GNOME_VFS_OK){
	TRACE("adding entry "<<info->name);

	if(!strcmp(info->name, ".") || !strcmp(info->name, ".."))
	    continue;

	info2fattr(info, &fattr);
	lu_cache_add2dir(ddir, info->name, (info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SYMLINK_NAME) ? info->symlink_name : NULL, &fattr);

    }

    gnome_vfs_directory_close(dir);
    res = 0;

  out_info:
    gnome_vfs_file_info_unref(info);
  out_uri:
    gnome_vfs_uri_unref(uri);
  out:
    return res;
}

int
GVFS::do_stat(char *nm, struct lufs_fattr *fattr){
    GnomeVFSURI *uri;
    GnomeVFSFileInfo *info;
    int res = -1;

    TRACE("file: "<<nm);

    if(!(uri = gnome_vfs_uri_new(nm + 1)))
	goto out;

    if(!(info = gnome_vfs_file_info_new()))
	goto out_uri;

    if(gnome_vfs_get_file_info_uri(uri, info, GNOME_VFS_FILE_INFO_DEFAULT) != GNOME_VFS_OK){
	WARN("get_file_info failed!");
	goto out_info;
    }

    info2fattr(info, fattr);
    res = 0;
    
  out_info:
    gnome_vfs_file_info_unref(info);
  out_uri:
    gnome_vfs_uri_unref(uri);
  out:
    return res;
}

int
GVFS::do_mkdir(char *dir, int mode){

    TRACE("dir: "<<dir<<", mode: "<<mode);

    if(gnome_vfs_make_directory(dir + 1, mode) == GNOME_VFS_OK)
	return 0;
    else
	return -1;
}

int
GVFS::do_rmdir(char *dir){
    TRACE("dir: "<<dir);

    if(gnome_vfs_remove_directory(dir + 1) == GNOME_VFS_OK)
	return 0;
    else
	return -1;
}

int
GVFS::do_create(char *file, int mode){
    GnomeVFSHandle *handle;
    GnomeVFSResult gres;

    TRACE("create "<<file<<",mode: "<<std::oct<<mode<<std::hex);

    if((gres = gnome_vfs_create(&handle, file + 1, GNOME_VFS_OPEN_NONE, 0, mode & 0777)) != GNOME_VFS_OK){
	WARN("couldn't create:"<<gnome_vfs_result_to_string(gres));
	return -1;
    }


    gnome_vfs_close(handle);

    return 0;
}

int
GVFS::do_unlink(char *file){
    TRACE("file: "<<file);

    if(gnome_vfs_unlink(file + 1) == GNOME_VFS_OK)
	return 0;
    else 
	return -1;
}

int
GVFS::do_rename(char *old, char *nnew){
    TRACE("old: "<<old<<",new: "<<nnew);

    if(gnome_vfs_move(old + 1, nnew + 1, 0) == GNOME_VFS_OK)
	return 0;
    else
	return -1;
}

int
GVFS::do_open(char *file, unsigned mode){
    TRACE("open "<<file<<" mode="<<std::oct<<mode<<std::hex); 

    return 1;
}

int
GVFS::do_release(char *file){
    TRACE("release "<<file);

    return 1;
}

int
GVFS::do_read(char *file, long long offset, unsigned long count, char *buf){
    GnomeVFSHandle *handle;
    GnomeVFSFileSize nread;
    int res = -1;

    TRACE("read: "<<file<<",off: "<<offset);

    if(gnome_vfs_open(&handle, file + 1, GNOME_VFS_OPEN_READ) != GNOME_VFS_OK){
	WARN("open failed!");
	goto out;
    }

    if(gnome_vfs_seek(handle, GNOME_VFS_SEEK_START, offset) != GNOME_VFS_OK){
	WARN("seek failed!");
	goto out_handle;
    }

    if(gnome_vfs_read(handle, buf, count, &nread) != GNOME_VFS_OK){
	WARN("read failed!");
    }else
	res = nread;
  
    TRACE("read "<<res<<"bytes");

  out_handle:
    gnome_vfs_close(handle);
  out:
    return res;
}

int
GVFS::do_write(char *file, long long offset, unsigned long count, char *buf){
    GnomeVFSHandle *handle;
    GnomeVFSFileSize nwrote;
    int res = -1;

    TRACE("read: "<<file<<",off: "<<offset);

    if(gnome_vfs_open(&handle, file + 1, GNOME_VFS_OPEN_WRITE) != GNOME_VFS_OK){
	WARN("open failed!");
	goto out;
    }

    if(gnome_vfs_seek(handle, GNOME_VFS_SEEK_START, offset) != GNOME_VFS_OK){
	WARN("seek failed!");
	goto out_handle;
    }

    if(gnome_vfs_write(handle, buf, count, &nwrote) != GNOME_VFS_OK){
	WARN("read failed!");
    }else
	res = nwrote;
  
    TRACE("read "<<res<<"bytes");

  out_handle:
    gnome_vfs_close(handle);
  out:
    return res;
}

int
GVFS::do_readlink(char *link, char *buf, int buflen){
    GnomeVFSURI *uri;
    GnomeVFSFileInfo *info;
    int res = -1;

    TRACE("link: "<<link);

    if(!(uri = gnome_vfs_uri_new(link + 1)))
	goto out;

    if(!(info = gnome_vfs_file_info_new()))
	goto out_uri;

    if(gnome_vfs_get_file_info_uri(uri, info, GNOME_VFS_FILE_INFO_DEFAULT) != GNOME_VFS_OK){
	WARN("get_file_info failed!");
	goto out_info;
    }

    if(info->valid_fields & GNOME_VFS_FILE_INFO_FIELDS_SYMLINK_NAME)
	if(strlen(info->symlink_name) < buflen){
	    strcpy(buf, info->symlink_name);
	    TRACE("target: " << buf);
	    res = 0;
	}
    

  out_info:
    gnome_vfs_file_info_unref(info);
  out_uri:
    gnome_vfs_uri_unref(uri);
  out:
    return res;
}

int
GVFS::do_link(char *old, char *nnew){
    TRACE("");
    return -1;
}

int
GVFS::do_symlink(char *old, char *nnew){
    TRACE("symlink "<<old<<" <=> "<<nnew);
    return -1;
}

int
GVFS::do_setattr(char *f, struct lufs_fattr *fattr){
    GnomeVFSFileInfo *info;
    int res = -1;
    
    TRACE("file: "<<f);

    if(!(info = gnome_vfs_file_info_new()))
	goto out;


    info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_PERMISSIONS;
    info->permissions = (GnomeVFSFilePermissions)(fattr->f_mode & 0777);

    if(gnome_vfs_set_file_info(f + 1, info, GNOME_VFS_SET_FILE_INFO_PERMISSIONS) != GNOME_VFS_OK)
	goto out_info;

    info->valid_fields = GNOME_VFS_FILE_INFO_FIELDS_CTIME;
    info->atime = fattr->f_atime;
    info->ctime = fattr->f_ctime;
    info->mtime = fattr->f_mtime;

    if(gnome_vfs_set_file_info(f + 1, info, GNOME_VFS_SET_FILE_INFO_TIME) != GNOME_VFS_OK)
	goto out_info;

    res = 0;

  out_info:
    gnome_vfs_file_info_unref(info);
  out:
    return res;
}
