/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.cluster.routing.allocation.decider;

import java.util.Map;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.cluster.ClusterInfo;
import org.elasticsearch.cluster.DiskUsage;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.routing.ExpectedShardSizeEstimator;
import org.elasticsearch.cluster.routing.RecoverySource;
import org.elasticsearch.cluster.routing.RoutingNode;
import org.elasticsearch.cluster.routing.RoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.cluster.routing.allocation.DiskThresholdSettings;
import org.elasticsearch.cluster.routing.allocation.RoutingAllocation;
import org.elasticsearch.cluster.routing.allocation.decider.AllocationDecider;
import org.elasticsearch.cluster.routing.allocation.decider.Decision;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.settings.ClusterSettings;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.snapshots.SnapshotShardSizeInfo;

public class DiskThresholdDecider
extends AllocationDecider {
    private static final Logger logger = LogManager.getLogger(DiskThresholdDecider.class);
    public static final String NAME = "disk_threshold";
    public static final Setting<Boolean> SETTING_IGNORE_DISK_WATERMARKS = Setting.boolSetting("index.routing.allocation.disk.watermark.ignore", false, Setting.Property.IndexScope, Setting.Property.PrivateIndex);
    private final DiskThresholdSettings diskThresholdSettings;
    private static final Decision YES_UNALLOCATED_PRIMARY_BETWEEN_WATERMARKS = Decision.single(Decision.Type.YES, "disk_threshold", "the node is above the low watermark, but less than the high watermark, and this primary shard has never been allocated before", new Object[0]);
    private static final Decision YES_DISK_WATERMARKS_IGNORED = Decision.single(Decision.Type.YES, "disk_threshold", "disk watermarks are ignored on this index", new Object[0]);
    private static final Decision YES_NOT_MOST_UTILIZED_DISK = Decision.single(Decision.Type.YES, "disk_threshold", "this shard is not allocated on the most utilized disk and can remain", new Object[0]);
    private static final Decision YES_DISABLED = Decision.single(Decision.Type.YES, "disk_threshold", "the disk threshold decider is disabled", new Object[0]);
    private static final Decision YES_USAGES_UNAVAILABLE = Decision.single(Decision.Type.YES, "disk_threshold", "disk usages are unavailable", new Object[0]);

    public DiskThresholdDecider(Settings settings, ClusterSettings clusterSettings) {
        this.diskThresholdSettings = new DiskThresholdSettings(settings, clusterSettings);
    }

    public static long sizeOfUnaccountedShards(RoutingNode node, boolean subtractShardsMovingAway, String dataPath, ClusterInfo clusterInfo, SnapshotShardSizeInfo snapshotShardSizeInfo, Metadata metadata, RoutingTable routingTable, long sizeOfUnaccountableSearchableSnapshotShards) {
        ClusterInfo.ReservedSpace reservedSpace = clusterInfo.getReservedSpace(node.nodeId(), dataPath);
        long totalSize = reservedSpace.total();
        for (ShardRouting routing : node.initializing()) {
            String actualPath;
            if (!ExpectedShardSizeEstimator.shouldReserveSpaceForInitializingShard(routing, metadata) || reservedSpace.containsShardId(routing.shardId()) || (actualPath = clusterInfo.getDataPath(routing)) != null && !actualPath.equals(dataPath)) continue;
            totalSize += Math.max(routing.getExpectedShardSize(), ExpectedShardSizeEstimator.getExpectedShardSize(routing, 0L, clusterInfo, snapshotShardSizeInfo, metadata, routingTable));
        }
        totalSize += sizeOfUnaccountableSearchableSnapshotShards;
        if (subtractShardsMovingAway) {
            for (ShardRouting routing : node.relocating()) {
                if (!dataPath.equals(clusterInfo.getDataPath(routing))) continue;
                totalSize -= ExpectedShardSizeEstimator.getExpectedShardSize(routing, 0L, clusterInfo, snapshotShardSizeInfo, metadata, routingTable);
            }
        }
        return totalSize;
    }

    @Override
    public Decision canAllocate(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        boolean skipLowThresholdChecks;
        Map<String, DiskUsage> usages = allocation.clusterInfo().getNodeMostAvailableDiskUsages();
        Decision decision = this.earlyTerminate(usages);
        if (decision != null) {
            return decision;
        }
        if (allocation.metadata().index(shardRouting.index()).ignoreDiskWatermarks()) {
            return YES_DISK_WATERMARKS_IGNORED;
        }
        DiskUsageWithRelocations usage = DiskThresholdDecider.getDiskUsage(node, allocation, usages, false);
        double usedDiskPercentage = usage.getUsedDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        ByteSizeValue total = ByteSizeValue.ofBytes(usage.getTotalBytes());
        if (freeBytes < 0L) {
            long sizeOfRelocatingShards = DiskThresholdDecider.sizeOfUnaccountedShards(node, false, usage.getPath(), allocation.clusterInfo(), allocation.snapshotShardSizeInfo(), allocation.metadata(), allocation.routingTable(), allocation.unaccountedSearchableSnapshotSize(node));
            logger.debug("fewer free bytes remaining than the size of all incoming shards: usage {} on node {} including {} bytes of relocations, preventing allocation", (Object)usage, (Object)node.nodeId(), (Object)sizeOfRelocatingShards);
            return allocation.decision(Decision.NO, NAME, "the node has fewer free bytes remaining than the total size of all incoming shards: free space [%sB], relocating shards [%sB]", freeBytes + sizeOfRelocatingShards, sizeOfRelocatingShards);
        }
        ByteSizeValue freeBytesValue = ByteSizeValue.ofBytes(freeBytes);
        if (logger.isTraceEnabled()) {
            logger.trace("node [{}] has {}% used disk", (Object)node.nodeId(), (Object)usedDiskPercentage);
        }
        boolean bl = skipLowThresholdChecks = shardRouting.primary() && !shardRouting.active() && shardRouting.recoverySource().getType() == RecoverySource.Type.EMPTY_STORE;
        if (freeBytes < this.diskThresholdSettings.getFreeBytesThresholdLowStage(total).getBytes()) {
            if (!skipLowThresholdChecks) {
                if (logger.isDebugEnabled()) {
                    logger.debug("less than the required {} free bytes threshold ({} free) on node {}, preventing allocation", (Object)this.diskThresholdSettings.getFreeBytesThresholdLowStage(total).getBytes(), (Object)freeBytesValue, (Object)node.nodeId());
                }
                return allocation.decision(Decision.NO, NAME, "the node is above the low watermark cluster setting [%s], having less than the minimum required [%s] free space, actual free: [%s], actual used: [%s]", this.diskThresholdSettings.describeLowThreshold(total, true), this.diskThresholdSettings.getFreeBytesThresholdLowStage(total), freeBytesValue, Strings.format1Decimals(usedDiskPercentage, "%"));
            }
            if (freeBytes > this.diskThresholdSettings.getFreeBytesThresholdHighStage(total).getBytes()) {
                if (logger.isDebugEnabled()) {
                    logger.debug("less than the required {} free bytes threshold ({} free) on node {}, but allowing allocation because primary has never been allocated", (Object)this.diskThresholdSettings.getFreeBytesThresholdLowStage(total), (Object)freeBytesValue, (Object)node.nodeId());
                }
                return YES_UNALLOCATED_PRIMARY_BETWEEN_WATERMARKS;
            }
            if (logger.isDebugEnabled()) {
                logger.debug("less than the required {} free bytes threshold ({} free) on node {}, preventing allocation even though primary has never been allocated", (Object)this.diskThresholdSettings.getFreeBytesThresholdHighStage(total).getBytes(), (Object)freeBytesValue, (Object)node.nodeId());
            }
            return allocation.decision(Decision.NO, NAME, "the node is above the high watermark cluster setting [%s], having less than the minimum required [%s] free space, actual free: [%s], actual used: [%s]", this.diskThresholdSettings.describeHighThreshold(total, true), this.diskThresholdSettings.getFreeBytesThresholdHighStage(total), freeBytesValue, Strings.format1Decimals(usedDiskPercentage, "%"));
        }
        long shardSize = ExpectedShardSizeEstimator.getExpectedShardSize(shardRouting, 0L, allocation);
        assert (shardSize >= 0L) : shardSize;
        long freeBytesAfterShard = freeBytes - shardSize;
        if (freeBytesAfterShard < this.diskThresholdSettings.getFreeBytesThresholdHighStage(total).getBytes()) {
            if (logger.isDebugEnabled()) {
                logger.debug("after allocating [{}] node [{}] would be above the high watermark setting [{}], having less than the minimum required {} of free space (actual free: {}, actual used: {}, estimated shard size: {}), preventing allocation", (Object)shardRouting, (Object)node.nodeId(), (Object)this.diskThresholdSettings.describeHighThreshold(total, false), (Object)this.diskThresholdSettings.getFreeBytesThresholdHighStage(total), (Object)freeBytesValue, (Object)Strings.format1Decimals(usedDiskPercentage, "%"), (Object)ByteSizeValue.ofBytes(shardSize));
            }
            return allocation.decision(Decision.NO, NAME, "allocating the shard to this node will bring the node above the high watermark cluster setting [%s] and cause it to have less than the minimum required [%s] of free space (free: [%s], used: [%s], estimated shard size: [%s])", this.diskThresholdSettings.describeHighThreshold(total, true), this.diskThresholdSettings.getFreeBytesThresholdHighStage(total), freeBytesValue, Strings.format1Decimals(usedDiskPercentage, "%"), ByteSizeValue.ofBytes(shardSize));
        }
        assert (freeBytesAfterShard >= 0L) : freeBytesAfterShard;
        return allocation.decision(Decision.YES, NAME, "enough disk for shard on node, free: [%s], used: [%s], shard size: [%s], free after allocating shard: [%s]", freeBytesValue, Strings.format1Decimals(usedDiskPercentage, "%"), ByteSizeValue.ofBytes(shardSize), ByteSizeValue.ofBytes(freeBytesAfterShard));
    }

    @Override
    public Decision canForceAllocateDuringReplace(ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        Map<String, DiskUsage> usages = allocation.clusterInfo().getNodeMostAvailableDiskUsages();
        Decision decision = this.earlyTerminate(usages);
        if (decision != null) {
            return decision;
        }
        if (allocation.metadata().index(shardRouting.index()).ignoreDiskWatermarks()) {
            return YES_DISK_WATERMARKS_IGNORED;
        }
        DiskUsageWithRelocations usage = DiskThresholdDecider.getDiskUsage(node, allocation, usages, false);
        long shardSize = ExpectedShardSizeEstimator.getExpectedShardSize(shardRouting, 0L, allocation);
        assert (shardSize >= 0L) : shardSize;
        long freeBytesAfterShard = usage.getFreeBytes() - shardSize;
        if (freeBytesAfterShard < 0L) {
            return Decision.single(Decision.Type.NO, NAME, "unable to force allocate shard to [%s] during replacement, as allocating to this node would cause disk usage to exceed 100%% ([%s] bytes above available disk space)", node.nodeId(), -freeBytesAfterShard);
        }
        return super.canForceAllocateDuringReplace(shardRouting, node, allocation);
    }

    @Override
    public Decision canRemain(IndexMetadata indexMetadata, ShardRouting shardRouting, RoutingNode node, RoutingAllocation allocation) {
        if (!shardRouting.currentNodeId().equals(node.nodeId())) {
            throw new IllegalArgumentException("Shard [" + String.valueOf(shardRouting) + "] is not allocated on node: [" + node.nodeId() + "]");
        }
        ClusterInfo clusterInfo = allocation.clusterInfo();
        Map<String, DiskUsage> usages = clusterInfo.getNodeLeastAvailableDiskUsages();
        Decision decision = this.earlyTerminate(usages);
        if (decision != null) {
            return decision;
        }
        if (indexMetadata.ignoreDiskWatermarks()) {
            return YES_DISK_WATERMARKS_IGNORED;
        }
        DiskUsageWithRelocations usage = DiskThresholdDecider.getDiskUsage(node, allocation, usages, true);
        String dataPath = clusterInfo.getDataPath(shardRouting);
        double freeDiskPercentage = usage.getFreeDiskAsPercentage();
        long freeBytes = usage.getFreeBytes();
        double usedDiskPercentage = usage.getUsedDiskAsPercentage();
        ByteSizeValue total = ByteSizeValue.ofBytes(usage.getTotalBytes());
        if (logger.isTraceEnabled()) {
            logger.trace("node [{}] has {}% free disk ({} bytes)", (Object)node.nodeId(), (Object)freeDiskPercentage, (Object)freeBytes);
        }
        if (dataPath == null || !usage.getPath().equals(dataPath)) {
            return YES_NOT_MOST_UTILIZED_DISK;
        }
        if (freeBytes < 0L) {
            long sizeOfRelocatingShards = DiskThresholdDecider.sizeOfUnaccountedShards(node, true, usage.getPath(), allocation.clusterInfo(), allocation.snapshotShardSizeInfo(), allocation.metadata(), allocation.routingTable(), allocation.unaccountedSearchableSnapshotSize(node));
            logger.debug("fewer free bytes remaining than the size of all incoming shards: usage {} on node {} including {} bytes of relocations, shard cannot remain", (Object)usage, (Object)node.nodeId(), (Object)sizeOfRelocatingShards);
            return allocation.decision(Decision.NO, NAME, "the shard cannot remain on this node because the node has fewer free bytes remaining than the total size of all incoming shards: free space [%s], relocating shards [%s]", freeBytes + sizeOfRelocatingShards, sizeOfRelocatingShards);
        }
        if (freeBytes < this.diskThresholdSettings.getFreeBytesThresholdHighStage(total).getBytes()) {
            if (logger.isDebugEnabled()) {
                logger.debug("node {} is over the high watermark setting [{}], having less than the required {} free space (actual free: {}, actual used: {}), shard cannot remain", (Object)node.nodeId(), (Object)this.diskThresholdSettings.describeHighThreshold(total, false), (Object)this.diskThresholdSettings.getFreeBytesThresholdHighStage(total), (Object)freeBytes, (Object)Strings.format1Decimals(usedDiskPercentage, "%"));
            }
            return allocation.decision(Decision.NO, NAME, "the shard cannot remain on this node because it is above the high watermark cluster setting [%s] and there is less than the required [%s] free space on node, actual free: [%s], actual used: [%s]", this.diskThresholdSettings.describeHighThreshold(total, true), this.diskThresholdSettings.getFreeBytesThresholdHighStage(total), ByteSizeValue.ofBytes(freeBytes), Strings.format1Decimals(usedDiskPercentage, "%"));
        }
        return allocation.decision(Decision.YES, NAME, "there is enough disk on this node for the shard to remain, free: [%s]", ByteSizeValue.ofBytes(freeBytes));
    }

    private static DiskUsageWithRelocations getDiskUsage(RoutingNode node, RoutingAllocation allocation, Map<String, DiskUsage> usages, boolean subtractLeavingShards) {
        DiskUsage usage = usages.get(node.nodeId());
        if (usage == null) {
            usage = DiskThresholdDecider.averageUsage(node, usages);
            logger.debug("unable to determine disk usage for {}, defaulting to average across nodes [{} total] [{} free] [{}% free]", (Object)node.nodeId(), (Object)usage.totalBytes(), (Object)usage.freeBytes(), (Object)usage.freeDiskAsPercentage());
        }
        DiskUsageWithRelocations diskUsageWithRelocations = new DiskUsageWithRelocations(usage, DiskThresholdDecider.sizeOfUnaccountedShards(node, subtractLeavingShards, usage.path(), allocation.clusterInfo(), allocation.snapshotShardSizeInfo(), allocation.metadata(), allocation.routingTable(), allocation.unaccountedSearchableSnapshotSize(node)));
        logger.trace("getDiskUsage(subtractLeavingShards={}) returning {}", (Object)subtractLeavingShards, (Object)diskUsageWithRelocations);
        return diskUsageWithRelocations;
    }

    static DiskUsage averageUsage(RoutingNode node, Map<String, DiskUsage> usages) {
        if (usages.size() == 0) {
            return new DiskUsage(node.nodeId(), node.node().getName(), "_na_", 0L, 0L);
        }
        long totalBytes = 0L;
        long freeBytes = 0L;
        for (DiskUsage du : usages.values()) {
            totalBytes += du.totalBytes();
            freeBytes += du.freeBytes();
        }
        return new DiskUsage(node.nodeId(), node.node().getName(), "_na_", totalBytes / (long)usages.size(), freeBytes / (long)usages.size());
    }

    private Decision earlyTerminate(Map<String, DiskUsage> usages) {
        if (!this.diskThresholdSettings.isEnabled()) {
            return YES_DISABLED;
        }
        if (usages.isEmpty()) {
            logger.trace("unable to determine disk usages for disk-aware allocation, allowing allocation");
            return YES_USAGES_UNAVAILABLE;
        }
        return null;
    }

    record DiskUsageWithRelocations(DiskUsage diskUsage, long relocatingShardSize) {
        double getFreeDiskAsPercentage() {
            if (this.getTotalBytes() == 0L) {
                return 100.0;
            }
            return 100.0 * (double)this.getFreeBytes() / (double)this.getTotalBytes();
        }

        double getUsedDiskAsPercentage() {
            return 100.0 - this.getFreeDiskAsPercentage();
        }

        long getFreeBytes() {
            try {
                return Math.subtractExact(this.diskUsage.freeBytes(), this.relocatingShardSize);
            }
            catch (ArithmeticException e) {
                return Long.MAX_VALUE;
            }
        }

        String getPath() {
            return this.diskUsage.path();
        }

        long getTotalBytes() {
            return this.diskUsage.totalBytes();
        }
    }
}

