/***************************************************************************
 *   crengine-ng                                                           *
 *   Copyright (C) 2007 Vadim Lopatin <coolreader.org@gmail.com>           *
 *   Copyright (C) 2022 Aleksey Chernov <valexlin@gmail.com>               *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU General Public License           *
 *   as published by the Free Software Foundation; either version 2        *
 *   of the License, or (at your option) any later version.                *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the Free Software           *
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,            *
 *   MA 02110-1301, USA.                                                   *
 ***************************************************************************/

#ifndef __LVCACHEDSTREAM_H_INCLUDED__
#define __LVCACHEDSTREAM_H_INCLUDED__

#include "lvnamedstream.h"

class LVCachedStream: public LVNamedStream
{
private:
#define CACHE_BUF_BLOCK_SHIFT 12
#define CACHE_BUF_BLOCK_SIZE  (1 << CACHE_BUF_BLOCK_SHIFT)
    class BufItem
    {
    public:
        lUInt32 start;
        lUInt32 size;
        BufItem* prev;
        BufItem* next;
        lUInt8 buf[CACHE_BUF_BLOCK_SIZE];

        int getIndex() {
            return start >> CACHE_BUF_BLOCK_SHIFT;
        }
        BufItem()
                : prev(NULL)
                , next(NULL) { }
    };

    LVStreamRef m_stream;
    int m_bufSize;
    lvsize_t m_size;
    lvpos_t m_pos;
    BufItem** m_buf;
    BufItem* m_head;
    BufItem* m_tail;
    int m_bufItems;
    int m_bufLen;

    /// add item to head
    BufItem* addNewItem(int start);
    /// move item to top
    void moveToTop(int index);
    /// reuse existing item from tail of list
    BufItem* reuseItem(int start);
    /// read item content from base stream
    bool fillItem(BufItem* item);
    BufItem* addOrReuseItem(int start) {
        //assert( !(m_head && !m_tail) );
        if (m_bufLen < m_bufSize)
            return addNewItem(start);
        else
            return reuseItem(start);
    }
    /// checks several items, loads if necessary
    bool fillFragment(int startIndex, int count);
public:
    LVCachedStream(LVStreamRef stream, int bufSize);
    virtual ~LVCachedStream();

    /// fastly return already known CRC
    virtual lverror_t getcrc32(lUInt32& dst) {
        return m_stream->getcrc32(dst);
    }

    /// fastly return already known SHA256
    virtual lverror_t getsha256(lString8& dst) {
        return m_stream->getsha256(dst);
    }

    virtual bool Eof() {
        return m_pos >= m_size;
    }
    virtual lvsize_t GetSize() {
        return m_size;
    }

    virtual lverror_t Seek(lvoffset_t offset, lvseek_origin_t origin, lvpos_t* newPos);

    virtual lverror_t Write(const void*, lvsize_t, lvsize_t*) {
        return LVERR_NOTIMPL;
    }

    virtual lverror_t Read(void* buf, lvsize_t size, lvsize_t* pBytesRead);

    virtual lverror_t SetSize(lvsize_t) {
        return LVERR_NOTIMPL;
    }
};

#endif // __LVCACHEDSTREAM_H_INCLUDED__
