/*  fermat.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifdef HAVE_FERMAT

#include "fermat.h"
#include <sstream>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <cerrno>
#include <cstring> // strerror
#include "fdstream.h"
#include <signal.h>
#include "files.h"

#include "functions.h" // for macros ERROR(), LOG(), LOGX()
//#define ERROR(str) (perror(str), exit(1))
//#define LOG(str) (cout << str << '\n')
//#define LOGX(str) (cout << str << '\n')

using namespace std;
using namespace GiNaC;

namespace Reduze {

struct FermatPrivate {
	pid_t fermat_pid;
	pid_t guard_pid; // guard to terminate fermat if parent dies unexpectedly
	GiNaC::lst symbols;
	std::string fermat_path;
	int fd_to_fermat[2]; // read/write side of pipe to fermat executable
	int fd_from_fermat[2]; // read/write side of pipe from fermat executable
	fdostream* to_fermat; // stream as convenience interface to pipe
	fdistream* from_fermat; // stream as convenience interface to pipe
};

Fermat::Fermat() {
	d = new FermatPrivate;
	d->fermat_pid = -1;
	d->guard_pid = -1;
	d->to_fermat = 0;
	d->from_fermat = 0;
}

Fermat::~Fermat() {
	close();
	delete d;
}

Fermat* Fermat::instance() {
	static Fermat* f = 0;
	if (f == 0)
		f = new Fermat();
	return f;
}

bool Fermat::is_initialized() const {
	return d->fermat_pid != -1;
}

bool Fermat::is_proper_executable_path(const string& exec, string& msg) {
	if (!is_readable_file(exec)) {
		msg = "Fermat exectuable not readable: " + exec;
		return false;
	}
	std::string dir = get_directory_of_filename(exec) + "BACKWARD";
	if (exec != "" && !is_readable_directory(dir)) {
		msg = "Fermat directory not readable: " + dir;
		return false;
	}
	return true;
}

void Fermat::init(const lst& symbols, const string& exec, bool use_laurent_polynomials) {
	if (is_initialized())
		throw std::runtime_error("Fermat already initialized.");
	string msg;
	if (!is_proper_executable_path(exec, msg))
		throw std::runtime_error(msg);
	for (lst::const_iterator s = symbols.begin(); s != symbols.end(); ++s) {
		VERIFY(is_a<symbol>(*s));
		string name = ex_to<symbol> (*s).get_name();
		if (name.empty() || islower(name[0]) == 0)
			ABORT("Fermat variable name \"" << name << "\" must start with a lower case letter.");
	}
	d->symbols = symbols;
	d->fermat_path = exec;
	fork_fermat_executable();
	configure_fermat_executable(use_laurent_polynomials);
}

void Fermat::fork_guard() { // to be called after fork_fermat_executable()
	ASSERT(d->fermat_pid > 0);
	int fd_dummy[2];
	int ret = pipe(fd_dummy);
	VERIFY(ret == 0);
	// start a guard which kills Fermat if we die unexpectedly
	d->guard_pid = fork();
	if (d->guard_pid == 0) { // we are the child
		::close(fd_dummy[1]);
		// ignore various signals to get a chance to someone killing Fermat
		sigset_t mask;
		sigfillset(&mask);
		sigprocmask(SIG_BLOCK, &mask, 0);
		//signal(SIGPIPE, SIG_IGN); // ignore broken pipe
		int dummy;
		int ret = ::read(fd_dummy[0], &dummy, sizeof(int)); // returns only upon death of parent
		VERIFY(ret >= 0);
		//cerr << "noticed parent's death, killing Fermat\n";
		kill(d->fermat_pid, SIGKILL);
		exit(0);
	} else if (d->guard_pid != -1) { // we are the parent
		::close(fd_dummy[0]);
		// guard might get killed first, then parent should kill Fermat
		//signal(SIGCHLD, process_sigchld);
	}
	if (errno)
		ERROR("fork of guard process failed\n"
				<< "error code " << errno << ": " << strerror(errno));
	//cerr << "forking guard succeeded\n";
}

void Fermat::fork_fermat_executable() {
	int ret1 = pipe(d->fd_to_fermat);
	int ret2 = pipe(d->fd_from_fermat);
	VERIFY(ret1 == 0 && ret2 == 0);

	d->fermat_pid = fork();

	if (d->fermat_pid == 0) { // we are the child, will become fermat executable
		// close STDIN and replace it with read side of pipe to fermat
		dup2(d->fd_to_fermat[0], fileno(stdin));
		// close STDOUT and replace it with write side of pipe from fermat
		dup2(d->fd_from_fermat[1], fileno(stdout));
		// replace current process with fermat executable
		execlp(d->fermat_path.c_str(), d->fermat_path.c_str(), NULL);
	} else if (d->fermat_pid != -1) { // we are the parent
		// close read side of pipe to fermat
		::close(d->fd_to_fermat[0]);
		// close write side of pipe from fermat
		::close(d->fd_from_fermat[1]);
		d->to_fermat = new fdostream(d->fd_to_fermat[1]);
		d->from_fermat = new fdistream(d->fd_from_fermat[0]);
		fork_guard();
	}
	if (errno)
		ERROR("fork of Fermat failed\n"
				<< "executable: \"" << d->fermat_path << "\"\n"
				<< "error code " << errno << ": " << strerror(errno));
	//cerr << "forking fermat succeeded\n";
}

// read from stream until a line (ignoring leading spaces) starts with given tag
void skip_upto(istream& is, const string& tag) {
	string line;
	size_t pos;
	do {
		getline(is, line);
		//cerr << "got line: '" << line << "'" << endl;
		pos = line.find_first_not_of(" ");
	} while (pos == string::npos || line.compare(pos, tag.size(), tag) != 0);
}

void Fermat::configure_fermat_executable(bool use_laurent_polynomials) {
	fdistream& from = *d->from_fermat;
	fdostream& to = *d->to_fermat;

	// send empty command to trigger full output of Fermat's welcome message
	to << "\n" << flush;
	skip_upto(from, "Elapsed CPU time"); // skip welcome message
	skip_upto(from, "Elapsed CPU time"); // skip confirmation of empty command

	// switch off floating point representation
	to << "&d\n0\n" << flush;
	skip_upto(from, "Elapsed CPU time"); // read &d confirmation

	// set prompt to empty line
	// do that before switching to "ugly" printing b.c. feris64 hangs (bug?)
	to << "&(M=' ')\n" << flush;
	skip_upto(from, "Elapsed CPU time");

	// switch on "ugly" printing: no spaces in integers, '*' for multiplication
	to << "&U\n" << flush;
	skip_upto(from, "Elapsed CPU time");

	// switch off suppression of output to terminal of long polys
	to << "&(_s=0)\n" << flush;
	skip_upto(from, "Elapsed CPU time");

	// switch off timing
	to << "&(t=0)\n" << flush;
	skip_upto(from, "0");

	// set polynomial variables
	//cerr << "configuring fermat with symbols: " << d->symbols;
	for (lst::const_iterator s = d->symbols.begin(); s != d->symbols.end(); ++s) {
		to << "&(J=" << *s << ")\n" << flush;
		skip_upto(from, "0");
	}

	if (use_laurent_polynomials)
	   ABORT("fermat 4.25 produced WRONG RESULTS with Laurent polys,"
			 " so please don't enabled it unless you know what you are doing !");

	if (use_laurent_polynomials) {
		// enable Laurent polynomials
		to << "&l\n" << flush;
		skip_upto(from, "0");
	}

	// Gausss-Bareiss
	//to << "&(K=1)\n" << flush;
	//to << "&(D=2)\n" << flush;

	// Gaussian elimination
	//to << "&(K=0)\n" << flush;
	//to << "&(D=1)\n" << flush;

	// Langrangian interpolation for dim > 3
	//to << "&(D=-1)\n" << flush;
	//to << "&(L=1)\n" << flush;

	verify_in_sync();
}

void Fermat::verify_in_sync() const {
	// check we are still in sync with the streams
	while (true) {
		fdistream& from = *d->from_fermat;
		fdostream& to = *d->to_fermat;

		/*
		 to << "2*3*5*7*11/(7-2)\n" << flush;
		 skip_upto(from, "462");
		 if (!from || !to)
		 break;
		 */

		string rstr = to_string(std::rand());
		to << rstr << "\n" << flush;
		skip_upto(from, rstr);
		if (!from || !to)
			break;

		return;
	}
	ERROR("communication problem with Fermat");
}

const GiNaC::lst& Fermat::get_symbols() const {
	ASSERT(is_initialized());
	return d->symbols;
}

void Fermat::set_symbols(const GiNaC::lst& symbs) {
	ASSERT(is_initialized());
	fdistream& from = *d->from_fermat;
	fdostream& to = *d->to_fermat;

	// delete old symbols
	for (lst::const_iterator s = d->symbols.begin(); s != d->symbols.end(); ++s) {
		to << "&(J=-" << *s << ")\n" << flush;
		skip_upto(from, "0");
	}

	// add new symbols
	d->symbols = symbs;
	for (lst::const_iterator s = d->symbols.begin(); s != d->symbols.end(); ++s) {
		to << "&(J=" << *s << ")\n" << flush;
		skip_upto(from, "0");
	}

}

ex Fermat::normalize(const ex& expr) {
	fdistream& from = *d->from_fermat;
	fdostream& to = *d->to_fermat;

	//cerr << "asking Fermat to simplify: '" << e << "'\n";
	// send all in one long line
	to << expr << "\n" << flush;
	// avoid long lines: split into multiple lines with line cont mark `
	/*
	 stringstream ss;
	 ss << expr;
	 string s = ss.str();
	 size_t chunksize = 80;
	 stringstream cto;
	 const char* sd = s.data();
	 size_t pos = 0, linesize = 0;
	 do {
	 size_t next;
	 if (pos + 1 >= s.size() || //
	 (next = s.find_first_of("+-*()/", pos + 1)) == string::npos)
	 next = s.size();
	 size_t len = next - pos;
	 if (linesize > 0 && linesize + len > chunksize) {
	 to << "`\n" << flush;
	 cto << "`\n";
	 linesize = 0;
	 }
	 to.write(&sd[pos], len);
	 //cto.write(&sd[pos], len);
	 linesize += len;
	 pos = next;
	 } while (pos < s.size());
	 to << "\n" << flush;
	 cto << "\n";
	 */
	//cerr << "sending to fermat: '" << cto.str() << "'" << endl;

	// assemble result:
	// ignore leading empty lines, read something non-empty, stop on '\n\n'
	string answer, l;
	string fullanswer; // including newlines to display errors in a nice way
	do {
		getline(from, l);
		size_t first = l.find_first_not_of(" "); // strip leading spaces
		size_t last = l.find_last_not_of(" `"); // strip trailing spaces and '
		if (first == string::npos || last == string::npos)
			continue;
		answer.append(l, first, last - first + 1);
		fullanswer.append(l + "\n");
	} while (answer.empty() || !l.empty());
	if (!from || !to)
		ERROR("communication problem with Fermat during normalize");
	// parse result with GiNaC
	ex res;
	try {
		res = ex(answer, d->symbols);
	} catch (exception& e) {
		LOGXX("failed Fermat request:\n" << expr);
		ERROR("Failed to parse expression simplified by Fermat:\n" << fullanswer
				<< "\n" << e.what() << "");
	}

	verify_in_sync();

	return res;
}

//std::string Fermat::determinant(const GiNaC::matrix& mat) {
//	VERIFY(mat.cols() == mat.rows());
//	fdistream& from = *d->from_fermat;
//	fdostream& to = *d->to_fermat;
//
//	string mat_name = "xmatrix";
//
//	unsigned dim = mat.cols();
//	to << "Array " << mat_name << "[" << dim << "," << dim << "]\n" << flush;
//	//skip_upto(from, "0");
//
//	for (unsigned i = 0; i < dim; ++i)
//		for (unsigned j = 0; j < dim; ++j) {
//			to << mat_name << "[" << i + 1 << "," << j + 1 << "] := ";
//			to << mat(i, j) << "\n" << flush;
//		}
//
//	to << "123456789\n" << flush;
//	skip_upto(from, "123456789");
//
//	to << "Det[" << mat_name << "]\n" << flush;
//
//	string answer, l;
//	do {
//		getline(from, l);
//		size_t first = l.find_first_not_of(" "); // strip leading spaces
//		size_t last = l.find_last_not_of(" `"); // strip trailing spaces and '
//		if (first == string::npos || last == string::npos)
//			continue;
//		answer.append(l, first, last - first + 1);
//	} while (answer.empty() || !l.empty());
//	//cerr << "got Fermat's answer: '" << answer << "'\n";
//
//	if (!from || !to)
//		ERROR("communication problem with Fermat during normalize");
//
//	//	ex res;
//	//	try {
//	//		res = ex(answer, d->symbols);
//	//	} catch (exception& e) {
//	//		//cerr << "request was: '" << cto.str() << "'" << endl;
//	//		throw std::runtime_error(
//	//				string(
//	//						"Failed to parse expression (determinant) simplified by Fermat:\n")
//	//						+ answer + "\n" + e.what());
//	//	}
//	//	return res.expand();
//
//	return answer;
//}

void Fermat::close() {
	delete d->from_fermat;
	delete d->to_fermat;
	d->from_fermat = 0;
	d->to_fermat = 0;
	if (d->fermat_pid != -1) {
		LOGX("Killing Fermat");
		kill(d->fermat_pid, SIGKILL);
		kill(d->guard_pid, SIGKILL);
		waitpid(d->fermat_pid, 0, 0); // cleanly wait for child's return
		waitpid(d->guard_pid, 0, 0); // cleanly wait for child's return
	}
	d->fermat_pid = -1;
	d->guard_pid = -1;
}

}

#endif // HAVE_FERMAT
