/*  filedata.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef FILEDATA_H_
#define FILEDATA_H_

#include <fstream>
#include <string>
#include <set>
#include <ginac/ginac.h>

namespace Reduze {

class LinearCombination;
class LinearCombinationGeneric;
class Identity;
class EquationHLight;
class LinCombHLight;

/// data output stream wrapper for different file formats
// note: alternative approaches would be possible for output format problem:
// a) of->mywrite(outidentity); <- needs one virtual method for each type
//    (can't template virtuals)
// b) of << outidentity; <- needs format info flag for of
class OutFileData {
public:
	static std::string default_filename_suffix(const std::string& format_str);

	OutFileData(const std::string& filename, const std::string& format_str);
	~OutFileData();

	/** closes the stream and renames the temporary file
	 ** must always be called when writing is complete **/
	void finalize();
	size_t tellp() {
		return raw_ostream_.tellp();
	}
	std::string format_string() const {
		switch (format_) {
		case MMA:
			return "mma";
		case MAPLE:
			return "maple";
		case FORM:
			return "form";
		case Reduze:
			return "reduze";
		}
	}
	void write_comment(const std::string& s);

protected:
	template<class T>
	OutFileData& operator<<(const T& t);
	// specialized Reduze output method
	template<class T>
	void write_reduze(const T& t);
	void close();

	/// output format
	enum Format {
		MMA, MAPLE, FORM, Reduze
	};
	inline Format format() const {
		return format_;
	}

private:
	Format format_;
	std::ofstream raw_ostream_;
	std::string filename_, tmp_filename_;
	bool empty_;
};

class InFileData {
public:
	InFileData(const std::string& filename);
	operator bool() const {
		return !raw_istream_.fail();
	}
	bool good() const {
		return raw_istream_.good();
	}
	void close() {
		raw_istream_.close();
	}
	bool fail() const {
		return raw_istream_.fail();
	}
	size_t tellg() {
		return raw_istream_.tellg();
	}
	size_t file_size();
	std::string filename() const {
		return filename_;
	}
protected:
	/// returns next non-empty line after stripping comments, lead/trail spaces
	/** empty return value signals end of stream or other failure **/
	InFileData& content_line(std::string&);
private:
	std::string filename_;
	std::ifstream raw_istream_;
};

/* some idea:
 class DataInfo: public YAMLAutoConfigurable {
 public:
 static YAMLSpec yaml_spec() {
 YAMLSpec s;
 s.set_keyword("data_info");
 s.set_short_description("Information about data file contents.");
 s.set_long_description(s.short_description());
 s.add_option("data_type", true, "string", ""//
 "Data type a file contains. Must be one out of:"
 "integrals, identities, linear_combinations");
 s.add_option("version", false, "string",
 "Version tag for the file format used. Must be 2.0");
 s.add_options(Job::yaml_spec());
 return s;
 }
 virtual YAMLSpec yaml_spec_link() const {
 return yaml_spec();
 }
 DataInfo() {
 add_auto_io("data_type", data_type_);
 add_auto_io("version", version_);
 }
 std::string data_type() const {
 return data_type_;
 }
 std::string version() const {
 return version_;
 }
 protected:
 void init() {
 if (data_type_ != "integrals" && data_type_ != "identities"
 && data_type_ != "linear_combinations")
 throw runtime_error("Unknown data type " + data_type_);
 if (version_ != "2.0")
 throw runtime_error("Unknown version " + version_);
 }
 private:
 std::string data_type_, version_;
 };
 */

template<class T>
void OutFileData::write_reduze(const T& t) {
	if (format_ != Reduze)
		throw std::runtime_error("wrong output format");
	if (!empty_)
		raw_ostream_ << "\n";
	raw_ostream_ << t;
	empty_ = false;
}

template<class T>
OutFileData& OutFileData::operator<<(const T& t) {
	switch (format_) {
	case Reduze:
		if (!empty_)
			raw_ostream_ << "\n";
		raw_ostream_ << t;
		break;
	case MMA:
		if (!empty_)
			raw_ostream_ << ",\n\n";
		t.to_mma_stream(raw_ostream_);
		break;
	case MAPLE:
		if (!empty_)
			raw_ostream_ << ",\n\n";
		t.to_maple_stream(raw_ostream_);
		break;
	case FORM:
		if (!empty_)
			raw_ostream_ << "\n";
		t.to_form_stream(raw_ostream_);
		break;
	}
	empty_ = false;
	return *this;
}

class INT;

class OutFileINTs: public OutFileData {
public:
	OutFileINTs(const std::string& filename, const std::string& format_str =
			"reduze") :
		OutFileData(filename, format_str) {
	}
	OutFileINTs& operator<<(const INT&);
	OutFileINTs& operator<<(const std::set<INT>&);
};

class InFileINTs: public InFileData {
public:
	InFileINTs(const std::string& filename) :
		InFileData(filename) {
		set_content_line();
	}
	/// get all integrals in a set
	void get_all(std::set<INT>& ints);
	/// get all integrals as a list
	void get_all(std::list<INT>& ints);
	/// returns next integral from stream
	// the stream InFileINTs must be tested on !fail() before using this function
	INT get_next();
private:
	void set_content_line();
	std::string line_;
};

class OutFileLinearCombinations: public OutFileData {
public:
	OutFileLinearCombinations(const std::string& filename,
			const std::string& format_str = "reduze") :
		OutFileData(filename, format_str) {
	}
	OutFileLinearCombinations& operator<<(const LinearCombination& lc);
};


class InFileLinearCombinations: public InFileData {
public:
	/// returns true if the file seems to contain linear combinations
	static bool check_filetype(const std::string& filename);
	/// finds all integrals in the file
	template<class TSet>
	static void find_INT(const std::string& filename, TSet& ints);
	InFileLinearCombinations(const std::string& filename);
	InFileLinearCombinations& get(LinearCombination& lc);
	InFileLinearCombinations& get(LinCombHLight& lc, bool discard_coeff=false);
    // same as get but without abort on failure
	InFileLinearCombinations& get_noabort(LinCombHLight& lc, bool disc_coeff);

private:
	GiNaC::lst symbols_;
};


class OutFileLinearCombinationsGeneric: public OutFileData {
public:
	OutFileLinearCombinationsGeneric(const std::string& filename,
			const std::string& format_str = "reduze") :
		OutFileData(filename, format_str) {
	}
	OutFileLinearCombinationsGeneric& operator<<(
			const LinearCombinationGeneric& lc);
};

class InFileLinearCombinationsGeneric: public InFileData {
public:
	InFileLinearCombinationsGeneric(const std::string& filename);
	InFileLinearCombinationsGeneric& get(LinearCombinationGeneric& lc);

private:
	GiNaC::lst symbols_;
};

class OutFileEquations: public OutFileData {
public:
	OutFileEquations(const std::string& filename,
			const std::string& format_str = "reduze") :
		OutFileData(filename, format_str) {
	}
	OutFileEquations& operator<<(const Identity& lc);
	OutFileEquations& operator<<(const EquationHLight& lc);
};

class InFileEquations: public InFileData {
public:
	/// returns true if the file seems to contain equations
	static bool check_filetype(const std::string& filename);
	InFileEquations(const std::string& filename);
	InFileEquations& get(Identity&);
	InFileEquations& get(EquationHLight&);
private:
	// same as get but without abort on failure
	InFileEquations& get_noabort(EquationHLight&);
	GiNaC::lst symbols_;
};

}

#endif /* FILE_DATA_H_ */
