/*  integralfamily.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "integralfamily.h"
#include "kinematics.h"
#include "equationlist.h"
#include "int.h"
#include "functions.h"
#include "sector.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "files.h"

using namespace std;

namespace Reduze {

namespace {
YAMLProxy<IntegralFamily> dummy;
}
/*
 struct IntegralFamilyPrivate {
 /// integrals which are preferred
 std::set<INT> preferred_integrals_;
 };

 IntegralFamily& IntegralFamily::operator=(const IntegralFamily& o) {
 if (&o == this)
 return *this;
 delete d;
 d = new IntegralFamilyPrivate(o.d);
 return *this;
 }
 */

// private default constructor
IntegralFamily::IntegralFamily() :
		source_(this), id_(0), kinematics_(0), cut_propagators_mask_(0) {
}

IntegralFamily::IntegralFamily(const Kinematics* kin) :
		source_(this), id_(0), kinematics_(kin), cut_propagators_mask_(0) {
}

/* see class Files
 IntegralFamily::IntegralFamily(const IntegralFamily* ic1,
 const IntegralFamily* ic2) :
 Kinematics(*Files::instance()->kinematics()) {
 using namespace GiNaC;

 string name1 = ic1->name();
 string name2 = ic2->name();

 LOGX("Creating product of integral families: " << name1 << ", " << name2);

 name_ = name1 + "_X_" + name2;

 // new symbols for the loop momenta
 exmap old2newA, old2newB;
 lst loop_momenta1 = ic1->loop_momenta();
 lst loop_momenta2 = ic2->loop_momenta();
 int num_mom1 = loop_momenta1.nops();
 int num_mom2 = loop_momenta2.nops();
 int num_mom = num_mom1 + num_mom2;

 for (int i = 1; i <= num_mom1; ++i) {
 symbol s(get_loop_momentum_prefix() + to_string(i));
 loop_momenta_.append(s);
 old2newA[loop_momenta1.op(i - 1)] = s;
 }

 for (int i = num_mom1 + 1; i <= num_mom; ++i) {
 symbol s(get_loop_momentum_prefix() + to_string(i));
 loop_momenta_.append(s);
 old2newB[loop_momenta2.op(i - num_mom1 - 1)] = s;
 }

 vector<Propagator> propagators1 = ic1->get_propagators();
 vector<Propagator> propagators2 = ic2->get_propagators();
 int num_props = propagators1.size() + propagators2.size();
 propagators_.reserve(num_props + num_mom1 * num_mom2);
 vector<Propagator>::const_iterator it;
 for (it = propagators1.begin(); it != propagators1.end(); ++it)
 propagators_.push_back(ex_to<Propagator> (it->subs(old2newA)));
 for (it = propagators2.begin(); it != propagators2.end(); ++it)
 propagators_.push_back(ex_to<Propagator> (it->subs(old2newB)));

 // additional propagators
 for (int i = 0; i < num_mom1; ++i) {
 for (int j = 0; i < num_mom2; ++j) {
 ex mom1 = loop_momenta1.op(i).subs(old2newA);
 ex mom2 = loop_momenta2.op(j).subs(old2newB);
 ex mom = mom1 - mom2;
 ex sqmass = 0;
 propagators_.push_back(Propagator(mom, sqmass));
 }
 }

 int num_props1 = propagators1.size();
 permutation_symmetries_ = ic1->permutation_symmetries_;
 map<int, vector<vector<int> > > permutation_symmetries_2 = ic2->permutation_symmetries_;
 map<int, vector<vector<int> > >::iterator mit;
 for (mit = permutation_symmetries_2.begin(); mit != permutation_symmetries_2.end(); ++mit) {
 vector<vector<int> >::iterator it1;
 for (it1 = mit->second.begin(); it1 != mit->second.end(); ++it1) {
 vector<int>::iterator it2;
 for (it2 = it1->begin(); it2 != it1->end(); ++it2) {
 *it2 += num_props1;
 }
 }
 permutation_symmetries_[permutation_symmetries_.size() + 1] = mit->second;
 }

 init();
 check();
 LOGX("Product integral family: " << name() << " created");
 }
 */

IntegralFamily::IntegralFamily(const Kinematics* kin, const std::string& name,
		const GiNaC::lst& loop_momenta,
		const std::vector<Propagator>& propagators,
		int cut_propagators_mask,
		const std::list<Permutation>* perms) :
		source_(this), name_(name), id_(0), kinematics_(kin),
		loop_momenta_(loop_momenta), propagators_(propagators),
		cut_propagators_mask_(cut_propagators_mask) {
	if (perms) {
		permutation_symmetries_ = *perms;
		incomplete_permutation_symmetries_ = *perms;
	}
	init(false);
}

IntegralFamily::~IntegralFamily() {
}

IntegralFamily* IntegralFamily::create_derived_integralfamily(const string& name) {
	LOGX("checking for derived integral family '" << name << "'");
	string basename, suffix;
	int shift;
	if (!Kinematics::search_dimension_shift_suffix(name, basename, suffix, shift))
		return 0;
	IntegralFamily* base = Files::instance()->integralfamily(basename);
	if (!base)
		return 0;
	const Kinematics* basekin = base->kinematics_;
	if (!basekin)
		return 0;
	Kinematics* kin = Kinematics::create_derived_kinematics(basekin->name() + suffix);
	if (!kin)
		return 0;
	LOGX("creating integral family from '" << basename << "' with dim shift " << shift);
	IntegralFamily* f = new IntegralFamily(*base);
	f->source_ = base;
    f->name_ = name;
    f->kinematics_ = kin;
    return f;
}

void IntegralFamily::set_id(int id) {
	id_ = id;
}

void IntegralFamily::set_name(const std::string& name) {
	name_ = name;
}

GiNaC::lst IntegralFamily::get_all_momenta() const {
	return add_lst(loop_momenta_, kinematics_->external_momenta());
}

int IntegralFamily::get_t(int sector_id) const {
	return Sector::t_of_id(sector_id);
}

unsigned IntegralFamily::num_scalar_products() const {
	return num_scalar_products(loop_momenta().nops(),
			kinematics_->independent_external_momenta().nops());
}

unsigned IntegralFamily::num_scalar_products(int NoLM, int NoEM) {
	int res = (NoLM + 1) * NoLM;
	res /= 2;
	res += NoLM * NoEM;
	return res;
}

int IntegralFamily::propagator_type(int i) const {
	if (i < 0 || i >= (int) propagators_.size())
		throw range_error("Propagator " + to_string(i) + " does not exist!");
	// standard
	if (propagators_[i].has_squared_momentum()) {
		if (is_cut_propagator(i))
			return PropagatorType::standard_cut;
		return PropagatorType::standard;
	}
	// bilinear
	if (is_cut_propagator(i))
		return PropagatorType::bilinear_cut;
	return PropagatorType::bilinear;
}

set<int> IntegralFamily::get_immediate_subsectors(int sector_id) const {
	set<int> subsectors;
	unsigned u_id = sector_id;
	for (unsigned i = 0; i < num_propagators(); ++i)
		subsectors.insert(u_id & ~(1 << i));
	subsectors.erase(u_id);
	return subsectors;
}

set<int> IntegralFamily::get_immediate_supersectors(int sector_id) const {
	set<int> supersectors;
	unsigned u_id = sector_id;
	for (unsigned i = 0; i < num_propagators(); ++i)
		supersectors.insert(u_id | (1 << i));
	supersectors.erase(u_id);
	return supersectors;
}

std::set<int> IntegralFamily::get_subsector_equivalents(int sector_id,
		bool recurse) const {
	set<int> res;
	stack<int> todo;
	int next = get_equivalent(sector_id);
	do {
		set<int> subs = get_immediate_subsectors(next);
		for (set<int>::const_iterator s = subs.begin(); s != subs.end(); ++s) {
			int equiv = get_equivalent(*s);
			if (res.insert(equiv).second)
				todo.push(equiv);
		}
	} while (recurse && !todo.empty() && (next = todo.top(), todo.pop(), true));
	return res;
}

void IntegralFamily::read(const YAML::Node& node) {
	using namespace GiNaC;

	LOGXX("Set up integral family from YAML config");
	verify_yaml_spec(node);

	if (node.FindValue("kinematics")) {
		string kinname;
		node["kinematics"] >> kinname;
		kinematics_ = Files::instance()->kinematics(kinname);
	}

	node["name"] >> name_;

	list < string > loop_momenta_str;
	node["loop_momenta"] >> loop_momenta_str;
	loop_momenta_ = create_symbols(loop_momenta_str);

	LOGXX("Parsing propagators");
	propagators_.clear();
	if (node.FindValue("propagators") != 0) {
		const YAML::Node& props_node = node["propagators"];
		for (YAML::Iterator n = props_node.begin(); n != props_node.end(); ++n)
			propagators_.push_back(
					Propagator::read(kinematics_, loop_momenta_, *n));
	}

	LOGXX("Parsing cut propagators");
	cut_propagators_mask_ = 0;
	if (node.FindValue("cut_propagators") != 0) {
		const YAML::Node& props_node = node["cut_propagators"];
		for (YAML::Iterator n = props_node.begin(); n != props_node.end(); ++n){
			int i;
			*n >> i;
            if (i < 1 || i > (int)propagators_.size())
            	throw runtime_error("invalid number '" + to_string(i) +
            		"' in cut_propagators");
			--i; // user input starts counting at 1, code starts at 0
            cut_propagators_mask_ |= (1 << i);
		}
	}

	LOGXX("Parsing permutation symmetries");
	if (node.FindValue("permutation_symmetries")) {
		const YAML::Node& plist = node["permutation_symmetries"];
		if (plist.Type() != YAML::NodeType::Sequence)
			throw runtime_error(
					"Node is not a sequence " + position_info(plist));
		for (YAML::Iterator p = plist.begin(); p != plist.end(); ++p) {
			Permutation permutation;
			permutation.read_from_onebased(*p); // internal props start with 0
			permutation_symmetries_.push_back(permutation);
		}
		incomplete_permutation_symmetries_ = permutation_symmetries_;
	}

	init();
}

void IntegralFamily::init(bool verbose) {
	using namespace GiNaC;
	try {
		if (verbose)
			LOG("Initialize integral family '" << name() << "'");
		LOGXX("Checking propagators");
		check_propagators();

		LOGXX("Init global propagator permutations");
		// check whether the propagators in the permutation exist
		const list<Permutation>& perms = permutation_symmetries_;
		list<Permutation>::const_iterator p;
		for (p = perms.begin(); p != perms.end(); ++p) {
			const vector<vector<int> >& pv = p->vector_rep();
			vector<vector<int> >::const_iterator c;
			vector<int>::const_iterator n;
			for (c = pv.begin(); c != pv.end(); ++c)
				for (n = c->begin(); n != c->end(); ++n)
					if (*n < 0 || *n >= (int) propagators_.size())
						throw runtime_error("Invalid permutation symmetry:"
								" propagator " + to_string(*n + 1) +
								" does not exist.");
		}
		// complete the global permutations
		PermutationSet aset;
		for (p = perms.begin(); p != perms.end(); ++p)
			aset.insert(*p);
		aset.complete_permutations();
		permutation_symmetries_.clear();
		copy(aset.pset().begin(), aset.pset().end(),
			 back_inserter(permutation_symmetries_));
		LOGXX("Init symbols for generic propagator exponents");
		propagator_exponent_symbols_.remove_all();
		propagator_exponents_.assign(num_propagators(), 0);
		for (size_t i = 0; i < num_propagators(); ++i) {
			propagator_exponents_[i] = GiNaC::symbol("a" + to_string(i + 1));
			propagator_exponent_symbols_.append(propagator_exponents_[i]);
		}

		LOGXX("Init mapping from scalar products to propagators");
		lst eqns; // equations: 1/Propagator(k, m2) == (ScalarProduct(k,k)-m2)
		for (size_t i = 0; i < num_propagators(); ++i) {
			// get propagator <-> sum of scalar product relation
			ex d = propagators_[i].inverse_ex();
			d = d.expand().subs(kinematics_->rules_sp_to_invariants());
			eqns.append((1 / ex(propagators_[i])) == d);
		}
		try {
			LOGXX("solving:\n" <<eqns);
			rules_sp_to_prop_ = lsolve_pattern(eqns,
					ScalarProduct(wild(1), wild(2)));
			LOGXX("solved");
		} catch (exception& e) {
			throw runtime_error(string("Can't determine scalar products from"
					" propagators\n") + e.what());
		}
		exmap::iterator r;
		for (r = rules_sp_to_prop_.begin(); r != rules_sp_to_prop_.end(); ++r)
			r->second = r->second.expand(); // expand (propi+propj)/(propi*propj)

	} catch (exception& e) {
		throw runtime_error("invalid integral family: " + name_ + ":\n" +
				e.what());
	}
}

void IntegralFamily::print(YAML::Emitter& os) const {
	using namespace YAML;
	os << BeginMap;

	os << Key << "name" << Value << name_;
	os << Key << "loop_momenta" << Value << Flow << loop_momenta_;

	os << Key << "propagators" << Value << BeginSeq;
	for (size_t i = 0; i < propagators_.size(); ++i)
		propagators_[i].print(os);
	os << EndSeq;

	os << Key << "cut_propagators" << Value << Flow << BeginSeq;
	for (size_t i = 0; i < propagators_.size(); ++i)
		if (is_cut_propagator(i))
			os << (i + 1); // internal counting starts with 0, YAML starts with 1
	os << EndSeq;

	os << Key << "permutation_symmetries" << Value << BeginSeq;
	const list<Permutation>& perms = permutation_symmetries_;
	list<Permutation>::const_iterator p;
	for (p = perms.begin(); p != perms.end(); ++p)
		p->print_to_onebased(os); // internal counting starts with 0, YAML starts with 1
	os << EndSeq;

	if (kinematics_ != 0)
		os << Key << "kinematics" << Value << kinematics_->name();
	else
		os << Key << "kinematics" << Value << "";
	os << EndMap;
}

const Kinematics* IntegralFamily::kinematics() const {
	return kinematics_;
}

int IntegralFamily::get_equivalent(int sector_id) const {
	return (INT(this, sector_id)).get_equivalent().id();
}

std::set<int> IntegralFamily::get_trivial_zero_sector_equivalents() const {
	// \todo optimize this function
	int num_loops = loop_momenta().nops();
	if (num_loops == 0)
		return set<int>();
	int max = max_sector_id();
	set<int> equivs = get_subsector_equivalents(max, true);
	equivs.insert(max);
	for (set<int>::iterator s = equivs.begin(); s != equivs.end();)
		if (!Sector(this, *s).is_obviously_zero())
			equivs.erase(s++);
		else
			++s;
	return equivs;
}

bool IntegralFamily::match_integrand(const GiNaC::ex& integrand, INT& integral,
		GiNaC::ex& coeff) const {
	using namespace GiNaC;
	integral = INT(this);
	coeff = 1;
	lst loop_moms = loop_momenta();
	lst powers;
	if (is_a < mul > (integrand)) {
		for (size_t i = 0; i < integrand.nops(); ++i)
			powers.append(integrand.op(i));
	} else if (is_a < power > (integrand) || is_a < Propagator > (integrand)
			|| freeof(integrand, loop_moms)) {
		powers.append(integrand);
	} else {
		return false;
	}
	vector<int8> v(propagators_.size(), 0);
	for (lst::const_iterator p = powers.begin(); p != powers.end(); ++p) {
		if (freeof(*p, loop_momenta())) {
			// currently we assume that all propagators should be matched
			if (p->has(Propagator(wild(1), wild(2), wild(3))))
				ABORT(
						"encountered " << *p << " with propagator free of " << name_ << " loop momenta");
			coeff *= *p;
		} else {
			exmap s;
			ex base, expo;
			if (p->match(pow(Propagator(wild(1), wild(2), wild(3)), wild(4)),
					s)) {
				base = Propagator(wild(1).subs(s), wild(2).subs(s),
						wild(3).subs(s));
				expo = wild(4).subs(s);
			} else if (p->match(Propagator(wild(1), wild(2), wild(3)), s)) {
				base = Propagator(wild(1).subs(s), wild(2).subs(s), //
				wild(3).subs(s));
				expo = 1;
			} else {
				return false;
			}
			ASSERT(!expo.is_zero());
			if (!(expo.info(info_flags::posint || expo.info(info_flags::negint))))
				return false;
			//if (!(expo.info(info_flags::integer)))
			//	return false;
			int num_expo = ex_to < numeric > (expo).to_int();
			size_t i;
			for (i = 0; i < propagators_.size(); ++i) {
				// if (base.match(propagators_[i])) {
				//ASSERT(is_a<Propagator>(base));
				if (base == propagators_[i]) {
					v[i] += num_expo;
					break;
				}
			}
			//LOG("problematic: " << base);
			if (i == propagators_.size())
				return false;
		}
	}
	integral = INT(this, v);
	return true;
}

bool IntegralFamily::have_non_standard_propagators() const {
	vector<Propagator>::const_iterator p;
	for (p = propagators_.begin(); p != propagators_.end(); ++p)
		if (!p->has_squared_momentum())
			return true;
	return false;
}

std::pair<bool, std::string> IntegralFamily::is_valid_perm_sym(
		const Permutation& perm_sym,
		bool no_perm_sym_shift_verification) const {
	using namespace GiNaC;
	const exmap& mom_cons = kinematics()->rule_momentum_conservation();
	const lst& iextm = kinematics()->independent_external_momenta();
	exmap sym2alt, alt2sym;
	lst altloopsym, altextsym;
	provide_alternative_symbols(loop_momenta(), altloopsym, sym2alt, alt2sym);
	provide_alternative_symbols(iextm, altextsym, sym2alt, alt2sym);

	YAML::Emitter ye;
	perm_sym.print_to_onebased(ye);
	const string yaml_perm = ye.c_str();

	// check whether the propagators in the permutation exist
	const vector<vector<int> >& pv = perm_sym.vector_rep();
	vector<vector<int> >::const_iterator c;
	for (c = pv.begin(); c != pv.end(); ++c)
		for (vector<int>::const_iterator n = c->begin(); n != c->end(); ++n)
			if (*n < 0 || *n >= (int) propagators_.size())
				return make_pair(false, "propagator " + to_string(*n + 1) + " does not exist.");
	// 2 equations for each relation prop1 -> prop2 (+/- sign)
	vector < list<ex> > eqns_list;
	// consistency check of the permutations and setup of the equations
	for (c = pv.begin(); c != pv.end(); ++c) {
		VERIFY(c->size() > 1);
		for (vector<int>::const_iterator n = c->begin(); n != c->end();) {
			int sn = *n;
			const Propagator& sp = propagators_[sn];
			++n;
			int dn = (n == c->end() ? *c->begin() : *n);
			const Propagator& dp = propagators_[dn];

			if (is_cut_propagator(sn) != is_cut_propagator(dn))
				return make_pair(false,
						"permutes cut with non-cut propagators.");
			if (!sp.squaredmass().is_equal(dp.squaredmass()))
				return make_pair(false,
						"permutes propagators with different masses.");

			eqns_list.push_back(list<ex>());
			const ex& s1 = sp.momentum1().subs(mom_cons);
			const ex& d1 = dp.momentum1().subs(mom_cons).subs(sym2alt);
			if (sp.has_squared_momentum() && dp.has_squared_momentum()) {
#ifdef NEW_GINAC
                eqns_list.back().push_back(lst({s1 == d1}));
                eqns_list.back().push_back(lst({s1 == -d1}));
#else
                eqns_list.back().push_back(lst(s1 == d1));
                eqns_list.back().push_back(lst(s1 == -d1));
#endif
			} else if (!sp.has_squared_momentum() && !dp.has_squared_momentum()) {
				const ex& s2 = sp.momentum2().subs(mom_cons);
				const ex& d2 = dp.momentum2().subs(mom_cons).subs(sym2alt);
#ifdef NEW_GINAC
                eqns_list.back().push_back(lst({s1 == d1, s2 == d2}));
                eqns_list.back().push_back(lst({s1 == -d1, s2 == -d2}));
#else
                eqns_list.back().push_back(lst(s1 == d1, s2 == d2));
                eqns_list.back().push_back(lst(s1 == -d1, s2 == -d2));
#endif
			} else {
				return make_pair(false,
						"permutes squared with non-squared propagators.");
			}

			if (n == c->begin())
				break;
		} // n: loop over propagators in a cycle
	} // c: loop over cycles in the permutation

	if (no_perm_sym_shift_verification)
		return make_pair(true, "");

	// add the equations of the momenta of the props not involved in the permutation
	set<int> permuted_props;
	for (c = pv.begin(); c != pv.end(); ++c)
		permuted_props.insert(c->begin(), c->end());
	for (unsigned i = 0; i < propagators_.size(); ++i) {
		if (permuted_props.find(i) == permuted_props.end()) {
			eqns_list.push_back(list<ex>());
			const ex& s1 = propagators_[i].momentum1().subs(mom_cons);
			const ex& d1 = s1.subs(sym2alt);
			if (propagators_[i].has_squared_momentum()) {
#ifdef NEW_GINAC
                eqns_list.back().push_back(lst({s1 == d1}));
                eqns_list.back().push_back(lst({s1 == -d1}));
#else
                eqns_list.back().push_back(lst(s1 == d1));
                eqns_list.back().push_back(lst(s1 == -d1));
#endif
			} else {
				const ex& s2 = propagators_[i].momentum2().subs(mom_cons);
				const ex& d2 = s2.subs(sym2alt);
#ifdef NEW_GINAC
                eqns_list.back().push_back(lst({s1 == d1, s2 == d2}));
                eqns_list.back().push_back(lst({s1 == -d1, s2 == -d2}));
#else
                eqns_list.back().push_back(lst(s1 == d1, s2 == d2));
                eqns_list.back().push_back(lst(s1 == -d1, s2 == -d2));
#endif
			}
		}
	}

	// find a shift to verify the permutation
	combinations<ex> combs(eqns_list);
	vector<ex> eqns_vec;
	bool perm_verified = false;
	while (combs.get_next(eqns_vec)) {
		lst eqns, sol_loop, sol_ext;
		for (vector<ex>::const_iterator v = eqns_vec.begin();
				v != eqns_vec.end(); ++v) {
			VERIFY(is_a<lst>(*v));
			for (size_t j = 0; j < v->nops(); ++j)
				eqns.append(v->op(j));
		}
		const lst sym = add_lst(loop_momenta(), iextm);
		LOGXX("Finding solutions for " << sym << " from equations " << eqns);
		ex solved = GiNaC::lsolve(eqns, sym);
		LOGXX("solution: " << solved);
		VERIFY(GiNaC::is_a<lst>(solved));
		if (solved.nops() == 0)
			continue;
		GiNaC::const_iterator r;
		for (r = solved.begin(); r != solved.end(); ++r) {
			VERIFY(GiNaC::is_a<relational>(*r));
			GiNaC::relational rel = GiNaC::ex_to < relational > (*r);
			VERIFY(GiNaC::is_a<symbol>(rel.lhs()));
			if (kinematics()->external_momenta().has(rel.lhs()))
				if (freeof(rel.rhs(), loop_momenta()))
					sol_ext.append(rel);
				else
					break;
			else
				sol_loop.append(rel);
		}
		if (r != solved.end()) {
			LOGXX(
					"   invalid solution: found loop momenta in right hand side of external momenta transformations.");
			continue;
		}
		ex det = GiNaC::abs(jacobi_determinant(sol_loop, altloopsym));
		if (!det.is_equal(1)) {
			LOGXX("   invalid solution: |det| = " << det);
			continue;
		}
		list<Crossing> crossings = Files::instance()->crossings(
				kinematics()->name())->ordered_crossings();
		crossings.push_front(Crossing(kinematics()));
		list<Crossing>::const_iterator cr;
		for (cr = crossings.begin(); cr != crossings.end(); ++cr) {
			if (!cr->is_equivalent_to_identity())
				continue;
			ex identity = ex(sol_ext).subs(cr->rules_momenta()).subs(alt2sym);
			GiNaC::const_iterator it;
			for (it = identity.begin(); it != identity.end(); ++it)
				if (!(it->op(0) - it->op(1)).expand().is_zero())
					break;
			if (it == identity.end())
				break; // valid crossing
		}
		if (cr == crossings.end()) {
			LOGXX("   invalid solution: invalid crossing.");
			continue;
		}
		LOGXX("Verified global permutation symmetry: " << yaml_perm);
		LOGXX("  Input equations from propagators:");
		LOGXX("  " << ex(eqns).subs(alt2sym));
		LOGX("    found shift (leaves kinematic invariants unchanged):");
		LOGX("    " << ex(solved).subs(alt2sym));
		perm_verified = true;
		break;
	}
	string msg;
	if (perm_verified)
		return make_pair(true, msg);
	msg = "no shift found to verify permutation";
	return make_pair(false, msg);
}

void IntegralFamily::check_propagators() const {
	//using namespace GiNaC;
	stringstream msg;
	size_t NoP = num_propagators();
	size_t NoLM = loop_momenta().nops();
	size_t NoEM = kinematics_->independent_external_momenta().nops();
	// check for multiple defined propagators
	GiNaC::lst props_unique;
	for (size_t i = 0; i < NoP; ++i)
		props_unique.append(propagators_[i]);
	props_unique.sort().unique();
	size_t NoP_unique = props_unique.nops();
	if (NoP != NoP_unique) {
		stringstream ss;
		for (size_t i = 0; i < NoP; ++i)
			ss << propagators_[i] << " ";
		throw runtime_error(
				"Multiple definitions of the same propagator: " + ss.str());
	}
	// check for the correct number of propagators
	if (NoP != num_scalar_products()) {
		msg << "With " << NoLM << " loop momenta " << loop_momenta() << " and "
				<< NoEM << " independent external momenta "
				<< kinematics_->independent_external_momenta()
				<< " one needs an integral family with "
				<< num_scalar_products() << " propagators.";
		throw runtime_error(msg.str());
	}
	// check if propagators contain loop momenta
	for (size_t i = 0; i < NoP; ++i) {
		const Propagator& p = propagators_[i];
		if (!freeof(p.squaredmass(), get_all_momenta())) {
			msg << "Propagator " << p << " has momentum dependent mass^2.";
			throw runtime_error(msg.str());
		} else if (freeof(p, loop_momenta())) {
			msg << "Propagator " << p << " contains no loop momenta\n" << "("
					<< loop_momenta() << ")";
			throw runtime_error(msg.str());
		}
	}
	// check if mass dimensions of propagators is 2
	for (size_t i = 0; i < NoP; ++i) {
		try {
			GiNaC::ex msq = propagators_[i].squaredmass();
			if (!msq.is_zero() && kinematics_->find_mass_dimension(msq) != 2)
				WARNING(
						"integral family " << name_ << ":" << " squared mass of propagator " << i << " specified as " << propagators_[i].squaredmass() << " has mass dimension different from 2");
		} catch (exception& e) {
			WARNING(
					"can't determine mass dimension of " << propagators_[i].squaredmass());
		}
	}
}

string IntegralFamily::t_tree_nice_string(const set<int>& sector_ids) const {
	ostringstream result;
	map<int, set<int> > ids_by_t;
	for (set<int>::const_iterator s = sector_ids.begin(); s != sector_ids.end();
			++s)
		ids_by_t[get_t(*s)].insert(*s);
	map<int, set<int> >::const_iterator t;
	for (t = ids_by_t.begin(); t != ids_by_t.end(); ++t) {
		result << "t = " << t->first << ":\n";
		const set<int>& ids = t->second;
		for (set<int>::const_iterator i = ids.begin(); i != ids.end();) {
			result << *i;
			if (++i != ids.end())
				result << ", ";
		}
		result << "\n";
	}
	return result.str();
}

void IntegralFamily::print_info() const {
	LOGX("Integral family '" << name_ << "':");
	YAML::Emitter e;
    print(e);
    LOGX(e.c_str());
    LOGX("");
    LOGX("Cut propagators form sector with id " << cut_propagators_mask_);
    LOGX("");
    LOGX("Rules scalar products to propagators:\n" << rules_sp_to_prop());
    LOGX("");
    LOG("Maximal sector id is " << max_sector_id() << ".");
    LOG(max_sector_id() + 1 << " sectors occur in total");
}

// CrossedIntegralFamily members

CrossedIntegralFamily::CrossedIntegralFamily(const IntegralFamily* source_ic,
		const Crossing& crossing) : //
		IntegralFamily(source_ic->kinematics()), //
		source_integralfamily_(source_ic), //
		crossing_(crossing) {
	using namespace GiNaC;

	string name = crossing.name_for_crossed_family(source_ic);

	vector<Propagator> props = source_ic->propagators();
	vector<Propagator>::iterator p;
	for (p = props.begin(); p != props.end(); ++p) {
		ex tmp = p->subs(crossing.rules_momenta());
		tmp = tmp.eval(); // bring Propagator into canonical form
		ASSERT(is_a<Propagator>(tmp));
		*p = ex_to < Propagator > (tmp);
	}

	*dynamic_cast<IntegralFamily*>(this) = //
			IntegralFamily(source_ic->kinematics(), name,
					source_ic->loop_momenta(), props,
					source_ic->cut_propagators_mask());
}

CrossedIntegralFamily::~CrossedIntegralFamily() {
}

void sequel(vector<int8> & vec, int8 sum, int8 min, int level) {
	// calculate the value of the first entry: sum - values of second ... last entries
	int8 first = sum;
	int length = static_cast<int>(vec.size());
	for (int i = 1; i < length; ++i) {
		first -= vec[i];
	}
	if (level == 0) {
		vec[0] = first;
		return;
	}
	if (first > min) {
		// increase entry at position level and set first entry
		vec[level]++;
		vec[0] = first - 1;
		return;
	} else {
		// set last entry to min and call the function on the next smaller level
		vec[level] = min;
		sequel(vec, sum, min, level - 1);
	}
}

void partition(vector<vector<int8> > & array, size_t lim, int length, int sum,
		bool bu) {
	if (length == 0 || lim == 0) {
		return;
	}
	int8 min, max;
	if (bu) { // for propagators in the denomoinator
		min = 1;
		max = static_cast<int8>(sum - length + 1);
	} else { // for propagators in the numerator
		min = 0;
		max = static_cast<int8>(sum);
	}
	// build the first vector {max, min, ..., min}
	array[0][0] = max;
	for (int i = 1; i < length; ++i) {
		array[0][i] = min;
	}
	// build all other vectors
	for (size_t i = 1; i < lim; ++i) {
		array[i] = array[i - 1];
		sequel(array[i], static_cast<int8>(sum), min, length - 1);
	}
}

}
// namespace Reduze
